import type * as ts from "./tsserverlibrary.shim"
import type {
  AreTypesMutuallyAssignableArguments,
  AreTypesMutuallyAssignableResponse,
  GetCompletionSymbolsArguments,
  GetCompletionSymbolsResponse,
  GetElementTypeArguments,
  GetElementTypeResponse,
  GetResolvedSignatureArguments,
  GetResolvedSignatureResponse,
  GetSymbolTypeArguments,
  GetSymbolTypeResponse,
  GetTypePropertiesArguments,
  GetTypePropertyArguments,
  GetTypeTextArguments,
  GetTypeTextResponse,
  SymbolResponse,
} from "./protocol"
import {
  areTypesMutuallyAssignable,
  getCompletionSymbols,
  getElementTypeByOffsets,
  getResolvedSignature,
  getSymbolType,
  getTypeProperties,
  getTypeProperty,
  getTypeText, ReverseMapper,
} from "./ide-get-element-type"
import {throwIdeError} from "./utils"

const decoratedLanguageServices = new WeakSet();

export function decorateLanguageService(languageService: ts.LanguageService) {
  if (decoratedLanguageServices.has(languageService)) {
    return
  }
  decoratedLanguageServices.add(languageService);

  languageService.webStormGetCompletionSymbols = (ts, ls, fileName, position, cancellationToken, reverseMapper): GetCompletionSymbolsResponse => {
    const program = languageService.getProgram();
    if (!program) {
      return undefined;
    }
    const sourceFile = program.getSourceFile(fileName);
    if (!sourceFile) {
      return undefined;
    }

    return getCompletionSymbols(
      ls,
      ts,
      program,
      sourceFile.fileName,
      position,
      cancellationToken,
      reverseMapper
    );
  }

  languageService.webStormGetElementType = (ts, fileName, startOffset, endOffset, typeRequestKind, forceReturnType, cancellationToken, reverseMapper) => {
    // see getQuickInfoAtPosition
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    const sourceFile = program.getSourceFile(fileName);
    if (!sourceFile) {
      return undefined
    }

    return getElementTypeByOffsets(ts, languageService.ideProjectId, program, sourceFile, startOffset, endOffset, typeRequestKind,
                                   forceReturnType, cancellationToken, reverseMapper)
  }

  languageService.webStormGetSymbolType = (ts, symbolId, cancellationToken, reverseMapper) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getSymbolType(ts, program, symbolId, cancellationToken, reverseMapper)
  }

  languageService.webStormGetTypeText = (ts, symbolId, flags, cancellationToken) => {
    const program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getTypeText(ts, program, symbolId, flags);
  }

  languageService.webStormGetTypeProperties = (ts, typeId, cancellationToken, reverseMapper) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getTypeProperties(ts, program, typeId, cancellationToken, reverseMapper)
  }

  languageService.webStormGetTypeProperty = (ts, typeId, propertyName, cancellationToken, reverseMapper) => {
    let program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    return getTypeProperty(ts, program, typeId, propertyName, cancellationToken, reverseMapper)
  }

  languageService.webStormAreTypesMutuallyAssignable = function(ts, type1Id, type2Id, cancellationToken): AreTypesMutuallyAssignableResponse {
    const program = languageService.getProgram();
    if (!program) {
      return undefined
    }

    return areTypesMutuallyAssignable(ts, program, type1Id, type2Id, cancellationToken)
  }

  languageService.webStormGetResolvedSignature = function(ts, fileName, startOffset, endOffset, cancellationToken, reverseMapper): GetResolvedSignatureResponse {
    const program = languageService.getProgram();
    if (!program) {
      return undefined
    }
    const sourceFile = program.getSourceFile(fileName);
    if (!sourceFile) {
      return undefined
    }

    // Convert offsets to Range
    const start = ts.getLineAndCharacterOfPosition(sourceFile, startOffset);
    const end = ts.getLineAndCharacterOfPosition(sourceFile, endOffset);
    const range = { start, end };

    return getResolvedSignature(ts, languageService.ideProjectId, program, sourceFile, range, cancellationToken, reverseMapper)
  }
}

export function getCompletionSymbolsTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetCompletionSymbolsArguments
): GetCompletionSymbolsResponse {
  let fileName = ts.server.toNormalizedPath(requestArguments.file)
  let {project, sourceFile} = projectService.ideProjectService.getProjectAndSourceFile(requestArguments.file,
    requestArguments.projectFileName)

  if (!project || !sourceFile)
    return undefined;

  let { line, character } = requestArguments.position
  let position = ts.getPositionOfLineAndCharacter(sourceFile, line, character)
  let languageService = project.getLanguageService()
  return languageService.webStormGetCompletionSymbols(
    ts,
    languageService,
    fileName,
    position,
    projectService.cancellationToken
  )
}

export function getElementTypeTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                       projectService: ts.server.ProjectService,
                                       requestArguments: GetElementTypeArguments): GetElementTypeResponse {
  let fileName = ts.server.toNormalizedPath(requestArguments.file)
  let {project, sourceFile} = projectService.ideProjectService.getProjectAndSourceFile(requestArguments.file,
                                                                                       requestArguments.projectFileName)

  if (!project || !sourceFile)
    return undefined

  let range = requestArguments.range
  let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)
  return project.getLanguageService().webStormGetElementType(ts, fileName, startOffset, endOffset, requestArguments.typeRequestKind,
                                                             requestArguments.forceReturnType, projectService.cancellationToken)
}

export function getSymbolTypeTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                      projectService: ts.server.ProjectService,
                                      requestArguments: GetSymbolTypeArguments): GetSymbolTypeResponse {
  const languageService = findLanguageService(
    projectService, requestArguments.ideProjectId, requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetSymbolType(ts, requestArguments.symbolId, projectService.cancellationToken)
}

export function getTypeTextTsServer(
  ts: typeof import("typescript/lib/tsserverlibrary"),
  projectService: ts.server.ProjectService,
  requestArguments: GetTypeTextArguments
): GetTypeTextResponse {
  const languageService = findLanguageService(
    projectService,
    requestArguments.ideProjectId,
    requestArguments.ideTypeCheckerId
  );

  if (!languageService) return undefined;

  return languageService.webStormGetTypeText(
    ts,
    requestArguments.symbolId,
    requestArguments.flags,
    projectService.cancellationToken,
  );
}

export function getTypePropertiesTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                          projectService: ts.server.ProjectService,
                                          requestArguments: GetTypePropertiesArguments): GetElementTypeResponse {
  const languageService = findLanguageService(
    projectService, requestArguments.ideProjectId, requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetTypeProperties(ts, requestArguments.typeId, projectService.cancellationToken)
}

export function getTypePropertyTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                        projectService: ts.server.ProjectService,
                                        requestArguments: GetTypePropertyArguments): SymbolResponse {
  const languageService = findLanguageService(
    projectService, requestArguments.ideProjectId, requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }
  return languageService.webStormGetTypeProperty(ts, requestArguments.typeId, requestArguments.propertyName, projectService.cancellationToken)
}

export function areTypesMutuallyAssignableTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                                   projectService: ts.server.ProjectService,
                                                   requestArguments: AreTypesMutuallyAssignableArguments): AreTypesMutuallyAssignableResponse {
  const languageService = findLanguageService(
    projectService,
    requestArguments.ideProjectId,
    requestArguments.ideTypeCheckerId,
  )
  if (!languageService) {
    return undefined
  }

  return languageService.webStormAreTypesMutuallyAssignable(
    ts,
    requestArguments.type1Id,
    requestArguments.type2Id,
    projectService.cancellationToken,
  )
}

export function getResolvedSignatureTsServer(ts: typeof import("typescript/lib/tsserverlibrary"),
                                         projectService: ts.server.ProjectService,
                                         requestArguments: GetResolvedSignatureArguments): GetResolvedSignatureResponse {
  let fileName = ts.server.toNormalizedPath(requestArguments.file)
  let {project, sourceFile} = projectService.ideProjectService.getProjectAndSourceFile(requestArguments.file,
                                                                                     requestArguments.projectFileName)

  if (!project || !sourceFile)
    return undefined

  let range = requestArguments.range
  let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)
  return project.getLanguageService().webStormGetResolvedSignature(ts, fileName, startOffset, endOffset,
                                                                 projectService.cancellationToken)
}

function findLanguageService(projectService: ts.server.ProjectService, ideProjectId: number, ideTypeCheckerId: number): ts.LanguageService | undefined {
  for (let [, project] of projectService.configuredProjects as Map<string, ts.server.ConfiguredProject>) {
    if (project.ideProjectId === ideProjectId) {
      return getLanguageService(project)
    }
  }

  for (let inferredProject of projectService.inferredProjects) {
    if (inferredProject.ideProjectId === ideProjectId) {
      return getLanguageService(inferredProject)
    }
  }

  for (let externalProject of projectService.externalProjects) {
    if (externalProject.ideProjectId === ideProjectId) {
      return getLanguageService(externalProject)
    }
  }

  function getLanguageService(project: ts.server.Project) {
    let program = project.getLanguageService().getProgram()
    if (program?.getTypeChecker()?.webStormCacheInfo?.ideTypeCheckerId == ideTypeCheckerId) {
      return project.getLanguageService()
    }
    throwIdeError("OutdatedTypeCheckerIdException")
  }

  console.error(`findLanguageService - failed to find language service for ideProjectId ${ideProjectId}`)
}
