/*
 * Copyright (c) 2010-2013, 2016, 2019, 2023, 2025 Eike Stepper (Loehne, Germany) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Eike Stepper - initial API and implementation
 */
package org.eclipse.emf.cdo.server.internal.db.mapping.horizontal;

import org.eclipse.emf.cdo.common.id.CDOID;
import org.eclipse.emf.cdo.common.protocol.CDODataInput;
import org.eclipse.emf.cdo.common.protocol.CDODataOutput;
import org.eclipse.emf.cdo.server.db.IDBStoreAccessor;
import org.eclipse.emf.cdo.server.db.IIDHandler;
import org.eclipse.emf.cdo.server.internal.db.IObjectTypeMapper;

import org.eclipse.net4j.util.lifecycle.LifecycleUtil;
import org.eclipse.net4j.util.om.monitor.OMMonitor;

import org.eclipse.emf.ecore.EClass;

import java.io.IOException;
import java.sql.Connection;

/**
 * @author Eike Stepper
 * @since 4.0
 */
public abstract class DelegatingObjectTypeMapper extends AbstractObjectTypeMapper
{
  private IObjectTypeMapper delegate;

  public DelegatingObjectTypeMapper()
  {
  }

  public IObjectTypeMapper getDelegate()
  {
    return delegate;
  }

  public void setDelegate(IObjectTypeMapper delegate)
  {
    this.delegate = delegate;
  }

  @Override
  public EClass getObjectType(IDBStoreAccessor accessor, CDOID id)
  {
    CDOID type = doGetObjectType(accessor, id);
    if (type != null)
    {
      return (EClass)getMetaDataManager().getMetaInstance(type);
    }

    return delegate.getObjectType(accessor, id);
  }

  @Override
  public boolean putObjectType(IDBStoreAccessor accessor, long timeStamp, CDOID id, EClass type)
  {
    CDOID classID = getMetaDataManager().getMetaID(type, timeStamp);
    if (!doPutObjectType(accessor, id, classID))
    {
      return false;
    }

    return delegate.putObjectType(accessor, timeStamp, id, type);
  }

  @Override
  public boolean removeObjectType(IDBStoreAccessor accessor, CDOID id)
  {
    doRemoveObjectType(accessor, id);
    return delegate.removeObjectType(accessor, id);
  }

  @Override
  public CDOID getMaxID(Connection connection, IIDHandler idHandler)
  {
    CDOID maxID = doGetMaxID(connection, idHandler);
    if (maxID != null)
    {
      return maxID;
    }

    return delegate.getMaxID(connection, idHandler);
  }

  @Override
  public void rawExport(Connection connection, CDODataOutput out, long fromCommitTime, long toCommitTime) throws IOException
  {
    delegate.rawExport(connection, out, fromCommitTime, toCommitTime);
  }

  @Override
  public void rawImport(Connection connection, CDODataInput in, OMMonitor monitor) throws IOException
  {
    delegate.rawImport(connection, in, monitor);
  }

  @Override
  protected void doBeforeActivate() throws Exception
  {
    super.doBeforeActivate();
    checkState(delegate, "delegate"); //$NON-NLS-1$
  }

  @Override
  protected void doActivate() throws Exception
  {
    super.doActivate();
    LifecycleUtil.activate(delegate);
  }

  @Override
  protected void doDeactivate() throws Exception
  {
    LifecycleUtil.deactivate(delegate);
    super.doDeactivate();
  }

  protected abstract CDOID doGetMaxID(Connection connection, IIDHandler idHandler);

  protected abstract CDOID doGetObjectType(IDBStoreAccessor accessor, CDOID id);

  protected abstract boolean doPutObjectType(IDBStoreAccessor accessor, CDOID id, CDOID type);

  protected abstract boolean doRemoveObjectType(IDBStoreAccessor accessor, CDOID id);
}
