/*
 * Copyright (c) 2010-2012, 2019 Eike Stepper (Loehne, Germany) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Eike Stepper - initial API and implementation
 */
package org.eclipse.net4j.util.collection;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;

/**
 * @author Eike Stepper
 * @since 3.0
 */
public abstract class IndexedList<E> implements List<E>
{
  public IndexedList()
  {
  }

  @Override
  public abstract E get(int index);

  @Override
  public abstract int size();

  @Override
  public boolean isEmpty()
  {
    return size() == 0;
  }

  @Override
  public boolean contains(Object o)
  {
    int size = size();
    for (int i = 0; i < size; i++)
    {
      if (get(i).equals(o))
      {
        return true;
      }
    }

    return false;
  }

  @Override
  public boolean containsAll(Collection<?> c)
  {
    for (Object object : c)
    {
      if (!contains(object))
      {
        return false;
      }
    }

    return true;
  }

  @Override
  public int indexOf(Object o)
  {
    return 0;
  }

  @Override
  public int lastIndexOf(Object o)
  {
    return 0;
  }

  @Override
  public Iterator<E> iterator()
  {
    return new IndexedIterator();
  }

  @Override
  public ListIterator<E> listIterator()
  {
    return new IndexedListIterator(0);
  }

  @Override
  public ListIterator<E> listIterator(int index)
  {
    if (index < 0 || index > size())
    {
      throw new IndexOutOfBoundsException("Index: " + index);
    }

    return new IndexedListIterator(index);
  }

  @Override
  public List<E> subList(int fromIndex, int toIndex)
  {
    return null;
  }

  @Override
  public Object[] toArray()
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public <T> T[] toArray(T[] a)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean add(E o)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean remove(Object o)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean addAll(Collection<? extends E> c)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean addAll(int index, Collection<? extends E> c)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean removeAll(Collection<?> c)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean retainAll(Collection<?> c)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public void clear()
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public E set(int index, E element)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public void add(int index, E element)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public E remove(int index)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public String toString()
  {
    StringBuilder builder = new StringBuilder();
    builder.append("{"); //$NON-NLS-1$
    for (int i = 0; i < size(); i++)
    {
      if (i != 0)
      {
        builder.append(", "); //$NON-NLS-1$
      }

      builder.append(get(i).toString());
    }

    builder.append("}"); //$NON-NLS-1$
    return builder.toString();
  }

  /**
   * @author Eike Stepper
   */
  private class IndexedIterator implements Iterator<E>
  {
    int pos = 0;

    @Override
    public boolean hasNext()
    {
      return pos != size();
    }

    @Override
    public E next()
    {
      try
      {
        return get(pos++);
      }
      catch (IndexOutOfBoundsException ex)
      {
        throw new NoSuchElementException();
      }
    }

    @Override
    public void remove()
    {
      throw new UnsupportedOperationException();
    }
  }

  /**
   * @author Eike Stepper
   */
  private class IndexedListIterator extends IndexedIterator implements ListIterator<E>
  {
    IndexedListIterator(int index)
    {
      pos = index;
    }

    @Override
    public boolean hasPrevious()
    {
      return pos != 0;
    }

    @Override
    public E previous()
    {
      try
      {
        return get(--pos);
      }
      catch (IndexOutOfBoundsException ex)
      {
        throw new NoSuchElementException();
      }
    }

    @Override
    public int nextIndex()
    {
      return pos;
    }

    @Override
    public int previousIndex()
    {
      return pos - 1;
    }

    @Override
    public void set(E o)
    {
      throw new UnsupportedOperationException();
    }

    @Override
    public void add(E o)
    {
      throw new UnsupportedOperationException();
    }
  }

  /**
   * @author Eike Stepper
   */
  public static abstract class ArrayBacked<E> extends IndexedList<E>
  {
    public ArrayBacked()
    {
    }

    protected abstract E[] getArray();

    @Override
    public E get(int i)
    {
      return getArray()[i];
    }

    @Override
    public int size()
    {
      return getArray().length;
    }
  }
}
