/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf.wal;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import java.util.zip.CRC32C;
import java.util.zip.GZIPInputStream;
import org.eclipse.rdf4j.model.Value;
import org.eclipse.rdf4j.model.impl.SimpleValueFactory;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalConfig;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalValueKind;

public final class ValueStoreWalSearch {
    private static final Pattern SEGMENT_PATTERN = Pattern.compile("wal-(\\d+)\\.v1(?:\\.gz)?");
    private final ValueStoreWalConfig config;
    private final JsonFactory jsonFactory = new JsonFactory();
    private volatile List<SegFirst> cachedSegments;

    private ValueStoreWalSearch(ValueStoreWalConfig config) {
        this.config = Objects.requireNonNull(config, "config");
    }

    public static ValueStoreWalSearch open(ValueStoreWalConfig config) {
        return new ValueStoreWalSearch(config);
    }

    public Value findValueById(int id) throws IOException {
        if (!Files.isDirectory(this.config.walDirectory(), new LinkOption[0])) {
            this.invalidateSegmentCache();
            return null;
        }
        LookupOutcome firstAttempt = this.locateCandidate(id, false);
        if (firstAttempt.value != null || !firstAttempt.retry) {
            return firstAttempt.value;
        }
        LookupOutcome secondAttempt = this.locateCandidate(id, true);
        return secondAttempt.value;
    }

    private LookupOutcome locateCandidate(int targetId, boolean forceRefresh) throws IOException {
        Optional<Value> value;
        List<SegFirst> segments = this.loadSegments(forceRefresh);
        if (segments.isEmpty()) {
            return LookupOutcome.miss(!forceRefresh);
        }
        SegFirst candidate = this.selectSegment(segments, targetId);
        if (candidate == null) {
            return LookupOutcome.miss(!forceRefresh);
        }
        try {
            value = this.scanSegmentForId(candidate.path, targetId);
        }
        catch (NoSuchFileException missingSegment) {
            this.invalidateSegmentCache();
            return LookupOutcome.miss(!forceRefresh);
        }
        if (value.isPresent()) {
            return LookupOutcome.hit(value.get());
        }
        return LookupOutcome.miss(!forceRefresh);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<SegFirst> loadSegments(boolean forceRefresh) throws IOException {
        List<SegFirst> snapshot;
        if (forceRefresh) {
            this.invalidateSegmentCache();
        }
        if ((snapshot = this.cachedSegments) != null) {
            return snapshot;
        }
        ValueStoreWalSearch valueStoreWalSearch = this;
        synchronized (valueStoreWalSearch) {
            snapshot = this.cachedSegments;
            if (snapshot == null) {
                this.cachedSegments = snapshot = this.readSegmentsFromDisk();
            }
            return snapshot;
        }
    }

    private List<SegFirst> readSegmentsFromDisk() throws IOException {
        if (!Files.isDirectory(this.config.walDirectory(), new LinkOption[0])) {
            return List.of();
        }
        ArrayList segments = new ArrayList();
        try (Stream<Path> stream = Files.list(this.config.walDirectory());){
            stream.forEach(p -> {
                long firstId1;
                Matcher m = SEGMENT_PATTERN.matcher(p.getFileName().toString());
                if (m.matches() && (firstId1 = Long.parseLong(m.group(1))) >= Integer.MIN_VALUE && firstId1 <= Integer.MAX_VALUE) {
                    segments.add(new SegFirst((Path)p, (int)firstId1));
                }
            });
        }
        return List.copyOf(segments);
    }

    private SegFirst selectSegment(List<SegFirst> segments, int targetId) {
        SegFirst best = null;
        for (SegFirst segment : segments) {
            if (segment.firstId > targetId || best != null && segment.firstId <= best.firstId) continue;
            best = segment;
        }
        return best;
    }

    private void invalidateSegmentCache() {
        this.cachedSegments = null;
    }

    private Optional<Value> scanSegmentForId(Path segment, int targetId) throws IOException {
        if (segment.getFileName().toString().endsWith(".gz")) {
            try (GZIPInputStream in = new GZIPInputStream(Files.newInputStream(segment, new OpenOption[0]));){
                while (true) {
                    Value value;
                    Optional<Value> optional;
                    int length;
                    if ((length = this.readIntLE(in)) == -1) {
                        optional = Optional.empty();
                        return optional;
                    }
                    if (length <= 0 || (long)length > 0x20000000L) {
                        optional = Optional.empty();
                        return optional;
                    }
                    byte[] data = in.readNBytes(length);
                    if (data.length < length) {
                        Optional<Value> optional2 = Optional.empty();
                        return optional2;
                    }
                    int expectedCrc = this.readIntLE(in);
                    CRC32C crc32c = new CRC32C();
                    crc32c.update(data, 0, data.length);
                    if ((int)crc32c.getValue() != expectedCrc) {
                        Optional<Value> optional3 = Optional.empty();
                        return optional3;
                    }
                    Parsed p = this.parseJson(data);
                    if (p.type != 'M' || p.id != targetId || (value = this.toValue(p)) == null) continue;
                    Optional<Value> optional4 = Optional.of(value);
                    return optional4;
                }
            }
        }
        try (FileChannel ch = FileChannel.open(segment, StandardOpenOption.READ);){
            ByteBuffer header = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
            while (true) {
                Value value;
                int n;
                header.clear();
                int r = ch.read(header);
                if (r == -1) {
                    Optional<Value> expectedCrc = Optional.empty();
                    return expectedCrc;
                }
                if (r < 4) {
                    Optional<Value> expectedCrc = Optional.empty();
                    return expectedCrc;
                }
                header.flip();
                int length = header.getInt();
                if (length <= 0 || (long)length > 0x20000000L) {
                    Optional<Value> crc32c = Optional.empty();
                    return crc32c;
                }
                byte[] data = new byte[length];
                ByteBuffer dataBuf = ByteBuffer.wrap(data);
                for (int total = 0; total < length; total += n) {
                    n = ch.read(dataBuf);
                    if (n >= 0) continue;
                    Optional<Value> optional = Optional.empty();
                    return optional;
                }
                ByteBuffer crcBuf = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
                int crcRead = ch.read(crcBuf);
                if (crcRead < 4) {
                    Optional<Value> optional = Optional.empty();
                    return optional;
                }
                crcBuf.flip();
                int expectedCrc = crcBuf.getInt();
                CRC32C crc32c = new CRC32C();
                crc32c.update(data, 0, data.length);
                if ((int)crc32c.getValue() != expectedCrc) {
                    Optional<Value> optional = Optional.empty();
                    return optional;
                }
                Parsed p = this.parseJson(data);
                if (p.type != 'M' || p.id != targetId || (value = this.toValue(p)) == null) continue;
                Optional<Value> optional = Optional.of(value);
                return optional;
            }
        }
    }

    private int readIntLE(InputStream in) throws IOException {
        byte[] b = in.readNBytes(4);
        if (b.length < 4) {
            return -1;
        }
        return b[0] & 0xFF | (b[1] & 0xFF) << 8 | (b[2] & 0xFF) << 16 | (b[3] & 0xFF) << 24;
    }

    private Parsed parseJson(byte[] jsonBytes) throws IOException {
        Parsed parsed = new Parsed();
        try (JsonParser jp = this.jsonFactory.createParser(jsonBytes);){
            if (jp.nextToken() != JsonToken.START_OBJECT) {
                Parsed parsed2 = parsed;
                return parsed2;
            }
            while (jp.nextToken() != JsonToken.END_OBJECT) {
                String field = jp.getCurrentName();
                jp.nextToken();
                if ("t".equals(field)) {
                    String t = jp.getValueAsString("");
                    parsed.type = (char)(t.isEmpty() ? 63 : (int)t.charAt(0));
                    continue;
                }
                if ("lsn".equals(field)) {
                    parsed.lsn = jp.getValueAsLong(-1L);
                    continue;
                }
                if ("id".equals(field)) {
                    parsed.id = jp.getValueAsInt(0);
                    continue;
                }
                if ("vk".equals(field)) {
                    String code = jp.getValueAsString("");
                    parsed.kind = ValueStoreWalValueKind.fromCode(code);
                    continue;
                }
                if ("lex".equals(field)) {
                    parsed.lex = jp.getValueAsString("");
                    continue;
                }
                if ("dt".equals(field)) {
                    parsed.dt = jp.getValueAsString("");
                    continue;
                }
                if ("lang".equals(field)) {
                    parsed.lang = jp.getValueAsString("");
                    continue;
                }
                if ("hash".equals(field)) {
                    parsed.hash = jp.getValueAsInt(0);
                    continue;
                }
                jp.skipChildren();
            }
        }
        return parsed;
    }

    private Value toValue(Parsed p) {
        SimpleValueFactory vf = SimpleValueFactory.getInstance();
        switch (p.kind) {
            case IRI: {
                return vf.createIRI(p.lex);
            }
            case BNODE: {
                return vf.createBNode(p.lex);
            }
            case LITERAL: {
                if (p.lang != null && !p.lang.isEmpty()) {
                    return vf.createLiteral(p.lex, p.lang);
                }
                if (p.dt != null && !p.dt.isEmpty()) {
                    return vf.createLiteral(p.lex, vf.createIRI(p.dt));
                }
                return vf.createLiteral(p.lex);
            }
        }
        return null;
    }

    private static final class LookupOutcome {
        final Value value;
        final boolean retry;

        private LookupOutcome(Value value, boolean retry) {
            this.value = value;
            this.retry = retry;
        }

        static LookupOutcome hit(Value value) {
            return new LookupOutcome(value, false);
        }

        static LookupOutcome miss(boolean retry) {
            return new LookupOutcome(null, retry);
        }
    }

    private static final class SegFirst {
        final Path path;
        final int firstId;

        SegFirst(Path p, int id) {
            this.path = p;
            this.firstId = id;
        }
    }

    private static final class Parsed {
        char type = (char)63;
        long lsn = -1L;
        int id = 0;
        ValueStoreWalValueKind kind = ValueStoreWalValueKind.NAMESPACE;
        String lex = "";
        String dt = "";
        String lang = "";
        int hash = 0;

        private Parsed() {
        }
    }
}

