package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link EnginePlugin}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableEnginePlugin.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableEnginePlugin implements EnginePlugin {
  private final String type;
  private final String name;

  private ImmutableEnginePlugin(String type, String name) {
    this.type = type;
    this.name = name;
  }

  /**
   * @return The value of the {@code type} attribute
   */
  @JsonProperty("Type")
  @Override
  public String type() {
    return type;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public String name() {
    return name;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EnginePlugin#type() type} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for type
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEnginePlugin withType(String value) {
    String newValue = Objects.requireNonNull(value, "type");
    if (this.type.equals(newValue)) return this;
    return new ImmutableEnginePlugin(newValue, this.name);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EnginePlugin#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEnginePlugin withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new ImmutableEnginePlugin(this.type, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableEnginePlugin} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableEnginePlugin
        && equalsByValue((ImmutableEnginePlugin) another);
  }

  private boolean equalsByValue(ImmutableEnginePlugin another) {
    return type.equals(another.type)
        && name.equals(another.name);
  }

  /**
   * Computes a hash code from attributes: {@code type}, {@code name}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + type.hashCode();
    h += (h << 5) + name.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code EnginePlugin} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "EnginePlugin{"
        + "type=" + type
        + ", name=" + name
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link EnginePlugin} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable EnginePlugin instance
   */
  public static ImmutableEnginePlugin copyOf(EnginePlugin instance) {
    if (instance instanceof ImmutableEnginePlugin) {
      return (ImmutableEnginePlugin) instance;
    }
    return ImmutableEnginePlugin.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableEnginePlugin ImmutableEnginePlugin}.
   * <pre>
   * ImmutableEnginePlugin.builder()
   *    .type(String) // required {@link EnginePlugin#type() type}
   *    .name(String) // required {@link EnginePlugin#name() name}
   *    .build();
   * </pre>
   * @return A new ImmutableEnginePlugin builder
   */
  public static ImmutableEnginePlugin.Builder builder() {
    return new ImmutableEnginePlugin.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableEnginePlugin ImmutableEnginePlugin}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_TYPE = 0x1L;
    private static final long INIT_BIT_NAME = 0x2L;
    private long initBits = 0x3L;

    private String type;
    private String name;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code EnginePlugin} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(EnginePlugin instance) {
      Objects.requireNonNull(instance, "instance");
      this.type(instance.type());
      this.name(instance.name());
      return this;
    }

    /**
     * Initializes the value for the {@link EnginePlugin#type() type} attribute.
     * @param type The value for type 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Type")
    public final Builder type(String type) {
      this.type = Objects.requireNonNull(type, "type");
      initBits &= ~INIT_BIT_TYPE;
      return this;
    }

    /**
     * Initializes the value for the {@link EnginePlugin#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Builds a new {@link ImmutableEnginePlugin ImmutableEnginePlugin}.
     * @return An immutable instance of EnginePlugin
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableEnginePlugin build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableEnginePlugin(type, name);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_TYPE) != 0) attributes.add("type");
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      return "Cannot build EnginePlugin, some of required attributes are not set " + attributes;
    }
  }
}
