/* -*- mode: C -*-  */
/*
   IGraph library.
   Copyright (C) 2006-2021 The igraph development team

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
   02110-1301 USA
*/

#include "igraph_topology.h"

#include "igraph_constructors.h"
#include "igraph_interface.h"

#include "isomorphism/isoclasses.h"

/*
 * Small labelled graphs are encoded into a compact representation, a "code",
 * that fits into a single integer value. Each non-loop edge corresponds to
 * a specific bit of the integer. The edge-to-bit mappings are stored in
 * the "isoclass_idx" arrays while the bit-to-edge mappings are in the "classedges"
 * arrays.
 *
 * The "isoclass2" array is a mapping from the code of each possible labelled
 * graph to its isomorphism class. A canonical representative of each isomorphism
 * class is stored in "isographs".
 *
 * In the names of arrays, the number refers to the vertex count, while "u"
 * indicates undirected graphs (the other arrays store directed ones).
 *
 * Description of each array for graphs of size n:
 *
 * isosclass_idx represents an n-by-n matrix stored in column-major order.
 * Element i,j of the matrix is an integer with a single bit set. This bit,
 * if set, represents edge i-j in the graph code.
 *
 * isoclass2[code] gives the isomorphism class of the graph represented by code.
 * Classes are labelled by integers starting at 0, after ordering them by the
 * graph code of their smallest-code representative.
 *
 * isographs[class] is the code of a graph belonging to the given class. For each
 * class, the representative with the smallest code is chosen.
 *
 * classedges[2*i] - classedges[2*i+1] are the endpoints of the edge represented
 * by bit i in the code. Bits are numbered from most to least significant, thus
 * the most significant one has index i=0.
 */

const unsigned int igraph_i_isoclass_3_idx[] = { 0, 4, 16, 1, 0, 32, 2, 8, 0 };

const unsigned int igraph_i_isoclass_4_idx[] = {
    0, 8, 64, 512, 1, 0, 128, 1024, 2, 16, 0, 2048, 4, 32, 256, 0
};

const unsigned int igraph_i_isoclass_3u_idx[] = { 0, 1, 2, 1, 0, 4, 2, 4, 0 };

const unsigned int igraph_i_isoclass_4u_idx[] = {
    0, 1, 2, 8, 1, 0, 4, 16, 2, 4, 0, 32, 8, 16, 32, 0
};

const unsigned int igraph_i_isoclass_5u_idx[] = {
    0, 1, 2, 8, 64, 1, 0, 4, 16, 128, 2, 4, 0, 32, 256, 8, 16, 32, 0, 512, 64, 128, 256, 512, 0
};

const unsigned int igraph_i_isoclass_6u_idx[] = {
    0, 1, 2, 8, 64, 1024, 1, 0, 4, 16, 128, 2048, 2, 4, 0, 32, 256, 4096,
    8, 16, 32, 0, 512, 8192, 64, 128, 256, 512, 0, 16384, 1024, 2048,
    4096, 8192, 16384, 0
};

const unsigned int igraph_i_isoclass2_3[] = {
    0, 1, 1, 2, 1, 3, 4, 5, 1, 4, 6, 7, 2, 5, 7, 8, 1, 4, 3, 5, 6, 9, 9, 10, 4, 11,
    9, 12, 7, 12, 13, 14, 1, 6, 4, 7, 4, 9, 11, 12, 3, 9, 9, 13, 5, 10, 12, 14, 2, 7, 5, 8,
    7, 13, 12, 14, 5, 12, 10, 14, 8, 14, 14, 15
};

const unsigned int igraph_i_isoclass2_3u[] = {
    0, 1, 1, 2, 1, 2, 2, 3
};

const unsigned int igraph_i_isoclass2_4u[] = {
    0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 4, 5, 6, 6, 7, 1, 2, 5, 6, 2, 4, 6, 7, 2, 3,
    6, 7, 6, 7, 8, 9, 1, 5, 2, 6, 2, 6, 4, 7, 2, 6, 3, 7, 6, 8, 7, 9, 2, 6, 6, 8,
    3, 7, 7, 9, 4, 7, 7, 9, 7, 9, 9, 10
};

const unsigned int igraph_i_isoclass2_4[] = {
    0,  1,  1,  2,  1,  2,  2,  3,  1,  4,  5,  6,  5,  6,  7,  8,  1,  5,  9, 10,
    11, 12, 13, 14,  2,  6, 10, 15, 12, 16, 17, 18,  1,  5, 11, 12,  9, 10, 13, 14,
    2,  6, 12, 16, 10, 15, 17, 18,  2,  7, 13, 17, 13, 17, 19, 20,  3,  8, 14, 18,
    14, 18, 20, 21,  1,  5,  4,  6,  5,  7,  6,  8,  9, 22, 22, 23, 24, 25, 25, 26,
    5, 27, 22, 28, 29, 30, 31, 32, 10, 28, 33, 34, 35, 36, 37, 38, 11, 29, 39, 40,
    41, 42, 43, 44, 13, 31, 45, 46, 47, 48, 49, 50, 12, 30, 45, 51, 52, 53, 54, 55,
    14, 32, 56, 57, 58, 59, 60, 61,  1,  9,  5, 10, 11, 13, 12, 14,  5, 22, 27, 28,
    29, 31, 30, 32,  4, 22, 22, 33, 39, 45, 45, 56,  6, 23, 28, 34, 40, 46, 51, 57,
    5, 24, 29, 35, 41, 47, 52, 58,  7, 25, 30, 36, 42, 48, 53, 59,  6, 25, 31, 37,
    43, 49, 54, 60,  8, 26, 32, 38, 44, 50, 55, 61,  2, 10,  6, 15, 12, 17, 16, 18,
    10, 33, 28, 34, 35, 37, 36, 38,  6, 28, 23, 34, 40, 51, 46, 57, 15, 34, 34, 62,
    63, 64, 64, 65, 12, 35, 40, 63, 66, 67, 68, 69, 17, 37, 51, 64, 67, 70, 71, 72,
    16, 36, 46, 64, 68, 71, 73, 74, 18, 38, 57, 65, 69, 72, 74, 75,  1, 11,  5, 12,
    9, 13, 10, 14, 11, 39, 29, 40, 41, 43, 42, 44,  5, 29, 24, 35, 41, 52, 47, 58,
    12, 40, 35, 63, 66, 68, 67, 69,  9, 41, 41, 66, 76, 77, 77, 78, 13, 43, 52, 68,
    77, 79, 80, 81, 10, 42, 47, 67, 77, 80, 82, 83, 14, 44, 58, 69, 78, 81, 83, 84,
    2, 12,  6, 16, 10, 17, 15, 18, 13, 45, 31, 46, 47, 49, 48, 50,  7, 30, 25, 36,
    42, 53, 48, 59, 17, 51, 37, 64, 67, 71, 70, 72, 13, 52, 43, 68, 77, 80, 79, 81,
    19, 54, 54, 73, 82, 85, 85, 86, 17, 53, 49, 71, 80, 87, 85, 88, 20, 55, 60, 74,
    83, 88, 89, 90,  2, 13,  7, 17, 13, 19, 17, 20, 12, 45, 30, 51, 52, 54, 53, 55,
    6, 31, 25, 37, 43, 54, 49, 60, 16, 46, 36, 64, 68, 73, 71, 74, 10, 47, 42, 67,
    77, 82, 80, 83, 17, 49, 53, 71, 80, 85, 87, 88, 15, 48, 48, 70, 79, 85, 85, 89,
    18, 50, 59, 72, 81, 86, 88, 90,  3, 14,  8, 18, 14, 20, 18, 21, 14, 56, 32, 57,
    58, 60, 59, 61,  8, 32, 26, 38, 44, 55, 50, 61, 18, 57, 38, 65, 69, 74, 72, 75,
    14, 58, 44, 69, 78, 83, 81, 84, 20, 60, 55, 74, 83, 89, 88, 90, 18, 59, 50, 72,
    81, 88, 86, 90, 21, 61, 61, 75, 84, 90, 90, 91,  1,  5,  5,  7,  4,  6,  6,  8,
    9, 22, 24, 25, 22, 23, 25, 26, 11, 29, 41, 42, 39, 40, 43, 44, 13, 31, 47, 48,
    45, 46, 49, 50,  5, 27, 29, 30, 22, 28, 31, 32, 10, 28, 35, 36, 33, 34, 37, 38,
    12, 30, 52, 53, 45, 51, 54, 55, 14, 32, 58, 59, 56, 57, 60, 61,  9, 24, 22, 25,
    22, 25, 23, 26, 76, 92, 92, 93, 92, 93, 93, 94, 41, 95, 96, 97, 98, 99, 100, 101,
    77, 102, 103, 104, 105, 106, 107, 108, 41, 95, 98, 99, 96, 97, 100, 101, 77, 102, 105, 106,
    103, 104, 107, 108, 66, 109, 110, 111, 110, 111, 112, 113, 78, 114, 115, 116, 115, 116, 117, 118,
    11, 41, 29, 42, 39, 43, 40, 44, 41, 96, 95, 97, 98, 100, 99, 101, 39, 98, 98, 119,
    120, 121, 121, 122, 43, 100, 123, 124, 121, 125, 126, 127, 29, 95, 128, 129, 98, 123, 130, 131,
    42, 97, 129, 132, 119, 124, 133, 134, 40, 99, 130, 133, 121, 126, 135, 136, 44, 101, 131, 134,
    122, 127, 136, 137, 13, 47, 31, 48, 45, 49, 46, 50, 77, 103, 102, 104, 105, 107, 106, 108,
    43, 123, 100, 124, 121, 126, 125, 127, 79, 138, 138, 139, 140, 141, 141, 142, 52, 143, 130, 144,
    110, 145, 146, 147, 80, 148, 149, 150, 151, 152, 153, 154, 68, 155, 146, 156, 157, 158, 159, 160,
    81, 161, 162, 163, 164, 165, 166, 167,  5, 29, 27, 30, 22, 31, 28, 32, 41, 98, 95, 99,
    96, 100, 97, 101, 29, 128, 95, 129, 98, 130, 123, 131, 52, 130, 143, 144, 110, 146, 145, 147,
    24, 95, 95, 109, 92, 102, 102, 114, 47, 123, 143, 155, 103, 138, 148, 161, 35, 129, 143, 168,
    105, 149, 169, 170, 58, 131, 171, 172, 115, 162, 173, 174, 10, 35, 28, 36, 33, 37, 34, 38,
    77, 105, 102, 106, 103, 107, 104, 108, 42, 129, 97, 132, 119, 133, 124, 134, 80, 149, 148, 150,
    151, 153, 152, 154, 47, 143, 123, 155, 103, 148, 138, 161, 82, 169, 169, 175, 176, 177, 177, 178,
    67, 168, 145, 179, 151, 180, 181, 182, 83, 170, 173, 183, 184, 185, 186, 187, 12, 52, 30, 53,
    45, 54, 51, 55, 66, 110, 109, 111, 110, 112, 111, 113, 40, 130, 99, 133, 121, 135, 126, 136,
    68, 146, 155, 156, 157, 159, 158, 160, 35, 143, 129, 168, 105, 169, 149, 170, 67, 145, 168, 179,
    151, 181, 180, 182, 63, 144, 144, 188, 140, 189, 189, 190, 69, 147, 172, 191, 164, 192, 193, 194,
    14, 58, 32, 59, 56, 60, 57, 61, 78, 115, 114, 116, 115, 117, 116, 118, 44, 131, 101, 134,
    122, 136, 127, 137, 81, 162, 161, 163, 164, 166, 165, 167, 58, 171, 131, 172, 115, 173, 162, 174,
    83, 173, 170, 183, 184, 186, 185, 187, 69, 172, 147, 191, 164, 193, 192, 194, 84, 174, 174, 195,
    196, 197, 197, 198,  1,  9, 11, 13,  5, 10, 12, 14,  5, 22, 29, 31, 27, 28, 30, 32,
    5, 24, 41, 47, 29, 35, 52, 58,  7, 25, 42, 48, 30, 36, 53, 59,  4, 22, 39, 45,
    22, 33, 45, 56,  6, 23, 40, 46, 28, 34, 51, 57,  6, 25, 43, 49, 31, 37, 54, 60,
    8, 26, 44, 50, 32, 38, 55, 61, 11, 41, 39, 43, 29, 42, 40, 44, 41, 96, 98, 100,
    95, 97, 99, 101, 29, 95, 98, 123, 128, 129, 130, 131, 42, 97, 119, 124, 129, 132, 133, 134,
    39, 98, 120, 121, 98, 119, 121, 122, 43, 100, 121, 125, 123, 124, 126, 127, 40, 99, 121, 126,
    130, 133, 135, 136, 44, 101, 122, 127, 131, 134, 136, 137,  9, 76, 41, 77, 41, 77, 66, 78,
    24, 92, 95, 102, 95, 102, 109, 114, 22, 92, 96, 103, 98, 105, 110, 115, 25, 93, 97, 104,
    99, 106, 111, 116, 22, 92, 98, 105, 96, 103, 110, 115, 25, 93, 99, 106, 97, 104, 111, 116,
    23, 93, 100, 107, 100, 107, 112, 117, 26, 94, 101, 108, 101, 108, 113, 118, 13, 77, 43, 79,
    52, 80, 68, 81, 47, 103, 123, 138, 143, 148, 155, 161, 31, 102, 100, 138, 130, 149, 146, 162,
    48, 104, 124, 139, 144, 150, 156, 163, 45, 105, 121, 140, 110, 151, 157, 164, 49, 107, 126, 141,
    145, 152, 158, 165, 46, 106, 125, 141, 146, 153, 159, 166, 50, 108, 127, 142, 147, 154, 160, 167,
    5, 41, 29, 52, 24, 47, 35, 58, 29, 98, 128, 130, 95, 123, 129, 131, 27, 95, 95, 143,
    95, 143, 143, 171, 30, 99, 129, 144, 109, 155, 168, 172, 22, 96, 98, 110, 92, 103, 105, 115,
    31, 100, 130, 146, 102, 138, 149, 162, 28, 97, 123, 145, 102, 148, 169, 173, 32, 101, 131, 147,
    114, 161, 170, 174, 12, 66, 40, 68, 35, 67, 63, 69, 52, 110, 130, 146, 143, 145, 144, 147,
    30, 109, 99, 155, 129, 168, 144, 172, 53, 111, 133, 156, 168, 179, 188, 191, 45, 110, 121, 157,
    105, 151, 140, 164, 54, 112, 135, 159, 169, 181, 189, 192, 51, 111, 126, 158, 149, 180, 189, 193,
    55, 113, 136, 160, 170, 182, 190, 194, 10, 77, 42, 80, 47, 82, 67, 83, 35, 105, 129, 149,
    143, 169, 168, 170, 28, 102, 97, 148, 123, 169, 145, 173, 36, 106, 132, 150, 155, 175, 179, 183,
    33, 103, 119, 151, 103, 176, 151, 184, 37, 107, 133, 153, 148, 177, 180, 185, 34, 104, 124, 152,
    138, 177, 181, 186, 38, 108, 134, 154, 161, 178, 182, 187, 14, 78, 44, 81, 58, 83, 69, 84,
    58, 115, 131, 162, 171, 173, 172, 174, 32, 114, 101, 161, 131, 170, 147, 174, 59, 116, 134, 163,
    172, 183, 191, 195, 56, 115, 122, 164, 115, 184, 164, 196, 60, 117, 136, 166, 173, 186, 193, 197,
    57, 116, 127, 165, 162, 185, 192, 197, 61, 118, 137, 167, 174, 187, 194, 198,  2, 10, 12, 17,
    6, 15, 16, 18, 10, 33, 35, 37, 28, 34, 36, 38, 12, 35, 66, 67, 40, 63, 68, 69,
    17, 37, 67, 70, 51, 64, 71, 72,  6, 28, 40, 51, 23, 34, 46, 57, 15, 34, 63, 64,
    34, 62, 64, 65, 16, 36, 68, 71, 46, 64, 73, 74, 18, 38, 69, 72, 57, 65, 74, 75,
    13, 47, 45, 49, 31, 48, 46, 50, 77, 103, 105, 107, 102, 104, 106, 108, 52, 143, 110, 145,
    130, 144, 146, 147, 80, 148, 151, 152, 149, 150, 153, 154, 43, 123, 121, 126, 100, 124, 125, 127,
    79, 138, 140, 141, 138, 139, 141, 142, 68, 155, 157, 158, 146, 156, 159, 160, 81, 161, 164, 165,
    162, 163, 166, 167, 13, 77, 52, 80, 43, 79, 68, 81, 47, 103, 143, 148, 123, 138, 155, 161,
    45, 105, 110, 151, 121, 140, 157, 164, 49, 107, 145, 152, 126, 141, 158, 165, 31, 102, 130, 149,
    100, 138, 146, 162, 48, 104, 144, 150, 124, 139, 156, 163, 46, 106, 146, 153, 125, 141, 159, 166,
    50, 108, 147, 154, 127, 142, 160, 167, 19, 82, 54, 85, 54, 85, 73, 86, 82, 176, 169, 177,
    169, 177, 175, 178, 54, 169, 112, 181, 135, 189, 159, 192, 85, 177, 181, 199, 189, 200, 201, 202,
    54, 169, 135, 189, 112, 181, 159, 192, 85, 177, 189, 200, 181, 199, 201, 202, 73, 175, 159, 201,
    159, 201, 203, 204, 86, 178, 192, 202, 192, 202, 204, 205,  7, 42, 30, 53, 25, 48, 36, 59,
    42, 119, 129, 133, 97, 124, 132, 134, 30, 129, 109, 168, 99, 144, 155, 172, 53, 133, 168, 188,
    111, 156, 179, 191, 25, 97, 99, 111, 93, 104, 106, 116, 48, 124, 144, 156, 104, 139, 150, 163,
    36, 132, 155, 179, 106, 150, 175, 183, 59, 134, 172, 191, 116, 163, 183, 195, 17, 67, 51, 71,
    37, 70, 64, 72, 80, 151, 149, 153, 148, 152, 150, 154, 53, 168, 111, 179, 133, 188, 156, 191,
    87, 180, 180, 206, 180, 206, 206, 207, 49, 145, 126, 158, 107, 152, 141, 165, 85, 181, 189, 201,
    177, 199, 200, 202, 71, 179, 158, 208, 153, 206, 201, 209, 88, 182, 193, 209, 185, 210, 211, 212,
    17, 80, 53, 87, 49, 85, 71, 88, 67, 151, 168, 180, 145, 181, 179, 182, 51, 149, 111, 180,
    126, 189, 158, 193, 71, 153, 179, 206, 158, 201, 208, 209, 37, 148, 133, 180, 107, 177, 153, 185,
    70, 152, 188, 206, 152, 199, 206, 210, 64, 150, 156, 206, 141, 200, 201, 211, 72, 154, 191, 207,
    165, 202, 209, 212, 20, 83, 55, 88, 60, 89, 74, 90, 83, 184, 170, 185, 173, 186, 183, 187,
    55, 170, 113, 182, 136, 190, 160, 194, 88, 185, 182, 210, 193, 211, 209, 212, 60, 173, 136, 193,
    117, 186, 166, 197, 89, 186, 190, 211, 186, 213, 211, 214, 74, 183, 160, 209, 166, 211, 204, 215,
    90, 187, 194, 212, 197, 214, 215, 216,  1, 11,  9, 13,  5, 12, 10, 14, 11, 39, 41, 43,
    29, 40, 42, 44,  9, 41, 76, 77, 41, 66, 77, 78, 13, 43, 77, 79, 52, 68, 80, 81,
    5, 29, 41, 52, 24, 35, 47, 58, 12, 40, 66, 68, 35, 63, 67, 69, 10, 42, 77, 80,
    47, 67, 82, 83, 14, 44, 78, 81, 58, 69, 83, 84,  5, 29, 22, 31, 27, 30, 28, 32,
    41, 98, 96, 100, 95, 99, 97, 101, 24, 95, 92, 102, 95, 109, 102, 114, 47, 123, 103, 138,
    143, 155, 148, 161, 29, 128, 98, 130, 95, 129, 123, 131, 52, 130, 110, 146, 143, 144, 145, 147,
    35, 129, 105, 149, 143, 168, 169, 170, 58, 131, 115, 162, 171, 172, 173, 174,  5, 41, 24, 47,
    29, 52, 35, 58, 29, 98, 95, 123, 128, 130, 129, 131, 22, 96, 92, 103, 98, 110, 105, 115,
    31, 100, 102, 138, 130, 146, 149, 162, 27, 95, 95, 143, 95, 143, 143, 171, 30, 99, 109, 155,
    129, 144, 168, 172, 28, 97, 102, 148, 123, 145, 169, 173, 32, 101, 114, 161, 131, 147, 170, 174,
    7, 42, 25, 48, 30, 53, 36, 59, 42, 119, 97, 124, 129, 133, 132, 134, 25, 97, 93, 104,
    99, 111, 106, 116, 48, 124, 104, 139, 144, 156, 150, 163, 30, 129, 99, 144, 109, 168, 155, 172,
    53, 133, 111, 156, 168, 188, 179, 191, 36, 132, 106, 150, 155, 179, 175, 183, 59, 134, 116, 163,
    172, 191, 183, 195,  4, 39, 22, 45, 22, 45, 33, 56, 39, 120, 98, 121, 98, 121, 119, 122,
    22, 98, 92, 105, 96, 110, 103, 115, 45, 121, 105, 140, 110, 157, 151, 164, 22, 98, 96, 110,
    92, 105, 103, 115, 45, 121, 110, 157, 105, 140, 151, 164, 33, 119, 103, 151, 103, 151, 176, 184,
    56, 122, 115, 164, 115, 164, 184, 196,  6, 40, 23, 46, 28, 51, 34, 57, 43, 121, 100, 125,
    123, 126, 124, 127, 25, 99, 93, 106, 97, 111, 104, 116, 49, 126, 107, 141, 145, 158, 152, 165,
    31, 130, 100, 146, 102, 149, 138, 162, 54, 135, 112, 159, 169, 189, 181, 192, 37, 133, 107, 153,
    148, 180, 177, 185, 60, 136, 117, 166, 173, 193, 186, 197,  6, 43, 25, 49, 31, 54, 37, 60,
    40, 121, 99, 126, 130, 135, 133, 136, 23, 100, 93, 107, 100, 112, 107, 117, 46, 125, 106, 141,
    146, 159, 153, 166, 28, 123, 97, 145, 102, 169, 148, 173, 51, 126, 111, 158, 149, 189, 180, 193,
    34, 124, 104, 152, 138, 181, 177, 186, 57, 127, 116, 165, 162, 192, 185, 197,  8, 44, 26, 50,
    32, 55, 38, 61, 44, 122, 101, 127, 131, 136, 134, 137, 26, 101, 94, 108, 101, 113, 108, 118,
    50, 127, 108, 142, 147, 160, 154, 167, 32, 131, 101, 147, 114, 170, 161, 174, 55, 136, 113, 160,
    170, 190, 182, 194, 38, 134, 108, 154, 161, 182, 178, 187, 61, 137, 118, 167, 174, 194, 187, 198,
    2, 12, 10, 17,  6, 16, 15, 18, 13, 45, 47, 49, 31, 46, 48, 50, 13, 52, 77, 80,
    43, 68, 79, 81, 19, 54, 82, 85, 54, 73, 85, 86,  7, 30, 42, 53, 25, 36, 48, 59,
    17, 51, 67, 71, 37, 64, 70, 72, 17, 53, 80, 87, 49, 71, 85, 88, 20, 55, 83, 88,
    60, 74, 89, 90, 10, 35, 33, 37, 28, 36, 34, 38, 77, 105, 103, 107, 102, 106, 104, 108,
    47, 143, 103, 148, 123, 155, 138, 161, 82, 169, 176, 177, 169, 175, 177, 178, 42, 129, 119, 133,
    97, 132, 124, 134, 80, 149, 151, 153, 148, 150, 152, 154, 67, 168, 151, 180, 145, 179, 181, 182,
    83, 170, 184, 185, 173, 183, 186, 187, 12, 66, 35, 67, 40, 68, 63, 69, 52, 110, 143, 145,
    130, 146, 144, 147, 45, 110, 105, 151, 121, 157, 140, 164, 54, 112, 169, 181, 135, 159, 189, 192,
    30, 109, 129, 168, 99, 155, 144, 172, 53, 111, 168, 179, 133, 156, 188, 191, 51, 111, 149, 180,
    126, 158, 189, 193, 55, 113, 170, 182, 136, 160, 190, 194, 17, 67, 37, 70, 51, 71, 64, 72,
    80, 151, 148, 152, 149, 153, 150, 154, 49, 145, 107, 152, 126, 158, 141, 165, 85, 181, 177, 199,
    189, 201, 200, 202, 53, 168, 133, 188, 111, 179, 156, 191, 87, 180, 180, 206, 180, 206, 206, 207,
    71, 179, 153, 206, 158, 208, 201, 209, 88, 182, 185, 210, 193, 209, 211, 212,  6, 40, 28, 51,
    23, 46, 34, 57, 43, 121, 123, 126, 100, 125, 124, 127, 31, 130, 102, 149, 100, 146, 138, 162,
    54, 135, 169, 189, 112, 159, 181, 192, 25, 99, 97, 111, 93, 106, 104, 116, 49, 126, 145, 158,
    107, 141, 152, 165, 37, 133, 148, 180, 107, 153, 177, 185, 60, 136, 173, 193, 117, 166, 186, 197,
    15, 63, 34, 64, 34, 64, 62, 65, 79, 140, 138, 141, 138, 141, 139, 142, 48, 144, 104, 150,
    124, 156, 139, 163, 85, 189, 177, 200, 181, 201, 199, 202, 48, 144, 124, 156, 104, 150, 139, 163,
    85, 189, 181, 201, 177, 200, 199, 202, 70, 188, 152, 206, 152, 206, 199, 210, 89, 190, 186, 211,
    186, 211, 213, 214, 16, 68, 36, 71, 46, 73, 64, 74, 68, 157, 155, 158, 146, 159, 156, 160,
    46, 146, 106, 153, 125, 159, 141, 166, 73, 159, 175, 201, 159, 203, 201, 204, 36, 155, 132, 179,
    106, 175, 150, 183, 71, 158, 179, 208, 153, 201, 206, 209, 64, 156, 150, 206, 141, 201, 200, 211,
    74, 160, 183, 209, 166, 204, 211, 215, 18, 69, 38, 72, 57, 74, 65, 75, 81, 164, 161, 165,
    162, 166, 163, 167, 50, 147, 108, 154, 127, 160, 142, 167, 86, 192, 178, 202, 192, 204, 202, 205,
    59, 172, 134, 191, 116, 183, 163, 195, 88, 193, 182, 209, 185, 211, 210, 212, 72, 191, 154, 207,
    165, 209, 202, 212, 90, 194, 187, 212, 197, 215, 214, 216,  2, 13, 13, 19,  7, 17, 17, 20,
    12, 45, 52, 54, 30, 51, 53, 55, 10, 47, 77, 82, 42, 67, 80, 83, 17, 49, 80, 85,
    53, 71, 87, 88,  6, 31, 43, 54, 25, 37, 49, 60, 16, 46, 68, 73, 36, 64, 71, 74,
    15, 48, 79, 85, 48, 70, 85, 89, 18, 50, 81, 86, 59, 72, 88, 90, 12, 52, 45, 54,
    30, 53, 51, 55, 66, 110, 110, 112, 109, 111, 111, 113, 35, 143, 105, 169, 129, 168, 149, 170,
    67, 145, 151, 181, 168, 179, 180, 182, 40, 130, 121, 135, 99, 133, 126, 136, 68, 146, 157, 159,
    155, 156, 158, 160, 63, 144, 140, 189, 144, 188, 189, 190, 69, 147, 164, 192, 172, 191, 193, 194,
    10, 77, 47, 82, 42, 80, 67, 83, 35, 105, 143, 169, 129, 149, 168, 170, 33, 103, 103, 176,
    119, 151, 151, 184, 37, 107, 148, 177, 133, 153, 180, 185, 28, 102, 123, 169, 97, 148, 145, 173,
    36, 106, 155, 175, 132, 150, 179, 183, 34, 104, 138, 177, 124, 152, 181, 186, 38, 108, 161, 178,
    134, 154, 182, 187, 17, 80, 49, 85, 53, 87, 71, 88, 67, 151, 145, 181, 168, 180, 179, 182,
    37, 148, 107, 177, 133, 180, 153, 185, 70, 152, 152, 199, 188, 206, 206, 210, 51, 149, 126, 189,
    111, 180, 158, 193, 71, 153, 158, 201, 179, 206, 208, 209, 64, 150, 141, 200, 156, 206, 201, 211,
    72, 154, 165, 202, 191, 207, 209, 212,  6, 43, 31, 54, 25, 49, 37, 60, 40, 121, 130, 135,
    99, 126, 133, 136, 28, 123, 102, 169, 97, 145, 148, 173, 51, 126, 149, 189, 111, 158, 180, 193,
    23, 100, 100, 112, 93, 107, 107, 117, 46, 125, 146, 159, 106, 141, 153, 166, 34, 124, 138, 181,
    104, 152, 177, 186, 57, 127, 162, 192, 116, 165, 185, 197, 16, 68, 46, 73, 36, 71, 64, 74,
    68, 157, 146, 159, 155, 158, 156, 160, 36, 155, 106, 175, 132, 179, 150, 183, 71, 158, 153, 201,
    179, 208, 206, 209, 46, 146, 125, 159, 106, 153, 141, 166, 73, 159, 159, 203, 175, 201, 201, 204,
    64, 156, 141, 201, 150, 206, 200, 211, 74, 160, 166, 204, 183, 209, 211, 215, 15, 79, 48, 85,
    48, 85, 70, 89, 63, 140, 144, 189, 144, 189, 188, 190, 34, 138, 104, 177, 124, 181, 152, 186,
    64, 141, 150, 200, 156, 201, 206, 211, 34, 138, 124, 181, 104, 177, 152, 186, 64, 141, 156, 201,
    150, 200, 206, 211, 62, 139, 139, 199, 139, 199, 199, 213, 65, 142, 163, 202, 163, 202, 210, 214,
    18, 81, 50, 86, 59, 88, 72, 90, 69, 164, 147, 192, 172, 193, 191, 194, 38, 161, 108, 178,
    134, 182, 154, 187, 72, 165, 154, 202, 191, 209, 207, 212, 57, 162, 127, 192, 116, 185, 165, 197,
    74, 166, 160, 204, 183, 211, 209, 215, 65, 163, 142, 202, 163, 210, 202, 214, 75, 167, 167, 205,
    195, 212, 212, 216,  3, 14, 14, 20,  8, 18, 18, 21, 14, 56, 58, 60, 32, 57, 59, 61,
    14, 58, 78, 83, 44, 69, 81, 84, 20, 60, 83, 89, 55, 74, 88, 90,  8, 32, 44, 55,
    26, 38, 50, 61, 18, 57, 69, 74, 38, 65, 72, 75, 18, 59, 81, 88, 50, 72, 86, 90,
    21, 61, 84, 90, 61, 75, 90, 91, 14, 58, 56, 60, 32, 59, 57, 61, 78, 115, 115, 117,
    114, 116, 116, 118, 58, 171, 115, 173, 131, 172, 162, 174, 83, 173, 184, 186, 170, 183, 185, 187,
    44, 131, 122, 136, 101, 134, 127, 137, 81, 162, 164, 166, 161, 163, 165, 167, 69, 172, 164, 193,
    147, 191, 192, 194, 84, 174, 196, 197, 174, 195, 197, 198, 14, 78, 58, 83, 44, 81, 69, 84,
    58, 115, 171, 173, 131, 162, 172, 174, 56, 115, 115, 184, 122, 164, 164, 196, 60, 117, 173, 186,
    136, 166, 193, 197, 32, 114, 131, 170, 101, 161, 147, 174, 59, 116, 172, 183, 134, 163, 191, 195,
    57, 116, 162, 185, 127, 165, 192, 197, 61, 118, 174, 187, 137, 167, 194, 198, 20, 83, 60, 89,
    55, 88, 74, 90, 83, 184, 173, 186, 170, 185, 183, 187, 60, 173, 117, 186, 136, 193, 166, 197,
    89, 186, 186, 213, 190, 211, 211, 214, 55, 170, 136, 190, 113, 182, 160, 194, 88, 185, 193, 211,
    182, 210, 209, 212, 74, 183, 166, 211, 160, 209, 204, 215, 90, 187, 197, 214, 194, 212, 215, 216,
    8, 44, 32, 55, 26, 50, 38, 61, 44, 122, 131, 136, 101, 127, 134, 137, 32, 131, 114, 170,
    101, 147, 161, 174, 55, 136, 170, 190, 113, 160, 182, 194, 26, 101, 101, 113, 94, 108, 108, 118,
    50, 127, 147, 160, 108, 142, 154, 167, 38, 134, 161, 182, 108, 154, 178, 187, 61, 137, 174, 194,
    118, 167, 187, 198, 18, 69, 57, 74, 38, 72, 65, 75, 81, 164, 162, 166, 161, 165, 163, 167,
    59, 172, 116, 183, 134, 191, 163, 195, 88, 193, 185, 211, 182, 209, 210, 212, 50, 147, 127, 160,
    108, 154, 142, 167, 86, 192, 192, 204, 178, 202, 202, 205, 72, 191, 165, 209, 154, 207, 202, 212,
    90, 194, 197, 215, 187, 212, 214, 216, 18, 81, 59, 88, 50, 86, 72, 90, 69, 164, 172, 193,
    147, 192, 191, 194, 57, 162, 116, 185, 127, 192, 165, 197, 74, 166, 183, 211, 160, 204, 209, 215,
    38, 161, 134, 182, 108, 178, 154, 187, 72, 165, 191, 209, 154, 202, 207, 212, 65, 163, 163, 210,
    142, 202, 202, 214, 75, 167, 195, 212, 167, 205, 212, 216, 21, 84, 61, 90, 61, 90, 75, 91,
    84, 196, 174, 197, 174, 197, 195, 198, 61, 174, 118, 187, 137, 194, 167, 198, 90, 197, 187, 214,
    194, 215, 212, 216, 61, 174, 137, 194, 118, 187, 167, 198, 90, 197, 194, 215, 187, 214, 212, 216,
    75, 195, 167, 212, 167, 212, 205, 216, 91, 198, 198, 216, 198, 216, 216, 217
};

const unsigned int igraph_i_isoclass2_5u[] = {
    0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 4, 5, 6, 6, 7, 1, 2, 5, 6, 2, 4, 6,
    7, 2, 3, 6, 7, 6, 7, 8, 9, 1, 5, 2, 6, 2, 6, 4, 7, 2, 6, 3, 7, 6, 8,
    7, 9, 2, 6, 6, 8, 3, 7, 7, 9, 4, 7, 7, 9, 7, 9, 9, 10, 1, 2, 2, 4, 5,
    6, 6, 7, 2, 4, 4, 11, 12, 13, 13, 14, 5, 6, 12, 13, 12, 13, 15, 16,
    6, 7, 13, 14, 15, 16, 17, 18, 5, 12, 6, 13, 12, 15, 13, 16, 6, 13, 7,
    14, 15, 17, 16, 18, 12, 15, 15, 17, 19, 20, 20, 21, 13, 16, 16, 18,
    20, 21, 21, 22, 1, 2, 5, 6, 2, 4, 6, 7, 5, 6, 12, 13, 12, 13, 15, 16,
    2, 4, 12, 13, 4, 11, 13, 14, 6, 7, 15, 16, 13, 14, 17, 18, 5, 12, 12,
    15, 6, 13, 13, 16, 12, 15, 19, 20, 15, 17, 20, 21, 6, 13, 15, 17, 7,
    14, 16, 18, 13, 16, 20, 21, 16, 18, 21, 22, 2, 3, 6, 7, 6, 7, 8, 9,
    6, 7, 13, 14, 15, 16, 17, 18, 6, 7, 15, 16, 13, 14, 17, 18, 8, 9, 17,
    18, 17, 18, 23, 24, 12, 19, 15, 20, 15, 20, 17, 21, 15, 20, 20, 25,
    26, 27, 27, 28, 15, 20, 26, 27, 20, 25, 27, 28, 17, 21, 27, 28, 27,
    28, 29, 30, 1, 5, 2, 6, 2, 6, 4, 7, 5, 12, 6, 13, 12, 15, 13, 16, 5,
    12, 12, 15, 6, 13, 13, 16, 12, 19, 15, 20, 15, 20, 17, 21, 2, 12, 4,
    13, 4, 13, 11, 14, 6, 15, 7, 16, 13, 17, 14, 18, 6, 15, 13, 17, 7,
    16, 14, 18, 13, 20, 16, 21, 16, 21, 18, 22, 2, 6, 3, 7, 6, 8, 7, 9,
    6, 13, 7, 14, 15, 17, 16, 18, 12, 15, 19, 20, 15, 17, 20, 21, 15, 20,
    20, 25, 26, 27, 27, 28, 6, 15, 7, 16, 13, 17, 14, 18, 8, 17, 9, 18,
    17, 23, 18, 24, 15, 26, 20, 27, 20, 27, 25, 28, 17, 27, 21, 28, 27,
    29, 28, 30, 2, 6, 6, 8, 3, 7, 7, 9, 12, 15, 15, 17, 19, 20, 20, 21,
    6, 13, 15, 17, 7, 14, 16, 18, 15, 20, 26, 27, 20, 25, 27, 28, 6, 15,
    13, 17, 7, 16, 14, 18, 15, 26, 20, 27, 20, 27, 25, 28, 8, 17, 17, 23,
    9, 18, 18, 24, 17, 27, 27, 29, 21, 28, 28, 30, 4, 7, 7, 9, 7, 9, 9,
    10, 13, 16, 16, 18, 20, 21, 21, 22, 13, 16, 20, 21, 16, 18, 21, 22,
    17, 21, 27, 28, 27, 28, 29, 30, 13, 20, 16, 21, 16, 21, 18, 22, 17,
    27, 21, 28, 27, 29, 28, 30, 17, 27, 27, 29, 21, 28, 28, 30, 23, 29,
    29, 31, 29, 31, 31, 32, 1, 5, 5, 12, 5, 12, 12, 19, 2, 6, 6, 13, 12,
    15, 15, 20, 2, 6, 12, 15, 6, 13, 15, 20, 4, 7, 13, 16, 13, 16, 17,
    21, 2, 12, 6, 15, 6, 15, 13, 20, 4, 13, 7, 16, 13, 17, 16, 21, 4, 13,
    13, 17, 7, 16, 16, 21, 11, 14, 14, 18, 14, 18, 18, 22, 2, 6, 6, 13,
    12, 15, 15, 20, 3, 7, 7, 14, 19, 20, 20, 25, 6, 8, 15, 17, 15, 17,
    26, 27, 7, 9, 16, 18, 20, 21, 27, 28, 6, 15, 8, 17, 15, 26, 17, 27,
    7, 16, 9, 18, 20, 27, 21, 28, 13, 17, 17, 23, 20, 27, 27, 29, 14, 18,
    18, 24, 25, 28, 28, 30, 2, 6, 12, 15, 6, 13, 15, 20, 6, 8, 15, 17,
    15, 17, 26, 27, 3, 7, 19, 20, 7, 14, 20, 25, 7, 9, 20, 21, 16, 18,
    27, 28, 6, 15, 15, 26, 8, 17, 17, 27, 13, 17, 20, 27, 17, 23, 27, 29,
    7, 16, 20, 27, 9, 18, 21, 28, 14, 18, 25, 28, 18, 24, 28, 30, 4, 7,
    13, 16, 13, 16, 17, 21, 7, 9, 16, 18, 20, 21, 27, 28, 7, 9, 20, 21,
    16, 18, 27, 28, 9, 10, 21, 22, 21, 22, 29, 30, 13, 20, 17, 27, 17,
    27, 23, 29, 16, 21, 21, 28, 27, 29, 29, 31, 16, 21, 27, 29, 21, 28,
    29, 31, 18, 22, 28, 30, 28, 30, 31, 32, 2, 12, 6, 15, 6, 15, 13, 20,
    6, 15, 8, 17, 15, 26, 17, 27, 6, 15, 15, 26, 8, 17, 17, 27, 13, 20,
    17, 27, 17, 27, 23, 29, 3, 19, 7, 20, 7, 20, 14, 25, 7, 20, 9, 21,
    16, 27, 18, 28, 7, 20, 16, 27, 9, 21, 18, 28, 14, 25, 18, 28, 18, 28,
    24, 30, 4, 13, 7, 16, 13, 17, 16, 21, 7, 16, 9, 18, 20, 27, 21, 28,
    13, 17, 20, 27, 17, 23, 27, 29, 16, 21, 21, 28, 27, 29, 29, 31, 7,
    20, 9, 21, 16, 27, 18, 28, 9, 21, 10, 22, 21, 29, 22, 30, 16, 27, 21,
    29, 21, 29, 28, 31, 18, 28, 22, 30, 28, 31, 30, 32, 4, 13, 13, 17, 7,
    16, 16, 21, 13, 17, 17, 23, 20, 27, 27, 29, 7, 16, 20, 27, 9, 18, 21,
    28, 16, 21, 27, 29, 21, 28, 29, 31, 7, 20, 16, 27, 9, 21, 18, 28, 16,
    27, 21, 29, 21, 29, 28, 31, 9, 21, 21, 29, 10, 22, 22, 30, 18, 28,
    28, 31, 22, 30, 30, 32, 11, 14, 14, 18, 14, 18, 18, 22, 14, 18, 18,
    24, 25, 28, 28, 30, 14, 18, 25, 28, 18, 24, 28, 30, 18, 22, 28, 30,
    28, 30, 31, 32, 14, 25, 18, 28, 18, 28, 24, 30, 18, 28, 22, 30, 28,
    31, 30, 32, 18, 28, 28, 31, 22, 30, 30, 32, 24, 30, 30, 32, 30, 32,
    32, 33
};

const unsigned int igraph_i_isoclass2_6u[] = {
    0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 4, 5, 6, 6, 7, 1, 2, 5, 6, 2, 4, 6,
    7, 2, 3, 6, 7, 6, 7, 8, 9, 1, 5, 2, 6, 2, 6, 4, 7, 2, 6, 3, 7, 6, 8,
    7, 9, 2, 6, 6, 8, 3, 7, 7, 9, 4, 7, 7, 9, 7, 9, 9, 10, 1, 2, 2, 4, 5,
    6, 6, 7, 2, 4, 4, 11, 12, 13, 13, 14, 5, 6, 12, 13, 12, 13, 15, 16,
    6, 7, 13, 14, 15, 16, 17, 18, 5, 12, 6, 13, 12, 15, 13, 16, 6, 13, 7,
    14, 15, 17, 16, 18, 12, 15, 15, 17, 19, 20, 20, 21, 13, 16, 16, 18,
    20, 21, 21, 22, 1, 2, 5, 6, 2, 4, 6, 7, 5, 6, 12, 13, 12, 13, 15, 16,
    2, 4, 12, 13, 4, 11, 13, 14, 6, 7, 15, 16, 13, 14, 17, 18, 5, 12, 12,
    15, 6, 13, 13, 16, 12, 15, 19, 20, 15, 17, 20, 21, 6, 13, 15, 17, 7,
    14, 16, 18, 13, 16, 20, 21, 16, 18, 21, 22, 2, 3, 6, 7, 6, 7, 8, 9,
    6, 7, 13, 14, 15, 16, 17, 18, 6, 7, 15, 16, 13, 14, 17, 18, 8, 9, 17,
    18, 17, 18, 23, 24, 12, 19, 15, 20, 15, 20, 17, 21, 15, 20, 20, 25,
    26, 27, 27, 28, 15, 20, 26, 27, 20, 25, 27, 28, 17, 21, 27, 28, 27,
    28, 29, 30, 1, 5, 2, 6, 2, 6, 4, 7, 5, 12, 6, 13, 12, 15, 13, 16, 5,
    12, 12, 15, 6, 13, 13, 16, 12, 19, 15, 20, 15, 20, 17, 21, 2, 12, 4,
    13, 4, 13, 11, 14, 6, 15, 7, 16, 13, 17, 14, 18, 6, 15, 13, 17, 7,
    16, 14, 18, 13, 20, 16, 21, 16, 21, 18, 22, 2, 6, 3, 7, 6, 8, 7, 9,
    6, 13, 7, 14, 15, 17, 16, 18, 12, 15, 19, 20, 15, 17, 20, 21, 15, 20,
    20, 25, 26, 27, 27, 28, 6, 15, 7, 16, 13, 17, 14, 18, 8, 17, 9, 18,
    17, 23, 18, 24, 15, 26, 20, 27, 20, 27, 25, 28, 17, 27, 21, 28, 27,
    29, 28, 30, 2, 6, 6, 8, 3, 7, 7, 9, 12, 15, 15, 17, 19, 20, 20, 21,
    6, 13, 15, 17, 7, 14, 16, 18, 15, 20, 26, 27, 20, 25, 27, 28, 6, 15,
    13, 17, 7, 16, 14, 18, 15, 26, 20, 27, 20, 27, 25, 28, 8, 17, 17, 23,
    9, 18, 18, 24, 17, 27, 27, 29, 21, 28, 28, 30, 4, 7, 7, 9, 7, 9, 9,
    10, 13, 16, 16, 18, 20, 21, 21, 22, 13, 16, 20, 21, 16, 18, 21, 22,
    17, 21, 27, 28, 27, 28, 29, 30, 13, 20, 16, 21, 16, 21, 18, 22, 17,
    27, 21, 28, 27, 29, 28, 30, 17, 27, 27, 29, 21, 28, 28, 30, 23, 29,
    29, 31, 29, 31, 31, 32, 1, 5, 5, 12, 5, 12, 12, 19, 2, 6, 6, 13, 12,
    15, 15, 20, 2, 6, 12, 15, 6, 13, 15, 20, 4, 7, 13, 16, 13, 16, 17,
    21, 2, 12, 6, 15, 6, 15, 13, 20, 4, 13, 7, 16, 13, 17, 16, 21, 4, 13,
    13, 17, 7, 16, 16, 21, 11, 14, 14, 18, 14, 18, 18, 22, 2, 6, 6, 13,
    12, 15, 15, 20, 3, 7, 7, 14, 19, 20, 20, 25, 6, 8, 15, 17, 15, 17,
    26, 27, 7, 9, 16, 18, 20, 21, 27, 28, 6, 15, 8, 17, 15, 26, 17, 27,
    7, 16, 9, 18, 20, 27, 21, 28, 13, 17, 17, 23, 20, 27, 27, 29, 14, 18,
    18, 24, 25, 28, 28, 30, 2, 6, 12, 15, 6, 13, 15, 20, 6, 8, 15, 17,
    15, 17, 26, 27, 3, 7, 19, 20, 7, 14, 20, 25, 7, 9, 20, 21, 16, 18,
    27, 28, 6, 15, 15, 26, 8, 17, 17, 27, 13, 17, 20, 27, 17, 23, 27, 29,
    7, 16, 20, 27, 9, 18, 21, 28, 14, 18, 25, 28, 18, 24, 28, 30, 4, 7,
    13, 16, 13, 16, 17, 21, 7, 9, 16, 18, 20, 21, 27, 28, 7, 9, 20, 21,
    16, 18, 27, 28, 9, 10, 21, 22, 21, 22, 29, 30, 13, 20, 17, 27, 17,
    27, 23, 29, 16, 21, 21, 28, 27, 29, 29, 31, 16, 21, 27, 29, 21, 28,
    29, 31, 18, 22, 28, 30, 28, 30, 31, 32, 2, 12, 6, 15, 6, 15, 13, 20,
    6, 15, 8, 17, 15, 26, 17, 27, 6, 15, 15, 26, 8, 17, 17, 27, 13, 20,
    17, 27, 17, 27, 23, 29, 3, 19, 7, 20, 7, 20, 14, 25, 7, 20, 9, 21,
    16, 27, 18, 28, 7, 20, 16, 27, 9, 21, 18, 28, 14, 25, 18, 28, 18, 28,
    24, 30, 4, 13, 7, 16, 13, 17, 16, 21, 7, 16, 9, 18, 20, 27, 21, 28,
    13, 17, 20, 27, 17, 23, 27, 29, 16, 21, 21, 28, 27, 29, 29, 31, 7,
    20, 9, 21, 16, 27, 18, 28, 9, 21, 10, 22, 21, 29, 22, 30, 16, 27, 21,
    29, 21, 29, 28, 31, 18, 28, 22, 30, 28, 31, 30, 32, 4, 13, 13, 17, 7,
    16, 16, 21, 13, 17, 17, 23, 20, 27, 27, 29, 7, 16, 20, 27, 9, 18, 21,
    28, 16, 21, 27, 29, 21, 28, 29, 31, 7, 20, 16, 27, 9, 21, 18, 28, 16,
    27, 21, 29, 21, 29, 28, 31, 9, 21, 21, 29, 10, 22, 22, 30, 18, 28,
    28, 31, 22, 30, 30, 32, 11, 14, 14, 18, 14, 18, 18, 22, 14, 18, 18,
    24, 25, 28, 28, 30, 14, 18, 25, 28, 18, 24, 28, 30, 18, 22, 28, 30,
    28, 30, 31, 32, 14, 25, 18, 28, 18, 28, 24, 30, 18, 28, 22, 30, 28,
    31, 30, 32, 18, 28, 28, 31, 22, 30, 30, 32, 24, 30, 30, 32, 30, 32,
    32, 33, 1, 2, 2, 4, 5, 6, 6, 7, 2, 4, 4, 11, 12, 13, 13, 14, 5, 6,
    12, 13, 12, 13, 15, 16, 6, 7, 13, 14, 15, 16, 17, 18, 5, 12, 6, 13,
    12, 15, 13, 16, 6, 13, 7, 14, 15, 17, 16, 18, 12, 15, 15, 17, 19, 20,
    20, 21, 13, 16, 16, 18, 20, 21, 21, 22, 2, 4, 4, 11, 12, 13, 13, 14,
    4, 11, 11, 34, 35, 36, 36, 37, 12, 13, 35, 36, 38, 39, 40, 41, 13,
    14, 36, 37, 40, 41, 42, 43, 12, 35, 13, 36, 38, 40, 39, 41, 13, 36,
    14, 37, 40, 42, 41, 43, 38, 40, 40, 42, 44, 45, 45, 46, 39, 41, 41,
    43, 45, 46, 46, 47, 5, 6, 12, 13, 12, 13, 15, 16, 12, 13, 35, 36, 38,
    39, 40, 41, 12, 13, 38, 39, 35, 36, 40, 41, 15, 16, 40, 41, 40, 41,
    48, 49, 50, 51, 51, 52, 51, 52, 52, 53, 51, 52, 54, 55, 56, 57, 58,
    59, 51, 52, 56, 57, 54, 55, 58, 59, 52, 53, 58, 59, 58, 59, 60, 61,
    6, 7, 13, 14, 15, 16, 17, 18, 13, 14, 36, 37, 40, 41, 42, 43, 15, 16,
    40, 41, 40, 41, 48, 49, 17, 18, 42, 43, 48, 49, 62, 63, 51, 54, 52,
    55, 56, 58, 57, 59, 52, 55, 55, 64, 65, 66, 66, 67, 56, 58, 65, 66,
    68, 69, 70, 71, 57, 59, 66, 67, 70, 71, 72, 73, 5, 12, 6, 13, 12, 15,
    13, 16, 12, 35, 13, 36, 38, 40, 39, 41, 50, 51, 51, 52, 51, 52, 52,
    53, 51, 54, 52, 55, 56, 58, 57, 59, 12, 38, 13, 39, 35, 40, 36, 41,
    15, 40, 16, 41, 40, 48, 41, 49, 51, 56, 52, 57, 54, 58, 55, 59, 52,
    58, 53, 59, 58, 60, 59, 61, 6, 13, 7, 14, 15, 17, 16, 18, 13, 36, 14,
    37, 40, 42, 41, 43, 51, 52, 54, 55, 56, 57, 58, 59, 52, 55, 55, 64,
    65, 66, 66, 67, 15, 40, 16, 41, 40, 48, 41, 49, 17, 42, 18, 43, 48,
    62, 49, 63, 56, 65, 58, 66, 68, 70, 69, 71, 57, 66, 59, 67, 70, 72,
    71, 73, 12, 15, 15, 17, 19, 20, 20, 21, 38, 40, 40, 42, 44, 45, 45,
    46, 51, 52, 56, 57, 54, 55, 58, 59, 56, 58, 65, 66, 68, 69, 70, 71,
    51, 56, 52, 57, 54, 58, 55, 59, 56, 65, 58, 66, 68, 70, 69, 71, 74,
    75, 75, 76, 77, 78, 78, 79, 75, 80, 80, 81, 82, 83, 83, 84, 13, 16,
    16, 18, 20, 21, 21, 22, 39, 41, 41, 43, 45, 46, 46, 47, 52, 53, 58,
    59, 58, 59, 60, 61, 57, 59, 66, 67, 70, 71, 72, 73, 52, 58, 53, 59,
    58, 60, 59, 61, 57, 66, 59, 67, 70, 72, 71, 73, 75, 80, 80, 81, 82,
    83, 83, 84, 76, 81, 81, 85, 86, 87, 87, 88, 5, 12, 12, 35, 50, 51,
    51, 54, 6, 13, 13, 36, 51, 52, 52, 55, 12, 15, 38, 40, 51, 52, 56,
    58, 13, 16, 39, 41, 52, 53, 57, 59, 12, 38, 15, 40, 51, 56, 52, 58,
    13, 39, 16, 41, 52, 57, 53, 59, 35, 40, 40, 48, 54, 58, 58, 60, 36,
    41, 41, 49, 55, 59, 59, 61, 6, 13, 13, 36, 51, 52, 52, 55, 7, 14, 14,
    37, 54, 55, 55, 64, 15, 17, 40, 42, 56, 57, 65, 66, 16, 18, 41, 43,
    58, 59, 66, 67, 15, 40, 17, 42, 56, 65, 57, 66, 16, 41, 18, 43, 58,
    66, 59, 67, 40, 48, 48, 62, 68, 70, 70, 72, 41, 49, 49, 63, 69, 71,
    71, 73, 12, 15, 38, 40, 51, 52, 56, 58, 15, 17, 40, 42, 56, 57, 65,
    66, 19, 20, 44, 45, 54, 55, 68, 69, 20, 21, 45, 46, 58, 59, 70, 71,
    51, 56, 56, 65, 74, 75, 75, 80, 52, 57, 58, 66, 75, 76, 80, 81, 54,
    58, 68, 70, 77, 78, 82, 83, 55, 59, 69, 71, 78, 79, 83, 84, 13, 16,
    39, 41, 52, 53, 57, 59, 16, 18, 41, 43, 58, 59, 66, 67, 20, 21, 45,
    46, 58, 59, 70, 71, 21, 22, 46, 47, 60, 61, 72, 73, 52, 58, 57, 66,
    75, 80, 76, 81, 53, 59, 59, 67, 80, 81, 81, 85, 58, 60, 70, 72, 82,
    83, 86, 87, 59, 61, 71, 73, 83, 84, 87, 88, 12, 38, 15, 40, 51, 56,
    52, 58, 15, 40, 17, 42, 56, 65, 57, 66, 51, 56, 56, 65, 74, 75, 75,
    80, 52, 58, 57, 66, 75, 80, 76, 81, 19, 44, 20, 45, 54, 68, 55, 69,
    20, 45, 21, 46, 58, 70, 59, 71, 54, 68, 58, 70, 77, 82, 78, 83, 55,
    69, 59, 71, 78, 83, 79, 84, 13, 39, 16, 41, 52, 57, 53, 59, 16, 41,
    18, 43, 58, 66, 59, 67, 52, 57, 58, 66, 75, 76, 80, 81, 53, 59, 59,
    67, 80, 81, 81, 85, 20, 45, 21, 46, 58, 70, 59, 71, 21, 46, 22, 47,
    60, 72, 61, 73, 58, 70, 60, 72, 82, 86, 83, 87, 59, 71, 61, 73, 83,
    87, 84, 88, 35, 40, 40, 48, 54, 58, 58, 60, 40, 48, 48, 62, 68, 70,
    70, 72, 54, 58, 68, 70, 77, 78, 82, 83, 58, 60, 70, 72, 82, 83, 86,
    87, 54, 68, 58, 70, 77, 82, 78, 83, 58, 70, 60, 72, 82, 86, 83, 87,
    77, 82, 82, 86, 89, 90, 90, 91, 78, 83, 83, 87, 90, 91, 91, 92, 36,
    41, 41, 49, 55, 59, 59, 61, 41, 49, 49, 63, 69, 71, 71, 73, 55, 59,
    69, 71, 78, 79, 83, 84, 59, 61, 71, 73, 83, 84, 87, 88, 55, 69, 59,
    71, 78, 83, 79, 84, 59, 71, 61, 73, 83, 87, 84, 88, 78, 83, 83, 87,
    90, 91, 91, 92, 79, 84, 84, 88, 91, 92, 92, 93, 1, 2, 5, 6, 2, 4, 6,
    7, 5, 6, 12, 13, 12, 13, 15, 16, 2, 4, 12, 13, 4, 11, 13, 14, 6, 7,
    15, 16, 13, 14, 17, 18, 5, 12, 12, 15, 6, 13, 13, 16, 12, 15, 19, 20,
    15, 17, 20, 21, 6, 13, 15, 17, 7, 14, 16, 18, 13, 16, 20, 21, 16, 18,
    21, 22, 5, 6, 12, 13, 12, 13, 15, 16, 12, 13, 35, 36, 38, 39, 40, 41,
    12, 13, 38, 39, 35, 36, 40, 41, 15, 16, 40, 41, 40, 41, 48, 49, 50,
    51, 51, 52, 51, 52, 52, 53, 51, 52, 54, 55, 56, 57, 58, 59, 51, 52,
    56, 57, 54, 55, 58, 59, 52, 53, 58, 59, 58, 59, 60, 61, 2, 4, 12, 13,
    4, 11, 13, 14, 12, 13, 38, 39, 35, 36, 40, 41, 4, 11, 35, 36, 11, 34,
    36, 37, 13, 14, 40, 41, 36, 37, 42, 43, 12, 35, 38, 40, 13, 36, 39,
    41, 38, 40, 44, 45, 40, 42, 45, 46, 13, 36, 40, 42, 14, 37, 41, 43,
    39, 41, 45, 46, 41, 43, 46, 47, 6, 7, 15, 16, 13, 14, 17, 18, 15, 16,
    40, 41, 40, 41, 48, 49, 13, 14, 40, 41, 36, 37, 42, 43, 17, 18, 48,
    49, 42, 43, 62, 63, 51, 54, 56, 58, 52, 55, 57, 59, 56, 58, 68, 69,
    65, 66, 70, 71, 52, 55, 65, 66, 55, 64, 66, 67, 57, 59, 70, 71, 66,
    67, 72, 73, 5, 12, 12, 15, 6, 13, 13, 16, 50, 51, 51, 52, 51, 52, 52,
    53, 12, 35, 38, 40, 13, 36, 39, 41, 51, 54, 56, 58, 52, 55, 57, 59,
    12, 38, 35, 40, 13, 39, 36, 41, 51, 56, 54, 58, 52, 57, 55, 59, 15,
    40, 40, 48, 16, 41, 41, 49, 52, 58, 58, 60, 53, 59, 59, 61, 12, 15,
    19, 20, 15, 17, 20, 21, 51, 52, 54, 55, 56, 57, 58, 59, 38, 40, 44,
    45, 40, 42, 45, 46, 56, 58, 68, 69, 65, 66, 70, 71, 51, 56, 54, 58,
    52, 57, 55, 59, 74, 75, 77, 78, 75, 76, 78, 79, 56, 65, 68, 70, 58,
    66, 69, 71, 75, 80, 82, 83, 80, 81, 83, 84, 6, 13, 15, 17, 7, 14, 16,
    18, 51, 52, 56, 57, 54, 55, 58, 59, 13, 36, 40, 42, 14, 37, 41, 43,
    52, 55, 65, 66, 55, 64, 66, 67, 15, 40, 40, 48, 16, 41, 41, 49, 56,
    65, 68, 70, 58, 66, 69, 71, 17, 42, 48, 62, 18, 43, 49, 63, 57, 66,
    70, 72, 59, 67, 71, 73, 13, 16, 20, 21, 16, 18, 21, 22, 52, 53, 58,
    59, 58, 59, 60, 61, 39, 41, 45, 46, 41, 43, 46, 47, 57, 59, 70, 71,
    66, 67, 72, 73, 52, 58, 58, 60, 53, 59, 59, 61, 75, 80, 82, 83, 80,
    81, 83, 84, 57, 66, 70, 72, 59, 67, 71, 73, 76, 81, 86, 87, 81, 85,
    87, 88, 5, 12, 50, 51, 12, 35, 51, 54, 12, 15, 51, 52, 38, 40, 56,
    58, 6, 13, 51, 52, 13, 36, 52, 55, 13, 16, 52, 53, 39, 41, 57, 59,
    12, 38, 51, 56, 15, 40, 52, 58, 35, 40, 54, 58, 40, 48, 58, 60, 13,
    39, 52, 57, 16, 41, 53, 59, 36, 41, 55, 59, 41, 49, 59, 61, 12, 15,
    51, 52, 38, 40, 56, 58, 19, 20, 54, 55, 44, 45, 68, 69, 15, 17, 56,
    57, 40, 42, 65, 66, 20, 21, 58, 59, 45, 46, 70, 71, 51, 56, 74, 75,
    56, 65, 75, 80, 54, 58, 77, 78, 68, 70, 82, 83, 52, 57, 75, 76, 58,
    66, 80, 81, 55, 59, 78, 79, 69, 71, 83, 84, 6, 13, 51, 52, 13, 36,
    52, 55, 15, 17, 56, 57, 40, 42, 65, 66, 7, 14, 54, 55, 14, 37, 55,
    64, 16, 18, 58, 59, 41, 43, 66, 67, 15, 40, 56, 65, 17, 42, 57, 66,
    40, 48, 68, 70, 48, 62, 70, 72, 16, 41, 58, 66, 18, 43, 59, 67, 41,
    49, 69, 71, 49, 63, 71, 73, 13, 16, 52, 53, 39, 41, 57, 59, 20, 21,
    58, 59, 45, 46, 70, 71, 16, 18, 58, 59, 41, 43, 66, 67, 21, 22, 60,
    61, 46, 47, 72, 73, 52, 58, 75, 80, 57, 66, 76, 81, 58, 60, 82, 83,
    70, 72, 86, 87, 53, 59, 80, 81, 59, 67, 81, 85, 59, 61, 83, 84, 71,
    73, 87, 88, 12, 38, 51, 56, 15, 40, 52, 58, 51, 56, 74, 75, 56, 65,
    75, 80, 15, 40, 56, 65, 17, 42, 57, 66, 52, 58, 75, 80, 57, 66, 76,
    81, 19, 44, 54, 68, 20, 45, 55, 69, 54, 68, 77, 82, 58, 70, 78, 83,
    20, 45, 58, 70, 21, 46, 59, 71, 55, 69, 78, 83, 59, 71, 79, 84, 35,
    40, 54, 58, 40, 48, 58, 60, 54, 58, 77, 78, 68, 70, 82, 83, 40, 48,
    68, 70, 48, 62, 70, 72, 58, 60, 82, 83, 70, 72, 86, 87, 54, 68, 77,
    82, 58, 70, 78, 83, 77, 82, 89, 90, 82, 86, 90, 91, 58, 70, 82, 86,
    60, 72, 83, 87, 78, 83, 90, 91, 83, 87, 91, 92, 13, 39, 52, 57, 16,
    41, 53, 59, 52, 57, 75, 76, 58, 66, 80, 81, 16, 41, 58, 66, 18, 43,
    59, 67, 53, 59, 80, 81, 59, 67, 81, 85, 20, 45, 58, 70, 21, 46, 59,
    71, 58, 70, 82, 86, 60, 72, 83, 87, 21, 46, 60, 72, 22, 47, 61, 73,
    59, 71, 83, 87, 61, 73, 84, 88, 36, 41, 55, 59, 41, 49, 59, 61, 55,
    59, 78, 79, 69, 71, 83, 84, 41, 49, 69, 71, 49, 63, 71, 73, 59, 61,
    83, 84, 71, 73, 87, 88, 55, 69, 78, 83, 59, 71, 79, 84, 78, 83, 90,
    91, 83, 87, 91, 92, 59, 71, 83, 87, 61, 73, 84, 88, 79, 84, 91, 92,
    84, 88, 92, 93, 2, 3, 6, 7, 6, 7, 8, 9, 6, 7, 13, 14, 15, 16, 17, 18,
    6, 7, 15, 16, 13, 14, 17, 18, 8, 9, 17, 18, 17, 18, 23, 24, 12, 19,
    15, 20, 15, 20, 17, 21, 15, 20, 20, 25, 26, 27, 27, 28, 15, 20, 26,
    27, 20, 25, 27, 28, 17, 21, 27, 28, 27, 28, 29, 30, 6, 7, 13, 14, 15,
    16, 17, 18, 13, 14, 36, 37, 40, 41, 42, 43, 15, 16, 40, 41, 40, 41,
    48, 49, 17, 18, 42, 43, 48, 49, 62, 63, 51, 54, 52, 55, 56, 58, 57,
    59, 52, 55, 55, 64, 65, 66, 66, 67, 56, 58, 65, 66, 68, 69, 70, 71,
    57, 59, 66, 67, 70, 71, 72, 73, 6, 7, 15, 16, 13, 14, 17, 18, 15, 16,
    40, 41, 40, 41, 48, 49, 13, 14, 40, 41, 36, 37, 42, 43, 17, 18, 48,
    49, 42, 43, 62, 63, 51, 54, 56, 58, 52, 55, 57, 59, 56, 58, 68, 69,
    65, 66, 70, 71, 52, 55, 65, 66, 55, 64, 66, 67, 57, 59, 70, 71, 66,
    67, 72, 73, 8, 9, 17, 18, 17, 18, 23, 24, 17, 18, 42, 43, 48, 49, 62,
    63, 17, 18, 48, 49, 42, 43, 62, 63, 23, 24, 62, 63, 62, 63, 94, 95,
    74, 77, 75, 78, 75, 78, 76, 79, 75, 78, 96, 97, 98, 99, 100, 101, 75,
    78, 98, 99, 96, 97, 100, 101, 76, 79, 100, 101, 100, 101, 102, 103,
    12, 19, 15, 20, 15, 20, 17, 21, 51, 54, 52, 55, 56, 58, 57, 59, 51,
    54, 56, 58, 52, 55, 57, 59, 74, 77, 75, 78, 75, 78, 76, 79, 38, 44,
    40, 45, 40, 45, 42, 46, 56, 68, 58, 69, 65, 70, 66, 71, 56, 68, 65,
    70, 58, 69, 66, 71, 75, 82, 80, 83, 80, 83, 81, 84, 15, 20, 20, 25,
    26, 27, 27, 28, 52, 55, 55, 64, 65, 66, 66, 67, 56, 58, 68, 69, 65,
    66, 70, 71, 75, 78, 96, 97, 98, 99, 100, 101, 56, 68, 58, 69, 65, 70,
    66, 71, 75, 96, 78, 97, 98, 100, 99, 101, 104, 105, 105, 106, 105,
    106, 106, 107, 108, 109, 109, 110, 111, 112, 112, 113, 15, 20, 26,
    27, 20, 25, 27, 28, 56, 58, 65, 66, 68, 69, 70, 71, 52, 55, 65, 66,
    55, 64, 66, 67, 75, 78, 98, 99, 96, 97, 100, 101, 56, 68, 65, 70, 58,
    69, 66, 71, 104, 105, 105, 106, 105, 106, 106, 107, 75, 96, 98, 100,
    78, 97, 99, 101, 108, 109, 111, 112, 109, 110, 112, 113, 17, 21, 27,
    28, 27, 28, 29, 30, 57, 59, 66, 67, 70, 71, 72, 73, 57, 59, 70, 71,
    66, 67, 72, 73, 76, 79, 100, 101, 100, 101, 102, 103, 75, 82, 80, 83,
    80, 83, 81, 84, 108, 109, 109, 110, 111, 112, 112, 113, 108, 109,
    111, 112, 109, 110, 112, 113, 114, 115, 116, 117, 116, 117, 118, 119,
    12, 19, 51, 54, 51, 54, 74, 77, 15, 20, 52, 55, 56, 58, 75, 78, 15,
    20, 56, 58, 52, 55, 75, 78, 17, 21, 57, 59, 57, 59, 76, 79, 38, 44,
    56, 68, 56, 68, 75, 82, 40, 45, 58, 69, 65, 70, 80, 83, 40, 45, 65,
    70, 58, 69, 80, 83, 42, 46, 66, 71, 66, 71, 81, 84, 15, 20, 52, 55,
    56, 58, 75, 78, 20, 25, 55, 64, 68, 69, 96, 97, 26, 27, 65, 66, 65,
    66, 98, 99, 27, 28, 66, 67, 70, 71, 100, 101, 56, 68, 75, 96, 104,
    105, 108, 109, 58, 69, 78, 97, 105, 106, 109, 110, 65, 70, 98, 100,
    105, 106, 111, 112, 66, 71, 99, 101, 106, 107, 112, 113, 15, 20, 56,
    58, 52, 55, 75, 78, 26, 27, 65, 66, 65, 66, 98, 99, 20, 25, 68, 69,
    55, 64, 96, 97, 27, 28, 70, 71, 66, 67, 100, 101, 56, 68, 104, 105,
    75, 96, 108, 109, 65, 70, 105, 106, 98, 100, 111, 112, 58, 69, 105,
    106, 78, 97, 109, 110, 66, 71, 106, 107, 99, 101, 112, 113, 17, 21,
    57, 59, 57, 59, 76, 79, 27, 28, 66, 67, 70, 71, 100, 101, 27, 28, 70,
    71, 66, 67, 100, 101, 29, 30, 72, 73, 72, 73, 102, 103, 75, 82, 108,
    109, 108, 109, 114, 115, 80, 83, 109, 110, 111, 112, 116, 117, 80,
    83, 111, 112, 109, 110, 116, 117, 81, 84, 112, 113, 112, 113, 118,
    119, 38, 44, 56, 68, 56, 68, 75, 82, 56, 68, 75, 96, 104, 105, 108,
    109, 56, 68, 104, 105, 75, 96, 108, 109, 75, 82, 108, 109, 108, 109,
    114, 115, 44, 120, 68, 121, 68, 121, 96, 122, 68, 121, 82, 122, 105,
    123, 109, 124, 68, 121, 105, 123, 82, 122, 109, 124, 96, 122, 109,
    124, 109, 124, 115, 125, 40, 45, 58, 69, 65, 70, 80, 83, 58, 69, 78,
    97, 105, 106, 109, 110, 65, 70, 105, 106, 98, 100, 111, 112, 80, 83,
    109, 110, 111, 112, 116, 117, 68, 121, 82, 122, 105, 123, 109, 124,
    82, 122, 90, 126, 127, 128, 129, 130, 105, 123, 127, 128, 127, 128,
    131, 132, 109, 124, 129, 130, 131, 132, 133, 134, 40, 45, 65, 70, 58,
    69, 80, 83, 65, 70, 98, 100, 105, 106, 111, 112, 58, 69, 105, 106,
    78, 97, 109, 110, 80, 83, 111, 112, 109, 110, 116, 117, 68, 121, 105,
    123, 82, 122, 109, 124, 105, 123, 127, 128, 127, 128, 131, 132, 82,
    122, 127, 128, 90, 126, 129, 130, 109, 124, 131, 132, 129, 130, 133,
    134, 42, 46, 66, 71, 66, 71, 81, 84, 66, 71, 99, 101, 106, 107, 112,
    113, 66, 71, 106, 107, 99, 101, 112, 113, 81, 84, 112, 113, 112, 113,
    118, 119, 96, 122, 109, 124, 109, 124, 115, 125, 109, 124, 129, 130,
    131, 132, 133, 134, 109, 124, 131, 132, 129, 130, 133, 134, 115, 125,
    133, 134, 133, 134, 135, 136, 1, 5, 2, 6, 2, 6, 4, 7, 5, 12, 6, 13,
    12, 15, 13, 16, 5, 12, 12, 15, 6, 13, 13, 16, 12, 19, 15, 20, 15, 20,
    17, 21, 2, 12, 4, 13, 4, 13, 11, 14, 6, 15, 7, 16, 13, 17, 14, 18, 6,
    15, 13, 17, 7, 16, 14, 18, 13, 20, 16, 21, 16, 21, 18, 22, 5, 12, 6,
    13, 12, 15, 13, 16, 12, 35, 13, 36, 38, 40, 39, 41, 50, 51, 51, 52,
    51, 52, 52, 53, 51, 54, 52, 55, 56, 58, 57, 59, 12, 38, 13, 39, 35,
    40, 36, 41, 15, 40, 16, 41, 40, 48, 41, 49, 51, 56, 52, 57, 54, 58,
    55, 59, 52, 58, 53, 59, 58, 60, 59, 61, 5, 12, 12, 15, 6, 13, 13, 16,
    50, 51, 51, 52, 51, 52, 52, 53, 12, 35, 38, 40, 13, 36, 39, 41, 51,
    54, 56, 58, 52, 55, 57, 59, 12, 38, 35, 40, 13, 39, 36, 41, 51, 56,
    54, 58, 52, 57, 55, 59, 15, 40, 40, 48, 16, 41, 41, 49, 52, 58, 58,
    60, 53, 59, 59, 61, 12, 19, 15, 20, 15, 20, 17, 21, 51, 54, 52, 55,
    56, 58, 57, 59, 51, 54, 56, 58, 52, 55, 57, 59, 74, 77, 75, 78, 75,
    78, 76, 79, 38, 44, 40, 45, 40, 45, 42, 46, 56, 68, 58, 69, 65, 70,
    66, 71, 56, 68, 65, 70, 58, 69, 66, 71, 75, 82, 80, 83, 80, 83, 81,
    84, 2, 12, 4, 13, 4, 13, 11, 14, 12, 38, 13, 39, 35, 40, 36, 41, 12,
    38, 35, 40, 13, 39, 36, 41, 38, 44, 40, 45, 40, 45, 42, 46, 4, 35,
    11, 36, 11, 36, 34, 37, 13, 40, 14, 41, 36, 42, 37, 43, 13, 40, 36,
    42, 14, 41, 37, 43, 39, 45, 41, 46, 41, 46, 43, 47, 6, 15, 7, 16, 13,
    17, 14, 18, 15, 40, 16, 41, 40, 48, 41, 49, 51, 56, 54, 58, 52, 57,
    55, 59, 56, 68, 58, 69, 65, 70, 66, 71, 13, 40, 14, 41, 36, 42, 37,
    43, 17, 48, 18, 49, 42, 62, 43, 63, 52, 65, 55, 66, 55, 66, 64, 67,
    57, 70, 59, 71, 66, 72, 67, 73, 6, 15, 13, 17, 7, 16, 14, 18, 51, 56,
    52, 57, 54, 58, 55, 59, 15, 40, 40, 48, 16, 41, 41, 49, 56, 68, 65,
    70, 58, 69, 66, 71, 13, 40, 36, 42, 14, 41, 37, 43, 52, 65, 55, 66,
    55, 66, 64, 67, 17, 48, 42, 62, 18, 49, 43, 63, 57, 70, 66, 72, 59,
    71, 67, 73, 13, 20, 16, 21, 16, 21, 18, 22, 52, 58, 53, 59, 58, 60,
    59, 61, 52, 58, 58, 60, 53, 59, 59, 61, 75, 82, 80, 83, 80, 83, 81,
    84, 39, 45, 41, 46, 41, 46, 43, 47, 57, 70, 59, 71, 66, 72, 67, 73,
    57, 70, 66, 72, 59, 71, 67, 73, 76, 86, 81, 87, 81, 87, 85, 88, 5,
    50, 12, 51, 12, 51, 35, 54, 12, 51, 15, 52, 38, 56, 40, 58, 12, 51,
    38, 56, 15, 52, 40, 58, 35, 54, 40, 58, 40, 58, 48, 60, 6, 51, 13,
    52, 13, 52, 36, 55, 13, 52, 16, 53, 39, 57, 41, 59, 13, 52, 39, 57,
    16, 53, 41, 59, 36, 55, 41, 59, 41, 59, 49, 61, 12, 51, 15, 52, 38,
    56, 40, 58, 19, 54, 20, 55, 44, 68, 45, 69, 51, 74, 56, 75, 56, 75,
    65, 80, 54, 77, 58, 78, 68, 82, 70, 83, 15, 56, 17, 57, 40, 65, 42,
    66, 20, 58, 21, 59, 45, 70, 46, 71, 52, 75, 57, 76, 58, 80, 66, 81,
    55, 78, 59, 79, 69, 83, 71, 84, 12, 51, 38, 56, 15, 52, 40, 58, 51,
    74, 56, 75, 56, 75, 65, 80, 19, 54, 44, 68, 20, 55, 45, 69, 54, 77,
    68, 82, 58, 78, 70, 83, 15, 56, 40, 65, 17, 57, 42, 66, 52, 75, 58,
    80, 57, 76, 66, 81, 20, 58, 45, 70, 21, 59, 46, 71, 55, 78, 69, 83,
    59, 79, 71, 84, 35, 54, 40, 58, 40, 58, 48, 60, 54, 77, 58, 78, 68,
    82, 70, 83, 54, 77, 68, 82, 58, 78, 70, 83, 77, 89, 82, 90, 82, 90,
    86, 91, 40, 68, 48, 70, 48, 70, 62, 72, 58, 82, 60, 83, 70, 86, 72,
    87, 58, 82, 70, 86, 60, 83, 72, 87, 78, 90, 83, 91, 83, 91, 87, 92,
    6, 51, 13, 52, 13, 52, 36, 55, 15, 56, 17, 57, 40, 65, 42, 66, 15,
    56, 40, 65, 17, 57, 42, 66, 40, 68, 48, 70, 48, 70, 62, 72, 7, 54,
    14, 55, 14, 55, 37, 64, 16, 58, 18, 59, 41, 66, 43, 67, 16, 58, 41,
    66, 18, 59, 43, 67, 41, 69, 49, 71, 49, 71, 63, 73, 13, 52, 16, 53,
    39, 57, 41, 59, 20, 58, 21, 59, 45, 70, 46, 71, 52, 75, 58, 80, 57,
    76, 66, 81, 58, 82, 60, 83, 70, 86, 72, 87, 16, 58, 18, 59, 41, 66,
    43, 67, 21, 60, 22, 61, 46, 72, 47, 73, 53, 80, 59, 81, 59, 81, 67,
    85, 59, 83, 61, 84, 71, 87, 73, 88, 13, 52, 39, 57, 16, 53, 41, 59,
    52, 75, 57, 76, 58, 80, 66, 81, 20, 58, 45, 70, 21, 59, 46, 71, 58,
    82, 70, 86, 60, 83, 72, 87, 16, 58, 41, 66, 18, 59, 43, 67, 53, 80,
    59, 81, 59, 81, 67, 85, 21, 60, 46, 72, 22, 61, 47, 73, 59, 83, 71,
    87, 61, 84, 73, 88, 36, 55, 41, 59, 41, 59, 49, 61, 55, 78, 59, 79,
    69, 83, 71, 84, 55, 78, 69, 83, 59, 79, 71, 84, 78, 90, 83, 91, 83,
    91, 87, 92, 41, 69, 49, 71, 49, 71, 63, 73, 59, 83, 61, 84, 71, 87,
    73, 88, 59, 83, 71, 87, 61, 84, 73, 88, 79, 91, 84, 92, 84, 92, 88,
    93, 2, 6, 3, 7, 6, 8, 7, 9, 6, 13, 7, 14, 15, 17, 16, 18, 12, 15, 19,
    20, 15, 17, 20, 21, 15, 20, 20, 25, 26, 27, 27, 28, 6, 15, 7, 16, 13,
    17, 14, 18, 8, 17, 9, 18, 17, 23, 18, 24, 15, 26, 20, 27, 20, 27, 25,
    28, 17, 27, 21, 28, 27, 29, 28, 30, 6, 13, 7, 14, 15, 17, 16, 18, 13,
    36, 14, 37, 40, 42, 41, 43, 51, 52, 54, 55, 56, 57, 58, 59, 52, 55,
    55, 64, 65, 66, 66, 67, 15, 40, 16, 41, 40, 48, 41, 49, 17, 42, 18,
    43, 48, 62, 49, 63, 56, 65, 58, 66, 68, 70, 69, 71, 57, 66, 59, 67,
    70, 72, 71, 73, 12, 15, 19, 20, 15, 17, 20, 21, 51, 52, 54, 55, 56,
    57, 58, 59, 38, 40, 44, 45, 40, 42, 45, 46, 56, 58, 68, 69, 65, 66,
    70, 71, 51, 56, 54, 58, 52, 57, 55, 59, 74, 75, 77, 78, 75, 76, 78,
    79, 56, 65, 68, 70, 58, 66, 69, 71, 75, 80, 82, 83, 80, 81, 83, 84,
    15, 20, 20, 25, 26, 27, 27, 28, 52, 55, 55, 64, 65, 66, 66, 67, 56,
    58, 68, 69, 65, 66, 70, 71, 75, 78, 96, 97, 98, 99, 100, 101, 56, 68,
    58, 69, 65, 70, 66, 71, 75, 96, 78, 97, 98, 100, 99, 101, 104, 105,
    105, 106, 105, 106, 106, 107, 108, 109, 109, 110, 111, 112, 112, 113,
    6, 15, 7, 16, 13, 17, 14, 18, 15, 40, 16, 41, 40, 48, 41, 49, 51, 56,
    54, 58, 52, 57, 55, 59, 56, 68, 58, 69, 65, 70, 66, 71, 13, 40, 14,
    41, 36, 42, 37, 43, 17, 48, 18, 49, 42, 62, 43, 63, 52, 65, 55, 66,
    55, 66, 64, 67, 57, 70, 59, 71, 66, 72, 67, 73, 8, 17, 9, 18, 17, 23,
    18, 24, 17, 42, 18, 43, 48, 62, 49, 63, 74, 75, 77, 78, 75, 76, 78,
    79, 75, 96, 78, 97, 98, 100, 99, 101, 17, 48, 18, 49, 42, 62, 43, 63,
    23, 62, 24, 63, 62, 94, 63, 95, 75, 98, 78, 99, 96, 100, 97, 101, 76,
    100, 79, 101, 100, 102, 101, 103, 15, 26, 20, 27, 20, 27, 25, 28, 56,
    65, 58, 66, 68, 70, 69, 71, 56, 65, 68, 70, 58, 66, 69, 71, 104, 105,
    105, 106, 105, 106, 106, 107, 52, 65, 55, 66, 55, 66, 64, 67, 75, 98,
    78, 99, 96, 100, 97, 101, 75, 98, 96, 100, 78, 99, 97, 101, 108, 111,
    109, 112, 109, 112, 110, 113, 17, 27, 21, 28, 27, 29, 28, 30, 57, 66,
    59, 67, 70, 72, 71, 73, 75, 80, 82, 83, 80, 81, 83, 84, 108, 109,
    109, 110, 111, 112, 112, 113, 57, 70, 59, 71, 66, 72, 67, 73, 76,
    100, 79, 101, 100, 102, 101, 103, 108, 111, 109, 112, 109, 112, 110,
    113, 114, 116, 115, 117, 116, 118, 117, 119, 12, 51, 19, 54, 51, 74,
    54, 77, 15, 52, 20, 55, 56, 75, 58, 78, 38, 56, 44, 68, 56, 75, 68,
    82, 40, 58, 45, 69, 65, 80, 70, 83, 15, 56, 20, 58, 52, 75, 55, 78,
    17, 57, 21, 59, 57, 76, 59, 79, 40, 65, 45, 70, 58, 80, 69, 83, 42,
    66, 46, 71, 66, 81, 71, 84, 15, 52, 20, 55, 56, 75, 58, 78, 20, 55,
    25, 64, 68, 96, 69, 97, 56, 75, 68, 96, 104, 108, 105, 109, 58, 78,
    69, 97, 105, 109, 106, 110, 26, 65, 27, 66, 65, 98, 66, 99, 27, 66,
    28, 67, 70, 100, 71, 101, 65, 98, 70, 100, 105, 111, 106, 112, 66,
    99, 71, 101, 106, 112, 107, 113, 38, 56, 44, 68, 56, 75, 68, 82, 56,
    75, 68, 96, 104, 108, 105, 109, 44, 68, 120, 121, 68, 96, 121, 122,
    68, 82, 121, 122, 105, 109, 123, 124, 56, 104, 68, 105, 75, 108, 96,
    109, 75, 108, 82, 109, 108, 114, 109, 115, 68, 105, 121, 123, 82,
    109, 122, 124, 96, 109, 122, 124, 109, 115, 124, 125, 40, 58, 45, 69,
    65, 80, 70, 83, 58, 78, 69, 97, 105, 109, 106, 110, 68, 82, 121, 122,
    105, 109, 123, 124, 82, 90, 122, 126, 127, 129, 128, 130, 65, 105,
    70, 106, 98, 111, 100, 112, 80, 109, 83, 110, 111, 116, 112, 117,
    105, 127, 123, 128, 127, 131, 128, 132, 109, 129, 124, 130, 131, 133,
    132, 134, 15, 56, 20, 58, 52, 75, 55, 78, 26, 65, 27, 66, 65, 98, 66,
    99, 56, 104, 68, 105, 75, 108, 96, 109, 65, 105, 70, 106, 98, 111,
    100, 112, 20, 68, 25, 69, 55, 96, 64, 97, 27, 70, 28, 71, 66, 100,
    67, 101, 58, 105, 69, 106, 78, 109, 97, 110, 66, 106, 71, 107, 99,
    112, 101, 113, 17, 57, 21, 59, 57, 76, 59, 79, 27, 66, 28, 67, 70,
    100, 71, 101, 75, 108, 82, 109, 108, 114, 109, 115, 80, 109, 83, 110,
    111, 116, 112, 117, 27, 70, 28, 71, 66, 100, 67, 101, 29, 72, 30, 73,
    72, 102, 73, 103, 80, 111, 83, 112, 109, 116, 110, 117, 81, 112, 84,
    113, 112, 118, 113, 119, 40, 65, 45, 70, 58, 80, 69, 83, 65, 98, 70,
    100, 105, 111, 106, 112, 68, 105, 121, 123, 82, 109, 122, 124, 105,
    127, 123, 128, 127, 131, 128, 132, 58, 105, 69, 106, 78, 109, 97,
    110, 80, 111, 83, 112, 109, 116, 110, 117, 82, 127, 122, 128, 90,
    129, 126, 130, 109, 131, 124, 132, 129, 133, 130, 134, 42, 66, 46,
    71, 66, 81, 71, 84, 66, 99, 71, 101, 106, 112, 107, 113, 96, 109,
    122, 124, 109, 115, 124, 125, 109, 129, 124, 130, 131, 133, 132, 134,
    66, 106, 71, 107, 99, 112, 101, 113, 81, 112, 84, 113, 112, 118, 113,
    119, 109, 131, 124, 132, 129, 133, 130, 134, 115, 133, 125, 134, 133,
    135, 134, 136, 2, 6, 6, 8, 3, 7, 7, 9, 12, 15, 15, 17, 19, 20, 20,
    21, 6, 13, 15, 17, 7, 14, 16, 18, 15, 20, 26, 27, 20, 25, 27, 28, 6,
    15, 13, 17, 7, 16, 14, 18, 15, 26, 20, 27, 20, 27, 25, 28, 8, 17, 17,
    23, 9, 18, 18, 24, 17, 27, 27, 29, 21, 28, 28, 30, 12, 15, 15, 17,
    19, 20, 20, 21, 38, 40, 40, 42, 44, 45, 45, 46, 51, 52, 56, 57, 54,
    55, 58, 59, 56, 58, 65, 66, 68, 69, 70, 71, 51, 56, 52, 57, 54, 58,
    55, 59, 56, 65, 58, 66, 68, 70, 69, 71, 74, 75, 75, 76, 77, 78, 78,
    79, 75, 80, 80, 81, 82, 83, 83, 84, 6, 13, 15, 17, 7, 14, 16, 18, 51,
    52, 56, 57, 54, 55, 58, 59, 13, 36, 40, 42, 14, 37, 41, 43, 52, 55,
    65, 66, 55, 64, 66, 67, 15, 40, 40, 48, 16, 41, 41, 49, 56, 65, 68,
    70, 58, 66, 69, 71, 17, 42, 48, 62, 18, 43, 49, 63, 57, 66, 70, 72,
    59, 67, 71, 73, 15, 20, 26, 27, 20, 25, 27, 28, 56, 58, 65, 66, 68,
    69, 70, 71, 52, 55, 65, 66, 55, 64, 66, 67, 75, 78, 98, 99, 96, 97,
    100, 101, 56, 68, 65, 70, 58, 69, 66, 71, 104, 105, 105, 106, 105,
    106, 106, 107, 75, 96, 98, 100, 78, 97, 99, 101, 108, 109, 111, 112,
    109, 110, 112, 113, 6, 15, 13, 17, 7, 16, 14, 18, 51, 56, 52, 57, 54,
    58, 55, 59, 15, 40, 40, 48, 16, 41, 41, 49, 56, 68, 65, 70, 58, 69,
    66, 71, 13, 40, 36, 42, 14, 41, 37, 43, 52, 65, 55, 66, 55, 66, 64,
    67, 17, 48, 42, 62, 18, 49, 43, 63, 57, 70, 66, 72, 59, 71, 67, 73,
    15, 26, 20, 27, 20, 27, 25, 28, 56, 65, 58, 66, 68, 70, 69, 71, 56,
    65, 68, 70, 58, 66, 69, 71, 104, 105, 105, 106, 105, 106, 106, 107,
    52, 65, 55, 66, 55, 66, 64, 67, 75, 98, 78, 99, 96, 100, 97, 101, 75,
    98, 96, 100, 78, 99, 97, 101, 108, 111, 109, 112, 109, 112, 110, 113,
    8, 17, 17, 23, 9, 18, 18, 24, 74, 75, 75, 76, 77, 78, 78, 79, 17, 42,
    48, 62, 18, 43, 49, 63, 75, 96, 98, 100, 78, 97, 99, 101, 17, 48, 42,
    62, 18, 49, 43, 63, 75, 98, 96, 100, 78, 99, 97, 101, 23, 62, 62, 94,
    24, 63, 63, 95, 76, 100, 100, 102, 79, 101, 101, 103, 17, 27, 27, 29,
    21, 28, 28, 30, 75, 80, 80, 81, 82, 83, 83, 84, 57, 66, 70, 72, 59,
    67, 71, 73, 108, 109, 111, 112, 109, 110, 112, 113, 57, 70, 66, 72,
    59, 71, 67, 73, 108, 111, 109, 112, 109, 112, 110, 113, 76, 100, 100,
    102, 79, 101, 101, 103, 114, 116, 116, 118, 115, 117, 117, 119, 12,
    51, 51, 74, 19, 54, 54, 77, 38, 56, 56, 75, 44, 68, 68, 82, 15, 52,
    56, 75, 20, 55, 58, 78, 40, 58, 65, 80, 45, 69, 70, 83, 15, 56, 52,
    75, 20, 58, 55, 78, 40, 65, 58, 80, 45, 70, 69, 83, 17, 57, 57, 76,
    21, 59, 59, 79, 42, 66, 66, 81, 46, 71, 71, 84, 38, 56, 56, 75, 44,
    68, 68, 82, 44, 68, 68, 96, 120, 121, 121, 122, 56, 75, 104, 108, 68,
    96, 105, 109, 68, 82, 105, 109, 121, 122, 123, 124, 56, 104, 75, 108,
    68, 105, 96, 109, 68, 105, 82, 109, 121, 123, 122, 124, 75, 108, 108,
    114, 82, 109, 109, 115, 96, 109, 109, 115, 122, 124, 124, 125, 15,
    52, 56, 75, 20, 55, 58, 78, 56, 75, 104, 108, 68, 96, 105, 109, 20,
    55, 68, 96, 25, 64, 69, 97, 58, 78, 105, 109, 69, 97, 106, 110, 26,
    65, 65, 98, 27, 66, 66, 99, 65, 98, 105, 111, 70, 100, 106, 112, 27,
    66, 70, 100, 28, 67, 71, 101, 66, 99, 106, 112, 71, 101, 107, 113,
    40, 58, 65, 80, 45, 69, 70, 83, 68, 82, 105, 109, 121, 122, 123, 124,
    58, 78, 105, 109, 69, 97, 106, 110, 82, 90, 127, 129, 122, 126, 128,
    130, 65, 105, 98, 111, 70, 106, 100, 112, 105, 127, 127, 131, 123,
    128, 128, 132, 80, 109, 111, 116, 83, 110, 112, 117, 109, 129, 131,
    133, 124, 130, 132, 134, 15, 56, 52, 75, 20, 58, 55, 78, 56, 104, 75,
    108, 68, 105, 96, 109, 26, 65, 65, 98, 27, 66, 66, 99, 65, 105, 98,
    111, 70, 106, 100, 112, 20, 68, 55, 96, 25, 69, 64, 97, 58, 105, 78,
    109, 69, 106, 97, 110, 27, 70, 66, 100, 28, 71, 67, 101, 66, 106, 99,
    112, 71, 107, 101, 113, 40, 65, 58, 80, 45, 70, 69, 83, 68, 105, 82,
    109, 121, 123, 122, 124, 65, 98, 105, 111, 70, 100, 106, 112, 105,
    127, 127, 131, 123, 128, 128, 132, 58, 105, 78, 109, 69, 106, 97,
    110, 82, 127, 90, 129, 122, 128, 126, 130, 80, 111, 109, 116, 83,
    112, 110, 117, 109, 131, 129, 133, 124, 132, 130, 134, 17, 57, 57,
    76, 21, 59, 59, 79, 75, 108, 108, 114, 82, 109, 109, 115, 27, 66, 70,
    100, 28, 67, 71, 101, 80, 109, 111, 116, 83, 110, 112, 117, 27, 70,
    66, 100, 28, 71, 67, 101, 80, 111, 109, 116, 83, 112, 110, 117, 29,
    72, 72, 102, 30, 73, 73, 103, 81, 112, 112, 118, 84, 113, 113, 119,
    42, 66, 66, 81, 46, 71, 71, 84, 96, 109, 109, 115, 122, 124, 124,
    125, 66, 99, 106, 112, 71, 101, 107, 113, 109, 129, 131, 133, 124,
    130, 132, 134, 66, 106, 99, 112, 71, 107, 101, 113, 109, 131, 129,
    133, 124, 132, 130, 134, 81, 112, 112, 118, 84, 113, 113, 119, 115,
    133, 133, 135, 125, 134, 134, 136, 4, 7, 7, 9, 7, 9, 9, 10, 13, 16,
    16, 18, 20, 21, 21, 22, 13, 16, 20, 21, 16, 18, 21, 22, 17, 21, 27,
    28, 27, 28, 29, 30, 13, 20, 16, 21, 16, 21, 18, 22, 17, 27, 21, 28,
    27, 29, 28, 30, 17, 27, 27, 29, 21, 28, 28, 30, 23, 29, 29, 31, 29,
    31, 31, 32, 13, 16, 16, 18, 20, 21, 21, 22, 39, 41, 41, 43, 45, 46,
    46, 47, 52, 53, 58, 59, 58, 59, 60, 61, 57, 59, 66, 67, 70, 71, 72,
    73, 52, 58, 53, 59, 58, 60, 59, 61, 57, 66, 59, 67, 70, 72, 71, 73,
    75, 80, 80, 81, 82, 83, 83, 84, 76, 81, 81, 85, 86, 87, 87, 88, 13,
    16, 20, 21, 16, 18, 21, 22, 52, 53, 58, 59, 58, 59, 60, 61, 39, 41,
    45, 46, 41, 43, 46, 47, 57, 59, 70, 71, 66, 67, 72, 73, 52, 58, 58,
    60, 53, 59, 59, 61, 75, 80, 82, 83, 80, 81, 83, 84, 57, 66, 70, 72,
    59, 67, 71, 73, 76, 81, 86, 87, 81, 85, 87, 88, 17, 21, 27, 28, 27,
    28, 29, 30, 57, 59, 66, 67, 70, 71, 72, 73, 57, 59, 70, 71, 66, 67,
    72, 73, 76, 79, 100, 101, 100, 101, 102, 103, 75, 82, 80, 83, 80, 83,
    81, 84, 108, 109, 109, 110, 111, 112, 112, 113, 108, 109, 111, 112,
    109, 110, 112, 113, 114, 115, 116, 117, 116, 117, 118, 119, 13, 20,
    16, 21, 16, 21, 18, 22, 52, 58, 53, 59, 58, 60, 59, 61, 52, 58, 58,
    60, 53, 59, 59, 61, 75, 82, 80, 83, 80, 83, 81, 84, 39, 45, 41, 46,
    41, 46, 43, 47, 57, 70, 59, 71, 66, 72, 67, 73, 57, 70, 66, 72, 59,
    71, 67, 73, 76, 86, 81, 87, 81, 87, 85, 88, 17, 27, 21, 28, 27, 29,
    28, 30, 57, 66, 59, 67, 70, 72, 71, 73, 75, 80, 82, 83, 80, 81, 83,
    84, 108, 109, 109, 110, 111, 112, 112, 113, 57, 70, 59, 71, 66, 72,
    67, 73, 76, 100, 79, 101, 100, 102, 101, 103, 108, 111, 109, 112,
    109, 112, 110, 113, 114, 116, 115, 117, 116, 118, 117, 119, 17, 27,
    27, 29, 21, 28, 28, 30, 75, 80, 80, 81, 82, 83, 83, 84, 57, 66, 70,
    72, 59, 67, 71, 73, 108, 109, 111, 112, 109, 110, 112, 113, 57, 70,
    66, 72, 59, 71, 67, 73, 108, 111, 109, 112, 109, 112, 110, 113, 76,
    100, 100, 102, 79, 101, 101, 103, 114, 116, 116, 118, 115, 117, 117,
    119, 23, 29, 29, 31, 29, 31, 31, 32, 76, 81, 81, 85, 86, 87, 87, 88,
    76, 81, 86, 87, 81, 85, 87, 88, 114, 115, 116, 117, 116, 117, 118,
    119, 76, 86, 81, 87, 81, 87, 85, 88, 114, 116, 115, 117, 116, 118,
    117, 119, 114, 116, 116, 118, 115, 117, 117, 119, 137, 138, 138, 139,
    138, 139, 139, 140, 35, 54, 54, 77, 54, 77, 77, 89, 40, 58, 58, 78,
    68, 82, 82, 90, 40, 58, 68, 82, 58, 78, 82, 90, 48, 60, 70, 83, 70,
    83, 86, 91, 40, 68, 58, 82, 58, 82, 78, 90, 48, 70, 60, 83, 70, 86,
    83, 91, 48, 70, 70, 86, 60, 83, 83, 91, 62, 72, 72, 87, 72, 87, 87,
    92, 40, 58, 58, 78, 68, 82, 82, 90, 45, 69, 69, 97, 121, 122, 122,
    126, 65, 80, 105, 109, 105, 109, 127, 129, 70, 83, 106, 110, 123,
    124, 128, 130, 65, 105, 80, 109, 105, 127, 109, 129, 70, 106, 83,
    110, 123, 128, 124, 130, 98, 111, 111, 116, 127, 131, 131, 133, 100,
    112, 112, 117, 128, 132, 132, 134, 40, 58, 68, 82, 58, 78, 82, 90,
    65, 80, 105, 109, 105, 109, 127, 129, 45, 69, 121, 122, 69, 97, 122,
    126, 70, 83, 123, 124, 106, 110, 128, 130, 65, 105, 105, 127, 80,
    109, 109, 129, 98, 111, 127, 131, 111, 116, 131, 133, 70, 106, 123,
    128, 83, 110, 124, 130, 100, 112, 128, 132, 112, 117, 132, 134, 48,
    60, 70, 83, 70, 83, 86, 91, 70, 83, 106, 110, 123, 124, 128, 130, 70,
    83, 123, 124, 106, 110, 128, 130, 86, 91, 128, 130, 128, 130, 141,
    142, 98, 127, 111, 131, 111, 131, 116, 133, 111, 131, 131, 143, 144,
    145, 145, 146, 111, 131, 144, 145, 131, 143, 145, 146, 116, 133, 145,
    146, 145, 146, 147, 148, 40, 68, 58, 82, 58, 82, 78, 90, 65, 105, 80,
    109, 105, 127, 109, 129, 65, 105, 105, 127, 80, 109, 109, 129, 98,
    127, 111, 131, 111, 131, 116, 133, 45, 121, 69, 122, 69, 122, 97,
    126, 70, 123, 83, 124, 106, 128, 110, 130, 70, 123, 106, 128, 83,
    124, 110, 130, 100, 128, 112, 132, 112, 132, 117, 134, 48, 70, 60,
    83, 70, 86, 83, 91, 70, 106, 83, 110, 123, 128, 124, 130, 98, 111,
    127, 131, 111, 116, 131, 133, 111, 131, 131, 143, 144, 145, 145, 146,
    70, 123, 83, 124, 106, 128, 110, 130, 86, 128, 91, 130, 128, 141,
    130, 142, 111, 144, 131, 145, 131, 145, 143, 146, 116, 145, 133, 146,
    145, 147, 146, 148, 48, 70, 70, 86, 60, 83, 83, 91, 98, 111, 111,
    116, 127, 131, 131, 133, 70, 106, 123, 128, 83, 110, 124, 130, 111,
    131, 144, 145, 131, 143, 145, 146, 70, 123, 106, 128, 83, 124, 110,
    130, 111, 144, 131, 145, 131, 145, 143, 146, 86, 128, 128, 141, 91,
    130, 130, 142, 116, 145, 145, 147, 133, 146, 146, 148, 62, 72, 72,
    87, 72, 87, 87, 92, 100, 112, 112, 117, 128, 132, 132, 134, 100, 112,
    128, 132, 112, 117, 132, 134, 116, 133, 145, 146, 145, 146, 147, 148,
    100, 128, 112, 132, 112, 132, 117, 134, 116, 145, 133, 146, 145, 147,
    146, 148, 116, 145, 145, 147, 133, 146, 146, 148, 138, 149, 149, 150,
    149, 150, 150, 151, 1, 5, 5, 12, 5, 12, 12, 19, 2, 6, 6, 13, 12, 15,
    15, 20, 2, 6, 12, 15, 6, 13, 15, 20, 4, 7, 13, 16, 13, 16, 17, 21, 2,
    12, 6, 15, 6, 15, 13, 20, 4, 13, 7, 16, 13, 17, 16, 21, 4, 13, 13,
    17, 7, 16, 16, 21, 11, 14, 14, 18, 14, 18, 18, 22, 5, 12, 12, 35, 50,
    51, 51, 54, 6, 13, 13, 36, 51, 52, 52, 55, 12, 15, 38, 40, 51, 52,
    56, 58, 13, 16, 39, 41, 52, 53, 57, 59, 12, 38, 15, 40, 51, 56, 52,
    58, 13, 39, 16, 41, 52, 57, 53, 59, 35, 40, 40, 48, 54, 58, 58, 60,
    36, 41, 41, 49, 55, 59, 59, 61, 5, 12, 50, 51, 12, 35, 51, 54, 12,
    15, 51, 52, 38, 40, 56, 58, 6, 13, 51, 52, 13, 36, 52, 55, 13, 16,
    52, 53, 39, 41, 57, 59, 12, 38, 51, 56, 15, 40, 52, 58, 35, 40, 54,
    58, 40, 48, 58, 60, 13, 39, 52, 57, 16, 41, 53, 59, 36, 41, 55, 59,
    41, 49, 59, 61, 12, 19, 51, 54, 51, 54, 74, 77, 15, 20, 52, 55, 56,
    58, 75, 78, 15, 20, 56, 58, 52, 55, 75, 78, 17, 21, 57, 59, 57, 59,
    76, 79, 38, 44, 56, 68, 56, 68, 75, 82, 40, 45, 58, 69, 65, 70, 80,
    83, 40, 45, 65, 70, 58, 69, 80, 83, 42, 46, 66, 71, 66, 71, 81, 84,
    5, 50, 12, 51, 12, 51, 35, 54, 12, 51, 15, 52, 38, 56, 40, 58, 12,
    51, 38, 56, 15, 52, 40, 58, 35, 54, 40, 58, 40, 58, 48, 60, 6, 51,
    13, 52, 13, 52, 36, 55, 13, 52, 16, 53, 39, 57, 41, 59, 13, 52, 39,
    57, 16, 53, 41, 59, 36, 55, 41, 59, 41, 59, 49, 61, 12, 51, 19, 54,
    51, 74, 54, 77, 15, 52, 20, 55, 56, 75, 58, 78, 38, 56, 44, 68, 56,
    75, 68, 82, 40, 58, 45, 69, 65, 80, 70, 83, 15, 56, 20, 58, 52, 75,
    55, 78, 17, 57, 21, 59, 57, 76, 59, 79, 40, 65, 45, 70, 58, 80, 69,
    83, 42, 66, 46, 71, 66, 81, 71, 84, 12, 51, 51, 74, 19, 54, 54, 77,
    38, 56, 56, 75, 44, 68, 68, 82, 15, 52, 56, 75, 20, 55, 58, 78, 40,
    58, 65, 80, 45, 69, 70, 83, 15, 56, 52, 75, 20, 58, 55, 78, 40, 65,
    58, 80, 45, 70, 69, 83, 17, 57, 57, 76, 21, 59, 59, 79, 42, 66, 66,
    81, 46, 71, 71, 84, 35, 54, 54, 77, 54, 77, 77, 89, 40, 58, 58, 78,
    68, 82, 82, 90, 40, 58, 68, 82, 58, 78, 82, 90, 48, 60, 70, 83, 70,
    83, 86, 91, 40, 68, 58, 82, 58, 82, 78, 90, 48, 70, 60, 83, 70, 86,
    83, 91, 48, 70, 70, 86, 60, 83, 83, 91, 62, 72, 72, 87, 72, 87, 87,
    92, 2, 12, 12, 38, 12, 38, 38, 44, 4, 13, 13, 39, 35, 40, 40, 45, 4,
    13, 35, 40, 13, 39, 40, 45, 11, 14, 36, 41, 36, 41, 42, 46, 4, 35,
    13, 40, 13, 40, 39, 45, 11, 36, 14, 41, 36, 42, 41, 46, 11, 36, 36,
    42, 14, 41, 41, 46, 34, 37, 37, 43, 37, 43, 43, 47, 6, 15, 15, 40,
    51, 56, 56, 68, 7, 16, 16, 41, 54, 58, 58, 69, 13, 17, 40, 48, 52,
    57, 65, 70, 14, 18, 41, 49, 55, 59, 66, 71, 13, 40, 17, 48, 52, 65,
    57, 70, 14, 41, 18, 49, 55, 66, 59, 71, 36, 42, 42, 62, 55, 66, 66,
    72, 37, 43, 43, 63, 64, 67, 67, 73, 6, 15, 51, 56, 15, 40, 56, 68,
    13, 17, 52, 57, 40, 48, 65, 70, 7, 16, 54, 58, 16, 41, 58, 69, 14,
    18, 55, 59, 41, 49, 66, 71, 13, 40, 52, 65, 17, 48, 57, 70, 36, 42,
    55, 66, 42, 62, 66, 72, 14, 41, 55, 66, 18, 49, 59, 71, 37, 43, 64,
    67, 43, 63, 67, 73, 13, 20, 52, 58, 52, 58, 75, 82, 16, 21, 53, 59,
    58, 60, 80, 83, 16, 21, 58, 60, 53, 59, 80, 83, 18, 22, 59, 61, 59,
    61, 81, 84, 39, 45, 57, 70, 57, 70, 76, 86, 41, 46, 59, 71, 66, 72,
    81, 87, 41, 46, 66, 72, 59, 71, 81, 87, 43, 47, 67, 73, 67, 73, 85,
    88, 6, 51, 15, 56, 15, 56, 40, 68, 13, 52, 17, 57, 40, 65, 48, 70,
    13, 52, 40, 65, 17, 57, 48, 70, 36, 55, 42, 66, 42, 66, 62, 72, 7,
    54, 16, 58, 16, 58, 41, 69, 14, 55, 18, 59, 41, 66, 49, 71, 14, 55,
    41, 66, 18, 59, 49, 71, 37, 64, 43, 67, 43, 67, 63, 73, 13, 52, 20,
    58, 52, 75, 58, 82, 16, 53, 21, 59, 58, 80, 60, 83, 39, 57, 45, 70,
    57, 76, 70, 86, 41, 59, 46, 71, 66, 81, 72, 87, 16, 58, 21, 60, 53,
    80, 59, 83, 18, 59, 22, 61, 59, 81, 61, 84, 41, 66, 46, 72, 59, 81,
    71, 87, 43, 67, 47, 73, 67, 85, 73, 88, 13, 52, 52, 75, 20, 58, 58,
    82, 39, 57, 57, 76, 45, 70, 70, 86, 16, 53, 58, 80, 21, 59, 60, 83,
    41, 59, 66, 81, 46, 71, 72, 87, 16, 58, 53, 80, 21, 60, 59, 83, 41,
    66, 59, 81, 46, 72, 71, 87, 18, 59, 59, 81, 22, 61, 61, 84, 43, 67,
    67, 85, 47, 73, 73, 88, 36, 55, 55, 78, 55, 78, 78, 90, 41, 59, 59,
    79, 69, 83, 83, 91, 41, 59, 69, 83, 59, 79, 83, 91, 49, 61, 71, 84,
    71, 84, 87, 92, 41, 69, 59, 83, 59, 83, 79, 91, 49, 71, 61, 84, 71,
    87, 84, 92, 49, 71, 71, 87, 61, 84, 84, 92, 63, 73, 73, 88, 73, 88,
    88, 93, 2, 6, 6, 13, 12, 15, 15, 20, 3, 7, 7, 14, 19, 20, 20, 25, 6,
    8, 15, 17, 15, 17, 26, 27, 7, 9, 16, 18, 20, 21, 27, 28, 6, 15, 8,
    17, 15, 26, 17, 27, 7, 16, 9, 18, 20, 27, 21, 28, 13, 17, 17, 23, 20,
    27, 27, 29, 14, 18, 18, 24, 25, 28, 28, 30, 6, 13, 13, 36, 51, 52,
    52, 55, 7, 14, 14, 37, 54, 55, 55, 64, 15, 17, 40, 42, 56, 57, 65,
    66, 16, 18, 41, 43, 58, 59, 66, 67, 15, 40, 17, 42, 56, 65, 57, 66,
    16, 41, 18, 43, 58, 66, 59, 67, 40, 48, 48, 62, 68, 70, 70, 72, 41,
    49, 49, 63, 69, 71, 71, 73, 12, 15, 51, 52, 38, 40, 56, 58, 19, 20,
    54, 55, 44, 45, 68, 69, 15, 17, 56, 57, 40, 42, 65, 66, 20, 21, 58,
    59, 45, 46, 70, 71, 51, 56, 74, 75, 56, 65, 75, 80, 54, 58, 77, 78,
    68, 70, 82, 83, 52, 57, 75, 76, 58, 66, 80, 81, 55, 59, 78, 79, 69,
    71, 83, 84, 15, 20, 52, 55, 56, 58, 75, 78, 20, 25, 55, 64, 68, 69,
    96, 97, 26, 27, 65, 66, 65, 66, 98, 99, 27, 28, 66, 67, 70, 71, 100,
    101, 56, 68, 75, 96, 104, 105, 108, 109, 58, 69, 78, 97, 105, 106,
    109, 110, 65, 70, 98, 100, 105, 106, 111, 112, 66, 71, 99, 101, 106,
    107, 112, 113, 12, 51, 15, 52, 38, 56, 40, 58, 19, 54, 20, 55, 44,
    68, 45, 69, 51, 74, 56, 75, 56, 75, 65, 80, 54, 77, 58, 78, 68, 82,
    70, 83, 15, 56, 17, 57, 40, 65, 42, 66, 20, 58, 21, 59, 45, 70, 46,
    71, 52, 75, 57, 76, 58, 80, 66, 81, 55, 78, 59, 79, 69, 83, 71, 84,
    15, 52, 20, 55, 56, 75, 58, 78, 20, 55, 25, 64, 68, 96, 69, 97, 56,
    75, 68, 96, 104, 108, 105, 109, 58, 78, 69, 97, 105, 109, 106, 110,
    26, 65, 27, 66, 65, 98, 66, 99, 27, 66, 28, 67, 70, 100, 71, 101, 65,
    98, 70, 100, 105, 111, 106, 112, 66, 99, 71, 101, 106, 112, 107, 113,
    38, 56, 56, 75, 44, 68, 68, 82, 44, 68, 68, 96, 120, 121, 121, 122,
    56, 75, 104, 108, 68, 96, 105, 109, 68, 82, 105, 109, 121, 122, 123,
    124, 56, 104, 75, 108, 68, 105, 96, 109, 68, 105, 82, 109, 121, 123,
    122, 124, 75, 108, 108, 114, 82, 109, 109, 115, 96, 109, 109, 115,
    122, 124, 124, 125, 40, 58, 58, 78, 68, 82, 82, 90, 45, 69, 69, 97,
    121, 122, 122, 126, 65, 80, 105, 109, 105, 109, 127, 129, 70, 83,
    106, 110, 123, 124, 128, 130, 65, 105, 80, 109, 105, 127, 109, 129,
    70, 106, 83, 110, 123, 128, 124, 130, 98, 111, 111, 116, 127, 131,
    131, 133, 100, 112, 112, 117, 128, 132, 132, 134, 6, 15, 15, 40, 51,
    56, 56, 68, 7, 16, 16, 41, 54, 58, 58, 69, 13, 17, 40, 48, 52, 57,
    65, 70, 14, 18, 41, 49, 55, 59, 66, 71, 13, 40, 17, 48, 52, 65, 57,
    70, 14, 41, 18, 49, 55, 66, 59, 71, 36, 42, 42, 62, 55, 66, 66, 72,
    37, 43, 43, 63, 64, 67, 67, 73, 8, 17, 17, 42, 74, 75, 75, 96, 9, 18,
    18, 43, 77, 78, 78, 97, 17, 23, 48, 62, 75, 76, 98, 100, 18, 24, 49,
    63, 78, 79, 99, 101, 17, 48, 23, 62, 75, 98, 76, 100, 18, 49, 24, 63,
    78, 99, 79, 101, 42, 62, 62, 94, 96, 100, 100, 102, 43, 63, 63, 95,
    97, 101, 101, 103, 15, 26, 56, 65, 56, 65, 104, 105, 20, 27, 58, 66,
    68, 70, 105, 106, 20, 27, 68, 70, 58, 66, 105, 106, 25, 28, 69, 71,
    69, 71, 106, 107, 52, 65, 75, 98, 75, 98, 108, 111, 55, 66, 78, 99,
    96, 100, 109, 112, 55, 66, 96, 100, 78, 99, 109, 112, 64, 67, 97,
    101, 97, 101, 110, 113, 17, 27, 57, 66, 75, 80, 108, 109, 21, 28, 59,
    67, 82, 83, 109, 110, 27, 29, 70, 72, 80, 81, 111, 112, 28, 30, 71,
    73, 83, 84, 112, 113, 57, 70, 76, 100, 108, 111, 114, 116, 59, 71,
    79, 101, 109, 112, 115, 117, 66, 72, 100, 102, 109, 112, 116, 118,
    67, 73, 101, 103, 110, 113, 117, 119, 15, 56, 26, 65, 56, 104, 65,
    105, 20, 58, 27, 66, 68, 105, 70, 106, 52, 75, 65, 98, 75, 108, 98,
    111, 55, 78, 66, 99, 96, 109, 100, 112, 20, 68, 27, 70, 58, 105, 66,
    106, 25, 69, 28, 71, 69, 106, 71, 107, 55, 96, 66, 100, 78, 109, 99,
    112, 64, 97, 67, 101, 97, 110, 101, 113, 17, 57, 27, 66, 75, 108, 80,
    109, 21, 59, 28, 67, 82, 109, 83, 110, 57, 76, 70, 100, 108, 114,
    111, 116, 59, 79, 71, 101, 109, 115, 112, 117, 27, 70, 29, 72, 80,
    111, 81, 112, 28, 71, 30, 73, 83, 112, 84, 113, 66, 100, 72, 102,
    109, 116, 112, 118, 67, 101, 73, 103, 110, 117, 113, 119, 40, 65, 65,
    98, 68, 105, 105, 127, 45, 70, 70, 100, 121, 123, 123, 128, 58, 80,
    105, 111, 82, 109, 127, 131, 69, 83, 106, 112, 122, 124, 128, 132,
    58, 105, 80, 111, 82, 127, 109, 131, 69, 106, 83, 112, 122, 128, 124,
    132, 78, 109, 109, 116, 90, 129, 129, 133, 97, 110, 110, 117, 126,
    130, 130, 134, 42, 66, 66, 99, 96, 109, 109, 129, 46, 71, 71, 101,
    122, 124, 124, 130, 66, 81, 106, 112, 109, 115, 131, 133, 71, 84,
    107, 113, 124, 125, 132, 134, 66, 106, 81, 112, 109, 131, 115, 133,
    71, 107, 84, 113, 124, 132, 125, 134, 99, 112, 112, 118, 129, 133,
    133, 135, 101, 113, 113, 119, 130, 134, 134, 136, 2, 6, 12, 15, 6,
    13, 15, 20, 6, 8, 15, 17, 15, 17, 26, 27, 3, 7, 19, 20, 7, 14, 20,
    25, 7, 9, 20, 21, 16, 18, 27, 28, 6, 15, 15, 26, 8, 17, 17, 27, 13,
    17, 20, 27, 17, 23, 27, 29, 7, 16, 20, 27, 9, 18, 21, 28, 14, 18, 25,
    28, 18, 24, 28, 30, 12, 15, 38, 40, 51, 52, 56, 58, 15, 17, 40, 42,
    56, 57, 65, 66, 19, 20, 44, 45, 54, 55, 68, 69, 20, 21, 45, 46, 58,
    59, 70, 71, 51, 56, 56, 65, 74, 75, 75, 80, 52, 57, 58, 66, 75, 76,
    80, 81, 54, 58, 68, 70, 77, 78, 82, 83, 55, 59, 69, 71, 78, 79, 83,
    84, 6, 13, 51, 52, 13, 36, 52, 55, 15, 17, 56, 57, 40, 42, 65, 66, 7,
    14, 54, 55, 14, 37, 55, 64, 16, 18, 58, 59, 41, 43, 66, 67, 15, 40,
    56, 65, 17, 42, 57, 66, 40, 48, 68, 70, 48, 62, 70, 72, 16, 41, 58,
    66, 18, 43, 59, 67, 41, 49, 69, 71, 49, 63, 71, 73, 15, 20, 56, 58,
    52, 55, 75, 78, 26, 27, 65, 66, 65, 66, 98, 99, 20, 25, 68, 69, 55,
    64, 96, 97, 27, 28, 70, 71, 66, 67, 100, 101, 56, 68, 104, 105, 75,
    96, 108, 109, 65, 70, 105, 106, 98, 100, 111, 112, 58, 69, 105, 106,
    78, 97, 109, 110, 66, 71, 106, 107, 99, 101, 112, 113, 12, 51, 38,
    56, 15, 52, 40, 58, 51, 74, 56, 75, 56, 75, 65, 80, 19, 54, 44, 68,
    20, 55, 45, 69, 54, 77, 68, 82, 58, 78, 70, 83, 15, 56, 40, 65, 17,
    57, 42, 66, 52, 75, 58, 80, 57, 76, 66, 81, 20, 58, 45, 70, 21, 59,
    46, 71, 55, 78, 69, 83, 59, 79, 71, 84, 38, 56, 44, 68, 56, 75, 68,
    82, 56, 75, 68, 96, 104, 108, 105, 109, 44, 68, 120, 121, 68, 96,
    121, 122, 68, 82, 121, 122, 105, 109, 123, 124, 56, 104, 68, 105, 75,
    108, 96, 109, 75, 108, 82, 109, 108, 114, 109, 115, 68, 105, 121,
    123, 82, 109, 122, 124, 96, 109, 122, 124, 109, 115, 124, 125, 15,
    52, 56, 75, 20, 55, 58, 78, 56, 75, 104, 108, 68, 96, 105, 109, 20,
    55, 68, 96, 25, 64, 69, 97, 58, 78, 105, 109, 69, 97, 106, 110, 26,
    65, 65, 98, 27, 66, 66, 99, 65, 98, 105, 111, 70, 100, 106, 112, 27,
    66, 70, 100, 28, 67, 71, 101, 66, 99, 106, 112, 71, 101, 107, 113,
    40, 58, 68, 82, 58, 78, 82, 90, 65, 80, 105, 109, 105, 109, 127, 129,
    45, 69, 121, 122, 69, 97, 122, 126, 70, 83, 123, 124, 106, 110, 128,
    130, 65, 105, 105, 127, 80, 109, 109, 129, 98, 111, 127, 131, 111,
    116, 131, 133, 70, 106, 123, 128, 83, 110, 124, 130, 100, 112, 128,
    132, 112, 117, 132, 134, 6, 15, 51, 56, 15, 40, 56, 68, 13, 17, 52,
    57, 40, 48, 65, 70, 7, 16, 54, 58, 16, 41, 58, 69, 14, 18, 55, 59,
    41, 49, 66, 71, 13, 40, 52, 65, 17, 48, 57, 70, 36, 42, 55, 66, 42,
    62, 66, 72, 14, 41, 55, 66, 18, 49, 59, 71, 37, 43, 64, 67, 43, 63,
    67, 73, 15, 26, 56, 65, 56, 65, 104, 105, 20, 27, 58, 66, 68, 70,
    105, 106, 20, 27, 68, 70, 58, 66, 105, 106, 25, 28, 69, 71, 69, 71,
    106, 107, 52, 65, 75, 98, 75, 98, 108, 111, 55, 66, 78, 99, 96, 100,
    109, 112, 55, 66, 96, 100, 78, 99, 109, 112, 64, 67, 97, 101, 97,
    101, 110, 113, 8, 17, 74, 75, 17, 42, 75, 96, 17, 23, 75, 76, 48, 62,
    98, 100, 9, 18, 77, 78, 18, 43, 78, 97, 18, 24, 78, 79, 49, 63, 99,
    101, 17, 48, 75, 98, 23, 62, 76, 100, 42, 62, 96, 100, 62, 94, 100,
    102, 18, 49, 78, 99, 24, 63, 79, 101, 43, 63, 97, 101, 63, 95, 101,
    103, 17, 27, 75, 80, 57, 66, 108, 109, 27, 29, 80, 81, 70, 72, 111,
    112, 21, 28, 82, 83, 59, 67, 109, 110, 28, 30, 83, 84, 71, 73, 112,
    113, 57, 70, 108, 111, 76, 100, 114, 116, 66, 72, 109, 112, 100, 102,
    116, 118, 59, 71, 109, 112, 79, 101, 115, 117, 67, 73, 110, 113, 101,
    103, 117, 119, 15, 56, 56, 104, 26, 65, 65, 105, 52, 75, 75, 108, 65,
    98, 98, 111, 20, 58, 68, 105, 27, 66, 70, 106, 55, 78, 96, 109, 66,
    99, 100, 112, 20, 68, 58, 105, 27, 70, 66, 106, 55, 96, 78, 109, 66,
    100, 99, 112, 25, 69, 69, 106, 28, 71, 71, 107, 64, 97, 97, 110, 67,
    101, 101, 113, 40, 65, 68, 105, 65, 98, 105, 127, 58, 80, 82, 109,
    105, 111, 127, 131, 45, 70, 121, 123, 70, 100, 123, 128, 69, 83, 122,
    124, 106, 112, 128, 132, 58, 105, 82, 127, 80, 111, 109, 131, 78,
    109, 90, 129, 109, 116, 129, 133, 69, 106, 122, 128, 83, 112, 124,
    132, 97, 110, 126, 130, 110, 117, 130, 134, 17, 57, 75, 108, 27, 66,
    80, 109, 57, 76, 108, 114, 70, 100, 111, 116, 21, 59, 82, 109, 28,
    67, 83, 110, 59, 79, 109, 115, 71, 101, 112, 117, 27, 70, 80, 111,
    29, 72, 81, 112, 66, 100, 109, 116, 72, 102, 112, 118, 28, 71, 83,
    112, 30, 73, 84, 113, 67, 101, 110, 117, 73, 103, 113, 119, 42, 66,
    96, 109, 66, 99, 109, 129, 66, 81, 109, 115, 106, 112, 131, 133, 46,
    71, 122, 124, 71, 101, 124, 130, 71, 84, 124, 125, 107, 113, 132,
    134, 66, 106, 109, 131, 81, 112, 115, 133, 99, 112, 129, 133, 112,
    118, 133, 135, 71, 107, 124, 132, 84, 113, 125, 134, 101, 113, 130,
    134, 113, 119, 134, 136, 4, 7, 13, 16, 13, 16, 17, 21, 7, 9, 16, 18,
    20, 21, 27, 28, 7, 9, 20, 21, 16, 18, 27, 28, 9, 10, 21, 22, 21, 22,
    29, 30, 13, 20, 17, 27, 17, 27, 23, 29, 16, 21, 21, 28, 27, 29, 29,
    31, 16, 21, 27, 29, 21, 28, 29, 31, 18, 22, 28, 30, 28, 30, 31, 32,
    13, 16, 39, 41, 52, 53, 57, 59, 16, 18, 41, 43, 58, 59, 66, 67, 20,
    21, 45, 46, 58, 59, 70, 71, 21, 22, 46, 47, 60, 61, 72, 73, 52, 58,
    57, 66, 75, 80, 76, 81, 53, 59, 59, 67, 80, 81, 81, 85, 58, 60, 70,
    72, 82, 83, 86, 87, 59, 61, 71, 73, 83, 84, 87, 88, 13, 16, 52, 53,
    39, 41, 57, 59, 20, 21, 58, 59, 45, 46, 70, 71, 16, 18, 58, 59, 41,
    43, 66, 67, 21, 22, 60, 61, 46, 47, 72, 73, 52, 58, 75, 80, 57, 66,
    76, 81, 58, 60, 82, 83, 70, 72, 86, 87, 53, 59, 80, 81, 59, 67, 81,
    85, 59, 61, 83, 84, 71, 73, 87, 88, 17, 21, 57, 59, 57, 59, 76, 79,
    27, 28, 66, 67, 70, 71, 100, 101, 27, 28, 70, 71, 66, 67, 100, 101,
    29, 30, 72, 73, 72, 73, 102, 103, 75, 82, 108, 109, 108, 109, 114,
    115, 80, 83, 109, 110, 111, 112, 116, 117, 80, 83, 111, 112, 109,
    110, 116, 117, 81, 84, 112, 113, 112, 113, 118, 119, 35, 54, 40, 58,
    40, 58, 48, 60, 54, 77, 58, 78, 68, 82, 70, 83, 54, 77, 68, 82, 58,
    78, 70, 83, 77, 89, 82, 90, 82, 90, 86, 91, 40, 68, 48, 70, 48, 70,
    62, 72, 58, 82, 60, 83, 70, 86, 72, 87, 58, 82, 70, 86, 60, 83, 72,
    87, 78, 90, 83, 91, 83, 91, 87, 92, 40, 58, 45, 69, 65, 80, 70, 83,
    58, 78, 69, 97, 105, 109, 106, 110, 68, 82, 121, 122, 105, 109, 123,
    124, 82, 90, 122, 126, 127, 129, 128, 130, 65, 105, 70, 106, 98, 111,
    100, 112, 80, 109, 83, 110, 111, 116, 112, 117, 105, 127, 123, 128,
    127, 131, 128, 132, 109, 129, 124, 130, 131, 133, 132, 134, 40, 58,
    65, 80, 45, 69, 70, 83, 68, 82, 105, 109, 121, 122, 123, 124, 58, 78,
    105, 109, 69, 97, 106, 110, 82, 90, 127, 129, 122, 126, 128, 130, 65,
    105, 98, 111, 70, 106, 100, 112, 105, 127, 127, 131, 123, 128, 128,
    132, 80, 109, 111, 116, 83, 110, 112, 117, 109, 129, 131, 133, 124,
    130, 132, 134, 48, 60, 70, 83, 70, 83, 86, 91, 70, 83, 106, 110, 123,
    124, 128, 130, 70, 83, 123, 124, 106, 110, 128, 130, 86, 91, 128,
    130, 128, 130, 141, 142, 98, 127, 111, 131, 111, 131, 116, 133, 111,
    131, 131, 143, 144, 145, 145, 146, 111, 131, 144, 145, 131, 143, 145,
    146, 116, 133, 145, 146, 145, 146, 147, 148, 13, 20, 52, 58, 52, 58,
    75, 82, 16, 21, 53, 59, 58, 60, 80, 83, 16, 21, 58, 60, 53, 59, 80,
    83, 18, 22, 59, 61, 59, 61, 81, 84, 39, 45, 57, 70, 57, 70, 76, 86,
    41, 46, 59, 71, 66, 72, 81, 87, 41, 46, 66, 72, 59, 71, 81, 87, 43,
    47, 67, 73, 67, 73, 85, 88, 17, 27, 57, 66, 75, 80, 108, 109, 21, 28,
    59, 67, 82, 83, 109, 110, 27, 29, 70, 72, 80, 81, 111, 112, 28, 30,
    71, 73, 83, 84, 112, 113, 57, 70, 76, 100, 108, 111, 114, 116, 59,
    71, 79, 101, 109, 112, 115, 117, 66, 72, 100, 102, 109, 112, 116,
    118, 67, 73, 101, 103, 110, 113, 117, 119, 17, 27, 75, 80, 57, 66,
    108, 109, 27, 29, 80, 81, 70, 72, 111, 112, 21, 28, 82, 83, 59, 67,
    109, 110, 28, 30, 83, 84, 71, 73, 112, 113, 57, 70, 108, 111, 76,
    100, 114, 116, 66, 72, 109, 112, 100, 102, 116, 118, 59, 71, 109,
    112, 79, 101, 115, 117, 67, 73, 110, 113, 101, 103, 117, 119, 23, 29,
    76, 81, 76, 81, 114, 115, 29, 31, 81, 85, 86, 87, 116, 117, 29, 31,
    86, 87, 81, 85, 116, 117, 31, 32, 87, 88, 87, 88, 118, 119, 76, 86,
    114, 116, 114, 116, 137, 138, 81, 87, 115, 117, 116, 118, 138, 139,
    81, 87, 116, 118, 115, 117, 138, 139, 85, 88, 117, 119, 117, 119,
    139, 140, 40, 68, 65, 105, 65, 105, 98, 127, 58, 82, 80, 109, 105,
    127, 111, 131, 58, 82, 105, 127, 80, 109, 111, 131, 78, 90, 109, 129,
    109, 129, 116, 133, 45, 121, 70, 123, 70, 123, 100, 128, 69, 122, 83,
    124, 106, 128, 112, 132, 69, 122, 106, 128, 83, 124, 112, 132, 97,
    126, 110, 130, 110, 130, 117, 134, 48, 70, 70, 106, 98, 111, 111,
    131, 60, 83, 83, 110, 127, 131, 131, 143, 70, 86, 123, 128, 111, 116,
    144, 145, 83, 91, 124, 130, 131, 133, 145, 146, 70, 123, 86, 128,
    111, 144, 116, 145, 83, 124, 91, 130, 131, 145, 133, 146, 106, 128,
    128, 141, 131, 145, 145, 147, 110, 130, 130, 142, 143, 146, 146, 148,
    48, 70, 98, 111, 70, 106, 111, 131, 70, 86, 111, 116, 123, 128, 144,
    145, 60, 83, 127, 131, 83, 110, 131, 143, 83, 91, 131, 133, 124, 130,
    145, 146, 70, 123, 111, 144, 86, 128, 116, 145, 106, 128, 131, 145,
    128, 141, 145, 147, 83, 124, 131, 145, 91, 130, 133, 146, 110, 130,
    143, 146, 130, 142, 146, 148, 62, 72, 100, 112, 100, 112, 116, 133,
    72, 87, 112, 117, 128, 132, 145, 146, 72, 87, 128, 132, 112, 117,
    145, 146, 87, 92, 132, 134, 132, 134, 147, 148, 100, 128, 116, 145,
    116, 145, 138, 149, 112, 132, 133, 146, 145, 147, 149, 150, 112, 132,
    145, 147, 133, 146, 149, 150, 117, 134, 146, 148, 146, 148, 150, 151,
    2, 12, 6, 15, 6, 15, 13, 20, 6, 15, 8, 17, 15, 26, 17, 27, 6, 15, 15,
    26, 8, 17, 17, 27, 13, 20, 17, 27, 17, 27, 23, 29, 3, 19, 7, 20, 7,
    20, 14, 25, 7, 20, 9, 21, 16, 27, 18, 28, 7, 20, 16, 27, 9, 21, 18,
    28, 14, 25, 18, 28, 18, 28, 24, 30, 12, 38, 15, 40, 51, 56, 52, 58,
    15, 40, 17, 42, 56, 65, 57, 66, 51, 56, 56, 65, 74, 75, 75, 80, 52,
    58, 57, 66, 75, 80, 76, 81, 19, 44, 20, 45, 54, 68, 55, 69, 20, 45,
    21, 46, 58, 70, 59, 71, 54, 68, 58, 70, 77, 82, 78, 83, 55, 69, 59,
    71, 78, 83, 79, 84, 12, 38, 51, 56, 15, 40, 52, 58, 51, 56, 74, 75,
    56, 65, 75, 80, 15, 40, 56, 65, 17, 42, 57, 66, 52, 58, 75, 80, 57,
    66, 76, 81, 19, 44, 54, 68, 20, 45, 55, 69, 54, 68, 77, 82, 58, 70,
    78, 83, 20, 45, 58, 70, 21, 46, 59, 71, 55, 69, 78, 83, 59, 71, 79,
    84, 38, 44, 56, 68, 56, 68, 75, 82, 56, 68, 75, 96, 104, 105, 108,
    109, 56, 68, 104, 105, 75, 96, 108, 109, 75, 82, 108, 109, 108, 109,
    114, 115, 44, 120, 68, 121, 68, 121, 96, 122, 68, 121, 82, 122, 105,
    123, 109, 124, 68, 121, 105, 123, 82, 122, 109, 124, 96, 122, 109,
    124, 109, 124, 115, 125, 6, 51, 13, 52, 13, 52, 36, 55, 15, 56, 17,
    57, 40, 65, 42, 66, 15, 56, 40, 65, 17, 57, 42, 66, 40, 68, 48, 70,
    48, 70, 62, 72, 7, 54, 14, 55, 14, 55, 37, 64, 16, 58, 18, 59, 41,
    66, 43, 67, 16, 58, 41, 66, 18, 59, 43, 67, 41, 69, 49, 71, 49, 71,
    63, 73, 15, 56, 20, 58, 52, 75, 55, 78, 26, 65, 27, 66, 65, 98, 66,
    99, 56, 104, 68, 105, 75, 108, 96, 109, 65, 105, 70, 106, 98, 111,
    100, 112, 20, 68, 25, 69, 55, 96, 64, 97, 27, 70, 28, 71, 66, 100,
    67, 101, 58, 105, 69, 106, 78, 109, 97, 110, 66, 106, 71, 107, 99,
    112, 101, 113, 15, 56, 52, 75, 20, 58, 55, 78, 56, 104, 75, 108, 68,
    105, 96, 109, 26, 65, 65, 98, 27, 66, 66, 99, 65, 105, 98, 111, 70,
    106, 100, 112, 20, 68, 55, 96, 25, 69, 64, 97, 58, 105, 78, 109, 69,
    106, 97, 110, 27, 70, 66, 100, 28, 71, 67, 101, 66, 106, 99, 112, 71,
    107, 101, 113, 40, 68, 58, 82, 58, 82, 78, 90, 65, 105, 80, 109, 105,
    127, 109, 129, 65, 105, 105, 127, 80, 109, 109, 129, 98, 127, 111,
    131, 111, 131, 116, 133, 45, 121, 69, 122, 69, 122, 97, 126, 70, 123,
    83, 124, 106, 128, 110, 130, 70, 123, 106, 128, 83, 124, 110, 130,
    100, 128, 112, 132, 112, 132, 117, 134, 6, 51, 15, 56, 15, 56, 40,
    68, 13, 52, 17, 57, 40, 65, 48, 70, 13, 52, 40, 65, 17, 57, 48, 70,
    36, 55, 42, 66, 42, 66, 62, 72, 7, 54, 16, 58, 16, 58, 41, 69, 14,
    55, 18, 59, 41, 66, 49, 71, 14, 55, 41, 66, 18, 59, 49, 71, 37, 64,
    43, 67, 43, 67, 63, 73, 15, 56, 26, 65, 56, 104, 65, 105, 20, 58, 27,
    66, 68, 105, 70, 106, 52, 75, 65, 98, 75, 108, 98, 111, 55, 78, 66,
    99, 96, 109, 100, 112, 20, 68, 27, 70, 58, 105, 66, 106, 25, 69, 28,
    71, 69, 106, 71, 107, 55, 96, 66, 100, 78, 109, 99, 112, 64, 97, 67,
    101, 97, 110, 101, 113, 15, 56, 56, 104, 26, 65, 65, 105, 52, 75, 75,
    108, 65, 98, 98, 111, 20, 58, 68, 105, 27, 66, 70, 106, 55, 78, 96,
    109, 66, 99, 100, 112, 20, 68, 58, 105, 27, 70, 66, 106, 55, 96, 78,
    109, 66, 100, 99, 112, 25, 69, 69, 106, 28, 71, 71, 107, 64, 97, 97,
    110, 67, 101, 101, 113, 40, 68, 65, 105, 65, 105, 98, 127, 58, 82,
    80, 109, 105, 127, 111, 131, 58, 82, 105, 127, 80, 109, 111, 131, 78,
    90, 109, 129, 109, 129, 116, 133, 45, 121, 70, 123, 70, 123, 100,
    128, 69, 122, 83, 124, 106, 128, 112, 132, 69, 122, 106, 128, 83,
    124, 112, 132, 97, 126, 110, 130, 110, 130, 117, 134, 8, 74, 17, 75,
    17, 75, 42, 96, 17, 75, 23, 76, 48, 98, 62, 100, 17, 75, 48, 98, 23,
    76, 62, 100, 42, 96, 62, 100, 62, 100, 94, 102, 9, 77, 18, 78, 18,
    78, 43, 97, 18, 78, 24, 79, 49, 99, 63, 101, 18, 78, 49, 99, 24, 79,
    63, 101, 43, 97, 63, 101, 63, 101, 95, 103, 17, 75, 27, 80, 57, 108,
    66, 109, 27, 80, 29, 81, 70, 111, 72, 112, 57, 108, 70, 111, 76, 114,
    100, 116, 66, 109, 72, 112, 100, 116, 102, 118, 21, 82, 28, 83, 59,
    109, 67, 110, 28, 83, 30, 84, 71, 112, 73, 113, 59, 109, 71, 112, 79,
    115, 101, 117, 67, 110, 73, 113, 101, 117, 103, 119, 17, 75, 57, 108,
    27, 80, 66, 109, 57, 108, 76, 114, 70, 111, 100, 116, 27, 80, 70,
    111, 29, 81, 72, 112, 66, 109, 100, 116, 72, 112, 102, 118, 21, 82,
    59, 109, 28, 83, 67, 110, 59, 109, 79, 115, 71, 112, 101, 117, 28,
    83, 71, 112, 30, 84, 73, 113, 67, 110, 101, 117, 73, 113, 103, 119,
    42, 96, 66, 109, 66, 109, 99, 129, 66, 109, 81, 115, 106, 131, 112,
    133, 66, 109, 106, 131, 81, 115, 112, 133, 99, 129, 112, 133, 112,
    133, 118, 135, 46, 122, 71, 124, 71, 124, 101, 130, 71, 124, 84, 125,
    107, 132, 113, 134, 71, 124, 107, 132, 84, 125, 113, 134, 101, 130,
    113, 134, 113, 134, 119, 136, 4, 13, 7, 16, 13, 17, 16, 21, 7, 16, 9,
    18, 20, 27, 21, 28, 13, 17, 20, 27, 17, 23, 27, 29, 16, 21, 21, 28,
    27, 29, 29, 31, 7, 20, 9, 21, 16, 27, 18, 28, 9, 21, 10, 22, 21, 29,
    22, 30, 16, 27, 21, 29, 21, 29, 28, 31, 18, 28, 22, 30, 28, 31, 30,
    32, 13, 39, 16, 41, 52, 57, 53, 59, 16, 41, 18, 43, 58, 66, 59, 67,
    52, 57, 58, 66, 75, 76, 80, 81, 53, 59, 59, 67, 80, 81, 81, 85, 20,
    45, 21, 46, 58, 70, 59, 71, 21, 46, 22, 47, 60, 72, 61, 73, 58, 70,
    60, 72, 82, 86, 83, 87, 59, 71, 61, 73, 83, 87, 84, 88, 35, 40, 54,
    58, 40, 48, 58, 60, 54, 58, 77, 78, 68, 70, 82, 83, 40, 48, 68, 70,
    48, 62, 70, 72, 58, 60, 82, 83, 70, 72, 86, 87, 54, 68, 77, 82, 58,
    70, 78, 83, 77, 82, 89, 90, 82, 86, 90, 91, 58, 70, 82, 86, 60, 72,
    83, 87, 78, 83, 90, 91, 83, 87, 91, 92, 40, 45, 58, 69, 65, 70, 80,
    83, 58, 69, 78, 97, 105, 106, 109, 110, 65, 70, 105, 106, 98, 100,
    111, 112, 80, 83, 109, 110, 111, 112, 116, 117, 68, 121, 82, 122,
    105, 123, 109, 124, 82, 122, 90, 126, 127, 128, 129, 130, 105, 123,
    127, 128, 127, 128, 131, 132, 109, 124, 129, 130, 131, 132, 133, 134,
    13, 52, 16, 53, 39, 57, 41, 59, 20, 58, 21, 59, 45, 70, 46, 71, 52,
    75, 58, 80, 57, 76, 66, 81, 58, 82, 60, 83, 70, 86, 72, 87, 16, 58,
    18, 59, 41, 66, 43, 67, 21, 60, 22, 61, 46, 72, 47, 73, 53, 80, 59,
    81, 59, 81, 67, 85, 59, 83, 61, 84, 71, 87, 73, 88, 17, 57, 21, 59,
    57, 76, 59, 79, 27, 66, 28, 67, 70, 100, 71, 101, 75, 108, 82, 109,
    108, 114, 109, 115, 80, 109, 83, 110, 111, 116, 112, 117, 27, 70, 28,
    71, 66, 100, 67, 101, 29, 72, 30, 73, 72, 102, 73, 103, 80, 111, 83,
    112, 109, 116, 110, 117, 81, 112, 84, 113, 112, 118, 113, 119, 40,
    65, 58, 80, 45, 70, 69, 83, 68, 105, 82, 109, 121, 123, 122, 124, 65,
    98, 105, 111, 70, 100, 106, 112, 105, 127, 127, 131, 123, 128, 128,
    132, 58, 105, 78, 109, 69, 106, 97, 110, 82, 127, 90, 129, 122, 128,
    126, 130, 80, 111, 109, 116, 83, 112, 110, 117, 109, 131, 129, 133,
    124, 132, 130, 134, 48, 70, 60, 83, 70, 86, 83, 91, 70, 106, 83, 110,
    123, 128, 124, 130, 98, 111, 127, 131, 111, 116, 131, 133, 111, 131,
    131, 143, 144, 145, 145, 146, 70, 123, 83, 124, 106, 128, 110, 130,
    86, 128, 91, 130, 128, 141, 130, 142, 111, 144, 131, 145, 131, 145,
    143, 146, 116, 145, 133, 146, 145, 147, 146, 148, 13, 52, 20, 58, 52,
    75, 58, 82, 16, 53, 21, 59, 58, 80, 60, 83, 39, 57, 45, 70, 57, 76,
    70, 86, 41, 59, 46, 71, 66, 81, 72, 87, 16, 58, 21, 60, 53, 80, 59,
    83, 18, 59, 22, 61, 59, 81, 61, 84, 41, 66, 46, 72, 59, 81, 71, 87,
    43, 67, 47, 73, 67, 85, 73, 88, 17, 57, 27, 66, 75, 108, 80, 109, 21,
    59, 28, 67, 82, 109, 83, 110, 57, 76, 70, 100, 108, 114, 111, 116,
    59, 79, 71, 101, 109, 115, 112, 117, 27, 70, 29, 72, 80, 111, 81,
    112, 28, 71, 30, 73, 83, 112, 84, 113, 66, 100, 72, 102, 109, 116,
    112, 118, 67, 101, 73, 103, 110, 117, 113, 119, 40, 65, 68, 105, 65,
    98, 105, 127, 58, 80, 82, 109, 105, 111, 127, 131, 45, 70, 121, 123,
    70, 100, 123, 128, 69, 83, 122, 124, 106, 112, 128, 132, 58, 105, 82,
    127, 80, 111, 109, 131, 78, 109, 90, 129, 109, 116, 129, 133, 69,
    106, 122, 128, 83, 112, 124, 132, 97, 110, 126, 130, 110, 117, 130,
    134, 48, 70, 70, 106, 98, 111, 111, 131, 60, 83, 83, 110, 127, 131,
    131, 143, 70, 86, 123, 128, 111, 116, 144, 145, 83, 91, 124, 130,
    131, 133, 145, 146, 70, 123, 86, 128, 111, 144, 116, 145, 83, 124,
    91, 130, 131, 145, 133, 146, 106, 128, 128, 141, 131, 145, 145, 147,
    110, 130, 130, 142, 143, 146, 146, 148, 17, 75, 27, 80, 57, 108, 66,
    109, 27, 80, 29, 81, 70, 111, 72, 112, 57, 108, 70, 111, 76, 114,
    100, 116, 66, 109, 72, 112, 100, 116, 102, 118, 21, 82, 28, 83, 59,
    109, 67, 110, 28, 83, 30, 84, 71, 112, 73, 113, 59, 109, 71, 112, 79,
    115, 101, 117, 67, 110, 73, 113, 101, 117, 103, 119, 23, 76, 29, 81,
    76, 114, 81, 115, 29, 81, 31, 85, 86, 116, 87, 117, 76, 114, 86, 116,
    114, 137, 116, 138, 81, 115, 87, 117, 116, 138, 118, 139, 29, 86, 31,
    87, 81, 116, 85, 117, 31, 87, 32, 88, 87, 118, 88, 119, 81, 116, 87,
    118, 115, 138, 117, 139, 85, 117, 88, 119, 117, 139, 119, 140, 48,
    98, 70, 111, 70, 111, 106, 131, 70, 111, 86, 116, 123, 144, 128, 145,
    70, 111, 123, 144, 86, 116, 128, 145, 106, 131, 128, 145, 128, 145,
    141, 147, 60, 127, 83, 131, 83, 131, 110, 143, 83, 131, 91, 133, 124,
    145, 130, 146, 83, 131, 124, 145, 91, 133, 130, 146, 110, 143, 130,
    146, 130, 146, 142, 148, 62, 100, 72, 112, 100, 116, 112, 133, 72,
    112, 87, 117, 128, 145, 132, 146, 100, 116, 128, 145, 116, 138, 145,
    149, 112, 133, 132, 146, 145, 149, 147, 150, 72, 128, 87, 132, 112,
    145, 117, 146, 87, 132, 92, 134, 132, 147, 134, 148, 112, 145, 132,
    147, 133, 149, 146, 150, 117, 146, 134, 148, 146, 150, 148, 151, 4,
    13, 13, 17, 7, 16, 16, 21, 13, 17, 17, 23, 20, 27, 27, 29, 7, 16, 20,
    27, 9, 18, 21, 28, 16, 21, 27, 29, 21, 28, 29, 31, 7, 20, 16, 27, 9,
    21, 18, 28, 16, 27, 21, 29, 21, 29, 28, 31, 9, 21, 21, 29, 10, 22,
    22, 30, 18, 28, 28, 31, 22, 30, 30, 32, 35, 40, 40, 48, 54, 58, 58,
    60, 40, 48, 48, 62, 68, 70, 70, 72, 54, 58, 68, 70, 77, 78, 82, 83,
    58, 60, 70, 72, 82, 83, 86, 87, 54, 68, 58, 70, 77, 82, 78, 83, 58,
    70, 60, 72, 82, 86, 83, 87, 77, 82, 82, 86, 89, 90, 90, 91, 78, 83,
    83, 87, 90, 91, 91, 92, 13, 39, 52, 57, 16, 41, 53, 59, 52, 57, 75,
    76, 58, 66, 80, 81, 16, 41, 58, 66, 18, 43, 59, 67, 53, 59, 80, 81,
    59, 67, 81, 85, 20, 45, 58, 70, 21, 46, 59, 71, 58, 70, 82, 86, 60,
    72, 83, 87, 21, 46, 60, 72, 22, 47, 61, 73, 59, 71, 83, 87, 61, 73,
    84, 88, 40, 45, 65, 70, 58, 69, 80, 83, 65, 70, 98, 100, 105, 106,
    111, 112, 58, 69, 105, 106, 78, 97, 109, 110, 80, 83, 111, 112, 109,
    110, 116, 117, 68, 121, 105, 123, 82, 122, 109, 124, 105, 123, 127,
    128, 127, 128, 131, 132, 82, 122, 127, 128, 90, 126, 129, 130, 109,
    124, 131, 132, 129, 130, 133, 134, 13, 52, 39, 57, 16, 53, 41, 59,
    52, 75, 57, 76, 58, 80, 66, 81, 20, 58, 45, 70, 21, 59, 46, 71, 58,
    82, 70, 86, 60, 83, 72, 87, 16, 58, 41, 66, 18, 59, 43, 67, 53, 80,
    59, 81, 59, 81, 67, 85, 21, 60, 46, 72, 22, 61, 47, 73, 59, 83, 71,
    87, 61, 84, 73, 88, 40, 65, 45, 70, 58, 80, 69, 83, 65, 98, 70, 100,
    105, 111, 106, 112, 68, 105, 121, 123, 82, 109, 122, 124, 105, 127,
    123, 128, 127, 131, 128, 132, 58, 105, 69, 106, 78, 109, 97, 110, 80,
    111, 83, 112, 109, 116, 110, 117, 82, 127, 122, 128, 90, 129, 126,
    130, 109, 131, 124, 132, 129, 133, 130, 134, 17, 57, 57, 76, 21, 59,
    59, 79, 75, 108, 108, 114, 82, 109, 109, 115, 27, 66, 70, 100, 28,
    67, 71, 101, 80, 109, 111, 116, 83, 110, 112, 117, 27, 70, 66, 100,
    28, 71, 67, 101, 80, 111, 109, 116, 83, 112, 110, 117, 29, 72, 72,
    102, 30, 73, 73, 103, 81, 112, 112, 118, 84, 113, 113, 119, 48, 70,
    70, 86, 60, 83, 83, 91, 98, 111, 111, 116, 127, 131, 131, 133, 70,
    106, 123, 128, 83, 110, 124, 130, 111, 131, 144, 145, 131, 143, 145,
    146, 70, 123, 106, 128, 83, 124, 110, 130, 111, 144, 131, 145, 131,
    145, 143, 146, 86, 128, 128, 141, 91, 130, 130, 142, 116, 145, 145,
    147, 133, 146, 146, 148, 13, 52, 52, 75, 20, 58, 58, 82, 39, 57, 57,
    76, 45, 70, 70, 86, 16, 53, 58, 80, 21, 59, 60, 83, 41, 59, 66, 81,
    46, 71, 72, 87, 16, 58, 53, 80, 21, 60, 59, 83, 41, 66, 59, 81, 46,
    72, 71, 87, 18, 59, 59, 81, 22, 61, 61, 84, 43, 67, 67, 85, 47, 73,
    73, 88, 40, 65, 65, 98, 68, 105, 105, 127, 45, 70, 70, 100, 121, 123,
    123, 128, 58, 80, 105, 111, 82, 109, 127, 131, 69, 83, 106, 112, 122,
    124, 128, 132, 58, 105, 80, 111, 82, 127, 109, 131, 69, 106, 83, 112,
    122, 128, 124, 132, 78, 109, 109, 116, 90, 129, 129, 133, 97, 110,
    110, 117, 126, 130, 130, 134, 17, 57, 75, 108, 27, 66, 80, 109, 57,
    76, 108, 114, 70, 100, 111, 116, 21, 59, 82, 109, 28, 67, 83, 110,
    59, 79, 109, 115, 71, 101, 112, 117, 27, 70, 80, 111, 29, 72, 81,
    112, 66, 100, 109, 116, 72, 102, 112, 118, 28, 71, 83, 112, 30, 73,
    84, 113, 67, 101, 110, 117, 73, 103, 113, 119, 48, 70, 98, 111, 70,
    106, 111, 131, 70, 86, 111, 116, 123, 128, 144, 145, 60, 83, 127,
    131, 83, 110, 131, 143, 83, 91, 131, 133, 124, 130, 145, 146, 70,
    123, 111, 144, 86, 128, 116, 145, 106, 128, 131, 145, 128, 141, 145,
    147, 83, 124, 131, 145, 91, 130, 133, 146, 110, 130, 143, 146, 130,
    142, 146, 148, 17, 75, 57, 108, 27, 80, 66, 109, 57, 108, 76, 114,
    70, 111, 100, 116, 27, 80, 70, 111, 29, 81, 72, 112, 66, 109, 100,
    116, 72, 112, 102, 118, 21, 82, 59, 109, 28, 83, 67, 110, 59, 109,
    79, 115, 71, 112, 101, 117, 28, 83, 71, 112, 30, 84, 73, 113, 67,
    110, 101, 117, 73, 113, 103, 119, 48, 98, 70, 111, 70, 111, 106, 131,
    70, 111, 86, 116, 123, 144, 128, 145, 70, 111, 123, 144, 86, 116,
    128, 145, 106, 131, 128, 145, 128, 145, 141, 147, 60, 127, 83, 131,
    83, 131, 110, 143, 83, 131, 91, 133, 124, 145, 130, 146, 83, 131,
    124, 145, 91, 133, 130, 146, 110, 143, 130, 146, 130, 146, 142, 148,
    23, 76, 76, 114, 29, 81, 81, 115, 76, 114, 114, 137, 86, 116, 116,
    138, 29, 81, 86, 116, 31, 85, 87, 117, 81, 115, 116, 138, 87, 117,
    118, 139, 29, 86, 81, 116, 31, 87, 85, 117, 81, 116, 115, 138, 87,
    118, 117, 139, 31, 87, 87, 118, 32, 88, 88, 119, 85, 117, 117, 139,
    88, 119, 119, 140, 62, 100, 100, 116, 72, 112, 112, 133, 100, 116,
    116, 138, 128, 145, 145, 149, 72, 112, 128, 145, 87, 117, 132, 146,
    112, 133, 145, 149, 132, 146, 147, 150, 72, 128, 112, 145, 87, 132,
    117, 146, 112, 145, 133, 149, 132, 147, 146, 150, 87, 132, 132, 147,
    92, 134, 134, 148, 117, 146, 146, 150, 134, 148, 148, 151, 11, 14,
    14, 18, 14, 18, 18, 22, 14, 18, 18, 24, 25, 28, 28, 30, 14, 18, 25,
    28, 18, 24, 28, 30, 18, 22, 28, 30, 28, 30, 31, 32, 14, 25, 18, 28,
    18, 28, 24, 30, 18, 28, 22, 30, 28, 31, 30, 32, 18, 28, 28, 31, 22,
    30, 30, 32, 24, 30, 30, 32, 30, 32, 32, 33, 36, 41, 41, 49, 55, 59,
    59, 61, 41, 49, 49, 63, 69, 71, 71, 73, 55, 59, 69, 71, 78, 79, 83,
    84, 59, 61, 71, 73, 83, 84, 87, 88, 55, 69, 59, 71, 78, 83, 79, 84,
    59, 71, 61, 73, 83, 87, 84, 88, 78, 83, 83, 87, 90, 91, 91, 92, 79,
    84, 84, 88, 91, 92, 92, 93, 36, 41, 55, 59, 41, 49, 59, 61, 55, 59,
    78, 79, 69, 71, 83, 84, 41, 49, 69, 71, 49, 63, 71, 73, 59, 61, 83,
    84, 71, 73, 87, 88, 55, 69, 78, 83, 59, 71, 79, 84, 78, 83, 90, 91,
    83, 87, 91, 92, 59, 71, 83, 87, 61, 73, 84, 88, 79, 84, 91, 92, 84,
    88, 92, 93, 42, 46, 66, 71, 66, 71, 81, 84, 66, 71, 99, 101, 106,
    107, 112, 113, 66, 71, 106, 107, 99, 101, 112, 113, 81, 84, 112, 113,
    112, 113, 118, 119, 96, 122, 109, 124, 109, 124, 115, 125, 109, 124,
    129, 130, 131, 132, 133, 134, 109, 124, 131, 132, 129, 130, 133, 134,
    115, 125, 133, 134, 133, 134, 135, 136, 36, 55, 41, 59, 41, 59, 49,
    61, 55, 78, 59, 79, 69, 83, 71, 84, 55, 78, 69, 83, 59, 79, 71, 84,
    78, 90, 83, 91, 83, 91, 87, 92, 41, 69, 49, 71, 49, 71, 63, 73, 59,
    83, 61, 84, 71, 87, 73, 88, 59, 83, 71, 87, 61, 84, 73, 88, 79, 91,
    84, 92, 84, 92, 88, 93, 42, 66, 46, 71, 66, 81, 71, 84, 66, 99, 71,
    101, 106, 112, 107, 113, 96, 109, 122, 124, 109, 115, 124, 125, 109,
    129, 124, 130, 131, 133, 132, 134, 66, 106, 71, 107, 99, 112, 101,
    113, 81, 112, 84, 113, 112, 118, 113, 119, 109, 131, 124, 132, 129,
    133, 130, 134, 115, 133, 125, 134, 133, 135, 134, 136, 42, 66, 66,
    81, 46, 71, 71, 84, 96, 109, 109, 115, 122, 124, 124, 125, 66, 99,
    106, 112, 71, 101, 107, 113, 109, 129, 131, 133, 124, 130, 132, 134,
    66, 106, 99, 112, 71, 107, 101, 113, 109, 131, 129, 133, 124, 132,
    130, 134, 81, 112, 112, 118, 84, 113, 113, 119, 115, 133, 133, 135,
    125, 134, 134, 136, 62, 72, 72, 87, 72, 87, 87, 92, 100, 112, 112,
    117, 128, 132, 132, 134, 100, 112, 128, 132, 112, 117, 132, 134, 116,
    133, 145, 146, 145, 146, 147, 148, 100, 128, 112, 132, 112, 132, 117,
    134, 116, 145, 133, 146, 145, 147, 146, 148, 116, 145, 145, 147, 133,
    146, 146, 148, 138, 149, 149, 150, 149, 150, 150, 151, 36, 55, 55,
    78, 55, 78, 78, 90, 41, 59, 59, 79, 69, 83, 83, 91, 41, 59, 69, 83,
    59, 79, 83, 91, 49, 61, 71, 84, 71, 84, 87, 92, 41, 69, 59, 83, 59,
    83, 79, 91, 49, 71, 61, 84, 71, 87, 84, 92, 49, 71, 71, 87, 61, 84,
    84, 92, 63, 73, 73, 88, 73, 88, 88, 93, 42, 66, 66, 99, 96, 109, 109,
    129, 46, 71, 71, 101, 122, 124, 124, 130, 66, 81, 106, 112, 109, 115,
    131, 133, 71, 84, 107, 113, 124, 125, 132, 134, 66, 106, 81, 112,
    109, 131, 115, 133, 71, 107, 84, 113, 124, 132, 125, 134, 99, 112,
    112, 118, 129, 133, 133, 135, 101, 113, 113, 119, 130, 134, 134, 136,
    42, 66, 96, 109, 66, 99, 109, 129, 66, 81, 109, 115, 106, 112, 131,
    133, 46, 71, 122, 124, 71, 101, 124, 130, 71, 84, 124, 125, 107, 113,
    132, 134, 66, 106, 109, 131, 81, 112, 115, 133, 99, 112, 129, 133,
    112, 118, 133, 135, 71, 107, 124, 132, 84, 113, 125, 134, 101, 113,
    130, 134, 113, 119, 134, 136, 62, 72, 100, 112, 100, 112, 116, 133,
    72, 87, 112, 117, 128, 132, 145, 146, 72, 87, 128, 132, 112, 117,
    145, 146, 87, 92, 132, 134, 132, 134, 147, 148, 100, 128, 116, 145,
    116, 145, 138, 149, 112, 132, 133, 146, 145, 147, 149, 150, 112, 132,
    145, 147, 133, 146, 149, 150, 117, 134, 146, 148, 146, 148, 150, 151,
    42, 96, 66, 109, 66, 109, 99, 129, 66, 109, 81, 115, 106, 131, 112,
    133, 66, 109, 106, 131, 81, 115, 112, 133, 99, 129, 112, 133, 112,
    133, 118, 135, 46, 122, 71, 124, 71, 124, 101, 130, 71, 124, 84, 125,
    107, 132, 113, 134, 71, 124, 107, 132, 84, 125, 113, 134, 101, 130,
    113, 134, 113, 134, 119, 136, 62, 100, 72, 112, 100, 116, 112, 133,
    72, 112, 87, 117, 128, 145, 132, 146, 100, 116, 128, 145, 116, 138,
    145, 149, 112, 133, 132, 146, 145, 149, 147, 150, 72, 128, 87, 132,
    112, 145, 117, 146, 87, 132, 92, 134, 132, 147, 134, 148, 112, 145,
    132, 147, 133, 149, 146, 150, 117, 146, 134, 148, 146, 150, 148, 151,
    62, 100, 100, 116, 72, 112, 112, 133, 100, 116, 116, 138, 128, 145,
    145, 149, 72, 112, 128, 145, 87, 117, 132, 146, 112, 133, 145, 149,
    132, 146, 147, 150, 72, 128, 112, 145, 87, 132, 117, 146, 112, 145,
    133, 149, 132, 147, 146, 150, 87, 132, 132, 147, 92, 134, 134, 148,
    117, 146, 146, 150, 134, 148, 148, 151, 94, 102, 102, 118, 102, 118,
    118, 135, 102, 118, 118, 139, 141, 147, 147, 150, 102, 118, 141, 147,
    118, 139, 147, 150, 118, 135, 147, 150, 147, 150, 152, 153, 102, 141,
    118, 147, 118, 147, 139, 150, 118, 147, 135, 150, 147, 152, 150, 153,
    118, 147, 147, 152, 135, 150, 150, 153, 139, 150, 150, 153, 150, 153,
    153, 154, 1, 5, 5, 12, 5, 12, 12, 19, 5, 12, 12, 35, 50, 51, 51, 54,
    5, 12, 50, 51, 12, 35, 51, 54, 12, 19, 51, 54, 51, 54, 74, 77, 5, 50,
    12, 51, 12, 51, 35, 54, 12, 51, 19, 54, 51, 74, 54, 77, 12, 51, 51,
    74, 19, 54, 54, 77, 35, 54, 54, 77, 54, 77, 77, 89, 2, 6, 6, 13, 12,
    15, 15, 20, 6, 13, 13, 36, 51, 52, 52, 55, 12, 15, 51, 52, 38, 40,
    56, 58, 15, 20, 52, 55, 56, 58, 75, 78, 12, 51, 15, 52, 38, 56, 40,
    58, 15, 52, 20, 55, 56, 75, 58, 78, 38, 56, 56, 75, 44, 68, 68, 82,
    40, 58, 58, 78, 68, 82, 82, 90, 2, 6, 12, 15, 6, 13, 15, 20, 12, 15,
    38, 40, 51, 52, 56, 58, 6, 13, 51, 52, 13, 36, 52, 55, 15, 20, 56,
    58, 52, 55, 75, 78, 12, 51, 38, 56, 15, 52, 40, 58, 38, 56, 44, 68,
    56, 75, 68, 82, 15, 52, 56, 75, 20, 55, 58, 78, 40, 58, 68, 82, 58,
    78, 82, 90, 4, 7, 13, 16, 13, 16, 17, 21, 13, 16, 39, 41, 52, 53, 57,
    59, 13, 16, 52, 53, 39, 41, 57, 59, 17, 21, 57, 59, 57, 59, 76, 79,
    35, 54, 40, 58, 40, 58, 48, 60, 40, 58, 45, 69, 65, 80, 70, 83, 40,
    58, 65, 80, 45, 69, 70, 83, 48, 60, 70, 83, 70, 83, 86, 91, 2, 12, 6,
    15, 6, 15, 13, 20, 12, 38, 15, 40, 51, 56, 52, 58, 12, 38, 51, 56,
    15, 40, 52, 58, 38, 44, 56, 68, 56, 68, 75, 82, 6, 51, 13, 52, 13, 52,
      36, 55, 15, 56, 20, 58, 52, 75, 55, 78, 15, 56, 52, 75, 20, 58, 55,
    78, 40, 68, 58, 82, 58, 82, 78, 90, 4, 13, 7, 16, 13, 17, 16, 21, 13,
    39, 16, 41, 52, 57, 53, 59, 35, 40, 54, 58, 40, 48, 58, 60, 40, 45,
    58, 69, 65, 70, 80, 83, 13, 52, 16, 53, 39, 57, 41, 59, 17, 57, 21,
    59, 57, 76, 59, 79, 40, 65, 58, 80, 45, 70, 69, 83, 48, 70, 60, 83,
    70, 86, 83, 91, 4, 13, 13, 17, 7, 16, 16, 21, 35, 40, 40, 48, 54, 58,
    58, 60, 13, 39, 52, 57, 16, 41, 53, 59, 40, 45, 65, 70, 58, 69, 80,
    83, 13, 52, 39, 57, 16, 53, 41, 59, 40, 65, 45, 70, 58, 80, 69, 83,
    17, 57, 57, 76, 21, 59, 59, 79, 48, 70, 70, 86, 60, 83, 83, 91, 11,
    14, 14, 18, 14, 18, 18, 22, 36, 41, 41, 49, 55, 59, 59, 61, 36, 41,
    55, 59, 41, 49, 59, 61, 42, 46, 66, 71, 66, 71, 81, 84, 36, 55, 41,
    59, 41, 59, 49, 61, 42, 66, 46, 71, 66, 81, 71, 84, 42, 66, 66, 81,
    46, 71, 71, 84, 62, 72, 72, 87, 72, 87, 87, 92, 2, 12, 12, 38, 12,
    38, 38, 44, 6, 15, 15, 40, 51, 56, 56, 68, 6, 15, 51, 56, 15, 40, 56,
    68, 13, 20, 52, 58, 52, 58, 75, 82, 6, 51, 15, 56, 15, 56, 40, 68,
    13, 52, 20, 58, 52, 75, 58, 82, 13, 52, 52, 75, 20, 58, 58, 82, 36,
    55, 55, 78, 55, 78, 78, 90, 4, 13, 13, 39, 35, 40, 40, 45, 7, 16, 16,
    41, 54, 58, 58, 69, 13, 17, 52, 57, 40, 48, 65, 70, 16, 21, 53, 59,
    58, 60, 80, 83, 13, 52, 17, 57, 40, 65, 48, 70, 16, 53, 21, 59, 58,
    80, 60, 83, 39, 57, 57, 76, 45, 70, 70, 86, 41, 59, 59, 79, 69, 83,
    83, 91, 4, 13, 35, 40, 13, 39, 40, 45, 13, 17, 40, 48, 52, 57, 65,
    70, 7, 16, 54, 58, 16, 41, 58, 69, 16, 21, 58, 60, 53, 59, 80, 83,
    13, 52, 40, 65, 17, 57, 48, 70, 39, 57, 45, 70, 57, 76, 70, 86, 16,
    53, 58, 80, 21, 59, 60, 83, 41, 59, 69, 83, 59, 79, 83, 91, 11, 14,
    36, 41, 36, 41, 42, 46, 14, 18, 41, 49, 55, 59, 66, 71, 14, 18, 55,
    59, 41, 49, 66, 71, 18, 22, 59, 61, 59, 61, 81, 84, 36, 55, 42, 66,
    42, 66, 62, 72, 41, 59, 46, 71, 66, 81, 72, 87, 41, 59, 66, 81, 46,
    71, 72, 87, 49, 61, 71, 84, 71, 84, 87, 92, 4, 35, 13, 40, 13, 40,
    39, 45, 13, 40, 17, 48, 52, 65, 57, 70, 13, 40, 52, 65, 17, 48, 57,
    70, 39, 45, 57, 70, 57, 70, 76, 86, 7, 54, 16, 58, 16, 58, 41, 69,
    16, 58, 21, 60, 53, 80, 59, 83, 16, 58, 53, 80, 21, 60, 59, 83, 41,
    69, 59, 83, 59, 83, 79, 91, 11, 36, 14, 41, 36, 42, 41, 46, 14, 41,
    18, 49, 55, 66, 59, 71, 36, 42, 55, 66, 42, 62, 66, 72, 41, 46, 59,
    71, 66, 72, 81, 87, 14, 55, 18, 59, 41, 66, 49, 71, 18, 59, 22, 61,
    59, 81, 61, 84, 41, 66, 59, 81, 46, 72, 71, 87, 49, 71, 61, 84, 71,
    87, 84, 92, 11, 36, 36, 42, 14, 41, 41, 46, 36, 42, 42, 62, 55, 66,
    66, 72, 14, 41, 55, 66, 18, 49, 59, 71, 41, 46, 66, 72, 59, 71, 81,
    87, 14, 55, 41, 66, 18, 59, 49, 71, 41, 66, 46, 72, 59, 81, 71, 87,
    18, 59, 59, 81, 22, 61, 61, 84, 49, 71, 71, 87, 61, 84, 84, 92, 34,
    37, 37, 43, 37, 43, 43, 47, 37, 43, 43, 63, 64, 67, 67, 73, 37, 43,
    64, 67, 43, 63, 67, 73, 43, 47, 67, 73, 67, 73, 85, 88, 37, 64, 43,
    67, 43, 67, 63, 73, 43, 67, 47, 73, 67, 85, 73, 88, 43, 67, 67, 85,
    47, 73, 73, 88, 63, 73, 73, 88, 73, 88, 88, 93, 2, 6, 6, 13, 12, 15,
    15, 20, 6, 13, 13, 36, 51, 52, 52, 55, 12, 15, 51, 52, 38, 40, 56,
    58, 15, 20, 52, 55, 56, 58, 75, 78, 12, 51, 15, 52, 38, 56, 40, 58,
    15, 52, 20, 55, 56, 75, 58, 78, 38, 56, 56, 75, 44, 68, 68, 82, 40,
    58, 58, 78, 68, 82, 82, 90, 3, 7, 7, 14, 19, 20, 20, 25, 7, 14, 14,
    37, 54, 55, 55, 64, 19, 20, 54, 55, 44, 45, 68, 69, 20, 25, 55, 64,
    68, 69, 96, 97, 19, 54, 20, 55, 44, 68, 45, 69, 20, 55, 25, 64, 68,
    96, 69, 97, 44, 68, 68, 96, 120, 121, 121, 122, 45, 69, 69, 97, 121,
    122, 122, 126, 6, 8, 15, 17, 15, 17, 26, 27, 15, 17, 40, 42, 56, 57,
    65, 66, 15, 17, 56, 57, 40, 42, 65, 66, 26, 27, 65, 66, 65, 66, 98,
    99, 51, 74, 56, 75, 56, 75, 65, 80, 56, 75, 68, 96, 104, 108, 105,
    109, 56, 75, 104, 108, 68, 96, 105, 109, 65, 80, 105, 109, 105, 109,
    127, 129, 7, 9, 16, 18, 20, 21, 27, 28, 16, 18, 41, 43, 58, 59, 66,
    67, 20, 21, 58, 59, 45, 46, 70, 71, 27, 28, 66, 67, 70, 71, 100, 101,
    54, 77, 58, 78, 68, 82, 70, 83, 58, 78, 69, 97, 105, 109, 106, 110,
    68, 82, 105, 109, 121, 122, 123, 124, 70, 83, 106, 110, 123, 124,
    128, 130, 6, 15, 8, 17, 15, 26, 17, 27, 15, 40, 17, 42, 56, 65, 57,
    66, 51, 56, 74, 75, 56, 65, 75, 80, 56, 68, 75, 96, 104, 105, 108,
    109, 15, 56, 17, 57, 40, 65, 42, 66, 26, 65, 27, 66, 65, 98, 66, 99,
    56, 104, 75, 108, 68, 105, 96, 109, 65, 105, 80, 109, 105, 127, 109,
    129, 7, 16, 9, 18, 20, 27, 21, 28, 16, 41, 18, 43, 58, 66, 59, 67,
    54, 58, 77, 78, 68, 70, 82, 83, 58, 69, 78, 97, 105, 106, 109, 110,
    20, 58, 21, 59, 45, 70, 46, 71, 27, 66, 28, 67, 70, 100, 71, 101, 68,
    105, 82, 109, 121, 123, 122, 124, 70, 106, 83, 110, 123, 128, 124,
    130, 13, 17, 17, 23, 20, 27, 27, 29, 40, 48, 48, 62, 68, 70, 70, 72,
    52, 57, 75, 76, 58, 66, 80, 81, 65, 70, 98, 100, 105, 106, 111, 112,
    52, 75, 57, 76, 58, 80, 66, 81, 65, 98, 70, 100, 105, 111, 106, 112,
    75, 108, 108, 114, 82, 109, 109, 115, 98, 111, 111, 116, 127, 131,
    131, 133, 14, 18, 18, 24, 25, 28, 28, 30, 41, 49, 49, 63, 69, 71, 71,
    73, 55, 59, 78, 79, 69, 71, 83, 84, 66, 71, 99, 101, 106, 107, 112,
    113, 55, 78, 59, 79, 69, 83, 71, 84, 66, 99, 71, 101, 106, 112, 107,
    113, 96, 109, 109, 115, 122, 124, 124, 125, 100, 112, 112, 117, 128,
    132, 132, 134, 6, 15, 15, 40, 51, 56, 56, 68, 8, 17, 17, 42, 74, 75,
    75, 96, 15, 26, 56, 65, 56, 65, 104, 105, 17, 27, 57, 66, 75, 80,
    108, 109, 15, 56, 26, 65, 56, 104, 65, 105, 17, 57, 27, 66, 75, 108,
    80, 109, 40, 65, 65, 98, 68, 105, 105, 127, 42, 66, 66, 99, 96, 109,
    109, 129, 7, 16, 16, 41, 54, 58, 58, 69, 9, 18, 18, 43, 77, 78, 78,
    97, 20, 27, 58, 66, 68, 70, 105, 106, 21, 28, 59, 67, 82, 83, 109,
    110, 20, 58, 27, 66, 68, 105, 70, 106, 21, 59, 28, 67, 82, 109, 83,
    110, 45, 70, 70, 100, 121, 123, 123, 128, 46, 71, 71, 101, 122, 124,
    124, 130, 13, 17, 40, 48, 52, 57, 65, 70, 17, 23, 48, 62, 75, 76, 98,
    100, 20, 27, 68, 70, 58, 66, 105, 106, 27, 29, 70, 72, 80, 81, 111,
    112, 52, 75, 65, 98, 75, 108, 98, 111, 57, 76, 70, 100, 108, 114,
    111, 116, 58, 80, 105, 111, 82, 109, 127, 131, 66, 81, 106, 112, 109,
    115, 131, 133, 14, 18, 41, 49, 55, 59, 66, 71, 18, 24, 49, 63, 78,
    79, 99, 101, 25, 28, 69, 71, 69, 71, 106, 107, 28, 30, 71, 73, 83,
    84, 112, 113, 55, 78, 66, 99, 96, 109, 100, 112, 59, 79, 71, 101,
    109, 115, 112, 117, 69, 83, 106, 112, 122, 124, 128, 132, 71, 84,
    107, 113, 124, 125, 132, 134, 13, 40, 17, 48, 52, 65, 57, 70, 17, 48,
    23, 62, 75, 98, 76, 100, 52, 65, 75, 98, 75, 98, 108, 111, 57, 70,
    76, 100, 108, 111, 114, 116, 20, 68, 27, 70, 58, 105, 66, 106, 27,
    70, 29, 72, 80, 111, 81, 112, 58, 105, 80, 111, 82, 127, 109, 131,
    66, 106, 81, 112, 109, 131, 115, 133, 14, 41, 18, 49, 55, 66, 59, 71,
    18, 49, 24, 63, 78, 99, 79, 101, 55, 66, 78, 99, 96, 100, 109, 112,
    59, 71, 79, 101, 109, 112, 115, 117, 25, 69, 28, 71, 69, 106, 71,
    107, 28, 71, 30, 73, 83, 112, 84, 113, 69, 106, 83, 112, 122, 128,
    124, 132, 71, 107, 84, 113, 124, 132, 125, 134, 36, 42, 42, 62, 55,
    66, 66, 72, 42, 62, 62, 94, 96, 100, 100, 102, 55, 66, 96, 100, 78,
    99, 109, 112, 66, 72, 100, 102, 109, 112, 116, 118, 55, 96, 66, 100,
    78, 109, 99, 112, 66, 100, 72, 102, 109, 116, 112, 118, 78, 109, 109,
    116, 90, 129, 129, 133, 99, 112, 112, 118, 129, 133, 133, 135, 37,
    43, 43, 63, 64, 67, 67, 73, 43, 63, 63, 95, 97, 101, 101, 103, 64,
    67, 97, 101, 97, 101, 110, 113, 67, 73, 101, 103, 110, 113, 117, 119,
    64, 97, 67, 101, 97, 110, 101, 113, 67, 101, 73, 103, 110, 117, 113,
    119, 97, 110, 110, 117, 126, 130, 130, 134, 101, 113, 113, 119, 130,
    134, 134, 136, 2, 6, 12, 15, 6, 13, 15, 20, 12, 15, 38, 40, 51, 52,
    56, 58, 6, 13, 51, 52, 13, 36, 52, 55, 15, 20, 56, 58, 52, 55, 75,
    78, 12, 51, 38, 56, 15, 52, 40, 58, 38, 56, 44, 68, 56, 75, 68, 82,
    15, 52, 56, 75, 20, 55, 58, 78, 40, 58, 68, 82, 58, 78, 82, 90, 6, 8,
    15, 17, 15, 17, 26, 27, 15, 17, 40, 42, 56, 57, 65, 66, 15, 17, 56,
    57, 40, 42, 65, 66, 26, 27, 65, 66, 65, 66, 98, 99, 51, 74, 56, 75,
    56, 75, 65, 80, 56, 75, 68, 96, 104, 108, 105, 109, 56, 75, 104, 108,
    68, 96, 105, 109, 65, 80, 105, 109, 105, 109, 127, 129, 3, 7, 19, 20,
    7, 14, 20, 25, 19, 20, 44, 45, 54, 55, 68, 69, 7, 14, 54, 55, 14, 37,
    55, 64, 20, 25, 68, 69, 55, 64, 96, 97, 19, 54, 44, 68, 20, 55, 45,
    69, 44, 68, 120, 121, 68, 96, 121, 122, 20, 55, 68, 96, 25, 64, 69,
    97, 45, 69, 121, 122, 69, 97, 122, 126, 7, 9, 20, 21, 16, 18, 27, 28,
    20, 21, 45, 46, 58, 59, 70, 71, 16, 18, 58, 59, 41, 43, 66, 67, 27,
    28, 70, 71, 66, 67, 100, 101, 54, 77, 68, 82, 58, 78, 70, 83, 68, 82,
    121, 122, 105, 109, 123, 124, 58, 78, 105, 109, 69, 97, 106, 110, 70,
    83, 123, 124, 106, 110, 128, 130, 6, 15, 15, 26, 8, 17, 17, 27, 51,
    56, 56, 65, 74, 75, 75, 80, 15, 40, 56, 65, 17, 42, 57, 66, 56, 68,
    104, 105, 75, 96, 108, 109, 15, 56, 40, 65, 17, 57, 42, 66, 56, 104,
    68, 105, 75, 108, 96, 109, 26, 65, 65, 98, 27, 66, 66, 99, 65, 105,
    105, 127, 80, 109, 109, 129, 13, 17, 20, 27, 17, 23, 27, 29, 52, 57,
    58, 66, 75, 76, 80, 81, 40, 48, 68, 70, 48, 62, 70, 72, 65, 70, 105,
    106, 98, 100, 111, 112, 52, 75, 58, 80, 57, 76, 66, 81, 75, 108, 82,
    109, 108, 114, 109, 115, 65, 98, 105, 111, 70, 100, 106, 112, 98,
    111, 127, 131, 111, 116, 131, 133, 7, 16, 20, 27, 9, 18, 21, 28, 54,
    58, 68, 70, 77, 78, 82, 83, 16, 41, 58, 66, 18, 43, 59, 67, 58, 69,
    105, 106, 78, 97, 109, 110, 20, 58, 45, 70, 21, 59, 46, 71, 68, 105,
    121, 123, 82, 109, 122, 124, 27, 66, 70, 100, 28, 67, 71, 101, 70,
    106, 123, 128, 83, 110, 124, 130, 14, 18, 25, 28, 18, 24, 28, 30, 55,
    59, 69, 71, 78, 79, 83, 84, 41, 49, 69, 71, 49, 63, 71, 73, 66, 71,
    106, 107, 99, 101, 112, 113, 55, 78, 69, 83, 59, 79, 71, 84, 96, 109,
    122, 124, 109, 115, 124, 125, 66, 99, 106, 112, 71, 101, 107, 113,
    100, 112, 128, 132, 112, 117, 132, 134, 6, 15, 51, 56, 15, 40, 56,
    68, 15, 26, 56, 65, 56, 65, 104, 105, 8, 17, 74, 75, 17, 42, 75, 96,
    17, 27, 75, 80, 57, 66, 108, 109, 15, 56, 56, 104, 26, 65, 65, 105,
    40, 65, 68, 105, 65, 98, 105, 127, 17, 57, 75, 108, 27, 66, 80, 109,
    42, 66, 96, 109, 66, 99, 109, 129, 13, 17, 52, 57, 40, 48, 65, 70,
    20, 27, 58, 66, 68, 70, 105, 106, 17, 23, 75, 76, 48, 62, 98, 100,
    27, 29, 80, 81, 70, 72, 111, 112, 52, 75, 75, 108, 65, 98, 98, 111,
    58, 80, 82, 109, 105, 111, 127, 131, 57, 76, 108, 114, 70, 100, 111,
    116, 66, 81, 109, 115, 106, 112, 131, 133, 7, 16, 54, 58, 16, 41, 58,
    69, 20, 27, 68, 70, 58, 66, 105, 106, 9, 18, 77, 78, 18, 43, 78, 97,
    21, 28, 82, 83, 59, 67, 109, 110, 20, 58, 68, 105, 27, 66, 70, 106,
    45, 70, 121, 123, 70, 100, 123, 128, 21, 59, 82, 109, 28, 67, 83,
    110, 46, 71, 122, 124, 71, 101, 124, 130, 14, 18, 55, 59, 41, 49, 66,
    71, 25, 28, 69, 71, 69, 71, 106, 107, 18, 24, 78, 79, 49, 63, 99,
    101, 28, 30, 83, 84, 71, 73, 112, 113, 55, 78, 96, 109, 66, 99, 100,
    112, 69, 83, 122, 124, 106, 112, 128, 132, 59, 79, 109, 115, 71, 101,
    112, 117, 71, 84, 124, 125, 107, 113, 132, 134, 13, 40, 52, 65, 17,
    48, 57, 70, 52, 65, 75, 98, 75, 98, 108, 111, 17, 48, 75, 98, 23, 62,
    76, 100, 57, 70, 108, 111, 76, 100, 114, 116, 20, 68, 58, 105, 27,
    70, 66, 106, 58, 105, 82, 127, 80, 111, 109, 131, 27, 70, 80, 111,
    29, 72, 81, 112, 66, 106, 109, 131, 81, 112, 115, 133, 36, 42, 55,
    66, 42, 62, 66, 72, 55, 66, 78, 99, 96, 100, 109, 112, 42, 62, 96,
    100, 62, 94, 100, 102, 66, 72, 109, 112, 100, 102, 116, 118, 55, 96,
    78, 109, 66, 100, 99, 112, 78, 109, 90, 129, 109, 116, 129, 133, 66,
    100, 109, 116, 72, 102, 112, 118, 99, 112, 129, 133, 112, 118, 133,
    135, 14, 41, 55, 66, 18, 49, 59, 71, 55, 66, 96, 100, 78, 99, 109,
    112, 18, 49, 78, 99, 24, 63, 79, 101, 59, 71, 109, 112, 79, 101, 115,
    117, 25, 69, 69, 106, 28, 71, 71, 107, 69, 106, 122, 128, 83, 112,
    124, 132, 28, 71, 83, 112, 30, 73, 84, 113, 71, 107, 124, 132, 84,
    113, 125, 134, 37, 43, 64, 67, 43, 63, 67, 73, 64, 67, 97, 101, 97,
    101, 110, 113, 43, 63, 97, 101, 63, 95, 101, 103, 67, 73, 110, 113,
    101, 103, 117, 119, 64, 97, 97, 110, 67, 101, 101, 113, 97, 110, 126,
    130, 110, 117, 130, 134, 67, 101, 110, 117, 73, 103, 113, 119, 101,
    113, 130, 134, 113, 119, 134, 136, 4, 7, 13, 16, 13, 16, 17, 21, 13,
    16, 39, 41, 52, 53, 57, 59, 13, 16, 52, 53, 39, 41, 57, 59, 17, 21,
    57, 59, 57, 59, 76, 79, 35, 54, 40, 58, 40, 58, 48, 60, 40, 58, 45,
    69, 65, 80, 70, 83, 40, 58, 65, 80, 45, 69, 70, 83, 48, 60, 70, 83,
    70, 83, 86, 91, 7, 9, 16, 18, 20, 21, 27, 28, 16, 18, 41, 43, 58, 59,
    66, 67, 20, 21, 58, 59, 45, 46, 70, 71, 27, 28, 66, 67, 70, 71, 100,
    101, 54, 77, 58, 78, 68, 82, 70, 83, 58, 78, 69, 97, 105, 109, 106,
    110, 68, 82, 105, 109, 121, 122, 123, 124, 70, 83, 106, 110, 123,
    124, 128, 130, 7, 9, 20, 21, 16, 18, 27, 28, 20, 21, 45, 46, 58, 59,
    70, 71, 16, 18, 58, 59, 41, 43, 66, 67, 27, 28, 70, 71, 66, 67, 100,
    101, 54, 77, 68, 82, 58, 78, 70, 83, 68, 82, 121, 122, 105, 109, 123,
    124, 58, 78, 105, 109, 69, 97, 106, 110, 70, 83, 123, 124, 106, 110,
    128, 130, 9, 10, 21, 22, 21, 22, 29, 30, 21, 22, 46, 47, 60, 61, 72,
    73, 21, 22, 60, 61, 46, 47, 72, 73, 29, 30, 72, 73, 72, 73, 102, 103,
    77, 89, 82, 90, 82, 90, 86, 91, 82, 90, 122, 126, 127, 129, 128, 130,
    82, 90, 127, 129, 122, 126, 128, 130, 86, 91, 128, 130, 128, 130,
    141, 142, 13, 20, 17, 27, 17, 27, 23, 29, 52, 58, 57, 66, 75, 80, 76,
    81, 52, 58, 75, 80, 57, 66, 76, 81, 75, 82, 108, 109, 108, 109, 114,
    115, 40, 68, 48, 70, 48, 70, 62, 72, 65, 105, 70, 106, 98, 111, 100,
    112, 65, 105, 98, 111, 70, 106, 100, 112, 98, 127, 111, 131, 111,
    131, 116, 133, 16, 21, 21, 28, 27, 29, 29, 31, 53, 59, 59, 67, 80,
    81, 81, 85, 58, 60, 82, 83, 70, 72, 86, 87, 80, 83, 109, 110, 111,
    112, 116, 117, 58, 82, 60, 83, 70, 86, 72, 87, 80, 109, 83, 110, 111,
    116, 112, 117, 105, 127, 127, 131, 123, 128, 128, 132, 111, 131, 131,
    143, 144, 145, 145, 146, 16, 21, 27, 29, 21, 28, 29, 31, 58, 60, 70,
    72, 82, 83, 86, 87, 53, 59, 80, 81, 59, 67, 81, 85, 80, 83, 111, 112,
    109, 110, 116, 117, 58, 82, 70, 86, 60, 83, 72, 87, 105, 127, 123,
    128, 127, 131, 128, 132, 80, 109, 111, 116, 83, 110, 112, 117, 111,
    131, 144, 145, 131, 143, 145, 146, 18, 22, 28, 30, 28, 30, 31, 32,
    59, 61, 71, 73, 83, 84, 87, 88, 59, 61, 83, 84, 71, 73, 87, 88, 81,
    84, 112, 113, 112, 113, 118, 119, 78, 90, 83, 91, 83, 91, 87, 92,
    109, 129, 124, 130, 131, 133, 132, 134, 109, 129, 131, 133, 124, 130,
    132, 134, 116, 133, 145, 146, 145, 146, 147, 148, 13, 20, 52, 58, 52,
    58, 75, 82, 17, 27, 57, 66, 75, 80, 108, 109, 17, 27, 75, 80, 57, 66,
    108, 109, 23, 29, 76, 81, 76, 81, 114, 115, 40, 68, 65, 105, 65, 105,
    98, 127, 48, 70, 70, 106, 98, 111, 111, 131, 48, 70, 98, 111, 70,
    106, 111, 131, 62, 72, 100, 112, 100, 112, 116, 133, 16, 21, 53, 59,
    58, 60, 80, 83, 21, 28, 59, 67, 82, 83, 109, 110, 27, 29, 80, 81, 70,
    72, 111, 112, 29, 31, 81, 85, 86, 87, 116, 117, 58, 82, 80, 109, 105,
    127, 111, 131, 60, 83, 83, 110, 127, 131, 131, 143, 70, 86, 111, 116,
    123, 128, 144, 145, 72, 87, 112, 117, 128, 132, 145, 146, 16, 21, 58,
    60, 53, 59, 80, 83, 27, 29, 70, 72, 80, 81, 111, 112, 21, 28, 82, 83,
    59, 67, 109, 110, 29, 31, 86, 87, 81, 85, 116, 117, 58, 82, 105, 127,
    80, 109, 111, 131, 70, 86, 123, 128, 111, 116, 144, 145, 60, 83, 127,
    131, 83, 110, 131, 143, 72, 87, 128, 132, 112, 117, 145, 146, 18, 22,
    59, 61, 59, 61, 81, 84, 28, 30, 71, 73, 83, 84, 112, 113, 28, 30, 83,
    84, 71, 73, 112, 113, 31, 32, 87, 88, 87, 88, 118, 119, 78, 90, 109,
    129, 109, 129, 116, 133, 83, 91, 124, 130, 131, 133, 145, 146, 83,
    91, 131, 133, 124, 130, 145, 146, 87, 92, 132, 134, 132, 134, 147,
    148, 39, 45, 57, 70, 57, 70, 76, 86, 57, 70, 76, 100, 108, 111, 114,
    116, 57, 70, 108, 111, 76, 100, 114, 116, 76, 86, 114, 116, 114, 116,
    137, 138, 45, 121, 70, 123, 70, 123, 100, 128, 70, 123, 86, 128, 111,
    144, 116, 145, 70, 123, 111, 144, 86, 128, 116, 145, 100, 128, 116,
    145, 116, 145, 138, 149, 41, 46, 59, 71, 66, 72, 81, 87, 59, 71, 79,
    101, 109, 112, 115, 117, 66, 72, 109, 112, 100, 102, 116, 118, 81,
    87, 115, 117, 116, 118, 138, 139, 69, 122, 83, 124, 106, 128, 112,
    132, 83, 124, 91, 130, 131, 145, 133, 146, 106, 128, 131, 145, 128,
    141, 145, 147, 112, 132, 133, 146, 145, 147, 149, 150, 41, 46, 66,
    72, 59, 71, 81, 87, 66, 72, 100, 102, 109, 112, 116, 118, 59, 71,
    109, 112, 79, 101, 115, 117, 81, 87, 116, 118, 115, 117, 138, 139,
    69, 122, 106, 128, 83, 124, 112, 132, 106, 128, 128, 141, 131, 145,
    145, 147, 83, 124, 131, 145, 91, 130, 133, 146, 112, 132, 145, 147,
    133, 146, 149, 150, 43, 47, 67, 73, 67, 73, 85, 88, 67, 73, 101, 103,
    110, 113, 117, 119, 67, 73, 110, 113, 101, 103, 117, 119, 85, 88,
    117, 119, 117, 119, 139, 140, 97, 126, 110, 130, 110, 130, 117, 134,
    110, 130, 130, 142, 143, 146, 146, 148, 110, 130, 143, 146, 130, 142,
    146, 148, 117, 134, 146, 148, 146, 148, 150, 151, 2, 12, 6, 15, 6,
    15, 13, 20, 12, 38, 15, 40, 51, 56, 52, 58, 12, 38, 51, 56, 15, 40,
    52, 58, 38, 44, 56, 68, 56, 68, 75, 82, 6, 51, 13, 52, 13, 52, 36,
    55, 15, 56, 20, 58, 52, 75, 55, 78, 15, 56, 52, 75, 20, 58, 55, 78,
    40, 68, 58, 82, 58, 82, 78, 90, 6, 15, 8, 17, 15, 26, 17, 27, 15, 40,
    17, 42, 56, 65, 57, 66, 51, 56, 74, 75, 56, 65, 75, 80, 56, 68, 75,
    96, 104, 105, 108, 109, 15, 56, 17, 57, 40, 65, 42, 66, 26, 65, 27,
    66, 65, 98, 66, 99, 56, 104, 75, 108, 68, 105, 96, 109, 65, 105, 80,
    109, 105, 127, 109, 129, 6, 15, 15, 26, 8, 17, 17, 27, 51, 56, 56,
    65, 74, 75, 75, 80, 15, 40, 56, 65, 17, 42, 57, 66, 56, 68, 104, 105,
    75, 96, 108, 109, 15, 56, 40, 65, 17, 57, 42, 66, 56, 104, 68, 105,
    75, 108, 96, 109, 26, 65, 65, 98, 27, 66, 66, 99, 65, 105, 105, 127,
    80, 109, 109, 129, 13, 20, 17, 27, 17, 27, 23, 29, 52, 58, 57, 66,
    75, 80, 76, 81, 52, 58, 75, 80, 57, 66, 76, 81, 75, 82, 108, 109,
    108, 109, 114, 115, 40, 68, 48, 70, 48, 70, 62, 72, 65, 105, 70, 106,
    98, 111, 100, 112, 65, 105, 98, 111, 70, 106, 100, 112, 98, 127, 111,
    131, 111, 131, 116, 133, 3, 19, 7, 20, 7, 20, 14, 25, 19, 44, 20, 45,
    54, 68, 55, 69, 19, 44, 54, 68, 20, 45, 55, 69, 44, 120, 68, 121, 68,
    121, 96, 122, 7, 54, 14, 55, 14, 55, 37, 64, 20, 68, 25, 69, 55, 96,
    64, 97, 20, 68, 55, 96, 25, 69, 64, 97, 45, 121, 69, 122, 69, 122,
    97, 126, 7, 20, 9, 21, 16, 27, 18, 28, 20, 45, 21, 46, 58, 70, 59,
    71, 54, 68, 77, 82, 58, 70, 78, 83, 68, 121, 82, 122, 105, 123, 109,
    124, 16, 58, 18, 59, 41, 66, 43, 67, 27, 70, 28, 71, 66, 100, 67,
    101, 58, 105, 78, 109, 69, 106, 97, 110, 70, 123, 83, 124, 106, 128,
    110, 130, 7, 20, 16, 27, 9, 21, 18, 28, 54, 68, 58, 70, 77, 82, 78,
    83, 20, 45, 58, 70, 21, 46, 59, 71, 68, 121, 105, 123, 82, 122, 109,
    124, 16, 58, 41, 66, 18, 59, 43, 67, 58, 105, 69, 106, 78, 109, 97,
    110, 27, 70, 66, 100, 28, 71, 67, 101, 70, 123, 106, 128, 83, 124,
    110, 130, 14, 25, 18, 28, 18, 28, 24, 30, 55, 69, 59, 71, 78, 83, 79,
    84, 55, 69, 78, 83, 59, 71, 79, 84, 96, 122, 109, 124, 109, 124, 115,
    125, 41, 69, 49, 71, 49, 71, 63, 73, 66, 106, 71, 107, 99, 112, 101,
    113, 66, 106, 99, 112, 71, 107, 101, 113, 100, 128, 112, 132, 112,
    132, 117, 134, 6, 51, 15, 56, 15, 56, 40, 68, 15, 56, 26, 65, 56,
    104, 65, 105, 15, 56, 56, 104, 26, 65, 65, 105, 40, 68, 65, 105, 65,
    105, 98, 127, 8, 74, 17, 75, 17, 75, 42, 96, 17, 75, 27, 80, 57, 108,
    66, 109, 17, 75, 57, 108, 27, 80, 66, 109, 42, 96, 66, 109, 66, 109,
    99, 129, 13, 52, 17, 57, 40, 65, 48, 70, 20, 58, 27, 66, 68, 105, 70,
    106, 52, 75, 75, 108, 65, 98, 98, 111, 58, 82, 80, 109, 105, 127,
    111, 131, 17, 75, 23, 76, 48, 98, 62, 100, 27, 80, 29, 81, 70, 111,
    72, 112, 57, 108, 76, 114, 70, 111, 100, 116, 66, 109, 81, 115, 106,
    131, 112, 133, 13, 52, 40, 65, 17, 57, 48, 70, 52, 75, 65, 98, 75,
    108, 98, 111, 20, 58, 68, 105, 27, 66, 70, 106, 58, 82, 105, 127, 80,
    109, 111, 131, 17, 75, 48, 98, 23, 76, 62, 100, 57, 108, 70, 111, 76,
    114, 100, 116, 27, 80, 70, 111, 29, 81, 72, 112, 66, 109, 106, 131,
    81, 115, 112, 133, 36, 55, 42, 66, 42, 66, 62, 72, 55, 78, 66, 99,
    96, 109, 100, 112, 55, 78, 96, 109, 66, 99, 100, 112, 78, 90, 109,
    129, 109, 129, 116, 133, 42, 96, 62, 100, 62, 100, 94, 102, 66, 109,
    72, 112, 100, 116, 102, 118, 66, 109, 100, 116, 72, 112, 102, 118,
    99, 129, 112, 133, 112, 133, 118, 135, 7, 54, 16, 58, 16, 58, 41, 69,
    20, 68, 27, 70, 58, 105, 66, 106, 20, 68, 58, 105, 27, 70, 66, 106,
    45, 121, 70, 123, 70, 123, 100, 128, 9, 77, 18, 78, 18, 78, 43, 97,
    21, 82, 28, 83, 59, 109, 67, 110, 21, 82, 59, 109, 28, 83, 67, 110,
    46, 122, 71, 124, 71, 124, 101, 130, 14, 55, 18, 59, 41, 66, 49, 71,
    25, 69, 28, 71, 69, 106, 71, 107, 55, 96, 78, 109, 66, 100, 99, 112,
    69, 122, 83, 124, 106, 128, 112, 132, 18, 78, 24, 79, 49, 99, 63,
    101, 28, 83, 30, 84, 71, 112, 73, 113, 59, 109, 79, 115, 71, 112,
    101, 117, 71, 124, 84, 125, 107, 132, 113, 134, 14, 55, 41, 66, 18,
    59, 49, 71, 55, 96, 66, 100, 78, 109, 99, 112, 25, 69, 69, 106, 28,
    71, 71, 107, 69, 122, 106, 128, 83, 124, 112, 132, 18, 78, 49, 99,
    24, 79, 63, 101, 59, 109, 71, 112, 79, 115, 101, 117, 28, 83, 71,
    112, 30, 84, 73, 113, 71, 124, 107, 132, 84, 125, 113, 134, 37, 64,
    43, 67, 43, 67, 63, 73, 64, 97, 67, 101, 97, 110, 101, 113, 64, 97,
    97, 110, 67, 101, 101, 113, 97, 126, 110, 130, 110, 130, 117, 134,
    43, 97, 63, 101, 63, 101, 95, 103, 67, 110, 73, 113, 101, 117, 103,
    119, 67, 110, 101, 117, 73, 113, 103, 119, 101, 130, 113, 134, 113,
    134, 119, 136, 4, 13, 7, 16, 13, 17, 16, 21, 13, 39, 16, 41, 52, 57,
    53, 59, 35, 40, 54, 58, 40, 48, 58, 60, 40, 45, 58, 69, 65, 70, 80,
    83, 13, 52, 16, 53, 39, 57, 41, 59, 17, 57, 21, 59, 57, 76, 59, 79,
    40, 65, 58, 80, 45, 70, 69, 83, 48, 70, 60, 83, 70, 86, 83, 91, 7,
    16, 9, 18, 20, 27, 21, 28, 16, 41, 18, 43, 58, 66, 59, 67, 54, 58,
    77, 78, 68, 70, 82, 83, 58, 69, 78, 97, 105, 106, 109, 110, 20, 58,
    21, 59, 45, 70, 46, 71, 27, 66, 28, 67, 70, 100, 71, 101, 68, 105,
    82, 109, 121, 123, 122, 124, 70, 106, 83, 110, 123, 128, 124, 130,
    13, 17, 20, 27, 17, 23, 27, 29, 52, 57, 58, 66, 75, 76, 80, 81, 40,
    48, 68, 70, 48, 62, 70, 72, 65, 70, 105, 106, 98, 100, 111, 112, 52,
    75, 58, 80, 57, 76, 66, 81, 75, 108, 82, 109, 108, 114, 109, 115, 65,
    98, 105, 111, 70, 100, 106, 112, 98, 111, 127, 131, 111, 116, 131,
    133, 16, 21, 21, 28, 27, 29, 29, 31, 53, 59, 59, 67, 80, 81, 81, 85,
    58, 60, 82, 83, 70, 72, 86, 87, 80, 83, 109, 110, 111, 112, 116, 117,
    58, 82, 60, 83, 70, 86, 72, 87, 80, 109, 83, 110, 111, 116, 112, 117,
    105, 127, 127, 131, 123, 128, 128, 132, 111, 131, 131, 143, 144, 145,
    145, 146, 7, 20, 9, 21, 16, 27, 18, 28, 20, 45, 21, 46, 58, 70, 59,
    71, 54, 68, 77, 82, 58, 70, 78, 83, 68, 121, 82, 122, 105, 123, 109,
    124, 16, 58, 18, 59, 41, 66, 43, 67, 27, 70, 28, 71, 66, 100, 67,
    101, 58, 105, 78, 109, 69, 106, 97, 110, 70, 123, 83, 124, 106, 128,
    110, 130, 9, 21, 10, 22, 21, 29, 22, 30, 21, 46, 22, 47, 60, 72, 61,
    73, 77, 82, 89, 90, 82, 86, 90, 91, 82, 122, 90, 126, 127, 128, 129,
    130, 21, 60, 22, 61, 46, 72, 47, 73, 29, 72, 30, 73, 72, 102, 73,
    103, 82, 127, 90, 129, 122, 128, 126, 130, 86, 128, 91, 130, 128,
    141, 130, 142, 16, 27, 21, 29, 21, 29, 28, 31, 58, 70, 60, 72, 82,
    86, 83, 87, 58, 70, 82, 86, 60, 72, 83, 87, 105, 123, 127, 128, 127,
    128, 131, 132, 53, 80, 59, 81, 59, 81, 67, 85, 80, 111, 83, 112, 109,
    116, 110, 117, 80, 111, 109, 116, 83, 112, 110, 117, 111, 144, 131,
    145, 131, 145, 143, 146, 18, 28, 22, 30, 28, 31, 30, 32, 59, 71, 61,
    73, 83, 87, 84, 88, 78, 83, 90, 91, 83, 87, 91, 92, 109, 124, 129,
    130, 131, 132, 133, 134, 59, 83, 61, 84, 71, 87, 73, 88, 81, 112, 84,
    113, 112, 118, 113, 119, 109, 131, 129, 133, 124, 132, 130, 134, 116,
    145, 133, 146, 145, 147, 146, 148, 13, 52, 20, 58, 52, 75, 58, 82,
    17, 57, 27, 66, 75, 108, 80, 109, 40, 65, 68, 105, 65, 98, 105, 127,
    48, 70, 70, 106, 98, 111, 111, 131, 17, 75, 27, 80, 57, 108, 66, 109,
    23, 76, 29, 81, 76, 114, 81, 115, 48, 98, 70, 111, 70, 111, 106, 131,
    62, 100, 72, 112, 100, 116, 112, 133, 16, 53, 21, 59, 58, 80, 60, 83,
    21, 59, 28, 67, 82, 109, 83, 110, 58, 80, 82, 109, 105, 111, 127,
    131, 60, 83, 83, 110, 127, 131, 131, 143, 27, 80, 29, 81, 70, 111,
    72, 112, 29, 81, 31, 85, 86, 116, 87, 117, 70, 111, 86, 116, 123,
    144, 128, 145, 72, 112, 87, 117, 128, 145, 132, 146, 39, 57, 45, 70,
    57, 76, 70, 86, 57, 76, 70, 100, 108, 114, 111, 116, 45, 70, 121,
    123, 70, 100, 123, 128, 70, 86, 123, 128, 111, 116, 144, 145, 57,
    108, 70, 111, 76, 114, 100, 116, 76, 114, 86, 116, 114, 137, 116,
    138, 70, 111, 123, 144, 86, 116, 128, 145, 100, 116, 128, 145, 116,
    138, 145, 149, 41, 59, 46, 71, 66, 81, 72, 87, 59, 79, 71, 101, 109,
    115, 112, 117, 69, 83, 122, 124, 106, 112, 128, 132, 83, 91, 124,
    130, 131, 133, 145, 146, 66, 109, 72, 112, 100, 116, 102, 118, 81,
    115, 87, 117, 116, 138, 118, 139, 106, 131, 128, 145, 128, 145, 141,
    147, 112, 133, 132, 146, 145, 149, 147, 150, 16, 58, 21, 60, 53, 80,
    59, 83, 27, 70, 29, 72, 80, 111, 81, 112, 58, 105, 82, 127, 80, 111,
    109, 131, 70, 123, 86, 128, 111, 144, 116, 145, 21, 82, 28, 83, 59,
    109, 67, 110, 29, 86, 31, 87, 81, 116, 85, 117, 60, 127, 83, 131, 83,
    131, 110, 143, 72, 128, 87, 132, 112, 145, 117, 146, 18, 59, 22, 61,
    59, 81, 61, 84, 28, 71, 30, 73, 83, 112, 84, 113, 78, 109, 90, 129,
    109, 116, 129, 133, 83, 124, 91, 130, 131, 145, 133, 146, 28, 83, 30,
    84, 71, 112, 73, 113, 31, 87, 32, 88, 87, 118, 88, 119, 83, 131, 91,
    133, 124, 145, 130, 146, 87, 132, 92, 134, 132, 147, 134, 148, 41,
    66, 46, 72, 59, 81, 71, 87, 66, 100, 72, 102, 109, 116, 112, 118, 69,
    106, 122, 128, 83, 112, 124, 132, 106, 128, 128, 141, 131, 145, 145,
    147, 59, 109, 71, 112, 79, 115, 101, 117, 81, 116, 87, 118, 115, 138,
    117, 139, 83, 131, 124, 145, 91, 133, 130, 146, 112, 145, 132, 147,
    133, 149, 146, 150, 43, 67, 47, 73, 67, 85, 73, 88, 67, 101, 73, 103,
    110, 117, 113, 119, 97, 110, 126, 130, 110, 117, 130, 134, 110, 130,
    130, 142, 143, 146, 146, 148, 67, 110, 73, 113, 101, 117, 103, 119,
    85, 117, 88, 119, 117, 139, 119, 140, 110, 143, 130, 146, 130, 146,
    142, 148, 117, 146, 134, 148, 146, 150, 148, 151, 4, 13, 13, 17, 7,
    16, 16, 21, 35, 40, 40, 48, 54, 58, 58, 60, 13, 39, 52, 57, 16, 41,
    53, 59, 40, 45, 65, 70, 58, 69, 80, 83, 13, 52, 39, 57, 16, 53, 41,
    59, 40, 65, 45, 70, 58, 80, 69, 83, 17, 57, 57, 76, 21, 59, 59, 79,
    48, 70, 70, 86, 60, 83, 83, 91, 13, 17, 17, 23, 20, 27, 27, 29, 40,
    48, 48, 62, 68, 70, 70, 72, 52, 57, 75, 76, 58, 66, 80, 81, 65, 70,
    98, 100, 105, 106, 111, 112, 52, 75, 57, 76, 58, 80, 66, 81, 65, 98,
    70, 100, 105, 111, 106, 112, 75, 108, 108, 114, 82, 109, 109, 115,
    98, 111, 111, 116, 127, 131, 131, 133, 7, 16, 20, 27, 9, 18, 21, 28,
    54, 58, 68, 70, 77, 78, 82, 83, 16, 41, 58, 66, 18, 43, 59, 67, 58,
    69, 105, 106, 78, 97, 109, 110, 20, 58, 45, 70, 21, 59, 46, 71, 68,
    105, 121, 123, 82, 109, 122, 124, 27, 66, 70, 100, 28, 67, 71, 101,
    70, 106, 123, 128, 83, 110, 124, 130, 16, 21, 27, 29, 21, 28, 29, 31,
    58, 60, 70, 72, 82, 83, 86, 87, 53, 59, 80, 81, 59, 67, 81, 85, 80,
    83, 111, 112, 109, 110, 116, 117, 58, 82, 70, 86, 60, 83, 72, 87,
    105, 127, 123, 128, 127, 131, 128, 132, 80, 109, 111, 116, 83, 110,
    112, 117, 111, 131, 144, 145, 131, 143, 145, 146, 7, 20, 16, 27, 9,
    21, 18, 28, 54, 68, 58, 70, 77, 82, 78, 83, 20, 45, 58, 70, 21, 46,
    59, 71, 68, 121, 105, 123, 82, 122, 109, 124, 16, 58, 41, 66, 18, 59,
    43, 67, 58, 105, 69, 106, 78, 109, 97, 110, 27, 70, 66, 100, 28, 71,
    67, 101, 70, 123, 106, 128, 83, 124, 110, 130, 16, 27, 21, 29, 21,
    29, 28, 31, 58, 70, 60, 72, 82, 86, 83, 87, 58, 70, 82, 86, 60, 72,
    83, 87, 105, 123, 127, 128, 127, 128, 131, 132, 53, 80, 59, 81, 59,
    81, 67, 85, 80, 111, 83, 112, 109, 116, 110, 117, 80, 111, 109, 116,
    83, 112, 110, 117, 111, 144, 131, 145, 131, 145, 143, 146, 9, 21, 21,
    29, 10, 22, 22, 30, 77, 82, 82, 86, 89, 90, 90, 91, 21, 46, 60, 72,
    22, 47, 61, 73, 82, 122, 127, 128, 90, 126, 129, 130, 21, 60, 46, 72,
    22, 61, 47, 73, 82, 127, 122, 128, 90, 129, 126, 130, 29, 72, 72,
    102, 30, 73, 73, 103, 86, 128, 128, 141, 91, 130, 130, 142, 18, 28,
    28, 31, 22, 30, 30, 32, 78, 83, 83, 87, 90, 91, 91, 92, 59, 71, 83,
    87, 61, 73, 84, 88, 109, 124, 131, 132, 129, 130, 133, 134, 59, 83,
    71, 87, 61, 84, 73, 88, 109, 131, 124, 132, 129, 133, 130, 134, 81,
    112, 112, 118, 84, 113, 113, 119, 116, 145, 145, 147, 133, 146, 146,
    148, 13, 52, 52, 75, 20, 58, 58, 82, 40, 65, 65, 98, 68, 105, 105,
    127, 17, 57, 75, 108, 27, 66, 80, 109, 48, 70, 98, 111, 70, 106, 111,
    131, 17, 75, 57, 108, 27, 80, 66, 109, 48, 98, 70, 111, 70, 111, 106,
    131, 23, 76, 76, 114, 29, 81, 81, 115, 62, 100, 100, 116, 72, 112,
    112, 133, 39, 57, 57, 76, 45, 70, 70, 86, 45, 70, 70, 100, 121, 123,
    123, 128, 57, 76, 108, 114, 70, 100, 111, 116, 70, 86, 111, 116, 123,
    128, 144, 145, 57, 108, 76, 114, 70, 111, 100, 116, 70, 111, 86, 116,
    123, 144, 128, 145, 76, 114, 114, 137, 86, 116, 116, 138, 100, 116,
    116, 138, 128, 145, 145, 149, 16, 53, 58, 80, 21, 59, 60, 83, 58, 80,
    105, 111, 82, 109, 127, 131, 21, 59, 82, 109, 28, 67, 83, 110, 60,
    83, 127, 131, 83, 110, 131, 143, 27, 80, 70, 111, 29, 81, 72, 112,
    70, 111, 123, 144, 86, 116, 128, 145, 29, 81, 86, 116, 31, 85, 87,
    117, 72, 112, 128, 145, 87, 117, 132, 146, 41, 59, 66, 81, 46, 71,
    72, 87, 69, 83, 106, 112, 122, 124, 128, 132, 59, 79, 109, 115, 71,
    101, 112, 117, 83, 91, 131, 133, 124, 130, 145, 146, 66, 109, 100,
    116, 72, 112, 102, 118, 106, 131, 128, 145, 128, 145, 141, 147, 81,
    115, 116, 138, 87, 117, 118, 139, 112, 133, 145, 149, 132, 146, 147,
    150, 16, 58, 53, 80, 21, 60, 59, 83, 58, 105, 80, 111, 82, 127, 109,
    131, 27, 70, 80, 111, 29, 72, 81, 112, 70, 123, 111, 144, 86, 128,
    116, 145, 21, 82, 59, 109, 28, 83, 67, 110, 60, 127, 83, 131, 83,
    131, 110, 143, 29, 86, 81, 116, 31, 87, 85, 117, 72, 128, 112, 145,
    87, 132, 117, 146, 41, 66, 59, 81, 46, 72, 71, 87, 69, 106, 83, 112,
    122, 128, 124, 132, 66, 100, 109, 116, 72, 102, 112, 118, 106, 128,
    131, 145, 128, 141, 145, 147, 59, 109, 79, 115, 71, 112, 101, 117,
    83, 131, 91, 133, 124, 145, 130, 146, 81, 116, 115, 138, 87, 118,
    117, 139, 112, 145, 133, 149, 132, 147, 146, 150, 18, 59, 59, 81, 22,
    61, 61, 84, 78, 109, 109, 116, 90, 129, 129, 133, 28, 71, 83, 112,
    30, 73, 84, 113, 83, 124, 131, 145, 91, 130, 133, 146, 28, 83, 71,
    112, 30, 84, 73, 113, 83, 131, 124, 145, 91, 133, 130, 146, 31, 87,
    87, 118, 32, 88, 88, 119, 87, 132, 132, 147, 92, 134, 134, 148, 43,
    67, 67, 85, 47, 73, 73, 88, 97, 110, 110, 117, 126, 130, 130, 134,
    67, 101, 110, 117, 73, 103, 113, 119, 110, 130, 143, 146, 130, 142,
    146, 148, 67, 110, 101, 117, 73, 113, 103, 119, 110, 143, 130, 146,
    130, 146, 142, 148, 85, 117, 117, 139, 88, 119, 119, 140, 117, 146,
    146, 150, 134, 148, 148, 151, 11, 14, 14, 18, 14, 18, 18, 22, 36, 41,
    41, 49, 55, 59, 59, 61, 36, 41, 55, 59, 41, 49, 59, 61, 42, 46, 66,
    71, 66, 71, 81, 84, 36, 55, 41, 59, 41, 59, 49, 61, 42, 66, 46, 71,
    66, 81, 71, 84, 42, 66, 66, 81, 46, 71, 71, 84, 62, 72, 72, 87, 72,
    87, 87, 92, 14, 18, 18, 24, 25, 28, 28, 30, 41, 49, 49, 63, 69, 71,
    71, 73, 55, 59, 78, 79, 69, 71, 83, 84, 66, 71, 99, 101, 106, 107,
    112, 113, 55, 78, 59, 79, 69, 83, 71, 84, 66, 99, 71, 101, 106, 112,
    107, 113, 96, 109, 109, 115, 122, 124, 124, 125, 100, 112, 112, 117,
    128, 132, 132, 134, 14, 18, 25, 28, 18, 24, 28, 30, 55, 59, 69, 71,
    78, 79, 83, 84, 41, 49, 69, 71, 49, 63, 71, 73, 66, 71, 106, 107, 99,
    101, 112, 113, 55, 78, 69, 83, 59, 79, 71, 84, 96, 109, 122, 124,
    109, 115, 124, 125, 66, 99, 106, 112, 71, 101, 107, 113, 100, 112,
    128, 132, 112, 117, 132, 134, 18, 22, 28, 30, 28, 30, 31, 32, 59, 61,
    71, 73, 83, 84, 87, 88, 59, 61, 83, 84, 71, 73, 87, 88, 81, 84, 112,
    113, 112, 113, 118, 119, 78, 90, 83, 91, 83, 91, 87, 92, 109, 129,
    124, 130, 131, 133, 132, 134, 109, 129, 131, 133, 124, 130, 132, 134,
    116, 133, 145, 146, 145, 146, 147, 148, 14, 25, 18, 28, 18, 28, 24,
    30, 55, 69, 59, 71, 78, 83, 79, 84, 55, 69, 78, 83, 59, 71, 79, 84,
    96, 122, 109, 124, 109, 124, 115, 125, 41, 69, 49, 71, 49, 71, 63,
    73, 66, 106, 71, 107, 99, 112, 101, 113, 66, 106, 99, 112, 71, 107,
    101, 113, 100, 128, 112, 132, 112, 132, 117, 134, 18, 28, 22, 30, 28,
    31, 30, 32, 59, 71, 61, 73, 83, 87, 84, 88, 78, 83, 90, 91, 83, 87,
    91, 92, 109, 124, 129, 130, 131, 132, 133, 134, 59, 83, 61, 84, 71,
    87, 73, 88, 81, 112, 84, 113, 112, 118, 113, 119, 109, 131, 129, 133,
    124, 132, 130, 134, 116, 145, 133, 146, 145, 147, 146, 148, 18, 28,
    28, 31, 22, 30, 30, 32, 78, 83, 83, 87, 90, 91, 91, 92, 59, 71, 83,
    87, 61, 73, 84, 88, 109, 124, 131, 132, 129, 130, 133, 134, 59, 83,
    71, 87, 61, 84, 73, 88, 109, 131, 124, 132, 129, 133, 130, 134, 81,
    112, 112, 118, 84, 113, 113, 119, 116, 145, 145, 147, 133, 146, 146,
    148, 24, 30, 30, 32, 30, 32, 32, 33, 79, 84, 84, 88, 91, 92, 92, 93,
    79, 84, 91, 92, 84, 88, 92, 93, 115, 125, 133, 134, 133, 134, 135,
    136, 79, 91, 84, 92, 84, 92, 88, 93, 115, 133, 125, 134, 133, 135,
    134, 136, 115, 133, 133, 135, 125, 134, 134, 136, 138, 149, 149, 150,
    149, 150, 150, 151, 36, 55, 55, 78, 55, 78, 78, 90, 42, 66, 66, 99,
    96, 109, 109, 129, 42, 66, 96, 109, 66, 99, 109, 129, 62, 72, 100,
    112, 100, 112, 116, 133, 42, 96, 66, 109, 66, 109, 99, 129, 62, 100,
    72, 112, 100, 116, 112, 133, 62, 100, 100, 116, 72, 112, 112, 133,
    94, 102, 102, 118, 102, 118, 118, 135, 41, 59, 59, 79, 69, 83, 83,
    91, 46, 71, 71, 101, 122, 124, 124, 130, 66, 81, 109, 115, 106, 112,
    131, 133, 72, 87, 112, 117, 128, 132, 145, 146, 66, 109, 81, 115,
    106, 131, 112, 133, 72, 112, 87, 117, 128, 145, 132, 146, 100, 116,
    116, 138, 128, 145, 145, 149, 102, 118, 118, 139, 141, 147, 147, 150,
    41, 59, 69, 83, 59, 79, 83, 91, 66, 81, 106, 112, 109, 115, 131, 133,
    46, 71, 122, 124, 71, 101, 124, 130, 72, 87, 128, 132, 112, 117, 145,
    146, 66, 109, 106, 131, 81, 115, 112, 133, 100, 116, 128, 145, 116,
    138, 145, 149, 72, 112, 128, 145, 87, 117, 132, 146, 102, 118, 141,
    147, 118, 139, 147, 150, 49, 61, 71, 84, 71, 84, 87, 92, 71, 84, 107,
    113, 124, 125, 132, 134, 71, 84, 124, 125, 107, 113, 132, 134, 87,
    92, 132, 134, 132, 134, 147, 148, 99, 129, 112, 133, 112, 133, 118,
    135, 112, 133, 132, 146, 145, 149, 147, 150, 112, 133, 145, 149, 132,
    146, 147, 150, 118, 135, 147, 150, 147, 150, 152, 153, 41, 69, 59,
    83, 59, 83, 79, 91, 66, 106, 81, 112, 109, 131, 115, 133, 66, 106,
    109, 131, 81, 112, 115, 133, 100, 128, 116, 145, 116, 145, 138, 149,
    46, 122, 71, 124, 71, 124, 101, 130, 72, 128, 87, 132, 112, 145, 117,
    146, 72, 128, 112, 145, 87, 132, 117, 146, 102, 141, 118, 147, 118,
    147, 139, 150, 49, 71, 61, 84, 71, 87, 84, 92, 71, 107, 84, 113, 124,
    132, 125, 134, 99, 112, 129, 133, 112, 118, 133, 135, 112, 132, 133,
    146, 145, 147, 149, 150, 71, 124, 84, 125, 107, 132, 113, 134, 87,
    132, 92, 134, 132, 147, 134, 148, 112, 145, 133, 149, 132, 147, 146,
    150, 118, 147, 135, 150, 147, 152, 150, 153, 49, 71, 71, 87, 61, 84,
    84, 92, 99, 112, 112, 118, 129, 133, 133, 135, 71, 107, 124, 132, 84,
    113, 125, 134, 112, 132, 145, 147, 133, 146, 149, 150, 71, 124, 107,
    132, 84, 125, 113, 134, 112, 145, 132, 147, 133, 149, 146, 150, 87,
    132, 132, 147, 92, 134, 134, 148, 118, 147, 147, 152, 135, 150, 150,
    153, 63, 73, 73, 88, 73, 88, 88, 93, 101, 113, 113, 119, 130, 134,
    134, 136, 101, 113, 130, 134, 113, 119, 134, 136, 117, 134, 146, 148,
    146, 148, 150, 151, 101, 130, 113, 134, 113, 134, 119, 136, 117, 146,
    134, 148, 146, 150, 148, 151, 117, 146, 146, 150, 134, 148, 148, 151,
    139, 150, 150, 153, 150, 153, 153, 154, 2, 12, 12, 38, 12, 38, 38,
    44, 6, 15, 15, 40, 51, 56, 56, 68, 6, 15, 51, 56, 15, 40, 56, 68, 13,
    20, 52, 58, 52, 58, 75, 82, 6, 51, 15, 56, 15, 56, 40, 68, 13, 52,
    20, 58, 52, 75, 58, 82, 13, 52, 52, 75, 20, 58, 58, 82, 36, 55, 55,
    78, 55, 78, 78, 90, 6, 15, 15, 40, 51, 56, 56, 68, 8, 17, 17, 42, 74,
    75, 75, 96, 15, 26, 56, 65, 56, 65, 104, 105, 17, 27, 57, 66, 75, 80,
    108, 109, 15, 56, 26, 65, 56, 104, 65, 105, 17, 57, 27, 66, 75, 108,
    80, 109, 40, 65, 65, 98, 68, 105, 105, 127, 42, 66, 66, 99, 96, 109,
    109, 129, 6, 15, 51, 56, 15, 40, 56, 68, 15, 26, 56, 65, 56, 65, 104,
    105, 8, 17, 74, 75, 17, 42, 75, 96, 17, 27, 75, 80, 57, 66, 108, 109,
    15, 56, 56, 104, 26, 65, 65, 105, 40, 65, 68, 105, 65, 98, 105, 127,
    17, 57, 75, 108, 27, 66, 80, 109, 42, 66, 96, 109, 66, 99, 109, 129,
    13, 20, 52, 58, 52, 58, 75, 82, 17, 27, 57, 66, 75, 80, 108, 109, 17,
    27, 75, 80, 57, 66, 108, 109, 23, 29, 76, 81, 76, 81, 114, 115, 40,
    68, 65, 105, 65, 105, 98, 127, 48, 70, 70, 106, 98, 111, 111, 131,
    48, 70, 98, 111, 70, 106, 111, 131, 62, 72, 100, 112, 100, 112, 116,
    133, 6, 51, 15, 56, 15, 56, 40, 68, 15, 56, 26, 65, 56, 104, 65, 105,
    15, 56, 56, 104, 26, 65, 65, 105, 40, 68, 65, 105, 65, 105, 98, 127,
    8, 74, 17, 75, 17, 75, 42, 96, 17, 75, 27, 80, 57, 108, 66, 109, 17,
    75, 57, 108, 27, 80, 66, 109, 42, 96, 66, 109, 66, 109, 99, 129, 13,
    52, 20, 58, 52, 75, 58, 82, 17, 57, 27, 66, 75, 108, 80, 109, 40, 65,
    68, 105, 65, 98, 105, 127, 48, 70, 70, 106, 98, 111, 111, 131, 17,
    75, 27, 80, 57, 108, 66, 109, 23, 76, 29, 81, 76, 114, 81, 115, 48,
    98, 70, 111, 70, 111, 106, 131, 62, 100, 72, 112, 100, 116, 112, 133,
    13, 52, 52, 75, 20, 58, 58, 82, 40, 65, 65, 98, 68, 105, 105, 127,
    17, 57, 75, 108, 27, 66, 80, 109, 48, 70, 98, 111, 70, 106, 111, 131,
    17, 75, 57, 108, 27, 80, 66, 109, 48, 98, 70, 111, 70, 111, 106, 131,
    23, 76, 76, 114, 29, 81, 81, 115, 62, 100, 100, 116, 72, 112, 112,
    133, 36, 55, 55, 78, 55, 78, 78, 90, 42, 66, 66, 99, 96, 109, 109,
    129, 42, 66, 96, 109, 66, 99, 109, 129, 62, 72, 100, 112, 100, 112,
    116, 133, 42, 96, 66, 109, 66, 109, 99, 129, 62, 100, 72, 112, 100,
    116, 112, 133, 62, 100, 100, 116, 72, 112, 112, 133, 94, 102, 102,
    118, 102, 118, 118, 135, 3, 19, 19, 44, 19, 44, 44, 120, 7, 20, 20,
    45, 54, 68, 68, 121, 7, 20, 54, 68, 20, 45, 68, 121, 14, 25, 55, 69,
    55, 69, 96, 122, 7, 54, 20, 68, 20, 68, 45, 121, 14, 55, 25, 69, 55,
    96, 69, 122, 14, 55, 55, 96, 25, 69, 69, 122, 37, 64, 64, 97, 64, 97,
    97, 126, 7, 20, 20, 45, 54, 68, 68, 121, 9, 21, 21, 46, 77, 82, 82,
    122, 16, 27, 58, 70, 58, 70, 105, 123, 18, 28, 59, 71, 78, 83, 109,
    124, 16, 58, 27, 70, 58, 105, 70, 123, 18, 59, 28, 71, 78, 109, 83,
    124, 41, 66, 66, 100, 69, 106, 106, 128, 43, 67, 67, 101, 97, 110,
    110, 130, 7, 20, 54, 68, 20, 45, 68, 121, 16, 27, 58, 70, 58, 70,
    105, 123, 9, 21, 77, 82, 21, 46, 82, 122, 18, 28, 78, 83, 59, 71,
    109, 124, 16, 58, 58, 105, 27, 70, 70, 123, 41, 66, 69, 106, 66, 100,
    106, 128, 18, 59, 78, 109, 28, 71, 83, 124, 43, 67, 97, 110, 67, 101,
    110, 130, 14, 25, 55, 69, 55, 69, 96, 122, 18, 28, 59, 71, 78, 83,
    109, 124, 18, 28, 78, 83, 59, 71, 109, 124, 24, 30, 79, 84, 79, 84,
    115, 125, 41, 69, 66, 106, 66, 106, 100, 128, 49, 71, 71, 107, 99,
    112, 112, 132, 49, 71, 99, 112, 71, 107, 112, 132, 63, 73, 101, 113,
    101, 113, 117, 134, 7, 54, 20, 68, 20, 68, 45, 121, 16, 58, 27, 70,
    58, 105, 70, 123, 16, 58, 58, 105, 27, 70, 70, 123, 41, 69, 66, 106,
    66, 106, 100, 128, 9, 77, 21, 82, 21, 82, 46, 122, 18, 78, 28, 83,
    59, 109, 71, 124, 18, 78, 59, 109, 28, 83, 71, 124, 43, 97, 67, 110,
    67, 110, 101, 130, 14, 55, 25, 69, 55, 96, 69, 122, 18, 59, 28, 71,
    78, 109, 83, 124, 41, 66, 69, 106, 66, 100, 106, 128, 49, 71, 71,
    107, 99, 112, 112, 132, 18, 78, 28, 83, 59, 109, 71, 124, 24, 79, 30,
    84, 79, 115, 84, 125, 49, 99, 71, 112, 71, 112, 107, 132, 63, 101,
    73, 113, 101, 117, 113, 134, 14, 55, 55, 96, 25, 69, 69, 122, 41, 66,
    66, 100, 69, 106, 106, 128, 18, 59, 78, 109, 28, 71, 83, 124, 49, 71,
    99, 112, 71, 107, 112, 132, 18, 78, 59, 109, 28, 83, 71, 124, 49, 99,
    71, 112, 71, 112, 107, 132, 24, 79, 79, 115, 30, 84, 84, 125, 63,
    101, 101, 117, 73, 113, 113, 134, 37, 64, 64, 97, 64, 97, 97, 126,
    43, 67, 67, 101, 97, 110, 110, 130, 43, 67, 97, 110, 67, 101, 110,
    130, 63, 73, 101, 113, 101, 113, 117, 134, 43, 97, 67, 110, 67, 110,
    101, 130, 63, 101, 73, 113, 101, 117, 113, 134, 63, 101, 101, 117,
    73, 113, 113, 134, 95, 103, 103, 119, 103, 119, 119, 136, 4, 13, 13,
    39, 35, 40, 40, 45, 7, 16, 16, 41, 54, 58, 58, 69, 13, 17, 52, 57,
    40, 48, 65, 70, 16, 21, 53, 59, 58, 60, 80, 83, 13, 52, 17, 57, 40,
    65, 48, 70, 16, 53, 21, 59, 58, 80, 60, 83, 39, 57, 57, 76, 45, 70,
    70, 86, 41, 59, 59, 79, 69, 83, 83, 91, 7, 16, 16, 41, 54, 58, 58,
    69, 9, 18, 18, 43, 77, 78, 78, 97, 20, 27, 58, 66, 68, 70, 105, 106,
    21, 28, 59, 67, 82, 83, 109, 110, 20, 58, 27, 66, 68, 105, 70, 106,
    21, 59, 28, 67, 82, 109, 83, 110, 45, 70, 70, 100, 121, 123, 123,
    128, 46, 71, 71, 101, 122, 124, 124, 130, 13, 17, 52, 57, 40, 48, 65,
    70, 20, 27, 58, 66, 68, 70, 105, 106, 17, 23, 75, 76, 48, 62, 98,
    100, 27, 29, 80, 81, 70, 72, 111, 112, 52, 75, 75, 108, 65, 98, 98,
    111, 58, 80, 82, 109, 105, 111, 127, 131, 57, 76, 108, 114, 70, 100,
    111, 116, 66, 81, 109, 115, 106, 112, 131, 133, 16, 21, 53, 59, 58,
    60, 80, 83, 21, 28, 59, 67, 82, 83, 109, 110, 27, 29, 80, 81, 70, 72,
    111, 112, 29, 31, 81, 85, 86, 87, 116, 117, 58, 82, 80, 109, 105,
    127, 111, 131, 60, 83, 83, 110, 127, 131, 131, 143, 70, 86, 111, 116,
    123, 128, 144, 145, 72, 87, 112, 117, 128, 132, 145, 146, 13, 52, 17,
    57, 40, 65, 48, 70, 20, 58, 27, 66, 68, 105, 70, 106, 52, 75, 75,
    108, 65, 98, 98, 111, 58, 82, 80, 109, 105, 127, 111, 131, 17, 75,
    23, 76, 48, 98, 62, 100, 27, 80, 29, 81, 70, 111, 72, 112, 57, 108,
    76, 114, 70, 111, 100, 116, 66, 109, 81, 115, 106, 131, 112, 133, 16,
    53, 21, 59, 58, 80, 60, 83, 21, 59, 28, 67, 82, 109, 83, 110, 58, 80,
    82, 109, 105, 111, 127, 131, 60, 83, 83, 110, 127, 131, 131, 143, 27,
    80, 29, 81, 70, 111, 72, 112, 29, 81, 31, 85, 86, 116, 87, 117, 70,
    111, 86, 116, 123, 144, 128, 145, 72, 112, 87, 117, 128, 145, 132,
    146, 39, 57, 57, 76, 45, 70, 70, 86, 45, 70, 70, 100, 121, 123, 123,
    128, 57, 76, 108, 114, 70, 100, 111, 116, 70, 86, 111, 116, 123, 128,
    144, 145, 57, 108, 76, 114, 70, 111, 100, 116, 70, 111, 86, 116, 123,
    144, 128, 145, 76, 114, 114, 137, 86, 116, 116, 138, 100, 116, 116,
    138, 128, 145, 145, 149, 41, 59, 59, 79, 69, 83, 83, 91, 46, 71, 71,
    101, 122, 124, 124, 130, 66, 81, 109, 115, 106, 112, 131, 133, 72,
    87, 112, 117, 128, 132, 145, 146, 66, 109, 81, 115, 106, 131, 112,
    133, 72, 112, 87, 117, 128, 145, 132, 146, 100, 116, 116, 138, 128,
    145, 145, 149, 102, 118, 118, 139, 141, 147, 147, 150, 7, 20, 20, 45,
    54, 68, 68, 121, 9, 21, 21, 46, 77, 82, 82, 122, 16, 27, 58, 70, 58,
    70, 105, 123, 18, 28, 59, 71, 78, 83, 109, 124, 16, 58, 27, 70, 58,
    105, 70, 123, 18, 59, 28, 71, 78, 109, 83, 124, 41, 66, 66, 100, 69,
    106, 106, 128, 43, 67, 67, 101, 97, 110, 110, 130, 9, 21, 21, 46, 77,
    82, 82, 122, 10, 22, 22, 47, 89, 90, 90, 126, 21, 29, 60, 72, 82, 86,
    127, 128, 22, 30, 61, 73, 90, 91, 129, 130, 21, 60, 29, 72, 82, 127,
    86, 128, 22, 61, 30, 73, 90, 129, 91, 130, 46, 72, 72, 102, 122, 128,
    128, 141, 47, 73, 73, 103, 126, 130, 130, 142, 16, 27, 58, 70, 58,
    70, 105, 123, 21, 29, 60, 72, 82, 86, 127, 128, 21, 29, 82, 86, 60,
    72, 127, 128, 28, 31, 83, 87, 83, 87, 131, 132, 53, 80, 80, 111, 80,
    111, 111, 144, 59, 81, 83, 112, 109, 116, 131, 145, 59, 81, 109, 116,
    83, 112, 131, 145, 67, 85, 110, 117, 110, 117, 143, 146, 18, 28, 59,
    71, 78, 83, 109, 124, 22, 30, 61, 73, 90, 91, 129, 130, 28, 31, 83,
    87, 83, 87, 131, 132, 30, 32, 84, 88, 91, 92, 133, 134, 59, 83, 81,
    112, 109, 131, 116, 145, 61, 84, 84, 113, 129, 133, 133, 146, 71, 87,
    112, 118, 124, 132, 145, 147, 73, 88, 113, 119, 130, 134, 146, 148,
    16, 58, 27, 70, 58, 105, 70, 123, 21, 60, 29, 72, 82, 127, 86, 128,
    53, 80, 80, 111, 80, 111, 111, 144, 59, 83, 81, 112, 109, 131, 116,
    145, 21, 82, 29, 86, 60, 127, 72, 128, 28, 83, 31, 87, 83, 131, 87,
    132, 59, 109, 81, 116, 83, 131, 112, 145, 67, 110, 85, 117, 110, 143,
    117, 146, 18, 59, 28, 71, 78, 109, 83, 124, 22, 61, 30, 73, 90, 129,
    91, 130, 59, 81, 83, 112, 109, 116, 131, 145, 61, 84, 84, 113, 129,
    133, 133, 146, 28, 83, 31, 87, 83, 131, 87, 132, 30, 84, 32, 88, 91,
    133, 92, 134, 71, 112, 87, 118, 124, 145, 132, 147, 73, 113, 88, 119,
    130, 146, 134, 148, 41, 66, 66, 100, 69, 106, 106, 128, 46, 72, 72,
    102, 122, 128, 128, 141, 59, 81, 109, 116, 83, 112, 131, 145, 71, 87,
    112, 118, 124, 132, 145, 147, 59, 109, 81, 116, 83, 131, 112, 145,
    71, 112, 87, 118, 124, 145, 132, 147, 79, 115, 115, 138, 91, 133,
    133, 149, 101, 117, 117, 139, 130, 146, 146, 150, 43, 67, 67, 101,
    97, 110, 110, 130, 47, 73, 73, 103, 126, 130, 130, 142, 67, 85, 110,
    117, 110, 117, 143, 146, 73, 88, 113, 119, 130, 134, 146, 148, 67,
    110, 85, 117, 110, 143, 117, 146, 73, 113, 88, 119, 130, 146, 134,
    148, 101, 117, 117, 139, 130, 146, 146, 150, 103, 119, 119, 140, 142,
    148, 148, 151, 4, 13, 35, 40, 13, 39, 40, 45, 13, 17, 40, 48, 52, 57,
    65, 70, 7, 16, 54, 58, 16, 41, 58, 69, 16, 21, 58, 60, 53, 59, 80,
    83, 13, 52, 40, 65, 17, 57, 48, 70, 39, 57, 45, 70, 57, 76, 70, 86,
    16, 53, 58, 80, 21, 59, 60, 83, 41, 59, 69, 83, 59, 79, 83, 91, 13,
    17, 40, 48, 52, 57, 65, 70, 17, 23, 48, 62, 75, 76, 98, 100, 20, 27,
    68, 70, 58, 66, 105, 106, 27, 29, 70, 72, 80, 81, 111, 112, 52, 75,
    65, 98, 75, 108, 98, 111, 57, 76, 70, 100, 108, 114, 111, 116, 58,
    80, 105, 111, 82, 109, 127, 131, 66, 81, 106, 112, 109, 115, 131,
    133, 7, 16, 54, 58, 16, 41, 58, 69, 20, 27, 68, 70, 58, 66, 105, 106,
    9, 18, 77, 78, 18, 43, 78, 97, 21, 28, 82, 83, 59, 67, 109, 110, 20,
    58, 68, 105, 27, 66, 70, 106, 45, 70, 121, 123, 70, 100, 123, 128,
    21, 59, 82, 109, 28, 67, 83, 110, 46, 71, 122, 124, 71, 101, 124,
    130, 16, 21, 58, 60, 53, 59, 80, 83, 27, 29, 70, 72, 80, 81, 111,
    112, 21, 28, 82, 83, 59, 67, 109, 110, 29, 31, 86, 87, 81, 85, 116,
    117, 58, 82, 105, 127, 80, 109, 111, 131, 70, 86, 123, 128, 111, 116,
    144, 145, 60, 83, 127, 131, 83, 110, 131, 143, 72, 87, 128, 132, 112,
    117, 145, 146, 13, 52, 40, 65, 17, 57, 48, 70, 52, 75, 65, 98, 75,
    108, 98, 111, 20, 58, 68, 105, 27, 66, 70, 106, 58, 82, 105, 127, 80,
    109, 111, 131, 17, 75, 48, 98, 23, 76, 62, 100, 57, 108, 70, 111, 76,
    114, 100, 116, 27, 80, 70, 111, 29, 81, 72, 112, 66, 109, 106, 131,
    81, 115, 112, 133, 39, 57, 45, 70, 57, 76, 70, 86, 57, 76, 70, 100,
    108, 114, 111, 116, 45, 70, 121, 123, 70, 100, 123, 128, 70, 86, 123,
    128, 111, 116, 144, 145, 57, 108, 70, 111, 76, 114, 100, 116, 76,
    114, 86, 116, 114, 137, 116, 138, 70, 111, 123, 144, 86, 116, 128,
    145, 100, 116, 128, 145, 116, 138, 145, 149, 16, 53, 58, 80, 21, 59,
    60, 83, 58, 80, 105, 111, 82, 109, 127, 131, 21, 59, 82, 109, 28, 67,
    83, 110, 60, 83, 127, 131, 83, 110, 131, 143, 27, 80, 70, 111, 29,
    81, 72, 112, 70, 111, 123, 144, 86, 116, 128, 145, 29, 81, 86, 116,
    31, 85, 87, 117, 72, 112, 128, 145, 87, 117, 132, 146, 41, 59, 69,
    83, 59, 79, 83, 91, 66, 81, 106, 112, 109, 115, 131, 133, 46, 71,
    122, 124, 71, 101, 124, 130, 72, 87, 128, 132, 112, 117, 145, 146,
    66, 109, 106, 131, 81, 115, 112, 133, 100, 116, 128, 145, 116, 138,
    145, 149, 72, 112, 128, 145, 87, 117, 132, 146, 102, 118, 141, 147,
    118, 139, 147, 150, 7, 20, 54, 68, 20, 45, 68, 121, 16, 27, 58, 70,
    58, 70, 105, 123, 9, 21, 77, 82, 21, 46, 82, 122, 18, 28, 78, 83, 59,
    71, 109, 124, 16, 58, 58, 105, 27, 70, 70, 123, 41, 66, 69, 106, 66,
    100, 106, 128, 18, 59, 78, 109, 28, 71, 83, 124, 43, 67, 97, 110, 67,
    101, 110, 130, 16, 27, 58, 70, 58, 70, 105, 123, 21, 29, 60, 72, 82,
    86, 127, 128, 21, 29, 82, 86, 60, 72, 127, 128, 28, 31, 83, 87, 83,
    87, 131, 132, 53, 80, 80, 111, 80, 111, 111, 144, 59, 81, 83, 112,
    109, 116, 131, 145, 59, 81, 109, 116, 83, 112, 131, 145, 67, 85, 110,
    117, 110, 117, 143, 146, 9, 21, 77, 82, 21, 46, 82, 122, 21, 29, 82,
    86, 60, 72, 127, 128, 10, 22, 89, 90, 22, 47, 90, 126, 22, 30, 90,
    91, 61, 73, 129, 130, 21, 60, 82, 127, 29, 72, 86, 128, 46, 72, 122,
    128, 72, 102, 128, 141, 22, 61, 90, 129, 30, 73, 91, 130, 47, 73,
    126, 130, 73, 103, 130, 142, 18, 28, 78, 83, 59, 71, 109, 124, 28,
    31, 83, 87, 83, 87, 131, 132, 22, 30, 90, 91, 61, 73, 129, 130, 30,
    32, 91, 92, 84, 88, 133, 134, 59, 83, 109, 131, 81, 112, 116, 145,
    71, 87, 124, 132, 112, 118, 145, 147, 61, 84, 129, 133, 84, 113, 133,
    146, 73, 88, 130, 134, 113, 119, 146, 148, 16, 58, 58, 105, 27, 70,
    70, 123, 53, 80, 80, 111, 80, 111, 111, 144, 21, 60, 82, 127, 29, 72,
    86, 128, 59, 83, 109, 131, 81, 112, 116, 145, 21, 82, 60, 127, 29,
    86, 72, 128, 59, 109, 83, 131, 81, 116, 112, 145, 28, 83, 83, 131,
    31, 87, 87, 132, 67, 110, 110, 143, 85, 117, 117, 146, 41, 66, 69,
    106, 66, 100, 106, 128, 59, 81, 83, 112, 109, 116, 131, 145, 46, 72,
    122, 128, 72, 102, 128, 141, 71, 87, 124, 132, 112, 118, 145, 147,
    59, 109, 83, 131, 81, 116, 112, 145, 79, 115, 91, 133, 115, 138, 133,
    149, 71, 112, 124, 145, 87, 118, 132, 147, 101, 117, 130, 146, 117,
    139, 146, 150, 18, 59, 78, 109, 28, 71, 83, 124, 59, 81, 109, 116,
    83, 112, 131, 145, 22, 61, 90, 129, 30, 73, 91, 130, 61, 84, 129,
    133, 84, 113, 133, 146, 28, 83, 83, 131, 31, 87, 87, 132, 71, 112,
    124, 145, 87, 118, 132, 147, 30, 84, 91, 133, 32, 88, 92, 134, 73,
    113, 130, 146, 88, 119, 134, 148, 43, 67, 97, 110, 67, 101, 110, 130,
    67, 85, 110, 117, 110, 117, 143, 146, 47, 73, 126, 130, 73, 103, 130,
    142, 73, 88, 130, 134, 113, 119, 146, 148, 67, 110, 110, 143, 85,
    117, 117, 146, 101, 117, 130, 146, 117, 139, 146, 150, 73, 113, 130,
    146, 88, 119, 134, 148, 103, 119, 142, 148, 119, 140, 148, 151, 11,
    14, 36, 41, 36, 41, 42, 46, 14, 18, 41, 49, 55, 59, 66, 71, 14, 18,
    55, 59, 41, 49, 66, 71, 18, 22, 59, 61, 59, 61, 81, 84, 36, 55, 42,
    66, 42, 66, 62, 72, 41, 59, 46, 71, 66, 81, 72, 87, 41, 59, 66, 81,
    46, 71, 72, 87, 49, 61, 71, 84, 71, 84, 87, 92, 14, 18, 41, 49, 55,
    59, 66, 71, 18, 24, 49, 63, 78, 79, 99, 101, 25, 28, 69, 71, 69, 71,
    106, 107, 28, 30, 71, 73, 83, 84, 112, 113, 55, 78, 66, 99, 96, 109,
    100, 112, 59, 79, 71, 101, 109, 115, 112, 117, 69, 83, 106, 112, 122,
    124, 128, 132, 71, 84, 107, 113, 124, 125, 132, 134, 14, 18, 55, 59,
    41, 49, 66, 71, 25, 28, 69, 71, 69, 71, 106, 107, 18, 24, 78, 79, 49,
    63, 99, 101, 28, 30, 83, 84, 71, 73, 112, 113, 55, 78, 96, 109, 66,
    99, 100, 112, 69, 83, 122, 124, 106, 112, 128, 132, 59, 79, 109, 115,
    71, 101, 112, 117, 71, 84, 124, 125, 107, 113, 132, 134, 18, 22, 59,
    61, 59, 61, 81, 84, 28, 30, 71, 73, 83, 84, 112, 113, 28, 30, 83, 84,
    71, 73, 112, 113, 31, 32, 87, 88, 87, 88, 118, 119, 78, 90, 109, 129,
    109, 129, 116, 133, 83, 91, 124, 130, 131, 133, 145, 146, 83, 91,
    131, 133, 124, 130, 145, 146, 87, 92, 132, 134, 132, 134, 147, 148,
    36, 55, 42, 66, 42, 66, 62, 72, 55, 78, 66, 99, 96, 109, 100, 112,
    55, 78, 96, 109, 66, 99, 100, 112, 78, 90, 109, 129, 109, 129, 116,
    133, 42, 96, 62, 100, 62, 100, 94, 102, 66, 109, 72, 112, 100, 116,
    102, 118, 66, 109, 100, 116, 72, 112, 102, 118, 99, 129, 112, 133,
    112, 133, 118, 135, 41, 59, 46, 71, 66, 81, 72, 87, 59, 79, 71, 101,
    109, 115, 112, 117, 69, 83, 122, 124, 106, 112, 128, 132, 83, 91,
    124, 130, 131, 133, 145, 146, 66, 109, 72, 112, 100, 116, 102, 118,
    81, 115, 87, 117, 116, 138, 118, 139, 106, 131, 128, 145, 128, 145,
    141, 147, 112, 133, 132, 146, 145, 149, 147, 150, 41, 59, 66, 81, 46,
    71, 72, 87, 69, 83, 106, 112, 122, 124, 128, 132, 59, 79, 109, 115,
    71, 101, 112, 117, 83, 91, 131, 133, 124, 130, 145, 146, 66, 109,
    100, 116, 72, 112, 102, 118, 106, 131, 128, 145, 128, 145, 141, 147,
    81, 115, 116, 138, 87, 117, 118, 139, 112, 133, 145, 149, 132, 146,
    147, 150, 49, 61, 71, 84, 71, 84, 87, 92, 71, 84, 107, 113, 124, 125,
    132, 134, 71, 84, 124, 125, 107, 113, 132, 134, 87, 92, 132, 134,
    132, 134, 147, 148, 99, 129, 112, 133, 112, 133, 118, 135, 112, 133,
    132, 146, 145, 149, 147, 150, 112, 133, 145, 149, 132, 146, 147, 150,
    118, 135, 147, 150, 147, 150, 152, 153, 14, 25, 55, 69, 55, 69, 96,
    122, 18, 28, 59, 71, 78, 83, 109, 124, 18, 28, 78, 83, 59, 71, 109,
    124, 24, 30, 79, 84, 79, 84, 115, 125, 41, 69, 66, 106, 66, 106, 100,
    128, 49, 71, 71, 107, 99, 112, 112, 132, 49, 71, 99, 112, 71, 107,
    112, 132, 63, 73, 101, 113, 101, 113, 117, 134, 18, 28, 59, 71, 78,
    83, 109, 124, 22, 30, 61, 73, 90, 91, 129, 130, 28, 31, 83, 87, 83,
    87, 131, 132, 30, 32, 84, 88, 91, 92, 133, 134, 59, 83, 81, 112, 109,
    131, 116, 145, 61, 84, 84, 113, 129, 133, 133, 146, 71, 87, 112, 118,
    124, 132, 145, 147, 73, 88, 113, 119, 130, 134, 146, 148, 18, 28, 78,
    83, 59, 71, 109, 124, 28, 31, 83, 87, 83, 87, 131, 132, 22, 30, 90,
    91, 61, 73, 129, 130, 30, 32, 91, 92, 84, 88, 133, 134, 59, 83, 109,
    131, 81, 112, 116, 145, 71, 87, 124, 132, 112, 118, 145, 147, 61, 84,
    129, 133, 84, 113, 133, 146, 73, 88, 130, 134, 113, 119, 146, 148,
    24, 30, 79, 84, 79, 84, 115, 125, 30, 32, 84, 88, 91, 92, 133, 134,
    30, 32, 91, 92, 84, 88, 133, 134, 32, 33, 92, 93, 92, 93, 135, 136,
    79, 91, 115, 133, 115, 133, 138, 149, 84, 92, 125, 134, 133, 135,
    149, 150, 84, 92, 133, 135, 125, 134, 149, 150, 88, 93, 134, 136,
    134, 136, 150, 151, 41, 69, 66, 106, 66, 106, 100, 128, 59, 83, 81,
    112, 109, 131, 116, 145, 59, 83, 109, 131, 81, 112, 116, 145, 79, 91,
    115, 133, 115, 133, 138, 149, 46, 122, 72, 128, 72, 128, 102, 141,
    71, 124, 87, 132, 112, 145, 118, 147, 71, 124, 112, 145, 87, 132,
    118, 147, 101, 130, 117, 146, 117, 146, 139, 150, 49, 71, 71, 107,
    99, 112, 112, 132, 61, 84, 84, 113, 129, 133, 133, 146, 71, 87, 124,
    132, 112, 118, 145, 147, 84, 92, 125, 134, 133, 135, 149, 150, 71,
    124, 87, 132, 112, 145, 118, 147, 84, 125, 92, 134, 133, 149, 135,
    150, 107, 132, 132, 147, 132, 147, 147, 152, 113, 134, 134, 148, 146,
    150, 150, 153, 49, 71, 99, 112, 71, 107, 112, 132, 71, 87, 112, 118,
    124, 132, 145, 147, 61, 84, 129, 133, 84, 113, 133, 146, 84, 92, 133,
    135, 125, 134, 149, 150, 71, 124, 112, 145, 87, 132, 118, 147, 107,
    132, 132, 147, 132, 147, 147, 152, 84, 125, 133, 149, 92, 134, 135,
    150, 113, 134, 146, 150, 134, 148, 150, 153, 63, 73, 101, 113, 101,
    113, 117, 134, 73, 88, 113, 119, 130, 134, 146, 148, 73, 88, 130,
    134, 113, 119, 146, 148, 88, 93, 134, 136, 134, 136, 150, 151, 101,
    130, 117, 146, 117, 146, 139, 150, 113, 134, 134, 148, 146, 150, 150,
    153, 113, 134, 146, 150, 134, 148, 150, 153, 119, 136, 148, 151, 148,
    151, 153, 154, 4, 35, 13, 40, 13, 40, 39, 45, 13, 40, 17, 48, 52, 65,
    57, 70, 13, 40, 52, 65, 17, 48, 57, 70, 39, 45, 57, 70, 57, 70, 76,
    86, 7, 54, 16, 58, 16, 58, 41, 69, 16, 58, 21, 60, 53, 80, 59, 83,
    16, 58, 53, 80, 21, 60, 59, 83, 41, 69, 59, 83, 59, 83, 79, 91, 13,
    40, 17, 48, 52, 65, 57, 70, 17, 48, 23, 62, 75, 98, 76, 100, 52, 65,
    75, 98, 75, 98, 108, 111, 57, 70, 76, 100, 108, 111, 114, 116, 20,
    68, 27, 70, 58, 105, 66, 106, 27, 70, 29, 72, 80, 111, 81, 112, 58,
    105, 80, 111, 82, 127, 109, 131, 66, 106, 81, 112, 109, 131, 115,
    133, 13, 40, 52, 65, 17, 48, 57, 70, 52, 65, 75, 98, 75, 98, 108,
    111, 17, 48, 75, 98, 23, 62, 76, 100, 57, 70, 108, 111, 76, 100, 114,
    116, 20, 68, 58, 105, 27, 70, 66, 106, 58, 105, 82, 127, 80, 111,
    109, 131, 27, 70, 80, 111, 29, 72, 81, 112, 66, 106, 109, 131, 81,
    112, 115, 133, 39, 45, 57, 70, 57, 70, 76, 86, 57, 70, 76, 100, 108,
    111, 114, 116, 57, 70, 108, 111, 76, 100, 114, 116, 76, 86, 114, 116,
    114, 116, 137, 138, 45, 121, 70, 123, 70, 123, 100, 128, 70, 123, 86,
    128, 111, 144, 116, 145, 70, 123, 111, 144, 86, 128, 116, 145, 100,
    128, 116, 145, 116, 145, 138, 149, 7, 54, 16, 58, 16, 58, 41, 69, 20,
    68, 27, 70, 58, 105, 66, 106, 20, 68, 58, 105, 27, 70, 66, 106, 45,
    121, 70, 123, 70, 123, 100, 128, 9, 77, 18, 78, 18, 78, 43, 97, 21,
    82, 28, 83, 59, 109, 67, 110, 21, 82, 59, 109, 28, 83, 67, 110, 46,
    122, 71, 124, 71, 124, 101, 130, 16, 58, 21, 60, 53, 80, 59, 83, 27,
    70, 29, 72, 80, 111, 81, 112, 58, 105, 82, 127, 80, 111, 109, 131,
    70, 123, 86, 128, 111, 144, 116, 145, 21, 82, 28, 83, 59, 109, 67,
    110, 29, 86, 31, 87, 81, 116, 85, 117, 60, 127, 83, 131, 83, 131,
    110, 143, 72, 128, 87, 132, 112, 145, 117, 146, 16, 58, 53, 80, 21,
    60, 59, 83, 58, 105, 80, 111, 82, 127, 109, 131, 27, 70, 80, 111, 29,
    72, 81, 112, 70, 123, 111, 144, 86, 128, 116, 145, 21, 82, 59, 109,
    28, 83, 67, 110, 60, 127, 83, 131, 83, 131, 110, 143, 29, 86, 81,
    116, 31, 87, 85, 117, 72, 128, 112, 145, 87, 132, 117, 146, 41, 69,
    59, 83, 59, 83, 79, 91, 66, 106, 81, 112, 109, 131, 115, 133, 66,
    106, 109, 131, 81, 112, 115, 133, 100, 128, 116, 145, 116, 145, 138,
    149, 46, 122, 71, 124, 71, 124, 101, 130, 72, 128, 87, 132, 112, 145,
    117, 146, 72, 128, 112, 145, 87, 132, 117, 146, 102, 141, 118, 147,
    118, 147, 139, 150, 7, 54, 20, 68, 20, 68, 45, 121, 16, 58, 27, 70,
    58, 105, 70, 123, 16, 58, 58, 105, 27, 70, 70, 123, 41, 69, 66, 106,
    66, 106, 100, 128, 9, 77, 21, 82, 21, 82, 46, 122, 18, 78, 28, 83,
    59, 109, 71, 124, 18, 78, 59, 109, 28, 83, 71, 124, 43, 97, 67, 110,
    67, 110, 101, 130, 16, 58, 27, 70, 58, 105, 70, 123, 21, 60, 29, 72,
    82, 127, 86, 128, 53, 80, 80, 111, 80, 111, 111, 144, 59, 83, 81,
    112, 109, 131, 116, 145, 21, 82, 29, 86, 60, 127, 72, 128, 28, 83,
    31, 87, 83, 131, 87, 132, 59, 109, 81, 116, 83, 131, 112, 145, 67,
    110, 85, 117, 110, 143, 117, 146, 16, 58, 58, 105, 27, 70, 70, 123,
    53, 80, 80, 111, 80, 111, 111, 144, 21, 60, 82, 127, 29, 72, 86, 128,
    59, 83, 109, 131, 81, 112, 116, 145, 21, 82, 60, 127, 29, 86, 72,
    128, 59, 109, 83, 131, 81, 116, 112, 145, 28, 83, 83, 131, 31, 87,
    87, 132, 67, 110, 110, 143, 85, 117, 117, 146, 41, 69, 66, 106, 66,
    106, 100, 128, 59, 83, 81, 112, 109, 131, 116, 145, 59, 83, 109, 131,
    81, 112, 116, 145, 79, 91, 115, 133, 115, 133, 138, 149, 46, 122, 72,
    128, 72, 128, 102, 141, 71, 124, 87, 132, 112, 145, 118, 147, 71,
    124, 112, 145, 87, 132, 118, 147, 101, 130, 117, 146, 117, 146, 139,
    150, 9, 77, 21, 82, 21, 82, 46, 122, 21, 82, 29, 86, 60, 127, 72,
    128, 21, 82, 60, 127, 29, 86, 72, 128, 46, 122, 72, 128, 72, 128,
    102, 141, 10, 89, 22, 90, 22, 90, 47, 126, 22, 90, 30, 91, 61, 129,
    73, 130, 22, 90, 61, 129, 30, 91, 73, 130, 47, 126, 73, 130, 73, 130,
    103, 142, 18, 78, 28, 83, 59, 109, 71, 124, 28, 83, 31, 87, 83, 131,
    87, 132, 59, 109, 83, 131, 81, 116, 112, 145, 71, 124, 87, 132, 112,
    145, 118, 147, 22, 90, 30, 91, 61, 129, 73, 130, 30, 91, 32, 92, 84,
    133, 88, 134, 61, 129, 84, 133, 84, 133, 113, 146, 73, 130, 88, 134,
    113, 146, 119, 148, 18, 78, 59, 109, 28, 83, 71, 124, 59, 109, 81,
    116, 83, 131, 112, 145, 28, 83, 83, 131, 31, 87, 87, 132, 71, 124,
    112, 145, 87, 132, 118, 147, 22, 90, 61, 129, 30, 91, 73, 130, 61,
    129, 84, 133, 84, 133, 113, 146, 30, 91, 84, 133, 32, 92, 88, 134,
    73, 130, 113, 146, 88, 134, 119, 148, 43, 97, 67, 110, 67, 110, 101,
    130, 67, 110, 85, 117, 110, 143, 117, 146, 67, 110, 110, 143, 85,
    117, 117, 146, 101, 130, 117, 146, 117, 146, 139, 150, 47, 126, 73,
    130, 73, 130, 103, 142, 73, 130, 88, 134, 113, 146, 119, 148, 73,
    130, 113, 146, 88, 134, 119, 148, 103, 142, 119, 148, 119, 148, 140,
    151, 11, 36, 14, 41, 36, 42, 41, 46, 14, 41, 18, 49, 55, 66, 59, 71,
    36, 42, 55, 66, 42, 62, 66, 72, 41, 46, 59, 71, 66, 72, 81, 87, 14,
    55, 18, 59, 41, 66, 49, 71, 18, 59, 22, 61, 59, 81, 61, 84, 41, 66,
    59, 81, 46, 72, 71, 87, 49, 71, 61, 84, 71, 87, 84, 92, 14, 41, 18,
    49, 55, 66, 59, 71, 18, 49, 24, 63, 78, 99, 79, 101, 55, 66, 78, 99,
    96, 100, 109, 112, 59, 71, 79, 101, 109, 112, 115, 117, 25, 69, 28,
    71, 69, 106, 71, 107, 28, 71, 30, 73, 83, 112, 84, 113, 69, 106, 83,
    112, 122, 128, 124, 132, 71, 107, 84, 113, 124, 132, 125, 134, 36,
    42, 55, 66, 42, 62, 66, 72, 55, 66, 78, 99, 96, 100, 109, 112, 42,
    62, 96, 100, 62, 94, 100, 102, 66, 72, 109, 112, 100, 102, 116, 118,
    55, 96, 78, 109, 66, 100, 99, 112, 78, 109, 90, 129, 109, 116, 129,
    133, 66, 100, 109, 116, 72, 102, 112, 118, 99, 112, 129, 133, 112,
    118, 133, 135, 41, 46, 59, 71, 66, 72, 81, 87, 59, 71, 79, 101, 109,
    112, 115, 117, 66, 72, 109, 112, 100, 102, 116, 118, 81, 87, 115,
    117, 116, 118, 138, 139, 69, 122, 83, 124, 106, 128, 112, 132, 83,
    124, 91, 130, 131, 145, 133, 146, 106, 128, 131, 145, 128, 141, 145,
    147, 112, 132, 133, 146, 145, 147, 149, 150, 14, 55, 18, 59, 41, 66,
    49, 71, 25, 69, 28, 71, 69, 106, 71, 107, 55, 96, 78, 109, 66, 100,
    99, 112, 69, 122, 83, 124, 106, 128, 112, 132, 18, 78, 24, 79, 49,
    99, 63, 101, 28, 83, 30, 84, 71, 112, 73, 113, 59, 109, 79, 115, 71,
    112, 101, 117, 71, 124, 84, 125, 107, 132, 113, 134, 18, 59, 22, 61,
    59, 81, 61, 84, 28, 71, 30, 73, 83, 112, 84, 113, 78, 109, 90, 129,
    109, 116, 129, 133, 83, 124, 91, 130, 131, 145, 133, 146, 28, 83, 30,
    84, 71, 112, 73, 113, 31, 87, 32, 88, 87, 118, 88, 119, 83, 131, 91,
    133, 124, 145, 130, 146, 87, 132, 92, 134, 132, 147, 134, 148, 41,
    66, 59, 81, 46, 72, 71, 87, 69, 106, 83, 112, 122, 128, 124, 132, 66,
    100, 109, 116, 72, 102, 112, 118, 106, 128, 131, 145, 128, 141, 145,
    147, 59, 109, 79, 115, 71, 112, 101, 117, 83, 131, 91, 133, 124, 145,
    130, 146, 81, 116, 115, 138, 87, 118, 117, 139, 112, 145, 133, 149,
    132, 147, 146, 150, 49, 71, 61, 84, 71, 87, 84, 92, 71, 107, 84, 113,
    124, 132, 125, 134, 99, 112, 129, 133, 112, 118, 133, 135, 112, 132,
    133, 146, 145, 147, 149, 150, 71, 124, 84, 125, 107, 132, 113, 134,
    87, 132, 92, 134, 132, 147, 134, 148, 112, 145, 133, 149, 132, 147,
    146, 150, 118, 147, 135, 150, 147, 152, 150, 153, 14, 55, 25, 69, 55,
    96, 69, 122, 18, 59, 28, 71, 78, 109, 83, 124, 41, 66, 69, 106, 66,
    100, 106, 128, 49, 71, 71, 107, 99, 112, 112, 132, 18, 78, 28, 83,
    59, 109, 71, 124, 24, 79, 30, 84, 79, 115, 84, 125, 49, 99, 71, 112,
    71, 112, 107, 132, 63, 101, 73, 113, 101, 117, 113, 134, 18, 59, 28,
    71, 78, 109, 83, 124, 22, 61, 30, 73, 90, 129, 91, 130, 59, 81, 83,
    112, 109, 116, 131, 145, 61, 84, 84, 113, 129, 133, 133, 146, 28, 83,
    31, 87, 83, 131, 87, 132, 30, 84, 32, 88, 91, 133, 92, 134, 71, 112,
    87, 118, 124, 145, 132, 147, 73, 113, 88, 119, 130, 146, 134, 148,
    41, 66, 69, 106, 66, 100, 106, 128, 59, 81, 83, 112, 109, 116, 131,
    145, 46, 72, 122, 128, 72, 102, 128, 141, 71, 87, 124, 132, 112, 118,
    145, 147, 59, 109, 83, 131, 81, 116, 112, 145, 79, 115, 91, 133, 115,
    138, 133, 149, 71, 112, 124, 145, 87, 118, 132, 147, 101, 117, 130,
    146, 117, 139, 146, 150, 49, 71, 71, 107, 99, 112, 112, 132, 61, 84,
    84, 113, 129, 133, 133, 146, 71, 87, 124, 132, 112, 118, 145, 147,
    84, 92, 125, 134, 133, 135, 149, 150, 71, 124, 87, 132, 112, 145,
    118, 147, 84, 125, 92, 134, 133, 149, 135, 150, 107, 132, 132, 147,
    132, 147, 147, 152, 113, 134, 134, 148, 146, 150, 150, 153, 18, 78,
    28, 83, 59, 109, 71, 124, 28, 83, 31, 87, 83, 131, 87, 132, 59, 109,
    83, 131, 81, 116, 112, 145, 71, 124, 87, 132, 112, 145, 118, 147, 22,
    90, 30, 91, 61, 129, 73, 130, 30, 91, 32, 92, 84, 133, 88, 134, 61,
    129, 84, 133, 84, 133, 113, 146, 73, 130, 88, 134, 113, 146, 119,
    148, 24, 79, 30, 84, 79, 115, 84, 125, 30, 84, 32, 88, 91, 133, 92,
    134, 79, 115, 91, 133, 115, 138, 133, 149, 84, 125, 92, 134, 133,
    149, 135, 150, 30, 91, 32, 92, 84, 133, 88, 134, 32, 92, 33, 93, 92,
    135, 93, 136, 84, 133, 92, 135, 125, 149, 134, 150, 88, 134, 93, 136,
    134, 150, 136, 151, 49, 99, 71, 112, 71, 112, 107, 132, 71, 112, 87,
    118, 124, 145, 132, 147, 71, 112, 124, 145, 87, 118, 132, 147, 107,
    132, 132, 147, 132, 147, 147, 152, 61, 129, 84, 133, 84, 133, 113,
    146, 84, 133, 92, 135, 125, 149, 134, 150, 84, 133, 125, 149, 92,
    135, 134, 150, 113, 146, 134, 150, 134, 150, 148, 153, 63, 101, 73,
    113, 101, 117, 113, 134, 73, 113, 88, 119, 130, 146, 134, 148, 101,
    117, 130, 146, 117, 139, 146, 150, 113, 134, 134, 148, 146, 150, 150,
    153, 73, 130, 88, 134, 113, 146, 119, 148, 88, 134, 93, 136, 134,
    150, 136, 151, 113, 146, 134, 150, 134, 150, 148, 153, 119, 148, 136,
    151, 148, 153, 151, 154, 11, 36, 36, 42, 14, 41, 41, 46, 36, 42, 42,
    62, 55, 66, 66, 72, 14, 41, 55, 66, 18, 49, 59, 71, 41, 46, 66, 72,
    59, 71, 81, 87, 14, 55, 41, 66, 18, 59, 49, 71, 41, 66, 46, 72, 59,
    81, 71, 87, 18, 59, 59, 81, 22, 61, 61, 84, 49, 71, 71, 87, 61, 84,
    84, 92, 36, 42, 42, 62, 55, 66, 66, 72, 42, 62, 62, 94, 96, 100, 100,
    102, 55, 66, 96, 100, 78, 99, 109, 112, 66, 72, 100, 102, 109, 112,
    116, 118, 55, 96, 66, 100, 78, 109, 99, 112, 66, 100, 72, 102, 109,
    116, 112, 118, 78, 109, 109, 116, 90, 129, 129, 133, 99, 112, 112,
    118, 129, 133, 133, 135, 14, 41, 55, 66, 18, 49, 59, 71, 55, 66, 96,
    100, 78, 99, 109, 112, 18, 49, 78, 99, 24, 63, 79, 101, 59, 71, 109,
    112, 79, 101, 115, 117, 25, 69, 69, 106, 28, 71, 71, 107, 69, 106,
    122, 128, 83, 112, 124, 132, 28, 71, 83, 112, 30, 73, 84, 113, 71,
    107, 124, 132, 84, 113, 125, 134, 41, 46, 66, 72, 59, 71, 81, 87, 66,
    72, 100, 102, 109, 112, 116, 118, 59, 71, 109, 112, 79, 101, 115,
    117, 81, 87, 116, 118, 115, 117, 138, 139, 69, 122, 106, 128, 83,
    124, 112, 132, 106, 128, 128, 141, 131, 145, 145, 147, 83, 124, 131,
    145, 91, 130, 133, 146, 112, 132, 145, 147, 133, 146, 149, 150, 14,
    55, 41, 66, 18, 59, 49, 71, 55, 96, 66, 100, 78, 109, 99, 112, 25,
    69, 69, 106, 28, 71, 71, 107, 69, 122, 106, 128, 83, 124, 112, 132,
    18, 78, 49, 99, 24, 79, 63, 101, 59, 109, 71, 112, 79, 115, 101, 117,
    28, 83, 71, 112, 30, 84, 73, 113, 71, 124, 107, 132, 84, 125, 113,
    134, 41, 66, 46, 72, 59, 81, 71, 87, 66, 100, 72, 102, 109, 116, 112,
    118, 69, 106, 122, 128, 83, 112, 124, 132, 106, 128, 128, 141, 131,
    145, 145, 147, 59, 109, 71, 112, 79, 115, 101, 117, 81, 116, 87, 118,
    115, 138, 117, 139, 83, 131, 124, 145, 91, 133, 130, 146, 112, 145,
    132, 147, 133, 149, 146, 150, 18, 59, 59, 81, 22, 61, 61, 84, 78,
    109, 109, 116, 90, 129, 129, 133, 28, 71, 83, 112, 30, 73, 84, 113,
    83, 124, 131, 145, 91, 130, 133, 146, 28, 83, 71, 112, 30, 84, 73,
    113, 83, 131, 124, 145, 91, 133, 130, 146, 31, 87, 87, 118, 32, 88,
    88, 119, 87, 132, 132, 147, 92, 134, 134, 148, 49, 71, 71, 87, 61,
    84, 84, 92, 99, 112, 112, 118, 129, 133, 133, 135, 71, 107, 124, 132,
    84, 113, 125, 134, 112, 132, 145, 147, 133, 146, 149, 150, 71, 124,
    107, 132, 84, 125, 113, 134, 112, 145, 132, 147, 133, 149, 146, 150,
    87, 132, 132, 147, 92, 134, 134, 148, 118, 147, 147, 152, 135, 150,
    150, 153, 14, 55, 55, 96, 25, 69, 69, 122, 41, 66, 66, 100, 69, 106,
    106, 128, 18, 59, 78, 109, 28, 71, 83, 124, 49, 71, 99, 112, 71, 107,
    112, 132, 18, 78, 59, 109, 28, 83, 71, 124, 49, 99, 71, 112, 71, 112,
    107, 132, 24, 79, 79, 115, 30, 84, 84, 125, 63, 101, 101, 117, 73,
    113, 113, 134, 41, 66, 66, 100, 69, 106, 106, 128, 46, 72, 72, 102,
    122, 128, 128, 141, 59, 81, 109, 116, 83, 112, 131, 145, 71, 87, 112,
    118, 124, 132, 145, 147, 59, 109, 81, 116, 83, 131, 112, 145, 71,
    112, 87, 118, 124, 145, 132, 147, 79, 115, 115, 138, 91, 133, 133,
    149, 101, 117, 117, 139, 130, 146, 146, 150, 18, 59, 78, 109, 28, 71,
    83, 124, 59, 81, 109, 116, 83, 112, 131, 145, 22, 61, 90, 129, 30,
    73, 91, 130, 61, 84, 129, 133, 84, 113, 133, 146, 28, 83, 83, 131,
    31, 87, 87, 132, 71, 112, 124, 145, 87, 118, 132, 147, 30, 84, 91,
    133, 32, 88, 92, 134, 73, 113, 130, 146, 88, 119, 134, 148, 49, 71,
    99, 112, 71, 107, 112, 132, 71, 87, 112, 118, 124, 132, 145, 147, 61,
    84, 129, 133, 84, 113, 133, 146, 84, 92, 133, 135, 125, 134, 149,
    150, 71, 124, 112, 145, 87, 132, 118, 147, 107, 132, 132, 147, 132,
    147, 147, 152, 84, 125, 133, 149, 92, 134, 135, 150, 113, 134, 146,
    150, 134, 148, 150, 153, 18, 78, 59, 109, 28, 83, 71, 124, 59, 109,
    81, 116, 83, 131, 112, 145, 28, 83, 83, 131, 31, 87, 87, 132, 71,
    124, 112, 145, 87, 132, 118, 147, 22, 90, 61, 129, 30, 91, 73, 130,
    61, 129, 84, 133, 84, 133, 113, 146, 30, 91, 84, 133, 32, 92, 88,
    134, 73, 130, 113, 146, 88, 134, 119, 148, 49, 99, 71, 112, 71, 112,
    107, 132, 71, 112, 87, 118, 124, 145, 132, 147, 71, 112, 124, 145,
    87, 118, 132, 147, 107, 132, 132, 147, 132, 147, 147, 152, 61, 129,
    84, 133, 84, 133, 113, 146, 84, 133, 92, 135, 125, 149, 134, 150, 84,
    133, 125, 149, 92, 135, 134, 150, 113, 146, 134, 150, 134, 150, 148,
    153, 24, 79, 79, 115, 30, 84, 84, 125, 79, 115, 115, 138, 91, 133,
    133, 149, 30, 84, 91, 133, 32, 88, 92, 134, 84, 125, 133, 149, 92,
    134, 135, 150, 30, 91, 84, 133, 32, 92, 88, 134, 84, 133, 125, 149,
    92, 135, 134, 150, 32, 92, 92, 135, 33, 93, 93, 136, 88, 134, 134,
    150, 93, 136, 136, 151, 63, 101, 101, 117, 73, 113, 113, 134, 101,
    117, 117, 139, 130, 146, 146, 150, 73, 113, 130, 146, 88, 119, 134,
    148, 113, 134, 146, 150, 134, 148, 150, 153, 73, 130, 113, 146, 88,
    134, 119, 148, 113, 146, 134, 150, 134, 150, 148, 153, 88, 134, 134,
    150, 93, 136, 136, 151, 119, 148, 148, 153, 136, 151, 151, 154, 34,
    37, 37, 43, 37, 43, 43, 47, 37, 43, 43, 63, 64, 67, 67, 73, 37, 43,
    64, 67, 43, 63, 67, 73, 43, 47, 67, 73, 67, 73, 85, 88, 37, 64, 43,
    67, 43, 67, 63, 73, 43, 67, 47, 73, 67, 85, 73, 88, 43, 67, 67, 85,
    47, 73, 73, 88, 63, 73, 73, 88, 73, 88, 88, 93, 37, 43, 43, 63, 64,
    67, 67, 73, 43, 63, 63, 95, 97, 101, 101, 103, 64, 67, 97, 101, 97,
    101, 110, 113, 67, 73, 101, 103, 110, 113, 117, 119, 64, 97, 67, 101,
    97, 110, 101, 113, 67, 101, 73, 103, 110, 117, 113, 119, 97, 110,
    110, 117, 126, 130, 130, 134, 101, 113, 113, 119, 130, 134, 134, 136,
    37, 43, 64, 67, 43, 63, 67, 73, 64, 67, 97, 101, 97, 101, 110, 113,
    43, 63, 97, 101, 63, 95, 101, 103, 67, 73, 110, 113, 101, 103, 117,
    119, 64, 97, 97, 110, 67, 101, 101, 113, 97, 110, 126, 130, 110, 117,
    130, 134, 67, 101, 110, 117, 73, 103, 113, 119, 101, 113, 130, 134,
    113, 119, 134, 136, 43, 47, 67, 73, 67, 73, 85, 88, 67, 73, 101, 103,
    110, 113, 117, 119, 67, 73, 110, 113, 101, 103, 117, 119, 85, 88,
    117, 119, 117, 119, 139, 140, 97, 126, 110, 130, 110, 130, 117, 134,
    110, 130, 130, 142, 143, 146, 146, 148, 110, 130, 143, 146, 130, 142,
    146, 148, 117, 134, 146, 148, 146, 148, 150, 151, 37, 64, 43, 67, 43,
    67, 63, 73, 64, 97, 67, 101, 97, 110, 101, 113, 64, 97, 97, 110, 67,
    101, 101, 113, 97, 126, 110, 130, 110, 130, 117, 134, 43, 97, 63,
    101, 63, 101, 95, 103, 67, 110, 73, 113, 101, 117, 103, 119, 67, 110,
    101, 117, 73, 113, 103, 119, 101, 130, 113, 134, 113, 134, 119, 136,
    43, 67, 47, 73, 67, 85, 73, 88, 67, 101, 73, 103, 110, 117, 113, 119,
    97, 110, 126, 130, 110, 117, 130, 134, 110, 130, 130, 142, 143, 146,
    146, 148, 67, 110, 73, 113, 101, 117, 103, 119, 85, 117, 88, 119,
    117, 139, 119, 140, 110, 143, 130, 146, 130, 146, 142, 148, 117, 146,
    134, 148, 146, 150, 148, 151, 43, 67, 67, 85, 47, 73, 73, 88, 97,
    110, 110, 117, 126, 130, 130, 134, 67, 101, 110, 117, 73, 103, 113,
    119, 110, 130, 143, 146, 130, 142, 146, 148, 67, 110, 101, 117, 73,
    113, 103, 119, 110, 143, 130, 146, 130, 146, 142, 148, 85, 117, 117,
    139, 88, 119, 119, 140, 117, 146, 146, 150, 134, 148, 148, 151, 63,
    73, 73, 88, 73, 88, 88, 93, 101, 113, 113, 119, 130, 134, 134, 136,
    101, 113, 130, 134, 113, 119, 134, 136, 117, 134, 146, 148, 146, 148,
    150, 151, 101, 130, 113, 134, 113, 134, 119, 136, 117, 146, 134, 148,
    146, 150, 148, 151, 117, 146, 146, 150, 134, 148, 148, 151, 139, 150,
    150, 153, 150, 153, 153, 154, 37, 64, 64, 97, 64, 97, 97, 126, 43,
    67, 67, 101, 97, 110, 110, 130, 43, 67, 97, 110, 67, 101, 110, 130,
    63, 73, 101, 113, 101, 113, 117, 134, 43, 97, 67, 110, 67, 110, 101,
    130, 63, 101, 73, 113, 101, 117, 113, 134, 63, 101, 101, 117, 73,
    113, 113, 134, 95, 103, 103, 119, 103, 119, 119, 136, 43, 67, 67,
    101, 97, 110, 110, 130, 47, 73, 73, 103, 126, 130, 130, 142, 67, 85,
    110, 117, 110, 117, 143, 146, 73, 88, 113, 119, 130, 134, 146, 148,
    67, 110, 85, 117, 110, 143, 117, 146, 73, 113, 88, 119, 130, 146,
    134, 148, 101, 117, 117, 139, 130, 146, 146, 150, 103, 119, 119, 140,
    142, 148, 148, 151, 43, 67, 97, 110, 67, 101, 110, 130, 67, 85, 110,
    117, 110, 117, 143, 146, 47, 73, 126, 130, 73, 103, 130, 142, 73, 88,
    130, 134, 113, 119, 146, 148, 67, 110, 110, 143, 85, 117, 117, 146,
    101, 117, 130, 146, 117, 139, 146, 150, 73, 113, 130, 146, 88, 119,
    134, 148, 103, 119, 142, 148, 119, 140, 148, 151, 63, 73, 101, 113,
    101, 113, 117, 134, 73, 88, 113, 119, 130, 134, 146, 148, 73, 88,
    130, 134, 113, 119, 146, 148, 88, 93, 134, 136, 134, 136, 150, 151,
    101, 130, 117, 146, 117, 146, 139, 150, 113, 134, 134, 148, 146, 150,
    150, 153, 113, 134, 146, 150, 134, 148, 150, 153, 119, 136, 148, 151,
    148, 151, 153, 154, 43, 97, 67, 110, 67, 110, 101, 130, 67, 110, 85,
    117, 110, 143, 117, 146, 67, 110, 110, 143, 85, 117, 117, 146, 101,
    130, 117, 146, 117, 146, 139, 150, 47, 126, 73, 130, 73, 130, 103,
    142, 73, 130, 88, 134, 113, 146, 119, 148, 73, 130, 113, 146, 88,
    134, 119, 148, 103, 142, 119, 148, 119, 148, 140, 151, 63, 101, 73,
    113, 101, 117, 113, 134, 73, 113, 88, 119, 130, 146, 134, 148, 101,
    117, 130, 146, 117, 139, 146, 150, 113, 134, 134, 148, 146, 150, 150,
    153, 73, 130, 88, 134, 113, 146, 119, 148, 88, 134, 93, 136, 134,
    150, 136, 151, 113, 146, 134, 150, 134, 150, 148, 153, 119, 148, 136,
    151, 148, 153, 151, 154, 63, 101, 101, 117, 73, 113, 113, 134, 101,
    117, 117, 139, 130, 146, 146, 150, 73, 113, 130, 146, 88, 119, 134,
    148, 113, 134, 146, 150, 134, 148, 150, 153, 73, 130, 113, 146, 88,
    134, 119, 148, 113, 146, 134, 150, 134, 150, 148, 153, 88, 134, 134,
    150, 93, 136, 136, 151, 119, 148, 148, 153, 136, 151, 151, 154, 95,
    103, 103, 119, 103, 119, 119, 136, 103, 119, 119, 140, 142, 148, 148,
    151, 103, 119, 142, 148, 119, 140, 148, 151, 119, 136, 148, 151, 148,
    151, 153, 154, 103, 142, 119, 148, 119, 148, 140, 151, 119, 148, 136,
    151, 148, 153, 151, 154, 119, 148, 148, 153, 136, 151, 151, 154, 140,
    151, 151, 154, 151, 154, 154, 155
};

const unsigned int igraph_i_isographs_3[] =  { 0, 1, 3, 5, 6, 7, 10, 11, 15, 21,
                                               23, 25, 27, 30, 31, 63
                                             };
const unsigned int igraph_i_isographs_3u[] = { 0, 1, 3, 7 };
const unsigned int igraph_i_isographs_4[] = {
    0,    1,    3,    7,    9,   10,   11,   14,   15,   18,   19,   20,   21,
    22,   23,   27,   29,   30,   31,   54,   55,   63,   73,   75,   76,   77,
    79,   81,   83,   84,   85,   86,   87,   90,   91,   92,   93,   94,   95,
    98,   99,  100,  101,  102,  103,  106,  107,  108,  109,  110,  111,  115,
    116,  117,  118,  119,  122,  123,  124,  125,  126,  127,  219,  220,  221,
    223,  228,  229,  230,  231,  237,  238,  239,  246,  247,  255,  292,  293,
    295,  301,  302,  303,  310,  311,  319,  365,  367,  373,  375,  382,  383,
    511,  585,  587,  591,  593,  594,  595,  596,  597,  598,  599,  601,  602,
    603,  604,  605,  606,  607,  625,  626,  627,  630,  631,  633,  634,  635,
    638,  639,  659,  660,  661,  663,  666,  667,  669,  670,  671,  674,  675,
    678,  679,  683,  686,  687,  694,  695,  703,  729,  731,  732,  733,  735,
    737,  739,  741,  742,  743,  745,  746,  747,  748,  749,  750,  751,  753,
    755,  756,  757,  758,  759,  761,  762,  763,  764,  765,  766,  767,  819,
    822,  823,  826,  827,  830,  831,  875,  876,  877,  879,  883,  885,  886,
    887,  891,  892,  893,  894,  895,  947,  949,  951,  955,  957,  958,  959,
    1019, 1020, 1021, 1023, 1755, 1757, 1758, 1759, 1782, 1783, 1791, 1883, 1887,
    1907, 1911, 1917, 1918, 1919, 2029, 2031, 2039, 2047, 4095
};
const unsigned int igraph_i_isographs_4u[] = { 0, 1, 3, 7, 11, 12, 13,
                                               15, 30, 31, 63
                                             };
const unsigned int igraph_i_isographs_5u[] = {
    0, 1, 3, 7, 11, 12, 13, 15, 30, 31, 63, 75, 76, 77, 79, 86, 87, 94,
    95, 116, 117, 119, 127, 222, 223, 235, 236, 237, 239, 254, 255, 507,
    511, 1023
};
const unsigned int igraph_i_isographs_6u[] = {
    0, 1, 3, 7, 11, 12, 13, 15, 30, 31, 63, 75, 76, 77, 79, 86, 87, 94,
    95, 116, 117, 119, 127, 222, 223, 235, 236, 237, 239, 254, 255, 507,
    511, 1023, 1099, 1100, 1101, 1103, 1108, 1109, 1110, 1111, 1118,
    1119, 1140, 1141, 1143, 1151, 1182, 1183, 1184, 1185, 1187, 1191,
    1194, 1195, 1196, 1197, 1198, 1199, 1214, 1215, 1246, 1247, 1259,
    1260, 1261, 1263, 1268, 1269, 1270, 1271, 1278, 1279, 1456, 1457,
    1459, 1460, 1461, 1463, 1465, 1467, 1468, 1469, 1471, 1531, 1532,
    1533, 1535, 1972, 1973, 1975, 1983, 2047, 3294, 3295, 3306, 3307,
    3308, 3309, 3310, 3311, 3326, 3327, 3440, 3441, 3443, 3447, 3448,
    3449, 3451, 3452, 3453, 3455, 3576, 3577, 3578, 3579, 3582, 3583,
    3873, 3875, 3879, 3885, 3887, 3903, 3947, 3948, 3949, 3950, 3951,
    3958, 3959, 3966, 3967, 4094, 4095, 7672, 7673, 7675, 7679, 7902,
    7903, 7915, 7916, 7917, 7919, 7934, 7935, 8185, 8187, 8191, 16350,
    16351, 16383, 32767
};

const unsigned int igraph_i_classedges_3[] = { 1, 2, 0, 2, 2, 1, 0, 1, 2, 0, 1, 0 };
const unsigned int igraph_i_classedges_3u[] = { 1, 2, 0, 2, 0, 1 };
const unsigned int igraph_i_classedges_4[] = { 2, 3, 1, 3, 0, 3, 3, 2, 1, 2, 0, 2,
                                               3, 1, 2, 1, 0, 1, 3, 0, 2, 0, 1, 0
                                             };
const unsigned int igraph_i_classedges_4u[] = { 2, 3, 1, 3, 0, 3, 1, 2, 0, 2, 0, 1 };
const unsigned int igraph_i_classedges_5u[] = { 3, 4, 2, 4, 1, 4, 0, 4, 2, 3, 1, 3,
                                                0, 3, 1, 2, 0, 2, 0, 1 };
const unsigned int igraph_i_classedges_6u[] = { 4, 5, 3, 5, 2, 5, 1, 5, 0, 5, 3, 4,
                                                2, 4, 1, 4, 0, 4, 2, 3, 1, 3, 0, 3,
                                                1, 2, 0, 2, 0, 1 };

/**
 * \function igraph_isoclass
 * \brief Determine the isomorphism class of small graphs.
 *
 * </para><para>
 * All graphs with a given number of vertices belong to a number of
 * isomorphism classes, with every graph in a given class being
 * isomorphic to each other.
 *
 * </para><para>
 * This function gives the isomorphism class (a number) of a
 * graph. Two graphs have the same isomorphism class if and only if
 * they are isomorphic.
 *
 * </para><para>
 * The first isomorphism class is numbered zero and it contains the edgeless
 * graph. The last isomorphism class contains the full graph. The number of
 * isomorphism classes for directed graphs with three vertices is 16
 * (between 0 and 15), for undirected graph it is only 4. For graphs
 * with four vertices it is 218 (directed) and 11 (undirected).
 * For 5 and 6 vertex undirected graphs, it is 34 and 156, respectively.
 * These values can also be retrieved using \ref igraph_graph_count().
 * For more information, see https://oeis.org/A000273 and https://oeis.org/A000088.
 *
 * </para><para>
 * At the moment, 3- and 4-vertex directed graphs and 3 to 6 vertex
 * undirected graphs are supported.
 *
 * </para><para>
 * Multi-edges and self-loops are ignored by this function.
 *
 * \param graph The graph object.
 * \param isoclass Pointer to an integer, the isomorphism class will
 *        be stored here.
 * \return Error code.
 * \sa \ref igraph_isomorphic(), \ref igraph_isoclass_subgraph(),
 * \ref igraph_isoclass_create(), \ref igraph_motifs_randesu().
 *
 * Because of some limitations this function works only for graphs
 * with three of four vertices.
 *
 * </para><para>
 * Time complexity: O(|E|), the number of edges in the graph.
 */
igraph_error_t igraph_isoclass(const igraph_t *graph, igraph_integer_t *isoclass) {
    igraph_integer_t e;
    igraph_integer_t no_of_nodes = igraph_vcount(graph);
    igraph_integer_t no_of_edges = igraph_ecount(graph);
    unsigned int idx, mul;
    const unsigned int *arr_idx, *arr_code;
    unsigned int code;

    if (igraph_is_directed(graph)) {
        switch (no_of_nodes) {
        case 3:
            arr_idx = igraph_i_isoclass_3_idx;
            arr_code = igraph_i_isoclass2_3;
            mul = 3;
            break;
        case 4:
            arr_idx = igraph_i_isoclass_4_idx;
            arr_code = igraph_i_isoclass2_4;
            mul = 4;
            break;
        default:
            IGRAPH_ERROR("Directed isoclass is only implemented for graphs with 3 or 4 vertices.",
                         IGRAPH_UNIMPLEMENTED);
        }
    } else {
        switch (no_of_nodes) {
        case 3:
            arr_idx = igraph_i_isoclass_3u_idx;
            arr_code = igraph_i_isoclass2_3u;
            mul = 3;
            break;
        case 4:
            arr_idx = igraph_i_isoclass_4u_idx;
            arr_code = igraph_i_isoclass2_4u;
            mul = 4;
            break;
        case 5:
            arr_idx = igraph_i_isoclass_5u_idx;
            arr_code = igraph_i_isoclass2_5u;
            mul = 5;
            break;
        case 6:
            arr_idx = igraph_i_isoclass_6u_idx;
            arr_code = igraph_i_isoclass2_6u;
            mul = 6;
            break;
        default:
            IGRAPH_ERROR("Undirected isoclass is only implemented for graphs with 3 to 6 vertices.",
                         IGRAPH_UNIMPLEMENTED);
        }
    }

    code = 0;
    for (e = 0; e < no_of_edges; e++) {
        idx = mul * IGRAPH_FROM(graph, e) + IGRAPH_TO(graph, e);
        code |= arr_idx[idx];
    }

    *isoclass = (igraph_integer_t) arr_code[code];

    return IGRAPH_SUCCESS;
}

/**
 * \function igraph_isoclass_subgraph
 * \brief The isomorphism class of a subgraph of a graph.
 *
 * This function identifies the isomorphism class of the subgraph
 * induced the vertices specified in \p vids.
 *
 * </para><para>
 * At the moment, 3- and 4-vertex directed graphs and 3 to 6 vertex
 * undirected graphs are supported.
 *
 * </para><para>
 * Multi-edges and self-loops are ignored by this function.
 *
 * \param graph The graph object.
 * \param vids A vector containing the vertex IDs to be considered as
 *        a subgraph. Each vertex ID should be included at most once.
 * \param isoclass Pointer to an integer, this will be set to the
 *        isomorphism class.
 * \return Error code.
 * \sa \ref igraph_isoclass(), \ref igraph_isomorphic(),
 * \ref igraph_isoclass_create().
 *
 * Time complexity: O((d+n)*n), d is the average degree in the network,
 * and n is the number of vertices in \c vids.
 */
igraph_error_t igraph_isoclass_subgraph(const igraph_t *graph, const igraph_vector_int_t *vids,
                             igraph_integer_t *isoclass) {
    igraph_integer_t subgraph_size = igraph_vector_int_size(vids);
    igraph_vector_int_t neis;

    unsigned int mul, idx;
    const unsigned int *arr_idx, *arr_code;
    unsigned int code = 0;

    igraph_integer_t i, j, s;

    IGRAPH_VECTOR_INT_INIT_FINALLY(&neis, 0);

    if (igraph_is_directed(graph)) {
        switch (subgraph_size) {
        case 3:
            arr_idx = igraph_i_isoclass_3_idx;
            arr_code = igraph_i_isoclass2_3;
            mul = 3;
            break;
        case 4:
            arr_idx = igraph_i_isoclass_4_idx;
            arr_code = igraph_i_isoclass2_4;
            mul = 4;
            break;
        default:
            IGRAPH_ERROR("Directed isoclass is only implemented for graphs with 3 or 4 vertices.",
                         IGRAPH_UNIMPLEMENTED);
        }
    } else {
        switch (subgraph_size) {
        case 3:
            arr_idx = igraph_i_isoclass_3u_idx;
            arr_code = igraph_i_isoclass2_3u;
            mul = 3;
            break;
        case 4:
            arr_idx = igraph_i_isoclass_4u_idx;
            arr_code = igraph_i_isoclass2_4u;
            mul = 4;
            break;
        case 5:
            arr_idx = igraph_i_isoclass_5u_idx;
            arr_code = igraph_i_isoclass2_5u;
            mul = 5;
            break;
        case 6:
            arr_idx = igraph_i_isoclass_6u_idx;
            arr_code = igraph_i_isoclass2_6u;
            mul = 6;
            break;
        default:
            IGRAPH_ERROR("Undirected isoclass is only implemented for graphs with 3 to 6 vertices.",
                         IGRAPH_UNIMPLEMENTED);
        }
    }

    for (i = 0; i < subgraph_size; i++) {
        igraph_integer_t from = VECTOR(*vids)[i];
        IGRAPH_CHECK(igraph_neighbors(graph, &neis, from, IGRAPH_OUT));
        s = igraph_vector_int_size(&neis);
        for (j = 0; j < s; j++) {
            igraph_integer_t nei = VECTOR(neis)[j], to;
            if (igraph_vector_int_search(vids, 0, nei, &to)) {
                idx = (mul * i + to);
                code |= arr_idx[idx];
            }
        }
    }

    *isoclass = arr_code[code];
    igraph_vector_int_destroy(&neis);
    IGRAPH_FINALLY_CLEAN(1);

    return IGRAPH_SUCCESS;
}

/**
 * \function igraph_isoclass_create
 * \brief Creates a graph from the given isomorphism class.
 *
 * </para><para>
 * This function creates the canonical representative graph of the
 * given isomorphism class.
 *
 * </para><para>
 * The isomorphism class is an integer between 0 and the number of
 * unique unlabeled (i.e. non-isomorphic) graphs on the given number
 * of vertices and give directedness. See https://oeis.org/A000273
 * and https://oeis.org/A000088 for the number of directed and
 * undirected graphs on \p size nodes.
 *
 * </para><para>
 * At the moment, 3- and 4-vertex directed graphs and 3 to 6 vertex
 * undirected graphs are supported.
 *
 * \param graph Pointer to an uninitialized graph object.
 * \param size The number of vertices to add to the graph.
 * \param number The isomorphism class.
 * \param directed Boolean constant, whether to create a directed
 *        graph.
 * \return Error code.
 * \sa \ref igraph_isoclass(),
 * \ref igraph_isoclass_subgraph(),
 * \ref igraph_isomorphic().
 *
 * Time complexity: O(|V|+|E|), the number of vertices plus the number
 * of edges in the graph to create.
 */
igraph_error_t igraph_isoclass_create(igraph_t *graph, igraph_integer_t size,
                           igraph_integer_t number, igraph_bool_t directed) {
    igraph_vector_int_t edges;
    const unsigned int *classedges;
    igraph_integer_t graphcount;
    igraph_integer_t pos;
    unsigned int power;
    unsigned int code;

    IGRAPH_VECTOR_INT_INIT_FINALLY(&edges, 0);

#define CHECK_ISOCLASS(number, directed, size, graphcount) \
    IGRAPH_ERRORF( \
        "Isoclass %" IGRAPH_PRId " requested, but there are only %" \
        IGRAPH_PRId " %s graphs of size %" IGRAPH_PRId ".", IGRAPH_EINVAL, \
        number, graphcount, directed ? "directed" : "undirected", size)

    if (directed) {
        switch (size) {
        case 3: {
            classedges = igraph_i_classedges_3;
            graphcount = sizeof(igraph_i_isographs_3) / sizeof(igraph_i_isographs_3[0]);

            if (number < 0 || number >= graphcount) {
                CHECK_ISOCLASS(number, directed, size, graphcount);
            }

            code = igraph_i_isographs_3[number];
            power = 32;

            break;
        }
        case 4: {
            classedges = igraph_i_classedges_4;
            graphcount = sizeof(igraph_i_isographs_4) / sizeof(igraph_i_isographs_4[0]);

            if (number < 0 || number >= graphcount) {
                CHECK_ISOCLASS(number, directed, size, graphcount);
            }

            code = igraph_i_isographs_4[number];
            power = 2048;

            break;
        }
        default:
            IGRAPH_ERROR("Directed isoclasses are supported only for graphs with 3 or 4 vertices.",
                         IGRAPH_UNIMPLEMENTED);
        }

    } else {
        switch (size) {
        case 3: {
            classedges = igraph_i_classedges_3u;
            graphcount = sizeof(igraph_i_isographs_3u) / sizeof(igraph_i_isographs_3u[0]);

            if (number < 0 || number >= graphcount) {
                CHECK_ISOCLASS(number, directed, size, graphcount);
            }

            code = igraph_i_isographs_3u[number];
            power = 4;

            break;
        }
        case 4: {
            classedges = igraph_i_classedges_4u;
            graphcount = sizeof(igraph_i_isographs_4u) / sizeof(igraph_i_isographs_4u[0]);

            if (number < 0 || number >= graphcount) {
                CHECK_ISOCLASS(number, directed, size, graphcount);
            }

            code = igraph_i_isographs_4u[number];
            power = 32;

            break;
        }
        case 5: {
            classedges = igraph_i_classedges_5u;
            graphcount = sizeof(igraph_i_isographs_5u) / sizeof(igraph_i_isographs_5u[0]);

            if (number < 0 || number >= graphcount) {
                CHECK_ISOCLASS(number, directed, size, graphcount);
            }

            code = igraph_i_isographs_5u[number];
            power = 512;

            break;
        }
        case 6: {
            classedges = igraph_i_classedges_6u;
            graphcount = sizeof(igraph_i_isographs_6u) / sizeof(igraph_i_isographs_6u[0]);

            if (number < 0 || number >= graphcount) {
                CHECK_ISOCLASS(number, directed, size, graphcount);
            }

            code = igraph_i_isographs_6u[number];
            power = 16384;

            break;
        }
        default:
            IGRAPH_ERROR("Undirected isoclasses are supported only for graphs with 3 to 6 vertices.",
                         IGRAPH_UNIMPLEMENTED);
        }
    }

#undef CHECK_ISOCLASS

    pos = 0;
    while (code > 0) {
        if (code >= power) {
            IGRAPH_CHECK(igraph_vector_int_push_back(&edges, classedges[2 * pos]));
            IGRAPH_CHECK(igraph_vector_int_push_back(&edges, classedges[2 * pos + 1]));
            code -= power;
        }
        power /= 2;
        pos++;
    }

    IGRAPH_CHECK(igraph_create(graph, &edges, size, directed));
    igraph_vector_int_destroy(&edges);
    IGRAPH_FINALLY_CLEAN(1);

    return IGRAPH_SUCCESS;
}

/* https://oeis.org/A000088 */
static igraph_integer_t undirected_graph_counts[] = {
    1, 1, 2, 4, 11, 34, 156, 1044, 12346, 274668, 12005168, 1018997864,
#if IGRAPH_INTEGER_SIZE == 64
    165091172592, 50502031367952, 29054155657235488
#endif
};

/* https://oeis.org/A000273 */
static igraph_integer_t directed_graph_counts[] = {
    1, 1, 3, 16, 218, 9608, 1540944, 882033440,
#if IGRAPH_INTEGER_SIZE == 64
    1793359192848, 13027956824399552
#endif
};

/**
 * \function igraph_graph_count
 * \brief The number of unlabelled graphs on the given number of vertices.
 *
 * Gives the number of unlabelled \em simple graphs on the specified number of vertices.
 * The "isoclass" of a graph of this size is at most one less than this value.
 *
 * </para><para>
 * This function is meant to be used in conjunction with isoclass and motif finder
 * functions. It will only work for small \p n values for which the result is
 * represetable in an \type igraph_integer_t. For larger \p n values, an overflow
 * error is raised.
 *
 * \param n The number of vertices.
 * \param directed Boolean, whether to consider directed graphs.
 * \param count Pointer to an integer, the result will be stored here.
 * \return Error code.
 *
 * \sa \ref igraph_isoclass(), \ref igraph_motifs_randesu_callback().
 *
 * Time complexity: O(1).
 */
igraph_error_t igraph_graph_count(igraph_integer_t n, igraph_bool_t directed, igraph_integer_t *count) {
    if (n < 0) {
        IGRAPH_ERROR("Graph size must not be negative.", IGRAPH_EINVAL);
    }
    if (directed) {
        if (n >= (igraph_integer_t) (sizeof directed_graph_counts / sizeof directed_graph_counts[0])) {
            IGRAPH_ERRORF("Graph size of % " IGRAPH_PRId " too large.", IGRAPH_EOVERFLOW, n);
        }
        *count = directed_graph_counts[n];
    } else {
        if (n >= (igraph_integer_t) (sizeof undirected_graph_counts / sizeof undirected_graph_counts[0])) {
            IGRAPH_ERRORF("Graph size of % " IGRAPH_PRId " too large.", IGRAPH_EOVERFLOW, n);
        }
        *count = undirected_graph_counts[n];
    }
    return IGRAPH_SUCCESS;
}
