-- autogenerated from https://www.unicode.org/Public/15.0.0/ucd/UnicodeData.txt
-- |
-- Module      : Unicode.Internal.Char.UnicodeData.GeneralCategory
-- Copyright   : (c) 2020 Composewell Technologies and Contributors
-- License     : Apache-2.0
-- Maintainer  : streamly@composewell.com
-- Stability   : experimental

{-# OPTIONS_HADDOCK hide #-}
{-# LANGUAGE PatternSynonyms #-}

module Unicode.Internal.Char.UnicodeData.GeneralCategory
( -- * Lookup functions
  generalCategory
, generalCategoryPlanes0To3

  -- * General categories
, pattern UppercaseLetter
, pattern LowercaseLetter
, pattern TitlecaseLetter
, pattern ModifierLetter
, pattern OtherLetter
, pattern NonSpacingMark
, pattern SpacingCombiningMark
, pattern EnclosingMark
, pattern DecimalNumber
, pattern LetterNumber
, pattern OtherNumber
, pattern ConnectorPunctuation
, pattern DashPunctuation
, pattern OpenPunctuation
, pattern ClosePunctuation
, pattern InitialQuote
, pattern FinalQuote
, pattern OtherPunctuation
, pattern MathSymbol
, pattern CurrencySymbol
, pattern ModifierSymbol
, pattern OtherSymbol
, pattern Space
, pattern LineSeparator
, pattern ParagraphSeparator
, pattern Control
, pattern Format
, pattern Surrogate
, pattern PrivateUse
, pattern NotAssigned

  -- * Characters bounds for predicates
, pattern MaxIsLetter
, pattern MaxIsAlphaNum
, pattern MaxIsLower
, pattern MaxIsUpper
, pattern MaxIsNumber
, pattern MaxIsSpace
, pattern MaxIsSeparator
) where

import Data.Bits (Bits(..))
import Data.Char (ord)
import Data.Int (Int8)
import Data.Word (Word16)
import GHC.Exts (Ptr(..))
import Unicode.Internal.Bits (lookupWord16AsInt, lookupWord8AsInt)

--------------------------------------------------------------------------------
-- General category patterns
--------------------------------------------------------------------------------

-- | General category Lu
pattern UppercaseLetter :: Int
pattern UppercaseLetter = 0

-- | General category Ll
pattern LowercaseLetter :: Int
pattern LowercaseLetter = 1

-- | General category Lt
pattern TitlecaseLetter :: Int
pattern TitlecaseLetter = 2

-- | General category Lm
pattern ModifierLetter :: Int
pattern ModifierLetter = 3

-- | General category Lo
pattern OtherLetter :: Int
pattern OtherLetter = 4

-- | General category Mn
pattern NonSpacingMark :: Int
pattern NonSpacingMark = 5

-- | General category Mc
pattern SpacingCombiningMark :: Int
pattern SpacingCombiningMark = 6

-- | General category Me
pattern EnclosingMark :: Int
pattern EnclosingMark = 7

-- | General category Nd
pattern DecimalNumber :: Int
pattern DecimalNumber = 8

-- | General category Nl
pattern LetterNumber :: Int
pattern LetterNumber = 9

-- | General category No
pattern OtherNumber :: Int
pattern OtherNumber = 10

-- | General category Pc
pattern ConnectorPunctuation :: Int
pattern ConnectorPunctuation = 11

-- | General category Pd
pattern DashPunctuation :: Int
pattern DashPunctuation = 12

-- | General category Ps
pattern OpenPunctuation :: Int
pattern OpenPunctuation = 13

-- | General category Pe
pattern ClosePunctuation :: Int
pattern ClosePunctuation = 14

-- | General category Pi
pattern InitialQuote :: Int
pattern InitialQuote = 15

-- | General category Pf
pattern FinalQuote :: Int
pattern FinalQuote = 16

-- | General category Po
pattern OtherPunctuation :: Int
pattern OtherPunctuation = 17

-- | General category Sm
pattern MathSymbol :: Int
pattern MathSymbol = 18

-- | General category Sc
pattern CurrencySymbol :: Int
pattern CurrencySymbol = 19

-- | General category Sk
pattern ModifierSymbol :: Int
pattern ModifierSymbol = 20

-- | General category So
pattern OtherSymbol :: Int
pattern OtherSymbol = 21

-- | General category Zs
pattern Space :: Int
pattern Space = 22

-- | General category Zl
pattern LineSeparator :: Int
pattern LineSeparator = 23

-- | General category Zp
pattern ParagraphSeparator :: Int
pattern ParagraphSeparator = 24

-- | General category Cc
pattern Control :: Int
pattern Control = 25

-- | General category Cf
pattern Format :: Int
pattern Format = 26

-- | General category Cs
pattern Surrogate :: Int
pattern Surrogate = 27

-- | General category Co
pattern PrivateUse :: Int
pattern PrivateUse = 28

-- | General category Cn
pattern NotAssigned :: Int
pattern NotAssigned = 29

--------------------------------------------------------------------------------
-- Characters bounds for predicates
--------------------------------------------------------------------------------

-- | Maximum codepoint satisfying @isLetter@
pattern MaxIsLetter :: Int
pattern MaxIsLetter = 0x323AF

-- | Maximum codepoint satisfying @isAlphaNum@
pattern MaxIsAlphaNum :: Int
pattern MaxIsAlphaNum = 0x323AF

-- | Maximum codepoint satisfying @isLower@
pattern MaxIsLower :: Int
pattern MaxIsLower = 0x1E943

-- | Maximum codepoint satisfying @isUpper@
pattern MaxIsUpper :: Int
pattern MaxIsUpper = 0x1E921

-- | Maximum codepoint satisfying @isNumber@
pattern MaxIsNumber :: Int
pattern MaxIsNumber = 0x1FBF9

-- | Maximum codepoint satisfying @isSpace@
pattern MaxIsSpace :: Int
pattern MaxIsSpace = 0x3000

-- | Maximum codepoint satisfying @isSeparator@
pattern MaxIsSeparator :: Int
pattern MaxIsSeparator = 0x3000

--------------------------------------------------------------------------------
-- Lookup functions
--------------------------------------------------------------------------------

-- | Return the general category of a code point in planes 0 to 3
--
-- The caller of this function must ensure its parameter is \< @0x40000@.
{-# INLINE generalCategoryPlanes0To3 #-}
generalCategoryPlanes0To3 :: Int -> Int
generalCategoryPlanes0To3 = lookupGeneralCategoryBitMap

-- | Return the general category of a character
{-# INLINE generalCategory #-}
generalCategory :: Char -> Int
generalCategory c
    -- Planes 0-3
    | cp < 0x323B0 = lookupGeneralCategoryBitMap cp
    -- Planes 4-13: Cn
    | cp < 0xE0000 = NotAssigned
    -- Plane 14
    | cp < 0xE01F0 = lookupGeneralCategoryBitMap (cp - 0xADC50)
    -- Plane 14: Cn
    | cp < 0xF0000 = NotAssigned
    -- Plane 15: Co
    | cp < 0xFFFFE = PrivateUse
    -- Plane 15: Cn
    | cp < 0x100000 = NotAssigned
    -- Plane 16: Co
    | cp < 0x10FFFE = PrivateUse
    -- Default: Cn
    | otherwise = NotAssigned
    where
    cp = ord c

{-# INLINE lookupGeneralCategoryBitMap #-}
lookupGeneralCategoryBitMap :: Int -> Int
lookupGeneralCategoryBitMap n =
    lookupWord8AsInt data# (
        lookupWord16AsInt offsets1# (
            lookupWord16AsInt offsets2# (
                n `shiftR` 8
            ) + ((n `shiftR` 3) .&. maskOffsets)
        ) + (n .&. maskData)
    )
    where
    maskData = (1 `shiftL` 3) - 1
    maskOffsets = (1 `shiftL` 5) - 1
    !(Ptr data#) = generalCategoryDataBitMap
    !(Ptr offsets1#) = generalCategoryOffsets1BitMap
    !(Ptr offsets2#) = generalCategoryOffsets2BitMap

generalCategoryDataBitMap :: Ptr Int8
generalCategoryDataBitMap = Ptr
    "\1\0\0\1\0\1\0\0\0\0\0\1\0\1\0\1\1\0\1\1\0\1\0\0\0\1\1\0\0\0\1\1\1\13\18\14\18\13\14\17\17\17\17\17\3\3\3\3\29\3\
    \\3\29\3\3\3\3\3\3\29\3\3\3\3\3\3\17\17\4\3\3\6\5\29\20\1\1\1\1\1\1\1\1\18\1\1\1\1\4\1\1\1\1\1\3\3\0\0\0\0\2\
    \\20\1\20\21\4\15\18\26\21\20\10\4\16\10\10\10\17\0\0\0\0\0\0\0\0\18\1\1\1\1\1\1\29\29\1\1\1\29\1\1\29\29\29\10\10\10\10\10\10\10\
    \\21\10\10\10\10\10\10\10\17\19\17\17\29\29\29\29\4\4\4\4\4\5\3\4\4\4\17\3\3\3\4\1\21\21\1\1\0\0\1\1\0\0\0\1\0\1\0\1\0\1\
    \\0\0\0\0\1\4\4\4\17\17\29\29\29\29\29\5\29\29\29\29\6\29\6\6\6\5\29\29\7\7\0\1\0\1\0\1\1\1\0\1\0\1\0\1\0\1\1\1\1\1\
    \\0\29\0\0\0\0\29\20\20\20\0\0\0\0\0\29\0\29\0\0\0\0\29\0\0\0\0\29\29\0\0\0\0\0\0\0\29\0\0\0\0\0\0\29\29\29\29\29\29\8\
    \\8\17\17\17\17\4\4\5\6\5\4\6\6\6\6\6\5\4\6\5\5\6\6\5\5\29\4\4\4\29\29\29\29\5\5\5\5\5\3\3\5\5\5\5\3\3\21\17\10\10\
    \\10\10\10\10\10\10\4\4\5\5\5\6\29\29\5\6\4\5\5\5\5\17\17\17\3\4\4\4\4\4\4\4\3\5\5\5\5\5\5\5\6\5\5\5\4\4\4\4\5\4\
    \\4\6\6\29\29\5\5\6\5\4\5\5\5\4\4\5\5\4\29\29\29\29\29\3\3\3\0\1\4\4\5\5\6\17\17\29\18\17\17\19\17\17\21\21\17\17\17\13\14\13\
    \\14\18\18\18\18\18\18\18\18\13\14\18\18\18\18\21\21\21\21\21\6\6\5\5\5\5\29\29\6\5\5\17\29\29\29\29\12\13\14\13\14\13\14\17\29\29\29\29\29\29\
    \\29\3\1\1\1\17\0\23\24\26\26\26\26\26\22\17\19\19\19\19\21\17\17\29\29\29\29\29\29\29\5\5\6\6\6\6\29\5\5\6\29\4\4\4\29\4\4\4\4\17\
    \\4\5\5\5\29\29\17\29\0\1\0\1\3\20\0\1\0\1\3\3\5\5\21\21\21\21\21\21\5\21\21\17\9\9\9\9\9\29\29\4\4\4\4\3\17\17\17\19\29\29\
    \\29\29\29\29\29\21\21\21\21\21\21\21\18\18\18\18\18\0\1\1\21\18\21\21\1\21\19\21\29\29\29\29\29\29\0\0\0\0\0\29\1\1\1\1\1\1\3\3\3\3\
    \\3\3\3\3\20\20\20\20\20\20\20\20\3\20\3\20\20\0\1\0\1\4\4\5\5\29\29\29\29\0\1\0\1\0\1\4\5\4\29\29\29\29\29\29\5\5\5\5\5\5\
    \\5\12\5\5\5\5\5\4\29\29\6\17\17\17\17\17\17\17\17\29\26\5\6\6\6\29\29\29\29\4\5\5\5\5\5\5\5\3\5\5\5\5\5\29\29\5\5\5\29\29\
    \\4\4\4\4\4\4\29\17\17\17\19\17\17\17\19\4\5\29\29\29\29\29\29\5\29\5\5\29\5\5\20\20\3\3\4\4\4\4\4\4\21\13\14\21\21\21\21\21\18\18\
    \\18\18\21\21\29\5\5\5\6\6\5\6\5\5\5\5\5\29\5\6\6\29\4\4\4\5\6\6\5\5\6\4\4\4\4\4\6\6\6\6\6\6\6\6\29\6\6\5\5\29\
    \\29\29\29\5\4\29\4\4\4\4\29\4\4\29\4\4\4\4\4\4\4\4\29\4\4\4\4\29\29\29\29\4\4\29\4\29\4\4\29\4\4\29\4\29\29\4\29\29\29\29\
    \\4\4\6\29\29\29\29\20\4\4\4\4\4\4\29\29\29\29\29\29\29\29\6\6\29\29\6\6\5\4\29\5\5\29\29\29\29\1\1\29\1\29\1\1\1\17\3\1\1\1\
    \\20\3\3\3\3\1\1\1\1\1\1\1\0\1\18\0\0\0\0\0\21\21\0\21\21\21\21\0\4\4\4\4\4\29\29\26\26\26\26\26\26\18\18\18\18\18\21\21\18\21\
    \\21\18\21\18\21\21\21\21\21\21\21\18\21\21\21\5\5\5\5\5\26\21\5\5\6\6\29\29\29\29\15\16\13\15\15\16\13\15\16\13\14\13\14\13\14\13\14\13\14\10\
    \\10\18\18\18\13\14\29\4\4\4\4\5\6\29\29\3\17\17\17\17\17\17\21\21\21\21\21\21\21\21\29\29\29\29\29\29\29\17\17\17\17\17\17\17\3\29\29\5\19\19\
    \\18\20\21\19\19\29\1\18\1\1\1\1\1\1\0\0\0\0\0\0\0\1\1\1\1\1\1\29\1\1\1\1\1\0\0\29\0\29\0\29\0\29\0\4\4\4\29\29\29\4\
    \\4\5\5\17\17\8\8\4\4\4\4\4\29\4\4\29\29\5\4\5\5\5\5\4\4\4\4\4\4\5\6\5\5\5\5\5\5\5\5\21\21\21\21\0\21\0\21\0\0\0\
    \\0\21\1\17\12\29\29\21\21\19\10\10\10\10\29\29\29\1\3\20\20\29\29\29\29\4\29\4\29\4\29\4\4\4\4\29\4\29\4\29\29\29\29\20\20\0\17\12\3\3\
    \\3\3\3\21\21\18\10\10\20\1\17\17\18\12\18\18\18\29\29\29\29\29\29\21\21\10\10\10\10\21\21\10\10\21\21\21\29\5\5\5\5\5\5\5\29\8\8\29\10\10\
    \\10\10\10\10\8\8\29\29\29\4\4\4\29\29\29\4\29\29\4\4\4\4\4\29\29\4\29\29\29\29\29\29\6\0\1\29\1\29\1\0\1\29\29\29\29\29\29\29\4\4\
    \\29\29\29\29\6\6\17\29\29\29\12\4\4\4\4\4\4\4\21\17\4\4\4\4\4\4\14\13\22\4\4\4\4\4\4\4\21\17\4\4\9\29\29\29\29\29\22\17\17\17\
    \\21\3\4\9\4\4\4\4\4\4\4\6\6\6\5\5\5\5\6\5\6\6\6\6\5\5\5\5\5\6\6\6\6\5\6\6\6\6\6\29\5\6\6\29\29\29\6\6\6\6\
    \\5\5\29\29\22\22\22\22\22\22\22\22\26\26\26\26\26\26\26\26\29\26\26\26\21\21\29\29\29\17\17\8\8\17\17\29\29\29\29\18\18\18\18\18\18\13\14\13\14\13\
    \\14\12\29\29\0\0\0\0\0\20\20\20\0\1\0\1\0\1\0\0\1\22\17\17\17\19\17\17\17\4\17\4\4\5\4\4\4\5\4\4\4\4\4\4\4\5\5\5\5\5\
    \\5\5\17\3\4\4\29\29\5\4\6\6\6\26\26\26\26\26\29\29\29\29\29\29\1\1\1\1\1\1\1\29\29\29\29\29\5\5\29\6\6\5\6\5\5\4\4\17\4\5\
    \\5\5\29\5\5\29\4\18\4\4\4\4\4\4\29\4\29\29\29\4\4\3\17\17\17\17\17\17\17\13\4\4\4\4\4\29\29\10\21\4\4\4\4\4\4\4\10\10\10\10\
    \\29\29\29\17\17\17\17\29\29\29\8\8\10\10\17\17\17\21\4\4\4\4\4\4\4\5\5\29\29\8\8\6\6\6\5\21\21\5\5\5\17\26\17\17\17\5\29\29\29\29\
    \\29\26\29\29\29\29\29\29\1\29\29\5\5\5\5\5\5\4\5\4\4\5\5\5\5\5\5\6\4\4\4\4\17\17\17\17\17\29\29\29\29\29\0\29\29\0\0\29\4\4\
    \\29\5\5\12\29\29\6\5\5\5\5\5\5\6\6\4\4\4\4\19\21\21\21\21\21\21\21\19\19\19\19\19\19\19\19\4\4\29\29\5\4\6\5\5\5\5\6\6\6\5\
    \\4\4\4\29\29\5\5\5\21\21\5\5\5\5\21\21\21\6\6\6\6\6\6\4\4\4\4\4\4\9\9\9\9\9\9\9\29\4\5\5\5\5\5\5\29\21\21\1\0\0\
    \\0\1\1\4\0\1\1\1\0\0\1\0\0\1\0\0\0\0\2\20\20\29\8\8\10\10\10\10\10\10\21\5\21\5\5\5\5\4\4\4\5\17\29\8\8\4\4\4\4\4\
    \\4\3\4\4\4\4\4\5\5\4\6\6\6\4\4\6\6\6\6\6\29\29\6\6\5\29\29\12\12\12\12\12\12\17\17\18\17\11\17\17\17\18\13\14\17\18\17\12\17\17\
    \\17\29\17\17\17\17\4\6\6\4\6\5\17\17\17\29\8\8\10\29\29\29\21\21\10\10\10\10\10\10\21\21\21\21\21\21\21\29\21\21\21\21\21\17\17\29\13\14\21\21\
    \\13\14\13\14\12\13\14\14\17\17\17\17\11\11\11\5\10\10\10\10\10\10\10\29\29\29\29\29\29\29\10\10\4\4\29\29\5\29\6\6\6\6\5\5\6\5\5\5\5\29\
    \\6\6\4\4\4\17\17\17\9\9\5\5\5\5\6\6\6\6\29\29\6\6\6\29\29\29\29\29\29\4\4\4\6\6\5\5\6\6\5\5\4\4\4\4\5\5\6\5\5\6\
    \\5\5\29\5\5\29\5\5\5\5\29\29\29\29\19\5\5\4\6\4\4\4\4\6\6\5\5\5\29\5\29\0\0\0\0\0\0\0\0\13\17\14\20\11\1\1\1\1\1\1\
    \\1\1\13\18\14\18\25\25\25\25\25\25\25\25\8\8\29\29\4\4\4\4\0\2\1\0\2\1\0\1\0\0\0\1\21\0\21\21\4\5\6\6\29\4\4\4\4\4\29\29\
    \\4\4\4\4\4\10\10\17\6\29\6\6\6\5\4\21\21\21\17\17\17\17\21\17\21\21\21\21\29\29\21\21\22\4\4\4\4\4\4\4\13\14\29\29\29\4\5\6\6\17\
    \\4\4\4\5\6\5\6\6\5\5\5\5\5\6\6\6\5\6\5\17\17\17\17\17\17\29\29\29\29\10\12\17\13\17\17\17\17\17\18\17\11\17\17\17\17\17\17\17\22\5\
    \\5\5\5\5\5\7\5\7\7\7\5\5\5\5\5\7\7\7\17\5\5\5\5\17\6\5\5\5\29\29\29\29\29\8\8\29\29\29\29\4\21\9\9\9\9\9\9\9\9\0\
    \\1\9\9\9\3\4\17\21\21\0\1\0\1\1\21\21\21\0\1\0\1\5\10\10\10\10\17\17\17\0\1\21\5\5\5\5\5\5\17\17\17\17\17\17\17\12\17\15\16\17\
    \\17\17\17\11\6\6\6\6\6\6\5\6\6\4\4\4\6\6\5\6\6\17\6\5\29\29\29\29\29\29\29\4\4\20\20\20\20\20\20\29\29\29\29\29\4\5\4\17\4\6\
    \\29\29\29\9\9\9\9\9\10\10\10\10\29\29\29\21\3\29\3\3\3\3\3\3\29\29\29\29\29\4\4\4\4\4\29\3\29\5\5\4\6\4\4\4\4\5\5\6\6\17\
    \\26\26\26\5\5\5\5\5\21\21\5\5\5\21\29\29\8\8\17\17\17\17\17\17\17\5\5\5\26\5\5\5\5\5\21\21\21\21\21\5\21\21\1\1\1\1\5\5\5\5\
    \\3\29\29\29\29\5\5\5\5\5\5\29\29\29\29\5\29\29\29\29\29\29\21\21\21\21\20\20\20\20\20\8\8\8\8\8\8\8\8\17\17\18\18\18\17\3\3\20\20\20\
    \\20\3\3\20\20\20\20\20\20\4\4\19\19\10\10\10\10\21\19\4\17\5\29\21\4\10\10\10\10\10\10\21\5\21\5\13\14\13\14\6\6\3\3\1\4\4\4\4\4\6\
    \\5\6\6\5\29\29\17\17\17\17\10\17\17\17\17\17\17\5\4\4\4\5\17\17\4\29\8\8\4\4\4\21\21\4\6\5\6\4\4\4\4\4\6\5\6\5\6\6\5\5\
    \\5\6\5\4\4\29\29\21\5\5\17\17\17\4\29\29\29\0\0\0\0\2\20\20\20\1\1\1\29\1\1\10\3\29\29\10\10\10\10\18\18\18\13\14\3\18\18\18\18\18\
    \\21\21\21\18\18\21\21\21\21\21\21\5\29\29\29\8\8\29\29\29\29\17\17\15\16\17\15\16\17\17\15\16\15\16\17\17\8\8\29\29\17\17\17\17\15\16\17\15\16\17\
    \\17\17\17\17\17\17\12\12\17\17\17\17\17\17\29\3\8\8\4\17\4\17\17\17\12\12\11\11\13\14\13\14\13\14\17\17\13\14\17\13\14\17\17\4\4\4\4\4\4\3\
    \\3\3\3\17\21\29\29\5\6\29\6\6\5\29\29\5\5\5\5\5\17\5\5\5\5\6\6\5\5\17\17\26\17\17\5\29\29\29\29\29\10\10\29\4\4\4\4\4\6\5\
    \\5\6\6\5\5\6\6\29\4\4\4\29\5\5\4\6\6\8\8\17\17\29\17\5\4\4\4\4\4\5\5\5\5\17\17\17\4\17\17\17\29\29\29\29\29\4\4\4\4\6\
    \\6\5\5\5\29\29\29\6\6\5\5\5\3\3\3\3\3\17\3\5\29\29\29\1\18\1\1\1\1\1\1\0\1\29\29\8\8\4\4\29\29\29\29\29\17\8\8\29\29\29\
    \\29\29\29\21\21\21\20\20\20\20\20\8\8\17\17\18\18\18\17\0\0\0\0\0\0\0\29\0\0\29\1\17\5\5\17\5\5\17\5\5\5\5\5\5\12\5\5\5\5\5\
    \\17\29\29\9\9\9\0\1\9\9\9\10\21\21\29\29\29\29\13\14\13\14\21\21\21\21\0\1\0\1\5\5\0\1\29\29\29\29\5\5\6\6\29\4\4\4\17\29\29\29\
    \\29\29\29\2\2\2\2\2\2\2\2\27\27\27\27\27\27\27\27\28\28\28\28\28\28\28\28\4\4\4\4\4\4\5\29\8\8\29\29\29\29\29\19"#

generalCategoryOffsets1BitMap :: Ptr Word16
generalCategoryOffsets1BitMap = Ptr
    "\8\6\146\4\96\3\96\3\96\3\96\3\6\6\108\1\214\9\146\3\184\6\134\1\96\3\96\3\96\3\96\3\96\3\7\6\127\1\115\11\198\6\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\145\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\54\0\44\0\26\8\96\3\96\3\96\3\96\3\96\3\95\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\97\3\249\4\48\4\96\3\96\3\96\3\96\3\48\4\48\4\48\4\48\4\52\4\146\3\96\3\96\3\104\7\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\
    \\48\4\48\4\50\4\146\3\146\3\146\3\48\4\52\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\146\3\48\4\54\4\48\4\48\4\48\4\48\4\48\4\146\3\48\4\48\4\48\4\50\4\54\4\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\103\3\97\3\103\3\
    \\96\3\96\3\96\3\96\3\96\3\103\3\96\3\96\3\96\3\96\3\103\3\97\3\103\3\96\3\97\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\144\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\142\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\188\9\237\1\
    \\96\3\100\3\96\3\96\3\97\3\96\3\96\3\89\3\96\3\140\3\96\3\140\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\143\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\164\6\146\3\146\3\146\3\182\5\182\5\182\5\182\5\182\5\182\5\96\3\96\3\96\3\140\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\144\3\146\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\140\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\144\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\97\3\146\5\209\2\151\5\20\9\195\5\21\10\96\3\96\3\96\3\140\3\141\3\146\3\96\3\96\3\96\3\96\3\
    \\96\3\142\3\96\3\96\3\96\3\144\3\21\10\174\7\48\4\48\4\48\4\48\4\48\4\48\4\195\1\195\1\223\3\185\10\48\4\48\4\48\4\48\4\176\8\48\4\48\4\48\4\189\7\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\236\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\
    \\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\236\3\48\4\234\3\48\4\48\4\48\4\48\4\48\4\48\4\195\1\199\2\9\2\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\207\0\146\3\96\3\96\3\96\3\96\3\96\3\96\3\182\5\182\5\148\4\157\4\251\9\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\207\0\146\3\96\3\96\3\96\3\96\3\143\3\146\3\43\5\146\3\146\3\146\3\40\5\146\3\171\9\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\142\3\146\3\96\3\96\3\97\3\172\9\96\3\96\3\96\3\96\3\96\3\142\3\96\3\96\3\96\3\96\3\96\3\141\3\218\2\164\9\21\10\29\9\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\55\12\146\3\146\3\96\3\96\3\96\3\96\3\96\3\5\6\21\10\63\12\96\3\96\3\104\7\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\141\3\48\4\48\4\48\4\48\4\48\4\48\4\49\4\146\3\96\3\96\3\96\3\96\3\96\3\59\0\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\7\196\6\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\144\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\141\3\96\3\96\3\96\3\96\3\96\3\96\3\145\3\146\3\223\7\92\1\92\1\227\7\117\0\117\0\117\0\117\0\95\4\74\0\74\0\74\0\238\9\243\9\21\10\205\10\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\142\0\92\1\92\1\92\1\92\1\92\1\92\1\146\0\185\4\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\124\5\214\1\246\7\
    \\83\9\199\2\199\2\135\6\146\3\146\3\146\3\146\3\142\0\92\1\92\1\92\1\92\1\144\0\92\1\225\7\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\142\0\92\1\92\1\92\1\92\1\92\1\92\1\146\0\185\4\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\123\1\21\10\179\11\80\5\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\83\5\96\3\96\3\98\5\96\3\96\3\189\8\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\4\8\123\5\145\3\48\4\48\4\48\4\49\4\92\1\186\7\48\4\48\4\48\4\92\1\150\0\92\1\48\4\48\4\48\4\48\4\92\1\186\7\
    \\48\4\48\4\48\4\48\4\150\0\92\1\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\236\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\17\4\19\4\92\1\92\1\92\1\184\7\48\4\48\4\48\4\48\4\48\4\198\1\
    \\195\1\195\1\195\1\197\1\17\4\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\212\5\16\4\16\4\194\1\195\1\195\1\195\1\195\1\195\1\195\1\195\1\191\1\195\1\
    \\195\1\195\1\199\1\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\17\4\19\4\92\1\92\1\92\1\184\7\48\4\48\4\48\4\48\4\48\4\198\1\195\1\195\1\195\1\
    \\197\1\17\4\195\1\195\1\105\4\18\1\25\1\122\2\74\0\74\0\74\0\14\1\4\1\21\1\126\2\74\0\74\0\86\4\117\0\117\0\91\4\74\0\74\0\74\0\117\0\117\0\117\0\95\4\74\0\74\0\86\4\117\0\117\0\91\4\74\0\74\0\74\0\117\0\117\0\117\0\95\4\74\0\74\0\126\0\117\0\117\0\117\0\124\0\74\0\74\0\
    \\79\0\88\4\117\0\117\0\122\0\96\3\96\3\96\3\96\3\5\6\146\3\21\10\179\11\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\92\1\92\1\92\1\224\7\96\3\96\3\96\3\96\3\96\3\212\6\144\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\177\2\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\144\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\145\3\146\3\146\3\146\3\146\3\146\3\146\3\117\0\117\0\117\0\117\0\117\0\117\0\34\1\146\3\74\0\74\0\74\0\74\0\74\0\74\0\42\6\141\0\182\5\182\5\182\5\182\5\182\5\182\5\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\46\2\243\1\215\4\6\1\89\4\117\0\28\1\93\4\74\0\74\0\74\0\187\3\25\0\192\0\192\0\192\0\
    \\190\3\185\0\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\9\37\7\196\6\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\142\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\97\3\88\3\96\3\97\3\34\5\88\3\96\3\96\3\96\3\70\3\146\1\166\7\146\3\21\10\179\11\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\236\2\96\3\96\3\96\3\96\3\130\5\16\2\141\9\146\3\146\3\146\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\9\6\44\4\33\11\146\3\21\10\15\5\74\11\96\3\96\3\26\8\96\3\96\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\117\0\117\0\117\0\117\0\74\0\74\0\74\0\74\0\92\1\92\1\151\0\198\6\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\96\3\96\3\96\3\140\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\144\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\142\3\56\4\117\0\27\1\117\0\27\1\207\11\74\0\101\4\74\0\101\4\135\0\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\117\0\117\0\117\0\117\0\117\0\
    \\117\0\74\0\74\0\74\0\74\0\74\0\74\0\192\0\192\0\192\0\192\0\75\9\228\0\192\0\192\0\192\0\192\0\192\0\192\0\191\0\239\0\192\0\192\0\192\0\192\0\192\0\192\0\22\1\117\0\117\0\117\0\23\1\40\4\74\0\74\0\74\0\74\0\74\0\178\4\181\2\157\4\157\4\157\4\157\4\183\2\215\11\146\3\96\3\96\3\96\3\
    \\106\3\205\0\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\164\6\146\3\146\3\146\3\182\5\182\5\182\5\182\5\182\5\182\5\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\158\4\244\3\157\4\157\4\157\4\157\4\157\4\160\4\61\2\48\4\62\2\197\6\146\3\179\2\181\2\157\4\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\38\6\146\3\35\6\135\9\96\3\70\6\97\3\73\6\90\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\127\9\142\2\132\9\146\3\172\9\96\3\96\3\96\3\96\3\96\3\96\3\96\3\90\5\48\4\48\4\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\236\2\96\3\96\3\96\3\96\3\96\3\96\3\86\2\146\3\146\3\146\3\146\3\96\3\229\6\48\4\252\11\48\4\48\4\
    \\193\10\22\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\239\3\48\4\48\4\48\4\97\2\195\1\195\1\205\1\48\4\48\4\48\4\48\4\96\2\193\10\48\4\48\4\48\4\48\4\49\4\146\3\146\3\146\3\48\4\53\4\146\3\146\3\92\1\92\1\92\1\92\1\92\1\92\1\92\1\184\7\48\4\48\4\48\4\48\4\48\4\
    \\48\4\48\4\48\4\48\4\180\7\92\1\92\1\96\3\96\3\104\6\110\6\96\3\96\3\96\3\171\11\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\234\7\92\1\92\1\141\0\92\1\92\1\92\1\92\1\92\1\183\7\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\
    \\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\50\4\146\3\146\3\146\3\146\3\146\3\146\3\243\4\48\4\48\4\48\4\48\4\48\4\52\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\52\4\146\3\48\4\49\4\92\2\48\4\92\2\48\4\92\2\48\4\48\4\48\4\50\4\146\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\143\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\140\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\80\5\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\103\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\8\7\199\2\91\1\92\1\226\7\96\3\96\3\48\4\54\4\117\0\117\0\117\0\117\0\117\0\117\0\117\0\117\0\117\0\117\0\31\1\126\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\240\0\188\3\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\
    \\192\0\192\0\192\0\192\0\193\0\193\0\236\0\192\0\192\0\192\0\192\0\192\0\9\0\0\0\21\0\66\7\62\7\194\0\16\0\189\0\58\7\120\8\125\8\193\0\11\0\192\0\192\0\126\8\192\0\192\0\192\0\192\0\192\0\192\0\242\0\64\7\197\0\192\0\74\0\74\0\74\0\74\0\74\0\74\0\74\0\74\0\83\0\74\0\74\0\74\0\
    \\134\2\134\2\35\10\134\2\140\2\142\2\137\2\146\2\142\2\142\2\143\2\134\2\230\5\192\0\236\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\186\3\233\5\192\0\152\2\232\0\192\0\192\0\4\0\6\0\192\0\196\0\58\5\53\5\59\5\146\3\146\3\165\1\81\10\96\3\96\3\96\3\105\6\93\1\146\3\96\3\96\3\3\6\157\4\
    \\67\9\9\2\146\3\146\3\96\3\96\3\111\11\9\2\146\3\146\3\146\3\146\3\96\3\96\3\105\6\227\7\146\3\146\3\96\3\96\3\97\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\92\1\92\1\92\1\224\7\96\3\96\3\96\3\96\3\96\3\212\6\144\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\177\2\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\70\1\67\3\68\3\68\3\68\3\68\3\68\3\68\3\11\2\143\11\134\2\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\149\11\146\3\23\8\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\117\0\117\0\117\0\117\0\74\0\74\0\74\0\74\0\
    \\92\1\92\1\151\0\198\6\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\108\8\108\8\108\8\108\8\241\5\150\7\21\10\27\10\116\0\117\0\117\0\88\8\73\0\74\0\74\0\101\8\108\8\108\8\108\8\108\8\2\2\102\0\230\4\109\0\117\0\117\0\118\0\90\4\74\0\74\0\75\0\74\0\217\3\180\1\
    \\157\4\151\6\96\3\96\3\96\3\96\3\118\1\6\6\157\4\157\4\21\10\43\1\1\6\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\33\2\247\3\77\1\89\7\21\10\116\10\199\2\201\2\0\6\96\3\96\3\96\3\157\4\157\4\157\4\233\2\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\3\6\157\4\174\2\146\3\21\10\101\7\96\3\96\3\96\3\6\6\84\1\68\4\96\3\96\3\3\6\223\2\221\2\225\2\199\2\200\2\96\3\96\3\96\3\38\2\96\3\143\3\96\3\96\3\96\3\139\3\30\6\157\4\96\3\96\3\96\3\96\3\96\3\125\1\157\4\157\4\222\9\157\4\157\4\157\4\188\6\96\3\96\3\96\3\96\3\96\3\
    \\96\3\49\1\156\4\148\5\8\6\96\3\125\4\21\10\117\1\96\3\140\8\38\5\35\5\96\3\96\3\94\3\77\6\18\6\215\1\155\3\147\3\109\3\137\6\21\10\49\10\55\10\24\2\106\3\35\5\96\3\96\3\94\3\116\3\240\7\78\3\230\2\4\10\206\4\37\1\21\10\93\7\146\3\24\2\98\3\93\3\96\3\96\3\94\3\91\3\18\6\
    \\43\3\41\11\145\3\146\3\137\6\21\10\84\2\216\2\49\3\38\5\35\5\96\3\96\3\94\3\91\3\249\6\215\1\127\7\13\2\109\3\137\6\21\10\63\10\146\3\83\3\119\4\103\3\112\3\65\5\119\4\96\3\68\5\160\5\220\0\45\5\146\3\37\1\21\10\185\7\113\2\3\7\99\3\94\3\96\3\96\3\94\3\96\3\139\4\19\2\56\8\
    \\45\6\40\5\137\6\21\10\56\4\143\0\197\8\99\3\94\3\96\3\96\3\94\3\100\3\249\6\154\5\75\3\149\3\63\5\137\6\21\10\131\3\146\3\225\6\99\3\94\3\96\3\96\3\96\3\96\3\112\7\252\7\160\8\27\8\93\1\137\6\21\10\92\1\101\6\49\3\96\3\97\3\236\2\96\3\96\3\93\3\74\6\97\3\213\0\75\8\68\3\
    \\37\1\21\10\72\5\146\3\95\3\96\3\96\3\96\3\96\3\96\3\181\6\59\8\120\1\9\6\21\10\201\5\146\3\146\3\146\3\146\3\112\3\101\3\96\3\96\3\207\4\96\3\181\6\190\2\175\9\7\5\21\10\116\8\146\3\146\3\146\3\146\3\166\8\199\2\171\8\58\2\21\10\80\7\84\7\73\10\96\3\95\3\96\3\96\3\96\3\141\3\
    \\181\2\127\1\49\11\4\6\157\4\181\2\157\4\157\4\157\4\47\7\60\2\190\7\43\4\8\2\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\72\8\155\4\35\8\21\10\212\9\125\5\40\8\116\7\25\7\144\4\96\3\55\3\55\1\21\10\143\6\117\0\117\0\117\0\117\0\24\1\201\6\74\0\74\0\74\0\74\0\74\0\174\3\
    \\96\3\96\3\100\1\61\5\96\3\96\3\172\1\146\3\96\3\96\3\159\2\146\3\96\3\99\3\161\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\42\8\221\6\100\9\156\4\111\1\249\2\21\10\179\11\92\1\229\7\85\9\218\9\21\10\179\11\96\3\96\3\96\3\96\3\106\7\96\3\96\3\96\3\96\3\96\3\96\3\145\3\110\7\96\3\
    \\96\3\96\3\96\3\173\2\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\140\3\146\3\96\3\96\3\2\6\90\10\96\3\96\3\96\3\96\3\96\3\96\3\84\10\7\5\205\8\144\5\141\5\226\2\21\10\179\11\21\10\179\11\63\4\221\8\157\4\251\8\157\4\7\5\146\3\146\3\146\3\146\3\146\3\146\3\187\6\96\3\96\3\96\3\
    \\96\3\96\3\151\4\135\5\104\9\141\3\21\10\212\9\47\4\244\3\161\4\196\7\60\3\96\3\96\3\96\3\254\6\132\1\21\10\101\7\96\3\96\3\96\3\96\3\149\4\213\8\254\3\59\4\96\3\96\3\96\3\96\3\64\3\140\5\55\11\60\4\21\10\24\5\21\10\101\7\96\3\96\3\96\3\59\0\74\0\60\5\117\0\117\0\117\0\117\0\
    \\117\0\18\1\199\2\146\3\51\11\157\4\155\4\137\1\138\1\159\1\74\0\74\0\74\0\74\0\74\0\130\2\134\2\134\2\134\2\134\2\134\2\134\2\134\2\184\3\74\0\186\3\74\0\74\0\74\0\131\2\134\2\134\2\134\2\134\2\157\4\157\4\157\4\157\4\157\4\157\4\157\4\157\4\74\0\117\0\126\0\31\1\74\0\117\0\74\0\117\0\
    \\126\0\31\1\74\0\111\4\74\0\117\0\74\0\126\0\74\0\31\12\74\0\31\12\74\0\31\12\98\4\95\0\163\10\157\10\128\0\252\0\74\0\224\5\132\0\72\7\174\5\179\5\135\7\6\4\199\2\251\1\199\2\92\9\145\7\199\2\141\7\243\8\185\5\182\5\171\10\177\10\92\1\25\4\134\2\165\9\241\6\241\6\241\6\241\6\85\2\146\3\
    \\157\4\6\9\1\9\157\4\119\9\146\3\202\3\53\7\132\8\196\3\165\4\171\4\200\0\180\0\100\2\106\2\92\1\92\1\37\9\37\9\37\9\37\9\42\9\244\11\185\10\191\10\227\3\237\3\48\4\48\4\48\4\94\2\231\3\48\4\48\4\48\4\96\2\195\1\117\0\117\0\117\0\117\0\117\0\117\0\74\0\74\0\74\0\74\0\74\0\74\0\
    \\20\0\195\0\12\0\89\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\192\0\55\9\60\9\8\12\95\10\74\0\74\0\74\0\74\0\97\4\167\6\96\3\96\3\96\3\96\3\96\3\96\3\96\3\238\1\237\1\11\2\96\3\96\3\97\3\146\3\97\3\97\3\97\3\97\3\97\3\97\3\97\3\97\3\157\4\157\4\
    \\157\4\157\4\218\10\212\10\84\9\89\9\13\4\37\0\199\2\244\10\232\8\199\2\186\1\216\5\146\3\146\3\146\3\146\3\48\4\48\4\48\4\193\7\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\52\4\146\3\116\5\17\4\204\7\208\7\36\9\10\8\223\4\47\9\95\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\97\3\9\3\80\5\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\173\0\96\3\77\2\96\3\96\3\21\10\169\11\146\3\146\3\192\0\192\0\192\0\192\0\192\0\167\2\11\9\10\6\192\0\192\0\192\0\52\2\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\31\7\37\9\82\9\146\3\251\5\254\5\
    \\96\3\96\3\31\8\197\10\182\7\114\2\96\3\96\3\96\3\96\3\96\3\96\3\192\6\146\3\29\7\96\3\96\3\96\3\96\3\96\3\64\3\68\3\166\5\57\4\21\10\179\11\157\4\157\4\147\4\246\5\21\10\101\7\96\3\96\3\3\6\78\9\96\3\96\3\2\6\157\4\254\3\56\4\96\3\96\3\96\3\141\3\188\6\96\3\96\3\96\3\
    \\96\3\96\3\54\3\83\11\198\2\248\10\21\10\205\10\166\0\96\3\21\10\131\4\96\3\96\3\96\3\96\3\96\3\184\6\62\1\146\3\254\5\33\4\21\10\226\10\96\3\96\3\15\3\121\10\96\3\96\3\96\3\96\3\96\3\96\3\152\1\107\11\173\2\146\3\146\3\80\6\96\3\79\11\65\0\146\3\237\2\237\2\237\2\146\3\97\3\97\3\
    \\74\0\74\0\74\0\74\0\74\0\179\3\74\0\193\4\74\0\74\0\74\0\74\0\74\0\74\0\74\0\74\0\74\0\74\0\96\3\96\3\96\3\96\3\109\9\114\9\21\10\179\11\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\
    \\39\12\39\12\39\12\39\12\39\12\39\12\39\12\39\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\47\12\157\4\157\4\86\6\236\1\157\4\157\4\7\11\16\4\13\11\215\7\
    \\156\7\230\1\236\4\158\0\99\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\210\3\244\2\150\7\21\10\27\10\116\0\117\0\117\0\88\8\73\0\74\0\74\0\30\0\21\11\96\3\118\1\96\3\96\3\96\3\96\3\27\11\96\3\96\3\96\3\97\3\236\2\236\2\236\2\27\5\
    \\74\4\29\3\146\3\190\5\224\8\92\1\92\1\92\1\92\1\92\1\154\9\48\4\37\9\37\9\37\9\37\9\37\9\37\9\149\9\187\7\48\4\255\4\48\4\51\4\55\4\146\3\146\3\146\3\146\3\146\3\48\4\48\4\48\4\48\4\48\4\196\10\96\3\96\3\96\3\96\3\227\7\26\8\96\3\96\3\122\5\108\5\96\3\96\3\96\3\96\3\
    \\3\6\21\9\96\3\96\3\96\3\238\2\96\3\96\3\96\3\96\3\142\3\96\3\69\2\146\3\146\3\146\3\146\3\146\3\117\0\117\0\117\0\117\0\117\0\74\0\74\0\74\0\74\0\74\0\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\140\3\21\10\179\11\117\0\117\0\117\0\117\0\33\1\74\0\74\0\74\0\74\0\41\6\
    \\96\3\96\3\96\3\96\3\96\3\96\3\97\3\146\3\96\3\96\3\140\3\146\3\96\3\146\3\146\3\146\3\52\0\134\2\134\2\134\2\134\2\134\2\50\0\167\9\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\140\3\94\3\96\3\96\3\96\3\96\3\98\3\31\5\96\3\96\3\238\2\92\1\96\3\96\3\81\5\150\0\96\3\96\3\
    \\96\3\97\3\231\7\92\1\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\135\4\140\0\62\6\178\2\100\3\95\3\96\3\96\3\140\3\254\9\92\1\230\7\199\2\237\1\96\3\96\3\96\3\152\8\96\3\96\3\96\3\105\6\146\3\146\3\146\3\146\3\96\3\102\6\96\3\96\3\134\6\140\0\200\2\146\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\140\3\62\4\96\3\96\3\140\3\92\1\96\3\96\3\143\3\92\1\96\3\96\3\144\3\116\6\146\3\142\0\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\124\10\96\3\96\3\96\3\96\3\96\3\96\3\157\4\9\6\221\8\141\0\92\1\18\5\21\10\156\1\11\2\60\3\96\3\96\3\96\3\96\3\96\3\
    \\131\5\60\11\157\6\160\6\96\3\96\3\96\3\145\3\21\10\179\11\146\4\96\3\96\3\96\3\2\6\130\1\9\5\21\10\160\7\146\3\96\3\96\3\96\3\96\3\108\10\146\3\60\3\96\3\96\3\96\3\96\3\96\3\128\5\127\1\191\6\14\9\21\10\0\11\142\0\92\1\226\7\146\3\96\3\96\3\93\3\96\3\96\3\127\5\38\3\101\10\
    \\253\2\146\3\146\3\146\3\146\3\146\3\146\3\146\3\97\3\86\3\96\3\98\3\96\3\23\12\96\3\96\3\96\3\96\3\96\3\2\6\141\5\21\9\21\10\179\11\87\11\38\5\35\5\96\3\96\3\94\3\91\3\93\11\15\8\19\8\45\5\26\8\143\1\252\9\252\9\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\126\5\157\4\136\10\193\6\21\10\101\11\144\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\141\5\137\5\55\6\146\3\21\10\179\11\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\141\5\37\3\149\10\146\3\21\10\179\11\
    \\199\2\196\6\146\3\146\3\96\3\96\3\96\3\96\3\96\3\202\8\128\1\23\12\21\10\179\11\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\8\7\132\5\58\8\21\10\124\6\97\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\96\3\96\3\96\3\96\3\96\3\127\5\157\4\222\1\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\117\0\117\0\117\0\117\0\74\0\74\0\74\0\74\0\21\10\80\7\228\7\61\5\96\3\94\3\96\3\96\3\96\3\124\5\7\5\128\1\195\6\146\3\21\10\80\7\92\1\226\7\25\11\96\3\
    \\96\3\96\3\180\2\157\4\157\4\219\6\47\8\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\97\3\93\3\96\3\96\3\96\3\96\3\45\7\3\3\175\6\146\3\21\10\179\11\98\3\94\3\96\3\96\3\96\3\121\7\50\6\145\3\21\10\179\11\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\67\8\96\3\
    \\94\3\96\3\96\3\96\3\71\8\135\11\218\8\199\2\21\10\179\11\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\145\3\146\3\92\1\92\1\183\7\236\6\233\6\48\4\54\4\56\4\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\97\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\145\3\96\3\96\3\96\3\97\3\21\10\205\10\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\97\3\21\10\179\11\96\3\96\3\96\3\140\3\229\11\146\3\96\3\96\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\140\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\143\3\96\3\141\3\96\3\145\3\96\3\143\10\28\6\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\157\4\157\4\157\4\157\4\157\4\251\9\157\4\157\4\7\5\146\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\52\4\146\3\146\3\146\3\146\3\146\3\146\3\146\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\
    \\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\50\4\146\3\48\4\48\4\48\4\48\4\49\4\92\2\48\4\48\4\48\4\48\4\48\4\48\4\209\1\20\7\24\6\196\9\13\7\161\4\48\4\48\4\48\4\16\7\48\4\48\4\48\4\48\4\48\4\48\4\
    \\48\4\53\4\146\3\146\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\204\9\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\92\1\92\1\227\7\146\3\92\1\92\1\227\7\146\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\49\4\146\3\92\1\92\1\
    \\92\1\230\7\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\117\0\117\0\117\0\95\4\74\0\74\0\86\4\117\0\117\0\91\4\98\4\74\0\74\0\117\0\117\0\117\0\95\4\74\0\74\0\246\0\204\6\11\1\91\4\169\3\99\4\74\0\117\0\117\0\117\0\95\4\74\0\74\0\
    \\74\0\74\0\77\0\86\4\117\0\117\0\120\0\74\0\74\0\75\0\84\4\117\0\117\0\118\0\74\0\74\0\74\0\82\4\117\0\117\0\117\0\124\0\74\0\74\0\79\0\163\11\21\10\21\10\21\10\21\10\21\10\21\10\74\0\85\0\74\0\38\6\33\6\42\6\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\
    \\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\7\5\157\4\157\4\46\11\52\8\21\9\134\2\134\2\134\2\134\2\134\2\134\2\134\2\164\9\146\3\146\3\146\3\11\2\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\96\3\96\3\96\3\96\3\
    \\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\96\3\94\6\92\1\7\5\146\3\146\3\146\3\146\3\146\3\100\3\96\3\96\3\96\3\120\3\95\3\101\3\211\4\125\3\202\4\120\3\200\4\120\3\101\3\101\3\206\4\96\3\93\3\96\3\142\3\28\2\93\3\
    \\96\3\142\3\146\3\146\3\146\3\146\3\146\3\146\3\241\4\146\3\53\4\146\3\48\4\48\4\48\4\48\4\48\4\52\4\48\4\55\4\54\4\146\3\50\4\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\146\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\
    \\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\13\10\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\
    \\48\4\89\2\48\4\51\4\48\4\51\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\49\4\90\2\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\54\4\48\4\52\4\55\4\146\3\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\52\4\146\3\
    \\48\4\50\4\48\4\51\4\48\4\55\4\48\4\48\4\48\4\48\4\48\4\189\7\50\4\243\4\48\4\52\4\48\4\55\4\48\4\55\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\48\4\192\7\48\4\48\4\48\4\48\4\48\4\48\4\53\4\146\3\146\3\146\3\146\3\
    \\21\10\179\11"#

generalCategoryOffsets2BitMap :: Ptr Word16
generalCategoryOffsets2BitMap = Ptr
    "\112\9\149\8\180\8\140\5\159\6\185\6\144\9\176\9\208\9\240\9\16\10\48\10\80\10\112\10\144\10\176\10\208\10\27\5\188\0\92\8\213\3\27\5\232\3\240\10\16\11\174\1\48\11\80\11\112\11\144\11\124\8\176\11\208\11\240\11\81\4\104\7\133\7\235\1\221\1\51\4\129\4\97\4\81\4\202\1\16\12\48\12\80\12\126\0\112\12\64\0\
    \\8\4\129\4\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\102\0\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\
    \\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\
    \\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\94\0\27\5\27\5\27\5\223\2\27\5\144\12\210\8\176\12\208\12\240\12\16\13\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\
    \\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\130\2\48\13\48\13\48\13\48\13\48\13\48\13\48\13\48\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\80\13\27\5\
    \\142\1\45\7\27\5\72\7\112\13\144\13\44\1\176\13\35\3\208\13\240\13\127\6\27\5\16\14\48\14\165\7\80\14\112\14\80\5\231\4\251\4\242\8\144\14\176\14\208\14\240\14\16\15\108\3\48\15\80\15\112\15\236\5\0\0\11\2\144\15\176\15\12\1\208\15\27\5\27\5\27\5\15\3\255\2\255\0\8\1\8\1\8\1\8\1\8\1\8\1\
    \\8\1\8\1\8\1\27\2\27\5\27\5\27\5\27\5\59\2\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\27\5\27\5\240\15\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\
    \\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\27\5\27\5\16\16\12\6\8\1\8\1\36\6\49\9\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\101\6\27\5\27\5\27\5\27\5\48\16\30\0\8\1\8\1\
    \\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\32\0\27\5\105\2\119\2\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\80\16\8\1\8\1\8\1\8\1\8\1\
    \\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\112\16\144\16\176\16\208\16\240\16\16\17\180\4\199\4\48\17\129\4\129\4\13\7\8\1\8\1\8\1\8\1\80\17\112\17\162\2\176\2\8\1\181\3\8\1\8\1\205\5\144\17\67\3\8\1\8\1\85\3\140\3\176\17\8\1\225\7\196\7\208\17\240\17\
    \\129\4\129\4\16\18\48\18\156\0\129\4\80\18\112\18\8\1\8\1\8\1\8\1\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\
    \\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\
    \\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\
    \\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\223\0\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\215\0\108\1\27\5\27\5\27\5\
    \\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\120\1\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\61\5\8\1\8\1\
    \\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\8\1\27\5\27\5\30\8\8\1\8\1\8\1\8\1\8\1\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\66\1\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\27\5\
    \\27\5\27\5\27\5\76\1\112\5\134\5"#

