{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The base class for all widgets.
-- 
-- It manages the widget lifecycle, layout, states and style.
-- 
-- === Height-for-width Geometry Management
-- 
-- GTK uses a height-for-width (and width-for-height) geometry management
-- system. Height-for-width means that a widget can change how much
-- vertical space it needs, depending on the amount of horizontal space
-- that it is given (and similar for width-for-height). The most common
-- example is a label that reflows to fill up the available width, wraps
-- to fewer lines, and therefore needs less height.
-- 
-- Height-for-width geometry management is implemented in GTK by way
-- of two virtual methods:
-- 
-- * t'GI.Gtk.Objects.Widget.Widget'.@/get_request_mode/@()
-- * t'GI.Gtk.Objects.Widget.Widget'.@/measure/@()
-- 
-- 
-- There are some important things to keep in mind when implementing
-- height-for-width and when using it in widget implementations.
-- 
-- If you implement a direct @GtkWidget@ subclass that supports
-- height-for-width or width-for-height geometry management for itself
-- or its child widgets, the t'GI.Gtk.Objects.Widget.Widget'.@/get_request_mode/@() virtual
-- function must be implemented as well and return the widget\'s preferred
-- request mode. The default implementation of this virtual function
-- returns 'GI.Gtk.Enums.SizeRequestModeConstantSize', which means that the widget will
-- only ever get -1 passed as the for_size value to its
-- t'GI.Gtk.Objects.Widget.Widget'.@/measure/@() implementation.
-- 
-- The geometry management system will query a widget hierarchy in
-- only one orientation at a time. When widgets are initially queried
-- for their minimum sizes it is generally done in two initial passes
-- in the t'GI.Gtk.Enums.SizeRequestMode' chosen by the toplevel.
-- 
-- For example, when queried in the normal 'GI.Gtk.Enums.SizeRequestModeHeightForWidth' mode:
-- 
-- First, the default minimum and natural width for each widget
-- in the interface will be computed using 'GI.Gtk.Objects.Widget.widgetMeasure' with an
-- orientation of 'GI.Gtk.Enums.OrientationHorizontal' and a for_size of -1.
-- Because the preferred widths for each widget depend on the preferred
-- widths of their children, this information propagates up the hierarchy,
-- and finally a minimum and natural width is determined for the entire
-- toplevel. Next, the toplevel will use the minimum width to query for the
-- minimum height contextual to that width using 'GI.Gtk.Objects.Widget.widgetMeasure' with an
-- orientation of 'GI.Gtk.Enums.OrientationVertical' and a for_size of the just computed
-- width. This will also be a highly recursive operation. The minimum height
-- for the minimum width is normally used to set the minimum size constraint
-- on the toplevel.
-- 
-- After the toplevel window has initially requested its size in both
-- dimensions it can go on to allocate itself a reasonable size (or a size
-- previously specified with 'GI.Gtk.Objects.Window.windowSetDefaultSize'). During the
-- recursive allocation process it’s important to note that request cycles
-- will be recursively executed while widgets allocate their children.
-- Each widget, once allocated a size, will go on to first share the
-- space in one orientation among its children and then request each child\'s
-- height for its target allocated width or its width for allocated height,
-- depending. In this way a widget will typically be requested its size
-- a number of times before actually being allocated a size. The size a
-- widget is finally allocated can of course differ from the size it has
-- requested. For this reason, @GtkWidget@ caches a  small number of results
-- to avoid re-querying for the same sizes in one allocation cycle.
-- 
-- If a widget does move content around to intelligently use up the
-- allocated size then it must support the request in both
-- @GtkSizeRequestMode@s even if the widget in question only
-- trades sizes in a single orientation.
-- 
-- For instance, a t'GI.Gtk.Objects.Label.Label' that does height-for-width word wrapping
-- will not expect to have t'GI.Gtk.Objects.Widget.Widget'.@/measure/@() with an orientation of
-- 'GI.Gtk.Enums.OrientationVertical' called because that call is specific to a
-- width-for-height request. In this case the label must return the height
-- required for its own minimum possible width. By following this rule any
-- widget that handles height-for-width or width-for-height requests will
-- always be allocated at least enough space to fit its own content.
-- 
-- Here are some examples of how a 'GI.Gtk.Enums.SizeRequestModeHeightForWidth' widget
-- generally deals with width-for-height requests:
-- 
-- 
-- === /c code/
-- >static void
-- >foo_widget_measure (GtkWidget      *widget,
-- >                    GtkOrientation  orientation,
-- >                    int             for_size,
-- >                    int            *minimum_size,
-- >                    int            *natural_size,
-- >                    int            *minimum_baseline,
-- >                    int            *natural_baseline)
-- >{
-- >  if (orientation == GTK_ORIENTATION_HORIZONTAL)
-- >    {
-- >      // Calculate minimum and natural width
-- >    }
-- >  else // VERTICAL
-- >    {
-- >      if (i_am_in_height_for_width_mode)
-- >        {
-- >          int min_width, dummy;
-- >
-- >          // First, get the minimum width of our widget
-- >          GTK_WIDGET_GET_CLASS (widget)->measure (widget, GTK_ORIENTATION_HORIZONTAL, -1,
-- >                                                  &min_width, &dummy, &dummy, &dummy);
-- >
-- >          // Now use the minimum width to retrieve the minimum and natural height to display
-- >          // that width.
-- >          GTK_WIDGET_GET_CLASS (widget)->measure (widget, GTK_ORIENTATION_VERTICAL, min_width,
-- >                                                  minimum_size, natural_size, &dummy, &dummy);
-- >        }
-- >      else
-- >        {
-- >          // ... some widgets do both.
-- >        }
-- >    }
-- >}
-- 
-- 
-- Often a widget needs to get its own request during size request or
-- allocation. For example, when computing height it may need to also
-- compute width. Or when deciding how to use an allocation, the widget
-- may need to know its natural size. In these cases, the widget should
-- be careful to call its virtual methods directly, like in the code
-- example above.
-- 
-- It will not work to use the wrapper function 'GI.Gtk.Objects.Widget.widgetMeasure'
-- inside your own t'GI.Gtk.Objects.Widget.Widget'.@/size_allocate/@() implementation.
-- These return a request adjusted by t'GI.Gtk.Objects.SizeGroup.SizeGroup', the widget\'s
-- align and expand flags, as well as its CSS style.
-- 
-- If a widget used the wrappers inside its virtual method implementations,
-- then the adjustments (such as widget margins) would be applied
-- twice. GTK therefore does not allow this and will warn if you try
-- to do it.
-- 
-- Of course if you are getting the size request for another widget, such
-- as a child widget, you must use 'GI.Gtk.Objects.Widget.widgetMeasure'; otherwise, you
-- would not properly consider widget margins, t'GI.Gtk.Objects.SizeGroup.SizeGroup', and
-- so forth.
-- 
-- GTK also supports baseline vertical alignment of widgets. This
-- means that widgets are positioned such that the typographical baseline of
-- widgets in the same row are aligned. This happens if a widget supports
-- baselines, has a vertical alignment using baselines, and is inside
-- a widget that supports baselines and has a natural “row” that it aligns to
-- the baseline, or a baseline assigned to it by the grandparent.
-- 
-- Baseline alignment support for a widget is also done by the
-- t'GI.Gtk.Objects.Widget.Widget'.@/measure/@() virtual function. It allows you to report
-- both a minimum and natural size.
-- 
-- If a widget ends up baseline aligned it will be allocated all the space in
-- the parent as if it was 'GI.Gtk.Enums.AlignFill', but the selected baseline can be
-- found via 'GI.Gtk.Objects.Widget.widgetGetBaseline'. If the baseline has a
-- value other than -1 you need to align the widget such that the baseline
-- appears at the position.
-- 
-- === GtkWidget as GtkBuildable
-- 
-- The @GtkWidget@ implementation of the @GtkBuildable@ interface
-- supports various custom elements to specify additional aspects of widgets
-- that are not directly expressed as properties.
-- 
-- If the widget uses a t'GI.Gtk.Objects.LayoutManager.LayoutManager', @GtkWidget@ supports
-- a custom @\<layout>@ element, used to define layout properties:
-- 
-- 
-- === /xml code/
-- ><object class="GtkGrid" id="my_grid">
-- >  <child>
-- >    <object class="GtkLabel" id="label1">
-- >      <property name="label">Description</property>
-- >      <layout>
-- >        <property name="column">0</property>
-- >        <property name="row">0</property>
-- >        <property name="row-span">1</property>
-- >        <property name="column-span">1</property>
-- >      </layout>
-- >    </object>
-- >  </child>
-- >  <child>
-- >    <object class="GtkEntry" id="description_entry">
-- >      <layout>
-- >        <property name="column">1</property>
-- >        <property name="row">0</property>
-- >        <property name="row-span">1</property>
-- >        <property name="column-span">1</property>
-- >      </layout>
-- >    </object>
-- >  </child>
-- ></object>
-- 
-- 
-- @GtkWidget@ allows style information such as style classes to
-- be associated with widgets, using the custom @\<style>@ element:
-- 
-- 
-- === /xml code/
-- ><object class="GtkButton" id="button1">
-- >  <style>
-- >    <class name="my-special-button-class"/>
-- >    <class name="dark-button"/>
-- >  </style>
-- ></object>
-- 
-- 
-- @GtkWidget@ allows defining accessibility information, such as properties,
-- relations, and states, using the custom @\<accessibility>@ element:
-- 
-- 
-- === /xml code/
-- ><object class="GtkButton" id="button1">
-- >  <accessibility>
-- >    <property name="label">Download</property>
-- >    <relation name="labelled-by">label1</relation>
-- >  </accessibility>
-- ></object>
-- 
-- 
-- === Building composite widgets from template XML
-- 
-- @GtkWidget @exposes some facilities to automate the procedure
-- of creating composite widgets using \"templates\".
-- 
-- To create composite widgets with @GtkBuilder@ XML, one must associate
-- the interface description with the widget class at class initialization
-- time using 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplate'.
-- 
-- The interface description semantics expected in composite template descriptions
-- is slightly different from regular t'GI.Gtk.Objects.Builder.Builder' XML.
-- 
-- Unlike regular interface descriptions, 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplate'
-- will expect a @\<template>@ tag as a direct child of the toplevel
-- @\<interface>@ tag. The @\<template>@ tag must specify the “class” attribute
-- which must be the type name of the widget. Optionally, the “parent”
-- attribute may be specified to specify the direct parent type of the widget
-- type; this is ignored by @GtkBuilder@ but can be used by UI design tools to
-- introspect what kind of properties and internal children exist for a given
-- type when the actual type does not exist.
-- 
-- The XML which is contained inside the @\<template>@ tag behaves as if it were
-- added to the @\<object>@ tag defining the widget itself. You may set properties
-- on a widget by inserting @\<property>@ tags into the @\<template>@ tag, and also
-- add @\<child>@ tags to add children and extend a widget in the normal way you
-- would with @\<object>@ tags.
-- 
-- Additionally, @\<object>@ tags can also be added before and after the initial
-- @\<template>@ tag in the normal way, allowing one to define auxiliary objects
-- which might be referenced by other widgets declared as children of the
-- @\<template>@ tag.
-- 
-- Since, unlike the @\<object>@ tag, the @\<template>@ tag does not contain an
-- “id” attribute, if you need to refer to the instance of the object itself that
-- the template will create, simply refer to the template class name in an
-- applicable element content.
-- 
-- Here is an example of a template definition, which includes an example of
-- this in the @\<signal>@ tag:
-- 
-- 
-- === /xml code/
-- ><interface>
-- >  <template class="FooWidget" parent="GtkBox">
-- >    <property name="orientation">horizontal</property>
-- >    <property name="spacing">4</property>
-- >    <child>
-- >      <object class="GtkButton" id="hello_button">
-- >        <property name="label">Hello World</property>
-- >        <signal name="clicked" handler="hello_button_clicked" object="FooWidget" swapped="yes"/>
-- >      </object>
-- >    </child>
-- >    <child>
-- >      <object class="GtkButton" id="goodbye_button">
-- >        <property name="label">Goodbye World</property>
-- >      </object>
-- >    </child>
-- >  </template>
-- ></interface>
-- 
-- 
-- Typically, you\'ll place the template fragment into a file that is
-- bundled with your project, using @GResource@. In order to load the
-- template, you need to call 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplateFromResource'
-- from the class initialization of your @GtkWidget@ type:
-- 
-- 
-- === /c code/
-- >static void
-- >foo_widget_class_init (FooWidgetClass *klass)
-- >{
-- >  // ...
-- >
-- >  gtk_widget_class_set_template_from_resource (GTK_WIDGET_CLASS (klass),
-- >                                               "/com/example/ui/foowidget.ui");
-- >}
-- 
-- 
-- You will also need to call 'GI.Gtk.Objects.Widget.widgetInitTemplate' from the
-- instance initialization function:
-- 
-- 
-- === /c code/
-- >static void
-- >foo_widget_init (FooWidget *self)
-- >{
-- >  gtk_widget_init_template (GTK_WIDGET (self));
-- >
-- >  // Initialize the rest of the widget...
-- >}
-- 
-- 
-- as well as calling 'GI.Gtk.Objects.Widget.widgetDisposeTemplate' from the dispose
-- function:
-- 
-- 
-- === /c code/
-- >static void
-- >foo_widget_dispose (GObject *gobject)
-- >{
-- >  FooWidget *self = FOO_WIDGET (gobject);
-- >
-- >  // Dispose objects for which you have a reference...
-- >
-- >  // Clear the template children for this widget type
-- >  gtk_widget_dispose_template (GTK_WIDGET (self), FOO_TYPE_WIDGET);
-- >
-- >  G_OBJECT_CLASS (foo_widget_parent_class)->dispose (gobject);
-- >}
-- 
-- 
-- You can access widgets defined in the template using the
-- 'GI.Gtk.Objects.Widget.widgetGetTemplateChild' function, but you will typically declare
-- a pointer in the instance private data structure of your type using the same
-- name as the widget in the template definition, and call
-- 'GI.Gtk.Structs.WidgetClass.widgetClassBindTemplateChildFull' (or one of its wrapper macros
-- @/Gtk.widget_class_bind_template_child/@ and @/Gtk.widget_class_bind_template_child_private/@)
-- with that name, e.g.
-- 
-- 
-- === /c code/
-- >typedef struct {
-- >  GtkWidget *hello_button;
-- >  GtkWidget *goodbye_button;
-- >} FooWidgetPrivate;
-- >
-- >G_DEFINE_TYPE_WITH_PRIVATE (FooWidget, foo_widget, GTK_TYPE_BOX)
-- >
-- >static void
-- >foo_widget_dispose (GObject *gobject)
-- >{
-- >  gtk_widget_dispose_template (GTK_WIDGET (gobject), FOO_TYPE_WIDGET);
-- >
-- >  G_OBJECT_CLASS (foo_widget_parent_class)->dispose (gobject);
-- >}
-- >
-- >static void
-- >foo_widget_class_init (FooWidgetClass *klass)
-- >{
-- >  // ...
-- >  G_OBJECT_CLASS (klass)->dispose = foo_widget_dispose;
-- >
-- >  gtk_widget_class_set_template_from_resource (GTK_WIDGET_CLASS (klass),
-- >                                               "/com/example/ui/foowidget.ui");
-- >  gtk_widget_class_bind_template_child_private (GTK_WIDGET_CLASS (klass),
-- >                                                FooWidget, hello_button);
-- >  gtk_widget_class_bind_template_child_private (GTK_WIDGET_CLASS (klass),
-- >                                                FooWidget, goodbye_button);
-- >}
-- >
-- >static void
-- >foo_widget_init (FooWidget *widget)
-- >{
-- >  gtk_widget_init_template (GTK_WIDGET (widget));
-- >}
-- 
-- 
-- You can also use 'GI.Gtk.Structs.WidgetClass.widgetClassBindTemplateCallbackFull' (or
-- is wrapper macro @/Gtk.widget_class_bind_template_callback/@) to connect
-- a signal callback defined in the template with a function visible in the
-- scope of the class, e.g.
-- 
-- 
-- === /c code/
-- >// the signal handler has the instance and user data swapped
-- >// because of the swapped="yes" attribute in the template XML
-- >static void
-- >hello_button_clicked (FooWidget *self,
-- >                      GtkButton *button)
-- >{
-- >  g_print ("Hello, world!\n");
-- >}
-- >
-- >static void
-- >foo_widget_class_init (FooWidgetClass *klass)
-- >{
-- >  // ...
-- >  gtk_widget_class_set_template_from_resource (GTK_WIDGET_CLASS (klass),
-- >                                               "/com/example/ui/foowidget.ui");
-- >  gtk_widget_class_bind_template_callback (GTK_WIDGET_CLASS (klass), hello_button_clicked);
-- >}
-- 

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.Widget
    ( 

-- * Exported types
    Widget(..)                              ,
    IsWidget                                ,
    toWidget                                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [actionSetEnabled]("GI.Gtk.Objects.Widget#g:method:actionSetEnabled"), [activate]("GI.Gtk.Objects.Widget#g:method:activate"), [activateAction]("GI.Gtk.Objects.Widget#g:method:activateAction"), [activateDefault]("GI.Gtk.Objects.Widget#g:method:activateDefault"), [addController]("GI.Gtk.Objects.Widget#g:method:addController"), [addCssClass]("GI.Gtk.Objects.Widget#g:method:addCssClass"), [addMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:addMnemonicLabel"), [addTickCallback]("GI.Gtk.Objects.Widget#g:method:addTickCallback"), [allocate]("GI.Gtk.Objects.Widget#g:method:allocate"), [announce]("GI.Gtk.Interfaces.Accessible#g:method:announce"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [childFocus]("GI.Gtk.Objects.Widget#g:method:childFocus"), [computeBounds]("GI.Gtk.Objects.Widget#g:method:computeBounds"), [computeExpand]("GI.Gtk.Objects.Widget#g:method:computeExpand"), [computePoint]("GI.Gtk.Objects.Widget#g:method:computePoint"), [computeTransform]("GI.Gtk.Objects.Widget#g:method:computeTransform"), [contains]("GI.Gtk.Objects.Widget#g:method:contains"), [createPangoContext]("GI.Gtk.Objects.Widget#g:method:createPangoContext"), [createPangoLayout]("GI.Gtk.Objects.Widget#g:method:createPangoLayout"), [disposeTemplate]("GI.Gtk.Objects.Widget#g:method:disposeTemplate"), [dragCheckThreshold]("GI.Gtk.Objects.Widget#g:method:dragCheckThreshold"), [errorBell]("GI.Gtk.Objects.Widget#g:method:errorBell"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [grabFocus]("GI.Gtk.Objects.Widget#g:method:grabFocus"), [hasCssClass]("GI.Gtk.Objects.Widget#g:method:hasCssClass"), [hasDefault]("GI.Gtk.Objects.Widget#g:method:hasDefault"), [hasFocus]("GI.Gtk.Objects.Widget#g:method:hasFocus"), [hasVisibleFocus]("GI.Gtk.Objects.Widget#g:method:hasVisibleFocus"), [hide]("GI.Gtk.Objects.Widget#g:method:hide"), [inDestruction]("GI.Gtk.Objects.Widget#g:method:inDestruction"), [initTemplate]("GI.Gtk.Objects.Widget#g:method:initTemplate"), [insertActionGroup]("GI.Gtk.Objects.Widget#g:method:insertActionGroup"), [insertAfter]("GI.Gtk.Objects.Widget#g:method:insertAfter"), [insertBefore]("GI.Gtk.Objects.Widget#g:method:insertBefore"), [isAncestor]("GI.Gtk.Objects.Widget#g:method:isAncestor"), [isDrawable]("GI.Gtk.Objects.Widget#g:method:isDrawable"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isFocus]("GI.Gtk.Objects.Widget#g:method:isFocus"), [isSensitive]("GI.Gtk.Objects.Widget#g:method:isSensitive"), [isVisible]("GI.Gtk.Objects.Widget#g:method:isVisible"), [keynavFailed]("GI.Gtk.Objects.Widget#g:method:keynavFailed"), [listMnemonicLabels]("GI.Gtk.Objects.Widget#g:method:listMnemonicLabels"), [map]("GI.Gtk.Objects.Widget#g:method:map"), [measure]("GI.Gtk.Objects.Widget#g:method:measure"), [mnemonicActivate]("GI.Gtk.Objects.Widget#g:method:mnemonicActivate"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [observeChildren]("GI.Gtk.Objects.Widget#g:method:observeChildren"), [observeControllers]("GI.Gtk.Objects.Widget#g:method:observeControllers"), [pick]("GI.Gtk.Objects.Widget#g:method:pick"), [queueAllocate]("GI.Gtk.Objects.Widget#g:method:queueAllocate"), [queueDraw]("GI.Gtk.Objects.Widget#g:method:queueDraw"), [queueResize]("GI.Gtk.Objects.Widget#g:method:queueResize"), [realize]("GI.Gtk.Objects.Widget#g:method:realize"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeController]("GI.Gtk.Objects.Widget#g:method:removeController"), [removeCssClass]("GI.Gtk.Objects.Widget#g:method:removeCssClass"), [removeMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:removeMnemonicLabel"), [removeTickCallback]("GI.Gtk.Objects.Widget#g:method:removeTickCallback"), [resetProperty]("GI.Gtk.Interfaces.Accessible#g:method:resetProperty"), [resetRelation]("GI.Gtk.Interfaces.Accessible#g:method:resetRelation"), [resetState]("GI.Gtk.Interfaces.Accessible#g:method:resetState"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [shouldLayout]("GI.Gtk.Objects.Widget#g:method:shouldLayout"), [show]("GI.Gtk.Objects.Widget#g:method:show"), [sizeAllocate]("GI.Gtk.Objects.Widget#g:method:sizeAllocate"), [snapshotChild]("GI.Gtk.Objects.Widget#g:method:snapshotChild"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [translateCoordinates]("GI.Gtk.Objects.Widget#g:method:translateCoordinates"), [triggerTooltipQuery]("GI.Gtk.Objects.Widget#g:method:triggerTooltipQuery"), [unmap]("GI.Gtk.Objects.Widget#g:method:unmap"), [unparent]("GI.Gtk.Objects.Widget#g:method:unparent"), [unrealize]("GI.Gtk.Objects.Widget#g:method:unrealize"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [unsetStateFlags]("GI.Gtk.Objects.Widget#g:method:unsetStateFlags"), [updateNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:updateNextAccessibleSibling"), [updatePlatformState]("GI.Gtk.Interfaces.Accessible#g:method:updatePlatformState"), [updateProperty]("GI.Gtk.Interfaces.Accessible#g:method:updateProperty"), [updateRelation]("GI.Gtk.Interfaces.Accessible#g:method:updateRelation"), [updateState]("GI.Gtk.Interfaces.Accessible#g:method:updateState"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleParent"), [getAccessibleRole]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleRole"), [getAllocatedBaseline]("GI.Gtk.Objects.Widget#g:method:getAllocatedBaseline"), [getAllocatedHeight]("GI.Gtk.Objects.Widget#g:method:getAllocatedHeight"), [getAllocatedWidth]("GI.Gtk.Objects.Widget#g:method:getAllocatedWidth"), [getAllocation]("GI.Gtk.Objects.Widget#g:method:getAllocation"), [getAncestor]("GI.Gtk.Objects.Widget#g:method:getAncestor"), [getAtContext]("GI.Gtk.Interfaces.Accessible#g:method:getAtContext"), [getBaseline]("GI.Gtk.Objects.Widget#g:method:getBaseline"), [getBounds]("GI.Gtk.Interfaces.Accessible#g:method:getBounds"), [getBuildableId]("GI.Gtk.Interfaces.Buildable#g:method:getBuildableId"), [getCanFocus]("GI.Gtk.Objects.Widget#g:method:getCanFocus"), [getCanTarget]("GI.Gtk.Objects.Widget#g:method:getCanTarget"), [getChildVisible]("GI.Gtk.Objects.Widget#g:method:getChildVisible"), [getClipboard]("GI.Gtk.Objects.Widget#g:method:getClipboard"), [getColor]("GI.Gtk.Objects.Widget#g:method:getColor"), [getCssClasses]("GI.Gtk.Objects.Widget#g:method:getCssClasses"), [getCssName]("GI.Gtk.Objects.Widget#g:method:getCssName"), [getCursor]("GI.Gtk.Objects.Widget#g:method:getCursor"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDirection]("GI.Gtk.Objects.Widget#g:method:getDirection"), [getDisplay]("GI.Gtk.Objects.Widget#g:method:getDisplay"), [getFirstAccessibleChild]("GI.Gtk.Interfaces.Accessible#g:method:getFirstAccessibleChild"), [getFirstChild]("GI.Gtk.Objects.Widget#g:method:getFirstChild"), [getFocusChild]("GI.Gtk.Objects.Widget#g:method:getFocusChild"), [getFocusOnClick]("GI.Gtk.Objects.Widget#g:method:getFocusOnClick"), [getFocusable]("GI.Gtk.Objects.Widget#g:method:getFocusable"), [getFontMap]("GI.Gtk.Objects.Widget#g:method:getFontMap"), [getFontOptions]("GI.Gtk.Objects.Widget#g:method:getFontOptions"), [getFrameClock]("GI.Gtk.Objects.Widget#g:method:getFrameClock"), [getHalign]("GI.Gtk.Objects.Widget#g:method:getHalign"), [getHasTooltip]("GI.Gtk.Objects.Widget#g:method:getHasTooltip"), [getHeight]("GI.Gtk.Objects.Widget#g:method:getHeight"), [getHexpand]("GI.Gtk.Objects.Widget#g:method:getHexpand"), [getHexpandSet]("GI.Gtk.Objects.Widget#g:method:getHexpandSet"), [getLastChild]("GI.Gtk.Objects.Widget#g:method:getLastChild"), [getLayoutManager]("GI.Gtk.Objects.Widget#g:method:getLayoutManager"), [getLimitEvents]("GI.Gtk.Objects.Widget#g:method:getLimitEvents"), [getMapped]("GI.Gtk.Objects.Widget#g:method:getMapped"), [getMarginBottom]("GI.Gtk.Objects.Widget#g:method:getMarginBottom"), [getMarginEnd]("GI.Gtk.Objects.Widget#g:method:getMarginEnd"), [getMarginStart]("GI.Gtk.Objects.Widget#g:method:getMarginStart"), [getMarginTop]("GI.Gtk.Objects.Widget#g:method:getMarginTop"), [getName]("GI.Gtk.Objects.Widget#g:method:getName"), [getNative]("GI.Gtk.Objects.Widget#g:method:getNative"), [getNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:getNextAccessibleSibling"), [getNextSibling]("GI.Gtk.Objects.Widget#g:method:getNextSibling"), [getOpacity]("GI.Gtk.Objects.Widget#g:method:getOpacity"), [getOverflow]("GI.Gtk.Objects.Widget#g:method:getOverflow"), [getPangoContext]("GI.Gtk.Objects.Widget#g:method:getPangoContext"), [getParent]("GI.Gtk.Objects.Widget#g:method:getParent"), [getPlatformState]("GI.Gtk.Interfaces.Accessible#g:method:getPlatformState"), [getPreferredSize]("GI.Gtk.Objects.Widget#g:method:getPreferredSize"), [getPrevSibling]("GI.Gtk.Objects.Widget#g:method:getPrevSibling"), [getPrimaryClipboard]("GI.Gtk.Objects.Widget#g:method:getPrimaryClipboard"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRealized]("GI.Gtk.Objects.Widget#g:method:getRealized"), [getReceivesDefault]("GI.Gtk.Objects.Widget#g:method:getReceivesDefault"), [getRequestMode]("GI.Gtk.Objects.Widget#g:method:getRequestMode"), [getRoot]("GI.Gtk.Objects.Widget#g:method:getRoot"), [getScaleFactor]("GI.Gtk.Objects.Widget#g:method:getScaleFactor"), [getSensitive]("GI.Gtk.Objects.Widget#g:method:getSensitive"), [getSettings]("GI.Gtk.Objects.Widget#g:method:getSettings"), [getSize]("GI.Gtk.Objects.Widget#g:method:getSize"), [getSizeRequest]("GI.Gtk.Objects.Widget#g:method:getSizeRequest"), [getStateFlags]("GI.Gtk.Objects.Widget#g:method:getStateFlags"), [getStyleContext]("GI.Gtk.Objects.Widget#g:method:getStyleContext"), [getTemplateChild]("GI.Gtk.Objects.Widget#g:method:getTemplateChild"), [getTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:getTooltipMarkup"), [getTooltipText]("GI.Gtk.Objects.Widget#g:method:getTooltipText"), [getValign]("GI.Gtk.Objects.Widget#g:method:getValign"), [getVexpand]("GI.Gtk.Objects.Widget#g:method:getVexpand"), [getVexpandSet]("GI.Gtk.Objects.Widget#g:method:getVexpandSet"), [getVisible]("GI.Gtk.Objects.Widget#g:method:getVisible"), [getWidth]("GI.Gtk.Objects.Widget#g:method:getWidth").
-- 
-- ==== Setters
-- [setAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:setAccessibleParent"), [setCanFocus]("GI.Gtk.Objects.Widget#g:method:setCanFocus"), [setCanTarget]("GI.Gtk.Objects.Widget#g:method:setCanTarget"), [setChildVisible]("GI.Gtk.Objects.Widget#g:method:setChildVisible"), [setCssClasses]("GI.Gtk.Objects.Widget#g:method:setCssClasses"), [setCursor]("GI.Gtk.Objects.Widget#g:method:setCursor"), [setCursorFromName]("GI.Gtk.Objects.Widget#g:method:setCursorFromName"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDirection]("GI.Gtk.Objects.Widget#g:method:setDirection"), [setFocusChild]("GI.Gtk.Objects.Widget#g:method:setFocusChild"), [setFocusOnClick]("GI.Gtk.Objects.Widget#g:method:setFocusOnClick"), [setFocusable]("GI.Gtk.Objects.Widget#g:method:setFocusable"), [setFontMap]("GI.Gtk.Objects.Widget#g:method:setFontMap"), [setFontOptions]("GI.Gtk.Objects.Widget#g:method:setFontOptions"), [setHalign]("GI.Gtk.Objects.Widget#g:method:setHalign"), [setHasTooltip]("GI.Gtk.Objects.Widget#g:method:setHasTooltip"), [setHexpand]("GI.Gtk.Objects.Widget#g:method:setHexpand"), [setHexpandSet]("GI.Gtk.Objects.Widget#g:method:setHexpandSet"), [setLayoutManager]("GI.Gtk.Objects.Widget#g:method:setLayoutManager"), [setLimitEvents]("GI.Gtk.Objects.Widget#g:method:setLimitEvents"), [setMarginBottom]("GI.Gtk.Objects.Widget#g:method:setMarginBottom"), [setMarginEnd]("GI.Gtk.Objects.Widget#g:method:setMarginEnd"), [setMarginStart]("GI.Gtk.Objects.Widget#g:method:setMarginStart"), [setMarginTop]("GI.Gtk.Objects.Widget#g:method:setMarginTop"), [setName]("GI.Gtk.Objects.Widget#g:method:setName"), [setOpacity]("GI.Gtk.Objects.Widget#g:method:setOpacity"), [setOverflow]("GI.Gtk.Objects.Widget#g:method:setOverflow"), [setParent]("GI.Gtk.Objects.Widget#g:method:setParent"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setReceivesDefault]("GI.Gtk.Objects.Widget#g:method:setReceivesDefault"), [setSensitive]("GI.Gtk.Objects.Widget#g:method:setSensitive"), [setSizeRequest]("GI.Gtk.Objects.Widget#g:method:setSizeRequest"), [setStateFlags]("GI.Gtk.Objects.Widget#g:method:setStateFlags"), [setTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:setTooltipMarkup"), [setTooltipText]("GI.Gtk.Objects.Widget#g:method:setTooltipText"), [setValign]("GI.Gtk.Objects.Widget#g:method:setValign"), [setVexpand]("GI.Gtk.Objects.Widget#g:method:setVexpand"), [setVexpandSet]("GI.Gtk.Objects.Widget#g:method:setVexpandSet"), [setVisible]("GI.Gtk.Objects.Widget#g:method:setVisible").

#if defined(ENABLE_OVERLOADING)
    ResolveWidgetMethod                     ,
#endif

-- ** actionSetEnabled #method:actionSetEnabled#

#if defined(ENABLE_OVERLOADING)
    WidgetActionSetEnabledMethodInfo        ,
#endif
    widgetActionSetEnabled                  ,


-- ** activate #method:activate#

#if defined(ENABLE_OVERLOADING)
    WidgetActivateMethodInfo                ,
#endif
    widgetActivate                          ,


-- ** activateAction #method:activateAction#

#if defined(ENABLE_OVERLOADING)
    WidgetActivateActionMethodInfo          ,
#endif
    widgetActivateAction                    ,


-- ** activateDefault #method:activateDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetActivateDefaultMethodInfo         ,
#endif
    widgetActivateDefault                   ,


-- ** addController #method:addController#

#if defined(ENABLE_OVERLOADING)
    WidgetAddControllerMethodInfo           ,
#endif
    widgetAddController                     ,


-- ** addCssClass #method:addCssClass#

#if defined(ENABLE_OVERLOADING)
    WidgetAddCssClassMethodInfo             ,
#endif
    widgetAddCssClass                       ,


-- ** addMnemonicLabel #method:addMnemonicLabel#

#if defined(ENABLE_OVERLOADING)
    WidgetAddMnemonicLabelMethodInfo        ,
#endif
    widgetAddMnemonicLabel                  ,


-- ** addTickCallback #method:addTickCallback#

#if defined(ENABLE_OVERLOADING)
    WidgetAddTickCallbackMethodInfo         ,
#endif
    widgetAddTickCallback                   ,


-- ** allocate #method:allocate#

#if defined(ENABLE_OVERLOADING)
    WidgetAllocateMethodInfo                ,
#endif
    widgetAllocate                          ,


-- ** childFocus #method:childFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetChildFocusMethodInfo              ,
#endif
    widgetChildFocus                        ,


-- ** computeBounds #method:computeBounds#

#if defined(ENABLE_OVERLOADING)
    WidgetComputeBoundsMethodInfo           ,
#endif
    widgetComputeBounds                     ,


-- ** computeExpand #method:computeExpand#

#if defined(ENABLE_OVERLOADING)
    WidgetComputeExpandMethodInfo           ,
#endif
    widgetComputeExpand                     ,


-- ** computePoint #method:computePoint#

#if defined(ENABLE_OVERLOADING)
    WidgetComputePointMethodInfo            ,
#endif
    widgetComputePoint                      ,


-- ** computeTransform #method:computeTransform#

#if defined(ENABLE_OVERLOADING)
    WidgetComputeTransformMethodInfo        ,
#endif
    widgetComputeTransform                  ,


-- ** contains #method:contains#

#if defined(ENABLE_OVERLOADING)
    WidgetContainsMethodInfo                ,
#endif
    widgetContains                          ,


-- ** createPangoContext #method:createPangoContext#

#if defined(ENABLE_OVERLOADING)
    WidgetCreatePangoContextMethodInfo      ,
#endif
    widgetCreatePangoContext                ,


-- ** createPangoLayout #method:createPangoLayout#

#if defined(ENABLE_OVERLOADING)
    WidgetCreatePangoLayoutMethodInfo       ,
#endif
    widgetCreatePangoLayout                 ,


-- ** disposeTemplate #method:disposeTemplate#

#if defined(ENABLE_OVERLOADING)
    WidgetDisposeTemplateMethodInfo         ,
#endif
    widgetDisposeTemplate                   ,


-- ** dragCheckThreshold #method:dragCheckThreshold#

#if defined(ENABLE_OVERLOADING)
    WidgetDragCheckThresholdMethodInfo      ,
#endif
    widgetDragCheckThreshold                ,


-- ** errorBell #method:errorBell#

#if defined(ENABLE_OVERLOADING)
    WidgetErrorBellMethodInfo               ,
#endif
    widgetErrorBell                         ,


-- ** getAllocatedBaseline #method:getAllocatedBaseline#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocatedBaselineMethodInfo    ,
#endif
    widgetGetAllocatedBaseline              ,


-- ** getAllocatedHeight #method:getAllocatedHeight#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocatedHeightMethodInfo      ,
#endif
    widgetGetAllocatedHeight                ,


-- ** getAllocatedWidth #method:getAllocatedWidth#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocatedWidthMethodInfo       ,
#endif
    widgetGetAllocatedWidth                 ,


-- ** getAllocation #method:getAllocation#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocationMethodInfo           ,
#endif
    widgetGetAllocation                     ,


-- ** getAncestor #method:getAncestor#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAncestorMethodInfo             ,
#endif
    widgetGetAncestor                       ,


-- ** getBaseline #method:getBaseline#

#if defined(ENABLE_OVERLOADING)
    WidgetGetBaselineMethodInfo             ,
#endif
    widgetGetBaseline                       ,


-- ** getCanFocus #method:getCanFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetGetCanFocusMethodInfo             ,
#endif
    widgetGetCanFocus                       ,


-- ** getCanTarget #method:getCanTarget#

#if defined(ENABLE_OVERLOADING)
    WidgetGetCanTargetMethodInfo            ,
#endif
    widgetGetCanTarget                      ,


-- ** getChildVisible #method:getChildVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetGetChildVisibleMethodInfo         ,
#endif
    widgetGetChildVisible                   ,


-- ** getClipboard #method:getClipboard#

#if defined(ENABLE_OVERLOADING)
    WidgetGetClipboardMethodInfo            ,
#endif
    widgetGetClipboard                      ,


-- ** getColor #method:getColor#

#if defined(ENABLE_OVERLOADING)
    WidgetGetColorMethodInfo                ,
#endif
    widgetGetColor                          ,


-- ** getCssClasses #method:getCssClasses#

#if defined(ENABLE_OVERLOADING)
    WidgetGetCssClassesMethodInfo           ,
#endif
    widgetGetCssClasses                     ,


-- ** getCssName #method:getCssName#

#if defined(ENABLE_OVERLOADING)
    WidgetGetCssNameMethodInfo              ,
#endif
    widgetGetCssName                        ,


-- ** getCursor #method:getCursor#

#if defined(ENABLE_OVERLOADING)
    WidgetGetCursorMethodInfo               ,
#endif
    widgetGetCursor                         ,


-- ** getDefaultDirection #method:getDefaultDirection#

    widgetGetDefaultDirection               ,


-- ** getDirection #method:getDirection#

#if defined(ENABLE_OVERLOADING)
    WidgetGetDirectionMethodInfo            ,
#endif
    widgetGetDirection                      ,


-- ** getDisplay #method:getDisplay#

#if defined(ENABLE_OVERLOADING)
    WidgetGetDisplayMethodInfo              ,
#endif
    widgetGetDisplay                        ,


-- ** getFirstChild #method:getFirstChild#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFirstChildMethodInfo           ,
#endif
    widgetGetFirstChild                     ,


-- ** getFocusChild #method:getFocusChild#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFocusChildMethodInfo           ,
#endif
    widgetGetFocusChild                     ,


-- ** getFocusOnClick #method:getFocusOnClick#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFocusOnClickMethodInfo         ,
#endif
    widgetGetFocusOnClick                   ,


-- ** getFocusable #method:getFocusable#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFocusableMethodInfo            ,
#endif
    widgetGetFocusable                      ,


-- ** getFontMap #method:getFontMap#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFontMapMethodInfo              ,
#endif
    widgetGetFontMap                        ,


-- ** getFontOptions #method:getFontOptions#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFontOptionsMethodInfo          ,
#endif
    widgetGetFontOptions                    ,


-- ** getFrameClock #method:getFrameClock#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFrameClockMethodInfo           ,
#endif
    widgetGetFrameClock                     ,


-- ** getHalign #method:getHalign#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHalignMethodInfo               ,
#endif
    widgetGetHalign                         ,


-- ** getHasTooltip #method:getHasTooltip#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHasTooltipMethodInfo           ,
#endif
    widgetGetHasTooltip                     ,


-- ** getHeight #method:getHeight#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHeightMethodInfo               ,
#endif
    widgetGetHeight                         ,


-- ** getHexpand #method:getHexpand#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHexpandMethodInfo              ,
#endif
    widgetGetHexpand                        ,


-- ** getHexpandSet #method:getHexpandSet#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHexpandSetMethodInfo           ,
#endif
    widgetGetHexpandSet                     ,


-- ** getLastChild #method:getLastChild#

#if defined(ENABLE_OVERLOADING)
    WidgetGetLastChildMethodInfo            ,
#endif
    widgetGetLastChild                      ,


-- ** getLayoutManager #method:getLayoutManager#

#if defined(ENABLE_OVERLOADING)
    WidgetGetLayoutManagerMethodInfo        ,
#endif
    widgetGetLayoutManager                  ,


-- ** getLimitEvents #method:getLimitEvents#

#if defined(ENABLE_OVERLOADING)
    WidgetGetLimitEventsMethodInfo          ,
#endif
    widgetGetLimitEvents                    ,


-- ** getMapped #method:getMapped#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMappedMethodInfo               ,
#endif
    widgetGetMapped                         ,


-- ** getMarginBottom #method:getMarginBottom#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginBottomMethodInfo         ,
#endif
    widgetGetMarginBottom                   ,


-- ** getMarginEnd #method:getMarginEnd#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginEndMethodInfo            ,
#endif
    widgetGetMarginEnd                      ,


-- ** getMarginStart #method:getMarginStart#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginStartMethodInfo          ,
#endif
    widgetGetMarginStart                    ,


-- ** getMarginTop #method:getMarginTop#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginTopMethodInfo            ,
#endif
    widgetGetMarginTop                      ,


-- ** getName #method:getName#

#if defined(ENABLE_OVERLOADING)
    WidgetGetNameMethodInfo                 ,
#endif
    widgetGetName                           ,


-- ** getNative #method:getNative#

#if defined(ENABLE_OVERLOADING)
    WidgetGetNativeMethodInfo               ,
#endif
    widgetGetNative                         ,


-- ** getNextSibling #method:getNextSibling#

#if defined(ENABLE_OVERLOADING)
    WidgetGetNextSiblingMethodInfo          ,
#endif
    widgetGetNextSibling                    ,


-- ** getOpacity #method:getOpacity#

#if defined(ENABLE_OVERLOADING)
    WidgetGetOpacityMethodInfo              ,
#endif
    widgetGetOpacity                        ,


-- ** getOverflow #method:getOverflow#

#if defined(ENABLE_OVERLOADING)
    WidgetGetOverflowMethodInfo             ,
#endif
    widgetGetOverflow                       ,


-- ** getPangoContext #method:getPangoContext#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPangoContextMethodInfo         ,
#endif
    widgetGetPangoContext                   ,


-- ** getParent #method:getParent#

#if defined(ENABLE_OVERLOADING)
    WidgetGetParentMethodInfo               ,
#endif
    widgetGetParent                         ,


-- ** getPreferredSize #method:getPreferredSize#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPreferredSizeMethodInfo        ,
#endif
    widgetGetPreferredSize                  ,


-- ** getPrevSibling #method:getPrevSibling#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPrevSiblingMethodInfo          ,
#endif
    widgetGetPrevSibling                    ,


-- ** getPrimaryClipboard #method:getPrimaryClipboard#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPrimaryClipboardMethodInfo     ,
#endif
    widgetGetPrimaryClipboard               ,


-- ** getRealized #method:getRealized#

#if defined(ENABLE_OVERLOADING)
    WidgetGetRealizedMethodInfo             ,
#endif
    widgetGetRealized                       ,


-- ** getReceivesDefault #method:getReceivesDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetGetReceivesDefaultMethodInfo      ,
#endif
    widgetGetReceivesDefault                ,


-- ** getRequestMode #method:getRequestMode#

#if defined(ENABLE_OVERLOADING)
    WidgetGetRequestModeMethodInfo          ,
#endif
    widgetGetRequestMode                    ,


-- ** getRoot #method:getRoot#

#if defined(ENABLE_OVERLOADING)
    WidgetGetRootMethodInfo                 ,
#endif
    widgetGetRoot                           ,


-- ** getScaleFactor #method:getScaleFactor#

#if defined(ENABLE_OVERLOADING)
    WidgetGetScaleFactorMethodInfo          ,
#endif
    widgetGetScaleFactor                    ,


-- ** getSensitive #method:getSensitive#

#if defined(ENABLE_OVERLOADING)
    WidgetGetSensitiveMethodInfo            ,
#endif
    widgetGetSensitive                      ,


-- ** getSettings #method:getSettings#

#if defined(ENABLE_OVERLOADING)
    WidgetGetSettingsMethodInfo             ,
#endif
    widgetGetSettings                       ,


-- ** getSize #method:getSize#

#if defined(ENABLE_OVERLOADING)
    WidgetGetSizeMethodInfo                 ,
#endif
    widgetGetSize                           ,


-- ** getSizeRequest #method:getSizeRequest#

#if defined(ENABLE_OVERLOADING)
    WidgetGetSizeRequestMethodInfo          ,
#endif
    widgetGetSizeRequest                    ,


-- ** getStateFlags #method:getStateFlags#

#if defined(ENABLE_OVERLOADING)
    WidgetGetStateFlagsMethodInfo           ,
#endif
    widgetGetStateFlags                     ,


-- ** getStyleContext #method:getStyleContext#

#if defined(ENABLE_OVERLOADING)
    WidgetGetStyleContextMethodInfo         ,
#endif
    widgetGetStyleContext                   ,


-- ** getTemplateChild #method:getTemplateChild#

#if defined(ENABLE_OVERLOADING)
    WidgetGetTemplateChildMethodInfo        ,
#endif
    widgetGetTemplateChild                  ,


-- ** getTooltipMarkup #method:getTooltipMarkup#

#if defined(ENABLE_OVERLOADING)
    WidgetGetTooltipMarkupMethodInfo        ,
#endif
    widgetGetTooltipMarkup                  ,


-- ** getTooltipText #method:getTooltipText#

#if defined(ENABLE_OVERLOADING)
    WidgetGetTooltipTextMethodInfo          ,
#endif
    widgetGetTooltipText                    ,


-- ** getValign #method:getValign#

#if defined(ENABLE_OVERLOADING)
    WidgetGetValignMethodInfo               ,
#endif
    widgetGetValign                         ,


-- ** getVexpand #method:getVexpand#

#if defined(ENABLE_OVERLOADING)
    WidgetGetVexpandMethodInfo              ,
#endif
    widgetGetVexpand                        ,


-- ** getVexpandSet #method:getVexpandSet#

#if defined(ENABLE_OVERLOADING)
    WidgetGetVexpandSetMethodInfo           ,
#endif
    widgetGetVexpandSet                     ,


-- ** getVisible #method:getVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetGetVisibleMethodInfo              ,
#endif
    widgetGetVisible                        ,


-- ** getWidth #method:getWidth#

#if defined(ENABLE_OVERLOADING)
    WidgetGetWidthMethodInfo                ,
#endif
    widgetGetWidth                          ,


-- ** grabFocus #method:grabFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetGrabFocusMethodInfo               ,
#endif
    widgetGrabFocus                         ,


-- ** hasCssClass #method:hasCssClass#

#if defined(ENABLE_OVERLOADING)
    WidgetHasCssClassMethodInfo             ,
#endif
    widgetHasCssClass                       ,


-- ** hasDefault #method:hasDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetHasDefaultMethodInfo              ,
#endif
    widgetHasDefault                        ,


-- ** hasFocus #method:hasFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetHasFocusMethodInfo                ,
#endif
    widgetHasFocus                          ,


-- ** hasVisibleFocus #method:hasVisibleFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetHasVisibleFocusMethodInfo         ,
#endif
    widgetHasVisibleFocus                   ,


-- ** hide #method:hide#

#if defined(ENABLE_OVERLOADING)
    WidgetHideMethodInfo                    ,
#endif
    widgetHide                              ,


-- ** inDestruction #method:inDestruction#

#if defined(ENABLE_OVERLOADING)
    WidgetInDestructionMethodInfo           ,
#endif
    widgetInDestruction                     ,


-- ** initTemplate #method:initTemplate#

#if defined(ENABLE_OVERLOADING)
    WidgetInitTemplateMethodInfo            ,
#endif
    widgetInitTemplate                      ,


-- ** insertActionGroup #method:insertActionGroup#

#if defined(ENABLE_OVERLOADING)
    WidgetInsertActionGroupMethodInfo       ,
#endif
    widgetInsertActionGroup                 ,


-- ** insertAfter #method:insertAfter#

#if defined(ENABLE_OVERLOADING)
    WidgetInsertAfterMethodInfo             ,
#endif
    widgetInsertAfter                       ,


-- ** insertBefore #method:insertBefore#

#if defined(ENABLE_OVERLOADING)
    WidgetInsertBeforeMethodInfo            ,
#endif
    widgetInsertBefore                      ,


-- ** isAncestor #method:isAncestor#

#if defined(ENABLE_OVERLOADING)
    WidgetIsAncestorMethodInfo              ,
#endif
    widgetIsAncestor                        ,


-- ** isDrawable #method:isDrawable#

#if defined(ENABLE_OVERLOADING)
    WidgetIsDrawableMethodInfo              ,
#endif
    widgetIsDrawable                        ,


-- ** isFocus #method:isFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetIsFocusMethodInfo                 ,
#endif
    widgetIsFocus                           ,


-- ** isSensitive #method:isSensitive#

#if defined(ENABLE_OVERLOADING)
    WidgetIsSensitiveMethodInfo             ,
#endif
    widgetIsSensitive                       ,


-- ** isVisible #method:isVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetIsVisibleMethodInfo               ,
#endif
    widgetIsVisible                         ,


-- ** keynavFailed #method:keynavFailed#

#if defined(ENABLE_OVERLOADING)
    WidgetKeynavFailedMethodInfo            ,
#endif
    widgetKeynavFailed                      ,


-- ** listMnemonicLabels #method:listMnemonicLabels#

#if defined(ENABLE_OVERLOADING)
    WidgetListMnemonicLabelsMethodInfo      ,
#endif
    widgetListMnemonicLabels                ,


-- ** map #method:map#

#if defined(ENABLE_OVERLOADING)
    WidgetMapMethodInfo                     ,
#endif
    widgetMap                               ,


-- ** measure #method:measure#

#if defined(ENABLE_OVERLOADING)
    WidgetMeasureMethodInfo                 ,
#endif
    widgetMeasure                           ,


-- ** mnemonicActivate #method:mnemonicActivate#

#if defined(ENABLE_OVERLOADING)
    WidgetMnemonicActivateMethodInfo        ,
#endif
    widgetMnemonicActivate                  ,


-- ** observeChildren #method:observeChildren#

#if defined(ENABLE_OVERLOADING)
    WidgetObserveChildrenMethodInfo         ,
#endif
    widgetObserveChildren                   ,


-- ** observeControllers #method:observeControllers#

#if defined(ENABLE_OVERLOADING)
    WidgetObserveControllersMethodInfo      ,
#endif
    widgetObserveControllers                ,


-- ** pick #method:pick#

#if defined(ENABLE_OVERLOADING)
    WidgetPickMethodInfo                    ,
#endif
    widgetPick                              ,


-- ** queueAllocate #method:queueAllocate#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueAllocateMethodInfo           ,
#endif
    widgetQueueAllocate                     ,


-- ** queueDraw #method:queueDraw#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueDrawMethodInfo               ,
#endif
    widgetQueueDraw                         ,


-- ** queueResize #method:queueResize#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueResizeMethodInfo             ,
#endif
    widgetQueueResize                       ,


-- ** realize #method:realize#

#if defined(ENABLE_OVERLOADING)
    WidgetRealizeMethodInfo                 ,
#endif
    widgetRealize                           ,


-- ** removeController #method:removeController#

#if defined(ENABLE_OVERLOADING)
    WidgetRemoveControllerMethodInfo        ,
#endif
    widgetRemoveController                  ,


-- ** removeCssClass #method:removeCssClass#

#if defined(ENABLE_OVERLOADING)
    WidgetRemoveCssClassMethodInfo          ,
#endif
    widgetRemoveCssClass                    ,


-- ** removeMnemonicLabel #method:removeMnemonicLabel#

#if defined(ENABLE_OVERLOADING)
    WidgetRemoveMnemonicLabelMethodInfo     ,
#endif
    widgetRemoveMnemonicLabel               ,


-- ** removeTickCallback #method:removeTickCallback#

#if defined(ENABLE_OVERLOADING)
    WidgetRemoveTickCallbackMethodInfo      ,
#endif
    widgetRemoveTickCallback                ,


-- ** setCanFocus #method:setCanFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetSetCanFocusMethodInfo             ,
#endif
    widgetSetCanFocus                       ,


-- ** setCanTarget #method:setCanTarget#

#if defined(ENABLE_OVERLOADING)
    WidgetSetCanTargetMethodInfo            ,
#endif
    widgetSetCanTarget                      ,


-- ** setChildVisible #method:setChildVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetSetChildVisibleMethodInfo         ,
#endif
    widgetSetChildVisible                   ,


-- ** setCssClasses #method:setCssClasses#

#if defined(ENABLE_OVERLOADING)
    WidgetSetCssClassesMethodInfo           ,
#endif
    widgetSetCssClasses                     ,


-- ** setCursor #method:setCursor#

#if defined(ENABLE_OVERLOADING)
    WidgetSetCursorMethodInfo               ,
#endif
    widgetSetCursor                         ,


-- ** setCursorFromName #method:setCursorFromName#

#if defined(ENABLE_OVERLOADING)
    WidgetSetCursorFromNameMethodInfo       ,
#endif
    widgetSetCursorFromName                 ,


-- ** setDefaultDirection #method:setDefaultDirection#

    widgetSetDefaultDirection               ,


-- ** setDirection #method:setDirection#

#if defined(ENABLE_OVERLOADING)
    WidgetSetDirectionMethodInfo            ,
#endif
    widgetSetDirection                      ,


-- ** setFocusChild #method:setFocusChild#

#if defined(ENABLE_OVERLOADING)
    WidgetSetFocusChildMethodInfo           ,
#endif
    widgetSetFocusChild                     ,


-- ** setFocusOnClick #method:setFocusOnClick#

#if defined(ENABLE_OVERLOADING)
    WidgetSetFocusOnClickMethodInfo         ,
#endif
    widgetSetFocusOnClick                   ,


-- ** setFocusable #method:setFocusable#

#if defined(ENABLE_OVERLOADING)
    WidgetSetFocusableMethodInfo            ,
#endif
    widgetSetFocusable                      ,


-- ** setFontMap #method:setFontMap#

#if defined(ENABLE_OVERLOADING)
    WidgetSetFontMapMethodInfo              ,
#endif
    widgetSetFontMap                        ,


-- ** setFontOptions #method:setFontOptions#

#if defined(ENABLE_OVERLOADING)
    WidgetSetFontOptionsMethodInfo          ,
#endif
    widgetSetFontOptions                    ,


-- ** setHalign #method:setHalign#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHalignMethodInfo               ,
#endif
    widgetSetHalign                         ,


-- ** setHasTooltip #method:setHasTooltip#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHasTooltipMethodInfo           ,
#endif
    widgetSetHasTooltip                     ,


-- ** setHexpand #method:setHexpand#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHexpandMethodInfo              ,
#endif
    widgetSetHexpand                        ,


-- ** setHexpandSet #method:setHexpandSet#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHexpandSetMethodInfo           ,
#endif
    widgetSetHexpandSet                     ,


-- ** setLayoutManager #method:setLayoutManager#

#if defined(ENABLE_OVERLOADING)
    WidgetSetLayoutManagerMethodInfo        ,
#endif
    widgetSetLayoutManager                  ,


-- ** setLimitEvents #method:setLimitEvents#

#if defined(ENABLE_OVERLOADING)
    WidgetSetLimitEventsMethodInfo          ,
#endif
    widgetSetLimitEvents                    ,


-- ** setMarginBottom #method:setMarginBottom#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginBottomMethodInfo         ,
#endif
    widgetSetMarginBottom                   ,


-- ** setMarginEnd #method:setMarginEnd#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginEndMethodInfo            ,
#endif
    widgetSetMarginEnd                      ,


-- ** setMarginStart #method:setMarginStart#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginStartMethodInfo          ,
#endif
    widgetSetMarginStart                    ,


-- ** setMarginTop #method:setMarginTop#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginTopMethodInfo            ,
#endif
    widgetSetMarginTop                      ,


-- ** setName #method:setName#

#if defined(ENABLE_OVERLOADING)
    WidgetSetNameMethodInfo                 ,
#endif
    widgetSetName                           ,


-- ** setOpacity #method:setOpacity#

#if defined(ENABLE_OVERLOADING)
    WidgetSetOpacityMethodInfo              ,
#endif
    widgetSetOpacity                        ,


-- ** setOverflow #method:setOverflow#

#if defined(ENABLE_OVERLOADING)
    WidgetSetOverflowMethodInfo             ,
#endif
    widgetSetOverflow                       ,


-- ** setParent #method:setParent#

#if defined(ENABLE_OVERLOADING)
    WidgetSetParentMethodInfo               ,
#endif
    widgetSetParent                         ,


-- ** setReceivesDefault #method:setReceivesDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetSetReceivesDefaultMethodInfo      ,
#endif
    widgetSetReceivesDefault                ,


-- ** setSensitive #method:setSensitive#

#if defined(ENABLE_OVERLOADING)
    WidgetSetSensitiveMethodInfo            ,
#endif
    widgetSetSensitive                      ,


-- ** setSizeRequest #method:setSizeRequest#

#if defined(ENABLE_OVERLOADING)
    WidgetSetSizeRequestMethodInfo          ,
#endif
    widgetSetSizeRequest                    ,


-- ** setStateFlags #method:setStateFlags#

#if defined(ENABLE_OVERLOADING)
    WidgetSetStateFlagsMethodInfo           ,
#endif
    widgetSetStateFlags                     ,


-- ** setTooltipMarkup #method:setTooltipMarkup#

#if defined(ENABLE_OVERLOADING)
    WidgetSetTooltipMarkupMethodInfo        ,
#endif
    widgetSetTooltipMarkup                  ,


-- ** setTooltipText #method:setTooltipText#

#if defined(ENABLE_OVERLOADING)
    WidgetSetTooltipTextMethodInfo          ,
#endif
    widgetSetTooltipText                    ,


-- ** setValign #method:setValign#

#if defined(ENABLE_OVERLOADING)
    WidgetSetValignMethodInfo               ,
#endif
    widgetSetValign                         ,


-- ** setVexpand #method:setVexpand#

#if defined(ENABLE_OVERLOADING)
    WidgetSetVexpandMethodInfo              ,
#endif
    widgetSetVexpand                        ,


-- ** setVexpandSet #method:setVexpandSet#

#if defined(ENABLE_OVERLOADING)
    WidgetSetVexpandSetMethodInfo           ,
#endif
    widgetSetVexpandSet                     ,


-- ** setVisible #method:setVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetSetVisibleMethodInfo              ,
#endif
    widgetSetVisible                        ,


-- ** shouldLayout #method:shouldLayout#

#if defined(ENABLE_OVERLOADING)
    WidgetShouldLayoutMethodInfo            ,
#endif
    widgetShouldLayout                      ,


-- ** show #method:show#

#if defined(ENABLE_OVERLOADING)
    WidgetShowMethodInfo                    ,
#endif
    widgetShow                              ,


-- ** sizeAllocate #method:sizeAllocate#

#if defined(ENABLE_OVERLOADING)
    WidgetSizeAllocateMethodInfo            ,
#endif
    widgetSizeAllocate                      ,


-- ** snapshotChild #method:snapshotChild#

#if defined(ENABLE_OVERLOADING)
    WidgetSnapshotChildMethodInfo           ,
#endif
    widgetSnapshotChild                     ,


-- ** translateCoordinates #method:translateCoordinates#

#if defined(ENABLE_OVERLOADING)
    WidgetTranslateCoordinatesMethodInfo    ,
#endif
    widgetTranslateCoordinates              ,


-- ** triggerTooltipQuery #method:triggerTooltipQuery#

#if defined(ENABLE_OVERLOADING)
    WidgetTriggerTooltipQueryMethodInfo     ,
#endif
    widgetTriggerTooltipQuery               ,


-- ** unmap #method:unmap#

#if defined(ENABLE_OVERLOADING)
    WidgetUnmapMethodInfo                   ,
#endif
    widgetUnmap                             ,


-- ** unparent #method:unparent#

#if defined(ENABLE_OVERLOADING)
    WidgetUnparentMethodInfo                ,
#endif
    widgetUnparent                          ,


-- ** unrealize #method:unrealize#

#if defined(ENABLE_OVERLOADING)
    WidgetUnrealizeMethodInfo               ,
#endif
    widgetUnrealize                         ,


-- ** unsetStateFlags #method:unsetStateFlags#

#if defined(ENABLE_OVERLOADING)
    WidgetUnsetStateFlagsMethodInfo         ,
#endif
    widgetUnsetStateFlags                   ,




 -- * Properties


-- ** canFocus #attr:canFocus#
-- | Whether the widget or any of its descendents can accept
-- the input focus.
-- 
-- This property is meant to be set by widget implementations,
-- typically in their instance init function.

#if defined(ENABLE_OVERLOADING)
    WidgetCanFocusPropertyInfo              ,
#endif
    constructWidgetCanFocus                 ,
    getWidgetCanFocus                       ,
    setWidgetCanFocus                       ,
#if defined(ENABLE_OVERLOADING)
    widgetCanFocus                          ,
#endif


-- ** canTarget #attr:canTarget#
-- | Whether the widget can receive pointer events.

#if defined(ENABLE_OVERLOADING)
    WidgetCanTargetPropertyInfo             ,
#endif
    constructWidgetCanTarget                ,
    getWidgetCanTarget                      ,
    setWidgetCanTarget                      ,
#if defined(ENABLE_OVERLOADING)
    widgetCanTarget                         ,
#endif


-- ** cssClasses #attr:cssClasses#
-- | A list of css classes applied to this widget.

#if defined(ENABLE_OVERLOADING)
    WidgetCssClassesPropertyInfo            ,
#endif
    constructWidgetCssClasses               ,
    getWidgetCssClasses                     ,
    setWidgetCssClasses                     ,
#if defined(ENABLE_OVERLOADING)
    widgetCssClasses                        ,
#endif


-- ** cssName #attr:cssName#
-- | The name of this widget in the CSS tree.
-- 
-- This property is meant to be set by widget implementations,
-- typically in their instance init function.

#if defined(ENABLE_OVERLOADING)
    WidgetCssNamePropertyInfo               ,
#endif
    constructWidgetCssName                  ,
    getWidgetCssName                        ,
#if defined(ENABLE_OVERLOADING)
    widgetCssName                           ,
#endif


-- ** cursor #attr:cursor#
-- | The cursor used by /@widget@/.

#if defined(ENABLE_OVERLOADING)
    WidgetCursorPropertyInfo                ,
#endif
    clearWidgetCursor                       ,
    constructWidgetCursor                   ,
    getWidgetCursor                         ,
    setWidgetCursor                         ,
#if defined(ENABLE_OVERLOADING)
    widgetCursor                            ,
#endif


-- ** focusOnClick #attr:focusOnClick#
-- | Whether the widget should grab focus when it is clicked with the mouse.
-- 
-- This property is only relevant for widgets that can take focus.

#if defined(ENABLE_OVERLOADING)
    WidgetFocusOnClickPropertyInfo          ,
#endif
    constructWidgetFocusOnClick             ,
    getWidgetFocusOnClick                   ,
    setWidgetFocusOnClick                   ,
#if defined(ENABLE_OVERLOADING)
    widgetFocusOnClick                      ,
#endif


-- ** focusable #attr:focusable#
-- | Whether this widget itself will accept the input focus.

#if defined(ENABLE_OVERLOADING)
    WidgetFocusablePropertyInfo             ,
#endif
    constructWidgetFocusable                ,
    getWidgetFocusable                      ,
    setWidgetFocusable                      ,
#if defined(ENABLE_OVERLOADING)
    widgetFocusable                         ,
#endif


-- ** halign #attr:halign#
-- | How to distribute horizontal space if widget gets extra space.

#if defined(ENABLE_OVERLOADING)
    WidgetHalignPropertyInfo                ,
#endif
    constructWidgetHalign                   ,
    getWidgetHalign                         ,
    setWidgetHalign                         ,
#if defined(ENABLE_OVERLOADING)
    widgetHalign                            ,
#endif


-- ** hasDefault #attr:hasDefault#
-- | Whether the widget is the default widget.

#if defined(ENABLE_OVERLOADING)
    WidgetHasDefaultPropertyInfo            ,
#endif
    getWidgetHasDefault                     ,


-- ** hasFocus #attr:hasFocus#
-- | Whether the widget has the input focus.

#if defined(ENABLE_OVERLOADING)
    WidgetHasFocusPropertyInfo              ,
#endif
    getWidgetHasFocus                       ,


-- ** hasTooltip #attr:hasTooltip#
-- | Enables or disables the emission of the [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip")
-- signal on /@widget@/.
-- 
-- A true value indicates that /@widget@/ can have a tooltip, in this case
-- the widget will be queried using [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") to
-- determine whether it will provide a tooltip or not.

#if defined(ENABLE_OVERLOADING)
    WidgetHasTooltipPropertyInfo            ,
#endif
    constructWidgetHasTooltip               ,
    getWidgetHasTooltip                     ,
    setWidgetHasTooltip                     ,
#if defined(ENABLE_OVERLOADING)
    widgetHasTooltip                        ,
#endif


-- ** heightRequest #attr:heightRequest#
-- | Overrides for height request of the widget.
-- 
-- If this is -1, the natural request will be used.

#if defined(ENABLE_OVERLOADING)
    WidgetHeightRequestPropertyInfo         ,
#endif
    constructWidgetHeightRequest            ,
    getWidgetHeightRequest                  ,
    setWidgetHeightRequest                  ,
#if defined(ENABLE_OVERLOADING)
    widgetHeightRequest                     ,
#endif


-- ** hexpand #attr:hexpand#
-- | Whether to expand horizontally.

#if defined(ENABLE_OVERLOADING)
    WidgetHexpandPropertyInfo               ,
#endif
    constructWidgetHexpand                  ,
    getWidgetHexpand                        ,
    setWidgetHexpand                        ,
#if defined(ENABLE_OVERLOADING)
    widgetHexpand                           ,
#endif


-- ** hexpandSet #attr:hexpandSet#
-- | Whether to use the @hexpand@ property.

#if defined(ENABLE_OVERLOADING)
    WidgetHexpandSetPropertyInfo            ,
#endif
    constructWidgetHexpandSet               ,
    getWidgetHexpandSet                     ,
    setWidgetHexpandSet                     ,
#if defined(ENABLE_OVERLOADING)
    widgetHexpandSet                        ,
#endif


-- ** layoutManager #attr:layoutManager#
-- | The t'GI.Gtk.Objects.LayoutManager.LayoutManager' instance to use to compute
-- the preferred size of the widget, and allocate its children.
-- 
-- This property is meant to be set by widget implementations,
-- typically in their instance init function.

#if defined(ENABLE_OVERLOADING)
    WidgetLayoutManagerPropertyInfo         ,
#endif
    clearWidgetLayoutManager                ,
    constructWidgetLayoutManager            ,
    getWidgetLayoutManager                  ,
    setWidgetLayoutManager                  ,
#if defined(ENABLE_OVERLOADING)
    widgetLayoutManager                     ,
#endif


-- ** limitEvents #attr:limitEvents#
-- | Makes this widget act like a modal dialog, with respect to
-- event delivery.
-- 
-- Global event controllers will not handle events with targets
-- inside the widget, unless they are set up to ignore propagation
-- limits. See 'GI.Gtk.Objects.EventController.eventControllerSetPropagationLimit'.
-- 
-- /Since: 4.18/

#if defined(ENABLE_OVERLOADING)
    WidgetLimitEventsPropertyInfo           ,
#endif
    constructWidgetLimitEvents              ,
    getWidgetLimitEvents                    ,
    setWidgetLimitEvents                    ,
#if defined(ENABLE_OVERLOADING)
    widgetLimitEvents                       ,
#endif


-- ** marginBottom #attr:marginBottom#
-- | Margin on bottom side of widget.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.

#if defined(ENABLE_OVERLOADING)
    WidgetMarginBottomPropertyInfo          ,
#endif
    constructWidgetMarginBottom             ,
    getWidgetMarginBottom                   ,
    setWidgetMarginBottom                   ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginBottom                      ,
#endif


-- ** marginEnd #attr:marginEnd#
-- | Margin on end of widget, horizontally.
-- 
-- This property supports left-to-right and right-to-left text
-- directions.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.

#if defined(ENABLE_OVERLOADING)
    WidgetMarginEndPropertyInfo             ,
#endif
    constructWidgetMarginEnd                ,
    getWidgetMarginEnd                      ,
    setWidgetMarginEnd                      ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginEnd                         ,
#endif


-- ** marginStart #attr:marginStart#
-- | Margin on start of widget, horizontally.
-- 
-- This property supports left-to-right and right-to-left text
-- directions.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.

#if defined(ENABLE_OVERLOADING)
    WidgetMarginStartPropertyInfo           ,
#endif
    constructWidgetMarginStart              ,
    getWidgetMarginStart                    ,
    setWidgetMarginStart                    ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginStart                       ,
#endif


-- ** marginTop #attr:marginTop#
-- | Margin on top side of widget.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.

#if defined(ENABLE_OVERLOADING)
    WidgetMarginTopPropertyInfo             ,
#endif
    constructWidgetMarginTop                ,
    getWidgetMarginTop                      ,
    setWidgetMarginTop                      ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginTop                         ,
#endif


-- ** name #attr:name#
-- | The name of the widget.

#if defined(ENABLE_OVERLOADING)
    WidgetNamePropertyInfo                  ,
#endif
    constructWidgetName                     ,
    getWidgetName                           ,
    setWidgetName                           ,
#if defined(ENABLE_OVERLOADING)
    widgetName                              ,
#endif


-- ** opacity #attr:opacity#
-- | The requested opacity of the widget.

#if defined(ENABLE_OVERLOADING)
    WidgetOpacityPropertyInfo               ,
#endif
    constructWidgetOpacity                  ,
    getWidgetOpacity                        ,
    setWidgetOpacity                        ,
#if defined(ENABLE_OVERLOADING)
    widgetOpacity                           ,
#endif


-- ** overflow #attr:overflow#
-- | How content outside the widget\'s content area is treated.
-- 
-- This property is meant to be set by widget implementations,
-- typically in their instance init function.

#if defined(ENABLE_OVERLOADING)
    WidgetOverflowPropertyInfo              ,
#endif
    constructWidgetOverflow                 ,
    getWidgetOverflow                       ,
    setWidgetOverflow                       ,
#if defined(ENABLE_OVERLOADING)
    widgetOverflow                          ,
#endif


-- ** parent #attr:parent#
-- | The parent widget of this widget.

#if defined(ENABLE_OVERLOADING)
    WidgetParentPropertyInfo                ,
#endif
    getWidgetParent                         ,
#if defined(ENABLE_OVERLOADING)
    widgetParent                            ,
#endif


-- ** receivesDefault #attr:receivesDefault#
-- | Whether the widget will receive the default action when it is focused.

#if defined(ENABLE_OVERLOADING)
    WidgetReceivesDefaultPropertyInfo       ,
#endif
    constructWidgetReceivesDefault          ,
    getWidgetReceivesDefault                ,
    setWidgetReceivesDefault                ,
#if defined(ENABLE_OVERLOADING)
    widgetReceivesDefault                   ,
#endif


-- ** root #attr:root#
-- | The @GtkRoot@ widget of the widget tree containing this widget.
-- 
-- This will be @NULL@ if the widget is not contained in a root widget.

#if defined(ENABLE_OVERLOADING)
    WidgetRootPropertyInfo                  ,
#endif
    getWidgetRoot                           ,
#if defined(ENABLE_OVERLOADING)
    widgetRoot                              ,
#endif


-- ** scaleFactor #attr:scaleFactor#
-- | The scale factor of the widget.

#if defined(ENABLE_OVERLOADING)
    WidgetScaleFactorPropertyInfo           ,
#endif
    getWidgetScaleFactor                    ,
#if defined(ENABLE_OVERLOADING)
    widgetScaleFactor                       ,
#endif


-- ** sensitive #attr:sensitive#
-- | Whether the widget responds to input.

#if defined(ENABLE_OVERLOADING)
    WidgetSensitivePropertyInfo             ,
#endif
    constructWidgetSensitive                ,
    getWidgetSensitive                      ,
    setWidgetSensitive                      ,
#if defined(ENABLE_OVERLOADING)
    widgetSensitive                         ,
#endif


-- ** tooltipMarkup #attr:tooltipMarkup#
-- | Sets the text of tooltip to be the given string, which is marked up
-- with Pango markup.
-- 
-- Also see 'GI.Gtk.Objects.Tooltip.tooltipSetMarkup'.
-- 
-- This is a convenience property which will take care of getting the
-- tooltip shown if the given string is not @NULL@:
-- [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip") will automatically be set to true
-- and there will be taken care of [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") in
-- the default signal handler.
-- 
-- Note that if both [Widget:tooltipText]("GI.Gtk.Objects.Widget#g:attr:tooltipText") and
-- [Widget:tooltipMarkup]("GI.Gtk.Objects.Widget#g:attr:tooltipMarkup") are set, the last one wins.

#if defined(ENABLE_OVERLOADING)
    WidgetTooltipMarkupPropertyInfo         ,
#endif
    clearWidgetTooltipMarkup                ,
    constructWidgetTooltipMarkup            ,
    getWidgetTooltipMarkup                  ,
    setWidgetTooltipMarkup                  ,
#if defined(ENABLE_OVERLOADING)
    widgetTooltipMarkup                     ,
#endif


-- ** tooltipText #attr:tooltipText#
-- | Sets the text of tooltip to be the given string.
-- 
-- Also see 'GI.Gtk.Objects.Tooltip.tooltipSetText'.
-- 
-- This is a convenience property which will take care of getting the
-- tooltip shown if the given string is not @NULL@:
-- [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip") will automatically be set to true
-- and there will be taken care of [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") in
-- the default signal handler.
-- 
-- Note that if both [Widget:tooltipText]("GI.Gtk.Objects.Widget#g:attr:tooltipText") and
-- [Widget:tooltipMarkup]("GI.Gtk.Objects.Widget#g:attr:tooltipMarkup") are set, the last one wins.

#if defined(ENABLE_OVERLOADING)
    WidgetTooltipTextPropertyInfo           ,
#endif
    clearWidgetTooltipText                  ,
    constructWidgetTooltipText              ,
    getWidgetTooltipText                    ,
    setWidgetTooltipText                    ,
#if defined(ENABLE_OVERLOADING)
    widgetTooltipText                       ,
#endif


-- ** valign #attr:valign#
-- | How to distribute vertical space if widget gets extra space.

#if defined(ENABLE_OVERLOADING)
    WidgetValignPropertyInfo                ,
#endif
    constructWidgetValign                   ,
    getWidgetValign                         ,
    setWidgetValign                         ,
#if defined(ENABLE_OVERLOADING)
    widgetValign                            ,
#endif


-- ** vexpand #attr:vexpand#
-- | Whether to expand vertically.

#if defined(ENABLE_OVERLOADING)
    WidgetVexpandPropertyInfo               ,
#endif
    constructWidgetVexpand                  ,
    getWidgetVexpand                        ,
    setWidgetVexpand                        ,
#if defined(ENABLE_OVERLOADING)
    widgetVexpand                           ,
#endif


-- ** vexpandSet #attr:vexpandSet#
-- | Whether to use the @vexpand@ property.

#if defined(ENABLE_OVERLOADING)
    WidgetVexpandSetPropertyInfo            ,
#endif
    constructWidgetVexpandSet               ,
    getWidgetVexpandSet                     ,
    setWidgetVexpandSet                     ,
#if defined(ENABLE_OVERLOADING)
    widgetVexpandSet                        ,
#endif


-- ** visible #attr:visible#
-- | Whether the widget is visible.

#if defined(ENABLE_OVERLOADING)
    WidgetVisiblePropertyInfo               ,
#endif
    constructWidgetVisible                  ,
    getWidgetVisible                        ,
    setWidgetVisible                        ,
#if defined(ENABLE_OVERLOADING)
    widgetVisible                           ,
#endif


-- ** widthRequest #attr:widthRequest#
-- | Overrides for width request of the widget.
-- 
-- If this is -1, the natural request will be used.

#if defined(ENABLE_OVERLOADING)
    WidgetWidthRequestPropertyInfo          ,
#endif
    constructWidgetWidthRequest             ,
    getWidgetWidthRequest                   ,
    setWidgetWidthRequest                   ,
#if defined(ENABLE_OVERLOADING)
    widgetWidthRequest                      ,
#endif




 -- * Signals


-- ** destroy #signal:destroy#

    WidgetDestroyCallback                   ,
#if defined(ENABLE_OVERLOADING)
    WidgetDestroySignalInfo                 ,
#endif
    afterWidgetDestroy                      ,
    onWidgetDestroy                         ,


-- ** directionChanged #signal:directionChanged#

    WidgetDirectionChangedCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetDirectionChangedSignalInfo        ,
#endif
    afterWidgetDirectionChanged             ,
    onWidgetDirectionChanged                ,


-- ** hide #signal:hide#

    WidgetHideCallback                      ,
#if defined(ENABLE_OVERLOADING)
    WidgetHideSignalInfo                    ,
#endif
    afterWidgetHide                         ,
    onWidgetHide                            ,


-- ** keynavFailed #signal:keynavFailed#

    WidgetKeynavFailedCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetKeynavFailedSignalInfo            ,
#endif
    afterWidgetKeynavFailed                 ,
    onWidgetKeynavFailed                    ,


-- ** map #signal:map#

    WidgetMapCallback                       ,
#if defined(ENABLE_OVERLOADING)
    WidgetMapSignalInfo                     ,
#endif
    afterWidgetMap                          ,
    onWidgetMap                             ,


-- ** mnemonicActivate #signal:mnemonicActivate#

    WidgetMnemonicActivateCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetMnemonicActivateSignalInfo        ,
#endif
    afterWidgetMnemonicActivate             ,
    onWidgetMnemonicActivate                ,


-- ** moveFocus #signal:moveFocus#

    WidgetMoveFocusCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetMoveFocusSignalInfo               ,
#endif
    afterWidgetMoveFocus                    ,
    onWidgetMoveFocus                       ,


-- ** queryTooltip #signal:queryTooltip#

    WidgetQueryTooltipCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetQueryTooltipSignalInfo            ,
#endif
    afterWidgetQueryTooltip                 ,
    onWidgetQueryTooltip                    ,


-- ** realize #signal:realize#

    WidgetRealizeCallback                   ,
#if defined(ENABLE_OVERLOADING)
    WidgetRealizeSignalInfo                 ,
#endif
    afterWidgetRealize                      ,
    onWidgetRealize                         ,


-- ** show #signal:show#

    WidgetShowCallback                      ,
#if defined(ENABLE_OVERLOADING)
    WidgetShowSignalInfo                    ,
#endif
    afterWidgetShow                         ,
    onWidgetShow                            ,


-- ** stateFlagsChanged #signal:stateFlagsChanged#

    WidgetStateFlagsChangedCallback         ,
#if defined(ENABLE_OVERLOADING)
    WidgetStateFlagsChangedSignalInfo       ,
#endif
    afterWidgetStateFlagsChanged            ,
    onWidgetStateFlagsChanged               ,


-- ** unmap #signal:unmap#

    WidgetUnmapCallback                     ,
#if defined(ENABLE_OVERLOADING)
    WidgetUnmapSignalInfo                   ,
#endif
    afterWidgetUnmap                        ,
    onWidgetUnmap                           ,


-- ** unrealize #signal:unrealize#

    WidgetUnrealizeCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetUnrealizeSignalInfo               ,
#endif
    afterWidgetUnrealize                    ,
    onWidgetUnrealize                       ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#endif

-- | Memory-managed wrapper type.
newtype Widget = Widget (SP.ManagedPtr Widget)
    deriving (Eq)

instance SP.ManagedPtrNewtype Widget where
    toManagedPtr (Widget p) = p

foreign import ccall "gtk_widget_get_type"
    c_gtk_widget_get_type :: IO B.Types.GType

instance B.Types.TypedObject Widget where
    glibType = c_gtk_widget_get_type

instance B.Types.GObject Widget

-- | Type class for types which can be safely cast to `Widget`, for instance with `toWidget`.
class (SP.GObject o, O.IsDescendantOf Widget o) => IsWidget o
instance (SP.GObject o, O.IsDescendantOf Widget o) => IsWidget o

instance O.HasParentTypes Widget
type instance O.ParentTypes Widget = '[GObject.Object.Object, Gtk.Accessible.Accessible, Gtk.Buildable.Buildable, Gtk.ConstraintTarget.ConstraintTarget]

-- | Cast to `Widget`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toWidget :: (MIO.MonadIO m, IsWidget o) => o -> m Widget
toWidget = MIO.liftIO . B.ManagedPtr.unsafeCastTo Widget

-- | Convert 'Widget' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Widget) where
    gvalueGType_ = c_gtk_widget_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Widget)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Widget)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Widget ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveWidgetMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveWidgetMethod "actionSetEnabled" o = WidgetActionSetEnabledMethodInfo
    ResolveWidgetMethod "activate" o = WidgetActivateMethodInfo
    ResolveWidgetMethod "activateAction" o = WidgetActivateActionMethodInfo
    ResolveWidgetMethod "activateDefault" o = WidgetActivateDefaultMethodInfo
    ResolveWidgetMethod "addController" o = WidgetAddControllerMethodInfo
    ResolveWidgetMethod "addCssClass" o = WidgetAddCssClassMethodInfo
    ResolveWidgetMethod "addMnemonicLabel" o = WidgetAddMnemonicLabelMethodInfo
    ResolveWidgetMethod "addTickCallback" o = WidgetAddTickCallbackMethodInfo
    ResolveWidgetMethod "allocate" o = WidgetAllocateMethodInfo
    ResolveWidgetMethod "announce" o = Gtk.Accessible.AccessibleAnnounceMethodInfo
    ResolveWidgetMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveWidgetMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveWidgetMethod "childFocus" o = WidgetChildFocusMethodInfo
    ResolveWidgetMethod "computeBounds" o = WidgetComputeBoundsMethodInfo
    ResolveWidgetMethod "computeExpand" o = WidgetComputeExpandMethodInfo
    ResolveWidgetMethod "computePoint" o = WidgetComputePointMethodInfo
    ResolveWidgetMethod "computeTransform" o = WidgetComputeTransformMethodInfo
    ResolveWidgetMethod "contains" o = WidgetContainsMethodInfo
    ResolveWidgetMethod "createPangoContext" o = WidgetCreatePangoContextMethodInfo
    ResolveWidgetMethod "createPangoLayout" o = WidgetCreatePangoLayoutMethodInfo
    ResolveWidgetMethod "disposeTemplate" o = WidgetDisposeTemplateMethodInfo
    ResolveWidgetMethod "dragCheckThreshold" o = WidgetDragCheckThresholdMethodInfo
    ResolveWidgetMethod "errorBell" o = WidgetErrorBellMethodInfo
    ResolveWidgetMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveWidgetMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveWidgetMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveWidgetMethod "grabFocus" o = WidgetGrabFocusMethodInfo
    ResolveWidgetMethod "hasCssClass" o = WidgetHasCssClassMethodInfo
    ResolveWidgetMethod "hasDefault" o = WidgetHasDefaultMethodInfo
    ResolveWidgetMethod "hasFocus" o = WidgetHasFocusMethodInfo
    ResolveWidgetMethod "hasVisibleFocus" o = WidgetHasVisibleFocusMethodInfo
    ResolveWidgetMethod "hide" o = WidgetHideMethodInfo
    ResolveWidgetMethod "inDestruction" o = WidgetInDestructionMethodInfo
    ResolveWidgetMethod "initTemplate" o = WidgetInitTemplateMethodInfo
    ResolveWidgetMethod "insertActionGroup" o = WidgetInsertActionGroupMethodInfo
    ResolveWidgetMethod "insertAfter" o = WidgetInsertAfterMethodInfo
    ResolveWidgetMethod "insertBefore" o = WidgetInsertBeforeMethodInfo
    ResolveWidgetMethod "isAncestor" o = WidgetIsAncestorMethodInfo
    ResolveWidgetMethod "isDrawable" o = WidgetIsDrawableMethodInfo
    ResolveWidgetMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveWidgetMethod "isFocus" o = WidgetIsFocusMethodInfo
    ResolveWidgetMethod "isSensitive" o = WidgetIsSensitiveMethodInfo
    ResolveWidgetMethod "isVisible" o = WidgetIsVisibleMethodInfo
    ResolveWidgetMethod "keynavFailed" o = WidgetKeynavFailedMethodInfo
    ResolveWidgetMethod "listMnemonicLabels" o = WidgetListMnemonicLabelsMethodInfo
    ResolveWidgetMethod "map" o = WidgetMapMethodInfo
    ResolveWidgetMethod "measure" o = WidgetMeasureMethodInfo
    ResolveWidgetMethod "mnemonicActivate" o = WidgetMnemonicActivateMethodInfo
    ResolveWidgetMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveWidgetMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveWidgetMethod "observeChildren" o = WidgetObserveChildrenMethodInfo
    ResolveWidgetMethod "observeControllers" o = WidgetObserveControllersMethodInfo
    ResolveWidgetMethod "pick" o = WidgetPickMethodInfo
    ResolveWidgetMethod "queueAllocate" o = WidgetQueueAllocateMethodInfo
    ResolveWidgetMethod "queueDraw" o = WidgetQueueDrawMethodInfo
    ResolveWidgetMethod "queueResize" o = WidgetQueueResizeMethodInfo
    ResolveWidgetMethod "realize" o = WidgetRealizeMethodInfo
    ResolveWidgetMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveWidgetMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveWidgetMethod "removeController" o = WidgetRemoveControllerMethodInfo
    ResolveWidgetMethod "removeCssClass" o = WidgetRemoveCssClassMethodInfo
    ResolveWidgetMethod "removeMnemonicLabel" o = WidgetRemoveMnemonicLabelMethodInfo
    ResolveWidgetMethod "removeTickCallback" o = WidgetRemoveTickCallbackMethodInfo
    ResolveWidgetMethod "resetProperty" o = Gtk.Accessible.AccessibleResetPropertyMethodInfo
    ResolveWidgetMethod "resetRelation" o = Gtk.Accessible.AccessibleResetRelationMethodInfo
    ResolveWidgetMethod "resetState" o = Gtk.Accessible.AccessibleResetStateMethodInfo
    ResolveWidgetMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveWidgetMethod "shouldLayout" o = WidgetShouldLayoutMethodInfo
    ResolveWidgetMethod "show" o = WidgetShowMethodInfo
    ResolveWidgetMethod "sizeAllocate" o = WidgetSizeAllocateMethodInfo
    ResolveWidgetMethod "snapshotChild" o = WidgetSnapshotChildMethodInfo
    ResolveWidgetMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveWidgetMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveWidgetMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveWidgetMethod "translateCoordinates" o = WidgetTranslateCoordinatesMethodInfo
    ResolveWidgetMethod "triggerTooltipQuery" o = WidgetTriggerTooltipQueryMethodInfo
    ResolveWidgetMethod "unmap" o = WidgetUnmapMethodInfo
    ResolveWidgetMethod "unparent" o = WidgetUnparentMethodInfo
    ResolveWidgetMethod "unrealize" o = WidgetUnrealizeMethodInfo
    ResolveWidgetMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveWidgetMethod "unsetStateFlags" o = WidgetUnsetStateFlagsMethodInfo
    ResolveWidgetMethod "updateNextAccessibleSibling" o = Gtk.Accessible.AccessibleUpdateNextAccessibleSiblingMethodInfo
    ResolveWidgetMethod "updatePlatformState" o = Gtk.Accessible.AccessibleUpdatePlatformStateMethodInfo
    ResolveWidgetMethod "updateProperty" o = Gtk.Accessible.AccessibleUpdatePropertyMethodInfo
    ResolveWidgetMethod "updateRelation" o = Gtk.Accessible.AccessibleUpdateRelationMethodInfo
    ResolveWidgetMethod "updateState" o = Gtk.Accessible.AccessibleUpdateStateMethodInfo
    ResolveWidgetMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveWidgetMethod "getAccessibleParent" o = Gtk.Accessible.AccessibleGetAccessibleParentMethodInfo
    ResolveWidgetMethod "getAccessibleRole" o = Gtk.Accessible.AccessibleGetAccessibleRoleMethodInfo
    ResolveWidgetMethod "getAllocatedBaseline" o = WidgetGetAllocatedBaselineMethodInfo
    ResolveWidgetMethod "getAllocatedHeight" o = WidgetGetAllocatedHeightMethodInfo
    ResolveWidgetMethod "getAllocatedWidth" o = WidgetGetAllocatedWidthMethodInfo
    ResolveWidgetMethod "getAllocation" o = WidgetGetAllocationMethodInfo
    ResolveWidgetMethod "getAncestor" o = WidgetGetAncestorMethodInfo
    ResolveWidgetMethod "getAtContext" o = Gtk.Accessible.AccessibleGetAtContextMethodInfo
    ResolveWidgetMethod "getBaseline" o = WidgetGetBaselineMethodInfo
    ResolveWidgetMethod "getBounds" o = Gtk.Accessible.AccessibleGetBoundsMethodInfo
    ResolveWidgetMethod "getBuildableId" o = Gtk.Buildable.BuildableGetBuildableIdMethodInfo
    ResolveWidgetMethod "getCanFocus" o = WidgetGetCanFocusMethodInfo
    ResolveWidgetMethod "getCanTarget" o = WidgetGetCanTargetMethodInfo
    ResolveWidgetMethod "getChildVisible" o = WidgetGetChildVisibleMethodInfo
    ResolveWidgetMethod "getClipboard" o = WidgetGetClipboardMethodInfo
    ResolveWidgetMethod "getColor" o = WidgetGetColorMethodInfo
    ResolveWidgetMethod "getCssClasses" o = WidgetGetCssClassesMethodInfo
    ResolveWidgetMethod "getCssName" o = WidgetGetCssNameMethodInfo
    ResolveWidgetMethod "getCursor" o = WidgetGetCursorMethodInfo
    ResolveWidgetMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveWidgetMethod "getDirection" o = WidgetGetDirectionMethodInfo
    ResolveWidgetMethod "getDisplay" o = WidgetGetDisplayMethodInfo
    ResolveWidgetMethod "getFirstAccessibleChild" o = Gtk.Accessible.AccessibleGetFirstAccessibleChildMethodInfo
    ResolveWidgetMethod "getFirstChild" o = WidgetGetFirstChildMethodInfo
    ResolveWidgetMethod "getFocusChild" o = WidgetGetFocusChildMethodInfo
    ResolveWidgetMethod "getFocusOnClick" o = WidgetGetFocusOnClickMethodInfo
    ResolveWidgetMethod "getFocusable" o = WidgetGetFocusableMethodInfo
    ResolveWidgetMethod "getFontMap" o = WidgetGetFontMapMethodInfo
    ResolveWidgetMethod "getFontOptions" o = WidgetGetFontOptionsMethodInfo
    ResolveWidgetMethod "getFrameClock" o = WidgetGetFrameClockMethodInfo
    ResolveWidgetMethod "getHalign" o = WidgetGetHalignMethodInfo
    ResolveWidgetMethod "getHasTooltip" o = WidgetGetHasTooltipMethodInfo
    ResolveWidgetMethod "getHeight" o = WidgetGetHeightMethodInfo
    ResolveWidgetMethod "getHexpand" o = WidgetGetHexpandMethodInfo
    ResolveWidgetMethod "getHexpandSet" o = WidgetGetHexpandSetMethodInfo
    ResolveWidgetMethod "getLastChild" o = WidgetGetLastChildMethodInfo
    ResolveWidgetMethod "getLayoutManager" o = WidgetGetLayoutManagerMethodInfo
    ResolveWidgetMethod "getLimitEvents" o = WidgetGetLimitEventsMethodInfo
    ResolveWidgetMethod "getMapped" o = WidgetGetMappedMethodInfo
    ResolveWidgetMethod "getMarginBottom" o = WidgetGetMarginBottomMethodInfo
    ResolveWidgetMethod "getMarginEnd" o = WidgetGetMarginEndMethodInfo
    ResolveWidgetMethod "getMarginStart" o = WidgetGetMarginStartMethodInfo
    ResolveWidgetMethod "getMarginTop" o = WidgetGetMarginTopMethodInfo
    ResolveWidgetMethod "getName" o = WidgetGetNameMethodInfo
    ResolveWidgetMethod "getNative" o = WidgetGetNativeMethodInfo
    ResolveWidgetMethod "getNextAccessibleSibling" o = Gtk.Accessible.AccessibleGetNextAccessibleSiblingMethodInfo
    ResolveWidgetMethod "getNextSibling" o = WidgetGetNextSiblingMethodInfo
    ResolveWidgetMethod "getOpacity" o = WidgetGetOpacityMethodInfo
    ResolveWidgetMethod "getOverflow" o = WidgetGetOverflowMethodInfo
    ResolveWidgetMethod "getPangoContext" o = WidgetGetPangoContextMethodInfo
    ResolveWidgetMethod "getParent" o = WidgetGetParentMethodInfo
    ResolveWidgetMethod "getPlatformState" o = Gtk.Accessible.AccessibleGetPlatformStateMethodInfo
    ResolveWidgetMethod "getPreferredSize" o = WidgetGetPreferredSizeMethodInfo
    ResolveWidgetMethod "getPrevSibling" o = WidgetGetPrevSiblingMethodInfo
    ResolveWidgetMethod "getPrimaryClipboard" o = WidgetGetPrimaryClipboardMethodInfo
    ResolveWidgetMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveWidgetMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveWidgetMethod "getRealized" o = WidgetGetRealizedMethodInfo
    ResolveWidgetMethod "getReceivesDefault" o = WidgetGetReceivesDefaultMethodInfo
    ResolveWidgetMethod "getRequestMode" o = WidgetGetRequestModeMethodInfo
    ResolveWidgetMethod "getRoot" o = WidgetGetRootMethodInfo
    ResolveWidgetMethod "getScaleFactor" o = WidgetGetScaleFactorMethodInfo
    ResolveWidgetMethod "getSensitive" o = WidgetGetSensitiveMethodInfo
    ResolveWidgetMethod "getSettings" o = WidgetGetSettingsMethodInfo
    ResolveWidgetMethod "getSize" o = WidgetGetSizeMethodInfo
    ResolveWidgetMethod "getSizeRequest" o = WidgetGetSizeRequestMethodInfo
    ResolveWidgetMethod "getStateFlags" o = WidgetGetStateFlagsMethodInfo
    ResolveWidgetMethod "getStyleContext" o = WidgetGetStyleContextMethodInfo
    ResolveWidgetMethod "getTemplateChild" o = WidgetGetTemplateChildMethodInfo
    ResolveWidgetMethod "getTooltipMarkup" o = WidgetGetTooltipMarkupMethodInfo
    ResolveWidgetMethod "getTooltipText" o = WidgetGetTooltipTextMethodInfo
    ResolveWidgetMethod "getValign" o = WidgetGetValignMethodInfo
    ResolveWidgetMethod "getVexpand" o = WidgetGetVexpandMethodInfo
    ResolveWidgetMethod "getVexpandSet" o = WidgetGetVexpandSetMethodInfo
    ResolveWidgetMethod "getVisible" o = WidgetGetVisibleMethodInfo
    ResolveWidgetMethod "getWidth" o = WidgetGetWidthMethodInfo
    ResolveWidgetMethod "setAccessibleParent" o = Gtk.Accessible.AccessibleSetAccessibleParentMethodInfo
    ResolveWidgetMethod "setCanFocus" o = WidgetSetCanFocusMethodInfo
    ResolveWidgetMethod "setCanTarget" o = WidgetSetCanTargetMethodInfo
    ResolveWidgetMethod "setChildVisible" o = WidgetSetChildVisibleMethodInfo
    ResolveWidgetMethod "setCssClasses" o = WidgetSetCssClassesMethodInfo
    ResolveWidgetMethod "setCursor" o = WidgetSetCursorMethodInfo
    ResolveWidgetMethod "setCursorFromName" o = WidgetSetCursorFromNameMethodInfo
    ResolveWidgetMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveWidgetMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveWidgetMethod "setDirection" o = WidgetSetDirectionMethodInfo
    ResolveWidgetMethod "setFocusChild" o = WidgetSetFocusChildMethodInfo
    ResolveWidgetMethod "setFocusOnClick" o = WidgetSetFocusOnClickMethodInfo
    ResolveWidgetMethod "setFocusable" o = WidgetSetFocusableMethodInfo
    ResolveWidgetMethod "setFontMap" o = WidgetSetFontMapMethodInfo
    ResolveWidgetMethod "setFontOptions" o = WidgetSetFontOptionsMethodInfo
    ResolveWidgetMethod "setHalign" o = WidgetSetHalignMethodInfo
    ResolveWidgetMethod "setHasTooltip" o = WidgetSetHasTooltipMethodInfo
    ResolveWidgetMethod "setHexpand" o = WidgetSetHexpandMethodInfo
    ResolveWidgetMethod "setHexpandSet" o = WidgetSetHexpandSetMethodInfo
    ResolveWidgetMethod "setLayoutManager" o = WidgetSetLayoutManagerMethodInfo
    ResolveWidgetMethod "setLimitEvents" o = WidgetSetLimitEventsMethodInfo
    ResolveWidgetMethod "setMarginBottom" o = WidgetSetMarginBottomMethodInfo
    ResolveWidgetMethod "setMarginEnd" o = WidgetSetMarginEndMethodInfo
    ResolveWidgetMethod "setMarginStart" o = WidgetSetMarginStartMethodInfo
    ResolveWidgetMethod "setMarginTop" o = WidgetSetMarginTopMethodInfo
    ResolveWidgetMethod "setName" o = WidgetSetNameMethodInfo
    ResolveWidgetMethod "setOpacity" o = WidgetSetOpacityMethodInfo
    ResolveWidgetMethod "setOverflow" o = WidgetSetOverflowMethodInfo
    ResolveWidgetMethod "setParent" o = WidgetSetParentMethodInfo
    ResolveWidgetMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveWidgetMethod "setReceivesDefault" o = WidgetSetReceivesDefaultMethodInfo
    ResolveWidgetMethod "setSensitive" o = WidgetSetSensitiveMethodInfo
    ResolveWidgetMethod "setSizeRequest" o = WidgetSetSizeRequestMethodInfo
    ResolveWidgetMethod "setStateFlags" o = WidgetSetStateFlagsMethodInfo
    ResolveWidgetMethod "setTooltipMarkup" o = WidgetSetTooltipMarkupMethodInfo
    ResolveWidgetMethod "setTooltipText" o = WidgetSetTooltipTextMethodInfo
    ResolveWidgetMethod "setValign" o = WidgetSetValignMethodInfo
    ResolveWidgetMethod "setVexpand" o = WidgetSetVexpandMethodInfo
    ResolveWidgetMethod "setVexpandSet" o = WidgetSetVexpandSetMethodInfo
    ResolveWidgetMethod "setVisible" o = WidgetSetVisibleMethodInfo
    ResolveWidgetMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveWidgetMethod t Widget, O.OverloadedMethod info Widget p) => OL.IsLabel t (Widget -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveWidgetMethod t Widget, O.OverloadedMethod info Widget p, R.HasField t Widget p) => R.HasField t Widget p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveWidgetMethod t Widget, O.OverloadedMethodInfo info Widget) => OL.IsLabel t (O.MethodProxy info Widget) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Widget::destroy
-- | Signals that all holders of a reference to the widget should release
-- the reference that they hold.
-- 
-- May result in finalization of the widget if all references are released.
-- 
-- This signal is not suitable for saving widget state.
type WidgetDestroyCallback =
    IO ()

type C_WidgetDestroyCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDestroyCallback`.
foreign import ccall "wrapper"
    mk_WidgetDestroyCallback :: C_WidgetDestroyCallback -> IO (FunPtr C_WidgetDestroyCallback)

wrap_WidgetDestroyCallback :: 
    GObject a => (a -> WidgetDestroyCallback) ->
    C_WidgetDestroyCallback
wrap_WidgetDestroyCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [destroy](#signal:destroy) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #destroy callback
-- @
-- 
-- 
onWidgetDestroy :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDestroyCallback) -> m SignalHandlerId
onWidgetDestroy obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDestroyCallback wrapped
    wrapped'' <- mk_WidgetDestroyCallback wrapped'
    connectSignalFunPtr obj "destroy" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [destroy](#signal:destroy) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #destroy callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDestroy :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDestroyCallback) -> m SignalHandlerId
afterWidgetDestroy obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDestroyCallback wrapped
    wrapped'' <- mk_WidgetDestroyCallback wrapped'
    connectSignalFunPtr obj "destroy" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDestroySignalInfo
instance SignalInfo WidgetDestroySignalInfo where
    type HaskellCallbackType WidgetDestroySignalInfo = WidgetDestroyCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDestroyCallback cb
        cb'' <- mk_WidgetDestroyCallback cb'
        connectSignalFunPtr obj "destroy" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::destroy"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:destroy"})

#endif

-- signal Widget::direction-changed
-- | Emitted when the text direction of a widget changes.
type WidgetDirectionChangedCallback =
    Gtk.Enums.TextDirection
    -- ^ /@previousDirection@/: the previous text direction
    -> IO ()

type C_WidgetDirectionChangedCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDirectionChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetDirectionChangedCallback :: C_WidgetDirectionChangedCallback -> IO (FunPtr C_WidgetDirectionChangedCallback)

wrap_WidgetDirectionChangedCallback :: 
    GObject a => (a -> WidgetDirectionChangedCallback) ->
    C_WidgetDirectionChangedCallback
wrap_WidgetDirectionChangedCallback gi'cb gi'selfPtr previousDirection _ = do
    let previousDirection' = (toEnum . fromIntegral) previousDirection
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  previousDirection'


-- | Connect a signal handler for the [directionChanged](#signal:directionChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #directionChanged callback
-- @
-- 
-- 
onWidgetDirectionChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDirectionChangedCallback) -> m SignalHandlerId
onWidgetDirectionChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDirectionChangedCallback wrapped
    wrapped'' <- mk_WidgetDirectionChangedCallback wrapped'
    connectSignalFunPtr obj "direction-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [directionChanged](#signal:directionChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #directionChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDirectionChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDirectionChangedCallback) -> m SignalHandlerId
afterWidgetDirectionChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDirectionChangedCallback wrapped
    wrapped'' <- mk_WidgetDirectionChangedCallback wrapped'
    connectSignalFunPtr obj "direction-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDirectionChangedSignalInfo
instance SignalInfo WidgetDirectionChangedSignalInfo where
    type HaskellCallbackType WidgetDirectionChangedSignalInfo = WidgetDirectionChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDirectionChangedCallback cb
        cb'' <- mk_WidgetDirectionChangedCallback cb'
        connectSignalFunPtr obj "direction-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::direction-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:directionChanged"})

#endif

-- signal Widget::hide
-- | Emitted when /@widget@/ is hidden.
type WidgetHideCallback =
    IO ()

type C_WidgetHideCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetHideCallback`.
foreign import ccall "wrapper"
    mk_WidgetHideCallback :: C_WidgetHideCallback -> IO (FunPtr C_WidgetHideCallback)

wrap_WidgetHideCallback :: 
    GObject a => (a -> WidgetHideCallback) ->
    C_WidgetHideCallback
wrap_WidgetHideCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [hide](#signal:hide) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #hide callback
-- @
-- 
-- 
onWidgetHide :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetHideCallback) -> m SignalHandlerId
onWidgetHide obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetHideCallback wrapped
    wrapped'' <- mk_WidgetHideCallback wrapped'
    connectSignalFunPtr obj "hide" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [hide](#signal:hide) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #hide callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetHide :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetHideCallback) -> m SignalHandlerId
afterWidgetHide obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetHideCallback wrapped
    wrapped'' <- mk_WidgetHideCallback wrapped'
    connectSignalFunPtr obj "hide" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetHideSignalInfo
instance SignalInfo WidgetHideSignalInfo where
    type HaskellCallbackType WidgetHideSignalInfo = WidgetHideCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetHideCallback cb
        cb'' <- mk_WidgetHideCallback cb'
        connectSignalFunPtr obj "hide" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::hide"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:hide"})

#endif

-- signal Widget::keynav-failed
-- | Emitted if keyboard navigation fails.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetKeynavFailed' for details.
type WidgetKeynavFailedCallback =
    Gtk.Enums.DirectionType
    -- ^ /@direction@/: the direction of movement
    -> IO Bool
    -- ^ __Returns:__ true if stopping keyboard navigation is fine, false
    --   if the emitting widget should try to handle the keyboard
    --   navigation attempt in its parent widget

type C_WidgetKeynavFailedCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetKeynavFailedCallback`.
foreign import ccall "wrapper"
    mk_WidgetKeynavFailedCallback :: C_WidgetKeynavFailedCallback -> IO (FunPtr C_WidgetKeynavFailedCallback)

wrap_WidgetKeynavFailedCallback :: 
    GObject a => (a -> WidgetKeynavFailedCallback) ->
    C_WidgetKeynavFailedCallback
wrap_WidgetKeynavFailedCallback gi'cb gi'selfPtr direction _ = do
    let direction' = (toEnum . fromIntegral) direction
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  direction'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [keynavFailed](#signal:keynavFailed) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #keynavFailed callback
-- @
-- 
-- 
onWidgetKeynavFailed :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetKeynavFailedCallback) -> m SignalHandlerId
onWidgetKeynavFailed obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetKeynavFailedCallback wrapped
    wrapped'' <- mk_WidgetKeynavFailedCallback wrapped'
    connectSignalFunPtr obj "keynav-failed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [keynavFailed](#signal:keynavFailed) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #keynavFailed callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetKeynavFailed :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetKeynavFailedCallback) -> m SignalHandlerId
afterWidgetKeynavFailed obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetKeynavFailedCallback wrapped
    wrapped'' <- mk_WidgetKeynavFailedCallback wrapped'
    connectSignalFunPtr obj "keynav-failed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetKeynavFailedSignalInfo
instance SignalInfo WidgetKeynavFailedSignalInfo where
    type HaskellCallbackType WidgetKeynavFailedSignalInfo = WidgetKeynavFailedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetKeynavFailedCallback cb
        cb'' <- mk_WidgetKeynavFailedCallback cb'
        connectSignalFunPtr obj "keynav-failed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::keynav-failed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:keynavFailed"})

#endif

-- signal Widget::map
-- | Emitted when /@widget@/ is going to be mapped.
-- 
-- A widget is mapped when the widget is visible (which is controlled with
-- [Widget:visible]("GI.Gtk.Objects.Widget#g:attr:visible")) and all its parents up to the toplevel widget
-- are also visible.
-- 
-- The @::map@ signal can be used to determine whether a widget will be drawn,
-- for instance it can resume an animation that was stopped during the
-- emission of [Widget::unmap]("GI.Gtk.Objects.Widget#g:signal:unmap").
type WidgetMapCallback =
    IO ()

type C_WidgetMapCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetMapCallback`.
foreign import ccall "wrapper"
    mk_WidgetMapCallback :: C_WidgetMapCallback -> IO (FunPtr C_WidgetMapCallback)

wrap_WidgetMapCallback :: 
    GObject a => (a -> WidgetMapCallback) ->
    C_WidgetMapCallback
wrap_WidgetMapCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [map](#signal:map) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #map callback
-- @
-- 
-- 
onWidgetMap :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMapCallback) -> m SignalHandlerId
onWidgetMap obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMapCallback wrapped
    wrapped'' <- mk_WidgetMapCallback wrapped'
    connectSignalFunPtr obj "map" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [map](#signal:map) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #map callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetMap :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMapCallback) -> m SignalHandlerId
afterWidgetMap obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMapCallback wrapped
    wrapped'' <- mk_WidgetMapCallback wrapped'
    connectSignalFunPtr obj "map" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetMapSignalInfo
instance SignalInfo WidgetMapSignalInfo where
    type HaskellCallbackType WidgetMapSignalInfo = WidgetMapCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetMapCallback cb
        cb'' <- mk_WidgetMapCallback cb'
        connectSignalFunPtr obj "map" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::map"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:map"})

#endif

-- signal Widget::mnemonic-activate
-- | Emitted when a widget is activated via a mnemonic.
-- 
-- The default handler for this signal activates /@widget@/ if /@groupCycling@/
-- is false, or just makes /@widget@/ grab focus if /@groupCycling@/ is true.
type WidgetMnemonicActivateCallback =
    Bool
    -- ^ /@groupCycling@/: true if there are other widgets with the same mnemonic
    -> IO Bool
    -- ^ __Returns:__ true to stop other handlers from being invoked for the event,
    --   false to propagate the event further

type C_WidgetMnemonicActivateCallback =
    Ptr Widget ->                           -- object
    CInt ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetMnemonicActivateCallback`.
foreign import ccall "wrapper"
    mk_WidgetMnemonicActivateCallback :: C_WidgetMnemonicActivateCallback -> IO (FunPtr C_WidgetMnemonicActivateCallback)

wrap_WidgetMnemonicActivateCallback :: 
    GObject a => (a -> WidgetMnemonicActivateCallback) ->
    C_WidgetMnemonicActivateCallback
wrap_WidgetMnemonicActivateCallback gi'cb gi'selfPtr groupCycling _ = do
    let groupCycling' = (/= 0) groupCycling
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  groupCycling'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [mnemonicActivate](#signal:mnemonicActivate) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #mnemonicActivate callback
-- @
-- 
-- 
onWidgetMnemonicActivate :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMnemonicActivateCallback) -> m SignalHandlerId
onWidgetMnemonicActivate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMnemonicActivateCallback wrapped
    wrapped'' <- mk_WidgetMnemonicActivateCallback wrapped'
    connectSignalFunPtr obj "mnemonic-activate" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [mnemonicActivate](#signal:mnemonicActivate) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #mnemonicActivate callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetMnemonicActivate :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMnemonicActivateCallback) -> m SignalHandlerId
afterWidgetMnemonicActivate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMnemonicActivateCallback wrapped
    wrapped'' <- mk_WidgetMnemonicActivateCallback wrapped'
    connectSignalFunPtr obj "mnemonic-activate" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetMnemonicActivateSignalInfo
instance SignalInfo WidgetMnemonicActivateSignalInfo where
    type HaskellCallbackType WidgetMnemonicActivateSignalInfo = WidgetMnemonicActivateCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetMnemonicActivateCallback cb
        cb'' <- mk_WidgetMnemonicActivateCallback cb'
        connectSignalFunPtr obj "mnemonic-activate" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::mnemonic-activate"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:mnemonicActivate"})

#endif

-- signal Widget::move-focus
-- | Emitted when the focus is moved.
-- 
-- The @::move-focus@ signal is a <https://docs.gtk.org/gtk4/class.SignalAction.html keybinding signal>.
-- 
-- The default bindings for this signal are \<kbd>Tab\<\/kbd> to move forward,
-- and \<kbd>Shift\<\/kbd>+\<kbd>Tab\<\/kbd> to move backward.
type WidgetMoveFocusCallback =
    Gtk.Enums.DirectionType
    -- ^ /@direction@/: the direction of the focus move
    -> IO ()

type C_WidgetMoveFocusCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetMoveFocusCallback`.
foreign import ccall "wrapper"
    mk_WidgetMoveFocusCallback :: C_WidgetMoveFocusCallback -> IO (FunPtr C_WidgetMoveFocusCallback)

wrap_WidgetMoveFocusCallback :: 
    GObject a => (a -> WidgetMoveFocusCallback) ->
    C_WidgetMoveFocusCallback
wrap_WidgetMoveFocusCallback gi'cb gi'selfPtr direction _ = do
    let direction' = (toEnum . fromIntegral) direction
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  direction'


-- | Connect a signal handler for the [moveFocus](#signal:moveFocus) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #moveFocus callback
-- @
-- 
-- 
onWidgetMoveFocus :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMoveFocusCallback) -> m SignalHandlerId
onWidgetMoveFocus obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMoveFocusCallback wrapped
    wrapped'' <- mk_WidgetMoveFocusCallback wrapped'
    connectSignalFunPtr obj "move-focus" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [moveFocus](#signal:moveFocus) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #moveFocus callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetMoveFocus :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMoveFocusCallback) -> m SignalHandlerId
afterWidgetMoveFocus obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMoveFocusCallback wrapped
    wrapped'' <- mk_WidgetMoveFocusCallback wrapped'
    connectSignalFunPtr obj "move-focus" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetMoveFocusSignalInfo
instance SignalInfo WidgetMoveFocusSignalInfo where
    type HaskellCallbackType WidgetMoveFocusSignalInfo = WidgetMoveFocusCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetMoveFocusCallback cb
        cb'' <- mk_WidgetMoveFocusCallback cb'
        connectSignalFunPtr obj "move-focus" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::move-focus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:moveFocus"})

#endif

-- signal Widget::query-tooltip
-- | Emitted when the widget’s tooltip is about to be shown.
-- 
-- This happens when the [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip") property
-- is true and the hover timeout has expired with the cursor hovering
-- above /@widget@/; or emitted when /@widget@/ got focus in keyboard mode.
-- 
-- Using the given coordinates, the signal handler should determine
-- whether a tooltip should be shown for /@widget@/. If this is the case
-- true should be returned, false otherwise. Note that if /@keyboardMode@/
-- is true, the values of /@x@/ and /@y@/ are undefined and should not be used.
-- 
-- The signal handler is free to manipulate /@tooltip@/ with the therefore
-- destined function calls.
type WidgetQueryTooltipCallback =
    Int32
    -- ^ /@x@/: the x coordinate of the cursor position in widget coordinates
    -> Int32
    -- ^ /@y@/: the y coordinate of the cursor position in widget coordinates
    -> Bool
    -- ^ /@keyboardMode@/: true if the tooltip was triggered using the keyboard
    -> Gtk.Tooltip.Tooltip
    -- ^ /@tooltip@/: a @GtkTooltip@
    -> IO Bool
    -- ^ __Returns:__ true if /@tooltip@/ should be shown right now, false otherwise

type C_WidgetQueryTooltipCallback =
    Ptr Widget ->                           -- object
    Int32 ->
    Int32 ->
    CInt ->
    Ptr Gtk.Tooltip.Tooltip ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetQueryTooltipCallback`.
foreign import ccall "wrapper"
    mk_WidgetQueryTooltipCallback :: C_WidgetQueryTooltipCallback -> IO (FunPtr C_WidgetQueryTooltipCallback)

wrap_WidgetQueryTooltipCallback :: 
    GObject a => (a -> WidgetQueryTooltipCallback) ->
    C_WidgetQueryTooltipCallback
wrap_WidgetQueryTooltipCallback gi'cb gi'selfPtr x y keyboardMode tooltip _ = do
    let keyboardMode' = (/= 0) keyboardMode
    tooltip' <- (newObject Gtk.Tooltip.Tooltip) tooltip
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  x y keyboardMode' tooltip'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [queryTooltip](#signal:queryTooltip) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #queryTooltip callback
-- @
-- 
-- 
onWidgetQueryTooltip :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetQueryTooltipCallback) -> m SignalHandlerId
onWidgetQueryTooltip obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetQueryTooltipCallback wrapped
    wrapped'' <- mk_WidgetQueryTooltipCallback wrapped'
    connectSignalFunPtr obj "query-tooltip" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [queryTooltip](#signal:queryTooltip) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #queryTooltip callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetQueryTooltip :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetQueryTooltipCallback) -> m SignalHandlerId
afterWidgetQueryTooltip obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetQueryTooltipCallback wrapped
    wrapped'' <- mk_WidgetQueryTooltipCallback wrapped'
    connectSignalFunPtr obj "query-tooltip" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetQueryTooltipSignalInfo
instance SignalInfo WidgetQueryTooltipSignalInfo where
    type HaskellCallbackType WidgetQueryTooltipSignalInfo = WidgetQueryTooltipCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetQueryTooltipCallback cb
        cb'' <- mk_WidgetQueryTooltipCallback cb'
        connectSignalFunPtr obj "query-tooltip" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::query-tooltip"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:queryTooltip"})

#endif

-- signal Widget::realize
-- | Emitted when /@widget@/ is associated with a @GdkSurface@.
-- 
-- This means that 'GI.Gtk.Objects.Widget.widgetRealize' has been called
-- or the widget has been mapped (that is, it is going to be drawn).
type WidgetRealizeCallback =
    IO ()

type C_WidgetRealizeCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetRealizeCallback`.
foreign import ccall "wrapper"
    mk_WidgetRealizeCallback :: C_WidgetRealizeCallback -> IO (FunPtr C_WidgetRealizeCallback)

wrap_WidgetRealizeCallback :: 
    GObject a => (a -> WidgetRealizeCallback) ->
    C_WidgetRealizeCallback
wrap_WidgetRealizeCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [realize](#signal:realize) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #realize callback
-- @
-- 
-- 
onWidgetRealize :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetRealizeCallback) -> m SignalHandlerId
onWidgetRealize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetRealizeCallback wrapped
    wrapped'' <- mk_WidgetRealizeCallback wrapped'
    connectSignalFunPtr obj "realize" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [realize](#signal:realize) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #realize callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetRealize :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetRealizeCallback) -> m SignalHandlerId
afterWidgetRealize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetRealizeCallback wrapped
    wrapped'' <- mk_WidgetRealizeCallback wrapped'
    connectSignalFunPtr obj "realize" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetRealizeSignalInfo
instance SignalInfo WidgetRealizeSignalInfo where
    type HaskellCallbackType WidgetRealizeSignalInfo = WidgetRealizeCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetRealizeCallback cb
        cb'' <- mk_WidgetRealizeCallback cb'
        connectSignalFunPtr obj "realize" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::realize"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:realize"})

#endif

-- signal Widget::show
-- | Emitted when /@widget@/ is shown.
type WidgetShowCallback =
    IO ()

type C_WidgetShowCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetShowCallback`.
foreign import ccall "wrapper"
    mk_WidgetShowCallback :: C_WidgetShowCallback -> IO (FunPtr C_WidgetShowCallback)

wrap_WidgetShowCallback :: 
    GObject a => (a -> WidgetShowCallback) ->
    C_WidgetShowCallback
wrap_WidgetShowCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [show](#signal:show) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #show callback
-- @
-- 
-- 
onWidgetShow :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetShowCallback) -> m SignalHandlerId
onWidgetShow obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetShowCallback wrapped
    wrapped'' <- mk_WidgetShowCallback wrapped'
    connectSignalFunPtr obj "show" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [show](#signal:show) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #show callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetShow :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetShowCallback) -> m SignalHandlerId
afterWidgetShow obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetShowCallback wrapped
    wrapped'' <- mk_WidgetShowCallback wrapped'
    connectSignalFunPtr obj "show" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetShowSignalInfo
instance SignalInfo WidgetShowSignalInfo where
    type HaskellCallbackType WidgetShowSignalInfo = WidgetShowCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetShowCallback cb
        cb'' <- mk_WidgetShowCallback cb'
        connectSignalFunPtr obj "show" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::show"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:show"})

#endif

-- signal Widget::state-flags-changed
-- | Emitted when the widget state changes.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetGetStateFlags'.
type WidgetStateFlagsChangedCallback =
    [Gtk.Flags.StateFlags]
    -- ^ /@flags@/: the previous state flags
    -> IO ()

type C_WidgetStateFlagsChangedCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetStateFlagsChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetStateFlagsChangedCallback :: C_WidgetStateFlagsChangedCallback -> IO (FunPtr C_WidgetStateFlagsChangedCallback)

wrap_WidgetStateFlagsChangedCallback :: 
    GObject a => (a -> WidgetStateFlagsChangedCallback) ->
    C_WidgetStateFlagsChangedCallback
wrap_WidgetStateFlagsChangedCallback gi'cb gi'selfPtr flags _ = do
    let flags' = wordToGFlags flags
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  flags'


-- | Connect a signal handler for the [stateFlagsChanged](#signal:stateFlagsChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #stateFlagsChanged callback
-- @
-- 
-- 
onWidgetStateFlagsChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStateFlagsChangedCallback) -> m SignalHandlerId
onWidgetStateFlagsChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStateFlagsChangedCallback wrapped
    wrapped'' <- mk_WidgetStateFlagsChangedCallback wrapped'
    connectSignalFunPtr obj "state-flags-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [stateFlagsChanged](#signal:stateFlagsChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #stateFlagsChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetStateFlagsChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStateFlagsChangedCallback) -> m SignalHandlerId
afterWidgetStateFlagsChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStateFlagsChangedCallback wrapped
    wrapped'' <- mk_WidgetStateFlagsChangedCallback wrapped'
    connectSignalFunPtr obj "state-flags-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetStateFlagsChangedSignalInfo
instance SignalInfo WidgetStateFlagsChangedSignalInfo where
    type HaskellCallbackType WidgetStateFlagsChangedSignalInfo = WidgetStateFlagsChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetStateFlagsChangedCallback cb
        cb'' <- mk_WidgetStateFlagsChangedCallback cb'
        connectSignalFunPtr obj "state-flags-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::state-flags-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:stateFlagsChanged"})

#endif

-- signal Widget::unmap
-- | Emitted when /@widget@/ is going to be unmapped.
-- 
-- A widget is unmapped when either it or any of its parents up to the
-- toplevel widget have been set as hidden.
-- 
-- As @::unmap@ indicates that a widget will not be shown any longer,
-- it can be used to, for example, stop an animation on the widget.
type WidgetUnmapCallback =
    IO ()

type C_WidgetUnmapCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetUnmapCallback`.
foreign import ccall "wrapper"
    mk_WidgetUnmapCallback :: C_WidgetUnmapCallback -> IO (FunPtr C_WidgetUnmapCallback)

wrap_WidgetUnmapCallback :: 
    GObject a => (a -> WidgetUnmapCallback) ->
    C_WidgetUnmapCallback
wrap_WidgetUnmapCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [unmap](#signal:unmap) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #unmap callback
-- @
-- 
-- 
onWidgetUnmap :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnmapCallback) -> m SignalHandlerId
onWidgetUnmap obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnmapCallback wrapped
    wrapped'' <- mk_WidgetUnmapCallback wrapped'
    connectSignalFunPtr obj "unmap" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [unmap](#signal:unmap) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #unmap callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetUnmap :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnmapCallback) -> m SignalHandlerId
afterWidgetUnmap obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnmapCallback wrapped
    wrapped'' <- mk_WidgetUnmapCallback wrapped'
    connectSignalFunPtr obj "unmap" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetUnmapSignalInfo
instance SignalInfo WidgetUnmapSignalInfo where
    type HaskellCallbackType WidgetUnmapSignalInfo = WidgetUnmapCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetUnmapCallback cb
        cb'' <- mk_WidgetUnmapCallback cb'
        connectSignalFunPtr obj "unmap" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::unmap"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:unmap"})

#endif

-- signal Widget::unrealize
-- | Emitted when the @GdkSurface@ associated with /@widget@/ is destroyed.
-- 
-- This means that 'GI.Gtk.Objects.Widget.widgetUnrealize' has been called
-- or the widget has been unmapped (that is, it is going to be hidden).
type WidgetUnrealizeCallback =
    IO ()

type C_WidgetUnrealizeCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetUnrealizeCallback`.
foreign import ccall "wrapper"
    mk_WidgetUnrealizeCallback :: C_WidgetUnrealizeCallback -> IO (FunPtr C_WidgetUnrealizeCallback)

wrap_WidgetUnrealizeCallback :: 
    GObject a => (a -> WidgetUnrealizeCallback) ->
    C_WidgetUnrealizeCallback
wrap_WidgetUnrealizeCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [unrealize](#signal:unrealize) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #unrealize callback
-- @
-- 
-- 
onWidgetUnrealize :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnrealizeCallback) -> m SignalHandlerId
onWidgetUnrealize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnrealizeCallback wrapped
    wrapped'' <- mk_WidgetUnrealizeCallback wrapped'
    connectSignalFunPtr obj "unrealize" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [unrealize](#signal:unrealize) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #unrealize callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetUnrealize :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnrealizeCallback) -> m SignalHandlerId
afterWidgetUnrealize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnrealizeCallback wrapped
    wrapped'' <- mk_WidgetUnrealizeCallback wrapped'
    connectSignalFunPtr obj "unrealize" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetUnrealizeSignalInfo
instance SignalInfo WidgetUnrealizeSignalInfo where
    type HaskellCallbackType WidgetUnrealizeSignalInfo = WidgetUnrealizeCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetUnrealizeCallback cb
        cb'' <- mk_WidgetUnrealizeCallback cb'
        connectSignalFunPtr obj "unrealize" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::unrealize"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:signal:unrealize"})

#endif

-- VVV Prop "can-focus"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@can-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #canFocus
-- @
getWidgetCanFocus :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetCanFocus obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "can-focus"

-- | Set the value of the “@can-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #canFocus 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetCanFocus :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetCanFocus obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "can-focus" val

-- | Construct a `GValueConstruct` with valid value for the “@can-focus@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetCanFocus :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetCanFocus val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "can-focus" val

#if defined(ENABLE_OVERLOADING)
data WidgetCanFocusPropertyInfo
instance AttrInfo WidgetCanFocusPropertyInfo where
    type AttrAllowedOps WidgetCanFocusPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetCanFocusPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetCanFocusPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetCanFocusPropertyInfo = (~) Bool
    type AttrTransferType WidgetCanFocusPropertyInfo = Bool
    type AttrGetType WidgetCanFocusPropertyInfo = Bool
    type AttrLabel WidgetCanFocusPropertyInfo = "can-focus"
    type AttrOrigin WidgetCanFocusPropertyInfo = Widget
    attrGet = getWidgetCanFocus
    attrSet = setWidgetCanFocus
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetCanFocus
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.canFocus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:canFocus"
        })
#endif

-- VVV Prop "can-target"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@can-target@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #canTarget
-- @
getWidgetCanTarget :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetCanTarget obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "can-target"

-- | Set the value of the “@can-target@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #canTarget 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetCanTarget :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetCanTarget obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "can-target" val

-- | Construct a `GValueConstruct` with valid value for the “@can-target@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetCanTarget :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetCanTarget val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "can-target" val

#if defined(ENABLE_OVERLOADING)
data WidgetCanTargetPropertyInfo
instance AttrInfo WidgetCanTargetPropertyInfo where
    type AttrAllowedOps WidgetCanTargetPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetCanTargetPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetCanTargetPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetCanTargetPropertyInfo = (~) Bool
    type AttrTransferType WidgetCanTargetPropertyInfo = Bool
    type AttrGetType WidgetCanTargetPropertyInfo = Bool
    type AttrLabel WidgetCanTargetPropertyInfo = "can-target"
    type AttrOrigin WidgetCanTargetPropertyInfo = Widget
    attrGet = getWidgetCanTarget
    attrSet = setWidgetCanTarget
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetCanTarget
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.canTarget"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:canTarget"
        })
#endif

-- VVV Prop "css-classes"
   -- Type: TCArray True (-1) (-1) (TBasicType TUTF8)
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Just False)

-- | Get the value of the “@css-classes@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #cssClasses
-- @
getWidgetCssClasses :: (MonadIO m, IsWidget o) => o -> m (Maybe [T.Text])
getWidgetCssClasses obj = MIO.liftIO $ B.Properties.getObjectPropertyStringArray obj "css-classes"

-- | Set the value of the “@css-classes@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #cssClasses 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetCssClasses :: (MonadIO m, IsWidget o) => o -> [T.Text] -> m ()
setWidgetCssClasses obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyStringArray obj "css-classes" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@css-classes@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetCssClasses :: (IsWidget o, MIO.MonadIO m) => [T.Text] -> m (GValueConstruct o)
constructWidgetCssClasses val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyStringArray "css-classes" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data WidgetCssClassesPropertyInfo
instance AttrInfo WidgetCssClassesPropertyInfo where
    type AttrAllowedOps WidgetCssClassesPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetCssClassesPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetCssClassesPropertyInfo = (~) [T.Text]
    type AttrTransferTypeConstraint WidgetCssClassesPropertyInfo = (~) [T.Text]
    type AttrTransferType WidgetCssClassesPropertyInfo = [T.Text]
    type AttrGetType WidgetCssClassesPropertyInfo = (Maybe [T.Text])
    type AttrLabel WidgetCssClassesPropertyInfo = "css-classes"
    type AttrOrigin WidgetCssClassesPropertyInfo = Widget
    attrGet = getWidgetCssClasses
    attrSet = setWidgetCssClasses
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetCssClasses
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.cssClasses"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:cssClasses"
        })
#endif

-- VVV Prop "css-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@css-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #cssName
-- @
getWidgetCssName :: (MonadIO m, IsWidget o) => o -> m T.Text
getWidgetCssName obj = MIO.liftIO $ checkUnexpectedNothing "getWidgetCssName" $ B.Properties.getObjectPropertyString obj "css-name"

-- | Construct a `GValueConstruct` with valid value for the “@css-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetCssName :: (IsWidget o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructWidgetCssName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "css-name" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data WidgetCssNamePropertyInfo
instance AttrInfo WidgetCssNamePropertyInfo where
    type AttrAllowedOps WidgetCssNamePropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetCssNamePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetCssNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint WidgetCssNamePropertyInfo = (~) T.Text
    type AttrTransferType WidgetCssNamePropertyInfo = T.Text
    type AttrGetType WidgetCssNamePropertyInfo = T.Text
    type AttrLabel WidgetCssNamePropertyInfo = "css-name"
    type AttrOrigin WidgetCssNamePropertyInfo = Widget
    attrGet = getWidgetCssName
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetCssName
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.cssName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:cssName"
        })
#endif

-- VVV Prop "cursor"
   -- Type: TInterface (Name {namespace = "Gdk", name = "Cursor"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@cursor@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #cursor
-- @
getWidgetCursor :: (MonadIO m, IsWidget o) => o -> m (Maybe Gdk.Cursor.Cursor)
getWidgetCursor obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "cursor" Gdk.Cursor.Cursor

-- | Set the value of the “@cursor@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #cursor 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetCursor :: (MonadIO m, IsWidget o, Gdk.Cursor.IsCursor a) => o -> a -> m ()
setWidgetCursor obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "cursor" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@cursor@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetCursor :: (IsWidget o, MIO.MonadIO m, Gdk.Cursor.IsCursor a) => a -> m (GValueConstruct o)
constructWidgetCursor val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "cursor" (P.Just val)

-- | Set the value of the “@cursor@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #cursor
-- @
clearWidgetCursor :: (MonadIO m, IsWidget o) => o -> m ()
clearWidgetCursor obj = liftIO $ B.Properties.setObjectPropertyObject obj "cursor" (Nothing :: Maybe Gdk.Cursor.Cursor)

#if defined(ENABLE_OVERLOADING)
data WidgetCursorPropertyInfo
instance AttrInfo WidgetCursorPropertyInfo where
    type AttrAllowedOps WidgetCursorPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetCursorPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetCursorPropertyInfo = Gdk.Cursor.IsCursor
    type AttrTransferTypeConstraint WidgetCursorPropertyInfo = Gdk.Cursor.IsCursor
    type AttrTransferType WidgetCursorPropertyInfo = Gdk.Cursor.Cursor
    type AttrGetType WidgetCursorPropertyInfo = (Maybe Gdk.Cursor.Cursor)
    type AttrLabel WidgetCursorPropertyInfo = "cursor"
    type AttrOrigin WidgetCursorPropertyInfo = Widget
    attrGet = getWidgetCursor
    attrSet = setWidgetCursor
    attrTransfer _ v = do
        unsafeCastTo Gdk.Cursor.Cursor v
    attrConstruct = constructWidgetCursor
    attrClear = clearWidgetCursor
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.cursor"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:cursor"
        })
#endif

-- VVV Prop "focus-on-click"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@focus-on-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #focusOnClick
-- @
getWidgetFocusOnClick :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetFocusOnClick obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "focus-on-click"

-- | Set the value of the “@focus-on-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #focusOnClick 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetFocusOnClick :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetFocusOnClick obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "focus-on-click" val

-- | Construct a `GValueConstruct` with valid value for the “@focus-on-click@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetFocusOnClick :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetFocusOnClick val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "focus-on-click" val

#if defined(ENABLE_OVERLOADING)
data WidgetFocusOnClickPropertyInfo
instance AttrInfo WidgetFocusOnClickPropertyInfo where
    type AttrAllowedOps WidgetFocusOnClickPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetFocusOnClickPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetFocusOnClickPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetFocusOnClickPropertyInfo = (~) Bool
    type AttrTransferType WidgetFocusOnClickPropertyInfo = Bool
    type AttrGetType WidgetFocusOnClickPropertyInfo = Bool
    type AttrLabel WidgetFocusOnClickPropertyInfo = "focus-on-click"
    type AttrOrigin WidgetFocusOnClickPropertyInfo = Widget
    attrGet = getWidgetFocusOnClick
    attrSet = setWidgetFocusOnClick
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetFocusOnClick
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.focusOnClick"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:focusOnClick"
        })
#endif

-- VVV Prop "focusable"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@focusable@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #focusable
-- @
getWidgetFocusable :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetFocusable obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "focusable"

-- | Set the value of the “@focusable@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #focusable 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetFocusable :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetFocusable obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "focusable" val

-- | Construct a `GValueConstruct` with valid value for the “@focusable@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetFocusable :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetFocusable val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "focusable" val

#if defined(ENABLE_OVERLOADING)
data WidgetFocusablePropertyInfo
instance AttrInfo WidgetFocusablePropertyInfo where
    type AttrAllowedOps WidgetFocusablePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetFocusablePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetFocusablePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetFocusablePropertyInfo = (~) Bool
    type AttrTransferType WidgetFocusablePropertyInfo = Bool
    type AttrGetType WidgetFocusablePropertyInfo = Bool
    type AttrLabel WidgetFocusablePropertyInfo = "focusable"
    type AttrOrigin WidgetFocusablePropertyInfo = Widget
    attrGet = getWidgetFocusable
    attrSet = setWidgetFocusable
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetFocusable
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.focusable"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:focusable"
        })
#endif

-- VVV Prop "halign"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Align"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@halign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #halign
-- @
getWidgetHalign :: (MonadIO m, IsWidget o) => o -> m Gtk.Enums.Align
getWidgetHalign obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "halign"

-- | Set the value of the “@halign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #halign 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHalign :: (MonadIO m, IsWidget o) => o -> Gtk.Enums.Align -> m ()
setWidgetHalign obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "halign" val

-- | Construct a `GValueConstruct` with valid value for the “@halign@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHalign :: (IsWidget o, MIO.MonadIO m) => Gtk.Enums.Align -> m (GValueConstruct o)
constructWidgetHalign val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "halign" val

#if defined(ENABLE_OVERLOADING)
data WidgetHalignPropertyInfo
instance AttrInfo WidgetHalignPropertyInfo where
    type AttrAllowedOps WidgetHalignPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHalignPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHalignPropertyInfo = (~) Gtk.Enums.Align
    type AttrTransferTypeConstraint WidgetHalignPropertyInfo = (~) Gtk.Enums.Align
    type AttrTransferType WidgetHalignPropertyInfo = Gtk.Enums.Align
    type AttrGetType WidgetHalignPropertyInfo = Gtk.Enums.Align
    type AttrLabel WidgetHalignPropertyInfo = "halign"
    type AttrOrigin WidgetHalignPropertyInfo = Widget
    attrGet = getWidgetHalign
    attrSet = setWidgetHalign
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHalign
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.halign"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:halign"
        })
#endif

-- VVV Prop "has-default"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@has-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hasDefault
-- @
getWidgetHasDefault :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHasDefault obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-default"

#if defined(ENABLE_OVERLOADING)
data WidgetHasDefaultPropertyInfo
instance AttrInfo WidgetHasDefaultPropertyInfo where
    type AttrAllowedOps WidgetHasDefaultPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint WidgetHasDefaultPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHasDefaultPropertyInfo = (~) ()
    type AttrTransferTypeConstraint WidgetHasDefaultPropertyInfo = (~) ()
    type AttrTransferType WidgetHasDefaultPropertyInfo = ()
    type AttrGetType WidgetHasDefaultPropertyInfo = Bool
    type AttrLabel WidgetHasDefaultPropertyInfo = "has-default"
    type AttrOrigin WidgetHasDefaultPropertyInfo = Widget
    attrGet = getWidgetHasDefault
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hasDefault"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:hasDefault"
        })
#endif

-- VVV Prop "has-focus"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@has-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hasFocus
-- @
getWidgetHasFocus :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHasFocus obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-focus"

#if defined(ENABLE_OVERLOADING)
data WidgetHasFocusPropertyInfo
instance AttrInfo WidgetHasFocusPropertyInfo where
    type AttrAllowedOps WidgetHasFocusPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint WidgetHasFocusPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHasFocusPropertyInfo = (~) ()
    type AttrTransferTypeConstraint WidgetHasFocusPropertyInfo = (~) ()
    type AttrTransferType WidgetHasFocusPropertyInfo = ()
    type AttrGetType WidgetHasFocusPropertyInfo = Bool
    type AttrLabel WidgetHasFocusPropertyInfo = "has-focus"
    type AttrOrigin WidgetHasFocusPropertyInfo = Widget
    attrGet = getWidgetHasFocus
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hasFocus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:hasFocus"
        })
#endif

-- VVV Prop "has-tooltip"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@has-tooltip@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hasTooltip
-- @
getWidgetHasTooltip :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHasTooltip obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-tooltip"

-- | Set the value of the “@has-tooltip@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #hasTooltip 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHasTooltip :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetHasTooltip obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "has-tooltip" val

-- | Construct a `GValueConstruct` with valid value for the “@has-tooltip@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHasTooltip :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetHasTooltip val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "has-tooltip" val

#if defined(ENABLE_OVERLOADING)
data WidgetHasTooltipPropertyInfo
instance AttrInfo WidgetHasTooltipPropertyInfo where
    type AttrAllowedOps WidgetHasTooltipPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHasTooltipPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHasTooltipPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetHasTooltipPropertyInfo = (~) Bool
    type AttrTransferType WidgetHasTooltipPropertyInfo = Bool
    type AttrGetType WidgetHasTooltipPropertyInfo = Bool
    type AttrLabel WidgetHasTooltipPropertyInfo = "has-tooltip"
    type AttrOrigin WidgetHasTooltipPropertyInfo = Widget
    attrGet = getWidgetHasTooltip
    attrSet = setWidgetHasTooltip
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHasTooltip
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hasTooltip"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:hasTooltip"
        })
#endif

-- VVV Prop "height-request"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@height-request@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #heightRequest
-- @
getWidgetHeightRequest :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetHeightRequest obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "height-request"

-- | Set the value of the “@height-request@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #heightRequest 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHeightRequest :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetHeightRequest obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "height-request" val

-- | Construct a `GValueConstruct` with valid value for the “@height-request@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHeightRequest :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetHeightRequest val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "height-request" val

#if defined(ENABLE_OVERLOADING)
data WidgetHeightRequestPropertyInfo
instance AttrInfo WidgetHeightRequestPropertyInfo where
    type AttrAllowedOps WidgetHeightRequestPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHeightRequestPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHeightRequestPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetHeightRequestPropertyInfo = (~) Int32
    type AttrTransferType WidgetHeightRequestPropertyInfo = Int32
    type AttrGetType WidgetHeightRequestPropertyInfo = Int32
    type AttrLabel WidgetHeightRequestPropertyInfo = "height-request"
    type AttrOrigin WidgetHeightRequestPropertyInfo = Widget
    attrGet = getWidgetHeightRequest
    attrSet = setWidgetHeightRequest
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHeightRequest
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.heightRequest"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:heightRequest"
        })
#endif

-- VVV Prop "hexpand"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@hexpand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hexpand
-- @
getWidgetHexpand :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHexpand obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "hexpand"

-- | Set the value of the “@hexpand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #hexpand 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHexpand :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetHexpand obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "hexpand" val

-- | Construct a `GValueConstruct` with valid value for the “@hexpand@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHexpand :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetHexpand val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "hexpand" val

#if defined(ENABLE_OVERLOADING)
data WidgetHexpandPropertyInfo
instance AttrInfo WidgetHexpandPropertyInfo where
    type AttrAllowedOps WidgetHexpandPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHexpandPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHexpandPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetHexpandPropertyInfo = (~) Bool
    type AttrTransferType WidgetHexpandPropertyInfo = Bool
    type AttrGetType WidgetHexpandPropertyInfo = Bool
    type AttrLabel WidgetHexpandPropertyInfo = "hexpand"
    type AttrOrigin WidgetHexpandPropertyInfo = Widget
    attrGet = getWidgetHexpand
    attrSet = setWidgetHexpand
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHexpand
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hexpand"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:hexpand"
        })
#endif

-- VVV Prop "hexpand-set"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@hexpand-set@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hexpandSet
-- @
getWidgetHexpandSet :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHexpandSet obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "hexpand-set"

-- | Set the value of the “@hexpand-set@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #hexpandSet 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHexpandSet :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetHexpandSet obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "hexpand-set" val

-- | Construct a `GValueConstruct` with valid value for the “@hexpand-set@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHexpandSet :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetHexpandSet val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "hexpand-set" val

#if defined(ENABLE_OVERLOADING)
data WidgetHexpandSetPropertyInfo
instance AttrInfo WidgetHexpandSetPropertyInfo where
    type AttrAllowedOps WidgetHexpandSetPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHexpandSetPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHexpandSetPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetHexpandSetPropertyInfo = (~) Bool
    type AttrTransferType WidgetHexpandSetPropertyInfo = Bool
    type AttrGetType WidgetHexpandSetPropertyInfo = Bool
    type AttrLabel WidgetHexpandSetPropertyInfo = "hexpand-set"
    type AttrOrigin WidgetHexpandSetPropertyInfo = Widget
    attrGet = getWidgetHexpandSet
    attrSet = setWidgetHexpandSet
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHexpandSet
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hexpandSet"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:hexpandSet"
        })
#endif

-- VVV Prop "layout-manager"
   -- Type: TInterface (Name {namespace = "Gtk", name = "LayoutManager"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@layout-manager@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #layoutManager
-- @
getWidgetLayoutManager :: (MonadIO m, IsWidget o) => o -> m (Maybe Gtk.LayoutManager.LayoutManager)
getWidgetLayoutManager obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "layout-manager" Gtk.LayoutManager.LayoutManager

-- | Set the value of the “@layout-manager@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #layoutManager 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetLayoutManager :: (MonadIO m, IsWidget o, Gtk.LayoutManager.IsLayoutManager a) => o -> a -> m ()
setWidgetLayoutManager obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "layout-manager" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@layout-manager@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetLayoutManager :: (IsWidget o, MIO.MonadIO m, Gtk.LayoutManager.IsLayoutManager a) => a -> m (GValueConstruct o)
constructWidgetLayoutManager val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "layout-manager" (P.Just val)

-- | Set the value of the “@layout-manager@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #layoutManager
-- @
clearWidgetLayoutManager :: (MonadIO m, IsWidget o) => o -> m ()
clearWidgetLayoutManager obj = liftIO $ B.Properties.setObjectPropertyObject obj "layout-manager" (Nothing :: Maybe Gtk.LayoutManager.LayoutManager)

#if defined(ENABLE_OVERLOADING)
data WidgetLayoutManagerPropertyInfo
instance AttrInfo WidgetLayoutManagerPropertyInfo where
    type AttrAllowedOps WidgetLayoutManagerPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetLayoutManagerPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetLayoutManagerPropertyInfo = Gtk.LayoutManager.IsLayoutManager
    type AttrTransferTypeConstraint WidgetLayoutManagerPropertyInfo = Gtk.LayoutManager.IsLayoutManager
    type AttrTransferType WidgetLayoutManagerPropertyInfo = Gtk.LayoutManager.LayoutManager
    type AttrGetType WidgetLayoutManagerPropertyInfo = (Maybe Gtk.LayoutManager.LayoutManager)
    type AttrLabel WidgetLayoutManagerPropertyInfo = "layout-manager"
    type AttrOrigin WidgetLayoutManagerPropertyInfo = Widget
    attrGet = getWidgetLayoutManager
    attrSet = setWidgetLayoutManager
    attrTransfer _ v = do
        unsafeCastTo Gtk.LayoutManager.LayoutManager v
    attrConstruct = constructWidgetLayoutManager
    attrClear = clearWidgetLayoutManager
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.layoutManager"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:layoutManager"
        })
#endif

-- VVV Prop "limit-events"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@limit-events@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #limitEvents
-- @
getWidgetLimitEvents :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetLimitEvents obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "limit-events"

-- | Set the value of the “@limit-events@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #limitEvents 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetLimitEvents :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetLimitEvents obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "limit-events" val

-- | Construct a `GValueConstruct` with valid value for the “@limit-events@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetLimitEvents :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetLimitEvents val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "limit-events" val

#if defined(ENABLE_OVERLOADING)
data WidgetLimitEventsPropertyInfo
instance AttrInfo WidgetLimitEventsPropertyInfo where
    type AttrAllowedOps WidgetLimitEventsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetLimitEventsPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetLimitEventsPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetLimitEventsPropertyInfo = (~) Bool
    type AttrTransferType WidgetLimitEventsPropertyInfo = Bool
    type AttrGetType WidgetLimitEventsPropertyInfo = Bool
    type AttrLabel WidgetLimitEventsPropertyInfo = "limit-events"
    type AttrOrigin WidgetLimitEventsPropertyInfo = Widget
    attrGet = getWidgetLimitEvents
    attrSet = setWidgetLimitEvents
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetLimitEvents
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.limitEvents"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:limitEvents"
        })
#endif

-- VVV Prop "margin-bottom"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-bottom@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginBottom
-- @
getWidgetMarginBottom :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginBottom obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-bottom"

-- | Set the value of the “@margin-bottom@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginBottom 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginBottom :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginBottom obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-bottom" val

-- | Construct a `GValueConstruct` with valid value for the “@margin-bottom@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginBottom :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginBottom val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-bottom" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginBottomPropertyInfo
instance AttrInfo WidgetMarginBottomPropertyInfo where
    type AttrAllowedOps WidgetMarginBottomPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginBottomPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginBottomPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginBottomPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginBottomPropertyInfo = Int32
    type AttrGetType WidgetMarginBottomPropertyInfo = Int32
    type AttrLabel WidgetMarginBottomPropertyInfo = "margin-bottom"
    type AttrOrigin WidgetMarginBottomPropertyInfo = Widget
    attrGet = getWidgetMarginBottom
    attrSet = setWidgetMarginBottom
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginBottom
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginBottom"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:marginBottom"
        })
#endif

-- VVV Prop "margin-end"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-end@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginEnd
-- @
getWidgetMarginEnd :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginEnd obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-end"

-- | Set the value of the “@margin-end@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginEnd 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginEnd :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginEnd obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-end" val

-- | Construct a `GValueConstruct` with valid value for the “@margin-end@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginEnd :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginEnd val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-end" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginEndPropertyInfo
instance AttrInfo WidgetMarginEndPropertyInfo where
    type AttrAllowedOps WidgetMarginEndPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginEndPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginEndPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginEndPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginEndPropertyInfo = Int32
    type AttrGetType WidgetMarginEndPropertyInfo = Int32
    type AttrLabel WidgetMarginEndPropertyInfo = "margin-end"
    type AttrOrigin WidgetMarginEndPropertyInfo = Widget
    attrGet = getWidgetMarginEnd
    attrSet = setWidgetMarginEnd
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginEnd
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginEnd"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:marginEnd"
        })
#endif

-- VVV Prop "margin-start"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-start@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginStart
-- @
getWidgetMarginStart :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginStart obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-start"

-- | Set the value of the “@margin-start@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginStart 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginStart :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginStart obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-start" val

-- | Construct a `GValueConstruct` with valid value for the “@margin-start@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginStart :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginStart val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-start" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginStartPropertyInfo
instance AttrInfo WidgetMarginStartPropertyInfo where
    type AttrAllowedOps WidgetMarginStartPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginStartPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginStartPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginStartPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginStartPropertyInfo = Int32
    type AttrGetType WidgetMarginStartPropertyInfo = Int32
    type AttrLabel WidgetMarginStartPropertyInfo = "margin-start"
    type AttrOrigin WidgetMarginStartPropertyInfo = Widget
    attrGet = getWidgetMarginStart
    attrSet = setWidgetMarginStart
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginStart
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginStart"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:marginStart"
        })
#endif

-- VVV Prop "margin-top"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-top@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginTop
-- @
getWidgetMarginTop :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginTop obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-top"

-- | Set the value of the “@margin-top@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginTop 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginTop :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginTop obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-top" val

-- | Construct a `GValueConstruct` with valid value for the “@margin-top@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginTop :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginTop val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-top" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginTopPropertyInfo
instance AttrInfo WidgetMarginTopPropertyInfo where
    type AttrAllowedOps WidgetMarginTopPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginTopPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginTopPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginTopPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginTopPropertyInfo = Int32
    type AttrGetType WidgetMarginTopPropertyInfo = Int32
    type AttrLabel WidgetMarginTopPropertyInfo = "margin-top"
    type AttrOrigin WidgetMarginTopPropertyInfo = Widget
    attrGet = getWidgetMarginTop
    attrSet = setWidgetMarginTop
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginTop
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginTop"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:marginTop"
        })
#endif

-- VVV Prop "name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #name
-- @
getWidgetName :: (MonadIO m, IsWidget o) => o -> m T.Text
getWidgetName obj = MIO.liftIO $ checkUnexpectedNothing "getWidgetName" $ B.Properties.getObjectPropertyString obj "name"

-- | Set the value of the “@name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #name 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetName :: (MonadIO m, IsWidget o) => o -> T.Text -> m ()
setWidgetName obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "name" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetName :: (IsWidget o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructWidgetName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "name" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data WidgetNamePropertyInfo
instance AttrInfo WidgetNamePropertyInfo where
    type AttrAllowedOps WidgetNamePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetNamePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint WidgetNamePropertyInfo = (~) T.Text
    type AttrTransferType WidgetNamePropertyInfo = T.Text
    type AttrGetType WidgetNamePropertyInfo = T.Text
    type AttrLabel WidgetNamePropertyInfo = "name"
    type AttrOrigin WidgetNamePropertyInfo = Widget
    attrGet = getWidgetName
    attrSet = setWidgetName
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetName
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.name"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:name"
        })
#endif

-- VVV Prop "opacity"
   -- Type: TBasicType TDouble
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@opacity@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #opacity
-- @
getWidgetOpacity :: (MonadIO m, IsWidget o) => o -> m Double
getWidgetOpacity obj = MIO.liftIO $ B.Properties.getObjectPropertyDouble obj "opacity"

-- | Set the value of the “@opacity@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #opacity 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetOpacity :: (MonadIO m, IsWidget o) => o -> Double -> m ()
setWidgetOpacity obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyDouble obj "opacity" val

-- | Construct a `GValueConstruct` with valid value for the “@opacity@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetOpacity :: (IsWidget o, MIO.MonadIO m) => Double -> m (GValueConstruct o)
constructWidgetOpacity val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyDouble "opacity" val

#if defined(ENABLE_OVERLOADING)
data WidgetOpacityPropertyInfo
instance AttrInfo WidgetOpacityPropertyInfo where
    type AttrAllowedOps WidgetOpacityPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetOpacityPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetOpacityPropertyInfo = (~) Double
    type AttrTransferTypeConstraint WidgetOpacityPropertyInfo = (~) Double
    type AttrTransferType WidgetOpacityPropertyInfo = Double
    type AttrGetType WidgetOpacityPropertyInfo = Double
    type AttrLabel WidgetOpacityPropertyInfo = "opacity"
    type AttrOrigin WidgetOpacityPropertyInfo = Widget
    attrGet = getWidgetOpacity
    attrSet = setWidgetOpacity
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetOpacity
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.opacity"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:opacity"
        })
#endif

-- VVV Prop "overflow"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Overflow"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@overflow@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #overflow
-- @
getWidgetOverflow :: (MonadIO m, IsWidget o) => o -> m Gtk.Enums.Overflow
getWidgetOverflow obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "overflow"

-- | Set the value of the “@overflow@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #overflow 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetOverflow :: (MonadIO m, IsWidget o) => o -> Gtk.Enums.Overflow -> m ()
setWidgetOverflow obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "overflow" val

-- | Construct a `GValueConstruct` with valid value for the “@overflow@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetOverflow :: (IsWidget o, MIO.MonadIO m) => Gtk.Enums.Overflow -> m (GValueConstruct o)
constructWidgetOverflow val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "overflow" val

#if defined(ENABLE_OVERLOADING)
data WidgetOverflowPropertyInfo
instance AttrInfo WidgetOverflowPropertyInfo where
    type AttrAllowedOps WidgetOverflowPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetOverflowPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetOverflowPropertyInfo = (~) Gtk.Enums.Overflow
    type AttrTransferTypeConstraint WidgetOverflowPropertyInfo = (~) Gtk.Enums.Overflow
    type AttrTransferType WidgetOverflowPropertyInfo = Gtk.Enums.Overflow
    type AttrGetType WidgetOverflowPropertyInfo = Gtk.Enums.Overflow
    type AttrLabel WidgetOverflowPropertyInfo = "overflow"
    type AttrOrigin WidgetOverflowPropertyInfo = Widget
    attrGet = getWidgetOverflow
    attrSet = setWidgetOverflow
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetOverflow
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.overflow"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:overflow"
        })
#endif

-- VVV Prop "parent"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Widget"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Just False)

-- | Get the value of the “@parent@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #parent
-- @
getWidgetParent :: (MonadIO m, IsWidget o) => o -> m (Maybe Widget)
getWidgetParent obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "parent" Widget

#if defined(ENABLE_OVERLOADING)
data WidgetParentPropertyInfo
instance AttrInfo WidgetParentPropertyInfo where
    type AttrAllowedOps WidgetParentPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint WidgetParentPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetParentPropertyInfo = (~) ()
    type AttrTransferTypeConstraint WidgetParentPropertyInfo = (~) ()
    type AttrTransferType WidgetParentPropertyInfo = ()
    type AttrGetType WidgetParentPropertyInfo = (Maybe Widget)
    type AttrLabel WidgetParentPropertyInfo = "parent"
    type AttrOrigin WidgetParentPropertyInfo = Widget
    attrGet = getWidgetParent
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.parent"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:parent"
        })
#endif

-- VVV Prop "receives-default"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@receives-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #receivesDefault
-- @
getWidgetReceivesDefault :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetReceivesDefault obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "receives-default"

-- | Set the value of the “@receives-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #receivesDefault 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetReceivesDefault :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetReceivesDefault obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "receives-default" val

-- | Construct a `GValueConstruct` with valid value for the “@receives-default@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetReceivesDefault :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetReceivesDefault val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "receives-default" val

#if defined(ENABLE_OVERLOADING)
data WidgetReceivesDefaultPropertyInfo
instance AttrInfo WidgetReceivesDefaultPropertyInfo where
    type AttrAllowedOps WidgetReceivesDefaultPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetReceivesDefaultPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetReceivesDefaultPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetReceivesDefaultPropertyInfo = (~) Bool
    type AttrTransferType WidgetReceivesDefaultPropertyInfo = Bool
    type AttrGetType WidgetReceivesDefaultPropertyInfo = Bool
    type AttrLabel WidgetReceivesDefaultPropertyInfo = "receives-default"
    type AttrOrigin WidgetReceivesDefaultPropertyInfo = Widget
    attrGet = getWidgetReceivesDefault
    attrSet = setWidgetReceivesDefault
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetReceivesDefault
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.receivesDefault"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:receivesDefault"
        })
#endif

-- VVV Prop "root"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Root"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@root@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #root
-- @
getWidgetRoot :: (MonadIO m, IsWidget o) => o -> m (Maybe Gtk.Root.Root)
getWidgetRoot obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "root" Gtk.Root.Root

#if defined(ENABLE_OVERLOADING)
data WidgetRootPropertyInfo
instance AttrInfo WidgetRootPropertyInfo where
    type AttrAllowedOps WidgetRootPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetRootPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetRootPropertyInfo = (~) ()
    type AttrTransferTypeConstraint WidgetRootPropertyInfo = (~) ()
    type AttrTransferType WidgetRootPropertyInfo = ()
    type AttrGetType WidgetRootPropertyInfo = (Maybe Gtk.Root.Root)
    type AttrLabel WidgetRootPropertyInfo = "root"
    type AttrOrigin WidgetRootPropertyInfo = Widget
    attrGet = getWidgetRoot
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.root"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:root"
        })
#endif

-- VVV Prop "scale-factor"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@scale-factor@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #scaleFactor
-- @
getWidgetScaleFactor :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetScaleFactor obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "scale-factor"

#if defined(ENABLE_OVERLOADING)
data WidgetScaleFactorPropertyInfo
instance AttrInfo WidgetScaleFactorPropertyInfo where
    type AttrAllowedOps WidgetScaleFactorPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint WidgetScaleFactorPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetScaleFactorPropertyInfo = (~) ()
    type AttrTransferTypeConstraint WidgetScaleFactorPropertyInfo = (~) ()
    type AttrTransferType WidgetScaleFactorPropertyInfo = ()
    type AttrGetType WidgetScaleFactorPropertyInfo = Int32
    type AttrLabel WidgetScaleFactorPropertyInfo = "scale-factor"
    type AttrOrigin WidgetScaleFactorPropertyInfo = Widget
    attrGet = getWidgetScaleFactor
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.scaleFactor"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:scaleFactor"
        })
#endif

-- VVV Prop "sensitive"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@sensitive@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #sensitive
-- @
getWidgetSensitive :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetSensitive obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "sensitive"

-- | Set the value of the “@sensitive@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #sensitive 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetSensitive :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetSensitive obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "sensitive" val

-- | Construct a `GValueConstruct` with valid value for the “@sensitive@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetSensitive :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetSensitive val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "sensitive" val

#if defined(ENABLE_OVERLOADING)
data WidgetSensitivePropertyInfo
instance AttrInfo WidgetSensitivePropertyInfo where
    type AttrAllowedOps WidgetSensitivePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetSensitivePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetSensitivePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetSensitivePropertyInfo = (~) Bool
    type AttrTransferType WidgetSensitivePropertyInfo = Bool
    type AttrGetType WidgetSensitivePropertyInfo = Bool
    type AttrLabel WidgetSensitivePropertyInfo = "sensitive"
    type AttrOrigin WidgetSensitivePropertyInfo = Widget
    attrGet = getWidgetSensitive
    attrSet = setWidgetSensitive
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetSensitive
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.sensitive"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:sensitive"
        })
#endif

-- VVV Prop "tooltip-markup"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@tooltip-markup@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #tooltipMarkup
-- @
getWidgetTooltipMarkup :: (MonadIO m, IsWidget o) => o -> m (Maybe T.Text)
getWidgetTooltipMarkup obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "tooltip-markup"

-- | Set the value of the “@tooltip-markup@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #tooltipMarkup 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetTooltipMarkup :: (MonadIO m, IsWidget o) => o -> T.Text -> m ()
setWidgetTooltipMarkup obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "tooltip-markup" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@tooltip-markup@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetTooltipMarkup :: (IsWidget o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructWidgetTooltipMarkup val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "tooltip-markup" (P.Just val)

-- | Set the value of the “@tooltip-markup@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #tooltipMarkup
-- @
clearWidgetTooltipMarkup :: (MonadIO m, IsWidget o) => o -> m ()
clearWidgetTooltipMarkup obj = liftIO $ B.Properties.setObjectPropertyString obj "tooltip-markup" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data WidgetTooltipMarkupPropertyInfo
instance AttrInfo WidgetTooltipMarkupPropertyInfo where
    type AttrAllowedOps WidgetTooltipMarkupPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetTooltipMarkupPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetTooltipMarkupPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint WidgetTooltipMarkupPropertyInfo = (~) T.Text
    type AttrTransferType WidgetTooltipMarkupPropertyInfo = T.Text
    type AttrGetType WidgetTooltipMarkupPropertyInfo = (Maybe T.Text)
    type AttrLabel WidgetTooltipMarkupPropertyInfo = "tooltip-markup"
    type AttrOrigin WidgetTooltipMarkupPropertyInfo = Widget
    attrGet = getWidgetTooltipMarkup
    attrSet = setWidgetTooltipMarkup
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetTooltipMarkup
    attrClear = clearWidgetTooltipMarkup
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.tooltipMarkup"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:tooltipMarkup"
        })
#endif

-- VVV Prop "tooltip-text"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@tooltip-text@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #tooltipText
-- @
getWidgetTooltipText :: (MonadIO m, IsWidget o) => o -> m (Maybe T.Text)
getWidgetTooltipText obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "tooltip-text"

-- | Set the value of the “@tooltip-text@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #tooltipText 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetTooltipText :: (MonadIO m, IsWidget o) => o -> T.Text -> m ()
setWidgetTooltipText obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "tooltip-text" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@tooltip-text@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetTooltipText :: (IsWidget o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructWidgetTooltipText val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "tooltip-text" (P.Just val)

-- | Set the value of the “@tooltip-text@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #tooltipText
-- @
clearWidgetTooltipText :: (MonadIO m, IsWidget o) => o -> m ()
clearWidgetTooltipText obj = liftIO $ B.Properties.setObjectPropertyString obj "tooltip-text" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data WidgetTooltipTextPropertyInfo
instance AttrInfo WidgetTooltipTextPropertyInfo where
    type AttrAllowedOps WidgetTooltipTextPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetTooltipTextPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetTooltipTextPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint WidgetTooltipTextPropertyInfo = (~) T.Text
    type AttrTransferType WidgetTooltipTextPropertyInfo = T.Text
    type AttrGetType WidgetTooltipTextPropertyInfo = (Maybe T.Text)
    type AttrLabel WidgetTooltipTextPropertyInfo = "tooltip-text"
    type AttrOrigin WidgetTooltipTextPropertyInfo = Widget
    attrGet = getWidgetTooltipText
    attrSet = setWidgetTooltipText
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetTooltipText
    attrClear = clearWidgetTooltipText
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.tooltipText"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:tooltipText"
        })
#endif

-- VVV Prop "valign"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Align"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@valign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #valign
-- @
getWidgetValign :: (MonadIO m, IsWidget o) => o -> m Gtk.Enums.Align
getWidgetValign obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "valign"

-- | Set the value of the “@valign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #valign 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetValign :: (MonadIO m, IsWidget o) => o -> Gtk.Enums.Align -> m ()
setWidgetValign obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "valign" val

-- | Construct a `GValueConstruct` with valid value for the “@valign@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetValign :: (IsWidget o, MIO.MonadIO m) => Gtk.Enums.Align -> m (GValueConstruct o)
constructWidgetValign val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "valign" val

#if defined(ENABLE_OVERLOADING)
data WidgetValignPropertyInfo
instance AttrInfo WidgetValignPropertyInfo where
    type AttrAllowedOps WidgetValignPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetValignPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetValignPropertyInfo = (~) Gtk.Enums.Align
    type AttrTransferTypeConstraint WidgetValignPropertyInfo = (~) Gtk.Enums.Align
    type AttrTransferType WidgetValignPropertyInfo = Gtk.Enums.Align
    type AttrGetType WidgetValignPropertyInfo = Gtk.Enums.Align
    type AttrLabel WidgetValignPropertyInfo = "valign"
    type AttrOrigin WidgetValignPropertyInfo = Widget
    attrGet = getWidgetValign
    attrSet = setWidgetValign
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetValign
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.valign"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:valign"
        })
#endif

-- VVV Prop "vexpand"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@vexpand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #vexpand
-- @
getWidgetVexpand :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetVexpand obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "vexpand"

-- | Set the value of the “@vexpand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #vexpand 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetVexpand :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetVexpand obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "vexpand" val

-- | Construct a `GValueConstruct` with valid value for the “@vexpand@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetVexpand :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetVexpand val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "vexpand" val

#if defined(ENABLE_OVERLOADING)
data WidgetVexpandPropertyInfo
instance AttrInfo WidgetVexpandPropertyInfo where
    type AttrAllowedOps WidgetVexpandPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetVexpandPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetVexpandPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetVexpandPropertyInfo = (~) Bool
    type AttrTransferType WidgetVexpandPropertyInfo = Bool
    type AttrGetType WidgetVexpandPropertyInfo = Bool
    type AttrLabel WidgetVexpandPropertyInfo = "vexpand"
    type AttrOrigin WidgetVexpandPropertyInfo = Widget
    attrGet = getWidgetVexpand
    attrSet = setWidgetVexpand
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetVexpand
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.vexpand"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:vexpand"
        })
#endif

-- VVV Prop "vexpand-set"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@vexpand-set@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #vexpandSet
-- @
getWidgetVexpandSet :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetVexpandSet obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "vexpand-set"

-- | Set the value of the “@vexpand-set@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #vexpandSet 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetVexpandSet :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetVexpandSet obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "vexpand-set" val

-- | Construct a `GValueConstruct` with valid value for the “@vexpand-set@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetVexpandSet :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetVexpandSet val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "vexpand-set" val

#if defined(ENABLE_OVERLOADING)
data WidgetVexpandSetPropertyInfo
instance AttrInfo WidgetVexpandSetPropertyInfo where
    type AttrAllowedOps WidgetVexpandSetPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetVexpandSetPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetVexpandSetPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetVexpandSetPropertyInfo = (~) Bool
    type AttrTransferType WidgetVexpandSetPropertyInfo = Bool
    type AttrGetType WidgetVexpandSetPropertyInfo = Bool
    type AttrLabel WidgetVexpandSetPropertyInfo = "vexpand-set"
    type AttrOrigin WidgetVexpandSetPropertyInfo = Widget
    attrGet = getWidgetVexpandSet
    attrSet = setWidgetVexpandSet
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetVexpandSet
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.vexpandSet"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:vexpandSet"
        })
#endif

-- VVV Prop "visible"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@visible@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #visible
-- @
getWidgetVisible :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetVisible obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "visible"

-- | Set the value of the “@visible@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #visible 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetVisible :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetVisible obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "visible" val

-- | Construct a `GValueConstruct` with valid value for the “@visible@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetVisible :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetVisible val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "visible" val

#if defined(ENABLE_OVERLOADING)
data WidgetVisiblePropertyInfo
instance AttrInfo WidgetVisiblePropertyInfo where
    type AttrAllowedOps WidgetVisiblePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetVisiblePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetVisiblePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetVisiblePropertyInfo = (~) Bool
    type AttrTransferType WidgetVisiblePropertyInfo = Bool
    type AttrGetType WidgetVisiblePropertyInfo = Bool
    type AttrLabel WidgetVisiblePropertyInfo = "visible"
    type AttrOrigin WidgetVisiblePropertyInfo = Widget
    attrGet = getWidgetVisible
    attrSet = setWidgetVisible
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetVisible
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.visible"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:visible"
        })
#endif

-- VVV Prop "width-request"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@width-request@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #widthRequest
-- @
getWidgetWidthRequest :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetWidthRequest obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "width-request"

-- | Set the value of the “@width-request@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #widthRequest 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetWidthRequest :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetWidthRequest obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "width-request" val

-- | Construct a `GValueConstruct` with valid value for the “@width-request@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetWidthRequest :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetWidthRequest val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "width-request" val

#if defined(ENABLE_OVERLOADING)
data WidgetWidthRequestPropertyInfo
instance AttrInfo WidgetWidthRequestPropertyInfo where
    type AttrAllowedOps WidgetWidthRequestPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetWidthRequestPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetWidthRequestPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetWidthRequestPropertyInfo = (~) Int32
    type AttrTransferType WidgetWidthRequestPropertyInfo = Int32
    type AttrGetType WidgetWidthRequestPropertyInfo = Int32
    type AttrLabel WidgetWidthRequestPropertyInfo = "width-request"
    type AttrOrigin WidgetWidthRequestPropertyInfo = Widget
    attrGet = getWidgetWidthRequest
    attrSet = setWidgetWidthRequest
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetWidthRequest
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widthRequest"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#g:attr:widthRequest"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Widget
type instance O.AttributeList Widget = WidgetAttributeList
type WidgetAttributeList = ('[ '("accessibleRole", Gtk.Accessible.AccessibleAccessibleRolePropertyInfo), '("canFocus", WidgetCanFocusPropertyInfo), '("canTarget", WidgetCanTargetPropertyInfo), '("cssClasses", WidgetCssClassesPropertyInfo), '("cssName", WidgetCssNamePropertyInfo), '("cursor", WidgetCursorPropertyInfo), '("focusOnClick", WidgetFocusOnClickPropertyInfo), '("focusable", WidgetFocusablePropertyInfo), '("halign", WidgetHalignPropertyInfo), '("hasDefault", WidgetHasDefaultPropertyInfo), '("hasFocus", WidgetHasFocusPropertyInfo), '("hasTooltip", WidgetHasTooltipPropertyInfo), '("heightRequest", WidgetHeightRequestPropertyInfo), '("hexpand", WidgetHexpandPropertyInfo), '("hexpandSet", WidgetHexpandSetPropertyInfo), '("layoutManager", WidgetLayoutManagerPropertyInfo), '("limitEvents", WidgetLimitEventsPropertyInfo), '("marginBottom", WidgetMarginBottomPropertyInfo), '("marginEnd", WidgetMarginEndPropertyInfo), '("marginStart", WidgetMarginStartPropertyInfo), '("marginTop", WidgetMarginTopPropertyInfo), '("name", WidgetNamePropertyInfo), '("opacity", WidgetOpacityPropertyInfo), '("overflow", WidgetOverflowPropertyInfo), '("parent", WidgetParentPropertyInfo), '("receivesDefault", WidgetReceivesDefaultPropertyInfo), '("root", WidgetRootPropertyInfo), '("scaleFactor", WidgetScaleFactorPropertyInfo), '("sensitive", WidgetSensitivePropertyInfo), '("tooltipMarkup", WidgetTooltipMarkupPropertyInfo), '("tooltipText", WidgetTooltipTextPropertyInfo), '("valign", WidgetValignPropertyInfo), '("vexpand", WidgetVexpandPropertyInfo), '("vexpandSet", WidgetVexpandSetPropertyInfo), '("visible", WidgetVisiblePropertyInfo), '("widthRequest", WidgetWidthRequestPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
widgetCanFocus :: AttrLabelProxy "canFocus"
widgetCanFocus = AttrLabelProxy

widgetCanTarget :: AttrLabelProxy "canTarget"
widgetCanTarget = AttrLabelProxy

widgetCssClasses :: AttrLabelProxy "cssClasses"
widgetCssClasses = AttrLabelProxy

widgetCssName :: AttrLabelProxy "cssName"
widgetCssName = AttrLabelProxy

widgetCursor :: AttrLabelProxy "cursor"
widgetCursor = AttrLabelProxy

widgetFocusOnClick :: AttrLabelProxy "focusOnClick"
widgetFocusOnClick = AttrLabelProxy

widgetFocusable :: AttrLabelProxy "focusable"
widgetFocusable = AttrLabelProxy

widgetHalign :: AttrLabelProxy "halign"
widgetHalign = AttrLabelProxy

widgetHasTooltip :: AttrLabelProxy "hasTooltip"
widgetHasTooltip = AttrLabelProxy

widgetHeightRequest :: AttrLabelProxy "heightRequest"
widgetHeightRequest = AttrLabelProxy

widgetHexpand :: AttrLabelProxy "hexpand"
widgetHexpand = AttrLabelProxy

widgetHexpandSet :: AttrLabelProxy "hexpandSet"
widgetHexpandSet = AttrLabelProxy

widgetLayoutManager :: AttrLabelProxy "layoutManager"
widgetLayoutManager = AttrLabelProxy

widgetLimitEvents :: AttrLabelProxy "limitEvents"
widgetLimitEvents = AttrLabelProxy

widgetMarginBottom :: AttrLabelProxy "marginBottom"
widgetMarginBottom = AttrLabelProxy

widgetMarginEnd :: AttrLabelProxy "marginEnd"
widgetMarginEnd = AttrLabelProxy

widgetMarginStart :: AttrLabelProxy "marginStart"
widgetMarginStart = AttrLabelProxy

widgetMarginTop :: AttrLabelProxy "marginTop"
widgetMarginTop = AttrLabelProxy

widgetName :: AttrLabelProxy "name"
widgetName = AttrLabelProxy

widgetOpacity :: AttrLabelProxy "opacity"
widgetOpacity = AttrLabelProxy

widgetOverflow :: AttrLabelProxy "overflow"
widgetOverflow = AttrLabelProxy

widgetParent :: AttrLabelProxy "parent"
widgetParent = AttrLabelProxy

widgetReceivesDefault :: AttrLabelProxy "receivesDefault"
widgetReceivesDefault = AttrLabelProxy

widgetRoot :: AttrLabelProxy "root"
widgetRoot = AttrLabelProxy

widgetScaleFactor :: AttrLabelProxy "scaleFactor"
widgetScaleFactor = AttrLabelProxy

widgetSensitive :: AttrLabelProxy "sensitive"
widgetSensitive = AttrLabelProxy

widgetTooltipMarkup :: AttrLabelProxy "tooltipMarkup"
widgetTooltipMarkup = AttrLabelProxy

widgetTooltipText :: AttrLabelProxy "tooltipText"
widgetTooltipText = AttrLabelProxy

widgetValign :: AttrLabelProxy "valign"
widgetValign = AttrLabelProxy

widgetVexpand :: AttrLabelProxy "vexpand"
widgetVexpand = AttrLabelProxy

widgetVexpandSet :: AttrLabelProxy "vexpandSet"
widgetVexpandSet = AttrLabelProxy

widgetVisible :: AttrLabelProxy "visible"
widgetVisible = AttrLabelProxy

widgetWidthRequest :: AttrLabelProxy "widthRequest"
widgetWidthRequest = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Widget = WidgetSignalList
type WidgetSignalList = ('[ '("destroy", WidgetDestroySignalInfo), '("directionChanged", WidgetDirectionChangedSignalInfo), '("hide", WidgetHideSignalInfo), '("keynavFailed", WidgetKeynavFailedSignalInfo), '("map", WidgetMapSignalInfo), '("mnemonicActivate", WidgetMnemonicActivateSignalInfo), '("moveFocus", WidgetMoveFocusSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("queryTooltip", WidgetQueryTooltipSignalInfo), '("realize", WidgetRealizeSignalInfo), '("show", WidgetShowSignalInfo), '("stateFlagsChanged", WidgetStateFlagsChangedSignalInfo), '("unmap", WidgetUnmapSignalInfo), '("unrealize", WidgetUnrealizeSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Widget::action_set_enabled
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "action name, such as \"clipboard.paste\""
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "enabled"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether the action is now enabled"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_action_set_enabled" gtk_widget_action_set_enabled :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- action_name : TBasicType TUTF8
    CInt ->                                 -- enabled : TBasicType TBoolean
    IO ()

-- | Enables or disables an action installed with
-- 'GI.Gtk.Structs.WidgetClass.widgetClassInstallAction'.
widgetActionSetEnabled ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> T.Text
    -- ^ /@actionName@/: action name, such as \"clipboard.paste\"
    -> Bool
    -- ^ /@enabled@/: whether the action is now enabled
    -> m ()
widgetActionSetEnabled widget actionName enabled = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    actionName' <- textToCString actionName
    let enabled' = (P.fromIntegral . P.fromEnum) enabled
    gtk_widget_action_set_enabled widget' actionName' enabled'
    touchManagedPtr widget
    freeMem actionName'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetActionSetEnabledMethodInfo
instance (signature ~ (T.Text -> Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetActionSetEnabledMethodInfo a signature where
    overloadedMethod = widgetActionSetEnabled

instance O.OverloadedMethodInfo WidgetActionSetEnabledMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetActionSetEnabled",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetActionSetEnabled"
        })


#endif

-- method Widget::activate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget that is activatable"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_activate" gtk_widget_activate :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Activates the widget.
-- 
-- The activation will emit the signal set using
-- 'GI.Gtk.Structs.WidgetClass.widgetClassSetActivateSignal'
-- during class initialization.
-- 
-- Activation is what happens when you press \<kbd>Enter\<\/kbd>
-- on a widget.
-- 
-- If you wish to handle the activation keybinding yourself,
-- it is recommended to use 'GI.Gtk.Structs.WidgetClass.widgetClassAddShortcut'
-- with an action created with 'GI.Gtk.Objects.SignalAction.signalActionNew'.
-- 
-- If /@widget@/ is not activatable, the function returns false.
widgetActivate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget that is activatable
    -> m Bool
    -- ^ __Returns:__ true if the widget was activated
widgetActivate widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_activate widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetActivateMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetActivateMethodInfo a signature where
    overloadedMethod = widgetActivate

instance O.OverloadedMethodInfo WidgetActivateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetActivate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetActivate"
        })


#endif

-- method Widget::activate_action
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to activate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "args"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "parameters to use" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_activate_action_variant" gtk_widget_activate_action_variant :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- name : TBasicType TUTF8
    Ptr GVariant ->                         -- args : TVariant
    IO CInt

-- | Activates an action for the widget.
-- 
-- The action is looked up in the action groups associated with
-- /@widget@/ and its ancestors.
-- 
-- If the action is in an action group added with
-- 'GI.Gtk.Objects.Widget.widgetInsertActionGroup', the /@name@/ is expected
-- to be prefixed with the prefix that was used when the group was
-- inserted.
-- 
-- The arguments must match the actions expected parameter type,
-- as returned by 'GI.Gio.Interfaces.Action.actionGetParameterType'.
widgetActivateAction ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> T.Text
    -- ^ /@name@/: the name of the action to activate
    -> Maybe (GVariant)
    -- ^ /@args@/: parameters to use
    -> m Bool
    -- ^ __Returns:__ true if the action was activated
widgetActivateAction widget name args = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    name' <- textToCString name
    maybeArgs <- case args of
        Nothing -> return FP.nullPtr
        Just jArgs -> do
            jArgs' <- unsafeManagedPtrGetPtr jArgs
            return jArgs'
    result <- gtk_widget_activate_action_variant widget' name' maybeArgs
    let result' = (/= 0) result
    touchManagedPtr widget
    whenJust args touchManagedPtr
    freeMem name'
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetActivateActionMethodInfo
instance (signature ~ (T.Text -> Maybe (GVariant) -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetActivateActionMethodInfo a signature where
    overloadedMethod = widgetActivateAction

instance O.OverloadedMethodInfo WidgetActivateActionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetActivateAction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetActivateAction"
        })


#endif

-- method Widget::activate_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_activate_default" gtk_widget_activate_default :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Activates the @default.activate@ action for the widget.
-- 
-- The action is looked up in the same was as for
-- 'GI.Gtk.Objects.Widget.widgetActivateAction'.
widgetActivateDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetActivateDefault widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_activate_default widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetActivateDefaultMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetActivateDefaultMethodInfo a signature where
    overloadedMethod = widgetActivateDefault

instance O.OverloadedMethodInfo WidgetActivateDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetActivateDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetActivateDefault"
        })


#endif

-- method Widget::add_controller
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "controller"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "EventController" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "an event controller that hasn't been\n  added to a widget yet"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_controller" gtk_widget_add_controller :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.EventController.EventController -> -- controller : TInterface (Name {namespace = "Gtk", name = "EventController"})
    IO ()

-- | Adds an event controller to the widget.
-- 
-- The event controllers of a widget handle the events that are
-- propagated to the widget.
-- 
-- You will usually want to call this function right after
-- creating any kind of t'GI.Gtk.Objects.EventController.EventController'.
widgetAddController ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gtk.EventController.IsEventController b) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@controller@/: an event controller that hasn\'t been
    --   added to a widget yet
    -> m ()
widgetAddController widget controller = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    controller' <- B.ManagedPtr.disownObject controller
    gtk_widget_add_controller widget' controller'
    touchManagedPtr widget
    touchManagedPtr controller
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetAddControllerMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsWidget a, Gtk.EventController.IsEventController b) => O.OverloadedMethod WidgetAddControllerMethodInfo a signature where
    overloadedMethod = widgetAddController

instance O.OverloadedMethodInfo WidgetAddControllerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddController",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetAddController"
        })


#endif

-- method Widget::add_css_class
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "css_class"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "style class to add to @widget, without the leading period"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_css_class" gtk_widget_add_css_class :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- css_class : TBasicType TUTF8
    IO ()

-- | Adds a style class to the widget.
-- 
-- After calling this function, the widget’s style will match
-- for /@cssClass@/, according to CSS matching rules.
-- 
-- Use 'GI.Gtk.Objects.Widget.widgetRemoveCssClass' to remove the
-- style again.
widgetAddCssClass ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> T.Text
    -- ^ /@cssClass@/: style class to add to /@widget@/, without the leading period
    -> m ()
widgetAddCssClass widget cssClass = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    cssClass' <- textToCString cssClass
    gtk_widget_add_css_class widget' cssClass'
    touchManagedPtr widget
    freeMem cssClass'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetAddCssClassMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetAddCssClassMethodInfo a signature where
    overloadedMethod = widgetAddCssClass

instance O.OverloadedMethodInfo WidgetAddCssClassMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddCssClass",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetAddCssClass"
        })


#endif

-- method Widget::add_mnemonic_label
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "label"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a widget that acts as a mnemonic label for @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_mnemonic_label" gtk_widget_add_mnemonic_label :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- label : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Adds a widget to the list of mnemonic labels for this widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetListMnemonicLabels'.
-- 
-- Note that the list of mnemonic labels for the widget is cleared
-- when the widget is destroyed, so the caller must make sure
-- to update its internal state at this point as well.
widgetAddMnemonicLabel ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@label@/: a widget that acts as a mnemonic label for /@widget@/
    -> m ()
widgetAddMnemonicLabel widget label = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    label' <- unsafeManagedPtrCastPtr label
    gtk_widget_add_mnemonic_label widget' label'
    touchManagedPtr widget
    touchManagedPtr label
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetAddMnemonicLabelMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetAddMnemonicLabelMethodInfo a signature where
    overloadedMethod = widgetAddMnemonicLabel

instance O.OverloadedMethodInfo WidgetAddMnemonicLabelMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddMnemonicLabel",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetAddMnemonicLabel"
        })


#endif

-- method Widget::add_tick_callback
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TickCallback" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "function\n  to call for updating animations"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeNotified
--           , argClosure = 2
--           , argDestroy = 3
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "notify"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DestroyNotify" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "function to call to free @user_data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_tick_callback" gtk_widget_add_tick_callback :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    FunPtr Gtk.Callbacks.C_TickCallback ->  -- callback : TInterface (Name {namespace = "Gtk", name = "TickCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    FunPtr GLib.Callbacks.C_DestroyNotify -> -- notify : TInterface (Name {namespace = "GLib", name = "DestroyNotify"})
    IO Word32

-- | Queues an animation frame update and adds a callback to be called
-- before each frame.
-- 
-- Until the tick callback is removed, it will be called frequently
-- (usually at the frame rate of the output device or as quickly as
-- the application can be repainted, whichever is slower). For this
-- reason, is most suitable for handling graphics that change every
-- frame or every few frames.
-- 
-- The tick callback does not automatically imply a relayout or repaint.
-- If you want a repaint or relayout, and aren’t changing widget properties
-- that would trigger that (for example, changing the text of a label),
-- then you will have to call 'GI.Gtk.Objects.Widget.widgetQueueResize' or
-- 'GI.Gtk.Objects.Widget.widgetQueueDraw' yourself.
-- 
-- 'GI.Gdk.Objects.FrameClock.frameClockGetFrameTime' should generally be used
-- for timing continuous animations and
-- 'GI.Gdk.Structs.FrameTimings.frameTimingsGetPredictedPresentationTime' should be
-- used if you are trying to display isolated frames at particular times.
-- 
-- This is a more convenient alternative to connecting directly to the
-- [FrameClock::update]("GI.Gdk.Objects.FrameClock#g:signal:update") signal of the frame clock, since you
-- don\'t have to worry about when a frame clock is assigned to a widget.
-- 
-- To remove a tick callback, pass the ID that is returned by this function
-- to 'GI.Gtk.Objects.Widget.widgetRemoveTickCallback'.
widgetAddTickCallback ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Callbacks.TickCallback
    -- ^ /@callback@/: function
    --   to call for updating animations
    -> m Word32
    -- ^ __Returns:__ an ID for this callback
widgetAddTickCallback widget callback = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    callback' <- Gtk.Callbacks.mk_TickCallback (Gtk.Callbacks.wrap_TickCallback Nothing (Gtk.Callbacks.drop_closures_TickCallback callback))
    let userData = castFunPtrToPtr callback'
    let notify = SP.safeFreeFunPtrPtr
    result <- gtk_widget_add_tick_callback widget' callback' userData notify
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetAddTickCallbackMethodInfo
instance (signature ~ (Gtk.Callbacks.TickCallback -> m Word32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetAddTickCallbackMethodInfo a signature where
    overloadedMethod = widgetAddTickCallback

instance O.OverloadedMethodInfo WidgetAddTickCallbackMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddTickCallback",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetAddTickCallback"
        })


#endif

-- method Widget::allocate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "new width" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "new height" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "new baseline, or -1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "transform"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "transformation to be applied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_allocate" gtk_widget_allocate :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- width : TBasicType TInt
    Int32 ->                                -- height : TBasicType TInt
    Int32 ->                                -- baseline : TBasicType TInt
    Ptr Gsk.Transform.Transform ->          -- transform : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO ()

-- | Assigns size, position, (optionally) a baseline and transform
-- to a child widget.
-- 
-- In this function, the allocation and baseline may be adjusted.
-- The given allocation will be forced to be bigger than the
-- widget\'s minimum size, as well as at least 0×0 in size.
-- 
-- This function is only used by widget implementations.
-- 
-- For a version that does not take a transform, see
-- 'GI.Gtk.Objects.Widget.widgetSizeAllocate'.
widgetAllocate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Int32
    -- ^ /@width@/: new width
    -> Int32
    -- ^ /@height@/: new height
    -> Int32
    -- ^ /@baseline@/: new baseline, or -1
    -> Maybe (Gsk.Transform.Transform)
    -- ^ /@transform@/: transformation to be applied
    -> m ()
widgetAllocate widget width height baseline transform = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeTransform <- case transform of
        Nothing -> return FP.nullPtr
        Just jTransform -> do
            jTransform' <- B.ManagedPtr.disownBoxed jTransform
            return jTransform'
    gtk_widget_allocate widget' width height baseline maybeTransform
    touchManagedPtr widget
    whenJust transform touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetAllocateMethodInfo
instance (signature ~ (Int32 -> Int32 -> Int32 -> Maybe (Gsk.Transform.Transform) -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetAllocateMethodInfo a signature where
    overloadedMethod = widgetAllocate

instance O.OverloadedMethodInfo WidgetAllocateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAllocate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetAllocate"
        })


#endif

-- method Widget::child_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "direction"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DirectionType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "direction of focus movement"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_child_focus" gtk_widget_child_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- direction : TInterface (Name {namespace = "Gtk", name = "DirectionType"})
    IO CInt

-- | Called by widgets as the user moves around the window using
-- keyboard shortcuts.
-- 
-- The /@direction@/ argument indicates what kind of motion is taking
-- place (up, down, left, right, tab forward, tab backward).
-- 
-- This function calls the t'GI.Gtk.Objects.Widget.Widget'.@/focus/@() virtual function;
-- widgets can override the virtual function in order to implement
-- appropriate focus behavior.
-- 
-- The default @focus()@ virtual function for a widget should return
-- true if moving in /@direction@/ left the focus on a focusable location
-- inside that widget, and false if moving in /@direction@/ moved the focus
-- outside the widget. When returning true, widgets normally call
-- 'GI.Gtk.Objects.Widget.widgetGrabFocus' to place the focus accordingly;
-- when returning false, they don’t modify the current focus location.
-- 
-- This function is used by custom widget implementations; if you\'re
-- writing an app, you’d use 'GI.Gtk.Objects.Widget.widgetGrabFocus' to move
-- the focus to a particular widget.
widgetChildFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Enums.DirectionType
    -- ^ /@direction@/: direction of focus movement
    -> m Bool
    -- ^ __Returns:__ true if focus ended up inside /@widget@/
widgetChildFocus widget direction = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let direction' = (fromIntegral . fromEnum) direction
    result <- gtk_widget_child_focus widget' direction'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetChildFocusMethodInfo
instance (signature ~ (Gtk.Enums.DirectionType -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetChildFocusMethodInfo a signature where
    overloadedMethod = widgetChildFocus

instance O.OverloadedMethodInfo WidgetChildFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetChildFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetChildFocus"
        })


#endif

-- method Widget::compute_bounds
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the target widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_bounds"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rectangle taking the bounds"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_compute_bounds" gtk_widget_compute_bounds :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- target : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Graphene.Rect.Rect ->               -- out_bounds : TInterface (Name {namespace = "Graphene", name = "Rect"})
    IO CInt

-- | Computes the bounds for /@widget@/ in the coordinate space of /@target@/.
-- 
-- The bounds of widget are (the bounding box of) the region that it is
-- expected to draw in. See the <https://docs.gtk.org/gtk4/coordinates.html coordinate system>
-- overview to learn more.
-- 
-- If the operation is successful, true is returned. If /@widget@/ has no
-- bounds or the bounds cannot be expressed in /@target@/\'s coordinate space
-- (for example if both widgets are in different windows), false is
-- returned and /@bounds@/ is set to the zero rectangle.
-- 
-- It is valid for /@widget@/ and /@target@/ to be the same widget.
widgetComputeBounds ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: the widget to query
    -> b
    -- ^ /@target@/: the target widget
    -> m ((Bool, Graphene.Rect.Rect))
    -- ^ __Returns:__ true if the bounds could be computed
widgetComputeBounds widget target = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    target' <- unsafeManagedPtrCastPtr target
    outBounds <- SP.callocBoxedBytes 16 :: IO (Ptr Graphene.Rect.Rect)
    result <- gtk_widget_compute_bounds widget' target' outBounds
    let result' = (/= 0) result
    outBounds' <- (wrapBoxed Graphene.Rect.Rect) outBounds
    touchManagedPtr widget
    touchManagedPtr target
    return (result', outBounds')

#if defined(ENABLE_OVERLOADING)
data WidgetComputeBoundsMethodInfo
instance (signature ~ (b -> m ((Bool, Graphene.Rect.Rect))), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetComputeBoundsMethodInfo a signature where
    overloadedMethod = widgetComputeBounds

instance O.OverloadedMethodInfo WidgetComputeBoundsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetComputeBounds",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetComputeBounds"
        })


#endif

-- method Widget::compute_expand
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "orientation"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Orientation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "expand direction" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_compute_expand" gtk_widget_compute_expand :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- orientation : TInterface (Name {namespace = "Gtk", name = "Orientation"})
    IO CInt

-- | Computes whether a parent widget should give this widget
-- extra space when possible.
-- 
-- Widgets with children should check this, rather than looking at
-- 'GI.Gtk.Objects.Widget.widgetGetHexpand' or 'GI.Gtk.Objects.Widget.widgetGetVexpand'.
-- 
-- This function already checks whether the widget is visible, so
-- visibility does not need to be checked separately. Non-visible
-- widgets are not expanded.
-- 
-- The computed expand value uses either the expand setting explicitly
-- set on the widget itself, or, if none has been explicitly set,
-- the widget may expand if some of its children do.
widgetComputeExpand ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Enums.Orientation
    -- ^ /@orientation@/: expand direction
    -> m Bool
    -- ^ __Returns:__ whether widget tree rooted here should be expanded
widgetComputeExpand widget orientation = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let orientation' = (fromIntegral . fromEnum) orientation
    result <- gtk_widget_compute_expand widget' orientation'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetComputeExpandMethodInfo
instance (signature ~ (Gtk.Enums.Orientation -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetComputeExpandMethodInfo a signature where
    overloadedMethod = widgetComputeExpand

instance O.OverloadedMethodInfo WidgetComputeExpandMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetComputeExpand",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetComputeExpand"
        })


#endif

-- method Widget::compute_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to transform into"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "point"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Point" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a point in @widget's coordinate system"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_point"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Point" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "set to the corresponding coordinates in\n  @target's coordinate system"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_compute_point" gtk_widget_compute_point :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- target : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Graphene.Point.Point ->             -- point : TInterface (Name {namespace = "Graphene", name = "Point"})
    Ptr Graphene.Point.Point ->             -- out_point : TInterface (Name {namespace = "Graphene", name = "Point"})
    IO CInt

-- | Translates the given /@point@/ in /@widget@/\'s coordinates to coordinates
-- in /@target@/’s coordinate system.
-- 
-- In order to perform this operation, both widgets must share a
-- a common ancestor. If that is not the case, /@outPoint@/ is set
-- to (0, 0) and false is returned.
widgetComputePoint ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: the widget to query
    -> b
    -- ^ /@target@/: the widget to transform into
    -> Graphene.Point.Point
    -- ^ /@point@/: a point in /@widget@/\'s coordinate system
    -> m ((Bool, Graphene.Point.Point))
    -- ^ __Returns:__ true if /@srcWidget@/ and /@destWidget@/ have a common
    --   ancestor, false otherwise
widgetComputePoint widget target point = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    target' <- unsafeManagedPtrCastPtr target
    point' <- unsafeManagedPtrGetPtr point
    outPoint <- SP.callocBoxedBytes 8 :: IO (Ptr Graphene.Point.Point)
    result <- gtk_widget_compute_point widget' target' point' outPoint
    let result' = (/= 0) result
    outPoint' <- (wrapBoxed Graphene.Point.Point) outPoint
    touchManagedPtr widget
    touchManagedPtr target
    touchManagedPtr point
    return (result', outPoint')

#if defined(ENABLE_OVERLOADING)
data WidgetComputePointMethodInfo
instance (signature ~ (b -> Graphene.Point.Point -> m ((Bool, Graphene.Point.Point))), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetComputePointMethodInfo a signature where
    overloadedMethod = widgetComputePoint

instance O.OverloadedMethodInfo WidgetComputePointMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetComputePoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetComputePoint"
        })


#endif

-- method Widget::compute_transform
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the target widget that the matrix will transform to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_transform"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Matrix" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to\n  store the final transformation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_compute_transform" gtk_widget_compute_transform :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- target : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Graphene.Matrix.Matrix ->           -- out_transform : TInterface (Name {namespace = "Graphene", name = "Matrix"})
    IO CInt

-- | Computes a matrix suitable to describe a transformation from
-- /@widget@/\'s coordinate system into /@target@/\'s coordinate system.
-- 
-- The transform can not be computed in certain cases, for example
-- when /@widget@/ and /@target@/ do not share a common ancestor. In that
-- case /@outTransform@/ gets set to the identity matrix.
-- 
-- To learn more about widget coordinate systems, see the coordinate
-- system <https://docs.gtk.org/gtk4/coordinates.html overview>.
widgetComputeTransform ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@target@/: the target widget that the matrix will transform to
    -> m ((Bool, Graphene.Matrix.Matrix))
    -- ^ __Returns:__ true if the transform could be computed
widgetComputeTransform widget target = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    target' <- unsafeManagedPtrCastPtr target
    outTransform <- SP.callocBoxedBytes 64 :: IO (Ptr Graphene.Matrix.Matrix)
    result <- gtk_widget_compute_transform widget' target' outTransform
    let result' = (/= 0) result
    outTransform' <- (wrapBoxed Graphene.Matrix.Matrix) outTransform
    touchManagedPtr widget
    touchManagedPtr target
    return (result', outTransform')

#if defined(ENABLE_OVERLOADING)
data WidgetComputeTransformMethodInfo
instance (signature ~ (b -> m ((Bool, Graphene.Matrix.Matrix))), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetComputeTransformMethodInfo a signature where
    overloadedMethod = widgetComputeTransform

instance O.OverloadedMethodInfo WidgetComputeTransformMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetComputeTransform",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetComputeTransform"
        })


#endif

-- method Widget::contains
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "X coordinate to test, relative to @widget's origin"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Y coordinate to test, relative to @widget's origin"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_contains" gtk_widget_contains :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CDouble ->                              -- x : TBasicType TDouble
    CDouble ->                              -- y : TBasicType TDouble
    IO CInt

-- | Tests if a given point is contained in the widget.
-- 
-- The coordinates for (x, y) must be in widget coordinates, so
-- (0, 0) is assumed to be the top left of /@widget@/\'s content area.
widgetContains ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> Double
    -- ^ /@x@/: X coordinate to test, relative to /@widget@/\'s origin
    -> Double
    -- ^ /@y@/: Y coordinate to test, relative to /@widget@/\'s origin
    -> m Bool
    -- ^ __Returns:__ true if /@widget@/ contains the point (x, y)
widgetContains widget x y = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let x' = realToFrac x
    let y' = realToFrac y
    result <- gtk_widget_contains widget' x' y'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetContainsMethodInfo
instance (signature ~ (Double -> Double -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetContainsMethodInfo a signature where
    overloadedMethod = widgetContains

instance O.OverloadedMethodInfo WidgetContainsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetContains",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetContains"
        })


#endif

-- method Widget::create_pango_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Context" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_create_pango_context" gtk_widget_create_pango_context :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Pango.Context.Context)

-- | Creates a new @PangoContext@ that is configured for the widget.
-- 
-- The @PangoContext@ will have the appropriate font map,
-- font options, font description, and base direction set.
-- 
-- See also 'GI.Gtk.Objects.Widget.widgetGetPangoContext'.
widgetCreatePangoContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Pango.Context.Context
    -- ^ __Returns:__ the new @PangoContext@
widgetCreatePangoContext widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_create_pango_context widget'
    checkUnexpectedReturnNULL "widgetCreatePangoContext" result
    result' <- (wrapObject Pango.Context.Context) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetCreatePangoContextMethodInfo
instance (signature ~ (m Pango.Context.Context), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetCreatePangoContextMethodInfo a signature where
    overloadedMethod = widgetCreatePangoContext

instance O.OverloadedMethodInfo WidgetCreatePangoContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetCreatePangoContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetCreatePangoContext"
        })


#endif

-- method Widget::create_pango_layout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text to set on the layout"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Layout" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_create_pango_layout" gtk_widget_create_pango_layout :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- text : TBasicType TUTF8
    IO (Ptr Pango.Layout.Layout)

-- | Creates a new @PangoLayout@ that is configured for the widget.
-- 
-- The @PangoLayout@ will have the appropriate font map,
-- font description, and base direction set.
-- 
-- If you keep a @PangoLayout@ created in this way around,
-- you need to re-create it when the widgets @PangoContext@
-- is replaced. This can be tracked by listening to changes
-- of the [Widget:root]("GI.Gtk.Objects.Widget#g:attr:root") property on the widget.
widgetCreatePangoLayout ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (T.Text)
    -- ^ /@text@/: text to set on the layout
    -> m Pango.Layout.Layout
    -- ^ __Returns:__ the new @PangoLayout@
widgetCreatePangoLayout widget text = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeText <- case text of
        Nothing -> return FP.nullPtr
        Just jText -> do
            jText' <- textToCString jText
            return jText'
    result <- gtk_widget_create_pango_layout widget' maybeText
    checkUnexpectedReturnNULL "widgetCreatePangoLayout" result
    result' <- (wrapObject Pango.Layout.Layout) result
    touchManagedPtr widget
    freeMem maybeText
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetCreatePangoLayoutMethodInfo
instance (signature ~ (Maybe (T.Text) -> m Pango.Layout.Layout), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetCreatePangoLayoutMethodInfo a signature where
    overloadedMethod = widgetCreatePangoLayout

instance O.OverloadedMethodInfo WidgetCreatePangoLayoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetCreatePangoLayout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetCreatePangoLayout"
        })


#endif

-- method Widget::dispose_template
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget with a template"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the type of the widget to finalize the template for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_dispose_template" gtk_widget_dispose_template :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CGType ->                               -- widget_type : TBasicType TGType
    IO ()

-- | Clears the template children for the widget.
-- 
-- This function is the opposite of 'GI.Gtk.Objects.Widget.widgetInitTemplate',
-- and it is used to clear all the template children from a widget
-- instance. If you bound a template child to a field in the instance
-- structure, or in the instance private data structure, the field will
-- be set to @NULL@ after this function returns.
-- 
-- You should call this function inside the @GObjectClass.dispose()@
-- implementation of any widget that called 'GI.Gtk.Objects.Widget.widgetInitTemplate'.
-- Typically, you will want to call this function last, right before
-- chaining up to the parent type\'s dispose implementation, e.g.
-- 
-- 
-- === /c code/
-- >static void
-- >some_widget_dispose (GObject *gobject)
-- >{
-- >  SomeWidget *self = SOME_WIDGET (gobject);
-- >
-- >  // Clear the template data for SomeWidget
-- >  gtk_widget_dispose_template (GTK_WIDGET (self), SOME_TYPE_WIDGET);
-- >
-- >  G_OBJECT_CLASS (some_widget_parent_class)->dispose (gobject);
-- >}
-- 
-- 
-- /Since: 4.8/
widgetDisposeTemplate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget with a template
    -> GType
    -- ^ /@widgetType@/: the type of the widget to finalize the template for
    -> m ()
widgetDisposeTemplate widget widgetType = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let widgetType' = gtypeToCGType widgetType
    gtk_widget_dispose_template widget' widgetType'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDisposeTemplateMethodInfo
instance (signature ~ (GType -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDisposeTemplateMethodInfo a signature where
    overloadedMethod = widgetDisposeTemplate

instance O.OverloadedMethodInfo WidgetDisposeTemplateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDisposeTemplate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetDisposeTemplate"
        })


#endif

-- method Widget::drag_check_threshold
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "X coordinate of start of drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Y coordinate of start of drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "current_x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "current X coordinate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "current_y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "current Y coordinate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_check_threshold" gtk_drag_check_threshold :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- start_x : TBasicType TInt
    Int32 ->                                -- start_y : TBasicType TInt
    Int32 ->                                -- current_x : TBasicType TInt
    Int32 ->                                -- current_y : TBasicType TInt
    IO CInt

-- | Checks to see if a drag movement has passed the GTK drag threshold.
widgetDragCheckThreshold ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Int32
    -- ^ /@startX@/: X coordinate of start of drag
    -> Int32
    -- ^ /@startY@/: Y coordinate of start of drag
    -> Int32
    -- ^ /@currentX@/: current X coordinate
    -> Int32
    -- ^ /@currentY@/: current Y coordinate
    -> m Bool
    -- ^ __Returns:__ true if the drag threshold has been passed
widgetDragCheckThreshold widget startX startY currentX currentY = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_drag_check_threshold widget' startX startY currentX currentY
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetDragCheckThresholdMethodInfo
instance (signature ~ (Int32 -> Int32 -> Int32 -> Int32 -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragCheckThresholdMethodInfo a signature where
    overloadedMethod = widgetDragCheckThreshold

instance O.OverloadedMethodInfo WidgetDragCheckThresholdMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragCheckThreshold",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetDragCheckThreshold"
        })


#endif

-- method Widget::error_bell
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_error_bell" gtk_widget_error_bell :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Notifies the user about an input-related error on the widget.
-- 
-- If the [Settings:gtkErrorBell]("GI.Gtk.Objects.Settings#g:attr:gtkErrorBell") setting is true,
-- it calls 'GI.Gdk.Objects.Surface.surfaceBeep', otherwise it does nothing.
-- 
-- Note that the effect of 'GI.Gdk.Objects.Surface.surfaceBeep' can be configured
-- in many ways, depending on the windowing backend and the desktop
-- environment or window manager that is used.
widgetErrorBell ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetErrorBell widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_error_bell widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetErrorBellMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetErrorBellMethodInfo a signature where
    overloadedMethod = widgetErrorBell

instance O.OverloadedMethodInfo WidgetErrorBellMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetErrorBell",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetErrorBell"
        })


#endif

-- method Widget::get_allocated_baseline
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocated_baseline" gtk_widget_get_allocated_baseline :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

{-# DEPRECATED widgetGetAllocatedBaseline ["(Since version 4.12)","Use 'GI.Gtk.Objects.Widget.widgetGetBaseline' instead"] #-}
-- | Returns the baseline that has currently been allocated to the widget.
-- 
-- This function is intended to be used when implementing handlers
-- for the @GtkWidget@Class.@/snapshot()/@ function, and when allocating
-- child widgets in @GtkWidget@Class.@/size_allocate()/@.
widgetGetAllocatedBaseline ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> m Int32
    -- ^ __Returns:__ the baseline of the /@widget@/, or -1 if none
widgetGetAllocatedBaseline widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_allocated_baseline widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocatedBaselineMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocatedBaselineMethodInfo a signature where
    overloadedMethod = widgetGetAllocatedBaseline

instance O.OverloadedMethodInfo WidgetGetAllocatedBaselineMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocatedBaseline",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocatedBaseline"
        })


#endif

-- method Widget::get_allocated_height
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocated_height" gtk_widget_get_allocated_height :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

{-# DEPRECATED widgetGetAllocatedHeight ["(Since version 4.12)","Use 'GI.Gtk.Objects.Widget.widgetGetHeight' instead"] #-}
-- | Returns the height that has currently been allocated to the widget.
-- 
-- To learn more about widget sizes, see the coordinate
-- system <https://docs.gtk.org/gtk4/coordinates.html overview>.
widgetGetAllocatedHeight ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> m Int32
    -- ^ __Returns:__ the height of the /@widget@/
widgetGetAllocatedHeight widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_allocated_height widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocatedHeightMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocatedHeightMethodInfo a signature where
    overloadedMethod = widgetGetAllocatedHeight

instance O.OverloadedMethodInfo WidgetGetAllocatedHeightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocatedHeight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocatedHeight"
        })


#endif

-- method Widget::get_allocated_width
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocated_width" gtk_widget_get_allocated_width :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

{-# DEPRECATED widgetGetAllocatedWidth ["(Since version 4.12)","Use 'GI.Gtk.Objects.Widget.widgetGetWidth' instead"] #-}
-- | Returns the width that has currently been allocated to the widget.
-- 
-- To learn more about widget sizes, see the coordinate
-- system <https://docs.gtk.org/gtk4/coordinates.html overview>.
widgetGetAllocatedWidth ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> m Int32
    -- ^ __Returns:__ the width of the /@widget@/
widgetGetAllocatedWidth widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_allocated_width widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocatedWidthMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocatedWidthMethodInfo a signature where
    overloadedMethod = widgetGetAllocatedWidth

instance O.OverloadedMethodInfo WidgetGetAllocatedWidthMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocatedWidth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocatedWidth"
        })


#endif

-- method Widget::get_allocation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "allocation"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Rectangle" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to a `GtkAllocation` to copy to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocation" gtk_widget_get_allocation :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Rectangle.Rectangle ->          -- allocation : TInterface (Name {namespace = "Gdk", name = "Rectangle"})
    IO ()

{-# DEPRECATED widgetGetAllocation ["(Since version 4.12)","Use 'GI.Gtk.Objects.Widget.widgetComputeBounds',","'GI.Gtk.Objects.Widget.widgetGetWidth' or 'GI.Gtk.Objects.Widget.widgetGetHeight' instead."] #-}
-- | Retrieves the widget’s allocation.
-- 
-- Note, when implementing a layout widget: a widget’s allocation
-- will be its “adjusted” allocation, that is, the widget’s parent
-- typically calls 'GI.Gtk.Objects.Widget.widgetSizeAllocate' with an allocation,
-- and that allocation is then adjusted (to handle margin
-- and alignment for example) before assignment to the widget.
-- 'GI.Gtk.Objects.Widget.widgetGetAllocation' returns the adjusted allocation that
-- was actually assigned to the widget. The adjusted allocation is
-- guaranteed to be completely contained within the
-- 'GI.Gtk.Objects.Widget.widgetSizeAllocate' allocation, however.
-- 
-- So a layout widget is guaranteed that its children stay inside
-- the assigned bounds, but not that they have exactly the bounds the
-- widget assigned.
widgetGetAllocation ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Gdk.Rectangle.Rectangle)
widgetGetAllocation widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    allocation <- SP.callocBoxedBytes 16 :: IO (Ptr Gdk.Rectangle.Rectangle)
    gtk_widget_get_allocation widget' allocation
    allocation' <- (wrapBoxed Gdk.Rectangle.Rectangle) allocation
    touchManagedPtr widget
    return allocation'

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocationMethodInfo
instance (signature ~ (m (Gdk.Rectangle.Rectangle)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocationMethodInfo a signature where
    overloadedMethod = widgetGetAllocation

instance O.OverloadedMethodInfo WidgetGetAllocationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocation"
        })


#endif

-- method Widget::get_ancestor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "ancestor type" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_ancestor" gtk_widget_get_ancestor :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CGType ->                               -- widget_type : TBasicType TGType
    IO (Ptr Widget)

-- | Gets the first ancestor of the widget with type /@widgetType@/.
-- 
-- For example, @gtk_widget_get_ancestor (widget, GTK_TYPE_BOX)@
-- gets the first @GtkBox@ that’s an ancestor of /@widget@/. No
-- reference will be added to the returned widget; it should
-- not be unreferenced.
-- 
-- Note that unlike 'GI.Gtk.Objects.Widget.widgetIsAncestor', this function
-- considers /@widget@/ to be an ancestor of itself.
widgetGetAncestor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> GType
    -- ^ /@widgetType@/: ancestor type
    -> m (Maybe Widget)
    -- ^ __Returns:__ the ancestor widget
widgetGetAncestor widget widgetType = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let widgetType' = gtypeToCGType widgetType
    result <- gtk_widget_get_ancestor widget' widgetType'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetAncestorMethodInfo
instance (signature ~ (GType -> m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAncestorMethodInfo a signature where
    overloadedMethod = widgetGetAncestor

instance O.OverloadedMethodInfo WidgetGetAncestorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAncestor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAncestor"
        })


#endif

-- method Widget::get_baseline
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_baseline" gtk_widget_get_baseline :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Returns the baseline that has currently been allocated to the widget.
-- 
-- This function is intended to be used when implementing handlers
-- for the @GtkWidgetClass.snapshot()@ function, and when allocating
-- child widgets in @GtkWidgetClass.size_allocate()@.
-- 
-- /Since: 4.12/
widgetGetBaseline ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> m Int32
    -- ^ __Returns:__ the baseline of the /@widget@/, or -1 if none
widgetGetBaseline widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_baseline widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetBaselineMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetBaselineMethodInfo a signature where
    overloadedMethod = widgetGetBaseline

instance O.OverloadedMethodInfo WidgetGetBaselineMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetBaseline",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetBaseline"
        })


#endif

-- method Widget::get_can_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_can_focus" gtk_widget_get_can_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the input focus can enter the widget or any
-- of its children.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetCanFocus'.
widgetGetCanFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the input focus can enter /@widget@/
widgetGetCanFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_can_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetCanFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetCanFocusMethodInfo a signature where
    overloadedMethod = widgetGetCanFocus

instance O.OverloadedMethodInfo WidgetGetCanFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetCanFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetCanFocus"
        })


#endif

-- method Widget::get_can_target
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_can_target" gtk_widget_get_can_target :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Queries whether the widget can be the target of pointer events.
widgetGetCanTarget ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if /@widget@/ can receive pointer events
widgetGetCanTarget widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_can_target widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetCanTargetMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetCanTargetMethodInfo a signature where
    overloadedMethod = widgetGetCanTarget

instance O.OverloadedMethodInfo WidgetGetCanTargetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetCanTarget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetCanTarget"
        })


#endif

-- method Widget::get_child_visible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_child_visible" gtk_widget_get_child_visible :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets the value set with 'GI.Gtk.Objects.Widget.widgetSetChildVisible'.
-- 
-- If you feel a need to use this function, your code probably
-- needs reorganization.
-- 
-- This function is only useful for widget implementations
-- and should never be called by an application.
widgetGetChildVisible ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget is mapped with the parent
widgetGetChildVisible widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_child_visible widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetChildVisibleMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetChildVisibleMethodInfo a signature where
    overloadedMethod = widgetGetChildVisible

instance O.OverloadedMethodInfo WidgetGetChildVisibleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetChildVisible",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetChildVisible"
        })


#endif

-- method Widget::get_clipboard
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Clipboard" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_clipboard" gtk_widget_get_clipboard :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Clipboard.Clipboard)

-- | Gets the clipboard object for the widget.
-- 
-- This is a utility function to get the clipboard object for the
-- display that /@widget@/ is using.
-- 
-- Note that this function always works, even when /@widget@/ is not
-- realized yet.
widgetGetClipboard ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gdk.Clipboard.Clipboard
    -- ^ __Returns:__ the appropriate clipboard object
widgetGetClipboard widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_clipboard widget'
    checkUnexpectedReturnNULL "widgetGetClipboard" result
    result' <- (newObject Gdk.Clipboard.Clipboard) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetClipboardMethodInfo
instance (signature ~ (m Gdk.Clipboard.Clipboard), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetClipboardMethodInfo a signature where
    overloadedMethod = widgetGetClipboard

instance O.OverloadedMethodInfo WidgetGetClipboardMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetClipboard",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetClipboard"
        })


#endif

-- method Widget::get_color
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "color"
--           , argType = TInterface Name { namespace = "Gdk" , name = "RGBA" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the color"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_color" gtk_widget_get_color :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.RGBA.RGBA ->                    -- color : TInterface (Name {namespace = "Gdk", name = "RGBA"})
    IO ()

-- | Gets the current foreground color for the widget’s style.
-- 
-- This function should only be used in snapshot
-- implementations that need to do custom drawing
-- with the foreground color.
-- 
-- /Since: 4.10/
widgetGetColor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Gdk.RGBA.RGBA)
widgetGetColor widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    color <- SP.callocBoxedBytes 16 :: IO (Ptr Gdk.RGBA.RGBA)
    gtk_widget_get_color widget' color
    color' <- (wrapBoxed Gdk.RGBA.RGBA) color
    touchManagedPtr widget
    return color'

#if defined(ENABLE_OVERLOADING)
data WidgetGetColorMethodInfo
instance (signature ~ (m (Gdk.RGBA.RGBA)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetColorMethodInfo a signature where
    overloadedMethod = widgetGetColor

instance O.OverloadedMethodInfo WidgetGetColorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetColor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetColor"
        })


#endif

-- method Widget::get_css_classes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_css_classes" gtk_widget_get_css_classes :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr CString)

-- | Returns the list of style classes applied to the widget.
widgetGetCssClasses ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m [T.Text]
    -- ^ __Returns:__ a @NULL@-terminated list of
    --   css classes currently applied to /@widget@/
widgetGetCssClasses widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_css_classes widget'
    checkUnexpectedReturnNULL "widgetGetCssClasses" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetCssClassesMethodInfo
instance (signature ~ (m [T.Text]), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetCssClassesMethodInfo a signature where
    overloadedMethod = widgetGetCssClasses

instance O.OverloadedMethodInfo WidgetGetCssClassesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetCssClasses",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetCssClasses"
        })


#endif

-- method Widget::get_css_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_css_name" gtk_widget_get_css_name :: 
    Ptr Widget ->                           -- self : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CString

-- | Returns the CSS name of the widget.
widgetGetCssName ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@self@/: a widget
    -> m T.Text
    -- ^ __Returns:__ the CSS name
widgetGetCssName self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_widget_get_css_name self'
    checkUnexpectedReturnNULL "widgetGetCssName" result
    result' <- cstringToText result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetCssNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetCssNameMethodInfo a signature where
    overloadedMethod = widgetGetCssName

instance O.OverloadedMethodInfo WidgetGetCssNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetCssName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetCssName"
        })


#endif

-- method Widget::get_cursor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Cursor" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_cursor" gtk_widget_get_cursor :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Cursor.Cursor)

-- | Gets the cursor set on the widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetCursor' for details.
widgetGetCursor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Gdk.Cursor.Cursor)
    -- ^ __Returns:__ the cursor
    --   that is set on /@widget@/
widgetGetCursor widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_cursor widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.Cursor.Cursor) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetCursorMethodInfo
instance (signature ~ (m (Maybe Gdk.Cursor.Cursor)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetCursorMethodInfo a signature where
    overloadedMethod = widgetGetCursor

instance O.OverloadedMethodInfo WidgetGetCursorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetCursor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetCursor"
        })


#endif

-- method Widget::get_direction
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "TextDirection" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_direction" gtk_widget_get_direction :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets the reading direction for the widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetDirection'.
widgetGetDirection ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gtk.Enums.TextDirection
    -- ^ __Returns:__ the reading direction for the widget
widgetGetDirection widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_direction widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetDirectionMethodInfo
instance (signature ~ (m Gtk.Enums.TextDirection), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetDirectionMethodInfo a signature where
    overloadedMethod = widgetGetDirection

instance O.OverloadedMethodInfo WidgetGetDirectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetDirection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetDirection"
        })


#endif

-- method Widget::get_display
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Display" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_display" gtk_widget_get_display :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Display.Display)

-- | Get the display for the window that the widget belongs to.
-- 
-- This function can only be called after the widget has been
-- added to a widget hierarchy with a @GtkRoot@ at the top.
-- 
-- In general, you should only create display-specific
-- resources when a widget has been realized, and you should
-- free those resources when the widget is unrealized.
widgetGetDisplay ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gdk.Display.Display
    -- ^ __Returns:__ the display for this widget
widgetGetDisplay widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_display widget'
    checkUnexpectedReturnNULL "widgetGetDisplay" result
    result' <- (newObject Gdk.Display.Display) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetDisplayMethodInfo
instance (signature ~ (m Gdk.Display.Display), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetDisplayMethodInfo a signature where
    overloadedMethod = widgetGetDisplay

instance O.OverloadedMethodInfo WidgetGetDisplayMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetDisplay",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetDisplay"
        })


#endif

-- method Widget::get_first_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_first_child" gtk_widget_get_first_child :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Widget)

-- | Returns the widget’s first child.
-- 
-- This function is primarily meant for widget implementations.
widgetGetFirstChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Widget)
    -- ^ __Returns:__ the widget\'s first child
widgetGetFirstChild widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_first_child widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetFirstChildMethodInfo
instance (signature ~ (m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFirstChildMethodInfo a signature where
    overloadedMethod = widgetGetFirstChild

instance O.OverloadedMethodInfo WidgetGetFirstChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFirstChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFirstChild"
        })


#endif

-- method Widget::get_focus_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_focus_child" gtk_widget_get_focus_child :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Widget)

-- | Returns the focus child of the widget.
widgetGetFocusChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Widget)
    -- ^ __Returns:__ the current focus
    --   child of /@widget@/
widgetGetFocusChild widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_focus_child widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetFocusChildMethodInfo
instance (signature ~ (m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFocusChildMethodInfo a signature where
    overloadedMethod = widgetGetFocusChild

instance O.OverloadedMethodInfo WidgetGetFocusChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFocusChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFocusChild"
        })


#endif

-- method Widget::get_focus_on_click
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_focus_on_click" gtk_widget_get_focus_on_click :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns whether the widget should grab focus when it is clicked
-- with the mouse.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetFocusOnClick'.
widgetGetFocusOnClick ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget should grab focus when it is
    --   clicked with the mouse
widgetGetFocusOnClick widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_focus_on_click widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetFocusOnClickMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFocusOnClickMethodInfo a signature where
    overloadedMethod = widgetGetFocusOnClick

instance O.OverloadedMethodInfo WidgetGetFocusOnClickMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFocusOnClick",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFocusOnClick"
        })


#endif

-- method Widget::get_focusable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_focusable" gtk_widget_get_focusable :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget can own the input focus.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetFocusable'.
widgetGetFocusable ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if /@widget@/ can own the input focus
widgetGetFocusable widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_focusable widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetFocusableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFocusableMethodInfo a signature where
    overloadedMethod = widgetGetFocusable

instance O.OverloadedMethodInfo WidgetGetFocusableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFocusable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFocusable"
        })


#endif

-- method Widget::get_font_map
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "FontMap" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_font_map" gtk_widget_get_font_map :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Pango.FontMap.FontMap)

-- | Gets the font map of the widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetFontMap'.
widgetGetFontMap ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Pango.FontMap.FontMap)
    -- ^ __Returns:__ the font map of /@widget@/
widgetGetFontMap widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_font_map widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Pango.FontMap.FontMap) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetFontMapMethodInfo
instance (signature ~ (m (Maybe Pango.FontMap.FontMap)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFontMapMethodInfo a signature where
    overloadedMethod = widgetGetFontMap

instance O.OverloadedMethodInfo WidgetGetFontMapMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFontMap",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFontMap"
        })


#endif

-- method Widget::get_font_options
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "cairo" , name = "FontOptions" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_font_options" gtk_widget_get_font_options :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Cairo.FontOptions.FontOptions)

{-# DEPRECATED widgetGetFontOptions ["(Since version 4.16)"] #-}
-- | Returns the @cairo_font_options_t@ of the widget.
-- 
-- Seee 'GI.Gtk.Objects.Widget.widgetSetFontOptions'.
widgetGetFontOptions ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Cairo.FontOptions.FontOptions)
    -- ^ __Returns:__ the @cairo_font_options_t@ of widget
widgetGetFontOptions widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_font_options widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Cairo.FontOptions.FontOptions) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetFontOptionsMethodInfo
instance (signature ~ (m (Maybe Cairo.FontOptions.FontOptions)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFontOptionsMethodInfo a signature where
    overloadedMethod = widgetGetFontOptions

instance O.OverloadedMethodInfo WidgetGetFontOptionsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFontOptions",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFontOptions"
        })


#endif

-- method Widget::get_frame_clock
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "FrameClock" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_frame_clock" gtk_widget_get_frame_clock :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.FrameClock.FrameClock)

-- | Obtains the frame clock for a widget.
-- 
-- The frame clock is a global “ticker” that can be used to drive
-- animations and repaints. The most common reason to get the frame
-- clock is to call 'GI.Gdk.Objects.FrameClock.frameClockGetFrameTime', in order
-- to get a time to use for animating. For example you might record
-- the start of the animation with an initial value from
-- 'GI.Gdk.Objects.FrameClock.frameClockGetFrameTime', and then update the animation
-- by calling 'GI.Gdk.Objects.FrameClock.frameClockGetFrameTime' again during each repaint.
-- 
-- 'GI.Gdk.Objects.FrameClock.frameClockRequestPhase' will result in a new frame on the
-- clock, but won’t necessarily repaint any widgets. To repaint a widget,
-- you have to use 'GI.Gtk.Objects.Widget.widgetQueueDraw' which invalidates the
-- widget (thus scheduling it to receive a draw on the next frame).
-- 'GI.Gtk.Objects.Widget.widgetQueueDraw' will also end up requesting a frame
-- on the appropriate frame clock.
-- 
-- A widget’s frame clock will not change while the widget is mapped.
-- Reparenting a widget (which implies a temporary unmap) can change
-- the widget’s frame clock.
-- 
-- Unrealized widgets do not have a frame clock.
widgetGetFrameClock ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Gdk.FrameClock.FrameClock)
    -- ^ __Returns:__ the frame clock
widgetGetFrameClock widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_frame_clock widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.FrameClock.FrameClock) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetFrameClockMethodInfo
instance (signature ~ (m (Maybe Gdk.FrameClock.FrameClock)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFrameClockMethodInfo a signature where
    overloadedMethod = widgetGetFrameClock

instance O.OverloadedMethodInfo WidgetGetFrameClockMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFrameClock",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFrameClock"
        })


#endif

-- method Widget::get_halign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Align" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_halign" gtk_widget_get_halign :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets the horizontal alignment of the widget.
-- 
-- For backwards compatibility reasons this method will never return
-- one of the baseline alignments, but instead it will convert it to
-- [enum/@gtk@/.Align.fill] or [enum/@gtk@/.Align.center].
-- 
-- Baselines are not supported for horizontal alignment.
widgetGetHalign ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gtk.Enums.Align
    -- ^ __Returns:__ the horizontal alignment of /@widget@/
widgetGetHalign widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_halign widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHalignMethodInfo
instance (signature ~ (m Gtk.Enums.Align), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHalignMethodInfo a signature where
    overloadedMethod = widgetGetHalign

instance O.OverloadedMethodInfo WidgetGetHalignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHalign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHalign"
        })


#endif

-- method Widget::get_has_tooltip
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_has_tooltip" gtk_widget_get_has_tooltip :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns the current value of the @has-tooltip@ property.
widgetGetHasTooltip ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ current value of @has-tooltip@ on /@widget@/
widgetGetHasTooltip widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_has_tooltip widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHasTooltipMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHasTooltipMethodInfo a signature where
    overloadedMethod = widgetGetHasTooltip

instance O.OverloadedMethodInfo WidgetGetHasTooltipMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHasTooltip",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHasTooltip"
        })


#endif

-- method Widget::get_height
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_height" gtk_widget_get_height :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Returns the content height of the widget.
-- 
-- This function returns the height passed to its
-- size-allocate implementation, which is the height you
-- should be using in t'GI.Gtk.Objects.Widget.Widget'.@/snapshot/@().
-- 
-- For pointer events, see 'GI.Gtk.Objects.Widget.widgetContains'.
-- 
-- To learn more about widget sizes, see the coordinate
-- system <https://docs.gtk.org/gtk4/coordinates.html overview>.
widgetGetHeight ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Int32
    -- ^ __Returns:__ The height of /@widget@/
widgetGetHeight widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_height widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetHeightMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHeightMethodInfo a signature where
    overloadedMethod = widgetGetHeight

instance O.OverloadedMethodInfo WidgetGetHeightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHeight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHeight"
        })


#endif

-- method Widget::get_hexpand
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_hexpand" gtk_widget_get_hexpand :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets whether the widget would like any available extra horizontal
-- space.
-- 
-- When a user resizes a window, widgets with expand set to true generally
-- receive the extra space. For example, a list or scrollable area
-- or document in your window would often be set to expand.
-- 
-- Widgets with children should use 'GI.Gtk.Objects.Widget.widgetComputeExpand'
-- rather than this function, to see whether any of its children,
-- has the expand flag set. If any child of a widget wants to
-- expand, the parent may ask to expand also.
-- 
-- This function only looks at the widget’s own hexpand flag, rather
-- than computing whether the entire widget tree rooted at this widget
-- wants to expand.
widgetGetHexpand ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ whether hexpand flag is set
widgetGetHexpand widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_hexpand widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHexpandMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHexpandMethodInfo a signature where
    overloadedMethod = widgetGetHexpand

instance O.OverloadedMethodInfo WidgetGetHexpandMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHexpand",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHexpand"
        })


#endif

-- method Widget::get_hexpand_set
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_hexpand_set" gtk_widget_get_hexpand_set :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets whether the @hexpand@ flag has been explicitly set.
-- 
-- If [Widget:hexpand]("GI.Gtk.Objects.Widget#g:attr:hexpand") property is set, then it
-- overrides any computed expand value based on child widgets.
-- If @hexpand@ is not set, then the expand value depends on
-- whether any children of the widget would like to expand.
-- 
-- There are few reasons to use this function, but it’s here
-- for completeness and consistency.
widgetGetHexpandSet ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ whether hexpand has been explicitly set
widgetGetHexpandSet widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_hexpand_set widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHexpandSetMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHexpandSetMethodInfo a signature where
    overloadedMethod = widgetGetHexpandSet

instance O.OverloadedMethodInfo WidgetGetHexpandSetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHexpandSet",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHexpandSet"
        })


#endif

-- method Widget::get_last_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_last_child" gtk_widget_get_last_child :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Widget)

-- | Returns the widget’s last child.
-- 
-- This function is primarily meant for widget implementations.
widgetGetLastChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Widget)
    -- ^ __Returns:__ the widget\'s last child
widgetGetLastChild widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_last_child widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetLastChildMethodInfo
instance (signature ~ (m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetLastChildMethodInfo a signature where
    overloadedMethod = widgetGetLastChild

instance O.OverloadedMethodInfo WidgetGetLastChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetLastChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetLastChild"
        })


#endif

-- method Widget::get_layout_manager
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "LayoutManager" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_layout_manager" gtk_widget_get_layout_manager :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.LayoutManager.LayoutManager)

-- | Retrieves the layout manager of the widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetLayoutManager'.
widgetGetLayoutManager ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Gtk.LayoutManager.LayoutManager)
    -- ^ __Returns:__ the layout manager of /@widget@/
widgetGetLayoutManager widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_layout_manager widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gtk.LayoutManager.LayoutManager) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetLayoutManagerMethodInfo
instance (signature ~ (m (Maybe Gtk.LayoutManager.LayoutManager)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetLayoutManagerMethodInfo a signature where
    overloadedMethod = widgetGetLayoutManager

instance O.OverloadedMethodInfo WidgetGetLayoutManagerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetLayoutManager",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetLayoutManager"
        })


#endif

-- method Widget::get_limit_events
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkWidget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_limit_events" gtk_widget_get_limit_events :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets the value of the [Widget:limitEvents]("GI.Gtk.Objects.Widget#g:attr:limitEvents") property.
-- 
-- /Since: 4.18/
widgetGetLimitEvents ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a @GtkWidget@
    -> m Bool
widgetGetLimitEvents widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_limit_events widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetLimitEventsMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetLimitEventsMethodInfo a signature where
    overloadedMethod = widgetGetLimitEvents

instance O.OverloadedMethodInfo WidgetGetLimitEventsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetLimitEvents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetLimitEvents"
        })


#endif

-- method Widget::get_mapped
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_mapped" gtk_widget_get_mapped :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns whether the widget is mapped.
widgetGetMapped ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget is mapped
widgetGetMapped widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_mapped widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetMappedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMappedMethodInfo a signature where
    overloadedMethod = widgetGetMapped

instance O.OverloadedMethodInfo WidgetGetMappedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMapped",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMapped"
        })


#endif

-- method Widget::get_margin_bottom
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_bottom" gtk_widget_get_margin_bottom :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Gets the bottom margin of the widget.
widgetGetMarginBottom ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Int32
    -- ^ __Returns:__ The bottom margin of /@widget@/
widgetGetMarginBottom widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_bottom widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginBottomMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginBottomMethodInfo a signature where
    overloadedMethod = widgetGetMarginBottom

instance O.OverloadedMethodInfo WidgetGetMarginBottomMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginBottom",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginBottom"
        })


#endif

-- method Widget::get_margin_end
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_end" gtk_widget_get_margin_end :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Gets the end margin of the widget.
widgetGetMarginEnd ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Int32
    -- ^ __Returns:__ The end margin of /@widget@/
widgetGetMarginEnd widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_end widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginEndMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginEndMethodInfo a signature where
    overloadedMethod = widgetGetMarginEnd

instance O.OverloadedMethodInfo WidgetGetMarginEndMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginEnd",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginEnd"
        })


#endif

-- method Widget::get_margin_start
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_start" gtk_widget_get_margin_start :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Gets the start margin of the widget.
widgetGetMarginStart ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Int32
    -- ^ __Returns:__ The start margin of /@widget@/
widgetGetMarginStart widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_start widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginStartMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginStartMethodInfo a signature where
    overloadedMethod = widgetGetMarginStart

instance O.OverloadedMethodInfo WidgetGetMarginStartMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginStart",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginStart"
        })


#endif

-- method Widget::get_margin_top
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_top" gtk_widget_get_margin_top :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Gets the top margin of the widget.
widgetGetMarginTop ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Int32
    -- ^ __Returns:__ The top margin of /@widget@/
widgetGetMarginTop widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_top widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginTopMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginTopMethodInfo a signature where
    overloadedMethod = widgetGetMarginTop

instance O.OverloadedMethodInfo WidgetGetMarginTopMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginTop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginTop"
        })


#endif

-- method Widget::get_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_name" gtk_widget_get_name :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CString

-- | Retrieves the name of a widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetName' for the significance of widget names.
widgetGetName ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m T.Text
    -- ^ __Returns:__ name of the widget
widgetGetName widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_name widget'
    checkUnexpectedReturnNULL "widgetGetName" result
    result' <- cstringToText result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetNameMethodInfo a signature where
    overloadedMethod = widgetGetName

instance O.OverloadedMethodInfo WidgetGetNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetName"
        })


#endif

-- method Widget::get_native
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Native" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_native" gtk_widget_get_native :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.Native.Native)

-- | Returns the nearest @GtkNative@ ancestor of the widget.
-- 
-- This function will return @NULL@ if the widget is not
-- contained inside a widget tree with a native ancestor.
-- 
-- @GtkNative@ widgets will return themselves here.
widgetGetNative ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Gtk.Native.Native)
    -- ^ __Returns:__ the @GtkNative@ ancestor of /@widget@/
widgetGetNative widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_native widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gtk.Native.Native) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetNativeMethodInfo
instance (signature ~ (m (Maybe Gtk.Native.Native)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetNativeMethodInfo a signature where
    overloadedMethod = widgetGetNative

instance O.OverloadedMethodInfo WidgetGetNativeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetNative",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetNative"
        })


#endif

-- method Widget::get_next_sibling
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_next_sibling" gtk_widget_get_next_sibling :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Widget)

-- | Returns the widget’s next sibling.
-- 
-- This function is primarily meant for widget implementations.
widgetGetNextSibling ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Widget)
    -- ^ __Returns:__ the widget\'s next sibling
widgetGetNextSibling widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_next_sibling widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetNextSiblingMethodInfo
instance (signature ~ (m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetNextSiblingMethodInfo a signature where
    overloadedMethod = widgetGetNextSibling

instance O.OverloadedMethodInfo WidgetGetNextSiblingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetNextSibling",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetNextSibling"
        })


#endif

-- method Widget::get_opacity
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TDouble)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_opacity" gtk_widget_get_opacity :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CDouble

-- | Fetches the requested opacity for the widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetOpacity'.
widgetGetOpacity ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Double
    -- ^ __Returns:__ the requested opacity for this widget
widgetGetOpacity widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_opacity widget'
    let result' = realToFrac result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetOpacityMethodInfo
instance (signature ~ (m Double), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetOpacityMethodInfo a signature where
    overloadedMethod = widgetGetOpacity

instance O.OverloadedMethodInfo WidgetGetOpacityMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetOpacity",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetOpacity"
        })


#endif

-- method Widget::get_overflow
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Overflow" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_overflow" gtk_widget_get_overflow :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Returns the widget’s overflow value.
widgetGetOverflow ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gtk.Enums.Overflow
    -- ^ __Returns:__ The widget\'s overflow value
widgetGetOverflow widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_overflow widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetOverflowMethodInfo
instance (signature ~ (m Gtk.Enums.Overflow), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetOverflowMethodInfo a signature where
    overloadedMethod = widgetGetOverflow

instance O.OverloadedMethodInfo WidgetGetOverflowMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetOverflow",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetOverflow"
        })


#endif

-- method Widget::get_pango_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Context" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_pango_context" gtk_widget_get_pango_context :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Pango.Context.Context)

-- | Gets a @PangoContext@ that is configured for the widget.
-- 
-- The @PangoContext@ will have the appropriate font map, font description,
-- and base direction set.
-- 
-- Unlike the context returned by 'GI.Gtk.Objects.Widget.widgetCreatePangoContext',
-- this context is owned by the widget (it can be used until the screen
-- for the widget changes or the widget is removed from its toplevel),
-- and will be updated to match any changes to the widget’s attributes.
-- This can be tracked by listening to changes of the
-- [Widget:root]("GI.Gtk.Objects.Widget#g:attr:root") property on the widget.
widgetGetPangoContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Pango.Context.Context
    -- ^ __Returns:__ the @PangoContext@ for the widget
widgetGetPangoContext widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_pango_context widget'
    checkUnexpectedReturnNULL "widgetGetPangoContext" result
    result' <- (newObject Pango.Context.Context) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetPangoContextMethodInfo
instance (signature ~ (m Pango.Context.Context), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPangoContextMethodInfo a signature where
    overloadedMethod = widgetGetPangoContext

instance O.OverloadedMethodInfo WidgetGetPangoContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPangoContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPangoContext"
        })


#endif

-- method Widget::get_parent
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_parent" gtk_widget_get_parent :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Widget)

-- | Returns the parent widget of the widget.
widgetGetParent ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Widget)
    -- ^ __Returns:__ the parent widget of /@widget@/
widgetGetParent widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_parent widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetParentMethodInfo
instance (signature ~ (m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetParentMethodInfo a signature where
    overloadedMethod = widgetGetParent

instance O.OverloadedMethodInfo WidgetGetParentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetParent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetParent"
        })


#endif

-- method Widget::get_preferred_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkWidget` instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum_size"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Requisition" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location for storing the minimum size"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "natural_size"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Requisition" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location for storing the natural size"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_preferred_size" gtk_widget_get_preferred_size :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.Requisition.Requisition ->      -- minimum_size : TInterface (Name {namespace = "Gtk", name = "Requisition"})
    Ptr Gtk.Requisition.Requisition ->      -- natural_size : TInterface (Name {namespace = "Gtk", name = "Requisition"})
    IO ()

-- | Retrieves the minimum and natural size of a widget, taking
-- into account the widget’s preference for height-for-width management.
-- 
-- This is used to retrieve a suitable size by container widgets which do
-- not impose any restrictions on the child placement. It can be used
-- to deduce toplevel window and menu sizes as well as child widgets in
-- free-form containers such as @GtkFixed@.
-- 
-- Handle with care. Note that the natural height of a height-for-width
-- widget will generally be a smaller size than the minimum height, since
-- the required height for the natural width is generally smaller than the
-- required height for the minimum width.
-- 
-- Use 'GI.Gtk.Objects.Widget.widgetMeasure' if you want to support baseline alignment.
widgetGetPreferredSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a @GtkWidget@ instance
    -> m ((Gtk.Requisition.Requisition, Gtk.Requisition.Requisition))
widgetGetPreferredSize widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    minimumSize <- SP.callocBoxedBytes 8 :: IO (Ptr Gtk.Requisition.Requisition)
    naturalSize <- SP.callocBoxedBytes 8 :: IO (Ptr Gtk.Requisition.Requisition)
    gtk_widget_get_preferred_size widget' minimumSize naturalSize
    minimumSize' <- (wrapBoxed Gtk.Requisition.Requisition) minimumSize
    naturalSize' <- (wrapBoxed Gtk.Requisition.Requisition) naturalSize
    touchManagedPtr widget
    return (minimumSize', naturalSize')

#if defined(ENABLE_OVERLOADING)
data WidgetGetPreferredSizeMethodInfo
instance (signature ~ (m ((Gtk.Requisition.Requisition, Gtk.Requisition.Requisition))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPreferredSizeMethodInfo a signature where
    overloadedMethod = widgetGetPreferredSize

instance O.OverloadedMethodInfo WidgetGetPreferredSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPreferredSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPreferredSize"
        })


#endif

-- method Widget::get_prev_sibling
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_prev_sibling" gtk_widget_get_prev_sibling :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Widget)

-- | Returns the widget’s previous sibling.
-- 
-- This function is primarily meant for widget implementations.
widgetGetPrevSibling ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Widget)
    -- ^ __Returns:__ the widget\'s previous sibling
widgetGetPrevSibling widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_prev_sibling widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetPrevSiblingMethodInfo
instance (signature ~ (m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPrevSiblingMethodInfo a signature where
    overloadedMethod = widgetGetPrevSibling

instance O.OverloadedMethodInfo WidgetGetPrevSiblingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPrevSibling",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPrevSibling"
        })


#endif

-- method Widget::get_primary_clipboard
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Clipboard" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_primary_clipboard" gtk_widget_get_primary_clipboard :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Clipboard.Clipboard)

-- | Gets the primary clipboard of the widget.
-- 
-- This is a utility function to get the primary clipboard object
-- for the display that /@widget@/ is using.
-- 
-- Note that this function always works, even when /@widget@/ is not
-- realized yet.
widgetGetPrimaryClipboard ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gdk.Clipboard.Clipboard
    -- ^ __Returns:__ the appropriate clipboard object
widgetGetPrimaryClipboard widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_primary_clipboard widget'
    checkUnexpectedReturnNULL "widgetGetPrimaryClipboard" result
    result' <- (newObject Gdk.Clipboard.Clipboard) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetPrimaryClipboardMethodInfo
instance (signature ~ (m Gdk.Clipboard.Clipboard), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPrimaryClipboardMethodInfo a signature where
    overloadedMethod = widgetGetPrimaryClipboard

instance O.OverloadedMethodInfo WidgetGetPrimaryClipboardMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPrimaryClipboard",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPrimaryClipboard"
        })


#endif

-- method Widget::get_realized
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_realized" gtk_widget_get_realized :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget is realized.
widgetGetRealized ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if /@widget@/ is realized
widgetGetRealized widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_realized widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetRealizedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetRealizedMethodInfo a signature where
    overloadedMethod = widgetGetRealized

instance O.OverloadedMethodInfo WidgetGetRealizedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetRealized",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetRealized"
        })


#endif

-- method Widget::get_receives_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_receives_default" gtk_widget_get_receives_default :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget is always treated as the default widget
-- within its toplevel when it has the focus, even if another widget
-- is the default.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetReceivesDefault'.
widgetGetReceivesDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if /@widget@/ acts as the default widget when focused
widgetGetReceivesDefault widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_receives_default widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetReceivesDefaultMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetReceivesDefaultMethodInfo a signature where
    overloadedMethod = widgetGetReceivesDefault

instance O.OverloadedMethodInfo WidgetGetReceivesDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetReceivesDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetReceivesDefault"
        })


#endif

-- method Widget::get_request_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkWidget` instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "SizeRequestMode" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_request_mode" gtk_widget_get_request_mode :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets whether the widget prefers a height-for-width layout
-- or a width-for-height layout.
-- 
-- Single-child widgets generally propagate the preference of
-- their child, more complex widgets need to request something
-- either in context of their children or in context of their
-- allocation capabilities.
widgetGetRequestMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a @GtkWidget@ instance
    -> m Gtk.Enums.SizeRequestMode
    -- ^ __Returns:__ The @GtkSizeRequestMode@ preferred by /@widget@/.
widgetGetRequestMode widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_request_mode widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetRequestModeMethodInfo
instance (signature ~ (m Gtk.Enums.SizeRequestMode), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetRequestModeMethodInfo a signature where
    overloadedMethod = widgetGetRequestMode

instance O.OverloadedMethodInfo WidgetGetRequestModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetRequestMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetRequestMode"
        })


#endif

-- method Widget::get_root
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Root" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_root" gtk_widget_get_root :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.Root.Root)

-- | Returns the @GtkRoot@ widget of the widget.
-- 
-- This function will return @NULL@ if the widget is not contained
-- inside a widget tree with a root widget.
-- 
-- @GtkRoot@ widgets will return themselves here.
widgetGetRoot ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe Gtk.Root.Root)
    -- ^ __Returns:__ the root widget of /@widget@/
widgetGetRoot widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_root widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gtk.Root.Root) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetRootMethodInfo
instance (signature ~ (m (Maybe Gtk.Root.Root)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetRootMethodInfo a signature where
    overloadedMethod = widgetGetRoot

instance O.OverloadedMethodInfo WidgetGetRootMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetRoot",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetRoot"
        })


#endif

-- method Widget::get_scale_factor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_scale_factor" gtk_widget_get_scale_factor :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Retrieves the internal scale factor that maps from window
-- coordinates to the actual device pixels.
-- 
-- On traditional systems this is 1, on high density outputs,
-- it can be a higher value (typically 2).
-- 
-- See 'GI.Gdk.Objects.Surface.surfaceGetScaleFactor'.
-- 
-- Note that modern systems may support *fractional* scaling,
-- where the scale factor is not an integer. On such systems,
-- this function will return the next higher integer value,
-- but you probably want to use 'GI.Gdk.Objects.Surface.surfaceGetScale'
-- to get the fractional scale value.
widgetGetScaleFactor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Int32
    -- ^ __Returns:__ the scale factor for /@widget@/
widgetGetScaleFactor widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_scale_factor widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetScaleFactorMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetScaleFactorMethodInfo a signature where
    overloadedMethod = widgetGetScaleFactor

instance O.OverloadedMethodInfo WidgetGetScaleFactorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetScaleFactor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetScaleFactor"
        })


#endif

-- method Widget::get_sensitive
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_sensitive" gtk_widget_get_sensitive :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns the widget’s sensitivity.
-- 
-- This function returns the value that has been set using
-- 'GI.Gtk.Objects.Widget.widgetSetSensitive').
-- 
-- The effective sensitivity of a widget is however determined
-- by both its own and its parent widget’s sensitivity.
-- See 'GI.Gtk.Objects.Widget.widgetIsSensitive'.
widgetGetSensitive ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget is sensitive
widgetGetSensitive widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_sensitive widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetSensitiveMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetSensitiveMethodInfo a signature where
    overloadedMethod = widgetGetSensitive

instance O.OverloadedMethodInfo WidgetGetSensitiveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetSensitive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetSensitive"
        })


#endif

-- method Widget::get_settings
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_settings" gtk_widget_get_settings :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.Settings.Settings)

-- | Gets the settings object holding the settings used for the widget.
-- 
-- Note that this function can only be called when the @GtkWidget@
-- is attached to a toplevel, since the settings object is specific
-- to a particular display. If you want to monitor the widget for
-- changes in its settings, connect to the @notify::display@ signal.
widgetGetSettings ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gtk.Settings.Settings
    -- ^ __Returns:__ the relevant settings object
widgetGetSettings widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_settings widget'
    checkUnexpectedReturnNULL "widgetGetSettings" result
    result' <- (newObject Gtk.Settings.Settings) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetSettingsMethodInfo
instance (signature ~ (m Gtk.Settings.Settings), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetSettingsMethodInfo a signature where
    overloadedMethod = widgetGetSettings

instance O.OverloadedMethodInfo WidgetGetSettingsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetSettings",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetSettings"
        })


#endif

-- method Widget::get_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "orientation"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Orientation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the orientation to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_size" gtk_widget_get_size :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- orientation : TInterface (Name {namespace = "Gtk", name = "Orientation"})
    IO Int32

-- | Returns the content width or height of the widget.
-- 
-- Which dimension is returned depends on /@orientation@/.
-- 
-- This is equivalent to calling 'GI.Gtk.Objects.Widget.widgetGetWidth'
-- for [enum/@gtk@/.Orientation.horizontal] or 'GI.Gtk.Objects.Widget.widgetGetHeight'
-- for [enum/@gtk@/.Orientation.vertical], but can be used when
-- writing orientation-independent code, such as when
-- implementing t'GI.Gtk.Interfaces.Orientable.Orientable' widgets.
-- 
-- To learn more about widget sizes, see the coordinate
-- system <https://docs.gtk.org/gtk4/coordinates.html overview>.
widgetGetSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Enums.Orientation
    -- ^ /@orientation@/: the orientation to query
    -> m Int32
    -- ^ __Returns:__ the size of /@widget@/ in /@orientation@/
widgetGetSize widget orientation = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let orientation' = (fromIntegral . fromEnum) orientation
    result <- gtk_widget_get_size widget' orientation'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetSizeMethodInfo
instance (signature ~ (Gtk.Enums.Orientation -> m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetSizeMethodInfo a signature where
    overloadedMethod = widgetGetSize

instance O.OverloadedMethodInfo WidgetGetSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetSize"
        })


#endif

-- method Widget::get_size_request
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for width"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for height"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_size_request" gtk_widget_get_size_request :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Int32 ->                            -- width : TBasicType TInt
    Ptr Int32 ->                            -- height : TBasicType TInt
    IO ()

-- | Gets the size request that was explicitly set for the widget.
-- 
-- A value of -1 stored in /@width@/ or /@height@/ indicates that that
-- dimension has not been set explicitly and the natural requisition
-- of the widget will be used instead.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetSizeRequest'.
-- 
-- To get the size a widget will actually request, call
-- 'GI.Gtk.Objects.Widget.widgetMeasure' instead of this function.
widgetGetSizeRequest ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ((Int32, Int32))
widgetGetSizeRequest widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    width <- allocMem :: IO (Ptr Int32)
    height <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_size_request widget' width height
    width' <- peek width
    height' <- peek height
    touchManagedPtr widget
    freeMem width
    freeMem height
    return (width', height')

#if defined(ENABLE_OVERLOADING)
data WidgetGetSizeRequestMethodInfo
instance (signature ~ (m ((Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetSizeRequestMethodInfo a signature where
    overloadedMethod = widgetGetSizeRequest

instance O.OverloadedMethodInfo WidgetGetSizeRequestMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetSizeRequest",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetSizeRequest"
        })


#endif

-- method Widget::get_state_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "StateFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_state_flags" gtk_widget_get_state_flags :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Returns the widget state as a flag set.
-- 
-- It is worth mentioning that the effective [flags/@gtk@/.StateFlags.insensitive]
-- state will be returned, that is, also based on parent insensitivity,
-- even if /@widget@/ itself is sensitive.
-- 
-- Also note that if you are looking for a way to obtain the
-- [flags/@gtk@/.StateFlags] to pass to a t'GI.Gtk.Objects.StyleContext.StyleContext'
-- method, you should look at 'GI.Gtk.Objects.StyleContext.styleContextGetState'.
widgetGetStateFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m [Gtk.Flags.StateFlags]
    -- ^ __Returns:__ the state flags of widget
widgetGetStateFlags widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_state_flags widget'
    let result' = wordToGFlags result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetStateFlagsMethodInfo
instance (signature ~ (m [Gtk.Flags.StateFlags]), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetStateFlagsMethodInfo a signature where
    overloadedMethod = widgetGetStateFlags

instance O.OverloadedMethodInfo WidgetGetStateFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetStateFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetStateFlags"
        })


#endif

-- method Widget::get_style_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "StyleContext" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_style_context" gtk_widget_get_style_context :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.StyleContext.StyleContext)

{-# DEPRECATED widgetGetStyleContext ["(Since version 4.10)","Style contexts will be removed in GTK 5"] #-}
-- | Returns the style context associated to the widget.
-- 
-- The returned object is guaranteed to be the same
-- for the lifetime of /@widget@/.
widgetGetStyleContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gtk.StyleContext.StyleContext
    -- ^ __Returns:__ the widgets style context
widgetGetStyleContext widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_style_context widget'
    checkUnexpectedReturnNULL "widgetGetStyleContext" result
    result' <- (newObject Gtk.StyleContext.StyleContext) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetStyleContextMethodInfo
instance (signature ~ (m Gtk.StyleContext.StyleContext), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetStyleContextMethodInfo a signature where
    overloadedMethod = widgetGetStyleContext

instance O.OverloadedMethodInfo WidgetGetStyleContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetStyleContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetStyleContext"
        })


#endif

-- method Widget::get_template_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The `GType` to get a template child for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "ID of the child defined in the template XML"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_template_child" gtk_widget_get_template_child :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CGType ->                               -- widget_type : TBasicType TGType
    CString ->                              -- name : TBasicType TUTF8
    IO (Ptr GObject.Object.Object)

-- | Fetches an object build from the template XML for /@widgetType@/ in
-- the widget.
-- 
-- This will only report children which were previously declared
-- with 'GI.Gtk.Structs.WidgetClass.widgetClassBindTemplateChildFull' or one of its
-- variants.
-- 
-- This function is only meant to be called for code which is private
-- to the /@widgetType@/ which declared the child and is meant for language
-- bindings which cannot easily make use of the GObject structure offsets.
widgetGetTemplateChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> GType
    -- ^ /@widgetType@/: The @GType@ to get a template child for
    -> T.Text
    -- ^ /@name@/: ID of the child defined in the template XML
    -> m GObject.Object.Object
    -- ^ __Returns:__ the object built in the template XML with
    --   the id /@name@/
widgetGetTemplateChild widget widgetType name = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let widgetType' = gtypeToCGType widgetType
    name' <- textToCString name
    result <- gtk_widget_get_template_child widget' widgetType' name'
    checkUnexpectedReturnNULL "widgetGetTemplateChild" result
    result' <- (newObject GObject.Object.Object) result
    touchManagedPtr widget
    freeMem name'
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetTemplateChildMethodInfo
instance (signature ~ (GType -> T.Text -> m GObject.Object.Object), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetTemplateChildMethodInfo a signature where
    overloadedMethod = widgetGetTemplateChild

instance O.OverloadedMethodInfo WidgetGetTemplateChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetTemplateChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetTemplateChild"
        })


#endif

-- method Widget::get_tooltip_markup
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_tooltip_markup" gtk_widget_get_tooltip_markup :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CString

-- | Gets the contents of the tooltip for the widget.
-- 
-- If the tooltip has not been set using
-- 'GI.Gtk.Objects.Widget.widgetSetTooltipMarkup', this
-- function returns @NULL@.
widgetGetTooltipMarkup ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the tooltip text
widgetGetTooltipMarkup widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_tooltip_markup widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetTooltipMarkupMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetTooltipMarkupMethodInfo a signature where
    overloadedMethod = widgetGetTooltipMarkup

instance O.OverloadedMethodInfo WidgetGetTooltipMarkupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetTooltipMarkup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetTooltipMarkup"
        })


#endif

-- method Widget::get_tooltip_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_tooltip_text" gtk_widget_get_tooltip_text :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CString

-- | Gets the contents of the tooltip for the widget.
-- 
-- If the /@widget@/\'s tooltip was set using
-- 'GI.Gtk.Objects.Widget.widgetSetTooltipMarkup',
-- this function will return the escaped text.
widgetGetTooltipText ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the tooltip text
widgetGetTooltipText widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_tooltip_text widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetTooltipTextMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetTooltipTextMethodInfo a signature where
    overloadedMethod = widgetGetTooltipText

instance O.OverloadedMethodInfo WidgetGetTooltipTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetTooltipText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetTooltipText"
        })


#endif

-- method Widget::get_valign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Align" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_valign" gtk_widget_get_valign :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets the vertical alignment of the widget.
widgetGetValign ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gtk.Enums.Align
    -- ^ __Returns:__ the vertical alignment of /@widget@/
widgetGetValign widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_valign widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetValignMethodInfo
instance (signature ~ (m Gtk.Enums.Align), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetValignMethodInfo a signature where
    overloadedMethod = widgetGetValign

instance O.OverloadedMethodInfo WidgetGetValignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetValign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetValign"
        })


#endif

-- method Widget::get_vexpand
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_vexpand" gtk_widget_get_vexpand :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets whether the widget would like any available extra vertical
-- space.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetGetHexpand' for more detail.
widgetGetVexpand ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ whether vexpand flag is set
widgetGetVexpand widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_vexpand widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetVexpandMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetVexpandMethodInfo a signature where
    overloadedMethod = widgetGetVexpand

instance O.OverloadedMethodInfo WidgetGetVexpandMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetVexpand",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetVexpand"
        })


#endif

-- method Widget::get_vexpand_set
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_vexpand_set" gtk_widget_get_vexpand_set :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets whether the @vexpand@ flag has been explicitly set.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetGetHexpandSet' for more detail.
widgetGetVexpandSet ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ whether vexpand has been explicitly set
widgetGetVexpandSet widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_vexpand_set widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetVexpandSetMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetVexpandSetMethodInfo a signature where
    overloadedMethod = widgetGetVexpandSet

instance O.OverloadedMethodInfo WidgetGetVexpandSetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetVexpandSet",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetVexpandSet"
        })


#endif

-- method Widget::get_visible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_visible" gtk_widget_get_visible :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget is visible.
-- 
-- If you want to take into account whether the widget’s
-- parent is also marked as visible, use
-- 'GI.Gtk.Objects.Widget.widgetIsVisible' instead.
-- 
-- This function does not check if the widget is
-- obscured in any way.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetVisible'.
widgetGetVisible ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget is visible
widgetGetVisible widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_visible widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetVisibleMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetVisibleMethodInfo a signature where
    overloadedMethod = widgetGetVisible

instance O.OverloadedMethodInfo WidgetGetVisibleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetVisible",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetVisible"
        })


#endif

-- method Widget::get_width
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_width" gtk_widget_get_width :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Returns the content width of the widget.
-- 
-- This function returns the width passed to its
-- size-allocate implementation, which is the width you
-- should be using in t'GI.Gtk.Objects.Widget.Widget'.@/snapshot/@().
-- 
-- For pointer events, see 'GI.Gtk.Objects.Widget.widgetContains'.
-- 
-- To learn more about widget sizes, see the coordinate
-- system <https://docs.gtk.org/gtk4/coordinates.html overview>.
widgetGetWidth ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Int32
    -- ^ __Returns:__ The width of /@widget@/
widgetGetWidth widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_width widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetWidthMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetWidthMethodInfo a signature where
    overloadedMethod = widgetGetWidth

instance O.OverloadedMethodInfo WidgetGetWidthMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetWidth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGetWidth"
        })


#endif

-- method Widget::grab_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_grab_focus" gtk_widget_grab_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Causes /@widget@/ to have the keyboard focus for the window
-- that it belongs to.
-- 
-- If /@widget@/ is not focusable, or its t'GI.Gtk.Objects.Widget.Widget'.@/grab_focus/@()
-- implementation cannot transfer the focus to a descendant of /@widget@/
-- that is focusable, it will not take focus and false will be returned.
-- 
-- Calling 'GI.Gtk.Objects.Widget.widgetGrabFocus' on an already focused widget
-- is allowed, should not have an effect, and return true.
widgetGrabFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if focus is now inside /@widget@/
widgetGrabFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_grab_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGrabFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGrabFocusMethodInfo a signature where
    overloadedMethod = widgetGrabFocus

instance O.OverloadedMethodInfo WidgetGrabFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGrabFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetGrabFocus"
        })


#endif

-- method Widget::has_css_class
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "css_class"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "style class, without the leading period"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_css_class" gtk_widget_has_css_class :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- css_class : TBasicType TUTF8
    IO CInt

-- | Returns whether a style class is currently applied to the widget.
widgetHasCssClass ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> T.Text
    -- ^ /@cssClass@/: style class, without the leading period
    -> m Bool
    -- ^ __Returns:__ true if /@cssClass@/ is currently applied to /@widget@/
widgetHasCssClass widget cssClass = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    cssClass' <- textToCString cssClass
    result <- gtk_widget_has_css_class widget' cssClass'
    let result' = (/= 0) result
    touchManagedPtr widget
    freeMem cssClass'
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasCssClassMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasCssClassMethodInfo a signature where
    overloadedMethod = widgetHasCssClass

instance O.OverloadedMethodInfo WidgetHasCssClassMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasCssClass",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetHasCssClass"
        })


#endif

-- method Widget::has_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_default" gtk_widget_has_default :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget is the current default widget
-- within its toplevel.
widgetHasDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if /@widget@/ is the current default widget
    --   within its toplevel
widgetHasDefault widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_default widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasDefaultMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasDefaultMethodInfo a signature where
    overloadedMethod = widgetHasDefault

instance O.OverloadedMethodInfo WidgetHasDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetHasDefault"
        })


#endif

-- method Widget::has_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_focus" gtk_widget_has_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines if the widget has the global input focus.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetIsFocus' for the difference between
-- having the global input focus, and only having the focus
-- within a toplevel.
widgetHasFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget has the global input focus
widgetHasFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasFocusMethodInfo a signature where
    overloadedMethod = widgetHasFocus

instance O.OverloadedMethodInfo WidgetHasFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetHasFocus"
        })


#endif

-- method Widget::has_visible_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_visible_focus" gtk_widget_has_visible_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines if the widget should show a visible indication that
-- it has the global input focus.
-- 
-- This is a convenience function that takes into account whether
-- focus indication should currently be shown in the toplevel window
-- of /@widget@/. See 'GI.Gtk.Objects.Window.windowGetFocusVisible' for more
-- information about focus indication.
-- 
-- To find out if the widget has the global input focus, use
-- 'GI.Gtk.Objects.Widget.widgetHasFocus'.
widgetHasVisibleFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget should display a “focus rectangle”
widgetHasVisibleFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_visible_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasVisibleFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasVisibleFocusMethodInfo a signature where
    overloadedMethod = widgetHasVisibleFocus

instance O.OverloadedMethodInfo WidgetHasVisibleFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasVisibleFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetHasVisibleFocus"
        })


#endif

-- method Widget::hide
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_hide" gtk_widget_hide :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

{-# DEPRECATED widgetHide ["(Since version 4.10)","Use 'GI.Gtk.Objects.Widget.widgetSetVisible' instead"] #-}
-- | Reverses the effects of [method.Gtk.Widget.show].
-- 
-- This is causing the widget to be hidden (invisible to the user).
widgetHide ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetHide widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_hide widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetHideMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHideMethodInfo a signature where
    overloadedMethod = widgetHide

instance O.OverloadedMethodInfo WidgetHideMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHide",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetHide"
        })


#endif

-- method Widget::in_destruction
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_in_destruction" gtk_widget_in_destruction :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns whether the widget is currently being destroyed.
-- 
-- This information can sometimes be used to avoid doing
-- unnecessary work.
widgetInDestruction ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if /@widget@/ is being destroyed
widgetInDestruction widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_in_destruction widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetInDestructionMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetInDestructionMethodInfo a signature where
    overloadedMethod = widgetInDestruction

instance O.OverloadedMethodInfo WidgetInDestructionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInDestruction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetInDestruction"
        })


#endif

-- method Widget::init_template
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_init_template" gtk_widget_init_template :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Creates and initializes child widgets defined in templates.
-- 
-- This function must be called in the instance initializer
-- for any class which assigned itself a template using
-- 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplate'.
-- 
-- It is important to call this function in the instance initializer
-- of a widget subclass and not in @GObject.constructed()@ or
-- @GObject.constructor()@ for two reasons:
-- 
--  - derived widgets will assume that the composite widgets
--    defined by its parent classes have been created in their
--    relative instance initializers
--  - when calling @g_object_new()@ on a widget with composite templates,
--    it’s important to build the composite widgets before the construct
--    properties are set. Properties passed to @g_object_new()@ should
--    take precedence over properties set in the private template XML
-- 
-- A good rule of thumb is to call this function as the first thing in
-- an instance initialization function.
widgetInitTemplate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetInitTemplate widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_init_template widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetInitTemplateMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetInitTemplateMethodInfo a signature where
    overloadedMethod = widgetInitTemplate

instance O.OverloadedMethodInfo WidgetInitTemplateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInitTemplate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetInitTemplate"
        })


#endif

-- method Widget::insert_action_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the prefix for actions in @group"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an action group" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_insert_action_group" gtk_widget_insert_action_group :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- name : TBasicType TUTF8
    Ptr Gio.ActionGroup.ActionGroup ->      -- group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    IO ()

-- | Inserts an action group into the widget\'s actions.
-- 
-- Children of /@widget@/ that implement t'GI.Gtk.Interfaces.Actionable.Actionable' can
-- then be associated with actions in /@group@/ by setting their
-- “action-name” to /@prefix@/.@action-name@.
-- 
-- Note that inheritance is defined for individual actions. I.e.
-- even if you insert a group with prefix /@prefix@/, actions with
-- the same prefix will still be inherited from the parent, unless
-- the group contains an action with the same name.
-- 
-- If /@group@/ is @NULL@, a previously inserted group for /@name@/ is
-- removed from /@widget@/.
widgetInsertActionGroup ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gio.ActionGroup.IsActionGroup b) =>
    a
    -- ^ /@widget@/: a widget
    -> T.Text
    -- ^ /@name@/: the prefix for actions in /@group@/
    -> Maybe (b)
    -- ^ /@group@/: an action group
    -> m ()
widgetInsertActionGroup widget name group = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    name' <- textToCString name
    maybeGroup <- case group of
        Nothing -> return FP.nullPtr
        Just jGroup -> do
            jGroup' <- unsafeManagedPtrCastPtr jGroup
            return jGroup'
    gtk_widget_insert_action_group widget' name' maybeGroup
    touchManagedPtr widget
    whenJust group touchManagedPtr
    freeMem name'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetInsertActionGroupMethodInfo
instance (signature ~ (T.Text -> Maybe (b) -> m ()), MonadIO m, IsWidget a, Gio.ActionGroup.IsActionGroup b) => O.OverloadedMethod WidgetInsertActionGroupMethodInfo a signature where
    overloadedMethod = widgetInsertActionGroup

instance O.OverloadedMethodInfo WidgetInsertActionGroupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInsertActionGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetInsertActionGroup"
        })


#endif

-- method Widget::insert_after
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parent"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the parent widget to insert @widget into"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "previous_sibling"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new previous sibling of @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_insert_after" gtk_widget_insert_after :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- parent : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- previous_sibling : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Sets the parent widget of the widget.
-- 
-- In contrast to 'GI.Gtk.Objects.Widget.widgetSetParent', this function
-- inserts /@widget@/ at a specific position into the list of children
-- of the /@parent@/ widget.
-- 
-- It will be placed after /@previousSibling@/, or at the beginning if
-- /@previousSibling@/ is @NULL@.
-- 
-- After calling this function, @gtk_widget_get_prev_sibling (widget)@
-- will return /@previousSibling@/.
-- 
-- If /@parent@/ is already set as the parent widget of /@widget@/, this
-- function can also be used to reorder /@widget@/ in the child widget
-- list of /@parent@/.
-- 
-- This function is primarily meant for widget implementations; if you are
-- just using a widget, you *must* use its own API for adding children.
widgetInsertAfter ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b, IsWidget c) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@parent@/: the parent widget to insert /@widget@/ into
    -> Maybe (c)
    -- ^ /@previousSibling@/: the new previous sibling of /@widget@/
    -> m ()
widgetInsertAfter widget parent previousSibling = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    parent' <- unsafeManagedPtrCastPtr parent
    maybePreviousSibling <- case previousSibling of
        Nothing -> return FP.nullPtr
        Just jPreviousSibling -> do
            jPreviousSibling' <- unsafeManagedPtrCastPtr jPreviousSibling
            return jPreviousSibling'
    gtk_widget_insert_after widget' parent' maybePreviousSibling
    touchManagedPtr widget
    touchManagedPtr parent
    whenJust previousSibling touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetInsertAfterMethodInfo
instance (signature ~ (b -> Maybe (c) -> m ()), MonadIO m, IsWidget a, IsWidget b, IsWidget c) => O.OverloadedMethod WidgetInsertAfterMethodInfo a signature where
    overloadedMethod = widgetInsertAfter

instance O.OverloadedMethodInfo WidgetInsertAfterMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInsertAfter",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetInsertAfter"
        })


#endif

-- method Widget::insert_before
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parent"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the parent widget to insert @widget into"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "next_sibling"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new next sibling of @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_insert_before" gtk_widget_insert_before :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- parent : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- next_sibling : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Sets the parent widget of the widget.
-- 
-- In contrast to 'GI.Gtk.Objects.Widget.widgetSetParent', this function
-- inserts /@widget@/ at a specific position into the list of children
-- of the /@parent@/ widget.
-- 
-- It will be placed before /@nextSibling@/, or at the end if
-- /@nextSibling@/ is @NULL@.
-- 
-- After calling this function, @gtk_widget_get_next_sibling (widget)@
-- will return /@nextSibling@/.
-- 
-- If /@parent@/ is already set as the parent widget of /@widget@/, this function
-- can also be used to reorder /@widget@/ in the child widget list of /@parent@/.
-- 
-- This function is primarily meant for widget implementations; if you are
-- just using a widget, you *must* use its own API for adding children.
widgetInsertBefore ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b, IsWidget c) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@parent@/: the parent widget to insert /@widget@/ into
    -> Maybe (c)
    -- ^ /@nextSibling@/: the new next sibling of /@widget@/
    -> m ()
widgetInsertBefore widget parent nextSibling = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    parent' <- unsafeManagedPtrCastPtr parent
    maybeNextSibling <- case nextSibling of
        Nothing -> return FP.nullPtr
        Just jNextSibling -> do
            jNextSibling' <- unsafeManagedPtrCastPtr jNextSibling
            return jNextSibling'
    gtk_widget_insert_before widget' parent' maybeNextSibling
    touchManagedPtr widget
    touchManagedPtr parent
    whenJust nextSibling touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetInsertBeforeMethodInfo
instance (signature ~ (b -> Maybe (c) -> m ()), MonadIO m, IsWidget a, IsWidget b, IsWidget c) => O.OverloadedMethod WidgetInsertBeforeMethodInfo a signature where
    overloadedMethod = widgetInsertBefore

instance O.OverloadedMethodInfo WidgetInsertBeforeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInsertBefore",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetInsertBefore"
        })


#endif

-- method Widget::is_ancestor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "ancestor"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another `GtkWidget`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_ancestor" gtk_widget_is_ancestor :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- ancestor : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget is a descendent of /@ancestor@/.
widgetIsAncestor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@ancestor@/: another @GtkWidget@
    -> m Bool
    -- ^ __Returns:__ true if /@ancestor@/ contains /@widget@/ as a child,
    --   grandchild, great grandchild, etc
widgetIsAncestor widget ancestor = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    ancestor' <- unsafeManagedPtrCastPtr ancestor
    result <- gtk_widget_is_ancestor widget' ancestor'
    let result' = (/= 0) result
    touchManagedPtr widget
    touchManagedPtr ancestor
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsAncestorMethodInfo
instance (signature ~ (b -> m Bool), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetIsAncestorMethodInfo a signature where
    overloadedMethod = widgetIsAncestor

instance O.OverloadedMethodInfo WidgetIsAncestorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsAncestor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetIsAncestor"
        })


#endif

-- method Widget::is_drawable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_drawable" gtk_widget_is_drawable :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget can be drawn to.
-- 
-- A widget can be drawn if it is mapped and visible.
widgetIsDrawable ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if /@widget@/ is drawable
widgetIsDrawable widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_drawable widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsDrawableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsDrawableMethodInfo a signature where
    overloadedMethod = widgetIsDrawable

instance O.OverloadedMethodInfo WidgetIsDrawableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsDrawable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetIsDrawable"
        })


#endif

-- method Widget::is_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_focus" gtk_widget_is_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines if the widget is the focus widget within its
-- toplevel.
-- 
-- This does not mean that the [Widget:hasFocus]("GI.Gtk.Objects.Widget#g:attr:hasFocus")
-- property is necessarily set; [Widget:hasFocus]("GI.Gtk.Objects.Widget#g:attr:hasFocus")
-- will only be set if the toplevel widget additionally has the
-- global input focus.
widgetIsFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget is the focus widget
widgetIsFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsFocusMethodInfo a signature where
    overloadedMethod = widgetIsFocus

instance O.OverloadedMethodInfo WidgetIsFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetIsFocus"
        })


#endif

-- method Widget::is_sensitive
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_sensitive" gtk_widget_is_sensitive :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns the widget’s effective sensitivity.
-- 
-- This means it is sensitive itself and also its
-- parent widget is sensitive.
widgetIsSensitive ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget is effectively sensitive
widgetIsSensitive widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_sensitive widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsSensitiveMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsSensitiveMethodInfo a signature where
    overloadedMethod = widgetIsSensitive

instance O.OverloadedMethodInfo WidgetIsSensitiveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsSensitive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetIsSensitive"
        })


#endif

-- method Widget::is_visible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_visible" gtk_widget_is_visible :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget and all its parents are marked as
-- visible.
-- 
-- This function does not check if the widget is obscured in any way.
-- 
-- See also 'GI.Gtk.Objects.Widget.widgetGetVisible' and
-- 'GI.Gtk.Objects.Widget.widgetSetVisible'.
widgetIsVisible ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if the widget and all its parents are visible
widgetIsVisible widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_visible widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsVisibleMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsVisibleMethodInfo a signature where
    overloadedMethod = widgetIsVisible

instance O.OverloadedMethodInfo WidgetIsVisibleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsVisible",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetIsVisible"
        })


#endif

-- method Widget::keynav_failed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "direction"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DirectionType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "direction of focus movement"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_keynav_failed" gtk_widget_keynav_failed :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- direction : TInterface (Name {namespace = "Gtk", name = "DirectionType"})
    IO CInt

-- | Emits the [Widget::keynavFailed]("GI.Gtk.Objects.Widget#g:signal:keynavFailed") signal on the widget.
-- 
-- This function should be called whenever keyboard navigation
-- within a single widget hits a boundary.
-- 
-- The return value of this function should be interpreted
-- in a way similar to the return value of
-- 'GI.Gtk.Objects.Widget.widgetChildFocus'. When true is returned,
-- stay in the widget, the failed keyboard navigation is ok
-- and\/or there is nowhere we can\/should move the focus to.
-- When false is returned, the caller should continue with
-- keyboard navigation outside the widget, e.g. by calling
-- 'GI.Gtk.Objects.Widget.widgetChildFocus' on the widget’s toplevel.
-- 
-- The default [Widget::keynavFailed]("GI.Gtk.Objects.Widget#g:signal:keynavFailed") handler returns
-- false for [enum/@gtk@/.DirectionType.tab-forward] and
-- [enum/@gtk@/.DirectionType.tab-backward]. For the other values
-- of t'GI.Gtk.Enums.DirectionType' it returns true.
-- 
-- Whenever the default handler returns true, it also calls
-- 'GI.Gtk.Objects.Widget.widgetErrorBell' to notify the user of the
-- failed keyboard navigation.
-- 
-- A use case for providing an own implementation of @::keynav-failed@
-- (either by connecting to it or by overriding it) would be a row of
-- t'GI.Gtk.Objects.Entry.Entry' widgets where the user should be able to navigate
-- the entire row with the cursor keys, as e.g. known from user
-- interfaces that require entering license keys.
widgetKeynavFailed ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Enums.DirectionType
    -- ^ /@direction@/: direction of focus movement
    -> m Bool
    -- ^ __Returns:__ true if stopping keyboard navigation is fine, false
    --   if the emitting widget should try to handle the keyboard
    --   navigation attempt in its parent widget
widgetKeynavFailed widget direction = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let direction' = (fromIntegral . fromEnum) direction
    result <- gtk_widget_keynav_failed widget' direction'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetKeynavFailedMethodInfo
instance (signature ~ (Gtk.Enums.DirectionType -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetKeynavFailedMethodInfo a signature where
    overloadedMethod = widgetKeynavFailed

instance O.OverloadedMethodInfo WidgetKeynavFailedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetKeynavFailed",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetKeynavFailed"
        })


#endif

-- method Widget::list_mnemonic_labels
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList (TInterface Name { namespace = "Gtk" , name = "Widget" }))
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_list_mnemonic_labels" gtk_widget_list_mnemonic_labels :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr (GList (Ptr Widget)))

-- | Returns the widgets for which this widget is the target of a
-- mnemonic.
-- 
-- Typically, these widgets will be labels. See, for example,
-- 'GI.Gtk.Objects.Label.labelSetMnemonicWidget'.
-- 
-- The widgets in the list are not individually referenced.
-- If you want to iterate through the list and perform actions
-- involving callbacks that might destroy the widgets, you
-- must call @g_list_foreach (result, (GFunc)g_object_ref, NULL)@
-- first, and then unref all the widgets afterwards.
widgetListMnemonicLabels ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m [Widget]
    -- ^ __Returns:__ the list
    --   of mnemonic labels
widgetListMnemonicLabels widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_list_mnemonic_labels widget'
    result' <- unpackGList result
    result'' <- mapM (newObject Widget) result'
    g_list_free result
    touchManagedPtr widget
    return result''

#if defined(ENABLE_OVERLOADING)
data WidgetListMnemonicLabelsMethodInfo
instance (signature ~ (m [Widget]), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetListMnemonicLabelsMethodInfo a signature where
    overloadedMethod = widgetListMnemonicLabels

instance O.OverloadedMethodInfo WidgetListMnemonicLabelsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetListMnemonicLabels",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetListMnemonicLabels"
        })


#endif

-- method Widget::map
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_map" gtk_widget_map :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Causes a widget to be mapped if it isn’t already.
-- 
-- This function is only for use in widget implementations.
widgetMap ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetMap widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_map widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetMapMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetMapMethodInfo a signature where
    overloadedMethod = widgetMap

instance O.OverloadedMethodInfo WidgetMapMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetMap",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetMap"
        })


#endif

-- method Widget::measure
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A `GtkWidget` instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "orientation"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Orientation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the orientation to measure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "for_size"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Size for the opposite of @orientation, i.e.\n  if @orientation is %GTK_ORIENTATION_HORIZONTAL, this is\n  the height the widget should be measured with. The %GTK_ORIENTATION_VERTICAL\n  case is analogous. This way, both height-for-width and width-for-height\n  requests can be implemented. If no size is known, -1 can be passed."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the minimum size"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the natural size"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "minimum_baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store the baseline\n  position for the minimum size, or -1 to report no baseline"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural_baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store the baseline\n  position for the natural size, or -1 to report no baseline"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_measure" gtk_widget_measure :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- orientation : TInterface (Name {namespace = "Gtk", name = "Orientation"})
    Int32 ->                                -- for_size : TBasicType TInt
    Ptr Int32 ->                            -- minimum : TBasicType TInt
    Ptr Int32 ->                            -- natural : TBasicType TInt
    Ptr Int32 ->                            -- minimum_baseline : TBasicType TInt
    Ptr Int32 ->                            -- natural_baseline : TBasicType TInt
    IO ()

-- | Measures /@widget@/ in the orientation /@orientation@/ and for the given /@forSize@/.
-- 
-- As an example, if /@orientation@/ is 'GI.Gtk.Enums.OrientationHorizontal' and /@forSize@/
-- is 300, this functions will compute the minimum and natural width of /@widget@/
-- if it is allocated at a height of 300 pixels.
-- 
-- See <https://docs.gtk.org/gtk4/class.Widget.html#height-for-width-geometry-management GtkWidget’s geometry management section> for
-- a more details on implementing @GtkWidgetClass.measure()@.
widgetMeasure ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: A @GtkWidget@ instance
    -> Gtk.Enums.Orientation
    -- ^ /@orientation@/: the orientation to measure
    -> Int32
    -- ^ /@forSize@/: Size for the opposite of /@orientation@/, i.e.
    --   if /@orientation@/ is 'GI.Gtk.Enums.OrientationHorizontal', this is
    --   the height the widget should be measured with. The 'GI.Gtk.Enums.OrientationVertical'
    --   case is analogous. This way, both height-for-width and width-for-height
    --   requests can be implemented. If no size is known, -1 can be passed.
    -> m ((Int32, Int32, Int32, Int32))
widgetMeasure widget orientation forSize = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let orientation' = (fromIntegral . fromEnum) orientation
    minimum <- allocMem :: IO (Ptr Int32)
    natural <- allocMem :: IO (Ptr Int32)
    minimumBaseline <- allocMem :: IO (Ptr Int32)
    naturalBaseline <- allocMem :: IO (Ptr Int32)
    gtk_widget_measure widget' orientation' forSize minimum natural minimumBaseline naturalBaseline
    minimum' <- peek minimum
    natural' <- peek natural
    minimumBaseline' <- peek minimumBaseline
    naturalBaseline' <- peek naturalBaseline
    touchManagedPtr widget
    freeMem minimum
    freeMem natural
    freeMem minimumBaseline
    freeMem naturalBaseline
    return (minimum', natural', minimumBaseline', naturalBaseline')

#if defined(ENABLE_OVERLOADING)
data WidgetMeasureMethodInfo
instance (signature ~ (Gtk.Enums.Orientation -> Int32 -> m ((Int32, Int32, Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetMeasureMethodInfo a signature where
    overloadedMethod = widgetMeasure

instance O.OverloadedMethodInfo WidgetMeasureMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetMeasure",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetMeasure"
        })


#endif

-- method Widget::mnemonic_activate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_cycling"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "true if there are other widgets with the same mnemonic"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_mnemonic_activate" gtk_widget_mnemonic_activate :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- group_cycling : TBasicType TBoolean
    IO CInt

-- | Emits the [Widget::mnemonicActivate]("GI.Gtk.Objects.Widget#g:signal:mnemonicActivate") signal.
widgetMnemonicActivate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@groupCycling@/: true if there are other widgets with the same mnemonic
    -> m Bool
    -- ^ __Returns:__ true if the signal has been handled
widgetMnemonicActivate widget groupCycling = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let groupCycling' = (P.fromIntegral . P.fromEnum) groupCycling
    result <- gtk_widget_mnemonic_activate widget' groupCycling'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetMnemonicActivateMethodInfo
instance (signature ~ (Bool -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetMnemonicActivateMethodInfo a signature where
    overloadedMethod = widgetMnemonicActivate

instance O.OverloadedMethodInfo WidgetMnemonicActivateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetMnemonicActivate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetMnemonicActivate"
        })


#endif

-- method Widget::observe_children
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "ListModel" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_observe_children" gtk_widget_observe_children :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gio.ListModel.ListModel)

-- | Returns a list model to track the children of the widget.
-- 
-- Calling this function will enable extra internal bookkeeping
-- to track children and emit signals on the returned listmodel.
-- It may slow down operations a lot.
-- 
-- Applications should try hard to avoid calling this function
-- because of the slowdowns.
widgetObserveChildren ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gio.ListModel.ListModel
    -- ^ __Returns:__ 
    --   a list model tracking /@widget@/\'s children
widgetObserveChildren widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_observe_children widget'
    checkUnexpectedReturnNULL "widgetObserveChildren" result
    result' <- (wrapObject Gio.ListModel.ListModel) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetObserveChildrenMethodInfo
instance (signature ~ (m Gio.ListModel.ListModel), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetObserveChildrenMethodInfo a signature where
    overloadedMethod = widgetObserveChildren

instance O.OverloadedMethodInfo WidgetObserveChildrenMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetObserveChildren",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetObserveChildren"
        })


#endif

-- method Widget::observe_controllers
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "ListModel" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_observe_controllers" gtk_widget_observe_controllers :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gio.ListModel.ListModel)

-- | Returns a list model to track the event controllers of the widget.
-- 
-- Calling this function will enable extra internal bookkeeping
-- to track controllers and emit signals on the returned listmodel.
-- It may slow down operations a lot.
-- 
-- Applications should try hard to avoid calling this function
-- because of the slowdowns.
widgetObserveControllers ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Gio.ListModel.ListModel
    -- ^ __Returns:__ 
    --   a list model tracking /@widget@/\'s controllers
widgetObserveControllers widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_observe_controllers widget'
    checkUnexpectedReturnNULL "widgetObserveControllers" result
    result' <- (wrapObject Gio.ListModel.ListModel) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetObserveControllersMethodInfo
instance (signature ~ (m Gio.ListModel.ListModel), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetObserveControllersMethodInfo a signature where
    overloadedMethod = widgetObserveControllers

instance O.OverloadedMethodInfo WidgetObserveControllersMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetObserveControllers",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetObserveControllers"
        })


#endif

-- method Widget::pick
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "x coordinate to test, relative to @widget's origin"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "y coordinate to test, relative to @widget's origin"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "PickFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags to influence what is picked"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_pick" gtk_widget_pick :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CDouble ->                              -- x : TBasicType TDouble
    CDouble ->                              -- y : TBasicType TDouble
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gtk", name = "PickFlags"})
    IO (Ptr Widget)

-- | Finds the descendant of the widget closest to a point.
-- 
-- The point (x, y) must be given in widget coordinates, so (0, 0)
-- is assumed to be the top left of /@widget@/\'s content area.
-- 
-- Usually widgets will return @NULL@ if the given coordinate is not
-- contained in /@widget@/ checked via 'GI.Gtk.Objects.Widget.widgetContains'.
-- Otherwise they will recursively try to find a child that does
-- not return @NULL@. Widgets are however free to customize their
-- picking algorithm.
-- 
-- This function is used on the toplevel to determine the widget
-- below the mouse cursor for purposes of hover highlighting and
-- delivering events.
widgetPick ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> Double
    -- ^ /@x@/: x coordinate to test, relative to /@widget@/\'s origin
    -> Double
    -- ^ /@y@/: y coordinate to test, relative to /@widget@/\'s origin
    -> [Gtk.Flags.PickFlags]
    -- ^ /@flags@/: flags to influence what is picked
    -> m (Maybe Widget)
    -- ^ __Returns:__ the widget\'s descendant at (x, y)
widgetPick widget x y flags = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let x' = realToFrac x
    let y' = realToFrac y
    let flags' = gflagsToWord flags
    result <- gtk_widget_pick widget' x' y' flags'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetPickMethodInfo
instance (signature ~ (Double -> Double -> [Gtk.Flags.PickFlags] -> m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetPickMethodInfo a signature where
    overloadedMethod = widgetPick

instance O.OverloadedMethodInfo WidgetPickMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetPick",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetPick"
        })


#endif

-- method Widget::queue_allocate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_queue_allocate" gtk_widget_queue_allocate :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Flags the widget for a rerun of the t'GI.Gtk.Objects.Widget.Widget'.@/size_allocate/@()
-- function.
-- 
-- Use this function instead of 'GI.Gtk.Objects.Widget.widgetQueueResize'
-- when the /@widget@/\'s size request didn\'t change but it wants to
-- reposition its contents.
-- 
-- An example user of this function is 'GI.Gtk.Objects.Widget.widgetSetHalign'.
-- 
-- This function is only for use in widget implementations.
widgetQueueAllocate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetQueueAllocate widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_queue_allocate widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetQueueAllocateMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetQueueAllocateMethodInfo a signature where
    overloadedMethod = widgetQueueAllocate

instance O.OverloadedMethodInfo WidgetQueueAllocateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetQueueAllocate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetQueueAllocate"
        })


#endif

-- method Widget::queue_draw
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_queue_draw" gtk_widget_queue_draw :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Schedules this widget to be redrawn.
-- 
-- The redraw will happen in the paint phase
-- of the current or the next frame.
-- 
-- This means /@widget@/\'s t'GI.Gtk.Objects.Widget.Widget'.@/snapshot/@()
-- implementation will be called.
widgetQueueDraw ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetQueueDraw widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_queue_draw widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetQueueDrawMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetQueueDrawMethodInfo a signature where
    overloadedMethod = widgetQueueDraw

instance O.OverloadedMethodInfo WidgetQueueDrawMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetQueueDraw",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetQueueDraw"
        })


#endif

-- method Widget::queue_resize
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_queue_resize" gtk_widget_queue_resize :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Flags a widget to have its size renegotiated.
-- 
-- This should be called when a widget for some reason has a new
-- size request. For example, when you change the text in a
-- t'GI.Gtk.Objects.Label.Label', the label queues a resize to ensure there’s
-- enough space for the new text.
-- 
-- Note that you cannot call 'GI.Gtk.Objects.Widget.widgetQueueResize' on a widget
-- from inside its implementation of the t'GI.Gtk.Objects.Widget.Widget'.@/size_allocate/@()
-- virtual method. Calls to 'GI.Gtk.Objects.Widget.widgetQueueResize' from inside
-- t'GI.Gtk.Objects.Widget.Widget'.@/size_allocate/@() will be silently ignored.
-- 
-- This function is only for use in widget implementations.
widgetQueueResize ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetQueueResize widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_queue_resize widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetQueueResizeMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetQueueResizeMethodInfo a signature where
    overloadedMethod = widgetQueueResize

instance O.OverloadedMethodInfo WidgetQueueResizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetQueueResize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetQueueResize"
        })


#endif

-- method Widget::realize
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_realize" gtk_widget_realize :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Creates the GDK resources associated with a widget.
-- 
-- Normally realization happens implicitly; if you show a widget
-- and all its parent containers, then the widget will be realized
-- and mapped automatically.
-- 
-- Realizing a widget requires all the widget’s parent widgets to be
-- realized; calling this function realizes the widget’s parents
-- in addition to /@widget@/ itself. If a widget is not yet inside a
-- toplevel window when you realize it, bad things will happen.
-- 
-- This function is primarily used in widget implementations, and
-- isn’t very useful otherwise. Many times when you think you might
-- need it, a better approach is to connect to a signal that will be
-- called after the widget is realized automatically, such as
-- [Widget::realize]("GI.Gtk.Objects.Widget#g:signal:realize").
widgetRealize ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetRealize widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_realize widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetRealizeMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetRealizeMethodInfo a signature where
    overloadedMethod = widgetRealize

instance O.OverloadedMethodInfo WidgetRealizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetRealize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetRealize"
        })


#endif

-- method Widget::remove_controller
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "controller"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "EventController" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an event controller"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_remove_controller" gtk_widget_remove_controller :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.EventController.EventController -> -- controller : TInterface (Name {namespace = "Gtk", name = "EventController"})
    IO ()

-- | Removes an event controller from the widget.
-- 
-- The removed event controller will not receive any more events,
-- and should not be used again.
-- 
-- Widgets will remove all event controllers automatically when they
-- are destroyed, there is normally no need to call this function.
widgetRemoveController ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gtk.EventController.IsEventController b) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@controller@/: an event controller
    -> m ()
widgetRemoveController widget controller = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    controller' <- unsafeManagedPtrCastPtr controller
    gtk_widget_remove_controller widget' controller'
    touchManagedPtr widget
    touchManagedPtr controller
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetRemoveControllerMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsWidget a, Gtk.EventController.IsEventController b) => O.OverloadedMethod WidgetRemoveControllerMethodInfo a signature where
    overloadedMethod = widgetRemoveController

instance O.OverloadedMethodInfo WidgetRemoveControllerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetRemoveController",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetRemoveController"
        })


#endif

-- method Widget::remove_css_class
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "css_class"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "style class to remove from @widget, without the leading period"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_remove_css_class" gtk_widget_remove_css_class :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- css_class : TBasicType TUTF8
    IO ()

-- | Removes a style from the widget.
-- 
-- After this, the style of /@widget@/ will stop matching for /@cssClass@/.
widgetRemoveCssClass ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> T.Text
    -- ^ /@cssClass@/: style class to remove from /@widget@/, without the leading period
    -> m ()
widgetRemoveCssClass widget cssClass = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    cssClass' <- textToCString cssClass
    gtk_widget_remove_css_class widget' cssClass'
    touchManagedPtr widget
    freeMem cssClass'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetRemoveCssClassMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetRemoveCssClassMethodInfo a signature where
    overloadedMethod = widgetRemoveCssClass

instance O.OverloadedMethodInfo WidgetRemoveCssClassMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetRemoveCssClass",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetRemoveCssClass"
        })


#endif

-- method Widget::remove_mnemonic_label
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "label"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget that is a mnemonic label for @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_remove_mnemonic_label" gtk_widget_remove_mnemonic_label :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- label : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Removes a widget from the list of mnemonic labels for this widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetListMnemonicLabels'.
-- 
-- The widget must have previously been added to the list with
-- 'GI.Gtk.Objects.Widget.widgetAddMnemonicLabel'.
widgetRemoveMnemonicLabel ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@label@/: a widget that is a mnemonic label for /@widget@/
    -> m ()
widgetRemoveMnemonicLabel widget label = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    label' <- unsafeManagedPtrCastPtr label
    gtk_widget_remove_mnemonic_label widget' label'
    touchManagedPtr widget
    touchManagedPtr label
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetRemoveMnemonicLabelMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetRemoveMnemonicLabelMethodInfo a signature where
    overloadedMethod = widgetRemoveMnemonicLabel

instance O.OverloadedMethodInfo WidgetRemoveMnemonicLabelMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetRemoveMnemonicLabel",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetRemoveMnemonicLabel"
        })


#endif

-- method Widget::remove_tick_callback
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "id"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "an ID returned by [method@Gtk.Widget.add_tick_callback]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_remove_tick_callback" gtk_widget_remove_tick_callback :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Word32 ->                               -- id : TBasicType TUInt
    IO ()

-- | Removes a tick callback previously registered with
-- 'GI.Gtk.Objects.Widget.widgetAddTickCallback'.
widgetRemoveTickCallback ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Word32
    -- ^ /@id@/: an ID returned by 'GI.Gtk.Objects.Widget.widgetAddTickCallback'
    -> m ()
widgetRemoveTickCallback widget id = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_remove_tick_callback widget' id
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetRemoveTickCallbackMethodInfo
instance (signature ~ (Word32 -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetRemoveTickCallbackMethodInfo a signature where
    overloadedMethod = widgetRemoveTickCallback

instance O.OverloadedMethodInfo WidgetRemoveTickCallbackMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetRemoveTickCallback",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetRemoveTickCallback"
        })


#endif

-- method Widget::set_can_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "can_focus"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "whether the input focus can enter\n  the widget or any of its children"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_can_focus" gtk_widget_set_can_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- can_focus : TBasicType TBoolean
    IO ()

-- | Sets whether the input focus can enter the widget or
-- any of its children.
-- 
-- Applications should set /@canFocus@/ to false to mark a
-- widget as for pointer\/touch use only.
-- 
-- Note that having /@canFocus@/ be true is only one of the
-- necessary conditions for being focusable. A widget must
-- also be sensitive and focusable and not have an ancestor
-- that is marked as not can-focus in order to receive input
-- focus.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetGrabFocus' for actually setting
-- the input focus on a widget.
widgetSetCanFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@canFocus@/: whether the input focus can enter
    --   the widget or any of its children
    -> m ()
widgetSetCanFocus widget canFocus = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let canFocus' = (P.fromIntegral . P.fromEnum) canFocus
    gtk_widget_set_can_focus widget' canFocus'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetCanFocusMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetCanFocusMethodInfo a signature where
    overloadedMethod = widgetSetCanFocus

instance O.OverloadedMethodInfo WidgetSetCanFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetCanFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetCanFocus"
        })


#endif

-- method Widget::set_can_target
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "can_target"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "whether this widget should be able to\n  receive pointer events"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_can_target" gtk_widget_set_can_target :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- can_target : TBasicType TBoolean
    IO ()

-- | Sets whether the widget can be the target of pointer events.
widgetSetCanTarget ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@canTarget@/: whether this widget should be able to
    --   receive pointer events
    -> m ()
widgetSetCanTarget widget canTarget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let canTarget' = (P.fromIntegral . P.fromEnum) canTarget
    gtk_widget_set_can_target widget' canTarget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetCanTargetMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetCanTargetMethodInfo a signature where
    overloadedMethod = widgetSetCanTarget

instance O.OverloadedMethodInfo WidgetSetCanTargetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetCanTarget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetCanTarget"
        })


#endif

-- method Widget::set_child_visible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "child_visible"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "whether @widget should be mapped along\n  with its parent"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_child_visible" gtk_widget_set_child_visible :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- child_visible : TBasicType TBoolean
    IO ()

-- | Sets whether the widget should be mapped along with its parent.
-- 
-- The child visibility can be set for widget before it is added
-- to a container with 'GI.Gtk.Objects.Widget.widgetSetParent', to avoid
-- mapping children unnecessary before immediately unmapping them.
-- However it will be reset to its default state of true when the
-- widget is removed from a container.
-- 
-- Note that changing the child visibility of a widget does not
-- queue a resize on the widget. Most of the time, the size of
-- a widget is computed from all visible children, whether or
-- not they are mapped. If this is not the case, the container
-- can queue a resize itself.
-- 
-- This function is only useful for widget implementations
-- and should never be called by an application.
widgetSetChildVisible ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@childVisible@/: whether /@widget@/ should be mapped along
    --   with its parent
    -> m ()
widgetSetChildVisible widget childVisible = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let childVisible' = (P.fromIntegral . P.fromEnum) childVisible
    gtk_widget_set_child_visible widget' childVisible'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetChildVisibleMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetChildVisibleMethodInfo a signature where
    overloadedMethod = widgetSetChildVisible

instance O.OverloadedMethodInfo WidgetSetChildVisibleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetChildVisible",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetChildVisible"
        })


#endif

-- method Widget::set_css_classes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "classes"
--           , argType = TCArray True (-1) (-1) (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "\n  `NULL`-terminated list of style classes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_css_classes" gtk_widget_set_css_classes :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr CString ->                          -- classes : TCArray True (-1) (-1) (TBasicType TUTF8)
    IO ()

-- | Replaces the current style classes of the widget with /@classes@/.
widgetSetCssClasses ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> [T.Text]
    -- ^ /@classes@/: 
    --   @NULL@-terminated list of style classes
    -> m ()
widgetSetCssClasses widget classes = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    classes' <- packZeroTerminatedUTF8CArray classes
    gtk_widget_set_css_classes widget' classes'
    touchManagedPtr widget
    mapZeroTerminatedCArray freeMem classes'
    freeMem classes'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetCssClassesMethodInfo
instance (signature ~ ([T.Text] -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetCssClassesMethodInfo a signature where
    overloadedMethod = widgetSetCssClasses

instance O.OverloadedMethodInfo WidgetSetCssClassesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetCssClasses",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetCssClasses"
        })


#endif

-- method Widget::set_cursor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cursor"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Cursor" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new cursor" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_cursor" gtk_widget_set_cursor :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Cursor.Cursor ->                -- cursor : TInterface (Name {namespace = "Gdk", name = "Cursor"})
    IO ()

-- | Sets the cursor to be shown when the pointer hovers over
-- the widget.
-- 
-- If the /@cursor@/ is @NULL@, /@widget@/ will use the cursor
-- inherited from its parent.
widgetSetCursor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gdk.Cursor.IsCursor b) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (b)
    -- ^ /@cursor@/: the new cursor
    -> m ()
widgetSetCursor widget cursor = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeCursor <- case cursor of
        Nothing -> return FP.nullPtr
        Just jCursor -> do
            jCursor' <- unsafeManagedPtrCastPtr jCursor
            return jCursor'
    gtk_widget_set_cursor widget' maybeCursor
    touchManagedPtr widget
    whenJust cursor touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetCursorMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsWidget a, Gdk.Cursor.IsCursor b) => O.OverloadedMethod WidgetSetCursorMethodInfo a signature where
    overloadedMethod = widgetSetCursor

instance O.OverloadedMethodInfo WidgetSetCursorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetCursor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetCursor"
        })


#endif

-- method Widget::set_cursor_from_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the cursor"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_cursor_from_name" gtk_widget_set_cursor_from_name :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- name : TBasicType TUTF8
    IO ()

-- | Sets the cursor to be shown when the pointer hovers over
-- the widget.
-- 
-- This is a utility function that creates a cursor via
-- 'GI.Gdk.Objects.Cursor.cursorNewFromName' and then sets it on /@widget@/
-- with 'GI.Gtk.Objects.Widget.widgetSetCursor'. See those functions for
-- details.
-- 
-- On top of that, this function allows /@name@/ to be @NULL@, which
-- will do the same as calling 'GI.Gtk.Objects.Widget.widgetSetCursor'
-- with a @NULL@ cursor.
widgetSetCursorFromName ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (T.Text)
    -- ^ /@name@/: the name of the cursor
    -> m ()
widgetSetCursorFromName widget name = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeName <- case name of
        Nothing -> return FP.nullPtr
        Just jName -> do
            jName' <- textToCString jName
            return jName'
    gtk_widget_set_cursor_from_name widget' maybeName
    touchManagedPtr widget
    freeMem maybeName
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetCursorFromNameMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetCursorFromNameMethodInfo a signature where
    overloadedMethod = widgetSetCursorFromName

instance O.OverloadedMethodInfo WidgetSetCursorFromNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetCursorFromName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetCursorFromName"
        })


#endif

-- method Widget::set_direction
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dir"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TextDirection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new direction" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_direction" gtk_widget_set_direction :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- dir : TInterface (Name {namespace = "Gtk", name = "TextDirection"})
    IO ()

-- | Sets the reading direction on the widget.
-- 
-- This direction controls the primary direction for widgets
-- containing text, and also the direction in which the children
-- of a container are packed. The ability to set the direction is
-- present in order so that correct localization into languages with
-- right-to-left reading directions can be done.
-- 
-- Generally, applications will let the default reading direction
-- prevail, except for widgets where the children are arranged in
-- an order that is explicitly visual rather than logical (such as
-- buttons for text justification).
-- 
-- If the direction is set to [enum/@gtk@/.TextDirection.none], then
-- the value set by [func/@gtk@/.Widget.set_default_direction] will be used.
widgetSetDirection ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Enums.TextDirection
    -- ^ /@dir@/: the new direction
    -> m ()
widgetSetDirection widget dir = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let dir' = (fromIntegral . fromEnum) dir
    gtk_widget_set_direction widget' dir'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetDirectionMethodInfo
instance (signature ~ (Gtk.Enums.TextDirection -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetDirectionMethodInfo a signature where
    overloadedMethod = widgetSetDirection

instance O.OverloadedMethodInfo WidgetSetDirectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetDirection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetDirection"
        })


#endif

-- method Widget::set_focus_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "child"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a direct child widget of @widget\n  or `NULL` to unset the focus child"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_focus_child" gtk_widget_set_focus_child :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- child : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Set the focus child of the widget.
-- 
-- This function is only suitable for widget implementations.
-- If you want a certain widget to get the input focus, call
-- 'GI.Gtk.Objects.Widget.widgetGrabFocus' on it.
widgetSetFocusChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (b)
    -- ^ /@child@/: a direct child widget of /@widget@/
    --   or @NULL@ to unset the focus child
    -> m ()
widgetSetFocusChild widget child = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeChild <- case child of
        Nothing -> return FP.nullPtr
        Just jChild -> do
            jChild' <- unsafeManagedPtrCastPtr jChild
            return jChild'
    gtk_widget_set_focus_child widget' maybeChild
    touchManagedPtr widget
    whenJust child touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetFocusChildMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetSetFocusChildMethodInfo a signature where
    overloadedMethod = widgetSetFocusChild

instance O.OverloadedMethodInfo WidgetSetFocusChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetFocusChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetFocusChild"
        })


#endif

-- method Widget::set_focus_on_click
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "focus_on_click"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "whether the widget should grab focus when clicked\n  with the mouse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_focus_on_click" gtk_widget_set_focus_on_click :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- focus_on_click : TBasicType TBoolean
    IO ()

-- | Sets whether the widget should grab focus when it is clicked
-- with the mouse.
-- 
-- Making mouse clicks not grab focus is useful in places like
-- toolbars where you don’t want the keyboard focus removed from
-- the main area of the application.
widgetSetFocusOnClick ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@focusOnClick@/: whether the widget should grab focus when clicked
    --   with the mouse
    -> m ()
widgetSetFocusOnClick widget focusOnClick = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let focusOnClick' = (P.fromIntegral . P.fromEnum) focusOnClick
    gtk_widget_set_focus_on_click widget' focusOnClick'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetFocusOnClickMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetFocusOnClickMethodInfo a signature where
    overloadedMethod = widgetSetFocusOnClick

instance O.OverloadedMethodInfo WidgetSetFocusOnClickMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetFocusOnClick",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetFocusOnClick"
        })


#endif

-- method Widget::set_focusable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "focusable"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "whether or not @widget can own the input focus"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_focusable" gtk_widget_set_focusable :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- focusable : TBasicType TBoolean
    IO ()

-- | Sets whether the widget can own the input focus.
-- 
-- Widget implementations should set /@focusable@/ to true in
-- their @/init()/@ function if they want to receive keyboard input.
-- 
-- Note that having /@focusable@/ be true is only one of the
-- necessary conditions for being focusable. A widget must
-- also be sensitive and can-focus and not have an ancestor
-- that is marked as not can-focus in order to receive input
-- focus.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetGrabFocus' for actually setting
-- the input focus on a widget.
widgetSetFocusable ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@focusable@/: whether or not /@widget@/ can own the input focus
    -> m ()
widgetSetFocusable widget focusable = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let focusable' = (P.fromIntegral . P.fromEnum) focusable
    gtk_widget_set_focusable widget' focusable'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetFocusableMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetFocusableMethodInfo a signature where
    overloadedMethod = widgetSetFocusable

instance O.OverloadedMethodInfo WidgetSetFocusableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetFocusable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetFocusable"
        })


#endif

-- method Widget::set_font_map
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "font_map"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "FontMap" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoFontMap`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_font_map" gtk_widget_set_font_map :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Pango.FontMap.FontMap ->            -- font_map : TInterface (Name {namespace = "Pango", name = "FontMap"})
    IO ()

-- | Sets the font map to use for text rendering in the widget.
-- 
-- The font map is the object that is used to look up fonts.
-- Setting a custom font map can be useful in special situations,
-- e.g. when you need to add application-specific fonts to the set
-- of available fonts.
-- 
-- When not set, the widget will inherit the font map from its parent.
widgetSetFontMap ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Pango.FontMap.IsFontMap b) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (b)
    -- ^ /@fontMap@/: a @PangoFontMap@
    -> m ()
widgetSetFontMap widget fontMap = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeFontMap <- case fontMap of
        Nothing -> return FP.nullPtr
        Just jFontMap -> do
            jFontMap' <- unsafeManagedPtrCastPtr jFontMap
            return jFontMap'
    gtk_widget_set_font_map widget' maybeFontMap
    touchManagedPtr widget
    whenJust fontMap touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetFontMapMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsWidget a, Pango.FontMap.IsFontMap b) => O.OverloadedMethod WidgetSetFontMapMethodInfo a signature where
    overloadedMethod = widgetSetFontMap

instance O.OverloadedMethodInfo WidgetSetFontMapMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetFontMap",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetFontMap"
        })


#endif

-- method Widget::set_font_options
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "options"
--           , argType =
--               TInterface Name { namespace = "cairo" , name = "FontOptions" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a `cairo_font_options_t` struct\n  to unset any previously set default font options"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_font_options" gtk_widget_set_font_options :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Cairo.FontOptions.FontOptions ->    -- options : TInterface (Name {namespace = "cairo", name = "FontOptions"})
    IO ()

{-# DEPRECATED widgetSetFontOptions ["(Since version 4.16)"] #-}
-- | Sets the @cairo_font_options_t@ used for text rendering
-- in the widget.
-- 
-- When not set, the default font options for the @GdkDisplay@
-- will be used.
widgetSetFontOptions ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (Cairo.FontOptions.FontOptions)
    -- ^ /@options@/: a @cairo_font_options_t@ struct
    --   to unset any previously set default font options
    -> m ()
widgetSetFontOptions widget options = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeOptions <- case options of
        Nothing -> return FP.nullPtr
        Just jOptions -> do
            jOptions' <- unsafeManagedPtrGetPtr jOptions
            return jOptions'
    gtk_widget_set_font_options widget' maybeOptions
    touchManagedPtr widget
    whenJust options touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetFontOptionsMethodInfo
instance (signature ~ (Maybe (Cairo.FontOptions.FontOptions) -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetFontOptionsMethodInfo a signature where
    overloadedMethod = widgetSetFontOptions

instance O.OverloadedMethodInfo WidgetSetFontOptionsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetFontOptions",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetFontOptions"
        })


#endif

-- method Widget::set_halign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "align"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Align" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the horizontal alignment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_halign" gtk_widget_set_halign :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- align : TInterface (Name {namespace = "Gtk", name = "Align"})
    IO ()

-- | Sets the horizontal alignment of the widget.
widgetSetHalign ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Enums.Align
    -- ^ /@align@/: the horizontal alignment
    -> m ()
widgetSetHalign widget align = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let align' = (fromIntegral . fromEnum) align
    gtk_widget_set_halign widget' align'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetHalignMethodInfo
instance (signature ~ (Gtk.Enums.Align -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetHalignMethodInfo a signature where
    overloadedMethod = widgetSetHalign

instance O.OverloadedMethodInfo WidgetSetHalignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetHalign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetHalign"
        })


#endif

-- method Widget::set_has_tooltip
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "has_tooltip"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether or not @widget has a tooltip"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_has_tooltip" gtk_widget_set_has_tooltip :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- has_tooltip : TBasicType TBoolean
    IO ()

-- | Sets the @has-tooltip@ property on the widget.
widgetSetHasTooltip ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@hasTooltip@/: whether or not /@widget@/ has a tooltip
    -> m ()
widgetSetHasTooltip widget hasTooltip = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let hasTooltip' = (P.fromIntegral . P.fromEnum) hasTooltip
    gtk_widget_set_has_tooltip widget' hasTooltip'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetHasTooltipMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetHasTooltipMethodInfo a signature where
    overloadedMethod = widgetSetHasTooltip

instance O.OverloadedMethodInfo WidgetSetHasTooltipMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetHasTooltip",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetHasTooltip"
        })


#endif

-- method Widget::set_hexpand
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "expand"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether to expand" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_hexpand" gtk_widget_set_hexpand :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- expand : TBasicType TBoolean
    IO ()

-- | Sets whether the widget would like any available extra horizontal
-- space.
-- 
-- When a user resizes a window, widgets with expand set to true generally
-- receive the extra space. For example, a list or scrollable area
-- or document in your window would often be set to expand.
-- 
-- Call this function to set the expand flag if you would like your
-- widget to become larger horizontally when the window has extra
-- room.
-- 
-- By default, widgets automatically expand if any of their children
-- want to expand. (To see if a widget will automatically expand given
-- its current children and state, call 'GI.Gtk.Objects.Widget.widgetComputeExpand'.
-- A widget can decide how the expandability of children affects its
-- own expansion by overriding the @compute_expand@ virtual method on
-- @GtkWidget@.).
-- 
-- Setting hexpand explicitly with this function will override the
-- automatic expand behavior.
-- 
-- This function forces the widget to expand or not to expand,
-- regardless of children. The override occurs because
-- 'GI.Gtk.Objects.Widget.widgetSetHexpand' sets the hexpand-set property (see
-- 'GI.Gtk.Objects.Widget.widgetSetHexpandSet') which causes the widget’s hexpand
-- value to be used, rather than looking at children and widget state.
widgetSetHexpand ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@expand@/: whether to expand
    -> m ()
widgetSetHexpand widget expand = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let expand' = (P.fromIntegral . P.fromEnum) expand
    gtk_widget_set_hexpand widget' expand'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetHexpandMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetHexpandMethodInfo a signature where
    overloadedMethod = widgetSetHexpand

instance O.OverloadedMethodInfo WidgetSetHexpandMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetHexpand",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetHexpand"
        })


#endif

-- method Widget::set_hexpand_set
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "set"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "value for hexpand-set property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_hexpand_set" gtk_widget_set_hexpand_set :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- set : TBasicType TBoolean
    IO ()

-- | Sets whether the hexpand flag will be used.
-- 
-- The [Widget:hexpandSet]("GI.Gtk.Objects.Widget#g:attr:hexpandSet") property will be set
-- automatically when you call 'GI.Gtk.Objects.Widget.widgetSetHexpand'
-- to set hexpand, so the most likely reason to use this function
-- would be to unset an explicit expand flag.
-- 
-- If hexpand is set, then it overrides any computed
-- expand value based on child widgets. If hexpand is not
-- set, then the expand value depends on whether any
-- children of the widget would like to expand.
-- 
-- There are few reasons to use this function, but it’s here
-- for completeness and consistency.
widgetSetHexpandSet ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@set@/: value for hexpand-set property
    -> m ()
widgetSetHexpandSet widget set = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let set' = (P.fromIntegral . P.fromEnum) set
    gtk_widget_set_hexpand_set widget' set'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetHexpandSetMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetHexpandSetMethodInfo a signature where
    overloadedMethod = widgetSetHexpandSet

instance O.OverloadedMethodInfo WidgetSetHexpandSetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetHexpandSet",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetHexpandSet"
        })


#endif

-- method Widget::set_layout_manager
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "layout_manager"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "LayoutManager" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a layout manager" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_layout_manager" gtk_widget_set_layout_manager :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.LayoutManager.LayoutManager ->  -- layout_manager : TInterface (Name {namespace = "Gtk", name = "LayoutManager"})
    IO ()

-- | Sets the layout manager to use for measuring and allocating children
-- of the widget.
widgetSetLayoutManager ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gtk.LayoutManager.IsLayoutManager b) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (b)
    -- ^ /@layoutManager@/: a layout manager
    -> m ()
widgetSetLayoutManager widget layoutManager = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeLayoutManager <- case layoutManager of
        Nothing -> return FP.nullPtr
        Just jLayoutManager -> do
            jLayoutManager' <- B.ManagedPtr.disownObject jLayoutManager
            return jLayoutManager'
    gtk_widget_set_layout_manager widget' maybeLayoutManager
    touchManagedPtr widget
    whenJust layoutManager touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetLayoutManagerMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsWidget a, Gtk.LayoutManager.IsLayoutManager b) => O.OverloadedMethod WidgetSetLayoutManagerMethodInfo a signature where
    overloadedMethod = widgetSetLayoutManager

instance O.OverloadedMethodInfo WidgetSetLayoutManagerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetLayoutManager",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetLayoutManager"
        })


#endif

-- method Widget::set_limit_events
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkWidget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "limit_events"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether to limit events"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_limit_events" gtk_widget_set_limit_events :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- limit_events : TBasicType TBoolean
    IO ()

-- | Sets whether the widget acts like a modal dialog,
-- with respect to event delivery.
-- 
-- /Since: 4.18/
widgetSetLimitEvents ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a @GtkWidget@
    -> Bool
    -- ^ /@limitEvents@/: whether to limit events
    -> m ()
widgetSetLimitEvents widget limitEvents = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let limitEvents' = (P.fromIntegral . P.fromEnum) limitEvents
    gtk_widget_set_limit_events widget' limitEvents'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetLimitEventsMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetLimitEventsMethodInfo a signature where
    overloadedMethod = widgetSetLimitEvents

instance O.OverloadedMethodInfo WidgetSetLimitEventsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetLimitEvents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetLimitEvents"
        })


#endif

-- method Widget::set_margin_bottom
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "margin"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the bottom margin" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_margin_bottom" gtk_widget_set_margin_bottom :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- margin : TBasicType TInt
    IO ()

-- | Sets the bottom margin of the widget.
widgetSetMarginBottom ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Int32
    -- ^ /@margin@/: the bottom margin
    -> m ()
widgetSetMarginBottom widget margin = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_set_margin_bottom widget' margin
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetMarginBottomMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetMarginBottomMethodInfo a signature where
    overloadedMethod = widgetSetMarginBottom

instance O.OverloadedMethodInfo WidgetSetMarginBottomMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetMarginBottom",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetMarginBottom"
        })


#endif

-- method Widget::set_margin_end
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "margin"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the end margin" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_margin_end" gtk_widget_set_margin_end :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- margin : TBasicType TInt
    IO ()

-- | Sets the end margin of the widget.
widgetSetMarginEnd ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Int32
    -- ^ /@margin@/: the end margin
    -> m ()
widgetSetMarginEnd widget margin = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_set_margin_end widget' margin
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetMarginEndMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetMarginEndMethodInfo a signature where
    overloadedMethod = widgetSetMarginEnd

instance O.OverloadedMethodInfo WidgetSetMarginEndMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetMarginEnd",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetMarginEnd"
        })


#endif

-- method Widget::set_margin_start
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "margin"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the start margin" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_margin_start" gtk_widget_set_margin_start :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- margin : TBasicType TInt
    IO ()

-- | Sets the start margin of the widget.
widgetSetMarginStart ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Int32
    -- ^ /@margin@/: the start margin
    -> m ()
widgetSetMarginStart widget margin = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_set_margin_start widget' margin
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetMarginStartMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetMarginStartMethodInfo a signature where
    overloadedMethod = widgetSetMarginStart

instance O.OverloadedMethodInfo WidgetSetMarginStartMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetMarginStart",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetMarginStart"
        })


#endif

-- method Widget::set_margin_top
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "margin"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the top margin" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_margin_top" gtk_widget_set_margin_top :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- margin : TBasicType TInt
    IO ()

-- | Sets the top margin of the widget.
widgetSetMarginTop ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Int32
    -- ^ /@margin@/: the top margin
    -> m ()
widgetSetMarginTop widget margin = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_set_margin_top widget' margin
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetMarginTopMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetMarginTopMethodInfo a signature where
    overloadedMethod = widgetSetMarginTop

instance O.OverloadedMethodInfo WidgetSetMarginTopMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetMarginTop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetMarginTop"
        })


#endif

-- method Widget::set_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name for the widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_name" gtk_widget_set_name :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- name : TBasicType TUTF8
    IO ()

-- | Sets a widgets name.
-- 
-- Setting a name allows you to refer to the widget from a
-- CSS file. You can apply a style to widgets with a particular name
-- in the CSS file. See the documentation for the CSS syntax (on the
-- same page as the docs for t'GI.Gtk.Objects.StyleContext.StyleContext'.
-- 
-- Note that the CSS syntax has certain special characters to delimit
-- and represent elements in a selector (period, #, >, *...), so using
-- these will make your widget impossible to match by name. Any combination
-- of alphanumeric symbols, dashes and underscores will suffice.
widgetSetName ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> T.Text
    -- ^ /@name@/: name for the widget
    -> m ()
widgetSetName widget name = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    name' <- textToCString name
    gtk_widget_set_name widget' name'
    touchManagedPtr widget
    freeMem name'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetNameMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetNameMethodInfo a signature where
    overloadedMethod = widgetSetName

instance O.OverloadedMethodInfo WidgetSetNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetName"
        })


#endif

-- method Widget::set_opacity
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "opacity"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "desired opacity, between 0 and 1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_opacity" gtk_widget_set_opacity :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CDouble ->                              -- opacity : TBasicType TDouble
    IO ()

-- | Requests the widget to be rendered partially transparent.
-- 
-- An opacity of 0 is fully transparent and an opacity of 1
-- is fully opaque.
-- 
-- Opacity works on both toplevel widgets and child widgets, although
-- there are some limitations: For toplevel widgets, applying opacity
-- depends on the capabilities of the windowing system. On X11, this
-- has any effect only on X displays with a compositing manager, see
-- 'GI.Gdk.Objects.Display.displayIsComposited'. On Windows and Wayland it will
-- always work, although setting a window’s opacity after the window
-- has been shown may cause some flicker.
-- 
-- Note that the opacity is inherited through inclusion — if you set
-- a toplevel to be partially translucent, all of its content will
-- appear translucent, since it is ultimatively rendered on that
-- toplevel. The opacity value itself is not inherited by child
-- widgets (since that would make widgets deeper in the hierarchy
-- progressively more translucent). As a consequence, t'GI.Gtk.Objects.Popover.Popover'
-- instances and other t'GI.Gtk.Interfaces.Native.Native' widgets with their own surface
-- will use their own opacity value, and thus by default appear
-- non-translucent, even if they are attached to a toplevel that
-- is translucent.
widgetSetOpacity ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Double
    -- ^ /@opacity@/: desired opacity, between 0 and 1
    -> m ()
widgetSetOpacity widget opacity = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let opacity' = realToFrac opacity
    gtk_widget_set_opacity widget' opacity'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetOpacityMethodInfo
instance (signature ~ (Double -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetOpacityMethodInfo a signature where
    overloadedMethod = widgetSetOpacity

instance O.OverloadedMethodInfo WidgetSetOpacityMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetOpacity",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetOpacity"
        })


#endif

-- method Widget::set_overflow
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "overflow"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Overflow" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "desired overflow value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_overflow" gtk_widget_set_overflow :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- overflow : TInterface (Name {namespace = "Gtk", name = "Overflow"})
    IO ()

-- | Sets how the widget treats content that is drawn outside the
-- it\'s content area.
-- 
-- See the definition of t'GI.Gtk.Enums.Overflow' for details.
-- 
-- This setting is provided for widget implementations and
-- should not be used by application code.
-- 
-- The default value is [enum/@gtk@/.Overflow.visible].
widgetSetOverflow ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Enums.Overflow
    -- ^ /@overflow@/: desired overflow value
    -> m ()
widgetSetOverflow widget overflow = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let overflow' = (fromIntegral . fromEnum) overflow
    gtk_widget_set_overflow widget' overflow'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetOverflowMethodInfo
instance (signature ~ (Gtk.Enums.Overflow -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetOverflowMethodInfo a signature where
    overloadedMethod = widgetSetOverflow

instance O.OverloadedMethodInfo WidgetSetOverflowMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetOverflow",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetOverflow"
        })


#endif

-- method Widget::set_parent
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parent"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "parent widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_parent" gtk_widget_set_parent :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- parent : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Sets the parent widget of the widget.
-- 
-- This takes care of details such as updating the state and style
-- of the child to reflect its new location and resizing the parent.
-- The opposite function is 'GI.Gtk.Objects.Widget.widgetUnparent'.
-- 
-- This function is useful only when implementing subclasses of
-- @GtkWidget@.
widgetSetParent ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@parent@/: parent widget
    -> m ()
widgetSetParent widget parent = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    parent' <- unsafeManagedPtrCastPtr parent
    gtk_widget_set_parent widget' parent'
    touchManagedPtr widget
    touchManagedPtr parent
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetParentMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetSetParentMethodInfo a signature where
    overloadedMethod = widgetSetParent

instance O.OverloadedMethodInfo WidgetSetParentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetParent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetParent"
        })


#endif

-- method Widget::set_receives_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "receives_default"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "whether or not @widget can be a default widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_receives_default" gtk_widget_set_receives_default :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- receives_default : TBasicType TBoolean
    IO ()

-- | Sets whether the widget will be treated as the default
-- widget within its toplevel when it has the focus, even if
-- another widget is the default.
widgetSetReceivesDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@receivesDefault@/: whether or not /@widget@/ can be a default widget
    -> m ()
widgetSetReceivesDefault widget receivesDefault = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let receivesDefault' = (P.fromIntegral . P.fromEnum) receivesDefault
    gtk_widget_set_receives_default widget' receivesDefault'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetReceivesDefaultMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetReceivesDefaultMethodInfo a signature where
    overloadedMethod = widgetSetReceivesDefault

instance O.OverloadedMethodInfo WidgetSetReceivesDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetReceivesDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetReceivesDefault"
        })


#endif

-- method Widget::set_sensitive
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "sensitive"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "true to make the widget sensitive"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_sensitive" gtk_widget_set_sensitive :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- sensitive : TBasicType TBoolean
    IO ()

-- | Sets the sensitivity of the widget.
-- 
-- A widget is sensitive if the user can interact with it.
-- Insensitive widgets are “grayed out” and the user can’t
-- interact with them. Insensitive widgets are known as
-- “inactive”, “disabled”, or “ghosted” in some other toolkits.
widgetSetSensitive ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@sensitive@/: true to make the widget sensitive
    -> m ()
widgetSetSensitive widget sensitive = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let sensitive' = (P.fromIntegral . P.fromEnum) sensitive
    gtk_widget_set_sensitive widget' sensitive'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetSensitiveMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetSensitiveMethodInfo a signature where
    overloadedMethod = widgetSetSensitive

instance O.OverloadedMethodInfo WidgetSetSensitiveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetSensitive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetSensitive"
        })


#endif

-- method Widget::set_size_request
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "width @widget should request, or -1 to unset"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "height @widget should request, or -1 to unset"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_size_request" gtk_widget_set_size_request :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- width : TBasicType TInt
    Int32 ->                                -- height : TBasicType TInt
    IO ()

-- | Sets the minimum size of the widget.
-- 
-- That is, the widget’s size request will be at least /@width@/
-- by /@height@/. You can use this function to force a widget to
-- be larger than it normally would be.
-- 
-- In most cases, 'GI.Gtk.Objects.Window.windowSetDefaultSize' is a better
-- choice for toplevel windows than this function; setting the default
-- size will still allow users to shrink the window. Setting the size
-- request will force them to leave the window at least as large as
-- the size request.
-- 
-- Note the inherent danger of setting any fixed size - themes,
-- translations into other languages, different fonts, and user action
-- can all change the appropriate size for a given widget. So, it is
-- basically impossible to hardcode a size that will always work.
-- 
-- The size request of a widget is the smallest size a widget can
-- accept while still functioning well and drawing itself correctly.
-- However in some strange cases a widget may be allocated less than
-- its requested size, and in many cases a widget may be allocated more
-- space than it requested.
-- 
-- If the size request in a given direction is -1 (unset), then
-- the “natural” size request of the widget will be used instead.
-- 
-- The size request set here does not include any margin from the
-- properties
-- [Widget:marginStart]("GI.Gtk.Objects.Widget#g:attr:marginStart"),
-- [Widget:marginEnd]("GI.Gtk.Objects.Widget#g:attr:marginEnd"),
-- [Widget:marginTop]("GI.Gtk.Objects.Widget#g:attr:marginTop"), and
-- [Widget:marginBottom]("GI.Gtk.Objects.Widget#g:attr:marginBottom"), but it does include pretty
-- much all other padding or border properties set by any subclass
-- of @GtkWidget@.
widgetSetSizeRequest ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Int32
    -- ^ /@width@/: width /@widget@/ should request, or -1 to unset
    -> Int32
    -- ^ /@height@/: height /@widget@/ should request, or -1 to unset
    -> m ()
widgetSetSizeRequest widget width height = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_set_size_request widget' width height
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetSizeRequestMethodInfo
instance (signature ~ (Int32 -> Int32 -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetSizeRequestMethodInfo a signature where
    overloadedMethod = widgetSetSizeRequest

instance O.OverloadedMethodInfo WidgetSetSizeRequestMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetSizeRequest",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetSizeRequest"
        })


#endif

-- method Widget::set_state_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "StateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "state flags to turn on"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "clear"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "whether to clear state before turning on @flags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_state_flags" gtk_widget_set_state_flags :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gtk", name = "StateFlags"})
    CInt ->                                 -- clear : TBasicType TBoolean
    IO ()

-- | Turns on flag values in the current widget state.
-- 
-- Typical widget states are insensitive, prelighted, etc.
-- 
-- This function accepts the values [flags/@gtk@/.StateFlags.dir-ltr] and
-- [flags/@gtk@/.StateFlags.dir-rtl] but ignores them. If you want to set
-- the widget\'s direction, use 'GI.Gtk.Objects.Widget.widgetSetDirection'.
-- 
-- This function is for use in widget implementations.
widgetSetStateFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> [Gtk.Flags.StateFlags]
    -- ^ /@flags@/: state flags to turn on
    -> Bool
    -- ^ /@clear@/: whether to clear state before turning on /@flags@/
    -> m ()
widgetSetStateFlags widget flags clear = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let flags' = gflagsToWord flags
    let clear' = (P.fromIntegral . P.fromEnum) clear
    gtk_widget_set_state_flags widget' flags' clear'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetStateFlagsMethodInfo
instance (signature ~ ([Gtk.Flags.StateFlags] -> Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetStateFlagsMethodInfo a signature where
    overloadedMethod = widgetSetStateFlags

instance O.OverloadedMethodInfo WidgetSetStateFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetStateFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetStateFlags"
        })


#endif

-- method Widget::set_tooltip_markup
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "markup"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the contents of the tooltip for @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_tooltip_markup" gtk_widget_set_tooltip_markup :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- markup : TBasicType TUTF8
    IO ()

-- | Sets the contents of the tooltip for widget.
-- 
-- /@markup@/ must contain Pango markup.
-- 
-- This function will take care of setting the
-- [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip") as a side effect, and of the
-- default handler for the [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") signal.
-- 
-- See also 'GI.Gtk.Objects.Tooltip.tooltipSetMarkup'.
widgetSetTooltipMarkup ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (T.Text)
    -- ^ /@markup@/: the contents of the tooltip for /@widget@/
    -> m ()
widgetSetTooltipMarkup widget markup = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeMarkup <- case markup of
        Nothing -> return FP.nullPtr
        Just jMarkup -> do
            jMarkup' <- textToCString jMarkup
            return jMarkup'
    gtk_widget_set_tooltip_markup widget' maybeMarkup
    touchManagedPtr widget
    freeMem maybeMarkup
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetTooltipMarkupMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetTooltipMarkupMethodInfo a signature where
    overloadedMethod = widgetSetTooltipMarkup

instance O.OverloadedMethodInfo WidgetSetTooltipMarkupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetTooltipMarkup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetTooltipMarkup"
        })


#endif

-- method Widget::set_tooltip_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the contents of the tooltip for @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_tooltip_text" gtk_widget_set_tooltip_text :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- text : TBasicType TUTF8
    IO ()

-- | Sets the contents of the tooltip for the widget.
-- 
-- If /@text@/ contains any markup, it will be escaped.
-- 
-- This function will take care of setting
-- [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip") as a side effect,
-- and of the default handler for the
-- [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") signal.
-- 
-- See also 'GI.Gtk.Objects.Tooltip.tooltipSetText'.
widgetSetTooltipText ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Maybe (T.Text)
    -- ^ /@text@/: the contents of the tooltip for /@widget@/
    -> m ()
widgetSetTooltipText widget text = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeText <- case text of
        Nothing -> return FP.nullPtr
        Just jText -> do
            jText' <- textToCString jText
            return jText'
    gtk_widget_set_tooltip_text widget' maybeText
    touchManagedPtr widget
    freeMem maybeText
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetTooltipTextMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetTooltipTextMethodInfo a signature where
    overloadedMethod = widgetSetTooltipText

instance O.OverloadedMethodInfo WidgetSetTooltipTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetTooltipText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetTooltipText"
        })


#endif

-- method Widget::set_valign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "align"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Align" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the vertical alignment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_valign" gtk_widget_set_valign :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- align : TInterface (Name {namespace = "Gtk", name = "Align"})
    IO ()

-- | Sets the vertical alignment of the widget.
widgetSetValign ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gtk.Enums.Align
    -- ^ /@align@/: the vertical alignment
    -> m ()
widgetSetValign widget align = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let align' = (fromIntegral . fromEnum) align
    gtk_widget_set_valign widget' align'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetValignMethodInfo
instance (signature ~ (Gtk.Enums.Align -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetValignMethodInfo a signature where
    overloadedMethod = widgetSetValign

instance O.OverloadedMethodInfo WidgetSetValignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetValign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetValign"
        })


#endif

-- method Widget::set_vexpand
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "expand"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether to expand" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_vexpand" gtk_widget_set_vexpand :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- expand : TBasicType TBoolean
    IO ()

-- | Sets whether the widget would like any available extra vertical
-- space.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetHexpand' for more detail.
widgetSetVexpand ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@expand@/: whether to expand
    -> m ()
widgetSetVexpand widget expand = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let expand' = (P.fromIntegral . P.fromEnum) expand
    gtk_widget_set_vexpand widget' expand'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetVexpandMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetVexpandMethodInfo a signature where
    overloadedMethod = widgetSetVexpand

instance O.OverloadedMethodInfo WidgetSetVexpandMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetVexpand",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetVexpand"
        })


#endif

-- method Widget::set_vexpand_set
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "set"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "value for vexpand-set property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_vexpand_set" gtk_widget_set_vexpand_set :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- set : TBasicType TBoolean
    IO ()

-- | Sets whether the vexpand flag will be used.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetHexpandSet' for more detail.
widgetSetVexpandSet ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget
    -> Bool
    -- ^ /@set@/: value for vexpand-set property
    -> m ()
widgetSetVexpandSet widget set = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let set' = (P.fromIntegral . P.fromEnum) set
    gtk_widget_set_vexpand_set widget' set'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetVexpandSetMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetVexpandSetMethodInfo a signature where
    overloadedMethod = widgetSetVexpandSet

instance O.OverloadedMethodInfo WidgetSetVexpandSetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetVexpandSet",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetVexpandSet"
        })


#endif

-- method Widget::set_visible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "visible"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether the widget should be shown or not"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_visible" gtk_widget_set_visible :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- visible : TBasicType TBoolean
    IO ()

-- | Sets the visibility state of /@widget@/.
-- 
-- Note that setting this to true doesn’t mean the widget is
-- actually viewable, see 'GI.Gtk.Objects.Widget.widgetGetVisible'.
widgetSetVisible ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Bool
    -- ^ /@visible@/: whether the widget should be shown or not
    -> m ()
widgetSetVisible widget visible = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let visible' = (P.fromIntegral . P.fromEnum) visible
    gtk_widget_set_visible widget' visible'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSetVisibleMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSetVisibleMethodInfo a signature where
    overloadedMethod = widgetSetVisible

instance O.OverloadedMethodInfo WidgetSetVisibleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSetVisible",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSetVisible"
        })


#endif

-- method Widget::should_layout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_should_layout" gtk_widget_should_layout :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns whether the widget should contribute to
-- the measuring and allocation of its parent.
-- 
-- This is false for invisible children, but also
-- for children that have their own surface, such
-- as t'GI.Gtk.Objects.Popover.Popover' instances.
widgetShouldLayout ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m Bool
    -- ^ __Returns:__ true if child should be included in
    --   measuring and allocating
widgetShouldLayout widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_should_layout widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetShouldLayoutMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetShouldLayoutMethodInfo a signature where
    overloadedMethod = widgetShouldLayout

instance O.OverloadedMethodInfo WidgetShouldLayoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetShouldLayout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetShouldLayout"
        })


#endif

-- method Widget::show
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_show" gtk_widget_show :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

{-# DEPRECATED widgetShow ["(Since version 4.10)","Use 'GI.Gtk.Objects.Widget.widgetSetVisible' instead"] #-}
-- | Flags a widget to be displayed.
-- 
-- Any widget that isn’t shown will not appear on the screen.
-- 
-- Remember that you have to show the containers containing a widget,
-- in addition to the widget itself, before it will appear onscreen.
-- 
-- When a toplevel widget is shown, it is immediately realized and
-- mapped; other shown widgets are realized and mapped when their
-- toplevel widget is realized and mapped.
widgetShow ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetShow widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_show widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetShowMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetShowMethodInfo a signature where
    overloadedMethod = widgetShow

instance O.OverloadedMethodInfo WidgetShowMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetShow",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetShow"
        })


#endif

-- method Widget::size_allocate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "allocation"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Rectangle" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "position and size to be allocated to @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the baseline of the child, or -1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_size_allocate" gtk_widget_size_allocate :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Rectangle.Rectangle ->          -- allocation : TInterface (Name {namespace = "Gdk", name = "Rectangle"})
    Int32 ->                                -- baseline : TBasicType TInt
    IO ()

-- | Allocates widget with a transformation that translates
-- the origin to the position in /@allocation@/.
-- 
-- This is a simple form of 'GI.Gtk.Objects.Widget.widgetAllocate'.
widgetSizeAllocate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> Gdk.Rectangle.Rectangle
    -- ^ /@allocation@/: position and size to be allocated to /@widget@/
    -> Int32
    -- ^ /@baseline@/: the baseline of the child, or -1
    -> m ()
widgetSizeAllocate widget allocation baseline = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    allocation' <- unsafeManagedPtrGetPtr allocation
    gtk_widget_size_allocate widget' allocation' baseline
    touchManagedPtr widget
    touchManagedPtr allocation
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSizeAllocateMethodInfo
instance (signature ~ (Gdk.Rectangle.Rectangle -> Int32 -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetSizeAllocateMethodInfo a signature where
    overloadedMethod = widgetSizeAllocate

instance O.OverloadedMethodInfo WidgetSizeAllocateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSizeAllocate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSizeAllocate"
        })


#endif

-- method Widget::snapshot_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "child"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a child of @widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "snapshot"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Snapshot" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "snapshot as passed to the widget. In particular, no\n  calls to [method@Gtk.Snapshot.translate] or other transform calls\n  should have been made"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_snapshot_child" gtk_widget_snapshot_child :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- child : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.Snapshot.Snapshot ->            -- snapshot : TInterface (Name {namespace = "Gtk", name = "Snapshot"})
    IO ()

-- | Snapshots a child of the widget.
-- 
-- When a widget receives a call to the snapshot function,
-- it must send synthetic t'GI.Gtk.Objects.Widget.Widget'.@/snapshot/@() calls
-- to all children. This function provides a convenient way
-- of doing this. A widget, when it receives a call to its
-- t'GI.Gtk.Objects.Widget.Widget'.@/snapshot/@() function, calls
-- 'GI.Gtk.Objects.Widget.widgetSnapshotChild' once for each child, passing in
-- the /@snapshot@/ the widget received.
-- 
-- This function takes care of translating the origin of /@snapshot@/,
-- and deciding whether the child needs to be snapshot.
-- 
-- It does nothing for children that implement @GtkNative@.
widgetSnapshotChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b, Gtk.Snapshot.IsSnapshot c) =>
    a
    -- ^ /@widget@/: a widget
    -> b
    -- ^ /@child@/: a child of /@widget@/
    -> c
    -- ^ /@snapshot@/: snapshot as passed to the widget. In particular, no
    --   calls to 'GI.Gtk.Objects.Snapshot.snapshotTranslate' or other transform calls
    --   should have been made
    -> m ()
widgetSnapshotChild widget child snapshot = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    child' <- unsafeManagedPtrCastPtr child
    snapshot' <- unsafeManagedPtrCastPtr snapshot
    gtk_widget_snapshot_child widget' child' snapshot'
    touchManagedPtr widget
    touchManagedPtr child
    touchManagedPtr snapshot
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetSnapshotChildMethodInfo
instance (signature ~ (b -> c -> m ()), MonadIO m, IsWidget a, IsWidget b, Gtk.Snapshot.IsSnapshot c) => O.OverloadedMethod WidgetSnapshotChildMethodInfo a signature where
    overloadedMethod = widgetSnapshotChild

instance O.OverloadedMethodInfo WidgetSnapshotChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetSnapshotChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetSnapshotChild"
        })


#endif

-- method Widget::translate_coordinates
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "src_widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dest_widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "src_x"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "X position in widget coordinates of @src_widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "src_y"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Y position in widget coordinates of @src_widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dest_x"
--           , argType = TBasicType TDouble
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store X position in widget coordinates of @dest_widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "dest_y"
--           , argType = TBasicType TDouble
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store Y position in widget coordinates of @dest_widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_translate_coordinates" gtk_widget_translate_coordinates :: 
    Ptr Widget ->                           -- src_widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- dest_widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CDouble ->                              -- src_x : TBasicType TDouble
    CDouble ->                              -- src_y : TBasicType TDouble
    Ptr CDouble ->                          -- dest_x : TBasicType TDouble
    Ptr CDouble ->                          -- dest_y : TBasicType TDouble
    IO CInt

{-# DEPRECATED widgetTranslateCoordinates ["(Since version 4.12)","Use 'GI.Gtk.Objects.Widget.widgetComputePoint' instead"] #-}
-- | Translates coordinates relative to /@srcWidget@/’s allocation
-- to coordinates relative to /@destWidget@/’s allocations.
-- 
-- In order to perform this operation, both widget must share
-- a common ancestor. If that is not the case, /@destX@/ and /@destY@/
-- are set to 0 and false is returned.
widgetTranslateCoordinates ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@srcWidget@/: a widget
    -> b
    -- ^ /@destWidget@/: another widget
    -> Double
    -- ^ /@srcX@/: X position in widget coordinates of /@srcWidget@/
    -> Double
    -- ^ /@srcY@/: Y position in widget coordinates of /@srcWidget@/
    -> m ((Bool, Double, Double))
    -- ^ __Returns:__ true if /@srcWidget@/ and /@destWidget@/ have a common
    --   ancestor, false otherwise
widgetTranslateCoordinates srcWidget destWidget srcX srcY = liftIO $ do
    srcWidget' <- unsafeManagedPtrCastPtr srcWidget
    destWidget' <- unsafeManagedPtrCastPtr destWidget
    let srcX' = realToFrac srcX
    let srcY' = realToFrac srcY
    destX <- allocMem :: IO (Ptr CDouble)
    destY <- allocMem :: IO (Ptr CDouble)
    result <- gtk_widget_translate_coordinates srcWidget' destWidget' srcX' srcY' destX destY
    let result' = (/= 0) result
    destX' <- peek destX
    let destX'' = realToFrac destX'
    destY' <- peek destY
    let destY'' = realToFrac destY'
    touchManagedPtr srcWidget
    touchManagedPtr destWidget
    freeMem destX
    freeMem destY
    return (result', destX'', destY'')

#if defined(ENABLE_OVERLOADING)
data WidgetTranslateCoordinatesMethodInfo
instance (signature ~ (b -> Double -> Double -> m ((Bool, Double, Double))), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetTranslateCoordinatesMethodInfo a signature where
    overloadedMethod = widgetTranslateCoordinates

instance O.OverloadedMethodInfo WidgetTranslateCoordinatesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetTranslateCoordinates",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetTranslateCoordinates"
        })


#endif

-- method Widget::trigger_tooltip_query
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_trigger_tooltip_query" gtk_widget_trigger_tooltip_query :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Triggers a tooltip query on the display of the widget.
widgetTriggerTooltipQuery ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetTriggerTooltipQuery widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_trigger_tooltip_query widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetTriggerTooltipQueryMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetTriggerTooltipQueryMethodInfo a signature where
    overloadedMethod = widgetTriggerTooltipQuery

instance O.OverloadedMethodInfo WidgetTriggerTooltipQueryMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetTriggerTooltipQuery",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetTriggerTooltipQuery"
        })


#endif

-- method Widget::unmap
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_unmap" gtk_widget_unmap :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Causes a widget to be unmapped if it’s currently mapped.
-- 
-- This function is only for use in widget implementations.
widgetUnmap ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetUnmap widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_unmap widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetUnmapMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetUnmapMethodInfo a signature where
    overloadedMethod = widgetUnmap

instance O.OverloadedMethodInfo WidgetUnmapMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetUnmap",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetUnmap"
        })


#endif

-- method Widget::unparent
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_unparent" gtk_widget_unparent :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Removes /@widget@/ from its parent.
-- 
-- This function is only for use in widget implementations,
-- typically in dispose.
widgetUnparent ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetUnparent widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_unparent widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetUnparentMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetUnparentMethodInfo a signature where
    overloadedMethod = widgetUnparent

instance O.OverloadedMethodInfo WidgetUnparentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetUnparent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetUnparent"
        })


#endif

-- method Widget::unrealize
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_unrealize" gtk_widget_unrealize :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Causes a widget to be unrealized.
-- 
-- This frees all GDK resources associated with the widget.
-- 
-- This function is only useful in widget implementations.
widgetUnrealize ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> m ()
widgetUnrealize widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_unrealize widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetUnrealizeMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetUnrealizeMethodInfo a signature where
    overloadedMethod = widgetUnrealize

instance O.OverloadedMethodInfo WidgetUnrealizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetUnrealize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetUnrealize"
        })


#endif

-- method Widget::unset_state_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "StateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "state flags to turn off"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_unset_state_flags" gtk_widget_unset_state_flags :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gtk", name = "StateFlags"})
    IO ()

-- | Turns off flag values for the current widget state.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetStateFlags'.
-- 
-- This function is for use in widget implementations.
widgetUnsetStateFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget
    -> [Gtk.Flags.StateFlags]
    -- ^ /@flags@/: state flags to turn off
    -> m ()
widgetUnsetStateFlags widget flags = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let flags' = gflagsToWord flags
    gtk_widget_unset_state_flags widget' flags'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetUnsetStateFlagsMethodInfo
instance (signature ~ ([Gtk.Flags.StateFlags] -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetUnsetStateFlagsMethodInfo a signature where
    overloadedMethod = widgetUnsetStateFlags

instance O.OverloadedMethodInfo WidgetUnsetStateFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetUnsetStateFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-Widget.html#v:widgetUnsetStateFlags"
        })


#endif

-- method Widget::get_default_direction
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "TextDirection" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_default_direction" gtk_widget_get_default_direction :: 
    IO CUInt

-- | Obtains the default reading direction.
-- 
-- See [func/@gtk@/.Widget.set_default_direction].
widgetGetDefaultDirection ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Gtk.Enums.TextDirection
    -- ^ __Returns:__ the current default direction
widgetGetDefaultDirection  = liftIO $ do
    result <- gtk_widget_get_default_direction
    let result' = (toEnum . fromIntegral) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Widget::set_default_direction
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "dir"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TextDirection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the new default direction, either [enum@Gtk.TextDirection.ltr]\n  or [enum@Gtk.TextDirection.rtl]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_set_default_direction" gtk_widget_set_default_direction :: 
    CUInt ->                                -- dir : TInterface (Name {namespace = "Gtk", name = "TextDirection"})
    IO ()

-- | Sets the default reading direction for widgets.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetDirection'.
widgetSetDefaultDirection ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Gtk.Enums.TextDirection
    -- ^ /@dir@/: the new default direction, either [enum/@gtk@/.TextDirection.ltr]
    --   or [enum/@gtk@/.TextDirection.rtl]
    -> m ()
widgetSetDefaultDirection dir = liftIO $ do
    let dir' = (fromIntegral . fromEnum) dir
    gtk_widget_set_default_direction dir'
    return ()

#if defined(ENABLE_OVERLOADING)
#endif


