{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Groups widgets together so they all request the same size.
-- 
-- This is typically useful when you want a column of widgets to have
-- the same size, but you can’t use a t'GI.Gtk.Objects.Grid.Grid' or t'GI.Gtk.Objects.Box.Box'.
-- 
-- In detail, the size requested for each widget in a @GtkSizeGroup@ is
-- the maximum of the sizes that would have been requested for each
-- widget in the size group if they were not in the size group. The
-- [mode]'GI.Gtk.Objects.SizeGroup.sizeGroupSetMode' of the size group determines
-- whether this applies to the horizontal size, the vertical size, or
-- both sizes.
-- 
-- Note that size groups only affect the amount of space requested, not
-- the size that the widgets finally receive. If you want the widgets in
-- a @GtkSizeGroup@ to actually be the same size, you need to pack them in
-- such a way that they get the size they request and not more. In
-- particular it doesn\'t make a lot of sense to set
-- [the expand flags]'GI.Gtk.Objects.Widget.widgetSetHexpand' on the widgets that
-- are members of a size group.
-- 
-- @GtkSizeGroup@ objects are referenced by each widget in the size group,
-- so once you have added all widgets to a @GtkSizeGroup@, you can drop
-- the initial reference to the size group with
-- 'GI.GObject.Objects.Object.objectUnref'. If the widgets in the size group are
-- subsequently destroyed, then they will be removed from the size group
-- and drop their references on the size group; when all widgets have been
-- removed, the size group will be freed.
-- 
-- Widgets can be part of multiple size groups; GTK will compute the
-- horizontal size of a widget from the horizontal requisition of all widgets
-- that can be reached from the widget by a chain of size groups with mode
-- [enum/@gtk@/.SizeGroupMode.HORIZONTAL] or [enum/@gtk@/.SizeGroupMode.BOTH], and
-- the vertical size from the vertical requisition of all widgets that can be
-- reached from the widget by a chain of size groups with mode
-- [enum/@gtk@/.SizeGroupMode.VERTICAL] or [enum/@gtk@/.SizeGroupMode.BOTH].
-- 
-- = Size groups and trading height-for-width
-- 
-- ::: warning
--     Generally, size groups don\'t interact well with widgets that
--     trade height for width (or width for height), such as wrappable
--     labels. Avoid using size groups with such widgets.
-- 
-- A size group with mode [enum/@gtk@/.SizeGroupMode.HORIZONTAL] or
-- [enum/@gtk@/.SizeGroupMode.VERTICAL] only consults non-contextual sizes
-- of widgets other than the one being measured, since it has no
-- knowledge of what size a widget will get allocated in the other
-- orientation. This can lead to widgets in a group actually requesting
-- different contextual sizes, contrary to the purpose of
-- @GtkSizeGroup@.
-- 
-- In contrast, a size group with mode [enum/@gtk@/.SizeGroupMode.BOTH] can
-- properly propagate the available size in the opposite orientation
-- when measuring widgets in the group, which results in consistent and
-- accurate measurements.
-- 
-- In case some mechanism other than a size group is already used to
-- ensure that widgets in a group all get the same size in one
-- orientation (for example, some common ancestor is known to allocate
-- the same width to all its children), and the size group is only
-- really needed to also make the widgets request the same size in the
-- other orientation, it is beneficial to still set the group\'s mode to
-- [enum/@gtk@/.SizeGroupMode.BOTH]. This lets the group assume and count
-- on sizes of the widgets in the former orientation being the same,
-- which enables it to propagate the available size as described above.
-- 
-- = Alternatives to size groups
-- 
-- Size groups have many limitations, such as only influencing size
-- requests but not allocations, and poor height-for-width support. When
-- possible, prefer using dedicated mechanisms that can properly ensure
-- that the widgets get the same size.
-- 
-- Various container widgets and layout managers support a homogeneous
-- layout mode, where they will explicitly give the same size to their
-- children (see [Box:homogeneous]("GI.Gtk.Objects.Box#g:attr:homogeneous")). Using homogeneous mode
-- can also have large performance benefits compared to either the same
-- container in non-homogeneous mode, or to size groups.
-- 
-- t'GI.Gtk.Objects.Grid.Grid' can be used to position widgets into rows and
-- columns. Members of each column will have the same width among them;
-- likewise, members of each row will have the same height. On top of
-- that, the heights can be made equal between all rows with
-- [Grid:rowHomogeneous]("GI.Gtk.Objects.Grid#g:attr:rowHomogeneous"), and the widths can be made equal
-- between all columns with [Grid:columnHomogeneous]("GI.Gtk.Objects.Grid#g:attr:columnHomogeneous").
-- 
-- = GtkSizeGroup as GtkBuildable
-- 
-- Size groups can be specified in a UI definition by placing an @\<object>@
-- element with @class=\"GtkSizeGroup\"@ somewhere in the UI definition. The
-- widgets that belong to the size group are specified by a @\<widgets>@ element
-- that may contain multiple @\<widget>@ elements, one for each member of the
-- size group. The ”name” attribute gives the id of the widget.
-- 
-- An example of a UI definition fragment with @GtkSizeGroup@:
-- 
-- === /xml code/
-- ><object class="GtkSizeGroup">
-- >  <property name="mode">horizontal</property>
-- >  <widgets>
-- >    <widget name="radio1"/>
-- >    <widget name="radio2"/>
-- >  </widgets>
-- ></object>
-- 

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.SizeGroup
    ( 

-- * Exported types
    SizeGroup(..)                           ,
    IsSizeGroup                             ,
    toSizeGroup                             ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [addWidget]("GI.Gtk.Objects.SizeGroup#g:method:addWidget"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeWidget]("GI.Gtk.Objects.SizeGroup#g:method:removeWidget"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getBuildableId]("GI.Gtk.Interfaces.Buildable#g:method:getBuildableId"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getMode]("GI.Gtk.Objects.SizeGroup#g:method:getMode"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getWidgets]("GI.Gtk.Objects.SizeGroup#g:method:getWidgets").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setMode]("GI.Gtk.Objects.SizeGroup#g:method:setMode"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveSizeGroupMethod                  ,
#endif

-- ** addWidget #method:addWidget#

#if defined(ENABLE_OVERLOADING)
    SizeGroupAddWidgetMethodInfo            ,
#endif
    sizeGroupAddWidget                      ,


-- ** getMode #method:getMode#

#if defined(ENABLE_OVERLOADING)
    SizeGroupGetModeMethodInfo              ,
#endif
    sizeGroupGetMode                        ,


-- ** getWidgets #method:getWidgets#

#if defined(ENABLE_OVERLOADING)
    SizeGroupGetWidgetsMethodInfo           ,
#endif
    sizeGroupGetWidgets                     ,


-- ** new #method:new#

    sizeGroupNew                            ,


-- ** removeWidget #method:removeWidget#

#if defined(ENABLE_OVERLOADING)
    SizeGroupRemoveWidgetMethodInfo         ,
#endif
    sizeGroupRemoveWidget                   ,


-- ** setMode #method:setMode#

#if defined(ENABLE_OVERLOADING)
    SizeGroupSetModeMethodInfo              ,
#endif
    sizeGroupSetMode                        ,




 -- * Properties


-- ** mode #attr:mode#
-- | The direction in which the size group affects requested sizes.

#if defined(ENABLE_OVERLOADING)
    SizeGroupModePropertyInfo               ,
#endif
    constructSizeGroupMode                  ,
    getSizeGroupMode                        ,
    setSizeGroupMode                        ,
#if defined(ENABLE_OVERLOADING)
    sizeGroupMode                           ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget

#endif

-- | Memory-managed wrapper type.
newtype SizeGroup = SizeGroup (SP.ManagedPtr SizeGroup)
    deriving (Eq)

instance SP.ManagedPtrNewtype SizeGroup where
    toManagedPtr (SizeGroup p) = p

foreign import ccall "gtk_size_group_get_type"
    c_gtk_size_group_get_type :: IO B.Types.GType

instance B.Types.TypedObject SizeGroup where
    glibType = c_gtk_size_group_get_type

instance B.Types.GObject SizeGroup

-- | Type class for types which can be safely cast to `SizeGroup`, for instance with `toSizeGroup`.
class (SP.GObject o, O.IsDescendantOf SizeGroup o) => IsSizeGroup o
instance (SP.GObject o, O.IsDescendantOf SizeGroup o) => IsSizeGroup o

instance O.HasParentTypes SizeGroup
type instance O.ParentTypes SizeGroup = '[GObject.Object.Object, Gtk.Buildable.Buildable]

-- | Cast to `SizeGroup`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toSizeGroup :: (MIO.MonadIO m, IsSizeGroup o) => o -> m SizeGroup
toSizeGroup = MIO.liftIO . B.ManagedPtr.unsafeCastTo SizeGroup

-- | Convert 'SizeGroup' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe SizeGroup) where
    gvalueGType_ = c_gtk_size_group_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr SizeGroup)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr SizeGroup)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject SizeGroup ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveSizeGroupMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveSizeGroupMethod "addWidget" o = SizeGroupAddWidgetMethodInfo
    ResolveSizeGroupMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveSizeGroupMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveSizeGroupMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveSizeGroupMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveSizeGroupMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveSizeGroupMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveSizeGroupMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveSizeGroupMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveSizeGroupMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveSizeGroupMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveSizeGroupMethod "removeWidget" o = SizeGroupRemoveWidgetMethodInfo
    ResolveSizeGroupMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveSizeGroupMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveSizeGroupMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveSizeGroupMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveSizeGroupMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveSizeGroupMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveSizeGroupMethod "getBuildableId" o = Gtk.Buildable.BuildableGetBuildableIdMethodInfo
    ResolveSizeGroupMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveSizeGroupMethod "getMode" o = SizeGroupGetModeMethodInfo
    ResolveSizeGroupMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveSizeGroupMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveSizeGroupMethod "getWidgets" o = SizeGroupGetWidgetsMethodInfo
    ResolveSizeGroupMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveSizeGroupMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveSizeGroupMethod "setMode" o = SizeGroupSetModeMethodInfo
    ResolveSizeGroupMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveSizeGroupMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveSizeGroupMethod t SizeGroup, O.OverloadedMethod info SizeGroup p) => OL.IsLabel t (SizeGroup -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveSizeGroupMethod t SizeGroup, O.OverloadedMethod info SizeGroup p, R.HasField t SizeGroup p) => R.HasField t SizeGroup p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveSizeGroupMethod t SizeGroup, O.OverloadedMethodInfo info SizeGroup) => OL.IsLabel t (O.MethodProxy info SizeGroup) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "mode"
   -- Type: TInterface (Name {namespace = "Gtk", name = "SizeGroupMode"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@mode@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' sizeGroup #mode
-- @
getSizeGroupMode :: (MonadIO m, IsSizeGroup o) => o -> m Gtk.Enums.SizeGroupMode
getSizeGroupMode obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "mode"

-- | Set the value of the “@mode@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' sizeGroup [ #mode 'Data.GI.Base.Attributes.:=' value ]
-- @
setSizeGroupMode :: (MonadIO m, IsSizeGroup o) => o -> Gtk.Enums.SizeGroupMode -> m ()
setSizeGroupMode obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "mode" val

-- | Construct a `GValueConstruct` with valid value for the “@mode@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSizeGroupMode :: (IsSizeGroup o, MIO.MonadIO m) => Gtk.Enums.SizeGroupMode -> m (GValueConstruct o)
constructSizeGroupMode val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "mode" val

#if defined(ENABLE_OVERLOADING)
data SizeGroupModePropertyInfo
instance AttrInfo SizeGroupModePropertyInfo where
    type AttrAllowedOps SizeGroupModePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint SizeGroupModePropertyInfo = IsSizeGroup
    type AttrSetTypeConstraint SizeGroupModePropertyInfo = (~) Gtk.Enums.SizeGroupMode
    type AttrTransferTypeConstraint SizeGroupModePropertyInfo = (~) Gtk.Enums.SizeGroupMode
    type AttrTransferType SizeGroupModePropertyInfo = Gtk.Enums.SizeGroupMode
    type AttrGetType SizeGroupModePropertyInfo = Gtk.Enums.SizeGroupMode
    type AttrLabel SizeGroupModePropertyInfo = "mode"
    type AttrOrigin SizeGroupModePropertyInfo = SizeGroup
    attrGet = getSizeGroupMode
    attrSet = setSizeGroupMode
    attrTransfer _ v = do
        return v
    attrConstruct = constructSizeGroupMode
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.SizeGroup.mode"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-SizeGroup.html#g:attr:mode"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList SizeGroup
type instance O.AttributeList SizeGroup = SizeGroupAttributeList
type SizeGroupAttributeList = ('[ '("mode", SizeGroupModePropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
sizeGroupMode :: AttrLabelProxy "mode"
sizeGroupMode = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList SizeGroup = SizeGroupSignalList
type SizeGroupSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method SizeGroup::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "mode"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "SizeGroupMode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the mode for the new size group."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "SizeGroup" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_size_group_new" gtk_size_group_new :: 
    CUInt ->                                -- mode : TInterface (Name {namespace = "Gtk", name = "SizeGroupMode"})
    IO (Ptr SizeGroup)

-- | Create a new @GtkSizeGroup@.
sizeGroupNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Gtk.Enums.SizeGroupMode
    -- ^ /@mode@/: the mode for the new size group.
    -> m SizeGroup
    -- ^ __Returns:__ a newly created @GtkSizeGroup@
sizeGroupNew mode = liftIO $ do
    let mode' = (fromIntegral . fromEnum) mode
    result <- gtk_size_group_new mode'
    checkUnexpectedReturnNULL "sizeGroupNew" result
    result' <- (wrapObject SizeGroup) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method SizeGroup::add_widget
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "size_group"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "SizeGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkSizeGroup`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GtkWidget` to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_size_group_add_widget" gtk_size_group_add_widget :: 
    Ptr SizeGroup ->                        -- size_group : TInterface (Name {namespace = "Gtk", name = "SizeGroup"})
    Ptr Gtk.Widget.Widget ->                -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Adds a widget to a @GtkSizeGroup@.
-- 
-- In the future, the requisition
-- of the widget will be determined as the maximum of its requisition
-- and the requisition of the other widgets in the size group.
-- Whether this applies horizontally, vertically, or in both directions
-- depends on the mode of the size group.
-- See 'GI.Gtk.Objects.SizeGroup.sizeGroupSetMode'.
-- 
-- When the widget is destroyed or no longer referenced elsewhere, it
-- will be removed from the size group.
sizeGroupAddWidget ::
    (B.CallStack.HasCallStack, MonadIO m, IsSizeGroup a, Gtk.Widget.IsWidget b) =>
    a
    -- ^ /@sizeGroup@/: a @GtkSizeGroup@
    -> b
    -- ^ /@widget@/: the @GtkWidget@ to add
    -> m ()
sizeGroupAddWidget sizeGroup widget = liftIO $ do
    sizeGroup' <- unsafeManagedPtrCastPtr sizeGroup
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_size_group_add_widget sizeGroup' widget'
    touchManagedPtr sizeGroup
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data SizeGroupAddWidgetMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsSizeGroup a, Gtk.Widget.IsWidget b) => O.OverloadedMethod SizeGroupAddWidgetMethodInfo a signature where
    overloadedMethod = sizeGroupAddWidget

instance O.OverloadedMethodInfo SizeGroupAddWidgetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.SizeGroup.sizeGroupAddWidget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-SizeGroup.html#v:sizeGroupAddWidget"
        })


#endif

-- method SizeGroup::get_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "size_group"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "SizeGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkSizeGroup`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "SizeGroupMode" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_size_group_get_mode" gtk_size_group_get_mode :: 
    Ptr SizeGroup ->                        -- size_group : TInterface (Name {namespace = "Gtk", name = "SizeGroup"})
    IO CUInt

-- | Gets the current mode of the size group.
sizeGroupGetMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsSizeGroup a) =>
    a
    -- ^ /@sizeGroup@/: a @GtkSizeGroup@
    -> m Gtk.Enums.SizeGroupMode
    -- ^ __Returns:__ the current mode of the size group.
sizeGroupGetMode sizeGroup = liftIO $ do
    sizeGroup' <- unsafeManagedPtrCastPtr sizeGroup
    result <- gtk_size_group_get_mode sizeGroup'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr sizeGroup
    return result'

#if defined(ENABLE_OVERLOADING)
data SizeGroupGetModeMethodInfo
instance (signature ~ (m Gtk.Enums.SizeGroupMode), MonadIO m, IsSizeGroup a) => O.OverloadedMethod SizeGroupGetModeMethodInfo a signature where
    overloadedMethod = sizeGroupGetMode

instance O.OverloadedMethodInfo SizeGroupGetModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.SizeGroup.sizeGroupGetMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-SizeGroup.html#v:sizeGroupGetMode"
        })


#endif

-- method SizeGroup::get_widgets
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "size_group"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "SizeGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkSizeGroup`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGSList (TInterface Name { namespace = "Gtk" , name = "Widget" }))
-- throws : False
-- Skip return : False

foreign import ccall "gtk_size_group_get_widgets" gtk_size_group_get_widgets :: 
    Ptr SizeGroup ->                        -- size_group : TInterface (Name {namespace = "Gtk", name = "SizeGroup"})
    IO (Ptr (GSList (Ptr Gtk.Widget.Widget)))

-- | Returns the list of widgets associated with /@sizeGroup@/.
sizeGroupGetWidgets ::
    (B.CallStack.HasCallStack, MonadIO m, IsSizeGroup a) =>
    a
    -- ^ /@sizeGroup@/: a @GtkSizeGroup@
    -> m [Gtk.Widget.Widget]
    -- ^ __Returns:__ a @GSList@ of
    --   widgets. The list is owned by GTK and should not be modified.
sizeGroupGetWidgets sizeGroup = liftIO $ do
    sizeGroup' <- unsafeManagedPtrCastPtr sizeGroup
    result <- gtk_size_group_get_widgets sizeGroup'
    result' <- unpackGSList result
    result'' <- mapM (newObject Gtk.Widget.Widget) result'
    touchManagedPtr sizeGroup
    return result''

#if defined(ENABLE_OVERLOADING)
data SizeGroupGetWidgetsMethodInfo
instance (signature ~ (m [Gtk.Widget.Widget]), MonadIO m, IsSizeGroup a) => O.OverloadedMethod SizeGroupGetWidgetsMethodInfo a signature where
    overloadedMethod = sizeGroupGetWidgets

instance O.OverloadedMethodInfo SizeGroupGetWidgetsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.SizeGroup.sizeGroupGetWidgets",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-SizeGroup.html#v:sizeGroupGetWidgets"
        })


#endif

-- method SizeGroup::remove_widget
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "size_group"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "SizeGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkSizeGroup`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GtkWidget` to remove"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_size_group_remove_widget" gtk_size_group_remove_widget :: 
    Ptr SizeGroup ->                        -- size_group : TInterface (Name {namespace = "Gtk", name = "SizeGroup"})
    Ptr Gtk.Widget.Widget ->                -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Removes a widget from a @GtkSizeGroup@.
sizeGroupRemoveWidget ::
    (B.CallStack.HasCallStack, MonadIO m, IsSizeGroup a, Gtk.Widget.IsWidget b) =>
    a
    -- ^ /@sizeGroup@/: a @GtkSizeGroup@
    -> b
    -- ^ /@widget@/: the @GtkWidget@ to remove
    -> m ()
sizeGroupRemoveWidget sizeGroup widget = liftIO $ do
    sizeGroup' <- unsafeManagedPtrCastPtr sizeGroup
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_size_group_remove_widget sizeGroup' widget'
    touchManagedPtr sizeGroup
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data SizeGroupRemoveWidgetMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsSizeGroup a, Gtk.Widget.IsWidget b) => O.OverloadedMethod SizeGroupRemoveWidgetMethodInfo a signature where
    overloadedMethod = sizeGroupRemoveWidget

instance O.OverloadedMethodInfo SizeGroupRemoveWidgetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.SizeGroup.sizeGroupRemoveWidget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-SizeGroup.html#v:sizeGroupRemoveWidget"
        })


#endif

-- method SizeGroup::set_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "size_group"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "SizeGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkSizeGroup`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mode"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "SizeGroupMode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the mode to set for the size group."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_size_group_set_mode" gtk_size_group_set_mode :: 
    Ptr SizeGroup ->                        -- size_group : TInterface (Name {namespace = "Gtk", name = "SizeGroup"})
    CUInt ->                                -- mode : TInterface (Name {namespace = "Gtk", name = "SizeGroupMode"})
    IO ()

-- | Sets the @GtkSizeGroupMode@ of the size group.
-- 
-- The mode of the size group determines whether the widgets in the
-- size group should all have the same horizontal requisition
-- ('GI.Gtk.Enums.SizeGroupModeHorizontal') all have the same vertical requisition
-- ('GI.Gtk.Enums.SizeGroupModeVertical'), or should all have the same requisition
-- in both directions ('GI.Gtk.Enums.SizeGroupModeBoth').
sizeGroupSetMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsSizeGroup a) =>
    a
    -- ^ /@sizeGroup@/: a @GtkSizeGroup@
    -> Gtk.Enums.SizeGroupMode
    -- ^ /@mode@/: the mode to set for the size group.
    -> m ()
sizeGroupSetMode sizeGroup mode = liftIO $ do
    sizeGroup' <- unsafeManagedPtrCastPtr sizeGroup
    let mode' = (fromIntegral . fromEnum) mode
    gtk_size_group_set_mode sizeGroup' mode'
    touchManagedPtr sizeGroup
    return ()

#if defined(ENABLE_OVERLOADING)
data SizeGroupSetModeMethodInfo
instance (signature ~ (Gtk.Enums.SizeGroupMode -> m ()), MonadIO m, IsSizeGroup a) => O.OverloadedMethod SizeGroupSetModeMethodInfo a signature where
    overloadedMethod = sizeGroupSetMode

instance O.OverloadedMethodInfo SizeGroupSetModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.SizeGroup.sizeGroupSetMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-SizeGroup.html#v:sizeGroupSetMode"
        })


#endif


