{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Manages keyboard shortcuts and their activation.
-- 
-- Most common shortcuts are using this controller implicitly, e.g. by
-- adding a mnemonic underline to a t'GI.Gtk.Objects.Label.Label', or by installing a key
-- binding using t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/add_binding/@(), or by adding accelerators
-- to global actions using 'GI.Gtk.Objects.Application.applicationSetAccelsForAction'.
-- 
-- But it is possible to create your own shortcut controller, and add
-- shortcuts to it.
-- 
-- @GtkShortcutController@ implements t'GI.Gio.Interfaces.ListModel.ListModel' for querying the
-- shortcuts that have been added to it.
-- 
-- = GtkShortcutController as GtkBuildable
-- 
-- @GtkShortcutController@s can be created in t'GI.Gtk.Objects.Builder.Builder' ui files, to set up
-- shortcuts in the same place as the widgets.
-- 
-- An example of a UI definition fragment with @GtkShortcutController@:
-- 
-- === /xml code/
-- >  <object class='GtkButton'>
-- >    <child>
-- >      <object class='GtkShortcutController'>
-- >        <property name='scope'>managed</property>
-- >        <child>
-- >          <object class='GtkShortcut'>
-- >            <property name='trigger'>&lt;Control&gt;k</property>
-- >            <property name='action'>activate</property>
-- >          </object>
-- >        </child>
-- >      </object>
-- >    </child>
-- >  </object>
-- 
-- 
-- This example creates a t'GI.Gtk.Objects.ActivateAction.ActivateAction' for triggering the
-- @activate@ signal of the t'GI.Gtk.Objects.Button.Button'. See 'GI.Gtk.Objects.ShortcutAction.shortcutActionParseString'
-- for the syntax for other kinds of t'GI.Gtk.Objects.ShortcutAction.ShortcutAction'. See
-- 'GI.Gtk.Objects.ShortcutTrigger.shortcutTriggerParseString' to learn more about the syntax
-- for triggers.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.ShortcutController
    ( 

-- * Exported types
    ShortcutController(..)                  ,
    IsShortcutController                    ,
    toShortcutController                    ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [addShortcut]("GI.Gtk.Objects.ShortcutController#g:method:addShortcut"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [itemsChanged]("GI.Gio.Interfaces.ListModel#g:method:itemsChanged"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeShortcut]("GI.Gtk.Objects.ShortcutController#g:method:removeShortcut"), [reset]("GI.Gtk.Objects.EventController#g:method:reset"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getBuildableId]("GI.Gtk.Interfaces.Buildable#g:method:getBuildableId"), [getCurrentEvent]("GI.Gtk.Objects.EventController#g:method:getCurrentEvent"), [getCurrentEventDevice]("GI.Gtk.Objects.EventController#g:method:getCurrentEventDevice"), [getCurrentEventState]("GI.Gtk.Objects.EventController#g:method:getCurrentEventState"), [getCurrentEventTime]("GI.Gtk.Objects.EventController#g:method:getCurrentEventTime"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getItem]("GI.Gio.Interfaces.ListModel#g:method:getItem"), [getItemType]("GI.Gio.Interfaces.ListModel#g:method:getItemType"), [getMnemonicsModifiers]("GI.Gtk.Objects.ShortcutController#g:method:getMnemonicsModifiers"), [getNItems]("GI.Gio.Interfaces.ListModel#g:method:getNItems"), [getName]("GI.Gtk.Objects.EventController#g:method:getName"), [getPropagationLimit]("GI.Gtk.Objects.EventController#g:method:getPropagationLimit"), [getPropagationPhase]("GI.Gtk.Objects.EventController#g:method:getPropagationPhase"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getScope]("GI.Gtk.Objects.ShortcutController#g:method:getScope"), [getWidget]("GI.Gtk.Objects.EventController#g:method:getWidget").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setMnemonicsModifiers]("GI.Gtk.Objects.ShortcutController#g:method:setMnemonicsModifiers"), [setName]("GI.Gtk.Objects.EventController#g:method:setName"), [setPropagationLimit]("GI.Gtk.Objects.EventController#g:method:setPropagationLimit"), [setPropagationPhase]("GI.Gtk.Objects.EventController#g:method:setPropagationPhase"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setScope]("GI.Gtk.Objects.ShortcutController#g:method:setScope"), [setStaticName]("GI.Gtk.Objects.EventController#g:method:setStaticName").

#if defined(ENABLE_OVERLOADING)
    ResolveShortcutControllerMethod         ,
#endif

-- ** addShortcut #method:addShortcut#

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerAddShortcutMethodInfo ,
#endif
    shortcutControllerAddShortcut           ,


-- ** getMnemonicsModifiers #method:getMnemonicsModifiers#

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerGetMnemonicsModifiersMethodInfo,
#endif
    shortcutControllerGetMnemonicsModifiers ,


-- ** getScope #method:getScope#

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerGetScopeMethodInfo    ,
#endif
    shortcutControllerGetScope              ,


-- ** new #method:new#

    shortcutControllerNew                   ,


-- ** newForModel #method:newForModel#

    shortcutControllerNewForModel           ,


-- ** removeShortcut #method:removeShortcut#

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerRemoveShortcutMethodInfo,
#endif
    shortcutControllerRemoveShortcut        ,


-- ** setMnemonicsModifiers #method:setMnemonicsModifiers#

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerSetMnemonicsModifiersMethodInfo,
#endif
    shortcutControllerSetMnemonicsModifiers ,


-- ** setScope #method:setScope#

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerSetScopeMethodInfo    ,
#endif
    shortcutControllerSetScope              ,




 -- * Properties


-- ** itemType #attr:itemType#
-- | The type of items. See 'GI.Gio.Interfaces.ListModel.listModelGetItemType'.
-- 
-- /Since: 4.8/

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerItemTypePropertyInfo  ,
#endif
    getShortcutControllerItemType           ,
#if defined(ENABLE_OVERLOADING)
    shortcutControllerItemType              ,
#endif


-- ** mnemonicModifiers #attr:mnemonicModifiers#
-- | The modifiers that need to be pressed to allow mnemonics activation.

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerMnemonicModifiersPropertyInfo,
#endif
    constructShortcutControllerMnemonicModifiers,
    getShortcutControllerMnemonicModifiers  ,
    setShortcutControllerMnemonicModifiers  ,
#if defined(ENABLE_OVERLOADING)
    shortcutControllerMnemonicModifiers     ,
#endif


-- ** model #attr:model#
-- | A list model to take shortcuts from.

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerModelPropertyInfo     ,
#endif
    constructShortcutControllerModel        ,
#if defined(ENABLE_OVERLOADING)
    shortcutControllerModel                 ,
#endif


-- ** nItems #attr:nItems#
-- | The number of items. See 'GI.Gio.Interfaces.ListModel.listModelGetNItems'.
-- 
-- /Since: 4.8/

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerNItemsPropertyInfo    ,
#endif
    getShortcutControllerNItems             ,
#if defined(ENABLE_OVERLOADING)
    shortcutControllerNItems                ,
#endif


-- ** scope #attr:scope#
-- | What scope the shortcuts will be handled in.

#if defined(ENABLE_OVERLOADING)
    ShortcutControllerScopePropertyInfo     ,
#endif
    constructShortcutControllerScope        ,
    getShortcutControllerScope              ,
    setShortcutControllerScope              ,
#if defined(ENABLE_OVERLOADING)
    shortcutControllerScope                 ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Enums as Gdk.Enums
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Shortcut as Gtk.Shortcut
import {-# SOURCE #-} qualified GI.Gtk.Objects.ShortcutAction as Gtk.ShortcutAction
import {-# SOURCE #-} qualified GI.Gtk.Objects.ShortcutTrigger as Gtk.ShortcutTrigger
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.Shortcut as Gtk.Shortcut

#endif

-- | Memory-managed wrapper type.
newtype ShortcutController = ShortcutController (SP.ManagedPtr ShortcutController)
    deriving (Eq)

instance SP.ManagedPtrNewtype ShortcutController where
    toManagedPtr (ShortcutController p) = p

foreign import ccall "gtk_shortcut_controller_get_type"
    c_gtk_shortcut_controller_get_type :: IO B.Types.GType

instance B.Types.TypedObject ShortcutController where
    glibType = c_gtk_shortcut_controller_get_type

instance B.Types.GObject ShortcutController

-- | Type class for types which can be safely cast to `ShortcutController`, for instance with `toShortcutController`.
class (SP.GObject o, O.IsDescendantOf ShortcutController o) => IsShortcutController o
instance (SP.GObject o, O.IsDescendantOf ShortcutController o) => IsShortcutController o

instance O.HasParentTypes ShortcutController
type instance O.ParentTypes ShortcutController = '[Gtk.EventController.EventController, GObject.Object.Object, Gio.ListModel.ListModel, Gtk.Buildable.Buildable]

-- | Cast to `ShortcutController`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toShortcutController :: (MIO.MonadIO m, IsShortcutController o) => o -> m ShortcutController
toShortcutController = MIO.liftIO . B.ManagedPtr.unsafeCastTo ShortcutController

-- | Convert 'ShortcutController' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe ShortcutController) where
    gvalueGType_ = c_gtk_shortcut_controller_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr ShortcutController)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr ShortcutController)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject ShortcutController ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveShortcutControllerMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveShortcutControllerMethod "addShortcut" o = ShortcutControllerAddShortcutMethodInfo
    ResolveShortcutControllerMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveShortcutControllerMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveShortcutControllerMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveShortcutControllerMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveShortcutControllerMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveShortcutControllerMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveShortcutControllerMethod "itemsChanged" o = Gio.ListModel.ListModelItemsChangedMethodInfo
    ResolveShortcutControllerMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveShortcutControllerMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveShortcutControllerMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveShortcutControllerMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveShortcutControllerMethod "removeShortcut" o = ShortcutControllerRemoveShortcutMethodInfo
    ResolveShortcutControllerMethod "reset" o = Gtk.EventController.EventControllerResetMethodInfo
    ResolveShortcutControllerMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveShortcutControllerMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveShortcutControllerMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveShortcutControllerMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveShortcutControllerMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveShortcutControllerMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveShortcutControllerMethod "getBuildableId" o = Gtk.Buildable.BuildableGetBuildableIdMethodInfo
    ResolveShortcutControllerMethod "getCurrentEvent" o = Gtk.EventController.EventControllerGetCurrentEventMethodInfo
    ResolveShortcutControllerMethod "getCurrentEventDevice" o = Gtk.EventController.EventControllerGetCurrentEventDeviceMethodInfo
    ResolveShortcutControllerMethod "getCurrentEventState" o = Gtk.EventController.EventControllerGetCurrentEventStateMethodInfo
    ResolveShortcutControllerMethod "getCurrentEventTime" o = Gtk.EventController.EventControllerGetCurrentEventTimeMethodInfo
    ResolveShortcutControllerMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveShortcutControllerMethod "getItem" o = Gio.ListModel.ListModelGetItemMethodInfo
    ResolveShortcutControllerMethod "getItemType" o = Gio.ListModel.ListModelGetItemTypeMethodInfo
    ResolveShortcutControllerMethod "getMnemonicsModifiers" o = ShortcutControllerGetMnemonicsModifiersMethodInfo
    ResolveShortcutControllerMethod "getNItems" o = Gio.ListModel.ListModelGetNItemsMethodInfo
    ResolveShortcutControllerMethod "getName" o = Gtk.EventController.EventControllerGetNameMethodInfo
    ResolveShortcutControllerMethod "getPropagationLimit" o = Gtk.EventController.EventControllerGetPropagationLimitMethodInfo
    ResolveShortcutControllerMethod "getPropagationPhase" o = Gtk.EventController.EventControllerGetPropagationPhaseMethodInfo
    ResolveShortcutControllerMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveShortcutControllerMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveShortcutControllerMethod "getScope" o = ShortcutControllerGetScopeMethodInfo
    ResolveShortcutControllerMethod "getWidget" o = Gtk.EventController.EventControllerGetWidgetMethodInfo
    ResolveShortcutControllerMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveShortcutControllerMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveShortcutControllerMethod "setMnemonicsModifiers" o = ShortcutControllerSetMnemonicsModifiersMethodInfo
    ResolveShortcutControllerMethod "setName" o = Gtk.EventController.EventControllerSetNameMethodInfo
    ResolveShortcutControllerMethod "setPropagationLimit" o = Gtk.EventController.EventControllerSetPropagationLimitMethodInfo
    ResolveShortcutControllerMethod "setPropagationPhase" o = Gtk.EventController.EventControllerSetPropagationPhaseMethodInfo
    ResolveShortcutControllerMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveShortcutControllerMethod "setScope" o = ShortcutControllerSetScopeMethodInfo
    ResolveShortcutControllerMethod "setStaticName" o = Gtk.EventController.EventControllerSetStaticNameMethodInfo
    ResolveShortcutControllerMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveShortcutControllerMethod t ShortcutController, O.OverloadedMethod info ShortcutController p) => OL.IsLabel t (ShortcutController -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveShortcutControllerMethod t ShortcutController, O.OverloadedMethod info ShortcutController p, R.HasField t ShortcutController p) => R.HasField t ShortcutController p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveShortcutControllerMethod t ShortcutController, O.OverloadedMethodInfo info ShortcutController) => OL.IsLabel t (O.MethodProxy info ShortcutController) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "item-type"
   -- Type: TBasicType TGType
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@item-type@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' shortcutController #itemType
-- @
getShortcutControllerItemType :: (MonadIO m, IsShortcutController o) => o -> m GType
getShortcutControllerItemType obj = MIO.liftIO $ B.Properties.getObjectPropertyGType obj "item-type"

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerItemTypePropertyInfo
instance AttrInfo ShortcutControllerItemTypePropertyInfo where
    type AttrAllowedOps ShortcutControllerItemTypePropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint ShortcutControllerItemTypePropertyInfo = IsShortcutController
    type AttrSetTypeConstraint ShortcutControllerItemTypePropertyInfo = (~) ()
    type AttrTransferTypeConstraint ShortcutControllerItemTypePropertyInfo = (~) ()
    type AttrTransferType ShortcutControllerItemTypePropertyInfo = ()
    type AttrGetType ShortcutControllerItemTypePropertyInfo = GType
    type AttrLabel ShortcutControllerItemTypePropertyInfo = "item-type"
    type AttrOrigin ShortcutControllerItemTypePropertyInfo = ShortcutController
    attrGet = getShortcutControllerItemType
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.itemType"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#g:attr:itemType"
        })
#endif

-- VVV Prop "mnemonic-modifiers"
   -- Type: TInterface (Name {namespace = "Gdk", name = "ModifierType"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@mnemonic-modifiers@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' shortcutController #mnemonicModifiers
-- @
getShortcutControllerMnemonicModifiers :: (MonadIO m, IsShortcutController o) => o -> m [Gdk.Flags.ModifierType]
getShortcutControllerMnemonicModifiers obj = MIO.liftIO $ B.Properties.getObjectPropertyFlags obj "mnemonic-modifiers"

-- | Set the value of the “@mnemonic-modifiers@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' shortcutController [ #mnemonicModifiers 'Data.GI.Base.Attributes.:=' value ]
-- @
setShortcutControllerMnemonicModifiers :: (MonadIO m, IsShortcutController o) => o -> [Gdk.Flags.ModifierType] -> m ()
setShortcutControllerMnemonicModifiers obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyFlags obj "mnemonic-modifiers" val

-- | Construct a `GValueConstruct` with valid value for the “@mnemonic-modifiers@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructShortcutControllerMnemonicModifiers :: (IsShortcutController o, MIO.MonadIO m) => [Gdk.Flags.ModifierType] -> m (GValueConstruct o)
constructShortcutControllerMnemonicModifiers val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFlags "mnemonic-modifiers" val

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerMnemonicModifiersPropertyInfo
instance AttrInfo ShortcutControllerMnemonicModifiersPropertyInfo where
    type AttrAllowedOps ShortcutControllerMnemonicModifiersPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint ShortcutControllerMnemonicModifiersPropertyInfo = IsShortcutController
    type AttrSetTypeConstraint ShortcutControllerMnemonicModifiersPropertyInfo = (~) [Gdk.Flags.ModifierType]
    type AttrTransferTypeConstraint ShortcutControllerMnemonicModifiersPropertyInfo = (~) [Gdk.Flags.ModifierType]
    type AttrTransferType ShortcutControllerMnemonicModifiersPropertyInfo = [Gdk.Flags.ModifierType]
    type AttrGetType ShortcutControllerMnemonicModifiersPropertyInfo = [Gdk.Flags.ModifierType]
    type AttrLabel ShortcutControllerMnemonicModifiersPropertyInfo = "mnemonic-modifiers"
    type AttrOrigin ShortcutControllerMnemonicModifiersPropertyInfo = ShortcutController
    attrGet = getShortcutControllerMnemonicModifiers
    attrSet = setShortcutControllerMnemonicModifiers
    attrTransfer _ v = do
        return v
    attrConstruct = constructShortcutControllerMnemonicModifiers
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.mnemonicModifiers"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#g:attr:mnemonicModifiers"
        })
#endif

-- VVV Prop "model"
   -- Type: TInterface (Name {namespace = "Gio", name = "ListModel"})
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a `GValueConstruct` with valid value for the “@model@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructShortcutControllerModel :: (IsShortcutController o, MIO.MonadIO m, Gio.ListModel.IsListModel a) => a -> m (GValueConstruct o)
constructShortcutControllerModel val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "model" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerModelPropertyInfo
instance AttrInfo ShortcutControllerModelPropertyInfo where
    type AttrAllowedOps ShortcutControllerModelPropertyInfo = '[ 'AttrConstruct, 'AttrClear]
    type AttrBaseTypeConstraint ShortcutControllerModelPropertyInfo = IsShortcutController
    type AttrSetTypeConstraint ShortcutControllerModelPropertyInfo = Gio.ListModel.IsListModel
    type AttrTransferTypeConstraint ShortcutControllerModelPropertyInfo = Gio.ListModel.IsListModel
    type AttrTransferType ShortcutControllerModelPropertyInfo = Gio.ListModel.ListModel
    type AttrGetType ShortcutControllerModelPropertyInfo = ()
    type AttrLabel ShortcutControllerModelPropertyInfo = "model"
    type AttrOrigin ShortcutControllerModelPropertyInfo = ShortcutController
    attrGet = undefined
    attrSet = undefined
    attrTransfer _ v = do
        unsafeCastTo Gio.ListModel.ListModel v
    attrConstruct = constructShortcutControllerModel
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.model"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#g:attr:model"
        })
#endif

-- VVV Prop "n-items"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@n-items@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' shortcutController #nItems
-- @
getShortcutControllerNItems :: (MonadIO m, IsShortcutController o) => o -> m Word32
getShortcutControllerNItems obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "n-items"

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerNItemsPropertyInfo
instance AttrInfo ShortcutControllerNItemsPropertyInfo where
    type AttrAllowedOps ShortcutControllerNItemsPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint ShortcutControllerNItemsPropertyInfo = IsShortcutController
    type AttrSetTypeConstraint ShortcutControllerNItemsPropertyInfo = (~) ()
    type AttrTransferTypeConstraint ShortcutControllerNItemsPropertyInfo = (~) ()
    type AttrTransferType ShortcutControllerNItemsPropertyInfo = ()
    type AttrGetType ShortcutControllerNItemsPropertyInfo = Word32
    type AttrLabel ShortcutControllerNItemsPropertyInfo = "n-items"
    type AttrOrigin ShortcutControllerNItemsPropertyInfo = ShortcutController
    attrGet = getShortcutControllerNItems
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.nItems"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#g:attr:nItems"
        })
#endif

-- VVV Prop "scope"
   -- Type: TInterface (Name {namespace = "Gtk", name = "ShortcutScope"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@scope@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' shortcutController #scope
-- @
getShortcutControllerScope :: (MonadIO m, IsShortcutController o) => o -> m Gtk.Enums.ShortcutScope
getShortcutControllerScope obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "scope"

-- | Set the value of the “@scope@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' shortcutController [ #scope 'Data.GI.Base.Attributes.:=' value ]
-- @
setShortcutControllerScope :: (MonadIO m, IsShortcutController o) => o -> Gtk.Enums.ShortcutScope -> m ()
setShortcutControllerScope obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "scope" val

-- | Construct a `GValueConstruct` with valid value for the “@scope@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructShortcutControllerScope :: (IsShortcutController o, MIO.MonadIO m) => Gtk.Enums.ShortcutScope -> m (GValueConstruct o)
constructShortcutControllerScope val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "scope" val

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerScopePropertyInfo
instance AttrInfo ShortcutControllerScopePropertyInfo where
    type AttrAllowedOps ShortcutControllerScopePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint ShortcutControllerScopePropertyInfo = IsShortcutController
    type AttrSetTypeConstraint ShortcutControllerScopePropertyInfo = (~) Gtk.Enums.ShortcutScope
    type AttrTransferTypeConstraint ShortcutControllerScopePropertyInfo = (~) Gtk.Enums.ShortcutScope
    type AttrTransferType ShortcutControllerScopePropertyInfo = Gtk.Enums.ShortcutScope
    type AttrGetType ShortcutControllerScopePropertyInfo = Gtk.Enums.ShortcutScope
    type AttrLabel ShortcutControllerScopePropertyInfo = "scope"
    type AttrOrigin ShortcutControllerScopePropertyInfo = ShortcutController
    attrGet = getShortcutControllerScope
    attrSet = setShortcutControllerScope
    attrTransfer _ v = do
        return v
    attrConstruct = constructShortcutControllerScope
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.scope"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#g:attr:scope"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList ShortcutController
type instance O.AttributeList ShortcutController = ShortcutControllerAttributeList
type ShortcutControllerAttributeList = ('[ '("itemType", ShortcutControllerItemTypePropertyInfo), '("mnemonicModifiers", ShortcutControllerMnemonicModifiersPropertyInfo), '("model", ShortcutControllerModelPropertyInfo), '("nItems", ShortcutControllerNItemsPropertyInfo), '("name", Gtk.EventController.EventControllerNamePropertyInfo), '("propagationLimit", Gtk.EventController.EventControllerPropagationLimitPropertyInfo), '("propagationPhase", Gtk.EventController.EventControllerPropagationPhasePropertyInfo), '("scope", ShortcutControllerScopePropertyInfo), '("widget", Gtk.EventController.EventControllerWidgetPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
shortcutControllerItemType :: AttrLabelProxy "itemType"
shortcutControllerItemType = AttrLabelProxy

shortcutControllerMnemonicModifiers :: AttrLabelProxy "mnemonicModifiers"
shortcutControllerMnemonicModifiers = AttrLabelProxy

shortcutControllerModel :: AttrLabelProxy "model"
shortcutControllerModel = AttrLabelProxy

shortcutControllerNItems :: AttrLabelProxy "nItems"
shortcutControllerNItems = AttrLabelProxy

shortcutControllerScope :: AttrLabelProxy "scope"
shortcutControllerScope = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList ShortcutController = ShortcutControllerSignalList
type ShortcutControllerSignalList = ('[ '("itemsChanged", Gio.ListModel.ListModelItemsChangedSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method ShortcutController::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gtk" , name = "ShortcutController" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_shortcut_controller_new" gtk_shortcut_controller_new :: 
    IO (Ptr ShortcutController)

-- | Creates a new shortcut controller.
shortcutControllerNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m ShortcutController
    -- ^ __Returns:__ a newly created shortcut controller
shortcutControllerNew  = liftIO $ do
    result <- gtk_shortcut_controller_new
    checkUnexpectedReturnNULL "shortcutControllerNew" result
    result' <- (wrapObject ShortcutController) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ShortcutController::new_for_model
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ListModel" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GListModel` containing shortcuts"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gtk" , name = "ShortcutController" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_shortcut_controller_new_for_model" gtk_shortcut_controller_new_for_model :: 
    Ptr Gio.ListModel.ListModel ->          -- model : TInterface (Name {namespace = "Gio", name = "ListModel"})
    IO (Ptr ShortcutController)

-- | Creates a new shortcut controller that takes its shortcuts from
-- the given list model.
-- 
-- A controller created by this function does not let you add or
-- remove individual shortcuts using the shortcut controller api,
-- but you can change the contents of the model.
shortcutControllerNewForModel ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.ListModel.IsListModel a) =>
    a
    -- ^ /@model@/: a @GListModel@ containing shortcuts
    -> m ShortcutController
    -- ^ __Returns:__ a newly created shortcut controller
shortcutControllerNewForModel model = liftIO $ do
    model' <- unsafeManagedPtrCastPtr model
    result <- gtk_shortcut_controller_new_for_model model'
    checkUnexpectedReturnNULL "shortcutControllerNewForModel" result
    result' <- (wrapObject ShortcutController) result
    touchManagedPtr model
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ShortcutController::add_shortcut
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ShortcutController" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the controller" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "shortcut"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Shortcut" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkShortcut`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_shortcut_controller_add_shortcut" gtk_shortcut_controller_add_shortcut :: 
    Ptr ShortcutController ->               -- self : TInterface (Name {namespace = "Gtk", name = "ShortcutController"})
    Ptr Gtk.Shortcut.Shortcut ->            -- shortcut : TInterface (Name {namespace = "Gtk", name = "Shortcut"})
    IO ()

-- | Adds /@shortcut@/ to the list of shortcuts handled by /@self@/.
-- 
-- If this controller uses an external shortcut list, this
-- function does nothing.
shortcutControllerAddShortcut ::
    (B.CallStack.HasCallStack, MonadIO m, IsShortcutController a, Gtk.Shortcut.IsShortcut b) =>
    a
    -- ^ /@self@/: the controller
    -> b
    -- ^ /@shortcut@/: a @GtkShortcut@
    -> m ()
shortcutControllerAddShortcut self shortcut = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    shortcut' <- B.ManagedPtr.disownObject shortcut
    gtk_shortcut_controller_add_shortcut self' shortcut'
    touchManagedPtr self
    touchManagedPtr shortcut
    return ()

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerAddShortcutMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsShortcutController a, Gtk.Shortcut.IsShortcut b) => O.OverloadedMethod ShortcutControllerAddShortcutMethodInfo a signature where
    overloadedMethod = shortcutControllerAddShortcut

instance O.OverloadedMethodInfo ShortcutControllerAddShortcutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.shortcutControllerAddShortcut",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#v:shortcutControllerAddShortcut"
        })


#endif

-- method ShortcutController::get_mnemonics_modifiers
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ShortcutController" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkShortcutController`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gdk" , name = "ModifierType" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_shortcut_controller_get_mnemonics_modifiers" gtk_shortcut_controller_get_mnemonics_modifiers :: 
    Ptr ShortcutController ->               -- self : TInterface (Name {namespace = "Gtk", name = "ShortcutController"})
    IO CUInt

-- | Gets the mnemonics modifiers for when this controller activates its shortcuts.
shortcutControllerGetMnemonicsModifiers ::
    (B.CallStack.HasCallStack, MonadIO m, IsShortcutController a) =>
    a
    -- ^ /@self@/: a @GtkShortcutController@
    -> m [Gdk.Flags.ModifierType]
    -- ^ __Returns:__ the controller\'s mnemonics modifiers
shortcutControllerGetMnemonicsModifiers self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_shortcut_controller_get_mnemonics_modifiers self'
    let result' = wordToGFlags result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerGetMnemonicsModifiersMethodInfo
instance (signature ~ (m [Gdk.Flags.ModifierType]), MonadIO m, IsShortcutController a) => O.OverloadedMethod ShortcutControllerGetMnemonicsModifiersMethodInfo a signature where
    overloadedMethod = shortcutControllerGetMnemonicsModifiers

instance O.OverloadedMethodInfo ShortcutControllerGetMnemonicsModifiersMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.shortcutControllerGetMnemonicsModifiers",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#v:shortcutControllerGetMnemonicsModifiers"
        })


#endif

-- method ShortcutController::get_scope
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ShortcutController" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkShortcutController`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "ShortcutScope" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_shortcut_controller_get_scope" gtk_shortcut_controller_get_scope :: 
    Ptr ShortcutController ->               -- self : TInterface (Name {namespace = "Gtk", name = "ShortcutController"})
    IO CUInt

-- | Gets the scope for when this controller activates its shortcuts.
-- 
-- See 'GI.Gtk.Objects.ShortcutController.shortcutControllerSetScope' for details.
shortcutControllerGetScope ::
    (B.CallStack.HasCallStack, MonadIO m, IsShortcutController a) =>
    a
    -- ^ /@self@/: a @GtkShortcutController@
    -> m Gtk.Enums.ShortcutScope
    -- ^ __Returns:__ the controller\'s scope
shortcutControllerGetScope self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_shortcut_controller_get_scope self'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerGetScopeMethodInfo
instance (signature ~ (m Gtk.Enums.ShortcutScope), MonadIO m, IsShortcutController a) => O.OverloadedMethod ShortcutControllerGetScopeMethodInfo a signature where
    overloadedMethod = shortcutControllerGetScope

instance O.OverloadedMethodInfo ShortcutControllerGetScopeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.shortcutControllerGetScope",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#v:shortcutControllerGetScope"
        })


#endif

-- method ShortcutController::remove_shortcut
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ShortcutController" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the controller" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "shortcut"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Shortcut" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkShortcut`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_shortcut_controller_remove_shortcut" gtk_shortcut_controller_remove_shortcut :: 
    Ptr ShortcutController ->               -- self : TInterface (Name {namespace = "Gtk", name = "ShortcutController"})
    Ptr Gtk.Shortcut.Shortcut ->            -- shortcut : TInterface (Name {namespace = "Gtk", name = "Shortcut"})
    IO ()

-- | Removes /@shortcut@/ from the list of shortcuts handled by /@self@/.
-- 
-- If /@shortcut@/ had not been added to /@controller@/ or this controller
-- uses an external shortcut list, this function does nothing.
shortcutControllerRemoveShortcut ::
    (B.CallStack.HasCallStack, MonadIO m, IsShortcutController a, Gtk.Shortcut.IsShortcut b) =>
    a
    -- ^ /@self@/: the controller
    -> b
    -- ^ /@shortcut@/: a @GtkShortcut@
    -> m ()
shortcutControllerRemoveShortcut self shortcut = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    shortcut' <- unsafeManagedPtrCastPtr shortcut
    gtk_shortcut_controller_remove_shortcut self' shortcut'
    touchManagedPtr self
    touchManagedPtr shortcut
    return ()

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerRemoveShortcutMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsShortcutController a, Gtk.Shortcut.IsShortcut b) => O.OverloadedMethod ShortcutControllerRemoveShortcutMethodInfo a signature where
    overloadedMethod = shortcutControllerRemoveShortcut

instance O.OverloadedMethodInfo ShortcutControllerRemoveShortcutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.shortcutControllerRemoveShortcut",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#v:shortcutControllerRemoveShortcut"
        })


#endif

-- method ShortcutController::set_mnemonics_modifiers
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ShortcutController" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkShortcutController`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "modifiers"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "ModifierType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new mnemonics_modifiers to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_shortcut_controller_set_mnemonics_modifiers" gtk_shortcut_controller_set_mnemonics_modifiers :: 
    Ptr ShortcutController ->               -- self : TInterface (Name {namespace = "Gtk", name = "ShortcutController"})
    CUInt ->                                -- modifiers : TInterface (Name {namespace = "Gdk", name = "ModifierType"})
    IO ()

-- | Sets the controller to use the given modifier for mnemonics.
-- 
-- The mnemonics modifiers determines which modifiers need to be pressed to allow
-- activation of shortcuts with mnemonics triggers.
-- 
-- GTK normally uses the Alt modifier for mnemonics, except in @GtkPopoverMenu@s,
-- where mnemonics can be triggered without any modifiers. It should be very
-- rarely necessary to change this, and doing so is likely to interfere with
-- other shortcuts.
-- 
-- This value is only relevant for local shortcut controllers. Global and managed
-- shortcut controllers will have their shortcuts activated from other places which
-- have their own modifiers for activating mnemonics.
shortcutControllerSetMnemonicsModifiers ::
    (B.CallStack.HasCallStack, MonadIO m, IsShortcutController a) =>
    a
    -- ^ /@self@/: a @GtkShortcutController@
    -> [Gdk.Flags.ModifierType]
    -- ^ /@modifiers@/: the new mnemonics_modifiers to use
    -> m ()
shortcutControllerSetMnemonicsModifiers self modifiers = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let modifiers' = gflagsToWord modifiers
    gtk_shortcut_controller_set_mnemonics_modifiers self' modifiers'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerSetMnemonicsModifiersMethodInfo
instance (signature ~ ([Gdk.Flags.ModifierType] -> m ()), MonadIO m, IsShortcutController a) => O.OverloadedMethod ShortcutControllerSetMnemonicsModifiersMethodInfo a signature where
    overloadedMethod = shortcutControllerSetMnemonicsModifiers

instance O.OverloadedMethodInfo ShortcutControllerSetMnemonicsModifiersMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.shortcutControllerSetMnemonicsModifiers",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#v:shortcutControllerSetMnemonicsModifiers"
        })


#endif

-- method ShortcutController::set_scope
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ShortcutController" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkShortcutController`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scope"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ShortcutScope" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new scope to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_shortcut_controller_set_scope" gtk_shortcut_controller_set_scope :: 
    Ptr ShortcutController ->               -- self : TInterface (Name {namespace = "Gtk", name = "ShortcutController"})
    CUInt ->                                -- scope : TInterface (Name {namespace = "Gtk", name = "ShortcutScope"})
    IO ()

-- | Sets the controller to have the given /@scope@/.
-- 
-- The scope allows shortcuts to be activated outside of the normal
-- event propagation. In particular, it allows installing global
-- keyboard shortcuts that can be activated even when a widget does
-- not have focus.
-- 
-- With 'GI.Gtk.Enums.ShortcutScopeLocal', shortcuts will only be activated
-- when the widget has focus.
shortcutControllerSetScope ::
    (B.CallStack.HasCallStack, MonadIO m, IsShortcutController a) =>
    a
    -- ^ /@self@/: a @GtkShortcutController@
    -> Gtk.Enums.ShortcutScope
    -- ^ /@scope@/: the new scope to use
    -> m ()
shortcutControllerSetScope self scope = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let scope' = (fromIntegral . fromEnum) scope
    gtk_shortcut_controller_set_scope self' scope'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data ShortcutControllerSetScopeMethodInfo
instance (signature ~ (Gtk.Enums.ShortcutScope -> m ()), MonadIO m, IsShortcutController a) => O.OverloadedMethod ShortcutControllerSetScopeMethodInfo a signature where
    overloadedMethod = shortcutControllerSetScope

instance O.OverloadedMethodInfo ShortcutControllerSetScopeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ShortcutController.shortcutControllerSetScope",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ShortcutController.html#v:shortcutControllerSetScope"
        })


#endif


