{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Handles the preferred size and allocation for children of a widget.
-- 
-- You typically subclass @GtkLayoutManager@ if you want to implement a
-- layout policy for the children of a widget, or if you want to determine
-- the size of a widget depending on its contents.
-- 
-- Each @GtkWidget@ can only have a @GtkLayoutManager@ instance associated
-- to it at any given time; it is possible, though, to replace the layout
-- manager instance using 'GI.Gtk.Objects.Widget.widgetSetLayoutManager'.
-- 
-- == Layout properties
-- 
-- A layout manager can expose properties for controlling the layout of
-- each child, by creating an object type derived from t'GI.Gtk.Objects.LayoutChild.LayoutChild'
-- and installing the properties on it as normal @GObject@ properties.
-- 
-- Each @GtkLayoutChild@ instance storing the layout properties for a
-- specific child is created through the 'GI.Gtk.Objects.LayoutManager.layoutManagerGetLayoutChild'
-- method; a @GtkLayoutManager@ controls the creation of its @GtkLayoutChild@
-- instances by overriding the GtkLayoutManagerClass.@/create_layout_child()/@
-- virtual function. The typical implementation should look like:
-- 
-- 
-- === /c code/
-- >static GtkLayoutChild *
-- >create_layout_child (GtkLayoutManager *manager,
-- >                     GtkWidget        *container,
-- >                     GtkWidget        *child)
-- >{
-- >  return g_object_new (your_layout_child_get_type (),
-- >                       "layout-manager", manager,
-- >                       "child-widget", child,
-- >                       NULL);
-- >}
-- 
-- 
-- The [LayoutChild:layoutManager]("GI.Gtk.Objects.LayoutChild#g:attr:layoutManager") and
-- [LayoutChild:childWidget]("GI.Gtk.Objects.LayoutChild#g:attr:childWidget") properties
-- on the newly created @GtkLayoutChild@ instance are mandatory. The
-- @GtkLayoutManager@ will cache the newly created @GtkLayoutChild@ instance
-- until the widget is removed from its parent, or the parent removes the
-- layout manager.
-- 
-- Each @GtkLayoutManager@ instance creating a @GtkLayoutChild@ should use
-- 'GI.Gtk.Objects.LayoutManager.layoutManagerGetLayoutChild' every time it needs to query
-- the layout properties; each @GtkLayoutChild@ instance should call
-- 'GI.Gtk.Objects.LayoutManager.layoutManagerLayoutChanged' every time a property is
-- updated, in order to queue a new size measuring and allocation.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.LayoutManager
    ( 

-- * Exported types
    LayoutManager(..)                       ,
    IsLayoutManager                         ,
    toLayoutManager                         ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [allocate]("GI.Gtk.Objects.LayoutManager#g:method:allocate"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [layoutChanged]("GI.Gtk.Objects.LayoutManager#g:method:layoutChanged"), [measure]("GI.Gtk.Objects.LayoutManager#g:method:measure"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getLayoutChild]("GI.Gtk.Objects.LayoutManager#g:method:getLayoutChild"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRequestMode]("GI.Gtk.Objects.LayoutManager#g:method:getRequestMode"), [getWidget]("GI.Gtk.Objects.LayoutManager#g:method:getWidget").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveLayoutManagerMethod              ,
#endif

-- ** allocate #method:allocate#

#if defined(ENABLE_OVERLOADING)
    LayoutManagerAllocateMethodInfo         ,
#endif
    layoutManagerAllocate                   ,


-- ** getLayoutChild #method:getLayoutChild#

#if defined(ENABLE_OVERLOADING)
    LayoutManagerGetLayoutChildMethodInfo   ,
#endif
    layoutManagerGetLayoutChild             ,


-- ** getRequestMode #method:getRequestMode#

#if defined(ENABLE_OVERLOADING)
    LayoutManagerGetRequestModeMethodInfo   ,
#endif
    layoutManagerGetRequestMode             ,


-- ** getWidget #method:getWidget#

#if defined(ENABLE_OVERLOADING)
    LayoutManagerGetWidgetMethodInfo        ,
#endif
    layoutManagerGetWidget                  ,


-- ** layoutChanged #method:layoutChanged#

#if defined(ENABLE_OVERLOADING)
    LayoutManagerLayoutChangedMethodInfo    ,
#endif
    layoutManagerLayoutChanged              ,


-- ** measure #method:measure#

#if defined(ENABLE_OVERLOADING)
    LayoutManagerMeasureMethodInfo          ,
#endif
    layoutManagerMeasure                    ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget

#endif

-- | Memory-managed wrapper type.
newtype LayoutManager = LayoutManager (SP.ManagedPtr LayoutManager)
    deriving (Eq)

instance SP.ManagedPtrNewtype LayoutManager where
    toManagedPtr (LayoutManager p) = p

foreign import ccall "gtk_layout_manager_get_type"
    c_gtk_layout_manager_get_type :: IO B.Types.GType

instance B.Types.TypedObject LayoutManager where
    glibType = c_gtk_layout_manager_get_type

instance B.Types.GObject LayoutManager

-- | Type class for types which can be safely cast to `LayoutManager`, for instance with `toLayoutManager`.
class (SP.GObject o, O.IsDescendantOf LayoutManager o) => IsLayoutManager o
instance (SP.GObject o, O.IsDescendantOf LayoutManager o) => IsLayoutManager o

instance O.HasParentTypes LayoutManager
type instance O.ParentTypes LayoutManager = '[GObject.Object.Object]

-- | Cast to `LayoutManager`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toLayoutManager :: (MIO.MonadIO m, IsLayoutManager o) => o -> m LayoutManager
toLayoutManager = MIO.liftIO . B.ManagedPtr.unsafeCastTo LayoutManager

-- | Convert 'LayoutManager' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe LayoutManager) where
    gvalueGType_ = c_gtk_layout_manager_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr LayoutManager)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr LayoutManager)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject LayoutManager ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveLayoutManagerMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveLayoutManagerMethod "allocate" o = LayoutManagerAllocateMethodInfo
    ResolveLayoutManagerMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveLayoutManagerMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveLayoutManagerMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveLayoutManagerMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveLayoutManagerMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveLayoutManagerMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveLayoutManagerMethod "layoutChanged" o = LayoutManagerLayoutChangedMethodInfo
    ResolveLayoutManagerMethod "measure" o = LayoutManagerMeasureMethodInfo
    ResolveLayoutManagerMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveLayoutManagerMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveLayoutManagerMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveLayoutManagerMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveLayoutManagerMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveLayoutManagerMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveLayoutManagerMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveLayoutManagerMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveLayoutManagerMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveLayoutManagerMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveLayoutManagerMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveLayoutManagerMethod "getLayoutChild" o = LayoutManagerGetLayoutChildMethodInfo
    ResolveLayoutManagerMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveLayoutManagerMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveLayoutManagerMethod "getRequestMode" o = LayoutManagerGetRequestModeMethodInfo
    ResolveLayoutManagerMethod "getWidget" o = LayoutManagerGetWidgetMethodInfo
    ResolveLayoutManagerMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveLayoutManagerMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveLayoutManagerMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveLayoutManagerMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveLayoutManagerMethod t LayoutManager, O.OverloadedMethod info LayoutManager p) => OL.IsLabel t (LayoutManager -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveLayoutManagerMethod t LayoutManager, O.OverloadedMethod info LayoutManager p, R.HasField t LayoutManager p) => R.HasField t LayoutManager p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveLayoutManagerMethod t LayoutManager, O.OverloadedMethodInfo info LayoutManager) => OL.IsLabel t (O.MethodProxy info LayoutManager) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList LayoutManager
type instance O.AttributeList LayoutManager = LayoutManagerAttributeList
type LayoutManagerAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList LayoutManager = LayoutManagerSignalList
type LayoutManagerSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method LayoutManager::allocate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "manager"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "LayoutManager" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkLayoutManager`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GtkWidget` using @manager"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new width of the @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new height of the @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the baseline position of the @widget, or -1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_layout_manager_allocate" gtk_layout_manager_allocate :: 
    Ptr LayoutManager ->                    -- manager : TInterface (Name {namespace = "Gtk", name = "LayoutManager"})
    Ptr Gtk.Widget.Widget ->                -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- width : TBasicType TInt
    Int32 ->                                -- height : TBasicType TInt
    Int32 ->                                -- baseline : TBasicType TInt
    IO ()

-- | Assigns the given /@width@/, /@height@/, and /@baseline@/ to
-- a /@widget@/, and computes the position and sizes of the children of
-- the /@widget@/ using the layout management policy of /@manager@/.
layoutManagerAllocate ::
    (B.CallStack.HasCallStack, MonadIO m, IsLayoutManager a, Gtk.Widget.IsWidget b) =>
    a
    -- ^ /@manager@/: a @GtkLayoutManager@
    -> b
    -- ^ /@widget@/: the @GtkWidget@ using /@manager@/
    -> Int32
    -- ^ /@width@/: the new width of the /@widget@/
    -> Int32
    -- ^ /@height@/: the new height of the /@widget@/
    -> Int32
    -- ^ /@baseline@/: the baseline position of the /@widget@/, or -1
    -> m ()
layoutManagerAllocate manager widget width height baseline = liftIO $ do
    manager' <- unsafeManagedPtrCastPtr manager
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_layout_manager_allocate manager' widget' width height baseline
    touchManagedPtr manager
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data LayoutManagerAllocateMethodInfo
instance (signature ~ (b -> Int32 -> Int32 -> Int32 -> m ()), MonadIO m, IsLayoutManager a, Gtk.Widget.IsWidget b) => O.OverloadedMethod LayoutManagerAllocateMethodInfo a signature where
    overloadedMethod = layoutManagerAllocate

instance O.OverloadedMethodInfo LayoutManagerAllocateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.LayoutManager.layoutManagerAllocate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-LayoutManager.html#v:layoutManagerAllocate"
        })


#endif

-- method LayoutManager::get_layout_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "manager"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "LayoutManager" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkLayoutManager`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "child"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkWidget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "LayoutChild" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_layout_manager_get_layout_child" gtk_layout_manager_get_layout_child :: 
    Ptr LayoutManager ->                    -- manager : TInterface (Name {namespace = "Gtk", name = "LayoutManager"})
    Ptr Gtk.Widget.Widget ->                -- child : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.LayoutChild.LayoutChild)

-- | Retrieves a @GtkLayoutChild@ instance for the @GtkLayoutManager@,
-- creating one if necessary.
-- 
-- The /@child@/ widget must be a child of the widget using /@manager@/.
-- 
-- The @GtkLayoutChild@ instance is owned by the @GtkLayoutManager@,
-- and is guaranteed to exist as long as /@child@/ is a child of the
-- @GtkWidget@ using the given @GtkLayoutManager@.
layoutManagerGetLayoutChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsLayoutManager a, Gtk.Widget.IsWidget b) =>
    a
    -- ^ /@manager@/: a @GtkLayoutManager@
    -> b
    -- ^ /@child@/: a @GtkWidget@
    -> m Gtk.LayoutChild.LayoutChild
    -- ^ __Returns:__ a @GtkLayoutChild@
layoutManagerGetLayoutChild manager child = liftIO $ do
    manager' <- unsafeManagedPtrCastPtr manager
    child' <- unsafeManagedPtrCastPtr child
    result <- gtk_layout_manager_get_layout_child manager' child'
    checkUnexpectedReturnNULL "layoutManagerGetLayoutChild" result
    result' <- (newObject Gtk.LayoutChild.LayoutChild) result
    touchManagedPtr manager
    touchManagedPtr child
    return result'

#if defined(ENABLE_OVERLOADING)
data LayoutManagerGetLayoutChildMethodInfo
instance (signature ~ (b -> m Gtk.LayoutChild.LayoutChild), MonadIO m, IsLayoutManager a, Gtk.Widget.IsWidget b) => O.OverloadedMethod LayoutManagerGetLayoutChildMethodInfo a signature where
    overloadedMethod = layoutManagerGetLayoutChild

instance O.OverloadedMethodInfo LayoutManagerGetLayoutChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.LayoutManager.layoutManagerGetLayoutChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-LayoutManager.html#v:layoutManagerGetLayoutChild"
        })


#endif

-- method LayoutManager::get_request_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "manager"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "LayoutManager" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkLayoutManager`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "SizeRequestMode" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_layout_manager_get_request_mode" gtk_layout_manager_get_request_mode :: 
    Ptr LayoutManager ->                    -- manager : TInterface (Name {namespace = "Gtk", name = "LayoutManager"})
    IO CUInt

-- | Retrieves the request mode of /@manager@/.
layoutManagerGetRequestMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsLayoutManager a) =>
    a
    -- ^ /@manager@/: a @GtkLayoutManager@
    -> m Gtk.Enums.SizeRequestMode
    -- ^ __Returns:__ a @GtkSizeRequestMode@
layoutManagerGetRequestMode manager = liftIO $ do
    manager' <- unsafeManagedPtrCastPtr manager
    result <- gtk_layout_manager_get_request_mode manager'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr manager
    return result'

#if defined(ENABLE_OVERLOADING)
data LayoutManagerGetRequestModeMethodInfo
instance (signature ~ (m Gtk.Enums.SizeRequestMode), MonadIO m, IsLayoutManager a) => O.OverloadedMethod LayoutManagerGetRequestModeMethodInfo a signature where
    overloadedMethod = layoutManagerGetRequestMode

instance O.OverloadedMethodInfo LayoutManagerGetRequestModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.LayoutManager.layoutManagerGetRequestMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-LayoutManager.html#v:layoutManagerGetRequestMode"
        })


#endif

-- method LayoutManager::get_widget
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "manager"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "LayoutManager" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkLayoutManager`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_layout_manager_get_widget" gtk_layout_manager_get_widget :: 
    Ptr LayoutManager ->                    -- manager : TInterface (Name {namespace = "Gtk", name = "LayoutManager"})
    IO (Ptr Gtk.Widget.Widget)

-- | Retrieves the @GtkWidget@ using the given @GtkLayoutManager@.
layoutManagerGetWidget ::
    (B.CallStack.HasCallStack, MonadIO m, IsLayoutManager a) =>
    a
    -- ^ /@manager@/: a @GtkLayoutManager@
    -> m (Maybe Gtk.Widget.Widget)
    -- ^ __Returns:__ a @GtkWidget@
layoutManagerGetWidget manager = liftIO $ do
    manager' <- unsafeManagedPtrCastPtr manager
    result <- gtk_layout_manager_get_widget manager'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gtk.Widget.Widget) result'
        return result''
    touchManagedPtr manager
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data LayoutManagerGetWidgetMethodInfo
instance (signature ~ (m (Maybe Gtk.Widget.Widget)), MonadIO m, IsLayoutManager a) => O.OverloadedMethod LayoutManagerGetWidgetMethodInfo a signature where
    overloadedMethod = layoutManagerGetWidget

instance O.OverloadedMethodInfo LayoutManagerGetWidgetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.LayoutManager.layoutManagerGetWidget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-LayoutManager.html#v:layoutManagerGetWidget"
        })


#endif

-- method LayoutManager::layout_changed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "manager"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "LayoutManager" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkLayoutManager`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_layout_manager_layout_changed" gtk_layout_manager_layout_changed :: 
    Ptr LayoutManager ->                    -- manager : TInterface (Name {namespace = "Gtk", name = "LayoutManager"})
    IO ()

-- | Queues a resize on the @GtkWidget@ using /@manager@/, if any.
-- 
-- This function should be called by subclasses of @GtkLayoutManager@
-- in response to changes to their layout management policies.
layoutManagerLayoutChanged ::
    (B.CallStack.HasCallStack, MonadIO m, IsLayoutManager a) =>
    a
    -- ^ /@manager@/: a @GtkLayoutManager@
    -> m ()
layoutManagerLayoutChanged manager = liftIO $ do
    manager' <- unsafeManagedPtrCastPtr manager
    gtk_layout_manager_layout_changed manager'
    touchManagedPtr manager
    return ()

#if defined(ENABLE_OVERLOADING)
data LayoutManagerLayoutChangedMethodInfo
instance (signature ~ (m ()), MonadIO m, IsLayoutManager a) => O.OverloadedMethod LayoutManagerLayoutChangedMethodInfo a signature where
    overloadedMethod = layoutManagerLayoutChanged

instance O.OverloadedMethodInfo LayoutManagerLayoutChangedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.LayoutManager.layoutManagerLayoutChanged",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-LayoutManager.html#v:layoutManagerLayoutChanged"
        })


#endif

-- method LayoutManager::measure
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "manager"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "LayoutManager" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkLayoutManager`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GtkWidget` using @manager"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "orientation"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Orientation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the orientation to measure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "for_size"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Size for the opposite of @orientation; for instance, if\n  the @orientation is %GTK_ORIENTATION_HORIZONTAL, this is the height\n  of the widget; if the @orientation is %GTK_ORIENTATION_VERTICAL, this\n  is the width of the widget. This allows to measure the height for the\n  given width, and the width for the given height. Use -1 if the size\n  is not known"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the minimum size for the given size and\n  orientation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the natural, or preferred size for the\n  given size and orientation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "minimum_baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the baseline position for the\n  minimum size"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural_baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the baseline position for the\n  natural size"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_layout_manager_measure" gtk_layout_manager_measure :: 
    Ptr LayoutManager ->                    -- manager : TInterface (Name {namespace = "Gtk", name = "LayoutManager"})
    Ptr Gtk.Widget.Widget ->                -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- orientation : TInterface (Name {namespace = "Gtk", name = "Orientation"})
    Int32 ->                                -- for_size : TBasicType TInt
    Ptr Int32 ->                            -- minimum : TBasicType TInt
    Ptr Int32 ->                            -- natural : TBasicType TInt
    Ptr Int32 ->                            -- minimum_baseline : TBasicType TInt
    Ptr Int32 ->                            -- natural_baseline : TBasicType TInt
    IO ()

-- | Measures the size of the /@widget@/ using /@manager@/, for the
-- given /@orientation@/ and size.
-- 
-- See the t'GI.Gtk.Objects.Widget.Widget' documentation on layout management for
-- more details.
layoutManagerMeasure ::
    (B.CallStack.HasCallStack, MonadIO m, IsLayoutManager a, Gtk.Widget.IsWidget b) =>
    a
    -- ^ /@manager@/: a @GtkLayoutManager@
    -> b
    -- ^ /@widget@/: the @GtkWidget@ using /@manager@/
    -> Gtk.Enums.Orientation
    -- ^ /@orientation@/: the orientation to measure
    -> Int32
    -- ^ /@forSize@/: Size for the opposite of /@orientation@/; for instance, if
    --   the /@orientation@/ is 'GI.Gtk.Enums.OrientationHorizontal', this is the height
    --   of the widget; if the /@orientation@/ is 'GI.Gtk.Enums.OrientationVertical', this
    --   is the width of the widget. This allows to measure the height for the
    --   given width, and the width for the given height. Use -1 if the size
    --   is not known
    -> m ((Int32, Int32, Int32, Int32))
layoutManagerMeasure manager widget orientation forSize = liftIO $ do
    manager' <- unsafeManagedPtrCastPtr manager
    widget' <- unsafeManagedPtrCastPtr widget
    let orientation' = (fromIntegral . fromEnum) orientation
    minimum <- allocMem :: IO (Ptr Int32)
    natural <- allocMem :: IO (Ptr Int32)
    minimumBaseline <- allocMem :: IO (Ptr Int32)
    naturalBaseline <- allocMem :: IO (Ptr Int32)
    gtk_layout_manager_measure manager' widget' orientation' forSize minimum natural minimumBaseline naturalBaseline
    minimum' <- peek minimum
    natural' <- peek natural
    minimumBaseline' <- peek minimumBaseline
    naturalBaseline' <- peek naturalBaseline
    touchManagedPtr manager
    touchManagedPtr widget
    freeMem minimum
    freeMem natural
    freeMem minimumBaseline
    freeMem naturalBaseline
    return (minimum', natural', minimumBaseline', naturalBaseline')

#if defined(ENABLE_OVERLOADING)
data LayoutManagerMeasureMethodInfo
instance (signature ~ (b -> Gtk.Enums.Orientation -> Int32 -> m ((Int32, Int32, Int32, Int32))), MonadIO m, IsLayoutManager a, Gtk.Widget.IsWidget b) => O.OverloadedMethod LayoutManagerMeasureMethodInfo a signature where
    overloadedMethod = layoutManagerMeasure

instance O.OverloadedMethodInfo LayoutManagerMeasureMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.LayoutManager.layoutManagerMeasure",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-LayoutManager.html#v:layoutManagerMeasure"
        })


#endif


