{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Uses constraints to describe relations between widgets.
-- 
-- @GtkConstraintLayout@ is a layout manager that uses relations between
-- widget attributes, expressed via t'GI.Gtk.Objects.Constraint.Constraint' instances, to
-- measure and allocate widgets.
-- 
-- === How do constraints work
-- 
-- Constraints are objects defining the relationship between attributes
-- of a widget; you can read the description of the t'GI.Gtk.Objects.Constraint.Constraint'
-- class to have a more in depth definition.
-- 
-- By taking multiple constraints and applying them to the children of
-- a widget using @GtkConstraintLayout@, it\'s possible to describe
-- complex layout policies; each constraint applied to a child or to the parent
-- widgets contributes to the full description of the layout, in terms of
-- parameters for resolving the value of each attribute.
-- 
-- It is important to note that a layout is defined by the totality of
-- constraints; removing a child, or a constraint, from an existing layout
-- without changing the remaining constraints may result in an unstable
-- or unsolvable layout.
-- 
-- Constraints have an implicit \"reading order\"; you should start describing
-- each edge of each child, as well as their relationship with the parent
-- container, from the top left (or top right, in RTL languages), horizontally
-- first, and then vertically.
-- 
-- A constraint-based layout with too few constraints can become \"unstable\",
-- that is: have more than one solution. The behavior of an unstable layout
-- is undefined.
-- 
-- A constraint-based layout with conflicting constraints may be unsolvable,
-- and lead to an unstable layout. You can use the [Constraint:strength]("GI.Gtk.Objects.Constraint#g:attr:strength")
-- property of t'GI.Gtk.Objects.Constraint.Constraint' to \"nudge\" the layout towards a solution.
-- 
-- === GtkConstraintLayout as GtkBuildable
-- 
-- @GtkConstraintLayout@ implements the t'GI.Gtk.Interfaces.Buildable.Buildable' interface and
-- has a custom \"constraints\" element which allows describing constraints in
-- a t'GI.Gtk.Objects.Builder.Builder' UI file.
-- 
-- An example of a UI definition fragment specifying a constraint:
-- 
-- 
-- === /xml code/
-- >  <object class="GtkConstraintLayout">
-- >    <constraints>
-- >      <constraint target="button" target-attribute="start"
-- >                  relation="eq"
-- >                  source="super" source-attribute="start"
-- >                  constant="12"
-- >                  strength="required" />
-- >      <constraint target="button" target-attribute="width"
-- >                  relation="ge"
-- >                  constant="250"
-- >                  strength="strong" />
-- >    </constraints>
-- >  </object>
-- 
-- 
-- The definition above will add two constraints to the GtkConstraintLayout:
-- 
--  - a required constraint between the leading edge of \"button\" and
--    the leading edge of the widget using the constraint layout, plus
--    12 pixels
--  - a strong, constant constraint making the width of \"button\" greater
--    than, or equal to 250 pixels
-- 
-- The \"target\" and \"target-attribute\" attributes are required.
-- 
-- The \"source\" and \"source-attribute\" attributes of the \"constraint\"
-- element are optional; if they are not specified, the constraint is
-- assumed to be a constant.
-- 
-- The \"relation\" attribute is optional; if not specified, the constraint
-- is assumed to be an equality.
-- 
-- The \"strength\" attribute is optional; if not specified, the constraint
-- is assumed to be required.
-- 
-- The \"source\" and \"target\" attributes can be set to \"super\" to indicate
-- that the constraint target is the widget using the GtkConstraintLayout.
-- 
-- There can be \"constant\" and \"multiplier\" attributes.
-- 
-- Additionally, the \"constraints\" element can also contain a description
-- of the @GtkConstraintGuides@ used by the layout:
-- 
-- 
-- === /xml code/
-- >  <constraints>
-- >    <guide min-width="100" max-width="500" name="hspace"/>
-- >    <guide min-height="64" nat-height="128" name="vspace" strength="strong"/>
-- >  </constraints>
-- 
-- 
-- The \"guide\" element has the following optional attributes:
-- 
--   - \"min-width\", \"nat-width\", and \"max-width\", describe the minimum,
--     natural, and maximum width of the guide, respectively
--   - \"min-height\", \"nat-height\", and \"max-height\", describe the minimum,
--     natural, and maximum height of the guide, respectively
--   - \"strength\" describes the strength of the constraint on the natural
--     size of the guide; if not specified, the constraint is assumed to
--     have a medium strength
--   - \"name\" describes a name for the guide, useful when debugging
-- 
-- === Using the Visual Format Language
-- 
-- Complex constraints can be described using a compact syntax called VFL,
-- or *Visual Format Language*.
-- 
-- The Visual Format Language describes all the constraints on a row or
-- column, typically starting from the leading edge towards the trailing
-- one. Each element of the layout is composed by \"views\", which identify
-- a t'GI.Gtk.Interfaces.ConstraintTarget.ConstraintTarget'.
-- 
-- For instance:
-- 
-- >  [button]-[textField]
-- 
-- 
-- Describes a constraint that binds the trailing edge of \"button\" to the
-- leading edge of \"textField\", leaving a default space between the two.
-- 
-- Using VFL is also possible to specify predicates that describe constraints
-- on attributes like width and height:
-- 
-- >  // Width must be greater than, or equal to 50
-- >  [button(>=50)]
-- >
-- >  // Width of button1 must be equal to width of button2
-- >  [button1(==button2)]
-- 
-- 
-- The default orientation for a VFL description is horizontal, unless
-- otherwise specified:
-- 
-- >  // horizontal orientation, default attribute: width
-- >  H:[button(>=150)]
-- >
-- >  // vertical orientation, default attribute: height
-- >  V:[button1(==button2)]
-- 
-- 
-- It\'s also possible to specify multiple predicates, as well as their
-- strength:
-- 
-- >  // minimum width of button must be 150
-- >  // natural width of button can be 250
-- >  [button(>=150@required, ==250@medium)]
-- 
-- 
-- Finally, it\'s also possible to use simple arithmetic operators:
-- 
-- >  // width of button1 must be equal to width of button2
-- >  // divided by 2 plus 12
-- >  [button1(button2 / 2 + 12)]
-- 

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.ConstraintLayout
    ( 
#if defined(ENABLE_OVERLOADING)
    ConstraintLayoutAddConstraintsFromDescriptionMethodInfo,
#endif

-- * Exported types
    ConstraintLayout(..)                    ,
    IsConstraintLayout                      ,
    toConstraintLayout                      ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [addConstraint]("GI.Gtk.Objects.ConstraintLayout#g:method:addConstraint"), [addConstraintsFromDescription]("GI.Gtk.Objects.ConstraintLayout#g:method:addConstraintsFromDescription"), [addGuide]("GI.Gtk.Objects.ConstraintLayout#g:method:addGuide"), [allocate]("GI.Gtk.Objects.LayoutManager#g:method:allocate"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [layoutChanged]("GI.Gtk.Objects.LayoutManager#g:method:layoutChanged"), [measure]("GI.Gtk.Objects.LayoutManager#g:method:measure"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [observeConstraints]("GI.Gtk.Objects.ConstraintLayout#g:method:observeConstraints"), [observeGuides]("GI.Gtk.Objects.ConstraintLayout#g:method:observeGuides"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeAllConstraints]("GI.Gtk.Objects.ConstraintLayout#g:method:removeAllConstraints"), [removeConstraint]("GI.Gtk.Objects.ConstraintLayout#g:method:removeConstraint"), [removeGuide]("GI.Gtk.Objects.ConstraintLayout#g:method:removeGuide"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getBuildableId]("GI.Gtk.Interfaces.Buildable#g:method:getBuildableId"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getLayoutChild]("GI.Gtk.Objects.LayoutManager#g:method:getLayoutChild"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRequestMode]("GI.Gtk.Objects.LayoutManager#g:method:getRequestMode"), [getWidget]("GI.Gtk.Objects.LayoutManager#g:method:getWidget").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveConstraintLayoutMethod           ,
#endif

-- ** addConstraint #method:addConstraint#

#if defined(ENABLE_OVERLOADING)
    ConstraintLayoutAddConstraintMethodInfo ,
#endif
    constraintLayoutAddConstraint           ,


-- ** addGuide #method:addGuide#

#if defined(ENABLE_OVERLOADING)
    ConstraintLayoutAddGuideMethodInfo      ,
#endif
    constraintLayoutAddGuide                ,


-- ** new #method:new#

    constraintLayoutNew                     ,


-- ** observeConstraints #method:observeConstraints#

#if defined(ENABLE_OVERLOADING)
    ConstraintLayoutObserveConstraintsMethodInfo,
#endif
    constraintLayoutObserveConstraints      ,


-- ** observeGuides #method:observeGuides#

#if defined(ENABLE_OVERLOADING)
    ConstraintLayoutObserveGuidesMethodInfo ,
#endif
    constraintLayoutObserveGuides           ,


-- ** removeAllConstraints #method:removeAllConstraints#

#if defined(ENABLE_OVERLOADING)
    ConstraintLayoutRemoveAllConstraintsMethodInfo,
#endif
    constraintLayoutRemoveAllConstraints    ,


-- ** removeConstraint #method:removeConstraint#

#if defined(ENABLE_OVERLOADING)
    ConstraintLayoutRemoveConstraintMethodInfo,
#endif
    constraintLayoutRemoveConstraint        ,


-- ** removeGuide #method:removeGuide#

#if defined(ENABLE_OVERLOADING)
    ConstraintLayoutRemoveGuideMethodInfo   ,
#endif
    constraintLayoutRemoveGuide             ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Constraint as Gtk.Constraint
import {-# SOURCE #-} qualified GI.Gtk.Objects.ConstraintGuide as Gtk.ConstraintGuide
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Objects.Constraint as Gtk.Constraint
import {-# SOURCE #-} qualified GI.Gtk.Objects.ConstraintGuide as Gtk.ConstraintGuide
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager

#endif

-- | Memory-managed wrapper type.
newtype ConstraintLayout = ConstraintLayout (SP.ManagedPtr ConstraintLayout)
    deriving (Eq)

instance SP.ManagedPtrNewtype ConstraintLayout where
    toManagedPtr (ConstraintLayout p) = p

foreign import ccall "gtk_constraint_layout_get_type"
    c_gtk_constraint_layout_get_type :: IO B.Types.GType

instance B.Types.TypedObject ConstraintLayout where
    glibType = c_gtk_constraint_layout_get_type

instance B.Types.GObject ConstraintLayout

-- | Type class for types which can be safely cast to `ConstraintLayout`, for instance with `toConstraintLayout`.
class (SP.GObject o, O.IsDescendantOf ConstraintLayout o) => IsConstraintLayout o
instance (SP.GObject o, O.IsDescendantOf ConstraintLayout o) => IsConstraintLayout o

instance O.HasParentTypes ConstraintLayout
type instance O.ParentTypes ConstraintLayout = '[Gtk.LayoutManager.LayoutManager, GObject.Object.Object, Gtk.Buildable.Buildable]

-- | Cast to `ConstraintLayout`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toConstraintLayout :: (MIO.MonadIO m, IsConstraintLayout o) => o -> m ConstraintLayout
toConstraintLayout = MIO.liftIO . B.ManagedPtr.unsafeCastTo ConstraintLayout

-- | Convert 'ConstraintLayout' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe ConstraintLayout) where
    gvalueGType_ = c_gtk_constraint_layout_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr ConstraintLayout)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr ConstraintLayout)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject ConstraintLayout ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveConstraintLayoutMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveConstraintLayoutMethod "addConstraint" o = ConstraintLayoutAddConstraintMethodInfo
    ResolveConstraintLayoutMethod "addConstraintsFromDescription" o = ConstraintLayoutAddConstraintsFromDescriptionMethodInfo
    ResolveConstraintLayoutMethod "addGuide" o = ConstraintLayoutAddGuideMethodInfo
    ResolveConstraintLayoutMethod "allocate" o = Gtk.LayoutManager.LayoutManagerAllocateMethodInfo
    ResolveConstraintLayoutMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveConstraintLayoutMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveConstraintLayoutMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveConstraintLayoutMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveConstraintLayoutMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveConstraintLayoutMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveConstraintLayoutMethod "layoutChanged" o = Gtk.LayoutManager.LayoutManagerLayoutChangedMethodInfo
    ResolveConstraintLayoutMethod "measure" o = Gtk.LayoutManager.LayoutManagerMeasureMethodInfo
    ResolveConstraintLayoutMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveConstraintLayoutMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveConstraintLayoutMethod "observeConstraints" o = ConstraintLayoutObserveConstraintsMethodInfo
    ResolveConstraintLayoutMethod "observeGuides" o = ConstraintLayoutObserveGuidesMethodInfo
    ResolveConstraintLayoutMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveConstraintLayoutMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveConstraintLayoutMethod "removeAllConstraints" o = ConstraintLayoutRemoveAllConstraintsMethodInfo
    ResolveConstraintLayoutMethod "removeConstraint" o = ConstraintLayoutRemoveConstraintMethodInfo
    ResolveConstraintLayoutMethod "removeGuide" o = ConstraintLayoutRemoveGuideMethodInfo
    ResolveConstraintLayoutMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveConstraintLayoutMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveConstraintLayoutMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveConstraintLayoutMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveConstraintLayoutMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveConstraintLayoutMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveConstraintLayoutMethod "getBuildableId" o = Gtk.Buildable.BuildableGetBuildableIdMethodInfo
    ResolveConstraintLayoutMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveConstraintLayoutMethod "getLayoutChild" o = Gtk.LayoutManager.LayoutManagerGetLayoutChildMethodInfo
    ResolveConstraintLayoutMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveConstraintLayoutMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveConstraintLayoutMethod "getRequestMode" o = Gtk.LayoutManager.LayoutManagerGetRequestModeMethodInfo
    ResolveConstraintLayoutMethod "getWidget" o = Gtk.LayoutManager.LayoutManagerGetWidgetMethodInfo
    ResolveConstraintLayoutMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveConstraintLayoutMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveConstraintLayoutMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveConstraintLayoutMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveConstraintLayoutMethod t ConstraintLayout, O.OverloadedMethod info ConstraintLayout p) => OL.IsLabel t (ConstraintLayout -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveConstraintLayoutMethod t ConstraintLayout, O.OverloadedMethod info ConstraintLayout p, R.HasField t ConstraintLayout p) => R.HasField t ConstraintLayout p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveConstraintLayoutMethod t ConstraintLayout, O.OverloadedMethodInfo info ConstraintLayout) => OL.IsLabel t (O.MethodProxy info ConstraintLayout) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList ConstraintLayout
type instance O.AttributeList ConstraintLayout = ConstraintLayoutAttributeList
type ConstraintLayoutAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList ConstraintLayout = ConstraintLayoutSignalList
type ConstraintLayoutSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method ConstraintLayout::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "ConstraintLayout" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_constraint_layout_new" gtk_constraint_layout_new :: 
    IO (Ptr ConstraintLayout)

-- | Creates a new @GtkConstraintLayout@ layout manager.
constraintLayoutNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m ConstraintLayout
    -- ^ __Returns:__ the newly created @GtkConstraintLayout@
constraintLayoutNew  = liftIO $ do
    result <- gtk_constraint_layout_new
    checkUnexpectedReturnNULL "constraintLayoutNew" result
    result' <- (wrapObject ConstraintLayout) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ConstraintLayout::add_constraint
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "layout"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintLayout" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkConstraintLayout`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "constraint"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Constraint" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [class@Gtk.Constraint]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_constraint_layout_add_constraint" gtk_constraint_layout_add_constraint :: 
    Ptr ConstraintLayout ->                 -- layout : TInterface (Name {namespace = "Gtk", name = "ConstraintLayout"})
    Ptr Gtk.Constraint.Constraint ->        -- constraint : TInterface (Name {namespace = "Gtk", name = "Constraint"})
    IO ()

-- | Adds a constraint to the layout manager.
-- 
-- The [Constraint:source]("GI.Gtk.Objects.Constraint#g:attr:source") and [Constraint:target]("GI.Gtk.Objects.Constraint#g:attr:target")
-- properties of @constraint@ can be:
-- 
--  - set to @NULL@ to indicate that the constraint refers to the
--    widget using @layout@
--  - set to the t'GI.Gtk.Objects.Widget.Widget' using @layout@
--  - set to a child of the t'GI.Gtk.Objects.Widget.Widget' using @layout@
--  - set to a t'GI.Gtk.Objects.ConstraintGuide.ConstraintGuide' that is part of @layout@
-- 
-- The /@layout@/ acquires the ownership of /@constraint@/ after calling
-- this function.
constraintLayoutAddConstraint ::
    (B.CallStack.HasCallStack, MonadIO m, IsConstraintLayout a, Gtk.Constraint.IsConstraint b) =>
    a
    -- ^ /@layout@/: a @GtkConstraintLayout@
    -> b
    -- ^ /@constraint@/: a t'GI.Gtk.Objects.Constraint.Constraint'
    -> m ()
constraintLayoutAddConstraint layout constraint = liftIO $ do
    layout' <- unsafeManagedPtrCastPtr layout
    constraint' <- B.ManagedPtr.disownObject constraint
    gtk_constraint_layout_add_constraint layout' constraint'
    touchManagedPtr layout
    touchManagedPtr constraint
    return ()

#if defined(ENABLE_OVERLOADING)
data ConstraintLayoutAddConstraintMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsConstraintLayout a, Gtk.Constraint.IsConstraint b) => O.OverloadedMethod ConstraintLayoutAddConstraintMethodInfo a signature where
    overloadedMethod = constraintLayoutAddConstraint

instance O.OverloadedMethodInfo ConstraintLayoutAddConstraintMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ConstraintLayout.constraintLayoutAddConstraint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ConstraintLayout.html#v:constraintLayoutAddConstraint"
        })


#endif

-- XXX Could not generate method ConstraintLayout::add_constraints_from_description
-- Not implemented: GHashTable element of type TInterface (Name {namespace = "Gtk", name = "ConstraintTarget"}) unsupported.
#if defined(ENABLE_OVERLOADING)
-- XXX: Dummy instance, since code generation failed.
-- Please file a bug at http://github.com/haskell-gi/haskell-gi.
data ConstraintLayoutAddConstraintsFromDescriptionMethodInfo
instance (p ~ (), o ~ O.UnsupportedMethodError "addConstraintsFromDescription" ConstraintLayout) => O.OverloadedMethod ConstraintLayoutAddConstraintsFromDescriptionMethodInfo o p where
    overloadedMethod = undefined

instance (o ~ O.UnsupportedMethodError "addConstraintsFromDescription" ConstraintLayout) => O.OverloadedMethodInfo ConstraintLayoutAddConstraintsFromDescriptionMethodInfo o where
    overloadedMethodInfo = undefined

#endif

-- method ConstraintLayout::add_guide
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "layout"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintLayout" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkConstraintLayout`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "guide"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintGuide" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [class@Gtk.ConstraintGuide] object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_constraint_layout_add_guide" gtk_constraint_layout_add_guide :: 
    Ptr ConstraintLayout ->                 -- layout : TInterface (Name {namespace = "Gtk", name = "ConstraintLayout"})
    Ptr Gtk.ConstraintGuide.ConstraintGuide -> -- guide : TInterface (Name {namespace = "Gtk", name = "ConstraintGuide"})
    IO ()

-- | Adds a guide to @layout@.
-- 
-- A guide can be used as the source or target of constraints,
-- like a widget, but it is not visible.
-- 
-- The @layout@ acquires the ownership of @guide@ after calling
-- this function.
constraintLayoutAddGuide ::
    (B.CallStack.HasCallStack, MonadIO m, IsConstraintLayout a, Gtk.ConstraintGuide.IsConstraintGuide b) =>
    a
    -- ^ /@layout@/: a @GtkConstraintLayout@
    -> b
    -- ^ /@guide@/: a t'GI.Gtk.Objects.ConstraintGuide.ConstraintGuide' object
    -> m ()
constraintLayoutAddGuide layout guide = liftIO $ do
    layout' <- unsafeManagedPtrCastPtr layout
    guide' <- B.ManagedPtr.disownObject guide
    gtk_constraint_layout_add_guide layout' guide'
    touchManagedPtr layout
    touchManagedPtr guide
    return ()

#if defined(ENABLE_OVERLOADING)
data ConstraintLayoutAddGuideMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsConstraintLayout a, Gtk.ConstraintGuide.IsConstraintGuide b) => O.OverloadedMethod ConstraintLayoutAddGuideMethodInfo a signature where
    overloadedMethod = constraintLayoutAddGuide

instance O.OverloadedMethodInfo ConstraintLayoutAddGuideMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ConstraintLayout.constraintLayoutAddGuide",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ConstraintLayout.html#v:constraintLayoutAddGuide"
        })


#endif

-- method ConstraintLayout::observe_constraints
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "layout"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintLayout" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkConstraintLayout`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "ListModel" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_constraint_layout_observe_constraints" gtk_constraint_layout_observe_constraints :: 
    Ptr ConstraintLayout ->                 -- layout : TInterface (Name {namespace = "Gtk", name = "ConstraintLayout"})
    IO (Ptr Gio.ListModel.ListModel)

-- | Returns a @GListModel@ to track the constraints that are
-- part of the layout.
-- 
-- Calling this function will enable extra internal bookkeeping
-- to track constraints and emit signals on the returned listmodel.
-- It may slow down operations a lot.
-- 
-- Applications should try hard to avoid calling this function
-- because of the slowdowns.
constraintLayoutObserveConstraints ::
    (B.CallStack.HasCallStack, MonadIO m, IsConstraintLayout a) =>
    a
    -- ^ /@layout@/: a @GtkConstraintLayout@
    -> m Gio.ListModel.ListModel
    -- ^ __Returns:__ a
    --   @GListModel@ tracking the layout\'s constraints
constraintLayoutObserveConstraints layout = liftIO $ do
    layout' <- unsafeManagedPtrCastPtr layout
    result <- gtk_constraint_layout_observe_constraints layout'
    checkUnexpectedReturnNULL "constraintLayoutObserveConstraints" result
    result' <- (wrapObject Gio.ListModel.ListModel) result
    touchManagedPtr layout
    return result'

#if defined(ENABLE_OVERLOADING)
data ConstraintLayoutObserveConstraintsMethodInfo
instance (signature ~ (m Gio.ListModel.ListModel), MonadIO m, IsConstraintLayout a) => O.OverloadedMethod ConstraintLayoutObserveConstraintsMethodInfo a signature where
    overloadedMethod = constraintLayoutObserveConstraints

instance O.OverloadedMethodInfo ConstraintLayoutObserveConstraintsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ConstraintLayout.constraintLayoutObserveConstraints",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ConstraintLayout.html#v:constraintLayoutObserveConstraints"
        })


#endif

-- method ConstraintLayout::observe_guides
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "layout"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintLayout" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkConstraintLayout`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "ListModel" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_constraint_layout_observe_guides" gtk_constraint_layout_observe_guides :: 
    Ptr ConstraintLayout ->                 -- layout : TInterface (Name {namespace = "Gtk", name = "ConstraintLayout"})
    IO (Ptr Gio.ListModel.ListModel)

-- | Returns a @GListModel@ to track the guides that are
-- part of the layout.
-- 
-- Calling this function will enable extra internal bookkeeping
-- to track guides and emit signals on the returned listmodel.
-- It may slow down operations a lot.
-- 
-- Applications should try hard to avoid calling this function
-- because of the slowdowns.
constraintLayoutObserveGuides ::
    (B.CallStack.HasCallStack, MonadIO m, IsConstraintLayout a) =>
    a
    -- ^ /@layout@/: a @GtkConstraintLayout@
    -> m Gio.ListModel.ListModel
    -- ^ __Returns:__ a
    --   @GListModel@ tracking the layout\'s guides
constraintLayoutObserveGuides layout = liftIO $ do
    layout' <- unsafeManagedPtrCastPtr layout
    result <- gtk_constraint_layout_observe_guides layout'
    checkUnexpectedReturnNULL "constraintLayoutObserveGuides" result
    result' <- (wrapObject Gio.ListModel.ListModel) result
    touchManagedPtr layout
    return result'

#if defined(ENABLE_OVERLOADING)
data ConstraintLayoutObserveGuidesMethodInfo
instance (signature ~ (m Gio.ListModel.ListModel), MonadIO m, IsConstraintLayout a) => O.OverloadedMethod ConstraintLayoutObserveGuidesMethodInfo a signature where
    overloadedMethod = constraintLayoutObserveGuides

instance O.OverloadedMethodInfo ConstraintLayoutObserveGuidesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ConstraintLayout.constraintLayoutObserveGuides",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ConstraintLayout.html#v:constraintLayoutObserveGuides"
        })


#endif

-- method ConstraintLayout::remove_all_constraints
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "layout"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintLayout" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkConstraintLayout`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_constraint_layout_remove_all_constraints" gtk_constraint_layout_remove_all_constraints :: 
    Ptr ConstraintLayout ->                 -- layout : TInterface (Name {namespace = "Gtk", name = "ConstraintLayout"})
    IO ()

-- | Removes all constraints from the layout manager.
constraintLayoutRemoveAllConstraints ::
    (B.CallStack.HasCallStack, MonadIO m, IsConstraintLayout a) =>
    a
    -- ^ /@layout@/: a @GtkConstraintLayout@
    -> m ()
constraintLayoutRemoveAllConstraints layout = liftIO $ do
    layout' <- unsafeManagedPtrCastPtr layout
    gtk_constraint_layout_remove_all_constraints layout'
    touchManagedPtr layout
    return ()

#if defined(ENABLE_OVERLOADING)
data ConstraintLayoutRemoveAllConstraintsMethodInfo
instance (signature ~ (m ()), MonadIO m, IsConstraintLayout a) => O.OverloadedMethod ConstraintLayoutRemoveAllConstraintsMethodInfo a signature where
    overloadedMethod = constraintLayoutRemoveAllConstraints

instance O.OverloadedMethodInfo ConstraintLayoutRemoveAllConstraintsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ConstraintLayout.constraintLayoutRemoveAllConstraints",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ConstraintLayout.html#v:constraintLayoutRemoveAllConstraints"
        })


#endif

-- method ConstraintLayout::remove_constraint
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "layout"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintLayout" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkConstraintLayout`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "constraint"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Constraint" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [class@Gtk.Constraint]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_constraint_layout_remove_constraint" gtk_constraint_layout_remove_constraint :: 
    Ptr ConstraintLayout ->                 -- layout : TInterface (Name {namespace = "Gtk", name = "ConstraintLayout"})
    Ptr Gtk.Constraint.Constraint ->        -- constraint : TInterface (Name {namespace = "Gtk", name = "Constraint"})
    IO ()

-- | Removes @constraint@ from the layout manager,
-- so that it no longer influences the layout.
constraintLayoutRemoveConstraint ::
    (B.CallStack.HasCallStack, MonadIO m, IsConstraintLayout a, Gtk.Constraint.IsConstraint b) =>
    a
    -- ^ /@layout@/: a @GtkConstraintLayout@
    -> b
    -- ^ /@constraint@/: a t'GI.Gtk.Objects.Constraint.Constraint'
    -> m ()
constraintLayoutRemoveConstraint layout constraint = liftIO $ do
    layout' <- unsafeManagedPtrCastPtr layout
    constraint' <- unsafeManagedPtrCastPtr constraint
    gtk_constraint_layout_remove_constraint layout' constraint'
    touchManagedPtr layout
    touchManagedPtr constraint
    return ()

#if defined(ENABLE_OVERLOADING)
data ConstraintLayoutRemoveConstraintMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsConstraintLayout a, Gtk.Constraint.IsConstraint b) => O.OverloadedMethod ConstraintLayoutRemoveConstraintMethodInfo a signature where
    overloadedMethod = constraintLayoutRemoveConstraint

instance O.OverloadedMethodInfo ConstraintLayoutRemoveConstraintMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ConstraintLayout.constraintLayoutRemoveConstraint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ConstraintLayout.html#v:constraintLayoutRemoveConstraint"
        })


#endif

-- method ConstraintLayout::remove_guide
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "layout"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintLayout" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkConstraintLayout`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "guide"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "ConstraintGuide" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [class@Gtk.ConstraintGuide] object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_constraint_layout_remove_guide" gtk_constraint_layout_remove_guide :: 
    Ptr ConstraintLayout ->                 -- layout : TInterface (Name {namespace = "Gtk", name = "ConstraintLayout"})
    Ptr Gtk.ConstraintGuide.ConstraintGuide -> -- guide : TInterface (Name {namespace = "Gtk", name = "ConstraintGuide"})
    IO ()

-- | Removes @guide@ from the layout manager,
-- so that it no longer influences the layout.
constraintLayoutRemoveGuide ::
    (B.CallStack.HasCallStack, MonadIO m, IsConstraintLayout a, Gtk.ConstraintGuide.IsConstraintGuide b) =>
    a
    -- ^ /@layout@/: a @GtkConstraintLayout@
    -> b
    -- ^ /@guide@/: a t'GI.Gtk.Objects.ConstraintGuide.ConstraintGuide' object
    -> m ()
constraintLayoutRemoveGuide layout guide = liftIO $ do
    layout' <- unsafeManagedPtrCastPtr layout
    guide' <- unsafeManagedPtrCastPtr guide
    gtk_constraint_layout_remove_guide layout' guide'
    touchManagedPtr layout
    touchManagedPtr guide
    return ()

#if defined(ENABLE_OVERLOADING)
data ConstraintLayoutRemoveGuideMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsConstraintLayout a, Gtk.ConstraintGuide.IsConstraintGuide b) => O.OverloadedMethod ConstraintLayoutRemoveGuideMethodInfo a signature where
    overloadedMethod = constraintLayoutRemoveGuide

instance O.OverloadedMethodInfo ConstraintLayoutRemoveGuideMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.ConstraintLayout.constraintLayoutRemoveGuide",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.9/docs/GI-Gtk-Objects-ConstraintLayout.html#v:constraintLayoutRemoveGuide"
        })


#endif


