{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The @GThreadPool@ struct represents a thread pool.
-- 
-- A thread pool is useful when you wish to asynchronously fork out the execution of work
-- and continue working in your own thread. If that will happen often, the overhead of starting
-- and destroying a thread each time might be too high. In such cases reusing already started
-- threads seems like a good idea. And it indeed is, but implementing this can be tedious
-- and error-prone.
-- 
-- Therefore GLib provides thread pools for your convenience. An added advantage is, that the
-- threads can be shared between the different subsystems of your program, when they are using GLib.
-- 
-- To create a new thread pool, you use [func/@gLib@/.ThreadPool.new].
-- It is destroyed by 'GI.GLib.Structs.ThreadPool.threadPoolFree'.
-- 
-- If you want to execute a certain task within a thread pool, use 'GI.GLib.Structs.ThreadPool.threadPoolPush'.
-- 
-- To get the current number of running threads you call 'GI.GLib.Structs.ThreadPool.threadPoolGetNumThreads'.
-- To get the number of still unprocessed tasks you call 'GI.GLib.Structs.ThreadPool.threadPoolUnprocessed'.
-- To control the maximum number of threads for a thread pool, you use
-- 'GI.GLib.Structs.ThreadPool.threadPoolGetMaxThreads'. and 'GI.GLib.Structs.ThreadPool.threadPoolSetMaxThreads'.
-- 
-- Finally you can control the number of unused threads, that are kept alive by GLib for future use.
-- The current number can be fetched with [func/@gLib@/.ThreadPool.get_num_unused_threads].
-- The maximum number can be controlled by [func/@gLib@/.ThreadPool.get_max_unused_threads] and
-- [func/@gLib@/.ThreadPool.set_max_unused_threads]. All currently unused threads
-- can be stopped by calling [func/@gLib@/.ThreadPool.stop_unused_threads].

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.ThreadPool
    ( 

-- * Exported types
    ThreadPool(..)                          ,
    newZeroThreadPool                       ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [free]("GI.GLib.Structs.ThreadPool#g:method:free"), [moveToFront]("GI.GLib.Structs.ThreadPool#g:method:moveToFront"), [push]("GI.GLib.Structs.ThreadPool#g:method:push"), [unprocessed]("GI.GLib.Structs.ThreadPool#g:method:unprocessed").
-- 
-- ==== Getters
-- [getMaxThreads]("GI.GLib.Structs.ThreadPool#g:method:getMaxThreads"), [getNumThreads]("GI.GLib.Structs.ThreadPool#g:method:getNumThreads").
-- 
-- ==== Setters
-- [setMaxThreads]("GI.GLib.Structs.ThreadPool#g:method:setMaxThreads").

#if defined(ENABLE_OVERLOADING)
    ResolveThreadPoolMethod                 ,
#endif

-- ** free #method:free#

#if defined(ENABLE_OVERLOADING)
    ThreadPoolFreeMethodInfo                ,
#endif
    threadPoolFree                          ,


-- ** getMaxIdleTime #method:getMaxIdleTime#

    threadPoolGetMaxIdleTime                ,


-- ** getMaxThreads #method:getMaxThreads#

#if defined(ENABLE_OVERLOADING)
    ThreadPoolGetMaxThreadsMethodInfo       ,
#endif
    threadPoolGetMaxThreads                 ,


-- ** getMaxUnusedThreads #method:getMaxUnusedThreads#

    threadPoolGetMaxUnusedThreads           ,


-- ** getNumThreads #method:getNumThreads#

#if defined(ENABLE_OVERLOADING)
    ThreadPoolGetNumThreadsMethodInfo       ,
#endif
    threadPoolGetNumThreads                 ,


-- ** getNumUnusedThreads #method:getNumUnusedThreads#

    threadPoolGetNumUnusedThreads           ,


-- ** moveToFront #method:moveToFront#

#if defined(ENABLE_OVERLOADING)
    ThreadPoolMoveToFrontMethodInfo         ,
#endif
    threadPoolMoveToFront                   ,


-- ** push #method:push#

#if defined(ENABLE_OVERLOADING)
    ThreadPoolPushMethodInfo                ,
#endif
    threadPoolPush                          ,


-- ** setMaxIdleTime #method:setMaxIdleTime#

    threadPoolSetMaxIdleTime                ,


-- ** setMaxThreads #method:setMaxThreads#

#if defined(ENABLE_OVERLOADING)
    ThreadPoolSetMaxThreadsMethodInfo       ,
#endif
    threadPoolSetMaxThreads                 ,


-- ** setMaxUnusedThreads #method:setMaxUnusedThreads#

    threadPoolSetMaxUnusedThreads           ,


-- ** stopUnusedThreads #method:stopUnusedThreads#

    threadPoolStopUnusedThreads             ,


-- ** unprocessed #method:unprocessed#

#if defined(ENABLE_OVERLOADING)
    ThreadPoolUnprocessedMethodInfo         ,
#endif
    threadPoolUnprocessed                   ,




 -- * Properties


-- ** exclusive #attr:exclusive#
-- | are all threads exclusive to this pool

    getThreadPoolExclusive                  ,
    setThreadPoolExclusive                  ,
#if defined(ENABLE_OVERLOADING)
    threadPool_exclusive                    ,
#endif


-- ** func #attr:func#
-- | the function to execute in the threads of this pool

    clearThreadPoolFunc                     ,
    getThreadPoolFunc                       ,
    setThreadPoolFunc                       ,
#if defined(ENABLE_OVERLOADING)
    threadPool_func                         ,
#endif


-- ** userData #attr:userData#
-- | the user data for the threads of this pool

    clearThreadPoolUserData                 ,
    getThreadPoolUserData                   ,
    setThreadPoolUserData                   ,
#if defined(ENABLE_OVERLOADING)
    threadPool_userData                     ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks

#else
import qualified GI.GLib.Callbacks as GLib.Callbacks

#endif

-- | Memory-managed wrapper type.
newtype ThreadPool = ThreadPool (SP.ManagedPtr ThreadPool)
    deriving (Eq)

instance SP.ManagedPtrNewtype ThreadPool where
    toManagedPtr (ThreadPool p) = p

instance BoxedPtr ThreadPool where
    boxedPtrCopy = \p -> B.ManagedPtr.withManagedPtr p (copyBytes 24 >=> B.ManagedPtr.wrapPtr ThreadPool)
    boxedPtrFree = \x -> SP.withManagedPtr x SP.freeMem
instance CallocPtr ThreadPool where
    boxedPtrCalloc = callocBytes 24


-- | Construct a `ThreadPool` struct initialized to zero.
newZeroThreadPool :: MonadIO m => m ThreadPool
newZeroThreadPool = liftIO $ boxedPtrCalloc >>= wrapPtr ThreadPool

instance tag ~ 'AttrSet => Constructible ThreadPool tag where
    new _ attrs = do
        o <- newZeroThreadPool
        GI.Attributes.set o attrs
        return o


-- | Get the value of the “@func@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' threadPool #func
-- @
getThreadPoolFunc :: MonadIO m => ThreadPool -> m (Maybe GLib.Callbacks.Func_WithClosures)
getThreadPoolFunc s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 0) :: IO (FunPtr GLib.Callbacks.C_Func)
    result <- SP.convertFunPtrIfNonNull val $ \val' -> do
        let val'' = GLib.Callbacks.dynamic_Func val'
        return val''
    return result

-- | Set the value of the “@func@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' threadPool [ #func 'Data.GI.Base.Attributes.:=' value ]
-- @
setThreadPoolFunc :: MonadIO m => ThreadPool -> FunPtr GLib.Callbacks.C_Func -> m ()
setThreadPoolFunc s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 0) (val :: FunPtr GLib.Callbacks.C_Func)

-- | Set the value of the “@func@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #func
-- @
clearThreadPoolFunc :: MonadIO m => ThreadPool -> m ()
clearThreadPoolFunc s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 0) (FP.nullFunPtr :: FunPtr GLib.Callbacks.C_Func)

#if defined(ENABLE_OVERLOADING)
data ThreadPoolFuncFieldInfo
instance AttrInfo ThreadPoolFuncFieldInfo where
    type AttrBaseTypeConstraint ThreadPoolFuncFieldInfo = (~) ThreadPool
    type AttrAllowedOps ThreadPoolFuncFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint ThreadPoolFuncFieldInfo = (~) (FunPtr GLib.Callbacks.C_Func)
    type AttrTransferTypeConstraint ThreadPoolFuncFieldInfo = (~)GLib.Callbacks.Func_WithClosures
    type AttrTransferType ThreadPoolFuncFieldInfo = (FunPtr GLib.Callbacks.C_Func)
    type AttrGetType ThreadPoolFuncFieldInfo = Maybe GLib.Callbacks.Func_WithClosures
    type AttrLabel ThreadPoolFuncFieldInfo = "func"
    type AttrOrigin ThreadPoolFuncFieldInfo = ThreadPool
    attrGet = getThreadPoolFunc
    attrSet = setThreadPoolFunc
    attrConstruct = undefined
    attrClear = clearThreadPoolFunc
    attrTransfer _ v = do
        GLib.Callbacks.mk_Func (GLib.Callbacks.wrap_Func Nothing v)
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.func"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#g:attr:func"
        })

threadPool_func :: AttrLabelProxy "func"
threadPool_func = AttrLabelProxy

#endif


-- | Get the value of the “@user_data@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' threadPool #userData
-- @
getThreadPoolUserData :: MonadIO m => ThreadPool -> m (Ptr ())
getThreadPoolUserData s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 8) :: IO (Ptr ())
    return val

-- | Set the value of the “@user_data@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' threadPool [ #userData 'Data.GI.Base.Attributes.:=' value ]
-- @
setThreadPoolUserData :: MonadIO m => ThreadPool -> Ptr () -> m ()
setThreadPoolUserData s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 8) (val :: Ptr ())

-- | Set the value of the “@user_data@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #userData
-- @
clearThreadPoolUserData :: MonadIO m => ThreadPool -> m ()
clearThreadPoolUserData s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 8) (FP.nullPtr :: Ptr ())

#if defined(ENABLE_OVERLOADING)
data ThreadPoolUserDataFieldInfo
instance AttrInfo ThreadPoolUserDataFieldInfo where
    type AttrBaseTypeConstraint ThreadPoolUserDataFieldInfo = (~) ThreadPool
    type AttrAllowedOps ThreadPoolUserDataFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint ThreadPoolUserDataFieldInfo = (~) (Ptr ())
    type AttrTransferTypeConstraint ThreadPoolUserDataFieldInfo = (~)(Ptr ())
    type AttrTransferType ThreadPoolUserDataFieldInfo = (Ptr ())
    type AttrGetType ThreadPoolUserDataFieldInfo = Ptr ()
    type AttrLabel ThreadPoolUserDataFieldInfo = "user_data"
    type AttrOrigin ThreadPoolUserDataFieldInfo = ThreadPool
    attrGet = getThreadPoolUserData
    attrSet = setThreadPoolUserData
    attrConstruct = undefined
    attrClear = clearThreadPoolUserData
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.userData"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#g:attr:userData"
        })

threadPool_userData :: AttrLabelProxy "userData"
threadPool_userData = AttrLabelProxy

#endif


-- | Get the value of the “@exclusive@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' threadPool #exclusive
-- @
getThreadPoolExclusive :: MonadIO m => ThreadPool -> m Bool
getThreadPoolExclusive s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 16) :: IO CInt
    let val' = (/= 0) val
    return val'

-- | Set the value of the “@exclusive@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' threadPool [ #exclusive 'Data.GI.Base.Attributes.:=' value ]
-- @
setThreadPoolExclusive :: MonadIO m => ThreadPool -> Bool -> m ()
setThreadPoolExclusive s val = liftIO $ withManagedPtr s $ \ptr -> do
    let val' = (P.fromIntegral . P.fromEnum) val
    poke (ptr `plusPtr` 16) (val' :: CInt)

#if defined(ENABLE_OVERLOADING)
data ThreadPoolExclusiveFieldInfo
instance AttrInfo ThreadPoolExclusiveFieldInfo where
    type AttrBaseTypeConstraint ThreadPoolExclusiveFieldInfo = (~) ThreadPool
    type AttrAllowedOps ThreadPoolExclusiveFieldInfo = '[ 'AttrSet, 'AttrGet]
    type AttrSetTypeConstraint ThreadPoolExclusiveFieldInfo = (~) Bool
    type AttrTransferTypeConstraint ThreadPoolExclusiveFieldInfo = (~)Bool
    type AttrTransferType ThreadPoolExclusiveFieldInfo = Bool
    type AttrGetType ThreadPoolExclusiveFieldInfo = Bool
    type AttrLabel ThreadPoolExclusiveFieldInfo = "exclusive"
    type AttrOrigin ThreadPoolExclusiveFieldInfo = ThreadPool
    attrGet = getThreadPoolExclusive
    attrSet = setThreadPoolExclusive
    attrConstruct = undefined
    attrClear = undefined
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.exclusive"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#g:attr:exclusive"
        })

threadPool_exclusive :: AttrLabelProxy "exclusive"
threadPool_exclusive = AttrLabelProxy

#endif



#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList ThreadPool
type instance O.AttributeList ThreadPool = ThreadPoolAttributeList
type ThreadPoolAttributeList = ('[ '("func", ThreadPoolFuncFieldInfo), '("userData", ThreadPoolUserDataFieldInfo), '("exclusive", ThreadPoolExclusiveFieldInfo)] :: [(Symbol, DK.Type)])
#endif

-- method ThreadPool::free
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pool"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadPool" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThreadPool" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "immediate"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "should @pool shut down immediately?"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "wait_"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "should the function wait for all tasks to be finished?"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_free" g_thread_pool_free :: 
    Ptr ThreadPool ->                       -- pool : TInterface (Name {namespace = "GLib", name = "ThreadPool"})
    CInt ->                                 -- immediate : TBasicType TBoolean
    CInt ->                                 -- wait_ : TBasicType TBoolean
    IO ()

-- | Frees all resources allocated for /@pool@/.
-- 
-- If /@immediate@/ is 'P.True', no new task is processed for /@pool@/.
-- Otherwise /@pool@/ is not freed before the last task is processed.
-- Note however, that no thread of this pool is interrupted while
-- processing a task. Instead at least all still running threads
-- can finish their tasks before the /@pool@/ is freed.
-- 
-- If /@wait_@/ is 'P.True', this function does not return before all
-- tasks to be processed (dependent on /@immediate@/, whether all
-- or only the currently running) are ready.
-- Otherwise this function returns immediately.
-- 
-- After calling this function /@pool@/ must not be used anymore.
threadPoolFree ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ThreadPool
    -- ^ /@pool@/: a t'GI.GLib.Structs.ThreadPool.ThreadPool'
    -> Bool
    -- ^ /@immediate@/: should /@pool@/ shut down immediately?
    -> Bool
    -- ^ /@wait_@/: should the function wait for all tasks to be finished?
    -> m ()
threadPoolFree pool immediate wait_ = liftIO $ do
    pool' <- unsafeManagedPtrGetPtr pool
    let immediate' = (P.fromIntegral . P.fromEnum) immediate
    let wait_' = (P.fromIntegral . P.fromEnum) wait_
    g_thread_pool_free pool' immediate' wait_'
    touchManagedPtr pool
    return ()

#if defined(ENABLE_OVERLOADING)
data ThreadPoolFreeMethodInfo
instance (signature ~ (Bool -> Bool -> m ()), MonadIO m) => O.OverloadedMethod ThreadPoolFreeMethodInfo ThreadPool signature where
    overloadedMethod = threadPoolFree

instance O.OverloadedMethodInfo ThreadPoolFreeMethodInfo ThreadPool where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.threadPoolFree",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#v:threadPoolFree"
        })


#endif

-- method ThreadPool::get_max_threads
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pool"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadPool" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThreadPool" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_get_max_threads" g_thread_pool_get_max_threads :: 
    Ptr ThreadPool ->                       -- pool : TInterface (Name {namespace = "GLib", name = "ThreadPool"})
    IO Int32

-- | Returns the maximal number of threads for /@pool@/.
threadPoolGetMaxThreads ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ThreadPool
    -- ^ /@pool@/: a t'GI.GLib.Structs.ThreadPool.ThreadPool'
    -> m Int32
    -- ^ __Returns:__ the maximal number of threads
threadPoolGetMaxThreads pool = liftIO $ do
    pool' <- unsafeManagedPtrGetPtr pool
    result <- g_thread_pool_get_max_threads pool'
    touchManagedPtr pool
    return result

#if defined(ENABLE_OVERLOADING)
data ThreadPoolGetMaxThreadsMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod ThreadPoolGetMaxThreadsMethodInfo ThreadPool signature where
    overloadedMethod = threadPoolGetMaxThreads

instance O.OverloadedMethodInfo ThreadPoolGetMaxThreadsMethodInfo ThreadPool where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.threadPoolGetMaxThreads",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#v:threadPoolGetMaxThreads"
        })


#endif

-- method ThreadPool::get_num_threads
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pool"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadPool" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThreadPool" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_get_num_threads" g_thread_pool_get_num_threads :: 
    Ptr ThreadPool ->                       -- pool : TInterface (Name {namespace = "GLib", name = "ThreadPool"})
    IO Word32

-- | Returns the number of threads currently running in /@pool@/.
threadPoolGetNumThreads ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ThreadPool
    -- ^ /@pool@/: a t'GI.GLib.Structs.ThreadPool.ThreadPool'
    -> m Word32
    -- ^ __Returns:__ the number of threads currently running
threadPoolGetNumThreads pool = liftIO $ do
    pool' <- unsafeManagedPtrGetPtr pool
    result <- g_thread_pool_get_num_threads pool'
    touchManagedPtr pool
    return result

#if defined(ENABLE_OVERLOADING)
data ThreadPoolGetNumThreadsMethodInfo
instance (signature ~ (m Word32), MonadIO m) => O.OverloadedMethod ThreadPoolGetNumThreadsMethodInfo ThreadPool signature where
    overloadedMethod = threadPoolGetNumThreads

instance O.OverloadedMethodInfo ThreadPoolGetNumThreadsMethodInfo ThreadPool where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.threadPoolGetNumThreads",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#v:threadPoolGetNumThreads"
        })


#endif

-- method ThreadPool::move_to_front
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pool"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadPool" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThreadPool" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an unprocessed item in the pool"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_move_to_front" g_thread_pool_move_to_front :: 
    Ptr ThreadPool ->                       -- pool : TInterface (Name {namespace = "GLib", name = "ThreadPool"})
    Ptr () ->                               -- data : TBasicType TPtr
    IO CInt

-- | Moves the item to the front of the queue of unprocessed
-- items, so that it will be processed next.
-- 
-- /Since: 2.46/
threadPoolMoveToFront ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ThreadPool
    -- ^ /@pool@/: a t'GI.GLib.Structs.ThreadPool.ThreadPool'
    -> Ptr ()
    -- ^ /@data@/: an unprocessed item in the pool
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the item was found and moved
threadPoolMoveToFront pool data_ = liftIO $ do
    pool' <- unsafeManagedPtrGetPtr pool
    result <- g_thread_pool_move_to_front pool' data_
    let result' = (/= 0) result
    touchManagedPtr pool
    return result'

#if defined(ENABLE_OVERLOADING)
data ThreadPoolMoveToFrontMethodInfo
instance (signature ~ (Ptr () -> m Bool), MonadIO m) => O.OverloadedMethod ThreadPoolMoveToFrontMethodInfo ThreadPool signature where
    overloadedMethod = threadPoolMoveToFront

instance O.OverloadedMethodInfo ThreadPoolMoveToFrontMethodInfo ThreadPool where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.threadPoolMoveToFront",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#v:threadPoolMoveToFront"
        })


#endif

-- method ThreadPool::push
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pool"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadPool" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThreadPool" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a new task for @pool"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_thread_pool_push" g_thread_pool_push :: 
    Ptr ThreadPool ->                       -- pool : TInterface (Name {namespace = "GLib", name = "ThreadPool"})
    Ptr () ->                               -- data : TBasicType TPtr
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Inserts /@data@/ into the list of tasks to be executed by /@pool@/.
-- 
-- When the number of currently running threads is lower than the
-- maximal allowed number of threads, a new thread is started (or
-- reused) with the properties given to @/g_thread_pool_new()/@.
-- Otherwise, /@data@/ stays in the queue until a thread in this pool
-- finishes its previous task and processes /@data@/.
-- 
-- /@error@/ can be 'P.Nothing' to ignore errors, or non-'P.Nothing' to report
-- errors. An error can only occur when a new thread couldn\'t be
-- created. In that case /@data@/ is simply appended to the queue of
-- work to do.
-- 
-- Before version 2.32, this function did not return a success status.
threadPoolPush ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ThreadPool
    -- ^ /@pool@/: a t'GI.GLib.Structs.ThreadPool.ThreadPool'
    -> Ptr ()
    -- ^ /@data@/: a new task for /@pool@/
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
threadPoolPush pool data_ = liftIO $ do
    pool' <- unsafeManagedPtrGetPtr pool
    onException (do
        _ <- propagateGError $ g_thread_pool_push pool' data_
        touchManagedPtr pool
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ThreadPoolPushMethodInfo
instance (signature ~ (Ptr () -> m ()), MonadIO m) => O.OverloadedMethod ThreadPoolPushMethodInfo ThreadPool signature where
    overloadedMethod = threadPoolPush

instance O.OverloadedMethodInfo ThreadPoolPushMethodInfo ThreadPool where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.threadPoolPush",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#v:threadPoolPush"
        })


#endif

-- method ThreadPool::set_max_threads
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pool"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadPool" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThreadPool" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "max_threads"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a new maximal number of threads for @pool,\n    or -1 for unlimited"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_thread_pool_set_max_threads" g_thread_pool_set_max_threads :: 
    Ptr ThreadPool ->                       -- pool : TInterface (Name {namespace = "GLib", name = "ThreadPool"})
    Int32 ->                                -- max_threads : TBasicType TInt
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets the maximal allowed number of threads for /@pool@/.
-- A value of -1 means that the maximal number of threads
-- is unlimited. If /@pool@/ is an exclusive thread pool, setting
-- the maximal number of threads to -1 is not allowed.
-- 
-- Setting /@maxThreads@/ to 0 means stopping all work for /@pool@/.
-- It is effectively frozen until /@maxThreads@/ is set to a non-zero
-- value again.
-- 
-- A thread is never terminated while calling /@func@/, as supplied by
-- @/g_thread_pool_new()/@. Instead the maximal number of threads only
-- has effect for the allocation of new threads in 'GI.GLib.Structs.ThreadPool.threadPoolPush'.
-- A new thread is allocated, whenever the number of currently
-- running threads in /@pool@/ is smaller than the maximal number.
-- 
-- /@error@/ can be 'P.Nothing' to ignore errors, or non-'P.Nothing' to report
-- errors. An error can only occur when a new thread couldn\'t be
-- created.
-- 
-- Before version 2.32, this function did not return a success status.
threadPoolSetMaxThreads ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ThreadPool
    -- ^ /@pool@/: a t'GI.GLib.Structs.ThreadPool.ThreadPool'
    -> Int32
    -- ^ /@maxThreads@/: a new maximal number of threads for /@pool@/,
    --     or -1 for unlimited
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
threadPoolSetMaxThreads pool maxThreads = liftIO $ do
    pool' <- unsafeManagedPtrGetPtr pool
    onException (do
        _ <- propagateGError $ g_thread_pool_set_max_threads pool' maxThreads
        touchManagedPtr pool
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ThreadPoolSetMaxThreadsMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m) => O.OverloadedMethod ThreadPoolSetMaxThreadsMethodInfo ThreadPool signature where
    overloadedMethod = threadPoolSetMaxThreads

instance O.OverloadedMethodInfo ThreadPoolSetMaxThreadsMethodInfo ThreadPool where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.threadPoolSetMaxThreads",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#v:threadPoolSetMaxThreads"
        })


#endif

-- method ThreadPool::unprocessed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pool"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadPool" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThreadPool" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_unprocessed" g_thread_pool_unprocessed :: 
    Ptr ThreadPool ->                       -- pool : TInterface (Name {namespace = "GLib", name = "ThreadPool"})
    IO Word32

-- | Returns the number of tasks still unprocessed in /@pool@/.
threadPoolUnprocessed ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ThreadPool
    -- ^ /@pool@/: a t'GI.GLib.Structs.ThreadPool.ThreadPool'
    -> m Word32
    -- ^ __Returns:__ the number of unprocessed tasks
threadPoolUnprocessed pool = liftIO $ do
    pool' <- unsafeManagedPtrGetPtr pool
    result <- g_thread_pool_unprocessed pool'
    touchManagedPtr pool
    return result

#if defined(ENABLE_OVERLOADING)
data ThreadPoolUnprocessedMethodInfo
instance (signature ~ (m Word32), MonadIO m) => O.OverloadedMethod ThreadPoolUnprocessedMethodInfo ThreadPool signature where
    overloadedMethod = threadPoolUnprocessed

instance O.OverloadedMethodInfo ThreadPoolUnprocessedMethodInfo ThreadPool where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.ThreadPool.threadPoolUnprocessed",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-ThreadPool.html#v:threadPoolUnprocessed"
        })


#endif

-- method ThreadPool::get_max_idle_time
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_get_max_idle_time" g_thread_pool_get_max_idle_time :: 
    IO Word32

-- | This function will return the maximum /@interval@/ that a
-- thread will wait in the thread pool for new tasks before
-- being stopped.
-- 
-- If this function returns 0, threads waiting in the thread
-- pool for new work are not stopped.
-- 
-- /Since: 2.10/
threadPoolGetMaxIdleTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Word32
    -- ^ __Returns:__ the maximum /@interval@/ (milliseconds) to wait
    --     for new tasks in the thread pool before stopping the
    --     thread
threadPoolGetMaxIdleTime  = liftIO $ do
    result <- g_thread_pool_get_max_idle_time
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method ThreadPool::get_max_unused_threads
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_get_max_unused_threads" g_thread_pool_get_max_unused_threads :: 
    IO Int32

-- | Returns the maximal allowed number of unused threads.
threadPoolGetMaxUnusedThreads ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Int32
    -- ^ __Returns:__ the maximal number of unused threads
threadPoolGetMaxUnusedThreads  = liftIO $ do
    result <- g_thread_pool_get_max_unused_threads
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method ThreadPool::get_num_unused_threads
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_get_num_unused_threads" g_thread_pool_get_num_unused_threads :: 
    IO Word32

-- | Returns the number of currently unused threads.
threadPoolGetNumUnusedThreads ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Word32
    -- ^ __Returns:__ the number of currently unused threads
threadPoolGetNumUnusedThreads  = liftIO $ do
    result <- g_thread_pool_get_num_unused_threads
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method ThreadPool::set_max_idle_time
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "interval"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the maximum @interval (in milliseconds)\n    a thread can be idle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_set_max_idle_time" g_thread_pool_set_max_idle_time :: 
    Word32 ->                               -- interval : TBasicType TUInt
    IO ()

-- | This function will set the maximum /@interval@/ that a thread
-- waiting in the pool for new tasks can be idle for before
-- being stopped. This function is similar to calling
-- 'GI.GLib.Functions.threadPoolStopUnusedThreads' on a regular timeout,
-- except this is done on a per thread basis.
-- 
-- By setting /@interval@/ to 0, idle threads will not be stopped.
-- 
-- The default value is 15000 (15 seconds).
-- 
-- /Since: 2.10/
threadPoolSetMaxIdleTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Word32
    -- ^ /@interval@/: the maximum /@interval@/ (in milliseconds)
    --     a thread can be idle
    -> m ()
threadPoolSetMaxIdleTime interval = liftIO $ do
    g_thread_pool_set_max_idle_time interval
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- method ThreadPool::set_max_unused_threads
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "max_threads"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "maximal number of unused threads"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_set_max_unused_threads" g_thread_pool_set_max_unused_threads :: 
    Int32 ->                                -- max_threads : TBasicType TInt
    IO ()

-- | Sets the maximal number of unused threads to /@maxThreads@/.
-- If /@maxThreads@/ is -1, no limit is imposed on the number
-- of unused threads.
-- 
-- The default value is 8 since GLib 2.84. Previously the default value was 2.
threadPoolSetMaxUnusedThreads ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@maxThreads@/: maximal number of unused threads
    -> m ()
threadPoolSetMaxUnusedThreads maxThreads = liftIO $ do
    g_thread_pool_set_max_unused_threads maxThreads
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- method ThreadPool::stop_unused_threads
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_pool_stop_unused_threads" g_thread_pool_stop_unused_threads :: 
    IO ()

-- | Stops all currently unused threads. This does not change the
-- maximal number of unused threads. This function can be used to
-- regularly stop all unused threads e.g. from @/g_timeout_add()/@.
threadPoolStopUnusedThreads ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m ()
threadPoolStopUnusedThreads  = liftIO $ do
    g_thread_pool_stop_unused_threads
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveThreadPoolMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveThreadPoolMethod "free" o = ThreadPoolFreeMethodInfo
    ResolveThreadPoolMethod "moveToFront" o = ThreadPoolMoveToFrontMethodInfo
    ResolveThreadPoolMethod "push" o = ThreadPoolPushMethodInfo
    ResolveThreadPoolMethod "unprocessed" o = ThreadPoolUnprocessedMethodInfo
    ResolveThreadPoolMethod "getMaxThreads" o = ThreadPoolGetMaxThreadsMethodInfo
    ResolveThreadPoolMethod "getNumThreads" o = ThreadPoolGetNumThreadsMethodInfo
    ResolveThreadPoolMethod "setMaxThreads" o = ThreadPoolSetMaxThreadsMethodInfo
    ResolveThreadPoolMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveThreadPoolMethod t ThreadPool, O.OverloadedMethod info ThreadPool p) => OL.IsLabel t (ThreadPool -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveThreadPoolMethod t ThreadPool, O.OverloadedMethod info ThreadPool p, R.HasField t ThreadPool p) => R.HasField t ThreadPool p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveThreadPoolMethod t ThreadPool, O.OverloadedMethodInfo info ThreadPool) => OL.IsLabel t (O.MethodProxy info ThreadPool) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


