{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The t'GI.GLib.Structs.Thread.Thread' struct represents a running thread. This struct
-- is returned by 'GI.GLib.Structs.Thread.threadNew' or 'GI.GLib.Structs.Thread.threadTryNew'. You can
-- obtain the t'GI.GLib.Structs.Thread.Thread' struct representing the current thread by
-- calling 'GI.GLib.Functions.threadSelf'.
-- 
-- GThread is refcounted, see 'GI.GLib.Structs.Thread.threadRef' and 'GI.GLib.Structs.Thread.threadUnref'.
-- The thread represented by it holds a reference while it is running,
-- and 'GI.GLib.Structs.Thread.threadJoin' consumes the reference that it is given, so
-- it is normally not necessary to manage GThread references
-- explicitly.
-- 
-- The structure is opaque -- none of its fields may be directly
-- accessed.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.Thread
    ( 

-- * Exported types
    Thread(..)                              ,
    newZeroThread                           ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [join]("GI.GLib.Structs.Thread#g:method:join"), [ref]("GI.GLib.Structs.Thread#g:method:ref"), [unref]("GI.GLib.Structs.Thread#g:method:unref").
-- 
-- ==== Getters
-- [getName]("GI.GLib.Structs.Thread#g:method:getName").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveThreadMethod                     ,
#endif

-- ** errorQuark #method:errorQuark#

    threadErrorQuark                        ,


-- ** exit #method:exit#

    threadExit                              ,


-- ** getName #method:getName#

#if defined(ENABLE_OVERLOADING)
    ThreadGetNameMethodInfo                 ,
#endif
    threadGetName                           ,


-- ** join #method:join#

#if defined(ENABLE_OVERLOADING)
    ThreadJoinMethodInfo                    ,
#endif
    threadJoin                              ,


-- ** new #method:new#

    threadNew                               ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    ThreadRefMethodInfo                     ,
#endif
    threadRef                               ,


-- ** self #method:self#

    threadSelf                              ,


-- ** tryNew #method:tryNew#

    threadTryNew                            ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    ThreadUnrefMethodInfo                   ,
#endif
    threadUnref                             ,


-- ** yield #method:yield#

    threadYield                             ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks

#else
import qualified GI.GLib.Callbacks as GLib.Callbacks

#endif

-- | Memory-managed wrapper type.
newtype Thread = Thread (SP.ManagedPtr Thread)
    deriving (Eq)

instance SP.ManagedPtrNewtype Thread where
    toManagedPtr (Thread p) = p

foreign import ccall "g_thread_get_type" c_g_thread_get_type :: 
    IO GType

type instance O.ParentTypes Thread = '[]
instance O.HasParentTypes Thread

instance B.Types.TypedObject Thread where
    glibType = c_g_thread_get_type

instance B.Types.GBoxed Thread

-- | Convert 'Thread' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Thread) where
    gvalueGType_ = c_g_thread_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr Thread)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr Thread)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed Thread ptr
        else return P.Nothing
        
    

-- | Construct a `Thread` struct initialized to zero.
newZeroThread :: MonadIO m => m Thread
newZeroThread = liftIO $ callocBoxedBytes 32 >>= wrapBoxed Thread

instance tag ~ 'AttrSet => Constructible Thread tag where
    new _ attrs = do
        o <- newZeroThread
        GI.Attributes.set o attrs
        return o



#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Thread
type instance O.AttributeList Thread = ThreadAttributeList
type ThreadAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method Thread::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an (optional) name for the new thread"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "func"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadFunc" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a function to execute in the new thread"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 2
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an argument to supply to the new thread"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Thread" })
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_new" g_thread_new :: 
    CString ->                              -- name : TBasicType TUTF8
    FunPtr GLib.Callbacks.C_ThreadFunc ->   -- func : TInterface (Name {namespace = "GLib", name = "ThreadFunc"})
    Ptr () ->                               -- data : TBasicType TPtr
    IO (Ptr Thread)

-- | This function creates a new thread. The new thread starts by invoking
-- /@func@/ with the argument data. The thread will run until /@func@/ returns
-- or until 'GI.GLib.Functions.threadExit' is called from the new thread. The return value
-- of /@func@/ becomes the return value of the thread, which can be obtained
-- with 'GI.GLib.Structs.Thread.threadJoin'.
-- 
-- The /@name@/ can be useful for discriminating threads in a debugger.
-- It is not used for other purposes and does not have to be unique.
-- Some systems restrict the length of /@name@/ to 16 bytes.
-- 
-- If the thread can not be created the program aborts. See
-- 'GI.GLib.Structs.Thread.threadTryNew' if you want to attempt to deal with failures.
-- 
-- If you are using threads to offload (potentially many) short-lived tasks,
-- t'GI.GLib.Structs.ThreadPool.ThreadPool' may be more appropriate than manually spawning and tracking
-- multiple @/GThreads/@.
-- 
-- To free the struct returned by this function, use 'GI.GLib.Structs.Thread.threadUnref'.
-- Note that 'GI.GLib.Structs.Thread.threadJoin' implicitly unrefs the t'GI.GLib.Structs.Thread.Thread' as well.
-- 
-- New threads by default inherit their scheduler policy (POSIX) or thread
-- priority (Windows) of the thread creating the new thread.
-- 
-- This behaviour changed in GLib 2.64: before threads on Windows were not
-- inheriting the thread priority but were spawned with the default priority.
-- Starting with GLib 2.64 the behaviour is now consistent between Windows and
-- POSIX and all threads inherit their parent thread\'s priority.
-- 
-- /Since: 2.32/
threadNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@name@/: an (optional) name for the new thread
    -> GLib.Callbacks.ThreadFunc
    -- ^ /@func@/: a function to execute in the new thread
    -> m Thread
    -- ^ __Returns:__ the new t'GI.GLib.Structs.Thread.Thread'
threadNew name func = liftIO $ do
    maybeName <- case name of
        Nothing -> return FP.nullPtr
        Just jName -> do
            jName' <- textToCString jName
            return jName'
    ptrfunc <- callocMem :: IO (Ptr (FunPtr GLib.Callbacks.C_ThreadFunc))
    func' <- GLib.Callbacks.mk_ThreadFunc (GLib.Callbacks.wrap_ThreadFunc (Just ptrfunc) (GLib.Callbacks.drop_closures_ThreadFunc func))
    poke ptrfunc func'
    let data_ = nullPtr
    result <- g_thread_new maybeName func' data_
    checkUnexpectedReturnNULL "threadNew" result
    result' <- (wrapBoxed Thread) result
    freeMem maybeName
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Thread::try_new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an (optional) name for the new thread"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "func"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ThreadFunc" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a function to execute in the new thread"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 2
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an argument to supply to the new thread"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Thread" })
-- throws : True
-- Skip return : False

foreign import ccall "g_thread_try_new" g_thread_try_new :: 
    CString ->                              -- name : TBasicType TUTF8
    FunPtr GLib.Callbacks.C_ThreadFunc ->   -- func : TInterface (Name {namespace = "GLib", name = "ThreadFunc"})
    Ptr () ->                               -- data : TBasicType TPtr
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Thread)

-- | This function is the same as 'GI.GLib.Structs.Thread.threadNew' except that
-- it allows for the possibility of failure.
-- 
-- If a thread can not be created (due to resource limits),
-- /@error@/ is set and 'P.Nothing' is returned.
-- 
-- /Since: 2.32/
threadTryNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@name@/: an (optional) name for the new thread
    -> GLib.Callbacks.ThreadFunc
    -- ^ /@func@/: a function to execute in the new thread
    -> m Thread
    -- ^ __Returns:__ the new t'GI.GLib.Structs.Thread.Thread', or 'P.Nothing' if an error occurred /(Can throw 'Data.GI.Base.GError.GError')/
threadTryNew name func = liftIO $ do
    maybeName <- case name of
        Nothing -> return FP.nullPtr
        Just jName -> do
            jName' <- textToCString jName
            return jName'
    ptrfunc <- callocMem :: IO (Ptr (FunPtr GLib.Callbacks.C_ThreadFunc))
    func' <- GLib.Callbacks.mk_ThreadFunc (GLib.Callbacks.wrap_ThreadFunc (Just ptrfunc) (GLib.Callbacks.drop_closures_ThreadFunc func))
    poke ptrfunc func'
    let data_ = nullPtr
    onException (do
        result <- propagateGError $ g_thread_try_new maybeName func' data_
        checkUnexpectedReturnNULL "threadTryNew" result
        result' <- (wrapBoxed Thread) result
        freeMem maybeName
        return result'
     ) (do
        freeMem maybeName
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Thread::get_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "thread"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "Thread" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a thread" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_get_name" g_thread_get_name :: 
    Ptr Thread ->                           -- thread : TInterface (Name {namespace = "GLib", name = "Thread"})
    IO CString

-- | Gets the name of the thread.
-- 
-- This function is intended for debugging purposes.
-- 
-- /Since: 2.84/
threadGetName ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Thread
    -- ^ /@thread@/: a thread
    -> m T.Text
    -- ^ __Returns:__ the name of the thread
threadGetName thread = liftIO $ do
    thread' <- unsafeManagedPtrGetPtr thread
    result <- g_thread_get_name thread'
    checkUnexpectedReturnNULL "threadGetName" result
    result' <- cstringToText result
    touchManagedPtr thread
    return result'

#if defined(ENABLE_OVERLOADING)
data ThreadGetNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod ThreadGetNameMethodInfo Thread signature where
    overloadedMethod = threadGetName

instance O.OverloadedMethodInfo ThreadGetNameMethodInfo Thread where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Thread.threadGetName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Thread.html#v:threadGetName"
        })


#endif

-- method Thread::join
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "thread"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "Thread" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThread" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_join" g_thread_join :: 
    Ptr Thread ->                           -- thread : TInterface (Name {namespace = "GLib", name = "Thread"})
    IO (Ptr ())

-- | Waits until /@thread@/ finishes, i.e. the function /@func@/, as
-- given to 'GI.GLib.Structs.Thread.threadNew', returns or 'GI.GLib.Functions.threadExit' is called.
-- If /@thread@/ has already terminated, then 'GI.GLib.Structs.Thread.threadJoin'
-- returns immediately.
-- 
-- Any thread can wait for any other thread by calling 'GI.GLib.Structs.Thread.threadJoin',
-- not just its \'creator\'. Calling 'GI.GLib.Structs.Thread.threadJoin' from multiple threads
-- for the same /@thread@/ leads to undefined behaviour.
-- 
-- The value returned by /@func@/ or given to 'GI.GLib.Functions.threadExit' is
-- returned by this function.
-- 
-- 'GI.GLib.Structs.Thread.threadJoin' consumes the reference to the passed-in /@thread@/.
-- This will usually cause the t'GI.GLib.Structs.Thread.Thread' struct and associated resources
-- to be freed. Use 'GI.GLib.Structs.Thread.threadRef' to obtain an extra reference if you
-- want to keep the GThread alive beyond the 'GI.GLib.Structs.Thread.threadJoin' call.
threadJoin ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Thread
    -- ^ /@thread@/: a t'GI.GLib.Structs.Thread.Thread'
    -> m (Ptr ())
    -- ^ __Returns:__ the return value of the thread
threadJoin thread = liftIO $ do
    thread' <- B.ManagedPtr.disownBoxed thread
    result <- g_thread_join thread'
    touchManagedPtr thread
    return result

#if defined(ENABLE_OVERLOADING)
data ThreadJoinMethodInfo
instance (signature ~ (m (Ptr ())), MonadIO m) => O.OverloadedMethod ThreadJoinMethodInfo Thread signature where
    overloadedMethod = threadJoin

instance O.OverloadedMethodInfo ThreadJoinMethodInfo Thread where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Thread.threadJoin",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Thread.html#v:threadJoin"
        })


#endif

-- method Thread::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "thread"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "Thread" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThread" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Thread" })
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_ref" g_thread_ref :: 
    Ptr Thread ->                           -- thread : TInterface (Name {namespace = "GLib", name = "Thread"})
    IO (Ptr Thread)

-- | Increase the reference count on /@thread@/.
-- 
-- /Since: 2.32/
threadRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Thread
    -- ^ /@thread@/: a t'GI.GLib.Structs.Thread.Thread'
    -> m Thread
    -- ^ __Returns:__ a new reference to /@thread@/
threadRef thread = liftIO $ do
    thread' <- unsafeManagedPtrGetPtr thread
    result <- g_thread_ref thread'
    checkUnexpectedReturnNULL "threadRef" result
    result' <- (wrapBoxed Thread) result
    touchManagedPtr thread
    return result'

#if defined(ENABLE_OVERLOADING)
data ThreadRefMethodInfo
instance (signature ~ (m Thread), MonadIO m) => O.OverloadedMethod ThreadRefMethodInfo Thread signature where
    overloadedMethod = threadRef

instance O.OverloadedMethodInfo ThreadRefMethodInfo Thread where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Thread.threadRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Thread.html#v:threadRef"
        })


#endif

-- method Thread::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "thread"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "Thread" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GThread" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_unref" g_thread_unref :: 
    Ptr Thread ->                           -- thread : TInterface (Name {namespace = "GLib", name = "Thread"})
    IO ()

-- | Decrease the reference count on /@thread@/, possibly freeing all
-- resources associated with it.
-- 
-- Note that each thread holds a reference to its t'GI.GLib.Structs.Thread.Thread' while
-- it is running, so it is safe to drop your own reference to it
-- if you don\'t need it anymore.
-- 
-- /Since: 2.32/
threadUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Thread
    -- ^ /@thread@/: a t'GI.GLib.Structs.Thread.Thread'
    -> m ()
threadUnref thread = liftIO $ do
    thread' <- B.ManagedPtr.disownBoxed thread
    g_thread_unref thread'
    touchManagedPtr thread
    return ()

#if defined(ENABLE_OVERLOADING)
data ThreadUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod ThreadUnrefMethodInfo Thread signature where
    overloadedMethod = threadUnref

instance O.OverloadedMethodInfo ThreadUnrefMethodInfo Thread where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Thread.threadUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Thread.html#v:threadUnref"
        })


#endif

-- method Thread::error_quark
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TUInt32)
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_error_quark" g_thread_error_quark :: 
    IO Word32

-- | /No description available in the introspection data./
threadErrorQuark ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Word32
threadErrorQuark  = liftIO $ do
    result <- g_thread_error_quark
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method Thread::exit
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "retval"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the return value of this thread"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_exit" g_thread_exit :: 
    Ptr () ->                               -- retval : TBasicType TPtr
    IO ()

-- | Terminates the current thread.
-- 
-- If another thread is waiting for us using 'GI.GLib.Structs.Thread.threadJoin' then the
-- waiting thread will be woken up and get /@retval@/ as the return value
-- of 'GI.GLib.Structs.Thread.threadJoin'.
-- 
-- Calling 'GI.GLib.Functions.threadExit' with a parameter /@retval@/ is equivalent to
-- returning /@retval@/ from the function /@func@/, as given to 'GI.GLib.Structs.Thread.threadNew'.
-- 
-- You must only call 'GI.GLib.Functions.threadExit' from a thread that you created
-- yourself with 'GI.GLib.Structs.Thread.threadNew' or related APIs. You must not call
-- this function from a thread created with another threading library
-- or or from within a t'GI.GLib.Structs.ThreadPool.ThreadPool'.
threadExit ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Ptr ()
    -- ^ /@retval@/: the return value of this thread
    -> m ()
threadExit retval = liftIO $ do
    g_thread_exit retval
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- method Thread::self
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Thread" })
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_self" g_thread_self :: 
    IO (Ptr Thread)

-- | This function returns the t'GI.GLib.Structs.Thread.Thread' corresponding to the
-- current thread. Note that this function does not increase
-- the reference count of the returned struct.
-- 
-- This function will return a t'GI.GLib.Structs.Thread.Thread' even for threads that
-- were not created by GLib (i.e. those created by other threading
-- APIs). This may be useful for thread identification purposes
-- (i.e. comparisons) but you must not use GLib functions (such
-- as 'GI.GLib.Structs.Thread.threadJoin') on these threads.
threadSelf ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Thread
    -- ^ __Returns:__ the t'GI.GLib.Structs.Thread.Thread' representing the current thread
threadSelf  = liftIO $ do
    result <- g_thread_self
    checkUnexpectedReturnNULL "threadSelf" result
    result' <- (newBoxed Thread) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Thread::yield
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_thread_yield" g_thread_yield :: 
    IO ()

-- | Causes the calling thread to voluntarily relinquish the CPU, so
-- that other threads can run.
-- 
-- This function is often used as a method to make busy wait less evil.
threadYield ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m ()
threadYield  = liftIO $ do
    g_thread_yield
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveThreadMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveThreadMethod "join" o = ThreadJoinMethodInfo
    ResolveThreadMethod "ref" o = ThreadRefMethodInfo
    ResolveThreadMethod "unref" o = ThreadUnrefMethodInfo
    ResolveThreadMethod "getName" o = ThreadGetNameMethodInfo
    ResolveThreadMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveThreadMethod t Thread, O.OverloadedMethod info Thread p) => OL.IsLabel t (Thread -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveThreadMethod t Thread, O.OverloadedMethod info Thread p, R.HasField t Thread p) => R.HasField t Thread p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveThreadMethod t Thread, O.OverloadedMethodInfo info Thread) => OL.IsLabel t (O.MethodProxy info Thread) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


