{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A @GPatternSpec@ struct is the ‘compiled’ form of a glob-style pattern.
-- 
-- The 'GI.GLib.Functions.patternMatchSimple' and 'GI.GLib.Structs.PatternSpec.patternSpecMatch' functions
-- match a string against a pattern containing @*@ and @?@ wildcards with similar
-- semantics as the standard @glob()@ function: @*@ matches an arbitrary,
-- possibly empty, string, @?@ matches an arbitrary character.
-- 
-- Note that in contrast to <http://developer.gnome.org/glib/stable/man:glob(3 `glob()`>), the @\/@ character can be
-- matched by the wildcards, there are no @[…]@ character ranges and @*@ and @?@
-- can not be escaped to include them literally in a pattern.
-- 
-- When multiple strings must be matched against the same pattern, it is better
-- to compile the pattern to a [struct/@gLib@/.PatternSpec] using
-- 'GI.GLib.Structs.PatternSpec.patternSpecNew' and use 'GI.GLib.Structs.PatternSpec.patternSpecMatchString'
-- instead of 'GI.GLib.Functions.patternMatchSimple'. This avoids the overhead of repeated
-- pattern compilation.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.PatternSpec
    ( 

-- * Exported types
    PatternSpec(..)                         ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [copy]("GI.GLib.Structs.PatternSpec#g:method:copy"), [equal]("GI.GLib.Structs.PatternSpec#g:method:equal"), [free]("GI.GLib.Structs.PatternSpec#g:method:free"), [match]("GI.GLib.Structs.PatternSpec#g:method:match"), [matchString]("GI.GLib.Structs.PatternSpec#g:method:matchString").
-- 
-- ==== Getters
-- /None/.
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolvePatternSpecMethod                ,
#endif

-- ** copy #method:copy#

#if defined(ENABLE_OVERLOADING)
    PatternSpecCopyMethodInfo               ,
#endif
    patternSpecCopy                         ,


-- ** equal #method:equal#

#if defined(ENABLE_OVERLOADING)
    PatternSpecEqualMethodInfo              ,
#endif
    patternSpecEqual                        ,


-- ** free #method:free#

#if defined(ENABLE_OVERLOADING)
    PatternSpecFreeMethodInfo               ,
#endif
    patternSpecFree                         ,


-- ** match #method:match#

#if defined(ENABLE_OVERLOADING)
    PatternSpecMatchMethodInfo              ,
#endif
    patternSpecMatch                        ,


-- ** matchString #method:matchString#

#if defined(ENABLE_OVERLOADING)
    PatternSpecMatchStringMethodInfo        ,
#endif
    patternSpecMatchString                  ,


-- ** new #method:new#

    patternSpecNew                          ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- | Memory-managed wrapper type.
newtype PatternSpec = PatternSpec (SP.ManagedPtr PatternSpec)
    deriving (Eq)

instance SP.ManagedPtrNewtype PatternSpec where
    toManagedPtr (PatternSpec p) = p

foreign import ccall "g_pattern_spec_get_type" c_g_pattern_spec_get_type :: 
    IO GType

type instance O.ParentTypes PatternSpec = '[]
instance O.HasParentTypes PatternSpec

instance B.Types.TypedObject PatternSpec where
    glibType = c_g_pattern_spec_get_type

instance B.Types.GBoxed PatternSpec

-- | Convert 'PatternSpec' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe PatternSpec) where
    gvalueGType_ = c_g_pattern_spec_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr PatternSpec)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr PatternSpec)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed PatternSpec ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList PatternSpec
type instance O.AttributeList PatternSpec = PatternSpecAttributeList
type PatternSpecAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method PatternSpec::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "pattern"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a zero-terminated UTF-8 encoded string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "PatternSpec" })
-- throws : False
-- Skip return : False

foreign import ccall "g_pattern_spec_new" g_pattern_spec_new :: 
    CString ->                              -- pattern : TBasicType TUTF8
    IO (Ptr PatternSpec)

-- | Compiles a pattern to a [type/@gLib@/.PatternSpec].
patternSpecNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@pattern@/: a zero-terminated UTF-8 encoded string
    -> m PatternSpec
    -- ^ __Returns:__ a newly-allocated [type/@gLib@/.PatternSpec]
patternSpecNew pattern = liftIO $ do
    pattern' <- textToCString pattern
    result <- g_pattern_spec_new pattern'
    checkUnexpectedReturnNULL "patternSpecNew" result
    result' <- (wrapBoxed PatternSpec) result
    freeMem pattern'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method PatternSpec::copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pspec"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PatternSpec" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GPatternSpec" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "PatternSpec" })
-- throws : False
-- Skip return : False

foreign import ccall "g_pattern_spec_copy" g_pattern_spec_copy :: 
    Ptr PatternSpec ->                      -- pspec : TInterface (Name {namespace = "GLib", name = "PatternSpec"})
    IO (Ptr PatternSpec)

-- | Copies /@pspec@/ in a new [type/@gLib@/.PatternSpec].
-- 
-- /Since: 2.70/
patternSpecCopy ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PatternSpec
    -- ^ /@pspec@/: a t'GI.GLib.Structs.PatternSpec.PatternSpec'
    -> m PatternSpec
    -- ^ __Returns:__ a copy of /@pspec@/.
patternSpecCopy pspec = liftIO $ do
    pspec' <- unsafeManagedPtrGetPtr pspec
    result <- g_pattern_spec_copy pspec'
    checkUnexpectedReturnNULL "patternSpecCopy" result
    result' <- (wrapBoxed PatternSpec) result
    touchManagedPtr pspec
    return result'

#if defined(ENABLE_OVERLOADING)
data PatternSpecCopyMethodInfo
instance (signature ~ (m PatternSpec), MonadIO m) => O.OverloadedMethod PatternSpecCopyMethodInfo PatternSpec signature where
    overloadedMethod = patternSpecCopy

instance O.OverloadedMethodInfo PatternSpecCopyMethodInfo PatternSpec where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PatternSpec.patternSpecCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PatternSpec.html#v:patternSpecCopy"
        })


#endif

-- method PatternSpec::equal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pspec1"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PatternSpec" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GPatternSpec" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pspec2"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PatternSpec" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another #GPatternSpec"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_pattern_spec_equal" g_pattern_spec_equal :: 
    Ptr PatternSpec ->                      -- pspec1 : TInterface (Name {namespace = "GLib", name = "PatternSpec"})
    Ptr PatternSpec ->                      -- pspec2 : TInterface (Name {namespace = "GLib", name = "PatternSpec"})
    IO CInt

-- | Compares two compiled pattern specs and returns whether they will
-- match the same set of strings.
patternSpecEqual ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PatternSpec
    -- ^ /@pspec1@/: a t'GI.GLib.Structs.PatternSpec.PatternSpec'
    -> PatternSpec
    -- ^ /@pspec2@/: another t'GI.GLib.Structs.PatternSpec.PatternSpec'
    -> m Bool
    -- ^ __Returns:__ Whether the compiled patterns are equal
patternSpecEqual pspec1 pspec2 = liftIO $ do
    pspec1' <- unsafeManagedPtrGetPtr pspec1
    pspec2' <- unsafeManagedPtrGetPtr pspec2
    result <- g_pattern_spec_equal pspec1' pspec2'
    let result' = (/= 0) result
    touchManagedPtr pspec1
    touchManagedPtr pspec2
    return result'

#if defined(ENABLE_OVERLOADING)
data PatternSpecEqualMethodInfo
instance (signature ~ (PatternSpec -> m Bool), MonadIO m) => O.OverloadedMethod PatternSpecEqualMethodInfo PatternSpec signature where
    overloadedMethod = patternSpecEqual

instance O.OverloadedMethodInfo PatternSpecEqualMethodInfo PatternSpec where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PatternSpec.patternSpecEqual",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PatternSpec.html#v:patternSpecEqual"
        })


#endif

-- method PatternSpec::free
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pspec"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PatternSpec" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GPatternSpec" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_pattern_spec_free" g_pattern_spec_free :: 
    Ptr PatternSpec ->                      -- pspec : TInterface (Name {namespace = "GLib", name = "PatternSpec"})
    IO ()

-- | Frees the memory allocated for the [type/@gLib@/.PatternSpec].
patternSpecFree ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PatternSpec
    -- ^ /@pspec@/: a t'GI.GLib.Structs.PatternSpec.PatternSpec'
    -> m ()
patternSpecFree pspec = liftIO $ do
    pspec' <- unsafeManagedPtrGetPtr pspec
    g_pattern_spec_free pspec'
    touchManagedPtr pspec
    return ()

#if defined(ENABLE_OVERLOADING)
data PatternSpecFreeMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod PatternSpecFreeMethodInfo PatternSpec signature where
    overloadedMethod = patternSpecFree

instance O.OverloadedMethodInfo PatternSpecFreeMethodInfo PatternSpec where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PatternSpec.patternSpecFree",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PatternSpec.html#v:patternSpecFree"
        })


#endif

-- method PatternSpec::match
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pspec"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PatternSpec" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GPatternSpec" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string_length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @string (in bytes, i.e. `strlen()`,\n   not [func@GLib.utf8_strlen])"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the UTF-8 encoded string to match"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string_reversed"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the reverse of @string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_pattern_spec_match" g_pattern_spec_match :: 
    Ptr PatternSpec ->                      -- pspec : TInterface (Name {namespace = "GLib", name = "PatternSpec"})
    FCT.CSize ->                            -- string_length : TBasicType TSize
    CString ->                              -- string : TBasicType TUTF8
    CString ->                              -- string_reversed : TBasicType TUTF8
    IO CInt

-- | Matches a string against a compiled pattern.
-- 
-- Passing the correct
-- length of the string given is mandatory. The reversed string can be
-- omitted by passing @NULL@, this is more efficient if the reversed
-- version of the string to be matched is not at hand, as
-- 'GI.GLib.Structs.PatternSpec.patternSpecMatch' will only construct it if the compiled pattern
-- requires reverse matches.
-- 
-- Note that, if the user code will (possibly) match a string against a
-- multitude of patterns containing wildcards, chances are high that
-- some patterns will require a reversed string. In this case, it’s
-- more efficient to provide the reversed string to avoid multiple
-- constructions thereof in the various calls to 'GI.GLib.Structs.PatternSpec.patternSpecMatch'.
-- 
-- Note also that the reverse of a UTF-8 encoded string can in general
-- not be obtained by 'GI.GLib.Functions.strreverse'. This works only if the string
-- does not contain any multibyte characters. GLib offers the
-- 'GI.GLib.Functions.utf8Strreverse' function to reverse UTF-8 encoded strings.
-- 
-- /Since: 2.70/
patternSpecMatch ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PatternSpec
    -- ^ /@pspec@/: a t'GI.GLib.Structs.PatternSpec.PatternSpec'
    -> FCT.CSize
    -- ^ /@stringLength@/: the length of /@string@/ (in bytes, i.e. @strlen()@,
    --    not 'GI.GLib.Functions.utf8Strlen')
    -> T.Text
    -- ^ /@string@/: the UTF-8 encoded string to match
    -> Maybe (T.Text)
    -- ^ /@stringReversed@/: the reverse of /@string@/
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@string@/ matches /@pspec@/
patternSpecMatch pspec stringLength string stringReversed = liftIO $ do
    pspec' <- unsafeManagedPtrGetPtr pspec
    string' <- textToCString string
    maybeStringReversed <- case stringReversed of
        Nothing -> return FP.nullPtr
        Just jStringReversed -> do
            jStringReversed' <- textToCString jStringReversed
            return jStringReversed'
    result <- g_pattern_spec_match pspec' stringLength string' maybeStringReversed
    let result' = (/= 0) result
    touchManagedPtr pspec
    freeMem string'
    freeMem maybeStringReversed
    return result'

#if defined(ENABLE_OVERLOADING)
data PatternSpecMatchMethodInfo
instance (signature ~ (FCT.CSize -> T.Text -> Maybe (T.Text) -> m Bool), MonadIO m) => O.OverloadedMethod PatternSpecMatchMethodInfo PatternSpec signature where
    overloadedMethod = patternSpecMatch

instance O.OverloadedMethodInfo PatternSpecMatchMethodInfo PatternSpec where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PatternSpec.patternSpecMatch",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PatternSpec.html#v:patternSpecMatch"
        })


#endif

-- method PatternSpec::match_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "pspec"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PatternSpec" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GPatternSpec" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the UTF-8 encoded string to match"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_pattern_spec_match_string" g_pattern_spec_match_string :: 
    Ptr PatternSpec ->                      -- pspec : TInterface (Name {namespace = "GLib", name = "PatternSpec"})
    CString ->                              -- string : TBasicType TUTF8
    IO CInt

-- | Matches a string against a compiled pattern.
-- 
-- If the string is to be
-- matched against more than one pattern, consider using
-- 'GI.GLib.Structs.PatternSpec.patternSpecMatch' instead while supplying the reversed string.
-- 
-- /Since: 2.70/
patternSpecMatchString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PatternSpec
    -- ^ /@pspec@/: a t'GI.GLib.Structs.PatternSpec.PatternSpec'
    -> T.Text
    -- ^ /@string@/: the UTF-8 encoded string to match
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@string@/ matches /@pspec@/
patternSpecMatchString pspec string = liftIO $ do
    pspec' <- unsafeManagedPtrGetPtr pspec
    string' <- textToCString string
    result <- g_pattern_spec_match_string pspec' string'
    let result' = (/= 0) result
    touchManagedPtr pspec
    freeMem string'
    return result'

#if defined(ENABLE_OVERLOADING)
data PatternSpecMatchStringMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m) => O.OverloadedMethod PatternSpecMatchStringMethodInfo PatternSpec signature where
    overloadedMethod = patternSpecMatchString

instance O.OverloadedMethodInfo PatternSpecMatchStringMethodInfo PatternSpec where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PatternSpec.patternSpecMatchString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PatternSpec.html#v:patternSpecMatchString"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type family ResolvePatternSpecMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolvePatternSpecMethod "copy" o = PatternSpecCopyMethodInfo
    ResolvePatternSpecMethod "equal" o = PatternSpecEqualMethodInfo
    ResolvePatternSpecMethod "free" o = PatternSpecFreeMethodInfo
    ResolvePatternSpecMethod "match" o = PatternSpecMatchMethodInfo
    ResolvePatternSpecMethod "matchString" o = PatternSpecMatchStringMethodInfo
    ResolvePatternSpecMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolvePatternSpecMethod t PatternSpec, O.OverloadedMethod info PatternSpec p) => OL.IsLabel t (PatternSpec -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolvePatternSpecMethod t PatternSpec, O.OverloadedMethod info PatternSpec p, R.HasField t PatternSpec p) => R.HasField t PatternSpec p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolvePatternSpecMethod t PatternSpec, O.OverloadedMethodInfo info PatternSpec) => OL.IsLabel t (O.MethodProxy info PatternSpec) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


