{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GPathBuf@ is a helper type that allows you to easily build paths from
-- individual elements, using the platform specific conventions for path
-- separators.
-- 
-- 
-- === /c code/
-- >g_auto (GPathBuf) path;
-- >
-- >g_path_buf_init (&path);
-- >
-- >g_path_buf_push (&path, "usr");
-- >g_path_buf_push (&path, "bin");
-- >g_path_buf_push (&path, "echo");
-- >
-- >g_autofree char *echo = g_path_buf_to_path (&path);
-- >g_assert_cmpstr (echo, ==, "/usr/bin/echo");
-- 
-- 
-- You can also load a full path and then operate on its components:
-- 
-- 
-- === /c code/
-- >g_auto (GPathBuf) path;
-- >
-- >g_path_buf_init_from_path (&path, "/usr/bin/echo");
-- >
-- >g_path_buf_pop (&path);
-- >g_path_buf_push (&path, "sh");
-- >
-- >g_autofree char *sh = g_path_buf_to_path (&path);
-- >g_assert_cmpstr (sh, ==, "/usr/bin/sh");
-- 
-- 
-- /Since: 2.76/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.PathBuf
    ( 

-- * Exported types
    PathBuf(..)                             ,
    newZeroPathBuf                          ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [clear]("GI.GLib.Structs.PathBuf#g:method:clear"), [clearToPath]("GI.GLib.Structs.PathBuf#g:method:clearToPath"), [free]("GI.GLib.Structs.PathBuf#g:method:free"), [freeToPath]("GI.GLib.Structs.PathBuf#g:method:freeToPath"), [init]("GI.GLib.Structs.PathBuf#g:method:init"), [initFromPath]("GI.GLib.Structs.PathBuf#g:method:initFromPath"), [pop]("GI.GLib.Structs.PathBuf#g:method:pop"), [push]("GI.GLib.Structs.PathBuf#g:method:push"), [toPath]("GI.GLib.Structs.PathBuf#g:method:toPath").
-- 
-- ==== Getters
-- /None/.
-- 
-- ==== Setters
-- [setExtension]("GI.GLib.Structs.PathBuf#g:method:setExtension"), [setFilename]("GI.GLib.Structs.PathBuf#g:method:setFilename").

#if defined(ENABLE_OVERLOADING)
    ResolvePathBufMethod                    ,
#endif

-- ** clear #method:clear#

#if defined(ENABLE_OVERLOADING)
    PathBufClearMethodInfo                  ,
#endif
    pathBufClear                            ,


-- ** clearToPath #method:clearToPath#

#if defined(ENABLE_OVERLOADING)
    PathBufClearToPathMethodInfo            ,
#endif
    pathBufClearToPath                      ,


-- ** equal #method:equal#

    pathBufEqual                            ,


-- ** free #method:free#

#if defined(ENABLE_OVERLOADING)
    PathBufFreeMethodInfo                   ,
#endif
    pathBufFree                             ,


-- ** freeToPath #method:freeToPath#

#if defined(ENABLE_OVERLOADING)
    PathBufFreeToPathMethodInfo             ,
#endif
    pathBufFreeToPath                       ,


-- ** init #method:init#

#if defined(ENABLE_OVERLOADING)
    PathBufInitMethodInfo                   ,
#endif
    pathBufInit                             ,


-- ** initFromPath #method:initFromPath#

#if defined(ENABLE_OVERLOADING)
    PathBufInitFromPathMethodInfo           ,
#endif
    pathBufInitFromPath                     ,


-- ** pop #method:pop#

#if defined(ENABLE_OVERLOADING)
    PathBufPopMethodInfo                    ,
#endif
    pathBufPop                              ,


-- ** push #method:push#

#if defined(ENABLE_OVERLOADING)
    PathBufPushMethodInfo                   ,
#endif
    pathBufPush                             ,


-- ** setExtension #method:setExtension#

#if defined(ENABLE_OVERLOADING)
    PathBufSetExtensionMethodInfo           ,
#endif
    pathBufSetExtension                     ,


-- ** setFilename #method:setFilename#

#if defined(ENABLE_OVERLOADING)
    PathBufSetFilenameMethodInfo            ,
#endif
    pathBufSetFilename                      ,


-- ** toPath #method:toPath#

#if defined(ENABLE_OVERLOADING)
    PathBufToPathMethodInfo                 ,
#endif
    pathBufToPath                           ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- | Memory-managed wrapper type.
newtype PathBuf = PathBuf (SP.ManagedPtr PathBuf)
    deriving (Eq)

instance SP.ManagedPtrNewtype PathBuf where
    toManagedPtr (PathBuf p) = p

instance BoxedPtr PathBuf where
    boxedPtrCopy = \p -> B.ManagedPtr.withManagedPtr p (copyBytes 64 >=> B.ManagedPtr.wrapPtr PathBuf)
    boxedPtrFree = \x -> SP.withManagedPtr x SP.freeMem
instance CallocPtr PathBuf where
    boxedPtrCalloc = callocBytes 64


-- | Construct a `PathBuf` struct initialized to zero.
newZeroPathBuf :: MonadIO m => m PathBuf
newZeroPathBuf = liftIO $ boxedPtrCalloc >>= wrapPtr PathBuf

instance tag ~ 'AttrSet => Constructible PathBuf tag where
    new _ attrs = do
        o <- newZeroPathBuf
        GI.Attributes.set o attrs
        return o



#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList PathBuf
type instance O.AttributeList PathBuf = PathBufAttributeList
type PathBufAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method PathBuf::clear
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_clear" g_path_buf_clear :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    IO ()

-- | Clears the contents of the path buffer.
-- 
-- This function should be use to free the resources in a stack-allocated
-- @GPathBuf@ initialized using 'GI.GLib.Structs.PathBuf.pathBufInit' or
-- 'GI.GLib.Structs.PathBuf.pathBufInitFromPath'.
-- 
-- /Since: 2.76/
pathBufClear ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> m ()
pathBufClear buf = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    g_path_buf_clear buf'
    touchManagedPtr buf
    return ()

#if defined(ENABLE_OVERLOADING)
data PathBufClearMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod PathBufClearMethodInfo PathBuf signature where
    overloadedMethod = pathBufClear

instance O.OverloadedMethodInfo PathBufClearMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufClear",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufClear"
        })


#endif

-- method PathBuf::clear_to_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFileName)
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_clear_to_path" g_path_buf_clear_to_path :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    IO CString

-- | Clears the contents of the path buffer and returns the built path.
-- 
-- This function returns @NULL@ if the @GPathBuf@ is empty.
-- 
-- See also: 'GI.GLib.Structs.PathBuf.pathBufToPath'
-- 
-- /Since: 2.76/
pathBufClearToPath ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> m (Maybe [Char])
    -- ^ __Returns:__ the built path
pathBufClearToPath buf = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    result <- g_path_buf_clear_to_path buf'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToString result'
        freeMem result'
        return result''
    touchManagedPtr buf
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data PathBufClearToPathMethodInfo
instance (signature ~ (m (Maybe [Char])), MonadIO m) => O.OverloadedMethod PathBufClearToPathMethodInfo PathBuf signature where
    overloadedMethod = pathBufClearToPath

instance O.OverloadedMethodInfo PathBufClearToPathMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufClearToPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufClearToPath"
        })


#endif

-- method PathBuf::free
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_free" g_path_buf_free :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    IO ()

-- | Frees a @GPathBuf@ allocated by @/g_path_buf_new()/@.
-- 
-- /Since: 2.76/
pathBufFree ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> m ()
pathBufFree buf = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    g_path_buf_free buf'
    touchManagedPtr buf
    return ()

#if defined(ENABLE_OVERLOADING)
data PathBufFreeMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod PathBufFreeMethodInfo PathBuf signature where
    overloadedMethod = pathBufFree

instance O.OverloadedMethodInfo PathBufFreeMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufFree",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufFree"
        })


#endif

-- method PathBuf::free_to_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFileName)
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_free_to_path" g_path_buf_free_to_path :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    IO CString

-- | Frees a @GPathBuf@ allocated by @/g_path_buf_new()/@, and
-- returns the path inside the buffer.
-- 
-- This function returns @NULL@ if the @GPathBuf@ is empty.
-- 
-- See also: 'GI.GLib.Structs.PathBuf.pathBufToPath'
-- 
-- /Since: 2.76/
pathBufFreeToPath ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> m (Maybe [Char])
    -- ^ __Returns:__ the path
pathBufFreeToPath buf = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    result <- g_path_buf_free_to_path buf'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToString result'
        freeMem result'
        return result''
    touchManagedPtr buf
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data PathBufFreeToPathMethodInfo
instance (signature ~ (m (Maybe [Char])), MonadIO m) => O.OverloadedMethod PathBufFreeToPathMethodInfo PathBuf signature where
    overloadedMethod = pathBufFreeToPath

instance O.OverloadedMethodInfo PathBufFreeToPathMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufFreeToPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufFreeToPath"
        })


#endif

-- method PathBuf::init
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "PathBuf" })
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_init" g_path_buf_init :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    IO (Ptr PathBuf)

-- | Initializes a @GPathBuf@ instance.
-- 
-- /Since: 2.76/
pathBufInit ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> m PathBuf
    -- ^ __Returns:__ the initialized path builder
pathBufInit buf = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    result <- g_path_buf_init buf'
    checkUnexpectedReturnNULL "pathBufInit" result
    result' <- (newPtr PathBuf) result
    touchManagedPtr buf
    return result'

#if defined(ENABLE_OVERLOADING)
data PathBufInitMethodInfo
instance (signature ~ (m PathBuf), MonadIO m) => O.OverloadedMethod PathBufInitMethodInfo PathBuf signature where
    overloadedMethod = pathBufInit

instance O.OverloadedMethodInfo PathBufInitMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufInit",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufInit"
        })


#endif

-- method PathBuf::init_from_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a file system path" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "PathBuf" })
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_init_from_path" g_path_buf_init_from_path :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    CString ->                              -- path : TBasicType TFileName
    IO (Ptr PathBuf)

-- | Initializes a @GPathBuf@ instance with the given path.
-- 
-- /Since: 2.76/
pathBufInitFromPath ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> Maybe ([Char])
    -- ^ /@path@/: a file system path
    -> m PathBuf
    -- ^ __Returns:__ the initialized path builder
pathBufInitFromPath buf path = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    maybePath <- case path of
        Nothing -> return FP.nullPtr
        Just jPath -> do
            jPath' <- stringToCString jPath
            return jPath'
    result <- g_path_buf_init_from_path buf' maybePath
    checkUnexpectedReturnNULL "pathBufInitFromPath" result
    result' <- (newPtr PathBuf) result
    touchManagedPtr buf
    freeMem maybePath
    return result'

#if defined(ENABLE_OVERLOADING)
data PathBufInitFromPathMethodInfo
instance (signature ~ (Maybe ([Char]) -> m PathBuf), MonadIO m) => O.OverloadedMethod PathBufInitFromPathMethodInfo PathBuf signature where
    overloadedMethod = pathBufInitFromPath

instance O.OverloadedMethodInfo PathBufInitFromPathMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufInitFromPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufInitFromPath"
        })


#endif

-- method PathBuf::pop
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_pop" g_path_buf_pop :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    IO CInt

-- | Removes the last element of the path buffer.
-- 
-- If there is only one element in the path buffer (for example, @\/@ on
-- Unix-like operating systems or the drive on Windows systems), it will
-- not be removed and 'P.False' will be returned instead.
-- 
-- 
-- === /C code/
-- >
-- >GPathBuf buf, cmp;
-- >
-- >g_path_buf_init_from_path (&buf, "/bin/sh");
-- >
-- >g_path_buf_pop (&buf);
-- >g_path_buf_init_from_path (&cmp, "/bin");
-- >g_assert_true (g_path_buf_equal (&buf, &cmp));
-- >g_path_buf_clear (&cmp);
-- >
-- >g_path_buf_pop (&buf);
-- >g_path_buf_init_from_path (&cmp, "/");
-- >g_assert_true (g_path_buf_equal (&buf, &cmp));
-- >g_path_buf_clear (&cmp);
-- >
-- >g_path_buf_clear (&buf);
-- 
-- 
-- /Since: 2.76/
pathBufPop ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> m Bool
    -- ^ __Returns:__ @TRUE@ if the buffer was modified and @FALSE@ otherwise
pathBufPop buf = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    result <- g_path_buf_pop buf'
    let result' = (/= 0) result
    touchManagedPtr buf
    return result'

#if defined(ENABLE_OVERLOADING)
data PathBufPopMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod PathBufPopMethodInfo PathBuf signature where
    overloadedMethod = pathBufPop

instance O.OverloadedMethodInfo PathBufPopMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufPop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufPop"
        })


#endif

-- method PathBuf::push
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "PathBuf" })
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_push" g_path_buf_push :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    CString ->                              -- path : TBasicType TFileName
    IO (Ptr PathBuf)

-- | Extends the given path buffer with /@path@/.
-- 
-- If /@path@/ is absolute, it replaces the current path.
-- 
-- If /@path@/ contains a directory separator, the buffer is extended by
-- as many elements the path provides.
-- 
-- On Windows, both forward slashes and backslashes are treated as
-- directory separators. On other platforms, 'GI.GLib.Constants.DIR_SEPARATOR_S' is the
-- only directory separator.
-- 
-- 
-- === /C code/
-- >
-- >GPathBuf buf, cmp;
-- >
-- >g_path_buf_init_from_path (&buf, "/tmp");
-- >g_path_buf_push (&buf, ".X11-unix/X0");
-- >g_path_buf_init_from_path (&cmp, "/tmp/.X11-unix/X0");
-- >g_assert_true (g_path_buf_equal (&buf, &cmp));
-- >g_path_buf_clear (&cmp);
-- >
-- >g_path_buf_push (&buf, "/etc/locale.conf");
-- >g_path_buf_init_from_path (&cmp, "/etc/locale.conf");
-- >g_assert_true (g_path_buf_equal (&buf, &cmp));
-- >g_path_buf_clear (&cmp);
-- >
-- >g_path_buf_clear (&buf);
-- 
-- 
-- /Since: 2.76/
pathBufPush ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> [Char]
    -- ^ /@path@/: a path
    -> m PathBuf
    -- ^ __Returns:__ the same pointer to /@buf@/, for convenience
pathBufPush buf path = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    path' <- stringToCString path
    result <- g_path_buf_push buf' path'
    checkUnexpectedReturnNULL "pathBufPush" result
    result' <- (newPtr PathBuf) result
    touchManagedPtr buf
    freeMem path'
    return result'

#if defined(ENABLE_OVERLOADING)
data PathBufPushMethodInfo
instance (signature ~ ([Char] -> m PathBuf), MonadIO m) => O.OverloadedMethod PathBufPushMethodInfo PathBuf signature where
    overloadedMethod = pathBufPush

instance O.OverloadedMethodInfo PathBufPushMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufPush",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufPush"
        })


#endif

-- method PathBuf::set_extension
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "extension"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the file extension" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_set_extension" g_path_buf_set_extension :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    CString ->                              -- extension : TBasicType TFileName
    IO CInt

-- | Adds an extension to the file name in the path buffer.
-- 
-- If /@extension@/ is @NULL@, the extension will be unset.
-- 
-- If the path buffer does not have a file name set, this function returns
-- @FALSE@ and leaves the path buffer unmodified.
-- 
-- /Since: 2.76/
pathBufSetExtension ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> Maybe ([Char])
    -- ^ /@extension@/: the file extension
    -> m Bool
    -- ^ __Returns:__ @TRUE@ if the extension was replaced, and @FALSE@ otherwise
pathBufSetExtension buf extension = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    maybeExtension <- case extension of
        Nothing -> return FP.nullPtr
        Just jExtension -> do
            jExtension' <- stringToCString jExtension
            return jExtension'
    result <- g_path_buf_set_extension buf' maybeExtension
    let result' = (/= 0) result
    touchManagedPtr buf
    freeMem maybeExtension
    return result'

#if defined(ENABLE_OVERLOADING)
data PathBufSetExtensionMethodInfo
instance (signature ~ (Maybe ([Char]) -> m Bool), MonadIO m) => O.OverloadedMethod PathBufSetExtensionMethodInfo PathBuf signature where
    overloadedMethod = pathBufSetExtension

instance O.OverloadedMethodInfo PathBufSetExtensionMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufSetExtension",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufSetExtension"
        })


#endif

-- method PathBuf::set_filename
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "file_name"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the file name in the path"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_set_filename" g_path_buf_set_filename :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    CString ->                              -- file_name : TBasicType TFileName
    IO CInt

-- | Sets the file name of the path.
-- 
-- If the path buffer is empty, the filename is left unset and this
-- function returns @FALSE@.
-- 
-- If the path buffer only contains the root element (on Unix-like operating
-- systems) or the drive (on Windows), this is the equivalent of pushing
-- the new /@fileName@/.
-- 
-- If the path buffer contains a path, this is the equivalent of
-- popping the path buffer and pushing /@fileName@/, creating a
-- sibling of the original path.
-- 
-- 
-- === /C code/
-- >
-- >GPathBuf buf, cmp;
-- >
-- >g_path_buf_init_from_path (&buf, "/");
-- >
-- >g_path_buf_set_filename (&buf, "bar");
-- >g_path_buf_init_from_path (&cmp, "/bar");
-- >g_assert_true (g_path_buf_equal (&buf, &cmp));
-- >g_path_buf_clear (&cmp);
-- >
-- >g_path_buf_set_filename (&buf, "baz.txt");
-- >g_path_buf_init_from_path (&cmp, "/baz.txt");
-- >g_assert_true (g_path_buf_equal (&buf, &cmp);
-- >g_path_buf_clear (&cmp);
-- >
-- >g_path_buf_clear (&buf);
-- 
-- 
-- /Since: 2.76/
pathBufSetFilename ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> [Char]
    -- ^ /@fileName@/: the file name in the path
    -> m Bool
    -- ^ __Returns:__ @TRUE@ if the file name was replaced, and @FALSE@ otherwise
pathBufSetFilename buf fileName = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    fileName' <- stringToCString fileName
    result <- g_path_buf_set_filename buf' fileName'
    let result' = (/= 0) result
    touchManagedPtr buf
    freeMem fileName'
    return result'

#if defined(ENABLE_OVERLOADING)
data PathBufSetFilenameMethodInfo
instance (signature ~ ([Char] -> m Bool), MonadIO m) => O.OverloadedMethod PathBufSetFilenameMethodInfo PathBuf signature where
    overloadedMethod = pathBufSetFilename

instance O.OverloadedMethodInfo PathBufSetFilenameMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufSetFilename",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufSetFilename"
        })


#endif

-- method PathBuf::to_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "buf"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "PathBuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFileName)
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_to_path" g_path_buf_to_path :: 
    Ptr PathBuf ->                          -- buf : TInterface (Name {namespace = "GLib", name = "PathBuf"})
    IO CString

-- | Retrieves the built path from the path buffer.
-- 
-- On Windows, the result contains backslashes as directory separators,
-- even if forward slashes were used in input.
-- 
-- If the path buffer is empty, this function returns @NULL@.
-- 
-- /Since: 2.76/
pathBufToPath ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathBuf
    -- ^ /@buf@/: a path buffer
    -> m (Maybe [Char])
    -- ^ __Returns:__ the path
pathBufToPath buf = liftIO $ do
    buf' <- unsafeManagedPtrGetPtr buf
    result <- g_path_buf_to_path buf'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToString result'
        freeMem result'
        return result''
    touchManagedPtr buf
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data PathBufToPathMethodInfo
instance (signature ~ (m (Maybe [Char])), MonadIO m) => O.OverloadedMethod PathBufToPathMethodInfo PathBuf signature where
    overloadedMethod = pathBufToPath

instance O.OverloadedMethodInfo PathBufToPathMethodInfo PathBuf where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.PathBuf.pathBufToPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-PathBuf.html#v:pathBufToPath"
        })


#endif

-- method PathBuf::equal
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "v1"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer to compare"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "v2"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path buffer to compare"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_path_buf_equal" g_path_buf_equal :: 
    Ptr () ->                               -- v1 : TBasicType TPtr
    Ptr () ->                               -- v2 : TBasicType TPtr
    IO CInt

-- | Compares two path buffers for equality and returns @TRUE@
-- if they are equal.
-- 
-- The path inside the paths buffers are not going to be normalized,
-- so @X\/Y\/Z\/A\/..@, @X\/.\/Y\/Z@ and @X\/Y\/Z@ are not going to be considered
-- equal.
-- 
-- This function can be passed to @/g_hash_table_new()/@ as the
-- @key_equal_func@ parameter.
-- 
-- /Since: 2.76/
pathBufEqual ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Ptr ()
    -- ^ /@v1@/: a path buffer to compare
    -> Ptr ()
    -- ^ /@v2@/: a path buffer to compare
    -> m Bool
    -- ^ __Returns:__ @TRUE@ if the two path buffers are equal,
    --   and @FALSE@ otherwise
pathBufEqual v1 v2 = liftIO $ do
    result <- g_path_buf_equal v1 v2
    let result' = (/= 0) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolvePathBufMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolvePathBufMethod "clear" o = PathBufClearMethodInfo
    ResolvePathBufMethod "clearToPath" o = PathBufClearToPathMethodInfo
    ResolvePathBufMethod "free" o = PathBufFreeMethodInfo
    ResolvePathBufMethod "freeToPath" o = PathBufFreeToPathMethodInfo
    ResolvePathBufMethod "init" o = PathBufInitMethodInfo
    ResolvePathBufMethod "initFromPath" o = PathBufInitFromPathMethodInfo
    ResolvePathBufMethod "pop" o = PathBufPopMethodInfo
    ResolvePathBufMethod "push" o = PathBufPushMethodInfo
    ResolvePathBufMethod "toPath" o = PathBufToPathMethodInfo
    ResolvePathBufMethod "setExtension" o = PathBufSetExtensionMethodInfo
    ResolvePathBufMethod "setFilename" o = PathBufSetFilenameMethodInfo
    ResolvePathBufMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolvePathBufMethod t PathBuf, O.OverloadedMethod info PathBuf p) => OL.IsLabel t (PathBuf -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolvePathBufMethod t PathBuf, O.OverloadedMethod info PathBuf p, R.HasField t PathBuf p) => R.HasField t PathBuf p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolvePathBufMethod t PathBuf, O.OverloadedMethodInfo info PathBuf) => OL.IsLabel t (O.MethodProxy info PathBuf) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


