{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Structure representing a single field in a structured log entry. See
-- @/g_log_structured()/@ for details.
-- 
-- Log fields may contain arbitrary values, including binary with embedded nul
-- bytes. If the field contains a string, the string must be UTF-8 encoded and
-- have a trailing nul byte. Otherwise, /@length@/ must be set to a non-negative
-- value.
-- 
-- /Since: 2.50/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.LogField
    ( 

-- * Exported types
    LogField(..)                            ,
    newZeroLogField                         ,


 -- * Methods

#if defined(ENABLE_OVERLOADING)
    ResolveLogFieldMethod                   ,
#endif



 -- * Properties


-- ** key #attr:key#
-- | field name (UTF-8 string)

    clearLogFieldKey                        ,
    getLogFieldKey                          ,
#if defined(ENABLE_OVERLOADING)
    logField_key                            ,
#endif
    setLogFieldKey                          ,


-- ** length #attr:length#
-- | length of /@value@/, in bytes, or -1 if it is nul-terminated

    getLogFieldLength                       ,
#if defined(ENABLE_OVERLOADING)
    logField_length                         ,
#endif
    setLogFieldLength                       ,


-- ** value #attr:value#
-- | field value (arbitrary bytes)

    clearLogFieldValue                      ,
    getLogFieldValue                        ,
#if defined(ENABLE_OVERLOADING)
    logField_value                          ,
#endif
    setLogFieldValue                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- | Memory-managed wrapper type.
newtype LogField = LogField (SP.ManagedPtr LogField)
    deriving (Eq)

instance SP.ManagedPtrNewtype LogField where
    toManagedPtr (LogField p) = p

instance BoxedPtr LogField where
    boxedPtrCopy = \p -> B.ManagedPtr.withManagedPtr p (copyBytes 24 >=> B.ManagedPtr.wrapPtr LogField)
    boxedPtrFree = \x -> SP.withManagedPtr x SP.freeMem
instance CallocPtr LogField where
    boxedPtrCalloc = callocBytes 24


-- | Construct a `LogField` struct initialized to zero.
newZeroLogField :: MonadIO m => m LogField
newZeroLogField = liftIO $ boxedPtrCalloc >>= wrapPtr LogField

instance tag ~ 'AttrSet => Constructible LogField tag where
    new _ attrs = do
        o <- newZeroLogField
        GI.Attributes.set o attrs
        return o


-- | Get the value of the “@key@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' logField #key
-- @
getLogFieldKey :: MonadIO m => LogField -> m (Maybe T.Text)
getLogFieldKey s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 0) :: IO CString
    result <- SP.convertIfNonNull val $ \val' -> do
        val'' <- cstringToText val'
        return val''
    return result

-- | Set the value of the “@key@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' logField [ #key 'Data.GI.Base.Attributes.:=' value ]
-- @
setLogFieldKey :: MonadIO m => LogField -> CString -> m ()
setLogFieldKey s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 0) (val :: CString)

-- | Set the value of the “@key@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #key
-- @
clearLogFieldKey :: MonadIO m => LogField -> m ()
clearLogFieldKey s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 0) (FP.nullPtr :: CString)

#if defined(ENABLE_OVERLOADING)
data LogFieldKeyFieldInfo
instance AttrInfo LogFieldKeyFieldInfo where
    type AttrBaseTypeConstraint LogFieldKeyFieldInfo = (~) LogField
    type AttrAllowedOps LogFieldKeyFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint LogFieldKeyFieldInfo = (~) CString
    type AttrTransferTypeConstraint LogFieldKeyFieldInfo = (~)CString
    type AttrTransferType LogFieldKeyFieldInfo = CString
    type AttrGetType LogFieldKeyFieldInfo = Maybe T.Text
    type AttrLabel LogFieldKeyFieldInfo = "key"
    type AttrOrigin LogFieldKeyFieldInfo = LogField
    attrGet = getLogFieldKey
    attrSet = setLogFieldKey
    attrConstruct = undefined
    attrClear = clearLogFieldKey
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.LogField.key"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-LogField.html#g:attr:key"
        })

logField_key :: AttrLabelProxy "key"
logField_key = AttrLabelProxy

#endif


-- | Get the value of the “@value@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' logField #value
-- @
getLogFieldValue :: MonadIO m => LogField -> m (Ptr ())
getLogFieldValue s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 8) :: IO (Ptr ())
    return val

-- | Set the value of the “@value@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' logField [ #value 'Data.GI.Base.Attributes.:=' value ]
-- @
setLogFieldValue :: MonadIO m => LogField -> Ptr () -> m ()
setLogFieldValue s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 8) (val :: Ptr ())

-- | Set the value of the “@value@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #value
-- @
clearLogFieldValue :: MonadIO m => LogField -> m ()
clearLogFieldValue s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 8) (FP.nullPtr :: Ptr ())

#if defined(ENABLE_OVERLOADING)
data LogFieldValueFieldInfo
instance AttrInfo LogFieldValueFieldInfo where
    type AttrBaseTypeConstraint LogFieldValueFieldInfo = (~) LogField
    type AttrAllowedOps LogFieldValueFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint LogFieldValueFieldInfo = (~) (Ptr ())
    type AttrTransferTypeConstraint LogFieldValueFieldInfo = (~)(Ptr ())
    type AttrTransferType LogFieldValueFieldInfo = (Ptr ())
    type AttrGetType LogFieldValueFieldInfo = Ptr ()
    type AttrLabel LogFieldValueFieldInfo = "value"
    type AttrOrigin LogFieldValueFieldInfo = LogField
    attrGet = getLogFieldValue
    attrSet = setLogFieldValue
    attrConstruct = undefined
    attrClear = clearLogFieldValue
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.LogField.value"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-LogField.html#g:attr:value"
        })

logField_value :: AttrLabelProxy "value"
logField_value = AttrLabelProxy

#endif


-- | Get the value of the “@length@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' logField #length
-- @
getLogFieldLength :: MonadIO m => LogField -> m DI.Int64
getLogFieldLength s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 16) :: IO DI.Int64
    return val

-- | Set the value of the “@length@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' logField [ #length 'Data.GI.Base.Attributes.:=' value ]
-- @
setLogFieldLength :: MonadIO m => LogField -> DI.Int64 -> m ()
setLogFieldLength s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 16) (val :: DI.Int64)

#if defined(ENABLE_OVERLOADING)
data LogFieldLengthFieldInfo
instance AttrInfo LogFieldLengthFieldInfo where
    type AttrBaseTypeConstraint LogFieldLengthFieldInfo = (~) LogField
    type AttrAllowedOps LogFieldLengthFieldInfo = '[ 'AttrSet, 'AttrGet]
    type AttrSetTypeConstraint LogFieldLengthFieldInfo = (~) DI.Int64
    type AttrTransferTypeConstraint LogFieldLengthFieldInfo = (~)DI.Int64
    type AttrTransferType LogFieldLengthFieldInfo = DI.Int64
    type AttrGetType LogFieldLengthFieldInfo = DI.Int64
    type AttrLabel LogFieldLengthFieldInfo = "length"
    type AttrOrigin LogFieldLengthFieldInfo = LogField
    attrGet = getLogFieldLength
    attrSet = setLogFieldLength
    attrConstruct = undefined
    attrClear = undefined
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.LogField.length"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-LogField.html#g:attr:length"
        })

logField_length :: AttrLabelProxy "length"
logField_length = AttrLabelProxy

#endif



#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList LogField
type instance O.AttributeList LogField = LogFieldAttributeList
type LogFieldAttributeList = ('[ '("key", LogFieldKeyFieldInfo), '("value", LogFieldValueFieldInfo), '("length", LogFieldLengthFieldInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveLogFieldMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveLogFieldMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveLogFieldMethod t LogField, O.OverloadedMethod info LogField p) => OL.IsLabel t (LogField -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveLogFieldMethod t LogField, O.OverloadedMethod info LogField p, R.HasField t LogField p) => R.HasField t LogField p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveLogFieldMethod t LogField, O.OverloadedMethodInfo info LogField) => OL.IsLabel t (O.MethodProxy info LogField) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


