{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GKeyFile@ parses .ini-like config files.
-- 
-- @GKeyFile@ lets you parse, edit or create files containing groups of
-- key-value pairs, which we call ‘key files’ for lack of a better name.
-- Several freedesktop.org specifications use key files. For example, the
-- <https://specifications.freedesktop.org/desktop-entry-spec/latest/ Desktop Entry Specification>
-- and the <https://specifications.freedesktop.org/icon-theme-spec/latest/ Icon Theme Specification>.
-- 
-- The syntax of key files is described in detail in the
-- <https://specifications.freedesktop.org/desktop-entry-spec/latest/ Desktop Entry Specification>,
-- here is a quick summary: Key files consists of groups of key-value pairs, interspersed
-- with comments.
-- 
-- 
-- === /txt code/
-- ># this is just an example
-- ># there can be comments before the first group
-- >
-- >[First Group]
-- >
-- >Name=Key File Example\tthis value shows\nescaping
-- >
-- ># localized strings are stored in multiple key-value pairs
-- >Welcome=Hello
-- >Welcome[de]=Hallo
-- >Welcome[fr_FR]=Bonjour
-- >Welcome[it]=Ciao
-- >
-- >[Another Group]
-- >
-- >Numbers=2;20;-200;0
-- >
-- >Booleans=true;false;true;true
-- 
-- 
-- Lines beginning with a @#@ and blank lines are considered comments.
-- 
-- Groups are started by a header line containing the group name enclosed
-- in @[@ and @]@, and ended implicitly by the start of the next group or
-- the end of the file. Each key-value pair must be contained in a group.
-- 
-- Key-value pairs generally have the form @key=value@, with the exception
-- of localized strings, which have the form @key[locale]=value@, with a
-- locale identifier of the form @lang_COUNTRY\@MODIFIER@ where @COUNTRY@
-- and @MODIFIER@ are optional. As a special case, the locale @C@ is associated
-- with the untranslated pair @key=value@ (since GLib 2.84). Space before and
-- after the @=@ character is ignored. Newline, tab, carriage return and
-- backslash characters in value are escaped as @\\n@, @\\t@, @\\r@, and @\\\\\\\\@,
-- respectively. To preserve leading spaces in values, these can also be escaped
-- as @\\s@.
-- 
-- Key files can store strings (possibly with localized variants), integers,
-- booleans and lists of these. Lists are separated by a separator character,
-- typically @;@ or @,@. To use the list separator character in a value in
-- a list, it has to be escaped by prefixing it with a backslash.
-- 
-- This syntax is obviously inspired by the .ini files commonly met
-- on Windows, but there are some important differences:
-- 
-- * .ini files use the @;@ character to begin comments,
-- key files use the @#@ character.
-- * Key files do not allow for ungrouped keys meaning only
-- comments can precede the first group.
-- * Key files are always encoded in UTF-8.
-- * Key and Group names are case-sensitive. For example, a group called
-- @[GROUP]@ is a different from @[group]@.
-- * .ini files don’t have a strongly typed boolean entry type,
--  they only have @GetProfileInt()@. In key files, only
--  @true@ and @false@ (in lower case) are allowed.
-- 
-- 
-- Note that in contrast to the
-- <https://specifications.freedesktop.org/desktop-entry-spec/latest/ Desktop Entry Specification>,
-- groups in key files may contain the same key multiple times; the last entry wins.
-- Key files may also contain multiple groups with the same name; they are merged
-- together. Another difference is that keys and group names in key files are not
-- restricted to ASCII characters.
-- 
-- Here is an example of loading a key file and reading a value:
-- 
-- 
-- === /c code/
-- >g_autoptr(GError) error = NULL;
-- >g_autoptr(GKeyFile) key_file = g_key_file_new ();
-- >
-- >if (!g_key_file_load_from_file (key_file, "key-file.ini", flags, &error))
-- >  {
-- >    if (!g_error_matches (error, G_FILE_ERROR, G_FILE_ERROR_NOENT))
-- >      g_warning ("Error loading key file: %s", error->message);
-- >    return;
-- >  }
-- >
-- >g_autofree gchar *val = g_key_file_get_string (key_file, "Group Name", "SomeKey", &error);
-- >if (val == NULL &&
-- >    !g_error_matches (error, G_KEY_FILE_ERROR, G_KEY_FILE_ERROR_KEY_NOT_FOUND))
-- >  {
-- >    g_warning ("Error finding key in key file: %s", error->message);
-- >    return;
-- >  }
-- >else if (val == NULL)
-- >  {
-- >    // Fall back to a default value.
-- >    val = g_strdup ("default-value");
-- >  }
-- 
-- 
-- Here is an example of creating and saving a key file:
-- 
-- 
-- === /c code/
-- >g_autoptr(GKeyFile) key_file = g_key_file_new ();
-- >const gchar *val = …;
-- >g_autoptr(GError) error = NULL;
-- >
-- >g_key_file_set_string (key_file, "Group Name", "SomeKey", val);
-- >
-- >// Save as a file.
-- >if (!g_key_file_save_to_file (key_file, "key-file.ini", &error))
-- >  {
-- >    g_warning ("Error saving key file: %s", error->message);
-- >    return;
-- >  }
-- >
-- >// Or store to a GBytes for use elsewhere.
-- >gsize data_len;
-- >g_autofree guint8 *data = (guint8 *) g_key_file_to_data (key_file, &data_len, &error);
-- >if (data == NULL)
-- >  {
-- >    g_warning ("Error saving key file: %s", error->message);
-- >    return;
-- >  }
-- >g_autoptr(GBytes) bytes = g_bytes_new_take (g_steal_pointer (&data), data_len);
-- 

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.KeyFile
    ( 

-- * Exported types
    KeyFile(..)                             ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [hasGroup]("GI.GLib.Structs.KeyFile#g:method:hasGroup"), [loadFromBytes]("GI.GLib.Structs.KeyFile#g:method:loadFromBytes"), [loadFromData]("GI.GLib.Structs.KeyFile#g:method:loadFromData"), [loadFromDataDirs]("GI.GLib.Structs.KeyFile#g:method:loadFromDataDirs"), [loadFromDirs]("GI.GLib.Structs.KeyFile#g:method:loadFromDirs"), [loadFromFile]("GI.GLib.Structs.KeyFile#g:method:loadFromFile"), [removeComment]("GI.GLib.Structs.KeyFile#g:method:removeComment"), [removeGroup]("GI.GLib.Structs.KeyFile#g:method:removeGroup"), [removeKey]("GI.GLib.Structs.KeyFile#g:method:removeKey"), [saveToFile]("GI.GLib.Structs.KeyFile#g:method:saveToFile"), [toData]("GI.GLib.Structs.KeyFile#g:method:toData"), [unref]("GI.GLib.Structs.KeyFile#g:method:unref").
-- 
-- ==== Getters
-- [getBoolean]("GI.GLib.Structs.KeyFile#g:method:getBoolean"), [getBooleanList]("GI.GLib.Structs.KeyFile#g:method:getBooleanList"), [getComment]("GI.GLib.Structs.KeyFile#g:method:getComment"), [getDouble]("GI.GLib.Structs.KeyFile#g:method:getDouble"), [getDoubleList]("GI.GLib.Structs.KeyFile#g:method:getDoubleList"), [getGroups]("GI.GLib.Structs.KeyFile#g:method:getGroups"), [getInt64]("GI.GLib.Structs.KeyFile#g:method:getInt64"), [getInteger]("GI.GLib.Structs.KeyFile#g:method:getInteger"), [getIntegerList]("GI.GLib.Structs.KeyFile#g:method:getIntegerList"), [getKeys]("GI.GLib.Structs.KeyFile#g:method:getKeys"), [getLocaleForKey]("GI.GLib.Structs.KeyFile#g:method:getLocaleForKey"), [getLocaleString]("GI.GLib.Structs.KeyFile#g:method:getLocaleString"), [getLocaleStringList]("GI.GLib.Structs.KeyFile#g:method:getLocaleStringList"), [getStartGroup]("GI.GLib.Structs.KeyFile#g:method:getStartGroup"), [getString]("GI.GLib.Structs.KeyFile#g:method:getString"), [getStringList]("GI.GLib.Structs.KeyFile#g:method:getStringList"), [getUint64]("GI.GLib.Structs.KeyFile#g:method:getUint64"), [getValue]("GI.GLib.Structs.KeyFile#g:method:getValue").
-- 
-- ==== Setters
-- [setBoolean]("GI.GLib.Structs.KeyFile#g:method:setBoolean"), [setBooleanList]("GI.GLib.Structs.KeyFile#g:method:setBooleanList"), [setComment]("GI.GLib.Structs.KeyFile#g:method:setComment"), [setDouble]("GI.GLib.Structs.KeyFile#g:method:setDouble"), [setDoubleList]("GI.GLib.Structs.KeyFile#g:method:setDoubleList"), [setInt64]("GI.GLib.Structs.KeyFile#g:method:setInt64"), [setInteger]("GI.GLib.Structs.KeyFile#g:method:setInteger"), [setIntegerList]("GI.GLib.Structs.KeyFile#g:method:setIntegerList"), [setListSeparator]("GI.GLib.Structs.KeyFile#g:method:setListSeparator"), [setLocaleString]("GI.GLib.Structs.KeyFile#g:method:setLocaleString"), [setLocaleStringList]("GI.GLib.Structs.KeyFile#g:method:setLocaleStringList"), [setString]("GI.GLib.Structs.KeyFile#g:method:setString"), [setStringList]("GI.GLib.Structs.KeyFile#g:method:setStringList"), [setUint64]("GI.GLib.Structs.KeyFile#g:method:setUint64"), [setValue]("GI.GLib.Structs.KeyFile#g:method:setValue").

#if defined(ENABLE_OVERLOADING)
    ResolveKeyFileMethod                    ,
#endif

-- ** errorQuark #method:errorQuark#

    keyFileErrorQuark                       ,


-- ** getBoolean #method:getBoolean#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetBooleanMethodInfo             ,
#endif
    keyFileGetBoolean                       ,


-- ** getBooleanList #method:getBooleanList#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetBooleanListMethodInfo         ,
#endif
    keyFileGetBooleanList                   ,


-- ** getComment #method:getComment#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetCommentMethodInfo             ,
#endif
    keyFileGetComment                       ,


-- ** getDouble #method:getDouble#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetDoubleMethodInfo              ,
#endif
    keyFileGetDouble                        ,


-- ** getDoubleList #method:getDoubleList#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetDoubleListMethodInfo          ,
#endif
    keyFileGetDoubleList                    ,


-- ** getGroups #method:getGroups#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetGroupsMethodInfo              ,
#endif
    keyFileGetGroups                        ,


-- ** getInt64 #method:getInt64#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetInt64MethodInfo               ,
#endif
    keyFileGetInt64                         ,


-- ** getInteger #method:getInteger#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetIntegerMethodInfo             ,
#endif
    keyFileGetInteger                       ,


-- ** getIntegerList #method:getIntegerList#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetIntegerListMethodInfo         ,
#endif
    keyFileGetIntegerList                   ,


-- ** getKeys #method:getKeys#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetKeysMethodInfo                ,
#endif
    keyFileGetKeys                          ,


-- ** getLocaleForKey #method:getLocaleForKey#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetLocaleForKeyMethodInfo        ,
#endif
    keyFileGetLocaleForKey                  ,


-- ** getLocaleString #method:getLocaleString#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetLocaleStringMethodInfo        ,
#endif
    keyFileGetLocaleString                  ,


-- ** getLocaleStringList #method:getLocaleStringList#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetLocaleStringListMethodInfo    ,
#endif
    keyFileGetLocaleStringList              ,


-- ** getStartGroup #method:getStartGroup#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetStartGroupMethodInfo          ,
#endif
    keyFileGetStartGroup                    ,


-- ** getString #method:getString#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetStringMethodInfo              ,
#endif
    keyFileGetString                        ,


-- ** getStringList #method:getStringList#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetStringListMethodInfo          ,
#endif
    keyFileGetStringList                    ,


-- ** getUint64 #method:getUint64#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetUint64MethodInfo              ,
#endif
    keyFileGetUint64                        ,


-- ** getValue #method:getValue#

#if defined(ENABLE_OVERLOADING)
    KeyFileGetValueMethodInfo               ,
#endif
    keyFileGetValue                         ,


-- ** hasGroup #method:hasGroup#

#if defined(ENABLE_OVERLOADING)
    KeyFileHasGroupMethodInfo               ,
#endif
    keyFileHasGroup                         ,


-- ** loadFromBytes #method:loadFromBytes#

#if defined(ENABLE_OVERLOADING)
    KeyFileLoadFromBytesMethodInfo          ,
#endif
    keyFileLoadFromBytes                    ,


-- ** loadFromData #method:loadFromData#

#if defined(ENABLE_OVERLOADING)
    KeyFileLoadFromDataMethodInfo           ,
#endif
    keyFileLoadFromData                     ,


-- ** loadFromDataDirs #method:loadFromDataDirs#

#if defined(ENABLE_OVERLOADING)
    KeyFileLoadFromDataDirsMethodInfo       ,
#endif
    keyFileLoadFromDataDirs                 ,


-- ** loadFromDirs #method:loadFromDirs#

#if defined(ENABLE_OVERLOADING)
    KeyFileLoadFromDirsMethodInfo           ,
#endif
    keyFileLoadFromDirs                     ,


-- ** loadFromFile #method:loadFromFile#

#if defined(ENABLE_OVERLOADING)
    KeyFileLoadFromFileMethodInfo           ,
#endif
    keyFileLoadFromFile                     ,


-- ** new #method:new#

    keyFileNew                              ,


-- ** removeComment #method:removeComment#

#if defined(ENABLE_OVERLOADING)
    KeyFileRemoveCommentMethodInfo          ,
#endif
    keyFileRemoveComment                    ,


-- ** removeGroup #method:removeGroup#

#if defined(ENABLE_OVERLOADING)
    KeyFileRemoveGroupMethodInfo            ,
#endif
    keyFileRemoveGroup                      ,


-- ** removeKey #method:removeKey#

#if defined(ENABLE_OVERLOADING)
    KeyFileRemoveKeyMethodInfo              ,
#endif
    keyFileRemoveKey                        ,


-- ** saveToFile #method:saveToFile#

#if defined(ENABLE_OVERLOADING)
    KeyFileSaveToFileMethodInfo             ,
#endif
    keyFileSaveToFile                       ,


-- ** setBoolean #method:setBoolean#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetBooleanMethodInfo             ,
#endif
    keyFileSetBoolean                       ,


-- ** setBooleanList #method:setBooleanList#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetBooleanListMethodInfo         ,
#endif
    keyFileSetBooleanList                   ,


-- ** setComment #method:setComment#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetCommentMethodInfo             ,
#endif
    keyFileSetComment                       ,


-- ** setDouble #method:setDouble#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetDoubleMethodInfo              ,
#endif
    keyFileSetDouble                        ,


-- ** setDoubleList #method:setDoubleList#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetDoubleListMethodInfo          ,
#endif
    keyFileSetDoubleList                    ,


-- ** setInt64 #method:setInt64#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetInt64MethodInfo               ,
#endif
    keyFileSetInt64                         ,


-- ** setInteger #method:setInteger#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetIntegerMethodInfo             ,
#endif
    keyFileSetInteger                       ,


-- ** setIntegerList #method:setIntegerList#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetIntegerListMethodInfo         ,
#endif
    keyFileSetIntegerList                   ,


-- ** setListSeparator #method:setListSeparator#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetListSeparatorMethodInfo       ,
#endif
    keyFileSetListSeparator                 ,


-- ** setLocaleString #method:setLocaleString#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetLocaleStringMethodInfo        ,
#endif
    keyFileSetLocaleString                  ,


-- ** setLocaleStringList #method:setLocaleStringList#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetLocaleStringListMethodInfo    ,
#endif
    keyFileSetLocaleStringList              ,


-- ** setString #method:setString#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetStringMethodInfo              ,
#endif
    keyFileSetString                        ,


-- ** setStringList #method:setStringList#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetStringListMethodInfo          ,
#endif
    keyFileSetStringList                    ,


-- ** setUint64 #method:setUint64#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetUint64MethodInfo              ,
#endif
    keyFileSetUint64                        ,


-- ** setValue #method:setValue#

#if defined(ENABLE_OVERLOADING)
    KeyFileSetValueMethodInfo               ,
#endif
    keyFileSetValue                         ,


-- ** toData #method:toData#

#if defined(ENABLE_OVERLOADING)
    KeyFileToDataMethodInfo                 ,
#endif
    keyFileToData                           ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    KeyFileUnrefMethodInfo                  ,
#endif
    keyFileUnref                            ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.GLib.Flags as GLib.Flags
import {-# SOURCE #-} qualified GI.GLib.Structs.Bytes as GLib.Bytes

#else
import {-# SOURCE #-} qualified GI.GLib.Flags as GLib.Flags
import {-# SOURCE #-} qualified GI.GLib.Structs.Bytes as GLib.Bytes

#endif

-- | Memory-managed wrapper type.
newtype KeyFile = KeyFile (SP.ManagedPtr KeyFile)
    deriving (Eq)

instance SP.ManagedPtrNewtype KeyFile where
    toManagedPtr (KeyFile p) = p

foreign import ccall "g_key_file_get_type" c_g_key_file_get_type :: 
    IO GType

type instance O.ParentTypes KeyFile = '[]
instance O.HasParentTypes KeyFile

instance B.Types.TypedObject KeyFile where
    glibType = c_g_key_file_get_type

instance B.Types.GBoxed KeyFile

-- | Convert 'KeyFile' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe KeyFile) where
    gvalueGType_ = c_g_key_file_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr KeyFile)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr KeyFile)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed KeyFile ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList KeyFile
type instance O.AttributeList KeyFile = KeyFileAttributeList
type KeyFileAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method KeyFile::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "KeyFile" })
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_new" g_key_file_new :: 
    IO (Ptr KeyFile)

-- | Creates a new empty [struct/@gLib@/.KeyFile] object.
-- 
-- Use 'GI.GLib.Structs.KeyFile.keyFileLoadFromFile',
-- 'GI.GLib.Structs.KeyFile.keyFileLoadFromData', 'GI.GLib.Structs.KeyFile.keyFileLoadFromDirs' or
-- 'GI.GLib.Structs.KeyFile.keyFileLoadFromDataDirs' to
-- read an existing key file.
-- 
-- /Since: 2.6/
keyFileNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m KeyFile
    -- ^ __Returns:__ an empty [struct/@gLib@/.KeyFile].
keyFileNew  = liftIO $ do
    result <- g_key_file_new
    checkUnexpectedReturnNULL "keyFileNew" result
    result' <- (wrapBoxed KeyFile) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method KeyFile::get_boolean
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_boolean" g_key_file_get_boolean :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Returns the value associated with /@key@/ under /@groupName@/ as a
-- boolean.
-- 
-- If /@key@/ cannot be found then [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. Likewise, if the value associated with /@key@/ cannot be interpreted
-- as a boolean then [error/@gLib@/.KeyFileError.INVALID_VALUE] is returned.
-- 
-- /Since: 2.6/
keyFileGetBoolean ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetBoolean keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    onException (do
        _ <- propagateGError $ g_key_file_get_boolean keyFile' groupName' key'
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        return ()
     ) (do
        freeMem groupName'
        freeMem key'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetBooleanMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod KeyFileGetBooleanMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetBoolean

instance O.OverloadedMethodInfo KeyFileGetBooleanMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetBoolean",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetBoolean"
        })


#endif

-- method KeyFile::get_boolean_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of booleans returned"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of booleans returned"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 3 (TBasicType TBoolean))
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_boolean_list" g_key_file_get_boolean_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr CInt)

-- | Returns the values associated with /@key@/ under /@groupName@/ as
-- booleans.
-- 
-- If /@key@/ cannot be found then [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. Likewise, if the values associated with /@key@/ cannot be interpreted
-- as booleans then [error/@gLib@/.KeyFileError.INVALID_VALUE] is returned.
-- 
-- /Since: 2.6/
keyFileGetBooleanList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m [Bool]
    -- ^ __Returns:__ 
    --    the values associated with the key as a list of booleans, or @NULL@ if the
    --    key was not found or could not be parsed. The returned list of booleans
    --    should be freed with 'GI.GLib.Functions.free' when no longer needed. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetBooleanList keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_key_file_get_boolean_list keyFile' groupName' key' length_
        length_' <- peek length_
        checkUnexpectedReturnNULL "keyFileGetBooleanList" result
        result' <- (unpackMapStorableArrayWithLength (/= 0) length_') result
        freeMem result
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        freeMem length_
        return result'
     ) (do
        freeMem groupName'
        freeMem key'
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetBooleanListMethodInfo
instance (signature ~ (T.Text -> T.Text -> m [Bool]), MonadIO m) => O.OverloadedMethod KeyFileGetBooleanListMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetBooleanList

instance O.OverloadedMethodInfo KeyFileGetBooleanListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetBooleanList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetBooleanList"
        })


#endif

-- method KeyFile::get_comment
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a group name, or `NULL` to get a top-level comment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key, or `NULL` to get a group comment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_comment" g_key_file_get_comment :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Retrieves a comment above /@key@/ from /@groupName@/.
-- 
-- If /@key@/ is @NULL@ then /@comment@/ will be read from above
-- /@groupName@/. If both /@key@/ and /@groupName@/ are @NULL@, then
-- /@comment@/ will be read from above the first group in the file.
-- 
-- Note that the returned string does not include the @#@ comment markers,
-- but does include any whitespace after them (on each line). It includes
-- the line breaks between lines, but does not include the final line break.
-- 
-- /Since: 2.6/
keyFileGetComment ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> Maybe (T.Text)
    -- ^ /@groupName@/: a group name, or @NULL@ to get a top-level comment
    -> Maybe (T.Text)
    -- ^ /@key@/: a key, or @NULL@ to get a group comment
    -> m T.Text
    -- ^ __Returns:__ a comment that should be freed with 'GI.GLib.Functions.free' /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetComment keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    maybeGroupName <- case groupName of
        Nothing -> return FP.nullPtr
        Just jGroupName -> do
            jGroupName' <- textToCString jGroupName
            return jGroupName'
    maybeKey <- case key of
        Nothing -> return FP.nullPtr
        Just jKey -> do
            jKey' <- textToCString jKey
            return jKey'
    onException (do
        result <- propagateGError $ g_key_file_get_comment keyFile' maybeGroupName maybeKey
        checkUnexpectedReturnNULL "keyFileGetComment" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr keyFile
        freeMem maybeGroupName
        freeMem maybeKey
        return result'
     ) (do
        freeMem maybeGroupName
        freeMem maybeKey
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetCommentMethodInfo
instance (signature ~ (Maybe (T.Text) -> Maybe (T.Text) -> m T.Text), MonadIO m) => O.OverloadedMethod KeyFileGetCommentMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetComment

instance O.OverloadedMethodInfo KeyFileGetCommentMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetComment",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetComment"
        })


#endif

-- method KeyFile::get_double
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TDouble)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_double" g_key_file_get_double :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CDouble

-- | Returns the value associated with /@key@/ under /@groupName@/ as a double.
-- 
-- If /@key@/ cannot be found then [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. Likewise, if the value associated with /@key@/ cannot be interpreted
-- as a double then [error/@gLib@/.KeyFileError.INVALID_VALUE] is returned.
-- 
-- /Since: 2.12/
keyFileGetDouble ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m Double
    -- ^ __Returns:__ the value associated with the key as a double, or
    --     @0.0@ if the key was not found or could not be parsed. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetDouble keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    onException (do
        result <- propagateGError $ g_key_file_get_double keyFile' groupName' key'
        let result' = realToFrac result
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        return result'
     ) (do
        freeMem groupName'
        freeMem key'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetDoubleMethodInfo
instance (signature ~ (T.Text -> T.Text -> m Double), MonadIO m) => O.OverloadedMethod KeyFileGetDoubleMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetDouble

instance O.OverloadedMethodInfo KeyFileGetDoubleMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetDouble",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetDouble"
        })


#endif

-- method KeyFile::get_double_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of doubles returned"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of doubles returned"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 3 (TBasicType TDouble))
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_double_list" g_key_file_get_double_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr CDouble)

-- | Returns the values associated with /@key@/ under /@groupName@/ as
-- doubles.
-- 
-- If /@key@/ cannot be found then [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. Likewise, if the values associated with /@key@/ cannot be interpreted
-- as doubles then [error/@gLib@/.KeyFileError.INVALID_VALUE] is returned.
-- 
-- /Since: 2.12/
keyFileGetDoubleList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m [Double]
    -- ^ __Returns:__ 
    --     the values associated with the key as a list of doubles, or @NULL@ if the
    --     key was not found or could not be parsed. The returned list of doubles
    --     should be freed with 'GI.GLib.Functions.free' when no longer needed. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetDoubleList keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_key_file_get_double_list keyFile' groupName' key' length_
        length_' <- peek length_
        checkUnexpectedReturnNULL "keyFileGetDoubleList" result
        result' <- (unpackMapStorableArrayWithLength realToFrac length_') result
        freeMem result
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        freeMem length_
        return result'
     ) (do
        freeMem groupName'
        freeMem key'
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetDoubleListMethodInfo
instance (signature ~ (T.Text -> T.Text -> m [Double]), MonadIO m) => O.OverloadedMethod KeyFileGetDoubleListMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetDoubleList

instance O.OverloadedMethodInfo KeyFileGetDoubleListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetDoubleList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetDoubleList"
        })


#endif

-- method KeyFile::get_groups
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the number of returned groups,\n   or `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_get_groups" g_key_file_get_groups :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    IO (Ptr CString)

-- | Returns all groups in the key file loaded with /@keyFile@/.
-- 
-- The array of returned groups will be @NULL@-terminated, so
-- /@length@/ may optionally be @NULL@.
-- 
-- /Since: 2.6/
keyFileGetGroups ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> m (([T.Text], FCT.CSize))
    -- ^ __Returns:__ a newly-allocated
    --    @NULL@-terminated array of strings. Use 'GI.GLib.Functions.strfreev' to free it.
keyFileGetGroups keyFile = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    result <- g_key_file_get_groups keyFile' length_
    checkUnexpectedReturnNULL "keyFileGetGroups" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    length_' <- peek length_
    touchManagedPtr keyFile
    freeMem length_
    return (result', length_')

#if defined(ENABLE_OVERLOADING)
data KeyFileGetGroupsMethodInfo
instance (signature ~ (m (([T.Text], FCT.CSize))), MonadIO m) => O.OverloadedMethod KeyFileGetGroupsMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetGroups

instance O.OverloadedMethodInfo KeyFileGetGroupsMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetGroups",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetGroups"
        })


#endif

-- method KeyFile::get_int64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt64)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_int64" g_key_file_get_int64 :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO Int64

-- | Returns the value associated with /@key@/ under /@groupName@/ as a signed
-- 64-bit integer.
-- 
-- This is similar to 'GI.GLib.Structs.KeyFile.keyFileGetInteger' but can return
-- 64-bit results without truncation.
-- 
-- /Since: 2.26/
keyFileGetInt64 ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m Int64
    -- ^ __Returns:__ the value associated with the key as a signed 64-bit integer, or
    --    @0@ if the key was not found or could not be parsed. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetInt64 keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    onException (do
        result <- propagateGError $ g_key_file_get_int64 keyFile' groupName' key'
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        return result
     ) (do
        freeMem groupName'
        freeMem key'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetInt64MethodInfo
instance (signature ~ (T.Text -> T.Text -> m Int64), MonadIO m) => O.OverloadedMethod KeyFileGetInt64MethodInfo KeyFile signature where
    overloadedMethod = keyFileGetInt64

instance O.OverloadedMethodInfo KeyFileGetInt64MethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetInt64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetInt64"
        })


#endif

-- method KeyFile::get_integer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_integer" g_key_file_get_integer :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO Int32

-- | Returns the value associated with /@key@/ under /@groupName@/ as an
-- integer.
-- 
-- If /@key@/ cannot be found then [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. Likewise, if the value associated with /@key@/ cannot be interpreted
-- as an integer, or is out of range for a @gint@, then
-- [error/@gLib@/.KeyFileError.INVALID_VALUE] is returned.
-- 
-- /Since: 2.6/
keyFileGetInteger ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m Int32
    -- ^ __Returns:__ the value associated with the key as an integer, or
    --     @0@ if the key was not found or could not be parsed. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetInteger keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    onException (do
        result <- propagateGError $ g_key_file_get_integer keyFile' groupName' key'
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        return result
     ) (do
        freeMem groupName'
        freeMem key'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetIntegerMethodInfo
instance (signature ~ (T.Text -> T.Text -> m Int32), MonadIO m) => O.OverloadedMethod KeyFileGetIntegerMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetInteger

instance O.OverloadedMethodInfo KeyFileGetIntegerMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetInteger",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetInteger"
        })


#endif

-- method KeyFile::get_integer_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of integers returned"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of integers returned"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 3 (TBasicType TInt))
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_integer_list" g_key_file_get_integer_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Int32)

-- | Returns the values associated with /@key@/ under /@groupName@/ as
-- integers.
-- 
-- If /@key@/ cannot be found then [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. Likewise, if the values associated with /@key@/ cannot be interpreted
-- as integers, or are out of range for @gint@, then
-- [error/@gLib@/.KeyFileError.INVALID_VALUE] is returned.
-- 
-- /Since: 2.6/
keyFileGetIntegerList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m [Int32]
    -- ^ __Returns:__ 
    --     the values associated with the key as a list of integers, or @NULL@ if
    --     the key was not found or could not be parsed. The returned list of
    --     integers should be freed with 'GI.GLib.Functions.free' when no longer needed. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetIntegerList keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_key_file_get_integer_list keyFile' groupName' key' length_
        length_' <- peek length_
        checkUnexpectedReturnNULL "keyFileGetIntegerList" result
        result' <- (unpackStorableArrayWithLength length_') result
        freeMem result
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        freeMem length_
        return result'
     ) (do
        freeMem groupName'
        freeMem key'
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetIntegerListMethodInfo
instance (signature ~ (T.Text -> T.Text -> m [Int32]), MonadIO m) => O.OverloadedMethod KeyFileGetIntegerListMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetIntegerList

instance O.OverloadedMethodInfo KeyFileGetIntegerListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetIntegerList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetIntegerList"
        })


#endif

-- method KeyFile::get_keys
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the number of keys returned,\n   or `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_keys" g_key_file_get_keys :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr CString)

-- | Returns all keys for the group name /@groupName@/.
-- 
-- The array of returned keys will be @NULL@-terminated, so /@length@/ may
-- optionally be @NULL@. If the /@groupName@/ cannot be found,
-- [error/@gLib@/.KeyFileError.GROUP_NOT_FOUND] is returned.
-- 
-- /Since: 2.6/
keyFileGetKeys ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> m (([T.Text], FCT.CSize))
    -- ^ __Returns:__ a newly-allocated
    --    @NULL@-terminated array of strings. Use 'GI.GLib.Functions.strfreev' to free it. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetKeys keyFile groupName = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_key_file_get_keys keyFile' groupName' length_
        checkUnexpectedReturnNULL "keyFileGetKeys" result
        result' <- unpackZeroTerminatedUTF8CArray result
        mapZeroTerminatedCArray freeMem result
        freeMem result
        length_' <- peek length_
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem length_
        return (result', length_')
     ) (do
        freeMem groupName'
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetKeysMethodInfo
instance (signature ~ (T.Text -> m (([T.Text], FCT.CSize))), MonadIO m) => O.OverloadedMethod KeyFileGetKeysMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetKeys

instance O.OverloadedMethodInfo KeyFileGetKeysMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetKeys",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetKeys"
        })


#endif

-- method KeyFile::get_locale_for_key
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "locale"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a locale identifier or `NULL` to use the current locale"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_get_locale_for_key" g_key_file_get_locale_for_key :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- locale : TBasicType TUTF8
    IO CString

-- | Returns the actual locale which the result of
-- 'GI.GLib.Structs.KeyFile.keyFileGetLocaleString' or
-- 'GI.GLib.Structs.KeyFile.keyFileGetLocaleStringList' came from.
-- 
-- If calling 'GI.GLib.Structs.KeyFile.keyFileGetLocaleString' or
-- 'GI.GLib.Structs.KeyFile.keyFileGetLocaleStringList' with exactly the same /@keyFile@/,
-- /@groupName@/, /@key@/ and /@locale@/, the result of those functions will
-- have originally been tagged with the locale that is the result of
-- this function.
-- 
-- /Since: 2.56/
keyFileGetLocaleForKey ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> Maybe (T.Text)
    -- ^ /@locale@/: a locale identifier or @NULL@ to use the current locale
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the locale from the file, or @NULL@ if the key was not
    --   found or the entry in the file was was untranslated
keyFileGetLocaleForKey keyFile groupName key locale = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    maybeLocale <- case locale of
        Nothing -> return FP.nullPtr
        Just jLocale -> do
            jLocale' <- textToCString jLocale
            return jLocale'
    result <- g_key_file_get_locale_for_key keyFile' groupName' key' maybeLocale
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    freeMem maybeLocale
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data KeyFileGetLocaleForKeyMethodInfo
instance (signature ~ (T.Text -> T.Text -> Maybe (T.Text) -> m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod KeyFileGetLocaleForKeyMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetLocaleForKey

instance O.OverloadedMethodInfo KeyFileGetLocaleForKeyMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetLocaleForKey",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetLocaleForKey"
        })


#endif

-- method KeyFile::get_locale_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "locale"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a locale identifier or `NULL` to use the current locale"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_locale_string" g_key_file_get_locale_string :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- locale : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Returns the value associated with /@key@/ under /@groupName@/
-- translated in the given /@locale@/ if available.
-- 
-- If /@locale@/ is @C@ then the untranslated value is returned (since GLib 2.84).
-- 
-- If /@locale@/ is @NULL@ then the current locale is assumed.
-- 
-- If /@locale@/ is to be non-@NULL@, or if the current locale will change over
-- the lifetime of the [struct/@gLib@/.KeyFile], it must be loaded with
-- [flags/@gLib@/.KeyFileFlags.KEEP_TRANSLATIONS] in order to load strings for all
-- locales.
-- 
-- If /@key@/ cannot be found then [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. If the value associated
-- with /@key@/ cannot be interpreted or no suitable translation can
-- be found then the untranslated value is returned.
-- 
-- /Since: 2.6/
keyFileGetLocaleString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> Maybe (T.Text)
    -- ^ /@locale@/: a locale identifier or @NULL@ to use the current locale
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string or @NULL@ if the specified
    --   key cannot be found. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetLocaleString keyFile groupName key locale = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    maybeLocale <- case locale of
        Nothing -> return FP.nullPtr
        Just jLocale -> do
            jLocale' <- textToCString jLocale
            return jLocale'
    onException (do
        result <- propagateGError $ g_key_file_get_locale_string keyFile' groupName' key' maybeLocale
        checkUnexpectedReturnNULL "keyFileGetLocaleString" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        freeMem maybeLocale
        return result'
     ) (do
        freeMem groupName'
        freeMem key'
        freeMem maybeLocale
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetLocaleStringMethodInfo
instance (signature ~ (T.Text -> T.Text -> Maybe (T.Text) -> m T.Text), MonadIO m) => O.OverloadedMethod KeyFileGetLocaleStringMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetLocaleString

instance O.OverloadedMethodInfo KeyFileGetLocaleStringMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetLocaleString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetLocaleString"
        })


#endif

-- method KeyFile::get_locale_string_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "locale"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a locale identifier or `NULL` to use the current locale"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the number of returned strings\n   or `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) 4 (TBasicType TUTF8))
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_locale_string_list" g_key_file_get_locale_string_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- locale : TBasicType TUTF8
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr CString)

-- | Returns the values associated with /@key@/ under /@groupName@/
-- translated in the given /@locale@/ if available.
-- 
-- If /@locale@/ is @C@ then the untranslated value is returned (since GLib 2.84).
-- 
-- If /@locale@/ is @NULL@ then the current locale is assumed.
-- 
-- If /@locale@/ is to be non-@NULL@, or if the current locale will change over
-- the lifetime of the [struct/@gLib@/.KeyFile], it must be loaded with
-- [flags/@gLib@/.KeyFileFlags.KEEP_TRANSLATIONS] in order to load strings for all
-- locales.
-- 
-- If /@key@/ cannot be found then [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. If the values associated
-- with /@key@/ cannot be interpreted or no suitable translations
-- can be found then the untranslated values are returned. The
-- returned array is @NULL@-terminated, so /@length@/ may optionally
-- be @NULL@.
-- 
-- /Since: 2.6/
keyFileGetLocaleStringList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> Maybe (T.Text)
    -- ^ /@locale@/: a locale identifier or @NULL@ to use the current locale
    -> m (([T.Text], FCT.CSize))
    -- ^ __Returns:__ 
    --    a newly allocated @NULL@-terminated string array or @NULL@ if the key
    --    isn’t found. The string array should be freed with 'GI.GLib.Functions.strfreev'. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetLocaleStringList keyFile groupName key locale = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    maybeLocale <- case locale of
        Nothing -> return FP.nullPtr
        Just jLocale -> do
            jLocale' <- textToCString jLocale
            return jLocale'
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_key_file_get_locale_string_list keyFile' groupName' key' maybeLocale length_
        checkUnexpectedReturnNULL "keyFileGetLocaleStringList" result
        result' <- unpackZeroTerminatedUTF8CArray result
        mapZeroTerminatedCArray freeMem result
        freeMem result
        length_' <- peek length_
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        freeMem maybeLocale
        freeMem length_
        return (result', length_')
     ) (do
        freeMem groupName'
        freeMem key'
        freeMem maybeLocale
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetLocaleStringListMethodInfo
instance (signature ~ (T.Text -> T.Text -> Maybe (T.Text) -> m (([T.Text], FCT.CSize))), MonadIO m) => O.OverloadedMethod KeyFileGetLocaleStringListMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetLocaleStringList

instance O.OverloadedMethodInfo KeyFileGetLocaleStringListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetLocaleStringList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetLocaleStringList"
        })


#endif

-- method KeyFile::get_start_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_get_start_group" g_key_file_get_start_group :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    IO CString

-- | Returns the name of the start group of the file.
-- 
-- /Since: 2.6/
keyFileGetStartGroup ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> m (Maybe T.Text)
    -- ^ __Returns:__ The start group of the key file.
keyFileGetStartGroup keyFile = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    result <- g_key_file_get_start_group keyFile'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr keyFile
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data KeyFileGetStartGroupMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod KeyFileGetStartGroupMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetStartGroup

instance O.OverloadedMethodInfo KeyFileGetStartGroupMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetStartGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetStartGroup"
        })


#endif

-- method KeyFile::get_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_string" g_key_file_get_string :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Returns the string value associated with /@key@/ under /@groupName@/.
-- 
-- Unlike 'GI.GLib.Structs.KeyFile.keyFileGetValue', this function handles escape
-- sequences like @\\s@.
-- 
-- If the key cannot be found, [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. If the /@groupName@/ cannot be found,
-- [error/@gLib@/.KeyFileError.GROUP_NOT_FOUND] is returned.
-- 
-- /Since: 2.6/
keyFileGetString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string or @NULL@ if the specified
    --   key cannot be found. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetString keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    onException (do
        result <- propagateGError $ g_key_file_get_string keyFile' groupName' key'
        checkUnexpectedReturnNULL "keyFileGetString" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        return result'
     ) (do
        freeMem groupName'
        freeMem key'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetStringMethodInfo
instance (signature ~ (T.Text -> T.Text -> m T.Text), MonadIO m) => O.OverloadedMethod KeyFileGetStringMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetString

instance O.OverloadedMethodInfo KeyFileGetStringMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetString"
        })


#endif

-- method KeyFile::get_string_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the number of returned\n   strings, or `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) 3 (TBasicType TUTF8))
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_string_list" g_key_file_get_string_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr CString)

-- | Returns the values associated with /@key@/ under /@groupName@/.
-- 
-- If the key cannot be found, [error/@gLib@/.KeyFileError.KEY_NOT_FOUND] is
-- returned. If the /@groupName@/ cannot be found,
-- [error/@gLib@/.KeyFileError.GROUP_NOT_FOUND] is returned.
-- 
-- /Since: 2.6/
keyFileGetStringList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m (([T.Text], FCT.CSize))
    -- ^ __Returns:__ 
    --  a @NULL@-terminated string array or @NULL@ if the specified
    --  key cannot be found. The array should be freed with 'GI.GLib.Functions.strfreev'. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetStringList keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_key_file_get_string_list keyFile' groupName' key' length_
        checkUnexpectedReturnNULL "keyFileGetStringList" result
        result' <- unpackZeroTerminatedUTF8CArray result
        mapZeroTerminatedCArray freeMem result
        freeMem result
        length_' <- peek length_
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        freeMem length_
        return (result', length_')
     ) (do
        freeMem groupName'
        freeMem key'
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetStringListMethodInfo
instance (signature ~ (T.Text -> T.Text -> m (([T.Text], FCT.CSize))), MonadIO m) => O.OverloadedMethod KeyFileGetStringListMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetStringList

instance O.OverloadedMethodInfo KeyFileGetStringListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetStringList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetStringList"
        })


#endif

-- method KeyFile::get_uint64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt64)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_uint64" g_key_file_get_uint64 :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO Word64

-- | Returns the value associated with /@key@/ under /@groupName@/ as an unsigned
-- 64-bit integer.
-- 
-- This is similar to 'GI.GLib.Structs.KeyFile.keyFileGetInteger' but can return
-- large positive results without truncation.
-- 
-- /Since: 2.26/
keyFileGetUint64 ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m Word64
    -- ^ __Returns:__ the value associated with the key as an unsigned 64-bit integer,
    --    or @0@ if the key was not found or could not be parsed. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetUint64 keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    onException (do
        result <- propagateGError $ g_key_file_get_uint64 keyFile' groupName' key'
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        return result
     ) (do
        freeMem groupName'
        freeMem key'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetUint64MethodInfo
instance (signature ~ (T.Text -> T.Text -> m Word64), MonadIO m) => O.OverloadedMethod KeyFileGetUint64MethodInfo KeyFile signature where
    overloadedMethod = keyFileGetUint64

instance O.OverloadedMethodInfo KeyFileGetUint64MethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetUint64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetUint64"
        })


#endif

-- method KeyFile::get_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_get_value" g_key_file_get_value :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Returns the raw value associated with /@key@/ under /@groupName@/.
-- 
-- Use 'GI.GLib.Structs.KeyFile.keyFileGetString' to retrieve an unescaped UTF-8 string.
-- 
-- If the key cannot be found, [error/@gLib@/.KeyFileError.KEY_NOT_FOUND]
-- is returned.  If the /@groupName@/ cannot be found,
-- [error/@gLib@/.KeyFileError.GROUP_NOT_FOUND] is returned.
-- 
-- /Since: 2.6/
keyFileGetValue ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string or @NULL@ if the specified
    --  key cannot be found. /(Can throw 'Data.GI.Base.GError.GError')/
keyFileGetValue keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    onException (do
        result <- propagateGError $ g_key_file_get_value keyFile' groupName' key'
        checkUnexpectedReturnNULL "keyFileGetValue" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        return result'
     ) (do
        freeMem groupName'
        freeMem key'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileGetValueMethodInfo
instance (signature ~ (T.Text -> T.Text -> m T.Text), MonadIO m) => O.OverloadedMethod KeyFileGetValueMethodInfo KeyFile signature where
    overloadedMethod = keyFileGetValue

instance O.OverloadedMethodInfo KeyFileGetValueMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileGetValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileGetValue"
        })


#endif

-- method KeyFile::has_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_has_group" g_key_file_has_group :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    IO CInt

-- | Looks whether the key file has the group /@groupName@/.
-- 
-- /Since: 2.6/
keyFileHasGroup ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> m Bool
    -- ^ __Returns:__ true if /@groupName@/ is a part of /@keyFile@/, false otherwise.
keyFileHasGroup keyFile groupName = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    result <- g_key_file_has_group keyFile' groupName'
    let result' = (/= 0) result
    touchManagedPtr keyFile
    freeMem groupName'
    return result'

#if defined(ENABLE_OVERLOADING)
data KeyFileHasGroupMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m) => O.OverloadedMethod KeyFileHasGroupMethodInfo KeyFile signature where
    overloadedMethod = keyFileHasGroup

instance O.OverloadedMethodInfo KeyFileHasGroupMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileHasGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileHasGroup"
        })


#endif

-- method KeyFile::load_from_bytes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an empty [struct@GLib.KeyFile] struct"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFileFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags from [flags@GLib.KeyFileFlags]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_load_from_bytes" g_key_file_load_from_bytes :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    Ptr GLib.Bytes.Bytes ->                 -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "KeyFileFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Loads a key file from the data in /@bytes@/ into an empty [struct/@gLib@/.KeyFile]
-- structure.
-- 
-- If the object cannot be created then a [error/@gLib@/.KeyFileError] is returned.
-- 
-- /Since: 2.50/
keyFileLoadFromBytes ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: an empty [struct/@gLib@/.KeyFile] struct
    -> GLib.Bytes.Bytes
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> [GLib.Flags.KeyFileFlags]
    -- ^ /@flags@/: flags from [flags/@gLib@/.KeyFileFlags]
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileLoadFromBytes keyFile bytes flags = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    bytes' <- unsafeManagedPtrGetPtr bytes
    let flags' = gflagsToWord flags
    onException (do
        _ <- propagateGError $ g_key_file_load_from_bytes keyFile' bytes' flags'
        touchManagedPtr keyFile
        touchManagedPtr bytes
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileLoadFromBytesMethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> [GLib.Flags.KeyFileFlags] -> m ()), MonadIO m) => O.OverloadedMethod KeyFileLoadFromBytesMethodInfo KeyFile signature where
    overloadedMethod = keyFileLoadFromBytes

instance O.OverloadedMethodInfo KeyFileLoadFromBytesMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileLoadFromBytes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileLoadFromBytes"
        })


#endif

-- method KeyFile::load_from_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an empty key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "key file loaded in memory"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @data in bytes (or `(gsize)-1` if data is nul-terminated)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFileFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags from [flags@GLib.KeyFileFlags]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_load_from_data" g_key_file_load_from_data :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- data : TBasicType TUTF8
    FCT.CSize ->                            -- length : TBasicType TSize
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "KeyFileFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Loads a key file from memory into an empty [struct/@gLib@/.KeyFile] structure.
-- 
-- If the object cannot be created then a [error/@gLib@/.KeyFileError is returned.
-- 
-- /Since: 2.6/
keyFileLoadFromData ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: an empty key file
    -> T.Text
    -- ^ /@data@/: key file loaded in memory
    -> FCT.CSize
    -- ^ /@length@/: the length of /@data@/ in bytes (or @(gsize)-1@ if data is nul-terminated)
    -> [GLib.Flags.KeyFileFlags]
    -- ^ /@flags@/: flags from [flags/@gLib@/.KeyFileFlags]
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileLoadFromData keyFile data_ length_ flags = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    data_' <- textToCString data_
    let flags' = gflagsToWord flags
    onException (do
        _ <- propagateGError $ g_key_file_load_from_data keyFile' data_' length_ flags'
        touchManagedPtr keyFile
        freeMem data_'
        return ()
     ) (do
        freeMem data_'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileLoadFromDataMethodInfo
instance (signature ~ (T.Text -> FCT.CSize -> [GLib.Flags.KeyFileFlags] -> m ()), MonadIO m) => O.OverloadedMethod KeyFileLoadFromDataMethodInfo KeyFile signature where
    overloadedMethod = keyFileLoadFromData

instance O.OverloadedMethodInfo KeyFileLoadFromDataMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileLoadFromData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileLoadFromData"
        })


#endif

-- method KeyFile::load_from_data_dirs
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an empty [struct@GLib.KeyFile] struct"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "file"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a relative path to a filename to open and parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "full_path"
--           , argType = TBasicType TFileName
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for a string\n   containing the full path of the file, or `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFileFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags from [flags@GLib.KeyFileFlags]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_load_from_data_dirs" g_key_file_load_from_data_dirs :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- file : TBasicType TFileName
    Ptr CString ->                          -- full_path : TBasicType TFileName
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "KeyFileFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Looks for a key file named /@file@/ in the paths returned from
-- 'GI.GLib.Functions.getUserDataDir' and 'GI.GLib.Functions.getSystemDataDirs'.
-- 
-- The search algorithm from 'GI.GLib.Structs.KeyFile.keyFileLoadFromDirs' is used. If
-- /@file@/ is found, it’s loaded into /@keyFile@/ and its full path is returned in
-- /@fullPath@/.
-- 
-- If the file could not be loaded then either a [error/@gLib@/.FileError] or
-- [error/@gLib@/.KeyFileError] is returned.
-- 
-- /Since: 2.6/
keyFileLoadFromDataDirs ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: an empty [struct/@gLib@/.KeyFile] struct
    -> [Char]
    -- ^ /@file@/: a relative path to a filename to open and parse
    -> [GLib.Flags.KeyFileFlags]
    -- ^ /@flags@/: flags from [flags/@gLib@/.KeyFileFlags]
    -> m ([Char])
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileLoadFromDataDirs keyFile file flags = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    file' <- stringToCString file
    fullPath <- callocMem :: IO (Ptr CString)
    let flags' = gflagsToWord flags
    onException (do
        _ <- propagateGError $ g_key_file_load_from_data_dirs keyFile' file' fullPath flags'
        fullPath' <- peek fullPath
        fullPath'' <- cstringToString fullPath'
        freeMem fullPath'
        touchManagedPtr keyFile
        freeMem file'
        freeMem fullPath
        return fullPath''
     ) (do
        freeMem file'
        freeMem fullPath
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileLoadFromDataDirsMethodInfo
instance (signature ~ ([Char] -> [GLib.Flags.KeyFileFlags] -> m ([Char])), MonadIO m) => O.OverloadedMethod KeyFileLoadFromDataDirsMethodInfo KeyFile signature where
    overloadedMethod = keyFileLoadFromDataDirs

instance O.OverloadedMethodInfo KeyFileLoadFromDataDirsMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileLoadFromDataDirs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileLoadFromDataDirs"
        })


#endif

-- method KeyFile::load_from_dirs
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an empty [struct@GLib.KeyFile] struct"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "file"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a relative path to a filename to open and parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "search_dirs"
--           , argType = TCArray True (-1) (-1) (TBasicType TFileName)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "`NULL`-terminated\n   array of directories to search"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "full_path"
--           , argType = TBasicType TFileName
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for a string\n   containing the full path of the file, or `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFileFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags from [flags@GLib.KeyFileFlags]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_load_from_dirs" g_key_file_load_from_dirs :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- file : TBasicType TFileName
    Ptr CString ->                          -- search_dirs : TCArray True (-1) (-1) (TBasicType TFileName)
    Ptr CString ->                          -- full_path : TBasicType TFileName
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "KeyFileFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Looks for a key file named /@file@/ in the paths specified in /@searchDirs@/,
-- loads the file into /@keyFile@/ and returns the file’s full path in /@fullPath@/.
-- 
-- /@searchDirs@/ are checked in the order listed in the array, with the highest
-- priority directory listed first. Within each directory, /@file@/ is looked for.
-- If it’s not found, @-@ characters in /@file@/ are progressively replaced with
-- directory separators to search subdirectories of the search directory. If the
-- file has not been found after all @-@ characters have been replaced, the next
-- search directory in /@searchDirs@/ is checked.
-- 
-- If the file could not be found in any of the /@searchDirs@/,
-- [error/@gLib@/.KeyFileError.NOT_FOUND] is returned. If
-- the file is found but the OS returns an error when opening or reading the
-- file, a [error/@gLib@/.FileError] is returned. If there is a problem parsing the
-- file, a [error/@gLib@/.KeyFileError] is returned.
-- 
-- /Since: 2.14/
keyFileLoadFromDirs ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: an empty [struct/@gLib@/.KeyFile] struct
    -> [Char]
    -- ^ /@file@/: a relative path to a filename to open and parse
    -> [[Char]]
    -- ^ /@searchDirs@/: @NULL@-terminated
    --    array of directories to search
    -> [GLib.Flags.KeyFileFlags]
    -- ^ /@flags@/: flags from [flags/@gLib@/.KeyFileFlags]
    -> m ([Char])
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileLoadFromDirs keyFile file searchDirs flags = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    file' <- stringToCString file
    searchDirs' <- packZeroTerminatedFileNameArray searchDirs
    fullPath <- callocMem :: IO (Ptr CString)
    let flags' = gflagsToWord flags
    onException (do
        _ <- propagateGError $ g_key_file_load_from_dirs keyFile' file' searchDirs' fullPath flags'
        fullPath' <- peek fullPath
        fullPath'' <- cstringToString fullPath'
        freeMem fullPath'
        touchManagedPtr keyFile
        freeMem file'
        mapZeroTerminatedCArray freeMem searchDirs'
        freeMem searchDirs'
        freeMem fullPath
        return fullPath''
     ) (do
        freeMem file'
        mapZeroTerminatedCArray freeMem searchDirs'
        freeMem searchDirs'
        freeMem fullPath
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileLoadFromDirsMethodInfo
instance (signature ~ ([Char] -> [[Char]] -> [GLib.Flags.KeyFileFlags] -> m ([Char])), MonadIO m) => O.OverloadedMethod KeyFileLoadFromDirsMethodInfo KeyFile signature where
    overloadedMethod = keyFileLoadFromDirs

instance O.OverloadedMethodInfo KeyFileLoadFromDirsMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileLoadFromDirs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileLoadFromDirs"
        })


#endif

-- method KeyFile::load_from_file
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an empty key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "file"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the path of a filename to load, in the GLib filename encoding"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFileFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags from [flags@GLib.KeyFileFlags]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_load_from_file" g_key_file_load_from_file :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- file : TBasicType TFileName
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "KeyFileFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Loads a key file into an empty [struct/@gLib@/.KeyFile] structure.
-- 
-- If the OS returns an error when opening or reading the file, a
-- [error/@gLib@/.FileError] is returned. If there is a problem parsing the file,
-- a [error/@gLib@/.KeyFileError] is returned.
-- 
-- This function will never return a [error/@gLib@/.KeyFileError.NOT_FOUND]
-- error. If the /@file@/ is not found, [error/@gLib@/.FileError.NOENT] is returned.
-- 
-- /Since: 2.6/
keyFileLoadFromFile ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: an empty key file
    -> [Char]
    -- ^ /@file@/: the path of a filename to load, in the GLib filename encoding
    -> [GLib.Flags.KeyFileFlags]
    -- ^ /@flags@/: flags from [flags/@gLib@/.KeyFileFlags]
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileLoadFromFile keyFile file flags = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    file' <- stringToCString file
    let flags' = gflagsToWord flags
    onException (do
        _ <- propagateGError $ g_key_file_load_from_file keyFile' file' flags'
        touchManagedPtr keyFile
        freeMem file'
        return ()
     ) (do
        freeMem file'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileLoadFromFileMethodInfo
instance (signature ~ ([Char] -> [GLib.Flags.KeyFileFlags] -> m ()), MonadIO m) => O.OverloadedMethod KeyFileLoadFromFileMethodInfo KeyFile signature where
    overloadedMethod = keyFileLoadFromFile

instance O.OverloadedMethodInfo KeyFileLoadFromFileMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileLoadFromFile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileLoadFromFile"
        })


#endif

-- method KeyFile::remove_comment
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a group name, or `NULL` to get a top-level comment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key, or `NULL` to get a group comment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_remove_comment" g_key_file_remove_comment :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Removes a comment above /@key@/ from /@groupName@/.
-- 
-- If /@key@/ is @NULL@ then /@comment@/ will be removed above /@groupName@/.
-- If both /@key@/ and /@groupName@/ are @NULL@, then /@comment@/ will
-- be removed above the first group in the file.
-- 
-- /Since: 2.6/
keyFileRemoveComment ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> Maybe (T.Text)
    -- ^ /@groupName@/: a group name, or @NULL@ to get a top-level comment
    -> Maybe (T.Text)
    -- ^ /@key@/: a key, or @NULL@ to get a group comment
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileRemoveComment keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    maybeGroupName <- case groupName of
        Nothing -> return FP.nullPtr
        Just jGroupName -> do
            jGroupName' <- textToCString jGroupName
            return jGroupName'
    maybeKey <- case key of
        Nothing -> return FP.nullPtr
        Just jKey -> do
            jKey' <- textToCString jKey
            return jKey'
    onException (do
        _ <- propagateGError $ g_key_file_remove_comment keyFile' maybeGroupName maybeKey
        touchManagedPtr keyFile
        freeMem maybeGroupName
        freeMem maybeKey
        return ()
     ) (do
        freeMem maybeGroupName
        freeMem maybeKey
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileRemoveCommentMethodInfo
instance (signature ~ (Maybe (T.Text) -> Maybe (T.Text) -> m ()), MonadIO m) => O.OverloadedMethod KeyFileRemoveCommentMethodInfo KeyFile signature where
    overloadedMethod = keyFileRemoveComment

instance O.OverloadedMethodInfo KeyFileRemoveCommentMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileRemoveComment",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileRemoveComment"
        })


#endif

-- method KeyFile::remove_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_remove_group" g_key_file_remove_group :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Removes the specified group, /@groupName@/,
-- from the key file.
-- 
-- /Since: 2.6/
keyFileRemoveGroup ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileRemoveGroup keyFile groupName = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    onException (do
        _ <- propagateGError $ g_key_file_remove_group keyFile' groupName'
        touchManagedPtr keyFile
        freeMem groupName'
        return ()
     ) (do
        freeMem groupName'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileRemoveGroupMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m) => O.OverloadedMethod KeyFileRemoveGroupMethodInfo KeyFile signature where
    overloadedMethod = keyFileRemoveGroup

instance O.OverloadedMethodInfo KeyFileRemoveGroupMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileRemoveGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileRemoveGroup"
        })


#endif

-- method KeyFile::remove_key
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key name to remove"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_remove_key" g_key_file_remove_key :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Removes /@key@/ in /@groupName@/ from the key file.
-- 
-- /Since: 2.6/
keyFileRemoveKey ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key name to remove
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileRemoveKey keyFile groupName key = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    onException (do
        _ <- propagateGError $ g_key_file_remove_key keyFile' groupName' key'
        touchManagedPtr keyFile
        freeMem groupName'
        freeMem key'
        return ()
     ) (do
        freeMem groupName'
        freeMem key'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileRemoveKeyMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod KeyFileRemoveKeyMethodInfo KeyFile signature where
    overloadedMethod = keyFileRemoveKey

instance O.OverloadedMethodInfo KeyFileRemoveKeyMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileRemoveKey",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileRemoveKey"
        })


#endif

-- method KeyFile::save_to_file
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "filename"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the file to write to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_save_to_file" g_key_file_save_to_file :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- filename : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Writes the contents of /@keyFile@/ to /@filename@/ using
-- 'GI.GLib.Functions.fileSetContents'.
-- 
-- If you need stricter guarantees about durability of
-- the written file than are provided by 'GI.GLib.Functions.fileSetContents', use
-- 'GI.GLib.Functions.fileSetContentsFull' with the return value of
-- 'GI.GLib.Structs.KeyFile.keyFileToData'.
-- 
-- This function can fail for any of the reasons that
-- 'GI.GLib.Functions.fileSetContents' may fail.
-- 
-- /Since: 2.40/
keyFileSaveToFile ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@filename@/: the name of the file to write to
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileSaveToFile keyFile filename = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    filename' <- textToCString filename
    onException (do
        _ <- propagateGError $ g_key_file_save_to_file keyFile' filename'
        touchManagedPtr keyFile
        freeMem filename'
        return ()
     ) (do
        freeMem filename'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileSaveToFileMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSaveToFileMethodInfo KeyFile signature where
    overloadedMethod = keyFileSaveToFile

instance O.OverloadedMethodInfo KeyFileSaveToFileMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSaveToFile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSaveToFile"
        })


#endif

-- method KeyFile::set_boolean
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "true or false" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_boolean" g_key_file_set_boolean :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CInt ->                                 -- value : TBasicType TBoolean
    IO ()

-- | Associates a new boolean value with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.6/
keyFileSetBoolean ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> Bool
    -- ^ /@value@/: true or false
    -> m ()
keyFileSetBoolean keyFile groupName key value = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    let value' = (P.fromIntegral . P.fromEnum) value
    g_key_file_set_boolean keyFile' groupName' key' value'
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetBooleanMethodInfo
instance (signature ~ (T.Text -> T.Text -> Bool -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetBooleanMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetBoolean

instance O.OverloadedMethodInfo KeyFileSetBooleanMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetBoolean",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetBoolean"
        })


#endif

-- method KeyFile::set_boolean_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "list"
--           , argType = TCArray False (-1) 4 (TBasicType TBoolean)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of boolean values"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "length of @list" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "length of @list" , sinceVersion = Nothing }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_boolean_list" g_key_file_set_boolean_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr CInt ->                             -- list : TCArray False (-1) 4 (TBasicType TBoolean)
    FCT.CSize ->                            -- length : TBasicType TSize
    IO ()

-- | Associates a list of boolean values with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.6/
keyFileSetBooleanList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> [Bool]
    -- ^ /@list@/: an array of boolean values
    -> m ()
keyFileSetBooleanList keyFile groupName key list = liftIO $ do
    let length_ = fromIntegral $ P.length list
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    list' <- (packMapStorableArray (P.fromIntegral . P.fromEnum)) list
    g_key_file_set_boolean_list keyFile' groupName' key' list' length_
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    freeMem list'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetBooleanListMethodInfo
instance (signature ~ (T.Text -> T.Text -> [Bool] -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetBooleanListMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetBooleanList

instance O.OverloadedMethodInfo KeyFileSetBooleanListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetBooleanList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetBooleanList"
        })


#endif

-- method KeyFile::set_comment
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a group name, or `NULL` to write a top-level comment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key, or `NULL` to write a group comment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "comment"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a comment" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_set_comment" g_key_file_set_comment :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- comment : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Places a comment above /@key@/ from /@groupName@/.
-- 
-- If /@key@/ is @NULL@ then /@comment@/ will be written above /@groupName@/.
-- If both /@key@/ and /@groupName@/ are @NULL@, then /@comment@/ will be
-- written above the first group in the file.
-- 
-- Note that this function prepends a @#@ comment marker to
-- each line of /@comment@/.
-- 
-- /Since: 2.6/
keyFileSetComment ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> Maybe (T.Text)
    -- ^ /@groupName@/: a group name, or @NULL@ to write a top-level comment
    -> Maybe (T.Text)
    -- ^ /@key@/: a key, or @NULL@ to write a group comment
    -> T.Text
    -- ^ /@comment@/: a comment
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
keyFileSetComment keyFile groupName key comment = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    maybeGroupName <- case groupName of
        Nothing -> return FP.nullPtr
        Just jGroupName -> do
            jGroupName' <- textToCString jGroupName
            return jGroupName'
    maybeKey <- case key of
        Nothing -> return FP.nullPtr
        Just jKey -> do
            jKey' <- textToCString jKey
            return jKey'
    comment' <- textToCString comment
    onException (do
        _ <- propagateGError $ g_key_file_set_comment keyFile' maybeGroupName maybeKey comment'
        touchManagedPtr keyFile
        freeMem maybeGroupName
        freeMem maybeKey
        freeMem comment'
        return ()
     ) (do
        freeMem maybeGroupName
        freeMem maybeKey
        freeMem comment'
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileSetCommentMethodInfo
instance (signature ~ (Maybe (T.Text) -> Maybe (T.Text) -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetCommentMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetComment

instance O.OverloadedMethodInfo KeyFileSetCommentMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetComment",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetComment"
        })


#endif

-- method KeyFile::set_double
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a double value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_double" g_key_file_set_double :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CDouble ->                              -- value : TBasicType TDouble
    IO ()

-- | Associates a new double value with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.12/
keyFileSetDouble ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> Double
    -- ^ /@value@/: a double value
    -> m ()
keyFileSetDouble keyFile groupName key value = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    let value' = realToFrac value
    g_key_file_set_double keyFile' groupName' key' value'
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetDoubleMethodInfo
instance (signature ~ (T.Text -> T.Text -> Double -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetDoubleMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetDouble

instance O.OverloadedMethodInfo KeyFileSetDoubleMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetDouble",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetDouble"
        })


#endif

-- method KeyFile::set_double_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "list"
--           , argType = TCArray False (-1) 4 (TBasicType TDouble)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of double values"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "number of double values in @list"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "number of double values in @list"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_double_list" g_key_file_set_double_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr CDouble ->                          -- list : TCArray False (-1) 4 (TBasicType TDouble)
    FCT.CSize ->                            -- length : TBasicType TSize
    IO ()

-- | Associates a list of double values with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.12/
keyFileSetDoubleList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> [Double]
    -- ^ /@list@/: an array of double values
    -> m ()
keyFileSetDoubleList keyFile groupName key list = liftIO $ do
    let length_ = fromIntegral $ P.length list
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    list' <- (packMapStorableArray realToFrac) list
    g_key_file_set_double_list keyFile' groupName' key' list' length_
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    freeMem list'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetDoubleListMethodInfo
instance (signature ~ (T.Text -> T.Text -> [Double] -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetDoubleListMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetDoubleList

instance O.OverloadedMethodInfo KeyFileSetDoubleListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetDoubleList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetDoubleList"
        })


#endif

-- method KeyFile::set_int64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an integer value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_int64" g_key_file_set_int64 :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Int64 ->                                -- value : TBasicType TInt64
    IO ()

-- | Associates a new integer value with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.26/
keyFileSetInt64 ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> Int64
    -- ^ /@value@/: an integer value
    -> m ()
keyFileSetInt64 keyFile groupName key value = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    g_key_file_set_int64 keyFile' groupName' key' value
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetInt64MethodInfo
instance (signature ~ (T.Text -> T.Text -> Int64 -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetInt64MethodInfo KeyFile signature where
    overloadedMethod = keyFileSetInt64

instance O.OverloadedMethodInfo KeyFileSetInt64MethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetInt64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetInt64"
        })


#endif

-- method KeyFile::set_integer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an integer value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_integer" g_key_file_set_integer :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Int32 ->                                -- value : TBasicType TInt
    IO ()

-- | Associates a new integer value with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.6/
keyFileSetInteger ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> Int32
    -- ^ /@value@/: an integer value
    -> m ()
keyFileSetInteger keyFile groupName key value = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    g_key_file_set_integer keyFile' groupName' key' value
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetIntegerMethodInfo
instance (signature ~ (T.Text -> T.Text -> Int32 -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetIntegerMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetInteger

instance O.OverloadedMethodInfo KeyFileSetIntegerMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetInteger",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetInteger"
        })


#endif

-- method KeyFile::set_integer_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "list"
--           , argType = TCArray False (-1) 4 (TBasicType TInt)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of integer values"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "number of integer values in @list"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "number of integer values in @list"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_integer_list" g_key_file_set_integer_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr Int32 ->                            -- list : TCArray False (-1) 4 (TBasicType TInt)
    FCT.CSize ->                            -- length : TBasicType TSize
    IO ()

-- | Associates a list of integer values with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.6/
keyFileSetIntegerList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> [Int32]
    -- ^ /@list@/: an array of integer values
    -> m ()
keyFileSetIntegerList keyFile groupName key list = liftIO $ do
    let length_ = fromIntegral $ P.length list
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    list' <- packStorableArray list
    g_key_file_set_integer_list keyFile' groupName' key' list' length_
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    freeMem list'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetIntegerListMethodInfo
instance (signature ~ (T.Text -> T.Text -> [Int32] -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetIntegerListMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetIntegerList

instance O.OverloadedMethodInfo KeyFileSetIntegerListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetIntegerList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetIntegerList"
        })


#endif

-- method KeyFile::set_list_separator
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "separator"
--           , argType = TBasicType TInt8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the separator" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_list_separator" g_key_file_set_list_separator :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    Int8 ->                                 -- separator : TBasicType TInt8
    IO ()

-- | Sets the character which is used to separate values in lists.
-- 
-- Typically @;@ or @,@ are used as separators. The default list separator
-- is @;@.
-- 
-- /Since: 2.6/
keyFileSetListSeparator ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> Int8
    -- ^ /@separator@/: the separator
    -> m ()
keyFileSetListSeparator keyFile separator = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    g_key_file_set_list_separator keyFile' separator
    touchManagedPtr keyFile
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetListSeparatorMethodInfo
instance (signature ~ (Int8 -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetListSeparatorMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetListSeparator

instance O.OverloadedMethodInfo KeyFileSetListSeparatorMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetListSeparator",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetListSeparator"
        })


#endif

-- method KeyFile::set_locale_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "locale"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a locale identifier"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_locale_string" g_key_file_set_locale_string :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- locale : TBasicType TUTF8
    CString ->                              -- string : TBasicType TUTF8
    IO ()

-- | Associates a string value for /@key@/ and /@locale@/ under /@groupName@/.
-- 
-- If the translation for /@key@/ cannot be found then it is created.
-- 
-- If /@locale@/ is @C@ then the untranslated value is set (since GLib 2.84).
-- 
-- /Since: 2.6/
keyFileSetLocaleString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> T.Text
    -- ^ /@locale@/: a locale identifier
    -> T.Text
    -- ^ /@string@/: a string
    -> m ()
keyFileSetLocaleString keyFile groupName key locale string = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    locale' <- textToCString locale
    string' <- textToCString string
    g_key_file_set_locale_string keyFile' groupName' key' locale' string'
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    freeMem locale'
    freeMem string'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetLocaleStringMethodInfo
instance (signature ~ (T.Text -> T.Text -> T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetLocaleStringMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetLocaleString

instance O.OverloadedMethodInfo KeyFileSetLocaleStringMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetLocaleString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetLocaleString"
        })


#endif

-- method KeyFile::set_locale_string_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "locale"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a locale identifier"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "list"
--           , argType = TCArray True (-1) 5 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a `NULL`-terminated array of\n   locale string values"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of @list"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_locale_string_list" g_key_file_set_locale_string_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- locale : TBasicType TUTF8
    Ptr CString ->                          -- list : TCArray True (-1) 5 (TBasicType TUTF8)
    FCT.CSize ->                            -- length : TBasicType TSize
    IO ()

-- | Associates a list of string values for /@key@/ and /@locale@/ under
-- /@groupName@/.
-- 
-- If /@locale@/ is @C@ then the untranslated value is set (since GLib 2.84).
-- 
-- If the translation for /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.6/
keyFileSetLocaleStringList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> T.Text
    -- ^ /@locale@/: a locale identifier
    -> [T.Text]
    -- ^ /@list@/: a @NULL@-terminated array of
    --    locale string values
    -> FCT.CSize
    -- ^ /@length@/: the length of /@list@/
    -> m ()
keyFileSetLocaleStringList keyFile groupName key locale list length_ = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    locale' <- textToCString locale
    list' <- packZeroTerminatedUTF8CArray list
    g_key_file_set_locale_string_list keyFile' groupName' key' locale' list' length_
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    freeMem locale'
    mapZeroTerminatedCArray freeMem list'
    freeMem list'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetLocaleStringListMethodInfo
instance (signature ~ (T.Text -> T.Text -> T.Text -> [T.Text] -> FCT.CSize -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetLocaleStringListMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetLocaleStringList

instance O.OverloadedMethodInfo KeyFileSetLocaleStringListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetLocaleStringList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetLocaleStringList"
        })


#endif

-- method KeyFile::set_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_string" g_key_file_set_string :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- string : TBasicType TUTF8
    IO ()

-- | Associates a new string value with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- If /@groupName@/ cannot be found then it is created.
-- Unlike 'GI.GLib.Structs.KeyFile.keyFileSetValue', this function handles characters
-- that need escaping, such as newlines.
-- 
-- /Since: 2.6/
keyFileSetString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> T.Text
    -- ^ /@string@/: a string
    -> m ()
keyFileSetString keyFile groupName key string = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    string' <- textToCString string
    g_key_file_set_string keyFile' groupName' key' string'
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    freeMem string'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetStringMethodInfo
instance (signature ~ (T.Text -> T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetStringMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetString

instance O.OverloadedMethodInfo KeyFileSetStringMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetString"
        })


#endif

-- method KeyFile::set_string_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "list"
--           , argType = TCArray True (-1) 4 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array\n   of string values"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "number of string values in @list"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_string_list" g_key_file_set_string_list :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Ptr CString ->                          -- list : TCArray True (-1) 4 (TBasicType TUTF8)
    FCT.CSize ->                            -- length : TBasicType TSize
    IO ()

-- | Associates a list of string values for /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- If /@groupName@/ cannot be found then it is created.
-- 
-- /Since: 2.6/
keyFileSetStringList ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> [T.Text]
    -- ^ /@list@/: an array
    --    of string values
    -> FCT.CSize
    -- ^ /@length@/: number of string values in /@list@/
    -> m ()
keyFileSetStringList keyFile groupName key list length_ = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    list' <- packZeroTerminatedUTF8CArray list
    g_key_file_set_string_list keyFile' groupName' key' list' length_
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    mapZeroTerminatedCArray freeMem list'
    freeMem list'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetStringListMethodInfo
instance (signature ~ (T.Text -> T.Text -> [T.Text] -> FCT.CSize -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetStringListMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetStringList

instance O.OverloadedMethodInfo KeyFileSetStringListMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetStringList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetStringList"
        })


#endif

-- method KeyFile::set_uint64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an integer value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_uint64" g_key_file_set_uint64 :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    Word64 ->                               -- value : TBasicType TUInt64
    IO ()

-- | Associates a new integer value with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created.
-- 
-- /Since: 2.26/
keyFileSetUint64 ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> Word64
    -- ^ /@value@/: an integer value
    -> m ()
keyFileSetUint64 keyFile groupName key value = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    g_key_file_set_uint64 keyFile' groupName' key' value
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetUint64MethodInfo
instance (signature ~ (T.Text -> T.Text -> Word64 -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetUint64MethodInfo KeyFile signature where
    overloadedMethod = keyFileSetUint64

instance O.OverloadedMethodInfo KeyFileSetUint64MethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetUint64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetUint64"
        })


#endif

-- method KeyFile::set_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a group name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_set_value" g_key_file_set_value :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    CString ->                              -- group_name : TBasicType TUTF8
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- value : TBasicType TUTF8
    IO ()

-- | Associates a new value with /@key@/ under /@groupName@/.
-- 
-- If /@key@/ cannot be found then it is created. If /@groupName@/ cannot
-- be found then it is created. To set an UTF-8 string which may contain
-- characters that need escaping (such as newlines or spaces), use
-- 'GI.GLib.Structs.KeyFile.keyFileSetString'.
-- 
-- /Since: 2.6/
keyFileSetValue ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> T.Text
    -- ^ /@groupName@/: a group name
    -> T.Text
    -- ^ /@key@/: a key
    -> T.Text
    -- ^ /@value@/: a string
    -> m ()
keyFileSetValue keyFile groupName key value = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    groupName' <- textToCString groupName
    key' <- textToCString key
    value' <- textToCString value
    g_key_file_set_value keyFile' groupName' key' value'
    touchManagedPtr keyFile
    freeMem groupName'
    freeMem key'
    freeMem value'
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileSetValueMethodInfo
instance (signature ~ (T.Text -> T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod KeyFileSetValueMethodInfo KeyFile signature where
    overloadedMethod = keyFileSetValue

instance O.OverloadedMethodInfo KeyFileSetValueMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileSetValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileSetValue"
        })


#endif

-- method KeyFile::to_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the length of the\n  returned string, or `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_key_file_to_data" g_key_file_to_data :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Outputs /@keyFile@/ as a string.
-- 
-- Note that this function never reports an error.
-- 
-- /Since: 2.6/
keyFileToData ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> m ((T.Text, FCT.CSize))
    -- ^ __Returns:__ a newly allocated string holding the contents of the key file /(Can throw 'Data.GI.Base.GError.GError')/
keyFileToData keyFile = liftIO $ do
    keyFile' <- unsafeManagedPtrGetPtr keyFile
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_key_file_to_data keyFile' length_
        checkUnexpectedReturnNULL "keyFileToData" result
        result' <- cstringToText result
        freeMem result
        length_' <- peek length_
        touchManagedPtr keyFile
        freeMem length_
        return (result', length_')
     ) (do
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data KeyFileToDataMethodInfo
instance (signature ~ (m ((T.Text, FCT.CSize))), MonadIO m) => O.OverloadedMethod KeyFileToDataMethodInfo KeyFile signature where
    overloadedMethod = keyFileToData

instance O.OverloadedMethodInfo KeyFileToDataMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileToData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileToData"
        })


#endif

-- method KeyFile::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "key_file"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "KeyFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a key file" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_unref" g_key_file_unref :: 
    Ptr KeyFile ->                          -- key_file : TInterface (Name {namespace = "GLib", name = "KeyFile"})
    IO ()

-- | Decreases the reference count of /@keyFile@/ by 1.
-- 
-- If the reference count reaches zero, frees the key file and all its allocated
-- memory.
-- 
-- /Since: 2.32/
keyFileUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    KeyFile
    -- ^ /@keyFile@/: a key file
    -> m ()
keyFileUnref keyFile = liftIO $ do
    keyFile' <- B.ManagedPtr.disownBoxed keyFile
    g_key_file_unref keyFile'
    touchManagedPtr keyFile
    return ()

#if defined(ENABLE_OVERLOADING)
data KeyFileUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod KeyFileUnrefMethodInfo KeyFile signature where
    overloadedMethod = keyFileUnref

instance O.OverloadedMethodInfo KeyFileUnrefMethodInfo KeyFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.KeyFile.keyFileUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-KeyFile.html#v:keyFileUnref"
        })


#endif

-- method KeyFile::error_quark
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TUInt32)
-- throws : False
-- Skip return : False

foreign import ccall "g_key_file_error_quark" g_key_file_error_quark :: 
    IO Word32

-- | /No description available in the introspection data./
keyFileErrorQuark ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Word32
keyFileErrorQuark  = liftIO $ do
    result <- g_key_file_error_quark
    return result

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveKeyFileMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveKeyFileMethod "hasGroup" o = KeyFileHasGroupMethodInfo
    ResolveKeyFileMethod "loadFromBytes" o = KeyFileLoadFromBytesMethodInfo
    ResolveKeyFileMethod "loadFromData" o = KeyFileLoadFromDataMethodInfo
    ResolveKeyFileMethod "loadFromDataDirs" o = KeyFileLoadFromDataDirsMethodInfo
    ResolveKeyFileMethod "loadFromDirs" o = KeyFileLoadFromDirsMethodInfo
    ResolveKeyFileMethod "loadFromFile" o = KeyFileLoadFromFileMethodInfo
    ResolveKeyFileMethod "removeComment" o = KeyFileRemoveCommentMethodInfo
    ResolveKeyFileMethod "removeGroup" o = KeyFileRemoveGroupMethodInfo
    ResolveKeyFileMethod "removeKey" o = KeyFileRemoveKeyMethodInfo
    ResolveKeyFileMethod "saveToFile" o = KeyFileSaveToFileMethodInfo
    ResolveKeyFileMethod "toData" o = KeyFileToDataMethodInfo
    ResolveKeyFileMethod "unref" o = KeyFileUnrefMethodInfo
    ResolveKeyFileMethod "getBoolean" o = KeyFileGetBooleanMethodInfo
    ResolveKeyFileMethod "getBooleanList" o = KeyFileGetBooleanListMethodInfo
    ResolveKeyFileMethod "getComment" o = KeyFileGetCommentMethodInfo
    ResolveKeyFileMethod "getDouble" o = KeyFileGetDoubleMethodInfo
    ResolveKeyFileMethod "getDoubleList" o = KeyFileGetDoubleListMethodInfo
    ResolveKeyFileMethod "getGroups" o = KeyFileGetGroupsMethodInfo
    ResolveKeyFileMethod "getInt64" o = KeyFileGetInt64MethodInfo
    ResolveKeyFileMethod "getInteger" o = KeyFileGetIntegerMethodInfo
    ResolveKeyFileMethod "getIntegerList" o = KeyFileGetIntegerListMethodInfo
    ResolveKeyFileMethod "getKeys" o = KeyFileGetKeysMethodInfo
    ResolveKeyFileMethod "getLocaleForKey" o = KeyFileGetLocaleForKeyMethodInfo
    ResolveKeyFileMethod "getLocaleString" o = KeyFileGetLocaleStringMethodInfo
    ResolveKeyFileMethod "getLocaleStringList" o = KeyFileGetLocaleStringListMethodInfo
    ResolveKeyFileMethod "getStartGroup" o = KeyFileGetStartGroupMethodInfo
    ResolveKeyFileMethod "getString" o = KeyFileGetStringMethodInfo
    ResolveKeyFileMethod "getStringList" o = KeyFileGetStringListMethodInfo
    ResolveKeyFileMethod "getUint64" o = KeyFileGetUint64MethodInfo
    ResolveKeyFileMethod "getValue" o = KeyFileGetValueMethodInfo
    ResolveKeyFileMethod "setBoolean" o = KeyFileSetBooleanMethodInfo
    ResolveKeyFileMethod "setBooleanList" o = KeyFileSetBooleanListMethodInfo
    ResolveKeyFileMethod "setComment" o = KeyFileSetCommentMethodInfo
    ResolveKeyFileMethod "setDouble" o = KeyFileSetDoubleMethodInfo
    ResolveKeyFileMethod "setDoubleList" o = KeyFileSetDoubleListMethodInfo
    ResolveKeyFileMethod "setInt64" o = KeyFileSetInt64MethodInfo
    ResolveKeyFileMethod "setInteger" o = KeyFileSetIntegerMethodInfo
    ResolveKeyFileMethod "setIntegerList" o = KeyFileSetIntegerListMethodInfo
    ResolveKeyFileMethod "setListSeparator" o = KeyFileSetListSeparatorMethodInfo
    ResolveKeyFileMethod "setLocaleString" o = KeyFileSetLocaleStringMethodInfo
    ResolveKeyFileMethod "setLocaleStringList" o = KeyFileSetLocaleStringListMethodInfo
    ResolveKeyFileMethod "setString" o = KeyFileSetStringMethodInfo
    ResolveKeyFileMethod "setStringList" o = KeyFileSetStringListMethodInfo
    ResolveKeyFileMethod "setUint64" o = KeyFileSetUint64MethodInfo
    ResolveKeyFileMethod "setValue" o = KeyFileSetValueMethodInfo
    ResolveKeyFileMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveKeyFileMethod t KeyFile, O.OverloadedMethod info KeyFile p) => OL.IsLabel t (KeyFile -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveKeyFileMethod t KeyFile, O.OverloadedMethod info KeyFile p, R.HasField t KeyFile p) => R.HasField t KeyFile p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveKeyFileMethod t KeyFile, O.OverloadedMethodInfo info KeyFile) => OL.IsLabel t (O.MethodProxy info KeyFile) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


