{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A simple reference counted data type representing an immutable sequence of
-- zero or more bytes from an unspecified origin.
-- 
-- The purpose of a @GBytes@ is to keep the memory region that it holds
-- alive for as long as anyone holds a reference to the bytes.  When
-- the last reference count is dropped, the memory is released. Multiple
-- unrelated callers can use byte data in the @GBytes@ without coordinating
-- their activities, resting assured that the byte data will not change or
-- move while they hold a reference.
-- 
-- A @GBytes@ can come from many different origins that may have
-- different procedures for freeing the memory region.  Examples are
-- memory from 'GI.GLib.Functions.malloc', from memory slices, from a
-- [struct/@gLib@/.MappedFile] or memory from other allocators.
-- 
-- @GBytes@ work well as keys in [struct/@gLib@/.HashTable]. Use
-- 'GI.GLib.Structs.Bytes.bytesEqual' and 'GI.GLib.Structs.Bytes.bytesHash' as parameters to
-- [func/@gLib@/.HashTable.new] or [func/@gLib@/.HashTable.new_full].
-- @GBytes@ can also be used as keys in a [struct/@gLib@/.Tree] by passing the
-- 'GI.GLib.Structs.Bytes.bytesCompare' function to t'GI.GLib.Structs.Tree.Tree'.@/new/@().
-- 
-- The data pointed to by this bytes must not be modified. For a mutable
-- array of bytes see [struct/@gLib@/.ByteArray]. Use
-- 'GI.GLib.Structs.Bytes.bytesUnrefToArray' to create a mutable array for a @GBytes@
-- sequence. To create an immutable @GBytes@ from a mutable
-- [struct/@gLib@/.ByteArray], use the [func/@gLib@/.ByteArray.free_to_bytes]
-- function.
-- 
-- /Since: 2.32/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.Bytes
    ( 

-- * Exported types
    Bytes(..)                               ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [compare]("GI.GLib.Structs.Bytes#g:method:compare"), [equal]("GI.GLib.Structs.Bytes#g:method:equal"), [hash]("GI.GLib.Structs.Bytes#g:method:hash"), [newFromBytes]("GI.GLib.Structs.Bytes#g:method:newFromBytes"), [ref]("GI.GLib.Structs.Bytes#g:method:ref"), [unref]("GI.GLib.Structs.Bytes#g:method:unref"), [unrefToArray]("GI.GLib.Structs.Bytes#g:method:unrefToArray"), [unrefToData]("GI.GLib.Structs.Bytes#g:method:unrefToData").
-- 
-- ==== Getters
-- [getData]("GI.GLib.Structs.Bytes#g:method:getData"), [getRegion]("GI.GLib.Structs.Bytes#g:method:getRegion"), [getSize]("GI.GLib.Structs.Bytes#g:method:getSize").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveBytesMethod                      ,
#endif

-- ** compare #method:compare#

#if defined(ENABLE_OVERLOADING)
    BytesCompareMethodInfo                  ,
#endif
    bytesCompare                            ,


-- ** equal #method:equal#

#if defined(ENABLE_OVERLOADING)
    BytesEqualMethodInfo                    ,
#endif
    bytesEqual                              ,


-- ** getData #method:getData#

#if defined(ENABLE_OVERLOADING)
    BytesGetDataMethodInfo                  ,
#endif
    bytesGetData                            ,


-- ** getRegion #method:getRegion#

#if defined(ENABLE_OVERLOADING)
    BytesGetRegionMethodInfo                ,
#endif
    bytesGetRegion                          ,


-- ** getSize #method:getSize#

#if defined(ENABLE_OVERLOADING)
    BytesGetSizeMethodInfo                  ,
#endif
    bytesGetSize                            ,


-- ** hash #method:hash#

#if defined(ENABLE_OVERLOADING)
    BytesHashMethodInfo                     ,
#endif
    bytesHash                               ,


-- ** new #method:new#

    bytesNew                                ,


-- ** newFromBytes #method:newFromBytes#

#if defined(ENABLE_OVERLOADING)
    BytesNewFromBytesMethodInfo             ,
#endif
    bytesNewFromBytes                       ,


-- ** newTake #method:newTake#

    bytesNewTake                            ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    BytesRefMethodInfo                      ,
#endif
    bytesRef                                ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    BytesUnrefMethodInfo                    ,
#endif
    bytesUnref                              ,


-- ** unrefToArray #method:unrefToArray#

#if defined(ENABLE_OVERLOADING)
    BytesUnrefToArrayMethodInfo             ,
#endif
    bytesUnrefToArray                       ,


-- ** unrefToData #method:unrefToData#

#if defined(ENABLE_OVERLOADING)
    BytesUnrefToDataMethodInfo              ,
#endif
    bytesUnrefToData                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- | Memory-managed wrapper type.
newtype Bytes = Bytes (SP.ManagedPtr Bytes)
    deriving (Eq)

instance SP.ManagedPtrNewtype Bytes where
    toManagedPtr (Bytes p) = p

foreign import ccall "g_bytes_get_type" c_g_bytes_get_type :: 
    IO GType

type instance O.ParentTypes Bytes = '[]
instance O.HasParentTypes Bytes

instance B.Types.TypedObject Bytes where
    glibType = c_g_bytes_get_type

instance B.Types.GBoxed Bytes

-- | Convert 'Bytes' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Bytes) where
    gvalueGType_ = c_g_bytes_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr Bytes)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr Bytes)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed Bytes ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Bytes
type instance O.AttributeList Bytes = BytesAttributeList
type BytesAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method Bytes::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "data"
--           , argType = TCArray False (-1) 1 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "\n  the data to be used for the bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "size"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the size of @data" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "size"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the size of @data" , sinceVersion = Nothing }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_new" g_bytes_new :: 
    Ptr Word8 ->                            -- data : TCArray False (-1) 1 (TBasicType TUInt8)
    FCT.CSize ->                            -- size : TBasicType TSize
    IO (Ptr Bytes)

-- | Creates a new [struct/@gLib@/.Bytes] from /@data@/.
-- 
-- /@data@/ is copied. If /@size@/ is 0, /@data@/ may be @NULL@.
-- 
-- As an optimization, 'GI.GLib.Structs.Bytes.bytesNew' may avoid an extra allocation by
-- copying the data within the resulting bytes structure if sufficiently small
-- (since GLib 2.84).
-- 
-- /Since: 2.32/
bytesNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (ByteString)
    -- ^ /@data@/: 
    --   the data to be used for the bytes
    -> m Bytes
    -- ^ __Returns:__ a new [struct/@gLib@/.Bytes]
bytesNew data_ = liftIO $ do
    let size = case data_ of
            Nothing -> 0
            Just jData_ -> fromIntegral $ B.length jData_
    maybeData_ <- case data_ of
        Nothing -> return FP.nullPtr
        Just jData_ -> do
            jData_' <- packByteString jData_
            return jData_'
    result <- g_bytes_new maybeData_ size
    checkUnexpectedReturnNULL "bytesNew" result
    result' <- (wrapBoxed Bytes) result
    freeMem maybeData_
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Bytes::new_take
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "data"
--           , argType = TCArray False (-1) 1 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "\n  the data to be used for the bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "size"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the size of @data" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "size"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the size of @data" , sinceVersion = Nothing }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_new_take" g_bytes_new_take :: 
    Ptr Word8 ->                            -- data : TCArray False (-1) 1 (TBasicType TUInt8)
    FCT.CSize ->                            -- size : TBasicType TSize
    IO (Ptr Bytes)

-- | Creates a new [struct/@gLib@/.Bytes] from /@data@/.
-- 
-- After this call, /@data@/ belongs to the @GBytes@ and may no longer be
-- modified by the caller. The memory of /@data@/ has to be dynamically
-- allocated and will eventually be freed with 'GI.GLib.Functions.free'.
-- 
-- For creating @GBytes@ with memory from other allocators, see
-- t'GI.GLib.Structs.Bytes.Bytes'.@/new_with_free_func/@().
-- 
-- /@data@/ may be @NULL@ if /@size@/ is 0.
-- 
-- /Since: 2.32/
bytesNewTake ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (ByteString)
    -- ^ /@data@/: 
    --   the data to be used for the bytes
    -> m Bytes
    -- ^ __Returns:__ a new [struct/@gLib@/.Bytes]
bytesNewTake data_ = liftIO $ do
    let size = case data_ of
            Nothing -> 0
            Just jData_ -> fromIntegral $ B.length jData_
    maybeData_ <- case data_ of
        Nothing -> return FP.nullPtr
        Just jData_ -> do
            jData_' <- packByteString jData_
            return jData_'
    result <- g_bytes_new_take maybeData_ size
    checkUnexpectedReturnNULL "bytesNewTake" result
    result' <- (wrapBoxed Bytes) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Bytes::compare
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes1"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bytes2"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a pointer to a [struct@GLib.Bytes] to compare with @bytes1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_compare" g_bytes_compare :: 
    Ptr Bytes ->                            -- bytes1 : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr Bytes ->                            -- bytes2 : TInterface (Name {namespace = "GLib", name = "Bytes"})
    IO Int32

-- | Compares the two [struct/@gLib@/.Bytes] values.
-- 
-- This function can be used to sort @GBytes@ instances in lexicographical
-- order.
-- 
-- If /@bytes1@/ and /@bytes2@/ have different length but the shorter one is a
-- prefix of the longer one then the shorter one is considered to be less than
-- the longer one. Otherwise the first byte where both differ is used for
-- comparison. If /@bytes1@/ has a smaller value at that position it is
-- considered less, otherwise greater than /@bytes2@/.
-- 
-- /Since: 2.32/
bytesCompare ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes1@/: a pointer to a [struct/@gLib@/.Bytes]
    -> Bytes
    -- ^ /@bytes2@/: a pointer to a [struct/@gLib@/.Bytes] to compare with /@bytes1@/
    -> m Int32
    -- ^ __Returns:__ a negative value if /@bytes1@/ is less than /@bytes2@/, a positive value
    --   if /@bytes1@/ is greater than /@bytes2@/, and zero if /@bytes1@/ is equal to /@bytes2@/
bytesCompare bytes1 bytes2 = liftIO $ do
    bytes1' <- unsafeManagedPtrGetPtr bytes1
    bytes2' <- unsafeManagedPtrGetPtr bytes2
    result <- g_bytes_compare bytes1' bytes2'
    touchManagedPtr bytes1
    touchManagedPtr bytes2
    return result

#if defined(ENABLE_OVERLOADING)
data BytesCompareMethodInfo
instance (signature ~ (Bytes -> m Int32), MonadIO m) => O.OverloadedMethod BytesCompareMethodInfo Bytes signature where
    overloadedMethod = bytesCompare

instance O.OverloadedMethodInfo BytesCompareMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesCompare",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesCompare"
        })


#endif

-- method Bytes::equal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes1"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bytes2"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a pointer to a [struct@GLib.Bytes] to compare with @bytes1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_equal" g_bytes_equal :: 
    Ptr Bytes ->                            -- bytes1 : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr Bytes ->                            -- bytes2 : TInterface (Name {namespace = "GLib", name = "Bytes"})
    IO CInt

-- | Compares the two [struct/@gLib@/.Bytes] values being pointed to and returns
-- @TRUE@ if they are equal.
-- 
-- This function can be passed to [func/@gLib@/.HashTable.new] as the
-- /@keyEqualFunc@/ parameter, when using non-@NULL@ @GBytes@ pointers as keys in
-- a [struct/@gLib@/.HashTable].
-- 
-- /Since: 2.32/
bytesEqual ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes1@/: a pointer to a [struct/@gLib@/.Bytes]
    -> Bytes
    -- ^ /@bytes2@/: a pointer to a [struct/@gLib@/.Bytes] to compare with /@bytes1@/
    -> m Bool
    -- ^ __Returns:__ @TRUE@ if the two keys match.
bytesEqual bytes1 bytes2 = liftIO $ do
    bytes1' <- unsafeManagedPtrGetPtr bytes1
    bytes2' <- unsafeManagedPtrGetPtr bytes2
    result <- g_bytes_equal bytes1' bytes2'
    let result' = (/= 0) result
    touchManagedPtr bytes1
    touchManagedPtr bytes2
    return result'

#if defined(ENABLE_OVERLOADING)
data BytesEqualMethodInfo
instance (signature ~ (Bytes -> m Bool), MonadIO m) => O.OverloadedMethod BytesEqualMethodInfo Bytes signature where
    overloadedMethod = bytesEqual

instance O.OverloadedMethodInfo BytesEqualMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesEqual",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesEqual"
        })


#endif

-- method Bytes::get_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "size"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to return size of byte data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "size"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "location to return size of byte data"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 1 (TBasicType TUInt8))
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_get_data" g_bytes_get_data :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr FCT.CSize ->                        -- size : TBasicType TSize
    IO (Ptr Word8)

-- | Get the byte data in the [struct/@gLib@/.Bytes].
-- 
-- This data should not be modified.
-- 
-- This function will always return the same pointer for a given @GBytes@.
-- 
-- @NULL@ may be returned if /@size@/ is 0. This is not guaranteed, as the @GBytes@
-- may represent an empty string with /@data@/ non-@NULL@ and /@size@/ as 0. @NULL@
-- will not be returned if /@size@/ is non-zero.
-- 
-- /Since: 2.32/
bytesGetData ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> m (Maybe ByteString)
    -- ^ __Returns:__ 
    --   a pointer to the byte data
bytesGetData bytes = liftIO $ do
    bytes' <- unsafeManagedPtrGetPtr bytes
    size <- allocMem :: IO (Ptr FCT.CSize)
    result <- g_bytes_get_data bytes' size
    size' <- peek size
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (unpackByteStringWithLength size') result'
        return result''
    touchManagedPtr bytes
    freeMem size
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data BytesGetDataMethodInfo
instance (signature ~ (m (Maybe ByteString)), MonadIO m) => O.OverloadedMethod BytesGetDataMethodInfo Bytes signature where
    overloadedMethod = bytesGetData

instance O.OverloadedMethodInfo BytesGetDataMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesGetData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesGetData"
        })


#endif

-- method Bytes::get_region
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "element_size"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a non-zero element size"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "an offset to the start of the region within the @bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_elements"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of elements in the region"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_get_region" g_bytes_get_region :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    FCT.CSize ->                            -- element_size : TBasicType TSize
    FCT.CSize ->                            -- offset : TBasicType TSize
    FCT.CSize ->                            -- n_elements : TBasicType TSize
    IO (Ptr ())

-- | Gets a pointer to a region in /@bytes@/.
-- 
-- The region starts at /@offset@/ many bytes from the start of the data
-- and contains /@nElements@/ many elements of /@elementSize@/ size.
-- 
-- /@nElements@/ may be zero, but /@elementSize@/ must always be non-zero.
-- Ideally, /@elementSize@/ is a static constant (eg: @sizeof@ a struct).
-- 
-- This function does careful bounds checking (including checking for
-- arithmetic overflows) and returns a non-@NULL@ pointer if the
-- specified region lies entirely within the /@bytes@/. If the region is
-- in some way out of range, or if an overflow has occurred, then @NULL@
-- is returned.
-- 
-- Note: it is possible to have a valid zero-size region. In this case,
-- the returned pointer will be equal to the base pointer of the data of
-- /@bytes@/, plus /@offset@/.  This will be non-@NULL@ except for the case
-- where /@bytes@/ itself was a zero-sized region.  Since it is unlikely
-- that you will be using this function to check for a zero-sized region
-- in a zero-sized /@bytes@/, @NULL@ effectively always means ‘error’.
-- 
-- /Since: 2.70/
bytesGetRegion ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> FCT.CSize
    -- ^ /@elementSize@/: a non-zero element size
    -> FCT.CSize
    -- ^ /@offset@/: an offset to the start of the region within the /@bytes@/
    -> FCT.CSize
    -- ^ /@nElements@/: the number of elements in the region
    -> m (Ptr ())
    -- ^ __Returns:__ the requested region, or @NULL@ in case of an error
bytesGetRegion bytes elementSize offset nElements = liftIO $ do
    bytes' <- unsafeManagedPtrGetPtr bytes
    result <- g_bytes_get_region bytes' elementSize offset nElements
    touchManagedPtr bytes
    return result

#if defined(ENABLE_OVERLOADING)
data BytesGetRegionMethodInfo
instance (signature ~ (FCT.CSize -> FCT.CSize -> FCT.CSize -> m (Ptr ())), MonadIO m) => O.OverloadedMethod BytesGetRegionMethodInfo Bytes signature where
    overloadedMethod = bytesGetRegion

instance O.OverloadedMethodInfo BytesGetRegionMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesGetRegion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesGetRegion"
        })


#endif

-- method Bytes::get_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSize)
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_get_size" g_bytes_get_size :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    IO FCT.CSize

-- | Get the size of the byte data in the [struct/@gLib@/.Bytes].
-- 
-- This function will always return the same value for a given @GBytes@.
-- 
-- /Since: 2.32/
bytesGetSize ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> m FCT.CSize
    -- ^ __Returns:__ the size
bytesGetSize bytes = liftIO $ do
    bytes' <- unsafeManagedPtrGetPtr bytes
    result <- g_bytes_get_size bytes'
    touchManagedPtr bytes
    return result

#if defined(ENABLE_OVERLOADING)
data BytesGetSizeMethodInfo
instance (signature ~ (m FCT.CSize), MonadIO m) => O.OverloadedMethod BytesGetSizeMethodInfo Bytes signature where
    overloadedMethod = bytesGetSize

instance O.OverloadedMethodInfo BytesGetSizeMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesGetSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesGetSize"
        })


#endif

-- method Bytes::hash
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to a [struct@GLib.Bytes] key"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_hash" g_bytes_hash :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    IO Word32

-- | Creates an integer hash code for the byte data in the [struct/@gLib@/.Bytes].
-- 
-- This function can be passed to [func/@gLib@/.HashTable.new] as the
-- /@keyHashFunc@/ parameter, when using non-@NULL@ @GBytes@ pointers as keys in
-- a [struct/@gLib@/.HashTable].
-- 
-- /Since: 2.32/
bytesHash ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes@/: a pointer to a [struct/@gLib@/.Bytes] key
    -> m Word32
    -- ^ __Returns:__ a hash value corresponding to the key.
bytesHash bytes = liftIO $ do
    bytes' <- unsafeManagedPtrGetPtr bytes
    result <- g_bytes_hash bytes'
    touchManagedPtr bytes
    return result

#if defined(ENABLE_OVERLOADING)
data BytesHashMethodInfo
instance (signature ~ (m Word32), MonadIO m) => O.OverloadedMethod BytesHashMethodInfo Bytes signature where
    overloadedMethod = bytesHash

instance O.OverloadedMethodInfo BytesHashMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesHash",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesHash"
        })


#endif

-- method Bytes::new_from_bytes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "offset which subsection starts at"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "length of subsection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_new_from_bytes" g_bytes_new_from_bytes :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    FCT.CSize ->                            -- offset : TBasicType TSize
    FCT.CSize ->                            -- length : TBasicType TSize
    IO (Ptr Bytes)

-- | Creates a [struct/@gLib@/.Bytes] which is a subsection of another @GBytes@.
-- 
-- The /@offset@/ + /@length@/ may not be longer than the size of /@bytes@/.
-- 
-- A reference to /@bytes@/ will be held by the newly created @GBytes@ until
-- the byte data is no longer needed.
-- 
-- Since 2.56, if /@offset@/ is 0 and /@length@/ matches the size of /@bytes@/, then
-- /@bytes@/ will be returned with the reference count incremented by 1. If /@bytes@/
-- is a slice of another @GBytes@, then the resulting @GBytes@ will reference
-- the same @GBytes@ instead of /@bytes@/. This allows consumers to simplify the
-- usage of @GBytes@ when asynchronously writing to streams.
-- 
-- /Since: 2.32/
bytesNewFromBytes ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> FCT.CSize
    -- ^ /@offset@/: offset which subsection starts at
    -> FCT.CSize
    -- ^ /@length@/: length of subsection
    -> m Bytes
    -- ^ __Returns:__ a new [struct/@gLib@/.Bytes]
bytesNewFromBytes bytes offset length_ = liftIO $ do
    bytes' <- unsafeManagedPtrGetPtr bytes
    result <- g_bytes_new_from_bytes bytes' offset length_
    checkUnexpectedReturnNULL "bytesNewFromBytes" result
    result' <- (wrapBoxed Bytes) result
    touchManagedPtr bytes
    return result'

#if defined(ENABLE_OVERLOADING)
data BytesNewFromBytesMethodInfo
instance (signature ~ (FCT.CSize -> FCT.CSize -> m Bytes), MonadIO m) => O.OverloadedMethod BytesNewFromBytesMethodInfo Bytes signature where
    overloadedMethod = bytesNewFromBytes

instance O.OverloadedMethodInfo BytesNewFromBytesMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesNewFromBytes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesNewFromBytes"
        })


#endif

-- method Bytes::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_ref" g_bytes_ref :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    IO (Ptr Bytes)

-- | Increase the reference count on /@bytes@/.
-- 
-- /Since: 2.32/
bytesRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> m Bytes
    -- ^ __Returns:__ the [struct/@gLib@/.Bytes]
bytesRef bytes = liftIO $ do
    bytes' <- unsafeManagedPtrGetPtr bytes
    result <- g_bytes_ref bytes'
    checkUnexpectedReturnNULL "bytesRef" result
    result' <- (wrapBoxed Bytes) result
    touchManagedPtr bytes
    return result'

#if defined(ENABLE_OVERLOADING)
data BytesRefMethodInfo
instance (signature ~ (m Bytes), MonadIO m) => O.OverloadedMethod BytesRefMethodInfo Bytes signature where
    overloadedMethod = bytesRef

instance O.OverloadedMethodInfo BytesRefMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesRef"
        })


#endif

-- method Bytes::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_unref" g_bytes_unref :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    IO ()

-- | Releases a reference on /@bytes@/.
-- 
-- This may result in the bytes being freed. If /@bytes@/ is @NULL@, it will
-- return immediately.
-- 
-- /Since: 2.32/
bytesUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Bytes)
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> m ()
bytesUnref bytes = liftIO $ do
    maybeBytes <- case bytes of
        Nothing -> return FP.nullPtr
        Just jBytes -> do
            jBytes' <- unsafeManagedPtrGetPtr jBytes
            return jBytes'
    g_bytes_unref maybeBytes
    whenJust bytes touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data BytesUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod BytesUnrefMethodInfo Bytes signature where
    overloadedMethod i = bytesUnref (Just i)

instance O.OverloadedMethodInfo BytesUnrefMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesUnref"
        })


#endif

-- method Bytes::unref_to_array
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just TByteArray
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_unref_to_array" g_bytes_unref_to_array :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    IO (Ptr GByteArray)

-- | Unreferences the bytes, and returns a new mutable [struct/@gLib@/.ByteArray]
-- containing the same byte data.
-- 
-- As an optimization, the byte data is transferred to the array without copying
-- if this was the last reference to /@bytes@/ and /@bytes@/ was created with
-- 'GI.GLib.Structs.Bytes.bytesNew', 'GI.GLib.Structs.Bytes.bytesNewTake' or
-- [func/@gLib@/.ByteArray.free_to_bytes] and the buffer was larger than the size
-- [struct/@gLib@/.Bytes] may internalize within its allocation. In all other cases
-- the data is copied.
-- 
-- Do not use it if /@bytes@/ contains more than @/G_MAXUINT/@
-- bytes. [struct/@gLib@/.ByteArray] stores the length of its data in @guint@,
-- which may be shorter than @gsize@, that /@bytes@/ is using.
-- 
-- /Since: 2.32/
bytesUnrefToArray ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> m ByteString
    -- ^ __Returns:__ a new mutable [struct/@gLib@/.ByteArray] containing
    --   the same byte data
bytesUnrefToArray bytes = liftIO $ do
    bytes' <- B.ManagedPtr.disownBoxed bytes
    result <- g_bytes_unref_to_array bytes'
    checkUnexpectedReturnNULL "bytesUnrefToArray" result
    result' <- unpackGByteArray result
    unrefGByteArray result
    touchManagedPtr bytes
    return result'

#if defined(ENABLE_OVERLOADING)
data BytesUnrefToArrayMethodInfo
instance (signature ~ (m ByteString), MonadIO m) => O.OverloadedMethod BytesUnrefToArrayMethodInfo Bytes signature where
    overloadedMethod = bytesUnrefToArray

instance O.OverloadedMethodInfo BytesUnrefToArrayMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesUnrefToArray",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesUnrefToArray"
        })


#endif

-- method Bytes::unref_to_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Bytes]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "size"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location to place the length of the returned data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "size"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "location to place the length of the returned data"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 1 (TBasicType TUInt8))
-- throws : False
-- Skip return : False

foreign import ccall "g_bytes_unref_to_data" g_bytes_unref_to_data :: 
    Ptr Bytes ->                            -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr FCT.CSize ->                        -- size : TBasicType TSize
    IO (Ptr Word8)

-- | Unreferences the bytes, and returns a pointer the same byte data
-- contents.
-- 
-- As an optimization, the byte data is returned without copying if this was
-- the last reference to /@bytes@/ and /@bytes@/ was created with
-- 'GI.GLib.Structs.Bytes.bytesNew', 'GI.GLib.Structs.Bytes.bytesNewTake' or
-- [func/@gLib@/.ByteArray.free_to_bytes] and the buffer was larger than the size
-- [struct/@gLib@/.Bytes] may internalize within its allocation. In all other cases
-- the data is copied.
-- 
-- /Since: 2.32/
bytesUnrefToData ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Bytes
    -- ^ /@bytes@/: a [struct/@gLib@/.Bytes]
    -> m ByteString
    -- ^ __Returns:__ 
    --   a pointer to the same byte data, which should be freed with 'GI.GLib.Functions.free'
bytesUnrefToData bytes = liftIO $ do
    bytes' <- B.ManagedPtr.disownBoxed bytes
    size <- allocMem :: IO (Ptr FCT.CSize)
    result <- g_bytes_unref_to_data bytes' size
    size' <- peek size
    checkUnexpectedReturnNULL "bytesUnrefToData" result
    result' <- (unpackByteStringWithLength size') result
    freeMem result
    touchManagedPtr bytes
    freeMem size
    return result'

#if defined(ENABLE_OVERLOADING)
data BytesUnrefToDataMethodInfo
instance (signature ~ (m ByteString), MonadIO m) => O.OverloadedMethod BytesUnrefToDataMethodInfo Bytes signature where
    overloadedMethod = bytesUnrefToData

instance O.OverloadedMethodInfo BytesUnrefToDataMethodInfo Bytes where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Bytes.bytesUnrefToData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.29/docs/GI-GLib-Structs-Bytes.html#v:bytesUnrefToData"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveBytesMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveBytesMethod "compare" o = BytesCompareMethodInfo
    ResolveBytesMethod "equal" o = BytesEqualMethodInfo
    ResolveBytesMethod "hash" o = BytesHashMethodInfo
    ResolveBytesMethod "newFromBytes" o = BytesNewFromBytesMethodInfo
    ResolveBytesMethod "ref" o = BytesRefMethodInfo
    ResolveBytesMethod "unref" o = BytesUnrefMethodInfo
    ResolveBytesMethod "unrefToArray" o = BytesUnrefToArrayMethodInfo
    ResolveBytesMethod "unrefToData" o = BytesUnrefToDataMethodInfo
    ResolveBytesMethod "getData" o = BytesGetDataMethodInfo
    ResolveBytesMethod "getRegion" o = BytesGetRegionMethodInfo
    ResolveBytesMethod "getSize" o = BytesGetSizeMethodInfo
    ResolveBytesMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveBytesMethod t Bytes, O.OverloadedMethod info Bytes p) => OL.IsLabel t (Bytes -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveBytesMethod t Bytes, O.OverloadedMethod info Bytes p, R.HasField t Bytes p) => R.HasField t Bytes p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveBytesMethod t Bytes, O.OverloadedMethodInfo info Bytes) => OL.IsLabel t (O.MethodProxy info Bytes) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


