{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GTlsInteraction@ provides a mechanism for the TLS connection and database
-- code to interact with the user. It can be used to ask the user for passwords.
-- 
-- To use a @GTlsInteraction@ with a TLS connection use
-- 'GI.Gio.Objects.TlsConnection.tlsConnectionSetInteraction'.
-- 
-- Callers should instantiate a derived class that implements the various
-- interaction methods to show the required dialogs.
-- 
-- Callers should use the \'invoke\' functions like
-- 'GI.Gio.Objects.TlsInteraction.tlsInteractionInvokeAskPassword' to run interaction methods.
-- These functions make sure that the interaction is invoked in the main loop
-- and not in the current thread, if the current thread is not running the
-- main loop.
-- 
-- Derived classes can choose to implement whichever interactions methods they’d
-- like to support by overriding those virtual methods in their class
-- initialization function. Any interactions not implemented will return
-- @G_TLS_INTERACTION_UNHANDLED@. If a derived class implements an async method,
-- it must also implement the corresponding finish method.
-- 
-- /Since: 2.30/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.TlsInteraction
    ( 

-- * Exported types
    TlsInteraction(..)                      ,
    IsTlsInteraction                        ,
    toTlsInteraction                        ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [askPassword]("GI.Gio.Objects.TlsInteraction#g:method:askPassword"), [askPasswordAsync]("GI.Gio.Objects.TlsInteraction#g:method:askPasswordAsync"), [askPasswordFinish]("GI.Gio.Objects.TlsInteraction#g:method:askPasswordFinish"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [invokeAskPassword]("GI.Gio.Objects.TlsInteraction#g:method:invokeAskPassword"), [invokeRequestCertificate]("GI.Gio.Objects.TlsInteraction#g:method:invokeRequestCertificate"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [requestCertificate]("GI.Gio.Objects.TlsInteraction#g:method:requestCertificate"), [requestCertificateAsync]("GI.Gio.Objects.TlsInteraction#g:method:requestCertificateAsync"), [requestCertificateFinish]("GI.Gio.Objects.TlsInteraction#g:method:requestCertificateFinish"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveTlsInteractionMethod             ,
#endif

-- ** askPassword #method:askPassword#

#if defined(ENABLE_OVERLOADING)
    TlsInteractionAskPasswordMethodInfo     ,
#endif
    tlsInteractionAskPassword               ,


-- ** askPasswordAsync #method:askPasswordAsync#

#if defined(ENABLE_OVERLOADING)
    TlsInteractionAskPasswordAsyncMethodInfo,
#endif
    tlsInteractionAskPasswordAsync          ,


-- ** askPasswordFinish #method:askPasswordFinish#

#if defined(ENABLE_OVERLOADING)
    TlsInteractionAskPasswordFinishMethodInfo,
#endif
    tlsInteractionAskPasswordFinish         ,


-- ** invokeAskPassword #method:invokeAskPassword#

#if defined(ENABLE_OVERLOADING)
    TlsInteractionInvokeAskPasswordMethodInfo,
#endif
    tlsInteractionInvokeAskPassword         ,


-- ** invokeRequestCertificate #method:invokeRequestCertificate#

#if defined(ENABLE_OVERLOADING)
    TlsInteractionInvokeRequestCertificateMethodInfo,
#endif
    tlsInteractionInvokeRequestCertificate  ,


-- ** requestCertificate #method:requestCertificate#

#if defined(ENABLE_OVERLOADING)
    TlsInteractionRequestCertificateMethodInfo,
#endif
    tlsInteractionRequestCertificate        ,


-- ** requestCertificateAsync #method:requestCertificateAsync#

#if defined(ENABLE_OVERLOADING)
    TlsInteractionRequestCertificateAsyncMethodInfo,
#endif
    tlsInteractionRequestCertificateAsync   ,


-- ** requestCertificateFinish #method:requestCertificateFinish#

#if defined(ENABLE_OVERLOADING)
    TlsInteractionRequestCertificateFinishMethodInfo,
#endif
    tlsInteractionRequestCertificateFinish  ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.DateTime as GLib.DateTime
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.SocketConnectable as Gio.SocketConnectable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InetAddress as Gio.InetAddress
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.SocketAddress as Gio.SocketAddress
import {-# SOURCE #-} qualified GI.Gio.Objects.SocketAddressEnumerator as Gio.SocketAddressEnumerator
import {-# SOURCE #-} qualified GI.Gio.Objects.TlsCertificate as Gio.TlsCertificate
import {-# SOURCE #-} qualified GI.Gio.Objects.TlsConnection as Gio.TlsConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.TlsDatabase as Gio.TlsDatabase
import {-# SOURCE #-} qualified GI.Gio.Objects.TlsPassword as Gio.TlsPassword
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.TlsConnection as Gio.TlsConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.TlsPassword as Gio.TlsPassword

#endif

-- | Memory-managed wrapper type.
newtype TlsInteraction = TlsInteraction (SP.ManagedPtr TlsInteraction)
    deriving (Eq)

instance SP.ManagedPtrNewtype TlsInteraction where
    toManagedPtr (TlsInteraction p) = p

foreign import ccall "g_tls_interaction_get_type"
    c_g_tls_interaction_get_type :: IO B.Types.GType

instance B.Types.TypedObject TlsInteraction where
    glibType = c_g_tls_interaction_get_type

instance B.Types.GObject TlsInteraction

-- | Type class for types which can be safely cast to `TlsInteraction`, for instance with `toTlsInteraction`.
class (SP.GObject o, O.IsDescendantOf TlsInteraction o) => IsTlsInteraction o
instance (SP.GObject o, O.IsDescendantOf TlsInteraction o) => IsTlsInteraction o

instance O.HasParentTypes TlsInteraction
type instance O.ParentTypes TlsInteraction = '[GObject.Object.Object]

-- | Cast to `TlsInteraction`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toTlsInteraction :: (MIO.MonadIO m, IsTlsInteraction o) => o -> m TlsInteraction
toTlsInteraction = MIO.liftIO . B.ManagedPtr.unsafeCastTo TlsInteraction

-- | Convert 'TlsInteraction' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe TlsInteraction) where
    gvalueGType_ = c_g_tls_interaction_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr TlsInteraction)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr TlsInteraction)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject TlsInteraction ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveTlsInteractionMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveTlsInteractionMethod "askPassword" o = TlsInteractionAskPasswordMethodInfo
    ResolveTlsInteractionMethod "askPasswordAsync" o = TlsInteractionAskPasswordAsyncMethodInfo
    ResolveTlsInteractionMethod "askPasswordFinish" o = TlsInteractionAskPasswordFinishMethodInfo
    ResolveTlsInteractionMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveTlsInteractionMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveTlsInteractionMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveTlsInteractionMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveTlsInteractionMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveTlsInteractionMethod "invokeAskPassword" o = TlsInteractionInvokeAskPasswordMethodInfo
    ResolveTlsInteractionMethod "invokeRequestCertificate" o = TlsInteractionInvokeRequestCertificateMethodInfo
    ResolveTlsInteractionMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveTlsInteractionMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveTlsInteractionMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveTlsInteractionMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveTlsInteractionMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveTlsInteractionMethod "requestCertificate" o = TlsInteractionRequestCertificateMethodInfo
    ResolveTlsInteractionMethod "requestCertificateAsync" o = TlsInteractionRequestCertificateAsyncMethodInfo
    ResolveTlsInteractionMethod "requestCertificateFinish" o = TlsInteractionRequestCertificateFinishMethodInfo
    ResolveTlsInteractionMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveTlsInteractionMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveTlsInteractionMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveTlsInteractionMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveTlsInteractionMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveTlsInteractionMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveTlsInteractionMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveTlsInteractionMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveTlsInteractionMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveTlsInteractionMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveTlsInteractionMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveTlsInteractionMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveTlsInteractionMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveTlsInteractionMethod t TlsInteraction, O.OverloadedMethod info TlsInteraction p) => OL.IsLabel t (TlsInteraction -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveTlsInteractionMethod t TlsInteraction, O.OverloadedMethod info TlsInteraction p, R.HasField t TlsInteraction p) => R.HasField t TlsInteraction p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveTlsInteractionMethod t TlsInteraction, O.OverloadedMethodInfo info TlsInteraction) => OL.IsLabel t (O.MethodProxy info TlsInteraction) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList TlsInteraction
type instance O.AttributeList TlsInteraction = TlsInteractionAttributeList
type TlsInteractionAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList TlsInteraction = TlsInteractionSignalList
type TlsInteractionSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method TlsInteraction::ask_password
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "interaction"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsInteraction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsInteraction object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "password"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsPassword" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsPassword object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an optional #GCancellable cancellation object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "TlsInteractionResult" })
-- throws : True
-- Skip return : False

foreign import ccall "g_tls_interaction_ask_password" g_tls_interaction_ask_password :: 
    Ptr TlsInteraction ->                   -- interaction : TInterface (Name {namespace = "Gio", name = "TlsInteraction"})
    Ptr Gio.TlsPassword.TlsPassword ->      -- password : TInterface (Name {namespace = "Gio", name = "TlsPassword"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CUInt

-- | Run synchronous interaction to ask the user for a password. In general,
-- 'GI.Gio.Objects.TlsInteraction.tlsInteractionInvokeAskPassword' should be used instead of this
-- function.
-- 
-- Derived subclasses usually implement a password prompt, although they may
-- also choose to provide a password from elsewhere. The /@password@/ value will
-- be filled in and then /@callback@/ will be called. Alternatively the user may
-- abort this password request, which will usually abort the TLS connection.
-- 
-- If the interaction is cancelled by the cancellation object, or by the
-- user then 'GI.Gio.Enums.TlsInteractionResultFailed' will be returned with an error that
-- contains a 'GI.Gio.Enums.IOErrorEnumCancelled' error code. Certain implementations may
-- not support immediate cancellation.
-- 
-- /Since: 2.30/
tlsInteractionAskPassword ::
    (B.CallStack.HasCallStack, MonadIO m, IsTlsInteraction a, Gio.TlsPassword.IsTlsPassword b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@interaction@/: a t'GI.Gio.Objects.TlsInteraction.TlsInteraction' object
    -> b
    -- ^ /@password@/: a t'GI.Gio.Objects.TlsPassword.TlsPassword' object
    -> Maybe (c)
    -- ^ /@cancellable@/: an optional t'GI.Gio.Objects.Cancellable.Cancellable' cancellation object
    -> m Gio.Enums.TlsInteractionResult
    -- ^ __Returns:__ The status of the ask password interaction. /(Can throw 'Data.GI.Base.GError.GError')/
tlsInteractionAskPassword interaction password cancellable = liftIO $ do
    interaction' <- unsafeManagedPtrCastPtr interaction
    password' <- unsafeManagedPtrCastPtr password
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_tls_interaction_ask_password interaction' password' maybeCancellable
        let result' = (toEnum . fromIntegral) result
        touchManagedPtr interaction
        touchManagedPtr password
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data TlsInteractionAskPasswordMethodInfo
instance (signature ~ (b -> Maybe (c) -> m Gio.Enums.TlsInteractionResult), MonadIO m, IsTlsInteraction a, Gio.TlsPassword.IsTlsPassword b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod TlsInteractionAskPasswordMethodInfo a signature where
    overloadedMethod = tlsInteractionAskPassword

instance O.OverloadedMethodInfo TlsInteractionAskPasswordMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.TlsInteraction.tlsInteractionAskPassword",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-TlsInteraction.html#v:tlsInteractionAskPassword"
        })


#endif

-- method TlsInteraction::ask_password_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "interaction"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsInteraction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsInteraction object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "password"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsPassword" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsPassword object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an optional #GCancellable cancellation object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "will be called when the interaction completes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to pass to the @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_tls_interaction_ask_password_async" g_tls_interaction_ask_password_async :: 
    Ptr TlsInteraction ->                   -- interaction : TInterface (Name {namespace = "Gio", name = "TlsInteraction"})
    Ptr Gio.TlsPassword.TlsPassword ->      -- password : TInterface (Name {namespace = "Gio", name = "TlsPassword"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Run asynchronous interaction to ask the user for a password. In general,
-- 'GI.Gio.Objects.TlsInteraction.tlsInteractionInvokeAskPassword' should be used instead of this
-- function.
-- 
-- Derived subclasses usually implement a password prompt, although they may
-- also choose to provide a password from elsewhere. The /@password@/ value will
-- be filled in and then /@callback@/ will be called. Alternatively the user may
-- abort this password request, which will usually abort the TLS connection.
-- 
-- If the interaction is cancelled by the cancellation object, or by the
-- user then 'GI.Gio.Enums.TlsInteractionResultFailed' will be returned with an error that
-- contains a 'GI.Gio.Enums.IOErrorEnumCancelled' error code. Certain implementations may
-- not support immediate cancellation.
-- 
-- Certain implementations may not support immediate cancellation.
-- 
-- /Since: 2.30/
tlsInteractionAskPasswordAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsTlsInteraction a, Gio.TlsPassword.IsTlsPassword b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@interaction@/: a t'GI.Gio.Objects.TlsInteraction.TlsInteraction' object
    -> b
    -- ^ /@password@/: a t'GI.Gio.Objects.TlsPassword.TlsPassword' object
    -> Maybe (c)
    -- ^ /@cancellable@/: an optional t'GI.Gio.Objects.Cancellable.Cancellable' cancellation object
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: will be called when the interaction completes
    -> m ()
tlsInteractionAskPasswordAsync interaction password cancellable callback = liftIO $ do
    interaction' <- unsafeManagedPtrCastPtr interaction
    password' <- unsafeManagedPtrCastPtr password
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_tls_interaction_ask_password_async interaction' password' maybeCancellable maybeCallback userData
    touchManagedPtr interaction
    touchManagedPtr password
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data TlsInteractionAskPasswordAsyncMethodInfo
instance (signature ~ (b -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsTlsInteraction a, Gio.TlsPassword.IsTlsPassword b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod TlsInteractionAskPasswordAsyncMethodInfo a signature where
    overloadedMethod = tlsInteractionAskPasswordAsync

instance O.OverloadedMethodInfo TlsInteractionAskPasswordAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.TlsInteraction.tlsInteractionAskPasswordAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-TlsInteraction.html#v:tlsInteractionAskPasswordAsync"
        })


#endif

-- method TlsInteraction::ask_password_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "interaction"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsInteraction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsInteraction object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the result passed to the callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "TlsInteractionResult" })
-- throws : True
-- Skip return : False

foreign import ccall "g_tls_interaction_ask_password_finish" g_tls_interaction_ask_password_finish :: 
    Ptr TlsInteraction ->                   -- interaction : TInterface (Name {namespace = "Gio", name = "TlsInteraction"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CUInt

-- | Complete an ask password user interaction request. This should be once
-- the 'GI.Gio.Objects.TlsInteraction.tlsInteractionAskPasswordAsync' completion callback is called.
-- 
-- If 'GI.Gio.Enums.TlsInteractionResultHandled' is returned, then the t'GI.Gio.Objects.TlsPassword.TlsPassword' passed
-- to 'GI.Gio.Objects.TlsInteraction.tlsInteractionAskPassword' will have its password filled in.
-- 
-- If the interaction is cancelled by the cancellation object, or by the
-- user then 'GI.Gio.Enums.TlsInteractionResultFailed' will be returned with an error that
-- contains a 'GI.Gio.Enums.IOErrorEnumCancelled' error code.
-- 
-- /Since: 2.30/
tlsInteractionAskPasswordFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsTlsInteraction a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@interaction@/: a t'GI.Gio.Objects.TlsInteraction.TlsInteraction' object
    -> b
    -- ^ /@result@/: the result passed to the callback
    -> m Gio.Enums.TlsInteractionResult
    -- ^ __Returns:__ The status of the ask password interaction. /(Can throw 'Data.GI.Base.GError.GError')/
tlsInteractionAskPasswordFinish interaction result_ = liftIO $ do
    interaction' <- unsafeManagedPtrCastPtr interaction
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_tls_interaction_ask_password_finish interaction' result_'
        let result' = (toEnum . fromIntegral) result
        touchManagedPtr interaction
        touchManagedPtr result_
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data TlsInteractionAskPasswordFinishMethodInfo
instance (signature ~ (b -> m Gio.Enums.TlsInteractionResult), MonadIO m, IsTlsInteraction a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod TlsInteractionAskPasswordFinishMethodInfo a signature where
    overloadedMethod = tlsInteractionAskPasswordFinish

instance O.OverloadedMethodInfo TlsInteractionAskPasswordFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.TlsInteraction.tlsInteractionAskPasswordFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-TlsInteraction.html#v:tlsInteractionAskPasswordFinish"
        })


#endif

-- method TlsInteraction::invoke_ask_password
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "interaction"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsInteraction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsInteraction object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "password"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsPassword" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsPassword object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an optional #GCancellable cancellation object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "TlsInteractionResult" })
-- throws : True
-- Skip return : False

foreign import ccall "g_tls_interaction_invoke_ask_password" g_tls_interaction_invoke_ask_password :: 
    Ptr TlsInteraction ->                   -- interaction : TInterface (Name {namespace = "Gio", name = "TlsInteraction"})
    Ptr Gio.TlsPassword.TlsPassword ->      -- password : TInterface (Name {namespace = "Gio", name = "TlsPassword"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CUInt

-- | Invoke the interaction to ask the user for a password. It invokes this
-- interaction in the main loop, specifically the t'GI.GLib.Structs.MainContext.MainContext' returned by
-- 'GI.GLib.Functions.mainContextGetThreadDefault' when the interaction is created. This
-- is called by called by t'GI.Gio.Objects.TlsConnection.TlsConnection' or t'GI.Gio.Objects.TlsDatabase.TlsDatabase' to ask the user
-- for a password.
-- 
-- Derived subclasses usually implement a password prompt, although they may
-- also choose to provide a password from elsewhere. The /@password@/ value will
-- be filled in and then /@callback@/ will be called. Alternatively the user may
-- abort this password request, which will usually abort the TLS connection.
-- 
-- The implementation can either be a synchronous (eg: modal dialog) or an
-- asynchronous one (eg: modeless dialog). This function will take care of
-- calling which ever one correctly.
-- 
-- If the interaction is cancelled by the cancellation object, or by the
-- user then 'GI.Gio.Enums.TlsInteractionResultFailed' will be returned with an error that
-- contains a 'GI.Gio.Enums.IOErrorEnumCancelled' error code. Certain implementations may
-- not support immediate cancellation.
-- 
-- /Since: 2.30/
tlsInteractionInvokeAskPassword ::
    (B.CallStack.HasCallStack, MonadIO m, IsTlsInteraction a, Gio.TlsPassword.IsTlsPassword b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@interaction@/: a t'GI.Gio.Objects.TlsInteraction.TlsInteraction' object
    -> b
    -- ^ /@password@/: a t'GI.Gio.Objects.TlsPassword.TlsPassword' object
    -> Maybe (c)
    -- ^ /@cancellable@/: an optional t'GI.Gio.Objects.Cancellable.Cancellable' cancellation object
    -> m Gio.Enums.TlsInteractionResult
    -- ^ __Returns:__ The status of the ask password interaction. /(Can throw 'Data.GI.Base.GError.GError')/
tlsInteractionInvokeAskPassword interaction password cancellable = liftIO $ do
    interaction' <- unsafeManagedPtrCastPtr interaction
    password' <- unsafeManagedPtrCastPtr password
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_tls_interaction_invoke_ask_password interaction' password' maybeCancellable
        let result' = (toEnum . fromIntegral) result
        touchManagedPtr interaction
        touchManagedPtr password
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data TlsInteractionInvokeAskPasswordMethodInfo
instance (signature ~ (b -> Maybe (c) -> m Gio.Enums.TlsInteractionResult), MonadIO m, IsTlsInteraction a, Gio.TlsPassword.IsTlsPassword b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod TlsInteractionInvokeAskPasswordMethodInfo a signature where
    overloadedMethod = tlsInteractionInvokeAskPassword

instance O.OverloadedMethodInfo TlsInteractionInvokeAskPasswordMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.TlsInteraction.tlsInteractionInvokeAskPassword",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-TlsInteraction.html#v:tlsInteractionInvokeAskPassword"
        })


#endif

-- method TlsInteraction::invoke_request_certificate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "interaction"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsInteraction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsInteraction object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsConnection object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "TlsCertificateRequestFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags providing more information about the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an optional #GCancellable cancellation object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "TlsInteractionResult" })
-- throws : True
-- Skip return : False

foreign import ccall "g_tls_interaction_invoke_request_certificate" g_tls_interaction_invoke_request_certificate :: 
    Ptr TlsInteraction ->                   -- interaction : TInterface (Name {namespace = "Gio", name = "TlsInteraction"})
    Ptr Gio.TlsConnection.TlsConnection ->  -- connection : TInterface (Name {namespace = "Gio", name = "TlsConnection"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "TlsCertificateRequestFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CUInt

-- | Invoke the interaction to ask the user to choose a certificate to
-- use with the connection. It invokes this interaction in the main
-- loop, specifically the t'GI.GLib.Structs.MainContext.MainContext' returned by
-- 'GI.GLib.Functions.mainContextGetThreadDefault' when the interaction is
-- created. This is called by called by t'GI.Gio.Objects.TlsConnection.TlsConnection' when the peer
-- requests a certificate during the handshake.
-- 
-- Derived subclasses usually implement a certificate selector,
-- although they may also choose to provide a certificate from
-- elsewhere. Alternatively the user may abort this certificate
-- request, which may or may not abort the TLS connection.
-- 
-- The implementation can either be a synchronous (eg: modal dialog) or an
-- asynchronous one (eg: modeless dialog). This function will take care of
-- calling which ever one correctly.
-- 
-- If the interaction is cancelled by the cancellation object, or by the
-- user then 'GI.Gio.Enums.TlsInteractionResultFailed' will be returned with an error that
-- contains a 'GI.Gio.Enums.IOErrorEnumCancelled' error code. Certain implementations may
-- not support immediate cancellation.
-- 
-- /Since: 2.40/
tlsInteractionInvokeRequestCertificate ::
    (B.CallStack.HasCallStack, MonadIO m, IsTlsInteraction a, Gio.TlsConnection.IsTlsConnection b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@interaction@/: a t'GI.Gio.Objects.TlsInteraction.TlsInteraction' object
    -> b
    -- ^ /@connection@/: a t'GI.Gio.Objects.TlsConnection.TlsConnection' object
    -> Gio.Enums.TlsCertificateRequestFlags
    -- ^ /@flags@/: flags providing more information about the request
    -> Maybe (c)
    -- ^ /@cancellable@/: an optional t'GI.Gio.Objects.Cancellable.Cancellable' cancellation object
    -> m Gio.Enums.TlsInteractionResult
    -- ^ __Returns:__ The status of the certificate request interaction. /(Can throw 'Data.GI.Base.GError.GError')/
tlsInteractionInvokeRequestCertificate interaction connection flags cancellable = liftIO $ do
    interaction' <- unsafeManagedPtrCastPtr interaction
    connection' <- unsafeManagedPtrCastPtr connection
    let flags' = (fromIntegral . fromEnum) flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_tls_interaction_invoke_request_certificate interaction' connection' flags' maybeCancellable
        let result' = (toEnum . fromIntegral) result
        touchManagedPtr interaction
        touchManagedPtr connection
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data TlsInteractionInvokeRequestCertificateMethodInfo
instance (signature ~ (b -> Gio.Enums.TlsCertificateRequestFlags -> Maybe (c) -> m Gio.Enums.TlsInteractionResult), MonadIO m, IsTlsInteraction a, Gio.TlsConnection.IsTlsConnection b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod TlsInteractionInvokeRequestCertificateMethodInfo a signature where
    overloadedMethod = tlsInteractionInvokeRequestCertificate

instance O.OverloadedMethodInfo TlsInteractionInvokeRequestCertificateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.TlsInteraction.tlsInteractionInvokeRequestCertificate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-TlsInteraction.html#v:tlsInteractionInvokeRequestCertificate"
        })


#endif

-- method TlsInteraction::request_certificate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "interaction"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsInteraction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsInteraction object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsConnection object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "TlsCertificateRequestFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags providing more information about the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an optional #GCancellable cancellation object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "TlsInteractionResult" })
-- throws : True
-- Skip return : False

foreign import ccall "g_tls_interaction_request_certificate" g_tls_interaction_request_certificate :: 
    Ptr TlsInteraction ->                   -- interaction : TInterface (Name {namespace = "Gio", name = "TlsInteraction"})
    Ptr Gio.TlsConnection.TlsConnection ->  -- connection : TInterface (Name {namespace = "Gio", name = "TlsConnection"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "TlsCertificateRequestFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CUInt

-- | Run synchronous interaction to ask the user to choose a certificate to use
-- with the connection. In general, 'GI.Gio.Objects.TlsInteraction.tlsInteractionInvokeRequestCertificate'
-- should be used instead of this function.
-- 
-- Derived subclasses usually implement a certificate selector, although they may
-- also choose to provide a certificate from elsewhere. Alternatively the user may
-- abort this certificate request, which will usually abort the TLS connection.
-- 
-- If 'GI.Gio.Enums.TlsInteractionResultHandled' is returned, then the t'GI.Gio.Objects.TlsConnection.TlsConnection'
-- passed to 'GI.Gio.Objects.TlsInteraction.tlsInteractionRequestCertificate' will have had its
-- [TlsConnection:certificate]("GI.Gio.Objects.TlsConnection#g:attr:certificate") filled in.
-- 
-- If the interaction is cancelled by the cancellation object, or by the
-- user then 'GI.Gio.Enums.TlsInteractionResultFailed' will be returned with an error that
-- contains a 'GI.Gio.Enums.IOErrorEnumCancelled' error code. Certain implementations may
-- not support immediate cancellation.
-- 
-- /Since: 2.40/
tlsInteractionRequestCertificate ::
    (B.CallStack.HasCallStack, MonadIO m, IsTlsInteraction a, Gio.TlsConnection.IsTlsConnection b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@interaction@/: a t'GI.Gio.Objects.TlsInteraction.TlsInteraction' object
    -> b
    -- ^ /@connection@/: a t'GI.Gio.Objects.TlsConnection.TlsConnection' object
    -> Gio.Enums.TlsCertificateRequestFlags
    -- ^ /@flags@/: flags providing more information about the request
    -> Maybe (c)
    -- ^ /@cancellable@/: an optional t'GI.Gio.Objects.Cancellable.Cancellable' cancellation object
    -> m Gio.Enums.TlsInteractionResult
    -- ^ __Returns:__ The status of the request certificate interaction. /(Can throw 'Data.GI.Base.GError.GError')/
tlsInteractionRequestCertificate interaction connection flags cancellable = liftIO $ do
    interaction' <- unsafeManagedPtrCastPtr interaction
    connection' <- unsafeManagedPtrCastPtr connection
    let flags' = (fromIntegral . fromEnum) flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_tls_interaction_request_certificate interaction' connection' flags' maybeCancellable
        let result' = (toEnum . fromIntegral) result
        touchManagedPtr interaction
        touchManagedPtr connection
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data TlsInteractionRequestCertificateMethodInfo
instance (signature ~ (b -> Gio.Enums.TlsCertificateRequestFlags -> Maybe (c) -> m Gio.Enums.TlsInteractionResult), MonadIO m, IsTlsInteraction a, Gio.TlsConnection.IsTlsConnection b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod TlsInteractionRequestCertificateMethodInfo a signature where
    overloadedMethod = tlsInteractionRequestCertificate

instance O.OverloadedMethodInfo TlsInteractionRequestCertificateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.TlsInteraction.tlsInteractionRequestCertificate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-TlsInteraction.html#v:tlsInteractionRequestCertificate"
        })


#endif

-- method TlsInteraction::request_certificate_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "interaction"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsInteraction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsInteraction object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsConnection object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "TlsCertificateRequestFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags providing more information about the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an optional #GCancellable cancellation object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "will be called when the interaction completes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to pass to the @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_tls_interaction_request_certificate_async" g_tls_interaction_request_certificate_async :: 
    Ptr TlsInteraction ->                   -- interaction : TInterface (Name {namespace = "Gio", name = "TlsInteraction"})
    Ptr Gio.TlsConnection.TlsConnection ->  -- connection : TInterface (Name {namespace = "Gio", name = "TlsConnection"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "TlsCertificateRequestFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Run asynchronous interaction to ask the user for a certificate to use with
-- the connection. In general, 'GI.Gio.Objects.TlsInteraction.tlsInteractionInvokeRequestCertificate' should
-- be used instead of this function.
-- 
-- Derived subclasses usually implement a certificate selector, although they may
-- also choose to provide a certificate from elsewhere. /@callback@/ will be called
-- when the operation completes. Alternatively the user may abort this certificate
-- request, which will usually abort the TLS connection.
-- 
-- /Since: 2.40/
tlsInteractionRequestCertificateAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsTlsInteraction a, Gio.TlsConnection.IsTlsConnection b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@interaction@/: a t'GI.Gio.Objects.TlsInteraction.TlsInteraction' object
    -> b
    -- ^ /@connection@/: a t'GI.Gio.Objects.TlsConnection.TlsConnection' object
    -> Gio.Enums.TlsCertificateRequestFlags
    -- ^ /@flags@/: flags providing more information about the request
    -> Maybe (c)
    -- ^ /@cancellable@/: an optional t'GI.Gio.Objects.Cancellable.Cancellable' cancellation object
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: will be called when the interaction completes
    -> m ()
tlsInteractionRequestCertificateAsync interaction connection flags cancellable callback = liftIO $ do
    interaction' <- unsafeManagedPtrCastPtr interaction
    connection' <- unsafeManagedPtrCastPtr connection
    let flags' = (fromIntegral . fromEnum) flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_tls_interaction_request_certificate_async interaction' connection' flags' maybeCancellable maybeCallback userData
    touchManagedPtr interaction
    touchManagedPtr connection
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data TlsInteractionRequestCertificateAsyncMethodInfo
instance (signature ~ (b -> Gio.Enums.TlsCertificateRequestFlags -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsTlsInteraction a, Gio.TlsConnection.IsTlsConnection b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod TlsInteractionRequestCertificateAsyncMethodInfo a signature where
    overloadedMethod = tlsInteractionRequestCertificateAsync

instance O.OverloadedMethodInfo TlsInteractionRequestCertificateAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.TlsInteraction.tlsInteractionRequestCertificateAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-TlsInteraction.html#v:tlsInteractionRequestCertificateAsync"
        })


#endif

-- method TlsInteraction::request_certificate_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "interaction"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "TlsInteraction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTlsInteraction object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the result passed to the callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "TlsInteractionResult" })
-- throws : True
-- Skip return : False

foreign import ccall "g_tls_interaction_request_certificate_finish" g_tls_interaction_request_certificate_finish :: 
    Ptr TlsInteraction ->                   -- interaction : TInterface (Name {namespace = "Gio", name = "TlsInteraction"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CUInt

-- | Complete a request certificate user interaction request. This should be once
-- the 'GI.Gio.Objects.TlsInteraction.tlsInteractionRequestCertificateAsync' completion callback is called.
-- 
-- If 'GI.Gio.Enums.TlsInteractionResultHandled' is returned, then the t'GI.Gio.Objects.TlsConnection.TlsConnection'
-- passed to 'GI.Gio.Objects.TlsInteraction.tlsInteractionRequestCertificateAsync' will have had its
-- [TlsConnection:certificate]("GI.Gio.Objects.TlsConnection#g:attr:certificate") filled in.
-- 
-- If the interaction is cancelled by the cancellation object, or by the
-- user then 'GI.Gio.Enums.TlsInteractionResultFailed' will be returned with an error that
-- contains a 'GI.Gio.Enums.IOErrorEnumCancelled' error code.
-- 
-- /Since: 2.40/
tlsInteractionRequestCertificateFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsTlsInteraction a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@interaction@/: a t'GI.Gio.Objects.TlsInteraction.TlsInteraction' object
    -> b
    -- ^ /@result@/: the result passed to the callback
    -> m Gio.Enums.TlsInteractionResult
    -- ^ __Returns:__ The status of the request certificate interaction. /(Can throw 'Data.GI.Base.GError.GError')/
tlsInteractionRequestCertificateFinish interaction result_ = liftIO $ do
    interaction' <- unsafeManagedPtrCastPtr interaction
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_tls_interaction_request_certificate_finish interaction' result_'
        let result' = (toEnum . fromIntegral) result
        touchManagedPtr interaction
        touchManagedPtr result_
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data TlsInteractionRequestCertificateFinishMethodInfo
instance (signature ~ (b -> m Gio.Enums.TlsInteractionResult), MonadIO m, IsTlsInteraction a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod TlsInteractionRequestCertificateFinishMethodInfo a signature where
    overloadedMethod = tlsInteractionRequestCertificateFinish

instance O.OverloadedMethodInfo TlsInteractionRequestCertificateFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.TlsInteraction.tlsInteractionRequestCertificateFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-TlsInteraction.html#v:tlsInteractionRequestCertificateFinish"
        })


#endif


