{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GSocketConnection@ is a t'GI.Gio.Objects.IOStream.IOStream' for a connected socket. They
-- can be created either by t'GI.Gio.Objects.SocketClient.SocketClient' when connecting to a host,
-- or by t'GI.Gio.Objects.SocketListener.SocketListener' when accepting a new client.
-- 
-- The type of the @GSocketConnection@ object returned from these calls
-- depends on the type of the underlying socket that is in use. For
-- instance, for a TCP\/IP connection it will be a t'GI.Gio.Objects.TcpConnection.TcpConnection'.
-- 
-- Choosing what type of object to construct is done with the socket
-- connection factory, and it is possible for third parties to register
-- custom socket connection types for specific combination of socket
-- family\/type\/protocol using [func/@gio@/.SocketConnection.factory_register_type].
-- 
-- To close a @GSocketConnection@, use 'GI.Gio.Objects.IOStream.iOStreamClose'. Closing both
-- substreams of the t'GI.Gio.Objects.IOStream.IOStream' separately will not close the
-- underlying t'GI.Gio.Objects.Socket.Socket'.
-- 
-- /Since: 2.22/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.SocketConnection
    ( 

-- * Exported types
    SocketConnection(..)                    ,
    IsSocketConnection                      ,
    toSocketConnection                      ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [clearPending]("GI.Gio.Objects.IOStream#g:method:clearPending"), [close]("GI.Gio.Objects.IOStream#g:method:close"), [closeAsync]("GI.Gio.Objects.IOStream#g:method:closeAsync"), [closeFinish]("GI.Gio.Objects.IOStream#g:method:closeFinish"), [connect]("GI.Gio.Objects.SocketConnection#g:method:connect"), [connectAsync]("GI.Gio.Objects.SocketConnection#g:method:connectAsync"), [connectFinish]("GI.Gio.Objects.SocketConnection#g:method:connectFinish"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [hasPending]("GI.Gio.Objects.IOStream#g:method:hasPending"), [isClosed]("GI.Gio.Objects.IOStream#g:method:isClosed"), [isConnected]("GI.Gio.Objects.SocketConnection#g:method:isConnected"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [spliceAsync]("GI.Gio.Objects.IOStream#g:method:spliceAsync"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getInputStream]("GI.Gio.Objects.IOStream#g:method:getInputStream"), [getLocalAddress]("GI.Gio.Objects.SocketConnection#g:method:getLocalAddress"), [getOutputStream]("GI.Gio.Objects.IOStream#g:method:getOutputStream"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRemoteAddress]("GI.Gio.Objects.SocketConnection#g:method:getRemoteAddress"), [getSocket]("GI.Gio.Objects.SocketConnection#g:method:getSocket").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setPending]("GI.Gio.Objects.IOStream#g:method:setPending"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveSocketConnectionMethod           ,
#endif

-- ** connect #method:connect#

#if defined(ENABLE_OVERLOADING)
    SocketConnectionConnectMethodInfo       ,
#endif
    socketConnectionConnect                 ,


-- ** connectAsync #method:connectAsync#

#if defined(ENABLE_OVERLOADING)
    SocketConnectionConnectAsyncMethodInfo  ,
#endif
    socketConnectionConnectAsync            ,


-- ** connectFinish #method:connectFinish#

#if defined(ENABLE_OVERLOADING)
    SocketConnectionConnectFinishMethodInfo ,
#endif
    socketConnectionConnectFinish           ,


-- ** factoryLookupType #method:factoryLookupType#

    socketConnectionFactoryLookupType       ,


-- ** factoryRegisterType #method:factoryRegisterType#

    socketConnectionFactoryRegisterType     ,


-- ** getLocalAddress #method:getLocalAddress#

#if defined(ENABLE_OVERLOADING)
    SocketConnectionGetLocalAddressMethodInfo,
#endif
    socketConnectionGetLocalAddress         ,


-- ** getRemoteAddress #method:getRemoteAddress#

#if defined(ENABLE_OVERLOADING)
    SocketConnectionGetRemoteAddressMethodInfo,
#endif
    socketConnectionGetRemoteAddress        ,


-- ** getSocket #method:getSocket#

#if defined(ENABLE_OVERLOADING)
    SocketConnectionGetSocketMethodInfo     ,
#endif
    socketConnectionGetSocket               ,


-- ** isConnected #method:isConnected#

#if defined(ENABLE_OVERLOADING)
    SocketConnectionIsConnectedMethodInfo   ,
#endif
    socketConnectionIsConnected             ,




 -- * Properties


-- ** socket #attr:socket#
-- | The underlying t'GI.Gio.Objects.Socket.Socket'.
-- 
-- /Since: 2.22/

#if defined(ENABLE_OVERLOADING)
    SocketConnectionSocketPropertyInfo      ,
#endif
    constructSocketConnectionSocket         ,
    getSocketConnectionSocket               ,
#if defined(ENABLE_OVERLOADING)
    socketConnectionSocket                  ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Flags as GLib.Flags
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.DatagramBased as Gio.DatagramBased
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.SocketConnectable as Gio.SocketConnectable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.Credentials as Gio.Credentials
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InetAddress as Gio.InetAddress
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.Socket as Gio.Socket
import {-# SOURCE #-} qualified GI.Gio.Objects.SocketAddress as Gio.SocketAddress
import {-# SOURCE #-} qualified GI.Gio.Objects.SocketAddressEnumerator as Gio.SocketAddressEnumerator
import {-# SOURCE #-} qualified GI.Gio.Objects.SocketControlMessage as Gio.SocketControlMessage
import {-# SOURCE #-} qualified GI.Gio.Structs.InputMessage as Gio.InputMessage
import {-# SOURCE #-} qualified GI.Gio.Structs.InputVector as Gio.InputVector
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputMessage as Gio.OutputMessage
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.Socket as Gio.Socket
import {-# SOURCE #-} qualified GI.Gio.Objects.SocketAddress as Gio.SocketAddress

#endif

-- | Memory-managed wrapper type.
newtype SocketConnection = SocketConnection (SP.ManagedPtr SocketConnection)
    deriving (Eq)

instance SP.ManagedPtrNewtype SocketConnection where
    toManagedPtr (SocketConnection p) = p

foreign import ccall "g_socket_connection_get_type"
    c_g_socket_connection_get_type :: IO B.Types.GType

instance B.Types.TypedObject SocketConnection where
    glibType = c_g_socket_connection_get_type

instance B.Types.GObject SocketConnection

-- | Type class for types which can be safely cast to `SocketConnection`, for instance with `toSocketConnection`.
class (SP.GObject o, O.IsDescendantOf SocketConnection o) => IsSocketConnection o
instance (SP.GObject o, O.IsDescendantOf SocketConnection o) => IsSocketConnection o

instance O.HasParentTypes SocketConnection
type instance O.ParentTypes SocketConnection = '[Gio.IOStream.IOStream, GObject.Object.Object]

-- | Cast to `SocketConnection`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toSocketConnection :: (MIO.MonadIO m, IsSocketConnection o) => o -> m SocketConnection
toSocketConnection = MIO.liftIO . B.ManagedPtr.unsafeCastTo SocketConnection

-- | Convert 'SocketConnection' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe SocketConnection) where
    gvalueGType_ = c_g_socket_connection_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr SocketConnection)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr SocketConnection)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject SocketConnection ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveSocketConnectionMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveSocketConnectionMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveSocketConnectionMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveSocketConnectionMethod "clearPending" o = Gio.IOStream.IOStreamClearPendingMethodInfo
    ResolveSocketConnectionMethod "close" o = Gio.IOStream.IOStreamCloseMethodInfo
    ResolveSocketConnectionMethod "closeAsync" o = Gio.IOStream.IOStreamCloseAsyncMethodInfo
    ResolveSocketConnectionMethod "closeFinish" o = Gio.IOStream.IOStreamCloseFinishMethodInfo
    ResolveSocketConnectionMethod "connect" o = SocketConnectionConnectMethodInfo
    ResolveSocketConnectionMethod "connectAsync" o = SocketConnectionConnectAsyncMethodInfo
    ResolveSocketConnectionMethod "connectFinish" o = SocketConnectionConnectFinishMethodInfo
    ResolveSocketConnectionMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveSocketConnectionMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveSocketConnectionMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveSocketConnectionMethod "hasPending" o = Gio.IOStream.IOStreamHasPendingMethodInfo
    ResolveSocketConnectionMethod "isClosed" o = Gio.IOStream.IOStreamIsClosedMethodInfo
    ResolveSocketConnectionMethod "isConnected" o = SocketConnectionIsConnectedMethodInfo
    ResolveSocketConnectionMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveSocketConnectionMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveSocketConnectionMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveSocketConnectionMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveSocketConnectionMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveSocketConnectionMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveSocketConnectionMethod "spliceAsync" o = Gio.IOStream.IOStreamSpliceAsyncMethodInfo
    ResolveSocketConnectionMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveSocketConnectionMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveSocketConnectionMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveSocketConnectionMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveSocketConnectionMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveSocketConnectionMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveSocketConnectionMethod "getInputStream" o = Gio.IOStream.IOStreamGetInputStreamMethodInfo
    ResolveSocketConnectionMethod "getLocalAddress" o = SocketConnectionGetLocalAddressMethodInfo
    ResolveSocketConnectionMethod "getOutputStream" o = Gio.IOStream.IOStreamGetOutputStreamMethodInfo
    ResolveSocketConnectionMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveSocketConnectionMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveSocketConnectionMethod "getRemoteAddress" o = SocketConnectionGetRemoteAddressMethodInfo
    ResolveSocketConnectionMethod "getSocket" o = SocketConnectionGetSocketMethodInfo
    ResolveSocketConnectionMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveSocketConnectionMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveSocketConnectionMethod "setPending" o = Gio.IOStream.IOStreamSetPendingMethodInfo
    ResolveSocketConnectionMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveSocketConnectionMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveSocketConnectionMethod t SocketConnection, O.OverloadedMethod info SocketConnection p) => OL.IsLabel t (SocketConnection -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveSocketConnectionMethod t SocketConnection, O.OverloadedMethod info SocketConnection p, R.HasField t SocketConnection p) => R.HasField t SocketConnection p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveSocketConnectionMethod t SocketConnection, O.OverloadedMethodInfo info SocketConnection) => OL.IsLabel t (O.MethodProxy info SocketConnection) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "socket"
   -- Type: TInterface (Name {namespace = "Gio", name = "Socket"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@socket@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' socketConnection #socket
-- @
getSocketConnectionSocket :: (MonadIO m, IsSocketConnection o) => o -> m Gio.Socket.Socket
getSocketConnectionSocket obj = MIO.liftIO $ checkUnexpectedNothing "getSocketConnectionSocket" $ B.Properties.getObjectPropertyObject obj "socket" Gio.Socket.Socket

-- | Construct a `GValueConstruct` with valid value for the “@socket@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSocketConnectionSocket :: (IsSocketConnection o, MIO.MonadIO m, Gio.Socket.IsSocket a) => a -> m (GValueConstruct o)
constructSocketConnectionSocket val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "socket" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data SocketConnectionSocketPropertyInfo
instance AttrInfo SocketConnectionSocketPropertyInfo where
    type AttrAllowedOps SocketConnectionSocketPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SocketConnectionSocketPropertyInfo = IsSocketConnection
    type AttrSetTypeConstraint SocketConnectionSocketPropertyInfo = Gio.Socket.IsSocket
    type AttrTransferTypeConstraint SocketConnectionSocketPropertyInfo = Gio.Socket.IsSocket
    type AttrTransferType SocketConnectionSocketPropertyInfo = Gio.Socket.Socket
    type AttrGetType SocketConnectionSocketPropertyInfo = Gio.Socket.Socket
    type AttrLabel SocketConnectionSocketPropertyInfo = "socket"
    type AttrOrigin SocketConnectionSocketPropertyInfo = SocketConnection
    attrGet = getSocketConnectionSocket
    attrSet = undefined
    attrTransfer _ v = do
        unsafeCastTo Gio.Socket.Socket v
    attrConstruct = constructSocketConnectionSocket
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.SocketConnection.socket"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-SocketConnection.html#g:attr:socket"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList SocketConnection
type instance O.AttributeList SocketConnection = SocketConnectionAttributeList
type SocketConnectionAttributeList = ('[ '("closed", Gio.IOStream.IOStreamClosedPropertyInfo), '("inputStream", Gio.IOStream.IOStreamInputStreamPropertyInfo), '("outputStream", Gio.IOStream.IOStreamOutputStreamPropertyInfo), '("socket", SocketConnectionSocketPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
socketConnectionSocket :: AttrLabelProxy "socket"
socketConnectionSocket = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList SocketConnection = SocketConnectionSignalList
type SocketConnectionSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method SocketConnection::connect
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "address"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketAddress" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GSocketAddress specifying the remote address."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a %GCancellable or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_socket_connection_connect" g_socket_connection_connect :: 
    Ptr SocketConnection ->                 -- connection : TInterface (Name {namespace = "Gio", name = "SocketConnection"})
    Ptr Gio.SocketAddress.SocketAddress ->  -- address : TInterface (Name {namespace = "Gio", name = "SocketAddress"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Connect /@connection@/ to the specified remote address.
-- 
-- /Since: 2.32/
socketConnectionConnect ::
    (B.CallStack.HasCallStack, MonadIO m, IsSocketConnection a, Gio.SocketAddress.IsSocketAddress b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@connection@/: a t'GI.Gio.Objects.SocketConnection.SocketConnection'
    -> b
    -- ^ /@address@/: a t'GI.Gio.Objects.SocketAddress.SocketAddress' specifying the remote address.
    -> Maybe (c)
    -- ^ /@cancellable@/: a @/GCancellable/@ or 'P.Nothing'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
socketConnectionConnect connection address cancellable = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    address' <- unsafeManagedPtrCastPtr address
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_socket_connection_connect connection' address' maybeCancellable
        touchManagedPtr connection
        touchManagedPtr address
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data SocketConnectionConnectMethodInfo
instance (signature ~ (b -> Maybe (c) -> m ()), MonadIO m, IsSocketConnection a, Gio.SocketAddress.IsSocketAddress b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod SocketConnectionConnectMethodInfo a signature where
    overloadedMethod = socketConnectionConnect

instance O.OverloadedMethodInfo SocketConnectionConnectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.SocketConnection.socketConnectionConnect",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-SocketConnection.html#v:socketConnectionConnect"
        })


#endif

-- method SocketConnection::connect_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "address"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketAddress" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GSocketAddress specifying the remote address."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a %GCancellable or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data for the callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_socket_connection_connect_async" g_socket_connection_connect_async :: 
    Ptr SocketConnection ->                 -- connection : TInterface (Name {namespace = "Gio", name = "SocketConnection"})
    Ptr Gio.SocketAddress.SocketAddress ->  -- address : TInterface (Name {namespace = "Gio", name = "SocketAddress"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously connect /@connection@/ to the specified remote address.
-- 
-- This clears the [Socket:blocking]("GI.Gio.Objects.Socket#g:attr:blocking") flag on /@connection@/\'s underlying
-- socket if it is currently set.
-- 
-- If [Socket:timeout]("GI.Gio.Objects.Socket#g:attr:timeout") is set, the operation will time out and return
-- 'GI.Gio.Enums.IOErrorEnumTimedOut' after that period. Otherwise, it will continue
-- indefinitely until operating system timeouts (if any) are hit.
-- 
-- Use 'GI.Gio.Objects.SocketConnection.socketConnectionConnectFinish' to retrieve the result.
-- 
-- /Since: 2.32/
socketConnectionConnectAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsSocketConnection a, Gio.SocketAddress.IsSocketAddress b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@connection@/: a t'GI.Gio.Objects.SocketConnection.SocketConnection'
    -> b
    -- ^ /@address@/: a t'GI.Gio.Objects.SocketAddress.SocketAddress' specifying the remote address.
    -> Maybe (c)
    -- ^ /@cancellable@/: a @/GCancellable/@ or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m ()
socketConnectionConnectAsync connection address cancellable callback = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    address' <- unsafeManagedPtrCastPtr address
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_socket_connection_connect_async connection' address' maybeCancellable maybeCallback userData
    touchManagedPtr connection
    touchManagedPtr address
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data SocketConnectionConnectAsyncMethodInfo
instance (signature ~ (b -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsSocketConnection a, Gio.SocketAddress.IsSocketAddress b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod SocketConnectionConnectAsyncMethodInfo a signature where
    overloadedMethod = socketConnectionConnectAsync

instance O.OverloadedMethodInfo SocketConnectionConnectAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.SocketConnection.socketConnectionConnectAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-SocketConnection.html#v:socketConnectionConnectAsync"
        })


#endif

-- method SocketConnection::connect_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_socket_connection_connect_finish" g_socket_connection_connect_finish :: 
    Ptr SocketConnection ->                 -- connection : TInterface (Name {namespace = "Gio", name = "SocketConnection"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Gets the result of a 'GI.Gio.Objects.SocketConnection.socketConnectionConnectAsync' call.
-- 
-- /Since: 2.32/
socketConnectionConnectFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsSocketConnection a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@connection@/: a t'GI.Gio.Objects.SocketConnection.SocketConnection'
    -> b
    -- ^ /@result@/: the t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
socketConnectionConnectFinish connection result_ = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_socket_connection_connect_finish connection' result_'
        touchManagedPtr connection
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data SocketConnectionConnectFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsSocketConnection a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod SocketConnectionConnectFinishMethodInfo a signature where
    overloadedMethod = socketConnectionConnectFinish

instance O.OverloadedMethodInfo SocketConnectionConnectFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.SocketConnection.socketConnectionConnectFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-SocketConnection.html#v:socketConnectionConnectFinish"
        })


#endif

-- method SocketConnection::get_local_address
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "SocketAddress" })
-- throws : True
-- Skip return : False

foreign import ccall "g_socket_connection_get_local_address" g_socket_connection_get_local_address :: 
    Ptr SocketConnection ->                 -- connection : TInterface (Name {namespace = "Gio", name = "SocketConnection"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.SocketAddress.SocketAddress)

-- | Try to get the local address of a socket connection.
-- 
-- /Since: 2.22/
socketConnectionGetLocalAddress ::
    (B.CallStack.HasCallStack, MonadIO m, IsSocketConnection a) =>
    a
    -- ^ /@connection@/: a t'GI.Gio.Objects.SocketConnection.SocketConnection'
    -> m Gio.SocketAddress.SocketAddress
    -- ^ __Returns:__ a t'GI.Gio.Objects.SocketAddress.SocketAddress' or 'P.Nothing' on error.
    --     Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
socketConnectionGetLocalAddress connection = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    onException (do
        result <- propagateGError $ g_socket_connection_get_local_address connection'
        checkUnexpectedReturnNULL "socketConnectionGetLocalAddress" result
        result' <- (wrapObject Gio.SocketAddress.SocketAddress) result
        touchManagedPtr connection
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data SocketConnectionGetLocalAddressMethodInfo
instance (signature ~ (m Gio.SocketAddress.SocketAddress), MonadIO m, IsSocketConnection a) => O.OverloadedMethod SocketConnectionGetLocalAddressMethodInfo a signature where
    overloadedMethod = socketConnectionGetLocalAddress

instance O.OverloadedMethodInfo SocketConnectionGetLocalAddressMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.SocketConnection.socketConnectionGetLocalAddress",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-SocketConnection.html#v:socketConnectionGetLocalAddress"
        })


#endif

-- method SocketConnection::get_remote_address
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "SocketAddress" })
-- throws : True
-- Skip return : False

foreign import ccall "g_socket_connection_get_remote_address" g_socket_connection_get_remote_address :: 
    Ptr SocketConnection ->                 -- connection : TInterface (Name {namespace = "Gio", name = "SocketConnection"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.SocketAddress.SocketAddress)

-- | Try to get the remote address of a socket connection.
-- 
-- Since GLib 2.40, when used with 'GI.Gio.Objects.SocketClient.socketClientConnect' or
-- 'GI.Gio.Objects.SocketClient.socketClientConnectAsync', during emission of
-- 'GI.Gio.Enums.SocketClientEventConnecting', this function will return the remote
-- address that will be used for the connection.  This allows
-- applications to print e.g. \"Connecting to example.com
-- (10.42.77.3)...\".
-- 
-- /Since: 2.22/
socketConnectionGetRemoteAddress ::
    (B.CallStack.HasCallStack, MonadIO m, IsSocketConnection a) =>
    a
    -- ^ /@connection@/: a t'GI.Gio.Objects.SocketConnection.SocketConnection'
    -> m Gio.SocketAddress.SocketAddress
    -- ^ __Returns:__ a t'GI.Gio.Objects.SocketAddress.SocketAddress' or 'P.Nothing' on error.
    --     Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
socketConnectionGetRemoteAddress connection = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    onException (do
        result <- propagateGError $ g_socket_connection_get_remote_address connection'
        checkUnexpectedReturnNULL "socketConnectionGetRemoteAddress" result
        result' <- (wrapObject Gio.SocketAddress.SocketAddress) result
        touchManagedPtr connection
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data SocketConnectionGetRemoteAddressMethodInfo
instance (signature ~ (m Gio.SocketAddress.SocketAddress), MonadIO m, IsSocketConnection a) => O.OverloadedMethod SocketConnectionGetRemoteAddressMethodInfo a signature where
    overloadedMethod = socketConnectionGetRemoteAddress

instance O.OverloadedMethodInfo SocketConnectionGetRemoteAddressMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.SocketConnection.socketConnectionGetRemoteAddress",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-SocketConnection.html#v:socketConnectionGetRemoteAddress"
        })


#endif

-- method SocketConnection::get_socket
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Socket" })
-- throws : False
-- Skip return : False

foreign import ccall "g_socket_connection_get_socket" g_socket_connection_get_socket :: 
    Ptr SocketConnection ->                 -- connection : TInterface (Name {namespace = "Gio", name = "SocketConnection"})
    IO (Ptr Gio.Socket.Socket)

-- | Gets the underlying t'GI.Gio.Objects.Socket.Socket' object of the connection.
-- This can be useful if you want to do something unusual on it
-- not supported by the t'GI.Gio.Objects.SocketConnection.SocketConnection' APIs.
-- 
-- /Since: 2.22/
socketConnectionGetSocket ::
    (B.CallStack.HasCallStack, MonadIO m, IsSocketConnection a) =>
    a
    -- ^ /@connection@/: a t'GI.Gio.Objects.SocketConnection.SocketConnection'
    -> m Gio.Socket.Socket
    -- ^ __Returns:__ a t'GI.Gio.Objects.Socket.Socket' or 'P.Nothing' on error.
socketConnectionGetSocket connection = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    result <- g_socket_connection_get_socket connection'
    checkUnexpectedReturnNULL "socketConnectionGetSocket" result
    result' <- (newObject Gio.Socket.Socket) result
    touchManagedPtr connection
    return result'

#if defined(ENABLE_OVERLOADING)
data SocketConnectionGetSocketMethodInfo
instance (signature ~ (m Gio.Socket.Socket), MonadIO m, IsSocketConnection a) => O.OverloadedMethod SocketConnectionGetSocketMethodInfo a signature where
    overloadedMethod = socketConnectionGetSocket

instance O.OverloadedMethodInfo SocketConnectionGetSocketMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.SocketConnection.socketConnectionGetSocket",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-SocketConnection.html#v:socketConnectionGetSocket"
        })


#endif

-- method SocketConnection::is_connected
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketConnection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_socket_connection_is_connected" g_socket_connection_is_connected :: 
    Ptr SocketConnection ->                 -- connection : TInterface (Name {namespace = "Gio", name = "SocketConnection"})
    IO CInt

-- | Checks if /@connection@/ is connected. This is equivalent to calling
-- 'GI.Gio.Objects.Socket.socketIsConnected' on /@connection@/\'s underlying t'GI.Gio.Objects.Socket.Socket'.
-- 
-- /Since: 2.32/
socketConnectionIsConnected ::
    (B.CallStack.HasCallStack, MonadIO m, IsSocketConnection a) =>
    a
    -- ^ /@connection@/: a t'GI.Gio.Objects.SocketConnection.SocketConnection'
    -> m Bool
    -- ^ __Returns:__ whether /@connection@/ is connected
socketConnectionIsConnected connection = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    result <- g_socket_connection_is_connected connection'
    let result' = (/= 0) result
    touchManagedPtr connection
    return result'

#if defined(ENABLE_OVERLOADING)
data SocketConnectionIsConnectedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsSocketConnection a) => O.OverloadedMethod SocketConnectionIsConnectedMethodInfo a signature where
    overloadedMethod = socketConnectionIsConnected

instance O.OverloadedMethodInfo SocketConnectionIsConnectedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.SocketConnection.socketConnectionIsConnected",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-SocketConnection.html#v:socketConnectionIsConnected"
        })


#endif

-- method SocketConnection::factory_lookup_type
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "family"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketFamily" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketFamily" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketType" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "protocol_id"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a protocol id" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TGType)
-- throws : False
-- Skip return : False

foreign import ccall "g_socket_connection_factory_lookup_type" g_socket_connection_factory_lookup_type :: 
    CUInt ->                                -- family : TInterface (Name {namespace = "Gio", name = "SocketFamily"})
    CUInt ->                                -- type : TInterface (Name {namespace = "Gio", name = "SocketType"})
    Int32 ->                                -- protocol_id : TBasicType TInt
    IO CGType

-- | Looks up the t'GType' to be used when creating socket connections on
-- sockets with the specified /@family@/, /@type@/ and /@protocolId@/.
-- 
-- If no type is registered, the t'GI.Gio.Objects.SocketConnection.SocketConnection' base type is returned.
-- 
-- /Since: 2.22/
socketConnectionFactoryLookupType ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Gio.Enums.SocketFamily
    -- ^ /@family@/: a t'GI.Gio.Enums.SocketFamily'
    -> Gio.Enums.SocketType
    -- ^ /@type@/: a t'GI.Gio.Enums.SocketType'
    -> Int32
    -- ^ /@protocolId@/: a protocol id
    -> m GType
    -- ^ __Returns:__ a t'GType'
socketConnectionFactoryLookupType family type_ protocolId = liftIO $ do
    let family' = (fromIntegral . fromEnum) family
    let type_' = (fromIntegral . fromEnum) type_
    result <- g_socket_connection_factory_lookup_type family' type_' protocolId
    let result' = GType result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method SocketConnection::factory_register_type
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "g_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GType, inheriting from %G_TYPE_SOCKET_CONNECTION"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "family"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketFamily" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketFamily" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SocketType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSocketType" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "protocol"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a protocol id" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_socket_connection_factory_register_type" g_socket_connection_factory_register_type :: 
    CGType ->                               -- g_type : TBasicType TGType
    CUInt ->                                -- family : TInterface (Name {namespace = "Gio", name = "SocketFamily"})
    CUInt ->                                -- type : TInterface (Name {namespace = "Gio", name = "SocketType"})
    Int32 ->                                -- protocol : TBasicType TInt
    IO ()

-- | Looks up the t'GType' to be used when creating socket connections on
-- sockets with the specified /@family@/, /@type@/ and /@protocol@/.
-- 
-- If no type is registered, the t'GI.Gio.Objects.SocketConnection.SocketConnection' base type is returned.
-- 
-- /Since: 2.22/
socketConnectionFactoryRegisterType ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GType
    -- ^ /@gType@/: a t'GType', inheriting from @/G_TYPE_SOCKET_CONNECTION/@
    -> Gio.Enums.SocketFamily
    -- ^ /@family@/: a t'GI.Gio.Enums.SocketFamily'
    -> Gio.Enums.SocketType
    -- ^ /@type@/: a t'GI.Gio.Enums.SocketType'
    -> Int32
    -- ^ /@protocol@/: a protocol id
    -> m ()
socketConnectionFactoryRegisterType gType family type_ protocol = liftIO $ do
    let gType' = gtypeToCGType gType
    let family' = (fromIntegral . fromEnum) family
    let type_' = (fromIntegral . fromEnum) type_
    g_socket_connection_factory_register_type gType' family' type_' protocol
    return ()

#if defined(ENABLE_OVERLOADING)
#endif


