{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A @GPropertyAction@ is a way to get a t'GI.Gio.Interfaces.Action.Action' with a state value
-- reflecting and controlling the value of a t'GI.GObject.Objects.Object.Object' property.
-- 
-- The state of the action will correspond to the value of the property.
-- Changing it will change the property (assuming the requested value
-- matches the requirements as specified in the [type/@gObject@/.ParamSpec]).
-- 
-- Only the most common types are presently supported.  Booleans are
-- mapped to booleans, strings to strings, signed\/unsigned integers to
-- int32\/uint32 and floats and doubles to doubles.
-- 
-- If the property is an enum then the state will be string-typed and
-- conversion will automatically be performed between the enum value and
-- ‘nick’ string as per the [type/@gObject@/.EnumValue] table.
-- 
-- Flags types are not currently supported.
-- 
-- Properties of object types, boxed types and pointer types are not
-- supported and probably never will be.
-- 
-- Properties of [type/@gLib@/.Variant] types are not currently supported.
-- 
-- If the property is boolean-valued then the action will have a @NULL@
-- parameter type, and activating the action (with no parameter) will
-- toggle the value of the property.
-- 
-- In all other cases, the parameter type will correspond to the type of
-- the property.
-- 
-- The general idea here is to reduce the number of locations where a
-- particular piece of state is kept (and therefore has to be synchronised
-- between). @GPropertyAction@ does not have a separate state that is kept
-- in sync with the property value — its state is the property value.
-- 
-- For example, it might be useful to create a t'GI.Gio.Interfaces.Action.Action' corresponding
-- to the @visible-child-name@ property of a <https://docs.gtk.org/gtk4/class.Stack.html `GtkStack`>
-- so that the current page can be switched from a menu.  The active radio
-- indication in the menu is then directly determined from the active page of
-- the @GtkStack@.
-- 
-- An anti-example would be binding the @active-id@ property on a
-- <https://docs.gtk.org/gtk4/class.ComboBox.html `GtkComboBox`>. This is
-- because the state of the combo box itself is probably uninteresting and is
-- actually being used to control something else.
-- 
-- Another anti-example would be to bind to the @visible-child-name@
-- property of a <https://docs.gtk.org/gtk4/class.Stack.html `GtkStack`> if
-- this value is actually stored in t'GI.Gio.Objects.Settings.Settings'.  In that case, the
-- real source of the value is* t'GI.Gio.Objects.Settings.Settings'.  If you want
-- a t'GI.Gio.Interfaces.Action.Action' to control a setting stored in t'GI.Gio.Objects.Settings.Settings',
-- see 'GI.Gio.Objects.Settings.settingsCreateAction' instead, and possibly combine its
-- use with 'GI.Gio.Objects.Settings.settingsBind'.
-- 
-- /Since: 2.38/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.PropertyAction
    ( 

-- * Exported types
    PropertyAction(..)                      ,
    IsPropertyAction                        ,
    toPropertyAction                        ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [activate]("GI.Gio.Interfaces.Action#g:method:activate"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [changeState]("GI.Gio.Interfaces.Action#g:method:changeState"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getEnabled]("GI.Gio.Interfaces.Action#g:method:getEnabled"), [getName]("GI.Gio.Interfaces.Action#g:method:getName"), [getParameterType]("GI.Gio.Interfaces.Action#g:method:getParameterType"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getState]("GI.Gio.Interfaces.Action#g:method:getState"), [getStateHint]("GI.Gio.Interfaces.Action#g:method:getStateHint"), [getStateType]("GI.Gio.Interfaces.Action#g:method:getStateType").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolvePropertyActionMethod             ,
#endif

-- ** new #method:new#

    propertyActionNew                       ,




 -- * Properties


-- ** enabled #attr:enabled#
-- | If /@action@/ is currently enabled.
-- 
-- If the action is disabled then calls to 'GI.Gio.Interfaces.Action.actionActivate' and
-- 'GI.Gio.Interfaces.Action.actionChangeState' have no effect.
-- 
-- /Since: 2.38/

#if defined(ENABLE_OVERLOADING)
    PropertyActionEnabledPropertyInfo       ,
#endif
    getPropertyActionEnabled                ,
#if defined(ENABLE_OVERLOADING)
    propertyActionEnabled                   ,
#endif


-- ** invertBoolean #attr:invertBoolean#
-- | If 'P.True', the state of the action will be the negation of the
-- property value, provided the property is boolean.
-- 
-- /Since: 2.46/

#if defined(ENABLE_OVERLOADING)
    PropertyActionInvertBooleanPropertyInfo ,
#endif
    constructPropertyActionInvertBoolean    ,
    getPropertyActionInvertBoolean          ,
#if defined(ENABLE_OVERLOADING)
    propertyActionInvertBoolean             ,
#endif


-- ** name #attr:name#
-- | The name of the action.  This is mostly meaningful for identifying
-- the action once it has been added to a t'GI.Gio.Interfaces.ActionMap.ActionMap'.
-- 
-- /Since: 2.38/

#if defined(ENABLE_OVERLOADING)
    PropertyActionNamePropertyInfo          ,
#endif
    constructPropertyActionName             ,
    getPropertyActionName                   ,
#if defined(ENABLE_OVERLOADING)
    propertyActionName                      ,
#endif


-- ** object #attr:object#
-- | The object to wrap a property on.
-- 
-- The object must be a non-'P.Nothing' t'GI.GObject.Objects.Object.Object' with properties.
-- 
-- /Since: 2.38/

#if defined(ENABLE_OVERLOADING)
    PropertyActionObjectPropertyInfo        ,
#endif
    constructPropertyActionObject           ,
#if defined(ENABLE_OVERLOADING)
    propertyActionObject                    ,
#endif


-- ** parameterType #attr:parameterType#
-- | The type of the parameter that must be given when activating the
-- action.
-- 
-- /Since: 2.38/

#if defined(ENABLE_OVERLOADING)
    PropertyActionParameterTypePropertyInfo ,
#endif
    getPropertyActionParameterType          ,
#if defined(ENABLE_OVERLOADING)
    propertyActionParameterType             ,
#endif


-- ** propertyName #attr:propertyName#
-- | The name of the property to wrap on the object.
-- 
-- The property must exist on the passed-in object and it must be
-- readable and writable (and not construct-only).
-- 
-- /Since: 2.38/

#if defined(ENABLE_OVERLOADING)
    PropertyActionPropertyNamePropertyInfo  ,
#endif
    constructPropertyActionPropertyName     ,
#if defined(ENABLE_OVERLOADING)
    propertyActionPropertyName              ,
#endif


-- ** state #attr:state#
-- | The state of the action, or 'P.Nothing' if the action is stateless.
-- 
-- /Since: 2.38/

#if defined(ENABLE_OVERLOADING)
    PropertyActionStatePropertyInfo         ,
#endif
    getPropertyActionState                  ,
#if defined(ENABLE_OVERLOADING)
    propertyActionState                     ,
#endif


-- ** stateType #attr:stateType#
-- | The t'GI.GLib.Structs.VariantType.VariantType' of the state that the action has, or 'P.Nothing' if the
-- action is stateless.
-- 
-- /Since: 2.38/

#if defined(ENABLE_OVERLOADING)
    PropertyActionStateTypePropertyInfo     ,
#endif
    getPropertyActionStateType              ,
#if defined(ENABLE_OVERLOADING)
    propertyActionStateType                 ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Action as Gio.Action

#else
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Action as Gio.Action

#endif

-- | Memory-managed wrapper type.
newtype PropertyAction = PropertyAction (SP.ManagedPtr PropertyAction)
    deriving (Eq)

instance SP.ManagedPtrNewtype PropertyAction where
    toManagedPtr (PropertyAction p) = p

foreign import ccall "g_property_action_get_type"
    c_g_property_action_get_type :: IO B.Types.GType

instance B.Types.TypedObject PropertyAction where
    glibType = c_g_property_action_get_type

instance B.Types.GObject PropertyAction

-- | Type class for types which can be safely cast to `PropertyAction`, for instance with `toPropertyAction`.
class (SP.GObject o, O.IsDescendantOf PropertyAction o) => IsPropertyAction o
instance (SP.GObject o, O.IsDescendantOf PropertyAction o) => IsPropertyAction o

instance O.HasParentTypes PropertyAction
type instance O.ParentTypes PropertyAction = '[GObject.Object.Object, Gio.Action.Action]

-- | Cast to `PropertyAction`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toPropertyAction :: (MIO.MonadIO m, IsPropertyAction o) => o -> m PropertyAction
toPropertyAction = MIO.liftIO . B.ManagedPtr.unsafeCastTo PropertyAction

-- | Convert 'PropertyAction' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe PropertyAction) where
    gvalueGType_ = c_g_property_action_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr PropertyAction)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr PropertyAction)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject PropertyAction ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolvePropertyActionMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolvePropertyActionMethod "activate" o = Gio.Action.ActionActivateMethodInfo
    ResolvePropertyActionMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolvePropertyActionMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolvePropertyActionMethod "changeState" o = Gio.Action.ActionChangeStateMethodInfo
    ResolvePropertyActionMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolvePropertyActionMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolvePropertyActionMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolvePropertyActionMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolvePropertyActionMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolvePropertyActionMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolvePropertyActionMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolvePropertyActionMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolvePropertyActionMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolvePropertyActionMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolvePropertyActionMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolvePropertyActionMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolvePropertyActionMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolvePropertyActionMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolvePropertyActionMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolvePropertyActionMethod "getEnabled" o = Gio.Action.ActionGetEnabledMethodInfo
    ResolvePropertyActionMethod "getName" o = Gio.Action.ActionGetNameMethodInfo
    ResolvePropertyActionMethod "getParameterType" o = Gio.Action.ActionGetParameterTypeMethodInfo
    ResolvePropertyActionMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolvePropertyActionMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolvePropertyActionMethod "getState" o = Gio.Action.ActionGetStateMethodInfo
    ResolvePropertyActionMethod "getStateHint" o = Gio.Action.ActionGetStateHintMethodInfo
    ResolvePropertyActionMethod "getStateType" o = Gio.Action.ActionGetStateTypeMethodInfo
    ResolvePropertyActionMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolvePropertyActionMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolvePropertyActionMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolvePropertyActionMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolvePropertyActionMethod t PropertyAction, O.OverloadedMethod info PropertyAction p) => OL.IsLabel t (PropertyAction -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolvePropertyActionMethod t PropertyAction, O.OverloadedMethod info PropertyAction p, R.HasField t PropertyAction p) => R.HasField t PropertyAction p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolvePropertyActionMethod t PropertyAction, O.OverloadedMethodInfo info PropertyAction) => OL.IsLabel t (O.MethodProxy info PropertyAction) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

--- XXX Duplicated object with different types:
  --- Name {namespace = "Gio", name = "PropertyAction"} -> Property {propName = "enabled", propType = TBasicType TBoolean, propFlags = [PropertyReadable], propReadNullable = Nothing, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "If @action is currently enabled.\n\nIf the action is disabled then calls to g_action_activate() and\ng_action_change_state() have no effect.", sinceVersion = Just "2.38"}, propDeprecated = Nothing}
  --- Name {namespace = "Gio", name = "Action"} -> Property {propName = "enabled", propType = TBasicType TBoolean, propFlags = [PropertyReadable], propReadNullable = Just False, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "If @action is currently enabled.\n\nIf the action is disabled then calls to [method@Gio.Action.activate] and\n[method@Gio.Action.change_state] have no effect.", sinceVersion = Just "2.28"}, propDeprecated = Nothing}
--- XXX Duplicated object with different types:
  --- Name {namespace = "Gio", name = "PropertyAction"} -> Property {propName = "name", propType = TBasicType TUTF8, propFlags = [PropertyReadable,PropertyWritable,PropertyConstructOnly], propReadNullable = Nothing, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "The name of the action.  This is mostly meaningful for identifying\nthe action once it has been added to a #GActionMap.", sinceVersion = Just "2.38"}, propDeprecated = Nothing}
  --- Name {namespace = "Gio", name = "Action"} -> Property {propName = "name", propType = TBasicType TUTF8, propFlags = [PropertyReadable], propReadNullable = Just False, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "The name of the action.  This is mostly meaningful for identifying\nthe action once it has been added to a [type@Gio.ActionGroup]. It is immutable.", sinceVersion = Just "2.28"}, propDeprecated = Nothing}
--- XXX Duplicated object with different types:
  --- Name {namespace = "Gio", name = "PropertyAction"} -> Property {propName = "parameter-type", propType = TInterface (Name {namespace = "GLib", name = "VariantType"}), propFlags = [PropertyReadable], propReadNullable = Nothing, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "The type of the parameter that must be given when activating the\naction.", sinceVersion = Just "2.38"}, propDeprecated = Nothing}
  --- Name {namespace = "Gio", name = "Action"} -> Property {propName = "parameter-type", propType = TInterface (Name {namespace = "GLib", name = "VariantType"}), propFlags = [PropertyReadable], propReadNullable = Just True, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "The type of the parameter that must be given when activating the\naction. This is immutable, and may be `NULL` if no parameter is needed when\nactivating the action.", sinceVersion = Just "2.28"}, propDeprecated = Nothing}
--- XXX Duplicated object with different types:
  --- Name {namespace = "Gio", name = "PropertyAction"} -> Property {propName = "state", propType = TVariant, propFlags = [PropertyReadable], propReadNullable = Nothing, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "The state of the action, or %NULL if the action is stateless.", sinceVersion = Just "2.38"}, propDeprecated = Nothing}
  --- Name {namespace = "Gio", name = "Action"} -> Property {propName = "state", propType = TVariant, propFlags = [PropertyReadable], propReadNullable = Nothing, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "The state of the action, or `NULL` if the action is stateless.", sinceVersion = Just "2.28"}, propDeprecated = Nothing}
--- XXX Duplicated object with different types:
  --- Name {namespace = "Gio", name = "PropertyAction"} -> Property {propName = "state-type", propType = TInterface (Name {namespace = "GLib", name = "VariantType"}), propFlags = [PropertyReadable], propReadNullable = Nothing, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "The #GVariantType of the state that the action has, or %NULL if the\naction is stateless.", sinceVersion = Just "2.38"}, propDeprecated = Nothing}
  --- Name {namespace = "Gio", name = "Action"} -> Property {propName = "state-type", propType = TInterface (Name {namespace = "GLib", name = "VariantType"}), propFlags = [PropertyReadable], propReadNullable = Just True, propWriteNullable = Nothing, propTransfer = TransferNothing, propDoc = Documentation {rawDocText = Just "The [type@GLib.VariantType] of the state that the action has, or `NULL` if the\naction is stateless. This is immutable.", sinceVersion = Just "2.28"}, propDeprecated = Nothing}
-- VVV Prop "enabled"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@enabled@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' propertyAction #enabled
-- @
getPropertyActionEnabled :: (MonadIO m, IsPropertyAction o) => o -> m Bool
getPropertyActionEnabled obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "enabled"

#if defined(ENABLE_OVERLOADING)
data PropertyActionEnabledPropertyInfo
instance AttrInfo PropertyActionEnabledPropertyInfo where
    type AttrAllowedOps PropertyActionEnabledPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint PropertyActionEnabledPropertyInfo = IsPropertyAction
    type AttrSetTypeConstraint PropertyActionEnabledPropertyInfo = (~) ()
    type AttrTransferTypeConstraint PropertyActionEnabledPropertyInfo = (~) ()
    type AttrTransferType PropertyActionEnabledPropertyInfo = ()
    type AttrGetType PropertyActionEnabledPropertyInfo = Bool
    type AttrLabel PropertyActionEnabledPropertyInfo = "enabled"
    type AttrOrigin PropertyActionEnabledPropertyInfo = PropertyAction
    attrGet = getPropertyActionEnabled
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.PropertyAction.enabled"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-PropertyAction.html#g:attr:enabled"
        })
#endif

-- VVV Prop "invert-boolean"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@invert-boolean@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' propertyAction #invertBoolean
-- @
getPropertyActionInvertBoolean :: (MonadIO m, IsPropertyAction o) => o -> m Bool
getPropertyActionInvertBoolean obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "invert-boolean"

-- | Construct a `GValueConstruct` with valid value for the “@invert-boolean@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructPropertyActionInvertBoolean :: (IsPropertyAction o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructPropertyActionInvertBoolean val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "invert-boolean" val

#if defined(ENABLE_OVERLOADING)
data PropertyActionInvertBooleanPropertyInfo
instance AttrInfo PropertyActionInvertBooleanPropertyInfo where
    type AttrAllowedOps PropertyActionInvertBooleanPropertyInfo = '[ 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint PropertyActionInvertBooleanPropertyInfo = IsPropertyAction
    type AttrSetTypeConstraint PropertyActionInvertBooleanPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint PropertyActionInvertBooleanPropertyInfo = (~) Bool
    type AttrTransferType PropertyActionInvertBooleanPropertyInfo = Bool
    type AttrGetType PropertyActionInvertBooleanPropertyInfo = Bool
    type AttrLabel PropertyActionInvertBooleanPropertyInfo = "invert-boolean"
    type AttrOrigin PropertyActionInvertBooleanPropertyInfo = PropertyAction
    attrGet = getPropertyActionInvertBoolean
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructPropertyActionInvertBoolean
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.PropertyAction.invertBoolean"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-PropertyAction.html#g:attr:invertBoolean"
        })
#endif

-- VVV Prop "name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' propertyAction #name
-- @
getPropertyActionName :: (MonadIO m, IsPropertyAction o) => o -> m (Maybe T.Text)
getPropertyActionName obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "name"

-- | Construct a `GValueConstruct` with valid value for the “@name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructPropertyActionName :: (IsPropertyAction o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructPropertyActionName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "name" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data PropertyActionNamePropertyInfo
instance AttrInfo PropertyActionNamePropertyInfo where
    type AttrAllowedOps PropertyActionNamePropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint PropertyActionNamePropertyInfo = IsPropertyAction
    type AttrSetTypeConstraint PropertyActionNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint PropertyActionNamePropertyInfo = (~) T.Text
    type AttrTransferType PropertyActionNamePropertyInfo = T.Text
    type AttrGetType PropertyActionNamePropertyInfo = (Maybe T.Text)
    type AttrLabel PropertyActionNamePropertyInfo = "name"
    type AttrOrigin PropertyActionNamePropertyInfo = PropertyAction
    attrGet = getPropertyActionName
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructPropertyActionName
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.PropertyAction.name"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-PropertyAction.html#g:attr:name"
        })
#endif

-- VVV Prop "object"
   -- Type: TInterface (Name {namespace = "GObject", name = "Object"})
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a `GValueConstruct` with valid value for the “@object@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructPropertyActionObject :: (IsPropertyAction o, MIO.MonadIO m, GObject.Object.IsObject a) => a -> m (GValueConstruct o)
constructPropertyActionObject val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "object" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data PropertyActionObjectPropertyInfo
instance AttrInfo PropertyActionObjectPropertyInfo where
    type AttrAllowedOps PropertyActionObjectPropertyInfo = '[ 'AttrConstruct, 'AttrClear]
    type AttrBaseTypeConstraint PropertyActionObjectPropertyInfo = IsPropertyAction
    type AttrSetTypeConstraint PropertyActionObjectPropertyInfo = GObject.Object.IsObject
    type AttrTransferTypeConstraint PropertyActionObjectPropertyInfo = GObject.Object.IsObject
    type AttrTransferType PropertyActionObjectPropertyInfo = GObject.Object.Object
    type AttrGetType PropertyActionObjectPropertyInfo = ()
    type AttrLabel PropertyActionObjectPropertyInfo = "object"
    type AttrOrigin PropertyActionObjectPropertyInfo = PropertyAction
    attrGet = undefined
    attrSet = undefined
    attrTransfer _ v = do
        unsafeCastTo GObject.Object.Object v
    attrConstruct = constructPropertyActionObject
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.PropertyAction.object"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-PropertyAction.html#g:attr:object"
        })
#endif

-- VVV Prop "parameter-type"
   -- Type: TInterface (Name {namespace = "GLib", name = "VariantType"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@parameter-type@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' propertyAction #parameterType
-- @
getPropertyActionParameterType :: (MonadIO m, IsPropertyAction o) => o -> m (Maybe GLib.VariantType.VariantType)
getPropertyActionParameterType obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "parameter-type" GLib.VariantType.VariantType

#if defined(ENABLE_OVERLOADING)
data PropertyActionParameterTypePropertyInfo
instance AttrInfo PropertyActionParameterTypePropertyInfo where
    type AttrAllowedOps PropertyActionParameterTypePropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint PropertyActionParameterTypePropertyInfo = IsPropertyAction
    type AttrSetTypeConstraint PropertyActionParameterTypePropertyInfo = (~) ()
    type AttrTransferTypeConstraint PropertyActionParameterTypePropertyInfo = (~) ()
    type AttrTransferType PropertyActionParameterTypePropertyInfo = ()
    type AttrGetType PropertyActionParameterTypePropertyInfo = (Maybe GLib.VariantType.VariantType)
    type AttrLabel PropertyActionParameterTypePropertyInfo = "parameter-type"
    type AttrOrigin PropertyActionParameterTypePropertyInfo = PropertyAction
    attrGet = getPropertyActionParameterType
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.PropertyAction.parameterType"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-PropertyAction.html#g:attr:parameterType"
        })
#endif

-- VVV Prop "property-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a `GValueConstruct` with valid value for the “@property-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructPropertyActionPropertyName :: (IsPropertyAction o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructPropertyActionPropertyName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "property-name" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data PropertyActionPropertyNamePropertyInfo
instance AttrInfo PropertyActionPropertyNamePropertyInfo where
    type AttrAllowedOps PropertyActionPropertyNamePropertyInfo = '[ 'AttrConstruct, 'AttrClear]
    type AttrBaseTypeConstraint PropertyActionPropertyNamePropertyInfo = IsPropertyAction
    type AttrSetTypeConstraint PropertyActionPropertyNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint PropertyActionPropertyNamePropertyInfo = (~) T.Text
    type AttrTransferType PropertyActionPropertyNamePropertyInfo = T.Text
    type AttrGetType PropertyActionPropertyNamePropertyInfo = ()
    type AttrLabel PropertyActionPropertyNamePropertyInfo = "property-name"
    type AttrOrigin PropertyActionPropertyNamePropertyInfo = PropertyAction
    attrGet = undefined
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructPropertyActionPropertyName
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.PropertyAction.propertyName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-PropertyAction.html#g:attr:propertyName"
        })
#endif

-- VVV Prop "state"
   -- Type: TVariant
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@state@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' propertyAction #state
-- @
getPropertyActionState :: (MonadIO m, IsPropertyAction o) => o -> m (Maybe GVariant)
getPropertyActionState obj = MIO.liftIO $ B.Properties.getObjectPropertyVariant obj "state"

#if defined(ENABLE_OVERLOADING)
data PropertyActionStatePropertyInfo
instance AttrInfo PropertyActionStatePropertyInfo where
    type AttrAllowedOps PropertyActionStatePropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint PropertyActionStatePropertyInfo = IsPropertyAction
    type AttrSetTypeConstraint PropertyActionStatePropertyInfo = (~) ()
    type AttrTransferTypeConstraint PropertyActionStatePropertyInfo = (~) ()
    type AttrTransferType PropertyActionStatePropertyInfo = ()
    type AttrGetType PropertyActionStatePropertyInfo = (Maybe GVariant)
    type AttrLabel PropertyActionStatePropertyInfo = "state"
    type AttrOrigin PropertyActionStatePropertyInfo = PropertyAction
    attrGet = getPropertyActionState
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.PropertyAction.state"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-PropertyAction.html#g:attr:state"
        })
#endif

-- VVV Prop "state-type"
   -- Type: TInterface (Name {namespace = "GLib", name = "VariantType"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@state-type@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' propertyAction #stateType
-- @
getPropertyActionStateType :: (MonadIO m, IsPropertyAction o) => o -> m (Maybe GLib.VariantType.VariantType)
getPropertyActionStateType obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "state-type" GLib.VariantType.VariantType

#if defined(ENABLE_OVERLOADING)
data PropertyActionStateTypePropertyInfo
instance AttrInfo PropertyActionStateTypePropertyInfo where
    type AttrAllowedOps PropertyActionStateTypePropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint PropertyActionStateTypePropertyInfo = IsPropertyAction
    type AttrSetTypeConstraint PropertyActionStateTypePropertyInfo = (~) ()
    type AttrTransferTypeConstraint PropertyActionStateTypePropertyInfo = (~) ()
    type AttrTransferType PropertyActionStateTypePropertyInfo = ()
    type AttrGetType PropertyActionStateTypePropertyInfo = (Maybe GLib.VariantType.VariantType)
    type AttrLabel PropertyActionStateTypePropertyInfo = "state-type"
    type AttrOrigin PropertyActionStateTypePropertyInfo = PropertyAction
    attrGet = getPropertyActionStateType
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.PropertyAction.stateType"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-PropertyAction.html#g:attr:stateType"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList PropertyAction
type instance O.AttributeList PropertyAction = PropertyActionAttributeList
type PropertyActionAttributeList = ('[ '("enabled", PropertyActionEnabledPropertyInfo), '("invertBoolean", PropertyActionInvertBooleanPropertyInfo), '("name", PropertyActionNamePropertyInfo), '("object", PropertyActionObjectPropertyInfo), '("parameterType", PropertyActionParameterTypePropertyInfo), '("propertyName", PropertyActionPropertyNamePropertyInfo), '("state", PropertyActionStatePropertyInfo), '("stateType", PropertyActionStateTypePropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
propertyActionEnabled :: AttrLabelProxy "enabled"
propertyActionEnabled = AttrLabelProxy

propertyActionInvertBoolean :: AttrLabelProxy "invertBoolean"
propertyActionInvertBoolean = AttrLabelProxy

propertyActionName :: AttrLabelProxy "name"
propertyActionName = AttrLabelProxy

propertyActionObject :: AttrLabelProxy "object"
propertyActionObject = AttrLabelProxy

propertyActionParameterType :: AttrLabelProxy "parameterType"
propertyActionParameterType = AttrLabelProxy

propertyActionPropertyName :: AttrLabelProxy "propertyName"
propertyActionPropertyName = AttrLabelProxy

propertyActionState :: AttrLabelProxy "state"
propertyActionState = AttrLabelProxy

propertyActionStateType :: AttrLabelProxy "stateType"
propertyActionStateType = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList PropertyAction = PropertyActionSignalList
type PropertyActionSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method PropertyAction::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to create"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the object that has the property\n  to wrap"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "PropertyAction" })
-- throws : False
-- Skip return : False

foreign import ccall "g_property_action_new" g_property_action_new :: 
    CString ->                              -- name : TBasicType TUTF8
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property_name : TBasicType TUTF8
    IO (Ptr PropertyAction)

-- | Creates a t'GI.Gio.Interfaces.Action.Action' corresponding to the value of property
-- /@propertyName@/ on /@object@/.
-- 
-- The property must be existent and readable and writable (and not
-- construct-only).
-- 
-- This function takes a reference on /@object@/ and doesn\'t release it
-- until the action is destroyed.
-- 
-- /Since: 2.38/
propertyActionNew ::
    (B.CallStack.HasCallStack, MonadIO m, GObject.Object.IsObject a) =>
    T.Text
    -- ^ /@name@/: the name of the action to create
    -> a
    -- ^ /@object@/: the object that has the property
    --   to wrap
    -> T.Text
    -- ^ /@propertyName@/: the name of the property
    -> m PropertyAction
    -- ^ __Returns:__ a new t'GI.Gio.Objects.PropertyAction.PropertyAction'
propertyActionNew name object propertyName = liftIO $ do
    name' <- textToCString name
    object' <- unsafeManagedPtrCastPtr object
    propertyName' <- textToCString propertyName
    result <- g_property_action_new name' object' propertyName'
    checkUnexpectedReturnNULL "propertyActionNew" result
    result' <- (wrapObject PropertyAction) result
    touchManagedPtr object
    freeMem name'
    freeMem propertyName'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif


