{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GMountOperation@ provides a mechanism for interacting with the user.
-- It can be used for authenticating mountable operations, such as loop
-- mounting files, hard drive partitions or server locations. It can
-- also be used to ask the user questions or show a list of applications
-- preventing unmount or eject operations from completing.
-- 
-- Note that @GMountOperation@ is used for more than just t'GI.Gio.Interfaces.Mount.Mount'
-- objects – for example it is also used in 'GI.Gio.Interfaces.Drive.driveStart' and
-- 'GI.Gio.Interfaces.Drive.driveStop'.
-- 
-- Users should instantiate a subclass of this that implements all the
-- various callbacks to show the required dialogs, such as
-- <https://docs.gtk.org/gtk4/class.MountOperation.html `GtkMountOperation`>.
-- If no user interaction is desired (for example when automounting
-- filesystems at login time), usually @NULL@ can be passed, see each method
-- taking a @GMountOperation@ for details.
-- 
-- Throughout the API, the term ‘TCRYPT’ is used to mean ‘compatible with TrueCrypt and VeraCrypt’.
-- <https://en.wikipedia.org/wiki/TrueCrypt TrueCrypt> is a discontinued system for
-- encrypting file containers, partitions or whole disks, typically used with Windows.
-- <https://www.veracrypt.fr/ VeraCrypt> is a maintained fork of TrueCrypt with various
-- improvements and auditing fixes.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.MountOperation
    ( 

-- * Exported types
    MountOperation(..)                      ,
    IsMountOperation                        ,
    toMountOperation                        ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [reply]("GI.Gio.Objects.MountOperation#g:method:reply"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAnonymous]("GI.Gio.Objects.MountOperation#g:method:getAnonymous"), [getChoice]("GI.Gio.Objects.MountOperation#g:method:getChoice"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDomain]("GI.Gio.Objects.MountOperation#g:method:getDomain"), [getIsTcryptHiddenVolume]("GI.Gio.Objects.MountOperation#g:method:getIsTcryptHiddenVolume"), [getIsTcryptSystemVolume]("GI.Gio.Objects.MountOperation#g:method:getIsTcryptSystemVolume"), [getPassword]("GI.Gio.Objects.MountOperation#g:method:getPassword"), [getPasswordSave]("GI.Gio.Objects.MountOperation#g:method:getPasswordSave"), [getPim]("GI.Gio.Objects.MountOperation#g:method:getPim"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getUsername]("GI.Gio.Objects.MountOperation#g:method:getUsername").
-- 
-- ==== Setters
-- [setAnonymous]("GI.Gio.Objects.MountOperation#g:method:setAnonymous"), [setChoice]("GI.Gio.Objects.MountOperation#g:method:setChoice"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDomain]("GI.Gio.Objects.MountOperation#g:method:setDomain"), [setIsTcryptHiddenVolume]("GI.Gio.Objects.MountOperation#g:method:setIsTcryptHiddenVolume"), [setIsTcryptSystemVolume]("GI.Gio.Objects.MountOperation#g:method:setIsTcryptSystemVolume"), [setPassword]("GI.Gio.Objects.MountOperation#g:method:setPassword"), [setPasswordSave]("GI.Gio.Objects.MountOperation#g:method:setPasswordSave"), [setPim]("GI.Gio.Objects.MountOperation#g:method:setPim"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setUsername]("GI.Gio.Objects.MountOperation#g:method:setUsername").

#if defined(ENABLE_OVERLOADING)
    ResolveMountOperationMethod             ,
#endif

-- ** getAnonymous #method:getAnonymous#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetAnonymousMethodInfo    ,
#endif
    mountOperationGetAnonymous              ,


-- ** getChoice #method:getChoice#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetChoiceMethodInfo       ,
#endif
    mountOperationGetChoice                 ,


-- ** getDomain #method:getDomain#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetDomainMethodInfo       ,
#endif
    mountOperationGetDomain                 ,


-- ** getIsTcryptHiddenVolume #method:getIsTcryptHiddenVolume#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetIsTcryptHiddenVolumeMethodInfo,
#endif
    mountOperationGetIsTcryptHiddenVolume   ,


-- ** getIsTcryptSystemVolume #method:getIsTcryptSystemVolume#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetIsTcryptSystemVolumeMethodInfo,
#endif
    mountOperationGetIsTcryptSystemVolume   ,


-- ** getPassword #method:getPassword#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetPasswordMethodInfo     ,
#endif
    mountOperationGetPassword               ,


-- ** getPasswordSave #method:getPasswordSave#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetPasswordSaveMethodInfo ,
#endif
    mountOperationGetPasswordSave           ,


-- ** getPim #method:getPim#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetPimMethodInfo          ,
#endif
    mountOperationGetPim                    ,


-- ** getUsername #method:getUsername#

#if defined(ENABLE_OVERLOADING)
    MountOperationGetUsernameMethodInfo     ,
#endif
    mountOperationGetUsername               ,


-- ** new #method:new#

    mountOperationNew                       ,


-- ** reply #method:reply#

#if defined(ENABLE_OVERLOADING)
    MountOperationReplyMethodInfo           ,
#endif
    mountOperationReply                     ,


-- ** setAnonymous #method:setAnonymous#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetAnonymousMethodInfo    ,
#endif
    mountOperationSetAnonymous              ,


-- ** setChoice #method:setChoice#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetChoiceMethodInfo       ,
#endif
    mountOperationSetChoice                 ,


-- ** setDomain #method:setDomain#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetDomainMethodInfo       ,
#endif
    mountOperationSetDomain                 ,


-- ** setIsTcryptHiddenVolume #method:setIsTcryptHiddenVolume#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetIsTcryptHiddenVolumeMethodInfo,
#endif
    mountOperationSetIsTcryptHiddenVolume   ,


-- ** setIsTcryptSystemVolume #method:setIsTcryptSystemVolume#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetIsTcryptSystemVolumeMethodInfo,
#endif
    mountOperationSetIsTcryptSystemVolume   ,


-- ** setPassword #method:setPassword#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetPasswordMethodInfo     ,
#endif
    mountOperationSetPassword               ,


-- ** setPasswordSave #method:setPasswordSave#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetPasswordSaveMethodInfo ,
#endif
    mountOperationSetPasswordSave           ,


-- ** setPim #method:setPim#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetPimMethodInfo          ,
#endif
    mountOperationSetPim                    ,


-- ** setUsername #method:setUsername#

#if defined(ENABLE_OVERLOADING)
    MountOperationSetUsernameMethodInfo     ,
#endif
    mountOperationSetUsername               ,




 -- * Properties


-- ** anonymous #attr:anonymous#
-- | Whether to use an anonymous user when authenticating.

#if defined(ENABLE_OVERLOADING)
    MountOperationAnonymousPropertyInfo     ,
#endif
    constructMountOperationAnonymous        ,
    getMountOperationAnonymous              ,
#if defined(ENABLE_OVERLOADING)
    mountOperationAnonymous                 ,
#endif
    setMountOperationAnonymous              ,


-- ** choice #attr:choice#
-- | The index of the user\'s choice when a question is asked during the
-- mount operation. See the [MountOperation::askQuestion]("GI.Gio.Objects.MountOperation#g:signal:askQuestion") signal.

#if defined(ENABLE_OVERLOADING)
    MountOperationChoicePropertyInfo        ,
#endif
    constructMountOperationChoice           ,
    getMountOperationChoice                 ,
#if defined(ENABLE_OVERLOADING)
    mountOperationChoice                    ,
#endif
    setMountOperationChoice                 ,


-- ** domain #attr:domain#
-- | The domain to use for the mount operation.

#if defined(ENABLE_OVERLOADING)
    MountOperationDomainPropertyInfo        ,
#endif
    clearMountOperationDomain               ,
    constructMountOperationDomain           ,
    getMountOperationDomain                 ,
#if defined(ENABLE_OVERLOADING)
    mountOperationDomain                    ,
#endif
    setMountOperationDomain                 ,


-- ** isTcryptHiddenVolume #attr:isTcryptHiddenVolume#
-- | Whether the device to be unlocked is a TCRYPT hidden volume.
-- See <https://www.veracrypt.fr/en/Hidden%20Volume.html the VeraCrypt documentation>.
-- 
-- /Since: 2.58/

#if defined(ENABLE_OVERLOADING)
    MountOperationIsTcryptHiddenVolumePropertyInfo,
#endif
    constructMountOperationIsTcryptHiddenVolume,
    getMountOperationIsTcryptHiddenVolume   ,
#if defined(ENABLE_OVERLOADING)
    mountOperationIsTcryptHiddenVolume      ,
#endif
    setMountOperationIsTcryptHiddenVolume   ,


-- ** isTcryptSystemVolume #attr:isTcryptSystemVolume#
-- | Whether the device to be unlocked is a TCRYPT system volume.
-- In this context, a system volume is a volume with a bootloader
-- and operating system installed. This is only supported for Windows
-- operating systems. For further documentation, see
-- <https://www.veracrypt.fr/en/System%20Encryption.html the VeraCrypt documentation>.
-- 
-- /Since: 2.58/

#if defined(ENABLE_OVERLOADING)
    MountOperationIsTcryptSystemVolumePropertyInfo,
#endif
    constructMountOperationIsTcryptSystemVolume,
    getMountOperationIsTcryptSystemVolume   ,
#if defined(ENABLE_OVERLOADING)
    mountOperationIsTcryptSystemVolume      ,
#endif
    setMountOperationIsTcryptSystemVolume   ,


-- ** password #attr:password#
-- | The password that is used for authentication when carrying out
-- the mount operation.

#if defined(ENABLE_OVERLOADING)
    MountOperationPasswordPropertyInfo      ,
#endif
    clearMountOperationPassword             ,
    constructMountOperationPassword         ,
    getMountOperationPassword               ,
#if defined(ENABLE_OVERLOADING)
    mountOperationPassword                  ,
#endif
    setMountOperationPassword               ,


-- ** passwordSave #attr:passwordSave#
-- | Determines if and how the password information should be saved.

#if defined(ENABLE_OVERLOADING)
    MountOperationPasswordSavePropertyInfo  ,
#endif
    constructMountOperationPasswordSave     ,
    getMountOperationPasswordSave           ,
#if defined(ENABLE_OVERLOADING)
    mountOperationPasswordSave              ,
#endif
    setMountOperationPasswordSave           ,


-- ** pim #attr:pim#
-- | The VeraCrypt PIM value, when unlocking a VeraCrypt volume. See
-- <https://www.veracrypt.fr/en/Personal%20Iterations%20Multiplier%20(PIM the VeraCrypt documentation>.html).
-- 
-- /Since: 2.58/

#if defined(ENABLE_OVERLOADING)
    MountOperationPimPropertyInfo           ,
#endif
    constructMountOperationPim              ,
    getMountOperationPim                    ,
#if defined(ENABLE_OVERLOADING)
    mountOperationPim                       ,
#endif
    setMountOperationPim                    ,


-- ** username #attr:username#
-- | The user name that is used for authentication when carrying out
-- the mount operation.

#if defined(ENABLE_OVERLOADING)
    MountOperationUsernamePropertyInfo      ,
#endif
    clearMountOperationUsername             ,
    constructMountOperationUsername         ,
    getMountOperationUsername               ,
#if defined(ENABLE_OVERLOADING)
    mountOperationUsername                  ,
#endif
    setMountOperationUsername               ,




 -- * Signals


-- ** aborted #signal:aborted#

    MountOperationAbortedCallback           ,
#if defined(ENABLE_OVERLOADING)
    MountOperationAbortedSignalInfo         ,
#endif
    afterMountOperationAborted              ,
    onMountOperationAborted                 ,


-- ** askPassword #signal:askPassword#

    MountOperationAskPasswordCallback       ,
#if defined(ENABLE_OVERLOADING)
    MountOperationAskPasswordSignalInfo     ,
#endif
    afterMountOperationAskPassword          ,
    onMountOperationAskPassword             ,


-- ** askQuestion #signal:askQuestion#

    MountOperationAskQuestionCallback       ,
#if defined(ENABLE_OVERLOADING)
    MountOperationAskQuestionSignalInfo     ,
#endif
    afterMountOperationAskQuestion          ,
    onMountOperationAskQuestion             ,


-- ** reply #signal:reply#

    MountOperationReplyCallback             ,
#if defined(ENABLE_OVERLOADING)
    MountOperationReplySignalInfo           ,
#endif
    afterMountOperationReply                ,
    onMountOperationReply                   ,


-- ** showProcesses #signal:showProcesses#

    MountOperationShowProcessesCallback     ,
#if defined(ENABLE_OVERLOADING)
    MountOperationShowProcessesSignalInfo   ,
#endif
    afterMountOperationShowProcesses        ,
    onMountOperationShowProcesses           ,


-- ** showUnmountProgress #signal:showUnmountProgress#

    MountOperationShowUnmountProgressCallback,
#if defined(ENABLE_OVERLOADING)
    MountOperationShowUnmountProgressSignalInfo,
#endif
    afterMountOperationShowUnmountProgress  ,
    onMountOperationShowUnmountProgress     ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags

#endif

-- | Memory-managed wrapper type.
newtype MountOperation = MountOperation (SP.ManagedPtr MountOperation)
    deriving (Eq)

instance SP.ManagedPtrNewtype MountOperation where
    toManagedPtr (MountOperation p) = p

foreign import ccall "g_mount_operation_get_type"
    c_g_mount_operation_get_type :: IO B.Types.GType

instance B.Types.TypedObject MountOperation where
    glibType = c_g_mount_operation_get_type

instance B.Types.GObject MountOperation

-- | Type class for types which can be safely cast to `MountOperation`, for instance with `toMountOperation`.
class (SP.GObject o, O.IsDescendantOf MountOperation o) => IsMountOperation o
instance (SP.GObject o, O.IsDescendantOf MountOperation o) => IsMountOperation o

instance O.HasParentTypes MountOperation
type instance O.ParentTypes MountOperation = '[GObject.Object.Object]

-- | Cast to `MountOperation`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toMountOperation :: (MIO.MonadIO m, IsMountOperation o) => o -> m MountOperation
toMountOperation = MIO.liftIO . B.ManagedPtr.unsafeCastTo MountOperation

-- | Convert 'MountOperation' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe MountOperation) where
    gvalueGType_ = c_g_mount_operation_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr MountOperation)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr MountOperation)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject MountOperation ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveMountOperationMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveMountOperationMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveMountOperationMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveMountOperationMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveMountOperationMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveMountOperationMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveMountOperationMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveMountOperationMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveMountOperationMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveMountOperationMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveMountOperationMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveMountOperationMethod "reply" o = MountOperationReplyMethodInfo
    ResolveMountOperationMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveMountOperationMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveMountOperationMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveMountOperationMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveMountOperationMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveMountOperationMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveMountOperationMethod "getAnonymous" o = MountOperationGetAnonymousMethodInfo
    ResolveMountOperationMethod "getChoice" o = MountOperationGetChoiceMethodInfo
    ResolveMountOperationMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveMountOperationMethod "getDomain" o = MountOperationGetDomainMethodInfo
    ResolveMountOperationMethod "getIsTcryptHiddenVolume" o = MountOperationGetIsTcryptHiddenVolumeMethodInfo
    ResolveMountOperationMethod "getIsTcryptSystemVolume" o = MountOperationGetIsTcryptSystemVolumeMethodInfo
    ResolveMountOperationMethod "getPassword" o = MountOperationGetPasswordMethodInfo
    ResolveMountOperationMethod "getPasswordSave" o = MountOperationGetPasswordSaveMethodInfo
    ResolveMountOperationMethod "getPim" o = MountOperationGetPimMethodInfo
    ResolveMountOperationMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveMountOperationMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveMountOperationMethod "getUsername" o = MountOperationGetUsernameMethodInfo
    ResolveMountOperationMethod "setAnonymous" o = MountOperationSetAnonymousMethodInfo
    ResolveMountOperationMethod "setChoice" o = MountOperationSetChoiceMethodInfo
    ResolveMountOperationMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveMountOperationMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveMountOperationMethod "setDomain" o = MountOperationSetDomainMethodInfo
    ResolveMountOperationMethod "setIsTcryptHiddenVolume" o = MountOperationSetIsTcryptHiddenVolumeMethodInfo
    ResolveMountOperationMethod "setIsTcryptSystemVolume" o = MountOperationSetIsTcryptSystemVolumeMethodInfo
    ResolveMountOperationMethod "setPassword" o = MountOperationSetPasswordMethodInfo
    ResolveMountOperationMethod "setPasswordSave" o = MountOperationSetPasswordSaveMethodInfo
    ResolveMountOperationMethod "setPim" o = MountOperationSetPimMethodInfo
    ResolveMountOperationMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveMountOperationMethod "setUsername" o = MountOperationSetUsernameMethodInfo
    ResolveMountOperationMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveMountOperationMethod t MountOperation, O.OverloadedMethod info MountOperation p) => OL.IsLabel t (MountOperation -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveMountOperationMethod t MountOperation, O.OverloadedMethod info MountOperation p, R.HasField t MountOperation p) => R.HasField t MountOperation p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveMountOperationMethod t MountOperation, O.OverloadedMethodInfo info MountOperation) => OL.IsLabel t (O.MethodProxy info MountOperation) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal MountOperation::aborted
-- | Emitted by the backend when e.g. a device becomes unavailable
-- while a mount operation is in progress.
-- 
-- Implementations of GMountOperation should handle this signal
-- by dismissing open password dialogs.
-- 
-- /Since: 2.20/
type MountOperationAbortedCallback =
    IO ()

type C_MountOperationAbortedCallback =
    Ptr MountOperation ->                   -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_MountOperationAbortedCallback`.
foreign import ccall "wrapper"
    mk_MountOperationAbortedCallback :: C_MountOperationAbortedCallback -> IO (FunPtr C_MountOperationAbortedCallback)

wrap_MountOperationAbortedCallback :: 
    GObject a => (a -> MountOperationAbortedCallback) ->
    C_MountOperationAbortedCallback
wrap_MountOperationAbortedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [aborted](#signal:aborted) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' mountOperation #aborted callback
-- @
-- 
-- 
onMountOperationAborted :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationAbortedCallback) -> m SignalHandlerId
onMountOperationAborted obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationAbortedCallback wrapped
    wrapped'' <- mk_MountOperationAbortedCallback wrapped'
    connectSignalFunPtr obj "aborted" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [aborted](#signal:aborted) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' mountOperation #aborted callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterMountOperationAborted :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationAbortedCallback) -> m SignalHandlerId
afterMountOperationAborted obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationAbortedCallback wrapped
    wrapped'' <- mk_MountOperationAbortedCallback wrapped'
    connectSignalFunPtr obj "aborted" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data MountOperationAbortedSignalInfo
instance SignalInfo MountOperationAbortedSignalInfo where
    type HaskellCallbackType MountOperationAbortedSignalInfo = MountOperationAbortedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_MountOperationAbortedCallback cb
        cb'' <- mk_MountOperationAbortedCallback cb'
        connectSignalFunPtr obj "aborted" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation::aborted"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:signal:aborted"})

#endif

-- signal MountOperation::ask-password
-- | Emitted when a mount operation asks the user for a password.
-- 
-- If the message contains a line break, the first line should be
-- presented as a heading. For example, it may be used as the
-- primary text in a @/GtkMessageDialog/@.
type MountOperationAskPasswordCallback =
    T.Text
    -- ^ /@message@/: string containing a message to display to the user.
    -> T.Text
    -- ^ /@defaultUser@/: string containing the default user name.
    -> T.Text
    -- ^ /@defaultDomain@/: string containing the default domain.
    -> [Gio.Flags.AskPasswordFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.AskPasswordFlags'.
    -> IO ()

type C_MountOperationAskPasswordCallback =
    Ptr MountOperation ->                   -- object
    CString ->
    CString ->
    CString ->
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_MountOperationAskPasswordCallback`.
foreign import ccall "wrapper"
    mk_MountOperationAskPasswordCallback :: C_MountOperationAskPasswordCallback -> IO (FunPtr C_MountOperationAskPasswordCallback)

wrap_MountOperationAskPasswordCallback :: 
    GObject a => (a -> MountOperationAskPasswordCallback) ->
    C_MountOperationAskPasswordCallback
wrap_MountOperationAskPasswordCallback gi'cb gi'selfPtr message defaultUser defaultDomain flags _ = do
    message' <- cstringToText message
    defaultUser' <- cstringToText defaultUser
    defaultDomain' <- cstringToText defaultDomain
    let flags' = wordToGFlags flags
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  message' defaultUser' defaultDomain' flags'


-- | Connect a signal handler for the [askPassword](#signal:askPassword) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' mountOperation #askPassword callback
-- @
-- 
-- 
onMountOperationAskPassword :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationAskPasswordCallback) -> m SignalHandlerId
onMountOperationAskPassword obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationAskPasswordCallback wrapped
    wrapped'' <- mk_MountOperationAskPasswordCallback wrapped'
    connectSignalFunPtr obj "ask-password" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [askPassword](#signal:askPassword) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' mountOperation #askPassword callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterMountOperationAskPassword :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationAskPasswordCallback) -> m SignalHandlerId
afterMountOperationAskPassword obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationAskPasswordCallback wrapped
    wrapped'' <- mk_MountOperationAskPasswordCallback wrapped'
    connectSignalFunPtr obj "ask-password" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data MountOperationAskPasswordSignalInfo
instance SignalInfo MountOperationAskPasswordSignalInfo where
    type HaskellCallbackType MountOperationAskPasswordSignalInfo = MountOperationAskPasswordCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_MountOperationAskPasswordCallback cb
        cb'' <- mk_MountOperationAskPasswordCallback cb'
        connectSignalFunPtr obj "ask-password" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation::ask-password"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:signal:askPassword"})

#endif

-- signal MountOperation::ask-question
-- | Emitted when asking the user a question and gives a list of
-- choices for the user to choose from.
-- 
-- If the message contains a line break, the first line should be
-- presented as a heading. For example, it may be used as the
-- primary text in a @/GtkMessageDialog/@.
type MountOperationAskQuestionCallback =
    T.Text
    -- ^ /@message@/: string containing a message to display to the user.
    -> [T.Text]
    -- ^ /@choices@/: an array of strings for each possible choice.
    -> IO ()

type C_MountOperationAskQuestionCallback =
    Ptr MountOperation ->                   -- object
    CString ->
    Ptr CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_MountOperationAskQuestionCallback`.
foreign import ccall "wrapper"
    mk_MountOperationAskQuestionCallback :: C_MountOperationAskQuestionCallback -> IO (FunPtr C_MountOperationAskQuestionCallback)

wrap_MountOperationAskQuestionCallback :: 
    GObject a => (a -> MountOperationAskQuestionCallback) ->
    C_MountOperationAskQuestionCallback
wrap_MountOperationAskQuestionCallback gi'cb gi'selfPtr message choices _ = do
    message' <- cstringToText message
    choices' <- unpackZeroTerminatedUTF8CArray choices
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  message' choices'


-- | Connect a signal handler for the [askQuestion](#signal:askQuestion) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' mountOperation #askQuestion callback
-- @
-- 
-- 
onMountOperationAskQuestion :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationAskQuestionCallback) -> m SignalHandlerId
onMountOperationAskQuestion obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationAskQuestionCallback wrapped
    wrapped'' <- mk_MountOperationAskQuestionCallback wrapped'
    connectSignalFunPtr obj "ask-question" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [askQuestion](#signal:askQuestion) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' mountOperation #askQuestion callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterMountOperationAskQuestion :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationAskQuestionCallback) -> m SignalHandlerId
afterMountOperationAskQuestion obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationAskQuestionCallback wrapped
    wrapped'' <- mk_MountOperationAskQuestionCallback wrapped'
    connectSignalFunPtr obj "ask-question" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data MountOperationAskQuestionSignalInfo
instance SignalInfo MountOperationAskQuestionSignalInfo where
    type HaskellCallbackType MountOperationAskQuestionSignalInfo = MountOperationAskQuestionCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_MountOperationAskQuestionCallback cb
        cb'' <- mk_MountOperationAskQuestionCallback cb'
        connectSignalFunPtr obj "ask-question" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation::ask-question"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:signal:askQuestion"})

#endif

-- signal MountOperation::reply
-- | Emitted when the user has replied to the mount operation.
type MountOperationReplyCallback =
    Gio.Enums.MountOperationResult
    -- ^ /@result@/: a t'GI.Gio.Enums.MountOperationResult' indicating how the request was handled
    -> IO ()

type C_MountOperationReplyCallback =
    Ptr MountOperation ->                   -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_MountOperationReplyCallback`.
foreign import ccall "wrapper"
    mk_MountOperationReplyCallback :: C_MountOperationReplyCallback -> IO (FunPtr C_MountOperationReplyCallback)

wrap_MountOperationReplyCallback :: 
    GObject a => (a -> MountOperationReplyCallback) ->
    C_MountOperationReplyCallback
wrap_MountOperationReplyCallback gi'cb gi'selfPtr result_ _ = do
    let result_' = (toEnum . fromIntegral) result_
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  result_'


-- | Connect a signal handler for the [reply](#signal:reply) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' mountOperation #reply callback
-- @
-- 
-- 
onMountOperationReply :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationReplyCallback) -> m SignalHandlerId
onMountOperationReply obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationReplyCallback wrapped
    wrapped'' <- mk_MountOperationReplyCallback wrapped'
    connectSignalFunPtr obj "reply" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [reply](#signal:reply) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' mountOperation #reply callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterMountOperationReply :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationReplyCallback) -> m SignalHandlerId
afterMountOperationReply obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationReplyCallback wrapped
    wrapped'' <- mk_MountOperationReplyCallback wrapped'
    connectSignalFunPtr obj "reply" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data MountOperationReplySignalInfo
instance SignalInfo MountOperationReplySignalInfo where
    type HaskellCallbackType MountOperationReplySignalInfo = MountOperationReplyCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_MountOperationReplyCallback cb
        cb'' <- mk_MountOperationReplyCallback cb'
        connectSignalFunPtr obj "reply" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation::reply"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:signal:reply"})

#endif

-- signal MountOperation::show-processes
-- | Emitted when one or more processes are blocking an operation
-- e.g. unmounting\/ejecting a t'GI.Gio.Interfaces.Mount.Mount' or stopping a t'GI.Gio.Interfaces.Drive.Drive'.
-- 
-- Note that this signal may be emitted several times to update the
-- list of blocking processes as processes close files. The
-- application should only respond with 'GI.Gio.Objects.MountOperation.mountOperationReply' to
-- the latest signal (setting [MountOperation:choice]("GI.Gio.Objects.MountOperation#g:attr:choice") to the choice
-- the user made).
-- 
-- If the message contains a line break, the first line should be
-- presented as a heading. For example, it may be used as the
-- primary text in a @/GtkMessageDialog/@.
-- 
-- /Since: 2.22/
type MountOperationShowProcessesCallback =
    T.Text
    -- ^ /@message@/: string containing a message to display to the user.
    -> [Int32]
    -- ^ /@processes@/: an array of @/GPid/@ for processes
    --   blocking the operation.
    -> [T.Text]
    -- ^ /@choices@/: an array of strings for each possible choice.
    -> IO ()

type C_MountOperationShowProcessesCallback =
    Ptr MountOperation ->                   -- object
    CString ->
    Ptr (GArray Int32) ->
    Ptr CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_MountOperationShowProcessesCallback`.
foreign import ccall "wrapper"
    mk_MountOperationShowProcessesCallback :: C_MountOperationShowProcessesCallback -> IO (FunPtr C_MountOperationShowProcessesCallback)

wrap_MountOperationShowProcessesCallback :: 
    GObject a => (a -> MountOperationShowProcessesCallback) ->
    C_MountOperationShowProcessesCallback
wrap_MountOperationShowProcessesCallback gi'cb gi'selfPtr message processes choices _ = do
    message' <- cstringToText message
    processes' <- unpackGArray processes
    choices' <- unpackZeroTerminatedUTF8CArray choices
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  message' processes' choices'


-- | Connect a signal handler for the [showProcesses](#signal:showProcesses) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' mountOperation #showProcesses callback
-- @
-- 
-- 
onMountOperationShowProcesses :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationShowProcessesCallback) -> m SignalHandlerId
onMountOperationShowProcesses obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationShowProcessesCallback wrapped
    wrapped'' <- mk_MountOperationShowProcessesCallback wrapped'
    connectSignalFunPtr obj "show-processes" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [showProcesses](#signal:showProcesses) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' mountOperation #showProcesses callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterMountOperationShowProcesses :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationShowProcessesCallback) -> m SignalHandlerId
afterMountOperationShowProcesses obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationShowProcessesCallback wrapped
    wrapped'' <- mk_MountOperationShowProcessesCallback wrapped'
    connectSignalFunPtr obj "show-processes" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data MountOperationShowProcessesSignalInfo
instance SignalInfo MountOperationShowProcessesSignalInfo where
    type HaskellCallbackType MountOperationShowProcessesSignalInfo = MountOperationShowProcessesCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_MountOperationShowProcessesCallback cb
        cb'' <- mk_MountOperationShowProcessesCallback cb'
        connectSignalFunPtr obj "show-processes" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation::show-processes"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:signal:showProcesses"})

#endif

-- signal MountOperation::show-unmount-progress
-- | Emitted when an unmount operation has been busy for more than some time
-- (typically 1.5 seconds).
-- 
-- When unmounting or ejecting a volume, the kernel might need to flush
-- pending data in its buffers to the volume stable storage, and this operation
-- can take a considerable amount of time. This signal may be emitted several
-- times as long as the unmount operation is outstanding, and then one
-- last time when the operation is completed, with /@bytesLeft@/ set to zero.
-- 
-- Implementations of GMountOperation should handle this signal by
-- showing an UI notification, and then dismiss it, or show another notification
-- of completion, when /@bytesLeft@/ reaches zero.
-- 
-- If the message contains a line break, the first line should be
-- presented as a heading. For example, it may be used as the
-- primary text in a @/GtkMessageDialog/@.
-- 
-- /Since: 2.34/
type MountOperationShowUnmountProgressCallback =
    T.Text
    -- ^ /@message@/: string containing a message to display to the user
    -> Int64
    -- ^ /@timeLeft@/: the estimated time left before the operation completes,
    --     in microseconds, or -1
    -> Int64
    -- ^ /@bytesLeft@/: the amount of bytes to be written before the operation
    --     completes (or -1 if such amount is not known), or zero if the operation
    --     is completed
    -> IO ()

type C_MountOperationShowUnmountProgressCallback =
    Ptr MountOperation ->                   -- object
    CString ->
    Int64 ->
    Int64 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_MountOperationShowUnmountProgressCallback`.
foreign import ccall "wrapper"
    mk_MountOperationShowUnmountProgressCallback :: C_MountOperationShowUnmountProgressCallback -> IO (FunPtr C_MountOperationShowUnmountProgressCallback)

wrap_MountOperationShowUnmountProgressCallback :: 
    GObject a => (a -> MountOperationShowUnmountProgressCallback) ->
    C_MountOperationShowUnmountProgressCallback
wrap_MountOperationShowUnmountProgressCallback gi'cb gi'selfPtr message timeLeft bytesLeft _ = do
    message' <- cstringToText message
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  message' timeLeft bytesLeft


-- | Connect a signal handler for the [showUnmountProgress](#signal:showUnmountProgress) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' mountOperation #showUnmountProgress callback
-- @
-- 
-- 
onMountOperationShowUnmountProgress :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationShowUnmountProgressCallback) -> m SignalHandlerId
onMountOperationShowUnmountProgress obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationShowUnmountProgressCallback wrapped
    wrapped'' <- mk_MountOperationShowUnmountProgressCallback wrapped'
    connectSignalFunPtr obj "show-unmount-progress" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [showUnmountProgress](#signal:showUnmountProgress) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' mountOperation #showUnmountProgress callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterMountOperationShowUnmountProgress :: (IsMountOperation a, MonadIO m) => a -> ((?self :: a) => MountOperationShowUnmountProgressCallback) -> m SignalHandlerId
afterMountOperationShowUnmountProgress obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MountOperationShowUnmountProgressCallback wrapped
    wrapped'' <- mk_MountOperationShowUnmountProgressCallback wrapped'
    connectSignalFunPtr obj "show-unmount-progress" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data MountOperationShowUnmountProgressSignalInfo
instance SignalInfo MountOperationShowUnmountProgressSignalInfo where
    type HaskellCallbackType MountOperationShowUnmountProgressSignalInfo = MountOperationShowUnmountProgressCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_MountOperationShowUnmountProgressCallback cb
        cb'' <- mk_MountOperationShowUnmountProgressCallback cb'
        connectSignalFunPtr obj "show-unmount-progress" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation::show-unmount-progress"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:signal:showUnmountProgress"})

#endif

-- VVV Prop "anonymous"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@anonymous@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #anonymous
-- @
getMountOperationAnonymous :: (MonadIO m, IsMountOperation o) => o -> m Bool
getMountOperationAnonymous obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "anonymous"

-- | Set the value of the “@anonymous@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #anonymous 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationAnonymous :: (MonadIO m, IsMountOperation o) => o -> Bool -> m ()
setMountOperationAnonymous obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "anonymous" val

-- | Construct a `GValueConstruct` with valid value for the “@anonymous@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationAnonymous :: (IsMountOperation o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructMountOperationAnonymous val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "anonymous" val

#if defined(ENABLE_OVERLOADING)
data MountOperationAnonymousPropertyInfo
instance AttrInfo MountOperationAnonymousPropertyInfo where
    type AttrAllowedOps MountOperationAnonymousPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint MountOperationAnonymousPropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationAnonymousPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint MountOperationAnonymousPropertyInfo = (~) Bool
    type AttrTransferType MountOperationAnonymousPropertyInfo = Bool
    type AttrGetType MountOperationAnonymousPropertyInfo = Bool
    type AttrLabel MountOperationAnonymousPropertyInfo = "anonymous"
    type AttrOrigin MountOperationAnonymousPropertyInfo = MountOperation
    attrGet = getMountOperationAnonymous
    attrSet = setMountOperationAnonymous
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationAnonymous
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.anonymous"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:anonymous"
        })
#endif

-- VVV Prop "choice"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@choice@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #choice
-- @
getMountOperationChoice :: (MonadIO m, IsMountOperation o) => o -> m Int32
getMountOperationChoice obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "choice"

-- | Set the value of the “@choice@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #choice 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationChoice :: (MonadIO m, IsMountOperation o) => o -> Int32 -> m ()
setMountOperationChoice obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "choice" val

-- | Construct a `GValueConstruct` with valid value for the “@choice@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationChoice :: (IsMountOperation o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructMountOperationChoice val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "choice" val

#if defined(ENABLE_OVERLOADING)
data MountOperationChoicePropertyInfo
instance AttrInfo MountOperationChoicePropertyInfo where
    type AttrAllowedOps MountOperationChoicePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint MountOperationChoicePropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationChoicePropertyInfo = (~) Int32
    type AttrTransferTypeConstraint MountOperationChoicePropertyInfo = (~) Int32
    type AttrTransferType MountOperationChoicePropertyInfo = Int32
    type AttrGetType MountOperationChoicePropertyInfo = Int32
    type AttrLabel MountOperationChoicePropertyInfo = "choice"
    type AttrOrigin MountOperationChoicePropertyInfo = MountOperation
    attrGet = getMountOperationChoice
    attrSet = setMountOperationChoice
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationChoice
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.choice"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:choice"
        })
#endif

-- VVV Prop "domain"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@domain@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #domain
-- @
getMountOperationDomain :: (MonadIO m, IsMountOperation o) => o -> m (Maybe T.Text)
getMountOperationDomain obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "domain"

-- | Set the value of the “@domain@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #domain 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationDomain :: (MonadIO m, IsMountOperation o) => o -> T.Text -> m ()
setMountOperationDomain obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "domain" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@domain@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationDomain :: (IsMountOperation o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructMountOperationDomain val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "domain" (P.Just val)

-- | Set the value of the “@domain@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #domain
-- @
clearMountOperationDomain :: (MonadIO m, IsMountOperation o) => o -> m ()
clearMountOperationDomain obj = liftIO $ B.Properties.setObjectPropertyString obj "domain" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data MountOperationDomainPropertyInfo
instance AttrInfo MountOperationDomainPropertyInfo where
    type AttrAllowedOps MountOperationDomainPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint MountOperationDomainPropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationDomainPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint MountOperationDomainPropertyInfo = (~) T.Text
    type AttrTransferType MountOperationDomainPropertyInfo = T.Text
    type AttrGetType MountOperationDomainPropertyInfo = (Maybe T.Text)
    type AttrLabel MountOperationDomainPropertyInfo = "domain"
    type AttrOrigin MountOperationDomainPropertyInfo = MountOperation
    attrGet = getMountOperationDomain
    attrSet = setMountOperationDomain
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationDomain
    attrClear = clearMountOperationDomain
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.domain"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:domain"
        })
#endif

-- VVV Prop "is-tcrypt-hidden-volume"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@is-tcrypt-hidden-volume@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #isTcryptHiddenVolume
-- @
getMountOperationIsTcryptHiddenVolume :: (MonadIO m, IsMountOperation o) => o -> m Bool
getMountOperationIsTcryptHiddenVolume obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "is-tcrypt-hidden-volume"

-- | Set the value of the “@is-tcrypt-hidden-volume@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #isTcryptHiddenVolume 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationIsTcryptHiddenVolume :: (MonadIO m, IsMountOperation o) => o -> Bool -> m ()
setMountOperationIsTcryptHiddenVolume obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "is-tcrypt-hidden-volume" val

-- | Construct a `GValueConstruct` with valid value for the “@is-tcrypt-hidden-volume@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationIsTcryptHiddenVolume :: (IsMountOperation o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructMountOperationIsTcryptHiddenVolume val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "is-tcrypt-hidden-volume" val

#if defined(ENABLE_OVERLOADING)
data MountOperationIsTcryptHiddenVolumePropertyInfo
instance AttrInfo MountOperationIsTcryptHiddenVolumePropertyInfo where
    type AttrAllowedOps MountOperationIsTcryptHiddenVolumePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint MountOperationIsTcryptHiddenVolumePropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationIsTcryptHiddenVolumePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint MountOperationIsTcryptHiddenVolumePropertyInfo = (~) Bool
    type AttrTransferType MountOperationIsTcryptHiddenVolumePropertyInfo = Bool
    type AttrGetType MountOperationIsTcryptHiddenVolumePropertyInfo = Bool
    type AttrLabel MountOperationIsTcryptHiddenVolumePropertyInfo = "is-tcrypt-hidden-volume"
    type AttrOrigin MountOperationIsTcryptHiddenVolumePropertyInfo = MountOperation
    attrGet = getMountOperationIsTcryptHiddenVolume
    attrSet = setMountOperationIsTcryptHiddenVolume
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationIsTcryptHiddenVolume
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.isTcryptHiddenVolume"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:isTcryptHiddenVolume"
        })
#endif

-- VVV Prop "is-tcrypt-system-volume"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@is-tcrypt-system-volume@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #isTcryptSystemVolume
-- @
getMountOperationIsTcryptSystemVolume :: (MonadIO m, IsMountOperation o) => o -> m Bool
getMountOperationIsTcryptSystemVolume obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "is-tcrypt-system-volume"

-- | Set the value of the “@is-tcrypt-system-volume@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #isTcryptSystemVolume 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationIsTcryptSystemVolume :: (MonadIO m, IsMountOperation o) => o -> Bool -> m ()
setMountOperationIsTcryptSystemVolume obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "is-tcrypt-system-volume" val

-- | Construct a `GValueConstruct` with valid value for the “@is-tcrypt-system-volume@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationIsTcryptSystemVolume :: (IsMountOperation o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructMountOperationIsTcryptSystemVolume val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "is-tcrypt-system-volume" val

#if defined(ENABLE_OVERLOADING)
data MountOperationIsTcryptSystemVolumePropertyInfo
instance AttrInfo MountOperationIsTcryptSystemVolumePropertyInfo where
    type AttrAllowedOps MountOperationIsTcryptSystemVolumePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint MountOperationIsTcryptSystemVolumePropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationIsTcryptSystemVolumePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint MountOperationIsTcryptSystemVolumePropertyInfo = (~) Bool
    type AttrTransferType MountOperationIsTcryptSystemVolumePropertyInfo = Bool
    type AttrGetType MountOperationIsTcryptSystemVolumePropertyInfo = Bool
    type AttrLabel MountOperationIsTcryptSystemVolumePropertyInfo = "is-tcrypt-system-volume"
    type AttrOrigin MountOperationIsTcryptSystemVolumePropertyInfo = MountOperation
    attrGet = getMountOperationIsTcryptSystemVolume
    attrSet = setMountOperationIsTcryptSystemVolume
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationIsTcryptSystemVolume
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.isTcryptSystemVolume"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:isTcryptSystemVolume"
        })
#endif

-- VVV Prop "password"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@password@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #password
-- @
getMountOperationPassword :: (MonadIO m, IsMountOperation o) => o -> m (Maybe T.Text)
getMountOperationPassword obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "password"

-- | Set the value of the “@password@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #password 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationPassword :: (MonadIO m, IsMountOperation o) => o -> T.Text -> m ()
setMountOperationPassword obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "password" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@password@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationPassword :: (IsMountOperation o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructMountOperationPassword val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "password" (P.Just val)

-- | Set the value of the “@password@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #password
-- @
clearMountOperationPassword :: (MonadIO m, IsMountOperation o) => o -> m ()
clearMountOperationPassword obj = liftIO $ B.Properties.setObjectPropertyString obj "password" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data MountOperationPasswordPropertyInfo
instance AttrInfo MountOperationPasswordPropertyInfo where
    type AttrAllowedOps MountOperationPasswordPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint MountOperationPasswordPropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationPasswordPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint MountOperationPasswordPropertyInfo = (~) T.Text
    type AttrTransferType MountOperationPasswordPropertyInfo = T.Text
    type AttrGetType MountOperationPasswordPropertyInfo = (Maybe T.Text)
    type AttrLabel MountOperationPasswordPropertyInfo = "password"
    type AttrOrigin MountOperationPasswordPropertyInfo = MountOperation
    attrGet = getMountOperationPassword
    attrSet = setMountOperationPassword
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationPassword
    attrClear = clearMountOperationPassword
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.password"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:password"
        })
#endif

-- VVV Prop "password-save"
   -- Type: TInterface (Name {namespace = "Gio", name = "PasswordSave"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@password-save@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #passwordSave
-- @
getMountOperationPasswordSave :: (MonadIO m, IsMountOperation o) => o -> m Gio.Enums.PasswordSave
getMountOperationPasswordSave obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "password-save"

-- | Set the value of the “@password-save@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #passwordSave 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationPasswordSave :: (MonadIO m, IsMountOperation o) => o -> Gio.Enums.PasswordSave -> m ()
setMountOperationPasswordSave obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "password-save" val

-- | Construct a `GValueConstruct` with valid value for the “@password-save@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationPasswordSave :: (IsMountOperation o, MIO.MonadIO m) => Gio.Enums.PasswordSave -> m (GValueConstruct o)
constructMountOperationPasswordSave val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "password-save" val

#if defined(ENABLE_OVERLOADING)
data MountOperationPasswordSavePropertyInfo
instance AttrInfo MountOperationPasswordSavePropertyInfo where
    type AttrAllowedOps MountOperationPasswordSavePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint MountOperationPasswordSavePropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationPasswordSavePropertyInfo = (~) Gio.Enums.PasswordSave
    type AttrTransferTypeConstraint MountOperationPasswordSavePropertyInfo = (~) Gio.Enums.PasswordSave
    type AttrTransferType MountOperationPasswordSavePropertyInfo = Gio.Enums.PasswordSave
    type AttrGetType MountOperationPasswordSavePropertyInfo = Gio.Enums.PasswordSave
    type AttrLabel MountOperationPasswordSavePropertyInfo = "password-save"
    type AttrOrigin MountOperationPasswordSavePropertyInfo = MountOperation
    attrGet = getMountOperationPasswordSave
    attrSet = setMountOperationPasswordSave
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationPasswordSave
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.passwordSave"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:passwordSave"
        })
#endif

-- VVV Prop "pim"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@pim@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #pim
-- @
getMountOperationPim :: (MonadIO m, IsMountOperation o) => o -> m Word32
getMountOperationPim obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "pim"

-- | Set the value of the “@pim@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #pim 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationPim :: (MonadIO m, IsMountOperation o) => o -> Word32 -> m ()
setMountOperationPim obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "pim" val

-- | Construct a `GValueConstruct` with valid value for the “@pim@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationPim :: (IsMountOperation o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructMountOperationPim val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "pim" val

#if defined(ENABLE_OVERLOADING)
data MountOperationPimPropertyInfo
instance AttrInfo MountOperationPimPropertyInfo where
    type AttrAllowedOps MountOperationPimPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint MountOperationPimPropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationPimPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint MountOperationPimPropertyInfo = (~) Word32
    type AttrTransferType MountOperationPimPropertyInfo = Word32
    type AttrGetType MountOperationPimPropertyInfo = Word32
    type AttrLabel MountOperationPimPropertyInfo = "pim"
    type AttrOrigin MountOperationPimPropertyInfo = MountOperation
    attrGet = getMountOperationPim
    attrSet = setMountOperationPim
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationPim
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.pim"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:pim"
        })
#endif

-- VVV Prop "username"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@username@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' mountOperation #username
-- @
getMountOperationUsername :: (MonadIO m, IsMountOperation o) => o -> m (Maybe T.Text)
getMountOperationUsername obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "username"

-- | Set the value of the “@username@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' mountOperation [ #username 'Data.GI.Base.Attributes.:=' value ]
-- @
setMountOperationUsername :: (MonadIO m, IsMountOperation o) => o -> T.Text -> m ()
setMountOperationUsername obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "username" (Just val)

-- | Construct a `GValueConstruct` with valid value for the “@username@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructMountOperationUsername :: (IsMountOperation o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructMountOperationUsername val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "username" (P.Just val)

-- | Set the value of the “@username@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #username
-- @
clearMountOperationUsername :: (MonadIO m, IsMountOperation o) => o -> m ()
clearMountOperationUsername obj = liftIO $ B.Properties.setObjectPropertyString obj "username" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data MountOperationUsernamePropertyInfo
instance AttrInfo MountOperationUsernamePropertyInfo where
    type AttrAllowedOps MountOperationUsernamePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint MountOperationUsernamePropertyInfo = IsMountOperation
    type AttrSetTypeConstraint MountOperationUsernamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint MountOperationUsernamePropertyInfo = (~) T.Text
    type AttrTransferType MountOperationUsernamePropertyInfo = T.Text
    type AttrGetType MountOperationUsernamePropertyInfo = (Maybe T.Text)
    type AttrLabel MountOperationUsernamePropertyInfo = "username"
    type AttrOrigin MountOperationUsernamePropertyInfo = MountOperation
    attrGet = getMountOperationUsername
    attrSet = setMountOperationUsername
    attrTransfer _ v = do
        return v
    attrConstruct = constructMountOperationUsername
    attrClear = clearMountOperationUsername
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.username"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#g:attr:username"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList MountOperation
type instance O.AttributeList MountOperation = MountOperationAttributeList
type MountOperationAttributeList = ('[ '("anonymous", MountOperationAnonymousPropertyInfo), '("choice", MountOperationChoicePropertyInfo), '("domain", MountOperationDomainPropertyInfo), '("isTcryptHiddenVolume", MountOperationIsTcryptHiddenVolumePropertyInfo), '("isTcryptSystemVolume", MountOperationIsTcryptSystemVolumePropertyInfo), '("password", MountOperationPasswordPropertyInfo), '("passwordSave", MountOperationPasswordSavePropertyInfo), '("pim", MountOperationPimPropertyInfo), '("username", MountOperationUsernamePropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
mountOperationAnonymous :: AttrLabelProxy "anonymous"
mountOperationAnonymous = AttrLabelProxy

mountOperationChoice :: AttrLabelProxy "choice"
mountOperationChoice = AttrLabelProxy

mountOperationDomain :: AttrLabelProxy "domain"
mountOperationDomain = AttrLabelProxy

mountOperationIsTcryptHiddenVolume :: AttrLabelProxy "isTcryptHiddenVolume"
mountOperationIsTcryptHiddenVolume = AttrLabelProxy

mountOperationIsTcryptSystemVolume :: AttrLabelProxy "isTcryptSystemVolume"
mountOperationIsTcryptSystemVolume = AttrLabelProxy

mountOperationPassword :: AttrLabelProxy "password"
mountOperationPassword = AttrLabelProxy

mountOperationPasswordSave :: AttrLabelProxy "passwordSave"
mountOperationPasswordSave = AttrLabelProxy

mountOperationPim :: AttrLabelProxy "pim"
mountOperationPim = AttrLabelProxy

mountOperationUsername :: AttrLabelProxy "username"
mountOperationUsername = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList MountOperation = MountOperationSignalList
type MountOperationSignalList = ('[ '("aborted", MountOperationAbortedSignalInfo), '("askPassword", MountOperationAskPasswordSignalInfo), '("askQuestion", MountOperationAskQuestionSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("reply", MountOperationReplySignalInfo), '("showProcesses", MountOperationShowProcessesSignalInfo), '("showUnmountProgress", MountOperationShowUnmountProgressSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method MountOperation::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "MountOperation" })
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_new" g_mount_operation_new :: 
    IO (Ptr MountOperation)

-- | Creates a new mount operation.
mountOperationNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m MountOperation
    -- ^ __Returns:__ a t'GI.Gio.Objects.MountOperation.MountOperation'.
mountOperationNew  = liftIO $ do
    result <- g_mount_operation_new
    checkUnexpectedReturnNULL "mountOperationNew" result
    result' <- (wrapObject MountOperation) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method MountOperation::get_anonymous
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_anonymous" g_mount_operation_get_anonymous :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO CInt

-- | Check to see whether the mount operation is being used
-- for an anonymous user.
mountOperationGetAnonymous ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if mount operation is anonymous.
mountOperationGetAnonymous op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_anonymous op'
    let result' = (/= 0) result
    touchManagedPtr op
    return result'

#if defined(ENABLE_OVERLOADING)
data MountOperationGetAnonymousMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetAnonymousMethodInfo a signature where
    overloadedMethod = mountOperationGetAnonymous

instance O.OverloadedMethodInfo MountOperationGetAnonymousMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetAnonymous",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetAnonymous"
        })


#endif

-- method MountOperation::get_choice
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_choice" g_mount_operation_get_choice :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO Int32

-- | Gets a choice from the mount operation.
mountOperationGetChoice ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m Int32
    -- ^ __Returns:__ an integer containing an index of the user\'s choice from
    -- the choice\'s list, or @0@.
mountOperationGetChoice op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_choice op'
    touchManagedPtr op
    return result

#if defined(ENABLE_OVERLOADING)
data MountOperationGetChoiceMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetChoiceMethodInfo a signature where
    overloadedMethod = mountOperationGetChoice

instance O.OverloadedMethodInfo MountOperationGetChoiceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetChoice",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetChoice"
        })


#endif

-- method MountOperation::get_domain
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_domain" g_mount_operation_get_domain :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO CString

-- | Gets the domain of the mount operation.
mountOperationGetDomain ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ a string set to the domain.
mountOperationGetDomain op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_domain op'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr op
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data MountOperationGetDomainMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetDomainMethodInfo a signature where
    overloadedMethod = mountOperationGetDomain

instance O.OverloadedMethodInfo MountOperationGetDomainMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetDomain",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetDomain"
        })


#endif

-- method MountOperation::get_is_tcrypt_hidden_volume
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_is_tcrypt_hidden_volume" g_mount_operation_get_is_tcrypt_hidden_volume :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO CInt

-- | Check to see whether the mount operation is being used
-- for a TCRYPT hidden volume.
-- 
-- /Since: 2.58/
mountOperationGetIsTcryptHiddenVolume ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if mount operation is for hidden volume.
mountOperationGetIsTcryptHiddenVolume op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_is_tcrypt_hidden_volume op'
    let result' = (/= 0) result
    touchManagedPtr op
    return result'

#if defined(ENABLE_OVERLOADING)
data MountOperationGetIsTcryptHiddenVolumeMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetIsTcryptHiddenVolumeMethodInfo a signature where
    overloadedMethod = mountOperationGetIsTcryptHiddenVolume

instance O.OverloadedMethodInfo MountOperationGetIsTcryptHiddenVolumeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetIsTcryptHiddenVolume",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetIsTcryptHiddenVolume"
        })


#endif

-- method MountOperation::get_is_tcrypt_system_volume
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_is_tcrypt_system_volume" g_mount_operation_get_is_tcrypt_system_volume :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO CInt

-- | Check to see whether the mount operation is being used
-- for a TCRYPT system volume.
-- 
-- /Since: 2.58/
mountOperationGetIsTcryptSystemVolume ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if mount operation is for system volume.
mountOperationGetIsTcryptSystemVolume op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_is_tcrypt_system_volume op'
    let result' = (/= 0) result
    touchManagedPtr op
    return result'

#if defined(ENABLE_OVERLOADING)
data MountOperationGetIsTcryptSystemVolumeMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetIsTcryptSystemVolumeMethodInfo a signature where
    overloadedMethod = mountOperationGetIsTcryptSystemVolume

instance O.OverloadedMethodInfo MountOperationGetIsTcryptSystemVolumeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetIsTcryptSystemVolume",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetIsTcryptSystemVolume"
        })


#endif

-- method MountOperation::get_password
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_password" g_mount_operation_get_password :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO CString

-- | Gets a password from the mount operation.
mountOperationGetPassword ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ a string containing the password within /@op@/.
mountOperationGetPassword op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_password op'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr op
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data MountOperationGetPasswordMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetPasswordMethodInfo a signature where
    overloadedMethod = mountOperationGetPassword

instance O.OverloadedMethodInfo MountOperationGetPasswordMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetPassword",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetPassword"
        })


#endif

-- method MountOperation::get_password_save
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "PasswordSave" })
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_password_save" g_mount_operation_get_password_save :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO CUInt

-- | Gets the state of saving passwords for the mount operation.
mountOperationGetPasswordSave ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m Gio.Enums.PasswordSave
    -- ^ __Returns:__ a t'GI.Gio.Enums.PasswordSave' flag.
mountOperationGetPasswordSave op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_password_save op'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr op
    return result'

#if defined(ENABLE_OVERLOADING)
data MountOperationGetPasswordSaveMethodInfo
instance (signature ~ (m Gio.Enums.PasswordSave), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetPasswordSaveMethodInfo a signature where
    overloadedMethod = mountOperationGetPasswordSave

instance O.OverloadedMethodInfo MountOperationGetPasswordSaveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetPasswordSave",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetPasswordSave"
        })


#endif

-- method MountOperation::get_pim
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_pim" g_mount_operation_get_pim :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO Word32

-- | Gets a PIM from the mount operation.
-- 
-- /Since: 2.58/
mountOperationGetPim ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m Word32
    -- ^ __Returns:__ The VeraCrypt PIM within /@op@/.
mountOperationGetPim op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_pim op'
    touchManagedPtr op
    return result

#if defined(ENABLE_OVERLOADING)
data MountOperationGetPimMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetPimMethodInfo a signature where
    overloadedMethod = mountOperationGetPim

instance O.OverloadedMethodInfo MountOperationGetPimMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetPim",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetPim"
        })


#endif

-- method MountOperation::get_username
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_get_username" g_mount_operation_get_username :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    IO CString

-- | Get the user name from the mount operation.
mountOperationGetUsername ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ a string containing the user name.
mountOperationGetUsername op = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    result <- g_mount_operation_get_username op'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr op
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data MountOperationGetUsernameMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationGetUsernameMethodInfo a signature where
    overloadedMethod = mountOperationGetUsername

instance O.OverloadedMethodInfo MountOperationGetUsernameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationGetUsername",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationGetUsername"
        })


#endif

-- method MountOperation::reply
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "MountOperationResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperationResult"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_reply" g_mount_operation_reply :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    CUInt ->                                -- result : TInterface (Name {namespace = "Gio", name = "MountOperationResult"})
    IO ()

-- | Emits the [MountOperation::reply]("GI.Gio.Objects.MountOperation#g:signal:reply") signal.
mountOperationReply ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'
    -> Gio.Enums.MountOperationResult
    -- ^ /@result@/: a t'GI.Gio.Enums.MountOperationResult'
    -> m ()
mountOperationReply op result_ = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    let result_' = (fromIntegral . fromEnum) result_
    g_mount_operation_reply op' result_'
    touchManagedPtr op
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationReplyMethodInfo
instance (signature ~ (Gio.Enums.MountOperationResult -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationReplyMethodInfo a signature where
    overloadedMethod = mountOperationReply

instance O.OverloadedMethodInfo MountOperationReplyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationReply",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationReply"
        })


#endif

-- method MountOperation::set_anonymous
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "anonymous"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "boolean value." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_anonymous" g_mount_operation_set_anonymous :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    CInt ->                                 -- anonymous : TBasicType TBoolean
    IO ()

-- | Sets the mount operation to use an anonymous user if /@anonymous@/ is 'P.True'.
mountOperationSetAnonymous ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Bool
    -- ^ /@anonymous@/: boolean value.
    -> m ()
mountOperationSetAnonymous op anonymous = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    let anonymous' = (P.fromIntegral . P.fromEnum) anonymous
    g_mount_operation_set_anonymous op' anonymous'
    touchManagedPtr op
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetAnonymousMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetAnonymousMethodInfo a signature where
    overloadedMethod = mountOperationSetAnonymous

instance O.OverloadedMethodInfo MountOperationSetAnonymousMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetAnonymous",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetAnonymous"
        })


#endif

-- method MountOperation::set_choice
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "choice"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an integer." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_choice" g_mount_operation_set_choice :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Int32 ->                                -- choice : TBasicType TInt
    IO ()

-- | Sets a default choice for the mount operation.
mountOperationSetChoice ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Int32
    -- ^ /@choice@/: an integer.
    -> m ()
mountOperationSetChoice op choice = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    g_mount_operation_set_choice op' choice
    touchManagedPtr op
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetChoiceMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetChoiceMethodInfo a signature where
    overloadedMethod = mountOperationSetChoice

instance O.OverloadedMethodInfo MountOperationSetChoiceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetChoice",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetChoice"
        })


#endif

-- method MountOperation::set_domain
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "domain"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the domain to set." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_domain" g_mount_operation_set_domain :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    CString ->                              -- domain : TBasicType TUTF8
    IO ()

-- | Sets the mount operation\'s domain.
mountOperationSetDomain ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Maybe (T.Text)
    -- ^ /@domain@/: the domain to set.
    -> m ()
mountOperationSetDomain op domain = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    maybeDomain <- case domain of
        Nothing -> return FP.nullPtr
        Just jDomain -> do
            jDomain' <- textToCString jDomain
            return jDomain'
    g_mount_operation_set_domain op' maybeDomain
    touchManagedPtr op
    freeMem maybeDomain
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetDomainMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetDomainMethodInfo a signature where
    overloadedMethod = mountOperationSetDomain

instance O.OverloadedMethodInfo MountOperationSetDomainMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetDomain",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetDomain"
        })


#endif

-- method MountOperation::set_is_tcrypt_hidden_volume
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hidden_volume"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "boolean value." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_is_tcrypt_hidden_volume" g_mount_operation_set_is_tcrypt_hidden_volume :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    CInt ->                                 -- hidden_volume : TBasicType TBoolean
    IO ()

-- | Sets the mount operation to use a hidden volume if /@hiddenVolume@/ is 'P.True'.
-- 
-- /Since: 2.58/
mountOperationSetIsTcryptHiddenVolume ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Bool
    -- ^ /@hiddenVolume@/: boolean value.
    -> m ()
mountOperationSetIsTcryptHiddenVolume op hiddenVolume = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    let hiddenVolume' = (P.fromIntegral . P.fromEnum) hiddenVolume
    g_mount_operation_set_is_tcrypt_hidden_volume op' hiddenVolume'
    touchManagedPtr op
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetIsTcryptHiddenVolumeMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetIsTcryptHiddenVolumeMethodInfo a signature where
    overloadedMethod = mountOperationSetIsTcryptHiddenVolume

instance O.OverloadedMethodInfo MountOperationSetIsTcryptHiddenVolumeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetIsTcryptHiddenVolume",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetIsTcryptHiddenVolume"
        })


#endif

-- method MountOperation::set_is_tcrypt_system_volume
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "system_volume"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "boolean value." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_is_tcrypt_system_volume" g_mount_operation_set_is_tcrypt_system_volume :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    CInt ->                                 -- system_volume : TBasicType TBoolean
    IO ()

-- | Sets the mount operation to use a system volume if /@systemVolume@/ is 'P.True'.
-- 
-- /Since: 2.58/
mountOperationSetIsTcryptSystemVolume ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Bool
    -- ^ /@systemVolume@/: boolean value.
    -> m ()
mountOperationSetIsTcryptSystemVolume op systemVolume = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    let systemVolume' = (P.fromIntegral . P.fromEnum) systemVolume
    g_mount_operation_set_is_tcrypt_system_volume op' systemVolume'
    touchManagedPtr op
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetIsTcryptSystemVolumeMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetIsTcryptSystemVolumeMethodInfo a signature where
    overloadedMethod = mountOperationSetIsTcryptSystemVolume

instance O.OverloadedMethodInfo MountOperationSetIsTcryptSystemVolumeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetIsTcryptSystemVolume",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetIsTcryptSystemVolume"
        })


#endif

-- method MountOperation::set_password
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "password"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "password to set." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_password" g_mount_operation_set_password :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    CString ->                              -- password : TBasicType TUTF8
    IO ()

-- | Sets the mount operation\'s password to /@password@/.
mountOperationSetPassword ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Maybe (T.Text)
    -- ^ /@password@/: password to set.
    -> m ()
mountOperationSetPassword op password = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    maybePassword <- case password of
        Nothing -> return FP.nullPtr
        Just jPassword -> do
            jPassword' <- textToCString jPassword
            return jPassword'
    g_mount_operation_set_password op' maybePassword
    touchManagedPtr op
    freeMem maybePassword
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetPasswordMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetPasswordMethodInfo a signature where
    overloadedMethod = mountOperationSetPassword

instance O.OverloadedMethodInfo MountOperationSetPasswordMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetPassword",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetPassword"
        })


#endif

-- method MountOperation::set_password_save
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "save"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "PasswordSave" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GPasswordSave flags."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_password_save" g_mount_operation_set_password_save :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    CUInt ->                                -- save : TInterface (Name {namespace = "Gio", name = "PasswordSave"})
    IO ()

-- | Sets the state of saving passwords for the mount operation.
mountOperationSetPasswordSave ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Gio.Enums.PasswordSave
    -- ^ /@save@/: a set of t'GI.Gio.Enums.PasswordSave' flags.
    -> m ()
mountOperationSetPasswordSave op save = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    let save' = (fromIntegral . fromEnum) save
    g_mount_operation_set_password_save op' save'
    touchManagedPtr op
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetPasswordSaveMethodInfo
instance (signature ~ (Gio.Enums.PasswordSave -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetPasswordSaveMethodInfo a signature where
    overloadedMethod = mountOperationSetPasswordSave

instance O.OverloadedMethodInfo MountOperationSetPasswordSaveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetPasswordSave",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetPasswordSave"
        })


#endif

-- method MountOperation::set_pim
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pim"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an unsigned integer."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_pim" g_mount_operation_set_pim :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Word32 ->                               -- pim : TBasicType TUInt
    IO ()

-- | Sets the mount operation\'s PIM to /@pim@/.
-- 
-- /Since: 2.58/
mountOperationSetPim ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Word32
    -- ^ /@pim@/: an unsigned integer.
    -> m ()
mountOperationSetPim op pim = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    g_mount_operation_set_pim op' pim
    touchManagedPtr op
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetPimMethodInfo
instance (signature ~ (Word32 -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetPimMethodInfo a signature where
    overloadedMethod = mountOperationSetPim

instance O.OverloadedMethodInfo MountOperationSetPimMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetPim",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetPim"
        })


#endif

-- method MountOperation::set_username
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "op"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMountOperation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "username"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input username." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_mount_operation_set_username" g_mount_operation_set_username :: 
    Ptr MountOperation ->                   -- op : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    CString ->                              -- username : TBasicType TUTF8
    IO ()

-- | Sets the user name within /@op@/ to /@username@/.
mountOperationSetUsername ::
    (B.CallStack.HasCallStack, MonadIO m, IsMountOperation a) =>
    a
    -- ^ /@op@/: a t'GI.Gio.Objects.MountOperation.MountOperation'.
    -> Maybe (T.Text)
    -- ^ /@username@/: input username.
    -> m ()
mountOperationSetUsername op username = liftIO $ do
    op' <- unsafeManagedPtrCastPtr op
    maybeUsername <- case username of
        Nothing -> return FP.nullPtr
        Just jUsername -> do
            jUsername' <- textToCString jUsername
            return jUsername'
    g_mount_operation_set_username op' maybeUsername
    touchManagedPtr op
    freeMem maybeUsername
    return ()

#if defined(ENABLE_OVERLOADING)
data MountOperationSetUsernameMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsMountOperation a) => O.OverloadedMethod MountOperationSetUsernameMethodInfo a signature where
    overloadedMethod = mountOperationSetUsername

instance O.OverloadedMethodInfo MountOperationSetUsernameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MountOperation.mountOperationSetUsername",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MountOperation.html#v:mountOperationSetUsername"
        })


#endif


