{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GMenuModel@ represents the contents of a menu — an ordered list of
-- menu items. The items are associated with actions, which can be
-- activated through them. Items can be grouped in sections, and may
-- have submenus associated with them. Both items and sections usually
-- have some representation data, such as labels or icons. The type of
-- the associated action (ie whether it is stateful, and what kind of
-- state it has) can influence the representation of the item.
-- 
-- The conceptual model of menus in @GMenuModel@ is hierarchical:
-- sections and submenus are again represented by @GMenuModel@s.
-- Menus themselves do not define their own roles. Rather, the role
-- of a particular @GMenuModel@ is defined by the item that references
-- it (or, in the case of the ‘root’ menu, is defined by the context
-- in which it is used).
-- 
-- As an example, consider the visible portions of this menu:
-- 
-- == An example menu
-- 
-- <<http://developer.gnome.org/gio/stable/menu-example.png>>
-- 
-- While this kind of deeply nested menu is no longer considered good UI
-- practice, it serves as a good example of the concepts in @GMenuModel@.
-- There are 8 ‘menus’ visible in the screenshot: one menubar, two
-- submenus and 5 sections:
-- 
-- * the toplevel menubar (containing 4 items)
-- * the View submenu (containing 3 sections)
-- * the first section of the View submenu (containing 2 items)
-- * the second section of the View submenu (containing 1 item)
-- * the final section of the View submenu (containing 1 item)
-- * the Highlight Mode submenu (containing 2 sections)
-- * the Sources section (containing 2 items)
-- * the Markup section (containing 2 items)
-- 
-- 
-- The <http://developer.gnome.org/gio/stable/#a-menu-example example> illustrates the conceptual connection between
-- these 8 menus. Each large block in the figure represents a menu and the
-- smaller blocks within the large block represent items in that menu. Some
-- items contain references to other menus.
-- 
-- == A menu example
-- 
-- \<picture>
--   \<source srcset=\"menu-model-dark.svg\" media=\"(prefers-color-scheme: dark)\">
--   \<img src=\"menu-model-light.svg\" alt=\"menu model\">
-- \<\/picture>
-- 
-- Notice that the separators visible in the <http://developer.gnome.org/gio/stable/#an-example-menu example>
-- appear nowhere in the <http://developer.gnome.org/gio/stable/#a-menu-example menu model>. This is because
-- separators are not explicitly represented in the menu model. Instead,
-- a separator is inserted between any two non-empty sections of a menu.
-- Section items can have labels just like any other item. In that case,
-- a display system may show a section header instead of a separator.
-- 
-- The motivation for this abstract model of application controls is
-- that modern user interfaces tend to make these controls available
-- outside the application. Examples include global menus, jumplists,
-- dash boards, etc. To support such uses, it is necessary to ‘export’
-- information about actions and their representation in menus, which
-- is exactly what the action group exporter and the menu model exporter do for
-- t'GI.Gio.Interfaces.ActionGroup.ActionGroup' and t'GI.Gio.Objects.MenuModel.MenuModel'. The client-side
-- counterparts to make use of the exported information are
-- t'GI.Gio.Objects.DBusActionGroup.DBusActionGroup' and t'GI.Gio.Objects.DBusMenuModel.DBusMenuModel'.
-- 
-- The API of @GMenuModel@ is very generic, with iterators for the
-- attributes and links of an item, see
-- 'GI.Gio.Objects.MenuModel.menuModelIterateItemAttributes' and
-- 'GI.Gio.Objects.MenuModel.menuModelIterateItemLinks'. The ‘standard’ attributes and
-- link types have predefined names: @G_MENU_ATTRIBUTE_LABEL@,
-- @G_MENU_ATTRIBUTE_ACTION@, @G_MENU_ATTRIBUTE_TARGET@, @G_MENU_LINK_SECTION@
-- and @G_MENU_LINK_SUBMENU@.
-- 
-- Items in a @GMenuModel@ represent active controls if they refer to
-- an action that can get activated when the user interacts with the
-- menu item. The reference to the action is encoded by the string ID
-- in the @G_MENU_ATTRIBUTE_ACTION@ attribute. An action ID uniquely
-- identifies an action in an action group. Which action group(s) provide
-- actions depends on the context in which the menu model is used.
-- E.g. when the model is exported as the application menu of a
-- <https://docs.gtk.org/gtk4/class.Application.html `GtkApplication`>,
-- actions can be application-wide or window-specific (and thus come from
-- two different action groups). By convention, the application-wide actions
-- have names that start with @app.@, while the names of window-specific
-- actions start with @win.@.
-- 
-- While a wide variety of stateful actions is possible, the following
-- is the minimum that is expected to be supported by all users of exported
-- menu information:
-- 
-- * an action with no parameter type and no state
-- * an action with no parameter type and boolean state
-- * an action with string parameter type and string state
-- 
-- 
-- == Stateless
-- 
-- A stateless action typically corresponds to an ordinary menu item.
-- 
-- Selecting such a menu item will activate the action (with no parameter).
-- 
-- == Boolean State
-- 
-- An action with a boolean state will most typically be used with a ‘toggle’
-- or ‘switch’ menu item. The state can be set directly, but activating the
-- action (with no parameter) results in the state being toggled.
-- 
-- Selecting a toggle menu item will activate the action. The menu item should
-- be rendered as ‘checked’ when the state is true.
-- 
-- == String Parameter and State
-- 
-- Actions with string parameters and state will most typically be used to
-- represent an enumerated choice over the items available for a group of
-- radio menu items. Activating the action with a string parameter is
-- equivalent to setting that parameter as the state.
-- 
-- Radio menu items, in addition to being associated with the action, will
-- have a target value. Selecting that menu item will result in activation
-- of the action with the target value as the parameter. The menu item should
-- be rendered as ‘selected’ when the state of the action is equal to the
-- target value of the menu item.
-- 
-- /Since: 2.32/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.MenuModel
    ( 

-- * Exported types
    MenuModel(..)                           ,
    IsMenuModel                             ,
    toMenuModel                             ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isMutable]("GI.Gio.Objects.MenuModel#g:method:isMutable"), [itemsChanged]("GI.Gio.Objects.MenuModel#g:method:itemsChanged"), [iterateItemAttributes]("GI.Gio.Objects.MenuModel#g:method:iterateItemAttributes"), [iterateItemLinks]("GI.Gio.Objects.MenuModel#g:method:iterateItemLinks"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getItemAttributeValue]("GI.Gio.Objects.MenuModel#g:method:getItemAttributeValue"), [getItemLink]("GI.Gio.Objects.MenuModel#g:method:getItemLink"), [getNItems]("GI.Gio.Objects.MenuModel#g:method:getNItems"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveMenuModelMethod                  ,
#endif

-- ** getItemAttributeValue #method:getItemAttributeValue#

#if defined(ENABLE_OVERLOADING)
    MenuModelGetItemAttributeValueMethodInfo,
#endif
    menuModelGetItemAttributeValue          ,


-- ** getItemLink #method:getItemLink#

#if defined(ENABLE_OVERLOADING)
    MenuModelGetItemLinkMethodInfo          ,
#endif
    menuModelGetItemLink                    ,


-- ** getNItems #method:getNItems#

#if defined(ENABLE_OVERLOADING)
    MenuModelGetNItemsMethodInfo            ,
#endif
    menuModelGetNItems                      ,


-- ** isMutable #method:isMutable#

#if defined(ENABLE_OVERLOADING)
    MenuModelIsMutableMethodInfo            ,
#endif
    menuModelIsMutable                      ,


-- ** itemsChanged #method:itemsChanged#

#if defined(ENABLE_OVERLOADING)
    MenuModelItemsChangedMethodInfo         ,
#endif
    menuModelItemsChanged                   ,


-- ** iterateItemAttributes #method:iterateItemAttributes#

#if defined(ENABLE_OVERLOADING)
    MenuModelIterateItemAttributesMethodInfo,
#endif
    menuModelIterateItemAttributes          ,


-- ** iterateItemLinks #method:iterateItemLinks#

#if defined(ENABLE_OVERLOADING)
    MenuModelIterateItemLinksMethodInfo     ,
#endif
    menuModelIterateItemLinks               ,




 -- * Signals


-- ** itemsChanged #signal:itemsChanged#

    MenuModelItemsChangedCallback           ,
#if defined(ENABLE_OVERLOADING)
    MenuModelItemsChangedSignalInfo         ,
#endif
    afterMenuModelItemsChanged              ,
    onMenuModelItemsChanged                 ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuAttributeIter as Gio.MenuAttributeIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuLinkIter as Gio.MenuLinkIter

#else
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuAttributeIter as Gio.MenuAttributeIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuLinkIter as Gio.MenuLinkIter

#endif

-- | Memory-managed wrapper type.
newtype MenuModel = MenuModel (SP.ManagedPtr MenuModel)
    deriving (Eq)

instance SP.ManagedPtrNewtype MenuModel where
    toManagedPtr (MenuModel p) = p

foreign import ccall "g_menu_model_get_type"
    c_g_menu_model_get_type :: IO B.Types.GType

instance B.Types.TypedObject MenuModel where
    glibType = c_g_menu_model_get_type

instance B.Types.GObject MenuModel

-- | Type class for types which can be safely cast to `MenuModel`, for instance with `toMenuModel`.
class (SP.GObject o, O.IsDescendantOf MenuModel o) => IsMenuModel o
instance (SP.GObject o, O.IsDescendantOf MenuModel o) => IsMenuModel o

instance O.HasParentTypes MenuModel
type instance O.ParentTypes MenuModel = '[GObject.Object.Object]

-- | Cast to `MenuModel`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toMenuModel :: (MIO.MonadIO m, IsMenuModel o) => o -> m MenuModel
toMenuModel = MIO.liftIO . B.ManagedPtr.unsafeCastTo MenuModel

-- | Convert 'MenuModel' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe MenuModel) where
    gvalueGType_ = c_g_menu_model_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr MenuModel)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr MenuModel)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject MenuModel ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveMenuModelMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveMenuModelMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveMenuModelMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveMenuModelMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveMenuModelMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveMenuModelMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveMenuModelMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveMenuModelMethod "isMutable" o = MenuModelIsMutableMethodInfo
    ResolveMenuModelMethod "itemsChanged" o = MenuModelItemsChangedMethodInfo
    ResolveMenuModelMethod "iterateItemAttributes" o = MenuModelIterateItemAttributesMethodInfo
    ResolveMenuModelMethod "iterateItemLinks" o = MenuModelIterateItemLinksMethodInfo
    ResolveMenuModelMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveMenuModelMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveMenuModelMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveMenuModelMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveMenuModelMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveMenuModelMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveMenuModelMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveMenuModelMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveMenuModelMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveMenuModelMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveMenuModelMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveMenuModelMethod "getItemAttributeValue" o = MenuModelGetItemAttributeValueMethodInfo
    ResolveMenuModelMethod "getItemLink" o = MenuModelGetItemLinkMethodInfo
    ResolveMenuModelMethod "getNItems" o = MenuModelGetNItemsMethodInfo
    ResolveMenuModelMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveMenuModelMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveMenuModelMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveMenuModelMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveMenuModelMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveMenuModelMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveMenuModelMethod t MenuModel, O.OverloadedMethod info MenuModel p) => OL.IsLabel t (MenuModel -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveMenuModelMethod t MenuModel, O.OverloadedMethod info MenuModel p, R.HasField t MenuModel p) => R.HasField t MenuModel p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveMenuModelMethod t MenuModel, O.OverloadedMethodInfo info MenuModel) => OL.IsLabel t (O.MethodProxy info MenuModel) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal MenuModel::items-changed
-- | Emitted when a change has occurred to the menu.
-- 
-- The only changes that can occur to a menu is that items are removed
-- or added.  Items may not change (except by being removed and added
-- back in the same location).  This signal is capable of describing
-- both of those changes (at the same time).
-- 
-- The signal means that starting at the index /@position@/, /@removed@/
-- items were removed and /@added@/ items were added in their place.  If
-- /@removed@/ is zero then only items were added.  If /@added@/ is zero
-- then only items were removed.
-- 
-- As an example, if the menu contains items a, b, c, d (in that
-- order) and the signal (2, 1, 3) occurs then the new composition of
-- the menu will be a, b, _, _, _, d (with each _ representing some
-- new item).
-- 
-- Signal handlers may query the model (particularly the added items)
-- and expect to see the results of the modification that is being
-- reported.  The signal is emitted after the modification.
type MenuModelItemsChangedCallback =
    Int32
    -- ^ /@position@/: the position of the change
    -> Int32
    -- ^ /@removed@/: the number of items removed
    -> Int32
    -- ^ /@added@/: the number of items added
    -> IO ()

type C_MenuModelItemsChangedCallback =
    Ptr MenuModel ->                        -- object
    Int32 ->
    Int32 ->
    Int32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_MenuModelItemsChangedCallback`.
foreign import ccall "wrapper"
    mk_MenuModelItemsChangedCallback :: C_MenuModelItemsChangedCallback -> IO (FunPtr C_MenuModelItemsChangedCallback)

wrap_MenuModelItemsChangedCallback :: 
    GObject a => (a -> MenuModelItemsChangedCallback) ->
    C_MenuModelItemsChangedCallback
wrap_MenuModelItemsChangedCallback gi'cb gi'selfPtr position removed added _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  position removed added


-- | Connect a signal handler for the [itemsChanged](#signal:itemsChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' menuModel #itemsChanged callback
-- @
-- 
-- 
onMenuModelItemsChanged :: (IsMenuModel a, MonadIO m) => a -> ((?self :: a) => MenuModelItemsChangedCallback) -> m SignalHandlerId
onMenuModelItemsChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MenuModelItemsChangedCallback wrapped
    wrapped'' <- mk_MenuModelItemsChangedCallback wrapped'
    connectSignalFunPtr obj "items-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [itemsChanged](#signal:itemsChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' menuModel #itemsChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterMenuModelItemsChanged :: (IsMenuModel a, MonadIO m) => a -> ((?self :: a) => MenuModelItemsChangedCallback) -> m SignalHandlerId
afterMenuModelItemsChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_MenuModelItemsChangedCallback wrapped
    wrapped'' <- mk_MenuModelItemsChangedCallback wrapped'
    connectSignalFunPtr obj "items-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data MenuModelItemsChangedSignalInfo
instance SignalInfo MenuModelItemsChangedSignalInfo where
    type HaskellCallbackType MenuModelItemsChangedSignalInfo = MenuModelItemsChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_MenuModelItemsChangedCallback cb
        cb'' <- mk_MenuModelItemsChangedCallback cb'
        connectSignalFunPtr obj "items-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MenuModel::items-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MenuModel.html#g:signal:itemsChanged"})

#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList MenuModel
type instance O.AttributeList MenuModel = MenuModelAttributeList
type MenuModelAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList MenuModel = MenuModelSignalList
type MenuModelSignalList = ('[ '("itemsChanged", MenuModelItemsChangedSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method MenuModel::get_item_attribute_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MenuModel" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMenuModel" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "item_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the index of the item"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attribute"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the attribute to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "expected_type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the expected type of the attribute, or\n    %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_menu_model_get_item_attribute_value" g_menu_model_get_item_attribute_value :: 
    Ptr MenuModel ->                        -- model : TInterface (Name {namespace = "Gio", name = "MenuModel"})
    Int32 ->                                -- item_index : TBasicType TInt
    CString ->                              -- attribute : TBasicType TUTF8
    Ptr GLib.VariantType.VariantType ->     -- expected_type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr GVariant)

-- | Queries the item at position /@itemIndex@/ in /@model@/ for the attribute
-- specified by /@attribute@/.
-- 
-- If /@expectedType@/ is non-'P.Nothing' then it specifies the expected type of
-- the attribute.  If it is 'P.Nothing' then any type will be accepted.
-- 
-- If the attribute exists and matches /@expectedType@/ (or if the
-- expected type is unspecified) then the value is returned.
-- 
-- If the attribute does not exist, or does not match the expected type
-- then 'P.Nothing' is returned.
-- 
-- /Since: 2.32/
menuModelGetItemAttributeValue ::
    (B.CallStack.HasCallStack, MonadIO m, IsMenuModel a) =>
    a
    -- ^ /@model@/: a t'GI.Gio.Objects.MenuModel.MenuModel'
    -> Int32
    -- ^ /@itemIndex@/: the index of the item
    -> T.Text
    -- ^ /@attribute@/: the attribute to query
    -> Maybe (GLib.VariantType.VariantType)
    -- ^ /@expectedType@/: the expected type of the attribute, or
    --     'P.Nothing'
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the value of the attribute
menuModelGetItemAttributeValue model itemIndex attribute expectedType = liftIO $ do
    model' <- unsafeManagedPtrCastPtr model
    attribute' <- textToCString attribute
    maybeExpectedType <- case expectedType of
        Nothing -> return FP.nullPtr
        Just jExpectedType -> do
            jExpectedType' <- unsafeManagedPtrGetPtr jExpectedType
            return jExpectedType'
    result <- g_menu_model_get_item_attribute_value model' itemIndex attribute' maybeExpectedType
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr model
    whenJust expectedType touchManagedPtr
    freeMem attribute'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data MenuModelGetItemAttributeValueMethodInfo
instance (signature ~ (Int32 -> T.Text -> Maybe (GLib.VariantType.VariantType) -> m (Maybe GVariant)), MonadIO m, IsMenuModel a) => O.OverloadedMethod MenuModelGetItemAttributeValueMethodInfo a signature where
    overloadedMethod = menuModelGetItemAttributeValue

instance O.OverloadedMethodInfo MenuModelGetItemAttributeValueMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MenuModel.menuModelGetItemAttributeValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MenuModel.html#v:menuModelGetItemAttributeValue"
        })


#endif

-- method MenuModel::get_item_link
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MenuModel" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMenuModel" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "item_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the index of the item"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "link"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the link to query" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "MenuModel" })
-- throws : False
-- Skip return : False

foreign import ccall "g_menu_model_get_item_link" g_menu_model_get_item_link :: 
    Ptr MenuModel ->                        -- model : TInterface (Name {namespace = "Gio", name = "MenuModel"})
    Int32 ->                                -- item_index : TBasicType TInt
    CString ->                              -- link : TBasicType TUTF8
    IO (Ptr MenuModel)

-- | Queries the item at position /@itemIndex@/ in /@model@/ for the link
-- specified by /@link@/.
-- 
-- If the link exists, the linked t'GI.Gio.Objects.MenuModel.MenuModel' is returned.  If the link
-- does not exist, 'P.Nothing' is returned.
-- 
-- /Since: 2.32/
menuModelGetItemLink ::
    (B.CallStack.HasCallStack, MonadIO m, IsMenuModel a) =>
    a
    -- ^ /@model@/: a t'GI.Gio.Objects.MenuModel.MenuModel'
    -> Int32
    -- ^ /@itemIndex@/: the index of the item
    -> T.Text
    -- ^ /@link@/: the link to query
    -> m (Maybe MenuModel)
    -- ^ __Returns:__ the linked t'GI.Gio.Objects.MenuModel.MenuModel', or 'P.Nothing'
menuModelGetItemLink model itemIndex link = liftIO $ do
    model' <- unsafeManagedPtrCastPtr model
    link' <- textToCString link
    result <- g_menu_model_get_item_link model' itemIndex link'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject MenuModel) result'
        return result''
    touchManagedPtr model
    freeMem link'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data MenuModelGetItemLinkMethodInfo
instance (signature ~ (Int32 -> T.Text -> m (Maybe MenuModel)), MonadIO m, IsMenuModel a) => O.OverloadedMethod MenuModelGetItemLinkMethodInfo a signature where
    overloadedMethod = menuModelGetItemLink

instance O.OverloadedMethodInfo MenuModelGetItemLinkMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MenuModel.menuModelGetItemLink",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MenuModel.html#v:menuModelGetItemLink"
        })


#endif

-- method MenuModel::get_n_items
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MenuModel" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMenuModel" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_menu_model_get_n_items" g_menu_model_get_n_items :: 
    Ptr MenuModel ->                        -- model : TInterface (Name {namespace = "Gio", name = "MenuModel"})
    IO Int32

-- | Query the number of items in /@model@/.
-- 
-- /Since: 2.32/
menuModelGetNItems ::
    (B.CallStack.HasCallStack, MonadIO m, IsMenuModel a) =>
    a
    -- ^ /@model@/: a t'GI.Gio.Objects.MenuModel.MenuModel'
    -> m Int32
    -- ^ __Returns:__ the number of items
menuModelGetNItems model = liftIO $ do
    model' <- unsafeManagedPtrCastPtr model
    result <- g_menu_model_get_n_items model'
    touchManagedPtr model
    return result

#if defined(ENABLE_OVERLOADING)
data MenuModelGetNItemsMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsMenuModel a) => O.OverloadedMethod MenuModelGetNItemsMethodInfo a signature where
    overloadedMethod = menuModelGetNItems

instance O.OverloadedMethodInfo MenuModelGetNItemsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MenuModel.menuModelGetNItems",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MenuModel.html#v:menuModelGetNItems"
        })


#endif

-- method MenuModel::is_mutable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MenuModel" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMenuModel" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_menu_model_is_mutable" g_menu_model_is_mutable :: 
    Ptr MenuModel ->                        -- model : TInterface (Name {namespace = "Gio", name = "MenuModel"})
    IO CInt

-- | Queries if /@model@/ is mutable.
-- 
-- An immutable t'GI.Gio.Objects.MenuModel.MenuModel' will never emit the [MenuModel::itemsChanged]("GI.Gio.Objects.MenuModel#g:signal:itemsChanged")
-- signal. Consumers of the model may make optimisations accordingly.
-- 
-- /Since: 2.32/
menuModelIsMutable ::
    (B.CallStack.HasCallStack, MonadIO m, IsMenuModel a) =>
    a
    -- ^ /@model@/: a t'GI.Gio.Objects.MenuModel.MenuModel'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the model is mutable (ie: \"items-changed\" may be
    --     emitted).
menuModelIsMutable model = liftIO $ do
    model' <- unsafeManagedPtrCastPtr model
    result <- g_menu_model_is_mutable model'
    let result' = (/= 0) result
    touchManagedPtr model
    return result'

#if defined(ENABLE_OVERLOADING)
data MenuModelIsMutableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsMenuModel a) => O.OverloadedMethod MenuModelIsMutableMethodInfo a signature where
    overloadedMethod = menuModelIsMutable

instance O.OverloadedMethodInfo MenuModelIsMutableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MenuModel.menuModelIsMutable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MenuModel.html#v:menuModelIsMutable"
        })


#endif

-- method MenuModel::items_changed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MenuModel" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMenuModel" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "position"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the position of the change"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "removed"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of items removed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "added"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of items added"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_menu_model_items_changed" g_menu_model_items_changed :: 
    Ptr MenuModel ->                        -- model : TInterface (Name {namespace = "Gio", name = "MenuModel"})
    Int32 ->                                -- position : TBasicType TInt
    Int32 ->                                -- removed : TBasicType TInt
    Int32 ->                                -- added : TBasicType TInt
    IO ()

-- | Requests emission of the [MenuModel::itemsChanged]("GI.Gio.Objects.MenuModel#g:signal:itemsChanged") signal on /@model@/.
-- 
-- This function should never be called except by t'GI.Gio.Objects.MenuModel.MenuModel'
-- subclasses.  Any other calls to this function will very likely lead
-- to a violation of the interface of the model.
-- 
-- The implementation should update its internal representation of the
-- menu before emitting the signal.  The implementation should further
-- expect to receive queries about the new state of the menu (and
-- particularly added menu items) while signal handlers are running.
-- 
-- The implementation must dispatch this call directly from a mainloop
-- entry and not in response to calls -- particularly those from the
-- t'GI.Gio.Objects.MenuModel.MenuModel' API.  Said another way: the menu must not change while
-- user code is running without returning to the mainloop.
-- 
-- /Since: 2.32/
menuModelItemsChanged ::
    (B.CallStack.HasCallStack, MonadIO m, IsMenuModel a) =>
    a
    -- ^ /@model@/: a t'GI.Gio.Objects.MenuModel.MenuModel'
    -> Int32
    -- ^ /@position@/: the position of the change
    -> Int32
    -- ^ /@removed@/: the number of items removed
    -> Int32
    -- ^ /@added@/: the number of items added
    -> m ()
menuModelItemsChanged model position removed added = liftIO $ do
    model' <- unsafeManagedPtrCastPtr model
    g_menu_model_items_changed model' position removed added
    touchManagedPtr model
    return ()

#if defined(ENABLE_OVERLOADING)
data MenuModelItemsChangedMethodInfo
instance (signature ~ (Int32 -> Int32 -> Int32 -> m ()), MonadIO m, IsMenuModel a) => O.OverloadedMethod MenuModelItemsChangedMethodInfo a signature where
    overloadedMethod = menuModelItemsChanged

instance O.OverloadedMethodInfo MenuModelItemsChangedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MenuModel.menuModelItemsChanged",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MenuModel.html#v:menuModelItemsChanged"
        })


#endif

-- method MenuModel::iterate_item_attributes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MenuModel" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMenuModel" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "item_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the index of the item"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "MenuAttributeIter" })
-- throws : False
-- Skip return : False

foreign import ccall "g_menu_model_iterate_item_attributes" g_menu_model_iterate_item_attributes :: 
    Ptr MenuModel ->                        -- model : TInterface (Name {namespace = "Gio", name = "MenuModel"})
    Int32 ->                                -- item_index : TBasicType TInt
    IO (Ptr Gio.MenuAttributeIter.MenuAttributeIter)

-- | Creates a t'GI.Gio.Objects.MenuAttributeIter.MenuAttributeIter' to iterate over the attributes of
-- the item at position /@itemIndex@/ in /@model@/.
-- 
-- You must free the iterator with 'GI.GObject.Objects.Object.objectUnref' when you are done.
-- 
-- /Since: 2.32/
menuModelIterateItemAttributes ::
    (B.CallStack.HasCallStack, MonadIO m, IsMenuModel a) =>
    a
    -- ^ /@model@/: a t'GI.Gio.Objects.MenuModel.MenuModel'
    -> Int32
    -- ^ /@itemIndex@/: the index of the item
    -> m Gio.MenuAttributeIter.MenuAttributeIter
    -- ^ __Returns:__ a new t'GI.Gio.Objects.MenuAttributeIter.MenuAttributeIter'
menuModelIterateItemAttributes model itemIndex = liftIO $ do
    model' <- unsafeManagedPtrCastPtr model
    result <- g_menu_model_iterate_item_attributes model' itemIndex
    checkUnexpectedReturnNULL "menuModelIterateItemAttributes" result
    result' <- (wrapObject Gio.MenuAttributeIter.MenuAttributeIter) result
    touchManagedPtr model
    return result'

#if defined(ENABLE_OVERLOADING)
data MenuModelIterateItemAttributesMethodInfo
instance (signature ~ (Int32 -> m Gio.MenuAttributeIter.MenuAttributeIter), MonadIO m, IsMenuModel a) => O.OverloadedMethod MenuModelIterateItemAttributesMethodInfo a signature where
    overloadedMethod = menuModelIterateItemAttributes

instance O.OverloadedMethodInfo MenuModelIterateItemAttributesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MenuModel.menuModelIterateItemAttributes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MenuModel.html#v:menuModelIterateItemAttributes"
        })


#endif

-- method MenuModel::iterate_item_links
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MenuModel" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GMenuModel" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "item_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the index of the item"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "MenuLinkIter" })
-- throws : False
-- Skip return : False

foreign import ccall "g_menu_model_iterate_item_links" g_menu_model_iterate_item_links :: 
    Ptr MenuModel ->                        -- model : TInterface (Name {namespace = "Gio", name = "MenuModel"})
    Int32 ->                                -- item_index : TBasicType TInt
    IO (Ptr Gio.MenuLinkIter.MenuLinkIter)

-- | Creates a t'GI.Gio.Objects.MenuLinkIter.MenuLinkIter' to iterate over the links of the item at
-- position /@itemIndex@/ in /@model@/.
-- 
-- You must free the iterator with 'GI.GObject.Objects.Object.objectUnref' when you are done.
-- 
-- /Since: 2.32/
menuModelIterateItemLinks ::
    (B.CallStack.HasCallStack, MonadIO m, IsMenuModel a) =>
    a
    -- ^ /@model@/: a t'GI.Gio.Objects.MenuModel.MenuModel'
    -> Int32
    -- ^ /@itemIndex@/: the index of the item
    -> m Gio.MenuLinkIter.MenuLinkIter
    -- ^ __Returns:__ a new t'GI.Gio.Objects.MenuLinkIter.MenuLinkIter'
menuModelIterateItemLinks model itemIndex = liftIO $ do
    model' <- unsafeManagedPtrCastPtr model
    result <- g_menu_model_iterate_item_links model' itemIndex
    checkUnexpectedReturnNULL "menuModelIterateItemLinks" result
    result' <- (wrapObject Gio.MenuLinkIter.MenuLinkIter) result
    touchManagedPtr model
    return result'

#if defined(ENABLE_OVERLOADING)
data MenuModelIterateItemLinksMethodInfo
instance (signature ~ (Int32 -> m Gio.MenuLinkIter.MenuLinkIter), MonadIO m, IsMenuModel a) => O.OverloadedMethod MenuModelIterateItemLinksMethodInfo a signature where
    overloadedMethod = menuModelIterateItemLinks

instance O.OverloadedMethodInfo MenuModelIterateItemLinksMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.MenuModel.menuModelIterateItemLinks",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-MenuModel.html#v:menuModelIterateItemLinks"
        })


#endif


