{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GDBusServer@ is a helper for listening to and accepting D-Bus
-- connections. This can be used to create a new D-Bus server, allowing two
-- peers to use the D-Bus protocol for their own specialized communication.
-- A server instance provided in this way will not perform message routing or
-- implement the
-- <https://dbus.freedesktop.org/doc/dbus-specification.html#message-bus-messages `org.freedesktop.DBus` interface>.
-- 
-- To just export an object on a well-known name on a message bus, such as the
-- session or system bus, you should instead use 'GI.Gio.Functions.busOwnName'.
-- 
-- An example of peer-to-peer communication with GDBus can be found
-- in <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-peer.c gdbus-example-peer.c>.
-- 
-- Note that a minimal @GDBusServer@ will accept connections from any
-- peer. In many use-cases it will be necessary to add a
-- t'GI.Gio.Objects.DBusAuthObserver.DBusAuthObserver' that only accepts connections that have
-- successfully authenticated as the same user that is running the
-- @GDBusServer@. Since GLib 2.68 this can be achieved more simply by passing
-- the @G_DBUS_SERVER_FLAGS_AUTHENTICATION_REQUIRE_SAME_USER@ flag to the
-- server.
-- 
-- /Since: 2.26/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.DBusServer
    ( 

-- * Exported types
    DBusServer(..)                          ,
    IsDBusServer                            ,
    toDBusServer                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [init]("GI.Gio.Interfaces.Initable#g:method:init"), [isActive]("GI.Gio.Objects.DBusServer#g:method:isActive"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [start]("GI.Gio.Objects.DBusServer#g:method:start"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [stop]("GI.Gio.Objects.DBusServer#g:method:stop"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getClientAddress]("GI.Gio.Objects.DBusServer#g:method:getClientAddress"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getFlags]("GI.Gio.Objects.DBusServer#g:method:getFlags"), [getGuid]("GI.Gio.Objects.DBusServer#g:method:getGuid"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveDBusServerMethod                 ,
#endif

-- ** getClientAddress #method:getClientAddress#

#if defined(ENABLE_OVERLOADING)
    DBusServerGetClientAddressMethodInfo    ,
#endif
    dBusServerGetClientAddress              ,


-- ** getFlags #method:getFlags#

#if defined(ENABLE_OVERLOADING)
    DBusServerGetFlagsMethodInfo            ,
#endif
    dBusServerGetFlags                      ,


-- ** getGuid #method:getGuid#

#if defined(ENABLE_OVERLOADING)
    DBusServerGetGuidMethodInfo             ,
#endif
    dBusServerGetGuid                       ,


-- ** isActive #method:isActive#

#if defined(ENABLE_OVERLOADING)
    DBusServerIsActiveMethodInfo            ,
#endif
    dBusServerIsActive                      ,


-- ** newSync #method:newSync#

    dBusServerNewSync                       ,


-- ** start #method:start#

#if defined(ENABLE_OVERLOADING)
    DBusServerStartMethodInfo               ,
#endif
    dBusServerStart                         ,


-- ** stop #method:stop#

#if defined(ENABLE_OVERLOADING)
    DBusServerStopMethodInfo                ,
#endif
    dBusServerStop                          ,




 -- * Properties


-- ** active #attr:active#
-- | Whether the server is currently active.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusServerActivePropertyInfo            ,
#endif
#if defined(ENABLE_OVERLOADING)
    dBusServerActive                        ,
#endif
    getDBusServerActive                     ,


-- ** address #attr:address#
-- | The D-Bus address to listen on.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusServerAddressPropertyInfo           ,
#endif
    constructDBusServerAddress              ,
#if defined(ENABLE_OVERLOADING)
    dBusServerAddress                       ,
#endif
    getDBusServerAddress                    ,


-- ** authenticationObserver #attr:authenticationObserver#
-- | A t'GI.Gio.Objects.DBusAuthObserver.DBusAuthObserver' object to assist in the authentication process or 'P.Nothing'.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusServerAuthenticationObserverPropertyInfo,
#endif
    constructDBusServerAuthenticationObserver,
#if defined(ENABLE_OVERLOADING)
    dBusServerAuthenticationObserver        ,
#endif
    getDBusServerAuthenticationObserver     ,


-- ** clientAddress #attr:clientAddress#
-- | The D-Bus address that clients can use.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusServerClientAddressPropertyInfo     ,
#endif
#if defined(ENABLE_OVERLOADING)
    dBusServerClientAddress                 ,
#endif
    getDBusServerClientAddress              ,


-- ** flags #attr:flags#
-- | Flags from the t'GI.Gio.Flags.DBusServerFlags' enumeration.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusServerFlagsPropertyInfo             ,
#endif
    constructDBusServerFlags                ,
#if defined(ENABLE_OVERLOADING)
    dBusServerFlags                         ,
#endif
    getDBusServerFlags                      ,


-- ** guid #attr:guid#
-- | The GUID of the server.
-- 
-- See [DBusConnection:guid]("GI.Gio.Objects.DBusConnection#g:attr:guid") for more details.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusServerGuidPropertyInfo              ,
#endif
    constructDBusServerGuid                 ,
#if defined(ENABLE_OVERLOADING)
    dBusServerGuid                          ,
#endif
    getDBusServerGuid                       ,




 -- * Signals


-- ** newConnection #signal:newConnection#

    DBusServerNewConnectionCallback         ,
#if defined(ENABLE_OVERLOADING)
    DBusServerNewConnectionSignalInfo       ,
#endif
    afterDBusServerNewConnection            ,
    onDBusServerNewConnection               ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncInitable as Gio.AsyncInitable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.Credentials as Gio.Credentials
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusAuthObserver as Gio.DBusAuthObserver
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusConnection as Gio.DBusConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusMessage as Gio.DBusMessage
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuAttributeIter as Gio.MenuAttributeIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuLinkIter as Gio.MenuLinkIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuModel as Gio.MenuModel
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.UnixFDList as Gio.UnixFDList
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusAnnotationInfo as Gio.DBusAnnotationInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusArgInfo as Gio.DBusArgInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusInterfaceInfo as Gio.DBusInterfaceInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusMethodInfo as Gio.DBusMethodInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusPropertyInfo as Gio.DBusPropertyInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusSignalInfo as Gio.DBusSignalInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusSubtreeVTable as Gio.DBusSubtreeVTable
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusAuthObserver as Gio.DBusAuthObserver
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusConnection as Gio.DBusConnection

#endif

-- | Memory-managed wrapper type.
newtype DBusServer = DBusServer (SP.ManagedPtr DBusServer)
    deriving (Eq)

instance SP.ManagedPtrNewtype DBusServer where
    toManagedPtr (DBusServer p) = p

foreign import ccall "g_dbus_server_get_type"
    c_g_dbus_server_get_type :: IO B.Types.GType

instance B.Types.TypedObject DBusServer where
    glibType = c_g_dbus_server_get_type

instance B.Types.GObject DBusServer

-- | Type class for types which can be safely cast to `DBusServer`, for instance with `toDBusServer`.
class (SP.GObject o, O.IsDescendantOf DBusServer o) => IsDBusServer o
instance (SP.GObject o, O.IsDescendantOf DBusServer o) => IsDBusServer o

instance O.HasParentTypes DBusServer
type instance O.ParentTypes DBusServer = '[GObject.Object.Object, Gio.Initable.Initable]

-- | Cast to `DBusServer`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toDBusServer :: (MIO.MonadIO m, IsDBusServer o) => o -> m DBusServer
toDBusServer = MIO.liftIO . B.ManagedPtr.unsafeCastTo DBusServer

-- | Convert 'DBusServer' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe DBusServer) where
    gvalueGType_ = c_g_dbus_server_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr DBusServer)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr DBusServer)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject DBusServer ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveDBusServerMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveDBusServerMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveDBusServerMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveDBusServerMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveDBusServerMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveDBusServerMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveDBusServerMethod "init" o = Gio.Initable.InitableInitMethodInfo
    ResolveDBusServerMethod "isActive" o = DBusServerIsActiveMethodInfo
    ResolveDBusServerMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveDBusServerMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveDBusServerMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveDBusServerMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveDBusServerMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveDBusServerMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveDBusServerMethod "start" o = DBusServerStartMethodInfo
    ResolveDBusServerMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveDBusServerMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveDBusServerMethod "stop" o = DBusServerStopMethodInfo
    ResolveDBusServerMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveDBusServerMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveDBusServerMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveDBusServerMethod "getClientAddress" o = DBusServerGetClientAddressMethodInfo
    ResolveDBusServerMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveDBusServerMethod "getFlags" o = DBusServerGetFlagsMethodInfo
    ResolveDBusServerMethod "getGuid" o = DBusServerGetGuidMethodInfo
    ResolveDBusServerMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveDBusServerMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveDBusServerMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveDBusServerMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveDBusServerMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveDBusServerMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveDBusServerMethod t DBusServer, O.OverloadedMethod info DBusServer p) => OL.IsLabel t (DBusServer -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveDBusServerMethod t DBusServer, O.OverloadedMethod info DBusServer p, R.HasField t DBusServer p) => R.HasField t DBusServer p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveDBusServerMethod t DBusServer, O.OverloadedMethodInfo info DBusServer) => OL.IsLabel t (O.MethodProxy info DBusServer) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal DBusServer::new-connection
-- | Emitted when a new authenticated connection has been made. Use
-- 'GI.Gio.Objects.DBusConnection.dBusConnectionGetPeerCredentials' to figure out what
-- identity (if any), was authenticated.
-- 
-- If you want to accept the connection, take a reference to the
-- /@connection@/ object and return 'P.True'. When you are done with the
-- connection call 'GI.Gio.Objects.DBusConnection.dBusConnectionClose' and give up your
-- reference. Note that the other peer may disconnect at any time -
-- a typical thing to do when accepting a connection is to listen to
-- the [DBusConnection::closed]("GI.Gio.Objects.DBusConnection#g:signal:closed") signal.
-- 
-- If [DBusServer:flags]("GI.Gio.Objects.DBusServer#g:attr:flags") contains 'GI.Gio.Flags.DBusServerFlagsRunInThread'
-- then the signal is emitted in a new thread dedicated to the
-- connection. Otherwise the signal is emitted in the thread-default
-- main context (see 'GI.GLib.Structs.MainContext.mainContextPushThreadDefault')
-- of the thread that /@server@/ was constructed in.
-- 
-- You are guaranteed that signal handlers for this signal runs
-- before incoming messages on /@connection@/ are processed. This means
-- that it\'s suitable to call @/g_dbus_connection_register_object()/@ or
-- similar from the signal handler.
-- 
-- /Since: 2.26/
type DBusServerNewConnectionCallback =
    Gio.DBusConnection.DBusConnection
    -- ^ /@connection@/: A t'GI.Gio.Objects.DBusConnection.DBusConnection' for the new connection.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to claim /@connection@/, 'P.False' to let other handlers
    -- run.

type C_DBusServerNewConnectionCallback =
    Ptr DBusServer ->                       -- object
    Ptr Gio.DBusConnection.DBusConnection ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_DBusServerNewConnectionCallback`.
foreign import ccall "wrapper"
    mk_DBusServerNewConnectionCallback :: C_DBusServerNewConnectionCallback -> IO (FunPtr C_DBusServerNewConnectionCallback)

wrap_DBusServerNewConnectionCallback :: 
    GObject a => (a -> DBusServerNewConnectionCallback) ->
    C_DBusServerNewConnectionCallback
wrap_DBusServerNewConnectionCallback gi'cb gi'selfPtr connection _ = do
    connection' <- (newObject Gio.DBusConnection.DBusConnection) connection
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  connection'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [newConnection](#signal:newConnection) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dBusServer #newConnection callback
-- @
-- 
-- 
onDBusServerNewConnection :: (IsDBusServer a, MonadIO m) => a -> ((?self :: a) => DBusServerNewConnectionCallback) -> m SignalHandlerId
onDBusServerNewConnection obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusServerNewConnectionCallback wrapped
    wrapped'' <- mk_DBusServerNewConnectionCallback wrapped'
    connectSignalFunPtr obj "new-connection" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [newConnection](#signal:newConnection) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dBusServer #newConnection callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDBusServerNewConnection :: (IsDBusServer a, MonadIO m) => a -> ((?self :: a) => DBusServerNewConnectionCallback) -> m SignalHandlerId
afterDBusServerNewConnection obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusServerNewConnectionCallback wrapped
    wrapped'' <- mk_DBusServerNewConnectionCallback wrapped'
    connectSignalFunPtr obj "new-connection" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DBusServerNewConnectionSignalInfo
instance SignalInfo DBusServerNewConnectionSignalInfo where
    type HaskellCallbackType DBusServerNewConnectionSignalInfo = DBusServerNewConnectionCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DBusServerNewConnectionCallback cb
        cb'' <- mk_DBusServerNewConnectionCallback cb'
        connectSignalFunPtr obj "new-connection" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer::new-connection"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#g:signal:newConnection"})

#endif

-- VVV Prop "active"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@active@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusServer #active
-- @
getDBusServerActive :: (MonadIO m, IsDBusServer o) => o -> m Bool
getDBusServerActive obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "active"

#if defined(ENABLE_OVERLOADING)
data DBusServerActivePropertyInfo
instance AttrInfo DBusServerActivePropertyInfo where
    type AttrAllowedOps DBusServerActivePropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint DBusServerActivePropertyInfo = IsDBusServer
    type AttrSetTypeConstraint DBusServerActivePropertyInfo = (~) ()
    type AttrTransferTypeConstraint DBusServerActivePropertyInfo = (~) ()
    type AttrTransferType DBusServerActivePropertyInfo = ()
    type AttrGetType DBusServerActivePropertyInfo = Bool
    type AttrLabel DBusServerActivePropertyInfo = "active"
    type AttrOrigin DBusServerActivePropertyInfo = DBusServer
    attrGet = getDBusServerActive
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.active"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#g:attr:active"
        })
#endif

-- VVV Prop "address"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@address@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusServer #address
-- @
getDBusServerAddress :: (MonadIO m, IsDBusServer o) => o -> m (Maybe T.Text)
getDBusServerAddress obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "address"

-- | Construct a `GValueConstruct` with valid value for the “@address@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusServerAddress :: (IsDBusServer o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructDBusServerAddress val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "address" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data DBusServerAddressPropertyInfo
instance AttrInfo DBusServerAddressPropertyInfo where
    type AttrAllowedOps DBusServerAddressPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusServerAddressPropertyInfo = IsDBusServer
    type AttrSetTypeConstraint DBusServerAddressPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint DBusServerAddressPropertyInfo = (~) T.Text
    type AttrTransferType DBusServerAddressPropertyInfo = T.Text
    type AttrGetType DBusServerAddressPropertyInfo = (Maybe T.Text)
    type AttrLabel DBusServerAddressPropertyInfo = "address"
    type AttrOrigin DBusServerAddressPropertyInfo = DBusServer
    attrGet = getDBusServerAddress
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusServerAddress
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.address"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#g:attr:address"
        })
#endif

-- VVV Prop "authentication-observer"
   -- Type: TInterface (Name {namespace = "Gio", name = "DBusAuthObserver"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@authentication-observer@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusServer #authenticationObserver
-- @
getDBusServerAuthenticationObserver :: (MonadIO m, IsDBusServer o) => o -> m (Maybe Gio.DBusAuthObserver.DBusAuthObserver)
getDBusServerAuthenticationObserver obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "authentication-observer" Gio.DBusAuthObserver.DBusAuthObserver

-- | Construct a `GValueConstruct` with valid value for the “@authentication-observer@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusServerAuthenticationObserver :: (IsDBusServer o, MIO.MonadIO m, Gio.DBusAuthObserver.IsDBusAuthObserver a) => a -> m (GValueConstruct o)
constructDBusServerAuthenticationObserver val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "authentication-observer" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data DBusServerAuthenticationObserverPropertyInfo
instance AttrInfo DBusServerAuthenticationObserverPropertyInfo where
    type AttrAllowedOps DBusServerAuthenticationObserverPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusServerAuthenticationObserverPropertyInfo = IsDBusServer
    type AttrSetTypeConstraint DBusServerAuthenticationObserverPropertyInfo = Gio.DBusAuthObserver.IsDBusAuthObserver
    type AttrTransferTypeConstraint DBusServerAuthenticationObserverPropertyInfo = Gio.DBusAuthObserver.IsDBusAuthObserver
    type AttrTransferType DBusServerAuthenticationObserverPropertyInfo = Gio.DBusAuthObserver.DBusAuthObserver
    type AttrGetType DBusServerAuthenticationObserverPropertyInfo = (Maybe Gio.DBusAuthObserver.DBusAuthObserver)
    type AttrLabel DBusServerAuthenticationObserverPropertyInfo = "authentication-observer"
    type AttrOrigin DBusServerAuthenticationObserverPropertyInfo = DBusServer
    attrGet = getDBusServerAuthenticationObserver
    attrSet = undefined
    attrTransfer _ v = do
        unsafeCastTo Gio.DBusAuthObserver.DBusAuthObserver v
    attrConstruct = constructDBusServerAuthenticationObserver
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.authenticationObserver"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#g:attr:authenticationObserver"
        })
#endif

-- VVV Prop "client-address"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@client-address@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusServer #clientAddress
-- @
getDBusServerClientAddress :: (MonadIO m, IsDBusServer o) => o -> m T.Text
getDBusServerClientAddress obj = MIO.liftIO $ checkUnexpectedNothing "getDBusServerClientAddress" $ B.Properties.getObjectPropertyString obj "client-address"

#if defined(ENABLE_OVERLOADING)
data DBusServerClientAddressPropertyInfo
instance AttrInfo DBusServerClientAddressPropertyInfo where
    type AttrAllowedOps DBusServerClientAddressPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusServerClientAddressPropertyInfo = IsDBusServer
    type AttrSetTypeConstraint DBusServerClientAddressPropertyInfo = (~) ()
    type AttrTransferTypeConstraint DBusServerClientAddressPropertyInfo = (~) ()
    type AttrTransferType DBusServerClientAddressPropertyInfo = ()
    type AttrGetType DBusServerClientAddressPropertyInfo = T.Text
    type AttrLabel DBusServerClientAddressPropertyInfo = "client-address"
    type AttrOrigin DBusServerClientAddressPropertyInfo = DBusServer
    attrGet = getDBusServerClientAddress
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.clientAddress"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#g:attr:clientAddress"
        })
#endif

-- VVV Prop "flags"
   -- Type: TInterface (Name {namespace = "Gio", name = "DBusServerFlags"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@flags@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusServer #flags
-- @
getDBusServerFlags :: (MonadIO m, IsDBusServer o) => o -> m [Gio.Flags.DBusServerFlags]
getDBusServerFlags obj = MIO.liftIO $ B.Properties.getObjectPropertyFlags obj "flags"

-- | Construct a `GValueConstruct` with valid value for the “@flags@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusServerFlags :: (IsDBusServer o, MIO.MonadIO m) => [Gio.Flags.DBusServerFlags] -> m (GValueConstruct o)
constructDBusServerFlags val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFlags "flags" val

#if defined(ENABLE_OVERLOADING)
data DBusServerFlagsPropertyInfo
instance AttrInfo DBusServerFlagsPropertyInfo where
    type AttrAllowedOps DBusServerFlagsPropertyInfo = '[ 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint DBusServerFlagsPropertyInfo = IsDBusServer
    type AttrSetTypeConstraint DBusServerFlagsPropertyInfo = (~) [Gio.Flags.DBusServerFlags]
    type AttrTransferTypeConstraint DBusServerFlagsPropertyInfo = (~) [Gio.Flags.DBusServerFlags]
    type AttrTransferType DBusServerFlagsPropertyInfo = [Gio.Flags.DBusServerFlags]
    type AttrGetType DBusServerFlagsPropertyInfo = [Gio.Flags.DBusServerFlags]
    type AttrLabel DBusServerFlagsPropertyInfo = "flags"
    type AttrOrigin DBusServerFlagsPropertyInfo = DBusServer
    attrGet = getDBusServerFlags
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusServerFlags
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.flags"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#g:attr:flags"
        })
#endif

-- VVV Prop "guid"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@guid@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusServer #guid
-- @
getDBusServerGuid :: (MonadIO m, IsDBusServer o) => o -> m T.Text
getDBusServerGuid obj = MIO.liftIO $ checkUnexpectedNothing "getDBusServerGuid" $ B.Properties.getObjectPropertyString obj "guid"

-- | Construct a `GValueConstruct` with valid value for the “@guid@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusServerGuid :: (IsDBusServer o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructDBusServerGuid val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "guid" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data DBusServerGuidPropertyInfo
instance AttrInfo DBusServerGuidPropertyInfo where
    type AttrAllowedOps DBusServerGuidPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusServerGuidPropertyInfo = IsDBusServer
    type AttrSetTypeConstraint DBusServerGuidPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint DBusServerGuidPropertyInfo = (~) T.Text
    type AttrTransferType DBusServerGuidPropertyInfo = T.Text
    type AttrGetType DBusServerGuidPropertyInfo = T.Text
    type AttrLabel DBusServerGuidPropertyInfo = "guid"
    type AttrOrigin DBusServerGuidPropertyInfo = DBusServer
    attrGet = getDBusServerGuid
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusServerGuid
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.guid"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#g:attr:guid"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList DBusServer
type instance O.AttributeList DBusServer = DBusServerAttributeList
type DBusServerAttributeList = ('[ '("active", DBusServerActivePropertyInfo), '("address", DBusServerAddressPropertyInfo), '("authenticationObserver", DBusServerAuthenticationObserverPropertyInfo), '("clientAddress", DBusServerClientAddressPropertyInfo), '("flags", DBusServerFlagsPropertyInfo), '("guid", DBusServerGuidPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
dBusServerActive :: AttrLabelProxy "active"
dBusServerActive = AttrLabelProxy

dBusServerAddress :: AttrLabelProxy "address"
dBusServerAddress = AttrLabelProxy

dBusServerAuthenticationObserver :: AttrLabelProxy "authenticationObserver"
dBusServerAuthenticationObserver = AttrLabelProxy

dBusServerClientAddress :: AttrLabelProxy "clientAddress"
dBusServerClientAddress = AttrLabelProxy

dBusServerFlags :: AttrLabelProxy "flags"
dBusServerFlags = AttrLabelProxy

dBusServerGuid :: AttrLabelProxy "guid"
dBusServerGuid = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList DBusServer = DBusServerSignalList
type DBusServerSignalList = ('[ '("newConnection", DBusServerNewConnectionSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method DBusServer::new_sync
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "address"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A D-Bus address." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusServerFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags from the #GDBusServerFlags enumeration."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "guid"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A D-Bus GUID." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "observer"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusAuthObserver" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusAuthObserver or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "DBusServer" })
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_server_new_sync" g_dbus_server_new_sync :: 
    CString ->                              -- address : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusServerFlags"})
    CString ->                              -- guid : TBasicType TUTF8
    Ptr Gio.DBusAuthObserver.DBusAuthObserver -> -- observer : TInterface (Name {namespace = "Gio", name = "DBusAuthObserver"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr DBusServer)

-- | Creates a new D-Bus server that listens on the first address in
-- /@address@/ that works.
-- 
-- Once constructed, you can use 'GI.Gio.Objects.DBusServer.dBusServerGetClientAddress' to
-- get a D-Bus address string that clients can use to connect.
-- 
-- To have control over the available authentication mechanisms and
-- the users that are authorized to connect, it is strongly recommended
-- to provide a non-'P.Nothing' t'GI.Gio.Objects.DBusAuthObserver.DBusAuthObserver'.
-- 
-- Connect to the [DBusServer::newConnection]("GI.Gio.Objects.DBusServer#g:signal:newConnection") signal to handle
-- incoming connections.
-- 
-- The returned t'GI.Gio.Objects.DBusServer.DBusServer' isn\'t active - you have to start it with
-- 'GI.Gio.Objects.DBusServer.dBusServerStart'.
-- 
-- t'GI.Gio.Objects.DBusServer.DBusServer' is used in this <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-peer.c example>.
-- 
-- This is a synchronous failable constructor. There is currently no
-- asynchronous version.
-- 
-- /Since: 2.26/
dBusServerNewSync ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.DBusAuthObserver.IsDBusAuthObserver a, Gio.Cancellable.IsCancellable b) =>
    T.Text
    -- ^ /@address@/: A D-Bus address.
    -> [Gio.Flags.DBusServerFlags]
    -- ^ /@flags@/: Flags from the t'GI.Gio.Flags.DBusServerFlags' enumeration.
    -> T.Text
    -- ^ /@guid@/: A D-Bus GUID.
    -> Maybe (a)
    -- ^ /@observer@/: A t'GI.Gio.Objects.DBusAuthObserver.DBusAuthObserver' or 'P.Nothing'.
    -> Maybe (b)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> m DBusServer
    -- ^ __Returns:__ A t'GI.Gio.Objects.DBusServer.DBusServer' or 'P.Nothing' if /@error@/ is set. Free with
    -- 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusServerNewSync address flags guid observer cancellable = liftIO $ do
    address' <- textToCString address
    let flags' = gflagsToWord flags
    guid' <- textToCString guid
    maybeObserver <- case observer of
        Nothing -> return FP.nullPtr
        Just jObserver -> do
            jObserver' <- unsafeManagedPtrCastPtr jObserver
            return jObserver'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_dbus_server_new_sync address' flags' guid' maybeObserver maybeCancellable
        checkUnexpectedReturnNULL "dBusServerNewSync" result
        result' <- (wrapObject DBusServer) result
        whenJust observer touchManagedPtr
        whenJust cancellable touchManagedPtr
        freeMem address'
        freeMem guid'
        return result'
     ) (do
        freeMem address'
        freeMem guid'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method DBusServer::get_client_address
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "server"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusServer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusServer." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_server_get_client_address" g_dbus_server_get_client_address :: 
    Ptr DBusServer ->                       -- server : TInterface (Name {namespace = "Gio", name = "DBusServer"})
    IO CString

-- | Gets a
-- <https://dbus.freedesktop.org/doc/dbus-specification.html#addresses D-Bus address>
-- string that can be used by clients to connect to /@server@/.
-- 
-- This is valid and non-empty if initializing the t'GI.Gio.Objects.DBusServer.DBusServer' succeeded.
-- 
-- /Since: 2.26/
dBusServerGetClientAddress ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusServer a) =>
    a
    -- ^ /@server@/: A t'GI.Gio.Objects.DBusServer.DBusServer'.
    -> m T.Text
    -- ^ __Returns:__ A D-Bus address string. Do not free, the string is owned
    -- by /@server@/.
dBusServerGetClientAddress server = liftIO $ do
    server' <- unsafeManagedPtrCastPtr server
    result <- g_dbus_server_get_client_address server'
    checkUnexpectedReturnNULL "dBusServerGetClientAddress" result
    result' <- cstringToText result
    touchManagedPtr server
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusServerGetClientAddressMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsDBusServer a) => O.OverloadedMethod DBusServerGetClientAddressMethodInfo a signature where
    overloadedMethod = dBusServerGetClientAddress

instance O.OverloadedMethodInfo DBusServerGetClientAddressMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.dBusServerGetClientAddress",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#v:dBusServerGetClientAddress"
        })


#endif

-- method DBusServer::get_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "server"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusServer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusServer." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "DBusServerFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_server_get_flags" g_dbus_server_get_flags :: 
    Ptr DBusServer ->                       -- server : TInterface (Name {namespace = "Gio", name = "DBusServer"})
    IO CUInt

-- | Gets the flags for /@server@/.
-- 
-- /Since: 2.26/
dBusServerGetFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusServer a) =>
    a
    -- ^ /@server@/: A t'GI.Gio.Objects.DBusServer.DBusServer'.
    -> m [Gio.Flags.DBusServerFlags]
    -- ^ __Returns:__ A set of flags from the t'GI.Gio.Flags.DBusServerFlags' enumeration.
dBusServerGetFlags server = liftIO $ do
    server' <- unsafeManagedPtrCastPtr server
    result <- g_dbus_server_get_flags server'
    let result' = wordToGFlags result
    touchManagedPtr server
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusServerGetFlagsMethodInfo
instance (signature ~ (m [Gio.Flags.DBusServerFlags]), MonadIO m, IsDBusServer a) => O.OverloadedMethod DBusServerGetFlagsMethodInfo a signature where
    overloadedMethod = dBusServerGetFlags

instance O.OverloadedMethodInfo DBusServerGetFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.dBusServerGetFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#v:dBusServerGetFlags"
        })


#endif

-- method DBusServer::get_guid
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "server"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusServer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusServer." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_server_get_guid" g_dbus_server_get_guid :: 
    Ptr DBusServer ->                       -- server : TInterface (Name {namespace = "Gio", name = "DBusServer"})
    IO CString

-- | Gets the GUID for /@server@/, as provided to 'GI.Gio.Objects.DBusServer.dBusServerNewSync'.
-- 
-- /Since: 2.26/
dBusServerGetGuid ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusServer a) =>
    a
    -- ^ /@server@/: A t'GI.Gio.Objects.DBusServer.DBusServer'.
    -> m T.Text
    -- ^ __Returns:__ A D-Bus GUID. Do not free this string, it is owned by /@server@/.
dBusServerGetGuid server = liftIO $ do
    server' <- unsafeManagedPtrCastPtr server
    result <- g_dbus_server_get_guid server'
    checkUnexpectedReturnNULL "dBusServerGetGuid" result
    result' <- cstringToText result
    touchManagedPtr server
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusServerGetGuidMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsDBusServer a) => O.OverloadedMethod DBusServerGetGuidMethodInfo a signature where
    overloadedMethod = dBusServerGetGuid

instance O.OverloadedMethodInfo DBusServerGetGuidMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.dBusServerGetGuid",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#v:dBusServerGetGuid"
        })


#endif

-- method DBusServer::is_active
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "server"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusServer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusServer." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_server_is_active" g_dbus_server_is_active :: 
    Ptr DBusServer ->                       -- server : TInterface (Name {namespace = "Gio", name = "DBusServer"})
    IO CInt

-- | Gets whether /@server@/ is active.
-- 
-- /Since: 2.26/
dBusServerIsActive ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusServer a) =>
    a
    -- ^ /@server@/: A t'GI.Gio.Objects.DBusServer.DBusServer'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if server is active, 'P.False' otherwise.
dBusServerIsActive server = liftIO $ do
    server' <- unsafeManagedPtrCastPtr server
    result <- g_dbus_server_is_active server'
    let result' = (/= 0) result
    touchManagedPtr server
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusServerIsActiveMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDBusServer a) => O.OverloadedMethod DBusServerIsActiveMethodInfo a signature where
    overloadedMethod = dBusServerIsActive

instance O.OverloadedMethodInfo DBusServerIsActiveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.dBusServerIsActive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#v:dBusServerIsActive"
        })


#endif

-- method DBusServer::start
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "server"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusServer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusServer." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_server_start" g_dbus_server_start :: 
    Ptr DBusServer ->                       -- server : TInterface (Name {namespace = "Gio", name = "DBusServer"})
    IO ()

-- | Starts /@server@/.
-- 
-- /Since: 2.26/
dBusServerStart ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusServer a) =>
    a
    -- ^ /@server@/: A t'GI.Gio.Objects.DBusServer.DBusServer'.
    -> m ()
dBusServerStart server = liftIO $ do
    server' <- unsafeManagedPtrCastPtr server
    g_dbus_server_start server'
    touchManagedPtr server
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusServerStartMethodInfo
instance (signature ~ (m ()), MonadIO m, IsDBusServer a) => O.OverloadedMethod DBusServerStartMethodInfo a signature where
    overloadedMethod = dBusServerStart

instance O.OverloadedMethodInfo DBusServerStartMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.dBusServerStart",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#v:dBusServerStart"
        })


#endif

-- method DBusServer::stop
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "server"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusServer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusServer." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_server_stop" g_dbus_server_stop :: 
    Ptr DBusServer ->                       -- server : TInterface (Name {namespace = "Gio", name = "DBusServer"})
    IO ()

-- | Stops /@server@/.
-- 
-- /Since: 2.26/
dBusServerStop ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusServer a) =>
    a
    -- ^ /@server@/: A t'GI.Gio.Objects.DBusServer.DBusServer'.
    -> m ()
dBusServerStop server = liftIO $ do
    server' <- unsafeManagedPtrCastPtr server
    g_dbus_server_stop server'
    touchManagedPtr server
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusServerStopMethodInfo
instance (signature ~ (m ()), MonadIO m, IsDBusServer a) => O.OverloadedMethod DBusServerStopMethodInfo a signature where
    overloadedMethod = dBusServerStop

instance O.OverloadedMethodInfo DBusServerStopMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusServer.dBusServerStop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusServer.html#v:dBusServerStop"
        })


#endif


