{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GDBusAuthObserver@ provides a mechanism for participating
-- in how a t'GI.Gio.Objects.DBusServer.DBusServer' (or a t'GI.Gio.Objects.DBusConnection.DBusConnection')
-- authenticates remote peers.
-- 
-- Simply instantiate a @GDBusAuthObserver@ and connect to the
-- signals you are interested in. Note that new signals may be added
-- in the future.
-- 
-- == Controlling Authentication Mechanisms
-- 
-- By default, a @GDBusServer@ or server-side @GDBusConnection@ will allow
-- any authentication mechanism to be used. If you only want to allow D-Bus
-- connections with the @EXTERNAL@ mechanism, which makes use of credentials
-- passing and is the recommended mechanism for modern Unix platforms such
-- as Linux and the BSD family, you would use a signal handler like this:
-- 
-- 
-- === /c code/
-- >static gboolean
-- >on_allow_mechanism (GDBusAuthObserver *observer,
-- >                    const gchar       *mechanism,
-- >                    gpointer           user_data)
-- >{
-- >  if (g_strcmp0 (mechanism, "EXTERNAL") == 0)
-- >    {
-- >      return TRUE;
-- >    }
-- >
-- >  return FALSE;
-- >}
-- 
-- 
-- == Controlling Authorization
-- 
-- By default, a @GDBusServer@ or server-side @GDBusConnection@ will accept
-- connections from any successfully authenticated user (but not from
-- anonymous connections using the @ANONYMOUS@ mechanism). If you only
-- want to allow D-Bus connections from processes owned by the same uid
-- as the server, since GLib 2.68, you should use the
-- @G_DBUS_SERVER_FLAGS_AUTHENTICATION_REQUIRE_SAME_USER@ flag. It’s equivalent
-- to the following signal handler:
-- 
-- 
-- === /c code/
-- >static gboolean
-- >on_authorize_authenticated_peer (GDBusAuthObserver *observer,
-- >                                 GIOStream         *stream,
-- >                                 GCredentials      *credentials,
-- >                                 gpointer           user_data)
-- >{
-- >  gboolean authorized;
-- >
-- >  authorized = FALSE;
-- >  if (credentials != NULL)
-- >    {
-- >      GCredentials *own_credentials;
-- >      own_credentials = g_credentials_new ();
-- >      if (g_credentials_is_same_user (credentials, own_credentials, NULL))
-- >        authorized = TRUE;
-- >      g_object_unref (own_credentials);
-- >    }
-- >
-- >  return authorized;
-- >}
-- 
-- 
-- /Since: 2.26/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.DBusAuthObserver
    ( 

-- * Exported types
    DBusAuthObserver(..)                    ,
    IsDBusAuthObserver                      ,
    toDBusAuthObserver                      ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [allowMechanism]("GI.Gio.Objects.DBusAuthObserver#g:method:allowMechanism"), [authorizeAuthenticatedPeer]("GI.Gio.Objects.DBusAuthObserver#g:method:authorizeAuthenticatedPeer"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveDBusAuthObserverMethod           ,
#endif

-- ** allowMechanism #method:allowMechanism#

#if defined(ENABLE_OVERLOADING)
    DBusAuthObserverAllowMechanismMethodInfo,
#endif
    dBusAuthObserverAllowMechanism          ,


-- ** authorizeAuthenticatedPeer #method:authorizeAuthenticatedPeer#

#if defined(ENABLE_OVERLOADING)
    DBusAuthObserverAuthorizeAuthenticatedPeerMethodInfo,
#endif
    dBusAuthObserverAuthorizeAuthenticatedPeer,


-- ** new #method:new#

    dBusAuthObserverNew                     ,




 -- * Signals


-- ** allowMechanism #signal:allowMechanism#

    DBusAuthObserverAllowMechanismCallback  ,
#if defined(ENABLE_OVERLOADING)
    DBusAuthObserverAllowMechanismSignalInfo,
#endif
    afterDBusAuthObserverAllowMechanism     ,
    onDBusAuthObserverAllowMechanism        ,


-- ** authorizeAuthenticatedPeer #signal:authorizeAuthenticatedPeer#

    DBusAuthObserverAuthorizeAuthenticatedPeerCallback,
#if defined(ENABLE_OVERLOADING)
    DBusAuthObserverAuthorizeAuthenticatedPeerSignalInfo,
#endif
    afterDBusAuthObserverAuthorizeAuthenticatedPeer,
    onDBusAuthObserverAuthorizeAuthenticatedPeer,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.Credentials as Gio.Credentials
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Objects.Credentials as Gio.Credentials
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream

#endif

-- | Memory-managed wrapper type.
newtype DBusAuthObserver = DBusAuthObserver (SP.ManagedPtr DBusAuthObserver)
    deriving (Eq)

instance SP.ManagedPtrNewtype DBusAuthObserver where
    toManagedPtr (DBusAuthObserver p) = p

foreign import ccall "g_dbus_auth_observer_get_type"
    c_g_dbus_auth_observer_get_type :: IO B.Types.GType

instance B.Types.TypedObject DBusAuthObserver where
    glibType = c_g_dbus_auth_observer_get_type

instance B.Types.GObject DBusAuthObserver

-- | Type class for types which can be safely cast to `DBusAuthObserver`, for instance with `toDBusAuthObserver`.
class (SP.GObject o, O.IsDescendantOf DBusAuthObserver o) => IsDBusAuthObserver o
instance (SP.GObject o, O.IsDescendantOf DBusAuthObserver o) => IsDBusAuthObserver o

instance O.HasParentTypes DBusAuthObserver
type instance O.ParentTypes DBusAuthObserver = '[GObject.Object.Object]

-- | Cast to `DBusAuthObserver`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toDBusAuthObserver :: (MIO.MonadIO m, IsDBusAuthObserver o) => o -> m DBusAuthObserver
toDBusAuthObserver = MIO.liftIO . B.ManagedPtr.unsafeCastTo DBusAuthObserver

-- | Convert 'DBusAuthObserver' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe DBusAuthObserver) where
    gvalueGType_ = c_g_dbus_auth_observer_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr DBusAuthObserver)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr DBusAuthObserver)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject DBusAuthObserver ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveDBusAuthObserverMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveDBusAuthObserverMethod "allowMechanism" o = DBusAuthObserverAllowMechanismMethodInfo
    ResolveDBusAuthObserverMethod "authorizeAuthenticatedPeer" o = DBusAuthObserverAuthorizeAuthenticatedPeerMethodInfo
    ResolveDBusAuthObserverMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveDBusAuthObserverMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveDBusAuthObserverMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveDBusAuthObserverMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveDBusAuthObserverMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveDBusAuthObserverMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveDBusAuthObserverMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveDBusAuthObserverMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveDBusAuthObserverMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveDBusAuthObserverMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveDBusAuthObserverMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveDBusAuthObserverMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveDBusAuthObserverMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveDBusAuthObserverMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveDBusAuthObserverMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveDBusAuthObserverMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveDBusAuthObserverMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveDBusAuthObserverMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveDBusAuthObserverMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveDBusAuthObserverMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveDBusAuthObserverMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveDBusAuthObserverMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveDBusAuthObserverMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveDBusAuthObserverMethod t DBusAuthObserver, O.OverloadedMethod info DBusAuthObserver p) => OL.IsLabel t (DBusAuthObserver -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveDBusAuthObserverMethod t DBusAuthObserver, O.OverloadedMethod info DBusAuthObserver p, R.HasField t DBusAuthObserver p) => R.HasField t DBusAuthObserver p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveDBusAuthObserverMethod t DBusAuthObserver, O.OverloadedMethodInfo info DBusAuthObserver) => OL.IsLabel t (O.MethodProxy info DBusAuthObserver) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal DBusAuthObserver::allow-mechanism
-- | Emitted to check if /@mechanism@/ is allowed to be used.
-- 
-- /Since: 2.34/
type DBusAuthObserverAllowMechanismCallback =
    T.Text
    -- ^ /@mechanism@/: The name of the mechanism, e.g. @DBUS_COOKIE_SHA1@.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' if /@mechanism@/ can be used to authenticate the other peer, 'P.False' if not.

type C_DBusAuthObserverAllowMechanismCallback =
    Ptr DBusAuthObserver ->                 -- object
    CString ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_DBusAuthObserverAllowMechanismCallback`.
foreign import ccall "wrapper"
    mk_DBusAuthObserverAllowMechanismCallback :: C_DBusAuthObserverAllowMechanismCallback -> IO (FunPtr C_DBusAuthObserverAllowMechanismCallback)

wrap_DBusAuthObserverAllowMechanismCallback :: 
    GObject a => (a -> DBusAuthObserverAllowMechanismCallback) ->
    C_DBusAuthObserverAllowMechanismCallback
wrap_DBusAuthObserverAllowMechanismCallback gi'cb gi'selfPtr mechanism _ = do
    mechanism' <- cstringToText mechanism
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  mechanism'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [allowMechanism](#signal:allowMechanism) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dBusAuthObserver #allowMechanism callback
-- @
-- 
-- 
onDBusAuthObserverAllowMechanism :: (IsDBusAuthObserver a, MonadIO m) => a -> ((?self :: a) => DBusAuthObserverAllowMechanismCallback) -> m SignalHandlerId
onDBusAuthObserverAllowMechanism obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusAuthObserverAllowMechanismCallback wrapped
    wrapped'' <- mk_DBusAuthObserverAllowMechanismCallback wrapped'
    connectSignalFunPtr obj "allow-mechanism" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [allowMechanism](#signal:allowMechanism) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dBusAuthObserver #allowMechanism callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDBusAuthObserverAllowMechanism :: (IsDBusAuthObserver a, MonadIO m) => a -> ((?self :: a) => DBusAuthObserverAllowMechanismCallback) -> m SignalHandlerId
afterDBusAuthObserverAllowMechanism obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusAuthObserverAllowMechanismCallback wrapped
    wrapped'' <- mk_DBusAuthObserverAllowMechanismCallback wrapped'
    connectSignalFunPtr obj "allow-mechanism" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DBusAuthObserverAllowMechanismSignalInfo
instance SignalInfo DBusAuthObserverAllowMechanismSignalInfo where
    type HaskellCallbackType DBusAuthObserverAllowMechanismSignalInfo = DBusAuthObserverAllowMechanismCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DBusAuthObserverAllowMechanismCallback cb
        cb'' <- mk_DBusAuthObserverAllowMechanismCallback cb'
        connectSignalFunPtr obj "allow-mechanism" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusAuthObserver::allow-mechanism"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusAuthObserver.html#g:signal:allowMechanism"})

#endif

-- signal DBusAuthObserver::authorize-authenticated-peer
-- | Emitted to check if a peer that is successfully authenticated
-- is authorized.
-- 
-- /Since: 2.26/
type DBusAuthObserverAuthorizeAuthenticatedPeerCallback =
    Gio.IOStream.IOStream
    -- ^ /@stream@/: A t'GI.Gio.Objects.IOStream.IOStream' for the t'GI.Gio.Objects.DBusConnection.DBusConnection'.
    -> Maybe Gio.Credentials.Credentials
    -- ^ /@credentials@/: Credentials received from the peer or 'P.Nothing'.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' if the peer is authorized, 'P.False' if not.

type C_DBusAuthObserverAuthorizeAuthenticatedPeerCallback =
    Ptr DBusAuthObserver ->                 -- object
    Ptr Gio.IOStream.IOStream ->
    Ptr Gio.Credentials.Credentials ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_DBusAuthObserverAuthorizeAuthenticatedPeerCallback`.
foreign import ccall "wrapper"
    mk_DBusAuthObserverAuthorizeAuthenticatedPeerCallback :: C_DBusAuthObserverAuthorizeAuthenticatedPeerCallback -> IO (FunPtr C_DBusAuthObserverAuthorizeAuthenticatedPeerCallback)

wrap_DBusAuthObserverAuthorizeAuthenticatedPeerCallback :: 
    GObject a => (a -> DBusAuthObserverAuthorizeAuthenticatedPeerCallback) ->
    C_DBusAuthObserverAuthorizeAuthenticatedPeerCallback
wrap_DBusAuthObserverAuthorizeAuthenticatedPeerCallback gi'cb gi'selfPtr stream credentials _ = do
    stream' <- (newObject Gio.IOStream.IOStream) stream
    maybeCredentials <-
        if credentials == FP.nullPtr
        then return Nothing
        else do
            credentials' <- (newObject Gio.Credentials.Credentials) credentials
            return $ Just credentials'
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  stream' maybeCredentials
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [authorizeAuthenticatedPeer](#signal:authorizeAuthenticatedPeer) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dBusAuthObserver #authorizeAuthenticatedPeer callback
-- @
-- 
-- 
onDBusAuthObserverAuthorizeAuthenticatedPeer :: (IsDBusAuthObserver a, MonadIO m) => a -> ((?self :: a) => DBusAuthObserverAuthorizeAuthenticatedPeerCallback) -> m SignalHandlerId
onDBusAuthObserverAuthorizeAuthenticatedPeer obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusAuthObserverAuthorizeAuthenticatedPeerCallback wrapped
    wrapped'' <- mk_DBusAuthObserverAuthorizeAuthenticatedPeerCallback wrapped'
    connectSignalFunPtr obj "authorize-authenticated-peer" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [authorizeAuthenticatedPeer](#signal:authorizeAuthenticatedPeer) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dBusAuthObserver #authorizeAuthenticatedPeer callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDBusAuthObserverAuthorizeAuthenticatedPeer :: (IsDBusAuthObserver a, MonadIO m) => a -> ((?self :: a) => DBusAuthObserverAuthorizeAuthenticatedPeerCallback) -> m SignalHandlerId
afterDBusAuthObserverAuthorizeAuthenticatedPeer obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusAuthObserverAuthorizeAuthenticatedPeerCallback wrapped
    wrapped'' <- mk_DBusAuthObserverAuthorizeAuthenticatedPeerCallback wrapped'
    connectSignalFunPtr obj "authorize-authenticated-peer" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DBusAuthObserverAuthorizeAuthenticatedPeerSignalInfo
instance SignalInfo DBusAuthObserverAuthorizeAuthenticatedPeerSignalInfo where
    type HaskellCallbackType DBusAuthObserverAuthorizeAuthenticatedPeerSignalInfo = DBusAuthObserverAuthorizeAuthenticatedPeerCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DBusAuthObserverAuthorizeAuthenticatedPeerCallback cb
        cb'' <- mk_DBusAuthObserverAuthorizeAuthenticatedPeerCallback cb'
        connectSignalFunPtr obj "authorize-authenticated-peer" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusAuthObserver::authorize-authenticated-peer"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusAuthObserver.html#g:signal:authorizeAuthenticatedPeer"})

#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList DBusAuthObserver
type instance O.AttributeList DBusAuthObserver = DBusAuthObserverAttributeList
type DBusAuthObserverAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList DBusAuthObserver = DBusAuthObserverSignalList
type DBusAuthObserverSignalList = ('[ '("allowMechanism", DBusAuthObserverAllowMechanismSignalInfo), '("authorizeAuthenticatedPeer", DBusAuthObserverAuthorizeAuthenticatedPeerSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method DBusAuthObserver::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "DBusAuthObserver" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_auth_observer_new" g_dbus_auth_observer_new :: 
    IO (Ptr DBusAuthObserver)

-- | Creates a new t'GI.Gio.Objects.DBusAuthObserver.DBusAuthObserver' object.
-- 
-- /Since: 2.26/
dBusAuthObserverNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m DBusAuthObserver
    -- ^ __Returns:__ A t'GI.Gio.Objects.DBusAuthObserver.DBusAuthObserver'. Free with 'GI.GObject.Objects.Object.objectUnref'.
dBusAuthObserverNew  = liftIO $ do
    result <- g_dbus_auth_observer_new
    checkUnexpectedReturnNULL "dBusAuthObserverNew" result
    result' <- (wrapObject DBusAuthObserver) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method DBusAuthObserver::allow_mechanism
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "observer"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusAuthObserver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusAuthObserver."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mechanism"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "The name of the mechanism, e.g. `DBUS_COOKIE_SHA1`."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_auth_observer_allow_mechanism" g_dbus_auth_observer_allow_mechanism :: 
    Ptr DBusAuthObserver ->                 -- observer : TInterface (Name {namespace = "Gio", name = "DBusAuthObserver"})
    CString ->                              -- mechanism : TBasicType TUTF8
    IO CInt

-- | Emits the [DBusAuthObserver::allowMechanism]("GI.Gio.Objects.DBusAuthObserver#g:signal:allowMechanism") signal on /@observer@/.
-- 
-- /Since: 2.34/
dBusAuthObserverAllowMechanism ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusAuthObserver a) =>
    a
    -- ^ /@observer@/: A t'GI.Gio.Objects.DBusAuthObserver.DBusAuthObserver'.
    -> T.Text
    -- ^ /@mechanism@/: The name of the mechanism, e.g. @DBUS_COOKIE_SHA1@.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@mechanism@/ can be used to authenticate the other peer, 'P.False' if not.
dBusAuthObserverAllowMechanism observer mechanism = liftIO $ do
    observer' <- unsafeManagedPtrCastPtr observer
    mechanism' <- textToCString mechanism
    result <- g_dbus_auth_observer_allow_mechanism observer' mechanism'
    let result' = (/= 0) result
    touchManagedPtr observer
    freeMem mechanism'
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusAuthObserverAllowMechanismMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m, IsDBusAuthObserver a) => O.OverloadedMethod DBusAuthObserverAllowMechanismMethodInfo a signature where
    overloadedMethod = dBusAuthObserverAllowMechanism

instance O.OverloadedMethodInfo DBusAuthObserverAllowMechanismMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusAuthObserver.dBusAuthObserverAllowMechanism",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusAuthObserver.html#v:dBusAuthObserverAllowMechanism"
        })


#endif

-- method DBusAuthObserver::authorize_authenticated_peer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "observer"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusAuthObserver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusAuthObserver."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GIOStream for the #GDBusConnection."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "credentials"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Credentials" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Credentials received from the peer or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_auth_observer_authorize_authenticated_peer" g_dbus_auth_observer_authorize_authenticated_peer :: 
    Ptr DBusAuthObserver ->                 -- observer : TInterface (Name {namespace = "Gio", name = "DBusAuthObserver"})
    Ptr Gio.IOStream.IOStream ->            -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    Ptr Gio.Credentials.Credentials ->      -- credentials : TInterface (Name {namespace = "Gio", name = "Credentials"})
    IO CInt

-- | Emits the [DBusAuthObserver::authorizeAuthenticatedPeer]("GI.Gio.Objects.DBusAuthObserver#g:signal:authorizeAuthenticatedPeer") signal on /@observer@/.
-- 
-- /Since: 2.26/
dBusAuthObserverAuthorizeAuthenticatedPeer ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusAuthObserver a, Gio.IOStream.IsIOStream b, Gio.Credentials.IsCredentials c) =>
    a
    -- ^ /@observer@/: A t'GI.Gio.Objects.DBusAuthObserver.DBusAuthObserver'.
    -> b
    -- ^ /@stream@/: A t'GI.Gio.Objects.IOStream.IOStream' for the t'GI.Gio.Objects.DBusConnection.DBusConnection'.
    -> Maybe (c)
    -- ^ /@credentials@/: Credentials received from the peer or 'P.Nothing'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the peer is authorized, 'P.False' if not.
dBusAuthObserverAuthorizeAuthenticatedPeer observer stream credentials = liftIO $ do
    observer' <- unsafeManagedPtrCastPtr observer
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCredentials <- case credentials of
        Nothing -> return FP.nullPtr
        Just jCredentials -> do
            jCredentials' <- unsafeManagedPtrCastPtr jCredentials
            return jCredentials'
    result <- g_dbus_auth_observer_authorize_authenticated_peer observer' stream' maybeCredentials
    let result' = (/= 0) result
    touchManagedPtr observer
    touchManagedPtr stream
    whenJust credentials touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusAuthObserverAuthorizeAuthenticatedPeerMethodInfo
instance (signature ~ (b -> Maybe (c) -> m Bool), MonadIO m, IsDBusAuthObserver a, Gio.IOStream.IsIOStream b, Gio.Credentials.IsCredentials c) => O.OverloadedMethod DBusAuthObserverAuthorizeAuthenticatedPeerMethodInfo a signature where
    overloadedMethod = dBusAuthObserverAuthorizeAuthenticatedPeer

instance O.OverloadedMethodInfo DBusAuthObserverAuthorizeAuthenticatedPeerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusAuthObserver.dBusAuthObserverAuthorizeAuthenticatedPeer",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-DBusAuthObserver.html#v:dBusAuthObserverAuthorizeAuthenticatedPeer"
        })


#endif


