{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GApplicationCommandLine@ represents a command-line invocation of
-- an application.
-- 
-- It is created by t'GI.Gio.Objects.Application.Application' and emitted
-- in the [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") signal and virtual function.
-- 
-- The class contains the list of arguments that the program was invoked
-- with. It is also possible to query if the commandline invocation was
-- local (ie: the current process is running in direct response to the
-- invocation) or remote (ie: some other process forwarded the
-- commandline to this process).
-- 
-- The @GApplicationCommandLine@ object can provide the /@argc@/ and /@argv@/
-- parameters for use with the [struct/@gLib@/.OptionContext] command-line parsing API,
-- with the 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetArguments' function. See
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-cmdline3.c gapplication-example-cmdline3.c>
-- for an example.
-- 
-- The exit status of the originally-invoked process may be set and
-- messages can be printed to stdout or stderr of that process.
-- 
-- For remote invocation, the originally-invoked process exits when
-- 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineDone' method is called. This method is
-- also automatically called when the object is disposed.
-- 
-- The main use for @GApplicationCommandLine@ (and the
-- [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") signal) is \'Emacs server\' like use cases:
-- You can set the @EDITOR@ environment variable to have e.g. git use
-- your favourite editor to edit commit messages, and if you already
-- have an instance of the editor running, the editing will happen
-- in the running instance, instead of opening a new one. An important
-- aspect of this use case is that the process that gets started by git
-- does not return until the editing is done.
-- 
-- Normally, the commandline is completely handled in the
-- [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") handler. The launching instance exits
-- once the signal handler in the primary instance has returned, and
-- the return value of the signal handler becomes the exit status
-- of the launching instance.
-- 
-- 
-- === /c code/
-- >static int
-- >command_line (GApplication            *application,
-- >              GApplicationCommandLine *cmdline)
-- >{
-- >  gchar **argv;
-- >  gint argc;
-- >  gint i;
-- >
-- >  argv = g_application_command_line_get_arguments (cmdline, &argc);
-- >
-- >  g_application_command_line_print (cmdline,
-- >                                    "This text is written back\n"
-- >                                    "to stdout of the caller\n");
-- >
-- >  for (i = 0; i < argc; i++)
-- >    g_print ("argument %d: %s\n", i, argv[i]);
-- >
-- >  g_strfreev (argv);
-- >
-- >  return 0;
-- >}
-- 
-- 
-- The complete example can be found here:
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-cmdline.c gapplication-example-cmdline.c>
-- 
-- In more complicated cases, the handling of the commandline can be
-- split between the launcher and the primary instance.
-- 
-- 
-- === /c code/
-- >static gboolean
-- > test_local_cmdline (GApplication   *application,
-- >                     gchar        ***arguments,
-- >                     gint           *exit_status)
-- >{
-- >  gint i, j;
-- >  gchar **argv;
-- >
-- >  argv = *arguments;
-- >
-- >  if (argv[0] == NULL)
-- >    {
-- >      *exit_status = 0;
-- >      return FALSE;
-- >    }
-- >
-- >  i = 1;
-- >  while (argv[i])
-- >    {
-- >      if (g_str_has_prefix (argv[i], "--local-"))
-- >        {
-- >          g_print ("handling argument %s locally\n", argv[i]);
-- >          g_free (argv[i]);
-- >          for (j = i; argv[j]; j++)
-- >            argv[j] = argv[j + 1];
-- >        }
-- >      else
-- >        {
-- >          g_print ("not handling argument %s locally\n", argv[i]);
-- >          i++;
-- >        }
-- >    }
-- >
-- >  *exit_status = 0;
-- >
-- >  return FALSE;
-- >}
-- >
-- >static void
-- >test_application_class_init (TestApplicationClass *class)
-- >{
-- >  G_APPLICATION_CLASS (class)->local_command_line = test_local_cmdline;
-- >
-- >  ...
-- >}
-- 
-- 
-- In this example of split commandline handling, options that start
-- with @--local-@ are handled locally, all other options are passed
-- to the [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") handler which runs in the primary
-- instance.
-- 
-- The complete example can be found here:
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-cmdline2.c gapplication-example-cmdline2.c>
-- 
-- If handling the commandline requires a lot of work, it may be better to defer it.
-- 
-- 
-- === /c code/
-- >static gboolean
-- >my_cmdline_handler (gpointer data)
-- >{
-- >  GApplicationCommandLine *cmdline = data;
-- >
-- >  // do the heavy lifting in an idle
-- >
-- >  g_application_command_line_set_exit_status (cmdline, 0);
-- >  g_object_unref (cmdline); // this releases the application
-- >
-- >  return G_SOURCE_REMOVE;
-- >}
-- >
-- >static int
-- >command_line (GApplication            *application,
-- >              GApplicationCommandLine *cmdline)
-- >{
-- >  // keep the application running until we are done with this commandline
-- >  g_application_hold (application);
-- >
-- >  g_object_set_data_full (G_OBJECT (cmdline),
-- >                          "application", application,
-- >                          (GDestroyNotify)g_application_release);
-- >
-- >  g_object_ref (cmdline);
-- >  g_idle_add (my_cmdline_handler, cmdline);
-- >
-- >  return 0;
-- >}
-- 
-- 
-- In this example the commandline is not completely handled before
-- the [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") handler returns. Instead, we keep
-- a reference to the @GApplicationCommandLine@ object and handle it
-- later (in this example, in an idle). Note that it is necessary to
-- hold the application until you are done with the commandline.
-- 
-- The complete example can be found here:
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-cmdline3.c gapplication-example-cmdline3.c>

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.ApplicationCommandLine
    ( 

-- * Exported types
    ApplicationCommandLine(..)              ,
    IsApplicationCommandLine                ,
    toApplicationCommandLine                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [createFileForArg]("GI.Gio.Objects.ApplicationCommandLine#g:method:createFileForArg"), [done]("GI.Gio.Objects.ApplicationCommandLine#g:method:done"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getenv]("GI.Gio.Objects.ApplicationCommandLine#g:method:getenv"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [printLiteral]("GI.Gio.Objects.ApplicationCommandLine#g:method:printLiteral"), [printerrLiteral]("GI.Gio.Objects.ApplicationCommandLine#g:method:printerrLiteral"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getArguments]("GI.Gio.Objects.ApplicationCommandLine#g:method:getArguments"), [getCwd]("GI.Gio.Objects.ApplicationCommandLine#g:method:getCwd"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getEnviron]("GI.Gio.Objects.ApplicationCommandLine#g:method:getEnviron"), [getExitStatus]("GI.Gio.Objects.ApplicationCommandLine#g:method:getExitStatus"), [getIsRemote]("GI.Gio.Objects.ApplicationCommandLine#g:method:getIsRemote"), [getOptionsDict]("GI.Gio.Objects.ApplicationCommandLine#g:method:getOptionsDict"), [getPlatformData]("GI.Gio.Objects.ApplicationCommandLine#g:method:getPlatformData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getStdin]("GI.Gio.Objects.ApplicationCommandLine#g:method:getStdin").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setExitStatus]("GI.Gio.Objects.ApplicationCommandLine#g:method:setExitStatus"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveApplicationCommandLineMethod     ,
#endif

-- ** createFileForArg #method:createFileForArg#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineCreateFileForArgMethodInfo,
#endif
    applicationCommandLineCreateFileForArg  ,


-- ** done #method:done#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineDoneMethodInfo    ,
#endif
    applicationCommandLineDone              ,


-- ** getArguments #method:getArguments#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetArgumentsMethodInfo,
#endif
    applicationCommandLineGetArguments      ,


-- ** getCwd #method:getCwd#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetCwdMethodInfo  ,
#endif
    applicationCommandLineGetCwd            ,


-- ** getEnviron #method:getEnviron#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetEnvironMethodInfo,
#endif
    applicationCommandLineGetEnviron        ,


-- ** getExitStatus #method:getExitStatus#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetExitStatusMethodInfo,
#endif
    applicationCommandLineGetExitStatus     ,


-- ** getIsRemote #method:getIsRemote#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetIsRemoteMethodInfo,
#endif
    applicationCommandLineGetIsRemote       ,


-- ** getOptionsDict #method:getOptionsDict#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetOptionsDictMethodInfo,
#endif
    applicationCommandLineGetOptionsDict    ,


-- ** getPlatformData #method:getPlatformData#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetPlatformDataMethodInfo,
#endif
    applicationCommandLineGetPlatformData   ,


-- ** getStdin #method:getStdin#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetStdinMethodInfo,
#endif
    applicationCommandLineGetStdin          ,


-- ** getenv #method:getenv#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineGetenvMethodInfo  ,
#endif
    applicationCommandLineGetenv            ,


-- ** printLiteral #method:printLiteral#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLinePrintLiteralMethodInfo,
#endif
    applicationCommandLinePrintLiteral      ,


-- ** printerrLiteral #method:printerrLiteral#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLinePrinterrLiteralMethodInfo,
#endif
    applicationCommandLinePrinterrLiteral   ,


-- ** setExitStatus #method:setExitStatus#

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineSetExitStatusMethodInfo,
#endif
    applicationCommandLineSetExitStatus     ,




 -- * Properties


-- ** arguments #attr:arguments#
-- | The commandline that caused this [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine")
-- signal emission.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineArgumentsPropertyInfo,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationCommandLineArguments         ,
#endif
    constructApplicationCommandLineArguments,


-- ** isRemote #attr:isRemote#
-- | Whether this is a remote commandline.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineIsRemotePropertyInfo,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationCommandLineIsRemote          ,
#endif
    getApplicationCommandLineIsRemote       ,


-- ** options #attr:options#
-- | The options sent along with the commandline.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineOptionsPropertyInfo,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationCommandLineOptions           ,
#endif
    constructApplicationCommandLineOptions  ,


-- ** platformData #attr:platformData#
-- | Platform-specific data for the commandline.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLinePlatformDataPropertyInfo,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationCommandLinePlatformData      ,
#endif
    constructApplicationCommandLinePlatformData,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Enums as GLib.Enums
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.DateTime as GLib.DateTime
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GLib.Structs.TimeVal as GLib.TimeVal
import qualified GI.GLib.Structs.VariantDict as GLib.VariantDict
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AppInfo as Gio.AppInfo
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Drive as Gio.Drive
import {-# SOURCE #-} qualified GI.Gio.Interfaces.File as Gio.File
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Icon as Gio.Icon
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Mount as Gio.Mount
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Seekable as Gio.Seekable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Volume as Gio.Volume
import {-# SOURCE #-} qualified GI.Gio.Objects.AppLaunchContext as Gio.AppLaunchContext
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.FileEnumerator as Gio.FileEnumerator
import {-# SOURCE #-} qualified GI.Gio.Objects.FileIOStream as Gio.FileIOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInfo as Gio.FileInfo
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInputStream as Gio.FileInputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileMonitor as Gio.FileMonitor
import {-# SOURCE #-} qualified GI.Gio.Objects.FileOutputStream as Gio.FileOutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MountOperation as Gio.MountOperation
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfo as Gio.FileAttributeInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfoList as Gio.FileAttributeInfoList
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeMatcher as Gio.FileAttributeMatcher
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GLib.Structs.VariantDict as GLib.VariantDict
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Interfaces.File as Gio.File
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream

#endif

-- | Memory-managed wrapper type.
newtype ApplicationCommandLine = ApplicationCommandLine (SP.ManagedPtr ApplicationCommandLine)
    deriving (Eq)

instance SP.ManagedPtrNewtype ApplicationCommandLine where
    toManagedPtr (ApplicationCommandLine p) = p

foreign import ccall "g_application_command_line_get_type"
    c_g_application_command_line_get_type :: IO B.Types.GType

instance B.Types.TypedObject ApplicationCommandLine where
    glibType = c_g_application_command_line_get_type

instance B.Types.GObject ApplicationCommandLine

-- | Type class for types which can be safely cast to `ApplicationCommandLine`, for instance with `toApplicationCommandLine`.
class (SP.GObject o, O.IsDescendantOf ApplicationCommandLine o) => IsApplicationCommandLine o
instance (SP.GObject o, O.IsDescendantOf ApplicationCommandLine o) => IsApplicationCommandLine o

instance O.HasParentTypes ApplicationCommandLine
type instance O.ParentTypes ApplicationCommandLine = '[GObject.Object.Object]

-- | Cast to `ApplicationCommandLine`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toApplicationCommandLine :: (MIO.MonadIO m, IsApplicationCommandLine o) => o -> m ApplicationCommandLine
toApplicationCommandLine = MIO.liftIO . B.ManagedPtr.unsafeCastTo ApplicationCommandLine

-- | Convert 'ApplicationCommandLine' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe ApplicationCommandLine) where
    gvalueGType_ = c_g_application_command_line_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr ApplicationCommandLine)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr ApplicationCommandLine)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject ApplicationCommandLine ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveApplicationCommandLineMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveApplicationCommandLineMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveApplicationCommandLineMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveApplicationCommandLineMethod "createFileForArg" o = ApplicationCommandLineCreateFileForArgMethodInfo
    ResolveApplicationCommandLineMethod "done" o = ApplicationCommandLineDoneMethodInfo
    ResolveApplicationCommandLineMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveApplicationCommandLineMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveApplicationCommandLineMethod "getenv" o = ApplicationCommandLineGetenvMethodInfo
    ResolveApplicationCommandLineMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveApplicationCommandLineMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveApplicationCommandLineMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveApplicationCommandLineMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveApplicationCommandLineMethod "printLiteral" o = ApplicationCommandLinePrintLiteralMethodInfo
    ResolveApplicationCommandLineMethod "printerrLiteral" o = ApplicationCommandLinePrinterrLiteralMethodInfo
    ResolveApplicationCommandLineMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveApplicationCommandLineMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveApplicationCommandLineMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveApplicationCommandLineMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveApplicationCommandLineMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveApplicationCommandLineMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveApplicationCommandLineMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveApplicationCommandLineMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveApplicationCommandLineMethod "getArguments" o = ApplicationCommandLineGetArgumentsMethodInfo
    ResolveApplicationCommandLineMethod "getCwd" o = ApplicationCommandLineGetCwdMethodInfo
    ResolveApplicationCommandLineMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveApplicationCommandLineMethod "getEnviron" o = ApplicationCommandLineGetEnvironMethodInfo
    ResolveApplicationCommandLineMethod "getExitStatus" o = ApplicationCommandLineGetExitStatusMethodInfo
    ResolveApplicationCommandLineMethod "getIsRemote" o = ApplicationCommandLineGetIsRemoteMethodInfo
    ResolveApplicationCommandLineMethod "getOptionsDict" o = ApplicationCommandLineGetOptionsDictMethodInfo
    ResolveApplicationCommandLineMethod "getPlatformData" o = ApplicationCommandLineGetPlatformDataMethodInfo
    ResolveApplicationCommandLineMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveApplicationCommandLineMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveApplicationCommandLineMethod "getStdin" o = ApplicationCommandLineGetStdinMethodInfo
    ResolveApplicationCommandLineMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveApplicationCommandLineMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveApplicationCommandLineMethod "setExitStatus" o = ApplicationCommandLineSetExitStatusMethodInfo
    ResolveApplicationCommandLineMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveApplicationCommandLineMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveApplicationCommandLineMethod t ApplicationCommandLine, O.OverloadedMethod info ApplicationCommandLine p) => OL.IsLabel t (ApplicationCommandLine -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveApplicationCommandLineMethod t ApplicationCommandLine, O.OverloadedMethod info ApplicationCommandLine p, R.HasField t ApplicationCommandLine p) => R.HasField t ApplicationCommandLine p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveApplicationCommandLineMethod t ApplicationCommandLine, O.OverloadedMethodInfo info ApplicationCommandLine) => OL.IsLabel t (O.MethodProxy info ApplicationCommandLine) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "arguments"
   -- Type: TVariant
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a `GValueConstruct` with valid value for the “@arguments@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationCommandLineArguments :: (IsApplicationCommandLine o, MIO.MonadIO m) => GVariant -> m (GValueConstruct o)
constructApplicationCommandLineArguments val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyVariant "arguments" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineArgumentsPropertyInfo
instance AttrInfo ApplicationCommandLineArgumentsPropertyInfo where
    type AttrAllowedOps ApplicationCommandLineArgumentsPropertyInfo = '[ 'AttrConstruct, 'AttrClear]
    type AttrBaseTypeConstraint ApplicationCommandLineArgumentsPropertyInfo = IsApplicationCommandLine
    type AttrSetTypeConstraint ApplicationCommandLineArgumentsPropertyInfo = (~) GVariant
    type AttrTransferTypeConstraint ApplicationCommandLineArgumentsPropertyInfo = (~) GVariant
    type AttrTransferType ApplicationCommandLineArgumentsPropertyInfo = GVariant
    type AttrGetType ApplicationCommandLineArgumentsPropertyInfo = ()
    type AttrLabel ApplicationCommandLineArgumentsPropertyInfo = "arguments"
    type AttrOrigin ApplicationCommandLineArgumentsPropertyInfo = ApplicationCommandLine
    attrGet = undefined
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructApplicationCommandLineArguments
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.arguments"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#g:attr:arguments"
        })
#endif

-- VVV Prop "is-remote"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@is-remote@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' applicationCommandLine #isRemote
-- @
getApplicationCommandLineIsRemote :: (MonadIO m, IsApplicationCommandLine o) => o -> m Bool
getApplicationCommandLineIsRemote obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "is-remote"

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineIsRemotePropertyInfo
instance AttrInfo ApplicationCommandLineIsRemotePropertyInfo where
    type AttrAllowedOps ApplicationCommandLineIsRemotePropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint ApplicationCommandLineIsRemotePropertyInfo = IsApplicationCommandLine
    type AttrSetTypeConstraint ApplicationCommandLineIsRemotePropertyInfo = (~) ()
    type AttrTransferTypeConstraint ApplicationCommandLineIsRemotePropertyInfo = (~) ()
    type AttrTransferType ApplicationCommandLineIsRemotePropertyInfo = ()
    type AttrGetType ApplicationCommandLineIsRemotePropertyInfo = Bool
    type AttrLabel ApplicationCommandLineIsRemotePropertyInfo = "is-remote"
    type AttrOrigin ApplicationCommandLineIsRemotePropertyInfo = ApplicationCommandLine
    attrGet = getApplicationCommandLineIsRemote
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.isRemote"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#g:attr:isRemote"
        })
#endif

-- VVV Prop "options"
   -- Type: TVariant
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a `GValueConstruct` with valid value for the “@options@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationCommandLineOptions :: (IsApplicationCommandLine o, MIO.MonadIO m) => GVariant -> m (GValueConstruct o)
constructApplicationCommandLineOptions val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyVariant "options" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineOptionsPropertyInfo
instance AttrInfo ApplicationCommandLineOptionsPropertyInfo where
    type AttrAllowedOps ApplicationCommandLineOptionsPropertyInfo = '[ 'AttrConstruct, 'AttrClear]
    type AttrBaseTypeConstraint ApplicationCommandLineOptionsPropertyInfo = IsApplicationCommandLine
    type AttrSetTypeConstraint ApplicationCommandLineOptionsPropertyInfo = (~) GVariant
    type AttrTransferTypeConstraint ApplicationCommandLineOptionsPropertyInfo = (~) GVariant
    type AttrTransferType ApplicationCommandLineOptionsPropertyInfo = GVariant
    type AttrGetType ApplicationCommandLineOptionsPropertyInfo = ()
    type AttrLabel ApplicationCommandLineOptionsPropertyInfo = "options"
    type AttrOrigin ApplicationCommandLineOptionsPropertyInfo = ApplicationCommandLine
    attrGet = undefined
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructApplicationCommandLineOptions
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.options"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#g:attr:options"
        })
#endif

-- VVV Prop "platform-data"
   -- Type: TVariant
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a `GValueConstruct` with valid value for the “@platform-data@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationCommandLinePlatformData :: (IsApplicationCommandLine o, MIO.MonadIO m) => GVariant -> m (GValueConstruct o)
constructApplicationCommandLinePlatformData val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyVariant "platform-data" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLinePlatformDataPropertyInfo
instance AttrInfo ApplicationCommandLinePlatformDataPropertyInfo where
    type AttrAllowedOps ApplicationCommandLinePlatformDataPropertyInfo = '[ 'AttrConstruct, 'AttrClear]
    type AttrBaseTypeConstraint ApplicationCommandLinePlatformDataPropertyInfo = IsApplicationCommandLine
    type AttrSetTypeConstraint ApplicationCommandLinePlatformDataPropertyInfo = (~) GVariant
    type AttrTransferTypeConstraint ApplicationCommandLinePlatformDataPropertyInfo = (~) GVariant
    type AttrTransferType ApplicationCommandLinePlatformDataPropertyInfo = GVariant
    type AttrGetType ApplicationCommandLinePlatformDataPropertyInfo = ()
    type AttrLabel ApplicationCommandLinePlatformDataPropertyInfo = "platform-data"
    type AttrOrigin ApplicationCommandLinePlatformDataPropertyInfo = ApplicationCommandLine
    attrGet = undefined
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructApplicationCommandLinePlatformData
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.platformData"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#g:attr:platformData"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList ApplicationCommandLine
type instance O.AttributeList ApplicationCommandLine = ApplicationCommandLineAttributeList
type ApplicationCommandLineAttributeList = ('[ '("arguments", ApplicationCommandLineArgumentsPropertyInfo), '("isRemote", ApplicationCommandLineIsRemotePropertyInfo), '("options", ApplicationCommandLineOptionsPropertyInfo), '("platformData", ApplicationCommandLinePlatformDataPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
applicationCommandLineArguments :: AttrLabelProxy "arguments"
applicationCommandLineArguments = AttrLabelProxy

applicationCommandLineIsRemote :: AttrLabelProxy "isRemote"
applicationCommandLineIsRemote = AttrLabelProxy

applicationCommandLineOptions :: AttrLabelProxy "options"
applicationCommandLineOptions = AttrLabelProxy

applicationCommandLinePlatformData :: AttrLabelProxy "platformData"
applicationCommandLinePlatformData = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList ApplicationCommandLine = ApplicationCommandLineSignalList
type ApplicationCommandLineSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method ApplicationCommandLine::create_file_for_arg
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "arg"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an argument from @cmdline"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_create_file_for_arg" g_application_command_line_create_file_for_arg :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    CString ->                              -- arg : TBasicType TFileName
    IO (Ptr Gio.File.File)

-- | Creates a t'GI.Gio.Interfaces.File.File' corresponding to a filename that was given as part
-- of the invocation of /@cmdline@/.
-- 
-- This differs from 'GI.Gio.Functions.fileNewForCommandlineArg' in that it
-- resolves relative pathnames using the current working directory of
-- the invoking process rather than the local process.
-- 
-- /Since: 2.36/
applicationCommandLineCreateFileForArg ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> [Char]
    -- ^ /@arg@/: an argument from /@cmdline@/
    -> m Gio.File.File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File'
applicationCommandLineCreateFileForArg cmdline arg = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    arg' <- stringToCString arg
    result <- g_application_command_line_create_file_for_arg cmdline' arg'
    checkUnexpectedReturnNULL "applicationCommandLineCreateFileForArg" result
    result' <- (wrapObject Gio.File.File) result
    touchManagedPtr cmdline
    freeMem arg'
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineCreateFileForArgMethodInfo
instance (signature ~ ([Char] -> m Gio.File.File), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineCreateFileForArgMethodInfo a signature where
    overloadedMethod = applicationCommandLineCreateFileForArg

instance O.OverloadedMethodInfo ApplicationCommandLineCreateFileForArgMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineCreateFileForArg",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineCreateFileForArg"
        })


#endif

-- method ApplicationCommandLine::done
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_done" g_application_command_line_done :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    IO ()

-- | Signals that command line processing is completed.
-- 
-- For remote invocation, it causes the invoking process to terminate.
-- 
-- For local invocation, it does nothing.
-- 
-- This method should be called in the [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine")
-- handler, after the exit status is set and all messages are printed.
-- 
-- After this call, 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineSetExitStatus' has no effect.
-- Subsequent calls to this method are no-ops.
-- 
-- This method is automatically called when the t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
-- object is disposed — so you can omit the call in non-garbage collected
-- languages.
-- 
-- /Since: 2.80/
applicationCommandLineDone ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m ()
applicationCommandLineDone cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    g_application_command_line_done cmdline'
    touchManagedPtr cmdline
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineDoneMethodInfo
instance (signature ~ (m ()), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineDoneMethodInfo a signature where
    overloadedMethod = applicationCommandLineDone

instance O.OverloadedMethodInfo ApplicationCommandLineDoneMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineDone",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineDone"
        })


#endif

-- method ApplicationCommandLine::get_arguments
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "argc"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of the arguments array, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "argc"
--              , argType = TBasicType TInt
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of the arguments array, or %NULL"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 1 (TBasicType TFileName))
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_get_arguments" g_application_command_line_get_arguments :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    Ptr Int32 ->                            -- argc : TBasicType TInt
    IO (Ptr CString)

-- | Gets the list of arguments that was passed on the command line.
-- 
-- The strings in the array may contain non-UTF-8 data on UNIX (such as
-- filenames or arguments given in the system locale) but are always in
-- UTF-8 on Windows.
-- 
-- If you wish to use the return value with t'GI.GLib.Structs.OptionContext.OptionContext', you must
-- use 'GI.GLib.Structs.OptionContext.optionContextParseStrv'.
-- 
-- The return value is 'P.Nothing'-terminated and should be freed using
-- 'GI.GLib.Functions.strfreev'.
-- 
-- /Since: 2.28/
applicationCommandLineGetArguments ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m [[Char]]
    -- ^ __Returns:__ 
    --      the string array containing the arguments (the argv)
applicationCommandLineGetArguments cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    argc <- allocMem :: IO (Ptr Int32)
    result <- g_application_command_line_get_arguments cmdline' argc
    argc' <- peek argc
    checkUnexpectedReturnNULL "applicationCommandLineGetArguments" result
    result' <- (unpackFileNameArrayWithLength argc') result
    (mapCArrayWithLength argc') freeMem result
    freeMem result
    touchManagedPtr cmdline
    freeMem argc
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetArgumentsMethodInfo
instance (signature ~ (m [[Char]]), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetArgumentsMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetArguments

instance O.OverloadedMethodInfo ApplicationCommandLineGetArgumentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetArguments",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetArguments"
        })


#endif

-- method ApplicationCommandLine::get_cwd
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFileName)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_get_cwd" g_application_command_line_get_cwd :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    IO CString

-- | Gets the working directory of the command line invocation.
-- The string may contain non-utf8 data.
-- 
-- It is possible that the remote application did not send a working
-- directory, so this may be 'P.Nothing'.
-- 
-- The return value should not be modified or freed and is valid for as
-- long as /@cmdline@/ exists.
-- 
-- /Since: 2.28/
applicationCommandLineGetCwd ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m (Maybe [Char])
    -- ^ __Returns:__ the current directory, or 'P.Nothing'
applicationCommandLineGetCwd cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    result <- g_application_command_line_get_cwd cmdline'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToString result'
        return result''
    touchManagedPtr cmdline
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetCwdMethodInfo
instance (signature ~ (m (Maybe [Char])), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetCwdMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetCwd

instance O.OverloadedMethodInfo ApplicationCommandLineGetCwdMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetCwd",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetCwd"
        })


#endif

-- method ApplicationCommandLine::get_environ
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TFileName))
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_get_environ" g_application_command_line_get_environ :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    IO (Ptr CString)

-- | Gets the contents of the \'environ\' variable of the command line
-- invocation, as would be returned by 'GI.GLib.Functions.getEnviron', ie as a
-- 'P.Nothing'-terminated list of strings in the form \'NAME=VALUE\'.
-- The strings may contain non-utf8 data.
-- 
-- The remote application usually does not send an environment.  Use
-- 'GI.Gio.Flags.ApplicationFlagsSendEnvironment' to affect that.  Even with this flag
-- set it is possible that the environment is still not available (due
-- to invocation messages from other applications).
-- 
-- The return value should not be modified or freed and is valid for as
-- long as /@cmdline@/ exists.
-- 
-- See 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetenv' if you are only interested
-- in the value of a single environment variable.
-- 
-- /Since: 2.28/
applicationCommandLineGetEnviron ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m [[Char]]
    -- ^ __Returns:__ 
    --     the environment strings, or 'P.Nothing' if they were not sent
applicationCommandLineGetEnviron cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    result <- g_application_command_line_get_environ cmdline'
    checkUnexpectedReturnNULL "applicationCommandLineGetEnviron" result
    result' <- unpackZeroTerminatedFileNameArray result
    touchManagedPtr cmdline
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetEnvironMethodInfo
instance (signature ~ (m [[Char]]), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetEnvironMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetEnviron

instance O.OverloadedMethodInfo ApplicationCommandLineGetEnvironMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetEnviron",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetEnviron"
        })


#endif

-- method ApplicationCommandLine::get_exit_status
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_get_exit_status" g_application_command_line_get_exit_status :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    IO Int32

-- | Gets the exit status of /@cmdline@/.  See
-- 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineSetExitStatus' for more information.
-- 
-- /Since: 2.28/
applicationCommandLineGetExitStatus ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m Int32
    -- ^ __Returns:__ the exit status
applicationCommandLineGetExitStatus cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    result <- g_application_command_line_get_exit_status cmdline'
    touchManagedPtr cmdline
    return result

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetExitStatusMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetExitStatusMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetExitStatus

instance O.OverloadedMethodInfo ApplicationCommandLineGetExitStatusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetExitStatus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetExitStatus"
        })


#endif

-- method ApplicationCommandLine::get_is_remote
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_get_is_remote" g_application_command_line_get_is_remote :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    IO CInt

-- | Determines if /@cmdline@/ represents a remote invocation.
-- 
-- /Since: 2.28/
applicationCommandLineGetIsRemote ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the invocation was remote
applicationCommandLineGetIsRemote cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    result <- g_application_command_line_get_is_remote cmdline'
    let result' = (/= 0) result
    touchManagedPtr cmdline
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetIsRemoteMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetIsRemoteMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetIsRemote

instance O.OverloadedMethodInfo ApplicationCommandLineGetIsRemoteMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetIsRemote",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetIsRemote"
        })


#endif

-- method ApplicationCommandLine::get_options_dict
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantDict" })
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_get_options_dict" g_application_command_line_get_options_dict :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    IO (Ptr GLib.VariantDict.VariantDict)

-- | Gets the options that were passed to @/g_application_command_line()/@.
-- 
-- If you did not override @/local_command_line()/@ then these are the same
-- options that were parsed according to the @/GOptionEntrys/@ added to the
-- application with 'GI.Gio.Objects.Application.applicationAddMainOptionEntries' and possibly
-- modified from your GApplication[handleLocalOptions](#g:signal:handleLocalOptions) handler.
-- 
-- If no options were sent then an empty dictionary is returned so that
-- you don\'t need to check for 'P.Nothing'.
-- 
-- The data has been passed via an untrusted external process, so the types of
-- all values must be checked before being used.
-- 
-- /Since: 2.40/
applicationCommandLineGetOptionsDict ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m GLib.VariantDict.VariantDict
    -- ^ __Returns:__ a t'GI.GLib.Structs.VariantDict.VariantDict' with the options
applicationCommandLineGetOptionsDict cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    result <- g_application_command_line_get_options_dict cmdline'
    checkUnexpectedReturnNULL "applicationCommandLineGetOptionsDict" result
    result' <- (newBoxed GLib.VariantDict.VariantDict) result
    touchManagedPtr cmdline
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetOptionsDictMethodInfo
instance (signature ~ (m GLib.VariantDict.VariantDict), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetOptionsDictMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetOptionsDict

instance O.OverloadedMethodInfo ApplicationCommandLineGetOptionsDictMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetOptionsDict",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetOptionsDict"
        })


#endif

-- method ApplicationCommandLine::get_platform_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_get_platform_data" g_application_command_line_get_platform_data :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    IO (Ptr GVariant)

-- | Gets the platform data associated with the invocation of /@cmdline@/.
-- 
-- This is a t'GVariant' dictionary containing information about the
-- context in which the invocation occurred.  It typically contains
-- information like the current working directory and the startup
-- notification ID.
-- 
-- It comes from an untrusted external process and hence the types of all
-- values must be validated before being used.
-- 
-- For local invocation, it will be 'P.Nothing'.
-- 
-- /Since: 2.28/
applicationCommandLineGetPlatformData ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the platform data, or 'P.Nothing'
applicationCommandLineGetPlatformData cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    result <- g_application_command_line_get_platform_data cmdline'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr cmdline
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetPlatformDataMethodInfo
instance (signature ~ (m (Maybe GVariant)), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetPlatformDataMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetPlatformData

instance O.OverloadedMethodInfo ApplicationCommandLineGetPlatformDataMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetPlatformData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetPlatformData"
        })


#endif

-- method ApplicationCommandLine::get_stdin
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "InputStream" })
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_get_stdin" g_application_command_line_get_stdin :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    IO (Ptr Gio.InputStream.InputStream)

-- | Gets the stdin of the invoking process.
-- 
-- The t'GI.Gio.Objects.InputStream.InputStream' can be used to read data passed to the standard
-- input of the invoking process.
-- This doesn\'t work on all platforms.  Presently, it is only available
-- on UNIX when using a D-Bus daemon capable of passing file descriptors.
-- If stdin is not available then 'P.Nothing' will be returned.  In the
-- future, support may be expanded to other platforms.
-- 
-- You must only call this function once per commandline invocation.
-- 
-- /Since: 2.34/
applicationCommandLineGetStdin ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> m (Maybe Gio.InputStream.InputStream)
    -- ^ __Returns:__ a t'GI.Gio.Objects.InputStream.InputStream' for stdin
applicationCommandLineGetStdin cmdline = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    result <- g_application_command_line_get_stdin cmdline'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Gio.InputStream.InputStream) result'
        return result''
    touchManagedPtr cmdline
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetStdinMethodInfo
instance (signature ~ (m (Maybe Gio.InputStream.InputStream)), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetStdinMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetStdin

instance O.OverloadedMethodInfo ApplicationCommandLineGetStdinMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetStdin",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetStdin"
        })


#endif

-- method ApplicationCommandLine::getenv
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the environment variable to get"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_getenv" g_application_command_line_getenv :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    CString ->                              -- name : TBasicType TFileName
    IO CString

-- | Gets the value of a particular environment variable of the command
-- line invocation, as would be returned by 'GI.GLib.Functions.getenv'.  The strings may
-- contain non-utf8 data.
-- 
-- The remote application usually does not send an environment.  Use
-- 'GI.Gio.Flags.ApplicationFlagsSendEnvironment' to affect that.  Even with this flag
-- set it is possible that the environment is still not available (due
-- to invocation messages from other applications).
-- 
-- The return value should not be modified or freed and is valid for as
-- long as /@cmdline@/ exists.
-- 
-- /Since: 2.28/
applicationCommandLineGetenv ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> [Char]
    -- ^ /@name@/: the environment variable to get
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the value of the variable, or 'P.Nothing' if unset or unsent
applicationCommandLineGetenv cmdline name = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    name' <- stringToCString name
    result <- g_application_command_line_getenv cmdline' name'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr cmdline
    freeMem name'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineGetenvMethodInfo
instance (signature ~ ([Char] -> m (Maybe T.Text)), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineGetenvMethodInfo a signature where
    overloadedMethod = applicationCommandLineGetenv

instance O.OverloadedMethodInfo ApplicationCommandLineGetenvMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetenv",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineGetenv"
        })


#endif

-- method ApplicationCommandLine::print_literal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "message"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the message" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_print_literal" g_application_command_line_print_literal :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    CString ->                              -- message : TBasicType TUTF8
    IO ()

-- | Prints a message using the stdout print handler in the invoking process.
-- 
-- Unlike @/g_application_command_line_print()/@, /@message@/ is not a @printf()@-style
-- format string. Use this function if /@message@/ contains text you don\'t have
-- control over, that could include @printf()@ escape sequences.
-- 
-- /Since: 2.80/
applicationCommandLinePrintLiteral ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> T.Text
    -- ^ /@message@/: the message
    -> m ()
applicationCommandLinePrintLiteral cmdline message = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    message' <- textToCString message
    g_application_command_line_print_literal cmdline' message'
    touchManagedPtr cmdline
    freeMem message'
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLinePrintLiteralMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLinePrintLiteralMethodInfo a signature where
    overloadedMethod = applicationCommandLinePrintLiteral

instance O.OverloadedMethodInfo ApplicationCommandLinePrintLiteralMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLinePrintLiteral",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLinePrintLiteral"
        })


#endif

-- method ApplicationCommandLine::printerr_literal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "message"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the message" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_printerr_literal" g_application_command_line_printerr_literal :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    CString ->                              -- message : TBasicType TUTF8
    IO ()

-- | Prints a message using the stderr print handler in the invoking process.
-- 
-- Unlike @/g_application_command_line_printerr()/@, /@message@/ is not
-- a @printf()@-style format string. Use this function if /@message@/ contains text
-- you don\'t have control over, that could include @printf()@ escape sequences.
-- 
-- /Since: 2.80/
applicationCommandLinePrinterrLiteral ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> T.Text
    -- ^ /@message@/: the message
    -> m ()
applicationCommandLinePrinterrLiteral cmdline message = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    message' <- textToCString message
    g_application_command_line_printerr_literal cmdline' message'
    touchManagedPtr cmdline
    freeMem message'
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLinePrinterrLiteralMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLinePrinterrLiteralMethodInfo a signature where
    overloadedMethod = applicationCommandLinePrinterrLiteral

instance O.OverloadedMethodInfo ApplicationCommandLinePrinterrLiteralMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLinePrinterrLiteral",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLinePrinterrLiteral"
        })


#endif

-- method ApplicationCommandLine::set_exit_status
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "cmdline"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ApplicationCommandLine" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplicationCommandLine"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "exit_status"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the exit status" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_command_line_set_exit_status" g_application_command_line_set_exit_status :: 
    Ptr ApplicationCommandLine ->           -- cmdline : TInterface (Name {namespace = "Gio", name = "ApplicationCommandLine"})
    Int32 ->                                -- exit_status : TBasicType TInt
    IO ()

-- | Sets the exit status that will be used when the invoking process
-- exits.
-- 
-- The return value of the [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") signal is
-- passed to this function when the handler returns.  This is the usual
-- way of setting the exit status.
-- 
-- In the event that you want the remote invocation to continue running
-- and want to decide on the exit status in the future, you can use this
-- call.  For the case of a remote invocation, the remote process will
-- typically exit when the last reference is dropped on /@cmdline@/.  The
-- exit status of the remote process will be equal to the last value
-- that was set with this function.
-- 
-- In the case that the commandline invocation is local, the situation
-- is slightly more complicated.  If the commandline invocation results
-- in the mainloop running (ie: because the use-count of the application
-- increased to a non-zero value) then the application is considered to
-- have been \'successful\' in a certain sense, and the exit status is
-- always zero.  If the application use count is zero, though, the exit
-- status of the local t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine' is used.
-- 
-- This method is a no-op if 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineDone' has
-- been called.
-- 
-- /Since: 2.28/
applicationCommandLineSetExitStatus ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplicationCommandLine a) =>
    a
    -- ^ /@cmdline@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine'
    -> Int32
    -- ^ /@exitStatus@/: the exit status
    -> m ()
applicationCommandLineSetExitStatus cmdline exitStatus = liftIO $ do
    cmdline' <- unsafeManagedPtrCastPtr cmdline
    g_application_command_line_set_exit_status cmdline' exitStatus
    touchManagedPtr cmdline
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineSetExitStatusMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsApplicationCommandLine a) => O.OverloadedMethod ApplicationCommandLineSetExitStatusMethodInfo a signature where
    overloadedMethod = applicationCommandLineSetExitStatus

instance O.OverloadedMethodInfo ApplicationCommandLineSetExitStatusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineSetExitStatus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Objects-ApplicationCommandLine.html#v:applicationCommandLineSetExitStatus"
        })


#endif


