{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The @GVolume@ interface represents user-visible objects that can be
-- mounted. For example, a file system partition on a USB flash drive, or an
-- optical disc inserted into a disc drive.
-- 
-- If a @GVolume@ is currently mounted, the corresponding t'GI.Gio.Interfaces.Mount.Mount' can
-- be retrieved using 'GI.Gio.Interfaces.Volume.volumeGetMount'.
-- 
-- Mounting a @GVolume@ instance is an asynchronous operation. For more
-- information about asynchronous operations, see t'GI.Gio.Interfaces.AsyncResult.AsyncResult' and
-- t'GI.Gio.Objects.Task.Task'. To mount a @GVolume@, first call 'GI.Gio.Interfaces.Volume.volumeMount'
-- with (at least) the @GVolume@ instance, optionally a
-- t'GI.Gio.Objects.MountOperation.MountOperation' object and a [type/@gio@/.AsyncReadyCallback].
-- 
-- Typically, one will only want to pass @NULL@ for the
-- t'GI.Gio.Objects.MountOperation.MountOperation' if automounting all volumes when a desktop session
-- starts since it’s not desirable to put up a lot of dialogs asking
-- for credentials.
-- 
-- The callback will be fired when the operation has resolved (either
-- with success or failure), and a t'GI.Gio.Interfaces.AsyncResult.AsyncResult' instance will be
-- passed to the callback.  That callback should then call
-- 'GI.Gio.Interfaces.Volume.volumeMountFinish' with the @GVolume@ instance and the
-- t'GI.Gio.Interfaces.AsyncResult.AsyncResult' data to see if the operation was completed
-- successfully.  If a [type/@gLib@/.Error] is present when
-- 'GI.Gio.Interfaces.Volume.volumeMountFinish' is called, then it will be filled with any
-- error information.
-- 
-- Note, when <http://developer.gnome.org/gio/stable/migrating-gnome-vfs.html porting from GnomeVFS>,
-- @GVolume@ is the moral equivalent of @GnomeVFSDrive@.
-- 
-- == Volume Identifiers
-- 
-- It is sometimes necessary to directly access the underlying
-- operating system object behind a volume (e.g. for passing a volume
-- to an application via the command line). For this purpose, GIO
-- allows to obtain an ‘identifier’ for the volume. There can be
-- different kinds of identifiers, such as Hal UDIs, filesystem labels,
-- traditional Unix devices (e.g. @\/dev\/sda2@), UUIDs. GIO uses predefined
-- strings as names for the different kinds of identifiers:
-- @G_VOLUME_IDENTIFIER_KIND_UUID@, @G_VOLUME_IDENTIFIER_KIND_LABEL@, etc.
-- Use 'GI.Gio.Interfaces.Volume.volumeGetIdentifier' to obtain an identifier for a volume.
-- 
-- Note that @G_VOLUME_IDENTIFIER_KIND_HAL_UDI@ will only be available
-- when the GVFS hal volume monitor is in use. Other volume monitors
-- will generally be able to provide the @G_VOLUME_IDENTIFIER_KIND_UNIX_DEVICE@
-- identifier, which can be used to obtain a hal device by means of
-- @libhal_manager_find_device_string_match()@.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Interfaces.Volume
    ( 

-- * Exported types
    Volume(..)                              ,
    IsVolume                                ,
    toVolume                                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [canEject]("GI.Gio.Interfaces.Volume#g:method:canEject"), [canMount]("GI.Gio.Interfaces.Volume#g:method:canMount"), [eject]("GI.Gio.Interfaces.Volume#g:method:eject"), [ejectFinish]("GI.Gio.Interfaces.Volume#g:method:ejectFinish"), [ejectWithOperation]("GI.Gio.Interfaces.Volume#g:method:ejectWithOperation"), [ejectWithOperationFinish]("GI.Gio.Interfaces.Volume#g:method:ejectWithOperationFinish"), [enumerateIdentifiers]("GI.Gio.Interfaces.Volume#g:method:enumerateIdentifiers"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [mount]("GI.Gio.Interfaces.Volume#g:method:mount"), [mountFinish]("GI.Gio.Interfaces.Volume#g:method:mountFinish"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [shouldAutomount]("GI.Gio.Interfaces.Volume#g:method:shouldAutomount"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getActivationRoot]("GI.Gio.Interfaces.Volume#g:method:getActivationRoot"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDrive]("GI.Gio.Interfaces.Volume#g:method:getDrive"), [getIcon]("GI.Gio.Interfaces.Volume#g:method:getIcon"), [getIdentifier]("GI.Gio.Interfaces.Volume#g:method:getIdentifier"), [getMount]("GI.Gio.Interfaces.Volume#g:method:getMount"), [getName]("GI.Gio.Interfaces.Volume#g:method:getName"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getSortKey]("GI.Gio.Interfaces.Volume#g:method:getSortKey"), [getSymbolicIcon]("GI.Gio.Interfaces.Volume#g:method:getSymbolicIcon"), [getUuid]("GI.Gio.Interfaces.Volume#g:method:getUuid").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveVolumeMethod                     ,
#endif

-- ** canEject #method:canEject#

#if defined(ENABLE_OVERLOADING)
    VolumeCanEjectMethodInfo                ,
#endif
    volumeCanEject                          ,


-- ** canMount #method:canMount#

#if defined(ENABLE_OVERLOADING)
    VolumeCanMountMethodInfo                ,
#endif
    volumeCanMount                          ,


-- ** eject #method:eject#

#if defined(ENABLE_OVERLOADING)
    VolumeEjectMethodInfo                   ,
#endif
    volumeEject                             ,


-- ** ejectFinish #method:ejectFinish#

#if defined(ENABLE_OVERLOADING)
    VolumeEjectFinishMethodInfo             ,
#endif
    volumeEjectFinish                       ,


-- ** ejectWithOperation #method:ejectWithOperation#

#if defined(ENABLE_OVERLOADING)
    VolumeEjectWithOperationMethodInfo      ,
#endif
    volumeEjectWithOperation                ,


-- ** ejectWithOperationFinish #method:ejectWithOperationFinish#

#if defined(ENABLE_OVERLOADING)
    VolumeEjectWithOperationFinishMethodInfo,
#endif
    volumeEjectWithOperationFinish          ,


-- ** enumerateIdentifiers #method:enumerateIdentifiers#

#if defined(ENABLE_OVERLOADING)
    VolumeEnumerateIdentifiersMethodInfo    ,
#endif
    volumeEnumerateIdentifiers              ,


-- ** getActivationRoot #method:getActivationRoot#

#if defined(ENABLE_OVERLOADING)
    VolumeGetActivationRootMethodInfo       ,
#endif
    volumeGetActivationRoot                 ,


-- ** getDrive #method:getDrive#

#if defined(ENABLE_OVERLOADING)
    VolumeGetDriveMethodInfo                ,
#endif
    volumeGetDrive                          ,


-- ** getIcon #method:getIcon#

#if defined(ENABLE_OVERLOADING)
    VolumeGetIconMethodInfo                 ,
#endif
    volumeGetIcon                           ,


-- ** getIdentifier #method:getIdentifier#

#if defined(ENABLE_OVERLOADING)
    VolumeGetIdentifierMethodInfo           ,
#endif
    volumeGetIdentifier                     ,


-- ** getMount #method:getMount#

#if defined(ENABLE_OVERLOADING)
    VolumeGetMountMethodInfo                ,
#endif
    volumeGetMount                          ,


-- ** getName #method:getName#

#if defined(ENABLE_OVERLOADING)
    VolumeGetNameMethodInfo                 ,
#endif
    volumeGetName                           ,


-- ** getSortKey #method:getSortKey#

#if defined(ENABLE_OVERLOADING)
    VolumeGetSortKeyMethodInfo              ,
#endif
    volumeGetSortKey                        ,


-- ** getSymbolicIcon #method:getSymbolicIcon#

#if defined(ENABLE_OVERLOADING)
    VolumeGetSymbolicIconMethodInfo         ,
#endif
    volumeGetSymbolicIcon                   ,


-- ** getUuid #method:getUuid#

#if defined(ENABLE_OVERLOADING)
    VolumeGetUuidMethodInfo                 ,
#endif
    volumeGetUuid                           ,


-- ** mount #method:mount#

#if defined(ENABLE_OVERLOADING)
    VolumeMountMethodInfo                   ,
#endif
    volumeMount                             ,


-- ** mountFinish #method:mountFinish#

#if defined(ENABLE_OVERLOADING)
    VolumeMountFinishMethodInfo             ,
#endif
    volumeMountFinish                       ,


-- ** shouldAutomount #method:shouldAutomount#

#if defined(ENABLE_OVERLOADING)
    VolumeShouldAutomountMethodInfo         ,
#endif
    volumeShouldAutomount                   ,




 -- * Signals


-- ** changed #signal:changed#

    VolumeChangedCallback                   ,
#if defined(ENABLE_OVERLOADING)
    VolumeChangedSignalInfo                 ,
#endif
    afterVolumeChanged                      ,
    onVolumeChanged                         ,


-- ** removed #signal:removed#

    VolumeRemovedCallback                   ,
#if defined(ENABLE_OVERLOADING)
    VolumeRemovedSignalInfo                 ,
#endif
    afterVolumeRemoved                      ,
    onVolumeRemoved                         ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Enums as GLib.Enums
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.DateTime as GLib.DateTime
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GLib.Structs.TimeVal as GLib.TimeVal
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AppInfo as Gio.AppInfo
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Drive as Gio.Drive
import {-# SOURCE #-} qualified GI.Gio.Interfaces.File as Gio.File
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Icon as Gio.Icon
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Mount as Gio.Mount
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Seekable as Gio.Seekable
import {-# SOURCE #-} qualified GI.Gio.Objects.AppLaunchContext as Gio.AppLaunchContext
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.FileEnumerator as Gio.FileEnumerator
import {-# SOURCE #-} qualified GI.Gio.Objects.FileIOStream as Gio.FileIOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInfo as Gio.FileInfo
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInputStream as Gio.FileInputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileMonitor as Gio.FileMonitor
import {-# SOURCE #-} qualified GI.Gio.Objects.FileOutputStream as Gio.FileOutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MountOperation as Gio.MountOperation
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfo as Gio.FileAttributeInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfoList as Gio.FileAttributeInfoList
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeMatcher as Gio.FileAttributeMatcher
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Drive as Gio.Drive
import {-# SOURCE #-} qualified GI.Gio.Interfaces.File as Gio.File
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Icon as Gio.Icon
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Mount as Gio.Mount
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.MountOperation as Gio.MountOperation

#endif

-- interface Volume 
-- | Memory-managed wrapper type.
newtype Volume = Volume (SP.ManagedPtr Volume)
    deriving (Eq)

instance SP.ManagedPtrNewtype Volume where
    toManagedPtr (Volume p) = p

foreign import ccall "g_volume_get_type"
    c_g_volume_get_type :: IO B.Types.GType

instance B.Types.TypedObject Volume where
    glibType = c_g_volume_get_type

instance B.Types.GObject Volume

-- | Type class for types which can be safely cast to `Volume`, for instance with `toVolume`.
class (SP.GObject o, O.IsDescendantOf Volume o) => IsVolume o
instance (SP.GObject o, O.IsDescendantOf Volume o) => IsVolume o

instance O.HasParentTypes Volume
type instance O.ParentTypes Volume = '[GObject.Object.Object]

-- | Cast to `Volume`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toVolume :: (MIO.MonadIO m, IsVolume o) => o -> m Volume
toVolume = MIO.liftIO . B.ManagedPtr.unsafeCastTo Volume

-- | Convert 'Volume' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Volume) where
    gvalueGType_ = c_g_volume_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Volume)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Volume)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Volume ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Volume
type instance O.AttributeList Volume = VolumeAttributeList
type VolumeAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveVolumeMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveVolumeMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveVolumeMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveVolumeMethod "canEject" o = VolumeCanEjectMethodInfo
    ResolveVolumeMethod "canMount" o = VolumeCanMountMethodInfo
    ResolveVolumeMethod "eject" o = VolumeEjectMethodInfo
    ResolveVolumeMethod "ejectFinish" o = VolumeEjectFinishMethodInfo
    ResolveVolumeMethod "ejectWithOperation" o = VolumeEjectWithOperationMethodInfo
    ResolveVolumeMethod "ejectWithOperationFinish" o = VolumeEjectWithOperationFinishMethodInfo
    ResolveVolumeMethod "enumerateIdentifiers" o = VolumeEnumerateIdentifiersMethodInfo
    ResolveVolumeMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveVolumeMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveVolumeMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveVolumeMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveVolumeMethod "mount" o = VolumeMountMethodInfo
    ResolveVolumeMethod "mountFinish" o = VolumeMountFinishMethodInfo
    ResolveVolumeMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveVolumeMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveVolumeMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveVolumeMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveVolumeMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveVolumeMethod "shouldAutomount" o = VolumeShouldAutomountMethodInfo
    ResolveVolumeMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveVolumeMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveVolumeMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveVolumeMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveVolumeMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveVolumeMethod "getActivationRoot" o = VolumeGetActivationRootMethodInfo
    ResolveVolumeMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveVolumeMethod "getDrive" o = VolumeGetDriveMethodInfo
    ResolveVolumeMethod "getIcon" o = VolumeGetIconMethodInfo
    ResolveVolumeMethod "getIdentifier" o = VolumeGetIdentifierMethodInfo
    ResolveVolumeMethod "getMount" o = VolumeGetMountMethodInfo
    ResolveVolumeMethod "getName" o = VolumeGetNameMethodInfo
    ResolveVolumeMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveVolumeMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveVolumeMethod "getSortKey" o = VolumeGetSortKeyMethodInfo
    ResolveVolumeMethod "getSymbolicIcon" o = VolumeGetSymbolicIconMethodInfo
    ResolveVolumeMethod "getUuid" o = VolumeGetUuidMethodInfo
    ResolveVolumeMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveVolumeMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveVolumeMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveVolumeMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveVolumeMethod t Volume, O.OverloadedMethod info Volume p) => OL.IsLabel t (Volume -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveVolumeMethod t Volume, O.OverloadedMethod info Volume p, R.HasField t Volume p) => R.HasField t Volume p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveVolumeMethod t Volume, O.OverloadedMethodInfo info Volume) => OL.IsLabel t (O.MethodProxy info Volume) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method Volume::can_eject
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_can_eject" g_volume_can_eject :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO CInt

-- | Checks if a volume can be ejected.
volumeCanEject ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@volume@/ can be ejected. 'P.False' otherwise
volumeCanEject volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_can_eject volume'
    let result' = (/= 0) result
    touchManagedPtr volume
    return result'

#if defined(ENABLE_OVERLOADING)
data VolumeCanEjectMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeCanEjectMethodInfo a signature where
    overloadedMethod = volumeCanEject

instance O.OverloadedMethodInfo VolumeCanEjectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeCanEject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeCanEject"
        })


#endif

-- method Volume::can_mount
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_can_mount" g_volume_can_mount :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO CInt

-- | Checks if a volume can be mounted.
volumeCanMount ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@volume@/ can be mounted. 'P.False' otherwise
volumeCanMount volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_can_mount volume'
    let result' = (/= 0) result
    touchManagedPtr volume
    return result'

#if defined(ENABLE_OVERLOADING)
data VolumeCanMountMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeCanMountMethodInfo a signature where
    overloadedMethod = volumeCanMount

instance O.OverloadedMethodInfo VolumeCanMountMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeCanMount",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeCanMount"
        })


#endif

-- method Volume::eject
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags affecting the unmount if required for eject"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data that gets passed to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_eject" g_volume_eject :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

{-# DEPRECATED volumeEject ["(Since version 2.22)","Use 'GI.Gio.Interfaces.Volume.volumeEjectWithOperation' instead."] #-}
-- | Ejects a volume. This is an asynchronous operation, and is
-- finished by calling 'GI.Gio.Interfaces.Volume.volumeEjectFinish' with the /@volume@/
-- and t'GI.Gio.Interfaces.AsyncResult.AsyncResult' returned in the /@callback@/.
volumeEject ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the unmount if required for eject
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback', or 'P.Nothing'
    -> m ()
volumeEject volume flags cancellable callback = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_volume_eject volume' flags' maybeCancellable maybeCallback userData
    touchManagedPtr volume
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data VolumeEjectMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsVolume a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod VolumeEjectMethodInfo a signature where
    overloadedMethod = volumeEject

instance O.OverloadedMethodInfo VolumeEjectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeEject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeEject"
        })


#endif

-- method Volume::eject_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "pointer to a #GVolume"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_volume_eject_finish" g_volume_eject_finish :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

{-# DEPRECATED volumeEjectFinish ["(Since version 2.22)","Use 'GI.Gio.Interfaces.Volume.volumeEjectWithOperationFinish' instead."] #-}
-- | Finishes ejecting a volume. If any errors occurred during the operation,
-- /@error@/ will be set to contain the errors and 'P.False' will be returned.
volumeEjectFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@volume@/: pointer to a t'GI.Gio.Interfaces.Volume.Volume'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
volumeEjectFinish volume result_ = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_volume_eject_finish volume' result_'
        touchManagedPtr volume
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data VolumeEjectFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsVolume a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod VolumeEjectFinishMethodInfo a signature where
    overloadedMethod = volumeEjectFinish

instance O.OverloadedMethodInfo VolumeEjectFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeEjectFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeEjectFinish"
        })


#endif

-- method Volume::eject_with_operation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags affecting the unmount if required for eject"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation or %NULL to\n    avoid user interaction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data passed to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_eject_with_operation" g_volume_eject_with_operation :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Ejects a volume. This is an asynchronous operation, and is
-- finished by calling 'GI.Gio.Interfaces.Volume.volumeEjectWithOperationFinish' with the /@volume@/
-- and t'GI.Gio.Interfaces.AsyncResult.AsyncResult' data returned in the /@callback@/.
-- 
-- /Since: 2.22/
volumeEjectWithOperation ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the unmount if required for eject
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation' or 'P.Nothing' to
    --     avoid user interaction
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback', or 'P.Nothing'
    -> m ()
volumeEjectWithOperation volume flags mountOperation cancellable callback = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_volume_eject_with_operation volume' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr volume
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data VolumeEjectWithOperationMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsVolume a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod VolumeEjectWithOperationMethodInfo a signature where
    overloadedMethod = volumeEjectWithOperation

instance O.OverloadedMethodInfo VolumeEjectWithOperationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeEjectWithOperation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeEjectWithOperation"
        })


#endif

-- method Volume::eject_with_operation_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_volume_eject_with_operation_finish" g_volume_eject_with_operation_finish :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes ejecting a volume. If any errors occurred during the operation,
-- /@error@/ will be set to contain the errors and 'P.False' will be returned.
-- 
-- /Since: 2.22/
volumeEjectWithOperationFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
volumeEjectWithOperationFinish volume result_ = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_volume_eject_with_operation_finish volume' result_'
        touchManagedPtr volume
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data VolumeEjectWithOperationFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsVolume a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod VolumeEjectWithOperationFinishMethodInfo a signature where
    overloadedMethod = volumeEjectWithOperationFinish

instance O.OverloadedMethodInfo VolumeEjectWithOperationFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeEjectWithOperationFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeEjectWithOperationFinish"
        })


#endif

-- method Volume::enumerate_identifiers
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_enumerate_identifiers" g_volume_enumerate_identifiers :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO (Ptr CString)

-- | Gets the kinds of <http://developer.gnome.org/gio/stable/#volume-identifiers identifiers> that /@volume@/ has.
-- Use 'GI.Gio.Interfaces.Volume.volumeGetIdentifier' to obtain the identifiers themselves.
volumeEnumerateIdentifiers ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m [T.Text]
    -- ^ __Returns:__ a 'P.Nothing'-terminated array
    --   of strings containing kinds of identifiers. Use 'GI.GLib.Functions.strfreev' to free.
volumeEnumerateIdentifiers volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_enumerate_identifiers volume'
    checkUnexpectedReturnNULL "volumeEnumerateIdentifiers" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    touchManagedPtr volume
    return result'

#if defined(ENABLE_OVERLOADING)
data VolumeEnumerateIdentifiersMethodInfo
instance (signature ~ (m [T.Text]), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeEnumerateIdentifiersMethodInfo a signature where
    overloadedMethod = volumeEnumerateIdentifiers

instance O.OverloadedMethodInfo VolumeEnumerateIdentifiersMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeEnumerateIdentifiers",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeEnumerateIdentifiers"
        })


#endif

-- method Volume::get_activation_root
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_activation_root" g_volume_get_activation_root :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO (Ptr Gio.File.File)

-- | Gets the activation root for a t'GI.Gio.Interfaces.Volume.Volume' if it is known ahead of
-- mount time. Returns 'P.Nothing' otherwise. If not 'P.Nothing' and if /@volume@/
-- is mounted, then the result of 'GI.Gio.Interfaces.Mount.mountGetRoot' on the
-- t'GI.Gio.Interfaces.Mount.Mount' object obtained from 'GI.Gio.Interfaces.Volume.volumeGetMount' will always
-- either be equal or a prefix of what this function returns. In
-- other words, in code
-- 
-- 
-- === /C code/
-- >
-- >  GMount *mount;
-- >  GFile *mount_root
-- >  GFile *volume_activation_root;
-- >
-- >  mount = g_volume_get_mount (volume); // mounted, so never NULL
-- >  mount_root = g_mount_get_root (mount);
-- >  volume_activation_root = g_volume_get_activation_root (volume); // assume not NULL
-- 
-- then the expression
-- 
-- === /C code/
-- >
-- >  (g_file_has_prefix (volume_activation_root, mount_root) ||
-- >   g_file_equal (volume_activation_root, mount_root))
-- 
-- will always be 'P.True'.
-- 
-- Activation roots are typically used in t'GI.Gio.Objects.VolumeMonitor.VolumeMonitor'
-- implementations to find the underlying mount to shadow, see
-- 'GI.Gio.Interfaces.Mount.mountIsShadowed' for more details.
-- 
-- /Since: 2.18/
volumeGetActivationRoot ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m (Maybe Gio.File.File)
    -- ^ __Returns:__ the activation root of /@volume@/
    --     or 'P.Nothing'. Use 'GI.GObject.Objects.Object.objectUnref' to free.
volumeGetActivationRoot volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_get_activation_root volume'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Gio.File.File) result'
        return result''
    touchManagedPtr volume
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VolumeGetActivationRootMethodInfo
instance (signature ~ (m (Maybe Gio.File.File)), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetActivationRootMethodInfo a signature where
    overloadedMethod = volumeGetActivationRoot

instance O.OverloadedMethodInfo VolumeGetActivationRootMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetActivationRoot",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetActivationRoot"
        })


#endif

-- method Volume::get_drive
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Drive" })
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_drive" g_volume_get_drive :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO (Ptr Gio.Drive.Drive)

-- | Gets the drive for the /@volume@/.
volumeGetDrive ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m (Maybe Gio.Drive.Drive)
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.Drive.Drive' or 'P.Nothing' if /@volume@/ is not
    --     associated with a drive. The returned object should be unreffed
    --     with 'GI.GObject.Objects.Object.objectUnref' when no longer needed.
volumeGetDrive volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_get_drive volume'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Gio.Drive.Drive) result'
        return result''
    touchManagedPtr volume
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VolumeGetDriveMethodInfo
instance (signature ~ (m (Maybe Gio.Drive.Drive)), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetDriveMethodInfo a signature where
    overloadedMethod = volumeGetDrive

instance O.OverloadedMethodInfo VolumeGetDriveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetDrive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetDrive"
        })


#endif

-- method Volume::get_icon
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Icon" })
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_icon" g_volume_get_icon :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO (Ptr Gio.Icon.Icon)

-- | Gets the icon for /@volume@/.
volumeGetIcon ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m Gio.Icon.Icon
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.Icon.Icon'.
    --     The returned object should be unreffed with 'GI.GObject.Objects.Object.objectUnref'
    --     when no longer needed.
volumeGetIcon volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_get_icon volume'
    checkUnexpectedReturnNULL "volumeGetIcon" result
    result' <- (wrapObject Gio.Icon.Icon) result
    touchManagedPtr volume
    return result'

#if defined(ENABLE_OVERLOADING)
data VolumeGetIconMethodInfo
instance (signature ~ (m Gio.Icon.Icon), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetIconMethodInfo a signature where
    overloadedMethod = volumeGetIcon

instance O.OverloadedMethodInfo VolumeGetIconMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetIcon",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetIcon"
        })


#endif

-- method Volume::get_identifier
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "kind"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the kind of identifier to return"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_identifier" g_volume_get_identifier :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    CString ->                              -- kind : TBasicType TUTF8
    IO CString

-- | Gets the identifier of the given kind for /@volume@/.
-- See the <http://developer.gnome.org/gio/stable/#volume-identifiers introduction> for more
-- information about volume identifiers.
volumeGetIdentifier ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> T.Text
    -- ^ /@kind@/: the kind of identifier to return
    -> m (Maybe T.Text)
    -- ^ __Returns:__ a newly allocated string containing the
    --     requested identifier, or 'P.Nothing' if the t'GI.Gio.Interfaces.Volume.Volume'
    --     doesn\'t have this kind of identifier
volumeGetIdentifier volume kind = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    kind' <- textToCString kind
    result <- g_volume_get_identifier volume' kind'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr volume
    freeMem kind'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VolumeGetIdentifierMethodInfo
instance (signature ~ (T.Text -> m (Maybe T.Text)), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetIdentifierMethodInfo a signature where
    overloadedMethod = volumeGetIdentifier

instance O.OverloadedMethodInfo VolumeGetIdentifierMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetIdentifier",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetIdentifier"
        })


#endif

-- method Volume::get_mount
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Mount" })
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_mount" g_volume_get_mount :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO (Ptr Gio.Mount.Mount)

-- | Gets the mount for the /@volume@/.
volumeGetMount ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m (Maybe Gio.Mount.Mount)
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.Mount.Mount' or 'P.Nothing' if /@volume@/ isn\'t mounted.
    --     The returned object should be unreffed with 'GI.GObject.Objects.Object.objectUnref'
    --     when no longer needed.
volumeGetMount volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_get_mount volume'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Gio.Mount.Mount) result'
        return result''
    touchManagedPtr volume
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VolumeGetMountMethodInfo
instance (signature ~ (m (Maybe Gio.Mount.Mount)), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetMountMethodInfo a signature where
    overloadedMethod = volumeGetMount

instance O.OverloadedMethodInfo VolumeGetMountMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetMount",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetMount"
        })


#endif

-- method Volume::get_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_name" g_volume_get_name :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO CString

-- | Gets the name of /@volume@/.
volumeGetName ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m T.Text
    -- ^ __Returns:__ the name for the given /@volume@/. The returned string should
    --     be freed with 'GI.GLib.Functions.free' when no longer needed.
volumeGetName volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_get_name volume'
    checkUnexpectedReturnNULL "volumeGetName" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr volume
    return result'

#if defined(ENABLE_OVERLOADING)
data VolumeGetNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetNameMethodInfo a signature where
    overloadedMethod = volumeGetName

instance O.OverloadedMethodInfo VolumeGetNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetName"
        })


#endif

-- method Volume::get_sort_key
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_sort_key" g_volume_get_sort_key :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO CString

-- | Gets the sort key for /@volume@/, if any.
-- 
-- /Since: 2.32/
volumeGetSortKey ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ Sorting key for /@volume@/ or 'P.Nothing' if no such key is available
volumeGetSortKey volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_get_sort_key volume'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr volume
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VolumeGetSortKeyMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetSortKeyMethodInfo a signature where
    overloadedMethod = volumeGetSortKey

instance O.OverloadedMethodInfo VolumeGetSortKeyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetSortKey",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetSortKey"
        })


#endif

-- method Volume::get_symbolic_icon
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Icon" })
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_symbolic_icon" g_volume_get_symbolic_icon :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO (Ptr Gio.Icon.Icon)

-- | Gets the symbolic icon for /@volume@/.
-- 
-- /Since: 2.34/
volumeGetSymbolicIcon ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m Gio.Icon.Icon
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.Icon.Icon'.
    --     The returned object should be unreffed with 'GI.GObject.Objects.Object.objectUnref'
    --     when no longer needed.
volumeGetSymbolicIcon volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_get_symbolic_icon volume'
    checkUnexpectedReturnNULL "volumeGetSymbolicIcon" result
    result' <- (wrapObject Gio.Icon.Icon) result
    touchManagedPtr volume
    return result'

#if defined(ENABLE_OVERLOADING)
data VolumeGetSymbolicIconMethodInfo
instance (signature ~ (m Gio.Icon.Icon), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetSymbolicIconMethodInfo a signature where
    overloadedMethod = volumeGetSymbolicIcon

instance O.OverloadedMethodInfo VolumeGetSymbolicIconMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetSymbolicIcon",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetSymbolicIcon"
        })


#endif

-- method Volume::get_uuid
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_get_uuid" g_volume_get_uuid :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO CString

-- | Gets the UUID for the /@volume@/. The reference is typically based on
-- the file system UUID for the volume in question and should be
-- considered an opaque string. Returns 'P.Nothing' if there is no UUID
-- available.
volumeGetUuid ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the UUID for /@volume@/ or 'P.Nothing' if no UUID
    --     can be computed.
    --     The returned string should be freed with 'GI.GLib.Functions.free'
    --     when no longer needed.
volumeGetUuid volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_get_uuid volume'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr volume
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VolumeGetUuidMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeGetUuidMethodInfo a signature where
    overloadedMethod = volumeGetUuid

instance O.OverloadedMethodInfo VolumeGetUuidMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeGetUuid",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeGetUuid"
        })


#endif

-- method Volume::mount
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountMountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation or %NULL to avoid user interaction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data that gets passed to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_mount" g_volume_mount :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountMountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Mounts a volume. This is an asynchronous operation, and is
-- finished by calling 'GI.Gio.Interfaces.Volume.volumeMountFinish' with the /@volume@/
-- and t'GI.Gio.Interfaces.AsyncResult.AsyncResult' returned in the /@callback@/.
volumeMount ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> [Gio.Flags.MountMountFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation' or 'P.Nothing' to avoid user interaction
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback', or 'P.Nothing'
    -> m ()
volumeMount volume flags mountOperation cancellable callback = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_volume_mount volume' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr volume
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data VolumeMountMethodInfo
instance (signature ~ ([Gio.Flags.MountMountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsVolume a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod VolumeMountMethodInfo a signature where
    overloadedMethod = volumeMount

instance O.OverloadedMethodInfo VolumeMountMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeMount",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeMount"
        })


#endif

-- method Volume::mount_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_volume_mount_finish" g_volume_mount_finish :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes mounting a volume. If any errors occurred during the operation,
-- /@error@/ will be set to contain the errors and 'P.False' will be returned.
-- 
-- If the mount operation succeeded, 'GI.Gio.Interfaces.Volume.volumeGetMount' on /@volume@/
-- is guaranteed to return the mount right after calling this
-- function; there\'s no need to listen for the \'mount-added\' signal on
-- t'GI.Gio.Objects.VolumeMonitor.VolumeMonitor'.
volumeMountFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
volumeMountFinish volume result_ = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_volume_mount_finish volume' result_'
        touchManagedPtr volume
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data VolumeMountFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsVolume a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod VolumeMountFinishMethodInfo a signature where
    overloadedMethod = volumeMountFinish

instance O.OverloadedMethodInfo VolumeMountFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeMountFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeMountFinish"
        })


#endif

-- method Volume::should_automount
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "volume"
--           , argType = TInterface Name { namespace = "Gio" , name = "Volume" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVolume" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_volume_should_automount" g_volume_should_automount :: 
    Ptr Volume ->                           -- volume : TInterface (Name {namespace = "Gio", name = "Volume"})
    IO CInt

-- | Returns whether the volume should be automatically mounted.
volumeShouldAutomount ::
    (B.CallStack.HasCallStack, MonadIO m, IsVolume a) =>
    a
    -- ^ /@volume@/: a t'GI.Gio.Interfaces.Volume.Volume'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the volume should be automatically mounted
volumeShouldAutomount volume = liftIO $ do
    volume' <- unsafeManagedPtrCastPtr volume
    result <- g_volume_should_automount volume'
    let result' = (/= 0) result
    touchManagedPtr volume
    return result'

#if defined(ENABLE_OVERLOADING)
data VolumeShouldAutomountMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsVolume a) => O.OverloadedMethod VolumeShouldAutomountMethodInfo a signature where
    overloadedMethod = volumeShouldAutomount

instance O.OverloadedMethodInfo VolumeShouldAutomountMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume.volumeShouldAutomount",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#v:volumeShouldAutomount"
        })


#endif

-- signal Volume::changed
-- | Emitted when the volume has been changed.
type VolumeChangedCallback =
    IO ()

type C_VolumeChangedCallback =
    Ptr Volume ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_VolumeChangedCallback`.
foreign import ccall "wrapper"
    mk_VolumeChangedCallback :: C_VolumeChangedCallback -> IO (FunPtr C_VolumeChangedCallback)

wrap_VolumeChangedCallback :: 
    GObject a => (a -> VolumeChangedCallback) ->
    C_VolumeChangedCallback
wrap_VolumeChangedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [changed](#signal:changed) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' volume #changed callback
-- @
-- 
-- 
onVolumeChanged :: (IsVolume a, MonadIO m) => a -> ((?self :: a) => VolumeChangedCallback) -> m SignalHandlerId
onVolumeChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_VolumeChangedCallback wrapped
    wrapped'' <- mk_VolumeChangedCallback wrapped'
    connectSignalFunPtr obj "changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [changed](#signal:changed) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' volume #changed callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterVolumeChanged :: (IsVolume a, MonadIO m) => a -> ((?self :: a) => VolumeChangedCallback) -> m SignalHandlerId
afterVolumeChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_VolumeChangedCallback wrapped
    wrapped'' <- mk_VolumeChangedCallback wrapped'
    connectSignalFunPtr obj "changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data VolumeChangedSignalInfo
instance SignalInfo VolumeChangedSignalInfo where
    type HaskellCallbackType VolumeChangedSignalInfo = VolumeChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_VolumeChangedCallback cb
        cb'' <- mk_VolumeChangedCallback cb'
        connectSignalFunPtr obj "changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume::changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#g:signal:changed"})

#endif

-- signal Volume::removed
-- | This signal is emitted when the t'GI.Gio.Interfaces.Volume.Volume' have been removed. If
-- the recipient is holding references to the object they should
-- release them so the object can be finalized.
type VolumeRemovedCallback =
    IO ()

type C_VolumeRemovedCallback =
    Ptr Volume ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_VolumeRemovedCallback`.
foreign import ccall "wrapper"
    mk_VolumeRemovedCallback :: C_VolumeRemovedCallback -> IO (FunPtr C_VolumeRemovedCallback)

wrap_VolumeRemovedCallback :: 
    GObject a => (a -> VolumeRemovedCallback) ->
    C_VolumeRemovedCallback
wrap_VolumeRemovedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [removed](#signal:removed) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' volume #removed callback
-- @
-- 
-- 
onVolumeRemoved :: (IsVolume a, MonadIO m) => a -> ((?self :: a) => VolumeRemovedCallback) -> m SignalHandlerId
onVolumeRemoved obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_VolumeRemovedCallback wrapped
    wrapped'' <- mk_VolumeRemovedCallback wrapped'
    connectSignalFunPtr obj "removed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [removed](#signal:removed) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' volume #removed callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterVolumeRemoved :: (IsVolume a, MonadIO m) => a -> ((?self :: a) => VolumeRemovedCallback) -> m SignalHandlerId
afterVolumeRemoved obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_VolumeRemovedCallback wrapped
    wrapped'' <- mk_VolumeRemovedCallback wrapped'
    connectSignalFunPtr obj "removed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data VolumeRemovedSignalInfo
instance SignalInfo VolumeRemovedSignalInfo where
    type HaskellCallbackType VolumeRemovedSignalInfo = VolumeRemovedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_VolumeRemovedCallback cb
        cb'' <- mk_VolumeRemovedCallback cb'
        connectSignalFunPtr obj "removed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Volume::removed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Volume.html#g:signal:removed"})

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Volume = VolumeSignalList
type VolumeSignalList = ('[ '("changed", VolumeChangedSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("removed", VolumeRemovedSignalInfo)] :: [(Symbol, DK.Type)])

#endif


