{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GFile@ is a high level abstraction for manipulating files on a
-- virtual file system. @GFile@s are lightweight, immutable objects
-- that do no I\/O upon creation. It is necessary to understand that
-- @GFile@ objects do not represent files, merely an identifier for a
-- file. All file content I\/O is implemented as streaming operations
-- (see t'GI.Gio.Objects.InputStream.InputStream' and t'GI.Gio.Objects.OutputStream.OutputStream').
-- 
-- To construct a @GFile@, you can use:
-- 
-- * [func/@gio@/.File.new_for_path] if you have a path.
-- * [func/@gio@/.File.new_for_uri] if you have a URI.
-- * [func/@gio@/.File.new_for_commandline_arg] or
-- [func/@gio@/.File.new_for_commandline_arg_and_cwd] for a command line
-- argument.
-- * [func/@gio@/.File.new_tmp] to create a temporary file from a template.
-- * [func/@gio@/.File.new_tmp_async] to asynchronously create a temporary file.
-- * [func/@gio@/.File.new_tmp_dir_async] to asynchronously create a temporary
-- directory.
-- * [func/@gio@/.File.parse_name] from a UTF-8 string gotten from
-- 'GI.Gio.Interfaces.File.fileGetParseName'.
-- * [func/@gio@/.File.new_build_filename] or [func/@gio@/.File.new_build_filenamev]
-- to create a file from path elements.
-- 
-- 
-- One way to think of a @GFile@ is as an abstraction of a pathname. For
-- normal files the system pathname is what is stored internally, but as
-- @GFile@s are extensible it could also be something else that corresponds
-- to a pathname in a userspace implementation of a filesystem.
-- 
-- @GFile@s make up hierarchies of directories and files that correspond to
-- the files on a filesystem. You can move through the file system with
-- @GFile@ using 'GI.Gio.Interfaces.File.fileGetParent' to get an identifier for the
-- parent directory, 'GI.Gio.Interfaces.File.fileGetChild' to get a child within a
-- directory, and 'GI.Gio.Interfaces.File.fileResolveRelativePath' to resolve a relative
-- path between two @GFile@s. There can be multiple hierarchies, so you may not
-- end up at the same root if you repeatedly call 'GI.Gio.Interfaces.File.fileGetParent'
-- on two different files.
-- 
-- All @GFile@s have a basename (get with 'GI.Gio.Interfaces.File.fileGetBasename'). These
-- names are byte strings that are used to identify the file on the filesystem
-- (relative to its parent directory) and there is no guarantees that they
-- have any particular charset encoding or even make any sense at all. If
-- you want to use filenames in a user interface you should use the display
-- name that you can get by requesting the
-- @G_FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME@ attribute with
-- 'GI.Gio.Interfaces.File.fileQueryInfo'. This is guaranteed to be in UTF-8 and can be
-- used in a user interface. But always store the real basename or the @GFile@
-- to use to actually access the file, because there is no way to go from a
-- display name to the actual name.
-- 
-- Using @GFile@ as an identifier has the same weaknesses as using a path
-- in that there may be multiple aliases for the same file. For instance,
-- hard or soft links may cause two different @GFile@s to refer to the same
-- file. Other possible causes for aliases are: case insensitive filesystems,
-- short and long names on FAT\/NTFS, or bind mounts in Linux. If you want to
-- check if two @GFile@s point to the same file you can query for the
-- @G_FILE_ATTRIBUTE_ID_FILE@ attribute. Note that @GFile@ does some trivial
-- canonicalization of pathnames passed in, so that trivial differences in
-- the path string used at creation (duplicated slashes, slash at end of
-- path, @.@ or @..@ path segments, etc) does not create different @GFile@s.
-- 
-- Many @GFile@ operations have both synchronous and asynchronous versions
-- to suit your application. Asynchronous versions of synchronous functions
-- simply have @_async()@ appended to their function names. The asynchronous
-- I\/O functions call a [callback/@gio@/.AsyncReadyCallback] which is then used to
-- finalize the operation, producing a t'GI.Gio.Interfaces.AsyncResult.AsyncResult' which is then
-- passed to the function’s matching @_finish()@ operation.
-- 
-- It is highly recommended to use asynchronous calls when running within a
-- shared main loop, such as in the main thread of an application. This avoids
-- I\/O operations blocking other sources on the main loop from being dispatched.
-- Synchronous I\/O operations should be performed from worker threads. See the
-- <http://developer.gnome.org/gio/stable/overview.html#asynchronous-programming introduction to asynchronous programming section>
-- for more.
-- 
-- Some @GFile@ operations almost always take a noticeable amount of time, and
-- so do not have synchronous analogs. Notable cases include:
-- 
-- * 'GI.Gio.Interfaces.File.fileMountMountable' to mount a mountable file.
-- * 'GI.Gio.Interfaces.File.fileUnmountMountableWithOperation' to unmount a mountable
-- file.
-- * 'GI.Gio.Interfaces.File.fileEjectMountableWithOperation' to eject a mountable file.
-- 
-- 
-- == Entity Tags
-- 
-- One notable feature of @GFile@s are entity tags, or ‘etags’ for
-- short. Entity tags are somewhat like a more abstract version of the
-- traditional mtime, and can be used to quickly determine if the file
-- has been modified from the version on the file system. See the
-- HTTP 1.1
-- <http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html specification>
-- for HTTP @ETag@ headers, which are a very similar concept.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Interfaces.File
    ( 

-- * Exported types
    File(..)                                ,
    IsFile                                  ,
    toFile                                  ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [appendTo]("GI.Gio.Interfaces.File#g:method:appendTo"), [appendToAsync]("GI.Gio.Interfaces.File#g:method:appendToAsync"), [appendToFinish]("GI.Gio.Interfaces.File#g:method:appendToFinish"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [buildAttributeListForCopy]("GI.Gio.Interfaces.File#g:method:buildAttributeListForCopy"), [copy]("GI.Gio.Interfaces.File#g:method:copy"), [copyAsync]("GI.Gio.Interfaces.File#g:method:copyAsync"), [copyAttributes]("GI.Gio.Interfaces.File#g:method:copyAttributes"), [copyFinish]("GI.Gio.Interfaces.File#g:method:copyFinish"), [create]("GI.Gio.Interfaces.File#g:method:create"), [createAsync]("GI.Gio.Interfaces.File#g:method:createAsync"), [createFinish]("GI.Gio.Interfaces.File#g:method:createFinish"), [createReadwrite]("GI.Gio.Interfaces.File#g:method:createReadwrite"), [createReadwriteAsync]("GI.Gio.Interfaces.File#g:method:createReadwriteAsync"), [createReadwriteFinish]("GI.Gio.Interfaces.File#g:method:createReadwriteFinish"), [delete]("GI.Gio.Interfaces.File#g:method:delete"), [deleteAsync]("GI.Gio.Interfaces.File#g:method:deleteAsync"), [deleteFinish]("GI.Gio.Interfaces.File#g:method:deleteFinish"), [dup]("GI.Gio.Interfaces.File#g:method:dup"), [ejectMountable]("GI.Gio.Interfaces.File#g:method:ejectMountable"), [ejectMountableFinish]("GI.Gio.Interfaces.File#g:method:ejectMountableFinish"), [ejectMountableWithOperation]("GI.Gio.Interfaces.File#g:method:ejectMountableWithOperation"), [ejectMountableWithOperationFinish]("GI.Gio.Interfaces.File#g:method:ejectMountableWithOperationFinish"), [enumerateChildren]("GI.Gio.Interfaces.File#g:method:enumerateChildren"), [enumerateChildrenAsync]("GI.Gio.Interfaces.File#g:method:enumerateChildrenAsync"), [enumerateChildrenFinish]("GI.Gio.Interfaces.File#g:method:enumerateChildrenFinish"), [equal]("GI.Gio.Interfaces.File#g:method:equal"), [findEnclosingMount]("GI.Gio.Interfaces.File#g:method:findEnclosingMount"), [findEnclosingMountAsync]("GI.Gio.Interfaces.File#g:method:findEnclosingMountAsync"), [findEnclosingMountFinish]("GI.Gio.Interfaces.File#g:method:findEnclosingMountFinish"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [hasParent]("GI.Gio.Interfaces.File#g:method:hasParent"), [hasPrefix]("GI.Gio.Interfaces.File#g:method:hasPrefix"), [hasUriScheme]("GI.Gio.Interfaces.File#g:method:hasUriScheme"), [hash]("GI.Gio.Interfaces.File#g:method:hash"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isNative]("GI.Gio.Interfaces.File#g:method:isNative"), [loadBytes]("GI.Gio.Interfaces.File#g:method:loadBytes"), [loadBytesAsync]("GI.Gio.Interfaces.File#g:method:loadBytesAsync"), [loadBytesFinish]("GI.Gio.Interfaces.File#g:method:loadBytesFinish"), [loadContents]("GI.Gio.Interfaces.File#g:method:loadContents"), [loadContentsAsync]("GI.Gio.Interfaces.File#g:method:loadContentsAsync"), [loadContentsFinish]("GI.Gio.Interfaces.File#g:method:loadContentsFinish"), [loadPartialContentsFinish]("GI.Gio.Interfaces.File#g:method:loadPartialContentsFinish"), [makeDirectory]("GI.Gio.Interfaces.File#g:method:makeDirectory"), [makeDirectoryAsync]("GI.Gio.Interfaces.File#g:method:makeDirectoryAsync"), [makeDirectoryFinish]("GI.Gio.Interfaces.File#g:method:makeDirectoryFinish"), [makeDirectoryWithParents]("GI.Gio.Interfaces.File#g:method:makeDirectoryWithParents"), [makeSymbolicLink]("GI.Gio.Interfaces.File#g:method:makeSymbolicLink"), [makeSymbolicLinkAsync]("GI.Gio.Interfaces.File#g:method:makeSymbolicLinkAsync"), [makeSymbolicLinkFinish]("GI.Gio.Interfaces.File#g:method:makeSymbolicLinkFinish"), [measureDiskUsage]("GI.Gio.Interfaces.File#g:method:measureDiskUsage"), [measureDiskUsageFinish]("GI.Gio.Interfaces.File#g:method:measureDiskUsageFinish"), [monitor]("GI.Gio.Interfaces.File#g:method:monitor"), [monitorDirectory]("GI.Gio.Interfaces.File#g:method:monitorDirectory"), [monitorFile]("GI.Gio.Interfaces.File#g:method:monitorFile"), [mountEnclosingVolume]("GI.Gio.Interfaces.File#g:method:mountEnclosingVolume"), [mountEnclosingVolumeFinish]("GI.Gio.Interfaces.File#g:method:mountEnclosingVolumeFinish"), [mountMountable]("GI.Gio.Interfaces.File#g:method:mountMountable"), [mountMountableFinish]("GI.Gio.Interfaces.File#g:method:mountMountableFinish"), [move]("GI.Gio.Interfaces.File#g:method:move"), [moveAsync]("GI.Gio.Interfaces.File#g:method:moveAsync"), [moveFinish]("GI.Gio.Interfaces.File#g:method:moveFinish"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [openReadwrite]("GI.Gio.Interfaces.File#g:method:openReadwrite"), [openReadwriteAsync]("GI.Gio.Interfaces.File#g:method:openReadwriteAsync"), [openReadwriteFinish]("GI.Gio.Interfaces.File#g:method:openReadwriteFinish"), [peekPath]("GI.Gio.Interfaces.File#g:method:peekPath"), [pollMountable]("GI.Gio.Interfaces.File#g:method:pollMountable"), [pollMountableFinish]("GI.Gio.Interfaces.File#g:method:pollMountableFinish"), [queryDefaultHandler]("GI.Gio.Interfaces.File#g:method:queryDefaultHandler"), [queryDefaultHandlerAsync]("GI.Gio.Interfaces.File#g:method:queryDefaultHandlerAsync"), [queryDefaultHandlerFinish]("GI.Gio.Interfaces.File#g:method:queryDefaultHandlerFinish"), [queryExists]("GI.Gio.Interfaces.File#g:method:queryExists"), [queryFileType]("GI.Gio.Interfaces.File#g:method:queryFileType"), [queryFilesystemInfo]("GI.Gio.Interfaces.File#g:method:queryFilesystemInfo"), [queryFilesystemInfoAsync]("GI.Gio.Interfaces.File#g:method:queryFilesystemInfoAsync"), [queryFilesystemInfoFinish]("GI.Gio.Interfaces.File#g:method:queryFilesystemInfoFinish"), [queryInfo]("GI.Gio.Interfaces.File#g:method:queryInfo"), [queryInfoAsync]("GI.Gio.Interfaces.File#g:method:queryInfoAsync"), [queryInfoFinish]("GI.Gio.Interfaces.File#g:method:queryInfoFinish"), [querySettableAttributes]("GI.Gio.Interfaces.File#g:method:querySettableAttributes"), [queryWritableNamespaces]("GI.Gio.Interfaces.File#g:method:queryWritableNamespaces"), [read]("GI.Gio.Interfaces.File#g:method:read"), [readAsync]("GI.Gio.Interfaces.File#g:method:readAsync"), [readFinish]("GI.Gio.Interfaces.File#g:method:readFinish"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [replace]("GI.Gio.Interfaces.File#g:method:replace"), [replaceAsync]("GI.Gio.Interfaces.File#g:method:replaceAsync"), [replaceContents]("GI.Gio.Interfaces.File#g:method:replaceContents"), [replaceContentsAsync]("GI.Gio.Interfaces.File#g:method:replaceContentsAsync"), [replaceContentsBytesAsync]("GI.Gio.Interfaces.File#g:method:replaceContentsBytesAsync"), [replaceContentsFinish]("GI.Gio.Interfaces.File#g:method:replaceContentsFinish"), [replaceFinish]("GI.Gio.Interfaces.File#g:method:replaceFinish"), [replaceReadwrite]("GI.Gio.Interfaces.File#g:method:replaceReadwrite"), [replaceReadwriteAsync]("GI.Gio.Interfaces.File#g:method:replaceReadwriteAsync"), [replaceReadwriteFinish]("GI.Gio.Interfaces.File#g:method:replaceReadwriteFinish"), [resolveRelativePath]("GI.Gio.Interfaces.File#g:method:resolveRelativePath"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [startMountable]("GI.Gio.Interfaces.File#g:method:startMountable"), [startMountableFinish]("GI.Gio.Interfaces.File#g:method:startMountableFinish"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [stopMountable]("GI.Gio.Interfaces.File#g:method:stopMountable"), [stopMountableFinish]("GI.Gio.Interfaces.File#g:method:stopMountableFinish"), [supportsThreadContexts]("GI.Gio.Interfaces.File#g:method:supportsThreadContexts"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [trash]("GI.Gio.Interfaces.File#g:method:trash"), [trashAsync]("GI.Gio.Interfaces.File#g:method:trashAsync"), [trashFinish]("GI.Gio.Interfaces.File#g:method:trashFinish"), [unmountMountable]("GI.Gio.Interfaces.File#g:method:unmountMountable"), [unmountMountableFinish]("GI.Gio.Interfaces.File#g:method:unmountMountableFinish"), [unmountMountableWithOperation]("GI.Gio.Interfaces.File#g:method:unmountMountableWithOperation"), [unmountMountableWithOperationFinish]("GI.Gio.Interfaces.File#g:method:unmountMountableWithOperationFinish"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getBasename]("GI.Gio.Interfaces.File#g:method:getBasename"), [getChild]("GI.Gio.Interfaces.File#g:method:getChild"), [getChildForDisplayName]("GI.Gio.Interfaces.File#g:method:getChildForDisplayName"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getParent]("GI.Gio.Interfaces.File#g:method:getParent"), [getParseName]("GI.Gio.Interfaces.File#g:method:getParseName"), [getPath]("GI.Gio.Interfaces.File#g:method:getPath"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRelativePath]("GI.Gio.Interfaces.File#g:method:getRelativePath"), [getUri]("GI.Gio.Interfaces.File#g:method:getUri"), [getUriScheme]("GI.Gio.Interfaces.File#g:method:getUriScheme").
-- 
-- ==== Setters
-- [setAttribute]("GI.Gio.Interfaces.File#g:method:setAttribute"), [setAttributeByteString]("GI.Gio.Interfaces.File#g:method:setAttributeByteString"), [setAttributeInt32]("GI.Gio.Interfaces.File#g:method:setAttributeInt32"), [setAttributeInt64]("GI.Gio.Interfaces.File#g:method:setAttributeInt64"), [setAttributeString]("GI.Gio.Interfaces.File#g:method:setAttributeString"), [setAttributeUint32]("GI.Gio.Interfaces.File#g:method:setAttributeUint32"), [setAttributeUint64]("GI.Gio.Interfaces.File#g:method:setAttributeUint64"), [setAttributesAsync]("GI.Gio.Interfaces.File#g:method:setAttributesAsync"), [setAttributesFinish]("GI.Gio.Interfaces.File#g:method:setAttributesFinish"), [setAttributesFromInfo]("GI.Gio.Interfaces.File#g:method:setAttributesFromInfo"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDisplayName]("GI.Gio.Interfaces.File#g:method:setDisplayName"), [setDisplayNameAsync]("GI.Gio.Interfaces.File#g:method:setDisplayNameAsync"), [setDisplayNameFinish]("GI.Gio.Interfaces.File#g:method:setDisplayNameFinish"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveFileMethod                       ,
#endif

-- ** appendTo #method:appendTo#

#if defined(ENABLE_OVERLOADING)
    FileAppendToMethodInfo                  ,
#endif
    fileAppendTo                            ,


-- ** appendToAsync #method:appendToAsync#

#if defined(ENABLE_OVERLOADING)
    FileAppendToAsyncMethodInfo             ,
#endif
    fileAppendToAsync                       ,


-- ** appendToFinish #method:appendToFinish#

#if defined(ENABLE_OVERLOADING)
    FileAppendToFinishMethodInfo            ,
#endif
    fileAppendToFinish                      ,


-- ** buildAttributeListForCopy #method:buildAttributeListForCopy#

#if defined(ENABLE_OVERLOADING)
    FileBuildAttributeListForCopyMethodInfo ,
#endif
    fileBuildAttributeListForCopy           ,


-- ** copy #method:copy#

#if defined(ENABLE_OVERLOADING)
    FileCopyMethodInfo                      ,
#endif
    fileCopy                                ,


-- ** copyAsync #method:copyAsync#

#if defined(ENABLE_OVERLOADING)
    FileCopyAsyncMethodInfo                 ,
#endif
    fileCopyAsync                           ,


-- ** copyAttributes #method:copyAttributes#

#if defined(ENABLE_OVERLOADING)
    FileCopyAttributesMethodInfo            ,
#endif
    fileCopyAttributes                      ,


-- ** copyFinish #method:copyFinish#

#if defined(ENABLE_OVERLOADING)
    FileCopyFinishMethodInfo                ,
#endif
    fileCopyFinish                          ,


-- ** create #method:create#

#if defined(ENABLE_OVERLOADING)
    FileCreateMethodInfo                    ,
#endif
    fileCreate                              ,


-- ** createAsync #method:createAsync#

#if defined(ENABLE_OVERLOADING)
    FileCreateAsyncMethodInfo               ,
#endif
    fileCreateAsync                         ,


-- ** createFinish #method:createFinish#

#if defined(ENABLE_OVERLOADING)
    FileCreateFinishMethodInfo              ,
#endif
    fileCreateFinish                        ,


-- ** createReadwrite #method:createReadwrite#

#if defined(ENABLE_OVERLOADING)
    FileCreateReadwriteMethodInfo           ,
#endif
    fileCreateReadwrite                     ,


-- ** createReadwriteAsync #method:createReadwriteAsync#

#if defined(ENABLE_OVERLOADING)
    FileCreateReadwriteAsyncMethodInfo      ,
#endif
    fileCreateReadwriteAsync                ,


-- ** createReadwriteFinish #method:createReadwriteFinish#

#if defined(ENABLE_OVERLOADING)
    FileCreateReadwriteFinishMethodInfo     ,
#endif
    fileCreateReadwriteFinish               ,


-- ** delete #method:delete#

#if defined(ENABLE_OVERLOADING)
    FileDeleteMethodInfo                    ,
#endif
    fileDelete                              ,


-- ** deleteAsync #method:deleteAsync#

#if defined(ENABLE_OVERLOADING)
    FileDeleteAsyncMethodInfo               ,
#endif
    fileDeleteAsync                         ,


-- ** deleteFinish #method:deleteFinish#

#if defined(ENABLE_OVERLOADING)
    FileDeleteFinishMethodInfo              ,
#endif
    fileDeleteFinish                        ,


-- ** dup #method:dup#

#if defined(ENABLE_OVERLOADING)
    FileDupMethodInfo                       ,
#endif
    fileDup                                 ,


-- ** ejectMountable #method:ejectMountable#

#if defined(ENABLE_OVERLOADING)
    FileEjectMountableMethodInfo            ,
#endif
    fileEjectMountable                      ,


-- ** ejectMountableFinish #method:ejectMountableFinish#

#if defined(ENABLE_OVERLOADING)
    FileEjectMountableFinishMethodInfo      ,
#endif
    fileEjectMountableFinish                ,


-- ** ejectMountableWithOperation #method:ejectMountableWithOperation#

#if defined(ENABLE_OVERLOADING)
    FileEjectMountableWithOperationMethodInfo,
#endif
    fileEjectMountableWithOperation         ,


-- ** ejectMountableWithOperationFinish #method:ejectMountableWithOperationFinish#

#if defined(ENABLE_OVERLOADING)
    FileEjectMountableWithOperationFinishMethodInfo,
#endif
    fileEjectMountableWithOperationFinish   ,


-- ** enumerateChildren #method:enumerateChildren#

#if defined(ENABLE_OVERLOADING)
    FileEnumerateChildrenMethodInfo         ,
#endif
    fileEnumerateChildren                   ,


-- ** enumerateChildrenAsync #method:enumerateChildrenAsync#

#if defined(ENABLE_OVERLOADING)
    FileEnumerateChildrenAsyncMethodInfo    ,
#endif
    fileEnumerateChildrenAsync              ,


-- ** enumerateChildrenFinish #method:enumerateChildrenFinish#

#if defined(ENABLE_OVERLOADING)
    FileEnumerateChildrenFinishMethodInfo   ,
#endif
    fileEnumerateChildrenFinish             ,


-- ** equal #method:equal#

#if defined(ENABLE_OVERLOADING)
    FileEqualMethodInfo                     ,
#endif
    fileEqual                               ,


-- ** findEnclosingMount #method:findEnclosingMount#

#if defined(ENABLE_OVERLOADING)
    FileFindEnclosingMountMethodInfo        ,
#endif
    fileFindEnclosingMount                  ,


-- ** findEnclosingMountAsync #method:findEnclosingMountAsync#

#if defined(ENABLE_OVERLOADING)
    FileFindEnclosingMountAsyncMethodInfo   ,
#endif
    fileFindEnclosingMountAsync             ,


-- ** findEnclosingMountFinish #method:findEnclosingMountFinish#

#if defined(ENABLE_OVERLOADING)
    FileFindEnclosingMountFinishMethodInfo  ,
#endif
    fileFindEnclosingMountFinish            ,


-- ** getBasename #method:getBasename#

#if defined(ENABLE_OVERLOADING)
    FileGetBasenameMethodInfo               ,
#endif
    fileGetBasename                         ,


-- ** getChild #method:getChild#

#if defined(ENABLE_OVERLOADING)
    FileGetChildMethodInfo                  ,
#endif
    fileGetChild                            ,


-- ** getChildForDisplayName #method:getChildForDisplayName#

#if defined(ENABLE_OVERLOADING)
    FileGetChildForDisplayNameMethodInfo    ,
#endif
    fileGetChildForDisplayName              ,


-- ** getParent #method:getParent#

#if defined(ENABLE_OVERLOADING)
    FileGetParentMethodInfo                 ,
#endif
    fileGetParent                           ,


-- ** getParseName #method:getParseName#

#if defined(ENABLE_OVERLOADING)
    FileGetParseNameMethodInfo              ,
#endif
    fileGetParseName                        ,


-- ** getPath #method:getPath#

#if defined(ENABLE_OVERLOADING)
    FileGetPathMethodInfo                   ,
#endif
    fileGetPath                             ,


-- ** getRelativePath #method:getRelativePath#

#if defined(ENABLE_OVERLOADING)
    FileGetRelativePathMethodInfo           ,
#endif
    fileGetRelativePath                     ,


-- ** getUri #method:getUri#

#if defined(ENABLE_OVERLOADING)
    FileGetUriMethodInfo                    ,
#endif
    fileGetUri                              ,


-- ** getUriScheme #method:getUriScheme#

#if defined(ENABLE_OVERLOADING)
    FileGetUriSchemeMethodInfo              ,
#endif
    fileGetUriScheme                        ,


-- ** hasParent #method:hasParent#

#if defined(ENABLE_OVERLOADING)
    FileHasParentMethodInfo                 ,
#endif
    fileHasParent                           ,


-- ** hasPrefix #method:hasPrefix#

#if defined(ENABLE_OVERLOADING)
    FileHasPrefixMethodInfo                 ,
#endif
    fileHasPrefix                           ,


-- ** hasUriScheme #method:hasUriScheme#

#if defined(ENABLE_OVERLOADING)
    FileHasUriSchemeMethodInfo              ,
#endif
    fileHasUriScheme                        ,


-- ** hash #method:hash#

#if defined(ENABLE_OVERLOADING)
    FileHashMethodInfo                      ,
#endif
    fileHash                                ,


-- ** isNative #method:isNative#

#if defined(ENABLE_OVERLOADING)
    FileIsNativeMethodInfo                  ,
#endif
    fileIsNative                            ,


-- ** loadBytes #method:loadBytes#

#if defined(ENABLE_OVERLOADING)
    FileLoadBytesMethodInfo                 ,
#endif
    fileLoadBytes                           ,


-- ** loadBytesAsync #method:loadBytesAsync#

#if defined(ENABLE_OVERLOADING)
    FileLoadBytesAsyncMethodInfo            ,
#endif
    fileLoadBytesAsync                      ,


-- ** loadBytesFinish #method:loadBytesFinish#

#if defined(ENABLE_OVERLOADING)
    FileLoadBytesFinishMethodInfo           ,
#endif
    fileLoadBytesFinish                     ,


-- ** loadContents #method:loadContents#

#if defined(ENABLE_OVERLOADING)
    FileLoadContentsMethodInfo              ,
#endif
    fileLoadContents                        ,


-- ** loadContentsAsync #method:loadContentsAsync#

#if defined(ENABLE_OVERLOADING)
    FileLoadContentsAsyncMethodInfo         ,
#endif
    fileLoadContentsAsync                   ,


-- ** loadContentsFinish #method:loadContentsFinish#

#if defined(ENABLE_OVERLOADING)
    FileLoadContentsFinishMethodInfo        ,
#endif
    fileLoadContentsFinish                  ,


-- ** loadPartialContentsFinish #method:loadPartialContentsFinish#

#if defined(ENABLE_OVERLOADING)
    FileLoadPartialContentsFinishMethodInfo ,
#endif
    fileLoadPartialContentsFinish           ,


-- ** makeDirectory #method:makeDirectory#

#if defined(ENABLE_OVERLOADING)
    FileMakeDirectoryMethodInfo             ,
#endif
    fileMakeDirectory                       ,


-- ** makeDirectoryAsync #method:makeDirectoryAsync#

#if defined(ENABLE_OVERLOADING)
    FileMakeDirectoryAsyncMethodInfo        ,
#endif
    fileMakeDirectoryAsync                  ,


-- ** makeDirectoryFinish #method:makeDirectoryFinish#

#if defined(ENABLE_OVERLOADING)
    FileMakeDirectoryFinishMethodInfo       ,
#endif
    fileMakeDirectoryFinish                 ,


-- ** makeDirectoryWithParents #method:makeDirectoryWithParents#

#if defined(ENABLE_OVERLOADING)
    FileMakeDirectoryWithParentsMethodInfo  ,
#endif
    fileMakeDirectoryWithParents            ,


-- ** makeSymbolicLink #method:makeSymbolicLink#

#if defined(ENABLE_OVERLOADING)
    FileMakeSymbolicLinkMethodInfo          ,
#endif
    fileMakeSymbolicLink                    ,


-- ** makeSymbolicLinkAsync #method:makeSymbolicLinkAsync#

#if defined(ENABLE_OVERLOADING)
    FileMakeSymbolicLinkAsyncMethodInfo     ,
#endif
    fileMakeSymbolicLinkAsync               ,


-- ** makeSymbolicLinkFinish #method:makeSymbolicLinkFinish#

#if defined(ENABLE_OVERLOADING)
    FileMakeSymbolicLinkFinishMethodInfo    ,
#endif
    fileMakeSymbolicLinkFinish              ,


-- ** measureDiskUsage #method:measureDiskUsage#

#if defined(ENABLE_OVERLOADING)
    FileMeasureDiskUsageMethodInfo          ,
#endif
    fileMeasureDiskUsage                    ,


-- ** measureDiskUsageFinish #method:measureDiskUsageFinish#

#if defined(ENABLE_OVERLOADING)
    FileMeasureDiskUsageFinishMethodInfo    ,
#endif
    fileMeasureDiskUsageFinish              ,


-- ** monitor #method:monitor#

#if defined(ENABLE_OVERLOADING)
    FileMonitorMethodInfo                   ,
#endif
    fileMonitor                             ,


-- ** monitorDirectory #method:monitorDirectory#

#if defined(ENABLE_OVERLOADING)
    FileMonitorDirectoryMethodInfo          ,
#endif
    fileMonitorDirectory                    ,


-- ** monitorFile #method:monitorFile#

#if defined(ENABLE_OVERLOADING)
    FileMonitorFileMethodInfo               ,
#endif
    fileMonitorFile                         ,


-- ** mountEnclosingVolume #method:mountEnclosingVolume#

#if defined(ENABLE_OVERLOADING)
    FileMountEnclosingVolumeMethodInfo      ,
#endif
    fileMountEnclosingVolume                ,


-- ** mountEnclosingVolumeFinish #method:mountEnclosingVolumeFinish#

#if defined(ENABLE_OVERLOADING)
    FileMountEnclosingVolumeFinishMethodInfo,
#endif
    fileMountEnclosingVolumeFinish          ,


-- ** mountMountable #method:mountMountable#

#if defined(ENABLE_OVERLOADING)
    FileMountMountableMethodInfo            ,
#endif
    fileMountMountable                      ,


-- ** mountMountableFinish #method:mountMountableFinish#

#if defined(ENABLE_OVERLOADING)
    FileMountMountableFinishMethodInfo      ,
#endif
    fileMountMountableFinish                ,


-- ** move #method:move#

#if defined(ENABLE_OVERLOADING)
    FileMoveMethodInfo                      ,
#endif
    fileMove                                ,


-- ** moveAsync #method:moveAsync#

#if defined(ENABLE_OVERLOADING)
    FileMoveAsyncMethodInfo                 ,
#endif
    fileMoveAsync                           ,


-- ** moveFinish #method:moveFinish#

#if defined(ENABLE_OVERLOADING)
    FileMoveFinishMethodInfo                ,
#endif
    fileMoveFinish                          ,


-- ** newBuildFilenamev #method:newBuildFilenamev#

    fileNewBuildFilenamev                   ,


-- ** newForCommandlineArg #method:newForCommandlineArg#

    fileNewForCommandlineArg                ,


-- ** newForCommandlineArgAndCwd #method:newForCommandlineArgAndCwd#

    fileNewForCommandlineArgAndCwd          ,


-- ** newForPath #method:newForPath#

    fileNewForPath                          ,


-- ** newForUri #method:newForUri#

    fileNewForUri                           ,


-- ** newTmp #method:newTmp#

    fileNewTmp                              ,


-- ** newTmpAsync #method:newTmpAsync#

    fileNewTmpAsync                         ,


-- ** newTmpDirAsync #method:newTmpDirAsync#

    fileNewTmpDirAsync                      ,


-- ** newTmpDirFinish #method:newTmpDirFinish#

    fileNewTmpDirFinish                     ,


-- ** newTmpFinish #method:newTmpFinish#

    fileNewTmpFinish                        ,


-- ** openReadwrite #method:openReadwrite#

#if defined(ENABLE_OVERLOADING)
    FileOpenReadwriteMethodInfo             ,
#endif
    fileOpenReadwrite                       ,


-- ** openReadwriteAsync #method:openReadwriteAsync#

#if defined(ENABLE_OVERLOADING)
    FileOpenReadwriteAsyncMethodInfo        ,
#endif
    fileOpenReadwriteAsync                  ,


-- ** openReadwriteFinish #method:openReadwriteFinish#

#if defined(ENABLE_OVERLOADING)
    FileOpenReadwriteFinishMethodInfo       ,
#endif
    fileOpenReadwriteFinish                 ,


-- ** parseName #method:parseName#

    fileParseName                           ,


-- ** peekPath #method:peekPath#

#if defined(ENABLE_OVERLOADING)
    FilePeekPathMethodInfo                  ,
#endif
    filePeekPath                            ,


-- ** pollMountable #method:pollMountable#

#if defined(ENABLE_OVERLOADING)
    FilePollMountableMethodInfo             ,
#endif
    filePollMountable                       ,


-- ** pollMountableFinish #method:pollMountableFinish#

#if defined(ENABLE_OVERLOADING)
    FilePollMountableFinishMethodInfo       ,
#endif
    filePollMountableFinish                 ,


-- ** queryDefaultHandler #method:queryDefaultHandler#

#if defined(ENABLE_OVERLOADING)
    FileQueryDefaultHandlerMethodInfo       ,
#endif
    fileQueryDefaultHandler                 ,


-- ** queryDefaultHandlerAsync #method:queryDefaultHandlerAsync#

#if defined(ENABLE_OVERLOADING)
    FileQueryDefaultHandlerAsyncMethodInfo  ,
#endif
    fileQueryDefaultHandlerAsync            ,


-- ** queryDefaultHandlerFinish #method:queryDefaultHandlerFinish#

#if defined(ENABLE_OVERLOADING)
    FileQueryDefaultHandlerFinishMethodInfo ,
#endif
    fileQueryDefaultHandlerFinish           ,


-- ** queryExists #method:queryExists#

#if defined(ENABLE_OVERLOADING)
    FileQueryExistsMethodInfo               ,
#endif
    fileQueryExists                         ,


-- ** queryFileType #method:queryFileType#

#if defined(ENABLE_OVERLOADING)
    FileQueryFileTypeMethodInfo             ,
#endif
    fileQueryFileType                       ,


-- ** queryFilesystemInfo #method:queryFilesystemInfo#

#if defined(ENABLE_OVERLOADING)
    FileQueryFilesystemInfoMethodInfo       ,
#endif
    fileQueryFilesystemInfo                 ,


-- ** queryFilesystemInfoAsync #method:queryFilesystemInfoAsync#

#if defined(ENABLE_OVERLOADING)
    FileQueryFilesystemInfoAsyncMethodInfo  ,
#endif
    fileQueryFilesystemInfoAsync            ,


-- ** queryFilesystemInfoFinish #method:queryFilesystemInfoFinish#

#if defined(ENABLE_OVERLOADING)
    FileQueryFilesystemInfoFinishMethodInfo ,
#endif
    fileQueryFilesystemInfoFinish           ,


-- ** queryInfo #method:queryInfo#

#if defined(ENABLE_OVERLOADING)
    FileQueryInfoMethodInfo                 ,
#endif
    fileQueryInfo                           ,


-- ** queryInfoAsync #method:queryInfoAsync#

#if defined(ENABLE_OVERLOADING)
    FileQueryInfoAsyncMethodInfo            ,
#endif
    fileQueryInfoAsync                      ,


-- ** queryInfoFinish #method:queryInfoFinish#

#if defined(ENABLE_OVERLOADING)
    FileQueryInfoFinishMethodInfo           ,
#endif
    fileQueryInfoFinish                     ,


-- ** querySettableAttributes #method:querySettableAttributes#

#if defined(ENABLE_OVERLOADING)
    FileQuerySettableAttributesMethodInfo   ,
#endif
    fileQuerySettableAttributes             ,


-- ** queryWritableNamespaces #method:queryWritableNamespaces#

#if defined(ENABLE_OVERLOADING)
    FileQueryWritableNamespacesMethodInfo   ,
#endif
    fileQueryWritableNamespaces             ,


-- ** read #method:read#

#if defined(ENABLE_OVERLOADING)
    FileReadMethodInfo                      ,
#endif
    fileRead                                ,


-- ** readAsync #method:readAsync#

#if defined(ENABLE_OVERLOADING)
    FileReadAsyncMethodInfo                 ,
#endif
    fileReadAsync                           ,


-- ** readFinish #method:readFinish#

#if defined(ENABLE_OVERLOADING)
    FileReadFinishMethodInfo                ,
#endif
    fileReadFinish                          ,


-- ** replace #method:replace#

#if defined(ENABLE_OVERLOADING)
    FileReplaceMethodInfo                   ,
#endif
    fileReplace                             ,


-- ** replaceAsync #method:replaceAsync#

#if defined(ENABLE_OVERLOADING)
    FileReplaceAsyncMethodInfo              ,
#endif
    fileReplaceAsync                        ,


-- ** replaceContents #method:replaceContents#

#if defined(ENABLE_OVERLOADING)
    FileReplaceContentsMethodInfo           ,
#endif
    fileReplaceContents                     ,


-- ** replaceContentsAsync #method:replaceContentsAsync#

#if defined(ENABLE_OVERLOADING)
    FileReplaceContentsAsyncMethodInfo      ,
#endif
    fileReplaceContentsAsync                ,


-- ** replaceContentsBytesAsync #method:replaceContentsBytesAsync#

#if defined(ENABLE_OVERLOADING)
    FileReplaceContentsBytesAsyncMethodInfo ,
#endif
    fileReplaceContentsBytesAsync           ,


-- ** replaceContentsFinish #method:replaceContentsFinish#

#if defined(ENABLE_OVERLOADING)
    FileReplaceContentsFinishMethodInfo     ,
#endif
    fileReplaceContentsFinish               ,


-- ** replaceFinish #method:replaceFinish#

#if defined(ENABLE_OVERLOADING)
    FileReplaceFinishMethodInfo             ,
#endif
    fileReplaceFinish                       ,


-- ** replaceReadwrite #method:replaceReadwrite#

#if defined(ENABLE_OVERLOADING)
    FileReplaceReadwriteMethodInfo          ,
#endif
    fileReplaceReadwrite                    ,


-- ** replaceReadwriteAsync #method:replaceReadwriteAsync#

#if defined(ENABLE_OVERLOADING)
    FileReplaceReadwriteAsyncMethodInfo     ,
#endif
    fileReplaceReadwriteAsync               ,


-- ** replaceReadwriteFinish #method:replaceReadwriteFinish#

#if defined(ENABLE_OVERLOADING)
    FileReplaceReadwriteFinishMethodInfo    ,
#endif
    fileReplaceReadwriteFinish              ,


-- ** resolveRelativePath #method:resolveRelativePath#

#if defined(ENABLE_OVERLOADING)
    FileResolveRelativePathMethodInfo       ,
#endif
    fileResolveRelativePath                 ,


-- ** setAttribute #method:setAttribute#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributeMethodInfo              ,
#endif
    fileSetAttribute                        ,


-- ** setAttributeByteString #method:setAttributeByteString#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributeByteStringMethodInfo    ,
#endif
    fileSetAttributeByteString              ,


-- ** setAttributeInt32 #method:setAttributeInt32#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributeInt32MethodInfo         ,
#endif
    fileSetAttributeInt32                   ,


-- ** setAttributeInt64 #method:setAttributeInt64#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributeInt64MethodInfo         ,
#endif
    fileSetAttributeInt64                   ,


-- ** setAttributeString #method:setAttributeString#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributeStringMethodInfo        ,
#endif
    fileSetAttributeString                  ,


-- ** setAttributeUint32 #method:setAttributeUint32#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributeUint32MethodInfo        ,
#endif
    fileSetAttributeUint32                  ,


-- ** setAttributeUint64 #method:setAttributeUint64#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributeUint64MethodInfo        ,
#endif
    fileSetAttributeUint64                  ,


-- ** setAttributesAsync #method:setAttributesAsync#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributesAsyncMethodInfo        ,
#endif
    fileSetAttributesAsync                  ,


-- ** setAttributesFinish #method:setAttributesFinish#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributesFinishMethodInfo       ,
#endif
    fileSetAttributesFinish                 ,


-- ** setAttributesFromInfo #method:setAttributesFromInfo#

#if defined(ENABLE_OVERLOADING)
    FileSetAttributesFromInfoMethodInfo     ,
#endif
    fileSetAttributesFromInfo               ,


-- ** setDisplayName #method:setDisplayName#

#if defined(ENABLE_OVERLOADING)
    FileSetDisplayNameMethodInfo            ,
#endif
    fileSetDisplayName                      ,


-- ** setDisplayNameAsync #method:setDisplayNameAsync#

#if defined(ENABLE_OVERLOADING)
    FileSetDisplayNameAsyncMethodInfo       ,
#endif
    fileSetDisplayNameAsync                 ,


-- ** setDisplayNameFinish #method:setDisplayNameFinish#

#if defined(ENABLE_OVERLOADING)
    FileSetDisplayNameFinishMethodInfo      ,
#endif
    fileSetDisplayNameFinish                ,


-- ** startMountable #method:startMountable#

#if defined(ENABLE_OVERLOADING)
    FileStartMountableMethodInfo            ,
#endif
    fileStartMountable                      ,


-- ** startMountableFinish #method:startMountableFinish#

#if defined(ENABLE_OVERLOADING)
    FileStartMountableFinishMethodInfo      ,
#endif
    fileStartMountableFinish                ,


-- ** stopMountable #method:stopMountable#

#if defined(ENABLE_OVERLOADING)
    FileStopMountableMethodInfo             ,
#endif
    fileStopMountable                       ,


-- ** stopMountableFinish #method:stopMountableFinish#

#if defined(ENABLE_OVERLOADING)
    FileStopMountableFinishMethodInfo       ,
#endif
    fileStopMountableFinish                 ,


-- ** supportsThreadContexts #method:supportsThreadContexts#

#if defined(ENABLE_OVERLOADING)
    FileSupportsThreadContextsMethodInfo    ,
#endif
    fileSupportsThreadContexts              ,


-- ** trash #method:trash#

#if defined(ENABLE_OVERLOADING)
    FileTrashMethodInfo                     ,
#endif
    fileTrash                               ,


-- ** trashAsync #method:trashAsync#

#if defined(ENABLE_OVERLOADING)
    FileTrashAsyncMethodInfo                ,
#endif
    fileTrashAsync                          ,


-- ** trashFinish #method:trashFinish#

#if defined(ENABLE_OVERLOADING)
    FileTrashFinishMethodInfo               ,
#endif
    fileTrashFinish                         ,


-- ** unmountMountable #method:unmountMountable#

#if defined(ENABLE_OVERLOADING)
    FileUnmountMountableMethodInfo          ,
#endif
    fileUnmountMountable                    ,


-- ** unmountMountableFinish #method:unmountMountableFinish#

#if defined(ENABLE_OVERLOADING)
    FileUnmountMountableFinishMethodInfo    ,
#endif
    fileUnmountMountableFinish              ,


-- ** unmountMountableWithOperation #method:unmountMountableWithOperation#

#if defined(ENABLE_OVERLOADING)
    FileUnmountMountableWithOperationMethodInfo,
#endif
    fileUnmountMountableWithOperation       ,


-- ** unmountMountableWithOperationFinish #method:unmountMountableWithOperationFinish#

#if defined(ENABLE_OVERLOADING)
    FileUnmountMountableWithOperationFinishMethodInfo,
#endif
    fileUnmountMountableWithOperationFinish ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Enums as GLib.Enums
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.DateTime as GLib.DateTime
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GLib.Structs.TimeVal as GLib.TimeVal
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AppInfo as Gio.AppInfo
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Drive as Gio.Drive
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Icon as Gio.Icon
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Mount as Gio.Mount
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Seekable as Gio.Seekable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Volume as Gio.Volume
import {-# SOURCE #-} qualified GI.Gio.Objects.AppLaunchContext as Gio.AppLaunchContext
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.FileEnumerator as Gio.FileEnumerator
import {-# SOURCE #-} qualified GI.Gio.Objects.FileIOStream as Gio.FileIOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInfo as Gio.FileInfo
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInputStream as Gio.FileInputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileMonitor as Gio.FileMonitor
import {-# SOURCE #-} qualified GI.Gio.Objects.FileOutputStream as Gio.FileOutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MountOperation as Gio.MountOperation
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfo as Gio.FileAttributeInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfoList as Gio.FileAttributeInfoList
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeMatcher as Gio.FileAttributeMatcher
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AppInfo as Gio.AppInfo
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Mount as Gio.Mount
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.FileEnumerator as Gio.FileEnumerator
import {-# SOURCE #-} qualified GI.Gio.Objects.FileIOStream as Gio.FileIOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInfo as Gio.FileInfo
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInputStream as Gio.FileInputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileMonitor as Gio.FileMonitor
import {-# SOURCE #-} qualified GI.Gio.Objects.FileOutputStream as Gio.FileOutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MountOperation as Gio.MountOperation
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfoList as Gio.FileAttributeInfoList

#endif

-- interface File 
-- | Memory-managed wrapper type.
newtype File = File (SP.ManagedPtr File)
    deriving (Eq)

instance SP.ManagedPtrNewtype File where
    toManagedPtr (File p) = p

foreign import ccall "g_file_get_type"
    c_g_file_get_type :: IO B.Types.GType

instance B.Types.TypedObject File where
    glibType = c_g_file_get_type

instance B.Types.GObject File

-- | Type class for types which can be safely cast to `File`, for instance with `toFile`.
class (SP.GObject o, O.IsDescendantOf File o) => IsFile o
instance (SP.GObject o, O.IsDescendantOf File o) => IsFile o

instance O.HasParentTypes File
type instance O.ParentTypes File = '[GObject.Object.Object]

-- | Cast to `File`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toFile :: (MIO.MonadIO m, IsFile o) => o -> m File
toFile = MIO.liftIO . B.ManagedPtr.unsafeCastTo File

-- | Convert 'File' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe File) where
    gvalueGType_ = c_g_file_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr File)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr File)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject File ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList File
type instance O.AttributeList File = FileAttributeList
type FileAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveFileMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveFileMethod "appendTo" o = FileAppendToMethodInfo
    ResolveFileMethod "appendToAsync" o = FileAppendToAsyncMethodInfo
    ResolveFileMethod "appendToFinish" o = FileAppendToFinishMethodInfo
    ResolveFileMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveFileMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveFileMethod "buildAttributeListForCopy" o = FileBuildAttributeListForCopyMethodInfo
    ResolveFileMethod "copy" o = FileCopyMethodInfo
    ResolveFileMethod "copyAsync" o = FileCopyAsyncMethodInfo
    ResolveFileMethod "copyAttributes" o = FileCopyAttributesMethodInfo
    ResolveFileMethod "copyFinish" o = FileCopyFinishMethodInfo
    ResolveFileMethod "create" o = FileCreateMethodInfo
    ResolveFileMethod "createAsync" o = FileCreateAsyncMethodInfo
    ResolveFileMethod "createFinish" o = FileCreateFinishMethodInfo
    ResolveFileMethod "createReadwrite" o = FileCreateReadwriteMethodInfo
    ResolveFileMethod "createReadwriteAsync" o = FileCreateReadwriteAsyncMethodInfo
    ResolveFileMethod "createReadwriteFinish" o = FileCreateReadwriteFinishMethodInfo
    ResolveFileMethod "delete" o = FileDeleteMethodInfo
    ResolveFileMethod "deleteAsync" o = FileDeleteAsyncMethodInfo
    ResolveFileMethod "deleteFinish" o = FileDeleteFinishMethodInfo
    ResolveFileMethod "dup" o = FileDupMethodInfo
    ResolveFileMethod "ejectMountable" o = FileEjectMountableMethodInfo
    ResolveFileMethod "ejectMountableFinish" o = FileEjectMountableFinishMethodInfo
    ResolveFileMethod "ejectMountableWithOperation" o = FileEjectMountableWithOperationMethodInfo
    ResolveFileMethod "ejectMountableWithOperationFinish" o = FileEjectMountableWithOperationFinishMethodInfo
    ResolveFileMethod "enumerateChildren" o = FileEnumerateChildrenMethodInfo
    ResolveFileMethod "enumerateChildrenAsync" o = FileEnumerateChildrenAsyncMethodInfo
    ResolveFileMethod "enumerateChildrenFinish" o = FileEnumerateChildrenFinishMethodInfo
    ResolveFileMethod "equal" o = FileEqualMethodInfo
    ResolveFileMethod "findEnclosingMount" o = FileFindEnclosingMountMethodInfo
    ResolveFileMethod "findEnclosingMountAsync" o = FileFindEnclosingMountAsyncMethodInfo
    ResolveFileMethod "findEnclosingMountFinish" o = FileFindEnclosingMountFinishMethodInfo
    ResolveFileMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveFileMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveFileMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveFileMethod "hasParent" o = FileHasParentMethodInfo
    ResolveFileMethod "hasPrefix" o = FileHasPrefixMethodInfo
    ResolveFileMethod "hasUriScheme" o = FileHasUriSchemeMethodInfo
    ResolveFileMethod "hash" o = FileHashMethodInfo
    ResolveFileMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveFileMethod "isNative" o = FileIsNativeMethodInfo
    ResolveFileMethod "loadBytes" o = FileLoadBytesMethodInfo
    ResolveFileMethod "loadBytesAsync" o = FileLoadBytesAsyncMethodInfo
    ResolveFileMethod "loadBytesFinish" o = FileLoadBytesFinishMethodInfo
    ResolveFileMethod "loadContents" o = FileLoadContentsMethodInfo
    ResolveFileMethod "loadContentsAsync" o = FileLoadContentsAsyncMethodInfo
    ResolveFileMethod "loadContentsFinish" o = FileLoadContentsFinishMethodInfo
    ResolveFileMethod "loadPartialContentsFinish" o = FileLoadPartialContentsFinishMethodInfo
    ResolveFileMethod "makeDirectory" o = FileMakeDirectoryMethodInfo
    ResolveFileMethod "makeDirectoryAsync" o = FileMakeDirectoryAsyncMethodInfo
    ResolveFileMethod "makeDirectoryFinish" o = FileMakeDirectoryFinishMethodInfo
    ResolveFileMethod "makeDirectoryWithParents" o = FileMakeDirectoryWithParentsMethodInfo
    ResolveFileMethod "makeSymbolicLink" o = FileMakeSymbolicLinkMethodInfo
    ResolveFileMethod "makeSymbolicLinkAsync" o = FileMakeSymbolicLinkAsyncMethodInfo
    ResolveFileMethod "makeSymbolicLinkFinish" o = FileMakeSymbolicLinkFinishMethodInfo
    ResolveFileMethod "measureDiskUsage" o = FileMeasureDiskUsageMethodInfo
    ResolveFileMethod "measureDiskUsageFinish" o = FileMeasureDiskUsageFinishMethodInfo
    ResolveFileMethod "monitor" o = FileMonitorMethodInfo
    ResolveFileMethod "monitorDirectory" o = FileMonitorDirectoryMethodInfo
    ResolveFileMethod "monitorFile" o = FileMonitorFileMethodInfo
    ResolveFileMethod "mountEnclosingVolume" o = FileMountEnclosingVolumeMethodInfo
    ResolveFileMethod "mountEnclosingVolumeFinish" o = FileMountEnclosingVolumeFinishMethodInfo
    ResolveFileMethod "mountMountable" o = FileMountMountableMethodInfo
    ResolveFileMethod "mountMountableFinish" o = FileMountMountableFinishMethodInfo
    ResolveFileMethod "move" o = FileMoveMethodInfo
    ResolveFileMethod "moveAsync" o = FileMoveAsyncMethodInfo
    ResolveFileMethod "moveFinish" o = FileMoveFinishMethodInfo
    ResolveFileMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveFileMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveFileMethod "openReadwrite" o = FileOpenReadwriteMethodInfo
    ResolveFileMethod "openReadwriteAsync" o = FileOpenReadwriteAsyncMethodInfo
    ResolveFileMethod "openReadwriteFinish" o = FileOpenReadwriteFinishMethodInfo
    ResolveFileMethod "peekPath" o = FilePeekPathMethodInfo
    ResolveFileMethod "pollMountable" o = FilePollMountableMethodInfo
    ResolveFileMethod "pollMountableFinish" o = FilePollMountableFinishMethodInfo
    ResolveFileMethod "queryDefaultHandler" o = FileQueryDefaultHandlerMethodInfo
    ResolveFileMethod "queryDefaultHandlerAsync" o = FileQueryDefaultHandlerAsyncMethodInfo
    ResolveFileMethod "queryDefaultHandlerFinish" o = FileQueryDefaultHandlerFinishMethodInfo
    ResolveFileMethod "queryExists" o = FileQueryExistsMethodInfo
    ResolveFileMethod "queryFileType" o = FileQueryFileTypeMethodInfo
    ResolveFileMethod "queryFilesystemInfo" o = FileQueryFilesystemInfoMethodInfo
    ResolveFileMethod "queryFilesystemInfoAsync" o = FileQueryFilesystemInfoAsyncMethodInfo
    ResolveFileMethod "queryFilesystemInfoFinish" o = FileQueryFilesystemInfoFinishMethodInfo
    ResolveFileMethod "queryInfo" o = FileQueryInfoMethodInfo
    ResolveFileMethod "queryInfoAsync" o = FileQueryInfoAsyncMethodInfo
    ResolveFileMethod "queryInfoFinish" o = FileQueryInfoFinishMethodInfo
    ResolveFileMethod "querySettableAttributes" o = FileQuerySettableAttributesMethodInfo
    ResolveFileMethod "queryWritableNamespaces" o = FileQueryWritableNamespacesMethodInfo
    ResolveFileMethod "read" o = FileReadMethodInfo
    ResolveFileMethod "readAsync" o = FileReadAsyncMethodInfo
    ResolveFileMethod "readFinish" o = FileReadFinishMethodInfo
    ResolveFileMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveFileMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveFileMethod "replace" o = FileReplaceMethodInfo
    ResolveFileMethod "replaceAsync" o = FileReplaceAsyncMethodInfo
    ResolveFileMethod "replaceContents" o = FileReplaceContentsMethodInfo
    ResolveFileMethod "replaceContentsAsync" o = FileReplaceContentsAsyncMethodInfo
    ResolveFileMethod "replaceContentsBytesAsync" o = FileReplaceContentsBytesAsyncMethodInfo
    ResolveFileMethod "replaceContentsFinish" o = FileReplaceContentsFinishMethodInfo
    ResolveFileMethod "replaceFinish" o = FileReplaceFinishMethodInfo
    ResolveFileMethod "replaceReadwrite" o = FileReplaceReadwriteMethodInfo
    ResolveFileMethod "replaceReadwriteAsync" o = FileReplaceReadwriteAsyncMethodInfo
    ResolveFileMethod "replaceReadwriteFinish" o = FileReplaceReadwriteFinishMethodInfo
    ResolveFileMethod "resolveRelativePath" o = FileResolveRelativePathMethodInfo
    ResolveFileMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveFileMethod "startMountable" o = FileStartMountableMethodInfo
    ResolveFileMethod "startMountableFinish" o = FileStartMountableFinishMethodInfo
    ResolveFileMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveFileMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveFileMethod "stopMountable" o = FileStopMountableMethodInfo
    ResolveFileMethod "stopMountableFinish" o = FileStopMountableFinishMethodInfo
    ResolveFileMethod "supportsThreadContexts" o = FileSupportsThreadContextsMethodInfo
    ResolveFileMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveFileMethod "trash" o = FileTrashMethodInfo
    ResolveFileMethod "trashAsync" o = FileTrashAsyncMethodInfo
    ResolveFileMethod "trashFinish" o = FileTrashFinishMethodInfo
    ResolveFileMethod "unmountMountable" o = FileUnmountMountableMethodInfo
    ResolveFileMethod "unmountMountableFinish" o = FileUnmountMountableFinishMethodInfo
    ResolveFileMethod "unmountMountableWithOperation" o = FileUnmountMountableWithOperationMethodInfo
    ResolveFileMethod "unmountMountableWithOperationFinish" o = FileUnmountMountableWithOperationFinishMethodInfo
    ResolveFileMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveFileMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveFileMethod "getBasename" o = FileGetBasenameMethodInfo
    ResolveFileMethod "getChild" o = FileGetChildMethodInfo
    ResolveFileMethod "getChildForDisplayName" o = FileGetChildForDisplayNameMethodInfo
    ResolveFileMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveFileMethod "getParent" o = FileGetParentMethodInfo
    ResolveFileMethod "getParseName" o = FileGetParseNameMethodInfo
    ResolveFileMethod "getPath" o = FileGetPathMethodInfo
    ResolveFileMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveFileMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveFileMethod "getRelativePath" o = FileGetRelativePathMethodInfo
    ResolveFileMethod "getUri" o = FileGetUriMethodInfo
    ResolveFileMethod "getUriScheme" o = FileGetUriSchemeMethodInfo
    ResolveFileMethod "setAttribute" o = FileSetAttributeMethodInfo
    ResolveFileMethod "setAttributeByteString" o = FileSetAttributeByteStringMethodInfo
    ResolveFileMethod "setAttributeInt32" o = FileSetAttributeInt32MethodInfo
    ResolveFileMethod "setAttributeInt64" o = FileSetAttributeInt64MethodInfo
    ResolveFileMethod "setAttributeString" o = FileSetAttributeStringMethodInfo
    ResolveFileMethod "setAttributeUint32" o = FileSetAttributeUint32MethodInfo
    ResolveFileMethod "setAttributeUint64" o = FileSetAttributeUint64MethodInfo
    ResolveFileMethod "setAttributesAsync" o = FileSetAttributesAsyncMethodInfo
    ResolveFileMethod "setAttributesFinish" o = FileSetAttributesFinishMethodInfo
    ResolveFileMethod "setAttributesFromInfo" o = FileSetAttributesFromInfoMethodInfo
    ResolveFileMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveFileMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveFileMethod "setDisplayName" o = FileSetDisplayNameMethodInfo
    ResolveFileMethod "setDisplayNameAsync" o = FileSetDisplayNameAsyncMethodInfo
    ResolveFileMethod "setDisplayNameFinish" o = FileSetDisplayNameFinishMethodInfo
    ResolveFileMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveFileMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveFileMethod t File, O.OverloadedMethod info File p) => OL.IsLabel t (File -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveFileMethod t File, O.OverloadedMethod info File p, R.HasField t File p) => R.HasField t File p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveFileMethod t File, O.OverloadedMethodInfo info File) => OL.IsLabel t (O.MethodProxy info File) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method File::append_to
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileOutputStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_append_to" g_file_append_to :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileOutputStream.FileOutputStream)

-- | Gets an output stream for appending data to the file.
-- If the file doesn\'t already exist it is created.
-- 
-- By default files created are generally readable by everyone,
-- but if you pass 'GI.Gio.Flags.FileCreateFlagsPrivate' in /@flags@/ the file
-- will be made readable only to the current user, to the level that
-- is supported on the target filesystem.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled
-- by triggering the cancellable object from another thread. If the
-- operation was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be
-- returned.
-- 
-- Some file systems don\'t allow all file names, and may return an
-- 'GI.Gio.Enums.IOErrorEnumInvalidFilename' error. If the file is a directory the
-- 'GI.Gio.Enums.IOErrorEnumIsDirectory' error will be returned. Other errors are
-- possible too, and depend on what kind of filesystem the file is on.
fileAppendTo ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileOutputStream.FileOutputStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileOutputStream.FileOutputStream', or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileAppendTo file flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_append_to file' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileAppendTo" result
        result' <- (wrapObject Gio.FileOutputStream.FileOutputStream) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileAppendToMethodInfo
instance (signature ~ ([Gio.Flags.FileCreateFlags] -> Maybe (b) -> m Gio.FileOutputStream.FileOutputStream), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileAppendToMethodInfo a signature where
    overloadedMethod = fileAppendTo

instance O.OverloadedMethodInfo FileAppendToMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileAppendTo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileAppendTo"
        })


#endif

-- method File::append_to_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_append_to_async" g_file_append_to_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously opens /@file@/ for appending.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileAppendTo' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileAppendToFinish' to get the result
-- of the operation.
fileAppendToAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileAppendToAsync file flags ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_append_to_async file' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileAppendToAsyncMethodInfo
instance (signature ~ ([Gio.Flags.FileCreateFlags] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileAppendToAsyncMethodInfo a signature where
    overloadedMethod = fileAppendToAsync

instance O.OverloadedMethodInfo FileAppendToAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileAppendToAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileAppendToAsync"
        })


#endif

-- method File::append_to_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileOutputStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_append_to_finish" g_file_append_to_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileOutputStream.FileOutputStream)

-- | Finishes an asynchronous file append operation started with
-- 'GI.Gio.Interfaces.File.fileAppendToAsync'.
fileAppendToFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileOutputStream.FileOutputStream
    -- ^ __Returns:__ a valid t'GI.Gio.Objects.FileOutputStream.FileOutputStream'
    --   or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileAppendToFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_append_to_finish file' res'
        checkUnexpectedReturnNULL "fileAppendToFinish" result
        result' <- (wrapObject Gio.FileOutputStream.FileOutputStream) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileAppendToFinishMethodInfo
instance (signature ~ (b -> m Gio.FileOutputStream.FileOutputStream), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileAppendToFinishMethodInfo a signature where
    overloadedMethod = fileAppendToFinish

instance O.OverloadedMethodInfo FileAppendToFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileAppendToFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileAppendToFinish"
        })


#endif

-- method File::build_attribute_list_for_copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile to copy attributes to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCopyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCopyFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_build_attribute_list_for_copy" g_file_build_attribute_list_for_copy :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCopyFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Prepares the file attribute query string for copying to /@file@/.
-- 
-- This function prepares an attribute query string to be
-- passed to 'GI.Gio.Interfaces.File.fileQueryInfo' to get a list of attributes
-- normally copied with the file (see 'GI.Gio.Interfaces.File.fileCopyAttributes'
-- for the detailed description). This function is used by the
-- implementation of 'GI.Gio.Interfaces.File.fileCopyAttributes' and is useful
-- when one needs to query and set the attributes in two
-- stages (e.g., for recursive move of a directory).
-- 
-- /Since: 2.68/
fileBuildAttributeListForCopy ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File' to copy attributes to
    -> [Gio.Flags.FileCopyFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCopyFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m T.Text
    -- ^ __Returns:__ an attribute query string for 'GI.Gio.Interfaces.File.fileQueryInfo',
    --   or 'P.Nothing' if an error occurs. /(Can throw 'Data.GI.Base.GError.GError')/
fileBuildAttributeListForCopy file flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_build_attribute_list_for_copy file' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileBuildAttributeListForCopy" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileBuildAttributeListForCopyMethodInfo
instance (signature ~ ([Gio.Flags.FileCopyFlags] -> Maybe (b) -> m T.Text), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileBuildAttributeListForCopyMethodInfo a signature where
    overloadedMethod = fileBuildAttributeListForCopy

instance O.OverloadedMethodInfo FileBuildAttributeListForCopyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileBuildAttributeListForCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileBuildAttributeListForCopy"
        })


#endif

-- method File::copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "source"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "destination"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "destination #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCopyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "set of #GFileCopyFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "progress_callback"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "FileProgressCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "function to callback with\n  progress information, or %NULL if progress information is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeCall
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "progress_callback_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data to pass to @progress_callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_copy" g_file_copy :: 
    Ptr File ->                             -- source : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- destination : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCopyFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_FileProgressCallback -> -- progress_callback : TInterface (Name {namespace = "Gio", name = "FileProgressCallback"})
    Ptr () ->                               -- progress_callback_data : TBasicType TPtr
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Copies the file /@source@/ to the location specified by /@destination@/.
-- Can not handle recursive copies of directories.
-- 
-- If the flag 'GI.Gio.Flags.FileCopyFlagsOverwrite' is specified an already
-- existing /@destination@/ file is overwritten.
-- 
-- If the flag 'GI.Gio.Flags.FileCopyFlagsNofollowSymlinks' is specified then symlinks
-- will be copied as symlinks, otherwise the target of the
-- /@source@/ symlink will be copied.
-- 
-- If the flag 'GI.Gio.Flags.FileCopyFlagsAllMetadata' is specified then all the metadata
-- that is possible to copy is copied, not just the default subset (which,
-- for instance, does not include the owner, see t'GI.Gio.Objects.FileInfo.FileInfo').
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- If /@progressCallback@/ is not 'P.Nothing', then the operation can be monitored
-- by setting this to a t'GI.Gio.Callbacks.FileProgressCallback' function.
-- /@progressCallbackData@/ will be passed to this function. It is guaranteed
-- that this callback will be called after all data has been transferred with
-- the total number of bytes copied during the operation.
-- 
-- If the /@source@/ file does not exist, then the 'GI.Gio.Enums.IOErrorEnumNotFound' error
-- is returned, independent on the status of the /@destination@/.
-- 
-- If 'GI.Gio.Flags.FileCopyFlagsOverwrite' is not specified and the target exists, then
-- the error 'GI.Gio.Enums.IOErrorEnumExists' is returned.
-- 
-- If trying to overwrite a file over a directory, the 'GI.Gio.Enums.IOErrorEnumIsDirectory'
-- error is returned. If trying to overwrite a directory with a directory the
-- 'GI.Gio.Enums.IOErrorEnumWouldMerge' error is returned.
-- 
-- If the source is a directory and the target does not exist, or
-- 'GI.Gio.Flags.FileCopyFlagsOverwrite' is specified and the target is a file, then the
-- 'GI.Gio.Enums.IOErrorEnumWouldRecurse' error is returned.
-- 
-- If you are interested in copying the t'GI.Gio.Interfaces.File.File' object itself (not the on-disk
-- file), see 'GI.Gio.Interfaces.File.fileDup'.
fileCopy ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@source@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@destination@/: destination t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileCopyFlags]
    -- ^ /@flags@/: set of t'GI.Gio.Flags.FileCopyFlags'
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.FileProgressCallback)
    -- ^ /@progressCallback@/: function to callback with
    --   progress information, or 'P.Nothing' if progress information is not needed
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileCopy source destination flags cancellable progressCallback = liftIO $ do
    source' <- unsafeManagedPtrCastPtr source
    destination' <- unsafeManagedPtrCastPtr destination
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeProgressCallback <- case progressCallback of
        Nothing -> return FP.nullFunPtr
        Just jProgressCallback -> do
            jProgressCallback' <- Gio.Callbacks.mk_FileProgressCallback (Gio.Callbacks.wrap_FileProgressCallback Nothing (Gio.Callbacks.drop_closures_FileProgressCallback jProgressCallback))
            return jProgressCallback'
    let progressCallbackData = nullPtr
    onException (do
        _ <- propagateGError $ g_file_copy source' destination' flags' maybeCancellable maybeProgressCallback progressCallbackData
        safeFreeFunPtr $ castFunPtrToPtr maybeProgressCallback
        touchManagedPtr source
        touchManagedPtr destination
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        safeFreeFunPtr $ castFunPtrToPtr maybeProgressCallback
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileCopyMethodInfo
instance (signature ~ (b -> [Gio.Flags.FileCopyFlags] -> Maybe (c) -> Maybe (Gio.Callbacks.FileProgressCallback) -> m ()), MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileCopyMethodInfo a signature where
    overloadedMethod = fileCopy

instance O.OverloadedMethodInfo FileCopyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCopy"
        })


#endif

-- method File::copy_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "source"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input [type@Gio.File]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "destination"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "destination [type@Gio.File]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCopyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "set of [flags@Gio.FileCopyFlags]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional [class@Gio.Cancellable] object,\n  `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "progress_callback_closure"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "[type@GObject.Closure] to invoke with progress\n  information, or `NULL` if progress information is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "ready_callback_closure"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "[type@GObject.Closure] to invoke when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_copy_async_with_closures" g_file_copy_async_with_closures :: 
    Ptr File ->                             -- source : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- destination : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCopyFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (GClosure ()) ->                    -- progress_callback_closure : TGClosure Nothing
    Ptr (GClosure ()) ->                    -- ready_callback_closure : TGClosure Nothing
    IO ()

-- | Version of 'GI.Gio.Interfaces.File.fileCopyAsync' using closures instead of callbacks for
-- easier binding in other languages.
-- 
-- /Since: 2.82/
fileCopyAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@source@/: input [type/@gio@/.File]
    -> b
    -- ^ /@destination@/: destination [type/@gio@/.File]
    -> [Gio.Flags.FileCopyFlags]
    -- ^ /@flags@/: set of [flags/@gio@/.FileCopyFlags]
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   @NULL@ to ignore
    -> Maybe (GClosure d)
    -- ^ /@progressCallbackClosure@/: [type/@gObject@/.Closure] to invoke with progress
    --   information, or @NULL@ if progress information is not needed
    -> GClosure e
    -- ^ /@readyCallbackClosure@/: [type/@gObject@/.Closure] to invoke when the request is satisfied
    -> m ()
fileCopyAsync source destination flags ioPriority cancellable progressCallbackClosure readyCallbackClosure = liftIO $ do
    source' <- unsafeManagedPtrCastPtr source
    destination' <- unsafeManagedPtrCastPtr destination
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeProgressCallbackClosure <- case progressCallbackClosure of
        Nothing -> return FP.nullPtr
        Just jProgressCallbackClosure -> do
            jProgressCallbackClosure' <- unsafeManagedPtrCastPtr jProgressCallbackClosure
            return jProgressCallbackClosure'
    readyCallbackClosure' <- unsafeManagedPtrCastPtr readyCallbackClosure
    g_file_copy_async_with_closures source' destination' flags' ioPriority maybeCancellable maybeProgressCallbackClosure readyCallbackClosure'
    touchManagedPtr source
    touchManagedPtr destination
    whenJust cancellable touchManagedPtr
    whenJust progressCallbackClosure touchManagedPtr
    touchManagedPtr readyCallbackClosure
    return ()

#if defined(ENABLE_OVERLOADING)
data FileCopyAsyncMethodInfo
instance (signature ~ (b -> [Gio.Flags.FileCopyFlags] -> Int32 -> Maybe (c) -> Maybe (GClosure d) -> GClosure e -> m ()), MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileCopyAsyncMethodInfo a signature where
    overloadedMethod = fileCopyAsync

instance O.OverloadedMethodInfo FileCopyAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCopyAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCopyAsync"
        })


#endif

-- method File::copy_attributes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "source"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile with attributes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "destination"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile to copy attributes to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCopyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCopyFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_copy_attributes" g_file_copy_attributes :: 
    Ptr File ->                             -- source : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- destination : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCopyFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Copies the file attributes from /@source@/ to /@destination@/.
-- 
-- Normally only a subset of the file attributes are copied,
-- those that are copies in a normal file copy operation
-- (which for instance does not include e.g. owner). However
-- if 'GI.Gio.Flags.FileCopyFlagsAllMetadata' is specified in /@flags@/, then
-- all the metadata that is possible to copy is copied. This
-- is useful when implementing move by copy + delete source.
fileCopyAttributes ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@source@/: a t'GI.Gio.Interfaces.File.File' with attributes
    -> b
    -- ^ /@destination@/: a t'GI.Gio.Interfaces.File.File' to copy attributes to
    -> [Gio.Flags.FileCopyFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCopyFlags'
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileCopyAttributes source destination flags cancellable = liftIO $ do
    source' <- unsafeManagedPtrCastPtr source
    destination' <- unsafeManagedPtrCastPtr destination
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_copy_attributes source' destination' flags' maybeCancellable
        touchManagedPtr source
        touchManagedPtr destination
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileCopyAttributesMethodInfo
instance (signature ~ (b -> [Gio.Flags.FileCopyFlags] -> Maybe (c) -> m ()), MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileCopyAttributesMethodInfo a signature where
    overloadedMethod = fileCopyAttributes

instance O.OverloadedMethodInfo FileCopyAttributesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCopyAttributes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCopyAttributes"
        })


#endif

-- method File::copy_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_copy_finish" g_file_copy_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes copying the file started with @/g_file_copy_async()/@.
fileCopyFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileCopyFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        _ <- propagateGError $ g_file_copy_finish file' res'
        touchManagedPtr file
        touchManagedPtr res
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileCopyFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileCopyFinishMethodInfo a signature where
    overloadedMethod = fileCopyFinish

instance O.OverloadedMethodInfo FileCopyFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCopyFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCopyFinish"
        })


#endif

-- method File::create
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileOutputStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_create" g_file_create :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileOutputStream.FileOutputStream)

-- | Creates a new file and returns an output stream for writing to it.
-- The file must not already exist.
-- 
-- By default files created are generally readable by everyone,
-- but if you pass 'GI.Gio.Flags.FileCreateFlagsPrivate' in /@flags@/ the file
-- will be made readable only to the current user, to the level
-- that is supported on the target filesystem.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled
-- by triggering the cancellable object from another thread. If the
-- operation was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be
-- returned.
-- 
-- If a file or directory with this name already exists the
-- 'GI.Gio.Enums.IOErrorEnumExists' error will be returned. Some file systems don\'t
-- allow all file names, and may return an 'GI.Gio.Enums.IOErrorEnumInvalidFilename'
-- error, and if the name is to long 'GI.Gio.Enums.IOErrorEnumFilenameTooLong' will
-- be returned. Other errors are possible too, and depend on what kind
-- of filesystem the file is on.
fileCreate ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileOutputStream.FileOutputStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileOutputStream.FileOutputStream' for the newly created
    --   file, or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileCreate file flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_create file' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileCreate" result
        result' <- (wrapObject Gio.FileOutputStream.FileOutputStream) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileCreateMethodInfo
instance (signature ~ ([Gio.Flags.FileCreateFlags] -> Maybe (b) -> m Gio.FileOutputStream.FileOutputStream), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileCreateMethodInfo a signature where
    overloadedMethod = fileCreate

instance O.OverloadedMethodInfo FileCreateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCreate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCreate"
        })


#endif

-- method File::create_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_create_async" g_file_create_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously creates a new file and returns an output stream
-- for writing to it. The file must not already exist.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileCreate' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileCreateFinish' to get the result
-- of the operation.
fileCreateAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileCreateAsync file flags ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_create_async file' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileCreateAsyncMethodInfo
instance (signature ~ ([Gio.Flags.FileCreateFlags] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileCreateAsyncMethodInfo a signature where
    overloadedMethod = fileCreateAsync

instance O.OverloadedMethodInfo FileCreateAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCreateAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCreateAsync"
        })


#endif

-- method File::create_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileOutputStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_create_finish" g_file_create_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileOutputStream.FileOutputStream)

-- | Finishes an asynchronous file create operation started with
-- 'GI.Gio.Interfaces.File.fileCreateAsync'.
fileCreateFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileOutputStream.FileOutputStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileOutputStream.FileOutputStream' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileCreateFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_create_finish file' res'
        checkUnexpectedReturnNULL "fileCreateFinish" result
        result' <- (wrapObject Gio.FileOutputStream.FileOutputStream) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileCreateFinishMethodInfo
instance (signature ~ (b -> m Gio.FileOutputStream.FileOutputStream), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileCreateFinishMethodInfo a signature where
    overloadedMethod = fileCreateFinish

instance O.OverloadedMethodInfo FileCreateFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCreateFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCreateFinish"
        })


#endif

-- method File::create_readwrite
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileIOStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_create_readwrite" g_file_create_readwrite :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileIOStream.FileIOStream)

-- | Creates a new file and returns a stream for reading and
-- writing to it. The file must not already exist.
-- 
-- By default files created are generally readable by everyone,
-- but if you pass 'GI.Gio.Flags.FileCreateFlagsPrivate' in /@flags@/ the file
-- will be made readable only to the current user, to the level
-- that is supported on the target filesystem.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled
-- by triggering the cancellable object from another thread. If the
-- operation was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be
-- returned.
-- 
-- If a file or directory with this name already exists, the
-- 'GI.Gio.Enums.IOErrorEnumExists' error will be returned. Some file systems don\'t
-- allow all file names, and may return an 'GI.Gio.Enums.IOErrorEnumInvalidFilename'
-- error, and if the name is too long, 'GI.Gio.Enums.IOErrorEnumFilenameTooLong'
-- will be returned. Other errors are possible too, and depend on what
-- kind of filesystem the file is on.
-- 
-- Note that in many non-local file cases read and write streams are
-- not supported, so make sure you really need to do read and write
-- streaming, rather than just opening for reading or writing.
-- 
-- /Since: 2.22/
fileCreateReadwrite ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileIOStream.FileIOStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileIOStream.FileIOStream' for the newly created
    --   file, or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileCreateReadwrite file flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_create_readwrite file' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileCreateReadwrite" result
        result' <- (wrapObject Gio.FileIOStream.FileIOStream) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileCreateReadwriteMethodInfo
instance (signature ~ ([Gio.Flags.FileCreateFlags] -> Maybe (b) -> m Gio.FileIOStream.FileIOStream), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileCreateReadwriteMethodInfo a signature where
    overloadedMethod = fileCreateReadwrite

instance O.OverloadedMethodInfo FileCreateReadwriteMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCreateReadwrite",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCreateReadwrite"
        })


#endif

-- method File::create_readwrite_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_create_readwrite_async" g_file_create_readwrite_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously creates a new file and returns a stream
-- for reading and writing to it. The file must not already exist.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileCreateReadwrite' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileCreateReadwriteFinish' to get
-- the result of the operation.
-- 
-- /Since: 2.22/
fileCreateReadwriteAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileCreateReadwriteAsync file flags ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_create_readwrite_async file' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileCreateReadwriteAsyncMethodInfo
instance (signature ~ ([Gio.Flags.FileCreateFlags] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileCreateReadwriteAsyncMethodInfo a signature where
    overloadedMethod = fileCreateReadwriteAsync

instance O.OverloadedMethodInfo FileCreateReadwriteAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCreateReadwriteAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCreateReadwriteAsync"
        })


#endif

-- method File::create_readwrite_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileIOStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_create_readwrite_finish" g_file_create_readwrite_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileIOStream.FileIOStream)

-- | Finishes an asynchronous file create operation started with
-- 'GI.Gio.Interfaces.File.fileCreateReadwriteAsync'.
-- 
-- /Since: 2.22/
fileCreateReadwriteFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileIOStream.FileIOStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileIOStream.FileIOStream' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileCreateReadwriteFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_create_readwrite_finish file' res'
        checkUnexpectedReturnNULL "fileCreateReadwriteFinish" result
        result' <- (wrapObject Gio.FileIOStream.FileIOStream) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileCreateReadwriteFinishMethodInfo
instance (signature ~ (b -> m Gio.FileIOStream.FileIOStream), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileCreateReadwriteFinishMethodInfo a signature where
    overloadedMethod = fileCreateReadwriteFinish

instance O.OverloadedMethodInfo FileCreateReadwriteFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileCreateReadwriteFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileCreateReadwriteFinish"
        })


#endif

-- method File::delete
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_delete" g_file_delete :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Deletes a file. If the /@file@/ is a directory, it will only be
-- deleted if it is empty. This has the same semantics as 'GI.GLib.Functions.unlink'.
-- 
-- If /@file@/ doesn’t exist, 'GI.Gio.Enums.IOErrorEnumNotFound' will be returned. This allows
-- for deletion to be implemented avoiding
-- <https://en.wikipedia.org/wiki/Time-of-check_to_time-of-use time-of-check to time-of-use races>:
-- >
-- >g_autoptr(GError) local_error = NULL;
-- >if (!g_file_delete (my_file, my_cancellable, &local_error) &&
-- >    !g_error_matches (local_error, G_IO_ERROR, G_IO_ERROR_NOT_FOUND))
-- >  {
-- >    // deletion failed for some reason other than the file not existing:
-- >    // so report the error
-- >    g_warning ("Failed to delete %s: %s",
-- >               g_file_peek_path (my_file), local_error->message);
-- >  }
-- 
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileDelete ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileDelete file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_delete file' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileDeleteMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileDeleteMethodInfo a signature where
    overloadedMethod = fileDelete

instance O.OverloadedMethodInfo FileDeleteMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileDelete",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileDelete"
        })


#endif

-- method File::delete_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call\n  when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_delete_async" g_file_delete_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously delete a file. If the /@file@/ is a directory, it will
-- only be deleted if it is empty.  This has the same semantics as
-- 'GI.GLib.Functions.unlink'.
-- 
-- /Since: 2.34/
fileDeleteAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call
    --   when the request is satisfied
    -> m ()
fileDeleteAsync file ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_delete_async file' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileDeleteAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileDeleteAsyncMethodInfo a signature where
    overloadedMethod = fileDeleteAsync

instance O.OverloadedMethodInfo FileDeleteAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileDeleteAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileDeleteAsync"
        })


#endif

-- method File::delete_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_delete_finish" g_file_delete_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes deleting a file started with 'GI.Gio.Interfaces.File.fileDeleteAsync'.
-- 
-- /Since: 2.34/
fileDeleteFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileDeleteFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_delete_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileDeleteFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileDeleteFinishMethodInfo a signature where
    overloadedMethod = fileDeleteFinish

instance O.OverloadedMethodInfo FileDeleteFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileDeleteFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileDeleteFinish"
        })


#endif

-- method File::dup
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_dup" g_file_dup :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO (Ptr File)

-- | Duplicates a t'GI.Gio.Interfaces.File.File' handle. This operation does not duplicate
-- the actual file or directory represented by the t'GI.Gio.Interfaces.File.File'; see
-- 'GI.Gio.Interfaces.File.fileCopy' if attempting to copy a file.
-- 
-- 'GI.Gio.Interfaces.File.fileDup' is useful when a second handle is needed to the same underlying
-- file, for use in a separate thread (t'GI.Gio.Interfaces.File.File' is not thread-safe). For use
-- within the same thread, use 'GI.GObject.Objects.Object.objectRef' to increment the existing object’s
-- reference count.
-- 
-- This call does no blocking I\/O.
fileDup ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File' that is a duplicate
    --   of the given t'GI.Gio.Interfaces.File.File'.
fileDup file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_dup file'
    checkUnexpectedReturnNULL "fileDup" result
    result' <- (wrapObject File) result
    touchManagedPtr file
    return result'

#if defined(ENABLE_OVERLOADING)
data FileDupMethodInfo
instance (signature ~ (m File), MonadIO m, IsFile a) => O.OverloadedMethod FileDupMethodInfo a signature where
    overloadedMethod = fileDup

instance O.OverloadedMethodInfo FileDupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileDup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileDup"
        })


#endif

-- method File::eject_mountable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_eject_mountable" g_file_eject_mountable :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

{-# DEPRECATED fileEjectMountable ["(Since version 2.22)","Use 'GI.Gio.Interfaces.File.fileEjectMountableWithOperation' instead."] #-}
-- | Starts an asynchronous eject on a mountable.
-- When this operation has completed, /@callback@/ will be called with
-- /@userUser@/ data, and the operation can be finalized with
-- 'GI.Gio.Interfaces.File.fileEjectMountableFinish'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileEjectMountable ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileEjectMountable file flags cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_eject_mountable file' flags' maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileEjectMountableMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileEjectMountableMethodInfo a signature where
    overloadedMethod = fileEjectMountable

instance O.OverloadedMethodInfo FileEjectMountableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileEjectMountable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileEjectMountable"
        })


#endif

-- method File::eject_mountable_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_eject_mountable_finish" g_file_eject_mountable_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

{-# DEPRECATED fileEjectMountableFinish ["(Since version 2.22)","Use 'GI.Gio.Interfaces.File.fileEjectMountableWithOperationFinish'","  instead."] #-}
-- | Finishes an asynchronous eject operation started by
-- 'GI.Gio.Interfaces.File.fileEjectMountable'.
fileEjectMountableFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileEjectMountableFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_eject_mountable_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileEjectMountableFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileEjectMountableFinishMethodInfo a signature where
    overloadedMethod = fileEjectMountableFinish

instance O.OverloadedMethodInfo FileEjectMountableFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileEjectMountableFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileEjectMountableFinish"
        })


#endif

-- method File::eject_mountable_with_operation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation,\n  or %NULL to avoid user interaction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_eject_mountable_with_operation" g_file_eject_mountable_with_operation :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Starts an asynchronous eject on a mountable.
-- When this operation has completed, /@callback@/ will be called with
-- /@userUser@/ data, and the operation can be finalized with
-- 'GI.Gio.Interfaces.File.fileEjectMountableWithOperationFinish'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- /Since: 2.22/
fileEjectMountableWithOperation ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation',
    --   or 'P.Nothing' to avoid user interaction
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileEjectMountableWithOperation file flags mountOperation cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_eject_mountable_with_operation file' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileEjectMountableWithOperationMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileEjectMountableWithOperationMethodInfo a signature where
    overloadedMethod = fileEjectMountableWithOperation

instance O.OverloadedMethodInfo FileEjectMountableWithOperationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileEjectMountableWithOperation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileEjectMountableWithOperation"
        })


#endif

-- method File::eject_mountable_with_operation_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_eject_mountable_with_operation_finish" g_file_eject_mountable_with_operation_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous eject operation started by
-- 'GI.Gio.Interfaces.File.fileEjectMountableWithOperation'.
-- 
-- /Since: 2.22/
fileEjectMountableWithOperationFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileEjectMountableWithOperationFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_eject_mountable_with_operation_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileEjectMountableWithOperationFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileEjectMountableWithOperationFinishMethodInfo a signature where
    overloadedMethod = fileEjectMountableWithOperationFinish

instance O.OverloadedMethodInfo FileEjectMountableWithOperationFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileEjectMountableWithOperationFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileEjectMountableWithOperationFinish"
        })


#endif

-- method File::enumerate_children
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attributes"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an attribute query string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileEnumerator" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_enumerate_children" g_file_enumerate_children :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attributes : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileEnumerator.FileEnumerator)

-- | Gets the requested information about the files in a directory.
-- The result is a t'GI.Gio.Objects.FileEnumerator.FileEnumerator' object that will give out
-- t'GI.Gio.Objects.FileInfo.FileInfo' objects for all the files in the directory.
-- 
-- The /@attributes@/ value is a string that specifies the file
-- attributes that should be gathered. It is not an error if
-- it\'s not possible to read a particular requested attribute
-- from a file - it just won\'t be set. /@attributes@/ should
-- be a comma-separated list of attributes or attribute wildcards.
-- The wildcard \"*\" means all attributes, and a wildcard like
-- \"standard::*\" means all attributes in the standard namespace.
-- An example attribute query be \"standard::*,owner[user](#g:signal:user)\".
-- The standard attributes are available as defines, like
-- 'GI.Gio.Constants.FILE_ATTRIBUTE_STANDARD_NAME'. 'GI.Gio.Constants.FILE_ATTRIBUTE_STANDARD_NAME' should
-- always be specified if you plan to call 'GI.Gio.Objects.FileEnumerator.fileEnumeratorGetChild' or
-- 'GI.Gio.Objects.FileEnumerator.fileEnumeratorIterate' on the returned enumerator.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled
-- by triggering the cancellable object from another thread. If the
-- operation was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be
-- returned.
-- 
-- If the file does not exist, the 'GI.Gio.Enums.IOErrorEnumNotFound' error will
-- be returned. If the file is not a directory, the 'GI.Gio.Enums.IOErrorEnumNotDirectory'
-- error will be returned. Other errors are possible too.
fileEnumerateChildren ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attributes@/: an attribute query string
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileEnumerator.FileEnumerator
    -- ^ __Returns:__ A t'GI.Gio.Objects.FileEnumerator.FileEnumerator' if successful,
    --   'P.Nothing' on error. Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileEnumerateChildren file attributes flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attributes' <- textToCString attributes
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_enumerate_children file' attributes' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileEnumerateChildren" result
        result' <- (wrapObject Gio.FileEnumerator.FileEnumerator) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attributes'
        return result'
     ) (do
        freeMem attributes'
     )

#if defined(ENABLE_OVERLOADING)
data FileEnumerateChildrenMethodInfo
instance (signature ~ (T.Text -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m Gio.FileEnumerator.FileEnumerator), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileEnumerateChildrenMethodInfo a signature where
    overloadedMethod = fileEnumerateChildren

instance O.OverloadedMethodInfo FileEnumerateChildrenMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileEnumerateChildren",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileEnumerateChildren"
        })


#endif

-- method File::enumerate_children_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attributes"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an attribute query string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_enumerate_children_async" g_file_enumerate_children_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attributes : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously gets the requested information about the files
-- in a directory. The result is a t'GI.Gio.Objects.FileEnumerator.FileEnumerator' object that will
-- give out t'GI.Gio.Objects.FileInfo.FileInfo' objects for all the files in the directory.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileEnumerateChildren' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called. You can
-- then call 'GI.Gio.Interfaces.File.fileEnumerateChildrenFinish' to get the result of
-- the operation.
fileEnumerateChildrenAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attributes@/: an attribute query string
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileEnumerateChildrenAsync file attributes flags ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attributes' <- textToCString attributes
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_enumerate_children_async file' attributes' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    freeMem attributes'
    return ()

#if defined(ENABLE_OVERLOADING)
data FileEnumerateChildrenAsyncMethodInfo
instance (signature ~ (T.Text -> [Gio.Flags.FileQueryInfoFlags] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileEnumerateChildrenAsyncMethodInfo a signature where
    overloadedMethod = fileEnumerateChildrenAsync

instance O.OverloadedMethodInfo FileEnumerateChildrenAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileEnumerateChildrenAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileEnumerateChildrenAsync"
        })


#endif

-- method File::enumerate_children_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileEnumerator" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_enumerate_children_finish" g_file_enumerate_children_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileEnumerator.FileEnumerator)

-- | Finishes an async enumerate children operation.
-- See 'GI.Gio.Interfaces.File.fileEnumerateChildrenAsync'.
fileEnumerateChildrenFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileEnumerator.FileEnumerator
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileEnumerator.FileEnumerator' or 'P.Nothing'
    --   if an error occurred.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileEnumerateChildrenFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_enumerate_children_finish file' res'
        checkUnexpectedReturnNULL "fileEnumerateChildrenFinish" result
        result' <- (wrapObject Gio.FileEnumerator.FileEnumerator) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileEnumerateChildrenFinishMethodInfo
instance (signature ~ (b -> m Gio.FileEnumerator.FileEnumerator), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileEnumerateChildrenFinishMethodInfo a signature where
    overloadedMethod = fileEnumerateChildrenFinish

instance O.OverloadedMethodInfo FileEnumerateChildrenFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileEnumerateChildrenFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileEnumerateChildrenFinish"
        })


#endif

-- method File::equal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file1"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the first #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "file2"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the second #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_equal" g_file_equal :: 
    Ptr File ->                             -- file1 : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- file2 : TInterface (Name {namespace = "Gio", name = "File"})
    IO CInt

-- | Checks if the two given @/GFiles/@ refer to the same file.
-- 
-- Note that two @/GFiles/@ that differ can still refer to the same
-- file on the filesystem due to various forms of filename
-- aliasing.
-- 
-- This call does no blocking I\/O.
fileEqual ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b) =>
    a
    -- ^ /@file1@/: the first t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@file2@/: the second t'GI.Gio.Interfaces.File.File'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@file1@/ and /@file2@/ are equal.
fileEqual file1 file2 = liftIO $ do
    file1' <- unsafeManagedPtrCastPtr file1
    file2' <- unsafeManagedPtrCastPtr file2
    result <- g_file_equal file1' file2'
    let result' = (/= 0) result
    touchManagedPtr file1
    touchManagedPtr file2
    return result'

#if defined(ENABLE_OVERLOADING)
data FileEqualMethodInfo
instance (signature ~ (b -> m Bool), MonadIO m, IsFile a, IsFile b) => O.OverloadedMethod FileEqualMethodInfo a signature where
    overloadedMethod = fileEqual

instance O.OverloadedMethodInfo FileEqualMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileEqual",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileEqual"
        })


#endif

-- method File::find_enclosing_mount
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Mount" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_find_enclosing_mount" g_file_find_enclosing_mount :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.Mount.Mount)

-- | Gets a t'GI.Gio.Interfaces.Mount.Mount' for the t'GI.Gio.Interfaces.File.File'.
-- 
-- t'GI.Gio.Interfaces.Mount.Mount' is returned only for user interesting locations, see
-- t'GI.Gio.Objects.VolumeMonitor.VolumeMonitor'. If the t'GI.Gio.Structs.FileIface.FileIface' for /@file@/ does not have a @/mount/@,
-- /@error@/ will be set to 'GI.Gio.Enums.IOErrorEnumNotFound' and 'P.Nothing' @/will/@ be returned.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileFindEnclosingMount ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.Mount.Mount
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.Mount.Mount' where the /@file@/ is located
    --   or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileFindEnclosingMount file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_find_enclosing_mount file' maybeCancellable
        checkUnexpectedReturnNULL "fileFindEnclosingMount" result
        result' <- (wrapObject Gio.Mount.Mount) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileFindEnclosingMountMethodInfo
instance (signature ~ (Maybe (b) -> m Gio.Mount.Mount), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileFindEnclosingMountMethodInfo a signature where
    overloadedMethod = fileFindEnclosingMount

instance O.OverloadedMethodInfo FileFindEnclosingMountMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileFindEnclosingMount",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileFindEnclosingMount"
        })


#endif

-- method File::find_enclosing_mount_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_find_enclosing_mount_async" g_file_find_enclosing_mount_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously gets the mount for the file.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileFindEnclosingMount' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileFindEnclosingMountFinish' to
-- get the result of the operation.
fileFindEnclosingMountAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileFindEnclosingMountAsync file ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_find_enclosing_mount_async file' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileFindEnclosingMountAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileFindEnclosingMountAsyncMethodInfo a signature where
    overloadedMethod = fileFindEnclosingMountAsync

instance O.OverloadedMethodInfo FileFindEnclosingMountAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileFindEnclosingMountAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileFindEnclosingMountAsync"
        })


#endif

-- method File::find_enclosing_mount_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Mount" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_find_enclosing_mount_finish" g_file_find_enclosing_mount_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.Mount.Mount)

-- | Finishes an asynchronous find mount request.
-- See 'GI.Gio.Interfaces.File.fileFindEnclosingMountAsync'.
fileFindEnclosingMountFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.Mount.Mount
    -- ^ __Returns:__ t'GI.Gio.Interfaces.Mount.Mount' for given /@file@/ or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileFindEnclosingMountFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_find_enclosing_mount_finish file' res'
        checkUnexpectedReturnNULL "fileFindEnclosingMountFinish" result
        result' <- (wrapObject Gio.Mount.Mount) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileFindEnclosingMountFinishMethodInfo
instance (signature ~ (b -> m Gio.Mount.Mount), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileFindEnclosingMountFinishMethodInfo a signature where
    overloadedMethod = fileFindEnclosingMountFinish

instance O.OverloadedMethodInfo FileFindEnclosingMountFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileFindEnclosingMountFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileFindEnclosingMountFinish"
        })


#endif

-- method File::get_basename
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFileName)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_get_basename" g_file_get_basename :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO CString

-- | Gets the base name (the last component of the path) for a given t'GI.Gio.Interfaces.File.File'.
-- 
-- If called for the top level of a system (such as the filesystem root
-- or a uri like sftp:\/\/host\/) it will return a single directory separator
-- (and on Windows, possibly a drive letter).
-- 
-- The base name is a byte string (not UTF-8). It has no defined encoding
-- or rules other than it may not contain zero bytes.  If you want to use
-- filenames in a user interface you should use the display name that you
-- can get by requesting the 'GI.Gio.Constants.FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME'
-- attribute with 'GI.Gio.Interfaces.File.fileQueryInfo'.
-- 
-- This call does no blocking I\/O.
fileGetBasename ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m (Maybe [Char])
    -- ^ __Returns:__ string containing the t'GI.Gio.Interfaces.File.File'\'s
    --   base name, or 'P.Nothing' if given t'GI.Gio.Interfaces.File.File' is invalid. The returned string
    --   should be freed with 'GI.GLib.Functions.free' when no longer needed.
fileGetBasename file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_get_basename file'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToString result'
        freeMem result'
        return result''
    touchManagedPtr file
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data FileGetBasenameMethodInfo
instance (signature ~ (m (Maybe [Char])), MonadIO m, IsFile a) => O.OverloadedMethod FileGetBasenameMethodInfo a signature where
    overloadedMethod = fileGetBasename

instance O.OverloadedMethodInfo FileGetBasenameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetBasename",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetBasename"
        })


#endif

-- method File::get_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "string containing the child's basename"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_get_child" g_file_get_child :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- name : TBasicType TFileName
    IO (Ptr File)

-- | Gets a child of /@file@/ with basename equal to /@name@/.
-- 
-- Note that the file with that specific name might not exist, but
-- you can still have a t'GI.Gio.Interfaces.File.File' that points to it. You can use this
-- for instance to create that file.
-- 
-- This call does no blocking I\/O.
fileGetChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Char]
    -- ^ /@name@/: string containing the child\'s basename
    -> m File
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.File.File' to a child specified by /@name@/.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'.
fileGetChild file name = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    name' <- stringToCString name
    result <- g_file_get_child file' name'
    checkUnexpectedReturnNULL "fileGetChild" result
    result' <- (wrapObject File) result
    touchManagedPtr file
    freeMem name'
    return result'

#if defined(ENABLE_OVERLOADING)
data FileGetChildMethodInfo
instance (signature ~ ([Char] -> m File), MonadIO m, IsFile a) => O.OverloadedMethod FileGetChildMethodInfo a signature where
    overloadedMethod = fileGetChild

instance O.OverloadedMethodInfo FileGetChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetChild"
        })


#endif

-- method File::get_child_for_display_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "display_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "string to a possible child"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_get_child_for_display_name" g_file_get_child_for_display_name :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- display_name : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr File)

-- | Gets the child of /@file@/ for a given /@displayName@/ (i.e. a UTF-8
-- version of the name). If this function fails, it returns 'P.Nothing'
-- and /@error@/ will be set. This is very useful when constructing a
-- t'GI.Gio.Interfaces.File.File' for a new file and the user entered the filename in the
-- user interface, for instance when you select a directory and
-- type a filename in the file selector.
-- 
-- This call does no blocking I\/O.
fileGetChildForDisplayName ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@displayName@/: string to a possible child
    -> m File
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.File.File' to the specified child, or
    --   'P.Nothing' if the display name couldn\'t be converted.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileGetChildForDisplayName file displayName = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    displayName' <- textToCString displayName
    onException (do
        result <- propagateGError $ g_file_get_child_for_display_name file' displayName'
        checkUnexpectedReturnNULL "fileGetChildForDisplayName" result
        result' <- (wrapObject File) result
        touchManagedPtr file
        freeMem displayName'
        return result'
     ) (do
        freeMem displayName'
     )

#if defined(ENABLE_OVERLOADING)
data FileGetChildForDisplayNameMethodInfo
instance (signature ~ (T.Text -> m File), MonadIO m, IsFile a) => O.OverloadedMethod FileGetChildForDisplayNameMethodInfo a signature where
    overloadedMethod = fileGetChildForDisplayName

instance O.OverloadedMethodInfo FileGetChildForDisplayNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetChildForDisplayName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetChildForDisplayName"
        })


#endif

-- method File::get_parent
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_get_parent" g_file_get_parent :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO (Ptr File)

-- | Gets the parent directory for the /@file@/.
-- If the /@file@/ represents the root directory of the
-- file system, then 'P.Nothing' will be returned.
-- 
-- This call does no blocking I\/O.
fileGetParent ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m (Maybe File)
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.File.File' structure to the
    --   parent of the given t'GI.Gio.Interfaces.File.File' or 'P.Nothing' if there is no parent. Free
    --   the returned object with 'GI.GObject.Objects.Object.objectUnref'.
fileGetParent file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_get_parent file'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject File) result'
        return result''
    touchManagedPtr file
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data FileGetParentMethodInfo
instance (signature ~ (m (Maybe File)), MonadIO m, IsFile a) => O.OverloadedMethod FileGetParentMethodInfo a signature where
    overloadedMethod = fileGetParent

instance O.OverloadedMethodInfo FileGetParentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetParent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetParent"
        })


#endif

-- method File::get_parse_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_get_parse_name" g_file_get_parse_name :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO CString

-- | Gets the parse name of the /@file@/.
-- A parse name is a UTF-8 string that describes the
-- file such that one can get the t'GI.Gio.Interfaces.File.File' back using
-- 'GI.Gio.Functions.fileParseName'.
-- 
-- This is generally used to show the t'GI.Gio.Interfaces.File.File' as a nice
-- full-pathname kind of string in a user interface,
-- like in a location entry.
-- 
-- For local files with names that can safely be converted
-- to UTF-8 the pathname is used, otherwise the IRI is used
-- (a form of URI that allows UTF-8 characters unescaped).
-- 
-- This call does no blocking I\/O.
fileGetParseName ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m T.Text
    -- ^ __Returns:__ a string containing the t'GI.Gio.Interfaces.File.File'\'s parse name.
    --   The returned string should be freed with 'GI.GLib.Functions.free'
    --   when no longer needed.
fileGetParseName file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_get_parse_name file'
    checkUnexpectedReturnNULL "fileGetParseName" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr file
    return result'

#if defined(ENABLE_OVERLOADING)
data FileGetParseNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsFile a) => O.OverloadedMethod FileGetParseNameMethodInfo a signature where
    overloadedMethod = fileGetParseName

instance O.OverloadedMethodInfo FileGetParseNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetParseName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetParseName"
        })


#endif

-- method File::get_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFileName)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_get_path" g_file_get_path :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO CString

-- | Gets the local pathname for t'GI.Gio.Interfaces.File.File', if one exists. If non-'P.Nothing', this is
-- guaranteed to be an absolute, canonical path. It might contain symlinks.
-- 
-- This call does no blocking I\/O.
fileGetPath ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m (Maybe [Char])
    -- ^ __Returns:__ string containing the t'GI.Gio.Interfaces.File.File'\'s path,
    --   or 'P.Nothing' if no such path exists. The returned string should be freed
    --   with 'GI.GLib.Functions.free' when no longer needed.
fileGetPath file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_get_path file'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToString result'
        freeMem result'
        return result''
    touchManagedPtr file
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data FileGetPathMethodInfo
instance (signature ~ (m (Maybe [Char])), MonadIO m, IsFile a) => O.OverloadedMethod FileGetPathMethodInfo a signature where
    overloadedMethod = fileGetPath

instance O.OverloadedMethodInfo FileGetPathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetPath"
        })


#endif

-- method File::get_relative_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "parent"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "descendant"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFileName)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_get_relative_path" g_file_get_relative_path :: 
    Ptr File ->                             -- parent : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- descendant : TInterface (Name {namespace = "Gio", name = "File"})
    IO CString

-- | Gets the path for /@descendant@/ relative to /@parent@/.
-- 
-- This call does no blocking I\/O.
fileGetRelativePath ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b) =>
    a
    -- ^ /@parent@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@descendant@/: input t'GI.Gio.Interfaces.File.File'
    -> m (Maybe [Char])
    -- ^ __Returns:__ string with the relative path from
    --   /@descendant@/ to /@parent@/, or 'P.Nothing' if /@descendant@/ doesn\'t have /@parent@/ as
    --   prefix. The returned string should be freed with 'GI.GLib.Functions.free' when
    --   no longer needed.
fileGetRelativePath parent descendant = liftIO $ do
    parent' <- unsafeManagedPtrCastPtr parent
    descendant' <- unsafeManagedPtrCastPtr descendant
    result <- g_file_get_relative_path parent' descendant'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToString result'
        freeMem result'
        return result''
    touchManagedPtr parent
    touchManagedPtr descendant
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data FileGetRelativePathMethodInfo
instance (signature ~ (b -> m (Maybe [Char])), MonadIO m, IsFile a, IsFile b) => O.OverloadedMethod FileGetRelativePathMethodInfo a signature where
    overloadedMethod = fileGetRelativePath

instance O.OverloadedMethodInfo FileGetRelativePathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetRelativePath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetRelativePath"
        })


#endif

-- method File::get_uri
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_get_uri" g_file_get_uri :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO CString

-- | Gets the URI for the /@file@/.
-- 
-- This call does no blocking I\/O.
fileGetUri ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m T.Text
    -- ^ __Returns:__ a string containing the t'GI.Gio.Interfaces.File.File'\'s URI. If the t'GI.Gio.Interfaces.File.File' was constructed
    --   with an invalid URI, an invalid URI is returned.
    --   The returned string should be freed with 'GI.GLib.Functions.free'
    --   when no longer needed.
fileGetUri file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_get_uri file'
    checkUnexpectedReturnNULL "fileGetUri" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr file
    return result'

#if defined(ENABLE_OVERLOADING)
data FileGetUriMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsFile a) => O.OverloadedMethod FileGetUriMethodInfo a signature where
    overloadedMethod = fileGetUri

instance O.OverloadedMethodInfo FileGetUriMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetUri",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetUri"
        })


#endif

-- method File::get_uri_scheme
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_get_uri_scheme" g_file_get_uri_scheme :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO CString

-- | Gets the URI scheme for a t'GI.Gio.Interfaces.File.File'.
-- RFC 3986 decodes the scheme as:
-- >
-- >URI = scheme ":" hier-part [ "?" query ] [ "#" fragment ]
-- 
-- Common schemes include \"file\", \"http\", \"ftp\", etc.
-- 
-- The scheme can be different from the one used to construct the t'GI.Gio.Interfaces.File.File',
-- in that it might be replaced with one that is logically equivalent to the t'GI.Gio.Interfaces.File.File'.
-- 
-- This call does no blocking I\/O.
fileGetUriScheme ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ a string containing the URI scheme for the given
    --   t'GI.Gio.Interfaces.File.File' or 'P.Nothing' if the t'GI.Gio.Interfaces.File.File' was constructed with an invalid URI. The
    --   returned string should be freed with 'GI.GLib.Functions.free' when no longer needed.
fileGetUriScheme file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_get_uri_scheme file'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr file
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data FileGetUriSchemeMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsFile a) => O.OverloadedMethod FileGetUriSchemeMethodInfo a signature where
    overloadedMethod = fileGetUriScheme

instance O.OverloadedMethodInfo FileGetUriSchemeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileGetUriScheme",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileGetUriScheme"
        })


#endif

-- method File::has_parent
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parent"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the parent to check for, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_has_parent" g_file_has_parent :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- parent : TInterface (Name {namespace = "Gio", name = "File"})
    IO CInt

-- | Checks if /@file@/ has a parent, and optionally, if it is /@parent@/.
-- 
-- If /@parent@/ is 'P.Nothing' then this function returns 'P.True' if /@file@/ has any
-- parent at all.  If /@parent@/ is non-'P.Nothing' then 'P.True' is only returned
-- if /@file@/ is an immediate child of /@parent@/.
-- 
-- /Since: 2.24/
fileHasParent ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@parent@/: the parent to check for, or 'P.Nothing'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@file@/ is an immediate child of /@parent@/ (or any parent in
    --   the case that /@parent@/ is 'P.Nothing').
fileHasParent file parent = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeParent <- case parent of
        Nothing -> return FP.nullPtr
        Just jParent -> do
            jParent' <- unsafeManagedPtrCastPtr jParent
            return jParent'
    result <- g_file_has_parent file' maybeParent
    let result' = (/= 0) result
    touchManagedPtr file
    whenJust parent touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data FileHasParentMethodInfo
instance (signature ~ (Maybe (b) -> m Bool), MonadIO m, IsFile a, IsFile b) => O.OverloadedMethod FileHasParentMethodInfo a signature where
    overloadedMethod = fileHasParent

instance O.OverloadedMethodInfo FileHasParentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileHasParent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileHasParent"
        })


#endif

-- method File::has_prefix
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "prefix"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_has_prefix" g_file_has_prefix :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- prefix : TInterface (Name {namespace = "Gio", name = "File"})
    IO CInt

-- | Checks whether /@file@/ has the prefix specified by /@prefix@/.
-- 
-- In other words, if the names of initial elements of /@file@/\'s
-- pathname match /@prefix@/. Only full pathname elements are matched,
-- so a path like \/foo is not considered a prefix of \/foobar, only
-- of \/foo\/bar.
-- 
-- A t'GI.Gio.Interfaces.File.File' is not a prefix of itself. If you want to check for
-- equality, use 'GI.Gio.Interfaces.File.fileEqual'.
-- 
-- This call does no I\/O, as it works purely on names. As such it can
-- sometimes return 'P.False' even if /@file@/ is inside a /@prefix@/ (from a
-- filesystem point of view), because the prefix of /@file@/ is an alias
-- of /@prefix@/.
fileHasPrefix ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@prefix@/: input t'GI.Gio.Interfaces.File.File'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@file@/\'s parent, grandparent, etc is /@prefix@/,
    --   'P.False' otherwise.
fileHasPrefix file prefix = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    prefix' <- unsafeManagedPtrCastPtr prefix
    result <- g_file_has_prefix file' prefix'
    let result' = (/= 0) result
    touchManagedPtr file
    touchManagedPtr prefix
    return result'

#if defined(ENABLE_OVERLOADING)
data FileHasPrefixMethodInfo
instance (signature ~ (b -> m Bool), MonadIO m, IsFile a, IsFile b) => O.OverloadedMethod FileHasPrefixMethodInfo a signature where
    overloadedMethod = fileHasPrefix

instance O.OverloadedMethodInfo FileHasPrefixMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileHasPrefix",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileHasPrefix"
        })


#endif

-- method File::has_uri_scheme
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri_scheme"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing a URI scheme"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_has_uri_scheme" g_file_has_uri_scheme :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- uri_scheme : TBasicType TUTF8
    IO CInt

-- | Checks to see if a t'GI.Gio.Interfaces.File.File' has a given URI scheme.
-- 
-- This call does no blocking I\/O.
fileHasUriScheme ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@uriScheme@/: a string containing a URI scheme
    -> m Bool
    -- ^ __Returns:__ 'P.True' if t'GI.Gio.Interfaces.File.File'\'s backend supports the
    --   given URI scheme, 'P.False' if URI scheme is 'P.Nothing',
    --   not supported, or t'GI.Gio.Interfaces.File.File' is invalid.
fileHasUriScheme file uriScheme = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    uriScheme' <- textToCString uriScheme
    result <- g_file_has_uri_scheme file' uriScheme'
    let result' = (/= 0) result
    touchManagedPtr file
    freeMem uriScheme'
    return result'

#if defined(ENABLE_OVERLOADING)
data FileHasUriSchemeMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m, IsFile a) => O.OverloadedMethod FileHasUriSchemeMethodInfo a signature where
    overloadedMethod = fileHasUriScheme

instance O.OverloadedMethodInfo FileHasUriSchemeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileHasUriScheme",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileHasUriScheme"
        })


#endif

-- method File::hash
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#gconstpointer to a #GFile"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_hash" g_file_hash :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO Word32

-- | Creates a hash value for a t'GI.Gio.Interfaces.File.File'.
-- 
-- This call does no blocking I\/O.
fileHash ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: @/gconstpointer/@ to a t'GI.Gio.Interfaces.File.File'
    -> m Word32
    -- ^ __Returns:__ 0 if /@file@/ is not a valid t'GI.Gio.Interfaces.File.File', otherwise an
    --   integer that can be used as hash value for the t'GI.Gio.Interfaces.File.File'.
    --   This function is intended for easily hashing a t'GI.Gio.Interfaces.File.File' to
    --   add to a t'GI.GLib.Structs.HashTable.HashTable' or similar data structure.
fileHash file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_hash file'
    touchManagedPtr file
    return result

#if defined(ENABLE_OVERLOADING)
data FileHashMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsFile a) => O.OverloadedMethod FileHashMethodInfo a signature where
    overloadedMethod = fileHash

instance O.OverloadedMethodInfo FileHashMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileHash",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileHash"
        })


#endif

-- method File::is_native
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_is_native" g_file_is_native :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO CInt

-- | Checks to see if a file is native to the platform.
-- 
-- A native file is one expressed in the platform-native filename format,
-- e.g. \"C:\\Windows\" or \"\/usr\/bin\/\". This does not mean the file is local,
-- as it might be on a locally mounted remote filesystem.
-- 
-- On some systems non-native files may be available using the native
-- filesystem via a userspace filesystem (FUSE), in these cases this call
-- will return 'P.False', but 'GI.Gio.Interfaces.File.fileGetPath' will still return a native path.
-- 
-- This call does no blocking I\/O.
fileIsNative ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@file@/ is native
fileIsNative file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_is_native file'
    let result' = (/= 0) result
    touchManagedPtr file
    return result'

#if defined(ENABLE_OVERLOADING)
data FileIsNativeMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsFile a) => O.OverloadedMethod FileIsNativeMethodInfo a signature where
    overloadedMethod = fileIsNative

instance O.OverloadedMethodInfo FileIsNativeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileIsNative",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileIsNative"
        })


#endif

-- method File::load_bytes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag_out"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to place the current\n  entity tag for the file, or %NULL if the entity tag is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_load_bytes" g_file_load_bytes :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr CString ->                          -- etag_out : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GLib.Bytes.Bytes)

-- | Loads the contents of /@file@/ and returns it as t'GI.GLib.Structs.Bytes.Bytes'.
-- 
-- If /@file@/ is a resource:\/\/ based URI, the resulting bytes will reference the
-- embedded resource instead of a copy. Otherwise, this is equivalent to calling
-- 'GI.Gio.Interfaces.File.fileLoadContents' and 'GI.GLib.Structs.Bytes.bytesNewTake'.
-- 
-- For resources, /@etagOut@/ will be set to 'P.Nothing'.
-- 
-- The data contained in the resulting t'GI.GLib.Structs.Bytes.Bytes' is always zero-terminated, but
-- this is not included in the t'GI.GLib.Structs.Bytes.Bytes' length. The resulting t'GI.GLib.Structs.Bytes.Bytes' should be
-- freed with 'GI.GLib.Structs.Bytes.bytesUnref' when no longer in use.
-- 
-- /Since: 2.56/
fileLoadBytes ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'
    -> m ((GLib.Bytes.Bytes, Maybe T.Text))
    -- ^ __Returns:__ a t'GI.GLib.Structs.Bytes.Bytes' or 'P.Nothing' and /@error@/ is set /(Can throw 'Data.GI.Base.GError.GError')/
fileLoadBytes file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    etagOut <- callocMem :: IO (Ptr CString)
    onException (do
        result <- propagateGError $ g_file_load_bytes file' maybeCancellable etagOut
        checkUnexpectedReturnNULL "fileLoadBytes" result
        result' <- (wrapBoxed GLib.Bytes.Bytes) result
        etagOut' <- peek etagOut
        maybeEtagOut' <- convertIfNonNull etagOut' $ \etagOut'' -> do
            etagOut''' <- cstringToText etagOut''
            return etagOut'''
        freeMem etagOut'
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem etagOut
        return (result', maybeEtagOut')
     ) (do
        freeMem etagOut
     )

#if defined(ENABLE_OVERLOADING)
data FileLoadBytesMethodInfo
instance (signature ~ (Maybe (b) -> m ((GLib.Bytes.Bytes, Maybe T.Text))), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileLoadBytesMethodInfo a signature where
    overloadedMethod = fileLoadBytes

instance O.OverloadedMethodInfo FileLoadBytesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileLoadBytes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileLoadBytes"
        })


#endif

-- method File::load_bytes_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 3
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_load_bytes_async" g_file_load_bytes_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously loads the contents of /@file@/ as t'GI.GLib.Structs.Bytes.Bytes'.
-- 
-- If /@file@/ is a resource:\/\/ based URI, the resulting bytes will reference the
-- embedded resource instead of a copy. Otherwise, this is equivalent to calling
-- 'GI.Gio.Interfaces.File.fileLoadContentsAsync' and 'GI.GLib.Structs.Bytes.bytesNewTake'.
-- 
-- /@callback@/ should call 'GI.Gio.Interfaces.File.fileLoadBytesFinish' to get the result of this
-- asynchronous operation.
-- 
-- See 'GI.Gio.Interfaces.File.fileLoadBytes' for more information.
-- 
-- /Since: 2.56/
fileLoadBytesAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileLoadBytesAsync file cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_load_bytes_async file' maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileLoadBytesAsyncMethodInfo
instance (signature ~ (Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileLoadBytesAsyncMethodInfo a signature where
    overloadedMethod = fileLoadBytesAsync

instance O.OverloadedMethodInfo FileLoadBytesAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileLoadBytesAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileLoadBytesAsync"
        })


#endif

-- method File::load_bytes_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult provided to the callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag_out"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to place the current\n  entity tag for the file, or %NULL if the entity tag is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_load_bytes_finish" g_file_load_bytes_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr CString ->                          -- etag_out : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GLib.Bytes.Bytes)

-- | Completes an asynchronous request to 'GI.Gio.Interfaces.File.fileLoadBytesAsync'.
-- 
-- For resources, /@etagOut@/ will be set to 'P.Nothing'.
-- 
-- The data contained in the resulting t'GI.GLib.Structs.Bytes.Bytes' is always zero-terminated, but
-- this is not included in the t'GI.GLib.Structs.Bytes.Bytes' length. The resulting t'GI.GLib.Structs.Bytes.Bytes' should be
-- freed with 'GI.GLib.Structs.Bytes.bytesUnref' when no longer in use.
-- 
-- See 'GI.Gio.Interfaces.File.fileLoadBytes' for more information.
-- 
-- /Since: 2.56/
fileLoadBytesFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult' provided to the callback
    -> m ((GLib.Bytes.Bytes, Maybe T.Text))
    -- ^ __Returns:__ a t'GI.GLib.Structs.Bytes.Bytes' or 'P.Nothing' and /@error@/ is set /(Can throw 'Data.GI.Base.GError.GError')/
fileLoadBytesFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    etagOut <- callocMem :: IO (Ptr CString)
    onException (do
        result <- propagateGError $ g_file_load_bytes_finish file' result_' etagOut
        checkUnexpectedReturnNULL "fileLoadBytesFinish" result
        result' <- (wrapBoxed GLib.Bytes.Bytes) result
        etagOut' <- peek etagOut
        maybeEtagOut' <- convertIfNonNull etagOut' $ \etagOut'' -> do
            etagOut''' <- cstringToText etagOut''
            return etagOut'''
        freeMem etagOut'
        touchManagedPtr file
        touchManagedPtr result_
        freeMem etagOut
        return (result', maybeEtagOut')
     ) (do
        freeMem etagOut
     )

#if defined(ENABLE_OVERLOADING)
data FileLoadBytesFinishMethodInfo
instance (signature ~ (b -> m ((GLib.Bytes.Bytes, Maybe T.Text))), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileLoadBytesFinishMethodInfo a signature where
    overloadedMethod = fileLoadBytesFinish

instance O.OverloadedMethodInfo FileLoadBytesFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileLoadBytesFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileLoadBytesFinish"
        })


#endif

-- method File::load_contents
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "contents"
--           , argType = TCArray False (-1) 3 (TBasicType TUInt8)
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a location to place the contents of the file"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to place the length of the contents of the file,\n  or %NULL if the length is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "etag_out"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to place the current entity tag for the file,\n  or %NULL if the entity tag is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "a location to place the length of the contents of the file,\n  or %NULL if the length is not needed"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_load_contents" g_file_load_contents :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr Word8) ->                      -- contents : TCArray False (-1) 3 (TBasicType TUInt8)
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr CString ->                          -- etag_out : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Loads the content of the file into memory. The data is always
-- zero-terminated, but this is not included in the resultant /@length@/.
-- The returned /@contents@/ should be freed with 'GI.GLib.Functions.free' when no longer
-- needed.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileLoadContents ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> m ((ByteString, Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileLoadContents file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    contents <- callocMem :: IO (Ptr (Ptr Word8))
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    etagOut <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_file_load_contents file' maybeCancellable contents length_ etagOut
        length_' <- peek length_
        contents' <- peek contents
        contents'' <- (unpackByteStringWithLength length_') contents'
        freeMem contents'
        etagOut' <- peek etagOut
        maybeEtagOut' <- convertIfNonNull etagOut' $ \etagOut'' -> do
            etagOut''' <- cstringToText etagOut''
            return etagOut'''
        freeMem etagOut'
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem contents
        freeMem length_
        freeMem etagOut
        return (contents'', maybeEtagOut')
     ) (do
        freeMem contents
        freeMem length_
        freeMem etagOut
     )

#if defined(ENABLE_OVERLOADING)
data FileLoadContentsMethodInfo
instance (signature ~ (Maybe (b) -> m ((ByteString, Maybe T.Text))), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileLoadContentsMethodInfo a signature where
    overloadedMethod = fileLoadContents

instance O.OverloadedMethodInfo FileLoadContentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileLoadContents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileLoadContents"
        })


#endif

-- method File::load_contents_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GAsyncReadyCallback to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 3
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_load_contents_async" g_file_load_contents_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Starts an asynchronous load of the /@file@/\'s contents.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileLoadContents' which is
-- the synchronous version of this call.
-- 
-- When the load operation has completed, /@callback@/ will be called
-- with /@user@/ data. To finish the operation, call
-- 'GI.Gio.Interfaces.File.fileLoadContentsFinish' with the t'GI.Gio.Interfaces.AsyncResult.AsyncResult' returned by
-- the /@callback@/.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileLoadContentsAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is satisfied
    -> m ()
fileLoadContentsAsync file cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_load_contents_async file' maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileLoadContentsAsyncMethodInfo
instance (signature ~ (Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileLoadContentsAsyncMethodInfo a signature where
    overloadedMethod = fileLoadContentsAsync

instance O.OverloadedMethodInfo FileLoadContentsAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileLoadContentsAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileLoadContentsAsync"
        })


#endif

-- method File::load_contents_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "contents"
--           , argType = TCArray False (-1) 3 (TBasicType TUInt8)
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a location to place the contents of the file"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to place the length of the contents of the file,\n  or %NULL if the length is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "etag_out"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to place the current entity tag for the file,\n  or %NULL if the entity tag is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "a location to place the length of the contents of the file,\n  or %NULL if the length is not needed"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_load_contents_finish" g_file_load_contents_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr Word8) ->                      -- contents : TCArray False (-1) 3 (TBasicType TUInt8)
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr CString ->                          -- etag_out : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous load of the /@file@/\'s contents.
-- The contents are placed in /@contents@/, and /@length@/ is set to the
-- size of the /@contents@/ string. The /@contents@/ should be freed with
-- 'GI.GLib.Functions.free' when no longer needed. If /@etagOut@/ is present, it will be
-- set to the new entity tag for the /@file@/.
fileLoadContentsFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ((ByteString, Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileLoadContentsFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    contents <- callocMem :: IO (Ptr (Ptr Word8))
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    etagOut <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_file_load_contents_finish file' res' contents length_ etagOut
        length_' <- peek length_
        contents' <- peek contents
        contents'' <- (unpackByteStringWithLength length_') contents'
        freeMem contents'
        etagOut' <- peek etagOut
        maybeEtagOut' <- convertIfNonNull etagOut' $ \etagOut'' -> do
            etagOut''' <- cstringToText etagOut''
            return etagOut'''
        freeMem etagOut'
        touchManagedPtr file
        touchManagedPtr res
        freeMem contents
        freeMem length_
        freeMem etagOut
        return (contents'', maybeEtagOut')
     ) (do
        freeMem contents
        freeMem length_
        freeMem etagOut
     )

#if defined(ENABLE_OVERLOADING)
data FileLoadContentsFinishMethodInfo
instance (signature ~ (b -> m ((ByteString, Maybe T.Text))), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileLoadContentsFinishMethodInfo a signature where
    overloadedMethod = fileLoadContentsFinish

instance O.OverloadedMethodInfo FileLoadContentsFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileLoadContentsFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileLoadContentsFinish"
        })


#endif

-- method File::load_partial_contents_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "contents"
--           , argType = TCArray False (-1) 3 (TBasicType TUInt8)
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a location to place the contents of the file"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to place the length of the contents of the file,\n  or %NULL if the length is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "etag_out"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to place the current entity tag for the file,\n  or %NULL if the entity tag is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "a location to place the length of the contents of the file,\n  or %NULL if the length is not needed"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_load_partial_contents_finish" g_file_load_partial_contents_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr Word8) ->                      -- contents : TCArray False (-1) 3 (TBasicType TUInt8)
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr CString ->                          -- etag_out : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous partial load operation that was started
-- with @/g_file_load_partial_contents_async()/@. The data is always
-- zero-terminated, but this is not included in the resultant /@length@/.
-- The returned /@contents@/ should be freed with 'GI.GLib.Functions.free' when no longer
-- needed.
fileLoadPartialContentsFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ((ByteString, Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileLoadPartialContentsFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    contents <- callocMem :: IO (Ptr (Ptr Word8))
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    etagOut <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_file_load_partial_contents_finish file' res' contents length_ etagOut
        length_' <- peek length_
        contents' <- peek contents
        contents'' <- (unpackByteStringWithLength length_') contents'
        freeMem contents'
        etagOut' <- peek etagOut
        maybeEtagOut' <- convertIfNonNull etagOut' $ \etagOut'' -> do
            etagOut''' <- cstringToText etagOut''
            return etagOut'''
        freeMem etagOut'
        touchManagedPtr file
        touchManagedPtr res
        freeMem contents
        freeMem length_
        freeMem etagOut
        return (contents'', maybeEtagOut')
     ) (do
        freeMem contents
        freeMem length_
        freeMem etagOut
     )

#if defined(ENABLE_OVERLOADING)
data FileLoadPartialContentsFinishMethodInfo
instance (signature ~ (b -> m ((ByteString, Maybe T.Text))), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileLoadPartialContentsFinishMethodInfo a signature where
    overloadedMethod = fileLoadPartialContentsFinish

instance O.OverloadedMethodInfo FileLoadPartialContentsFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileLoadPartialContentsFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileLoadPartialContentsFinish"
        })


#endif

-- method File::make_directory
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_make_directory" g_file_make_directory :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Creates a directory.
-- 
-- Note that this will only create a child directory
-- of the immediate parent directory of the path or URI given by the t'GI.Gio.Interfaces.File.File'.
-- To recursively create directories, see 'GI.Gio.Interfaces.File.fileMakeDirectoryWithParents'.
-- 
-- This function will fail if the parent directory does not exist, setting
-- /@error@/ to 'GI.Gio.Enums.IOErrorEnumNotFound'. If the file system doesn\'t support
-- creating directories, this function will fail, setting /@error@/ to
-- 'GI.Gio.Enums.IOErrorEnumNotSupported'. If the directory already exists,
-- [error/@gio@/.IOErrorEnum.EXISTS] will be returned.
-- 
-- For a local t'GI.Gio.Interfaces.File.File' the newly created directory will have the default
-- (current) ownership and permissions of the current process.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileMakeDirectory ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMakeDirectory file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_make_directory file' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMakeDirectoryMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMakeDirectoryMethodInfo a signature where
    overloadedMethod = fileMakeDirectory

instance O.OverloadedMethodInfo FileMakeDirectoryMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMakeDirectory",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMakeDirectory"
        })


#endif

-- method File::make_directory_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call\n  when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_make_directory_async" g_file_make_directory_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously creates a directory.
-- 
-- /Since: 2.38/
fileMakeDirectoryAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call
    --   when the request is satisfied
    -> m ()
fileMakeDirectoryAsync file ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_make_directory_async file' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileMakeDirectoryAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMakeDirectoryAsyncMethodInfo a signature where
    overloadedMethod = fileMakeDirectoryAsync

instance O.OverloadedMethodInfo FileMakeDirectoryAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMakeDirectoryAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMakeDirectoryAsync"
        })


#endif

-- method File::make_directory_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_make_directory_finish" g_file_make_directory_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous directory creation, started with
-- 'GI.Gio.Interfaces.File.fileMakeDirectoryAsync'.
-- 
-- /Since: 2.38/
fileMakeDirectoryFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMakeDirectoryFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_make_directory_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMakeDirectoryFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileMakeDirectoryFinishMethodInfo a signature where
    overloadedMethod = fileMakeDirectoryFinish

instance O.OverloadedMethodInfo FileMakeDirectoryFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMakeDirectoryFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMakeDirectoryFinish"
        })


#endif

-- method File::make_directory_with_parents
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_make_directory_with_parents" g_file_make_directory_with_parents :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Creates a directory and any parent directories that may not
-- exist similar to \'mkdir -p\'. If the file system does not support
-- creating directories, this function will fail, setting /@error@/ to
-- 'GI.Gio.Enums.IOErrorEnumNotSupported'. If the directory itself already exists,
-- this function will fail setting /@error@/ to 'GI.Gio.Enums.IOErrorEnumExists', unlike
-- the similar 'GI.GLib.Functions.mkdirWithParents'.
-- 
-- For a local t'GI.Gio.Interfaces.File.File' the newly created directories will have the default
-- (current) ownership and permissions of the current process.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- /Since: 2.18/
fileMakeDirectoryWithParents ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMakeDirectoryWithParents file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_make_directory_with_parents file' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMakeDirectoryWithParentsMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMakeDirectoryWithParentsMethodInfo a signature where
    overloadedMethod = fileMakeDirectoryWithParents

instance O.OverloadedMethodInfo FileMakeDirectoryWithParentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMakeDirectoryWithParents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMakeDirectoryWithParents"
        })


#endif

-- method File::make_symbolic_link
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GFile with the name of the symlink to create"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "symlink_value"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a string with the path for the target\n  of the new symlink"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_make_symbolic_link" g_file_make_symbolic_link :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- symlink_value : TBasicType TFileName
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Creates a symbolic link named /@file@/ which contains the string
-- /@symlinkValue@/.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileMakeSymbolicLink ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File' with the name of the symlink to create
    -> [Char]
    -- ^ /@symlinkValue@/: a string with the path for the target
    --   of the new symlink
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMakeSymbolicLink file symlinkValue cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    symlinkValue' <- stringToCString symlinkValue
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_make_symbolic_link file' symlinkValue' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem symlinkValue'
        return ()
     ) (do
        freeMem symlinkValue'
     )

#if defined(ENABLE_OVERLOADING)
data FileMakeSymbolicLinkMethodInfo
instance (signature ~ ([Char] -> Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMakeSymbolicLinkMethodInfo a signature where
    overloadedMethod = fileMakeSymbolicLink

instance O.OverloadedMethodInfo FileMakeSymbolicLinkMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMakeSymbolicLink",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMakeSymbolicLink"
        })


#endif

-- method File::make_symbolic_link_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GFile with the name of the symlink to create"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "symlink_value"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a string with the path for the target\n  of the new symlink"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call\n  when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_make_symbolic_link_async" g_file_make_symbolic_link_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- symlink_value : TBasicType TFileName
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously creates a symbolic link named /@file@/ which contains the
-- string /@symlinkValue@/.
-- 
-- /Since: 2.74/
fileMakeSymbolicLinkAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File' with the name of the symlink to create
    -> [Char]
    -- ^ /@symlinkValue@/: a string with the path for the target
    --   of the new symlink
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call
    --   when the request is satisfied
    -> m ()
fileMakeSymbolicLinkAsync file symlinkValue ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    symlinkValue' <- stringToCString symlinkValue
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_make_symbolic_link_async file' symlinkValue' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    freeMem symlinkValue'
    return ()

#if defined(ENABLE_OVERLOADING)
data FileMakeSymbolicLinkAsyncMethodInfo
instance (signature ~ ([Char] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMakeSymbolicLinkAsyncMethodInfo a signature where
    overloadedMethod = fileMakeSymbolicLinkAsync

instance O.OverloadedMethodInfo FileMakeSymbolicLinkAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMakeSymbolicLinkAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMakeSymbolicLinkAsync"
        })


#endif

-- method File::make_symbolic_link_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_make_symbolic_link_finish" g_file_make_symbolic_link_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous symbolic link creation, started with
-- 'GI.Gio.Interfaces.File.fileMakeSymbolicLinkAsync'.
-- 
-- /Since: 2.74/
fileMakeSymbolicLinkFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMakeSymbolicLinkFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_make_symbolic_link_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMakeSymbolicLinkFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileMakeSymbolicLinkFinishMethodInfo a signature where
    overloadedMethod = fileMakeSymbolicLinkFinish

instance O.OverloadedMethodInfo FileMakeSymbolicLinkFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMakeSymbolicLinkFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMakeSymbolicLinkFinish"
        })


#endif

-- method File::measure_disk_usage
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileMeasureFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GFileMeasureFlags" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "optional #GCancellable"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "progress_callback"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "FileMeasureProgressCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileMeasureProgressCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeCall
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "progress_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user_data for @progress_callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "disk_usage"
--           , argType = TBasicType TUInt64
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of bytes of disk space used"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "num_dirs"
--           , argType = TBasicType TUInt64
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of directories encountered"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "num_files"
--           , argType = TBasicType TUInt64
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of non-directories encountered"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_measure_disk_usage" g_file_measure_disk_usage :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileMeasureFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_FileMeasureProgressCallback -> -- progress_callback : TInterface (Name {namespace = "Gio", name = "FileMeasureProgressCallback"})
    Ptr () ->                               -- progress_data : TBasicType TPtr
    Ptr Word64 ->                           -- disk_usage : TBasicType TUInt64
    Ptr Word64 ->                           -- num_dirs : TBasicType TUInt64
    Ptr Word64 ->                           -- num_files : TBasicType TUInt64
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Recursively measures the disk usage of /@file@/.
-- 
-- This is essentially an analog of the \'du\' command, but it also
-- reports the number of directories and non-directory files encountered
-- (including things like symbolic links).
-- 
-- By default, errors are only reported against the toplevel file
-- itself.  Errors found while recursing are silently ignored, unless
-- 'GI.Gio.Flags.FileMeasureFlagsReportAnyError' is given in /@flags@/.
-- 
-- The returned size, /@diskUsage@/, is in bytes and should be formatted
-- with 'GI.GLib.Functions.formatSize' in order to get something reasonable for showing
-- in a user interface.
-- 
-- /@progressCallback@/ and /@progressData@/ can be given to request
-- periodic progress updates while scanning.  See the documentation for
-- t'GI.Gio.Callbacks.FileMeasureProgressCallback' for information about when and how the
-- callback will be invoked.
-- 
-- /Since: 2.38/
fileMeasureDiskUsage ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileMeasureFlags]
    -- ^ /@flags@/: t'GI.Gio.Flags.FileMeasureFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable'
    -> Maybe (Gio.Callbacks.FileMeasureProgressCallback)
    -- ^ /@progressCallback@/: a t'GI.Gio.Callbacks.FileMeasureProgressCallback'
    -> m ((Word64, Word64, Word64))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMeasureDiskUsage file flags cancellable progressCallback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeProgressCallback <- case progressCallback of
        Nothing -> return FP.nullFunPtr
        Just jProgressCallback -> do
            jProgressCallback' <- Gio.Callbacks.mk_FileMeasureProgressCallback (Gio.Callbacks.wrap_FileMeasureProgressCallback Nothing (Gio.Callbacks.drop_closures_FileMeasureProgressCallback jProgressCallback))
            return jProgressCallback'
    diskUsage <- allocMem :: IO (Ptr Word64)
    numDirs <- allocMem :: IO (Ptr Word64)
    numFiles <- allocMem :: IO (Ptr Word64)
    let progressData = nullPtr
    onException (do
        _ <- propagateGError $ g_file_measure_disk_usage file' flags' maybeCancellable maybeProgressCallback progressData diskUsage numDirs numFiles
        diskUsage' <- peek diskUsage
        numDirs' <- peek numDirs
        numFiles' <- peek numFiles
        safeFreeFunPtr $ castFunPtrToPtr maybeProgressCallback
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem diskUsage
        freeMem numDirs
        freeMem numFiles
        return (diskUsage', numDirs', numFiles')
     ) (do
        safeFreeFunPtr $ castFunPtrToPtr maybeProgressCallback
        freeMem diskUsage
        freeMem numDirs
        freeMem numFiles
     )

#if defined(ENABLE_OVERLOADING)
data FileMeasureDiskUsageMethodInfo
instance (signature ~ ([Gio.Flags.FileMeasureFlags] -> Maybe (b) -> Maybe (Gio.Callbacks.FileMeasureProgressCallback) -> m ((Word64, Word64, Word64))), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMeasureDiskUsageMethodInfo a signature where
    overloadedMethod = fileMeasureDiskUsage

instance O.OverloadedMethodInfo FileMeasureDiskUsageMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMeasureDiskUsage",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMeasureDiskUsage"
        })


#endif

-- method File::measure_disk_usage_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the #GAsyncResult passed to your #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "disk_usage"
--           , argType = TBasicType TUInt64
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of bytes of disk space used"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "num_dirs"
--           , argType = TBasicType TUInt64
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of directories encountered"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "num_files"
--           , argType = TBasicType TUInt64
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of non-directories encountered"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_measure_disk_usage_finish" g_file_measure_disk_usage_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr Word64 ->                           -- disk_usage : TBasicType TUInt64
    Ptr Word64 ->                           -- num_dirs : TBasicType TUInt64
    Ptr Word64 ->                           -- num_files : TBasicType TUInt64
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Collects the results from an earlier call to
-- @/g_file_measure_disk_usage_async()/@.  See 'GI.Gio.Interfaces.File.fileMeasureDiskUsage' for
-- more information.
-- 
-- /Since: 2.38/
fileMeasureDiskUsageFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: the t'GI.Gio.Interfaces.AsyncResult.AsyncResult' passed to your t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m ((Word64, Word64, Word64))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMeasureDiskUsageFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    diskUsage <- allocMem :: IO (Ptr Word64)
    numDirs <- allocMem :: IO (Ptr Word64)
    numFiles <- allocMem :: IO (Ptr Word64)
    onException (do
        _ <- propagateGError $ g_file_measure_disk_usage_finish file' result_' diskUsage numDirs numFiles
        diskUsage' <- peek diskUsage
        numDirs' <- peek numDirs
        numFiles' <- peek numFiles
        touchManagedPtr file
        touchManagedPtr result_
        freeMem diskUsage
        freeMem numDirs
        freeMem numFiles
        return (diskUsage', numDirs', numFiles')
     ) (do
        freeMem diskUsage
        freeMem numDirs
        freeMem numFiles
     )

#if defined(ENABLE_OVERLOADING)
data FileMeasureDiskUsageFinishMethodInfo
instance (signature ~ (b -> m ((Word64, Word64, Word64))), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileMeasureDiskUsageFinishMethodInfo a signature where
    overloadedMethod = fileMeasureDiskUsageFinish

instance O.OverloadedMethodInfo FileMeasureDiskUsageFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMeasureDiskUsageFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMeasureDiskUsageFinish"
        })


#endif

-- method File::monitor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileMonitorFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileMonitorFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "FileMonitor" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_monitor" g_file_monitor :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileMonitorFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileMonitor.FileMonitor)

-- | Obtains a file or directory monitor for the given file,
-- depending on the type of the file.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- /Since: 2.18/
fileMonitor ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileMonitorFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileMonitorFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileMonitor.FileMonitor
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileMonitor.FileMonitor' for the given /@file@/,
    --   or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileMonitor file flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_monitor file' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileMonitor" result
        result' <- (wrapObject Gio.FileMonitor.FileMonitor) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMonitorMethodInfo
instance (signature ~ ([Gio.Flags.FileMonitorFlags] -> Maybe (b) -> m Gio.FileMonitor.FileMonitor), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMonitorMethodInfo a signature where
    overloadedMethod = fileMonitor

instance O.OverloadedMethodInfo FileMonitorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMonitor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMonitor"
        })


#endif

-- method File::monitor_directory
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileMonitorFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileMonitorFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "FileMonitor" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_monitor_directory" g_file_monitor_directory :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileMonitorFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileMonitor.FileMonitor)

-- | Obtains a directory monitor for the given file.
-- This may fail if directory monitoring is not supported.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- It does not make sense for /@flags@/ to contain
-- 'GI.Gio.Flags.FileMonitorFlagsWatchHardLinks', since hard links can not be made to
-- directories.  It is not possible to monitor all the files in a
-- directory for changes made via hard links; if you want to do this then
-- you must register individual watches with 'GI.Gio.Interfaces.File.fileMonitor'.
fileMonitorDirectory ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileMonitorFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileMonitorFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileMonitor.FileMonitor
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileMonitor.FileMonitor' for the given /@file@/,
    --   or 'P.Nothing' on error. Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileMonitorDirectory file flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_monitor_directory file' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileMonitorDirectory" result
        result' <- (wrapObject Gio.FileMonitor.FileMonitor) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMonitorDirectoryMethodInfo
instance (signature ~ ([Gio.Flags.FileMonitorFlags] -> Maybe (b) -> m Gio.FileMonitor.FileMonitor), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMonitorDirectoryMethodInfo a signature where
    overloadedMethod = fileMonitorDirectory

instance O.OverloadedMethodInfo FileMonitorDirectoryMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMonitorDirectory",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMonitorDirectory"
        })


#endif

-- method File::monitor_file
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileMonitorFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileMonitorFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "FileMonitor" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_monitor_file" g_file_monitor_file :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileMonitorFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileMonitor.FileMonitor)

-- | Obtains a file monitor for the given file. If no file notification
-- mechanism exists, then regular polling of the file is used.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- If /@flags@/ contains 'GI.Gio.Flags.FileMonitorFlagsWatchHardLinks' then the monitor
-- will also attempt to report changes made to the file via another
-- filename (ie, a hard link). Without this flag, you can only rely on
-- changes made through the filename contained in /@file@/ to be
-- reported. Using this flag may result in an increase in resource
-- usage, and may not have any effect depending on the t'GI.Gio.Objects.FileMonitor.FileMonitor'
-- backend and\/or filesystem type.
fileMonitorFile ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileMonitorFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileMonitorFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileMonitor.FileMonitor
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileMonitor.FileMonitor' for the given /@file@/,
    --   or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileMonitorFile file flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_monitor_file file' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileMonitorFile" result
        result' <- (wrapObject Gio.FileMonitor.FileMonitor) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMonitorFileMethodInfo
instance (signature ~ ([Gio.Flags.FileMonitorFlags] -> Maybe (b) -> m Gio.FileMonitor.FileMonitor), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileMonitorFileMethodInfo a signature where
    overloadedMethod = fileMonitorFile

instance O.OverloadedMethodInfo FileMonitorFileMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMonitorFile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMonitorFile"
        })


#endif

-- method File::mount_enclosing_volume
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "location"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountMountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation\n  or %NULL to avoid user interaction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call\n  when the request is satisfied, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_mount_enclosing_volume" g_file_mount_enclosing_volume :: 
    Ptr File ->                             -- location : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountMountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Starts a /@mountOperation@/, mounting the volume that contains
-- the file /@location@/.
-- 
-- When this operation has completed, /@callback@/ will be called with
-- /@userUser@/ data, and the operation can be finalized with
-- 'GI.Gio.Interfaces.File.fileMountEnclosingVolumeFinish'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileMountEnclosingVolume ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@location@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.MountMountFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation'
    --   or 'P.Nothing' to avoid user interaction
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call
    --   when the request is satisfied, or 'P.Nothing'
    -> m ()
fileMountEnclosingVolume location flags mountOperation cancellable callback = liftIO $ do
    location' <- unsafeManagedPtrCastPtr location
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_mount_enclosing_volume location' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr location
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileMountEnclosingVolumeMethodInfo
instance (signature ~ ([Gio.Flags.MountMountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileMountEnclosingVolumeMethodInfo a signature where
    overloadedMethod = fileMountEnclosingVolume

instance O.OverloadedMethodInfo FileMountEnclosingVolumeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMountEnclosingVolume",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMountEnclosingVolume"
        })


#endif

-- method File::mount_enclosing_volume_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "location"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_mount_enclosing_volume_finish" g_file_mount_enclosing_volume_finish :: 
    Ptr File ->                             -- location : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes a mount operation started by 'GI.Gio.Interfaces.File.fileMountEnclosingVolume'.
fileMountEnclosingVolumeFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@location@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMountEnclosingVolumeFinish location result_ = liftIO $ do
    location' <- unsafeManagedPtrCastPtr location
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_mount_enclosing_volume_finish location' result_'
        touchManagedPtr location
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMountEnclosingVolumeFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileMountEnclosingVolumeFinishMethodInfo a signature where
    overloadedMethod = fileMountEnclosingVolumeFinish

instance O.OverloadedMethodInfo FileMountEnclosingVolumeFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMountEnclosingVolumeFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMountEnclosingVolumeFinish"
        })


#endif

-- method File::mount_mountable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountMountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation,\n  or %NULL to avoid user interaction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_mount_mountable" g_file_mount_mountable :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountMountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Mounts a file of type G_FILE_TYPE_MOUNTABLE.
-- Using /@mountOperation@/, you can request callbacks when, for instance,
-- passwords are needed during authentication.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileMountMountableFinish' to get
-- the result of the operation.
fileMountMountable ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.MountMountFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation',
    --   or 'P.Nothing' to avoid user interaction
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileMountMountable file flags mountOperation cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_mount_mountable file' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileMountMountableMethodInfo
instance (signature ~ ([Gio.Flags.MountMountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileMountMountableMethodInfo a signature where
    overloadedMethod = fileMountMountable

instance O.OverloadedMethodInfo FileMountMountableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMountMountable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMountMountable"
        })


#endif

-- method File::mount_mountable_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_mount_mountable_finish" g_file_mount_mountable_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr File)

-- | Finishes a mount operation. See 'GI.Gio.Interfaces.File.fileMountMountable' for details.
-- 
-- Finish an asynchronous mount operation that was started
-- with 'GI.Gio.Interfaces.File.fileMountMountable'.
fileMountMountableFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m File
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.File.File' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileMountMountableFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_file_mount_mountable_finish file' result_'
        checkUnexpectedReturnNULL "fileMountMountableFinish" result
        result' <- (wrapObject File) result
        touchManagedPtr file
        touchManagedPtr result_
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMountMountableFinishMethodInfo
instance (signature ~ (b -> m File), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileMountMountableFinishMethodInfo a signature where
    overloadedMethod = fileMountMountableFinish

instance O.OverloadedMethodInfo FileMountMountableFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMountMountableFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMountMountableFinish"
        })


#endif

-- method File::move
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "source"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GFile pointing to the source location"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "destination"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GFile pointing to the destination location"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCopyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "set of #GFileCopyFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "progress_callback"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "FileProgressCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "#GFileProgressCallback\n  function for updates"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeCall
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "progress_callback_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "gpointer to user data for\n  the callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_move" g_file_move :: 
    Ptr File ->                             -- source : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- destination : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCopyFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_FileProgressCallback -> -- progress_callback : TInterface (Name {namespace = "Gio", name = "FileProgressCallback"})
    Ptr () ->                               -- progress_callback_data : TBasicType TPtr
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Tries to move the file or directory /@source@/ to the location specified
-- by /@destination@/. If native move operations are supported then this is
-- used, otherwise a copy + delete fallback is used. The native
-- implementation may support moving directories (for instance on moves
-- inside the same filesystem), but the fallback code does not.
-- 
-- If the flag 'GI.Gio.Flags.FileCopyFlagsOverwrite' is specified an already
-- existing /@destination@/ file is overwritten.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- If /@progressCallback@/ is not 'P.Nothing', then the operation can be monitored
-- by setting this to a t'GI.Gio.Callbacks.FileProgressCallback' function.
-- /@progressCallbackData@/ will be passed to this function. It is
-- guaranteed that this callback will be called after all data has been
-- transferred with the total number of bytes copied during the operation.
-- 
-- If the /@source@/ file does not exist, then the 'GI.Gio.Enums.IOErrorEnumNotFound'
-- error is returned, independent on the status of the /@destination@/.
-- 
-- If 'GI.Gio.Flags.FileCopyFlagsOverwrite' is not specified and the target exists,
-- then the error 'GI.Gio.Enums.IOErrorEnumExists' is returned.
-- 
-- If trying to overwrite a file over a directory, the 'GI.Gio.Enums.IOErrorEnumIsDirectory'
-- error is returned. If trying to overwrite a directory with a directory the
-- 'GI.Gio.Enums.IOErrorEnumWouldMerge' error is returned.
-- 
-- If the source is a directory and the target does not exist, or
-- 'GI.Gio.Flags.FileCopyFlagsOverwrite' is specified and the target is a file, then
-- the 'GI.Gio.Enums.IOErrorEnumWouldRecurse' error may be returned (if the native
-- move operation isn\'t available).
fileMove ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@source@/: t'GI.Gio.Interfaces.File.File' pointing to the source location
    -> b
    -- ^ /@destination@/: t'GI.Gio.Interfaces.File.File' pointing to the destination location
    -> [Gio.Flags.FileCopyFlags]
    -- ^ /@flags@/: set of t'GI.Gio.Flags.FileCopyFlags'
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.FileProgressCallback)
    -- ^ /@progressCallback@/: t'GI.Gio.Callbacks.FileProgressCallback'
    --   function for updates
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMove source destination flags cancellable progressCallback = liftIO $ do
    source' <- unsafeManagedPtrCastPtr source
    destination' <- unsafeManagedPtrCastPtr destination
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeProgressCallback <- case progressCallback of
        Nothing -> return FP.nullFunPtr
        Just jProgressCallback -> do
            jProgressCallback' <- Gio.Callbacks.mk_FileProgressCallback (Gio.Callbacks.wrap_FileProgressCallback Nothing (Gio.Callbacks.drop_closures_FileProgressCallback jProgressCallback))
            return jProgressCallback'
    let progressCallbackData = nullPtr
    onException (do
        _ <- propagateGError $ g_file_move source' destination' flags' maybeCancellable maybeProgressCallback progressCallbackData
        safeFreeFunPtr $ castFunPtrToPtr maybeProgressCallback
        touchManagedPtr source
        touchManagedPtr destination
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        safeFreeFunPtr $ castFunPtrToPtr maybeProgressCallback
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMoveMethodInfo
instance (signature ~ (b -> [Gio.Flags.FileCopyFlags] -> Maybe (c) -> Maybe (Gio.Callbacks.FileProgressCallback) -> m ()), MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileMoveMethodInfo a signature where
    overloadedMethod = fileMove

instance O.OverloadedMethodInfo FileMoveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMove",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMove"
        })


#endif

-- method File::move_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "source"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input [type@Gio.File]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "destination"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "destination [type@Gio.File]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCopyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "set of [flags@Gio.FileCopyFlags]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional [class@Gio.Cancellable] object,\n  `NULL` to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "progress_callback_closure"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "[type@GObject.Closure] to invoke with progress\n  information, or `NULL` if progress information is not needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "ready_callback_closure"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "[type@GObject.Closure] to invoke when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_move_async_with_closures" g_file_move_async_with_closures :: 
    Ptr File ->                             -- source : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr File ->                             -- destination : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCopyFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (GClosure ()) ->                    -- progress_callback_closure : TGClosure Nothing
    Ptr (GClosure ()) ->                    -- ready_callback_closure : TGClosure Nothing
    IO ()

-- | Version of 'GI.Gio.Interfaces.File.fileMoveAsync' using closures instead of callbacks for
-- easier binding in other languages.
-- 
-- /Since: 2.82/
fileMoveAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@source@/: input [type/@gio@/.File]
    -> b
    -- ^ /@destination@/: destination [type/@gio@/.File]
    -> [Gio.Flags.FileCopyFlags]
    -- ^ /@flags@/: set of [flags/@gio@/.FileCopyFlags]
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   @NULL@ to ignore
    -> Maybe (GClosure d)
    -- ^ /@progressCallbackClosure@/: [type/@gObject@/.Closure] to invoke with progress
    --   information, or @NULL@ if progress information is not needed
    -> GClosure e
    -- ^ /@readyCallbackClosure@/: [type/@gObject@/.Closure] to invoke when the request is satisfied
    -> m ()
fileMoveAsync source destination flags ioPriority cancellable progressCallbackClosure readyCallbackClosure = liftIO $ do
    source' <- unsafeManagedPtrCastPtr source
    destination' <- unsafeManagedPtrCastPtr destination
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeProgressCallbackClosure <- case progressCallbackClosure of
        Nothing -> return FP.nullPtr
        Just jProgressCallbackClosure -> do
            jProgressCallbackClosure' <- unsafeManagedPtrCastPtr jProgressCallbackClosure
            return jProgressCallbackClosure'
    readyCallbackClosure' <- unsafeManagedPtrCastPtr readyCallbackClosure
    g_file_move_async_with_closures source' destination' flags' ioPriority maybeCancellable maybeProgressCallbackClosure readyCallbackClosure'
    touchManagedPtr source
    touchManagedPtr destination
    whenJust cancellable touchManagedPtr
    whenJust progressCallbackClosure touchManagedPtr
    touchManagedPtr readyCallbackClosure
    return ()

#if defined(ENABLE_OVERLOADING)
data FileMoveAsyncMethodInfo
instance (signature ~ (b -> [Gio.Flags.FileCopyFlags] -> Int32 -> Maybe (c) -> Maybe (GClosure d) -> GClosure e -> m ()), MonadIO m, IsFile a, IsFile b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileMoveAsyncMethodInfo a signature where
    overloadedMethod = fileMoveAsync

instance O.OverloadedMethodInfo FileMoveAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMoveAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMoveAsync"
        })


#endif

-- method File::move_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input source #GFile"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_move_finish" g_file_move_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous file movement, started with
-- @/g_file_move_async()/@.
-- 
-- /Since: 2.72/
fileMoveFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input source t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileMoveFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_move_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileMoveFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileMoveFinishMethodInfo a signature where
    overloadedMethod = fileMoveFinish

instance O.OverloadedMethodInfo FileMoveFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileMoveFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileMoveFinish"
        })


#endif

-- method File::open_readwrite
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GFile to open" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileIOStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_open_readwrite" g_file_open_readwrite :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileIOStream.FileIOStream)

-- | Opens an existing file for reading and writing. The result is
-- a t'GI.Gio.Objects.FileIOStream.FileIOStream' that can be used to read and write the contents
-- of the file.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled
-- by triggering the cancellable object from another thread. If the
-- operation was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be
-- returned.
-- 
-- If the file does not exist, the 'GI.Gio.Enums.IOErrorEnumNotFound' error will
-- be returned. If the file is a directory, the 'GI.Gio.Enums.IOErrorEnumIsDirectory'
-- error will be returned. Other errors are possible too, and depend on
-- what kind of filesystem the file is on. Note that in many non-local
-- file cases read and write streams are not supported, so make sure you
-- really need to do read and write streaming, rather than just opening
-- for reading or writing.
-- 
-- /Since: 2.22/
fileOpenReadwrite ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: t'GI.Gio.Interfaces.File.File' to open
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable'
    -> m Gio.FileIOStream.FileIOStream
    -- ^ __Returns:__ t'GI.Gio.Objects.FileIOStream.FileIOStream' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileOpenReadwrite file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_open_readwrite file' maybeCancellable
        checkUnexpectedReturnNULL "fileOpenReadwrite" result
        result' <- (wrapObject Gio.FileIOStream.FileIOStream) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileOpenReadwriteMethodInfo
instance (signature ~ (Maybe (b) -> m Gio.FileIOStream.FileIOStream), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileOpenReadwriteMethodInfo a signature where
    overloadedMethod = fileOpenReadwrite

instance O.OverloadedMethodInfo FileOpenReadwriteMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileOpenReadwrite",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileOpenReadwrite"
        })


#endif

-- method File::open_readwrite_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_open_readwrite_async" g_file_open_readwrite_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously opens /@file@/ for reading and writing.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileOpenReadwrite' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileOpenReadwriteFinish' to get
-- the result of the operation.
-- 
-- /Since: 2.22/
fileOpenReadwriteAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileOpenReadwriteAsync file ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_open_readwrite_async file' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileOpenReadwriteAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileOpenReadwriteAsyncMethodInfo a signature where
    overloadedMethod = fileOpenReadwriteAsync

instance O.OverloadedMethodInfo FileOpenReadwriteAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileOpenReadwriteAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileOpenReadwriteAsync"
        })


#endif

-- method File::open_readwrite_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileIOStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_open_readwrite_finish" g_file_open_readwrite_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileIOStream.FileIOStream)

-- | Finishes an asynchronous file read operation started with
-- 'GI.Gio.Interfaces.File.fileOpenReadwriteAsync'.
-- 
-- /Since: 2.22/
fileOpenReadwriteFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileIOStream.FileIOStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileIOStream.FileIOStream' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileOpenReadwriteFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_open_readwrite_finish file' res'
        checkUnexpectedReturnNULL "fileOpenReadwriteFinish" result
        result' <- (wrapObject Gio.FileIOStream.FileIOStream) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileOpenReadwriteFinishMethodInfo
instance (signature ~ (b -> m Gio.FileIOStream.FileIOStream), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileOpenReadwriteFinishMethodInfo a signature where
    overloadedMethod = fileOpenReadwriteFinish

instance O.OverloadedMethodInfo FileOpenReadwriteFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileOpenReadwriteFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileOpenReadwriteFinish"
        })


#endif

-- method File::peek_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFileName)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_peek_path" g_file_peek_path :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO CString

-- | Exactly like 'GI.Gio.Interfaces.File.fileGetPath', but caches the result via
-- @/g_object_set_qdata_full()/@.  This is useful for example in C
-- applications which mix @g_file_*@ APIs with native ones.  It
-- also avoids an extra duplicated string when possible, so will be
-- generally more efficient.
-- 
-- This call does no blocking I\/O.
-- 
-- /Since: 2.56/
filePeekPath ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> m (Maybe [Char])
    -- ^ __Returns:__ string containing the t'GI.Gio.Interfaces.File.File'\'s path,
    --   or 'P.Nothing' if no such path exists. The returned string is owned by /@file@/.
filePeekPath file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_peek_path file'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToString result'
        return result''
    touchManagedPtr file
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data FilePeekPathMethodInfo
instance (signature ~ (m (Maybe [Char])), MonadIO m, IsFile a) => O.OverloadedMethod FilePeekPathMethodInfo a signature where
    overloadedMethod = filePeekPath

instance O.OverloadedMethodInfo FilePeekPathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.filePeekPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:filePeekPath"
        })


#endif

-- method File::poll_mountable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call\n  when the request is satisfied, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 3
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_poll_mountable" g_file_poll_mountable :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Polls a file of type 'GI.Gio.Enums.FileTypeMountable'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileMountMountableFinish' to get
-- the result of the operation.
-- 
-- /Since: 2.22/
filePollMountable ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call
    --   when the request is satisfied, or 'P.Nothing'
    -> m ()
filePollMountable file cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_poll_mountable file' maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FilePollMountableMethodInfo
instance (signature ~ (Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FilePollMountableMethodInfo a signature where
    overloadedMethod = filePollMountable

instance O.OverloadedMethodInfo FilePollMountableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.filePollMountable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:filePollMountable"
        })


#endif

-- method File::poll_mountable_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_poll_mountable_finish" g_file_poll_mountable_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes a poll operation. See 'GI.Gio.Interfaces.File.filePollMountable' for details.
-- 
-- Finish an asynchronous poll operation that was polled
-- with 'GI.Gio.Interfaces.File.filePollMountable'.
-- 
-- /Since: 2.22/
filePollMountableFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
filePollMountableFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_poll_mountable_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FilePollMountableFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FilePollMountableFinishMethodInfo a signature where
    overloadedMethod = filePollMountableFinish

instance O.OverloadedMethodInfo FilePollMountableFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.filePollMountableFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:filePollMountableFinish"
        })


#endif

-- method File::query_default_handler
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile to open" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "AppInfo" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_query_default_handler" g_file_query_default_handler :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.AppInfo.AppInfo)

-- | Returns the t'GI.Gio.Interfaces.AppInfo.AppInfo' that is registered as the default
-- application to handle the file specified by /@file@/.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileQueryDefaultHandler ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File' to open
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> m Gio.AppInfo.AppInfo
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.AppInfo.AppInfo' if the handle was found,
    --   'P.Nothing' if there were errors.
    --   When you are done with it, release it with 'GI.GObject.Objects.Object.objectUnref' /(Can throw 'Data.GI.Base.GError.GError')/
fileQueryDefaultHandler file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_query_default_handler file' maybeCancellable
        checkUnexpectedReturnNULL "fileQueryDefaultHandler" result
        result' <- (wrapObject Gio.AppInfo.AppInfo) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileQueryDefaultHandlerMethodInfo
instance (signature ~ (Maybe (b) -> m Gio.AppInfo.AppInfo), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryDefaultHandlerMethodInfo a signature where
    overloadedMethod = fileQueryDefaultHandler

instance O.OverloadedMethodInfo FileQueryDefaultHandlerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryDefaultHandler",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryDefaultHandler"
        })


#endif

-- method File::query_default_handler_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile to open" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GAsyncReadyCallback to call when the request is done"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_query_default_handler_async" g_file_query_default_handler_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Async version of 'GI.Gio.Interfaces.File.fileQueryDefaultHandler'.
-- 
-- /Since: 2.60/
fileQueryDefaultHandlerAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File' to open
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is done
    -> m ()
fileQueryDefaultHandlerAsync file ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_query_default_handler_async file' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileQueryDefaultHandlerAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryDefaultHandlerAsyncMethodInfo a signature where
    overloadedMethod = fileQueryDefaultHandlerAsync

instance O.OverloadedMethodInfo FileQueryDefaultHandlerAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryDefaultHandlerAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryDefaultHandlerAsync"
        })


#endif

-- method File::query_default_handler_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile to open" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "AppInfo" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_query_default_handler_finish" g_file_query_default_handler_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.AppInfo.AppInfo)

-- | Finishes a 'GI.Gio.Interfaces.File.fileQueryDefaultHandlerAsync' operation.
-- 
-- /Since: 2.60/
fileQueryDefaultHandlerFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File' to open
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.AppInfo.AppInfo
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.AppInfo.AppInfo' if the handle was found,
    --   'P.Nothing' if there were errors.
    --   When you are done with it, release it with 'GI.GObject.Objects.Object.objectUnref' /(Can throw 'Data.GI.Base.GError.GError')/
fileQueryDefaultHandlerFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_file_query_default_handler_finish file' result_'
        checkUnexpectedReturnNULL "fileQueryDefaultHandlerFinish" result
        result' <- (wrapObject Gio.AppInfo.AppInfo) result
        touchManagedPtr file
        touchManagedPtr result_
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileQueryDefaultHandlerFinishMethodInfo
instance (signature ~ (b -> m Gio.AppInfo.AppInfo), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileQueryDefaultHandlerFinishMethodInfo a signature where
    overloadedMethod = fileQueryDefaultHandlerFinish

instance O.OverloadedMethodInfo FileQueryDefaultHandlerFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryDefaultHandlerFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryDefaultHandlerFinish"
        })


#endif

-- method File::query_exists
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_query_exists" g_file_query_exists :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    IO CInt

-- | Utility function to check if a particular file exists.
-- 
-- The fallback implementation of this API is using 'GI.Gio.Interfaces.File.fileQueryInfo'
-- and therefore may do blocking I\/O. To asynchronously query the existence
-- of a file, use 'GI.Gio.Interfaces.File.fileQueryInfoAsync'.
-- 
-- Note that in many cases it is <https://en.wikipedia.org/wiki/Time_of_check_to_time_of_use racy to first check for file existence>
-- and then execute something based on the outcome of that, because the
-- file might have been created or removed in between the operations. The
-- general approach to handling that is to not check, but just do the
-- operation and handle the errors as they come.
-- 
-- As an example of race-free checking, take the case of reading a file,
-- and if it doesn\'t exist, creating it. There are two racy versions: read
-- it, and on error create it; and: check if it exists, if not create it.
-- These can both result in two processes creating the file (with perhaps
-- a partially written file as the result). The correct approach is to
-- always try to create the file with 'GI.Gio.Interfaces.File.fileCreate' which will either
-- atomically create the file or fail with a 'GI.Gio.Enums.IOErrorEnumExists' error.
-- 
-- However, in many cases an existence check is useful in a user interface,
-- for instance to make a menu item sensitive\/insensitive, so that you don\'t
-- have to fool users that something is possible and then just show an error
-- dialog. If you do this, you should make sure to also handle the errors
-- that can happen due to races when you execute the operation.
fileQueryExists ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the file exists (and can be detected without error),
    --   'P.False' otherwise (or if cancelled).
fileQueryExists file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    result <- g_file_query_exists file' maybeCancellable
    let result' = (/= 0) result
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data FileQueryExistsMethodInfo
instance (signature ~ (Maybe (b) -> m Bool), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryExistsMethodInfo a signature where
    overloadedMethod = fileQueryExists

instance O.OverloadedMethodInfo FileQueryExistsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryExists",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryExists"
        })


#endif

-- method File::query_file_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a set of #GFileQueryInfoFlags passed to g_file_query_info()"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "FileType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_query_file_type" g_file_query_file_type :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    IO CUInt

-- | Utility function to inspect the t'GI.Gio.Enums.FileType' of a file. This is
-- implemented using 'GI.Gio.Interfaces.File.fileQueryInfo' and as such does blocking I\/O.
-- 
-- The primary use case of this method is to check if a file is
-- a regular file, directory, or symlink.
-- 
-- /Since: 2.18/
fileQueryFileType ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileQueryInfoFlags' passed to 'GI.Gio.Interfaces.File.fileQueryInfo'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.Enums.FileType
    -- ^ __Returns:__ The t'GI.Gio.Enums.FileType' of the file and 'GI.Gio.Enums.FileTypeUnknown'
    --   if the file does not exist
fileQueryFileType file flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    result <- g_file_query_file_type file' flags' maybeCancellable
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data FileQueryFileTypeMethodInfo
instance (signature ~ ([Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m Gio.Enums.FileType), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryFileTypeMethodInfo a signature where
    overloadedMethod = fileQueryFileType

instance O.OverloadedMethodInfo FileQueryFileTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryFileType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryFileType"
        })


#endif

-- method File::query_filesystem_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attributes"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an attribute query string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "FileInfo" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_query_filesystem_info" g_file_query_filesystem_info :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attributes : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileInfo.FileInfo)

-- | Similar to 'GI.Gio.Interfaces.File.fileQueryInfo', but obtains information
-- about the filesystem the /@file@/ is on, rather than the file itself.
-- For instance the amount of space available and the type of
-- the filesystem.
-- 
-- The /@attributes@/ value is a string that specifies the attributes
-- that should be gathered. It is not an error if it\'s not possible
-- to read a particular requested attribute from a file - it just
-- won\'t be set. /@attributes@/ should be a comma-separated list of
-- attributes or attribute wildcards. The wildcard \"*\" means all
-- attributes, and a wildcard like \"filesystem::*\" means all attributes
-- in the filesystem namespace. The standard namespace for filesystem
-- attributes is \"filesystem\". Common attributes of interest are
-- 'GI.Gio.Constants.FILE_ATTRIBUTE_FILESYSTEM_SIZE' (the total size of the filesystem
-- in bytes), 'GI.Gio.Constants.FILE_ATTRIBUTE_FILESYSTEM_FREE' (number of bytes available),
-- and 'GI.Gio.Constants.FILE_ATTRIBUTE_FILESYSTEM_TYPE' (type of the filesystem).
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled
-- by triggering the cancellable object from another thread. If the
-- operation was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be
-- returned.
-- 
-- If the file does not exist, the 'GI.Gio.Enums.IOErrorEnumNotFound' error will
-- be returned. Other errors are possible too, and depend on what
-- kind of filesystem the file is on.
fileQueryFilesystemInfo ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attributes@/: an attribute query string
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileInfo.FileInfo
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileInfo.FileInfo' or 'P.Nothing' if there was an error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileQueryFilesystemInfo file attributes cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attributes' <- textToCString attributes
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_query_filesystem_info file' attributes' maybeCancellable
        checkUnexpectedReturnNULL "fileQueryFilesystemInfo" result
        result' <- (wrapObject Gio.FileInfo.FileInfo) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attributes'
        return result'
     ) (do
        freeMem attributes'
     )

#if defined(ENABLE_OVERLOADING)
data FileQueryFilesystemInfoMethodInfo
instance (signature ~ (T.Text -> Maybe (b) -> m Gio.FileInfo.FileInfo), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryFilesystemInfoMethodInfo a signature where
    overloadedMethod = fileQueryFilesystemInfo

instance O.OverloadedMethodInfo FileQueryFilesystemInfoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryFilesystemInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryFilesystemInfo"
        })


#endif

-- method File::query_filesystem_info_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attributes"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an attribute query string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_query_filesystem_info_async" g_file_query_filesystem_info_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attributes : TBasicType TUTF8
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously gets the requested information about the filesystem
-- that the specified /@file@/ is on. The result is a t'GI.Gio.Objects.FileInfo.FileInfo' object
-- that contains key-value attributes (such as type or size for the
-- file).
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileQueryFilesystemInfo' which is the
-- synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called. You can
-- then call 'GI.Gio.Interfaces.File.fileQueryInfoFinish' to get the result of the
-- operation.
fileQueryFilesystemInfoAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attributes@/: an attribute query string
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileQueryFilesystemInfoAsync file attributes ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attributes' <- textToCString attributes
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_query_filesystem_info_async file' attributes' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    freeMem attributes'
    return ()

#if defined(ENABLE_OVERLOADING)
data FileQueryFilesystemInfoAsyncMethodInfo
instance (signature ~ (T.Text -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryFilesystemInfoAsyncMethodInfo a signature where
    overloadedMethod = fileQueryFilesystemInfoAsync

instance O.OverloadedMethodInfo FileQueryFilesystemInfoAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryFilesystemInfoAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryFilesystemInfoAsync"
        })


#endif

-- method File::query_filesystem_info_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "FileInfo" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_query_filesystem_info_finish" g_file_query_filesystem_info_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileInfo.FileInfo)

-- | Finishes an asynchronous filesystem info query.
-- See 'GI.Gio.Interfaces.File.fileQueryFilesystemInfoAsync'.
fileQueryFilesystemInfoFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileInfo.FileInfo
    -- ^ __Returns:__ t'GI.Gio.Objects.FileInfo.FileInfo' for given /@file@/
    --   or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileQueryFilesystemInfoFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_query_filesystem_info_finish file' res'
        checkUnexpectedReturnNULL "fileQueryFilesystemInfoFinish" result
        result' <- (wrapObject Gio.FileInfo.FileInfo) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileQueryFilesystemInfoFinishMethodInfo
instance (signature ~ (b -> m Gio.FileInfo.FileInfo), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileQueryFilesystemInfoFinishMethodInfo a signature where
    overloadedMethod = fileQueryFilesystemInfoFinish

instance O.OverloadedMethodInfo FileQueryFilesystemInfoFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryFilesystemInfoFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryFilesystemInfoFinish"
        })


#endif

-- method File::query_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attributes"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an attribute query string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "FileInfo" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_query_info" g_file_query_info :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attributes : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileInfo.FileInfo)

-- | Gets the requested information about specified /@file@/.
-- The result is a t'GI.Gio.Objects.FileInfo.FileInfo' object that contains key-value
-- attributes (such as the type or size of the file).
-- 
-- The /@attributes@/ value is a string that specifies the file
-- attributes that should be gathered. It is not an error if
-- it\'s not possible to read a particular requested attribute
-- from a file - it just won\'t be set. /@attributes@/ should be a
-- comma-separated list of attributes or attribute wildcards.
-- The wildcard \"*\" means all attributes, and a wildcard like
-- \"standard::*\" means all attributes in the standard namespace.
-- An example attribute query be \"standard::*,owner[user](#g:signal:user)\".
-- The standard attributes are available as defines, like
-- 'GI.Gio.Constants.FILE_ATTRIBUTE_STANDARD_NAME'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled
-- by triggering the cancellable object from another thread. If the
-- operation was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be
-- returned.
-- 
-- For symlinks, normally the information about the target of the
-- symlink is returned, rather than information about the symlink
-- itself. However if you pass 'GI.Gio.Flags.FileQueryInfoFlagsNofollowSymlinks'
-- in /@flags@/ the information about the symlink itself will be returned.
-- Also, for symlinks that point to non-existing files the information
-- about the symlink itself will be returned.
-- 
-- If the file does not exist, the 'GI.Gio.Enums.IOErrorEnumNotFound' error will be
-- returned. Other errors are possible too, and depend on what kind of
-- filesystem the file is on.
fileQueryInfo ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attributes@/: an attribute query string
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileInfo.FileInfo
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileInfo.FileInfo' for the given /@file@/, or 'P.Nothing'
    --   on error. Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileQueryInfo file attributes flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attributes' <- textToCString attributes
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_query_info file' attributes' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileQueryInfo" result
        result' <- (wrapObject Gio.FileInfo.FileInfo) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attributes'
        return result'
     ) (do
        freeMem attributes'
     )

#if defined(ENABLE_OVERLOADING)
data FileQueryInfoMethodInfo
instance (signature ~ (T.Text -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m Gio.FileInfo.FileInfo), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryInfoMethodInfo a signature where
    overloadedMethod = fileQueryInfo

instance O.OverloadedMethodInfo FileQueryInfoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryInfo"
        })


#endif

-- method File::query_info_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attributes"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an attribute query string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_query_info_async" g_file_query_info_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attributes : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously gets the requested information about specified /@file@/.
-- The result is a t'GI.Gio.Objects.FileInfo.FileInfo' object that contains key-value attributes
-- (such as type or size for the file).
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileQueryInfo' which is the synchronous
-- version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called. You can
-- then call 'GI.Gio.Interfaces.File.fileQueryInfoFinish' to get the result of the operation.
fileQueryInfoAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attributes@/: an attribute query string
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileQueryInfoAsync file attributes flags ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attributes' <- textToCString attributes
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_query_info_async file' attributes' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    freeMem attributes'
    return ()

#if defined(ENABLE_OVERLOADING)
data FileQueryInfoAsyncMethodInfo
instance (signature ~ (T.Text -> [Gio.Flags.FileQueryInfoFlags] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryInfoAsyncMethodInfo a signature where
    overloadedMethod = fileQueryInfoAsync

instance O.OverloadedMethodInfo FileQueryInfoAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryInfoAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryInfoAsync"
        })


#endif

-- method File::query_info_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "FileInfo" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_query_info_finish" g_file_query_info_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileInfo.FileInfo)

-- | Finishes an asynchronous file info query.
-- See 'GI.Gio.Interfaces.File.fileQueryInfoAsync'.
fileQueryInfoFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileInfo.FileInfo
    -- ^ __Returns:__ t'GI.Gio.Objects.FileInfo.FileInfo' for given /@file@/
    --   or 'P.Nothing' on error. Free the returned object with
    --   'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileQueryInfoFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_query_info_finish file' res'
        checkUnexpectedReturnNULL "fileQueryInfoFinish" result
        result' <- (wrapObject Gio.FileInfo.FileInfo) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileQueryInfoFinishMethodInfo
instance (signature ~ (b -> m Gio.FileInfo.FileInfo), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileQueryInfoFinishMethodInfo a signature where
    overloadedMethod = fileQueryInfoFinish

instance O.OverloadedMethodInfo FileQueryInfoFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryInfoFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryInfoFinish"
        })


#endif

-- method File::query_settable_attributes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "FileAttributeInfoList" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_query_settable_attributes" g_file_query_settable_attributes :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileAttributeInfoList.FileAttributeInfoList)

-- | Obtain the list of settable attributes for the file.
-- 
-- Returns the type and full attribute name of all the attributes
-- that can be set on this file. This doesn\'t mean setting it will
-- always succeed though, you might get an access failure, or some
-- specific file may not support a specific attribute.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileQuerySettableAttributes ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileAttributeInfoList.FileAttributeInfoList
    -- ^ __Returns:__ a t'GI.Gio.Structs.FileAttributeInfoList.FileAttributeInfoList' describing the settable attributes.
    --   When you are done with it, release it with
    --   'GI.Gio.Structs.FileAttributeInfoList.fileAttributeInfoListUnref' /(Can throw 'Data.GI.Base.GError.GError')/
fileQuerySettableAttributes file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_query_settable_attributes file' maybeCancellable
        checkUnexpectedReturnNULL "fileQuerySettableAttributes" result
        result' <- (wrapBoxed Gio.FileAttributeInfoList.FileAttributeInfoList) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileQuerySettableAttributesMethodInfo
instance (signature ~ (Maybe (b) -> m Gio.FileAttributeInfoList.FileAttributeInfoList), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQuerySettableAttributesMethodInfo a signature where
    overloadedMethod = fileQuerySettableAttributes

instance O.OverloadedMethodInfo FileQuerySettableAttributesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQuerySettableAttributes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQuerySettableAttributes"
        })


#endif

-- method File::query_writable_namespaces
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "FileAttributeInfoList" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_query_writable_namespaces" g_file_query_writable_namespaces :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileAttributeInfoList.FileAttributeInfoList)

-- | Obtain the list of attribute namespaces where new attributes
-- can be created by a user. An example of this is extended
-- attributes (in the \"xattr\" namespace).
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileQueryWritableNamespaces ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileAttributeInfoList.FileAttributeInfoList
    -- ^ __Returns:__ a t'GI.Gio.Structs.FileAttributeInfoList.FileAttributeInfoList' describing the writable namespaces.
    --   When you are done with it, release it with
    --   'GI.Gio.Structs.FileAttributeInfoList.fileAttributeInfoListUnref' /(Can throw 'Data.GI.Base.GError.GError')/
fileQueryWritableNamespaces file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_query_writable_namespaces file' maybeCancellable
        checkUnexpectedReturnNULL "fileQueryWritableNamespaces" result
        result' <- (wrapBoxed Gio.FileAttributeInfoList.FileAttributeInfoList) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileQueryWritableNamespacesMethodInfo
instance (signature ~ (Maybe (b) -> m Gio.FileAttributeInfoList.FileAttributeInfoList), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileQueryWritableNamespacesMethodInfo a signature where
    overloadedMethod = fileQueryWritableNamespaces

instance O.OverloadedMethodInfo FileQueryWritableNamespacesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileQueryWritableNamespaces",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileQueryWritableNamespaces"
        })


#endif

-- method File::read
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GFile to read" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileInputStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_read" g_file_read :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileInputStream.FileInputStream)

-- | Opens a file for reading. The result is a t'GI.Gio.Objects.FileInputStream.FileInputStream' that
-- can be used to read the contents of the file.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- If the file does not exist, the 'GI.Gio.Enums.IOErrorEnumNotFound' error will be
-- returned. If the file is a directory, the 'GI.Gio.Enums.IOErrorEnumIsDirectory'
-- error will be returned. Other errors are possible too, and depend
-- on what kind of filesystem the file is on.
fileRead ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: t'GI.Gio.Interfaces.File.File' to read
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable'
    -> m Gio.FileInputStream.FileInputStream
    -- ^ __Returns:__ t'GI.Gio.Objects.FileInputStream.FileInputStream' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileRead file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_read file' maybeCancellable
        checkUnexpectedReturnNULL "fileRead" result
        result' <- (wrapObject Gio.FileInputStream.FileInputStream) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileReadMethodInfo
instance (signature ~ (Maybe (b) -> m Gio.FileInputStream.FileInputStream), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReadMethodInfo a signature where
    overloadedMethod = fileRead

instance O.OverloadedMethodInfo FileReadMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileRead",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileRead"
        })


#endif

-- method File::read_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_read_async" g_file_read_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously opens /@file@/ for reading.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileRead' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileReadFinish' to get the result
-- of the operation.
fileReadAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileReadAsync file ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_read_async file' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileReadAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReadAsyncMethodInfo a signature where
    overloadedMethod = fileReadAsync

instance O.OverloadedMethodInfo FileReadAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReadAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReadAsync"
        })


#endif

-- method File::read_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileInputStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_read_finish" g_file_read_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileInputStream.FileInputStream)

-- | Finishes an asynchronous file read operation started with
-- 'GI.Gio.Interfaces.File.fileReadAsync'.
fileReadFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileInputStream.FileInputStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileInputStream.FileInputStream' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileReadFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_read_finish file' res'
        checkUnexpectedReturnNULL "fileReadFinish" result
        result' <- (wrapObject Gio.FileInputStream.FileInputStream) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileReadFinishMethodInfo
instance (signature ~ (b -> m Gio.FileInputStream.FileInputStream), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileReadFinishMethodInfo a signature where
    overloadedMethod = fileReadFinish

instance O.OverloadedMethodInfo FileReadFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReadFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReadFinish"
        })


#endif

-- method File::replace
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "an optional [entity tag](#entity-tags)\n  for the current #GFile, or #NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "make_backup"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if a backup should be created"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileOutputStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_replace" g_file_replace :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- etag : TBasicType TUTF8
    CInt ->                                 -- make_backup : TBasicType TBoolean
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileOutputStream.FileOutputStream)

-- | Returns an output stream for overwriting the file, possibly
-- creating a backup copy of the file first. If the file doesn\'t exist,
-- it will be created.
-- 
-- This will try to replace the file in the safest way possible so
-- that any errors during the writing will not affect an already
-- existing copy of the file. For instance, for local files it
-- may write to a temporary file and then atomically rename over
-- the destination when the stream is closed.
-- 
-- By default files created are generally readable by everyone,
-- but if you pass 'GI.Gio.Flags.FileCreateFlagsPrivate' in /@flags@/ the file
-- will be made readable only to the current user, to the level that
-- is supported on the target filesystem.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled
-- by triggering the cancellable object from another thread. If the
-- operation was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be
-- returned.
-- 
-- If you pass in a non-'P.Nothing' /@etag@/ value and /@file@/ already exists, then
-- this value is compared to the current entity tag of the file, and if
-- they differ an 'GI.Gio.Enums.IOErrorEnumWrongEtag' error is returned. This
-- generally means that the file has been changed since you last read
-- it. You can get the new etag from 'GI.Gio.Objects.FileOutputStream.fileOutputStreamGetEtag'
-- after you\'ve finished writing and closed the t'GI.Gio.Objects.FileOutputStream.FileOutputStream'. When
-- you load a new file you can use 'GI.Gio.Objects.FileInputStream.fileInputStreamQueryInfo' to
-- get the etag of the file.
-- 
-- If /@makeBackup@/ is 'P.True', this function will attempt to make a
-- backup of the current file before overwriting it. If this fails
-- a 'GI.Gio.Enums.IOErrorEnumCantCreateBackup' error will be returned. If you
-- want to replace anyway, try again with /@makeBackup@/ set to 'P.False'.
-- 
-- If the file is a directory the 'GI.Gio.Enums.IOErrorEnumIsDirectory' error will
-- be returned, and if the file is some other form of non-regular file
-- then a 'GI.Gio.Enums.IOErrorEnumNotRegularFile' error will be returned. Some
-- file systems don\'t allow all file names, and may return an
-- 'GI.Gio.Enums.IOErrorEnumInvalidFilename' error, and if the name is to long
-- 'GI.Gio.Enums.IOErrorEnumFilenameTooLong' will be returned. Other errors are
-- possible too, and depend on what kind of filesystem the file is on.
fileReplace ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (T.Text)
    -- ^ /@etag@/: an optional <http://developer.gnome.org/gio/stable/#entity-tags entity tag>
    --   for the current t'GI.Gio.Interfaces.File.File', or @/NULL/@ to ignore
    -> Bool
    -- ^ /@makeBackup@/: 'P.True' if a backup should be created
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileOutputStream.FileOutputStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileOutputStream.FileOutputStream' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileReplace file etag makeBackup flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeEtag <- case etag of
        Nothing -> return FP.nullPtr
        Just jEtag -> do
            jEtag' <- textToCString jEtag
            return jEtag'
    let makeBackup' = (P.fromIntegral . P.fromEnum) makeBackup
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_replace file' maybeEtag makeBackup' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileReplace" result
        result' <- (wrapObject Gio.FileOutputStream.FileOutputStream) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem maybeEtag
        return result'
     ) (do
        freeMem maybeEtag
     )

#if defined(ENABLE_OVERLOADING)
data FileReplaceMethodInfo
instance (signature ~ (Maybe (T.Text) -> Bool -> [Gio.Flags.FileCreateFlags] -> Maybe (b) -> m Gio.FileOutputStream.FileOutputStream), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReplaceMethodInfo a signature where
    overloadedMethod = fileReplace

instance O.OverloadedMethodInfo FileReplaceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplace",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplace"
        })


#endif

-- method File::replace_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "an [entity tag](#entity-tags) for the current #GFile,\n  or %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "make_backup"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if a backup should be created"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 7
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_replace_async" g_file_replace_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- etag : TBasicType TUTF8
    CInt ->                                 -- make_backup : TBasicType TBoolean
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously overwrites the file, replacing the contents,
-- possibly creating a backup copy of the file first.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileReplace' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileReplaceFinish' to get the result
-- of the operation.
fileReplaceAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (T.Text)
    -- ^ /@etag@/: an <http://developer.gnome.org/gio/stable/#entity-tags entity tag> for the current t'GI.Gio.Interfaces.File.File',
    --   or 'P.Nothing' to ignore
    -> Bool
    -- ^ /@makeBackup@/: 'P.True' if a backup should be created
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileReplaceAsync file etag makeBackup flags ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeEtag <- case etag of
        Nothing -> return FP.nullPtr
        Just jEtag -> do
            jEtag' <- textToCString jEtag
            return jEtag'
    let makeBackup' = (P.fromIntegral . P.fromEnum) makeBackup
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_replace_async file' maybeEtag makeBackup' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    freeMem maybeEtag
    return ()

#if defined(ENABLE_OVERLOADING)
data FileReplaceAsyncMethodInfo
instance (signature ~ (Maybe (T.Text) -> Bool -> [Gio.Flags.FileCreateFlags] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReplaceAsyncMethodInfo a signature where
    overloadedMethod = fileReplaceAsync

instance O.OverloadedMethodInfo FileReplaceAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceAsync"
        })


#endif

-- method File::replace_contents
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "contents"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a string containing the new contents for @file"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of @contents in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the old [entity-tag](#entity-tags) for the document,\n  or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "make_backup"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if a backup should be created"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "new_etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location to a new [entity tag](#entity-tags)\n  for the document. This should be freed with g_free() when no longer\n  needed, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of @contents in bytes"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_replace_contents" g_file_replace_contents :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Word8 ->                            -- contents : TCArray False (-1) 2 (TBasicType TUInt8)
    FCT.CSize ->                            -- length : TBasicType TSize
    CString ->                              -- etag : TBasicType TUTF8
    CInt ->                                 -- make_backup : TBasicType TBoolean
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Ptr CString ->                          -- new_etag : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Replaces the contents of /@file@/ with /@contents@/ of /@length@/ bytes.
-- 
-- If /@etag@/ is specified (not 'P.Nothing'), any existing file must have that etag,
-- or the error 'GI.Gio.Enums.IOErrorEnumWrongEtag' will be returned.
-- 
-- If /@makeBackup@/ is 'P.True', this function will attempt to make a backup
-- of /@file@/. Internally, it uses 'GI.Gio.Interfaces.File.fileReplace', so will try to replace the
-- file contents in the safest way possible. For example, atomic renames are
-- used when replacing local files’ contents.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- The returned /@newEtag@/ can be used to verify that the file hasn\'t
-- changed the next time it is saved over.
fileReplaceContents ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> ByteString
    -- ^ /@contents@/: a string containing the new contents for /@file@/
    -> Maybe (T.Text)
    -- ^ /@etag@/: the old <http://developer.gnome.org/gio/stable/#entity-tags entity-tag> for the document,
    --   or 'P.Nothing'
    -> Bool
    -- ^ /@makeBackup@/: 'P.True' if a backup should be created
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> m ((Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileReplaceContents file contents etag makeBackup flags cancellable = liftIO $ do
    let length_ = fromIntegral $ B.length contents
    file' <- unsafeManagedPtrCastPtr file
    contents' <- packByteString contents
    maybeEtag <- case etag of
        Nothing -> return FP.nullPtr
        Just jEtag -> do
            jEtag' <- textToCString jEtag
            return jEtag'
    let makeBackup' = (P.fromIntegral . P.fromEnum) makeBackup
    let flags' = gflagsToWord flags
    newEtag <- callocMem :: IO (Ptr CString)
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_replace_contents file' contents' length_ maybeEtag makeBackup' flags' newEtag maybeCancellable
        newEtag' <- peek newEtag
        maybeNewEtag' <- convertIfNonNull newEtag' $ \newEtag'' -> do
            newEtag''' <- cstringToText newEtag''
            return newEtag'''
        freeMem newEtag'
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem contents'
        freeMem maybeEtag
        freeMem newEtag
        return maybeNewEtag'
     ) (do
        freeMem contents'
        freeMem maybeEtag
        freeMem newEtag
     )

#if defined(ENABLE_OVERLOADING)
data FileReplaceContentsMethodInfo
instance (signature ~ (ByteString -> Maybe (T.Text) -> Bool -> [Gio.Flags.FileCreateFlags] -> Maybe (b) -> m ((Maybe T.Text))), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReplaceContentsMethodInfo a signature where
    overloadedMethod = fileReplaceContents

instance O.OverloadedMethodInfo FileReplaceContentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceContents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceContents"
        })


#endif

-- method File::replace_contents_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "contents"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "string of contents to replace the file with"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of @contents in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a new [entity tag](#entity-tags) for the @file, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "make_backup"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if a backup should be created"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GAsyncReadyCallback to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 8
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of @contents in bytes"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_replace_contents_async" g_file_replace_contents_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Word8 ->                            -- contents : TCArray False (-1) 2 (TBasicType TUInt8)
    FCT.CSize ->                            -- length : TBasicType TSize
    CString ->                              -- etag : TBasicType TUTF8
    CInt ->                                 -- make_backup : TBasicType TBoolean
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Starts an asynchronous replacement of /@file@/ with the given
-- /@contents@/ of /@length@/ bytes. /@etag@/ will replace the document\'s
-- current entity tag.
-- 
-- When this operation has completed, /@callback@/ will be called with
-- /@userUser@/ data, and the operation can be finalized with
-- 'GI.Gio.Interfaces.File.fileReplaceContentsFinish'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- If /@makeBackup@/ is 'P.True', this function will attempt to
-- make a backup of /@file@/.
-- 
-- Note that no copy of /@contents@/ will be made, so it must stay valid
-- until /@callback@/ is called. See 'GI.Gio.Interfaces.File.fileReplaceContentsBytesAsync'
-- for a t'GI.GLib.Structs.Bytes.Bytes' version that will automatically hold a reference to the
-- contents (without copying) for the duration of the call.
fileReplaceContentsAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> ByteString
    -- ^ /@contents@/: string of contents to replace the file with
    -> Maybe (T.Text)
    -- ^ /@etag@/: a new <http://developer.gnome.org/gio/stable/#entity-tags entity tag> for the /@file@/, or 'P.Nothing'
    -> Bool
    -- ^ /@makeBackup@/: 'P.True' if a backup should be created
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is satisfied
    -> m ()
fileReplaceContentsAsync file contents etag makeBackup flags cancellable callback = liftIO $ do
    let length_ = fromIntegral $ B.length contents
    file' <- unsafeManagedPtrCastPtr file
    contents' <- packByteString contents
    maybeEtag <- case etag of
        Nothing -> return FP.nullPtr
        Just jEtag -> do
            jEtag' <- textToCString jEtag
            return jEtag'
    let makeBackup' = (P.fromIntegral . P.fromEnum) makeBackup
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_replace_contents_async file' contents' length_ maybeEtag makeBackup' flags' maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    freeMem contents'
    freeMem maybeEtag
    return ()

#if defined(ENABLE_OVERLOADING)
data FileReplaceContentsAsyncMethodInfo
instance (signature ~ (ByteString -> Maybe (T.Text) -> Bool -> [Gio.Flags.FileCreateFlags] -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReplaceContentsAsyncMethodInfo a signature where
    overloadedMethod = fileReplaceContentsAsync

instance O.OverloadedMethodInfo FileReplaceContentsAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceContentsAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceContentsAsync"
        })


#endif

-- method File::replace_contents_bytes_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "contents"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBytes" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a new [entity tag](#entity-tags) for the @file, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "make_backup"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if a backup should be created"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GAsyncReadyCallback to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 7
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_replace_contents_bytes_async" g_file_replace_contents_bytes_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr GLib.Bytes.Bytes ->                 -- contents : TInterface (Name {namespace = "GLib", name = "Bytes"})
    CString ->                              -- etag : TBasicType TUTF8
    CInt ->                                 -- make_backup : TBasicType TBoolean
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Same as 'GI.Gio.Interfaces.File.fileReplaceContentsAsync' but takes a t'GI.GLib.Structs.Bytes.Bytes' input instead.
-- This function will keep a ref on /@contents@/ until the operation is done.
-- Unlike 'GI.Gio.Interfaces.File.fileReplaceContentsAsync' this allows forgetting about the
-- content without waiting for the callback.
-- 
-- When this operation has completed, /@callback@/ will be called with
-- /@userUser@/ data, and the operation can be finalized with
-- 'GI.Gio.Interfaces.File.fileReplaceContentsFinish'.
-- 
-- /Since: 2.40/
fileReplaceContentsBytesAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> GLib.Bytes.Bytes
    -- ^ /@contents@/: a t'GI.GLib.Structs.Bytes.Bytes'
    -> Maybe (T.Text)
    -- ^ /@etag@/: a new <http://developer.gnome.org/gio/stable/#entity-tags entity tag> for the /@file@/, or 'P.Nothing'
    -> Bool
    -- ^ /@makeBackup@/: 'P.True' if a backup should be created
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is satisfied
    -> m ()
fileReplaceContentsBytesAsync file contents etag makeBackup flags cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    contents' <- unsafeManagedPtrGetPtr contents
    maybeEtag <- case etag of
        Nothing -> return FP.nullPtr
        Just jEtag -> do
            jEtag' <- textToCString jEtag
            return jEtag'
    let makeBackup' = (P.fromIntegral . P.fromEnum) makeBackup
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_replace_contents_bytes_async file' contents' maybeEtag makeBackup' flags' maybeCancellable maybeCallback userData
    touchManagedPtr file
    touchManagedPtr contents
    whenJust cancellable touchManagedPtr
    freeMem maybeEtag
    return ()

#if defined(ENABLE_OVERLOADING)
data FileReplaceContentsBytesAsyncMethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> Maybe (T.Text) -> Bool -> [Gio.Flags.FileCreateFlags] -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReplaceContentsBytesAsyncMethodInfo a signature where
    overloadedMethod = fileReplaceContentsBytesAsync

instance O.OverloadedMethodInfo FileReplaceContentsBytesAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceContentsBytesAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceContentsBytesAsync"
        })


#endif

-- method File::replace_contents_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "new_etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a location of a new [entity tag](#entity-tags)\n  for the document. This should be freed with g_free() when it is no\n  longer needed, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_replace_contents_finish" g_file_replace_contents_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr CString ->                          -- new_etag : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous replace of the given /@file@/. See
-- 'GI.Gio.Interfaces.File.fileReplaceContentsAsync'. Sets /@newEtag@/ to the new entity
-- tag for the document, if present.
fileReplaceContentsFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ((Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileReplaceContentsFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    newEtag <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_file_replace_contents_finish file' res' newEtag
        newEtag' <- peek newEtag
        maybeNewEtag' <- convertIfNonNull newEtag' $ \newEtag'' -> do
            newEtag''' <- cstringToText newEtag''
            return newEtag'''
        freeMem newEtag'
        touchManagedPtr file
        touchManagedPtr res
        freeMem newEtag
        return maybeNewEtag'
     ) (do
        freeMem newEtag
     )

#if defined(ENABLE_OVERLOADING)
data FileReplaceContentsFinishMethodInfo
instance (signature ~ (b -> m ((Maybe T.Text))), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileReplaceContentsFinishMethodInfo a signature where
    overloadedMethod = fileReplaceContentsFinish

instance O.OverloadedMethodInfo FileReplaceContentsFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceContentsFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceContentsFinish"
        })


#endif

-- method File::replace_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileOutputStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_replace_finish" g_file_replace_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileOutputStream.FileOutputStream)

-- | Finishes an asynchronous file replace operation started with
-- 'GI.Gio.Interfaces.File.fileReplaceAsync'.
fileReplaceFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileOutputStream.FileOutputStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileOutputStream.FileOutputStream', or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileReplaceFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_replace_finish file' res'
        checkUnexpectedReturnNULL "fileReplaceFinish" result
        result' <- (wrapObject Gio.FileOutputStream.FileOutputStream) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileReplaceFinishMethodInfo
instance (signature ~ (b -> m Gio.FileOutputStream.FileOutputStream), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileReplaceFinishMethodInfo a signature where
    overloadedMethod = fileReplaceFinish

instance O.OverloadedMethodInfo FileReplaceFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceFinish"
        })


#endif

-- method File::replace_readwrite
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "an optional [entity tag](#entity-tags)\n  for the current #GFile, or #NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "make_backup"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if a backup should be created"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileIOStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_replace_readwrite" g_file_replace_readwrite :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- etag : TBasicType TUTF8
    CInt ->                                 -- make_backup : TBasicType TBoolean
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileIOStream.FileIOStream)

-- | Returns an output stream for overwriting the file in readwrite mode,
-- possibly creating a backup copy of the file first. If the file doesn\'t
-- exist, it will be created.
-- 
-- For details about the behaviour, see 'GI.Gio.Interfaces.File.fileReplace' which does the
-- same thing but returns an output stream only.
-- 
-- Note that in many non-local file cases read and write streams are not
-- supported, so make sure you really need to do read and write streaming,
-- rather than just opening for reading or writing.
-- 
-- /Since: 2.22/
fileReplaceReadwrite ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> Maybe (T.Text)
    -- ^ /@etag@/: an optional <http://developer.gnome.org/gio/stable/#entity-tags entity tag>
    --   for the current t'GI.Gio.Interfaces.File.File', or @/NULL/@ to ignore
    -> Bool
    -- ^ /@makeBackup@/: 'P.True' if a backup should be created
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m Gio.FileIOStream.FileIOStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileIOStream.FileIOStream' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileReplaceReadwrite file etag makeBackup flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeEtag <- case etag of
        Nothing -> return FP.nullPtr
        Just jEtag -> do
            jEtag' <- textToCString jEtag
            return jEtag'
    let makeBackup' = (P.fromIntegral . P.fromEnum) makeBackup
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_replace_readwrite file' maybeEtag makeBackup' flags' maybeCancellable
        checkUnexpectedReturnNULL "fileReplaceReadwrite" result
        result' <- (wrapObject Gio.FileIOStream.FileIOStream) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem maybeEtag
        return result'
     ) (do
        freeMem maybeEtag
     )

#if defined(ENABLE_OVERLOADING)
data FileReplaceReadwriteMethodInfo
instance (signature ~ (Maybe (T.Text) -> Bool -> [Gio.Flags.FileCreateFlags] -> Maybe (b) -> m Gio.FileIOStream.FileIOStream), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReplaceReadwriteMethodInfo a signature where
    overloadedMethod = fileReplaceReadwrite

instance O.OverloadedMethodInfo FileReplaceReadwriteMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceReadwrite",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceReadwrite"
        })


#endif

-- method File::replace_readwrite_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "etag"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "an [entity tag](#entity-tags) for the current #GFile,\n  or %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "make_backup"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE if a backup should be created"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileCreateFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileCreateFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 7
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_replace_readwrite_async" g_file_replace_readwrite_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- etag : TBasicType TUTF8
    CInt ->                                 -- make_backup : TBasicType TBoolean
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileCreateFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously overwrites the file in read-write mode,
-- replacing the contents, possibly creating a backup copy
-- of the file first.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileReplaceReadwrite' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileReplaceReadwriteFinish' to get
-- the result of the operation.
-- 
-- /Since: 2.22/
fileReplaceReadwriteAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Maybe (T.Text)
    -- ^ /@etag@/: an <http://developer.gnome.org/gio/stable/#entity-tags entity tag> for the current t'GI.Gio.Interfaces.File.File',
    --   or 'P.Nothing' to ignore
    -> Bool
    -- ^ /@makeBackup@/: 'P.True' if a backup should be created
    -> [Gio.Flags.FileCreateFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileCreateFlags'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileReplaceReadwriteAsync file etag makeBackup flags ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeEtag <- case etag of
        Nothing -> return FP.nullPtr
        Just jEtag -> do
            jEtag' <- textToCString jEtag
            return jEtag'
    let makeBackup' = (P.fromIntegral . P.fromEnum) makeBackup
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_replace_readwrite_async file' maybeEtag makeBackup' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    freeMem maybeEtag
    return ()

#if defined(ENABLE_OVERLOADING)
data FileReplaceReadwriteAsyncMethodInfo
instance (signature ~ (Maybe (T.Text) -> Bool -> [Gio.Flags.FileCreateFlags] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileReplaceReadwriteAsyncMethodInfo a signature where
    overloadedMethod = fileReplaceReadwriteAsync

instance O.OverloadedMethodInfo FileReplaceReadwriteAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceReadwriteAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceReadwriteAsync"
        })


#endif

-- method File::replace_readwrite_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "FileIOStream" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_replace_readwrite_finish" g_file_replace_readwrite_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Gio.FileIOStream.FileIOStream)

-- | Finishes an asynchronous file replace operation started with
-- 'GI.Gio.Interfaces.File.fileReplaceReadwriteAsync'.
-- 
-- /Since: 2.22/
fileReplaceReadwriteFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m Gio.FileIOStream.FileIOStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.FileIOStream.FileIOStream', or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileReplaceReadwriteFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_replace_readwrite_finish file' res'
        checkUnexpectedReturnNULL "fileReplaceReadwriteFinish" result
        result' <- (wrapObject Gio.FileIOStream.FileIOStream) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileReplaceReadwriteFinishMethodInfo
instance (signature ~ (b -> m Gio.FileIOStream.FileIOStream), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileReplaceReadwriteFinishMethodInfo a signature where
    overloadedMethod = fileReplaceReadwriteFinish

instance O.OverloadedMethodInfo FileReplaceReadwriteFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileReplaceReadwriteFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileReplaceReadwriteFinish"
        })


#endif

-- method File::resolve_relative_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "relative_path"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a given relative path string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_resolve_relative_path" g_file_resolve_relative_path :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- relative_path : TBasicType TFileName
    IO (Ptr File)

-- | Resolves a relative path for /@file@/ to an absolute path.
-- 
-- This call does no blocking I\/O.
-- 
-- If the /@relativePath@/ is an absolute path name, the resolution
-- is done absolutely (without taking /@file@/ path as base).
fileResolveRelativePath ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Char]
    -- ^ /@relativePath@/: a given relative path string
    -> m File
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.File.File' for the resolved path.
fileResolveRelativePath file relativePath = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    relativePath' <- stringToCString relativePath
    result <- g_file_resolve_relative_path file' relativePath'
    checkUnexpectedReturnNULL "fileResolveRelativePath" result
    result' <- (wrapObject File) result
    touchManagedPtr file
    freeMem relativePath'
    return result'

#if defined(ENABLE_OVERLOADING)
data FileResolveRelativePathMethodInfo
instance (signature ~ ([Char] -> m File), MonadIO m, IsFile a) => O.OverloadedMethod FileResolveRelativePathMethodInfo a signature where
    overloadedMethod = fileResolveRelativePath

instance O.OverloadedMethodInfo FileResolveRelativePathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileResolveRelativePath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileResolveRelativePath"
        })


#endif

-- method File::set_attribute
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attribute"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileAttributeType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The type of the attribute"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value_p"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a pointer to the value (or the pointer\n  itself if the type is a pointer type)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attribute" g_file_set_attribute :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attribute : TBasicType TUTF8
    CUInt ->                                -- type : TInterface (Name {namespace = "Gio", name = "FileAttributeType"})
    Ptr () ->                               -- value_p : TBasicType TPtr
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets an attribute in the file with attribute name /@attribute@/ to /@valueP@/.
-- 
-- Some attributes can be unset by setting /@type@/ to
-- 'GI.Gio.Enums.FileAttributeTypeInvalid' and /@valueP@/ to 'P.Nothing'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetAttribute ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attribute@/: a string containing the attribute\'s name
    -> Gio.Enums.FileAttributeType
    -- ^ /@type@/: The type of the attribute
    -> Ptr ()
    -- ^ /@valueP@/: a pointer to the value (or the pointer
    --   itself if the type is a pointer type)
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttribute file attribute type_ valueP flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attribute' <- textToCString attribute
    let type_' = (fromIntegral . fromEnum) type_
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_set_attribute file' attribute' type_' valueP flags' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attribute'
        return ()
     ) (do
        freeMem attribute'
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributeMethodInfo
instance (signature ~ (T.Text -> Gio.Enums.FileAttributeType -> Ptr () -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetAttributeMethodInfo a signature where
    overloadedMethod = fileSetAttribute

instance O.OverloadedMethodInfo FileSetAttributeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttribute",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttribute"
        })


#endif

-- method File::set_attribute_byte_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attribute"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's new value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attribute_byte_string" g_file_set_attribute_byte_string :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attribute : TBasicType TUTF8
    CString ->                              -- value : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets /@attribute@/ of type 'GI.Gio.Enums.FileAttributeTypeByteString' to /@value@/.
-- If /@attribute@/ is of a different type, this operation will fail,
-- returning 'P.False'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetAttributeByteString ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attribute@/: a string containing the attribute\'s name
    -> T.Text
    -- ^ /@value@/: a string containing the attribute\'s new value
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttributeByteString file attribute value flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attribute' <- textToCString attribute
    value' <- textToCString value
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_set_attribute_byte_string file' attribute' value' flags' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attribute'
        freeMem value'
        return ()
     ) (do
        freeMem attribute'
        freeMem value'
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributeByteStringMethodInfo
instance (signature ~ (T.Text -> T.Text -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetAttributeByteStringMethodInfo a signature where
    overloadedMethod = fileSetAttributeByteString

instance O.OverloadedMethodInfo FileSetAttributeByteStringMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributeByteString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributeByteString"
        })


#endif

-- method File::set_attribute_int32
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attribute"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TInt32
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #gint32 containing the attribute's new value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attribute_int32" g_file_set_attribute_int32 :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attribute : TBasicType TUTF8
    Int32 ->                                -- value : TBasicType TInt32
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets /@attribute@/ of type 'GI.Gio.Enums.FileAttributeTypeInt32' to /@value@/.
-- If /@attribute@/ is of a different type, this operation will fail.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetAttributeInt32 ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attribute@/: a string containing the attribute\'s name
    -> Int32
    -- ^ /@value@/: a @/gint32/@ containing the attribute\'s new value
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttributeInt32 file attribute value flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attribute' <- textToCString attribute
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_set_attribute_int32 file' attribute' value flags' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attribute'
        return ()
     ) (do
        freeMem attribute'
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributeInt32MethodInfo
instance (signature ~ (T.Text -> Int32 -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetAttributeInt32MethodInfo a signature where
    overloadedMethod = fileSetAttributeInt32

instance O.OverloadedMethodInfo FileSetAttributeInt32MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributeInt32",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributeInt32"
        })


#endif

-- method File::set_attribute_int64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attribute"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #guint64 containing the attribute's new value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attribute_int64" g_file_set_attribute_int64 :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attribute : TBasicType TUTF8
    Int64 ->                                -- value : TBasicType TInt64
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets /@attribute@/ of type 'GI.Gio.Enums.FileAttributeTypeInt64' to /@value@/.
-- If /@attribute@/ is of a different type, this operation will fail.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetAttributeInt64 ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attribute@/: a string containing the attribute\'s name
    -> Int64
    -- ^ /@value@/: a @/guint64/@ containing the attribute\'s new value
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttributeInt64 file attribute value flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attribute' <- textToCString attribute
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_set_attribute_int64 file' attribute' value flags' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attribute'
        return ()
     ) (do
        freeMem attribute'
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributeInt64MethodInfo
instance (signature ~ (T.Text -> Int64 -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetAttributeInt64MethodInfo a signature where
    overloadedMethod = fileSetAttributeInt64

instance O.OverloadedMethodInfo FileSetAttributeInt64MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributeInt64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributeInt64"
        })


#endif

-- method File::set_attribute_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attribute"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attribute_string" g_file_set_attribute_string :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attribute : TBasicType TUTF8
    CString ->                              -- value : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets /@attribute@/ of type 'GI.Gio.Enums.FileAttributeTypeString' to /@value@/.
-- If /@attribute@/ is of a different type, this operation will fail.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetAttributeString ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attribute@/: a string containing the attribute\'s name
    -> T.Text
    -- ^ /@value@/: a string containing the attribute\'s value
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttributeString file attribute value flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attribute' <- textToCString attribute
    value' <- textToCString value
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_set_attribute_string file' attribute' value' flags' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attribute'
        freeMem value'
        return ()
     ) (do
        freeMem attribute'
        freeMem value'
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributeStringMethodInfo
instance (signature ~ (T.Text -> T.Text -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetAttributeStringMethodInfo a signature where
    overloadedMethod = fileSetAttributeString

instance O.OverloadedMethodInfo FileSetAttributeStringMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributeString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributeString"
        })


#endif

-- method File::set_attribute_uint32
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attribute"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUInt32
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #guint32 containing the attribute's new value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attribute_uint32" g_file_set_attribute_uint32 :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attribute : TBasicType TUTF8
    Word32 ->                               -- value : TBasicType TUInt32
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets /@attribute@/ of type 'GI.Gio.Enums.FileAttributeTypeUint32' to /@value@/.
-- If /@attribute@/ is of a different type, this operation will fail.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetAttributeUint32 ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attribute@/: a string containing the attribute\'s name
    -> Word32
    -- ^ /@value@/: a @/guint32/@ containing the attribute\'s new value
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttributeUint32 file attribute value flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attribute' <- textToCString attribute
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_set_attribute_uint32 file' attribute' value flags' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attribute'
        return ()
     ) (do
        freeMem attribute'
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributeUint32MethodInfo
instance (signature ~ (T.Text -> Word32 -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetAttributeUint32MethodInfo a signature where
    overloadedMethod = fileSetAttributeUint32

instance O.OverloadedMethodInfo FileSetAttributeUint32MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributeUint32",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributeUint32"
        })


#endif

-- method File::set_attribute_uint64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attribute"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing the attribute's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #guint64 containing the attribute's new value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attribute_uint64" g_file_set_attribute_uint64 :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- attribute : TBasicType TUTF8
    Word64 ->                               -- value : TBasicType TUInt64
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets /@attribute@/ of type 'GI.Gio.Enums.FileAttributeTypeUint64' to /@value@/.
-- If /@attribute@/ is of a different type, this operation will fail.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetAttributeUint64 ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@attribute@/: a string containing the attribute\'s name
    -> Word64
    -- ^ /@value@/: a @/guint64/@ containing the attribute\'s new value
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttributeUint64 file attribute value flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    attribute' <- textToCString attribute
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_set_attribute_uint64 file' attribute' value flags' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem attribute'
        return ()
     ) (do
        freeMem attribute'
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributeUint64MethodInfo
instance (signature ~ (T.Text -> Word64 -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetAttributeUint64MethodInfo a signature where
    overloadedMethod = fileSetAttributeUint64

instance O.OverloadedMethodInfo FileSetAttributeUint64MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributeUint64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributeUint64"
        })


#endif

-- method File::set_attributes_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "info"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileInfo" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileInfo" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_set_attributes_async" g_file_set_attributes_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.FileInfo.FileInfo ->            -- info : TInterface (Name {namespace = "Gio", name = "FileInfo"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously sets the attributes of /@file@/ with /@info@/.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileSetAttributesFromInfo',
-- which is the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileSetAttributesFinish' to get
-- the result of the operation.
fileSetAttributesAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.FileInfo.IsFileInfo b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@info@/: a t'GI.Gio.Objects.FileInfo.FileInfo'
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: a t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileSetAttributesAsync file info flags ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    info' <- unsafeManagedPtrCastPtr info
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_set_attributes_async file' info' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    touchManagedPtr info
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileSetAttributesAsyncMethodInfo
instance (signature ~ (b -> [Gio.Flags.FileQueryInfoFlags] -> Int32 -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.FileInfo.IsFileInfo b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileSetAttributesAsyncMethodInfo a signature where
    overloadedMethod = fileSetAttributesAsync

instance O.OverloadedMethodInfo FileSetAttributesAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributesAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributesAsync"
        })


#endif

-- method File::set_attributes_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "info"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileInfo" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileInfo" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attributes_finish" g_file_set_attributes_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr Gio.FileInfo.FileInfo) ->      -- info : TInterface (Name {namespace = "Gio", name = "FileInfo"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes setting an attribute started in 'GI.Gio.Interfaces.File.fileSetAttributesAsync'.
fileSetAttributesFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m (Gio.FileInfo.FileInfo)
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttributesFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    info <- callocMem :: IO (Ptr (Ptr Gio.FileInfo.FileInfo))
    onException (do
        _ <- propagateGError $ g_file_set_attributes_finish file' result_' info
        info' <- peek info
        info'' <- (wrapObject Gio.FileInfo.FileInfo) info'
        touchManagedPtr file
        touchManagedPtr result_
        freeMem info
        return info''
     ) (do
        freeMem info
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributesFinishMethodInfo
instance (signature ~ (b -> m (Gio.FileInfo.FileInfo)), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileSetAttributesFinishMethodInfo a signature where
    overloadedMethod = fileSetAttributesFinish

instance O.OverloadedMethodInfo FileSetAttributesFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributesFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributesFinish"
        })


#endif

-- method File::set_attributes_from_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "info"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileInfo" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFileInfo" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileQueryInfoFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GFileQueryInfoFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_attributes_from_info" g_file_set_attributes_from_info :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.FileInfo.FileInfo ->            -- info : TInterface (Name {namespace = "Gio", name = "FileInfo"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "FileQueryInfoFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Tries to set all attributes in the t'GI.Gio.Objects.FileInfo.FileInfo' on the target
-- values, not stopping on the first error.
-- 
-- If there is any error during this operation then /@error@/ will
-- be set to the first error. Error on particular fields are flagged
-- by setting the \"status\" field in the attribute value to
-- 'GI.Gio.Enums.FileAttributeStatusErrorSetting', which means you can
-- also detect further errors.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetAttributesFromInfo ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.FileInfo.IsFileInfo b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@info@/: a t'GI.Gio.Objects.FileInfo.FileInfo'
    -> [Gio.Flags.FileQueryInfoFlags]
    -- ^ /@flags@/: t'GI.Gio.Flags.FileQueryInfoFlags'
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileSetAttributesFromInfo file info flags cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    info' <- unsafeManagedPtrCastPtr info
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_set_attributes_from_info file' info' flags' maybeCancellable
        touchManagedPtr file
        touchManagedPtr info
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileSetAttributesFromInfoMethodInfo
instance (signature ~ (b -> [Gio.Flags.FileQueryInfoFlags] -> Maybe (c) -> m ()), MonadIO m, IsFile a, Gio.FileInfo.IsFileInfo b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileSetAttributesFromInfoMethodInfo a signature where
    overloadedMethod = fileSetAttributesFromInfo

instance O.OverloadedMethodInfo FileSetAttributesFromInfoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetAttributesFromInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetAttributesFromInfo"
        })


#endif

-- method File::set_display_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "display_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_display_name" g_file_set_display_name :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- display_name : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr File)

-- | Renames /@file@/ to the specified display name.
-- 
-- The display name is converted from UTF-8 to the correct encoding
-- for the target filesystem if possible and the /@file@/ is renamed to this.
-- 
-- If you want to implement a rename operation in the user interface the
-- edit name ('GI.Gio.Constants.FILE_ATTRIBUTE_STANDARD_EDIT_NAME') should be used as the
-- initial value in the rename widget, and then the result after editing
-- should be passed to 'GI.Gio.Interfaces.File.fileSetDisplayName'.
-- 
-- On success the resulting converted filename is returned.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileSetDisplayName ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@displayName@/: a string
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m File
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.File.File' specifying what /@file@/ was renamed to,
    --   or 'P.Nothing' if there was an error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileSetDisplayName file displayName cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    displayName' <- textToCString displayName
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_file_set_display_name file' displayName' maybeCancellable
        checkUnexpectedReturnNULL "fileSetDisplayName" result
        result' <- (wrapObject File) result
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        freeMem displayName'
        return result'
     ) (do
        freeMem displayName'
     )

#if defined(ENABLE_OVERLOADING)
data FileSetDisplayNameMethodInfo
instance (signature ~ (T.Text -> Maybe (b) -> m File), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetDisplayNameMethodInfo a signature where
    overloadedMethod = fileSetDisplayName

instance O.OverloadedMethodInfo FileSetDisplayNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetDisplayName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetDisplayName"
        })


#endif

-- method File::set_display_name_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "display_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_set_display_name_async" g_file_set_display_name_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CString ->                              -- display_name : TBasicType TUTF8
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously sets the display name for a given t'GI.Gio.Interfaces.File.File'.
-- 
-- For more details, see 'GI.Gio.Interfaces.File.fileSetDisplayName' which is
-- the synchronous version of this call.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileSetDisplayNameFinish' to get
-- the result of the operation.
fileSetDisplayNameAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> T.Text
    -- ^ /@displayName@/: a string
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileSetDisplayNameAsync file displayName ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    displayName' <- textToCString displayName
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_set_display_name_async file' displayName' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    freeMem displayName'
    return ()

#if defined(ENABLE_OVERLOADING)
data FileSetDisplayNameAsyncMethodInfo
instance (signature ~ (T.Text -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileSetDisplayNameAsyncMethodInfo a signature where
    overloadedMethod = fileSetDisplayNameAsync

instance O.OverloadedMethodInfo FileSetDisplayNameAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetDisplayNameAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetDisplayNameAsync"
        })


#endif

-- method File::set_display_name_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_set_display_name_finish" g_file_set_display_name_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr File)

-- | Finishes setting a display name started with
-- 'GI.Gio.Interfaces.File.fileSetDisplayNameAsync'.
fileSetDisplayNameFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m File
    -- ^ __Returns:__ a t'GI.Gio.Interfaces.File.File' or 'P.Nothing' on error.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileSetDisplayNameFinish file res = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_file_set_display_name_finish file' res'
        checkUnexpectedReturnNULL "fileSetDisplayNameFinish" result
        result' <- (wrapObject File) result
        touchManagedPtr file
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileSetDisplayNameFinishMethodInfo
instance (signature ~ (b -> m File), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileSetDisplayNameFinishMethodInfo a signature where
    overloadedMethod = fileSetDisplayNameFinish

instance O.OverloadedMethodInfo FileSetDisplayNameFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSetDisplayNameFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSetDisplayNameFinish"
        })


#endif

-- method File::start_mountable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DriveStartFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation, or %NULL to avoid user interaction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call when the request is satisfied, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_start_mountable" g_file_start_mountable :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DriveStartFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- start_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Starts a file of type 'GI.Gio.Enums.FileTypeMountable'.
-- Using /@startOperation@/, you can request callbacks when, for instance,
-- passwords are needed during authentication.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileMountMountableFinish' to get
-- the result of the operation.
-- 
-- /Since: 2.22/
fileStartMountable ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.DriveStartFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@startOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation', or 'P.Nothing' to avoid user interaction
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is satisfied, or 'P.Nothing'
    -> m ()
fileStartMountable file flags startOperation cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeStartOperation <- case startOperation of
        Nothing -> return FP.nullPtr
        Just jStartOperation -> do
            jStartOperation' <- unsafeManagedPtrCastPtr jStartOperation
            return jStartOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_start_mountable file' flags' maybeStartOperation maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust startOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileStartMountableMethodInfo
instance (signature ~ ([Gio.Flags.DriveStartFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileStartMountableMethodInfo a signature where
    overloadedMethod = fileStartMountable

instance O.OverloadedMethodInfo FileStartMountableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileStartMountable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileStartMountable"
        })


#endif

-- method File::start_mountable_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_start_mountable_finish" g_file_start_mountable_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes a start operation. See 'GI.Gio.Interfaces.File.fileStartMountable' for details.
-- 
-- Finish an asynchronous start operation that was started
-- with 'GI.Gio.Interfaces.File.fileStartMountable'.
-- 
-- /Since: 2.22/
fileStartMountableFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileStartMountableFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_start_mountable_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileStartMountableFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileStartMountableFinishMethodInfo a signature where
    overloadedMethod = fileStartMountableFinish

instance O.OverloadedMethodInfo FileStartMountableFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileStartMountableFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileStartMountableFinish"
        })


#endif

-- method File::stop_mountable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation,\n  or %NULL to avoid user interaction."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call\n  when the request is satisfied, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_stop_mountable" g_file_stop_mountable :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Stops a file of type 'GI.Gio.Enums.FileTypeMountable'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileStopMountableFinish' to get
-- the result of the operation.
-- 
-- /Since: 2.22/
fileStopMountable ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation',
    --   or 'P.Nothing' to avoid user interaction.
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call
    --   when the request is satisfied, or 'P.Nothing'
    -> m ()
fileStopMountable file flags mountOperation cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_stop_mountable file' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileStopMountableMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileStopMountableMethodInfo a signature where
    overloadedMethod = fileStopMountable

instance O.OverloadedMethodInfo FileStopMountableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileStopMountable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileStopMountable"
        })


#endif

-- method File::stop_mountable_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_stop_mountable_finish" g_file_stop_mountable_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes a stop operation, see 'GI.Gio.Interfaces.File.fileStopMountable' for details.
-- 
-- Finish an asynchronous stop operation that was started
-- with 'GI.Gio.Interfaces.File.fileStopMountable'.
-- 
-- /Since: 2.22/
fileStopMountableFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileStopMountableFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_stop_mountable_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileStopMountableFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileStopMountableFinishMethodInfo a signature where
    overloadedMethod = fileStopMountableFinish

instance O.OverloadedMethodInfo FileStopMountableFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileStopMountableFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileStopMountableFinish"
        })


#endif

-- method File::supports_thread_contexts
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_file_supports_thread_contexts" g_file_supports_thread_contexts :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    IO CInt

-- | Checks if /@file@/ supports thread-default main contexts
-- (see 'GI.GLib.Structs.MainContext.mainContextPushThreadDefault')
-- If this returns 'P.False', you cannot perform asynchronous operations on
-- /@file@/ in a thread that has a thread-default context.
-- 
-- /Since: 2.22/
fileSupportsThreadContexts ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a) =>
    a
    -- ^ /@file@/: a t'GI.Gio.Interfaces.File.File'
    -> m Bool
    -- ^ __Returns:__ Whether or not /@file@/ supports thread-default contexts.
fileSupportsThreadContexts file = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result <- g_file_supports_thread_contexts file'
    let result' = (/= 0) result
    touchManagedPtr file
    return result'

#if defined(ENABLE_OVERLOADING)
data FileSupportsThreadContextsMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsFile a) => O.OverloadedMethod FileSupportsThreadContextsMethodInfo a signature where
    overloadedMethod = fileSupportsThreadContexts

instance O.OverloadedMethodInfo FileSupportsThreadContextsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileSupportsThreadContexts",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileSupportsThreadContexts"
        })


#endif

-- method File::trash
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GFile to send to trash"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_trash" g_file_trash :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sends /@file@/ to the \"Trashcan\", if possible. This is similar to
-- deleting it, but the user can recover it before emptying the trashcan.
-- Trashing is disabled for system mounts by default (see
-- 'GI.Gio.Structs.UnixMountEntry.unixMountEntryIsSystemInternal'), so this call can return the
-- 'GI.Gio.Enums.IOErrorEnumNotSupported' error. Since GLib 2.66, the @x-gvfs-notrash@ unix
-- mount option can be used to disable 'GI.Gio.Interfaces.File.fileTrash' support for particular
-- mounts, the 'GI.Gio.Enums.IOErrorEnumNotSupported' error will be returned in that case.
-- Since 2.82, the @x-gvfs-trash@ unix mount option can be used to enable
-- 'GI.Gio.Interfaces.File.fileTrash' support for particular system mounts.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
fileTrash ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: t'GI.Gio.Interfaces.File.File' to send to trash
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileTrash file cancellable = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_file_trash file' maybeCancellable
        touchManagedPtr file
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileTrashMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileTrashMethodInfo a signature where
    overloadedMethod = fileTrash

instance O.OverloadedMethodInfo FileTrashMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileTrash",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileTrash"
        })


#endif

-- method File::trash_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call\n  when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_trash_async" g_file_trash_async :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously sends /@file@/ to the Trash location, if possible.
-- 
-- /Since: 2.38/
fileTrashAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call
    --   when the request is satisfied
    -> m ()
fileTrashAsync file ioPriority cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_trash_async file' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileTrashAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileTrashAsyncMethodInfo a signature where
    overloadedMethod = fileTrashAsync

instance O.OverloadedMethodInfo FileTrashAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileTrashAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileTrashAsync"
        })


#endif

-- method File::trash_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_trash_finish" g_file_trash_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous file trashing operation, started with
-- 'GI.Gio.Interfaces.File.fileTrashAsync'.
-- 
-- /Since: 2.38/
fileTrashFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileTrashFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_trash_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileTrashFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileTrashFinishMethodInfo a signature where
    overloadedMethod = fileTrashFinish

instance O.OverloadedMethodInfo FileTrashFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileTrashFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileTrashFinish"
        })


#endif

-- method File::unmount_mountable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_unmount_mountable" g_file_unmount_mountable :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

{-# DEPRECATED fileUnmountMountable ["(Since version 2.22)","Use 'GI.Gio.Interfaces.File.fileUnmountMountableWithOperation' instead."] #-}
-- | Unmounts a file of type G_FILE_TYPE_MOUNTABLE.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileUnmountMountableFinish' to get
-- the result of the operation.
fileUnmountMountable ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileUnmountMountable file flags cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_unmount_mountable file' flags' maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileUnmountMountableMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod FileUnmountMountableMethodInfo a signature where
    overloadedMethod = fileUnmountMountable

instance O.OverloadedMethodInfo FileUnmountMountableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileUnmountMountable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileUnmountMountable"
        })


#endif

-- method File::unmount_mountable_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_unmount_mountable_finish" g_file_unmount_mountable_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

{-# DEPRECATED fileUnmountMountableFinish ["(Since version 2.22)","Use 'GI.Gio.Interfaces.File.fileUnmountMountableWithOperationFinish'","  instead."] #-}
-- | Finishes an unmount operation, see 'GI.Gio.Interfaces.File.fileUnmountMountable' for details.
-- 
-- Finish an asynchronous unmount operation that was started
-- with 'GI.Gio.Interfaces.File.fileUnmountMountable'.
fileUnmountMountableFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileUnmountMountableFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_unmount_mountable_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileUnmountMountableFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileUnmountMountableFinishMethodInfo a signature where
    overloadedMethod = fileUnmountMountableFinish

instance O.OverloadedMethodInfo FileUnmountMountableFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileUnmountMountableFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileUnmountMountableFinish"
        })


#endif

-- method File::unmount_mountable_with_operation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation,\n  or %NULL to avoid user interaction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object,\n  %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_unmount_mountable_with_operation" g_file_unmount_mountable_with_operation :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Unmounts a file of type 'GI.Gio.Enums.FileTypeMountable'.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.File.fileUnmountMountableFinish' to get
-- the result of the operation.
-- 
-- /Since: 2.22/
fileUnmountMountableWithOperation ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the operation
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation',
    --   or 'P.Nothing' to avoid user interaction
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object,
    --   'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
fileUnmountMountableWithOperation file flags mountOperation cancellable callback = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_unmount_mountable_with_operation file' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr file
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data FileUnmountMountableWithOperationMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsFile a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod FileUnmountMountableWithOperationMethodInfo a signature where
    overloadedMethod = fileUnmountMountableWithOperation

instance O.OverloadedMethodInfo FileUnmountMountableWithOperationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileUnmountMountableWithOperation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileUnmountMountableWithOperation"
        })


#endif

-- method File::unmount_mountable_with_operation_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "file"
--           , argType = TInterface Name { namespace = "Gio" , name = "File" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input #GFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_file_unmount_mountable_with_operation_finish" g_file_unmount_mountable_with_operation_finish :: 
    Ptr File ->                             -- file : TInterface (Name {namespace = "Gio", name = "File"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an unmount operation,
-- see 'GI.Gio.Interfaces.File.fileUnmountMountableWithOperation' for details.
-- 
-- Finish an asynchronous unmount operation that was started
-- with 'GI.Gio.Interfaces.File.fileUnmountMountableWithOperation'.
-- 
-- /Since: 2.22/
fileUnmountMountableWithOperationFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@file@/: input t'GI.Gio.Interfaces.File.File'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
fileUnmountMountableWithOperationFinish file result_ = liftIO $ do
    file' <- unsafeManagedPtrCastPtr file
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_file_unmount_mountable_with_operation_finish file' result_'
        touchManagedPtr file
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data FileUnmountMountableWithOperationFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsFile a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod FileUnmountMountableWithOperationFinishMethodInfo a signature where
    overloadedMethod = fileUnmountMountableWithOperationFinish

instance O.OverloadedMethodInfo FileUnmountMountableWithOperationFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.File.fileUnmountMountableWithOperationFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-File.html#v:fileUnmountMountableWithOperationFinish"
        })


#endif

-- method File::new_build_filenamev
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "args"
--           , argType = TCArray True (-1) (-1) (TBasicType TFileName)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "%NULL-terminated\n  array of strings containing the path elements."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_new_build_filenamev" g_file_new_build_filenamev :: 
    Ptr CString ->                          -- args : TCArray True (-1) (-1) (TBasicType TFileName)
    IO (Ptr File)

-- | Constructs a t'GI.Gio.Interfaces.File.File' from a vector of elements using the correct
-- separator for filenames.
-- 
-- Using this function is equivalent to calling 'GI.GLib.Functions.buildFilenamev',
-- followed by 'GI.Gio.Functions.fileNewForPath' on the result.
-- 
-- /Since: 2.78/
fileNewBuildFilenamev ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [[Char]]
    -- ^ /@args@/: 'P.Nothing'-terminated
    --   array of strings containing the path elements.
    -> m File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File'
fileNewBuildFilenamev args = liftIO $ do
    args' <- packZeroTerminatedFileNameArray args
    result <- g_file_new_build_filenamev args'
    checkUnexpectedReturnNULL "fileNewBuildFilenamev" result
    result' <- (wrapObject File) result
    mapZeroTerminatedCArray freeMem args'
    freeMem args'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_for_commandline_arg
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "arg"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a command line string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_new_for_commandline_arg" g_file_new_for_commandline_arg :: 
    CString ->                              -- arg : TBasicType TFileName
    IO (Ptr File)

-- | Creates a t'GI.Gio.Interfaces.File.File' with the given argument from the command line.
-- The value of /@arg@/ can be either a URI, an absolute path or a
-- relative path resolved relative to the current working directory.
-- This operation never fails, but the returned object might not
-- support any I\/O operation if /@arg@/ points to a malformed path.
-- 
-- Note that on Windows, this function expects its argument to be in
-- UTF-8 -- not the system code page.  This means that you
-- should not use this function with string from argv as it is passed
-- to @/main()/@.  @/g_win32_get_command_line()/@ will return a UTF-8 version of
-- the commandline.  t'GI.Gio.Objects.Application.Application' also uses UTF-8 but
-- 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineCreateFileForArg' may be more useful
-- for you there.  It is also always possible to use this function with
-- t'GI.GLib.Structs.OptionContext.OptionContext' arguments of type 'GI.GLib.Enums.OptionArgFilename'.
fileNewForCommandlineArg ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Char]
    -- ^ /@arg@/: a command line string
    -> m File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File'.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'.
fileNewForCommandlineArg arg = liftIO $ do
    arg' <- stringToCString arg
    result <- g_file_new_for_commandline_arg arg'
    checkUnexpectedReturnNULL "fileNewForCommandlineArg" result
    result' <- (wrapObject File) result
    freeMem arg'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_for_commandline_arg_and_cwd
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "arg"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a command line string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cwd"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the current working directory of the commandline"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_new_for_commandline_arg_and_cwd" g_file_new_for_commandline_arg_and_cwd :: 
    CString ->                              -- arg : TBasicType TFileName
    CString ->                              -- cwd : TBasicType TFileName
    IO (Ptr File)

-- | Creates a t'GI.Gio.Interfaces.File.File' with the given argument from the command line.
-- 
-- This function is similar to 'GI.Gio.Functions.fileNewForCommandlineArg' except
-- that it allows for passing the current working directory as an
-- argument instead of using the current working directory of the
-- process.
-- 
-- This is useful if the commandline argument was given in a context
-- other than the invocation of the current process.
-- 
-- See also 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineCreateFileForArg'.
-- 
-- /Since: 2.36/
fileNewForCommandlineArgAndCwd ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Char]
    -- ^ /@arg@/: a command line string
    -> [Char]
    -- ^ /@cwd@/: the current working directory of the commandline
    -> m File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File'
fileNewForCommandlineArgAndCwd arg cwd = liftIO $ do
    arg' <- stringToCString arg
    cwd' <- stringToCString cwd
    result <- g_file_new_for_commandline_arg_and_cwd arg' cwd'
    checkUnexpectedReturnNULL "fileNewForCommandlineArgAndCwd" result
    result' <- (wrapObject File) result
    freeMem arg'
    freeMem cwd'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_for_path
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "path"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a string containing a relative or absolute path.\n  The string must be encoded in the glib filename encoding."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_new_for_path" g_file_new_for_path :: 
    CString ->                              -- path : TBasicType TFileName
    IO (Ptr File)

-- | Constructs a t'GI.Gio.Interfaces.File.File' for a given path. This operation never
-- fails, but the returned object might not support any I\/O
-- operation if /@path@/ is malformed.
fileNewForPath ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Char]
    -- ^ /@path@/: a string containing a relative or absolute path.
    --   The string must be encoded in the glib filename encoding.
    -> m File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File' for the given /@path@/.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'.
fileNewForPath path = liftIO $ do
    path' <- stringToCString path
    result <- g_file_new_for_path path'
    checkUnexpectedReturnNULL "fileNewForPath" result
    result' <- (wrapObject File) result
    freeMem path'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_for_uri
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a UTF-8 string containing a URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_new_for_uri" g_file_new_for_uri :: 
    CString ->                              -- uri : TBasicType TUTF8
    IO (Ptr File)

-- | Constructs a t'GI.Gio.Interfaces.File.File' for a given URI. This operation never
-- fails, but the returned object might not support any I\/O
-- operation if /@uri@/ is malformed or if the uri type is
-- not supported.
fileNewForUri ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uri@/: a UTF-8 string containing a URI
    -> m File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File' for the given /@uri@/.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'.
fileNewForUri uri = liftIO $ do
    uri' <- textToCString uri
    result <- g_file_new_for_uri uri'
    checkUnexpectedReturnNULL "fileNewForUri" result
    result' <- (wrapObject File) result
    freeMem uri'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_tmp
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "tmpl"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Template for the file\n  name, as in g_file_open_tmp(), or %NULL for a default template"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "iostream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileIOStream" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "on return, a #GFileIOStream for the created file"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_new_tmp" g_file_new_tmp :: 
    CString ->                              -- tmpl : TBasicType TFileName
    Ptr (Ptr Gio.FileIOStream.FileIOStream) -> -- iostream : TInterface (Name {namespace = "Gio", name = "FileIOStream"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr File)

-- | Opens a file in the preferred directory for temporary files (as
-- returned by 'GI.GLib.Functions.getTmpDir') and returns a t'GI.Gio.Interfaces.File.File' and
-- t'GI.Gio.Objects.FileIOStream.FileIOStream' pointing to it.
-- 
-- /@tmpl@/ should be a string in the GLib file name encoding
-- containing a sequence of six \'X\' characters, and containing no
-- directory components. If it is 'P.Nothing', a default template is used.
-- 
-- Unlike the other t'GI.Gio.Interfaces.File.File' constructors, this will return 'P.Nothing' if
-- a temporary file could not be created.
-- 
-- /Since: 2.32/
fileNewTmp ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe ([Char])
    -- ^ /@tmpl@/: Template for the file
    --   name, as in 'GI.GLib.Functions.fileOpenTmp', or 'P.Nothing' for a default template
    -> m ((File, Gio.FileIOStream.FileIOStream))
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File'.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileNewTmp tmpl = liftIO $ do
    maybeTmpl <- case tmpl of
        Nothing -> return FP.nullPtr
        Just jTmpl -> do
            jTmpl' <- stringToCString jTmpl
            return jTmpl'
    iostream <- callocMem :: IO (Ptr (Ptr Gio.FileIOStream.FileIOStream))
    onException (do
        result <- propagateGError $ g_file_new_tmp maybeTmpl iostream
        checkUnexpectedReturnNULL "fileNewTmp" result
        result' <- (wrapObject File) result
        iostream' <- peek iostream
        iostream'' <- (wrapObject Gio.FileIOStream.FileIOStream) iostream'
        freeMem maybeTmpl
        freeMem iostream
        return (result', iostream'')
     ) (do
        freeMem maybeTmpl
        freeMem iostream
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_tmp_async
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "tmpl"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Template for the file\n  name, as in g_file_open_tmp(), or %NULL for a default template"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GAsyncReadyCallback to call when the request is done"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_new_tmp_async" g_file_new_tmp_async :: 
    CString ->                              -- tmpl : TBasicType TFileName
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously opens a file in the preferred directory for temporary files
--  (as returned by 'GI.GLib.Functions.getTmpDir') as 'GI.Gio.Functions.fileNewTmp'.
-- 
-- /@tmpl@/ should be a string in the GLib file name encoding
-- containing a sequence of six \'X\' characters, and containing no
-- directory components. If it is 'P.Nothing', a default template is used.
-- 
-- /Since: 2.74/
fileNewTmpAsync ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.Cancellable.IsCancellable a) =>
    Maybe ([Char])
    -- ^ /@tmpl@/: Template for the file
    --   name, as in 'GI.GLib.Functions.fileOpenTmp', or 'P.Nothing' for a default template
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (a)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is done
    -> m ()
fileNewTmpAsync tmpl ioPriority cancellable callback = liftIO $ do
    maybeTmpl <- case tmpl of
        Nothing -> return FP.nullPtr
        Just jTmpl -> do
            jTmpl' <- stringToCString jTmpl
            return jTmpl'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_new_tmp_async maybeTmpl ioPriority maybeCancellable maybeCallback userData
    whenJust cancellable touchManagedPtr
    freeMem maybeTmpl
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_tmp_dir_async
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "tmpl"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Template for the file\n  name, as in g_dir_make_tmp(), or %NULL for a default template"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GAsyncReadyCallback to call when the request is done"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_file_new_tmp_dir_async" g_file_new_tmp_dir_async :: 
    CString ->                              -- tmpl : TBasicType TFileName
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously creates a directory in the preferred directory for
-- temporary files (as returned by 'GI.GLib.Functions.getTmpDir') as 'GI.GLib.Functions.dirMakeTmp'.
-- 
-- /@tmpl@/ should be a string in the GLib file name encoding
-- containing a sequence of six \'X\' characters, and containing no
-- directory components. If it is 'P.Nothing', a default template is used.
-- 
-- /Since: 2.74/
fileNewTmpDirAsync ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.Cancellable.IsCancellable a) =>
    Maybe ([Char])
    -- ^ /@tmpl@/: Template for the file
    --   name, as in 'GI.GLib.Functions.dirMakeTmp', or 'P.Nothing' for a default template
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (a)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is done
    -> m ()
fileNewTmpDirAsync tmpl ioPriority cancellable callback = liftIO $ do
    maybeTmpl <- case tmpl of
        Nothing -> return FP.nullPtr
        Just jTmpl -> do
            jTmpl' <- stringToCString jTmpl
            return jTmpl'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_file_new_tmp_dir_async maybeTmpl ioPriority maybeCancellable maybeCallback userData
    whenJust cancellable touchManagedPtr
    freeMem maybeTmpl
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_tmp_dir_finish
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_new_tmp_dir_finish" g_file_new_tmp_dir_finish :: 
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr File)

-- | Finishes a temporary directory creation started by
-- 'GI.Gio.Functions.fileNewTmpDirAsync'.
-- 
-- /Since: 2.74/
fileNewTmpDirFinish ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.AsyncResult.IsAsyncResult a) =>
    a
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File'.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileNewTmpDirFinish result_ = liftIO $ do
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_file_new_tmp_dir_finish result_'
        checkUnexpectedReturnNULL "fileNewTmpDirFinish" result
        result' <- (wrapObject File) result
        touchManagedPtr result_
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::new_tmp_finish
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "iostream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "FileIOStream" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "on return, a #GFileIOStream for the created file"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : True
-- Skip return : False

foreign import ccall "g_file_new_tmp_finish" g_file_new_tmp_finish :: 
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr Gio.FileIOStream.FileIOStream) -> -- iostream : TInterface (Name {namespace = "Gio", name = "FileIOStream"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr File)

-- | Finishes a temporary file creation started by 'GI.Gio.Functions.fileNewTmpAsync'.
-- 
-- /Since: 2.74/
fileNewTmpFinish ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.AsyncResult.IsAsyncResult a) =>
    a
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ((File, Gio.FileIOStream.FileIOStream))
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File'.
    --   Free the returned object with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
fileNewTmpFinish result_ = liftIO $ do
    result_' <- unsafeManagedPtrCastPtr result_
    iostream <- callocMem :: IO (Ptr (Ptr Gio.FileIOStream.FileIOStream))
    onException (do
        result <- propagateGError $ g_file_new_tmp_finish result_' iostream
        checkUnexpectedReturnNULL "fileNewTmpFinish" result
        result' <- (wrapObject File) result
        iostream' <- peek iostream
        iostream'' <- (wrapObject Gio.FileIOStream.FileIOStream) iostream'
        touchManagedPtr result_
        freeMem iostream
        return (result', iostream'')
     ) (do
        freeMem iostream
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method File::parse_name
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "parse_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a file name or path to be parsed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "File" })
-- throws : False
-- Skip return : False

foreign import ccall "g_file_parse_name" g_file_parse_name :: 
    CString ->                              -- parse_name : TBasicType TUTF8
    IO (Ptr File)

-- | Constructs a t'GI.Gio.Interfaces.File.File' with the given /@parseName@/ (i.e. something
-- given by 'GI.Gio.Interfaces.File.fileGetParseName'). This operation never fails,
-- but the returned object might not support any I\/O operation if
-- the /@parseName@/ cannot be parsed.
fileParseName ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@parseName@/: a file name or path to be parsed
    -> m File
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.File.File'.
fileParseName parseName = liftIO $ do
    parseName' <- textToCString parseName
    result <- g_file_parse_name parseName'
    checkUnexpectedReturnNULL "fileParseName" result
    result' <- (wrapObject File) result
    freeMem parseName'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList File = FileSignalList
type FileSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif


