{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GDrive@ represents a piece of hardware connected to the machine.
-- It’s generally only created for removable hardware or hardware with
-- removable media. For example, an optical disc drive, or a USB flash drive.
-- 
-- @GDrive@ is a container class for t'GI.Gio.Interfaces.Volume.Volume' objects that stem from
-- the same piece of media. As such, @GDrive@ abstracts a drive with
-- (or without) removable media and provides operations for querying
-- whether media is available, determining whether media change is
-- automatically detected and ejecting the media.
-- 
-- If the @GDrive@ reports that media isn’t automatically detected, one
-- can poll for media; typically one should not do this periodically
-- as a poll for media operation is potentially expensive and may
-- spin up the drive creating noise.
-- 
-- @GDrive@ supports starting and stopping drives with authentication
-- support for the former. This can be used to support a diverse set
-- of use cases including connecting\/disconnecting iSCSI devices,
-- powering down external disk enclosures and starting\/stopping
-- multi-disk devices such as RAID devices. Note that the actual
-- semantics and side-effects of starting\/stopping a @GDrive@ may vary
-- according to implementation. To choose the correct verbs in e.g. a
-- file manager, use 'GI.Gio.Interfaces.Drive.driveGetStartStopType'.
-- 
-- For <http://developer.gnome.org/gio/stable/migrating-gnome-vfs.html porting from GnomeVFS> note that there is no
-- equivalent of @GDrive@ in that API.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Interfaces.Drive
    ( 

-- * Exported types
    Drive(..)                               ,
    IsDrive                                 ,
    toDrive                                 ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [canEject]("GI.Gio.Interfaces.Drive#g:method:canEject"), [canPollForMedia]("GI.Gio.Interfaces.Drive#g:method:canPollForMedia"), [canStart]("GI.Gio.Interfaces.Drive#g:method:canStart"), [canStartDegraded]("GI.Gio.Interfaces.Drive#g:method:canStartDegraded"), [canStop]("GI.Gio.Interfaces.Drive#g:method:canStop"), [eject]("GI.Gio.Interfaces.Drive#g:method:eject"), [ejectFinish]("GI.Gio.Interfaces.Drive#g:method:ejectFinish"), [ejectWithOperation]("GI.Gio.Interfaces.Drive#g:method:ejectWithOperation"), [ejectWithOperationFinish]("GI.Gio.Interfaces.Drive#g:method:ejectWithOperationFinish"), [enumerateIdentifiers]("GI.Gio.Interfaces.Drive#g:method:enumerateIdentifiers"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [hasMedia]("GI.Gio.Interfaces.Drive#g:method:hasMedia"), [hasVolumes]("GI.Gio.Interfaces.Drive#g:method:hasVolumes"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isMediaCheckAutomatic]("GI.Gio.Interfaces.Drive#g:method:isMediaCheckAutomatic"), [isMediaRemovable]("GI.Gio.Interfaces.Drive#g:method:isMediaRemovable"), [isRemovable]("GI.Gio.Interfaces.Drive#g:method:isRemovable"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [pollForMedia]("GI.Gio.Interfaces.Drive#g:method:pollForMedia"), [pollForMediaFinish]("GI.Gio.Interfaces.Drive#g:method:pollForMediaFinish"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [start]("GI.Gio.Interfaces.Drive#g:method:start"), [startFinish]("GI.Gio.Interfaces.Drive#g:method:startFinish"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [stop]("GI.Gio.Interfaces.Drive#g:method:stop"), [stopFinish]("GI.Gio.Interfaces.Drive#g:method:stopFinish"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getIcon]("GI.Gio.Interfaces.Drive#g:method:getIcon"), [getIdentifier]("GI.Gio.Interfaces.Drive#g:method:getIdentifier"), [getName]("GI.Gio.Interfaces.Drive#g:method:getName"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getSortKey]("GI.Gio.Interfaces.Drive#g:method:getSortKey"), [getStartStopType]("GI.Gio.Interfaces.Drive#g:method:getStartStopType"), [getSymbolicIcon]("GI.Gio.Interfaces.Drive#g:method:getSymbolicIcon"), [getVolumes]("GI.Gio.Interfaces.Drive#g:method:getVolumes").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveDriveMethod                      ,
#endif

-- ** canEject #method:canEject#

#if defined(ENABLE_OVERLOADING)
    DriveCanEjectMethodInfo                 ,
#endif
    driveCanEject                           ,


-- ** canPollForMedia #method:canPollForMedia#

#if defined(ENABLE_OVERLOADING)
    DriveCanPollForMediaMethodInfo          ,
#endif
    driveCanPollForMedia                    ,


-- ** canStart #method:canStart#

#if defined(ENABLE_OVERLOADING)
    DriveCanStartMethodInfo                 ,
#endif
    driveCanStart                           ,


-- ** canStartDegraded #method:canStartDegraded#

#if defined(ENABLE_OVERLOADING)
    DriveCanStartDegradedMethodInfo         ,
#endif
    driveCanStartDegraded                   ,


-- ** canStop #method:canStop#

#if defined(ENABLE_OVERLOADING)
    DriveCanStopMethodInfo                  ,
#endif
    driveCanStop                            ,


-- ** eject #method:eject#

#if defined(ENABLE_OVERLOADING)
    DriveEjectMethodInfo                    ,
#endif
    driveEject                              ,


-- ** ejectFinish #method:ejectFinish#

#if defined(ENABLE_OVERLOADING)
    DriveEjectFinishMethodInfo              ,
#endif
    driveEjectFinish                        ,


-- ** ejectWithOperation #method:ejectWithOperation#

#if defined(ENABLE_OVERLOADING)
    DriveEjectWithOperationMethodInfo       ,
#endif
    driveEjectWithOperation                 ,


-- ** ejectWithOperationFinish #method:ejectWithOperationFinish#

#if defined(ENABLE_OVERLOADING)
    DriveEjectWithOperationFinishMethodInfo ,
#endif
    driveEjectWithOperationFinish           ,


-- ** enumerateIdentifiers #method:enumerateIdentifiers#

#if defined(ENABLE_OVERLOADING)
    DriveEnumerateIdentifiersMethodInfo     ,
#endif
    driveEnumerateIdentifiers               ,


-- ** getIcon #method:getIcon#

#if defined(ENABLE_OVERLOADING)
    DriveGetIconMethodInfo                  ,
#endif
    driveGetIcon                            ,


-- ** getIdentifier #method:getIdentifier#

#if defined(ENABLE_OVERLOADING)
    DriveGetIdentifierMethodInfo            ,
#endif
    driveGetIdentifier                      ,


-- ** getName #method:getName#

#if defined(ENABLE_OVERLOADING)
    DriveGetNameMethodInfo                  ,
#endif
    driveGetName                            ,


-- ** getSortKey #method:getSortKey#

#if defined(ENABLE_OVERLOADING)
    DriveGetSortKeyMethodInfo               ,
#endif
    driveGetSortKey                         ,


-- ** getStartStopType #method:getStartStopType#

#if defined(ENABLE_OVERLOADING)
    DriveGetStartStopTypeMethodInfo         ,
#endif
    driveGetStartStopType                   ,


-- ** getSymbolicIcon #method:getSymbolicIcon#

#if defined(ENABLE_OVERLOADING)
    DriveGetSymbolicIconMethodInfo          ,
#endif
    driveGetSymbolicIcon                    ,


-- ** getVolumes #method:getVolumes#

#if defined(ENABLE_OVERLOADING)
    DriveGetVolumesMethodInfo               ,
#endif
    driveGetVolumes                         ,


-- ** hasMedia #method:hasMedia#

#if defined(ENABLE_OVERLOADING)
    DriveHasMediaMethodInfo                 ,
#endif
    driveHasMedia                           ,


-- ** hasVolumes #method:hasVolumes#

#if defined(ENABLE_OVERLOADING)
    DriveHasVolumesMethodInfo               ,
#endif
    driveHasVolumes                         ,


-- ** isMediaCheckAutomatic #method:isMediaCheckAutomatic#

#if defined(ENABLE_OVERLOADING)
    DriveIsMediaCheckAutomaticMethodInfo    ,
#endif
    driveIsMediaCheckAutomatic              ,


-- ** isMediaRemovable #method:isMediaRemovable#

#if defined(ENABLE_OVERLOADING)
    DriveIsMediaRemovableMethodInfo         ,
#endif
    driveIsMediaRemovable                   ,


-- ** isRemovable #method:isRemovable#

#if defined(ENABLE_OVERLOADING)
    DriveIsRemovableMethodInfo              ,
#endif
    driveIsRemovable                        ,


-- ** pollForMedia #method:pollForMedia#

#if defined(ENABLE_OVERLOADING)
    DrivePollForMediaMethodInfo             ,
#endif
    drivePollForMedia                       ,


-- ** pollForMediaFinish #method:pollForMediaFinish#

#if defined(ENABLE_OVERLOADING)
    DrivePollForMediaFinishMethodInfo       ,
#endif
    drivePollForMediaFinish                 ,


-- ** start #method:start#

#if defined(ENABLE_OVERLOADING)
    DriveStartMethodInfo                    ,
#endif
    driveStart                              ,


-- ** startFinish #method:startFinish#

#if defined(ENABLE_OVERLOADING)
    DriveStartFinishMethodInfo              ,
#endif
    driveStartFinish                        ,


-- ** stop #method:stop#

#if defined(ENABLE_OVERLOADING)
    DriveStopMethodInfo                     ,
#endif
    driveStop                               ,


-- ** stopFinish #method:stopFinish#

#if defined(ENABLE_OVERLOADING)
    DriveStopFinishMethodInfo               ,
#endif
    driveStopFinish                         ,




 -- * Signals


-- ** changed #signal:changed#

    DriveChangedCallback                    ,
#if defined(ENABLE_OVERLOADING)
    DriveChangedSignalInfo                  ,
#endif
    afterDriveChanged                       ,
    onDriveChanged                          ,


-- ** disconnected #signal:disconnected#

    DriveDisconnectedCallback               ,
#if defined(ENABLE_OVERLOADING)
    DriveDisconnectedSignalInfo             ,
#endif
    afterDriveDisconnected                  ,
    onDriveDisconnected                     ,


-- ** ejectButton #signal:ejectButton#

    DriveEjectButtonCallback                ,
#if defined(ENABLE_OVERLOADING)
    DriveEjectButtonSignalInfo              ,
#endif
    afterDriveEjectButton                   ,
    onDriveEjectButton                      ,


-- ** stopButton #signal:stopButton#

    DriveStopButtonCallback                 ,
#if defined(ENABLE_OVERLOADING)
    DriveStopButtonSignalInfo               ,
#endif
    afterDriveStopButton                    ,
    onDriveStopButton                       ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Enums as GLib.Enums
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.DateTime as GLib.DateTime
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GLib.Structs.TimeVal as GLib.TimeVal
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AppInfo as Gio.AppInfo
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.File as Gio.File
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Icon as Gio.Icon
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Mount as Gio.Mount
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Seekable as Gio.Seekable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Volume as Gio.Volume
import {-# SOURCE #-} qualified GI.Gio.Objects.AppLaunchContext as Gio.AppLaunchContext
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.FileEnumerator as Gio.FileEnumerator
import {-# SOURCE #-} qualified GI.Gio.Objects.FileIOStream as Gio.FileIOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInfo as Gio.FileInfo
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInputStream as Gio.FileInputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileMonitor as Gio.FileMonitor
import {-# SOURCE #-} qualified GI.Gio.Objects.FileOutputStream as Gio.FileOutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MountOperation as Gio.MountOperation
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfo as Gio.FileAttributeInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfoList as Gio.FileAttributeInfoList
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeMatcher as Gio.FileAttributeMatcher
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Icon as Gio.Icon
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Volume as Gio.Volume
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.MountOperation as Gio.MountOperation

#endif

-- interface Drive 
-- | Memory-managed wrapper type.
newtype Drive = Drive (SP.ManagedPtr Drive)
    deriving (Eq)

instance SP.ManagedPtrNewtype Drive where
    toManagedPtr (Drive p) = p

foreign import ccall "g_drive_get_type"
    c_g_drive_get_type :: IO B.Types.GType

instance B.Types.TypedObject Drive where
    glibType = c_g_drive_get_type

instance B.Types.GObject Drive

-- | Type class for types which can be safely cast to `Drive`, for instance with `toDrive`.
class (SP.GObject o, O.IsDescendantOf Drive o) => IsDrive o
instance (SP.GObject o, O.IsDescendantOf Drive o) => IsDrive o

instance O.HasParentTypes Drive
type instance O.ParentTypes Drive = '[GObject.Object.Object]

-- | Cast to `Drive`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toDrive :: (MIO.MonadIO m, IsDrive o) => o -> m Drive
toDrive = MIO.liftIO . B.ManagedPtr.unsafeCastTo Drive

-- | Convert 'Drive' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Drive) where
    gvalueGType_ = c_g_drive_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Drive)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Drive)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Drive ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Drive
type instance O.AttributeList Drive = DriveAttributeList
type DriveAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveDriveMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveDriveMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveDriveMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveDriveMethod "canEject" o = DriveCanEjectMethodInfo
    ResolveDriveMethod "canPollForMedia" o = DriveCanPollForMediaMethodInfo
    ResolveDriveMethod "canStart" o = DriveCanStartMethodInfo
    ResolveDriveMethod "canStartDegraded" o = DriveCanStartDegradedMethodInfo
    ResolveDriveMethod "canStop" o = DriveCanStopMethodInfo
    ResolveDriveMethod "eject" o = DriveEjectMethodInfo
    ResolveDriveMethod "ejectFinish" o = DriveEjectFinishMethodInfo
    ResolveDriveMethod "ejectWithOperation" o = DriveEjectWithOperationMethodInfo
    ResolveDriveMethod "ejectWithOperationFinish" o = DriveEjectWithOperationFinishMethodInfo
    ResolveDriveMethod "enumerateIdentifiers" o = DriveEnumerateIdentifiersMethodInfo
    ResolveDriveMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveDriveMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveDriveMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveDriveMethod "hasMedia" o = DriveHasMediaMethodInfo
    ResolveDriveMethod "hasVolumes" o = DriveHasVolumesMethodInfo
    ResolveDriveMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveDriveMethod "isMediaCheckAutomatic" o = DriveIsMediaCheckAutomaticMethodInfo
    ResolveDriveMethod "isMediaRemovable" o = DriveIsMediaRemovableMethodInfo
    ResolveDriveMethod "isRemovable" o = DriveIsRemovableMethodInfo
    ResolveDriveMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveDriveMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveDriveMethod "pollForMedia" o = DrivePollForMediaMethodInfo
    ResolveDriveMethod "pollForMediaFinish" o = DrivePollForMediaFinishMethodInfo
    ResolveDriveMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveDriveMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveDriveMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveDriveMethod "start" o = DriveStartMethodInfo
    ResolveDriveMethod "startFinish" o = DriveStartFinishMethodInfo
    ResolveDriveMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveDriveMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveDriveMethod "stop" o = DriveStopMethodInfo
    ResolveDriveMethod "stopFinish" o = DriveStopFinishMethodInfo
    ResolveDriveMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveDriveMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveDriveMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveDriveMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveDriveMethod "getIcon" o = DriveGetIconMethodInfo
    ResolveDriveMethod "getIdentifier" o = DriveGetIdentifierMethodInfo
    ResolveDriveMethod "getName" o = DriveGetNameMethodInfo
    ResolveDriveMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveDriveMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveDriveMethod "getSortKey" o = DriveGetSortKeyMethodInfo
    ResolveDriveMethod "getStartStopType" o = DriveGetStartStopTypeMethodInfo
    ResolveDriveMethod "getSymbolicIcon" o = DriveGetSymbolicIconMethodInfo
    ResolveDriveMethod "getVolumes" o = DriveGetVolumesMethodInfo
    ResolveDriveMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveDriveMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveDriveMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveDriveMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveDriveMethod t Drive, O.OverloadedMethod info Drive p) => OL.IsLabel t (Drive -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveDriveMethod t Drive, O.OverloadedMethod info Drive p, R.HasField t Drive p) => R.HasField t Drive p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveDriveMethod t Drive, O.OverloadedMethodInfo info Drive) => OL.IsLabel t (O.MethodProxy info Drive) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method Drive::can_eject
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_can_eject" g_drive_can_eject :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if a drive can be ejected.
driveCanEject ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@drive@/ can be ejected, 'P.False' otherwise.
driveCanEject drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_can_eject drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveCanEjectMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveCanEjectMethodInfo a signature where
    overloadedMethod = driveCanEject

instance O.OverloadedMethodInfo DriveCanEjectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveCanEject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveCanEject"
        })


#endif

-- method Drive::can_poll_for_media
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_can_poll_for_media" g_drive_can_poll_for_media :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if a drive can be polled for media changes.
driveCanPollForMedia ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@drive@/ can be polled for media changes,
    --     'P.False' otherwise.
driveCanPollForMedia drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_can_poll_for_media drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveCanPollForMediaMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveCanPollForMediaMethodInfo a signature where
    overloadedMethod = driveCanPollForMedia

instance O.OverloadedMethodInfo DriveCanPollForMediaMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveCanPollForMedia",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveCanPollForMedia"
        })


#endif

-- method Drive::can_start
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_can_start" g_drive_can_start :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if a drive can be started.
-- 
-- /Since: 2.22/
driveCanStart ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@drive@/ can be started, 'P.False' otherwise.
driveCanStart drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_can_start drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveCanStartMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveCanStartMethodInfo a signature where
    overloadedMethod = driveCanStart

instance O.OverloadedMethodInfo DriveCanStartMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveCanStart",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveCanStart"
        })


#endif

-- method Drive::can_start_degraded
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_can_start_degraded" g_drive_can_start_degraded :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if a drive can be started degraded.
-- 
-- /Since: 2.22/
driveCanStartDegraded ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@drive@/ can be started degraded, 'P.False' otherwise.
driveCanStartDegraded drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_can_start_degraded drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveCanStartDegradedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveCanStartDegradedMethodInfo a signature where
    overloadedMethod = driveCanStartDegraded

instance O.OverloadedMethodInfo DriveCanStartDegradedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveCanStartDegraded",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveCanStartDegraded"
        })


#endif

-- method Drive::can_stop
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_can_stop" g_drive_can_stop :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if a drive can be stopped.
-- 
-- /Since: 2.22/
driveCanStop ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@drive@/ can be stopped, 'P.False' otherwise.
driveCanStop drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_can_stop drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveCanStopMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveCanStopMethodInfo a signature where
    overloadedMethod = driveCanStop

instance O.OverloadedMethodInfo DriveCanStopMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveCanStop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveCanStop"
        })


#endif

-- method Drive::eject
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags affecting the unmount if required for eject"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_eject" g_drive_eject :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

{-# DEPRECATED driveEject ["(Since version 2.22)","Use 'GI.Gio.Interfaces.Drive.driveEjectWithOperation' instead."] #-}
-- | Asynchronously ejects a drive.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.Drive.driveEjectFinish' to obtain the
-- result of the operation.
driveEject ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the unmount if required for eject
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback', or 'P.Nothing'.
    -> m ()
driveEject drive flags cancellable callback = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_drive_eject drive' flags' maybeCancellable maybeCallback userData
    touchManagedPtr drive
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data DriveEjectMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsDrive a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod DriveEjectMethodInfo a signature where
    overloadedMethod = driveEject

instance O.OverloadedMethodInfo DriveEjectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveEject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveEject"
        })


#endif

-- method Drive::eject_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_drive_eject_finish" g_drive_eject_finish :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

{-# DEPRECATED driveEjectFinish ["(Since version 2.22)","Use 'GI.Gio.Interfaces.Drive.driveEjectWithOperationFinish' instead."] #-}
-- | Finishes ejecting a drive.
driveEjectFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
driveEjectFinish drive result_ = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_drive_eject_finish drive' result_'
        touchManagedPtr drive
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data DriveEjectFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod DriveEjectFinishMethodInfo a signature where
    overloadedMethod = driveEjectFinish

instance O.OverloadedMethodInfo DriveEjectFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveEjectFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveEjectFinish"
        })


#endif

-- method Drive::eject_with_operation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags affecting the unmount if required for eject"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation or %NULL to avoid\n    user interaction."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data passed to @callback."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_eject_with_operation" g_drive_eject_with_operation :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Ejects a drive. This is an asynchronous operation, and is
-- finished by calling 'GI.Gio.Interfaces.Drive.driveEjectWithOperationFinish' with the /@drive@/
-- and t'GI.Gio.Interfaces.AsyncResult.AsyncResult' data returned in the /@callback@/.
-- 
-- /Since: 2.22/
driveEjectWithOperation ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the unmount if required for eject
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation' or 'P.Nothing' to avoid
    --     user interaction.
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback', or 'P.Nothing'.
    -> m ()
driveEjectWithOperation drive flags mountOperation cancellable callback = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_drive_eject_with_operation drive' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr drive
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data DriveEjectWithOperationMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsDrive a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod DriveEjectWithOperationMethodInfo a signature where
    overloadedMethod = driveEjectWithOperation

instance O.OverloadedMethodInfo DriveEjectWithOperationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveEjectWithOperation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveEjectWithOperation"
        })


#endif

-- method Drive::eject_with_operation_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_drive_eject_with_operation_finish" g_drive_eject_with_operation_finish :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes ejecting a drive. If any errors occurred during the operation,
-- /@error@/ will be set to contain the errors and 'P.False' will be returned.
-- 
-- /Since: 2.22/
driveEjectWithOperationFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
driveEjectWithOperationFinish drive result_ = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_drive_eject_with_operation_finish drive' result_'
        touchManagedPtr drive
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data DriveEjectWithOperationFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod DriveEjectWithOperationFinishMethodInfo a signature where
    overloadedMethod = driveEjectWithOperationFinish

instance O.OverloadedMethodInfo DriveEjectWithOperationFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveEjectWithOperationFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveEjectWithOperationFinish"
        })


#endif

-- method Drive::enumerate_identifiers
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_enumerate_identifiers" g_drive_enumerate_identifiers :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO (Ptr CString)

-- | Gets the kinds of identifiers that /@drive@/ has.
-- Use 'GI.Gio.Interfaces.Drive.driveGetIdentifier' to obtain the identifiers
-- themselves.
driveEnumerateIdentifiers ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'
    -> m [T.Text]
    -- ^ __Returns:__ a 'P.Nothing'-terminated
    --     array of strings containing kinds of identifiers. Use 'GI.GLib.Functions.strfreev'
    --     to free.
driveEnumerateIdentifiers drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_enumerate_identifiers drive'
    checkUnexpectedReturnNULL "driveEnumerateIdentifiers" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveEnumerateIdentifiersMethodInfo
instance (signature ~ (m [T.Text]), MonadIO m, IsDrive a) => O.OverloadedMethod DriveEnumerateIdentifiersMethodInfo a signature where
    overloadedMethod = driveEnumerateIdentifiers

instance O.OverloadedMethodInfo DriveEnumerateIdentifiersMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveEnumerateIdentifiers",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveEnumerateIdentifiers"
        })


#endif

-- method Drive::get_icon
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Icon" })
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_get_icon" g_drive_get_icon :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO (Ptr Gio.Icon.Icon)

-- | Gets the icon for /@drive@/.
driveGetIcon ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Gio.Icon.Icon
    -- ^ __Returns:__ t'GI.Gio.Interfaces.Icon.Icon' for the /@drive@/.
    --    Free the returned object with 'GI.GObject.Objects.Object.objectUnref'.
driveGetIcon drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_get_icon drive'
    checkUnexpectedReturnNULL "driveGetIcon" result
    result' <- (wrapObject Gio.Icon.Icon) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveGetIconMethodInfo
instance (signature ~ (m Gio.Icon.Icon), MonadIO m, IsDrive a) => O.OverloadedMethod DriveGetIconMethodInfo a signature where
    overloadedMethod = driveGetIcon

instance O.OverloadedMethodInfo DriveGetIconMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveGetIcon",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveGetIcon"
        })


#endif

-- method Drive::get_identifier
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "kind"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the kind of identifier to return"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_get_identifier" g_drive_get_identifier :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    CString ->                              -- kind : TBasicType TUTF8
    IO CString

-- | Gets the identifier of the given kind for /@drive@/. The only
-- identifier currently available is
-- 'GI.Gio.Constants.DRIVE_IDENTIFIER_KIND_UNIX_DEVICE'.
driveGetIdentifier ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'
    -> T.Text
    -- ^ /@kind@/: the kind of identifier to return
    -> m (Maybe T.Text)
    -- ^ __Returns:__ a newly allocated string containing the
    --     requested identifier, or 'P.Nothing' if the t'GI.Gio.Interfaces.Drive.Drive'
    --     doesn\'t have this kind of identifier.
driveGetIdentifier drive kind = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    kind' <- textToCString kind
    result <- g_drive_get_identifier drive' kind'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr drive
    freeMem kind'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DriveGetIdentifierMethodInfo
instance (signature ~ (T.Text -> m (Maybe T.Text)), MonadIO m, IsDrive a) => O.OverloadedMethod DriveGetIdentifierMethodInfo a signature where
    overloadedMethod = driveGetIdentifier

instance O.OverloadedMethodInfo DriveGetIdentifierMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveGetIdentifier",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveGetIdentifier"
        })


#endif

-- method Drive::get_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_get_name" g_drive_get_name :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CString

-- | Gets the name of /@drive@/.
driveGetName ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m T.Text
    -- ^ __Returns:__ a string containing /@drive@/\'s name. The returned
    --     string should be freed when no longer needed.
driveGetName drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_get_name drive'
    checkUnexpectedReturnNULL "driveGetName" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveGetNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsDrive a) => O.OverloadedMethod DriveGetNameMethodInfo a signature where
    overloadedMethod = driveGetName

instance O.OverloadedMethodInfo DriveGetNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveGetName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveGetName"
        })


#endif

-- method Drive::get_sort_key
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_get_sort_key" g_drive_get_sort_key :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CString

-- | Gets the sort key for /@drive@/, if any.
-- 
-- /Since: 2.32/
driveGetSortKey ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: A t'GI.Gio.Interfaces.Drive.Drive'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ Sorting key for /@drive@/ or 'P.Nothing' if no such key is available.
driveGetSortKey drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_get_sort_key drive'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr drive
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DriveGetSortKeyMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsDrive a) => O.OverloadedMethod DriveGetSortKeyMethodInfo a signature where
    overloadedMethod = driveGetSortKey

instance O.OverloadedMethodInfo DriveGetSortKeyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveGetSortKey",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveGetSortKey"
        })


#endif

-- method Drive::get_start_stop_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "DriveStartStopType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_get_start_stop_type" g_drive_get_start_stop_type :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CUInt

-- | Gets a hint about how a drive can be started\/stopped.
-- 
-- /Since: 2.22/
driveGetStartStopType ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Gio.Enums.DriveStartStopType
    -- ^ __Returns:__ A value from the t'GI.Gio.Enums.DriveStartStopType' enumeration.
driveGetStartStopType drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_get_start_stop_type drive'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveGetStartStopTypeMethodInfo
instance (signature ~ (m Gio.Enums.DriveStartStopType), MonadIO m, IsDrive a) => O.OverloadedMethod DriveGetStartStopTypeMethodInfo a signature where
    overloadedMethod = driveGetStartStopType

instance O.OverloadedMethodInfo DriveGetStartStopTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveGetStartStopType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveGetStartStopType"
        })


#endif

-- method Drive::get_symbolic_icon
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Icon" })
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_get_symbolic_icon" g_drive_get_symbolic_icon :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO (Ptr Gio.Icon.Icon)

-- | Gets the icon for /@drive@/.
-- 
-- /Since: 2.34/
driveGetSymbolicIcon ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Gio.Icon.Icon
    -- ^ __Returns:__ symbolic t'GI.Gio.Interfaces.Icon.Icon' for the /@drive@/.
    --    Free the returned object with 'GI.GObject.Objects.Object.objectUnref'.
driveGetSymbolicIcon drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_get_symbolic_icon drive'
    checkUnexpectedReturnNULL "driveGetSymbolicIcon" result
    result' <- (wrapObject Gio.Icon.Icon) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveGetSymbolicIconMethodInfo
instance (signature ~ (m Gio.Icon.Icon), MonadIO m, IsDrive a) => O.OverloadedMethod DriveGetSymbolicIconMethodInfo a signature where
    overloadedMethod = driveGetSymbolicIcon

instance O.OverloadedMethodInfo DriveGetSymbolicIconMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveGetSymbolicIcon",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveGetSymbolicIcon"
        })


#endif

-- method Drive::get_volumes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList (TInterface Name { namespace = "Gio" , name = "Volume" }))
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_get_volumes" g_drive_get_volumes :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO (Ptr (GList (Ptr Gio.Volume.Volume)))

-- | Get a list of mountable volumes for /@drive@/.
-- 
-- The returned list should be freed with @/g_list_free()/@, after
-- its elements have been unreffed with 'GI.GObject.Objects.Object.objectUnref'.
driveGetVolumes ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m [Gio.Volume.Volume]
    -- ^ __Returns:__ t'GI.GLib.Structs.List.List' containing any t'GI.Gio.Interfaces.Volume.Volume' objects on the given /@drive@/.
driveGetVolumes drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_get_volumes drive'
    result' <- unpackGList result
    result'' <- mapM (wrapObject Gio.Volume.Volume) result'
    g_list_free result
    touchManagedPtr drive
    return result''

#if defined(ENABLE_OVERLOADING)
data DriveGetVolumesMethodInfo
instance (signature ~ (m [Gio.Volume.Volume]), MonadIO m, IsDrive a) => O.OverloadedMethod DriveGetVolumesMethodInfo a signature where
    overloadedMethod = driveGetVolumes

instance O.OverloadedMethodInfo DriveGetVolumesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveGetVolumes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveGetVolumes"
        })


#endif

-- method Drive::has_media
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_has_media" g_drive_has_media :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if the /@drive@/ has media. Note that the OS may not be polling
-- the drive for media changes; see 'GI.Gio.Interfaces.Drive.driveIsMediaCheckAutomatic'
-- for more details.
driveHasMedia ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@drive@/ has media, 'P.False' otherwise.
driveHasMedia drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_has_media drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveHasMediaMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveHasMediaMethodInfo a signature where
    overloadedMethod = driveHasMedia

instance O.OverloadedMethodInfo DriveHasMediaMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveHasMedia",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveHasMedia"
        })


#endif

-- method Drive::has_volumes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_has_volumes" g_drive_has_volumes :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Check if /@drive@/ has any mountable volumes.
driveHasVolumes ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@drive@/ contains volumes, 'P.False' otherwise.
driveHasVolumes drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_has_volumes drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveHasVolumesMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveHasVolumesMethodInfo a signature where
    overloadedMethod = driveHasVolumes

instance O.OverloadedMethodInfo DriveHasVolumesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveHasVolumes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveHasVolumes"
        })


#endif

-- method Drive::is_media_check_automatic
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_is_media_check_automatic" g_drive_is_media_check_automatic :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if /@drive@/ is capable of automatically detecting media changes.
driveIsMediaCheckAutomatic ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@drive@/ is capable of automatically detecting
    --     media changes, 'P.False' otherwise.
driveIsMediaCheckAutomatic drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_is_media_check_automatic drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveIsMediaCheckAutomaticMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveIsMediaCheckAutomaticMethodInfo a signature where
    overloadedMethod = driveIsMediaCheckAutomatic

instance O.OverloadedMethodInfo DriveIsMediaCheckAutomaticMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveIsMediaCheckAutomatic",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveIsMediaCheckAutomatic"
        })


#endif

-- method Drive::is_media_removable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_is_media_removable" g_drive_is_media_removable :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if the /@drive@/ supports removable media.
driveIsMediaRemovable ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@drive@/ supports removable media, 'P.False' otherwise.
driveIsMediaRemovable drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_is_media_removable drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveIsMediaRemovableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveIsMediaRemovableMethodInfo a signature where
    overloadedMethod = driveIsMediaRemovable

instance O.OverloadedMethodInfo DriveIsMediaRemovableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveIsMediaRemovable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveIsMediaRemovable"
        })


#endif

-- method Drive::is_removable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_is_removable" g_drive_is_removable :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    IO CInt

-- | Checks if the t'GI.Gio.Interfaces.Drive.Drive' and\/or its media is considered removable by the user.
-- See 'GI.Gio.Interfaces.Drive.driveIsMediaRemovable'.
-- 
-- /Since: 2.50/
driveIsRemovable ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@drive@/ and\/or its media is considered removable, 'P.False' otherwise.
driveIsRemovable drive = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result <- g_drive_is_removable drive'
    let result' = (/= 0) result
    touchManagedPtr drive
    return result'

#if defined(ENABLE_OVERLOADING)
data DriveIsRemovableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDrive a) => O.OverloadedMethod DriveIsRemovableMethodInfo a signature where
    overloadedMethod = driveIsRemovable

instance O.OverloadedMethodInfo DriveIsRemovableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveIsRemovable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveIsRemovable"
        })


#endif

-- method Drive::poll_for_media
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 3
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_poll_for_media" g_drive_poll_for_media :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously polls /@drive@/ to see if media has been inserted or removed.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.Drive.drivePollForMediaFinish' to obtain the
-- result of the operation.
drivePollForMedia ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback', or 'P.Nothing'.
    -> m ()
drivePollForMedia drive cancellable callback = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_drive_poll_for_media drive' maybeCancellable maybeCallback userData
    touchManagedPtr drive
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data DrivePollForMediaMethodInfo
instance (signature ~ (Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsDrive a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod DrivePollForMediaMethodInfo a signature where
    overloadedMethod = drivePollForMedia

instance O.OverloadedMethodInfo DrivePollForMediaMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.drivePollForMedia",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:drivePollForMedia"
        })


#endif

-- method Drive::poll_for_media_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_drive_poll_for_media_finish" g_drive_poll_for_media_finish :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an operation started with 'GI.Gio.Interfaces.Drive.drivePollForMedia' on a drive.
drivePollForMediaFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
drivePollForMediaFinish drive result_ = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_drive_poll_for_media_finish drive' result_'
        touchManagedPtr drive
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data DrivePollForMediaFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod DrivePollForMediaFinishMethodInfo a signature where
    overloadedMethod = drivePollForMediaFinish

instance O.OverloadedMethodInfo DrivePollForMediaFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.drivePollForMediaFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:drivePollForMediaFinish"
        })


#endif

-- method Drive::start
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DriveStartFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags affecting the start operation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation or %NULL to avoid\n    user interaction."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_start" g_drive_start :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DriveStartFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously starts a drive.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.Drive.driveStartFinish' to obtain the
-- result of the operation.
-- 
-- /Since: 2.22/
driveStart ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> [Gio.Flags.DriveStartFlags]
    -- ^ /@flags@/: flags affecting the start operation.
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation' or 'P.Nothing' to avoid
    --     user interaction.
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback', or 'P.Nothing'.
    -> m ()
driveStart drive flags mountOperation cancellable callback = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_drive_start drive' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr drive
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data DriveStartMethodInfo
instance (signature ~ ([Gio.Flags.DriveStartFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsDrive a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod DriveStartMethodInfo a signature where
    overloadedMethod = driveStart

instance O.OverloadedMethodInfo DriveStartMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveStart",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveStart"
        })


#endif

-- method Drive::start_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_drive_start_finish" g_drive_start_finish :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes starting a drive.
-- 
-- /Since: 2.22/
driveStartFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
driveStartFinish drive result_ = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_drive_start_finish drive' result_'
        touchManagedPtr drive
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data DriveStartFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod DriveStartFinishMethodInfo a signature where
    overloadedMethod = driveStartFinish

instance O.OverloadedMethodInfo DriveStartFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveStartFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveStartFinish"
        })


#endif

-- method Drive::stop
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountUnmountFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags affecting the unmount if required for stopping."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mount_operation"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MountOperation" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GMountOperation or %NULL to avoid\n    user interaction."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncReadyCallback, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_drive_stop" g_drive_stop :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "MountUnmountFlags"})
    Ptr Gio.MountOperation.MountOperation -> -- mount_operation : TInterface (Name {namespace = "Gio", name = "MountOperation"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously stops a drive.
-- 
-- When the operation is finished, /@callback@/ will be called.
-- You can then call 'GI.Gio.Interfaces.Drive.driveStopFinish' to obtain the
-- result of the operation.
-- 
-- /Since: 2.22/
driveStop ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> [Gio.Flags.MountUnmountFlags]
    -- ^ /@flags@/: flags affecting the unmount if required for stopping.
    -> Maybe (b)
    -- ^ /@mountOperation@/: a t'GI.Gio.Objects.MountOperation.MountOperation' or 'P.Nothing' to avoid
    --     user interaction.
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback', or 'P.Nothing'.
    -> m ()
driveStop drive flags mountOperation cancellable callback = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    let flags' = gflagsToWord flags
    maybeMountOperation <- case mountOperation of
        Nothing -> return FP.nullPtr
        Just jMountOperation -> do
            jMountOperation' <- unsafeManagedPtrCastPtr jMountOperation
            return jMountOperation'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_drive_stop drive' flags' maybeMountOperation maybeCancellable maybeCallback userData
    touchManagedPtr drive
    whenJust mountOperation touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data DriveStopMethodInfo
instance (signature ~ ([Gio.Flags.MountUnmountFlags] -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsDrive a, Gio.MountOperation.IsMountOperation b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod DriveStopMethodInfo a signature where
    overloadedMethod = driveStop

instance O.OverloadedMethodInfo DriveStopMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveStop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveStop"
        })


#endif

-- method Drive::stop_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "drive"
--           , argType = TInterface Name { namespace = "Gio" , name = "Drive" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDrive." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_drive_stop_finish" g_drive_stop_finish :: 
    Ptr Drive ->                            -- drive : TInterface (Name {namespace = "Gio", name = "Drive"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes stopping a drive.
-- 
-- /Since: 2.22/
driveStopFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@drive@/: a t'GI.Gio.Interfaces.Drive.Drive'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
driveStopFinish drive result_ = liftIO $ do
    drive' <- unsafeManagedPtrCastPtr drive
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_drive_stop_finish drive' result_'
        touchManagedPtr drive
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data DriveStopFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsDrive a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod DriveStopFinishMethodInfo a signature where
    overloadedMethod = driveStopFinish

instance O.OverloadedMethodInfo DriveStopFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive.driveStopFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#v:driveStopFinish"
        })


#endif

-- signal Drive::changed
-- | Emitted when the drive\'s state has changed.
type DriveChangedCallback =
    IO ()

type C_DriveChangedCallback =
    Ptr Drive ->                            -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DriveChangedCallback`.
foreign import ccall "wrapper"
    mk_DriveChangedCallback :: C_DriveChangedCallback -> IO (FunPtr C_DriveChangedCallback)

wrap_DriveChangedCallback :: 
    GObject a => (a -> DriveChangedCallback) ->
    C_DriveChangedCallback
wrap_DriveChangedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [changed](#signal:changed) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' drive #changed callback
-- @
-- 
-- 
onDriveChanged :: (IsDrive a, MonadIO m) => a -> ((?self :: a) => DriveChangedCallback) -> m SignalHandlerId
onDriveChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DriveChangedCallback wrapped
    wrapped'' <- mk_DriveChangedCallback wrapped'
    connectSignalFunPtr obj "changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [changed](#signal:changed) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' drive #changed callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDriveChanged :: (IsDrive a, MonadIO m) => a -> ((?self :: a) => DriveChangedCallback) -> m SignalHandlerId
afterDriveChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DriveChangedCallback wrapped
    wrapped'' <- mk_DriveChangedCallback wrapped'
    connectSignalFunPtr obj "changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DriveChangedSignalInfo
instance SignalInfo DriveChangedSignalInfo where
    type HaskellCallbackType DriveChangedSignalInfo = DriveChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DriveChangedCallback cb
        cb'' <- mk_DriveChangedCallback cb'
        connectSignalFunPtr obj "changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive::changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#g:signal:changed"})

#endif

-- signal Drive::disconnected
-- | This signal is emitted when the t'GI.Gio.Interfaces.Drive.Drive' have been
-- disconnected. If the recipient is holding references to the
-- object they should release them so the object can be
-- finalized.
type DriveDisconnectedCallback =
    IO ()

type C_DriveDisconnectedCallback =
    Ptr Drive ->                            -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DriveDisconnectedCallback`.
foreign import ccall "wrapper"
    mk_DriveDisconnectedCallback :: C_DriveDisconnectedCallback -> IO (FunPtr C_DriveDisconnectedCallback)

wrap_DriveDisconnectedCallback :: 
    GObject a => (a -> DriveDisconnectedCallback) ->
    C_DriveDisconnectedCallback
wrap_DriveDisconnectedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [disconnected](#signal:disconnected) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' drive #disconnected callback
-- @
-- 
-- 
onDriveDisconnected :: (IsDrive a, MonadIO m) => a -> ((?self :: a) => DriveDisconnectedCallback) -> m SignalHandlerId
onDriveDisconnected obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DriveDisconnectedCallback wrapped
    wrapped'' <- mk_DriveDisconnectedCallback wrapped'
    connectSignalFunPtr obj "disconnected" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [disconnected](#signal:disconnected) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' drive #disconnected callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDriveDisconnected :: (IsDrive a, MonadIO m) => a -> ((?self :: a) => DriveDisconnectedCallback) -> m SignalHandlerId
afterDriveDisconnected obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DriveDisconnectedCallback wrapped
    wrapped'' <- mk_DriveDisconnectedCallback wrapped'
    connectSignalFunPtr obj "disconnected" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DriveDisconnectedSignalInfo
instance SignalInfo DriveDisconnectedSignalInfo where
    type HaskellCallbackType DriveDisconnectedSignalInfo = DriveDisconnectedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DriveDisconnectedCallback cb
        cb'' <- mk_DriveDisconnectedCallback cb'
        connectSignalFunPtr obj "disconnected" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive::disconnected"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#g:signal:disconnected"})

#endif

-- signal Drive::eject-button
-- | Emitted when the physical eject button (if any) of a drive has
-- been pressed.
type DriveEjectButtonCallback =
    IO ()

type C_DriveEjectButtonCallback =
    Ptr Drive ->                            -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DriveEjectButtonCallback`.
foreign import ccall "wrapper"
    mk_DriveEjectButtonCallback :: C_DriveEjectButtonCallback -> IO (FunPtr C_DriveEjectButtonCallback)

wrap_DriveEjectButtonCallback :: 
    GObject a => (a -> DriveEjectButtonCallback) ->
    C_DriveEjectButtonCallback
wrap_DriveEjectButtonCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [ejectButton](#signal:ejectButton) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' drive #ejectButton callback
-- @
-- 
-- 
onDriveEjectButton :: (IsDrive a, MonadIO m) => a -> ((?self :: a) => DriveEjectButtonCallback) -> m SignalHandlerId
onDriveEjectButton obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DriveEjectButtonCallback wrapped
    wrapped'' <- mk_DriveEjectButtonCallback wrapped'
    connectSignalFunPtr obj "eject-button" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [ejectButton](#signal:ejectButton) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' drive #ejectButton callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDriveEjectButton :: (IsDrive a, MonadIO m) => a -> ((?self :: a) => DriveEjectButtonCallback) -> m SignalHandlerId
afterDriveEjectButton obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DriveEjectButtonCallback wrapped
    wrapped'' <- mk_DriveEjectButtonCallback wrapped'
    connectSignalFunPtr obj "eject-button" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DriveEjectButtonSignalInfo
instance SignalInfo DriveEjectButtonSignalInfo where
    type HaskellCallbackType DriveEjectButtonSignalInfo = DriveEjectButtonCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DriveEjectButtonCallback cb
        cb'' <- mk_DriveEjectButtonCallback cb'
        connectSignalFunPtr obj "eject-button" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive::eject-button"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#g:signal:ejectButton"})

#endif

-- signal Drive::stop-button
-- | Emitted when the physical stop button (if any) of a drive has
-- been pressed.
-- 
-- /Since: 2.22/
type DriveStopButtonCallback =
    IO ()

type C_DriveStopButtonCallback =
    Ptr Drive ->                            -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DriveStopButtonCallback`.
foreign import ccall "wrapper"
    mk_DriveStopButtonCallback :: C_DriveStopButtonCallback -> IO (FunPtr C_DriveStopButtonCallback)

wrap_DriveStopButtonCallback :: 
    GObject a => (a -> DriveStopButtonCallback) ->
    C_DriveStopButtonCallback
wrap_DriveStopButtonCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [stopButton](#signal:stopButton) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' drive #stopButton callback
-- @
-- 
-- 
onDriveStopButton :: (IsDrive a, MonadIO m) => a -> ((?self :: a) => DriveStopButtonCallback) -> m SignalHandlerId
onDriveStopButton obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DriveStopButtonCallback wrapped
    wrapped'' <- mk_DriveStopButtonCallback wrapped'
    connectSignalFunPtr obj "stop-button" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [stopButton](#signal:stopButton) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' drive #stopButton callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDriveStopButton :: (IsDrive a, MonadIO m) => a -> ((?self :: a) => DriveStopButtonCallback) -> m SignalHandlerId
afterDriveStopButton obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DriveStopButtonCallback wrapped
    wrapped'' <- mk_DriveStopButtonCallback wrapped'
    connectSignalFunPtr obj "stop-button" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DriveStopButtonSignalInfo
instance SignalInfo DriveStopButtonSignalInfo where
    type HaskellCallbackType DriveStopButtonSignalInfo = DriveStopButtonCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DriveStopButtonCallback cb
        cb'' <- mk_DriveStopButtonCallback cb'
        connectSignalFunPtr obj "stop-button" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Drive::stop-button"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-Drive.html#g:signal:stopButton"})

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Drive = DriveSignalList
type DriveSignalList = ('[ '("changed", DriveChangedSignalInfo), '("disconnected", DriveDisconnectedSignalInfo), '("ejectButton", DriveEjectButtonSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("stopButton", DriveStopButtonSignalInfo)] :: [(Symbol, DK.Type)])

#endif


