{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GActionGroup@ represents a group of actions.
-- 
-- Actions can be used to expose functionality in a structured way, either
-- from one part of a program to another, or to the outside world. Action
-- groups are often used together with a [type/@gio@/.MenuModel] that provides additional
-- representation data for displaying the actions to the user, e.g. in a menu.
-- 
-- The main way to interact with the actions in a @GActionGroup@ is to
-- activate them with 'GI.Gio.Interfaces.ActionGroup.actionGroupActivateAction'. Activating an
-- action may require a [type/@gLib@/.Variant] parameter. The required type of the
-- parameter can be inquired with 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionParameterType'.
-- Actions may be disabled, see 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionEnabled'.
-- Activating a disabled action has no effect.
-- 
-- Actions may optionally have a state in the form of a [type/@gLib@/.Variant]. The current
-- state of an action can be inquired with 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionState'.
-- Activating a stateful action may change its state, but it is also possible to
-- set the state by calling 'GI.Gio.Interfaces.ActionGroup.actionGroupChangeActionState'.
-- 
-- As typical example, consider a text editing application which has an
-- option to change the current font to ‘bold’. A good way to represent
-- this would be a stateful action, with a boolean state. Activating the
-- action would toggle the state.
-- 
-- Each action in the group has a unique name (which is a string).  All
-- method calls, except 'GI.Gio.Interfaces.ActionGroup.actionGroupListActions' take the name of
-- an action as an argument.
-- 
-- The @GActionGroup@ API is meant to be the ‘public’ API to the action
-- group. The calls here are exactly the interaction that ‘external
-- forces’ (eg: UI, incoming D-Bus messages, etc.) are supposed to have
-- with actions. ‘Internal’ APIs (ie: ones meant only to be accessed by
-- the action group implementation) are found on subclasses. This is
-- why you will find – for example – 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionEnabled'
-- but not an equivalent @set_action_enabled()@ method.
-- 
-- Signals are emitted on the action group in response to state changes
-- on individual actions.
-- 
-- Implementations of @GActionGroup@ should provide implementations for
-- the virtual functions 'GI.Gio.Interfaces.ActionGroup.actionGroupListActions' and
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupQueryAction'. The other virtual functions should
-- not be implemented — their ‘wrappers’ are actually implemented with
-- calls to 'GI.Gio.Interfaces.ActionGroup.actionGroupQueryAction'.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Interfaces.ActionGroup
    ( 

-- * Exported types
    ActionGroup(..)                         ,
    IsActionGroup                           ,
    toActionGroup                           ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [actionAdded]("GI.Gio.Interfaces.ActionGroup#g:method:actionAdded"), [actionEnabledChanged]("GI.Gio.Interfaces.ActionGroup#g:method:actionEnabledChanged"), [actionRemoved]("GI.Gio.Interfaces.ActionGroup#g:method:actionRemoved"), [actionStateChanged]("GI.Gio.Interfaces.ActionGroup#g:method:actionStateChanged"), [activateAction]("GI.Gio.Interfaces.ActionGroup#g:method:activateAction"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [changeActionState]("GI.Gio.Interfaces.ActionGroup#g:method:changeActionState"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [hasAction]("GI.Gio.Interfaces.ActionGroup#g:method:hasAction"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [listActions]("GI.Gio.Interfaces.ActionGroup#g:method:listActions"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [queryAction]("GI.Gio.Interfaces.ActionGroup#g:method:queryAction"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getActionEnabled]("GI.Gio.Interfaces.ActionGroup#g:method:getActionEnabled"), [getActionParameterType]("GI.Gio.Interfaces.ActionGroup#g:method:getActionParameterType"), [getActionState]("GI.Gio.Interfaces.ActionGroup#g:method:getActionState"), [getActionStateHint]("GI.Gio.Interfaces.ActionGroup#g:method:getActionStateHint"), [getActionStateType]("GI.Gio.Interfaces.ActionGroup#g:method:getActionStateType"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveActionGroupMethod                ,
#endif

-- ** actionAdded #method:actionAdded#

#if defined(ENABLE_OVERLOADING)
    ActionGroupActionAddedMethodInfo        ,
#endif
    actionGroupActionAdded                  ,


-- ** actionEnabledChanged #method:actionEnabledChanged#

#if defined(ENABLE_OVERLOADING)
    ActionGroupActionEnabledChangedMethodInfo,
#endif
    actionGroupActionEnabledChanged         ,


-- ** actionRemoved #method:actionRemoved#

#if defined(ENABLE_OVERLOADING)
    ActionGroupActionRemovedMethodInfo      ,
#endif
    actionGroupActionRemoved                ,


-- ** actionStateChanged #method:actionStateChanged#

#if defined(ENABLE_OVERLOADING)
    ActionGroupActionStateChangedMethodInfo ,
#endif
    actionGroupActionStateChanged           ,


-- ** activateAction #method:activateAction#

#if defined(ENABLE_OVERLOADING)
    ActionGroupActivateActionMethodInfo     ,
#endif
    actionGroupActivateAction               ,


-- ** changeActionState #method:changeActionState#

#if defined(ENABLE_OVERLOADING)
    ActionGroupChangeActionStateMethodInfo  ,
#endif
    actionGroupChangeActionState            ,


-- ** getActionEnabled #method:getActionEnabled#

#if defined(ENABLE_OVERLOADING)
    ActionGroupGetActionEnabledMethodInfo   ,
#endif
    actionGroupGetActionEnabled             ,


-- ** getActionParameterType #method:getActionParameterType#

#if defined(ENABLE_OVERLOADING)
    ActionGroupGetActionParameterTypeMethodInfo,
#endif
    actionGroupGetActionParameterType       ,


-- ** getActionState #method:getActionState#

#if defined(ENABLE_OVERLOADING)
    ActionGroupGetActionStateMethodInfo     ,
#endif
    actionGroupGetActionState               ,


-- ** getActionStateHint #method:getActionStateHint#

#if defined(ENABLE_OVERLOADING)
    ActionGroupGetActionStateHintMethodInfo ,
#endif
    actionGroupGetActionStateHint           ,


-- ** getActionStateType #method:getActionStateType#

#if defined(ENABLE_OVERLOADING)
    ActionGroupGetActionStateTypeMethodInfo ,
#endif
    actionGroupGetActionStateType           ,


-- ** hasAction #method:hasAction#

#if defined(ENABLE_OVERLOADING)
    ActionGroupHasActionMethodInfo          ,
#endif
    actionGroupHasAction                    ,


-- ** listActions #method:listActions#

#if defined(ENABLE_OVERLOADING)
    ActionGroupListActionsMethodInfo        ,
#endif
    actionGroupListActions                  ,


-- ** queryAction #method:queryAction#

#if defined(ENABLE_OVERLOADING)
    ActionGroupQueryActionMethodInfo        ,
#endif
    actionGroupQueryAction                  ,




 -- * Signals


-- ** actionAdded #signal:actionAdded#

    ActionGroupActionAddedCallback          ,
#if defined(ENABLE_OVERLOADING)
    ActionGroupActionAddedSignalInfo        ,
#endif
    afterActionGroupActionAdded             ,
    onActionGroupActionAdded                ,


-- ** actionEnabledChanged #signal:actionEnabledChanged#

    ActionGroupActionEnabledChangedCallback ,
#if defined(ENABLE_OVERLOADING)
    ActionGroupActionEnabledChangedSignalInfo,
#endif
    afterActionGroupActionEnabledChanged    ,
    onActionGroupActionEnabledChanged       ,


-- ** actionRemoved #signal:actionRemoved#

    ActionGroupActionRemovedCallback        ,
#if defined(ENABLE_OVERLOADING)
    ActionGroupActionRemovedSignalInfo      ,
#endif
    afterActionGroupActionRemoved           ,
    onActionGroupActionRemoved              ,


-- ** actionStateChanged #signal:actionStateChanged#

    ActionGroupActionStateChangedCallback   ,
#if defined(ENABLE_OVERLOADING)
    ActionGroupActionStateChangedSignalInfo ,
#endif
    afterActionGroupActionStateChanged      ,
    onActionGroupActionStateChanged         ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object

#else
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object

#endif

-- interface ActionGroup 
-- | Memory-managed wrapper type.
newtype ActionGroup = ActionGroup (SP.ManagedPtr ActionGroup)
    deriving (Eq)

instance SP.ManagedPtrNewtype ActionGroup where
    toManagedPtr (ActionGroup p) = p

foreign import ccall "g_action_group_get_type"
    c_g_action_group_get_type :: IO B.Types.GType

instance B.Types.TypedObject ActionGroup where
    glibType = c_g_action_group_get_type

instance B.Types.GObject ActionGroup

-- | Type class for types which can be safely cast to `ActionGroup`, for instance with `toActionGroup`.
class (SP.GObject o, O.IsDescendantOf ActionGroup o) => IsActionGroup o
instance (SP.GObject o, O.IsDescendantOf ActionGroup o) => IsActionGroup o

instance O.HasParentTypes ActionGroup
type instance O.ParentTypes ActionGroup = '[GObject.Object.Object]

-- | Cast to `ActionGroup`, for types for which this is known to be safe. For general casts, use `Data.GI.Base.ManagedPtr.castTo`.
toActionGroup :: (MIO.MonadIO m, IsActionGroup o) => o -> m ActionGroup
toActionGroup = MIO.liftIO . B.ManagedPtr.unsafeCastTo ActionGroup

-- | Convert 'ActionGroup' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe ActionGroup) where
    gvalueGType_ = c_g_action_group_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr ActionGroup)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr ActionGroup)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject ActionGroup ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList ActionGroup
type instance O.AttributeList ActionGroup = ActionGroupAttributeList
type ActionGroupAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveActionGroupMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveActionGroupMethod "actionAdded" o = ActionGroupActionAddedMethodInfo
    ResolveActionGroupMethod "actionEnabledChanged" o = ActionGroupActionEnabledChangedMethodInfo
    ResolveActionGroupMethod "actionRemoved" o = ActionGroupActionRemovedMethodInfo
    ResolveActionGroupMethod "actionStateChanged" o = ActionGroupActionStateChangedMethodInfo
    ResolveActionGroupMethod "activateAction" o = ActionGroupActivateActionMethodInfo
    ResolveActionGroupMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveActionGroupMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveActionGroupMethod "changeActionState" o = ActionGroupChangeActionStateMethodInfo
    ResolveActionGroupMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveActionGroupMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveActionGroupMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveActionGroupMethod "hasAction" o = ActionGroupHasActionMethodInfo
    ResolveActionGroupMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveActionGroupMethod "listActions" o = ActionGroupListActionsMethodInfo
    ResolveActionGroupMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveActionGroupMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveActionGroupMethod "queryAction" o = ActionGroupQueryActionMethodInfo
    ResolveActionGroupMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveActionGroupMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveActionGroupMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveActionGroupMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveActionGroupMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveActionGroupMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveActionGroupMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveActionGroupMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveActionGroupMethod "getActionEnabled" o = ActionGroupGetActionEnabledMethodInfo
    ResolveActionGroupMethod "getActionParameterType" o = ActionGroupGetActionParameterTypeMethodInfo
    ResolveActionGroupMethod "getActionState" o = ActionGroupGetActionStateMethodInfo
    ResolveActionGroupMethod "getActionStateHint" o = ActionGroupGetActionStateHintMethodInfo
    ResolveActionGroupMethod "getActionStateType" o = ActionGroupGetActionStateTypeMethodInfo
    ResolveActionGroupMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveActionGroupMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveActionGroupMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveActionGroupMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveActionGroupMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveActionGroupMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveActionGroupMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveActionGroupMethod t ActionGroup, O.OverloadedMethod info ActionGroup p) => OL.IsLabel t (ActionGroup -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveActionGroupMethod t ActionGroup, O.OverloadedMethod info ActionGroup p, R.HasField t ActionGroup p) => R.HasField t ActionGroup p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveActionGroupMethod t ActionGroup, O.OverloadedMethodInfo info ActionGroup) => OL.IsLabel t (O.MethodProxy info ActionGroup) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method ActionGroup::action_added
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of an action in the group"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_action_added" g_action_group_action_added :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    IO ()

-- | Emits the [ActionGroup::actionAdded]("GI.Gio.Interfaces.ActionGroup#g:signal:actionAdded") signal on /@actionGroup@/.
-- 
-- This function should only be called by [type/@gio@/.ActionGroup] implementations.
-- 
-- /Since: 2.28/
actionGroupActionAdded ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of an action in the group
    -> m ()
actionGroupActionAdded actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    g_action_group_action_added actionGroup' actionName'
    touchManagedPtr actionGroup
    freeMem actionName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ActionGroupActionAddedMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupActionAddedMethodInfo a signature where
    overloadedMethod = actionGroupActionAdded

instance O.OverloadedMethodInfo ActionGroupActionAddedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupActionAdded",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupActionAdded"
        })


#endif

-- method ActionGroup::action_enabled_changed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of an action in the group"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "enabled"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether the action is now enabled"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_action_enabled_changed" g_action_group_action_enabled_changed :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    CInt ->                                 -- enabled : TBasicType TBoolean
    IO ()

-- | Emits the [ActionGroup::actionEnabledChanged]("GI.Gio.Interfaces.ActionGroup#g:signal:actionEnabledChanged") signal on /@actionGroup@/.
-- 
-- This function should only be called by [type/@gio@/.ActionGroup] implementations.
-- 
-- /Since: 2.28/
actionGroupActionEnabledChanged ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of an action in the group
    -> Bool
    -- ^ /@enabled@/: whether the action is now enabled
    -> m ()
actionGroupActionEnabledChanged actionGroup actionName enabled = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    let enabled' = (P.fromIntegral . P.fromEnum) enabled
    g_action_group_action_enabled_changed actionGroup' actionName' enabled'
    touchManagedPtr actionGroup
    freeMem actionName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ActionGroupActionEnabledChangedMethodInfo
instance (signature ~ (T.Text -> Bool -> m ()), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupActionEnabledChangedMethodInfo a signature where
    overloadedMethod = actionGroupActionEnabledChanged

instance O.OverloadedMethodInfo ActionGroupActionEnabledChangedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupActionEnabledChanged",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupActionEnabledChanged"
        })


#endif

-- method ActionGroup::action_removed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of an action in the group"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_action_removed" g_action_group_action_removed :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    IO ()

-- | Emits the [ActionGroup::actionRemoved]("GI.Gio.Interfaces.ActionGroup#g:signal:actionRemoved") signal on /@actionGroup@/.
-- 
-- This function should only be called by [type/@gio@/.ActionGroup] implementations.
-- 
-- /Since: 2.28/
actionGroupActionRemoved ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of an action in the group
    -> m ()
actionGroupActionRemoved actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    g_action_group_action_removed actionGroup' actionName'
    touchManagedPtr actionGroup
    freeMem actionName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ActionGroupActionRemovedMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupActionRemovedMethodInfo a signature where
    overloadedMethod = actionGroupActionRemoved

instance O.OverloadedMethodInfo ActionGroupActionRemovedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupActionRemoved",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupActionRemoved"
        })


#endif

-- method ActionGroup::action_state_changed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of an action in the group"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new state of the named action"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_action_state_changed" g_action_group_action_state_changed :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    Ptr GVariant ->                         -- state : TVariant
    IO ()

-- | Emits the [ActionGroup::actionStateChanged]("GI.Gio.Interfaces.ActionGroup#g:signal:actionStateChanged") signal on /@actionGroup@/.
-- 
-- This function should only be called by [type/@gio@/.ActionGroup] implementations.
-- 
-- /Since: 2.28/
actionGroupActionStateChanged ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of an action in the group
    -> GVariant
    -- ^ /@state@/: the new state of the named action
    -> m ()
actionGroupActionStateChanged actionGroup actionName state = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    state' <- unsafeManagedPtrGetPtr state
    g_action_group_action_state_changed actionGroup' actionName' state'
    touchManagedPtr actionGroup
    touchManagedPtr state
    freeMem actionName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ActionGroupActionStateChangedMethodInfo
instance (signature ~ (T.Text -> GVariant -> m ()), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupActionStateChangedMethodInfo a signature where
    overloadedMethod = actionGroupActionStateChanged

instance O.OverloadedMethodInfo ActionGroupActionStateChangedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupActionStateChanged",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupActionStateChanged"
        })


#endif

-- method ActionGroup::activate_action
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to activate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameter"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "parameters to the activation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_activate_action" g_action_group_activate_action :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    Ptr GVariant ->                         -- parameter : TVariant
    IO ()

-- | Activate the named action within /@actionGroup@/.
-- 
-- If the action is expecting a parameter, then the correct type of
-- parameter must be given as /@parameter@/.  If the action is expecting no
-- parameters then /@parameter@/ must be @NULL@.  See
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionParameterType'.
-- 
-- If the [type/@gio@/.ActionGroup] implementation supports asynchronous remote
-- activation over D-Bus, this call may return before the relevant
-- D-Bus traffic has been sent, or any replies have been received. In
-- order to block on such asynchronous activation calls,
-- 'GI.Gio.Objects.DBusConnection.dBusConnectionFlush' should be called prior to the code, which
-- depends on the result of the action activation. Without flushing
-- the D-Bus connection, there is no guarantee that the action would
-- have been activated.
-- 
-- The following code which runs in a remote app instance, shows an
-- example of a ‘quit’ action being activated on the primary app
-- instance over D-Bus. Here 'GI.Gio.Objects.DBusConnection.dBusConnectionFlush' is called
-- before @exit()@. Without @g_dbus_connection_flush()@, the ‘quit’ action
-- may fail to be activated on the primary instance.
-- 
-- 
-- === /c code/
-- >// call ‘quit’ action on primary instance
-- >g_action_group_activate_action (G_ACTION_GROUP (app), "quit", NULL);
-- >
-- >// make sure the action is activated now
-- >g_dbus_connection_flush (…);
-- >
-- >g_debug ("Application has been terminated. Exiting.");
-- >
-- >exit (0);
-- 
-- 
-- /Since: 2.28/
actionGroupActivateAction ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of the action to activate
    -> Maybe (GVariant)
    -- ^ /@parameter@/: parameters to the activation
    -> m ()
actionGroupActivateAction actionGroup actionName parameter = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    maybeParameter <- case parameter of
        Nothing -> return FP.nullPtr
        Just jParameter -> do
            jParameter' <- unsafeManagedPtrGetPtr jParameter
            return jParameter'
    g_action_group_activate_action actionGroup' actionName' maybeParameter
    touchManagedPtr actionGroup
    whenJust parameter touchManagedPtr
    freeMem actionName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ActionGroupActivateActionMethodInfo
instance (signature ~ (T.Text -> Maybe (GVariant) -> m ()), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupActivateActionMethodInfo a signature where
    overloadedMethod = actionGroupActivateAction

instance O.OverloadedMethodInfo ActionGroupActivateActionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupActivateAction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupActivateAction"
        })


#endif

-- method ActionGroup::change_action_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the name of the action to request the change on"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new state" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_change_action_state" g_action_group_change_action_state :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    Ptr GVariant ->                         -- value : TVariant
    IO ()

-- | Request for the state of the named action within /@actionGroup@/ to be
-- changed to /@value@/.
-- 
-- The action must be stateful and /@value@/ must be of the correct type.
-- See 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionStateType'.
-- 
-- This call merely requests a change.  The action may refuse to change
-- its state or may change its state to something other than /@value@/.
-- See 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionStateHint'.
-- 
-- If the /@value@/ GVariant is floating, it is consumed.
-- 
-- /Since: 2.28/
actionGroupChangeActionState ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of the action to request the change on
    -> GVariant
    -- ^ /@value@/: the new state
    -> m ()
actionGroupChangeActionState actionGroup actionName value = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    value' <- unsafeManagedPtrGetPtr value
    g_action_group_change_action_state actionGroup' actionName' value'
    touchManagedPtr actionGroup
    touchManagedPtr value
    freeMem actionName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ActionGroupChangeActionStateMethodInfo
instance (signature ~ (T.Text -> GVariant -> m ()), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupChangeActionStateMethodInfo a signature where
    overloadedMethod = actionGroupChangeActionState

instance O.OverloadedMethodInfo ActionGroupChangeActionStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupChangeActionState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupChangeActionState"
        })


#endif

-- method ActionGroup::get_action_enabled
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_get_action_enabled" g_action_group_get_action_enabled :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    IO CInt

-- | Checks if the named action within /@actionGroup@/ is currently enabled.
-- 
-- An action must be enabled in order to be activated or in order to
-- have its state changed from outside callers.
-- 
-- /Since: 2.28/
actionGroupGetActionEnabled ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of the action to query
    -> m Bool
    -- ^ __Returns:__ whether the action is currently enabled
actionGroupGetActionEnabled actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    result <- g_action_group_get_action_enabled actionGroup' actionName'
    let result' = (/= 0) result
    touchManagedPtr actionGroup
    freeMem actionName'
    return result'

#if defined(ENABLE_OVERLOADING)
data ActionGroupGetActionEnabledMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupGetActionEnabledMethodInfo a signature where
    overloadedMethod = actionGroupGetActionEnabled

instance O.OverloadedMethodInfo ActionGroupGetActionEnabledMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupGetActionEnabled",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupGetActionEnabled"
        })


#endif

-- method ActionGroup::get_action_parameter_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_get_action_parameter_type" g_action_group_get_action_parameter_type :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    IO (Ptr GLib.VariantType.VariantType)

-- | Queries the type of the parameter that must be given when activating
-- the named action within /@actionGroup@/.
-- 
-- When activating the action using 'GI.Gio.Interfaces.ActionGroup.actionGroupActivateAction',
-- the [type/@gLib@/.Variant] given to that function must be of the type returned
-- by this function.
-- 
-- In the case that this function returns @NULL@, you must not give any
-- [type/@gLib@/.Variant], but @NULL@ instead.
-- 
-- The parameter type of a particular action will never change but it is
-- possible for an action to be removed and for a new action to be added
-- with the same name but a different parameter type.
-- 
-- /Since: 2.28/
actionGroupGetActionParameterType ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of the action to query
    -> m (Maybe GLib.VariantType.VariantType)
    -- ^ __Returns:__ the parameter type
actionGroupGetActionParameterType actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    result <- g_action_group_get_action_parameter_type actionGroup' actionName'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed GLib.VariantType.VariantType) result'
        return result''
    touchManagedPtr actionGroup
    freeMem actionName'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ActionGroupGetActionParameterTypeMethodInfo
instance (signature ~ (T.Text -> m (Maybe GLib.VariantType.VariantType)), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupGetActionParameterTypeMethodInfo a signature where
    overloadedMethod = actionGroupGetActionParameterType

instance O.OverloadedMethodInfo ActionGroupGetActionParameterTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupGetActionParameterType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupGetActionParameterType"
        })


#endif

-- method ActionGroup::get_action_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_get_action_state" g_action_group_get_action_state :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    IO (Ptr GVariant)

-- | Queries the current state of the named action within /@actionGroup@/.
-- 
-- If the action is not stateful then @NULL@ will be returned.  If the
-- action is stateful then the type of the return value is the type
-- given by 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionStateType'.
-- 
-- The return value (if non-@NULL@) should be freed with
-- 'GI.GLib.Structs.Variant.variantUnref' when it is no longer required.
-- 
-- /Since: 2.28/
actionGroupGetActionState ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of the action to query
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the current state of the action
actionGroupGetActionState actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    result <- g_action_group_get_action_state actionGroup' actionName'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr actionGroup
    freeMem actionName'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ActionGroupGetActionStateMethodInfo
instance (signature ~ (T.Text -> m (Maybe GVariant)), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupGetActionStateMethodInfo a signature where
    overloadedMethod = actionGroupGetActionState

instance O.OverloadedMethodInfo ActionGroupGetActionStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupGetActionState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupGetActionState"
        })


#endif

-- method ActionGroup::get_action_state_hint
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_get_action_state_hint" g_action_group_get_action_state_hint :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    IO (Ptr GVariant)

-- | Requests a hint about the valid range of values for the state of the
-- named action within /@actionGroup@/.
-- 
-- If @NULL@ is returned it either means that the action is not stateful
-- or that there is no hint about the valid range of values for the
-- state of the action.
-- 
-- If a [type/@gLib@/.Variant] array is returned then each item in the array is a
-- possible value for the state.  If a [type/@gLib@/.Variant] pair (ie: two-tuple) is
-- returned then the tuple specifies the inclusive lower and upper bound
-- of valid values for the state.
-- 
-- In any case, the information is merely a hint.  It may be possible to
-- have a state value outside of the hinted range and setting a value
-- within the range may fail.
-- 
-- The return value (if non-@NULL@) should be freed with
-- 'GI.GLib.Structs.Variant.variantUnref' when it is no longer required.
-- 
-- /Since: 2.28/
actionGroupGetActionStateHint ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of the action to query
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the state range hint
actionGroupGetActionStateHint actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    result <- g_action_group_get_action_state_hint actionGroup' actionName'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr actionGroup
    freeMem actionName'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ActionGroupGetActionStateHintMethodInfo
instance (signature ~ (T.Text -> m (Maybe GVariant)), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupGetActionStateHintMethodInfo a signature where
    overloadedMethod = actionGroupGetActionStateHint

instance O.OverloadedMethodInfo ActionGroupGetActionStateHintMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupGetActionStateHint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupGetActionStateHint"
        })


#endif

-- method ActionGroup::get_action_state_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_get_action_state_type" g_action_group_get_action_state_type :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    IO (Ptr GLib.VariantType.VariantType)

-- | Queries the type of the state of the named action within
-- /@actionGroup@/.
-- 
-- If the action is stateful then this function returns the
-- [type/@gLib@/.VariantType] of the state.  All calls to
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupChangeActionState' must give a [type/@gLib@/.Variant] of this
-- type and 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionState' will return a [type/@gLib@/.Variant]
-- of the same type.
-- 
-- If the action is not stateful then this function will return @NULL@.
-- In that case, 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionState' will return @NULL@
-- and you must not call 'GI.Gio.Interfaces.ActionGroup.actionGroupChangeActionState'.
-- 
-- The state type of a particular action will never change but it is
-- possible for an action to be removed and for a new action to be added
-- with the same name but a different state type.
-- 
-- /Since: 2.28/
actionGroupGetActionStateType ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of the action to query
    -> m (Maybe GLib.VariantType.VariantType)
    -- ^ __Returns:__ the state type, if the action is stateful
actionGroupGetActionStateType actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    result <- g_action_group_get_action_state_type actionGroup' actionName'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed GLib.VariantType.VariantType) result'
        return result''
    touchManagedPtr actionGroup
    freeMem actionName'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ActionGroupGetActionStateTypeMethodInfo
instance (signature ~ (T.Text -> m (Maybe GLib.VariantType.VariantType)), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupGetActionStateTypeMethodInfo a signature where
    overloadedMethod = actionGroupGetActionStateType

instance O.OverloadedMethodInfo ActionGroupGetActionStateTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupGetActionStateType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupGetActionStateType"
        })


#endif

-- method ActionGroup::has_action
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the action to check for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_has_action" g_action_group_has_action :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    IO CInt

-- | Checks if the named action exists within /@actionGroup@/.
-- 
-- /Since: 2.28/
actionGroupHasAction ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of the action to check for
    -> m Bool
    -- ^ __Returns:__ whether the named action exists
actionGroupHasAction actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    result <- g_action_group_has_action actionGroup' actionName'
    let result' = (/= 0) result
    touchManagedPtr actionGroup
    freeMem actionName'
    return result'

#if defined(ENABLE_OVERLOADING)
data ActionGroupHasActionMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupHasActionMethodInfo a signature where
    overloadedMethod = actionGroupHasAction

instance O.OverloadedMethodInfo ActionGroupHasActionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupHasAction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupHasAction"
        })


#endif

-- method ActionGroup::list_actions
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_list_actions" g_action_group_list_actions :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    IO (Ptr CString)

-- | Lists the actions contained within /@actionGroup@/.
-- 
-- The caller is responsible for freeing the list with 'GI.GLib.Functions.strfreev' when
-- it is no longer required.
-- 
-- /Since: 2.28/
actionGroupListActions ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> m [T.Text]
    -- ^ __Returns:__ a @NULL@-terminated array
    --   of the names of the actions in the group
actionGroupListActions actionGroup = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    result <- g_action_group_list_actions actionGroup'
    checkUnexpectedReturnNULL "actionGroupListActions" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    touchManagedPtr actionGroup
    return result'

#if defined(ENABLE_OVERLOADING)
data ActionGroupListActionsMethodInfo
instance (signature ~ (m [T.Text]), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupListActionsMethodInfo a signature where
    overloadedMethod = actionGroupListActions

instance O.OverloadedMethodInfo ActionGroupListActionsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupListActions",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupListActions"
        })


#endif

-- method ActionGroup::query_action
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.ActionGroup]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of an action in the group"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "enabled"
--           , argType = TBasicType TBoolean
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "if the action is presently enabled"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "parameter_type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the parameter type, or `NULL` if none needed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state_type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the state type, or `NULL` if stateless"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state_hint"
--           , argType = TVariant
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the state hint, or `NULL` if none"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "state"
--           , argType = TVariant
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the current state, or `NULL` if stateless"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_action_group_query_action" g_action_group_query_action :: 
    Ptr ActionGroup ->                      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    CString ->                              -- action_name : TBasicType TUTF8
    Ptr CInt ->                             -- enabled : TBasicType TBoolean
    Ptr (Ptr GLib.VariantType.VariantType) -> -- parameter_type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    Ptr (Ptr GLib.VariantType.VariantType) -> -- state_type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    Ptr (Ptr GVariant) ->                   -- state_hint : TVariant
    Ptr (Ptr GVariant) ->                   -- state : TVariant
    IO CInt

-- | Queries all aspects of the named action within an /@actionGroup@/.
-- 
-- This function acquires the information available from
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupHasAction', 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionEnabled',
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionParameterType',
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionStateType',
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionStateHint' and
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupGetActionState' with a single function call.
-- 
-- This provides two main benefits.
-- 
-- The first is the improvement in efficiency that comes with not having
-- to perform repeated lookups of the action in order to discover
-- different things about it.  The second is that implementing
-- [type/@gio@/.ActionGroup] can now be done by only overriding this one virtual
-- function.
-- 
-- The interface provides a default implementation of this function that
-- calls the individual functions, as required, to fetch the
-- information.  The interface also provides default implementations of
-- those functions that call this function.  All implementations,
-- therefore, must override either this function or all of the others.
-- 
-- If the action exists, @TRUE@ is returned and any of the requested
-- fields (as indicated by having a non-@NULL@ reference passed in) are
-- filled.  If the action doesn’t exist, @FALSE@ is returned and the
-- fields may or may not have been modified.
-- 
-- /Since: 2.32/
actionGroupQueryAction ::
    (B.CallStack.HasCallStack, MonadIO m, IsActionGroup a) =>
    a
    -- ^ /@actionGroup@/: a [type/@gio@/.ActionGroup]
    -> T.Text
    -- ^ /@actionName@/: the name of an action in the group
    -> m ((Bool, Bool, GLib.VariantType.VariantType, GLib.VariantType.VariantType, GVariant, GVariant))
    -- ^ __Returns:__ @TRUE@ if the action exists, else @FALSE@
actionGroupQueryAction actionGroup actionName = liftIO $ do
    actionGroup' <- unsafeManagedPtrCastPtr actionGroup
    actionName' <- textToCString actionName
    enabled <- allocMem :: IO (Ptr CInt)
    parameterType <- callocMem :: IO (Ptr (Ptr GLib.VariantType.VariantType))
    stateType <- callocMem :: IO (Ptr (Ptr GLib.VariantType.VariantType))
    stateHint <- callocMem :: IO (Ptr (Ptr GVariant))
    state <- callocMem :: IO (Ptr (Ptr GVariant))
    result <- g_action_group_query_action actionGroup' actionName' enabled parameterType stateType stateHint state
    let result' = (/= 0) result
    enabled' <- peek enabled
    let enabled'' = (/= 0) enabled'
    parameterType' <- peek parameterType
    parameterType'' <- (newBoxed GLib.VariantType.VariantType) parameterType'
    stateType' <- peek stateType
    stateType'' <- (newBoxed GLib.VariantType.VariantType) stateType'
    stateHint' <- peek stateHint
    stateHint'' <- B.GVariant.wrapGVariantPtr stateHint'
    state' <- peek state
    state'' <- B.GVariant.wrapGVariantPtr state'
    touchManagedPtr actionGroup
    freeMem actionName'
    freeMem enabled
    freeMem parameterType
    freeMem stateType
    freeMem stateHint
    freeMem state
    return (result', enabled'', parameterType'', stateType'', stateHint'', state'')

#if defined(ENABLE_OVERLOADING)
data ActionGroupQueryActionMethodInfo
instance (signature ~ (T.Text -> m ((Bool, Bool, GLib.VariantType.VariantType, GLib.VariantType.VariantType, GVariant, GVariant))), MonadIO m, IsActionGroup a) => O.OverloadedMethod ActionGroupQueryActionMethodInfo a signature where
    overloadedMethod = actionGroupQueryAction

instance O.OverloadedMethodInfo ActionGroupQueryActionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup.actionGroupQueryAction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#v:actionGroupQueryAction"
        })


#endif

-- signal ActionGroup::action-added
-- | Signals that a new action was just added to the group.
-- 
-- This signal is emitted after the action has been added
-- and is now visible.
-- 
-- /Since: 2.28/
type ActionGroupActionAddedCallback =
    T.Text
    -- ^ /@actionName@/: the name of the action in /@actionGroup@/
    -> IO ()

type C_ActionGroupActionAddedCallback =
    Ptr ActionGroup ->                      -- object
    CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ActionGroupActionAddedCallback`.
foreign import ccall "wrapper"
    mk_ActionGroupActionAddedCallback :: C_ActionGroupActionAddedCallback -> IO (FunPtr C_ActionGroupActionAddedCallback)

wrap_ActionGroupActionAddedCallback :: 
    GObject a => (a -> ActionGroupActionAddedCallback) ->
    C_ActionGroupActionAddedCallback
wrap_ActionGroupActionAddedCallback gi'cb gi'selfPtr actionName _ = do
    actionName' <- cstringToText actionName
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  actionName'


-- | Connect a signal handler for the [actionAdded](#signal:actionAdded) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' actionGroup #actionAdded callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@action-added::detail@” instead.
-- 
onActionGroupActionAdded :: (IsActionGroup a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ActionGroupActionAddedCallback) -> m SignalHandlerId
onActionGroupActionAdded obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ActionGroupActionAddedCallback wrapped
    wrapped'' <- mk_ActionGroupActionAddedCallback wrapped'
    connectSignalFunPtr obj "action-added" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [actionAdded](#signal:actionAdded) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' actionGroup #actionAdded callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@action-added::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterActionGroupActionAdded :: (IsActionGroup a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ActionGroupActionAddedCallback) -> m SignalHandlerId
afterActionGroupActionAdded obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ActionGroupActionAddedCallback wrapped
    wrapped'' <- mk_ActionGroupActionAddedCallback wrapped'
    connectSignalFunPtr obj "action-added" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data ActionGroupActionAddedSignalInfo
instance SignalInfo ActionGroupActionAddedSignalInfo where
    type HaskellCallbackType ActionGroupActionAddedSignalInfo = ActionGroupActionAddedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ActionGroupActionAddedCallback cb
        cb'' <- mk_ActionGroupActionAddedCallback cb'
        connectSignalFunPtr obj "action-added" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup::action-added"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#g:signal:actionAdded"})

#endif

-- signal ActionGroup::action-enabled-changed
-- | Signals that the enabled status of the named action has changed.
-- 
-- /Since: 2.28/
type ActionGroupActionEnabledChangedCallback =
    T.Text
    -- ^ /@actionName@/: the name of the action in /@actionGroup@/
    -> Bool
    -- ^ /@enabled@/: whether the action is enabled
    -> IO ()

type C_ActionGroupActionEnabledChangedCallback =
    Ptr ActionGroup ->                      -- object
    CString ->
    CInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ActionGroupActionEnabledChangedCallback`.
foreign import ccall "wrapper"
    mk_ActionGroupActionEnabledChangedCallback :: C_ActionGroupActionEnabledChangedCallback -> IO (FunPtr C_ActionGroupActionEnabledChangedCallback)

wrap_ActionGroupActionEnabledChangedCallback :: 
    GObject a => (a -> ActionGroupActionEnabledChangedCallback) ->
    C_ActionGroupActionEnabledChangedCallback
wrap_ActionGroupActionEnabledChangedCallback gi'cb gi'selfPtr actionName enabled _ = do
    actionName' <- cstringToText actionName
    let enabled' = (/= 0) enabled
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  actionName' enabled'


-- | Connect a signal handler for the [actionEnabledChanged](#signal:actionEnabledChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' actionGroup #actionEnabledChanged callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@action-enabled-changed::detail@” instead.
-- 
onActionGroupActionEnabledChanged :: (IsActionGroup a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ActionGroupActionEnabledChangedCallback) -> m SignalHandlerId
onActionGroupActionEnabledChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ActionGroupActionEnabledChangedCallback wrapped
    wrapped'' <- mk_ActionGroupActionEnabledChangedCallback wrapped'
    connectSignalFunPtr obj "action-enabled-changed" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [actionEnabledChanged](#signal:actionEnabledChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' actionGroup #actionEnabledChanged callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@action-enabled-changed::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterActionGroupActionEnabledChanged :: (IsActionGroup a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ActionGroupActionEnabledChangedCallback) -> m SignalHandlerId
afterActionGroupActionEnabledChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ActionGroupActionEnabledChangedCallback wrapped
    wrapped'' <- mk_ActionGroupActionEnabledChangedCallback wrapped'
    connectSignalFunPtr obj "action-enabled-changed" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data ActionGroupActionEnabledChangedSignalInfo
instance SignalInfo ActionGroupActionEnabledChangedSignalInfo where
    type HaskellCallbackType ActionGroupActionEnabledChangedSignalInfo = ActionGroupActionEnabledChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ActionGroupActionEnabledChangedCallback cb
        cb'' <- mk_ActionGroupActionEnabledChangedCallback cb'
        connectSignalFunPtr obj "action-enabled-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup::action-enabled-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#g:signal:actionEnabledChanged"})

#endif

-- signal ActionGroup::action-removed
-- | Signals that an action is just about to be removed from the group.
-- 
-- This signal is emitted before the action is removed, so the action
-- is still visible and can be queried from the signal handler.
-- 
-- /Since: 2.28/
type ActionGroupActionRemovedCallback =
    T.Text
    -- ^ /@actionName@/: the name of the action in /@actionGroup@/
    -> IO ()

type C_ActionGroupActionRemovedCallback =
    Ptr ActionGroup ->                      -- object
    CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ActionGroupActionRemovedCallback`.
foreign import ccall "wrapper"
    mk_ActionGroupActionRemovedCallback :: C_ActionGroupActionRemovedCallback -> IO (FunPtr C_ActionGroupActionRemovedCallback)

wrap_ActionGroupActionRemovedCallback :: 
    GObject a => (a -> ActionGroupActionRemovedCallback) ->
    C_ActionGroupActionRemovedCallback
wrap_ActionGroupActionRemovedCallback gi'cb gi'selfPtr actionName _ = do
    actionName' <- cstringToText actionName
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  actionName'


-- | Connect a signal handler for the [actionRemoved](#signal:actionRemoved) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' actionGroup #actionRemoved callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@action-removed::detail@” instead.
-- 
onActionGroupActionRemoved :: (IsActionGroup a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ActionGroupActionRemovedCallback) -> m SignalHandlerId
onActionGroupActionRemoved obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ActionGroupActionRemovedCallback wrapped
    wrapped'' <- mk_ActionGroupActionRemovedCallback wrapped'
    connectSignalFunPtr obj "action-removed" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [actionRemoved](#signal:actionRemoved) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' actionGroup #actionRemoved callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@action-removed::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterActionGroupActionRemoved :: (IsActionGroup a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ActionGroupActionRemovedCallback) -> m SignalHandlerId
afterActionGroupActionRemoved obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ActionGroupActionRemovedCallback wrapped
    wrapped'' <- mk_ActionGroupActionRemovedCallback wrapped'
    connectSignalFunPtr obj "action-removed" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data ActionGroupActionRemovedSignalInfo
instance SignalInfo ActionGroupActionRemovedSignalInfo where
    type HaskellCallbackType ActionGroupActionRemovedSignalInfo = ActionGroupActionRemovedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ActionGroupActionRemovedCallback cb
        cb'' <- mk_ActionGroupActionRemovedCallback cb'
        connectSignalFunPtr obj "action-removed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup::action-removed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#g:signal:actionRemoved"})

#endif

-- signal ActionGroup::action-state-changed
-- | Signals that the state of the named action has changed.
-- 
-- /Since: 2.28/
type ActionGroupActionStateChangedCallback =
    T.Text
    -- ^ /@actionName@/: the name of the action in /@actionGroup@/
    -> GVariant
    -- ^ /@value@/: the new value of the state
    -> IO ()

type C_ActionGroupActionStateChangedCallback =
    Ptr ActionGroup ->                      -- object
    CString ->
    Ptr GVariant ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ActionGroupActionStateChangedCallback`.
foreign import ccall "wrapper"
    mk_ActionGroupActionStateChangedCallback :: C_ActionGroupActionStateChangedCallback -> IO (FunPtr C_ActionGroupActionStateChangedCallback)

wrap_ActionGroupActionStateChangedCallback :: 
    GObject a => (a -> ActionGroupActionStateChangedCallback) ->
    C_ActionGroupActionStateChangedCallback
wrap_ActionGroupActionStateChangedCallback gi'cb gi'selfPtr actionName value _ = do
    actionName' <- cstringToText actionName
    value' <- B.GVariant.newGVariantFromPtr value
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  actionName' value'


-- | Connect a signal handler for the [actionStateChanged](#signal:actionStateChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' actionGroup #actionStateChanged callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@action-state-changed::detail@” instead.
-- 
onActionGroupActionStateChanged :: (IsActionGroup a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ActionGroupActionStateChangedCallback) -> m SignalHandlerId
onActionGroupActionStateChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ActionGroupActionStateChangedCallback wrapped
    wrapped'' <- mk_ActionGroupActionStateChangedCallback wrapped'
    connectSignalFunPtr obj "action-state-changed" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [actionStateChanged](#signal:actionStateChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' actionGroup #actionStateChanged callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@action-state-changed::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterActionGroupActionStateChanged :: (IsActionGroup a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ActionGroupActionStateChangedCallback) -> m SignalHandlerId
afterActionGroupActionStateChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ActionGroupActionStateChangedCallback wrapped
    wrapped'' <- mk_ActionGroupActionStateChangedCallback wrapped'
    connectSignalFunPtr obj "action-state-changed" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data ActionGroupActionStateChangedSignalInfo
instance SignalInfo ActionGroupActionStateChangedSignalInfo where
    type HaskellCallbackType ActionGroupActionStateChangedSignalInfo = ActionGroupActionStateChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ActionGroupActionStateChangedCallback cb
        cb'' <- mk_ActionGroupActionStateChangedCallback cb'
        connectSignalFunPtr obj "action-state-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.ActionGroup::action-state-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.35/docs/GI-Gio-Interfaces-ActionGroup.html#g:signal:actionStateChanged"})

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList ActionGroup = ActionGroupSignalList
type ActionGroupSignalList = ('[ '("actionAdded", ActionGroupActionAddedSignalInfo), '("actionEnabledChanged", ActionGroupActionEnabledChangedSignalInfo), '("actionRemoved", ActionGroupActionRemovedSignalInfo), '("actionStateChanged", ActionGroupActionStateChangedSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif


