package org.tanukisoftware.wrapper;

/*
 * Copyright (c) 1999, 2025 Tanuki Software, Ltd.
 * http://www.tanukisoftware.com
 * All rights reserved.
 *
 * This software is the proprietary information of Tanuki Software.
 * You shall use it only in accordance with the terms of the
 * license agreement you entered into with Tanuki Software.
 * http://wrapper.tanukisoftware.com/doc/english/licenseOverview.html
 * 
 * 
 * Portions of the Software have been derived from source code
 * developed by Silver Egg Technology under the following license:
 * 
 * Copyright (c) 2001 Silver Egg Technology
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without 
 * restriction, including without limitation the rights to use, 
 * copy, modify, merge, publish, distribute, sub-license, and/or 
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following 
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 */

import java.io.File;
import java.io.PrintStream;
import java.lang.reflect.Method;
import java.util.StringTokenizer;

class WrapperNativeLibrary
{
    private static boolean m_libraryLoaded = false;

    private static boolean m_libraryOK = false;

    private static String m_libraryName = null;

    /** Stores the resolved OS used in naming of native library. */
    private static String m_os = null;
    
    /** Stores the resolved Architecture used in naming of native library. */
    private static String m_arch = null;
    
    /** Flag to remember whether or not this is Windows. */
    private static boolean m_windows = false;
    
    /** Flag to remember whether or not this is MacOSX. */
    private static boolean m_macosx = false;
    
    /** Flag to remember whether or not this is AIX. */
    private static boolean m_aix = false;
    
    /** Flag to remember whether or not this is z/OS. */
    private static boolean m_zos = false;
    
    /** Info level log channel */
    private static PrintStream m_outInfo = System.out;
    
    /** Error level log channel */
    private static PrintStream m_outError = System.out;
    
    /** Debug level log channel */
    private static PrintStream m_outDebug = System.out;
    
    /** message resources: eventually these will be split up */
    private static WrapperResources m_res = null;
    
    /** Set on startup to show whether or not a Professional edition library is being used.
     *   Overriding this makes no difference as the underlying library also contains these checks.
     *   Caching this value makes it possible to reduce the number of JNI calls and avoid confusing
     *   errors on shutdown. */
    private static boolean m_professionalEdition;
    
    /** Set on startup to show whether or not a Standard edition library is being used.
     *   Overriding this makes no difference as the underlying library also contains these checks.
     *   Caching this value makes it possible to reduce the number of JNI calls and avoid confusing
     *   errors on shutdown. */
    private static boolean m_standardEdition;

    private static native void nativeInit( boolean debug );
    private static native void nativeDispose( boolean debug );
    private static native String nativeGetLibraryPath( boolean debug ) throws NullPointerException;
    private static native String nativeGetLibraryVersion();
    private static native boolean nativeIsProfessionalEdition();
    private static native boolean nativeIsStandardEdition();
    private static native WrapperResources nativeLoadWrapperResources( String domain, String folder, boolean makeActive );

    static void setPrintStreams( PrintStream outInfo, PrintStream outError, PrintStream outDebug ) {
        if ( outInfo != null )
        {
            m_outInfo = outInfo;
        }
        if ( outError != null )
        {
            m_outError = outError;
        }
        if ( outDebug != null )
        {
            m_outDebug = outDebug;
        }
    }

    /**
     * Searches for and then loads the native library.  This method will attempt
     *  locate the wrapper library using one of the following 3 naming 
     */
    static boolean load( int jvmBits, boolean doInit, boolean debug, boolean errorAsInfo )
    {
        // Look for the base name of the library.
        String baseName = System.getProperty( "wrapper.native_library" );
        if ( baseName == null )
        {
            // This should only happen if an old version of the Wrapper binary is being used.
            // This message is logged when localization is not yet initialized.
            m_outInfo.println( "WARNING - The wrapper.native_library system property was not" );
            m_outInfo.println( "          set. Using the default value, 'wrapper'." );
            baseName = "wrapper";
        }
        String[] detailedNames = new String[2];
        if ( jvmBits > 0 )
        {
            detailedNames[0] = generateDetailedNativeLibraryBaseName( baseName, jvmBits );
        }
        else
        {
            detailedNames[0] = generateDetailedNativeLibraryBaseName( baseName, 32 );
            detailedNames[1] = generateDetailedNativeLibraryBaseName( baseName, 64 );
        }
        
        // Construct brief and detailed native library file names.
        String file = mapSharedLibraryName( baseName );
        String[] detailedFiles = new String[detailedNames.length];
        for ( int i = 0; i < detailedNames.length; i++ )
        {
            if ( detailedNames[i] != null )
            {
                detailedFiles[i] = mapSharedLibraryName( detailedNames[i] );
            }
        }
        
        String[] detailedErrors = new String[detailedNames.length];
        String baseError = null;
        // Try loading the native library using the detailed name first.  If that fails, use
        //  the brief name.
        if ( debug )
        {
            // This message is logged when localization is not yet initialized.
            m_outDebug.println( "Load native library.  There are multiple possible file names and the first to be found will be used.  Errors loading non-existing files is normal and is only a problem if they all fail." ); 
        }

        for ( int i = 0; i < detailedNames.length; i++ )
        {
            if ( detailedNames[i] != null )
            {
                detailedErrors[i] = loadNativeLibrary( detailedNames[i], detailedFiles[i], debug );
                if ( detailedErrors[i] == null )
                {
                    m_libraryName = detailedFiles[i];
                    m_libraryOK = true;
                    break;
                }
            }
        }
        if ( ( !m_libraryOK ) && ( ( baseError = loadNativeLibrary( baseName, file, debug ) ) == null ) )
        {
            m_libraryName = file;
            m_libraryOK = true;
        }
        if ( m_libraryOK )
        {
            m_libraryLoaded = true;
            if ( debug )
            {
                // This message is logged when localization is not yet initialized.
                m_outDebug.println( "  Successfully loaded native library." );
            }
            
            // Cache the values of the professional and standard edition flags.
            try
            {
                m_professionalEdition = nativeIsProfessionalEdition();
            }
            catch ( Throwable e )
            {
                if ( debug )
                {
                    // This message is logged when localization is not yet initialized.
                    m_outDebug.println( "Call to nativeIsProfessionalEdition() failed: " + e );
                }
                m_professionalEdition = false;
            }
            try
            {
                m_standardEdition = nativeIsStandardEdition();
            }
            catch ( Throwable e )
            {
                if ( debug )
                {
                    // This message is logged when localization is not yet initialized.
                    m_outDebug.println( "Call to nativeIsStandardEdition() failed: " + e );
                }
                m_standardEdition = false;
            }
            
            // Try reloading the resources once the library is initialized so we get actual localized content.
            //  We do this before trying to initialize the native library intentionally so any messages will be
            //  localized correctly.  This one call is designed to handle this state.
            m_res = loadWrapperResourcesInner( System.getProperty( "wrapper.lang.domain" ) + "jni",
                WrapperSystemPropertyUtil.getStringProperty( "wrapper.lang.folder", "../lang" ), true );
            
            if ( debug )
            {
                m_outDebug.println( getRes().getString( "Loaded localized resources." ) );
            }
            
            if ( doInit )
            {
                // The library was loaded correctly, so initialize it.
                if ( debug )
                {
                    m_outDebug.println( getRes().getString( "Calling native initialization method." ) );
                }
                nativeInit( debug );
            }
        }
        else
        {
            m_libraryLoaded = false;
            
            // The library could not be loaded, so we want to give the user a useful
            //  clue as to why not.
            PrintStream out = errorAsInfo ? m_outInfo : m_outError;
            out.println();
            
            String libPath = System.getProperty( "java.library.path" );
            if ( libPath.equals( "" ) )
            {
                // No library path
                // This message is logged when localization is unavailable.
                out.println( "ERROR - Unable to load the Wrapper's native library because the" );
                out.println( "        java.library.path was set to ''.  Please see the" );
                out.println( "        documentation for the wrapper.java.library.path" );
                out.println( "        configuration property." );
            }
            else
            {
                // Attempt to locate the actual files on the path.
                String error = null;
                File libFile = null;
                for ( int i = 0; i < detailedNames.length; i++ )
                {
                    if ( detailedFiles[i] != null )
                    {
                        libFile = locateFileOnPath( detailedFiles[i], libPath );
                        if ( libFile != null )
                        {
                            error = detailedErrors[i];
                            break;
                        }
                    }
                }
                if ( libFile == null )
                {
                    libFile = locateFileOnPath( file, libPath );
                    if ( libFile != null )
                    {
                        error = baseError;
                    }
                }
                if ( libFile == null )
                {
                    // The library could not be located on the library path.
                    // This message is logged when localization is unavailable.
                    out.println( "ERROR - Unable to load the Wrapper's native library because none of the" );
                    out.println( "        following files:" );
                    for ( int i = 0; i < detailedNames.length; i++ )
                    {
                        if ( detailedFiles[i] != null )
                        {
                            out.println( "          " + detailedFiles[i] );
                        }
                    }
                    out.println( "          " + file );
                    out.println( "        could be located on the following java.library.path:" );
                    
                    String pathSep = System.getProperty( "path.separator" );
                    StringTokenizer st = new StringTokenizer( libPath, pathSep );
                    while ( st.hasMoreTokens() )
                    {
                        File pathElement = new File( st.nextToken() );
                        out.println( "          " + pathElement.getAbsolutePath() );
                    }
                    out.println( "        Please see the documentation for the wrapper.java.library.path" );
                    out.println( "        configuration property." );
                }
                else
                {
                    // The library file was found but could not be loaded for some reason.
                    // This message is logged when localization is unavailable.
                    out.println( "ERROR - Unable to load the Wrapper's native library '" + libFile.getName() + "'." );
                    out.println( "        The file is located on the path at the following location but" );
                    out.println( "        could not be loaded:" );
                    out.println( "          " + libFile.getAbsolutePath() );
                    out.println( "        Please verify that the file is both readable and executable by the" );
                    out.println( "        current user and that the file has not been corrupted in any way." );
                    out.println( "        One common cause of this problem is running a 32-bit version" );
                    out.println( "        of the Wrapper with a 64-bit version of Java, or vice versa." );
                    if ( jvmBits > 0 )
                    {
                        out.println( "        This is a " + jvmBits + "-bit JVM." );
                    }
                    else
                    {
                        out.println( "        The bit depth of this JVM could not be determined." );
                    }
                    out.println( "        Reported cause:" );
                    out.println( "          " + error );
                }
            }
            out.println();
        }

        return m_libraryLoaded;
    }

    static void dispose( boolean debug )
    {
        if ( m_libraryOK )
        {
            nativeDispose( debug );
        }
    }

    /**
     * Generates a detailed native library base name which is made up of the
     *  base name, the os name, architecture, and the bits of the current JVM,
     *  not the platform.
     *
     * @return A detailed native library base name.
     */
    static String generateDetailedNativeLibraryBaseName( String baseName, int jvmBits )
    {
        // Generate an os name.  Most names are used as is, but some are modified.
        String os = System.getProperty( "os.name", "" ).toLowerCase();
        if ( os.startsWith( "windows" ) )
        {
            os = "windows";
            m_windows = true;
        }
        else if ( os.equals( "sunos" ) )
        {
            os = "solaris";
        }
        else if ( os.equals( "hp-ux" ) || os.equals( "hp-ux64" ) )
        {
            os = "hpux";
        }
        else if ( os.equals( "mac os x" ) )
        {
            os = "macosx";
            m_macosx = true;
        }
        else if ( os.equals( "unix_sv" ) )
        {
            os = "unixware";
        }
        else if ( os.equals( "os/400" ) )
        {
            os = "os400";
        }
        else if ( os.equals( "z/os" ) )
        {
            os = "zos";
            m_zos = true;
        }
        else if ( os.indexOf("aix") > -1 )
        {
            m_aix = true;
        }
        m_os = os;
        
        // Generate an architecture name.
        String arch = System.getProperty( "os.arch", "" ).toLowerCase();
        if ( m_macosx )
        {
            if ( arch.startsWith( "arm" ) || arch.startsWith( "aarch" ) )
            {
                arch = "arm";
            }
            else if ( arch.equals( "ppc" ) || arch.equals( "ppc64" ) || arch.equals( "x86" ) || arch.equals( "x86_64" ) )
            {
                arch = "universal";
            }
        }
        else
        {
            if ( arch.equals( "amd64" ) || arch.equals( "athlon" ) || arch.equals( "x86_64" ) ||
                arch.equals( "i686" ) || arch.equals( "i586" ) || arch.equals( "i486" ) ||
                arch.equals( "i386" ) )
            {
                arch = "x86";
            }
            else if ( arch.startsWith( "ia32" ) || arch.startsWith( "ia64" ) )
            {
                arch = "ia";
            }
            else if ( arch.startsWith( "sparc" ) )
            {
                arch = "sparc";
            }
            else if ( arch.startsWith( "ppc64le" ) )
            {
                arch = "ppcle";
            }
            else if ( arch.equals( "power" ) || arch.equals( "powerpc" ) || arch.equals( "ppc64" ) )
            {
                if ( m_aix )
                {
                    arch = "ppc";
                }
                else
                {
                    arch = "ppcbe";
                }
            }
            else if ( arch.startsWith( "pa_risc" ) || arch.startsWith( "pa-risc" ) )
            {
                arch = "parisc";
            }
            else if ( arch.startsWith( "arm" ) || arch.startsWith( "aarch" ) )
            {
                if ( jvmBits == 64 )
                {
                    arch = "arm";
                }
                else
                {
                    arch = System.getProperty( "wrapper.arch" );
                }
            }
            else if ( arch.equals( "s390" ) || arch.equals( "s390x" ) )
            {
                arch = "390";
            }
        }
        m_arch = arch;
        
        return baseName + "-" + os + "-" + arch + "-" + jvmBits;
    }
    
    /** 
     * On AIX, mapLibraryName() will return "lib*.a" with the IBM SDK, and "lib*.so" with the openJDK. 
     * Shared libraries can either have .so or .a suffix on AIX, but System.loadLibrary() in openJDK only accepts '.so' file.
     *   Note : After calling this function, we will always log the libraries names with a '.so' suffix even though the loaded lib was '.a'. 
     *   This may not be perfect, but there is no way to control whether System.loadLibrary() loaded a '.a' or a '.so' file. 
     *   We could use System.load() to control the path & suffix, but we don't want to make a special implementation for AIX.
     *   We could also loop through the lib paths and check if libraries with same names & different suffix coexist, but it assumes loadLibrary() always load '.a' in priority, which is not guaranteed in future versions of Java.
     */
    private static String mapSharedLibraryName(String name) 
    {
        String result = System.mapLibraryName( name );
        if ( isAIX() && result.endsWith(".a") )
        {
            result = result.substring( 0, result.length() - 2 ).concat( ".so" );
        }
        if ( isMacOSX() && "universal".equalsIgnoreCase( System.getProperty( "wrapper.arch" ) ) && result.endsWith( ".dylib" ) )
        {
            /* For universal distributions, the native library has a '.jnilib' extension for compatibility with Java 6 and earlier versions.
             *  Note: The above condition uses wrapper.arch, and not getArch(), to not change the extension when running a ARM Wrapper with a x86 jvm (in compatibility mode). */
            result = result.substring( 0, result.length() - 6 ).concat( ".jnilib" );
        }
        return result;
    }
    
    /**
     * Attempts to load the a native library file.
     *
     * @param name Name of the library to load.
     * @param file Name of the actual library file.
     *
     * @return null if the library was successfully loaded, an error message
     *         otherwise.
     */
    private static String loadNativeLibrary( String name, String file, boolean debug )
    {
        try
        {
            checkOldLibraryOnAix(file);
            System.loadLibrary( name );
            
            if ( debug )
            {
                m_outDebug.println( getRes().getString( "  Attempt to load native library with name: {0}  Result: {1}" , file, getRes().getString( "Success!" ) ) );
            }
            
            return null;
        }
        catch ( UnsatisfiedLinkError e )
        {
            if ( debug )
            {
                m_outDebug.println( getRes().getString( "  Attempt to load native library with name: {0}  Result: {1}" , file, e.getMessage() ) );
            }
            String error = e.getMessage();
            if ( error == null )
            {
                error = e.toString();
            }
            return error;
        }
        catch ( Throwable e )
        {
            if ( debug )
            {
                m_outDebug.println( getRes().getString( "  Attempt to load native library with name: {0}  Result: {1}" , file, e.getMessage() ) );
            }
            String error = e.toString();
            return error;
        }
    }
    
    /** 
     * On AIX, '.a' libraries are loaded prior to '.so' libraries if both coexist on the same folder. 
     * To help the user, lets warn if any '.a' file are found in the lib folders.
     */
    private static void checkOldLibraryOnAix(String libName)
    {
        if (isAIX())
        {
            if (libName.endsWith(".so")) 
            {
                libName = libName.substring(0, libName.length() - 3).concat(".a");
            }
            // We want to log the exact path of the library, so we don't pass libPaths with separators but rather loop on each path
            String pathSep = System.getProperty( "path.separator" );
            String[] libPaths = System.getProperty( "java.library.path" ).split(pathSep);
            for ( int j = 0; j < libPaths.length; j++ ) 
            {
                File libFile = locateFileOnPath( libName, libPaths[j] );
                if ( libFile != null )
                {
                    m_outInfo.println( getRes().getString( "WARNING - {0} was found in {1}.", libName, libPaths[j] ));
                    m_outInfo.println( getRes().getString( "          Recent Wrapper''s native libraries have a ''.so'' suffix." ));
                    m_outInfo.println( getRes().getString( "          Depending on the version of Java that is used, {0}", libName ));
                    m_outInfo.println( getRes().getString( "          may be loaded instead of a more recent library." ));
                    m_outInfo.println( getRes().getString( "          Please remove {0} and make sure that the latest version", libName ));
                    m_outInfo.println( getRes().getString( "          of the Wrapper''s native library is in the lib folder." ));
                }
            }
        }
    }
    
    /**
     * Loads a WrapperResources based on the current locale of the JVM.
     *
     * @param domain Domain of the resource.
     * @param folder Location of the resource.
     *
     * @return The requested resource.
     */
    static WrapperResources loadWrapperResourcesInner( String domain, String folder, boolean makeActive )
    {
        if ( m_libraryLoaded && isStandardEdition() )
        {
            if ( folder == null )
            {
                folder = WrapperSystemPropertyUtil.getStringProperty( "wrapper.lang.folder", "../lang" );
            }
            
            WrapperResources res = nativeLoadWrapperResources( domain, folder, makeActive );
            if (res != null)
            {
                return res;
            }
            else
            {
                return new WrapperResources();
            }
        }
        else
        {
            return new WrapperResources();
        }
    }

    static WrapperResources getRes()
    {
        if ( m_res == null )
        {
            // Create a dummy resources file so initial localization will work until the native library is loaded.
            m_res = new WrapperResources();
        }            
        return m_res;
    }

    /**
     * Searches for a file on a path.
     *
     * @param file File to look for.
     * @param path Path to be searched.
     *
     * @return Reference to the file object if found, otherwise null.
     */
    private static File locateFileOnPath( String file, String path )
    {
        // A library path exists but the library was not found on it.
        String pathSep = System.getProperty( "path.separator" );
        
        // Search for the file on the library path to verify that it does not
        //  exist, it could be some other problem
        StringTokenizer st = new StringTokenizer( path, pathSep );
        while( st.hasMoreTokens() )
        {
            File libFile = new File( new File( st.nextToken() ), file );
            if ( libFile.exists() )
            {
                return libFile;
            }
        }
        
        return null;
    }
    
    /**
     * Returns true if the current Wrapper edition has support for Professional
     *  Edition features.
     *
     * False will also be returned if the native library has not been initialized correctly.
     *
     * @return True if professional features are supported.
     */
    static boolean isProfessionalEdition()
    {
        // Use a cached value rather than calling the JNI call here to avoid confusing errors on shutdown.
        return m_professionalEdition;
    }
    
    /**
     * Returns true if the current Wrapper edition has support for Standard
     *  Edition features.
     *
     * False will also be returned if the native library has not been initialized correctly.
     *
     * @return True if standard features are supported.
     */
    static boolean isStandardEdition()
    {
        // Use a cached value rather than calling the JNI call here to avoid confusing errors on shutdown.
        return m_standardEdition;
    }

    static boolean allowsNativeCalls()
    {
        return m_libraryOK;
    }

    static void disableNativeCalls()
    {
        m_libraryOK = false;
    }

    static String getVersion( boolean debug )
    {
        String jniVersion = null;

        if ( m_libraryOK )
        {
            // Request the version from the native library.  Be careful as the method
            //  will not exist if the library is older than 3.6.3.
            try
            {
                jniVersion = nativeGetLibraryVersion();
            }
            catch ( Throwable e )
            {
                try 
                {
                    // Fallback to the native function of WrapperManager.
                    //  Important: If this is a bootstrap call, isBootstrapCall() will return true
                    //  to prevent WrapperManager from initializing.
                    jniVersion = WrapperManager.nativeGetLibraryVersion();
                }
                catch ( Exception ex )
                {
                    if ( debug )
                    {
                        m_outDebug.println( getRes().getString( 
                                "Call to nativeGetLibraryVersion() failed: {0}",  ex ) );
                    }
                }
            }
        }
        return jniVersion;
    }

    static String getLibraryPath( boolean debug )
    {
        String jniPath = null;

        if ( m_libraryOK )
        {
            // Request the path to the native library.  Be careful as the method
            //  will not exist if the library is old.
            try
            {
                jniPath = nativeGetLibraryPath( debug );
            }
            catch ( Throwable e )
            {
                if ( debug )
                {
                    m_outDebug.println( getRes().getString( 
                            "Call to nativeGetLibraryPath() failed: {0}",  e ) );
                }
            }
        }
        return jniPath;
    }

    /**
     * Returns the OS that the Wrapper has resolved.
     *  This will be the same as the WRAPPER_OS environment variable.
     *  Used when resolving the platform specific native library name.
     *
     * @return The resolved Wrapper OS name.
     *
     * @since Wrapper 3.5.31
     */
    static String getOS()
    {
        return m_os;
    }
    
    /**
     * Returns the Architecture that the Wrapper has resolved.
     *  This will be the same as the WRAPPER_ARCH environment variable.
     *  Used when resolving the platform specific native library name.
     *
     * @return The resolved Wrapper Architecture name.
     *
     * @since Wrapper 3.5.31
     */
    static String getArch()
    {
        return m_arch;
    }
    
    /**
     * Returns true if the current JVM is Windows.
     *
     * @return True if this is Windows.
     *
     * @since Wrapper 3.5.1
     */
    static boolean isWindows()
    {
        return m_windows;
    }
    
    /**
     * Returns true if the current JVM is Mac OSX.
     *
     * @return True if this is Mac OSX.
     *
     * @since Wrapper 3.5.1
     */
    static boolean isMacOSX()
    {
        return m_macosx;
    }
    
    /**
     * Returns true if the current JVM is AIX.
     *
     * @return True if this is AIX.
     *
     * @since Wrapper 3.5.27
     */
    static boolean isAIX()
    {
        return m_aix;
    }
    
    /**
     * Returns true if the current JVM is z/OS.
     *
     * @return True if this is z/OS.
     *
     * @since Wrapper 3.5.35
     */
    static boolean isZOS()
    {
        return m_zos;
    }
}
