package org.tanukisoftware.wrapper;

/*
 * Copyright (c) 1999, 2023 Tanuki Software, Ltd.
 * http://www.tanukisoftware.com
 * All rights reserved.
 *
 * This software is the proprietary information of Tanuki Software.
 * You shall use it only in accordance with the terms of the
 * license agreement you entered into with Tanuki Software.
 * http://wrapper.tanukisoftware.com/doc/english/licenseOverview.html
 */

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.NoSuchElementException;

public class WrapperBootstrap
{
    private static boolean c_bootstrapCall;
    
    private static final int ENTRYPOINT_MAINCLASS = 1;
    // private static final int ENTRYPOINT_MODULE    = 2;
    private static final int ENTRYPOINT_JAR       = 3;

    private static boolean m_debug;
    
    static boolean isBootstrapCall()
    {
        return c_bootstrapCall;
    }

    /*---------------------------------------------------------------
     * Main Method
     *-------------------------------------------------------------*/
    /**
     * In general, this method should avoid failing with exit code 1 and
     *  instead print tokens that can be parsed by native code.
     *  Exit code 1 will cause to not parse the output at all, not even the
     *  version number.
     *  For some critical errors, it can be useful to print error messages or
     *  call stacks from this function and then return. This can be achieved
     *  with special tokens to inform the Wrapper to raise the loglevel.
     *
     * @param args arg1: 1: 'mainclass', 2: 'module' or 3: 'jar'
     *             arg2: name of the element which type is specified by arg1
     *             arg3: debug
     */
    public static void main( String args[] )
    {
        c_bootstrapCall = true;

        HashMap tm = new LinkedHashMap();

        tm.put( "wrapper_version", WrapperInfo.getVersion() );
        
        if ( args.length < 3 )
        {
            // This output will cause the error after to be printed at loglevel ERROR.
            tm.put( "wrong_argument_number", Integer.toString( args.length ) );
            printParseInformation( tm );
            return;
        }

        try
        {
            if ( Integer.parseInt( args[2] ) != 0 )
            {
                m_debug = true;
            }
            else
            {
                m_debug = false;
            }
        }
        catch ( NumberFormatException e )
        {
            m_debug = false;
        }

        // Decide whether this is a 32 or 64 bit version of Java.
        int jvmBits = WrapperSystemPropertyUtil.getIntProperty( "sun.arch.data.model", -1 );
        if ( jvmBits == -1 )
        {
            jvmBits = WrapperSystemPropertyUtil.getIntProperty( "com.ibm.vm.bitmode", -1 );
        }
        if ( m_debug )
        {
            if ( jvmBits == -1 )
            {
                // This message is logged when localization is not yet initialized.
                printDebug( "The bit depth of this JVM could not be determined." );
            }
        }

        // Create a stream to log errors in case the native library fails to load.
        WrapperPrintStream outError = null;
        if ( "true".equals( System.getProperty( "wrapper.use_sun_encoding" ) ) ) {
            String sunStdoutEncoding = System.getProperty( "sun.stdout.encoding" );
            if ( ( sunStdoutEncoding != null ) && !sunStdoutEncoding.equals( System.getProperty( "file.encoding" ) ) ) {
                // We need to create the stream using the same encoding as the one used for stdout, else this will lead to encoding issues.
                try
                {
                    outError = new WrapperPrintStream( System.out, false, sunStdoutEncoding, "WrapperBootstrap Error: " );
                }
                catch ( UnsupportedEncodingException e )
                {
                    // This should not happen because we always make sure the encoding exists before launching a JVM.
                }
            }
        }
        if ( outError == null )
        {
            outError = new WrapperPrintStream( System.out, "WrapperBootstrap Error: " );
        }
        WrapperNativeLibrary.setPrintStreams( System.out, outError, System.out );

        // Load the library and request its version.
        if ( WrapperNativeLibrary.load( jvmBits, false, m_debug, false ) )
        {
            tm.put( "library_status", "loaded" );

            String libVersion = WrapperNativeLibrary.getVersion( m_debug );
            if ( libVersion == null )
            {
                libVersion = "not found";
            }
            tm.put( "library_version", libVersion );

            if ( WrapperNativeLibrary.isStandardEdition() && WrapperNativeLibrary.isWindows() )
            {
                String libPath = WrapperNativeLibrary.getLibraryPath( m_debug );
                if ( libPath == null )
                {
                    libPath = "not found";
                }
                tm.put( "library_path", libPath );
            }
        }
        else
        {
            tm.put( "library_status", "load failed" );
        }

        ClassLoader baseClassLoader = WrapperBootstrap.class.getClassLoader();
        Class mainClassClass = null;

        boolean findMainModule = false;

        int javaVersion = getJavaVersion();

        int id = Integer.parseInt( args[0] );
        if ( id == ENTRYPOINT_MAINCLASS )
        {
            // Output:
            // ------------------------------
            //     mainclass: not found
            //   or (Java >= 9)
            //     mainmodule: not found
            //     package: <package>
            //   or (Java >= 9)
            //     mainmodule: <modulename>
            //     package: <package>
            //   or (Java >= 9)
            //     mainmodule: unnamed
            //     package: <package>
            try
            {
                // Load the class without resolving it (this is not necessary, as it will be done with the --dry-run call), and without initializing it (as we do not want to execute any static code).
                mainClassClass = baseClassLoader.loadClass( args[1] );
            }
            catch ( ClassNotFoundException e )
            {
                tm.put( "mainclass", "not found" );
                mainClassClass = null;
            }
            catch ( Throwable t )
            {
                // This output will cause the stack below to be printed at loglevel ERROR.
                tm.put( "mainclass", "load failed" );
                printParseInformation( tm );
                t.printStackTrace();
                return;
            }
            if ( javaVersion >= 9 )
            {
                findMainModule = true;
            }
        }
        else if ( id == ENTRYPOINT_JAR )
        {
            // Output:
            // ------------------------------
            //     mainjar: not found
            //   or
            //     mainjar: can't open
            //   or
            //     mainjar: no manifest
            //   or
            //     [
            //         mainjar: no mainclass
            //       or
            //         mainclass: not found - <classname>
            //       or
            //         mainclass: <classname>
            //         mainmodule: <mainmodule>
            //         package: <package>
            //     ]
            //     and
            //     [
            //         mainjar: no classpath
            //       or
            //         classpath: <classpath>
            //     ]
            
            // Look for the specified jar file.
            File file = new File( args[1] );
            if ( !file.exists() )
            {
                tm.put( "mainjar", "not found" );
            }
            else
            {
                JarFile jarFile;
                try
                {
                    jarFile = new JarFile( file );
                }
                catch ( IOException e )
                {
                    if ( m_debug )
                    {
                        printDebug( WrapperNativeLibrary.getRes().getString( "Can't open jar file: {0}", e.getMessage() ) ); 
                    }
                    tm.put( "mainjar", "can't open" );
                    jarFile = null;
                }
                catch ( SecurityException e )
                {
                    if ( m_debug )
                    {
                        printDebug( WrapperNativeLibrary.getRes().getString( "Can't open jar file: {0}", e.getMessage() ) ); 
                    }
                    tm.put( "mainjar", "access denied" );
                    jarFile = null;
                }
                if ( jarFile != null )
                {
                    Manifest manifest;
                    try
                    {
                        manifest = jarFile.getManifest();
                    }
                    catch ( IOException e )
                    {
                        if ( m_debug )
                        {
                            printDebug( WrapperNativeLibrary.getRes().getString( "Jar file doesn't have a manifest: {0}", e.getMessage() ) ); 
                        }
                        tm.put( "mainjar", "no manifest" );
                        manifest = null;
                    }
                    
                    if ( manifest != null )
                    {
                        Attributes attributes = manifest.getMainAttributes();

                        String mainClassName = attributes.getValue( "Main-Class" );
                        if ( mainClassName == null ) 
                        {
                            tm.put( "mainjar", "no mainclass" );
                        }
                        else
                        {
                            URLClassLoader jarClassLoader;

                            // Store the main jar in the classpath.
                            try
                            {
                                URL[] classURLs = new URL[1];
                                classURLs[0] = new URL( "file:" + file.getAbsolutePath() );
                                jarClassLoader = URLClassLoader.newInstance( classURLs, baseClassLoader );
                            }
                            catch ( MalformedURLException e )
                            {
                                if ( m_debug )
                                {
                                    printDebug( WrapperNativeLibrary.getRes().getString( "Unable to add jar to classpath: {0}", e.getMessage() ) ); 
                                }
                                jarClassLoader = null;
                            }
                            try
                            {
                                // Load the class without resolving it or initializing it.
                                if ( jarClassLoader == null )
                                {
                                    mainClassClass = baseClassLoader.loadClass( mainClassName );
                                }
                                else
                                {
                                    mainClassClass = jarClassLoader.loadClass( mainClassName );
                                }
                                if ( javaVersion >= 9 )
                                {
                                    findMainModule = true;
                                }
                            }
                            catch ( ClassNotFoundException e )
                            {
                                if ( m_debug )
                                {
                                    printDebug( WrapperNativeLibrary.getRes().getString( "Failed to retrieve main class: {0}", e.getMessage() ) ); 
                                }
                                mainClassClass = null;
                            }
                            catch ( Throwable t )
                            {
                                // This output will cause the stack below to be printed at loglevel ERROR.
                                tm.put( "mainclass", "load failed - " + mainClassName );
                                printParseInformation( tm );
                                t.printStackTrace();
                                return;
                            }

                            if ( mainClassClass != null )
                            {
                                tm.put( "mainclass", mainClassName );
                            }
                            else
                            {
                                tm.put( "mainclass", "not found - " + mainClassName );
                            }
                        }

                        String classPath = attributes.getValue( "Class-Path" );
                        if ( classPath == null )
                        {
                            tm.put( "mainjar", "no classpath" );
                        }
                        else
                        {
                            tm.put( "classpath", classPath );
                        }
                    }
                }
            }
        }
        else
        {
            // This output will cause the error after to be printed at loglevel ERROR.
            tm.put( "invalid_argument", "" );
            printParseInformation( tm );
            return;
        }

        if ( mainClassClass != null )
        {
            if ( findMainModule )
            {
                Object moduleName = null;
                try
                {
                    // use reflection as this requires java 9
                    Method m = Class.class.getMethod( "getModule", new Class[] { } );
                    Object module = m.invoke( mainClassClass, new Object[] { } );
                    m = module.getClass().getMethod( "getName", new Class[] { } );
                    moduleName = (String)m.invoke( module, new Object[] { } );
                    if ( moduleName != null )
                    {
                        tm.put( "mainmodule", moduleName );
                    }
                    else
                    {
                        tm.put( "mainmodule", "unnamed" );
                    }
                }
                catch ( Exception e )
                {
                    if ( m_debug )
                    {
                        printDebug( WrapperNativeLibrary.getRes().getString( "Failed to retrieve main module: {0}", e.getMessage() ) ); 
                    }
                    tm.put( "mainmodule", "not found" );
                }
            }

            Package pkg = mainClassClass.getPackage();
            if ( pkg != null )
            {
                tm.put( "package", pkg.getName() );
            }
            else
            {
                // should not happen because the class was retrieved using the fully qualified name.
                tm.put( "package", "not found" );
            }
        }

        // check that modules used by the Wrapper are available (might not be the case if running with a custom image)
        if ( javaVersion >= 9 )
        {
            if ( getModuleByName( "java.management" ) == null )
            {
                tm.put( "java.management", "not available" );
            }
        }

        // TODO: Make sure that the method is public and static?

        printParseInformation( tm );

        // Normally, no other non-daemon threads should be running. However, as a precaution, call System.exit() to ensure the process exits as quickly as possible.
        System.exit( 0 );
    }

    private static void printParseInformation( HashMap tm )
    {
        StringBuffer sb = new StringBuffer();
        for ( Iterator iter = tm.keySet().iterator(); iter.hasNext(); )
        {
            String name = (String)iter.next();
            sb.append( "WrapperBootstrap: " );
            sb.append( name );
            sb.append( ": " );
            sb.append( (String)tm.get( name ) );
            sb.append( System.getProperty( "line.separator" ) );
        }
        // This token is used to confirm that we have finished printing all the information.
        sb.append( "WrapperBootstrap: --" );

        System.out.println( sb.toString() );
    }

    private static void printDebug( String message )
    {
        System.out.println( WrapperNativeLibrary.getRes().getString( "WrapperBootstrap Debug: {0}", message ) );
    }

    private static int getJavaVersion()
    {
        String version = System.getProperty( "java.version" );
        if ( version.startsWith( "1." ) )
        {
            // Java 8 or lower
            version = version.substring( 2, 3 );
        }
        else
        {
            // Java 9 or higher
            int i = version.indexOf( "." );
            if ( i != -1 ) 
            {
                version = version.substring( 0, i );
            }
        }
        return Integer.parseInt( version );
    }

    private static Object getModuleByName( String moduleName )
    {
        Object module = null;
        Class moduleLayerClass;
        try
        {
            moduleLayerClass = Class.forName( "java.lang.ModuleLayer" );
        }
        catch ( ClassNotFoundException e )
        {
            if ( m_debug )
            {
                printDebug( WrapperNativeLibrary.getRes().getString( "java.lang.ModuleLayer not found" ) ); 
            }
            moduleLayerClass = null;
        }
        if ( moduleLayerClass != null )
        {
            try
            {
                Method m = moduleLayerClass.getMethod( "boot", new Class[] { } );
                Object boot = m.invoke( null, new Object[] { } );
                if ( boot != null )
                {
                    m = moduleLayerClass.getMethod( "findModule", new Class[] { String.class } );
                    Object optional = m.invoke( boot, new Object[] { moduleName } );
                    m = optional.getClass().getMethod( "get", new Class[] { });
                    try 
                    {
                        module = m.invoke( optional, new Object[] { } );
                    }
                    catch ( InvocationTargetException e ) {
                        if ( m_debug && e.getCause() instanceof NoSuchElementException ) {
                            printDebug( WrapperNativeLibrary.getRes().getString( "Failed to find module ''{0}''.", moduleName ) );
                        }
                    }
                }
                else
                {
                    // boot layer not initialized (should not happen)
                }
            }
            catch ( Exception e )
            {
                if ( m_debug )
                {
                    printDebug( WrapperNativeLibrary.getRes().getString( "Failed to retrieve module ''{0}'': {1}", moduleName, e.getMessage() ) ); 
                }
            }
        }
        return module;
    }
}
