
import Component from '@glimmer/component';
import { HdsButtonSet, HdsFormRadioGroup, HdsAlert, HdsButton, HdsTextDisplay, HdsCardContainer, HdsModal } from '@hashicorp/design-system-components/components';
import { on } from '@ember/modifier';
import { tracked } from '@glimmer/tracking';
import { get } from '@ember/object';
import { precompileTemplate } from '@ember/template-compilation';
import { setComponentTemplate } from '@ember/component';
import { g, i } from 'decorator-transforms/runtime';

class DownloadRecordsModal extends Component {
  static {
    g(this.prototype, "format", [tracked], function () {
      return 'json';
    });
  }
  #format = (i(this, "format"), void 0);
  static {
    g(this.prototype, "isLoading", [tracked], function () {
      return false;
    });
  }
  #isLoading = (i(this, "isLoading"), void 0);
  static {
    g(this.prototype, "hasError", [tracked], function () {
      return false;
    });
  }
  #hasError = (i(this, "hasError"), void 0);
  static {
    g(this.prototype, "downloadUrl", [tracked]);
  }
  #downloadUrl = (i(this, "downloadUrl"), void 0);
  static {
    g(this.prototype, "hasData", [tracked], function () {
      return false;
    });
  }
  #hasData = (i(this, "hasData"), void 0);
  get isContinueShown() {
    return !this.isLoading && !this.hasData;
  }
  get downloadName() {
    return `download.${this.format}`;
  }
  get isCSV() {
    return this.format === 'csv';
  }
  handleFormatChange = e => {
    const target = e.target;
    this.format = target?.value;
  };
  startDownload = async () => {
    try {
      this.isLoading = true;
      const listData = await this.args.onDownload();
      this.hasData = true;
      if (this.format === 'json') {
        this.downloadUrl = this.createJSONDownloadUrl(listData);
      } else if (this.format === 'csv') {
        this.downloadUrl = this.createCSVDownloadUrl(listData);
      }
    } catch (e) {
      console.error('Error creating download', e);
      this.hasError = true;
    } finally {
      this.isLoading = false;
    }
  };
  onFileDownloaded = () => {
    this.args.onFileDownloaded(this.format);
  };
  createJSONDownloadUrl = listData => {
    return `data:text/json;charset=utf-8,${encodeURIComponent(JSON.stringify(listData))}`;
  };
  /**
  * Converts list of data to a CSV
  * - Each cell is quoted to allow for newlines and commas in the dta
  * - Quotes within are escaped with a double quote
  * https://www.rfc-editor.org/rfc/rfc4180
  * @param {ListData} listData
  */
  createCSVDownloadUrl = listData => {
    const headers = this.args.csvFields.map(({
      label
    }) => label).join(',');
    const rows = listData.map(datum => this.args.csvFields.filter(({
      key
    }) => Boolean(key)).map(({
      key
    }) => {
      let cellAsString;
      const rawCell = get(datum, key || '');
      // Handle any undefined or null values but let other falsey values (0, '', false) through
      if (rawCell === undefined || rawCell === null) {
        cellAsString = '';
      } else if (typeof rawCell !== 'string' && typeof rawCell !== 'number') {
        cellAsString = JSON.stringify(rawCell);
      } else {
        cellAsString = String(rawCell);
      }
      const escapedCell = cellAsString.replace(/"/g, '""');
      return `"${escapedCell}"`;
    }).join(',')).join('\r\n');
    const csv = `${headers}\r\n${rows}`;
    return `data:text/csv;charset=utf-8,${encodeURIComponent(csv)}`;
  };
  static {
    setComponentTemplate(precompileTemplate("\n    <HdsModal @onClose={{@onClose}} data-test-download-records-modal id=\"download-records\" as |M|>\n      <M.Header>\n        Download records\n      </M.Header>\n      <M.Body>\n        {{#if this.hasData}}\n          <HdsCardContainer {{!-- @glint-expect-error --}} @background=\"neutral-primary\" @hasBorder={{true}} {{!-- @glint-expect-error --}} @level=\"mid\" class=\"download-records__download-card\">\n            <HdsTextDisplay @size=\"200\">\n              Download records\n            </HdsTextDisplay>\n            <HdsButton @color=\"secondary\" @href={{this.downloadUrl}} @icon=\"download\" @isIconOnly={{true}} @text=\"Download\" data-test-download-records-modal-download-button download={{this.downloadName}} {{on \"click\" this.onFileDownloaded}} />\n          </HdsCardContainer>\n          <HdsAlert class=\"download-records__disclaimer\" @type=\"compact\" @color=\"warning\" as |A|>\n            <A.Description>\n              Please download the file before closing the popup.\n            </A.Description>\n          </HdsAlert>\n\n        {{else if this.isLoading}}\n          <div data-test-download-records-modal-loader>\n\n            <HdsTextDisplay @size=\"200\">\n              {{#if this.isCSV}}\n                Generating CSV file...\n              {{else}}\n                Generating JSON file...\n              {{/if}}\n            </HdsTextDisplay>\n            <HdsAlert class=\"download-records__disclaimer\" @type=\"compact\" @color=\"warning\" as |A|>\n              <A.Description>\n                Do not close this popup or refresh your browser.\n              </A.Description>\n            </HdsAlert>\n\n          </div>\n        {{else}}\n          <HdsFormRadioGroup @name=\"format\" as |G|>\n            <G.Legend class=\"hds-typography-body-100-font-size\">\n              Select the download file format and click \"Continue\" to proceed.\n            </G.Legend>\n            <G.RadioField @value=\"json\" data-test-download-records-modal-radio-json checked {{on \"change\" this.handleFormatChange}} as |F|>\n              <F.Label>\n                JSON\n              </F.Label>\n            </G.RadioField>\n            <G.RadioField @value=\"csv\" data-test-download-records-modal-radio-csv {{on \"change\" this.handleFormatChange}} as |F|>\n              <F.Label>\n                CSV\n              </F.Label>\n            </G.RadioField>\n          </HdsFormRadioGroup>\n          <HdsAlert class=\"download-records__disclaimer\" @type=\"compact\" @color=\"warning\" as |A|>\n            <A.Description>\n              Downloads are limited to 1000 records at this time.\n            </A.Description>\n          </HdsAlert>\n        {{/if}}\n      </M.Body>\n      <M.Footer as |F|>\n        <HdsButtonSet>\n          {{#if this.isContinueShown}}\n            <HdsButton type=\"button\" data-test-download-records-modal-submit @text=\"Continue\" {{on \"click\" this.startDownload}} />\n          {{/if}}\n          <HdsButton type=\"button\" data-test-download-records-modal-cancel @color=\"secondary\" @text=\"Cancel\" {{on \"click\" F.close}} />\n        </HdsButtonSet>\n      </M.Footer>\n    </HdsModal>\n  ", {
      strictMode: true,
      scope: () => ({
        HdsModal,
        HdsCardContainer,
        HdsTextDisplay,
        HdsButton,
        on,
        HdsAlert,
        HdsFormRadioGroup,
        HdsButtonSet
      })
    }), this);
  }
}

export { DownloadRecordsModal as default };
//# sourceMappingURL=download-records.js.map
