
import { isTesting } from '@embroider/macros';

/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: BUSL-1.1
 */


// When under test we will use UTC to make it easier to have consistent assertions
const defaultTimeZone = isTesting() ? 'Etc/UTC' : Intl.DateTimeFormat().resolvedOptions().timeZone;

/**
 * Formats a Date object into a human-readable date-time string
 * @param date - The Date object to format
 * @param timeZone - The timezone to use for formatting (defaults to UTC in tests, local timezone otherwise)
 * @returns A formatted string like "Jan 15, 2025, 14:30 UTC"
 */
const formatDateTime = (date, timeZone = defaultTimeZone) => {
  return new Intl.DateTimeFormat('en-US', {
    year: 'numeric',
    month: 'short',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
    hour12: false,
    timeZoneName: 'short',
    timeZone
  }).format(date);
};

/**
 * Converts a number of days to a gRPC duration string format
 * @param days - The number of days to convert
 * @returns A duration string in hours format (e.g., "72h" for 3 days)
 */
const daysToGrpcDuration = days => {
  const hours = days * 24;
  return `${hours}h`;
};

/**
 * Calculates and formats the time remaining until a future date
 * @param futureDate - A Date object or ISO string representing the target date
 * @returns A human-readable string describing time until the date (e.g., "2 days, 5 hours", "Overdue")
 */
const timeUntil = futureDate => {
  const targetDate = typeof futureDate === 'string' ? new Date(futureDate) : futureDate;
  const now = new Date();
  const diffMs = targetDate.getTime() - now.getTime();
  if (diffMs <= 0) return 'Overdue';
  const diffHours = Math.floor(diffMs / (1000 * 60 * 60));
  const diffDays = Math.floor(diffHours / 24);
  const remainingHours = diffHours % 24;
  if (diffDays > 0) {
    if (remainingHours > 0) {
      return `${diffDays} day${diffDays !== 1 ? 's' : ''}, ${remainingHours} hour${remainingHours !== 1 ? 's' : ''}`;
    }
    return `${diffDays} day${diffDays !== 1 ? 's' : ''}`;
  } else if (diffHours > 0) {
    return `${diffHours} hour${diffHours !== 1 ? 's' : ''}`;
  } else {
    const diffMinutes = Math.floor(diffMs / (1000 * 60));
    return `${diffMinutes} minute${diffMinutes !== 1 ? 's' : ''}`;
  }
};

/**
 * Formats a duration string from seconds format to human-readable format
 * @param duration - A duration string in seconds format (e.g., "3600s") or null
 * @returns A human-readable duration string (e.g., "1 hour") or empty string if null/invalid
 */
const formatDuration = duration => {
  if (!duration) {
    return '';
  }
  const secondsMatch = duration.match(/^(\d+)s$/);
  if (secondsMatch && secondsMatch[1]) {
    const seconds = parseInt(secondsMatch[1], 10);
    if (seconds < 60) {
      return `${seconds} second${seconds !== 1 ? 's' : ''}`;
    } else if (seconds < 3600) {
      const minutes = Math.floor(seconds / 60);
      return `${minutes} minute${minutes !== 1 ? 's' : ''}`;
    } else if (seconds < 86400) {
      const hours = Math.floor(seconds / 3600);
      return `${hours} hour${hours !== 1 ? 's' : ''}`;
    } else if (seconds < 2592000) {
      const days = Math.floor(seconds / 86400);
      return `${days} day${days !== 1 ? 's' : ''}`;
    } else if (seconds < 31536000) {
      const months = Math.floor(seconds / 2592000);
      return `${months} month${months !== 1 ? 's' : ''}`;
    } else {
      const years = Math.floor(seconds / 31536000);
      return `${years} year${years !== 1 ? 's' : ''}`;
    }
  }
  return duration;
};
const dateTimeUtils = {
  formatDateTime,
  daysToGrpcDuration,
  formatDuration,
  timeUntil
};

export { daysToGrpcDuration, dateTimeUtils as default, formatDateTime, formatDuration, timeUntil };
//# sourceMappingURL=date-time.js.map
