
import { HdsButton, HdsLinkInline, HdsTextBody, HdsPageHeader } from '@hashicorp/design-system-components/components';
import Component from '@glimmer/component';
import { flattenFilters } from '../../../utils/filters.js';
import InventoryTable from '../inventory-table.js';
import '../filter-bar.js';
import updateData from '../../../modifiers/update-data.js';
import { tracked } from '@glimmer/tracking';
import { service } from '@ember/service';
import { on } from '@ember/modifier';
import { daysToGrpcDuration, formatDateTime, timeUntil, formatDuration } from '../../../utils/date-time.js';
import DownloadRecordsModal from '../modals/download-records.js';
import { precompileTemplate } from '@ember/template-compilation';
import { setComponentTemplate } from '@ember/component';
import { g, i } from 'decorator-transforms/runtime';

/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: BUSL-1.1
 */
class SecretInventory extends Component {
  static {
    g(this.prototype, "reportingApi", [service]);
  }
  #reportingApi = (i(this, "reportingApi"), void 0);
  static {
    g(this.prototype, "page", [tracked], function () {
      return [];
    });
  }
  #page = (i(this, "page"), void 0);
  static {
    g(this.prototype, "isLoading", [tracked], function () {
      return false;
    });
  }
  #isLoading = (i(this, "isLoading"), void 0);
  static {
    g(this.prototype, "isError", [tracked], function () {
      return false;
    });
  }
  #isError = (i(this, "isError"), void 0);
  static {
    g(this.prototype, "isEmpty", [tracked], function () {
      return false;
    });
  }
  #isEmpty = (i(this, "isEmpty"), void 0);
  static {
    g(this.prototype, "nextPageToken", [tracked]);
  }
  #nextPageToken = (i(this, "nextPageToken"), void 0);
  static {
    g(this.prototype, "previousPageToken", [tracked]);
  }
  #previousPageToken = (i(this, "previousPageToken"), void 0);
  static {
    g(this.prototype, "pageSize", [tracked], function () {
      return 10;
    });
  }
  #pageSize = (i(this, "pageSize"), void 0);
  static {
    g(this.prototype, "totalUnfilteredCount", [tracked], function () {
      return 0;
    });
  }
  #totalUnfilteredCount = (i(this, "totalUnfilteredCount"), void 0);
  static {
    g(this.prototype, "isDownloadModalOpen", [tracked], function () {
      return false;
    });
  }
  #isDownloadModalOpen = (i(this, "isDownloadModalOpen"), void 0);
  lastRequest;
  handlePageSizeChange = newPageSize => {
    this.pageSize = newPageSize;
    this.nextPageToken = undefined;
    this.previousPageToken = undefined;
    this.args.onPageChange({
      page_size: newPageSize,
      next_page_token: undefined,
      previous_page_token: undefined
    });
  };
  static {
    g(this.prototype, "lastUpdatedTime", [tracked], function () {
      return '';
    });
  }
  #lastUpdatedTime = (i(this, "lastUpdatedTime"), void 0);
  static {
    g(this.prototype, "quickFilters", [tracked], function () {
      return [{
        label: 'Stale secrets',
        description: 'Static secrets that have not been accessed in 90+ days.',
        getParams: () => {
          const filters = [{
            field: 'secretLastAccessedAt',
            operator: '<',
            value: {
              type: 'duration',
              value: daysToGrpcDuration(-90)
            }
          }];
          return {
            filters
          };
        }
      }, {
        label: 'Long-lived secrets',
        description: 'All secrets that have not been updated in 90+ days',
        getParams: () => {
          const filters = [{
            field: 'secretLastModifiedAt',
            operator: '<',
            value: {
              type: 'duration',
              value: daysToGrpcDuration(-90)
            }
          }, {
            field: 'secretTypes',
            operator: '=',
            value: {
              type: 'list',
              value: ['static']
            }
          }];
          return {
            filters
          };
        }
      }, {
        label: 'Upcoming secret rotations',
        description: 'Auto-rotating secrets scheduled to rotate in less than 30 days.',
        getParams: () => {
          const filters = [{
            field: 'expiresOrNextRotationAt',
            operator: '<',
            value: {
              type: 'duration',
              value: daysToGrpcDuration(90)
            }
          }, {
            field: 'secretTypes',
            operator: '=',
            value: {
              type: 'list',
              value: ['auto-rotating']
            }
          }];
          return {
            filters
          };
        }
      }];
    });
  }
  #quickFilters = (i(this, "quickFilters"), void 0);
  filterFieldDefinitions = [{
    label: 'Secret name',
    level: 'primary',
    controls: [{
      name: 'secretNames',
      type: 'search'
    }]
  }, {
    label: 'Secret engine',
    controls: [{
      name: 'pluginNames',
      type: 'single-select',
      options: [{
        name: 'KV-v1',
        value: 'kv-v1'
      }, {
        name: 'KV-v2',
        value: 'kv-v2'
      }]
    }],
    level: 'primary'
  }, {
    label: 'Secret type',
    controls: [{
      label: 'Secret type',
      name: 'secretTypes',
      type: 'multi-select',
      options: [{
        name: 'Static (KV)',
        value: 'static'
      }, {
        name: 'Dynamic',
        value: 'dynamic'
      }, {
        name: 'Auto-rotating',
        value: 'auto-rotating'
      }]
    }]
  }, {
    label: 'Namespace',
    controls: [{
      label: 'Namespace',
      name: 'namespaces',
      type: 'searchable-multi-select',
      placeholder: 'Search for namespace',
      onSearch: async search => {
        const {
          namespaces = []
        } = await this.reportingApi.reporting.fetchDistinctNamespaces({
          organizationId: this.reportingApi.organizationId,
          projectId: this.reportingApi.projectId,
          clusterId: this.args.activeCluster,
          namespacePathPrefix: search
        });
        return namespaces.map(namespace => ({
          name: namespace.path ?? '',
          value: namespace.path ?? ''
        })).filter(namespace => Boolean(namespace.name));
      }
    }]
  }, {
    label: 'Mount path',
    controls: [{
      label: 'Mount path',
      name: 'mountPaths',
      type: 'searchable-multi-select',
      placeholder: 'Search for mount path',
      onSearch: async search => {
        const {
          mounts = []
        } = await this.reportingApi.reporting.fetchDistinctMounts({
          organizationId: this.reportingApi.organizationId,
          projectId: this.reportingApi.projectId,
          clusterId: this.args.activeCluster,
          mountPathPrefix: search
        });
        return mounts.map(mount => ({
          name: mount.mountPath ?? '',
          value: mount.mountPath ?? ''
        })).filter(mount => Boolean(mount.name));
      }
    }]
  }, {
    label: 'Created',
    controls: [{
      name: 'createdAt',
      label: 'Created in the last',
      type: 'lookback',
      options: [{
        name: '7 days',
        value: daysToGrpcDuration(-7)
      }, {
        name: '30 days',
        value: daysToGrpcDuration(-30)
      }, {
        name: '60 days',
        value: daysToGrpcDuration(-60)
      }, {
        name: '90 days',
        value: daysToGrpcDuration(-90)
      }]
    }, {
      name: 'createdByIds',
      label: 'Created by',
      type: 'text'
    }]
  }, {
    label: 'Last accessed',
    controls: [{
      label: 'Accessed in the last',
      name: 'secretLastAccessedAt',
      type: 'lookback',
      options: [{
        name: '7 days',
        value: daysToGrpcDuration(-7)
      }, {
        name: '30 days',
        value: daysToGrpcDuration(-30)
      }, {
        name: '60 days',
        value: daysToGrpcDuration(-60)
      }, {
        name: '90 days',
        value: daysToGrpcDuration(-90)
      }]
    }, {
      label: 'Last accessed by',
      name: 'secretLastAccessedByIds',
      type: 'text'
    }]
  }, {
    label: 'Last modified',
    controls: [{
      label: 'Modified in the last',
      name: 'secretLastModifiedAt',
      type: 'lookback',
      options: [{
        name: '7 days',
        value: daysToGrpcDuration(-7)
      }, {
        name: '30 days',
        value: daysToGrpcDuration(-30)
      }, {
        name: '60 days',
        value: daysToGrpcDuration(-60)
      }, {
        name: '90 days',
        value: daysToGrpcDuration(-90)
      }]
    }, {
      label: 'Last modified by',
      name: 'secretLastModifiedByIds',
      type: 'text'
    }]
  }, {
    label: 'TTL',
    controls: [{
      label: 'Secrets that expire in',
      name: 'expiresOrNextRotationAt',
      type: 'lookback',
      options: [{
        name: 'Less than 1 hour',
        value: '1h'
      }, {
        name: 'Less than 24 hours',
        value: '24h'
      }, {
        name: 'Less than 7 days',
        value: daysToGrpcDuration(7)
      }, {
        name: 'Less than 30 days',
        value: daysToGrpcDuration(30)
      }, {
        name: 'Less than 60 days',
        value: daysToGrpcDuration(60)
      }, {
        name: 'Less than 90 days',
        value: daysToGrpcDuration(90)
      }]
    }]
  }, {
    label: 'Deleted',
    controls: [{
      label: 'Include deleted secrets',
      name: 'includeDeleted',
      type: 'boolean'
    }, {
      label: 'Deleted in the last',
      name: 'secretDeletedAt',
      dependsOn: filters => {
        return filters['includeDeleted']?.value.value === true;
      },
      type: 'lookback',
      options: [{
        name: 'Last 7 days',
        value: daysToGrpcDuration(-7)
      }, {
        name: 'Last 30 days',
        value: daysToGrpcDuration(-30)
      }, {
        name: 'Last 60 days',
        value: daysToGrpcDuration(-60)
      }, {
        name: 'Last 90 days',
        value: daysToGrpcDuration(-90)
      }]
    }, {
      label: 'Deleted by',
      name: 'secretDeletedByIds',
      dependsOn: filters => filters['includeDeleted']?.value.value === true,
      type: 'text'
    }]
  }];
  columns = [{
    label: 'Secret name',
    key: 'secretName',
    secondaryKey: 'secretType',
    isSortable: true,
    width: '200px'
  }, {
    label: 'Engine',
    key: 'pluginName',
    width: '100px',
    minWidth: '100px'
  }, {
    label: 'Namespace',
    key: 'namespace',
    isSortable: true,
    width: '200px'
  }, {
    label: 'Mount path',
    key: 'mountPath',
    isSortable: true,
    width: '200px'
  }, {
    label: 'Created',
    key: 'createdAt',
    isSortable: true,
    cellFormatter: row => {
      return formatDateTime(new Date(row['createdAt']));
    },
    secondaryKey: 'createdById',
    width: '200px'
  }, {
    label: 'Last accessed',
    key: 'secretLastAccessedAt',
    cellFormatter: row => {
      return formatDateTime(new Date(row['secretLastAccessedAt']));
    },
    secondaryKey: 'secretLastAccessedById',
    isSortable: true,
    tooltip: 'Shows when this secret was last viewed or used and by whom.',
    width: '200px'
  }, {
    label: 'Last modified',
    key: 'secretLastModifiedAt',
    cellFormatter: row => {
      return formatDateTime(new Date(row['secretLastModifiedAt']));
    },
    secondaryKey: 'secretLastModifiedById',
    isSortable: true,
    tooltip: 'Shows when this secret was manually changed and by whom. Includes updating secret values, manual rotations, and deletions. ',
    width: '200px'
  }, {
    label: 'Versions',
    key: 'versionCount',
    width: '100px',
    minWidth: '100px'
  }, {
    label: 'Next rotation',
    key: 'nextRotationAt',
    isSortable: true,
    cellFormatter: row => {
      const date = row['nextRotationAt'];
      return date ? timeUntil(date) : '';
    },
    secondaryKey: 'nextRotationAt',
    secondaryCellFormatter: row => {
      const date = row['nextRotationAt'];
      return date ? formatDateTime(new Date(date)) : '';
    },
    tooltip: 'Next scheduled rotation date for auto-rotating secrets.',
    width: '200px'
  }, {
    label: 'TTL',
    key: 'ttl',
    cellFormatter: row => {
      const ttl = row['ttl'];
      return ttl ? formatDuration(ttl) : '';
    },
    isSortable: true,
    tooltip: 'Shows how long a secret will remain valid from the moment it is created',
    width: '150px'
  }, {
    label: 'Deleted',
    key: 'secretDeletedAt',
    cellFormatter: row => {
      const date = row['secretDeletedAt'];
      return date ? formatDateTime(new Date(date)) : '';
    },
    secondaryKey: 'secretDeletedById',
    isSortable: true,
    tooltip: 'Indicates this KV v2 secret is hidden but recoverable',
    width: '200px'
  }];
  get visibleColumnKeys() {
    // If visibleColumns is provided from query params, use it
    if (this.args.visibleColumns && this.args.visibleColumns.length > 0) {
      return this.args.visibleColumns;
    }
    // Default to first 7 columns when no query params
    return this.columns.slice(0, 7).map(col => col.key);
  }
  handleColumnsChanged = visibleColumnKeys => {
    if (this.args.onColumnsChanged) {
      this.args.onColumnsChanged(visibleColumnKeys);
    }
  };
  get visibleColumns() {
    return this.columns.filter(col => this.visibleColumnKeys.includes(col.key));
  }
  fetchQuickFilterCount = async filter => {
    const params = filter.getParams();
    const {
      reporting
    } = this.reportingApi;
    const baseFilters = {
      ...flattenFilters(params.filters || []),
      organizationId: this.reportingApi.organizationId,
      projectId: this.reportingApi.projectId,
      clusterId: this.args.activeCluster
    };
    const {
      count
    } = await reporting.fetchSecretsInventoryCount(baseFilters);
    const updatedFilter = {
      ...filter,
      count: Number(count ?? 0)
    };
    return updatedFilter;
  };
  fetchQuickFilterCounts = async () => {
    const {
      count: totalUnfilteredCount
    } = await this.reportingApi.reporting.fetchSecretsInventoryCount({
      organizationId: this.reportingApi.organizationId,
      projectId: this.reportingApi.projectId,
      clusterId: this.args.activeCluster
    });
    this.totalUnfilteredCount = totalUnfilteredCount ? Number(totalUnfilteredCount) : 0;
    this.quickFilters = await Promise.all(this.quickFilters.map(filter => {
      return this.fetchQuickFilterCount(filter);
    }));
  };
  handleDataUpdate = async filters => {
    const {
      reporting
    } = this.reportingApi;
    const flattenedFilters = flattenFilters(filters);
    this.lastRequest = {
      ...flattenedFilters,
      organizationId: this.reportingApi.organizationId,
      projectId: this.reportingApi.projectId,
      clusterId: this.args.activeCluster,
      paginationPageSize: this.pageSize,
      paginationPreviousPageToken: this.args.previousPageToken,
      paginationNextPageToken: this.args.nextPageToken
    };
    if (this.args.appliedSort && this.args.appliedSort.length > 0) {
      const [sortBy, sortOrder] = (this.args.appliedSort?.[0] || '').split(' ');
      if (sortBy && sortOrder) {
        const sortByUnderscore = sortBy.replace(/([a-z])([A-Z])/g, '$1_$2').toLowerCase();
        this.lastRequest.sortingOrderBy = [`${sortByUnderscore} ${sortOrder}`];
      }
    }
    this.isLoading = true;
    try {
      const {
        data,
        pagination,
        reportAsOf
      } = await reporting.fetchSecretsInventory(this.lastRequest);
      this.fetchQuickFilterCounts().catch(error => {
        console.warn('Problem fetching quick filter counts:', error);
      });
      this.lastUpdatedTime = formatDateTime(reportAsOf || new Date());
      this.page = data;
      this.isEmpty = !data || data.length === 0;
      this.nextPageToken = pagination?.nextPageToken;
      this.previousPageToken = pagination?.previousPageToken;
    } catch {
      this.isError = true;
    } finally {
      this.isLoading = false;
    }
  };
  handleDownload = async () => {
    const {
      reporting
    } = this.reportingApi;
    const downloadRequest = {
      ...(this.lastRequest || {}),
      organizationId: this.reportingApi.organizationId,
      projectId: this.reportingApi.projectId,
      clusterId: this.args.activeCluster,
      paginationPageSize: 1000
    };
    delete downloadRequest.paginationPreviousPageToken;
    delete downloadRequest.paginationNextPageToken;
    const {
      data
    } = await reporting.fetchSecretsInventory(downloadRequest);
    return data;
  };
  closeDownloadModal = () => {
    this.isDownloadModalOpen = false;
  };
  openDownloadModal = () => {
    this.isDownloadModalOpen = true;
  };
  static {
    setComponentTemplate(precompileTemplate("\n    <div class=\"secret-inventory\" data-test-vault-reporting-secret-inventory {{updateData this.handleDataUpdate @appliedFilters}}>\n      <HdsPageHeader as |PH|>\n        <PH.Title data-test-vault-reporting-secret-inventory-page-title>Secret\n          Inventory\n        </PH.Title>\n        <PH.Description>\n          <HdsTextBody @tag=\"p\" @size=\"200\" @color=\"primary\" class=\"secret-inventory__description\">\n            {{#if @activeCluster}}\n              View all secrets in\n              <strong>{{@activeCluster}}</strong>\n              with filtering to support security assessments.\n            {{/if}}\n            {{#if this.lastUpdatedTime}}\n\n              Last updated\n              {{this.lastUpdatedTime}}.\n            {{/if}}\n            Can\u2019t find what you\u2019re looking for?\n            <HdsLinkInline class=\"dashboard__survey-link\" data-test-vault-reporting-dashboard-survey-link @icon=\"external-link\" @href=\"https://forms.gle/wMvjYN5cZQwgLWyRA\" target=\"_blank\">Share feedback</HdsLinkInline>\n          </HdsTextBody>\n\n        </PH.Description>\n        <PH.Actions>\n          {{yield to=\"actions\"}}\n\n          <HdsButton @color=\"primary\" @text=\"Export\" data-test-vault-reporting-secret-inventory-download-button {{on \"click\" this.openDownloadModal}} />\n        </PH.Actions>\n      </HdsPageHeader>\n      <InventoryTable data-test-vault-reporting-secret-inventory-table @isLoading={{this.isLoading}} @isError={{this.isError}} @isEmpty={{this.isEmpty}} @onFilterApplied={{@onFilterApplied}} @onSortApplied={{@onSortApplied}} @appliedFilters={{@appliedFilters}} @appliedSort={{@appliedSort}} @onPageChange={{@onPageChange}} @nextPageToken={{this.nextPageToken}} @previousPageToken={{this.previousPageToken}} @pageSize={{this.pageSize}} @onPageSizeChange={{this.handlePageSizeChange}} @filterFieldDefinitions={{this.filterFieldDefinitions}} @quickFilters={{this.quickFilters}} @columns={{this.columns}} @visibleColumnKeys={{this.visibleColumnKeys}} @onColumnsChanged={{this.handleColumnsChanged}} @rows={{this.page}} @totalUnfilteredCount={{this.totalUnfilteredCount}} />\n      {{#if this.isDownloadModalOpen}}\n        <DownloadRecordsModal @onDownload={{this.handleDownload}} @csvFields={{this.columns}} @onClose={{this.closeDownloadModal}} />\n      {{/if}}\n    </div>\n  ", {
      strictMode: true,
      scope: () => ({
        updateData,
        HdsPageHeader,
        HdsTextBody,
        HdsLinkInline,
        HdsButton,
        on,
        InventoryTable,
        DownloadRecordsModal
      })
    }), this);
  }
}

export { SecretInventory as default };
//# sourceMappingURL=secret-inventory.js.map
