/**
 * Copyright IBM Corp. 2016, 2025
 * SPDX-License-Identifier: BUSL-1.1
 */

/*
Authentication requests return authentication information in either the "auth" or "data" key, 
depending on the authentication method. 

The stubbed responses below are used to test and compare authentication logic across different method types:
- If the method has `{ mount_type: "token" }`, the authentication details are returned inside the "data" key.
- Otherwise, `mount_type` is an empty string (""), and the authentication details are found in the "auth" key.

Some values depend on the method's mount configuration (such as `ttl` and `lease_duration`). 
For consistency, all methods stubbed here were mounted using default settings.
*/

const BASE_REQUEST_DATA = {
  request_id: 'cbca58ca-8b53-76e7-3d07-6d72f7c5affe',
  lease_id: '',
  renewable: false,
  lease_duration: 0,
  wrap_info: null,
  warnings: null,
};

export const RESPONSE_STUBS = {
  github: {
    ...BASE_REQUEST_DATA,
    data: null,
    auth: {
      client_token: 'hvs.myvaultgeneratedgithubtoken',
      accessor: 'm9UFOzTtVahYxs8nQe6Su73r',
      policies: ['default'],
      token_policies: ['default'],
      metadata: {
        org: 'hashicorp',
        username: 'github-user99',
      },
      lease_duration: 2764800,
      renewable: true,
      entity_id: 'd3396007-4d0e-33f9-7e4e-beb2e87c3518',
      token_type: 'service',
      orphan: true,
      mfa_requirement: null,
      num_uses: 0,
    },
    mount_type: '',
  },
  jwt: {
    ['lookup-self']: {
      ...BASE_REQUEST_DATA,
      data: {
        accessor: 'MkjSR78ducuarJ6ypCDbHhBp',
        creation_time: 1749659696,
        creation_ttl: 2764800,
        display_name: 'jwt-ugaKjSEAKwQkiGh1rbnGkp39oCSe3LQ2@clients',
        entity_id: 'b6061dc8-a19e-195e-43a8-43d37f4625dd',
        expire_time: '2025-07-13T12:34:56.345108-04:00',
        explicit_max_ttl: 0,
        id: 'hvs.myvaultgeneratedjwttoken',
        issue_time: '2025-06-11T12:34:56.345113-04:00',
        meta: {
          role: 'reader',
        },
        num_uses: 0,
        orphan: true,
        path: 'auth/jwt/login',
        policies: ['default', 'reader'],
        renewable: true,
        ttl: 2764800,
        type: 'service',
      },
      auth: null,
      mount_type: 'token',
    },
    login: {
      ...BASE_REQUEST_DATA,
      data: null,
      auth: {
        client_token: 'hvs.myvaultgeneratedjwttoken',
        accessor: 'wzT0PZBYP0ZWyXI0Cst3UlsY',
        policies: ['default', 'reader'],
        token_policies: ['default', 'reader'],
        metadata: {
          role: 'reader',
        },
        lease_duration: 2764800,
        renewable: true,
        entity_id: 'b6061dc8-a19e-195e-43a8-43d37f4625dd',
        token_type: 'service',
        orphan: true,
        mfa_requirement: null,
        num_uses: 0,
      },
      mount_type: '',
    },
  },
  ldap: {
    ...BASE_REQUEST_DATA,
    data: {}, // empty object instead of null
    auth: {
      client_token: 'hvs.myvaultgeneratedldaptoken',
      accessor: 'aIFJgy2Eo6qgIUx9bAuOKC6y',
      policies: ['default'],
      token_policies: ['default'],
      metadata: {
        username: 'bob.johnson',
      },
      lease_duration: 2764800,
      renewable: true,
      entity_id: '998d4fb7-c7db-8d81-c34d-dc1754103510',
      token_type: 'service',
      orphan: true,
      mfa_requirement: null,
      num_uses: 0,
    },
    mount_type: '',
  },
  oidc: {
    // Response from the OIDC token exchange (happens first)
    ['oidc/callback']: {
      ...BASE_REQUEST_DATA,
      data: null,
      auth: {
        client_token: 'hvs.myvaultgeneratedoidctoken',
        accessor: 'AjTM1Ec825ZJCg4xEVxbdPmf',
        policies: ['default'],
        token_policies: ['default'],
        metadata: {
          role: 'reader',
        },
        lease_duration: 2764800,
        renewable: true,
        entity_id: '18b57edf-acff-3e65-2ff2-6c772ce44924',
        token_type: 'service',
        orphan: true,
        mfa_requirement: null,
        num_uses: 0,
      },
      mount_type: '',
    },
    // Response from token lookup (after OIDC token exchange)
    'lookup-self': {
      ...BASE_REQUEST_DATA,
      data: {
        accessor: 'ew50HTqF2xgsmaKIsdKpJtTc',
        creation_time: 1749584514,
        creation_ttl: 2764800,
        display_name: 'my-oidc-google-oauth2|105299854624506884705',
        entity_id: '18b57edf-acff-3e65-2ff2-6c772ce44924',
        expire_time: '2025-07-12T15:41:54.961915-04:00',
        explicit_max_ttl: 0,
        id: 'hvs.myvaultgeneratedoidctoken',
        issue_time: '2025-06-10T15:41:54.961919-04:00',
        meta: {
          role: 'reader',
        },
        num_uses: 0,
        orphan: true,
        path: 'auth/my_oidc/oidc/callback',
        policies: ['default'],
        renewable: true,
        ttl: 2764799,
        type: 'service',
      },
      auth: null,
      mount_type: 'token',
    },
  },
  okta: {
    ...BASE_REQUEST_DATA,
    data: {}, // empty object instead of null
    auth: {
      client_token: 'hvs.myvaultgeneratedoktatoken',
      accessor: 'bnCp5tEioxHJXgSXbKowYoZj',
      policies: ['default'],
      token_policies: ['default'],
      metadata: {
        policies: '',
        username: 'vaultuser@gmail.com',
      },
      lease_duration: 2592000,
      renewable: true,
      entity_id: 'cb1ed44d-d3fb-5fd4-62cf-e027f84f35f6',
      token_type: 'service',
      orphan: true,
      mfa_requirement: null,
      num_uses: 0,
    },
    mount_type: '',
  },
  radius: {
    ...BASE_REQUEST_DATA,
    data: null,
    auth: {
      client_token: 'hvs.myvaultgeneratedradiustoken',
      accessor: 'wx4Df1iktankETDXZ67tpGgo',
      policies: ['default'],
      token_policies: ['default'],
      metadata: {
        policies: '',
        username: 'vaultuser',
      },
      lease_duration: 2764800,
      renewable: true,
      entity_id: '7056887c-7e54-3f76-e498-1f76fc0d0e2c',
      token_type: 'service',
      orphan: true,
      mfa_requirement: null,
      num_uses: 0,
    },
    mount_type: '',
  },
  token: {
    request_id: '6a7c7b72-6f0b-1700-5a4e-cd82e768b5d8',
    lease_id: '',
    renewable: false,
    lease_duration: 0,
    data: {
      accessor: '3tl0hAUwdDJVduSEnIca7Tr6',
      creation_time: 1744649084,
      creation_ttl: 2764800,
      display_name: 'token',
      entity_id: '',
      expire_time: '2025-05-16T09:44:44.837733-07:00',
      explicit_max_ttl: 0,
      id: 'hvs.myvaultgeneratedtoken',
      issue_time: '2025-04-14T09:44:44.837735-07:00',
      meta: null,
      num_uses: 0,
      orphan: false,
      path: 'auth/token/create',
      policies: ['default'],
      renewable: true,
      ttl: 2764785,
      type: 'service',
    },
    wrap_info: null,
    warnings: null,
    auth: null,
    mount_type: 'token',
  },
  userpass: {
    ...BASE_REQUEST_DATA,
    data: null,
    auth: {
      client_token: 'hvs.myvaultgenerateduserpasstoken',
      accessor: 'WSm7g8UzWEXhMO7g8C1zggDU',
      policies: ['default'],
      token_policies: ['default'],
      metadata: {
        username: 'bob',
      },
      lease_duration: 2764800,
      renewable: true,
      entity_id: 'fa17f31c-41b0-c927-7b2b-d905200bb95c',
      token_type: 'service',
      orphan: true,
      mfa_requirement: null,
      num_uses: 0,
    },
    mount_type: '',
  },
  // ENTERPRISE ONLY
  saml: {
    ['saml/token']: {
      ...BASE_REQUEST_DATA,
      data: null,
      auth: {
        client_token: 'hvs.myvaultgeneratedsamltoken',
        accessor: 'kHiH5wwqClnujASsKalca1T6',
        policies: ['default'],
        token_policies: ['default'],
        metadata: {
          role: 'dev',
        },
        lease_duration: 1800,
        renewable: true,
        entity_id: '81fc10e5-49a3-d0a2-9835-ac6b551ee266',
        token_type: 'service',
        orphan: true,
        mfa_requirement: null,
        num_uses: 0,
      },
      mount_type: '',
    },
    ['lookup-self']: {
      ...BASE_REQUEST_DATA,
      data: {
        accessor: 'H4fWtQaYX3aaEg1JIPSWiK9v',
        creation_time: 1749585309,
        creation_ttl: 1800,
        display_name: 'saml-vaultuser@hashicorp.com',
        entity_id: '81fc10e5-49a3-d0a2-9835-ac6b551ee266',
        expire_time: '2025-06-10T16:25:09.246659-04:00',
        explicit_max_ttl: 0,
        id: 'hvs.myvaultgeneratedsamltoken',
        issue_time: '2025-06-10T15:55:09.246666-04:00',
        meta: {
          role: 'dev',
        },
        num_uses: 0,
        orphan: true,
        path: 'auth/saml/token',
        policies: ['default'],
        renewable: true,
        ttl: 1800,
        type: 'service',
      },
      auth: null,
      mount_type: 'token',
    },
  },
};

// Once the auth service authentication method is simplified and no longer handles every auth type
// the "backend" key should be completely removable
export const TOKEN_DATA = {
  github: {
    authMethodType: 'github',
    authMountPath: 'github',
    displayName: `${RESPONSE_STUBS.github.auth.metadata.org}/${RESPONSE_STUBS.github.auth.metadata.username}`,
    entityId: RESPONSE_STUBS.github.auth.entity_id,
    policies: RESPONSE_STUBS.github.auth.policies,
    renewable: RESPONSE_STUBS.github.auth.renewable,
    token: RESPONSE_STUBS.github.auth.client_token,
    tokenExpirationEpoch: 1752352843223,
    ttl: RESPONSE_STUBS.github.auth.lease_duration,
    userRootNamespace: '',
  },
  ldap: {
    authMethodType: 'ldap',
    authMountPath: 'ldap',
    displayName: RESPONSE_STUBS.ldap.auth.metadata.username,
    entityId: RESPONSE_STUBS.ldap.auth.entity_id,
    policies: RESPONSE_STUBS.ldap.auth.policies,
    renewable: RESPONSE_STUBS.ldap.auth.renewable,
    token: RESPONSE_STUBS.ldap.auth.client_token,
    tokenExpirationEpoch: 1752352843696,
    ttl: RESPONSE_STUBS.ldap.auth.lease_duration,
    userRootNamespace: '',
  },
  jwt: {
    authMethodType: 'jwt',
    authMountPath: 'jwt',
    displayName: RESPONSE_STUBS.jwt['lookup-self'].data.display_name,
    entityId: RESPONSE_STUBS.jwt.login.auth.entity_id,
    policies: RESPONSE_STUBS.jwt.login.auth.policies,
    renewable: RESPONSE_STUBS.jwt.login.auth.renewable,
    token: RESPONSE_STUBS.jwt.login.auth.client_token,
    tokenExpirationEpoch: 1752425319766,
    ttl: RESPONSE_STUBS.jwt.login.auth.lease_duration,
    userRootNamespace: '',
  },
  oidc: {
    authMethodType: 'oidc',
    authMountPath: 'oidc',
    displayName: RESPONSE_STUBS.oidc['lookup-self'].data.display_name,
    entityId: RESPONSE_STUBS.oidc['oidc/callback'].auth.entity_id,
    policies: RESPONSE_STUBS.oidc['oidc/callback'].auth.policies,
    renewable: RESPONSE_STUBS.oidc['oidc/callback'].auth.renewable,
    token: RESPONSE_STUBS.oidc['oidc/callback'].auth.client_token,
    tokenExpirationEpoch: 1752349314961,
    ttl: RESPONSE_STUBS.oidc['oidc/callback'].auth.lease_duration,
    userRootNamespace: '',
  },
  okta: {
    authMethodType: 'okta',
    authMountPath: 'okta',
    displayName: RESPONSE_STUBS.okta.auth.metadata.username,
    entityId: RESPONSE_STUBS.okta.auth.entity_id,
    policies: RESPONSE_STUBS.okta.auth.policies,
    renewable: RESPONSE_STUBS.okta.auth.renewable,
    token: RESPONSE_STUBS.okta.auth.client_token,
    tokenExpirationEpoch: 1752180044950,
    ttl: RESPONSE_STUBS.okta.auth.lease_duration,
    userRootNamespace: '',
  },
  radius: {
    authMethodType: 'radius',
    authMountPath: 'radius',
    displayName: RESPONSE_STUBS.radius.auth.metadata.username,
    entityId: RESPONSE_STUBS.radius.auth.entity_id,
    policies: RESPONSE_STUBS.radius.auth.policies,
    renewable: RESPONSE_STUBS.radius.auth.renewable,
    token: RESPONSE_STUBS.radius.auth.client_token,
    tokenExpirationEpoch: 1752352846180,
    ttl: RESPONSE_STUBS.radius.auth.lease_duration,
    userRootNamespace: '',
  },
  token: {
    authMethodType: 'token',
    authMountPath: '',
    displayName: 'token',
    entityId: RESPONSE_STUBS.token.data.entity_id,
    policies: RESPONSE_STUBS.token.data.policies,
    renewable: RESPONSE_STUBS.token.data.renewable,
    token: RESPONSE_STUBS.token.data.id,
    tokenExpirationEpoch: 1747413884837,
    ttl: RESPONSE_STUBS.token.data.ttl,
    userRootNamespace: '',
  },
  userpass: {
    authMethodType: 'userpass',
    authMountPath: 'userpass',
    displayName: RESPONSE_STUBS.userpass.auth.metadata.username,
    entityId: RESPONSE_STUBS.userpass.auth.entity_id,
    policies: RESPONSE_STUBS.userpass.auth.policies,
    renewable: RESPONSE_STUBS.userpass.auth.renewable,
    token: RESPONSE_STUBS.userpass.auth.client_token,
    tokenExpirationEpoch: 1752352843463,
    ttl: RESPONSE_STUBS.userpass.auth.lease_duration,
    userRootNamespace: '',
  },
  // ENTERPRISE ONLY
  saml: {
    authMethodType: 'saml',
    authMountPath: 'saml',
    displayName: RESPONSE_STUBS.saml['lookup-self'].data.display_name,
    entityId: RESPONSE_STUBS.saml['lookup-self'].data.entity_id,
    policies: RESPONSE_STUBS.saml['lookup-self'].data.policies,
    renewable: RESPONSE_STUBS.saml['lookup-self'].data.renewable,
    token: RESPONSE_STUBS.saml['lookup-self'].data.id,
    tokenExpirationEpoch: 1749587109246,
    ttl: RESPONSE_STUBS.saml['lookup-self'].data.ttl,
    userRootNamespace: '',
  },
};
