/* Copyright (C) 2014 J.F.Dockes
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef _DIRREADER_H_INCLUDED_
#define _DIRREADER_H_INCLUDED_

#include <string>
#include <vector>
#include <iostream>

#include <QThread>
#include <QString>

#include "libupnpp/log.h"
#include "libupnpp/control/cdirectory.hxx"

// Ensure that we build with older libupnpp versions which don't export the UPNP_E_XX error codes
#ifndef UPNP_E_SUCCESS
#define UPNP_E_SUCCESS       0
#endif
#ifndef UPNP_E_OUTOF_MEMORY
#define UPNP_E_OUTOF_MEMORY  -104
#endif
#ifndef UPNP_E_CANCELED
#define UPNP_E_CANCELED      -210
#endif

// A thread object for performing container browse or search operation(s).
class CDBrowseQO : public QThread {
    Q_OBJECT;

public:
    // We will perform a browse if there a no search strings, else search(es).
    CDBrowseQO(UPnPClient::CDSH server, std::string objid, std::vector<std::string> ss,
               QObject *parent = 0)
        : QThread(parent), m_serv(server), m_objid(objid), m_searchStrings(ss) {
        LOGDEB1("CDBrowseQO::init. objid " << m_objid << '\n');
    }

    ~CDBrowseQO() {
        LOGDEB1("CDBrowseQO::~. objid " << m_objid << '\n');
    }

    virtual void run() {
        LOGDEB("CDBrowseQO::run: " << (m_searchStrings.empty() ? "browse" : "search") <<
               " objid " << m_objid << '\n');
        if (m_searchStrings.empty()) {
            runBrowseOrSearch("");
        } else {
            for (const auto& ss : m_searchStrings) {
                if (m_cancel) {
                    LOGINF("reader: objid " << m_objid << " Cancelled\n");
                    break;
                }
                if (!ss.empty()) {
                    // ss.empty() should not happen...
                    runBrowseOrSearch(ss);
                }
            }
        }
        LOGDEB("Reader: objid " << m_objid << " emit done, status " << m_status << "\n");
        emit done(m_cancel ? UPNP_E_CANCELED : m_status);
    }

    void setCancel() {
        LOGDEB("Reader: objid " << m_objid << " set cancel\n");
        m_cancel = true;
    }
    const std::string& getObjid() {
        return m_objid;
    }
    const std::string& getModelName() {
        return m_serv->getModelName();
    }
    UPnPClient::ContentDirectory::ServiceKind getKind() {
        return m_serv->getKind();
    }

signals:
    void sliceAvailable(UPnPClient::UPnPDirContent *);
    void done(int);

private:
    UPnPClient::CDSH m_serv;
    std::string m_objid;
    std::vector<std::string> m_searchStrings;
    int m_status;
    bool m_cancel{false};


    virtual void runBrowseOrSearch(const std::string& searchstr) {
        LOGDEB("CDBrowseQO::runBOS: " <<
               (searchstr.empty() ? "browse" : std::string("search [") +  searchstr) << "]\n");
        int offset = 0;
        int toread = 100; // read small count the first time
        int total = 0;// Updated on first read.
        int count;

        while (total == 0 || (offset < total)) {
            if (m_cancel) {
                LOGINF("reader: objid " << m_objid << " Cancelled\n");
                break;
            }
            UPnPClient::UPnPDirContent *slice = new UPnPClient::UPnPDirContent();
            if (slice == 0) {
                m_status = UPNP_E_OUTOF_MEMORY;
                break;
            }
            LOGDEB("dirreader: reading "<< toread << " total " << total << "\n");
            if (searchstr.empty()) {
                m_status = m_serv->readDirSlice(m_objid, offset, toread, *slice,  &count, &total);
            } else {
                m_status = m_serv->searchSlice(
                    m_objid, searchstr, offset, toread, *slice,  &count, &total);
            }
            if (m_cancel) {
                LOGINF("reader: objid " << m_objid << " Cancelled\n");
                break;
            }
            if (m_status != UPNP_E_SUCCESS) {
                break;
            }
            offset += count;
            emit sliceAvailable(slice);
            if (count != toread || m_cancel)
                break;
            toread = m_serv->goodSliceSize();
        }
    }

};


// The content directory object is used to generate SystemUpdateId and
// ContainerUpdateIDs events in case we might care
class ContentDirectoryQO : public QObject, public UPnPClient::VarEventReporter {
    Q_OBJECT;

public:
    ContentDirectoryQO(UPnPClient::CDSH service, QObject *parent = 0)
        : QObject(parent), m_srv(service) {
        m_srv->installReporter(this);
    }
    virtual ~ContentDirectoryQO() {
        m_srv->installReporter(0);
    }

    // SystemUpdateId
    virtual void changed(const char *nm, int value) {
        LOGDEB1("CDQO: Changed: " << nm << " (int): " << value << "\n");
        if (!strcmp(nm, "SystemUpdateID")) {
            emit systemUpdateIDChanged(value);
        }
    }
    virtual void changed(const char *nm, const char *value) {
        LOGDEB1("CDQO: Changed: " << n << " (char*): " << v << "\n");
        if (!strcmp(nm, "ContainerUpdateIDs")) {
            emit containerUpdateIDsChanged(QString::fromUtf8(value));
        }
    }

    UPnPClient::CDSH srv() {return m_srv;}
    
signals:
    void systemUpdateIDChanged(int);
    void containerUpdateIDsChanged(QString);

private:
    UPnPClient::CDSH m_srv;
};

#endif /* _DIRREADER_H_INCLUDED_ */
