/* Copyright (C) 2011  Lucio Carreras
 *
 * This file is part of sayonara player
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef PLAYLIST_H_
#define PLAYLIST_H_

#include <iostream>
#include <vector>

#include <QObject>
#include <QList>
#include <QMap>
#include <QStringList>

#include "HelperStructs/globals.h"
#include "HelperStructs/MetaData.h"
#include "HelperStructs/CSettingsStorage.h"

namespace UPnPP {
struct ProtocolinfoEntry;
}

class Playlist : public QObject {
    Q_OBJECT

public:
    Playlist(QObject *parent = 0)
        : QObject(parent) {}
        
    virtual ~Playlist() = default;

    virtual unsigned int get_num_tracks() {
        return int(m_meta.size());
    }
    virtual void get_metadata(MetaDataList&);
    virtual int mode();
    virtual void update_state() {}
    virtual void getprotoinfo(std::vector<UPnPP::ProtocolinfoEntry>&);
    AudioState getTpState() {return m_tpstate;}
    virtual std::string getFriendlyName() { return "";}
    
signals:
    // New playlist (for displaying in ui_playlist)
    void sig_playlist_updated(MetaDataList&, int playidx, int/*unused*/);
    // Inform the ui_playlist about what row to highlight
    void sig_playing_track_changed(int row);
    // Current track info for displaying in the control area
    void sig_track_metadata(const MetaData&);
    // Make sure we have the latest state of the world
    void sig_sync();
    // Stopped at end of Playlist
    void sig_playlist_done();
    
    // These request action from the audio part
    void sig_stop();
    void sig_pause();
    void sig_resume_play();
    void sig_forward();
    void sig_backward();
    // Only connect in OHPlaylist mode
    void sig_mode_changed(int);

    // These report a state transition of the audio part
    void sig_stopped();
    void sig_paused();
    void sig_playing();

    // Done inserting tracks in the playlist
    void sig_insert_done();

    void connectionLost();
    void renewSubscriptions();

    // Only ever emitted by the openhome variant
    void playlistModeChanged(int);
                                           
public slots:
    // afteridx is the position after which we insert tracks. [-1,size-1]
    virtual void psl_insert_tracks(const MetaDataList&, int afteridx) = 0;
    virtual void psl_add_tracks(const MetaDataList&, bool nodups=false);
    virtual void psl_sort_by_tno();
    virtual void psl_load_playlist();
    virtual void psl_do_load_playlist(const QString& fn);
    virtual void psl_save_playlist();

    virtual void psl_change_track(int num) final;
    virtual void psl_change_track_impl(int) = 0;

    virtual void psl_seek(int secs) = 0;

    // Information from the remote end
    virtual void onRemoteTpState(AudioState, const char *) final;
    // Maybe also tell the subclass about these
    virtual void onRemoteTpState_impl(AudioState, const char *) {}
    virtual void onRemoteSecsInSong(quint32 s) final;
    virtual void onRemoteSecsInSong_impl(quint32) {}
    
    // Mode change requested by UI
    virtual void psl_change_mode(int);
    virtual void psl_clear_playlist();
    virtual void psl_clear_playlist_impl() = 0;
    virtual void psl_play() = 0;
    virtual void psl_pause() final;
    virtual void psl_pause_impl() = 0;
    virtual void psl_stop() = 0;
    virtual void psl_forward() = 0;
    virtual void psl_backward() = 0;
    virtual void psl_remove_rows_impl(const QList<int> &, bool is_drag_drop) = 0;

    // Note: in openhome mode, drag-dropping a selection containing
    // the currently playing track results in a track change because
    // this is seen as 2 separate remove/insert operations. After the
    // remove, the playing track has changed, and there is no way to
    // restore it when the insert occurs. We would need additional
    // input from the playlist ui which is the only code with the info
    // to link the remove/insert ops. Even then, it would be
    // complicated to restore the right playing track because we'd
    // have to wait for the playlist update from the remote player to
    // do it. Even them this would cause the sound to flicker. It
    // seems that the only option is to acknowledge the issue. We now stop
    // playing in this situation (when removing the currently playing track)
    virtual void psl_remove_rows(const QList<int> &, bool is_drag_drop);

    virtual void psl_selected_list(const QList<int>&);

protected:

    MetaDataList m_meta;
    int m_play_idx{-1};
    AudioState m_tpstate{AUDIO_UNKNOWN};
    
    virtual void remove_row(int row);
    virtual bool valid_row(int row) {
        return row >= 0 && row < int(m_meta.size());
    }
    int get_selection_min_row() {return m_selection_min_row;}
private:
    // m_insertion_point is used when not in append mode. In this case, the first insertion
    // (m_insertion_point==-1) occurs after the playing track, and m_insertion_point is set after
    // the inserted track(s) so that further insertion will occur in order, and not in reverse
    // order, each new track after the playing one. It is reset when changing the mode, clearing the
    // playlist or when play_idx changes.
    int m_insertion_point{-1};
    int m_selection_min_row{-1};
};

#endif /* PLAYLIST_H_ */
