/* Copyright (C) 2014-2025 J.F.Dockes 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef _CDBROWSER_H_INCLUDED_
#define _CDBROWSER_H_INCLUDED_

/** CDBrowser displays Content Directory data inside a DirBrowser Tab */
#include <time.h>

#include <vector>
#include <iostream>
#include <unordered_set>
#include <unordered_map>

#ifdef USING_WEBENGINE
#include <QWebEngineView>
#include <QWebChannel>
#define QWEBVIEW QWebEngineView
#define QWEBPAGE QWebEnginePage
#define QWEBSETTINGS QWebEngineSettings
class PopupHelper: public QObject{
    Q_OBJECT
public:
    using QObject::QObject;
Q_SIGNALS:
    void triggered();
public Q_SLOTS:
    void onTriggered(){
        Q_EMIT triggered();
    }
};
#else
#include <QWebView>
#define QWEBVIEW QWebView
#define QWEBPAGE QWebPage
#define QWEBSETTINGS QWebSettings
#endif

#include <QVariant>
#include <QTimer>
#include <QPoint>
#include <QMouseEvent>

#include "libupnpp/control/description.hxx"
#include "libupnpp/control/mediaserver.hxx"
#include "libupnpp/control/cdirectory.hxx"
#include "libupnpp/control/cdircontent.hxx"

#include "HelperStructs/MetaData.h"

#include "randplayer.h"

class CDBrowseQO;
class RecursiveReaper;
class DirBrowser;
class QProgressDialog;
class ContentDirectoryQO;
class QGestureEvent;

class CDBrowser : public QWidget
{
    Q_OBJECT;

public:
    CDBrowser(QWidget *parent, DirBrowser* tabs = 0);
    virtual ~CDBrowser();
    void setDirBrowser(DirBrowser *db) {m_browsers = db;}
    void getSearchCaps(std::set<std::string>& caps) {caps = m_searchcaps;}

    // Entry for path inside current server: for going back later
    struct CtPathElt {
        CtPathElt() {}
        CtPathElt(const std::string& id, const std::string& tt, bool ispl,
                  const UPnPClient::UPnPDirObject& e = UPnPClient::UPnPDirObject(),
                  const std::vector<std::string>& ss = std::vector<std::string>()) 
            : objid(id), title(tt), isPlaylist(ispl), searchStr(ss), dirent(e) {}
        std::string objid;
        std::string title;
        bool isPlaylist;
        std::vector<std::string> searchStr;
        QPoint scrollpos{0,0};
        UPnPClient::UPnPDirObject dirent;
    };

    // Browse specified container in newly created tab.
    // We have no server link yet.
    void browseInNewTab(QString UDN, std::vector<CtPathElt> path);

    void findText(const QString&, bool reverse);

    std::string getServerUDN();
    const std::vector<CtPathElt>& getcurpath() {
        return m_curpath;
    }
    QPoint scrollpos();
    
    bool eventFilter(QObject *object, QEvent *event) override;
    void setElidePath(bool onoff);
                                 
    static const std::string plContainerClass;
    static const std::string albContainerClass;

public slots:
    void search(const std::vector<std::string>& iss, const std::string& objid = "");
    virtual void initialPage(bool redisplay = false);
    void onBrowseDone();
    void onSliceAvailable(UPnPClient::UPnPDirContent *);
    void onReaperSliceAvailable(UPnPClient::UPnPDirContent *);
    void setStyleSheet();
    void refresh();
    void onSysUpdIdChanged(int id);
    virtual void appendHtml(const QString&, const QString& html);
    virtual void onLinkClicked(const QUrl &);
    virtual void createPopupMenu(const QPoint& = QPoint());
    virtual bool popupOther(QAction *act);
    virtual void itemActions(QAction *);
    virtual void containerActions(QAction *);
    virtual void back(QAction *);
    virtual void saveAsCSV();
    virtual void rreaperDone(int);
    virtual void onContentsSizeChanged(const QSize&);
    virtual void takeFocus();
#ifdef USING_WEBENGINE
    virtual void onPopupJsDone(const QVariant&);
    virtual void onLoadFinished(bool);
    virtual void startScript();
#if (QT_VERSION >= QT_VERSION_CHECK(6, 2, 0))
    virtual void onNewWindowRequested(QWebEngineNewWindowRequest &request);
#endif
#endif
    virtual void onToggleMode();
    
signals:
    void sig_script_ran();
    void sig_tracks_to_playlist(const MetaDataList&);
    void sig_tracks_to_randplay(RandPlayer::PlayMode, const std::vector<UPnPClient::UPnPDirObject>&);
    void sig_now_in(QWidget *, const QString&);
    void sig_searchcaps_changed();
    void sig_browse_in_new_tab(QString UDN, std::vector<CDBrowser::CtPathElt>);
    void sig_rand_stop();
    void sig_sort_order();
    void sig_toggle_mode();
    
protected:
    bool event(QEvent *) override;

private:
    // This is updated before displaying a page and used to avoid
    // repeated calls to QSettings by the display routines.
    struct FormatParams {
        // Display some artist info after the album title
        bool artistWithAlbum;
        // Max length of artist info
        int artistWithAlbumBytes;
        // Show cover thumbnail before the album title
        bool coverWithAlbum;
        // Thumbnail pixel width
        int coverWithAlbumWidth;
        // Albums as sets of cover images not lists
        bool albumsAsCovers;
        int albumsAsCoversWidth;
        QString genericAlbumImageURI;
        float scalemultiplier;
    };

    std::string imgtag(const std::string& imgurl, int width, const std::string& tt="");
    virtual void processOnLinkClicked(const QUrl &);
    void initContainerHtml(const std::vector<std::string>& ss = std::vector<std::string>());
    QString bodyTopHtml();
    void browseContainer(std::string ctid, std::string title, bool ispl,
                         const UPnPClient::UPnPDirObject& e = UPnPClient::UPnPDirObject(),
                         QPoint scrollpos = QPoint());
    void search(const std::string& objid, const std::vector<std::string>& iss, QPoint scrollpos);
    void curpathClicked(unsigned int i);
    void waitForPage();
    void mySetHtml(const QString&);
    void doCreatePopupMenu();
    void runJS(const QString&, QVariant* vop = nullptr);
    bool updateAlphamap(char& curinitial, const std::string& tt);
    void maybeShowAlphamap(unsigned int nct);
    QString alphalinks(const std::string& initials, bool nolinks=false);
    void alphaMessage(const std::string& msg);
    int serverFromUDN(const std::string& udn);
    void popupaddaction(QMenu *, const QString& txt, int val);
    void onSliceAvailableCovers(UPnPClient::UPnPDirContent *);
    
    void init_HTML();
    QString CTToHtml(int idx, const UPnPClient::UPnPDirObject& e, const std::string& alphid = "");
    enum ItemAlbumDisplay {IAD_NONE, IAD_REG, IAD_LINK};
    QString ItemToHtml(int idx, const UPnPClient::UPnPDirObject&, ItemAlbumDisplay albdisp=IAD_REG);
    static QString DSToHtml(int idx, const UPnPClient::UPnPDeviceDesc& dev);
    static QString initialServersPage();
    static QString emptyServersPage();
    void deleteReaders(const QString& from);
    bool newCds(int cdsidx);
    void updateFormatParams();
    bool gestureEvent(QGestureEvent *event);
    void popupSearch();

    QWEBVIEW *m_view{nullptr};
    // This is used to limit our automatic descent further down when a
    // container has a unique container entry. To defend against some
    // pathological trees
    int m_autoclickcnt{0};
    // When displaying the servers list, we periodically check the
    // server pool state. The last seen Media Server descriptions are
    // stored in m_msdescs for diffing with the new ones and deciding
    // if we need to redraw. Timer and servers list are only used
    // while we are displaying the servers page, the timer is stopped
    // as soon as a link is clicked.
    QTimer m_timer;
    std::vector<UPnPClient::UPnPDeviceDesc> m_msdescs;
    // Handle for the currently active media server
    std::shared_ptr<ContentDirectoryQO>  m_cds;
    // Search caps of current server
    std::set<std::string> m_searchcaps;
    std::vector<CtPathElt> m_curpath;
    FormatParams m_formatparams;
    std::map<char, std::string> m_alphamap;
    // Threaded objects to perform directory reads and recursive walks
    CDBrowseQO *m_reader{0};
    RecursiveReaper    *m_reaper{0};
    // Busy dialog for lengthy ops
    QProgressDialog *m_progressD{0};
    time_t           m_progstart;
    // Content of the currently visited container or search
    std::vector<UPnPClient::UPnPDirObject> m_entries;
    // Scroll position to be restored when we're done reading. This is
    // so that the user can scroll while we insert?
    QPoint m_savedscrollpos;
    // Recursive explore contents, for possible sorting before sending to pl
    std::vector<UPnPClient::UPnPDirObject> m_recwalkentries;
    // URL hashes for deduplication while walking the tree
    std::unordered_set<std::string> m_recwalkdedup;
    // Pointer to parent tabbed object for access to shared state (insertActive)
    DirBrowser *m_browsers{0};
    // Attributes for the popup menu click:
    std::string m_popupobjid;
    std::string m_popupobjtitle;
    std::string m_popupotype;
    int m_popupidx{-1};
    bool m_popupispl{false};
    bool m_popupispath{false};
    QPoint m_popupos;
    int m_popupmode; // now, next, at end
    bool m_pageloaded{false};
#ifdef USING_WEBENGINE
    // No use trying to insert stuff before the initial page insert ran
    bool m_jsbusy{false};
    // Have to start the js execution through a timer and slot. Need 2 members to store the data?
    // Maybe could be avoided with a lambda. TBchecked
    QString m_js;
    QVariant *m_vop;
    QWebChannel m_channel;
    PopupHelper m_puhelper;
#endif
    // Remember last click kind for detecting midclick
    Qt::MouseButton m_lastbutton{Qt::LeftButton};
    // State for init browsing in subdir instead of servers page (middle-click)
    QString m_initUDN;
    // SystemUpdateID for the connected content directory server
    int m_sysUpdId{0};
    bool m_forceRedisplay{false};
    bool m_elidePath{false};
    int m_maxartlen{0};
    int m_autotravmaxdepth{2};
    // Used while receiving cdir slices: add dots to show progress.
    std::string m_alphamsg;
    // Used while receiving slices to store the HTML after the first xxx (500) entries: keeping
    // updating the browser is too slow.
    QString m_residhtml;
    QUrl m_clickedurl;
    QWidget *m_eventFilterWidget;
};

#endif // CDBROWSER_INCLUDED
