/* Copyright (C) 2011  Lucio Carreras
 * Copyright (C) 2017  J.F. Dockes
 *
 * This file is part of Upplay
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include "HelperStructs/Helper.h"

#include <QDir>
#include <QString>
#include <QThread>

#include "HelperStructs/MetaData.h"
#include "HelperStructs/Style.h"

#include "upadapt/upputils.h"
#include "utils/pathut.h"

bool Helper::read_file_into_str(const QString& filename, QString& content)
{
    QFile file(filename);
    content.clear();
    if (!file.open(QIODevice::ReadOnly)) {
        return false;
    }
    const int BUFSZ{4*1024};
    char buffer[BUFSZ];
    while (!file.atEnd()) {
        auto cnt = file.read(buffer, BUFSZ);
        if (cnt <= 0)
            break;
        content.append(QString::fromUtf8(buffer, cnt));
    }
    file.close();
    if (content.size() > 0) {
        return true;
    }
    return false;
}

QString Helper::file_to_str(const QString& filename)
{
    QString temp;
    read_file_into_str(filename, temp);
    return temp;
}

static QString cvtNum2String(int num, int digits)
{
    QString str = QString::number(num);
    while (str.size() < digits) {
        str.prepend("0");
    }

    return str;
}

QString Helper::cvtMsecs2TitleLengthString(long int msec, bool colon,
                                           bool show_days)
{
    QString sign;
    if (msec < 0) {
        msec = -msec;
        sign = "-";
    }
        
    bool show_hrs = false;

    int sec = msec / 1000;
    int min = sec / 60;

    int secs = sec % 60;
    int hrs = min / 60;
    int days = hrs / 24;

    QString final_str;

    if (days > 0 && show_days) {
        final_str += QString::number(days) + "d ";
        hrs = hrs % 24;
        show_hrs = true;
    }

    if (!show_days) {
        hrs += (days * 24);
    }

    if (hrs > 0 || show_hrs) {
        final_str += QString::number(hrs) + "h ";
        min = min % 60;
    }

    if (colon) {
        final_str +=  cvtNum2String(min, 2) + ":" + cvtNum2String(secs, 2);
    } else {
        final_str +=  cvtNum2String(min, 2) + "m " + cvtNum2String(secs, 2);
    }

    return sign + final_str;
}

QString Helper::getSharePath()
{
    return u8s2qs(path_pkgdatadir("upplay", "UPPLAY_DATADIR",
#ifdef PREFIX
                                  PREFIX "/share/upplay"
#else
                                  ""
#endif
                      ));
}

QString Helper::getHomeDataPath()
{
    QDir homedir(QDir::home());

#ifndef Q_OS_WIN
    QString rpath = QString::fromUtf8(".local/share/upplay");
#else
    QString rpath = QString::fromUtf8("AppData/Local/upplay/");
#endif

    homedir.mkpath(rpath);
    return homedir.absoluteFilePath(rpath);
}

QString Helper::getDataFilePath(QString basename)
{
    QString udir = Helper::getHomeDataPath();
    QDir dir(udir);
    auto fn = dir.filePath(basename);
    if (QFile::exists(fn))
        return fn;

    QString sysdir = Helper::getSharePath();

    dir = QDir(sysdir);
    fn = dir.filePath(basename);
    return fn;
}

QString Helper::getPlayerHtmlFrag()
{
    static const char *confname = "/playerhtml.conf";
    QString qhtml;
    QString hdir = Helper::getHomeDataPath();
    QString confpath = hdir + confname;
    if (QFile::exists(confpath)) {
        Helper::read_file_into_str(confpath, qhtml);
    } else {
        QString sdir = Helper::getSharePath();
        confpath = sdir + confname;
        Helper::read_file_into_str(confpath, qhtml);
    }
    return qhtml;
}

static QString iconSubDir;
void Helper::setIconStyle(const QString& subd)
{
    iconSubDir = subd;
}

// We now use the qt resources instead of installed icons
QString Helper::getIconPath(const QString& icnm)
{
    QString path(":/icons/");
    if (!iconSubDir.isEmpty()) {
        QString pathdark = path + iconSubDir + "/" + icnm;
        QDir dir(":/");
        if (dir.exists(pathdark)) {
            return pathdark;
        }
    }
    return path + icnm;
}


#define DARK_BLUE(x) QString("<font color=#0000FF>") + x + QString("</font>")
#define LIGHT_BLUE(x) QString("<font color=#8888FF>") + x + QString("</font>")

QString Helper::createLink(QString name, QString target, bool underline)
{

    int dark = Style::use_dark_colors();
    if (target.size() == 0) {
        target = name;
    }

    QString content;
    QString style = "";

    if (!underline) {
        style = " style: \"text-decoration=none;\" ";
    }

    if (dark) {
        content = LIGHT_BLUE(name);
    } else {
        content = DARK_BLUE(name);
    }

    return QString("<a href=\"") + target + "\"" + style + ">" + content + "</a>";
}

void mdlSetCurPlayTrack(MetaDataList& lst, int idx)
{
    for (int i = 0; i < int(lst.size()); i++) {
        lst[i].pl_playing = false;
    }
    if (idx >= 0 && idx < int(lst.size()))
        lst[idx].pl_playing = true;
}

bool mdlContains(MetaDataList& lst, const MetaData& md, bool cs, MetaData **found)
{
    QString url = cs ? md.url.trimmed() : md.url.toLower().trimmed();
    for(int i = 0; i < int(lst.size()); i++) {
        QString url2 = cs ? lst[i].url.trimmed() : lst[i].url.toLower().trimmed();
        if(!url.compare(url2)) {
            if (found) 
                *found = &(lst[i]);
            return true;
        }
    }
    return false;
}

bool mdlSerialize(MetaDataList& lst, const QString& fn)
{
    QFile file(fn);
    file.open(QIODevice::WriteOnly);
    QDataStream out(&file);
    for (int i = 0; i < int(lst.size()); i++) {
        lst[i].serialize(out);
    }
    return true;
}

bool mdlUnSerialize(MetaDataList& lst, const QString& fn)
{
    QFile file(fn);
    file.open(QIODevice::ReadOnly);
    QDataStream in(&file);
    MetaData md;
    while (md.unSerialize(in)) {
        lst.push_back(md);
    }
    return true;
}
