/* Copyright (C) 2015-2025 J.F.Dockes
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "mdatawidget.h"

#include <iostream>
#include <set>

#include <QDebug>
#include <QLayout>
#include <QSettings>
#include <QTimer>
#include <QPoint>
#include <QAction>
#include <QMenu>
#ifdef USING_WEBENGINE
#include <QWebEnginePage>
#include <QWebEngineSettings>
#include <QWebEngineProfile>
#include <QtWebEngineWidgets>
#include <QLoggingCategory>
#else
#include <QWebFrame>
#include <QWebSettings>
#include <QWebElement>
#include <QNetworkDiskCache>
#endif

#include <libupnpp/log.h>
#include <libupnpp/control/cdircontent.hxx>

#include "HelperStructs/MetaData.h"
#include "HelperStructs/Helper.h"
#include "HelperStructs/Style.h"
#include "upadapt/upputils.h"
#include "GUI/widgets/tooltiplike.h"
#include "GUI/widgets/cdwebpage.h"
#include "utils/smallut.h"

static QString o_scaledcss;

// What follows (htmltemplate1+(js)+htmltemplate2) is the HTML template for the whole player display
// area: cover on the left, metadata on the right. (js) is the webchannel stuff only there for
// webengine.
static const QString htmltemplate1(R"(
<html>
  <head>
    <meta http-equiv='content-type' content='text/html; charset=utf-8'>
    <style>
    %4
    </style>
    <script type='text/javascript'>
)");

// The Qt qwebchannel.js code will be inserted here if we are using webengine

static const QString htmltemplate2(
"    let hoverTimeout;\n"
"    window.onload = function() {\n"
#ifdef USING_WEBENGINE
"        // Create the webchannel and store it in global variable\n"
"        window.webChannel = new QWebChannel(qt.webChannelTransport, function(channel) {});\n"
#endif
"        // Store a ref to our image\n"
"        var img = document.getElementById('coverArtImg');\n"
"        function showTooltip() {\n"
"            // Trigger the c++\n"
#ifdef USING_WEBENGINE
"            window.webChannel.objects.qjshelper.onCoverHover();\n"
#else
"            qjshelper.onCoverHover();\n"
#endif
"        }\n"
"        // When entering the image, start a timeout to call the trigger\n"
"        img.addEventListener('mouseenter', function() {\n"
"            hoverTimeout = setTimeout(showTooltip, 1000);\n"
"        });\n"
"        // When exiting the image, cancel the timeout. Short stays won't get a tooltip\n"
"        img.addEventListener('mouseleave', function() {\n"
"            clearTimeout(hoverTimeout);\n"
"        });\n"
"    };\n"
"    </script>\n"
"  </head>\n"
"  <body>\n"
"    <img align='left' id='coverArtImg'\n"
"      style='margin-left: 2px;margin-top: 3px;object-fit: scale-down; height: %5; width: %2'\n"
"      src='%1' />\n"
"    <div style='display: flex; justify-content: center; align-items: center;\n"
"                vertical-align: middle;height: %5px;'> \n"
"      <div class='playermeta'>\n"
"        %3\n"
"      </div>\n"
"    </div>\n"
"  </body>\n"
);

static QString htmltemplate;
static std::string playerhtmlfrag;

MDataWidget::MDataWidget(QWidget *parent)
    : QWidget(parent)
{
    m_view =  new QWEBVIEW(this);

    QVBoxLayout *verticalLayout = new QVBoxLayout(this);
    verticalLayout->setSpacing(0);
    verticalLayout->setContentsMargins(0, 0, 0, 0);
    verticalLayout->setObjectName(QString::fromUtf8("mdatavertLayout_1"));
    verticalLayout->addWidget(m_view);
    m_imgurl = QString("qrc:/icons/generic-cover.jpg");

    
    m_view->settings()->setAttribute(QWEBSETTINGS::JavascriptEnabled, true);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(m_view, SIGNAL(customContextMenuRequested(const QPoint&)),
            this, SLOT(createPopupMenu(const QPoint&)));

    htmltemplate = htmltemplate1;
    o_scaledcss = Style::getCSSData();
    
#ifdef USING_WEBENGINE
    auto profile = new QWebEngineProfile("upplay", this);
    profile->setHttpUserAgent("Upplay Control Point");
    auto page = new CDWebPage(profile, this);
#else
    auto page = new CDWebPage(this);
#endif
    m_view->setPage(page);
    
#ifdef USING_WEBENGINE
    m_channel.registerObject("qjshelper", &m_jslink);
    m_view->page()->setWebChannel(&m_channel);
    QString wcfn = Style::getCSSPath("qwebchannel.js");
    htmltemplate += Helper::file_to_str(wcfn);
#else
    auto frame = m_view->page()->mainFrame();
    connect(frame, SIGNAL(javaScriptWindowObjectCleared()), this, SLOT(connectJSHelper()));
#endif
    connect(&m_jslink, SIGNAL(coverHover()), this, SLOT(coverZoom()));

    htmltemplate += htmltemplate2;

    if (playerhtmlfrag.empty()) {
        playerhtmlfrag = qs2utf8s(Helper::getPlayerHtmlFrag());
        if (playerhtmlfrag.empty()) {
            playerhtmlfrag =
                "<p class='playertitle'>%(dc:title)</p>\n"
                "<p class='playerartist'>%(upnp:artist)</p>\n"
                "<p class='playeralbum'>%(upnp:album)</p>\n";
        }
    }
}

#ifndef USING_WEBENGINE
void MDataWidget::connectJSHelper()
{
    auto frame = m_view->page()->mainFrame();
    frame->addToJavaScriptWindowObject("qjshelper", &m_jslink, QWebFrame::QtOwnership);
}
#endif

// Function object for fetching the field value needed by pcSubst() for the player metadata base
// display
class FieldFetcher {
public:
    FieldFetcher(const MetaData& m)
        : m_md(m) {
        UPnPClient::UPnPDirContent dir;
        if (!m_md.didl.isEmpty() && dir.parse(qs2utf8s(m_md.didl)) && !dir.m_items.empty()) {
            okdidl = true;
            m_dirent = dir.m_items[0];
        }
    }

    const MetaData& m_md;
    UPnPClient::UPnPDirObject m_dirent;
    bool okdidl{false};
    std::set<std::string> m_values;
    std::string operator()(const std::string& key) {
        if (!okdidl) {
            // use MetaData should we?
            LOGDEB1("mdatawidget: NO DIDL\n");
            return "";
        } 
        std::string value;
        if (key == "upnp:artist") {
            value = m_dirent.getArtists();
        } else if (key == "dc:title") {
            value = m_dirent.m_title;
        } else {
            value = m_dirent.getprop(key);
        }
        // Only return something if we did not already see the value. Avoids e.g. upnp:artist
        // dc:creator dups.
        if (m_values.find(value) == m_values.end()) {
            m_values.insert(value);
            return value;
        }
        return "";
    }
};

const static QUrl baseUrl("file:///");
void MDataWidget::display()
{
    LOGDEB1("MDataWidget::display()\n");
    // Sometimes ignore the date
    QString albtxt;
    if (m_data.year < 1000 || m_data.album.contains(QString::number(m_data.year))) {
        albtxt = m_data.album.trimmed();
    } else {
        albtxt = m_data.album.trimmed() + " (" + QString::number(m_data.year) + ")";
    }

    int w = width() / 2;
    int h = height() - 6;
    if (w > h)
        w = h;

    std::string hfrag;
    FieldFetcher f(m_data);
    pcSubst(playerhtmlfrag, hfrag, f);
    LOGDEB1("mdatawidget: HTML FRAG [" << hfrag << "]\n");
    m_view->setHtml(htmltemplate
                    /*1*/.arg(m_imgurl)
                    /*2*/.arg(w)
                    /*3*/.arg(u8s2qs(hfrag))
                    /*4*/.arg(o_scaledcss)
                    /*5*/.arg(h)
                    ,
                    baseUrl);
}

void MDataWidget::refresh()
{
    LOGDEB1("MDataWidget::refresh()\n");
    o_scaledcss = Style::getCSSData();
    display();
}

enum popup_values{PUP_SHOWDETAILS};
void MDataWidget::createPopupMenu(const QPoint& pos)
{
    QMenu *popup = new QMenu(this);

    QAction *act = new QAction(tr("UPnP item details"), this);
    act->setData(PUP_SHOWDETAILS);
    popup->addAction(act);
    popup->connect(popup, SIGNAL(triggered(QAction *)), this, SLOT(onPopupTriggered(QAction *)));
    popup->popup(mapToGlobal(pos));
}

void MDataWidget::onPopupTriggered(QAction *act)
{
    auto what = act->data().toInt();
    switch (what) {
    case PUP_SHOWDETAILS:
    {
        QString head("<html><head></head><body>\n");
        if (Style::use_dark_colors()) {
            // The tooltip background stays light (because of the Qt::tooltip flag probably). But
            // the font somehow inherits the general style and becomes clear. Have to force it to
            // black to make things readable
            head += "<div style='color:black;'>";
        } else {
            head += "<div>";
        }
        QString tail("</div></body></html>\n");
        auto html = head + metaDataToHtml(&m_data) + tail;
        auto tooltipWindow = new TooltipWindow(html, this);
        tooltipWindow->move(QCursor::pos() + QPoint(-50, -50));
        tooltipWindow->show();
    }
    break;
    default:
        return;
    }
}

void MDataWidget::setData(const MetaData& md)
{
    LOGDEB0("MDataWidget::setData(). Title: " << qs2utf8s(md.title) <<
           " img " << qs2utf8s(md.albumArtURI) << '\n');
    m_data = md;
    display();
}

void MDataWidget::setCoverImage(QString url)
{
    if (url[0] == ':') // Coming from getIconPath(). HTML needs schema prefix
        m_imgurl = QString("qrc") + url;
    else
        m_imgurl = url;
    LOGDEB0("MDataWidget::setCoverImage: URL[" << qs2utf8s(m_imgurl) << "]\n");
    display();
}

void MDataWidget::resizeEvent(QResizeEvent *event)
{
    if (nullptr == m_timer) {
        m_timer = new QTimer(this);
        m_timer->setSingleShot(true);
        connect(m_timer, SIGNAL(timeout()), this, SLOT(display()));
    }
    m_timer->start(100);
    QWidget::resizeEvent(event);
}

void MDataWidget::coverZoom()
{
    LOGDEB0("coverZoom: url: " << qs2utf8s(m_imgurl) << "\n");
    
    if (m_imgurl == "qrc:/icons/logo.png") {
        return;
    }
    QString htmlfrag("<img src='");
    htmlfrag += m_imgurl + "' ";
    htmlfrag += "style='margin:0;object-fit: cover;'";
    
    int maxh = QSettings().value("maxcoverpopupsize").toInt();
    if (maxh == 0) {
        // Preference not set. Default:
        maxh = 600;
    }
    if (maxh) {
        htmlfrag += " height='" + QString::number(maxh) + "'";
    }
    htmlfrag += ">";
    
    auto tooltipWindow = new TooltipWindow(htmlfrag, this);
    tooltipWindow->move(mapToGlobal(pos()));
    tooltipWindow->show();
}
