#ifndef __UTIL_H__
#define __UTIL_H__

#include <cstdint>
#include <memory>
#include <string>
#include <vector>

#include "stable-diffusion.h"

#define SAFE_STR(s) ((s) ? (s) : "")
#define BOOL_STR(b) ((b) ? "true" : "false")

bool ends_with(const std::string& str, const std::string& ending);
bool starts_with(const std::string& str, const std::string& start);
bool contains(const std::string& str, const std::string& substr);

std::string sd_format(const char* fmt, ...);

void replace_all_chars(std::string& str, char target, char replacement);

int round_up_to(int value, int base);

bool file_exists(const std::string& filename);
bool is_directory(const std::string& path);

std::u32string utf8_to_utf32(const std::string& utf8_str);
std::string utf32_to_utf8(const std::u32string& utf32_str);
std::u32string unicode_value_to_utf32(int unicode_value);
// std::string sd_basename(const std::string& path);

typedef struct {
    uint32_t width;
    uint32_t height;
    uint32_t channel;
    float* data;
} sd_image_f32_t;

void normalize_sd_image_f32_t(sd_image_f32_t image, float means[3], float stds[3]);

sd_image_f32_t sd_image_t_to_sd_image_f32_t(sd_image_t image);

sd_image_f32_t resize_sd_image_f32_t(sd_image_f32_t image, int target_width, int target_height);

sd_image_f32_t clip_preprocess(sd_image_f32_t image, int target_width, int target_height);

class MmapWrapper {
public:
    static std::unique_ptr<MmapWrapper> create(const std::string& filename);

    virtual ~MmapWrapper() = default;

    MmapWrapper(const MmapWrapper&)            = delete;
    MmapWrapper& operator=(const MmapWrapper&) = delete;
    MmapWrapper(MmapWrapper&&)                 = delete;
    MmapWrapper& operator=(MmapWrapper&&)      = delete;

    const uint8_t* data() const { return static_cast<uint8_t*>(data_); }
    size_t size() const { return size_; }
    bool copy_data(void* buf, size_t n, size_t offset) const;

protected:
    MmapWrapper(void* data, size_t size)
        : data_(data), size_(size) {}
    void* data_  = nullptr;
    size_t size_ = 0;
};

std::string path_join(const std::string& p1, const std::string& p2);
std::vector<std::string> split_string(const std::string& str, char delimiter);
void pretty_progress(int step, int steps, float time);

void log_printf(sd_log_level_t level, const char* file, int line, const char* format, ...);

std::string trim(const std::string& s);

std::vector<std::pair<std::string, float>> parse_prompt_attention(const std::string& text);

sd_progress_cb_t sd_get_progress_callback();
void* sd_get_progress_callback_data();

sd_preview_cb_t sd_get_preview_callback();
void* sd_get_preview_callback_data();
preview_t sd_get_preview_mode();
int sd_get_preview_interval();
bool sd_should_preview_denoised();
bool sd_should_preview_noisy();

#define LOG_DEBUG(format, ...) log_printf(SD_LOG_DEBUG, __FILE__, __LINE__, format, ##__VA_ARGS__)
#define LOG_INFO(format, ...) log_printf(SD_LOG_INFO, __FILE__, __LINE__, format, ##__VA_ARGS__)
#define LOG_WARN(format, ...) log_printf(SD_LOG_WARN, __FILE__, __LINE__, format, ##__VA_ARGS__)
#define LOG_ERROR(format, ...) log_printf(SD_LOG_ERROR, __FILE__, __LINE__, format, ##__VA_ARGS__)
#endif  // __UTIL_H__
