//
// Copyright 2021 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//

import XCTest

@testable import LibSignalClient

class CryptoTests: TestCaseBase {
    func generateAesKey() -> [UInt8] {
        var key = Array(repeating: UInt8(0), count: 32)
        key.withUnsafeMutableBytes {
            try! fillRandom($0)
        }
        return key
    }

    func testAesGcmSiv() {
        let ptext = Data([0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00])
        let expected_ctext = Data([
            0x1D, 0xE2, 0x29, 0x67, 0x23, 0x7A, 0x81, 0x32, 0x91, 0x21, 0x3F, 0x26, 0x7E, 0x3B, 0x45, 0x2F, 0x02, 0xD0,
            0x1A, 0xE3, 0x3E, 0x4E, 0xC8, 0x54,
        ])
        let ad: [UInt8] = [0x01]
        let key: [UInt8] = [
            0x01,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
        ]
        let nonce: [UInt8] = [0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00]

        let gcm_siv = try! Aes256GcmSiv(key: key)

        let ctext = try! gcm_siv.encrypt(ptext, nonce: nonce, associatedData: ad)
        XCTAssertEqual(ctext, expected_ctext)

        let recovered = try! gcm_siv.decrypt(ctext, nonce: nonce, associatedData: ad)
        XCTAssertEqual(recovered, ptext)

        XCTAssertThrowsError(try gcm_siv.decrypt(ptext, nonce: nonce, associatedData: ad))
        XCTAssertThrowsError(try gcm_siv.decrypt(ctext, nonce: ad, associatedData: nonce))
    }

    func testAesGcm() {
        let plainTextData = Data("Super🔥secret🔥test🔥data🏁🏁".utf8)
        XCTAssertEqual(39, plainTextData.count)

        let key = self.generateAesKey()
        let encryptedParts = try! Aes256GcmEncryptedData.encrypt(plainTextData, key: key)
        let encryptedData = encryptedParts.concatenate()
        XCTAssertEqual(
            Aes256GcmEncryptedData.nonceLength + plainTextData.count + Aes256GcmEncryptedData.authenticationTagLength,
            encryptedData.count
        )

        let splitParts = try! Aes256GcmEncryptedData(concatenated: encryptedData)
        XCTAssertEqual(encryptedParts.nonce, splitParts.nonce)
        XCTAssertEqual(encryptedParts.ciphertext, splitParts.ciphertext)
        XCTAssertEqual(encryptedParts.authenticationTag, splitParts.authenticationTag)

        let decryptedData = try! splitParts.decrypt(key: key)
        XCTAssertEqual(39, decryptedData.count)
        XCTAssertEqual(plainTextData, decryptedData)

        var encryptedWithBadTag = encryptedData
        encryptedWithBadTag[encryptedWithBadTag.count - 1] ^= 0xFF
        XCTAssertThrowsError(try Aes256GcmEncryptedData(concatenated: encryptedWithBadTag).decrypt(key: key)) {
            guard case SignalError.invalidMessage(_) = $0 else {
                XCTFail("wrong error: \($0)")
                return
            }
        }
    }

    func testAesGcmKat() {
        let key: [UInt8] = [
            0xFE, 0xFF, 0xE9, 0x92, 0x86, 0x65, 0x73, 0x1C, 0x6D, 0x6A, 0x8F, 0x94, 0x67, 0x30, 0x83, 0x08, 0xFE, 0xFF,
            0xE9, 0x92, 0x86, 0x65, 0x73, 0x1C, 0x6D, 0x6A, 0x8F, 0x94, 0x67, 0x30, 0x83, 0x08,
        ]
        let plaintext = Data([
            0xD9, 0x31, 0x32, 0x25, 0xF8, 0x84, 0x06, 0xE5, 0xA5, 0x59, 0x09, 0xC5, 0xAF, 0xF5, 0x26, 0x9A, 0x86, 0xA7,
            0xA9, 0x53, 0x15, 0x34, 0xF7, 0xDA, 0x2E, 0x4C, 0x30, 0x3D, 0x8A, 0x31, 0x8A, 0x72, 0x1C, 0x3C, 0x0C, 0x95,
            0x95, 0x68, 0x09, 0x53, 0x2F, 0xCF, 0x0E, 0x24, 0x49, 0xA6, 0xB5, 0x25, 0xB1, 0x6A, 0xED, 0xF5, 0xAA, 0x0D,
            0xE6, 0x57, 0xBA, 0x63, 0x7B, 0x39,
        ])
        let expectedCiphertext = Data([
            0x52, 0x2D, 0xC1, 0xF0, 0x99, 0x56, 0x7D, 0x07, 0xF4, 0x7F, 0x37, 0xA3, 0x2A, 0x84, 0x42, 0x7D, 0x64, 0x3A,
            0x8C, 0xDC, 0xBF, 0xE5, 0xC0, 0xC9, 0x75, 0x98, 0xA2, 0xBD, 0x25, 0x55, 0xD1, 0xAA, 0x8C, 0xB0, 0x8E, 0x48,
            0x59, 0x0D, 0xBB, 0x3D, 0xA7, 0xB0, 0x8B, 0x10, 0x56, 0x82, 0x88, 0x38, 0xC5, 0xF6, 0x1E, 0x63, 0x93, 0xBA,
            0x7A, 0x0A, 0xBC, 0xC9, 0xF6, 0x62,
        ])
        let expectedTag = Data([
            0x76, 0xFC, 0x6E, 0xCE, 0x0F, 0x4E, 0x17, 0x68, 0xCD, 0xDF, 0x88, 0x53, 0xBB, 0x2D, 0x55, 0x1B,
        ])
        let nonce: [UInt8] = [0xCA, 0xFE, 0xBA, 0xBE, 0xFA, 0xCE, 0xDB, 0xAD, 0xDE, 0xCA, 0xF8, 0x88]
        let ad: [UInt8] = [
            0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE, 0xEF, 0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE, 0xEF, 0xAB, 0xAD,
            0xDA, 0xD2,
        ]

        let gcmEnc = try! Aes256GcmEncryption(key: key, nonce: nonce, associatedData: ad)
        var ciphertext = plaintext
        try! gcmEnc.encrypt(&ciphertext)
        let tag = try! gcmEnc.computeTag()
        XCTAssertEqual(ciphertext, expectedCiphertext)
        XCTAssertEqual(tag, expectedTag)

        let gcmDec = try! Aes256GcmDecryption(key: key, nonce: nonce, associatedData: ad)
        var decrypted = ciphertext
        try! gcmDec.decrypt(&decrypted)
        XCTAssertEqual(decrypted, plaintext)
        XCTAssert(try! gcmDec.verifyTag(tag))

        let gcmEnc2 = try! Aes256GcmEncryption(key: key, nonce: nonce, associatedData: ad)
        var ciphertextSplit = plaintext
        try! gcmEnc2.encrypt(&ciphertextSplit[..<1])
        try! gcmEnc2.encrypt(&ciphertextSplit[1...])
        let tag2 = try! gcmEnc2.computeTag()
        XCTAssertEqual(ciphertextSplit, expectedCiphertext)
        XCTAssertEqual(tag2, expectedTag)

        let gcmDec2 = try! Aes256GcmDecryption(key: key, nonce: nonce, associatedData: ad)
        var decryptedSplit = ciphertext
        try! gcmDec2.decrypt(&decryptedSplit[..<1])
        try! gcmDec2.decrypt(&decryptedSplit[1...])
        XCTAssertEqual(decryptedSplit, plaintext)
        XCTAssert(try! gcmDec2.verifyTag(tag))
    }

    func testAesCtr() {
        let plainTextData = Data("Super🔥secret🔥test🔥data🏁🏁".utf8)
        let key = self.generateAesKey()
        var nonce = Array(repeating: UInt8(0), count: Aes256Ctr32.nonceLength)
        nonce.withUnsafeMutableBytes {
            try! fillRandom($0)
        }

        var encryptedData = plainTextData
        try! Aes256Ctr32.process(&encryptedData, key: key, nonce: nonce)
        var decryptedData = encryptedData
        try! Aes256Ctr32.process(&decryptedData, key: key, nonce: nonce)
        XCTAssertEqual(plainTextData, decryptedData)
    }

    func testAesCtrKat() {
        let key: [UInt8] = [
            0x60, 0x3D, 0xEB, 0x10, 0x15, 0xCA, 0x71, 0xBE, 0x2B, 0x73, 0xAE, 0xF0, 0x85, 0x7D, 0x77, 0x81, 0x1F, 0x35,
            0x2C, 0x07, 0x3B, 0x61, 0x08, 0xD7, 0x2D, 0x98, 0x10, 0xA3, 0x09, 0x14, 0xDF, 0xF4,
        ]
        let plaintext = Data([
            0xFD, 0x4C, 0x14, 0x72, 0x9F, 0x50, 0x04, 0xBA, 0x49, 0xD8, 0x32, 0xAD, 0x7B, 0xE8, 0x7C, 0x18, 0xF4, 0xFA,
            0xFB, 0x58, 0x96, 0x2B, 0x9A, 0x43, 0xC3, 0xBE, 0x41, 0x71, 0x3D, 0xED, 0x93, 0xDB, 0xF8, 0x54, 0xAC, 0x4C,
            0xA2, 0x62, 0x85, 0xB7, 0xF7, 0x6E, 0x04, 0xB8, 0xF8, 0xD4, 0xE7, 0xD9, 0xF7, 0x54, 0x8F, 0x9B, 0x46, 0x5C,
            0x8F, 0x71, 0x3C, 0x10, 0x6E, 0x9F, 0x63, 0xF5, 0x43, 0x05, 0x33, 0x1A, 0x49, 0x83, 0xA2, 0xF4, 0xB7, 0x18,
            0xDE, 0x29, 0xFA, 0x79, 0x4D, 0xA1, 0x2E, 0xEE, 0x80, 0x86, 0x42, 0xFA, 0xEF, 0xF8, 0x27, 0x1A, 0x0E, 0xA2,
            0x8E, 0x3C, 0xC8, 0x0E, 0xEB, 0x65, 0xA8, 0xEB, 0x61, 0xF6, 0x9D, 0x8B, 0xA9, 0x7F, 0x6B, 0xF9, 0x05, 0x44,
            0x53, 0xF5, 0x5E, 0xFB, 0x8F, 0x94, 0x22, 0x08, 0x1F, 0x16, 0x20, 0xFE, 0x44, 0xAC, 0xF9, 0x9E, 0x81, 0x12,
            0x2F, 0x73, 0xD3, 0xF9, 0x21, 0xD5, 0xE3, 0x39, 0x16, 0x54, 0xE9, 0x94, 0x79, 0x04, 0x98, 0x43, 0x75, 0xB7,
            0x25, 0xFD, 0xFB, 0xA8, 0x95, 0xC5, 0xCD, 0xE3, 0xD2, 0x25, 0xD7, 0xBE, 0x3A, 0x21, 0x3C, 0x39, 0x65, 0x17,
            0x8A, 0x7D, 0xC1, 0xE3, 0xB5, 0x52, 0xEC, 0x7B, 0x2F, 0xFD, 0x9C, 0x77, 0xEB, 0xCC, 0x24, 0x3C, 0x45, 0x00,
            0xDF, 0xDF, 0xBE, 0x3B, 0x75, 0x54, 0xAA, 0x42, 0x7C, 0x01, 0x30, 0x5B, 0xEC, 0x48, 0xD7, 0x1A, 0xF2, 0x7C,
            0x59, 0x11, 0xD1, 0xE6, 0x49, 0xC6, 0x20, 0xD2, 0x2C, 0xF5, 0xF3, 0xA5, 0xAE, 0xB9, 0x46, 0x86, 0x51, 0xDA,
            0x79, 0x6F, 0x36, 0x95, 0x22, 0xFA, 0xF9, 0x1E, 0xFA, 0xBF, 0x0F, 0xEB, 0xD3, 0x3F, 0xCA, 0x41, 0xC9, 0x53,
            0x46, 0x06, 0xA4, 0xEA, 0x01, 0x99, 0xB9, 0x04, 0xB2, 0x43, 0xBA, 0x9C, 0xB8, 0xF3, 0x7A, 0x79, 0x2D, 0xF0,
            0x2E, 0xFA, 0xB8, 0xF0, 0xE2, 0xE0, 0xCF, 0x1D, 0x57, 0x9D, 0xAB, 0xA0, 0x42, 0xCF, 0xE4, 0xC9, 0x43, 0x0A,
            0xD4, 0xED, 0xA7, 0x86, 0x05, 0x2F, 0xCF, 0x15, 0xE7, 0xAC, 0xFA, 0x27, 0x36, 0xAA, 0xB4, 0x59, 0x0F, 0x73,
            0x67, 0x5F, 0xA1, 0x80, 0x5F, 0xE2, 0x38, 0x92, 0xC6, 0x3E, 0x0C, 0xD0, 0x1D, 0x00, 0x69, 0x35, 0xA6, 0xE3,
            0xF8, 0xE1, 0x05, 0xA7, 0x54, 0x80, 0x3D, 0x00, 0xD9, 0x85, 0x7E, 0x49, 0x63, 0x6A, 0xB0, 0x34, 0x16, 0x41,
            0x56, 0x85, 0x6D, 0x58, 0xA2, 0x44, 0xEA, 0xD4, 0x75, 0x30, 0x0D, 0x93, 0xB3, 0x1E, 0x44, 0xB5, 0xBE, 0x3B,
            0xBF, 0x69, 0x94, 0xED, 0xB8, 0x95, 0x80, 0x4B, 0x4F, 0x1B, 0xAD, 0x43, 0xEC, 0xFE, 0x08, 0xB4, 0xE1, 0x30,
            0x14, 0x8B, 0x66, 0x9F, 0xE6, 0x20, 0xE4, 0xF7, 0x30, 0x34, 0xFC, 0x3E, 0x74, 0x82, 0x37, 0x87, 0x0B, 0xEC,
            0x3B, 0x1F, 0x51, 0x76, 0x84, 0x65, 0x4D, 0x1D, 0x6B, 0xC0, 0x74, 0xDD, 0xF7, 0xB7, 0x59, 0xA2, 0x40, 0x5F,
            0x78, 0xED, 0x84, 0xD1, 0x00, 0x6D, 0x25, 0xAF, 0x9B, 0xBC, 0x12, 0xD6, 0xC6, 0x32, 0xF5, 0xD5, 0x43, 0xDA,
            0x0C, 0xBE, 0x9E, 0xA8, 0x66, 0xB2, 0xC9, 0x21, 0x26, 0x00, 0x9C, 0x27, 0xAD, 0x59, 0x39, 0x4B, 0x76, 0x33,
            0x7D, 0xE2, 0x46, 0xB5, 0x08, 0x95, 0x31, 0x7E, 0x2E, 0x34, 0x5D, 0xF3, 0x62, 0x9A, 0x5F, 0x62, 0x27, 0xF6,
            0x45, 0x22, 0x86, 0x6E, 0x7A, 0x39, 0x12, 0x1C, 0xCC, 0x55, 0x2E, 0x3D, 0xAB, 0xC9, 0x89, 0xDC, 0xE0, 0x66,
            0xDE, 0xA3, 0x55, 0xF7, 0x88, 0xC5, 0xD9, 0x2A, 0xDA, 0x09, 0x99, 0x17, 0xA2, 0x97, 0xCF, 0xEF, 0xA8, 0x67,
            0xCE, 0x37, 0x65, 0x6F, 0xAC, 0x6A, 0x50, 0x79, 0x8C, 0x10, 0xB3, 0x94, 0xD5, 0xBA, 0x54, 0xF8, 0x5C, 0xF0,
            0xF7, 0xEF, 0x1E, 0xED, 0xDF, 0xCA, 0x1E, 0x53, 0xE9, 0x3F, 0x13, 0x49, 0x88, 0x8C, 0xC7, 0x45, 0x19, 0x0C,
            0x19, 0x6F, 0x84, 0xEC, 0xF0, 0x72, 0x12, 0x87, 0xCC, 0x59, 0x2D, 0x40, 0x6F, 0x0A, 0x6C, 0xC5, 0xA5, 0x52,
            0x94, 0xBF, 0x7A, 0xA3, 0xB3, 0x5F, 0x6C, 0xEF, 0xC6, 0x1C, 0xAB, 0x79, 0x4B, 0x12, 0x44, 0x43, 0x12, 0xB5,
            0xE5, 0x0E, 0xC0, 0x71, 0x2E, 0x22, 0x1C, 0xC9, 0x5E, 0x9E, 0x26, 0xE9, 0xC3, 0xD0, 0x00, 0x88, 0x1E, 0x79,
            0x2A, 0xFC, 0xB5, 0x86, 0x41, 0xB1, 0xA9, 0x46, 0x13, 0xD6, 0x4E, 0xC7, 0x2F, 0x3D, 0xB9, 0xAB, 0x65, 0xBA,
            0x07, 0xA4, 0xF0, 0x5B, 0x7E, 0x9E, 0xE7, 0xB3, 0x35, 0xD8, 0x6A, 0x06, 0xFC, 0xBD, 0xB8, 0xCB, 0xD6, 0x95,
            0xAE, 0xEF, 0x53, 0x96, 0x4A, 0x96, 0x5F, 0xFE, 0x4C, 0x6D, 0x7B, 0x4E, 0x58, 0x0A, 0xB1, 0x39, 0xF8, 0x42,
            0x2A, 0x70, 0x2E, 0x09, 0xEA, 0xCB, 0xEA, 0x5D, 0x51, 0x2C, 0x31, 0xA9, 0x55, 0xB3, 0xD6, 0x03, 0x10, 0xBE,
            0x2B, 0xBD, 0xD7, 0x34, 0x84, 0xBA, 0xE6, 0x61, 0x27, 0x91, 0xA1, 0x9D, 0xA3, 0xC7, 0xB0, 0xFD, 0x14, 0x87,
            0xE7, 0x21, 0x31, 0xA8, 0xF9, 0xCB, 0x80, 0x17, 0x90, 0xCE, 0x8A, 0x6E, 0x1E, 0x37, 0x86, 0x62, 0xCE, 0xDC,
            0xD5, 0xEE, 0x82, 0xBD, 0x39, 0x05, 0x76, 0xAC, 0xFE, 0x53, 0x34, 0xEC, 0xD9, 0xD9, 0x07, 0x27, 0x3A, 0xEF,
            0xE6, 0x70, 0x58, 0x91, 0x63, 0x88, 0x21, 0x06, 0x38, 0xE5, 0xE6, 0x0F, 0x20, 0xEE, 0x92, 0x38, 0x9B, 0x35,
            0x33, 0xFD, 0x6A, 0xFF, 0xD3, 0x30, 0x95, 0xB2, 0x3D, 0x16, 0x9F, 0x09, 0x13, 0x65, 0x7F, 0x03, 0x3B, 0x8D,
            0x5C, 0x4E, 0xA5, 0x17, 0xF1, 0x67, 0xC1, 0xD5, 0x3E, 0x03, 0x17, 0x87, 0xBB, 0xE6, 0xD5, 0xB5, 0x77, 0x24,
            0x5F, 0xFF, 0x81, 0x51, 0xCD, 0x8F, 0xDC, 0xC5, 0xD6, 0xC3, 0x2D, 0xF7, 0x0F, 0xB8, 0x04, 0x3D, 0x42, 0xF8,
            0x96, 0xCD, 0x51, 0x3B, 0x4C, 0x85, 0xCF, 0xF2, 0x92, 0x67, 0x6C, 0xF1, 0x3B, 0x6A, 0x19, 0x31, 0xE8, 0x77,
            0x27, 0xA5, 0x61, 0x71, 0x1A, 0x31, 0x05, 0xD9, 0xF3, 0x51, 0x9B, 0x90, 0xC9, 0x42, 0x9B, 0x5C, 0xD3, 0xED,
            0xAA, 0xE3, 0xEE, 0x33, 0x48, 0x26, 0xA3, 0xFD, 0x74, 0xD6, 0x17, 0x5B, 0x55, 0x89, 0xDB, 0x39, 0x2F, 0x95,
            0x6A, 0x67, 0xC5, 0xE6, 0x7B, 0xE5, 0x96, 0x56, 0xF1, 0xCB, 0x37, 0xE5, 0x2C, 0x63, 0x6B, 0x26, 0x92, 0xA6,
            0x0C, 0x20, 0x44, 0x32, 0x74, 0x72, 0xFA, 0x9A, 0xF6, 0x51, 0xAF, 0xBC, 0xF5, 0x5D, 0x83, 0x98, 0xA3, 0x1D,
            0x34, 0x30, 0x74, 0x93, 0x1A, 0x72, 0xD5, 0x48, 0x33, 0xB2, 0x9E, 0xF2, 0x1F, 0xCB, 0x6E, 0xF4, 0x19, 0xBB,
            0x56, 0x31, 0x35, 0x13, 0xE4, 0x6C, 0x65, 0xD8, 0x33, 0x67, 0x7D, 0xBB, 0x0F, 0x28, 0x13, 0xE9, 0xCE, 0x5E,
            0xF7, 0x06, 0x76, 0x10, 0x2C, 0xA0, 0xD3, 0xC1, 0x4B, 0xBD, 0xD6, 0x59, 0xA7, 0x49, 0x8F, 0xA0, 0x8C, 0xD3,
            0x59, 0xD4, 0x28, 0xA8, 0x03, 0xAE, 0xFC, 0xC6, 0x60, 0xE9, 0xFC, 0x70, 0x4E, 0x9B, 0xAC, 0xC5, 0xF1, 0xD2,
            0x7F, 0x25, 0x28, 0xD4, 0x6B, 0x3F, 0xCA, 0xA2, 0xD4, 0x7D, 0xFA, 0x28, 0xBF, 0x4C,
        ])
        let expectedCiphertext = Data([
            0xF0, 0xC1, 0xDD, 0x48, 0xE5, 0x84, 0x3E, 0xB0, 0x3D, 0xE5, 0xAB, 0xB2, 0x98, 0x69, 0x7D, 0xC0, 0xF1, 0x03,
            0xA9, 0xD0, 0xC2, 0x30, 0x62, 0x0B, 0xCD, 0x86, 0x46, 0x77, 0x58, 0x37, 0x9D, 0xAA, 0x01, 0xAE, 0x18, 0x08,
            0x7D, 0x96, 0x09, 0x6A, 0x88, 0x14, 0xE9, 0x88, 0x08, 0xAB, 0x9B, 0x9C, 0x94, 0x39, 0x17, 0x27, 0x30, 0x54,
            0x20, 0x1C, 0xA3, 0xCD, 0xF2, 0xD4, 0x9F, 0x3A, 0xC7, 0x89, 0x6D, 0x34, 0xDB, 0x1C, 0xB1, 0xD7, 0x95, 0x9B,
            0x4D, 0xD5, 0x03, 0xF7, 0xB2, 0x5B, 0x33, 0x90, 0xE0, 0xDB, 0xCA, 0xCB, 0x15, 0xBB, 0xE8, 0x97, 0x82, 0x36,
            0xD7, 0x5A, 0xE2, 0x4D, 0x7C, 0xA0, 0xC4, 0xD5, 0x16, 0x84, 0x6E, 0xC0, 0xCC, 0x0E, 0x05, 0xB5, 0x05, 0xB3,
            0xD9, 0xD1, 0xC6, 0xE5, 0x01, 0x65, 0x91, 0x8C, 0x26, 0x67, 0x2E, 0xD1, 0x52, 0x52, 0x65, 0xB2, 0x9F, 0x63,
            0x36, 0x13, 0x8C, 0xED, 0xCA, 0x58, 0xE7, 0xF4, 0x47, 0xA8, 0x1B, 0x94, 0x85, 0xF7, 0x43, 0xB5, 0xE0, 0x1F,
            0xD5, 0xA5, 0x43, 0xF1, 0x8D, 0x93, 0x35, 0xC5, 0xE2, 0xD1, 0x9C, 0xAE, 0x82, 0x45, 0xA9, 0x22, 0x4A, 0x2B,
            0xAA, 0xBD, 0xF7, 0x67, 0x0E, 0x47, 0xBD, 0x22, 0xCF, 0x46, 0x5D, 0xF8, 0x56, 0x36, 0x21, 0x12, 0x4A, 0x80,
            0x91, 0x32, 0x5C, 0x67, 0x0E, 0x4F, 0x8F, 0xA0, 0x28, 0x68, 0x65, 0x05, 0xCE, 0xE8, 0x7D, 0x52, 0xD6, 0x3D,
            0x19, 0x65, 0xE6, 0x5D, 0xAF, 0x61, 0xF5, 0xE1, 0xB0, 0x0A, 0xE3, 0x3D, 0x4E, 0x5A, 0x42, 0x49, 0x69, 0x50,
            0xE8, 0xD7, 0x57, 0x10, 0xCF, 0x8C, 0x47, 0x71, 0x8F, 0x60, 0x71, 0x85, 0x0D, 0x11, 0xB5, 0x52, 0xE1, 0x9B,
            0xA0, 0xFA, 0xBE, 0xF5, 0xCC, 0xC7, 0x81, 0x3B, 0xA4, 0xBD, 0x0B, 0x59, 0x36, 0x94, 0xB3, 0x17, 0xF0, 0x4F,
            0xBE, 0x9C, 0xAF, 0x48, 0xAF, 0xF1, 0x4A, 0x45, 0x55, 0xF7, 0x8A, 0xB0, 0x56, 0xD4, 0x14, 0x87, 0x47, 0xC7,
            0xBD, 0x5A, 0x8B, 0x6E, 0x4B, 0xC8, 0x5D, 0x42, 0xAA, 0xE4, 0xE2, 0x63, 0x4A, 0xD9, 0x02, 0x8E, 0x5F, 0x32,
            0x34, 0x5A, 0x68, 0x13, 0xC2, 0x91, 0x58, 0x83, 0x62, 0xA7, 0xEC, 0xF6, 0xE0, 0xC3, 0xB3, 0xA3, 0xDB, 0x9D,
            0xBA, 0xA8, 0x2D, 0x27, 0x54, 0x96, 0x2F, 0x5D, 0x9B, 0x3E, 0x0F, 0xD1, 0x66, 0xCB, 0x11, 0xB5, 0x25, 0x40,
            0x81, 0x41, 0x7D, 0xAC, 0x0E, 0x35, 0xC0, 0x0B, 0x56, 0xEB, 0xEB, 0xD1, 0x21, 0x12, 0xAE, 0x20, 0x2C, 0x09,
            0x4F, 0xE3, 0xB2, 0x42, 0x52, 0xF0, 0x78, 0x7F, 0xB0, 0x9C, 0x6C, 0x51, 0x03, 0x6C, 0xEA, 0xC6, 0xDD, 0xDE,
            0x4A, 0xC5, 0x9A, 0xAD, 0xA7, 0xC7, 0x6B, 0xC7, 0x9E, 0x95, 0x0B, 0x66, 0xFF, 0xE6, 0xA0, 0x15, 0x45, 0x0E,
            0x87, 0x70, 0xC8, 0xB2, 0xB4, 0x91, 0xCC, 0xEC, 0x76, 0x10, 0xBF, 0x9A, 0x7F, 0x52, 0x3E, 0x5A, 0x57, 0x9F,
            0xF6, 0x4C, 0x62, 0x70, 0x0A, 0x7E, 0x83, 0x04, 0x13, 0x9C, 0x68, 0xCF, 0xDA, 0xB3, 0x4F, 0x7A, 0xD1, 0x8B,
            0x89, 0x89, 0xA9, 0x80, 0x2E, 0xD9, 0xDD, 0x39, 0x3D, 0x88, 0x9C, 0xF4, 0xD5, 0x26, 0xC9, 0xB5, 0x3F, 0xDB,
            0x0B, 0x78, 0xDC, 0xFA, 0xD4, 0x7B, 0x88, 0xC2, 0x3D, 0x69, 0x92, 0xE0, 0xE6, 0x3C, 0x31, 0xF8, 0x0D, 0x69,
            0xB4, 0x27, 0xEA, 0x7E, 0x71, 0x94, 0x4A, 0x61, 0x01, 0x3A, 0x0C, 0x70, 0xB2, 0xE9, 0xCF, 0xE2, 0x33, 0xA6,
            0x1C, 0xB4, 0x93, 0x9D, 0x2F, 0xDD, 0xE7, 0x5E, 0x6F, 0xF8, 0xFE, 0xE6, 0xB4, 0x5D, 0x48, 0x1A, 0xD0, 0xAD,
            0x01, 0x10, 0x46, 0x9E, 0xDF, 0xFC, 0x01, 0xB1, 0xBF, 0x2E, 0x4F, 0x14, 0x14, 0xF9, 0x25, 0xD8, 0x6A, 0xD1,
            0x98, 0xA2, 0x7A, 0x03, 0x88, 0x63, 0x7E, 0xDC, 0x7D, 0xD5, 0x47, 0xB8, 0xAE, 0xCA, 0x86, 0xEC, 0xCB, 0x3A,
            0xD5, 0xC0, 0x61, 0x5A, 0xF8, 0x42, 0x80, 0x96, 0xC8, 0x14, 0x2D, 0x75, 0x23, 0x5C, 0x46, 0x59, 0x95, 0xE5,
            0xEF, 0xF6, 0x22, 0x5E, 0x94, 0x91, 0x34, 0x57, 0x55, 0x1C, 0x1C, 0x18, 0x5E, 0x1D, 0x7B, 0xFA, 0x24, 0x37,
            0xAB, 0x56, 0xDA, 0x49, 0x95, 0x48, 0x34, 0x62, 0x8A, 0xC4, 0x80, 0xD7, 0xBA, 0xDA, 0x35, 0xEC, 0xBC, 0x34,
            0xDC, 0x6E, 0xFE, 0xB2, 0x60, 0x09, 0xC8, 0x2A, 0x0C, 0xC3, 0xF4, 0x77, 0x75, 0x7A, 0x91, 0xDC, 0x6D, 0x65,
            0x2C, 0xE7, 0xED, 0xD8, 0x2C, 0xB8, 0x91, 0xBA, 0x3B, 0x49, 0xBF, 0xEB, 0x74, 0xBD, 0x2A, 0x35, 0xB3, 0xF5,
            0xBC, 0xE7, 0x4A, 0x34, 0x35, 0x9D, 0xC0, 0x0D, 0xB8, 0xE0, 0x96, 0x1C, 0xB9, 0x75, 0x8C, 0xD9, 0x9E, 0xF2,
            0x5C, 0xF7, 0x18, 0x97, 0x4D, 0x60, 0xED, 0x5E, 0x77, 0x33, 0xF5, 0x25, 0xC8, 0x1E, 0xDB, 0x04, 0x64, 0xC7,
            0x93, 0x0A, 0xDD, 0x3E, 0x93, 0x36, 0xD8, 0x71, 0x5A, 0xEB, 0x37, 0xBB, 0x62, 0x48, 0x44, 0x24, 0x6A, 0x19,
            0xD4, 0x33, 0xC0, 0xED, 0x61, 0x5C, 0x22, 0x1E, 0x5E, 0x89, 0x74, 0x5D, 0x24, 0x67, 0x74, 0x37, 0x73, 0x56,
            0x06, 0x39, 0x89, 0x4B, 0x1A, 0xBD, 0x0F, 0x6E, 0x52, 0x89, 0xB5, 0x82, 0x6C, 0xEE, 0x5F, 0xCA, 0x76, 0xBD,
            0xD6, 0xD0, 0xD4, 0xDD, 0x69, 0xFB, 0x4A, 0x50, 0xD7, 0xD8, 0x14, 0xA4, 0x8C, 0x7E, 0x35, 0x92, 0x0A, 0xBB,
            0x8F, 0x0C, 0x1E, 0x60, 0xBA, 0x92, 0xD6, 0x12, 0xF4, 0xF4, 0xBF, 0x56, 0x95, 0xA0, 0x89, 0xDE, 0x63, 0x9B,
            0xFB, 0xC6, 0xF3, 0x17, 0xF4, 0xFD, 0x89, 0x5D, 0x32, 0x57, 0xEF, 0xBE, 0x1D, 0x49, 0xE9, 0x44, 0xB8, 0x2B,
            0xAD, 0xD4, 0xB2, 0x11, 0x64, 0xD4, 0xBA, 0xE7, 0xA8, 0x72, 0xF1, 0x83, 0xA3, 0xC8, 0x38, 0x5F, 0x54, 0xFD,
            0xD8, 0xF4, 0x71, 0x67, 0x21, 0x32, 0xDD, 0x44, 0xE5, 0x1C, 0xCD, 0xCF, 0xE1, 0x83, 0xC0, 0xCE, 0x00, 0x03,
            0x2A, 0x04, 0x88, 0x66, 0xAF, 0x6D, 0xFE, 0xA9, 0xE1, 0x5B, 0x58, 0xA1, 0x70, 0x93, 0x20, 0xE8, 0xFC, 0xA1,
            0x6D, 0xEF, 0xEA, 0xB2, 0x33, 0x02, 0x7A, 0x9E, 0xA3, 0x11, 0x8A, 0x52, 0x1C, 0x94, 0xBE, 0x5C, 0x48, 0xA7,
            0x2D, 0xE9, 0xC6, 0xFA, 0xBF, 0x21, 0x96, 0xE1, 0x23, 0xFC, 0x13, 0x56, 0xDE, 0xA2, 0x23, 0x71, 0x25, 0x99,
            0x75, 0x8A, 0x2F, 0x6F, 0xFE, 0x91, 0x92, 0x1C, 0x1A, 0xCE, 0xE3, 0xEC, 0x6C, 0x7B, 0x7A, 0x29, 0xA1, 0xD3,
            0xC5, 0xF8, 0x8A, 0xE6, 0xFB, 0x50, 0xB4, 0x2E, 0x36, 0xC0, 0x77, 0x37, 0x31, 0xE2, 0x8C, 0xA3, 0xC9, 0x3A,
            0x18, 0x62, 0x7D, 0x28, 0x7E, 0xD5, 0xF5, 0x38, 0x69, 0x14, 0x21, 0xDF, 0xFD, 0x36, 0xE3, 0xBB, 0x87, 0x18,
            0x54, 0xBC, 0x58, 0x5F, 0x36, 0x7E, 0xDB, 0xE7, 0x0B, 0x02, 0x9F, 0x81, 0xF3, 0x60, 0x59, 0x82, 0xEA, 0xFA,
            0x41, 0x35, 0xE5, 0x4B, 0x78, 0xD0, 0xC6, 0xCD, 0xF1, 0x8A, 0xFE, 0x22, 0xFF, 0x73, 0x08, 0xDA, 0x70, 0x11,
            0xF1, 0x5D, 0x35, 0x24, 0x90, 0x6F, 0x10, 0xFB, 0x6B, 0x78, 0x0F, 0xA9, 0xCC, 0x4B,
        ])
        let nonce: [UInt8] = [0xA6, 0xAA, 0xD9, 0xEC, 0xED, 0x14, 0xBF, 0x1C, 0x61, 0x91, 0x0D, 0xBA, 0, 0, 0, 35]

        var ciphertext = plaintext
        try! Aes256Ctr32.process(&ciphertext, key: key, nonce: nonce)
        XCTAssertEqual(ciphertext, expectedCiphertext)
    }
}
