//
// Copyright 2020-2021 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//

import Foundation
import SignalFfi

public enum Direction: Sendable {
    case sending
    case receiving
}

public enum IdentityChange: Sendable {
    case newOrUnchanged
    case replacedExisting
}

/// A marker protocol, which must be downcast to use in any particular store.
///
/// Essentially `Any`, but still able to catch typos when calling something that uses stores.
public protocol StoreContext {}

public protocol IdentityKeyStore: AnyObject {
    func identityKeyPair(context: StoreContext) throws -> IdentityKeyPair
    func localRegistrationId(context: StoreContext) throws -> UInt32
    func saveIdentity(
        _ identity: IdentityKey,
        for address: ProtocolAddress,
        context: StoreContext
    ) throws -> IdentityChange
    func isTrustedIdentity(
        _ identity: IdentityKey,
        for address: ProtocolAddress,
        direction: Direction,
        context: StoreContext
    ) throws -> Bool
    func identity(for address: ProtocolAddress, context: StoreContext) throws -> IdentityKey?
}

public protocol PreKeyStore: AnyObject {
    func loadPreKey(id: UInt32, context: StoreContext) throws -> PreKeyRecord
    func storePreKey(_ record: PreKeyRecord, id: UInt32, context: StoreContext) throws
    func removePreKey(id: UInt32, context: StoreContext) throws
}

public protocol SignedPreKeyStore: AnyObject {
    func loadSignedPreKey(id: UInt32, context: StoreContext) throws -> SignedPreKeyRecord
    func storeSignedPreKey(_ record: SignedPreKeyRecord, id: UInt32, context: StoreContext) throws
}

public protocol KyberPreKeyStore: AnyObject {
    func loadKyberPreKey(id: UInt32, context: StoreContext) throws -> KyberPreKeyRecord
    func storeKyberPreKey(_ record: KyberPreKeyRecord, id: UInt32, context: StoreContext) throws
    @available(*, deprecated, message: "use markKyberPreKeyUsed(id:signedPreKeyId:baseKey:context:) instead")
    func markKyberPreKeyUsed(id: UInt32, context: StoreContext) throws
    func markKyberPreKeyUsed(id: UInt32, signedPreKeyId: UInt32, baseKey: PublicKey, context: StoreContext) throws
}
extension KyberPreKeyStore {
    // Provide a default for backwards compatibility...
    @available(*, deprecated, message: "implement markKyberPreKeyUsed(id:signedPreKeyId:baseKey:context:) manually")
    public func markKyberPreKeyUsed(
        id: UInt32,
        signedPreKeyId: UInt32,
        baseKey: PublicKey,
        context: StoreContext
    ) throws {
        try markKyberPreKeyUsed(id: id, context: context)
    }

    // ...and one for *forwards* compatibility, until this overload gets removed.
    public func markKyberPreKeyUsed(id: UInt32, context: StoreContext) throws {
        fatalError("implement markKyberPreKeyUsed(id:signedPreKeyId:baseKey:context:)")
    }
}

public protocol SessionStore: AnyObject {
    func loadSession(for address: ProtocolAddress, context: StoreContext) throws -> SessionRecord?
    func loadExistingSessions(for addresses: [ProtocolAddress], context: StoreContext) throws -> [SessionRecord]
    func storeSession(_ record: SessionRecord, for address: ProtocolAddress, context: StoreContext) throws
}

public protocol SenderKeyStore: AnyObject {
    func storeSenderKey(
        from sender: ProtocolAddress,
        distributionId: UUID,
        record: SenderKeyRecord,
        context: StoreContext
    ) throws
    func loadSenderKey(
        from sender: ProtocolAddress,
        distributionId: UUID,
        context: StoreContext
    ) throws -> SenderKeyRecord?
}
