//
// Copyright 2024 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//

import ByteArray from '../internal/ByteArray.js';
import Native from '../../../Native.js';
import GroupSendFullToken from './GroupSendFullToken.js';

// For docs
import type GroupSendEndorsementsResponse from './GroupSendEndorsementsResponse.js';

/**
 * A minimal cacheable representation of an endorsement.
 *
 * This contains the minimal information needed to represent this specific endorsement; it must be
 * converted to a {@link GroupSendFullToken} before sending to the chat server. (It is valid to do
 * this immediately; it just uses up extra space.)
 *
 * Generated by {@link GroupSendEndorsement#toToken}.
 */
export default class GroupSendToken extends ByteArray {
  constructor(contents: Uint8Array) {
    super(contents, Native.GroupSendToken_CheckValidContents);
  }

  /**
   * Converts this token to a "full token", which can be sent to the chat server as authentication.
   *
   * {@code expiration} must be the same expiration that was in the original
   * {@link GroupSendEndorsementsResponse}, or the resulting token will fail to verify.
   */
  toFullToken(expiration: Date): GroupSendFullToken {
    return new GroupSendFullToken(
      Native.GroupSendToken_ToFullToken(
        this.contents,
        Math.floor(expiration.getTime() / 1000)
      )
    );
  }
}
