//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// 基本オブジェクト
//
// オブジェクトはログ出力機能 (とそのための自身のオブジェクト名) を持つ。

#include "nono.h"
#include "object.h"
#include "logger.h"
#include "mainapp.h"

// コンストラクタ
Object::Object(uint objid_)
{
	objid = objid_;
	assertmsg(objid < idname.size(),
		"objid=%u, size=%zu", objid, idname.size());

	// オブジェクト名(ログ表示名)は ID から引く。
	// (コンストラクタでは SetName() のオーバーライド版は呼ばれないことに注意)
	SetName(idname[objid]);

	// コンストラクタで指定した場合はエイリアスにも登録しておく。
	// ただし OBJ_NONE ならログを出すことも想定していないはずなので
	// エイリアスも追加しない。
	if (objid != OBJ_NONE) {
		AddAlias(GetName());
	}

	// 自分自身をリストに繋ぐ
	gMainApp.RegistObject(this);
}

// デストラクタ
Object::~Object()
{
	// 自分自身をリストから削除する
	gMainApp.UnregistObject(this);
}

// ログレベルを設定する。
void
Object::SetLogLevel(int loglevel_)
{
	loglevel = loglevel_;
}

// メッセージ表示
// 常に表示 (あらかじめ loglevel の評価をした後で呼び出す用)
void
Object::putmsgn(const char *fmt, ...) const
{
	char buf[1024];
	va_list ap;
	int len;

	// 名前だけ生表示されても分かりづらいので括弧でもつけておく
	len = snprintf(buf, sizeof(buf), "(%s) ", GetName().c_str());

	va_start(ap, fmt);
	vsnprintf(buf + len, sizeof(buf) - len, fmt, ap);
	va_end(ap);

	WriteLog(buf);
}

// ログ表示 (継承側で用意すること)
void
Object::putlogn(const char *fmt, ...) const
{
	PANIC("Object::putlogn called");
}

// ロガーに出力する共通部分
void
Object::WriteLog(const char *buf) const
{
	logger->WriteLog(buf);
}

// このオブジェクトの ID を文字列にして返す
const char *
Object::GetIdStr() const
{
	return GetIdStr(GetId());
}

// 指定のオブジェクト ID を文字列にして返す
/*static*/ const char *
Object::GetIdStr(uint target_id)
{
	if (target_id < idname.size()) {
		return idname[target_id];
	}
	return "unknownID?";
}

// 指定のオブジェクト ID を持つオブジェクトを返す。
/*static*/ Object *
Object::GetObject(uint target_id)
{
	return gMainApp.GetObject(target_id);
}
