/**
 * Copyright (C) 2022 MongoDB, Inc.  All Rights Reserved.
 */

#define MONGO_LOGV2_DEFAULT_COMPONENT ::mongo::logv2::LogComponent::kQuery

#include "mongot_cursor.h"

#include "document_source_internal_search_id_lookup.h"
#include "document_source_internal_search_mongot_remote.h"
#include "mongo/db/pipeline/document_source.h"
#include "mongo/db/pipeline/search_helper.h"
#include "mongo/logv2/log.h"
#include "mongo/rpc/get_status_from_command_result.h"
#include "mongot_options.h"
#include "mongot_task_executor.h"
#include "search/plan_sharded_search_gen.h"

namespace mongo::mongot_cursor {
MONGO_FAIL_POINT_DEFINE(shardedSearchOpCtxDisconnect);

boost::optional<std::string> SearchImplementedHelperFunctions::validatePipelineForShardedCollection(
    const Pipeline& pipeline) {
    auto firstStage = pipeline.peekFront();
    if (!firstStage ||
        StringData(firstStage->getSourceName()) !=
            DocumentSourceInternalSearchMongotRemote::kStageName) {
        return boost::none;
    }
    for (const auto& source : pipeline.getSources()) {
        DepsTracker dep;
        if (DepsTracker::State::NOT_SUPPORTED == source->getDependencies(&dep)) {
            // In a perfect world we would fail here as we don't know if $$SEARCH_META has been
            // accessed. However, we can't fail all queries with stages that don't support
            // dependency tracking.
            continue;
        } else if (dep.hasVariableReferenceTo({Variables::kSearchMetaId})) {
            return std::string("$$SEARCH_META cannot be used in a sharded environment");
        }
    }
    return boost::none;
}
namespace {
ServiceContext::ConstructorActionRegisterer searchQueryImplementation{
    "searchQueryImplementation", {"searchQueryHelperRegisterer"}, [](ServiceContext* context) {
        invariant(context);
        getSearchHelpers(context) = std::make_unique<SearchImplementedHelperFunctions>();
    }};
}  // namespace

/**
 * Create the RemoteCommandRequest for the provided command.
 */
executor::RemoteCommandRequest getRemoteCommandRequest(const ExpressionContext& expCtx,
                                                       const BSONObj& cmdObj) {
    uassert(7501001,
            str::stream() << "$search not enabled! "
                          << "Enable Search by setting serverParameter mongotHost to a valid "
                          << "\"host:port\" string",
            globalMongotParams.enabled);
    auto swHostAndPort = HostAndPort::parse(globalMongotParams.host);
    // This host and port string is configured and validated at startup.
    invariant(swHostAndPort.getStatus().isOK());
    executor::RemoteCommandRequest rcr(executor::RemoteCommandRequest(
        swHostAndPort.getValue(), expCtx.ns.db().toString(), cmdObj, expCtx.opCtx));
    rcr.sslMode = transport::ConnectSSLMode::kDisableSSL;
    return rcr;
}

void planShardedSearch(const ExpressionContext& expCtx, const BSONObj& searchRequest) {
    auto taskExecutor = executor::getMongotTaskExecutor(expCtx.opCtx->getServiceContext());

    // Create a PSS request.
    auto cmdObj = [&] {
        PlanShardedSearchSpec cmd;
        cmd.setPlanShardedSearch(expCtx.ns.coll());
        cmd.setQuery(searchRequest);
        cmd.setSearchFeatures(BSONObj());
        return cmd.toBSON();
    }();

    executor::RemoteCommandResponse response =
        Status(ErrorCodes::InternalError, "Internal error running search command");
    executor::TaskExecutor::CallbackHandle cbHnd =
        uassertStatusOKWithContext(taskExecutor->scheduleRemoteCommand(
                                       getRemoteCommandRequest(expCtx, cmdObj),
                                       [&response](const auto& args) { response = args.response; }),
                                   str::stream() << "Failed to execute search command " << cmdObj);

    if (MONGO_likely(shardedSearchOpCtxDisconnect.shouldFail())) {
        expCtx.opCtx->markKilled();
    }
    // It is imperative to wrap the wait() call in a try/catch. If an exception is thrown
    // and not caught, planShardedSearch will exit and all stack-allocated variables will be
    // destroyed. Then later when the executor thread tries to run the callbackFn of
    // scheduleRemoteCommand (the lambda above), it will try to access the `response` var,
    // which had been captured by reference and thus lived on the stack and therefore
    // destroyed as part of stack unwinding, and the server will segfault.

    // By catching the exception and then wait-ing for the callbackFn to run, we
    // ensure that planShardedSearch isn't exited (and the `response` object isn't
    // destroyed) before the callbackFn (which has a reference to `response`) executes.
    try {
        taskExecutor->wait(cbHnd, expCtx.opCtx);
    } catch (const DBException& exception) {
        LOGV2_ERROR(
            8049900,
            "An interruption occured while the MongotTaskExecutor was waiting for a response",
            "error"_attr = exception.toStatus());
        // If waiting for the response is interrupted, like by a ClientDisconnectError, then we
        // still have a callback out and registered with the TaskExecutor to run when the response
        // finally does come back. Since the callback references local state, cbResponse, it would
        // be invalid for the callback to run after leaving the this function. Therefore, we cancel
        // the callback and wait uninterruptably for the callback to be run.
        taskExecutor->cancel(cbHnd);
        taskExecutor->wait(cbHnd);
        throw;
    }
    uassertStatusOKWithContext(
        getStatusFromCommandResult(response.data),
        str::stream() << "mongot returned an error when executing planShardedSearch command: "
                      << response.toString());
}

}  // namespace mongo::mongot_cursor
