/**
 * Copyright (C) 2021 MongoDB, Inc.  All Rights Reserved.
 */

#include "document_source_search_meta.h"

#include "document_source_internal_search_mongot_remote.h"
#include "mongo/db/pipeline/document_source.h"
#include "mongo/db/pipeline/document_source_documents.h"
#include "mongo/db/pipeline/document_source_limit.h"
#include "mongo/db/pipeline/document_source_replace_root.h"
#include "mongo/db/pipeline/document_source_union_with.h"
#include "mongo/db/pipeline/expression_context.h"

namespace mongo {

using boost::intrusive_ptr;
using std::list;

REGISTER_DOCUMENT_SOURCE_WITH_MIN_VERSION(
    searchMeta,
    DocumentSourceSearchMeta::LiteParsed::parse,
    DocumentSourceSearchMeta::createFromBson,
    ServerGlobalParams::FeatureCompatibility::Version::kFullyUpgradedTo44);

const char* DocumentSourceSearchMeta::getSourceName() const {
    return kStageName.rawData();
}

std::list<intrusive_ptr<DocumentSource>> DocumentSourceSearchMeta::createFromBson(
    BSONElement elem, const intrusive_ptr<ExpressionContext>& pExpCtx) {

    uassert(5858804,
            "Must set 'enableSearchMeta' to true to use '$searchMeta'",
            enableSearchMeta.load());

    uassert(ErrorCodes::FailedToParse,
            str::stream() << "$searchMeta value must be an object. Found: "
                          << typeName(elem.type()),
            elem.type() == BSONType::Object);

    // If we are parsing a view, we don't actually want to make any calls to mongot which happen
    // during desugaring.
    if (pExpCtx->isParsingViewDefinition) {
        return {intrusive_ptr<DocumentSourceSearchMeta>(
            new DocumentSourceSearchMeta(elem.Obj().getOwned(), pExpCtx))};
    }

    uassert(6600902,
            "Running search command in non-allowed context (update pipeline)",
            !pExpCtx->isParsingPipelineUpdate);

    std::list<intrusive_ptr<DocumentSource>> res{
        // Fetch the search results and populate $$SEARCH_META.
        DocumentSourceInternalSearchMongotRemote::createFromBson(elem, pExpCtx),
        // Replace any documents returned by mongot with the SEARCH_META contents.
        DocumentSourceReplaceRoot::createFromBson(
            BSON("$replaceRoot" << BSON("newRoot" << std::string("$$SEARCH_META"))).firstElement(),
            pExpCtx),
        // If mongot returned no documents, generate a document to return. This depends on the
        // current, undocumented behavior of $unionWith that the outer pipeline is iterated before
        // the inner pipeline.
        DocumentSourceUnionWith::createFromBson(
            BSON(DocumentSourceUnionWith::kStageName
                 << BSON("pipeline" << BSON_ARRAY(BSON(DocumentSourceDocuments::kStageName
                                                       << BSON_ARRAY("$$SEARCH_META")))))
                .firstElement(),
            pExpCtx.get()),
        // Only return one copy of the meta results.
        DocumentSourceLimit::create(pExpCtx, 1)};
    return res;
}

StageConstraints DocumentSourceSearchMeta::constraints(Pipeline::SplitState pipeState) const {
    StageConstraints constraints(StreamType::kStreaming,
                                 PositionRequirement::kFirst,
                                 HostTypeRequirement::kAnyShard,
                                 DiskUseRequirement::kNoDiskUse,
                                 FacetRequirement::kNotAllowed,
                                 TransactionRequirement::kNotAllowed,
                                 LookupRequirement::kNotAllowed,
                                 UnionRequirement::kNotAllowed,
                                 ChangeStreamRequirement::kBlacklist);
    constraints.requiresInputDocSource = false;
    return constraints;
}

Value DocumentSourceSearchMeta::serialize(
    boost::optional<ExplainOptions::Verbosity> explain) const {
    return Value(DOC(kStageName << Value(_userObj)));
}

}  // namespace mongo
