/**
 * Copyright (C) 2019 MongoDB, Inc.  All Rights Reserved.
 */

#include "mongo/platform/basic.h"

#include "document_source_search.h"

#include "document_source_internal_search_id_lookup.h"
#include "document_source_internal_search_mongot_remote.h"
#include "mongo/db/pipeline/document_source.h"
#include "mongo/db/pipeline/document_source_add_fields.h"
#include "mongo/db/pipeline/document_source_internal_shard_filter.h"
#include "mongo/db/pipeline/document_source_replace_root.h"
#include "mongo/db/pipeline/expression_context.h"
#include "mongot_cursor.h"

namespace mongo {

using boost::intrusive_ptr;
using std::list;

REGISTER_MULTI_STAGE_ALIAS(search,
                           DocumentSourceSearch::LiteParsed::parse,
                           DocumentSourceSearch::createFromBson);

// $searchBeta is supported as an alias for $search for compatibility with applications that used
// search during its beta period.
REGISTER_MULTI_STAGE_ALIAS(searchBeta,
                           DocumentSourceSearch::LiteParsed::parse,
                           DocumentSourceSearch::createFromBson);

const char* DocumentSourceSearch::getSourceName() const {
    return kStageName.rawData();
}

Value DocumentSourceSearch::serialize(boost::optional<ExplainOptions::Verbosity>) const {
    return Value(DOC(kStageName << Value(_userObj)));
}

StageConstraints DocumentSourceSearch::constraints(Pipeline::SplitState) const {
    StageConstraints constraints(StreamType::kStreaming,
                                 PositionRequirement::kFirst,
                                 HostTypeRequirement::kAnyShard,
                                 DiskUseRequirement::kNoDiskUse,
                                 FacetRequirement::kNotAllowed,
                                 TransactionRequirement::kNotAllowed,
                                 LookupRequirement::kNotAllowed,
                                 UnionRequirement::kNotAllowed,
                                 ChangeStreamRequirement::kBlacklist);
    constraints.requiresInputDocSource = false;
    return constraints;
}

std::list<intrusive_ptr<DocumentSource>> DocumentSourceSearch::createFromBson(
    BSONElement elem, const intrusive_ptr<ExpressionContext>& pExpCtx) {

    uassert(ErrorCodes::FailedToParse,
            str::stream() << "$search value must be an object. Found: " << typeName(elem.type()),
            elem.type() == BSONType::Object);

    // If we are parsing a view, we don't actually want to make any calls to mongot which happen
    // during desugaring.
    if (pExpCtx->isParsingViewDefinition) {
        return {intrusive_ptr<DocumentSourceSearch>(
            new DocumentSourceSearch(elem.Obj().getOwned(), pExpCtx))};
    }

    uassert(6600901,
            "Running search command in non-allowed context (update pipeline)",
            !pExpCtx->isParsingPipelineUpdate);

    // On v5.0 and below, we use planShardedSearch as a mechanism for query validation and ignore
    // its response, only asserting that it returned a success status. For example, on v5.0 and
    // below, we want to allow $search queries with a sort specified only for replica sets and fail
    // in other scenerios (sharded and unsharded collections in a sharded cluster).
    if (pExpCtx->mongoProcessInterface->inShardedEnvironment(pExpCtx->opCtx) &&
        MONGO_likely(!searchReturnEofImmediately.shouldFail())) {
        mongot_cursor::planShardedSearch(*pExpCtx, elem.Obj());
    }

    // If 'returnStoredSource' is true, we don't want to do idLookup. Instead, promote the fields in
    // 'storedSource' to root.
    // 'getBoolField' returns false if the field is not present.
    if (elem.Obj().getBoolField(kReturnStoredSourceArg)) {
        // {$replaceRoot: {newRoot: {$ifNull: ["$storedSource", "$$ROOT"]}}
        // 'storedSource' is not always present in the document from mongot. If it's present, use it
        // as the root. Otherwise keep the original document.
        BSONObj replaceRootSpec =
            BSON("$replaceRoot" << BSON(
                     "newRoot" << BSON(
                         "$ifNull" << BSON_ARRAY("$" + kProtocolStoredFieldsName << "$$ROOT"))));
        return {DocumentSourceInternalSearchMongotRemote::createFromBson(elem, pExpCtx),
                DocumentSourceReplaceRoot::createFromBson(replaceRootSpec.firstElement(), pExpCtx)};
    }
    return {
        DocumentSourceInternalSearchMongotRemote::createFromBson(elem, pExpCtx),
        new DocumentSourceInternalSearchIdLookUp(pExpCtx),
    };
}

}  // namespace mongo
